from __future__ import annotations

import json
from typing import Any, Dict, List, Optional

import requests

from ..utils.io import ensure_parent_dir


def fetch_all_questions(
    base_url: str,
    params: Dict[str, Any],
    headers: Optional[Dict[str, str]] = None,
    output_file: Optional[str] = None,
    max_pages: Optional[int] = None,
) -> List[Dict[str, Any]]:
    """Fetch paginated Metaculus posts and optionally stream to a JSON array file.

    - base_url: e.g., "https://www.metaculus.com/api/posts/"
    - params: initial query params for the first page
    - headers: request headers; include Authorization if required, e.g. {"Authorization": "Token <KEY>", "Accept": "application/json"}
    - output_file: if provided, writes a JSON array containing all results progressively
    - max_pages: optional safety limit on number of pages
    """
    url = base_url
    all_questions: List[Dict[str, Any]] = []
    page_count = 0

    if output_file:
        ensure_parent_dir(output_file)
        f = open(output_file, "w", encoding="utf-8")
        f.write("[")
        first = True
    else:
        f = None
        first = False

    try:
        last_results: Optional[List[Any]] = None
        while url:
            resp = requests.get(url, headers=headers or {}, params=params if url == base_url else None, timeout=60)
            if resp.status_code != 200:
                raise RuntimeError(f"HTTP {resp.status_code}: {resp.text}")
            data = resp.json() or {}
            results = data.get("results", [])
            if not results or results == last_results:
                break
            last_results = results

            for entry in results:
                if f is not None:
                    if not first:
                        f.write(",\n")
                    json.dump(entry, f, ensure_ascii=False)
                    first = False
                all_questions.append(entry)

            url = data.get("next") or None
            page_count += 1
            if max_pages and page_count >= max_pages:
                break
    finally:
        if f is not None:
            f.write("]")
            f.close()

    return all_questions
