from __future__ import annotations

from typing import Any, Dict, List

from tqdm import tqdm

from ..utils.io import read_json, write_json


def reformat_analyzed(
    input_file: str,
    output_file: str,
) -> None:
    """From analyzed questions (with history.best_news and history.trend),
    produce a flattened list with fields: id (running), question_id, title, open_time, start_time, trend, best_news.
    """
    questions_data: List[Dict[str, Any]] = read_json(input_file)

    reformatted: List[Dict[str, Any]] = []
    entry_id = 1
    for entry in tqdm(questions_data, desc="Reformatting Entries"):
        qid = entry.get("id")
        title = entry.get("title")
        open_time = entry.get("open_time", "Unknown")
        for hist in entry.get("history", []) or []:
            if "best_news" not in hist:
                continue
            trend = hist.get("trend")
            if trend is None:
                continue
            reformatted.append(
                {
                    "id": entry_id,
                    "question_id": qid,
                    "title": title,
                    "open_time": open_time,
                    "start_time": hist.get("start_time"),
                    "trend": trend,
                    "best_news": hist.get("best_news"),
                }
            )
            entry_id += 1
    write_json(output_file, reformatted, indent=2)
