# -*- coding: utf-8 -*-
# Spearmint
#
# Academic and Non-Commercial Research Use Software License and Terms
# of Use
#
# Spearmint is a software package to perform Bayesian optimization
# according to specific algorithms (the “Software”).  The Software is
# designed to automatically run experiments (thus the code name
# 'spearmint') in a manner that iteratively adjusts a number of
# parameters so as to minimize some objective in as few runs as
# possible.
#
# The Software was developed by Ryan P. Adams, Michael Gelbart, and
# Jasper Snoek and at Harvard University, Kevin Swersky and Richard
# Zemel at the University of Toronto (“Toronto”), and Hugo Larochelle
# at the Université de Sherbrooke (“Sherbrooke”), which assigned its
# rights in the Software to Socpra Sciences et Génie
# S.E.C. (“Socpra”). Pursuant to an inter-institutional agreement
# between the parties, it is distributed for free academic and
# non-commercial research use by the President and Fellows of Harvard
# College (“Harvard”).
#
# Using the Software indicates your agreement to be bound by the terms
# of this Software Use Agreement (“Agreement”). Absent your agreement
# to the terms below, you (the “End User”) have no rights to hold or
# use the Software whatsoever.
#
# Harvard agrees to grant hereunder the limited non-exclusive license
# to End User for the use of the Software in the performance of End
# User’s internal, non-commercial research and academic use at End
# User’s academic or not-for-profit research institution
# (“Institution”) on the following terms and conditions:
#
# 1.  NO REDISTRIBUTION. The Software remains the property Harvard,
# Toronto and Socpra, and except as set forth in Section 4, End User
# shall not publish, distribute, or otherwise transfer or make
# available the Software to any other party.
#
# 2.  NO COMMERCIAL USE. End User shall not use the Software for
# commercial purposes and any such use of the Software is expressly
# prohibited. This includes, but is not limited to, use of the
# Software in fee-for-service arrangements, core facilities or
# laboratories or to provide research services to (or in collaboration
# with) third parties for a fee, and in industry-sponsored
# collaborative research projects where any commercial rights are
# granted to the sponsor. If End User wishes to use the Software for
# commercial purposes or for any other restricted purpose, End User
# must execute a separate license agreement with Harvard.
#
# Requests for use of the Software for commercial purposes, please
# contact:
#
# Office of Technology Development
# Harvard University
# Smith Campus Center, Suite 727E
# 1350 Massachusetts Avenue
# Cambridge, MA 02138 USA
# Telephone: (617) 495-3067
# Facsimile: (617) 495-9568
# E-mail: otd@harvard.edu
#
# 3.  OWNERSHIP AND COPYRIGHT NOTICE. Harvard, Toronto and Socpra own
# all intellectual property in the Software. End User shall gain no
# ownership to the Software. End User shall not remove or delete and
# shall retain in the Software, in any modifications to Software and
# in any Derivative Works, the copyright, trademark, or other notices
# pertaining to Software as provided with the Software.
#
# 4.  DERIVATIVE WORKS. End User may create and use Derivative Works,
# as such term is defined under U.S. copyright laws, provided that any
# such Derivative Works shall be restricted to non-commercial,
# internal research and academic use at End User’s Institution. End
# User may distribute Derivative Works to other Institutions solely
# for the performance of non-commercial, internal research and
# academic use on terms substantially similar to this License and
# Terms of Use.
#
# 5.  FEEDBACK. In order to improve the Software, comments from End
# Users may be useful. End User agrees to provide Harvard with
# feedback on the End User’s use of the Software (e.g., any bugs in
# the Software, the user experience, etc.).  Harvard is permitted to
# use such information provided by End User in making changes and
# improvements to the Software without compensation or an accounting
# to End User.
#
# 6.  NON ASSERT. End User acknowledges that Harvard, Toronto and/or
# Sherbrooke or Socpra may develop modifications to the Software that
# may be based on the feedback provided by End User under Section 5
# above. Harvard, Toronto and Sherbrooke/Socpra shall not be
# restricted in any way by End User regarding their use of such
# information.  End User acknowledges the right of Harvard, Toronto
# and Sherbrooke/Socpra to prepare, publish, display, reproduce,
# transmit and or use modifications to the Software that may be
# substantially similar or functionally equivalent to End User’s
# modifications and/or improvements if any.  In the event that End
# User obtains patent protection for any modification or improvement
# to Software, End User agrees not to allege or enjoin infringement of
# End User’s patent against Harvard, Toronto or Sherbrooke or Socpra,
# or any of the researchers, medical or research staff, officers,
# directors and employees of those institutions.
#
# 7.  PUBLICATION & ATTRIBUTION. End User has the right to publish,
# present, or share results from the use of the Software.  In
# accordance with customary academic practice, End User will
# acknowledge Harvard, Toronto and Sherbrooke/Socpra as the providers
# of the Software and may cite the relevant reference(s) from the
# following list of publications:
#
# Practical Bayesian Optimization of Machine Learning Algorithms
# Jasper Snoek, Hugo Larochelle and Ryan Prescott Adams
# Neural Information Processing Systems, 2012
#
# Multi-Task Bayesian Optimization
# Kevin Swersky, Jasper Snoek and Ryan Prescott Adams
# Advances in Neural Information Processing Systems, 2013
#
# Input Warping for Bayesian Optimization of Non-stationary Functions
# Jasper Snoek, Kevin Swersky, Richard Zemel and Ryan Prescott Adams
# Preprint, arXiv:1402.0929, http://arxiv.org/abs/1402.0929, 2013
#
# Bayesian Optimization and Semiparametric Models with Applications to
# Assistive Technology Jasper Snoek, PhD Thesis, University of
# Toronto, 2013
#
# 8.  NO WARRANTIES. THE SOFTWARE IS PROVIDED "AS IS." TO THE FULLEST
# EXTENT PERMITTED BY LAW, HARVARD, TORONTO AND SHERBROOKE AND SOCPRA
# HEREBY DISCLAIM ALL WARRANTIES OF ANY KIND (EXPRESS, IMPLIED OR
# OTHERWISE) REGARDING THE SOFTWARE, INCLUDING BUT NOT LIMITED TO ANY
# IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
# PURPOSE, OWNERSHIP, AND NON-INFRINGEMENT.  HARVARD, TORONTO AND
# SHERBROOKE AND SOCPRA MAKE NO WARRANTY ABOUT THE ACCURACY,
# RELIABILITY, COMPLETENESS, TIMELINESS, SUFFICIENCY OR QUALITY OF THE
# SOFTWARE.  HARVARD, TORONTO AND SHERBROOKE AND SOCPRA DO NOT WARRANT
# THAT THE SOFTWARE WILL OPERATE WITHOUT ERROR OR INTERRUPTION.
#
# 9.  LIMITATIONS OF LIABILITY AND REMEDIES. USE OF THE SOFTWARE IS AT
# END USER’S OWN RISK. IF END USER IS DISSATISFIED WITH THE SOFTWARE,
# ITS EXCLUSIVE REMEDY IS TO STOP USING IT.  IN NO EVENT SHALL
# HARVARD, TORONTO OR SHERBROOKE OR SOCPRA BE LIABLE TO END USER OR
# ITS INSTITUTION, IN CONTRACT, TORT OR OTHERWISE, FOR ANY DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, CONSEQUENTIAL, PUNITIVE OR OTHER
# DAMAGES OF ANY KIND WHATSOEVER ARISING OUT OF OR IN CONNECTION WITH
# THE SOFTWARE, EVEN IF HARVARD, TORONTO OR SHERBROOKE OR SOCPRA IS
# NEGLIGENT OR OTHERWISE AT FAULT, AND REGARDLESS OF WHETHER HARVARD,
# TORONTO OR SHERBROOKE OR SOCPRA IS ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGES.
#
# 10. INDEMNIFICATION. To the extent permitted by law, End User shall
# indemnify, defend and hold harmless Harvard, Toronto and Sherbrooke
# and Socpra, their corporate affiliates, current or future directors,
# trustees, officers, faculty, medical and professional staff,
# employees, students and agents and their respective successors,
# heirs and assigns (the "Indemnitees"), against any liability,
# damage, loss or expense (including reasonable attorney's fees and
# expenses of litigation) incurred by or imposed upon the Indemnitees
# or any one of them in connection with any claims, suits, actions,
# demands or judgments arising from End User’s breach of this
# Agreement or its Institution’s use of the Software except to the
# extent caused by the gross negligence or willful misconduct of
# Harvard, Toronto or Sherbrooke or Socpra. This indemnification
# provision shall survive expiration or termination of this Agreement.
#
# 11. GOVERNING LAW. This Agreement shall be construed and governed by
# the laws of the Commonwealth of Massachusetts regardless of
# otherwise applicable choice of law standards.
#
# 12. NON-USE OF NAME.  Nothing in this License and Terms of Use shall
# be construed as granting End Users or their Institutions any rights
# or licenses to use any trademarks, service marks or logos associated
# with the Software.  You may not use the terms “Harvard” or
# “University of Toronto” or “Université de Sherbrooke” or “Socpra
# Sciences et Génie S.E.C.” (or a substantially similar term) in any
# way that is inconsistent with the permitted uses described
# herein. You agree not to use any name or emblem of Harvard, Toronto
# or Sherbrooke, or any of their subdivisions for any purpose, or to
# falsely suggest any relationship between End User (or its
# Institution) and Harvard, Toronto and/or Sherbrooke, or in any
# manner that would infringe or violate any of their rights.
#
# 13. End User represents and warrants that it has the legal authority
# to enter into this License and Terms of Use on behalf of itself and
# its Institution.

import numpy as np
import cPickle as pickle

# Numba autojit might be nice.  Currently asplodes.
def sobol(num_points, num_dims):
    """Get Sobol sequence with num_points for num_dims."""

    # Total number of bits we need for a sequence of length num_points.
    num_bits = int(np.ceil(np.log2(num_points)))

    # Compute the c_i sequence.
    X = to_binary(np.arange(num_points, dtype=np.uint32), num_bits)
    C = num_bits - np.max((1-X) * (np.arange(num_bits)+1), axis=1) + 1
    
    # Direction numbers
    V = np.zeros((num_dims, num_bits), dtype=np.uint32)

    # Direction numbers for first dimension.
    V[0,:] = 1 << np.arange(31, 31-num_bits, -1, dtype=np.uint32)

    Z = np.zeros((num_points, num_dims))

    # Get the parameters for the Sobol sequence.
    # The first entry in this array (index 0) is the second dimension.
    params = get_params()

    # Loop over dimensions
    for dd in xrange(1,num_dims):
        s = params[dd-1]['s']
        a = params[dd-1]['a']
        m = params[dd-1]['m']

        # Direction numbers for dd-th dimension.
        if (num_bits <= s):
            V[dd,:] = m << np.arange(31, 31-num_bits, -1, dtype=np.uint32)
        else:
            V[dd,:s] = m << np.arange(31, 31-s, -1, dtype=np.uint32)
            for s0 in xrange(s, num_bits):
                V[dd,s0] = V[dd,s0-s] ^ (V[dd,s0-s] >> s)
                for s1 in xrange(s-1):
                    V[dd,s0] = V[dd,s0] ^ (((a >> (s-2-s1)) & 1) * V[dd,s0-s1-1])

    X = np.zeros((num_points, num_dims), dtype=np.uint32)

    # Wish we could do this without recursion.
    # Fancy loop unrolling?
    for nn in xrange(1,num_points):
        X[nn,:] = X[nn-1,:] ^ V[:,C[nn-1]-1]

    Z = X.astype('double') / float(1<<32)

    return Z

def to_binary(X, bits):
    return 1 & (X[:,np.newaxis]/2**np.arange(bits-1,-1,-1, dtype=np.uint32))

# These are the parameters for the Sobol sequence.
# This is hilarious.
params = """(lp1
(dp2
S'a'
I0
sS's'
I1
sS'm'
(lp3
I1
asS'd'
I2
sa(dp4
S'a'
I1
sS's'
I2
sS'm'
(lp5
I1
aI3
asS'd'
I3
sa(dp6
S'a'
I1
sS's'
I3
sS'm'
(lp7
I1
aI3
aI1
asS'd'
I4
sa(dp8
S'a'
I2
sS's'
I3
sS'm'
(lp9
I1
aI1
aI1
asS'd'
I5
sa(dp10
S'a'
I1
sS's'
I4
sS'm'
(lp11
I1
aI1
aI3
aI3
asS'd'
I6
sa(dp12
S'a'
I4
sS's'
I4
sS'm'
(lp13
I1
aI3
aI5
aI13
asS'd'
I7
sa(dp14
S'a'
I2
sS's'
I5
sS'm'
(lp15
I1
aI1
aI5
aI5
aI17
asS'd'
I8
sa(dp16
S'a'
I4
sS's'
I5
sS'm'
(lp17
I1
aI1
aI5
aI5
aI5
asS'd'
I9
sa(dp18
S'a'
I7
sS's'
I5
sS'm'
(lp19
I1
aI1
aI7
aI11
aI19
asS'd'
I10
sa(dp20
S'a'
I11
sS's'
I5
sS'm'
(lp21
I1
aI1
aI5
aI1
aI1
asS'd'
I11
sa(dp22
S'a'
I13
sS's'
I5
sS'm'
(lp23
I1
aI1
aI1
aI3
aI11
asS'd'
I12
sa(dp24
S'a'
I14
sS's'
I5
sS'm'
(lp25
I1
aI3
aI5
aI5
aI31
asS'd'
I13
sa(dp26
S'a'
I1
sS's'
I6
sS'm'
(lp27
I1
aI3
aI3
aI9
aI7
aI49
asS'd'
I14
sa(dp28
S'a'
I13
sS's'
I6
sS'm'
(lp29
I1
aI1
aI1
aI15
aI21
aI21
asS'd'
I15
sa(dp30
S'a'
I16
sS's'
I6
sS'm'
(lp31
I1
aI3
aI1
aI13
aI27
aI49
asS'd'
I16
sa(dp32
S'a'
I19
sS's'
I6
sS'm'
(lp33
I1
aI1
aI1
aI15
aI7
aI5
asS'd'
I17
sa(dp34
S'a'
I22
sS's'
I6
sS'm'
(lp35
I1
aI3
aI1
aI15
aI13
aI25
asS'd'
I18
sa(dp36
S'a'
I25
sS's'
I6
sS'm'
(lp37
I1
aI1
aI5
aI5
aI19
aI61
asS'd'
I19
sa(dp38
S'a'
I1
sS's'
I7
sS'm'
(lp39
I1
aI3
aI7
aI11
aI23
aI15
aI103
asS'd'
I20
sa(dp40
S'a'
I4
sS's'
I7
sS'm'
(lp41
I1
aI3
aI7
aI13
aI13
aI15
aI69
asS'd'
I21
sa(dp42
S'a'
I7
sS's'
I7
sS'm'
(lp43
I1
aI1
aI3
aI13
aI7
aI35
aI63
asS'd'
I22
sa(dp44
S'a'
I8
sS's'
I7
sS'm'
(lp45
I1
aI3
aI5
aI9
aI1
aI25
aI53
asS'd'
I23
sa(dp46
S'a'
I14
sS's'
I7
sS'm'
(lp47
I1
aI3
aI1
aI13
aI9
aI35
aI107
asS'd'
I24
sa(dp48
S'a'
I19
sS's'
I7
sS'm'
(lp49
I1
aI3
aI1
aI5
aI27
aI61
aI31
asS'd'
I25
sa(dp50
S'a'
I21
sS's'
I7
sS'm'
(lp51
I1
aI1
aI5
aI11
aI19
aI41
aI61
asS'd'
I26
sa(dp52
S'a'
I28
sS's'
I7
sS'm'
(lp53
I1
aI3
aI5
aI3
aI3
aI13
aI69
asS'd'
I27
sa(dp54
S'a'
I31
sS's'
I7
sS'm'
(lp55
I1
aI1
aI7
aI13
aI1
aI19
aI1
asS'd'
I28
sa(dp56
S'a'
I32
sS's'
I7
sS'm'
(lp57
I1
aI3
aI7
aI5
aI13
aI19
aI59
asS'd'
I29
sa(dp58
S'a'
I37
sS's'
I7
sS'm'
(lp59
I1
aI1
aI3
aI9
aI25
aI29
aI41
asS'd'
I30
sa(dp60
S'a'
I41
sS's'
I7
sS'm'
(lp61
I1
aI3
aI5
aI13
aI23
aI1
aI55
asS'd'
I31
sa(dp62
S'a'
I42
sS's'
I7
sS'm'
(lp63
I1
aI3
aI7
aI3
aI13
aI59
aI17
asS'd'
I32
sa(dp64
S'a'
I50
sS's'
I7
sS'm'
(lp65
I1
aI3
aI1
aI3
aI5
aI53
aI69
asS'd'
I33
sa(dp66
S'a'
I55
sS's'
I7
sS'm'
(lp67
I1
aI1
aI5
aI5
aI23
aI33
aI13
asS'd'
I34
sa(dp68
S'a'
I56
sS's'
I7
sS'm'
(lp69
I1
aI1
aI7
aI7
aI1
aI61
aI123
asS'd'
I35
sa(dp70
S'a'
I59
sS's'
I7
sS'm'
(lp71
I1
aI1
aI7
aI9
aI13
aI61
aI49
asS'd'
I36
sa(dp72
S'a'
I62
sS's'
I7
sS'm'
(lp73
I1
aI3
aI3
aI5
aI3
aI55
aI33
asS'd'
I37
sa(dp74
S'a'
I14
sS's'
I8
sS'm'
(lp75
I1
aI3
aI1
aI15
aI31
aI13
aI49
aI245
asS'd'
I38
sa(dp76
S'a'
I21
sS's'
I8
sS'm'
(lp77
I1
aI3
aI5
aI15
aI31
aI59
aI63
aI97
asS'd'
I39
sa(dp78
S'a'
I22
sS's'
I8
sS'm'
(lp79
I1
aI3
aI1
aI11
aI11
aI11
aI77
aI249
asS'd'
I40
sa(dp80
S'a'
I38
sS's'
I8
sS'm'
(lp81
I1
aI3
aI1
aI11
aI27
aI43
aI71
aI9
asS'd'
I41
sa(dp82
S'a'
I47
sS's'
I8
sS'm'
(lp83
I1
aI1
aI7
aI15
aI21
aI11
aI81
aI45
asS'd'
I42
sa(dp84
S'a'
I49
sS's'
I8
sS'm'
(lp85
I1
aI3
aI7
aI3
aI25
aI31
aI65
aI79
asS'd'
I43
sa(dp86
S'a'
I50
sS's'
I8
sS'm'
(lp87
I1
aI3
aI1
aI1
aI19
aI11
aI3
aI205
asS'd'
I44
sa(dp88
S'a'
I52
sS's'
I8
sS'm'
(lp89
I1
aI1
aI5
aI9
aI19
aI21
aI29
aI157
asS'd'
I45
sa(dp90
S'a'
I56
sS's'
I8
sS'm'
(lp91
I1
aI3
aI7
aI11
aI1
aI33
aI89
aI185
asS'd'
I46
sa(dp92
S'a'
I67
sS's'
I8
sS'm'
(lp93
I1
aI3
aI3
aI3
aI15
aI9
aI79
aI71
asS'd'
I47
sa(dp94
S'a'
I70
sS's'
I8
sS'm'
(lp95
I1
aI3
aI7
aI11
aI15
aI39
aI119
aI27
asS'd'
I48
sa(dp96
S'a'
I84
sS's'
I8
sS'm'
(lp97
I1
aI1
aI3
aI1
aI11
aI31
aI97
aI225
asS'd'
I49
sa(dp98
S'a'
I97
sS's'
I8
sS'm'
(lp99
I1
aI1
aI1
aI3
aI23
aI43
aI57
aI177
asS'd'
I50
sa(dp100
S'a'
I103
sS's'
I8
sS'm'
(lp101
I1
aI3
aI7
aI7
aI17
aI17
aI37
aI71
asS'd'
I51
sa(dp102
S'a'
I115
sS's'
I8
sS'm'
(lp103
I1
aI3
aI1
aI5
aI27
aI63
aI123
aI213
asS'd'
I52
sa(dp104
S'a'
I122
sS's'
I8
sS'm'
(lp105
I1
aI1
aI3
aI5
aI11
aI43
aI53
aI133
asS'd'
I53
sa(dp106
S'a'
I8
sS's'
I9
sS'm'
(lp107
I1
aI3
aI5
aI5
aI29
aI17
aI47
aI173
aI479
asS'd'
I54
sa(dp108
S'a'
I13
sS's'
I9
sS'm'
(lp109
I1
aI3
aI3
aI11
aI3
aI1
aI109
aI9
aI69
asS'd'
I55
sa(dp110
S'a'
I16
sS's'
I9
sS'm'
(lp111
I1
aI1
aI1
aI5
aI17
aI39
aI23
aI5
aI343
asS'd'
I56
sa(dp112
S'a'
I22
sS's'
I9
sS'm'
(lp113
I1
aI3
aI1
aI5
aI25
aI15
aI31
aI103
aI499
asS'd'
I57
sa(dp114
S'a'
I25
sS's'
I9
sS'm'
(lp115
I1
aI1
aI1
aI11
aI11
aI17
aI63
aI105
aI183
asS'd'
I58
sa(dp116
S'a'
I44
sS's'
I9
sS'm'
(lp117
I1
aI1
aI5
aI11
aI9
aI29
aI97
aI231
aI363
asS'd'
I59
sa(dp118
S'a'
I47
sS's'
I9
sS'm'
(lp119
I1
aI1
aI5
aI15
aI19
aI45
aI41
aI7
aI383
asS'd'
I60
sa(dp120
S'a'
I52
sS's'
I9
sS'm'
(lp121
I1
aI3
aI7
aI7
aI31
aI19
aI83
aI137
aI221
asS'd'
I61
sa(dp122
S'a'
I55
sS's'
I9
sS'm'
(lp123
I1
aI1
aI1
aI3
aI23
aI15
aI111
aI223
aI83
asS'd'
I62
sa(dp124
S'a'
I59
sS's'
I9
sS'm'
(lp125
I1
aI1
aI5
aI13
aI31
aI15
aI55
aI25
aI161
asS'd'
I63
sa(dp126
S'a'
I62
sS's'
I9
sS'm'
(lp127
I1
aI1
aI3
aI13
aI25
aI47
aI39
aI87
aI257
asS'd'
I64
sa(dp128
S'a'
I67
sS's'
I9
sS'm'
(lp129
I1
aI1
aI1
aI11
aI21
aI53
aI125
aI249
aI293
asS'd'
I65
sa(dp130
S'a'
I74
sS's'
I9
sS'm'
(lp131
I1
aI1
aI7
aI11
aI11
aI7
aI57
aI79
aI323
asS'd'
I66
sa(dp132
S'a'
I81
sS's'
I9
sS'm'
(lp133
I1
aI1
aI5
aI5
aI17
aI13
aI81
aI3
aI131
asS'd'
I67
sa(dp134
S'a'
I82
sS's'
I9
sS'm'
(lp135
I1
aI1
aI7
aI13
aI23
aI7
aI65
aI251
aI475
asS'd'
I68
sa(dp136
S'a'
I87
sS's'
I9
sS'm'
(lp137
I1
aI3
aI5
aI1
aI9
aI43
aI3
aI149
aI11
asS'd'
I69
sa(dp138
S'a'
I91
sS's'
I9
sS'm'
(lp139
I1
aI1
aI3
aI13
aI31
aI13
aI13
aI255
aI487
asS'd'
I70
sa(dp140
S'a'
I94
sS's'
I9
sS'm'
(lp141
I1
aI3
aI3
aI1
aI5
aI63
aI89
aI91
aI127
asS'd'
I71
sa(dp142
S'a'
I103
sS's'
I9
sS'm'
(lp143
I1
aI1
aI3
aI3
aI1
aI19
aI123
aI127
aI237
asS'd'
I72
sa(dp144
S'a'
I104
sS's'
I9
sS'm'
(lp145
I1
aI1
aI5
aI7
aI23
aI31
aI37
aI243
aI289
asS'd'
I73
sa(dp146
S'a'
I109
sS's'
I9
sS'm'
(lp147
I1
aI1
aI5
aI11
aI17
aI53
aI117
aI183
aI491
asS'd'
I74
sa(dp148
S'a'
I122
sS's'
I9
sS'm'
(lp149
I1
aI1
aI1
aI5
aI1
aI13
aI13
aI209
aI345
asS'd'
I75
sa(dp150
S'a'
I124
sS's'
I9
sS'm'
(lp151
I1
aI1
aI3
aI15
aI1
aI57
aI115
aI7
aI33
asS'd'
I76
sa(dp152
S'a'
I137
sS's'
I9
sS'm'
(lp153
I1
aI3
aI1
aI11
aI7
aI43
aI81
aI207
aI175
asS'd'
I77
sa(dp154
S'a'
I138
sS's'
I9
sS'm'
(lp155
I1
aI3
aI1
aI1
aI15
aI27
aI63
aI255
aI49
asS'd'
I78
sa(dp156
S'a'
I143
sS's'
I9
sS'm'
(lp157
I1
aI3
aI5
aI3
aI27
aI61
aI105
aI171
aI305
asS'd'
I79
sa(dp158
S'a'
I145
sS's'
I9
sS'm'
(lp159
I1
aI1
aI5
aI3
aI1
aI3
aI57
aI249
aI149
asS'd'
I80
sa(dp160
S'a'
I152
sS's'
I9
sS'm'
(lp161
I1
aI1
aI3
aI5
aI5
aI57
aI15
aI13
aI159
asS'd'
I81
sa(dp162
S'a'
I157
sS's'
I9
sS'm'
(lp163
I1
aI1
aI1
aI11
aI7
aI11
aI105
aI141
aI225
asS'd'
I82
sa(dp164
S'a'
I167
sS's'
I9
sS'm'
(lp165
I1
aI3
aI3
aI5
aI27
aI59
aI121
aI101
aI271
asS'd'
I83
sa(dp166
S'a'
I173
sS's'
I9
sS'm'
(lp167
I1
aI3
aI5
aI9
aI11
aI49
aI51
aI59
aI115
asS'd'
I84
sa(dp168
S'a'
I176
sS's'
I9
sS'm'
(lp169
I1
aI1
aI7
aI1
aI23
aI45
aI125
aI71
aI419
asS'd'
I85
sa(dp170
S'a'
I181
sS's'
I9
sS'm'
(lp171
I1
aI1
aI3
aI5
aI23
aI5
aI105
aI109
aI75
asS'd'
I86
sa(dp172
S'a'
I182
sS's'
I9
sS'm'
(lp173
I1
aI1
aI7
aI15
aI7
aI11
aI67
aI121
aI453
asS'd'
I87
sa(dp174
S'a'
I185
sS's'
I9
sS'm'
(lp175
I1
aI3
aI7
aI3
aI9
aI13
aI31
aI27
aI449
asS'd'
I88
sa(dp176
S'a'
I191
sS's'
I9
sS'm'
(lp177
I1
aI3
aI1
aI15
aI19
aI39
aI39
aI89
aI15
asS'd'
I89
sa(dp178
S'a'
I194
sS's'
I9
sS'm'
(lp179
I1
aI1
aI1
aI1
aI1
aI33
aI73
aI145
aI379
asS'd'
I90
sa(dp180
S'a'
I199
sS's'
I9
sS'm'
(lp181
I1
aI3
aI1
aI15
aI15
aI43
aI29
aI13
aI483
asS'd'
I91
sa(dp182
S'a'
I218
sS's'
I9
sS'm'
(lp183
I1
aI1
aI7
aI3
aI19
aI27
aI85
aI131
aI431
asS'd'
I92
sa(dp184
S'a'
I220
sS's'
I9
sS'm'
(lp185
I1
aI3
aI3
aI3
aI5
aI35
aI23
aI195
aI349
asS'd'
I93
sa(dp186
S'a'
I227
sS's'
I9
sS'm'
(lp187
I1
aI3
aI3
aI7
aI9
aI27
aI39
aI59
aI297
asS'd'
I94
sa(dp188
S'a'
I229
sS's'
I9
sS'm'
(lp189
I1
aI1
aI3
aI9
aI11
aI17
aI13
aI241
aI157
asS'd'
I95
sa(dp190
S'a'
I230
sS's'
I9
sS'm'
(lp191
I1
aI3
aI7
aI15
aI25
aI57
aI33
aI189
aI213
asS'd'
I96
sa(dp192
S'a'
I234
sS's'
I9
sS'm'
(lp193
I1
aI1
aI7
aI1
aI9
aI55
aI73
aI83
aI217
asS'd'
I97
sa(dp194
S'a'
I236
sS's'
I9
sS'm'
(lp195
I1
aI3
aI3
aI13
aI19
aI27
aI23
aI113
aI249
asS'd'
I98
sa(dp196
S'a'
I241
sS's'
I9
sS'm'
(lp197
I1
aI3
aI5
aI3
aI23
aI43
aI3
aI253
aI479
asS'd'
I99
sa(dp198
S'a'
I244
sS's'
I9
sS'm'
(lp199
I1
aI1
aI5
aI5
aI11
aI5
aI45
aI117
aI217
asS'd'
I100
sa(dp200
S'a'
I253
sS's'
I9
sS'm'
(lp201
I1
aI3
aI3
aI7
aI29
aI37
aI33
aI123
aI147
asS'd'
I101
sa(dp202
S'a'
I4
sS's'
I10
sS'm'
(lp203
I1
aI3
aI1
aI15
aI5
aI5
aI37
aI227
aI223
aI459
asS'd'
I102
sa(dp204
S'a'
I13
sS's'
I10
sS'm'
(lp205
I1
aI1
aI7
aI5
aI5
aI39
aI63
aI255
aI135
aI487
asS'd'
I103
sa(dp206
S'a'
I19
sS's'
I10
sS'm'
(lp207
I1
aI3
aI1
aI7
aI9
aI7
aI87
aI249
aI217
aI599
asS'd'
I104
sa(dp208
S'a'
I22
sS's'
I10
sS'm'
(lp209
I1
aI1
aI3
aI13
aI9
aI47
aI7
aI225
aI363
aI247
asS'd'
I105
sa(dp210
S'a'
I50
sS's'
I10
sS'm'
(lp211
I1
aI3
aI7
aI13
aI19
aI13
aI9
aI67
aI9
aI737
asS'd'
I106
sa(dp212
S'a'
I55
sS's'
I10
sS'm'
(lp213
I1
aI3
aI5
aI5
aI19
aI59
aI7
aI41
aI319
aI677
asS'd'
I107
sa(dp214
S'a'
I64
sS's'
I10
sS'm'
(lp215
I1
aI1
aI5
aI3
aI31
aI63
aI15
aI43
aI207
aI789
asS'd'
I108
sa(dp216
S'a'
I69
sS's'
I10
sS'm'
(lp217
I1
aI1
aI7
aI9
aI13
aI39
aI3
aI47
aI497
aI169
asS'd'
I109
sa(dp218
S'a'
I98
sS's'
I10
sS'm'
(lp219
I1
aI3
aI1
aI7
aI21
aI17
aI97
aI19
aI415
aI905
asS'd'
I110
sa(dp220
S'a'
I107
sS's'
I10
sS'm'
(lp221
I1
aI3
aI7
aI1
aI3
aI31
aI71
aI111
aI165
aI127
asS'd'
I111
sa(dp222
S'a'
I115
sS's'
I10
sS'm'
(lp223
I1
aI1
aI5
aI11
aI1
aI61
aI83
aI119
aI203
aI847
asS'd'
I112
sa(dp224
S'a'
I121
sS's'
I10
sS'm'
(lp225
I1
aI3
aI3
aI13
aI9
aI61
aI19
aI97
aI47
aI35
asS'd'
I113
sa(dp226
S'a'
I127
sS's'
I10
sS'm'
(lp227
I1
aI1
aI7
aI7
aI15
aI29
aI63
aI95
aI417
aI469
asS'd'
I114
sa(dp228
S'a'
I134
sS's'
I10
sS'm'
(lp229
I1
aI3
aI1
aI9
aI25
aI9
aI71
aI57
aI213
aI385
asS'd'
I115
sa(dp230
S'a'
I140
sS's'
I10
sS'm'
(lp231
I1
aI3
aI5
aI13
aI31
aI47
aI101
aI57
aI39
aI341
asS'd'
I116
sa(dp232
S'a'
I145
sS's'
I10
sS'm'
(lp233
I1
aI1
aI3
aI3
aI31
aI57
aI125
aI173
aI365
aI551
asS'd'
I117
sa(dp234
S'a'
I152
sS's'
I10
sS'm'
(lp235
I1
aI3
aI7
aI1
aI13
aI57
aI67
aI157
aI451
aI707
asS'd'
I118
sa(dp236
S'a'
I158
sS's'
I10
sS'm'
(lp237
I1
aI1
aI1
aI7
aI21
aI13
aI105
aI89
aI429
aI965
asS'd'
I119
sa(dp238
S'a'
I161
sS's'
I10
sS'm'
(lp239
I1
aI1
aI5
aI9
aI17
aI51
aI45
aI119
aI157
aI141
asS'd'
I120
sa(dp240
S'a'
I171
sS's'
I10
sS'm'
(lp241
I1
aI3
aI7
aI7
aI13
aI45
aI91
aI9
aI129
aI741
asS'd'
I121
sa(dp242
S'a'
I181
sS's'
I10
sS'm'
(lp243
I1
aI3
aI7
aI1
aI23
aI57
aI67
aI141
aI151
aI571
asS'd'
I122
sa(dp244
S'a'
I194
sS's'
I10
sS'm'
(lp245
I1
aI1
aI3
aI11
aI17
aI47
aI93
aI107
aI375
aI157
asS'd'
I123
sa(dp246
S'a'
I199
sS's'
I10
sS'm'
(lp247
I1
aI3
aI3
aI5
aI11
aI21
aI43
aI51
aI169
aI915
asS'd'
I124
sa(dp248
S'a'
I203
sS's'
I10
sS'm'
(lp249
I1
aI1
aI5
aI3
aI15
aI55
aI101
aI67
aI455
aI625
asS'd'
I125
sa(dp250
S'a'
I208
sS's'
I10
sS'm'
(lp251
I1
aI3
aI5
aI9
aI1
aI23
aI29
aI47
aI345
aI595
asS'd'
I126
sa(dp252
S'a'
I227
sS's'
I10
sS'm'
(lp253
I1
aI3
aI7
aI7
aI5
aI49
aI29
aI155
aI323
aI589
asS'd'
I127
sa(dp254
S'a'
I242
sS's'
I10
sS'm'
(lp255
I1
aI3
aI3
aI7
aI5
aI41
aI127
aI61
aI261
aI717
asS'd'
I128
sa(dp256
S'a'
I251
sS's'
I10
sS'm'
(lp257
I1
aI3
aI7
aI7
aI17
aI23
aI117
aI67
aI129
aI1009
asS'd'
I129
sa(dp258
S'a'
I253
sS's'
I10
sS'm'
(lp259
I1
aI1
aI3
aI13
aI11
aI39
aI21
aI207
aI123
aI305
asS'd'
I130
sa(dp260
S'a'
I265
sS's'
I10
sS'm'
(lp261
I1
aI1
aI3
aI9
aI29
aI3
aI95
aI47
aI231
aI73
asS'd'
I131
sa(dp262
S'a'
I266
sS's'
I10
sS'm'
(lp263
I1
aI3
aI1
aI9
aI1
aI29
aI117
aI21
aI441
aI259
asS'd'
I132
sa(dp264
S'a'
I274
sS's'
I10
sS'm'
(lp265
I1
aI3
aI1
aI13
aI21
aI39
aI125
aI211
aI439
aI723
asS'd'
I133
sa(dp266
S'a'
I283
sS's'
I10
sS'm'
(lp267
I1
aI1
aI7
aI3
aI17
aI63
aI115
aI89
aI49
aI773
asS'd'
I134
sa(dp268
S'a'
I289
sS's'
I10
sS'm'
(lp269
I1
aI3
aI7
aI13
aI11
aI33
aI101
aI107
aI63
aI73
asS'd'
I135
sa(dp270
S'a'
I295
sS's'
I10
sS'm'
(lp271
I1
aI1
aI5
aI5
aI13
aI57
aI63
aI135
aI437
aI177
asS'd'
I136
sa(dp272
S'a'
I301
sS's'
I10
sS'm'
(lp273
I1
aI1
aI3
aI7
aI27
aI63
aI93
aI47
aI417
aI483
asS'd'
I137
sa(dp274
S'a'
I316
sS's'
I10
sS'm'
(lp275
I1
aI1
aI3
aI1
aI23
aI29
aI1
aI191
aI49
aI23
asS'd'
I138
sa(dp276
S'a'
I319
sS's'
I10
sS'm'
(lp277
I1
aI1
aI3
aI15
aI25
aI55
aI9
aI101
aI219
aI607
asS'd'
I139
sa(dp278
S'a'
I324
sS's'
I10
sS'm'
(lp279
I1
aI3
aI1
aI7
aI7
aI19
aI51
aI251
aI393
aI307
asS'd'
I140
sa(dp280
S'a'
I346
sS's'
I10
sS'm'
(lp281
I1
aI3
aI3
aI3
aI25
aI55
aI17
aI75
aI337
aI3
asS'd'
I141
sa(dp282
S'a'
I352
sS's'
I10
sS'm'
(lp283
I1
aI1
aI1
aI13
aI25
aI17
aI65
aI45
aI479
aI413
asS'd'
I142
sa(dp284
S'a'
I361
sS's'
I10
sS'm'
(lp285
I1
aI1
aI7
aI7
aI27
aI49
aI99
aI161
aI213
aI727
asS'd'
I143
sa(dp286
S'a'
I367
sS's'
I10
sS'm'
(lp287
I1
aI3
aI5
aI1
aI23
aI5
aI43
aI41
aI251
aI857
asS'd'
I144
sa(dp288
S'a'
I382
sS's'
I10
sS'm'
(lp289
I1
aI3
aI3
aI7
aI11
aI61
aI39
aI87
aI383
aI835
asS'd'
I145
sa(dp290
S'a'
I395
sS's'
I10
sS'm'
(lp291
I1
aI1
aI3
aI15
aI13
aI7
aI29
aI7
aI505
aI923
asS'd'
I146
sa(dp292
S'a'
I398
sS's'
I10
sS'm'
(lp293
I1
aI3
aI7
aI1
aI5
aI31
aI47
aI157
aI445
aI501
asS'd'
I147
sa(dp294
S'a'
I400
sS's'
I10
sS'm'
(lp295
I1
aI1
aI3
aI7
aI1
aI43
aI9
aI147
aI115
aI605
asS'd'
I148
sa(dp296
S'a'
I412
sS's'
I10
sS'm'
(lp297
I1
aI3
aI3
aI13
aI5
aI1
aI119
aI211
aI455
aI1001
asS'd'
I149
sa(dp298
S'a'
I419
sS's'
I10
sS'm'
(lp299
I1
aI1
aI3
aI5
aI13
aI19
aI3
aI243
aI75
aI843
asS'd'
I150
sa(dp300
S'a'
I422
sS's'
I10
sS'm'
(lp301
I1
aI3
aI7
aI7
aI1
aI19
aI91
aI249
aI357
aI589
asS'd'
I151
sa(dp302
S'a'
I426
sS's'
I10
sS'm'
(lp303
I1
aI1
aI1
aI9
aI1
aI25
aI109
aI197
aI279
aI411
asS'd'
I152
sa(dp304
S'a'
I428
sS's'
I10
sS'm'
(lp305
I1
aI3
aI1
aI15
aI23
aI57
aI59
aI135
aI191
aI75
asS'd'
I153
sa(dp306
S'a'
I433
sS's'
I10
sS'm'
(lp307
I1
aI1
aI5
aI15
aI29
aI21
aI39
aI253
aI383
aI349
asS'd'
I154
sa(dp308
S'a'
I446
sS's'
I10
sS'm'
(lp309
I1
aI3
aI3
aI5
aI19
aI45
aI61
aI151
aI199
aI981
asS'd'
I155
sa(dp310
S'a'
I454
sS's'
I10
sS'm'
(lp311
I1
aI3
aI5
aI13
aI9
aI61
aI107
aI141
aI141
aI1
asS'd'
I156
sa(dp312
S'a'
I457
sS's'
I10
sS'm'
(lp313
I1
aI3
aI1
aI11
aI27
aI25
aI85
aI105
aI309
aI979
asS'd'
I157
sa(dp314
S'a'
I472
sS's'
I10
sS'm'
(lp315
I1
aI3
aI3
aI11
aI19
aI7
aI115
aI223
aI349
aI43
asS'd'
I158
sa(dp316
S'a'
I493
sS's'
I10
sS'm'
(lp317
I1
aI1
aI7
aI9
aI21
aI39
aI123
aI21
aI275
aI927
asS'd'
I159
sa(dp318
S'a'
I505
sS's'
I10
sS'm'
(lp319
I1
aI1
aI7
aI13
aI15
aI41
aI47
aI243
aI303
aI437
asS'd'
I160
sa(dp320
S'a'
I508
sS's'
I10
sS'm'
(lp321
I1
aI1
aI1
aI7
aI7
aI3
aI15
aI99
aI409
aI719
asS'd'
I161
sa(dp322
S'a'
I2
sS's'
I11
sS'm'
(lp323
I1
aI3
aI3
aI15
aI27
aI49
aI113
aI123
aI113
aI67
aI469
asS'd'
I162
sa(dp324
S'a'
I11
sS's'
I11
sS'm'
(lp325
I1
aI3
aI7
aI11
aI3
aI23
aI87
aI169
aI119
aI483
aI199
asS'd'
I163
sa(dp326
S'a'
I21
sS's'
I11
sS'm'
(lp327
I1
aI1
aI5
aI15
aI7
aI17
aI109
aI229
aI179
aI213
aI741
asS'd'
I164
sa(dp328
S'a'
I22
sS's'
I11
sS'm'
(lp329
I1
aI1
aI5
aI13
aI11
aI17
aI25
aI135
aI403
aI557
aI1433
asS'd'
I165
sa(dp330
S'a'
I35
sS's'
I11
sS'm'
(lp331
I1
aI3
aI1
aI1
aI1
aI61
aI67
aI215
aI189
aI945
aI1243
asS'd'
I166
sa(dp332
S'a'
I49
sS's'
I11
sS'm'
(lp333
I1
aI1
aI7
aI13
aI17
aI33
aI9
aI221
aI429
aI217
aI1679
asS'd'
I167
sa(dp334
S'a'
I50
sS's'
I11
sS'm'
(lp335
I1
aI1
aI3
aI11
aI27
aI3
aI15
aI93
aI93
aI865
aI1049
asS'd'
I168
sa(dp336
S'a'
I56
sS's'
I11
sS'm'
(lp337
I1
aI3
aI7
aI7
aI25
aI41
aI121
aI35
aI373
aI379
aI1547
asS'd'
I169
sa(dp338
S'a'
I61
sS's'
I11
sS'm'
(lp339
I1
aI3
aI3
aI9
aI11
aI35
aI45
aI205
aI241
aI9
aI59
asS'd'
I170
sa(dp340
S'a'
I70
sS's'
I11
sS'm'
(lp341
I1
aI3
aI1
aI7
aI3
aI51
aI7
aI177
aI53
aI975
aI89
asS'd'
I171
sa(dp342
S'a'
I74
sS's'
I11
sS'm'
(lp343
I1
aI1
aI3
aI5
aI27
aI1
aI113
aI231
aI299
aI759
aI861
asS'd'
I172
sa(dp344
S'a'
I79
sS's'
I11
sS'm'
(lp345
I1
aI3
aI3
aI15
aI25
aI29
aI5
aI255
aI139
aI891
aI2031
asS'd'
I173
sa(dp346
S'a'
I84
sS's'
I11
sS'm'
(lp347
I1
aI3
aI1
aI1
aI13
aI9
aI109
aI193
aI419
aI95
aI17
asS'd'
I174
sa(dp348
S'a'
I88
sS's'
I11
sS'm'
(lp349
I1
aI1
aI7
aI9
aI3
aI7
aI29
aI41
aI135
aI839
aI867
asS'd'
I175
sa(dp350
S'a'
I103
sS's'
I11
sS'm'
(lp351
I1
aI1
aI7
aI9
aI25
aI49
aI123
aI217
aI113
aI909
aI215
asS'd'
I176
sa(dp352
S'a'
I104
sS's'
I11
sS'm'
(lp353
I1
aI1
aI7
aI3
aI23
aI15
aI43
aI133
aI217
aI327
aI901
asS'd'
I177
sa(dp354
S'a'
I112
sS's'
I11
sS'm'
(lp355
I1
aI1
aI3
aI3
aI13
aI53
aI63
aI123
aI477
aI711
aI1387
asS'd'
I178
sa(dp356
S'a'
I115
sS's'
I11
sS'm'
(lp357
I1
aI1
aI3
aI15
aI7
aI29
aI75
aI119
aI181
aI957
aI247
asS'd'
I179
sa(dp358
S'a'
I117
sS's'
I11
sS'm'
(lp359
I1
aI1
aI1
aI11
aI27
aI25
aI109
aI151
aI267
aI99
aI1461
asS'd'
I180
sa(dp360
S'a'
I122
sS's'
I11
sS'm'
(lp361
I1
aI3
aI7
aI15
aI5
aI5
aI53
aI145
aI11
aI725
aI1501
asS'd'
I181
sa(dp362
S'a'
I134
sS's'
I11
sS'm'
(lp363
I1
aI3
aI7
aI1
aI9
aI43
aI71
aI229
aI157
aI607
aI1835
asS'd'
I182
sa(dp364
S'a'
I137
sS's'
I11
sS'm'
(lp365
I1
aI3
aI3
aI13
aI25
aI1
aI5
aI27
aI471
aI349
aI127
asS'd'
I183
sa(dp366
S'a'
I146
sS's'
I11
sS'm'
(lp367
I1
aI1
aI1
aI1
aI23
aI37
aI9
aI221
aI269
aI897
aI1685
asS'd'
I184
sa(dp368
S'a'
I148
sS's'
I11
sS'm'
(lp369
I1
aI1
aI3
aI3
aI31
aI29
aI51
aI19
aI311
aI553
aI1969
asS'd'
I185
sa(dp370
S'a'
I157
sS's'
I11
sS'm'
(lp371
I1
aI3
aI7
aI5
aI5
aI55
aI17
aI39
aI475
aI671
aI1529
asS'd'
I186
sa(dp372
S'a'
I158
sS's'
I11
sS'm'
(lp373
I1
aI1
aI7
aI1
aI1
aI35
aI47
aI27
aI437
aI395
aI1635
asS'd'
I187
sa(dp374
S'a'
I162
sS's'
I11
sS'm'
(lp375
I1
aI1
aI7
aI3
aI13
aI23
aI43
aI135
aI327
aI139
aI389
asS'd'
I188
sa(dp376
S'a'
I164
sS's'
I11
sS'm'
(lp377
I1
aI3
aI7
aI3
aI9
aI25
aI91
aI25
aI429
aI219
aI513
asS'd'
I189
sa(dp378
S'a'
I168
sS's'
I11
sS'm'
(lp379
I1
aI1
aI3
aI5
aI13
aI29
aI119
aI201
aI277
aI157
aI2043
asS'd'
I190
sa(dp380
S'a'
I173
sS's'
I11
sS'm'
(lp381
I1
aI3
aI5
aI3
aI29
aI57
aI13
aI17
aI167
aI739
aI1031
asS'd'
I191
sa(dp382
S'a'
I185
sS's'
I11
sS'm'
(lp383
I1
aI3
aI3
aI5
aI29
aI21
aI95
aI27
aI255
aI679
aI1531
asS'd'
I192
sa(dp384
S'a'
I186
sS's'
I11
sS'm'
(lp385
I1
aI3
aI7
aI15
aI9
aI5
aI21
aI71
aI61
aI961
aI1201
asS'd'
I193
sa(dp386
S'a'
I191
sS's'
I11
sS'm'
(lp387
I1
aI3
aI5
aI13
aI15
aI57
aI33
aI93
aI459
aI867
aI223
asS'd'
I194
sa(dp388
S'a'
I193
sS's'
I11
sS'm'
(lp389
I1
aI1
aI1
aI15
aI17
aI43
aI127
aI191
aI67
aI177
aI1073
asS'd'
I195
sa(dp390
S'a'
I199
sS's'
I11
sS'm'
(lp391
I1
aI1
aI1
aI15
aI23
aI7
aI21
aI199
aI75
aI293
aI1611
asS'd'
I196
sa(dp392
S'a'
I213
sS's'
I11
sS'm'
(lp393
I1
aI3
aI7
aI13
aI15
aI39
aI21
aI149
aI65
aI741
aI319
asS'd'
I197
sa(dp394
S'a'
I214
sS's'
I11
sS'm'
(lp395
I1
aI3
aI7
aI11
aI23
aI13
aI101
aI89
aI277
aI519
aI711
asS'd'
I198
sa(dp396
S'a'
I220
sS's'
I11
sS'm'
(lp397
I1
aI3
aI7
aI15
aI19
aI27
aI85
aI203
aI441
aI97
aI1895
asS'd'
I199
sa(dp398
S'a'
I227
sS's'
I11
sS'm'
(lp399
I1
aI3
aI1
aI3
aI29
aI25
aI21
aI155
aI11
aI191
aI197
asS'd'
I200
sa(dp400
S'a'
I236
sS's'
I11
sS'm'
(lp401
I1
aI1
aI7
aI5
aI27
aI11
aI81
aI101
aI457
aI675
aI1687
asS'd'
I201
sa(dp402
S'a'
I242
sS's'
I11
sS'm'
(lp403
I1
aI3
aI1
aI5
aI25
aI5
aI65
aI193
aI41
aI567
aI781
asS'd'
I202
sa(dp404
S'a'
I251
sS's'
I11
sS'm'
(lp405
I1
aI3
aI1
aI5
aI11
aI15
aI113
aI77
aI411
aI695
aI1111
asS'd'
I203
sa(dp406
S'a'
I256
sS's'
I11
sS'm'
(lp407
I1
aI1
aI3
aI9
aI11
aI53
aI119
aI171
aI55
aI297
aI509
asS'd'
I204
sa(dp408
S'a'
I259
sS's'
I11
sS'm'
(lp409
I1
aI1
aI1
aI1
aI11
aI39
aI113
aI139
aI165
aI347
aI595
asS'd'
I205
sa(dp410
S'a'
I265
sS's'
I11
sS'm'
(lp411
I1
aI3
aI7
aI11
aI9
aI17
aI101
aI13
aI81
aI325
aI1733
asS'd'
I206
sa(dp412
S'a'
I266
sS's'
I11
sS'm'
(lp413
I1
aI3
aI1
aI1
aI21
aI43
aI115
aI9
aI113
aI907
aI645
asS'd'
I207
sa(dp414
S'a'
I276
sS's'
I11
sS'm'
(lp415
I1
aI1
aI7
aI3
aI9
aI25
aI117
aI197
aI159
aI471
aI475
asS'd'
I208
sa(dp416
S'a'
I292
sS's'
I11
sS'm'
(lp417
I1
aI3
aI1
aI9
aI11
aI21
aI57
aI207
aI485
aI613
aI1661
asS'd'
I209
sa(dp418
S'a'
I304
sS's'
I11
sS'm'
(lp419
I1
aI1
aI7
aI7
aI27
aI55
aI49
aI223
aI89
aI85
aI1523
asS'd'
I210
sa(dp420
S'a'
I310
sS's'
I11
sS'm'
(lp421
I1
aI1
aI5
aI3
aI19
aI41
aI45
aI51
aI447
aI299
aI1355
asS'd'
I211
sa(dp422
S'a'
I316
sS's'
I11
sS'm'
(lp423
I1
aI3
aI1
aI13
aI1
aI33
aI117
aI143
aI313
aI187
aI1073
asS'd'
I212
sa(dp424
S'a'
I319
sS's'
I11
sS'm'
(lp425
I1
aI1
aI7
aI7
aI5
aI11
aI65
aI97
aI377
aI377
aI1501
asS'd'
I213
sa(dp426
S'a'
I322
sS's'
I11
sS'm'
(lp427
I1
aI3
aI1
aI1
aI21
aI35
aI95
aI65
aI99
aI23
aI1239
asS'd'
I214
sa(dp428
S'a'
I328
sS's'
I11
sS'm'
(lp429
I1
aI1
aI5
aI9
aI3
aI37
aI95
aI167
aI115
aI425
aI867
asS'd'
I215
sa(dp430
S'a'
I334
sS's'
I11
sS'm'
(lp431
I1
aI3
aI3
aI13
aI1
aI37
aI27
aI189
aI81
aI679
aI773
asS'd'
I216
sa(dp432
S'a'
I339
sS's'
I11
sS'm'
(lp433
I1
aI1
aI3
aI11
aI1
aI61
aI99
aI233
aI429
aI969
aI49
asS'd'
I217
sa(dp434
S'a'
I341
sS's'
I11
sS'm'
(lp435
I1
aI1
aI1
aI7
aI25
aI63
aI99
aI165
aI245
aI793
aI1143
asS'd'
I218
sa(dp436
S'a'
I345
sS's'
I11
sS'm'
(lp437
I1
aI1
aI5
aI11
aI11
aI43
aI55
aI65
aI71
aI283
aI273
asS'd'
I219
sa(dp438
S'a'
I346
sS's'
I11
sS'm'
(lp439
I1
aI1
aI5
aI5
aI9
aI3
aI101
aI251
aI355
aI379
aI1611
asS'd'
I220
sa(dp440
S'a'
I362
sS's'
I11
sS'm'
(lp441
I1
aI1
aI1
aI15
aI21
aI63
aI85
aI99
aI49
aI749
aI1335
asS'd'
I221
sa(dp442
S'a'
I367
sS's'
I11
sS'm'
(lp443
I1
aI1
aI5
aI13
aI27
aI9
aI121
aI43
aI255
aI715
aI289
asS'd'
I222
sa(dp444
S'a'
I372
sS's'
I11
sS'm'
(lp445
I1
aI3
aI1
aI5
aI27
aI19
aI17
aI223
aI77
aI571
aI1415
asS'd'
I223
sa(dp446
S'a'
I375
sS's'
I11
sS'm'
(lp447
I1
aI1
aI5
aI3
aI13
aI59
aI125
aI251
aI195
aI551
aI1737
asS'd'
I224
sa(dp448
S'a'
I376
sS's'
I11
sS'm'
(lp449
I1
aI3
aI3
aI15
aI13
aI27
aI49
aI105
aI389
aI971
aI755
asS'd'
I225
sa(dp450
S'a'
I381
sS's'
I11
sS'm'
(lp451
I1
aI3
aI5
aI15
aI23
aI43
aI35
aI107
aI447
aI763
aI253
asS'd'
I226
sa(dp452
S'a'
I385
sS's'
I11
sS'm'
(lp453
I1
aI3
aI5
aI11
aI21
aI3
aI17
aI39
aI497
aI407
aI611
asS'd'
I227
sa(dp454
S'a'
I388
sS's'
I11
sS'm'
(lp455
I1
aI1
aI7
aI13
aI15
aI31
aI113
aI17
aI23
aI507
aI1995
asS'd'
I228
sa(dp456
S'a'
I392
sS's'
I11
sS'm'
(lp457
I1
aI1
aI7
aI15
aI3
aI15
aI31
aI153
aI423
aI79
aI503
asS'd'
I229
sa(dp458
S'a'
I409
sS's'
I11
sS'm'
(lp459
I1
aI1
aI7
aI9
aI19
aI25
aI23
aI171
aI505
aI923
aI1989
asS'd'
I230
sa(dp460
S'a'
I415
sS's'
I11
sS'm'
(lp461
I1
aI1
aI5
aI9
aI21
aI27
aI121
aI223
aI133
aI87
aI697
asS'd'
I231
sa(dp462
S'a'
I416
sS's'
I11
sS'm'
(lp463
I1
aI1
aI5
aI5
aI9
aI19
aI107
aI99
aI319
aI765
aI1461
asS'd'
I232
sa(dp464
S'a'
I421
sS's'
I11
sS'm'
(lp465
I1
aI1
aI3
aI3
aI19
aI25
aI3
aI101
aI171
aI729
aI187
asS'd'
I233
sa(dp466
S'a'
I428
sS's'
I11
sS'm'
(lp467
I1
aI1
aI3
aI1
aI13
aI23
aI85
aI93
aI291
aI209
aI37
asS'd'
I234
sa(dp468
S'a'
I431
sS's'
I11
sS'm'
(lp469
I1
aI1
aI1
aI15
aI25
aI25
aI77
aI253
aI333
aI947
aI1073
asS'd'
I235
sa(dp470
S'a'
I434
sS's'
I11
sS'm'
(lp471
I1
aI1
aI3
aI9
aI17
aI29
aI55
aI47
aI255
aI305
aI2037
asS'd'
I236
sa(dp472
S'a'
I439
sS's'
I11
sS'm'
(lp473
I1
aI3
aI3
aI9
aI29
aI63
aI9
aI103
aI489
aI939
aI1523
asS'd'
I237
sa(dp474
S'a'
I446
sS's'
I11
sS'm'
(lp475
I1
aI3
aI7
aI15
aI7
aI31
aI89
aI175
aI369
aI339
aI595
asS'd'
I238
sa(dp476
S'a'
I451
sS's'
I11
sS'm'
(lp477
I1
aI3
aI7
aI13
aI25
aI5
aI71
aI207
aI251
aI367
aI665
asS'd'
I239
sa(dp478
S'a'
I453
sS's'
I11
sS'm'
(lp479
I1
aI3
aI3
aI3
aI21
aI25
aI75
aI35
aI31
aI321
aI1603
asS'd'
I240
sa(dp480
S'a'
I457
sS's'
I11
sS'm'
(lp481
I1
aI1
aI1
aI9
aI11
aI1
aI65
aI5
aI11
aI329
aI535
asS'd'
I241
sa(dp482
S'a'
I458
sS's'
I11
sS'm'
(lp483
I1
aI1
aI5
aI3
aI19
aI13
aI17
aI43
aI379
aI485
aI383
asS'd'
I242
sa(dp484
S'a'
I471
sS's'
I11
sS'm'
(lp485
I1
aI3
aI5
aI13
aI13
aI9
aI85
aI147
aI489
aI787
aI1133
asS'd'
I243
sa(dp486
S'a'
I475
sS's'
I11
sS'm'
(lp487
I1
aI3
aI1
aI1
aI5
aI51
aI37
aI129
aI195
aI297
aI1783
asS'd'
I244
sa(dp488
S'a'
I478
sS's'
I11
sS'm'
(lp489
I1
aI1
aI3
aI15
aI19
aI57
aI59
aI181
aI455
aI697
aI2033
asS'd'
I245
sa(dp490
S'a'
I484
sS's'
I11
sS'm'
(lp491
I1
aI3
aI7
aI1
aI27
aI9
aI65
aI145
aI325
aI189
aI201
asS'd'
I246
sa(dp492
S'a'
I493
sS's'
I11
sS'm'
(lp493
I1
aI3
aI1
aI15
aI31
aI23
aI19
aI5
aI485
aI581
aI539
asS'd'
I247
sa(dp494
S'a'
I494
sS's'
I11
sS'm'
(lp495
I1
aI1
aI7
aI13
aI11
aI15
aI65
aI83
aI185
aI847
aI831
asS'd'
I248
sa(dp496
S'a'
I499
sS's'
I11
sS'm'
(lp497
I1
aI3
aI5
aI7
aI7
aI55
aI73
aI15
aI303
aI511
aI1905
asS'd'
I249
sa(dp498
S'a'
I502
sS's'
I11
sS'm'
(lp499
I1
aI3
aI5
aI9
aI7
aI21
aI45
aI15
aI397
aI385
aI597
asS'd'
I250
sa(dp500
S'a'
I517
sS's'
I11
sS'm'
(lp501
I1
aI3
aI7
aI3
aI23
aI13
aI73
aI221
aI511
aI883
aI1265
asS'd'
I251
sa(dp502
S'a'
I518
sS's'
I11
sS'm'
(lp503
I1
aI1
aI3
aI11
aI1
aI51
aI73
aI185
aI33
aI975
aI1441
asS'd'
I252
sa(dp504
S'a'
I524
sS's'
I11
sS'm'
(lp505
I1
aI3
aI3
aI9
aI19
aI59
aI21
aI39
aI339
aI37
aI143
asS'd'
I253
sa(dp506
S'a'
I527
sS's'
I11
sS'm'
(lp507
I1
aI1
aI7
aI1
aI31
aI33
aI19
aI167
aI117
aI635
aI639
asS'd'
I254
sa(dp508
S'a'
I555
sS's'
I11
sS'm'
(lp509
I1
aI1
aI1
aI3
aI5
aI13
aI59
aI83
aI355
aI349
aI1967
asS'd'
I255
sa(dp510
S'a'
I560
sS's'
I11
sS'm'
(lp511
I1
aI1
aI1
aI5
aI19
aI3
aI53
aI133
aI97
aI863
aI983
asS'd'
I256
sa(dp512
S'a'
I565
sS's'
I11
sS'm'
(lp513
I1
aI3
aI1
aI13
aI9
aI41
aI91
aI105
aI173
aI97
aI625
asS'd'
I257
sa(dp514
S'a'
I569
sS's'
I11
sS'm'
(lp515
I1
aI1
aI5
aI3
aI7
aI49
aI115
aI133
aI71
aI231
aI1063
asS'd'
I258
sa(dp516
S'a'
I578
sS's'
I11
sS'm'
(lp517
I1
aI1
aI7
aI5
aI17
aI43
aI47
aI45
aI497
aI547
aI757
asS'd'
I259
sa(dp518
S'a'
I580
sS's'
I11
sS'm'
(lp519
I1
aI3
aI5
aI15
aI21
aI61
aI123
aI191
aI249
aI31
aI631
asS'd'
I260
sa(dp520
S'a'
I587
sS's'
I11
sS'm'
(lp521
I1
aI3
aI7
aI9
aI17
aI7
aI11
aI185
aI127
aI169
aI1951
asS'd'
I261
sa(dp522
S'a'
I589
sS's'
I11
sS'm'
(lp523
I1
aI1
aI5
aI13
aI11
aI11
aI9
aI49
aI29
aI125
aI791
asS'd'
I262
sa(dp524
S'a'
I590
sS's'
I11
sS'm'
(lp525
I1
aI1
aI1
aI15
aI31
aI41
aI13
aI167
aI273
aI429
aI57
asS'd'
I263
sa(dp526
S'a'
I601
sS's'
I11
sS'm'
(lp527
I1
aI3
aI5
aI3
aI27
aI7
aI35
aI209
aI65
aI265
aI1393
asS'd'
I264
sa(dp528
S'a'
I607
sS's'
I11
sS'm'
(lp529
I1
aI3
aI1
aI13
aI31
aI19
aI53
aI143
aI135
aI9
aI1021
asS'd'
I265
sa(dp530
S'a'
I611
sS's'
I11
sS'm'
(lp531
I1
aI1
aI7
aI13
aI31
aI5
aI115
aI153
aI143
aI957
aI623
asS'd'
I266
sa(dp532
S'a'
I614
sS's'
I11
sS'm'
(lp533
I1
aI1
aI5
aI11
aI25
aI19
aI29
aI31
aI297
aI943
aI443
asS'd'
I267
sa(dp534
S'a'
I617
sS's'
I11
sS'm'
(lp535
I1
aI3
aI3
aI5
aI21
aI11
aI127
aI81
aI479
aI25
aI699
asS'd'
I268
sa(dp536
S'a'
I618
sS's'
I11
sS'm'
(lp537
I1
aI1
aI3
aI11
aI25
aI31
aI97
aI19
aI195
aI781
aI705
asS'd'
I269
sa(dp538
S'a'
I625
sS's'
I11
sS'm'
(lp539
I1
aI1
aI5
aI5
aI31
aI11
aI75
aI207
aI197
aI885
aI2037
asS'd'
I270
sa(dp540
S'a'
I628
sS's'
I11
sS'm'
(lp541
I1
aI1
aI1
aI11
aI9
aI23
aI29
aI231
aI307
aI17
aI1497
asS'd'
I271
sa(dp542
S'a'
I635
sS's'
I11
sS'm'
(lp543
I1
aI1
aI5
aI11
aI11
aI43
aI111
aI233
aI307
aI523
aI1259
asS'd'
I272
sa(dp544
S'a'
I641
sS's'
I11
sS'm'
(lp545
I1
aI1
aI7
aI5
aI1
aI21
aI107
aI229
aI343
aI933
aI217
asS'd'
I273
sa(dp546
S'a'
I647
sS's'
I11
sS'm'
(lp547
I1
aI1
aI1
aI11
aI3
aI21
aI125
aI131
aI405
aI599
aI1469
asS'd'
I274
sa(dp548
S'a'
I654
sS's'
I11
sS'm'
(lp549
I1
aI3
aI5
aI5
aI9
aI39
aI33
aI81
aI389
aI151
aI811
asS'd'
I275
sa(dp550
S'a'
I659
sS's'
I11
sS'm'
(lp551
I1
aI1
aI7
aI7
aI7
aI1
aI59
aI223
aI265
aI529
aI2021
asS'd'
I276
sa(dp552
S'a'
I662
sS's'
I11
sS'm'
(lp553
I1
aI3
aI1
aI3
aI9
aI23
aI85
aI181
aI47
aI265
aI49
asS'd'
I277
sa(dp554
S'a'
I672
sS's'
I11
sS'm'
(lp555
I1
aI3
aI5
aI11
aI19
aI23
aI9
aI7
aI157
aI299
aI1983
asS'd'
I278
sa(dp556
S'a'
I675
sS's'
I11
sS'm'
(lp557
I1
aI3
aI1
aI5
aI15
aI5
aI21
aI105
aI29
aI339
aI1041
asS'd'
I279
sa(dp558
S'a'
I682
sS's'
I11
sS'm'
(lp559
I1
aI1
aI1
aI1
aI5
aI33
aI65
aI85
aI111
aI705
aI479
asS'd'
I280
sa(dp560
S'a'
I684
sS's'
I11
sS'm'
(lp561
I1
aI1
aI1
aI7
aI9
aI35
aI77
aI87
aI151
aI321
aI101
asS'd'
I281
sa(dp562
S'a'
I689
sS's'
I11
sS'm'
(lp563
I1
aI1
aI5
aI7
aI17
aI1
aI51
aI197
aI175
aI811
aI1229
asS'd'
I282
sa(dp564
S'a'
I695
sS's'
I11
sS'm'
(lp565
I1
aI3
aI3
aI15
aI23
aI37
aI85
aI185
aI239
aI543
aI731
asS'd'
I283
sa(dp566
S'a'
I696
sS's'
I11
sS'm'
(lp567
I1
aI3
aI1
aI7
aI7
aI55
aI111
aI109
aI289
aI439
aI243
asS'd'
I284
sa(dp568
S'a'
I713
sS's'
I11
sS'm'
(lp569
I1
aI1
aI7
aI11
aI17
aI53
aI35
aI217
aI259
aI853
aI1667
asS'd'
I285
sa(dp570
S'a'
I719
sS's'
I11
sS'm'
(lp571
I1
aI3
aI1
aI9
aI1
aI63
aI87
aI17
aI73
aI565
aI1091
asS'd'
I286
sa(dp572
S'a'
I724
sS's'
I11
sS'm'
(lp573
I1
aI1
aI3
aI3
aI11
aI41
aI1
aI57
aI295
aI263
aI1029
asS'd'
I287
sa(dp574
S'a'
I733
sS's'
I11
sS'm'
(lp575
I1
aI1
aI5
aI1
aI27
aI45
aI109
aI161
aI411
aI421
aI1395
asS'd'
I288
sa(dp576
S'a'
I734
sS's'
I11
sS'm'
(lp577
I1
aI3
aI5
aI11
aI25
aI35
aI47
aI191
aI339
aI417
aI1727
asS'd'
I289
sa(dp578
S'a'
I740
sS's'
I11
sS'm'
(lp579
I1
aI1
aI5
aI15
aI21
aI1
aI93
aI251
aI351
aI217
aI1767
asS'd'
I290
sa(dp580
S'a'
I747
sS's'
I11
sS'm'
(lp581
I1
aI3
aI3
aI11
aI3
aI7
aI75
aI155
aI313
aI211
aI491
asS'd'
I291
sa(dp582
S'a'
I749
sS's'
I11
sS'm'
(lp583
I1
aI3
aI3
aI5
aI11
aI9
aI101
aI161
aI453
aI913
aI1067
asS'd'
I292
sa(dp584
S'a'
I752
sS's'
I11
sS'm'
(lp585
I1
aI1
aI3
aI1
aI15
aI45
aI127
aI141
aI163
aI727
aI1597
asS'd'
I293
sa(dp586
S'a'
I755
sS's'
I11
sS'm'
(lp587
I1
aI3
aI3
aI7
aI1
aI33
aI63
aI73
aI73
aI341
aI1691
asS'd'
I294
sa(dp588
S'a'
I762
sS's'
I11
sS'm'
(lp589
I1
aI3
aI5
aI13
aI15
aI39
aI53
aI235
aI77
aI99
aI949
asS'd'
I295
sa(dp590
S'a'
I770
sS's'
I11
sS'm'
(lp591
I1
aI1
aI5
aI13
aI31
aI17
aI97
aI13
aI215
aI301
aI1927
asS'd'
I296
sa(dp592
S'a'
I782
sS's'
I11
sS'm'
(lp593
I1
aI1
aI7
aI1
aI1
aI37
aI91
aI93
aI441
aI251
aI1131
asS'd'
I297
sa(dp594
S'a'
I784
sS's'
I11
sS'm'
(lp595
I1
aI3
aI7
aI9
aI25
aI5
aI105
aI69
aI81
aI943
aI1459
asS'd'
I298
sa(dp596
S'a'
I787
sS's'
I11
sS'm'
(lp597
I1
aI3
aI7
aI11
aI31
aI43
aI13
aI209
aI27
aI1017
aI501
asS'd'
I299
sa(dp598
S'a'
I789
sS's'
I11
sS'm'
(lp599
I1
aI1
aI7
aI15
aI1
aI33
aI31
aI233
aI161
aI507
aI387
asS'd'
I300
sa(dp600
S'a'
I793
sS's'
I11
sS'm'
(lp601
I1
aI3
aI3
aI5
aI5
aI53
aI33
aI177
aI503
aI627
aI1927
asS'd'
I301
sa(dp602
S'a'
I796
sS's'
I11
sS'm'
(lp603
I1
aI1
aI7
aI11
aI7
aI61
aI119
aI31
aI457
aI229
aI1875
asS'd'
I302
sa(dp604
S'a'
I803
sS's'
I11
sS'm'
(lp605
I1
aI1
aI5
aI15
aI19
aI5
aI53
aI201
aI157
aI885
aI1057
asS'd'
I303
sa(dp606
S'a'
I805
sS's'
I11
sS'm'
(lp607
I1
aI3
aI7
aI9
aI1
aI35
aI51
aI113
aI249
aI425
aI1009
asS'd'
I304
sa(dp608
S'a'
I810
sS's'
I11
sS'm'
(lp609
I1
aI3
aI5
aI7
aI21
aI53
aI37
aI155
aI119
aI345
aI631
asS'd'
I305
sa(dp610
S'a'
I815
sS's'
I11
sS'm'
(lp611
I1
aI3
aI5
aI7
aI15
aI31
aI109
aI69
aI503
aI595
aI1879
asS'd'
I306
sa(dp612
S'a'
I824
sS's'
I11
sS'm'
(lp613
I1
aI3
aI3
aI1
aI25
aI35
aI65
aI131
aI403
aI705
aI503
asS'd'
I307
sa(dp614
S'a'
I829
sS's'
I11
sS'm'
(lp615
I1
aI3
aI7
aI7
aI19
aI33
aI11
aI153
aI45
aI633
aI499
asS'd'
I308
sa(dp616
S'a'
I830
sS's'
I11
sS'm'
(lp617
I1
aI3
aI3
aI5
aI11
aI3
aI29
aI93
aI487
aI33
aI703
asS'd'
I309
sa(dp618
S'a'
I832
sS's'
I11
sS'm'
(lp619
I1
aI1
aI3
aI15
aI21
aI53
aI107
aI179
aI387
aI927
aI1757
asS'd'
I310
sa(dp620
S'a'
I841
sS's'
I11
sS'm'
(lp621
I1
aI1
aI3
aI7
aI21
aI45
aI51
aI147
aI175
aI317
aI361
asS'd'
I311
sa(dp622
S'a'
I847
sS's'
I11
sS'm'
(lp623
I1
aI1
aI1
aI7
aI7
aI13
aI15
aI243
aI269
aI795
aI1965
asS'd'
I312
sa(dp624
S'a'
I849
sS's'
I11
sS'm'
(lp625
I1
aI1
aI3
aI5
aI19
aI33
aI57
aI115
aI443
aI537
aI627
asS'd'
I313
sa(dp626
S'a'
I861
sS's'
I11
sS'm'
(lp627
I1
aI3
aI3
aI9
aI3
aI39
aI25
aI61
aI185
aI717
aI1049
asS'd'
I314
sa(dp628
S'a'
I871
sS's'
I11
sS'm'
(lp629
I1
aI3
aI7
aI3
aI7
aI37
aI107
aI153
aI7
aI269
aI1581
asS'd'
I315
sa(dp630
S'a'
I878
sS's'
I11
sS'm'
(lp631
I1
aI1
aI7
aI3
aI7
aI41
aI91
aI41
aI145
aI489
aI1245
asS'd'
I316
sa(dp632
S'a'
I889
sS's'
I11
sS'm'
(lp633
I1
aI1
aI5
aI9
aI7
aI7
aI105
aI81
aI403
aI407
aI283
asS'd'
I317
sa(dp634
S'a'
I892
sS's'
I11
sS'm'
(lp635
I1
aI1
aI7
aI9
aI27
aI55
aI29
aI77
aI193
aI963
aI949
asS'd'
I318
sa(dp636
S'a'
I901
sS's'
I11
sS'm'
(lp637
I1
aI1
aI5
aI3
aI25
aI51
aI107
aI63
aI403
aI917
aI815
asS'd'
I319
sa(dp638
S'a'
I908
sS's'
I11
sS'm'
(lp639
I1
aI1
aI7
aI3
aI7
aI61
aI19
aI51
aI457
aI599
aI535
asS'd'
I320
sa(dp640
S'a'
I920
sS's'
I11
sS'm'
(lp641
I1
aI3
aI7
aI1
aI23
aI51
aI105
aI153
aI239
aI215
aI1847
asS'd'
I321
sa(dp642
S'a'
I923
sS's'
I11
sS'm'
(lp643
I1
aI1
aI3
aI5
aI27
aI23
aI79
aI49
aI495
aI45
aI1935
asS'd'
I322
sa(dp644
S'a'
I942
sS's'
I11
sS'm'
(lp645
I1
aI1
aI1
aI11
aI11
aI47
aI55
aI133
aI495
aI999
aI1461
asS'd'
I323
sa(dp646
S'a'
I949
sS's'
I11
sS'm'
(lp647
I1
aI1
aI3
aI15
aI27
aI51
aI93
aI17
aI355
aI763
aI1675
asS'd'
I324
sa(dp648
S'a'
I950
sS's'
I11
sS'm'
(lp649
I1
aI3
aI1
aI3
aI1
aI3
aI79
aI119
aI499
aI17
aI995
asS'd'
I325
sa(dp650
S'a'
I954
sS's'
I11
sS'm'
(lp651
I1
aI1
aI1
aI1
aI15
aI43
aI45
aI17
aI167
aI973
aI799
asS'd'
I326
sa(dp652
S'a'
I961
sS's'
I11
sS'm'
(lp653
I1
aI1
aI1
aI3
aI27
aI49
aI89
aI29
aI483
aI913
aI2023
asS'd'
I327
sa(dp654
S'a'
I968
sS's'
I11
sS'm'
(lp655
I1
aI1
aI3
aI3
aI5
aI11
aI75
aI7
aI41
aI851
aI611
asS'd'
I328
sa(dp656
S'a'
I971
sS's'
I11
sS'm'
(lp657
I1
aI3
aI1
aI3
aI7
aI57
aI39
aI123
aI257
aI283
aI507
asS'd'
I329
sa(dp658
S'a'
I973
sS's'
I11
sS'm'
(lp659
I1
aI3
aI3
aI11
aI27
aI23
aI113
aI229
aI187
aI299
aI133
asS'd'
I330
sa(dp660
S'a'
I979
sS's'
I11
sS'm'
(lp661
I1
aI1
aI3
aI13
aI9
aI63
aI101
aI77
aI451
aI169
aI337
asS'd'
I331
sa(dp662
S'a'
I982
sS's'
I11
sS'm'
(lp663
I1
aI3
aI7
aI3
aI3
aI59
aI45
aI195
aI229
aI415
aI409
asS'd'
I332
sa(dp664
S'a'
I986
sS's'
I11
sS'm'
(lp665
I1
aI3
aI5
aI3
aI11
aI19
aI71
aI93
aI43
aI857
aI369
asS'd'
I333
sa(dp666
S'a'
I998
sS's'
I11
sS'm'
(lp667
I1
aI3
aI7
aI9
aI19
aI33
aI115
aI19
aI241
aI703
aI247
asS'd'
I334
sa(dp668
S'a'
I1001
sS's'
I11
sS'm'
(lp669
I1
aI3
aI5
aI11
aI5
aI35
aI21
aI155
aI463
aI1005
aI1073
asS'd'
I335
sa(dp670
S'a'
I1010
sS's'
I11
sS'm'
(lp671
I1
aI3
aI7
aI3
aI25
aI15
aI109
aI83
aI93
aI69
aI1189
asS'd'
I336
sa(dp672
S'a'
I1012
sS's'
I11
sS'm'
(lp673
I1
aI3
aI5
aI7
aI5
aI21
aI93
aI133
aI135
aI167
aI903
asS'd'
I337
sa(dp674
S'a'
I41
sS's'
I12
sS'm'
(lp675
I1
aI1
aI7
aI7
aI3
aI59
aI121
aI161
aI285
aI815
aI1769
aI3705
asS'd'
I338
sa(dp676
S'a'
I52
sS's'
I12
sS'm'
(lp677
I1
aI3
aI1
aI1
aI3
aI47
aI103
aI171
aI381
aI609
aI185
aI373
asS'd'
I339
sa(dp678
S'a'
I61
sS's'
I12
sS'm'
(lp679
I1
aI3
aI3
aI15
aI23
aI33
aI107
aI131
aI441
aI445
aI689
aI2059
asS'd'
I340
sa(dp680
S'a'
I62
sS's'
I12
sS'm'
(lp681
I1
aI3
aI3
aI11
aI7
aI53
aI101
aI167
aI435
aI803
aI1255
aI3781
asS'd'
I341
sa(dp682
S'a'
I76
sS's'
I12
sS'm'
(lp683
I1
aI1
aI5
aI11
aI15
aI59
aI41
aI19
aI135
aI835
aI1263
aI505
asS'd'
I342
sa(dp684
S'a'
I104
sS's'
I12
sS'm'
(lp685
I1
aI1
aI7
aI11
aI21
aI49
aI23
aI219
aI127
aI961
aI1065
aI385
asS'd'
I343
sa(dp686
S'a'
I117
sS's'
I12
sS'm'
(lp687
I1
aI3
aI5
aI15
aI7
aI47
aI117
aI217
aI45
aI731
aI1639
aI733
asS'd'
I344
sa(dp688
S'a'
I131
sS's'
I12
sS'm'
(lp689
I1
aI1
aI7
aI11
aI27
aI57
aI91
aI87
aI81
aI35
aI1269
aI1007
asS'd'
I345
sa(dp690
S'a'
I143
sS's'
I12
sS'm'
(lp691
I1
aI1
aI3
aI11
aI15
aI37
aI53
aI219
aI193
aI937
aI1899
aI3733
asS'd'
I346
sa(dp692
S'a'
I145
sS's'
I12
sS'm'
(lp693
I1
aI3
aI5
aI3
aI13
aI11
aI27
aI19
aI199
aI393
aI965
aI2195
asS'd'
I347
sa(dp694
S'a'
I157
sS's'
I12
sS'm'
(lp695
I1
aI3
aI1
aI3
aI5
aI1
aI37
aI173
aI413
aI1023
aI553
aI409
asS'd'
I348
sa(dp696
S'a'
I167
sS's'
I12
sS'm'
(lp697
I1
aI3
aI1
aI7
aI15
aI29
aI123
aI95
aI255
aI373
aI1799
aI3841
asS'd'
I349
sa(dp698
S'a'
I171
sS's'
I12
sS'm'
(lp699
I1
aI3
aI5
aI13
aI21
aI57
aI51
aI17
aI511
aI195
aI1157
aI1831
asS'd'
I350
sa(dp700
S'a'
I176
sS's'
I12
sS'm'
(lp701
I1
aI1
aI1
aI15
aI29
aI19
aI7
aI73
aI295
aI519
aI587
aI3523
asS'd'
I351
sa(dp702
S'a'
I181
sS's'
I12
sS'm'
(lp703
I1
aI1
aI5
aI13
aI13
aI35
aI115
aI191
aI123
aI535
aI717
aI1661
asS'd'
I352
sa(dp704
S'a'
I194
sS's'
I12
sS'm'
(lp705
I1
aI3
aI3
aI5
aI23
aI21
aI47
aI251
aI379
aI921
aI1119
aI297
asS'd'
I353
sa(dp706
S'a'
I217
sS's'
I12
sS'm'
(lp707
I1
aI3
aI3
aI9
aI29
aI53
aI121
aI201
aI135
aI193
aI523
aI2943
asS'd'
I354
sa(dp708
S'a'
I236
sS's'
I12
sS'm'
(lp709
I1
aI1
aI1
aI7
aI29
aI45
aI125
aI9
aI99
aI867
aI425
aI601
asS'd'
I355
sa(dp710
S'a'
I239
sS's'
I12
sS'm'
(lp711
I1
aI3
aI1
aI9
aI13
aI15
aI67
aI181
aI109
aI293
aI1305
aI3079
asS'd'
I356
sa(dp712
S'a'
I262
sS's'
I12
sS'm'
(lp713
I1
aI3
aI3
aI9
aI5
aI35
aI15
aI209
aI305
aI87
aI767
aI2795
asS'd'
I357
sa(dp714
S'a'
I283
sS's'
I12
sS'm'
(lp715
I1
aI3
aI3
aI11
aI27
aI57
aI113
aI123
aI179
aI643
aI149
aI523
asS'd'
I358
sa(dp716
S'a'
I286
sS's'
I12
sS'm'
(lp717
I1
aI1
aI3
aI15
aI11
aI17
aI67
aI223
aI63
aI657
aI335
aI3309
asS'd'
I359
sa(dp718
S'a'
I307
sS's'
I12
sS'm'
(lp719
I1
aI1
aI1
aI9
aI25
aI29
aI109
aI159
aI39
aI513
aI571
aI1761
asS'd'
I360
sa(dp720
S'a'
I313
sS's'
I12
sS'm'
(lp721
I1
aI1
aI3
aI1
aI5
aI63
aI75
aI19
aI455
aI601
aI123
aI691
asS'd'
I361
sa(dp722
S'a'
I319
sS's'
I12
sS'm'
(lp723
I1
aI1
aI1
aI3
aI21
aI5
aI45
aI169
aI377
aI513
aI1951
aI2565
asS'd'
I362
sa(dp724
S'a'
I348
sS's'
I12
sS'm'
(lp725
I1
aI1
aI3
aI11
aI3
aI33
aI119
aI69
aI253
aI907
aI805
aI1449
asS'd'
I363
sa(dp726
S'a'
I352
sS's'
I12
sS'm'
(lp727
I1
aI1
aI5
aI13
aI31
aI15
aI17
aI7
aI499
aI61
aI687
aI1867
asS'd'
I364
sa(dp728
S'a'
I357
sS's'
I12
sS'm'
(lp729
I1
aI3
aI7
aI11
aI17
aI33
aI73
aI77
aI299
aI243
aI641
aI2345
asS'd'
I365
sa(dp730
S'a'
I391
sS's'
I12
sS'm'
(lp731
I1
aI1
aI7
aI11
aI9
aI35
aI31
aI235
aI359
aI647
aI379
aI1161
asS'd'
I366
sa(dp732
S'a'
I398
sS's'
I12
sS'm'
(lp733
I1
aI3
aI3
aI15
aI31
aI25
aI5
aI67
aI33
aI45
aI437
aI4067
asS'd'
I367
sa(dp734
S'a'
I400
sS's'
I12
sS'm'
(lp735
I1
aI1
aI3
aI11
aI7
aI17
aI37
aI87
aI333
aI253
aI1517
aI2921
asS'd'
I368
sa(dp736
S'a'
I412
sS's'
I12
sS'm'
(lp737
I1
aI1
aI7
aI15
aI7
aI15
aI107
aI189
aI153
aI769
aI1521
aI3427
asS'd'
I369
sa(dp738
S'a'
I415
sS's'
I12
sS'm'
(lp739
I1
aI3
aI5
aI13
aI5
aI61
aI113
aI37
aI293
aI393
aI113
aI43
asS'd'
I370
sa(dp740
S'a'
I422
sS's'
I12
sS'm'
(lp741
I1
aI1
aI1
aI15
aI29
aI43
aI107
aI31
aI167
aI147
aI301
aI1021
asS'd'
I371
sa(dp742
S'a'
I440
sS's'
I12
sS'm'
(lp743
I1
aI1
aI1
aI13
aI3
aI1
aI35
aI93
aI195
aI181
aI2027
aI1491
asS'd'
I372
sa(dp744
S'a'
I460
sS's'
I12
sS'm'
(lp745
I1
aI3
aI3
aI3
aI13
aI33
aI77
aI199
aI153
aI221
aI1699
aI3671
asS'd'
I373
sa(dp746
S'a'
I465
sS's'
I12
sS'm'
(lp747
I1
aI3
aI5
aI13
aI7
aI49
aI123
aI155
aI495
aI681
aI819
aI809
asS'd'
I374
sa(dp748
S'a'
I468
sS's'
I12
sS'm'
(lp749
I1
aI3
aI5
aI15
aI27
aI61
aI117
aI189
aI183
aI887
aI617
aI4053
asS'd'
I375
sa(dp750
S'a'
I515
sS's'
I12
sS'm'
(lp751
I1
aI1
aI1
aI7
aI31
aI59
aI125
aI235
aI389
aI369
aI447
aI1039
asS'd'
I376
sa(dp752
S'a'
I536
sS's'
I12
sS'm'
(lp753
I1
aI3
aI5
aI1
aI5
aI39
aI115
aI89
aI249
aI377
aI431
aI3747
asS'd'
I377
sa(dp754
S'a'
I539
sS's'
I12
sS'm'
(lp755
I1
aI1
aI1
aI5
aI7
aI47
aI59
aI157
aI77
aI445
aI699
aI3439
asS'd'
I378
sa(dp756
S'a'
I551
sS's'
I12
sS'm'
(lp757
I1
aI1
aI3
aI5
aI11
aI21
aI19
aI75
aI11
aI599
aI1575
aI735
asS'd'
I379
sa(dp758
S'a'
I558
sS's'
I12
sS'm'
(lp759
I1
aI3
aI5
aI3
aI19
aI13
aI41
aI69
aI199
aI143
aI1761
aI3215
asS'd'
I380
sa(dp760
S'a'
I563
sS's'
I12
sS'm'
(lp761
I1
aI3
aI5
aI7
aI19
aI43
aI25
aI41
aI41
aI11
aI1647
aI2783
asS'd'
I381
sa(dp762
S'a'
I570
sS's'
I12
sS'm'
(lp763
I1
aI3
aI1
aI9
aI19
aI45
aI111
aI97
aI405
aI399
aI457
aI3219
asS'd'
I382
sa(dp764
S'a'
I595
sS's'
I12
sS'm'
(lp765
I1
aI1
aI3
aI1
aI23
aI15
aI65
aI121
aI59
aI985
aI829
aI2259
asS'd'
I383
sa(dp766
S'a'
I598
sS's'
I12
sS'm'
(lp767
I1
aI1
aI3
aI7
aI17
aI13
aI107
aI229
aI75
aI551
aI1299
aI2363
asS'd'
I384
sa(dp768
S'a'
I617
sS's'
I12
sS'm'
(lp769
I1
aI1
aI5
aI5
aI21
aI57
aI23
aI199
aI509
aI139
aI2007
aI3875
asS'd'
I385
sa(dp770
S'a'
I647
sS's'
I12
sS'm'
(lp771
I1
aI3
aI1
aI11
aI19
aI53
aI15
aI229
aI215
aI741
aI695
aI823
asS'd'
I386
sa(dp772
S'a'
I654
sS's'
I12
sS'm'
(lp773
I1
aI3
aI7
aI1
aI29
aI3
aI17
aI163
aI417
aI559
aI549
aI319
asS'd'
I387
sa(dp774
S'a'
I678
sS's'
I12
sS'm'
(lp775
I1
aI3
aI1
aI13
aI17
aI9
aI47
aI133
aI365
aI7
aI1937
aI1071
asS'd'
I388
sa(dp776
S'a'
I713
sS's'
I12
sS'm'
(lp777
I1
aI3
aI5
aI7
aI19
aI37
aI55
aI163
aI301
aI249
aI689
aI2327
asS'd'
I389
sa(dp778
S'a'
I738
sS's'
I12
sS'm'
(lp779
I1
aI3
aI5
aI13
aI11
aI23
aI61
aI205
aI257
aI377
aI615
aI1457
asS'd'
I390
sa(dp780
S'a'
I747
sS's'
I12
sS'm'
(lp781
I1
aI3
aI5
aI1
aI23
aI37
aI13
aI75
aI331
aI495
aI579
aI3367
asS'd'
I391
sa(dp782
S'a'
I750
sS's'
I12
sS'm'
(lp783
I1
aI1
aI1
aI9
aI1
aI23
aI49
aI129
aI475
aI543
aI883
aI2531
asS'd'
I392
sa(dp784
S'a'
I757
sS's'
I12
sS'm'
(lp785
I1
aI3
aI1
aI5
aI23
aI59
aI51
aI35
aI343
aI695
aI219
aI369
asS'd'
I393
sa(dp786
S'a'
I772
sS's'
I12
sS'm'
(lp787
I1
aI3
aI3
aI1
aI27
aI17
aI63
aI97
aI71
aI507
aI1929
aI613
asS'd'
I394
sa(dp788
S'a'
I803
sS's'
I12
sS'm'
(lp789
I1
aI1
aI5
aI1
aI21
aI31
aI11
aI109
aI247
aI409
aI1817
aI2173
asS'd'
I395
sa(dp790
S'a'
I810
sS's'
I12
sS'm'
(lp791
I1
aI1
aI3
aI15
aI23
aI9
aI7
aI209
aI301
aI23
aI147
aI1691
asS'd'
I396
sa(dp792
S'a'
I812
sS's'
I12
sS'm'
(lp793
I1
aI1
aI7
aI5
aI5
aI19
aI37
aI229
aI249
aI277
aI1115
aI2309
asS'd'
I397
sa(dp794
S'a'
I850
sS's'
I12
sS'm'
(lp795
I1
aI1
aI1
aI5
aI5
aI63
aI5
aI249
aI285
aI431
aI343
aI2467
asS'd'
I398
sa(dp796
S'a'
I862
sS's'
I12
sS'm'
(lp797
I1
aI1
aI1
aI11
aI7
aI45
aI35
aI75
aI505
aI537
aI29
aI2919
asS'd'
I399
sa(dp798
S'a'
I906
sS's'
I12
sS'm'
(lp799
I1
aI3
aI5
aI15
aI11
aI39
aI15
aI63
aI263
aI9
aI199
aI445
asS'd'
I400
sa(dp800
S'a'
I908
sS's'
I12
sS'm'
(lp801
I1
aI3
aI3
aI3
aI27
aI63
aI53
aI171
aI227
aI63
aI1049
aI827
asS'd'
I401
sa(dp802
S'a'
I929
sS's'
I12
sS'm'
(lp803
I1
aI1
aI3
aI13
aI7
aI11
aI115
aI183
aI179
aI937
aI1785
aI381
asS'd'
I402
sa(dp804
S'a'
I930
sS's'
I12
sS'm'
(lp805
I1
aI3
aI1
aI11
aI13
aI15
aI107
aI81
aI53
aI295
aI1785
aI3757
asS'd'
I403
sa(dp806
S'a'
I954
sS's'
I12
sS'm'
(lp807
I1
aI3
aI3
aI13
aI11
aI5
aI109
aI243
aI3
aI505
aI323
aI1373
asS'd'
I404
sa(dp808
S'a'
I964
sS's'
I12
sS'm'
(lp809
I1
aI3
aI3
aI11
aI21
aI51
aI17
aI177
aI381
aI937
aI1263
aI3889
asS'd'
I405
sa(dp810
S'a'
I982
sS's'
I12
sS'm'
(lp811
I1
aI3
aI5
aI9
aI27
aI25
aI85
aI193
aI143
aI573
aI1189
aI2995
asS'd'
I406
sa(dp812
S'a'
I985
sS's'
I12
sS'm'
(lp813
I1
aI3
aI5
aI11
aI13
aI9
aI81
aI21
aI159
aI953
aI91
aI1751
asS'd'
I407
sa(dp814
S'a'
I991
sS's'
I12
sS'm'
(lp815
I1
aI1
aI3
aI3
aI27
aI61
aI11
aI253
aI391
aI333
aI1105
aI635
asS'd'
I408
sa(dp816
S'a'
I992
sS's'
I12
sS'm'
(lp817
I1
aI3
aI3
aI15
aI9
aI57
aI95
aI81
aI419
aI735
aI251
aI1141
asS'd'
I409
sa(dp818
S'a'
I1067
sS's'
I12
sS'm'
(lp819
I1
aI1
aI5
aI9
aI31
aI39
aI59
aI13
aI319
aI807
aI1241
aI2433
asS'd'
I410
sa(dp820
S'a'
I1070
sS's'
I12
sS'm'
(lp821
I1
aI3
aI3
aI5
aI27
aI13
aI107
aI141
aI423
aI937
aI2027
aI3233
asS'd'
I411
sa(dp822
S'a'
I1096
sS's'
I12
sS'm'
(lp823
I1
aI3
aI3
aI9
aI9
aI25
aI125
aI23
aI443
aI835
aI1245
aI847
asS'd'
I412
sa(dp824
S'a'
I1099
sS's'
I12
sS'm'
(lp825
I1
aI1
aI7
aI15
aI17
aI17
aI83
aI107
aI411
aI285
aI847
aI1571
asS'd'
I413
sa(dp826
S'a'
I1116
sS's'
I12
sS'm'
(lp827
I1
aI1
aI3
aI13
aI29
aI61
aI37
aI81
aI349
aI727
aI1453
aI1957
asS'd'
I414
sa(dp828
S'a'
I1143
sS's'
I12
sS'm'
(lp829
I1
aI3
aI7
aI11
aI31
aI13
aI59
aI77
aI273
aI591
aI1265
aI1533
asS'd'
I415
sa(dp830
S'a'
I1165
sS's'
I12
sS'm'
(lp831
I1
aI1
aI7
aI7
aI13
aI17
aI25
aI25
aI187
aI329
aI347
aI1473
asS'd'
I416
sa(dp832
S'a'
I1178
sS's'
I12
sS'm'
(lp833
I1
aI3
aI7
aI7
aI5
aI51
aI37
aI99
aI221
aI153
aI503
aI2583
asS'd'
I417
sa(dp834
S'a'
I1184
sS's'
I12
sS'm'
(lp835
I1
aI3
aI1
aI13
aI19
aI27
aI11
aI69
aI181
aI479
aI1183
aI3229
asS'd'
I418
sa(dp836
S'a'
I1202
sS's'
I12
sS'm'
(lp837
I1
aI3
aI3
aI13
aI23
aI21
aI103
aI147
aI323
aI909
aI947
aI315
asS'd'
I419
sa(dp838
S'a'
I1213
sS's'
I12
sS'm'
(lp839
I1
aI3
aI1
aI3
aI23
aI1
aI31
aI59
aI93
aI513
aI45
aI2271
asS'd'
I420
sa(dp840
S'a'
I1221
sS's'
I12
sS'm'
(lp841
I1
aI3
aI5
aI1
aI7
aI43
aI109
aI59
aI231
aI41
aI1515
aI2385
asS'd'
I421
sa(dp842
S'a'
I1240
sS's'
I12
sS'm'
(lp843
I1
aI3
aI1
aI5
aI31
aI57
aI49
aI223
aI283
aI1013
aI11
aI701
asS'd'
I422
sa(dp844
S'a'
I1246
sS's'
I12
sS'm'
(lp845
I1
aI1
aI5
aI1
aI19
aI53
aI55
aI31
aI31
aI299
aI495
aI693
asS'd'
I423
sa(dp846
S'a'
I1252
sS's'
I12
sS'm'
(lp847
I1
aI3
aI3
aI9
aI5
aI33
aI77
aI253
aI427
aI791
aI731
aI1019
asS'd'
I424
sa(dp848
S'a'
I1255
sS's'
I12
sS'm'
(lp849
I1
aI3
aI7
aI11
aI1
aI9
aI119
aI203
aI53
aI877
aI1707
aI3499
asS'd'
I425
sa(dp850
S'a'
I1267
sS's'
I12
sS'm'
(lp851
I1
aI1
aI3
aI7
aI13
aI39
aI55
aI159
aI423
aI113
aI1653
aI3455
asS'd'
I426
sa(dp852
S'a'
I1293
sS's'
I12
sS'm'
(lp853
I1
aI1
aI3
aI5
aI21
aI47
aI51
aI59
aI55
aI411
aI931
aI251
asS'd'
I427
sa(dp854
S'a'
I1301
sS's'
I12
sS'm'
(lp855
I1
aI3
aI7
aI3
aI31
aI25
aI81
aI115
aI405
aI239
aI741
aI455
asS'd'
I428
sa(dp856
S'a'
I1305
sS's'
I12
sS'm'
(lp857
I1
aI1
aI5
aI1
aI31
aI3
aI101
aI83
aI479
aI491
aI1779
aI2225
asS'd'
I429
sa(dp858
S'a'
I1332
sS's'
I12
sS'm'
(lp859
I1
aI3
aI3
aI3
aI9
aI37
aI107
aI161
aI203
aI503
aI767
aI3435
asS'd'
I430
sa(dp860
S'a'
I1349
sS's'
I12
sS'm'
(lp861
I1
aI3
aI7
aI9
aI1
aI27
aI61
aI119
aI233
aI39
aI1375
aI4089
asS'd'
I431
sa(dp862
S'a'
I1384
sS's'
I12
sS'm'
(lp863
I1
aI1
aI5
aI9
aI1
aI31
aI45
aI51
aI369
aI587
aI383
aI2813
asS'd'
I432
sa(dp864
S'a'
I1392
sS's'
I12
sS'm'
(lp865
I1
aI3
aI7
aI5
aI31
aI7
aI49
aI119
aI487
aI591
aI1627
aI53
asS'd'
I433
sa(dp866
S'a'
I1402
sS's'
I12
sS'm'
(lp867
I1
aI1
aI7
aI1
aI9
aI47
aI1
aI223
aI369
aI711
aI1603
aI1917
asS'd'
I434
sa(dp868
S'a'
I1413
sS's'
I12
sS'm'
(lp869
I1
aI3
aI5
aI3
aI21
aI37
aI111
aI17
aI483
aI739
aI1193
aI2775
asS'd'
I435
sa(dp870
S'a'
I1417
sS's'
I12
sS'm'
(lp871
I1
aI3
aI3
aI7
aI17
aI11
aI51
aI117
aI455
aI191
aI1493
aI3821
asS'd'
I436
sa(dp872
S'a'
I1423
sS's'
I12
sS'm'
(lp873
I1
aI1
aI5
aI9
aI23
aI39
aI99
aI181
aI343
aI485
aI99
aI1931
asS'd'
I437
sa(dp874
S'a'
I1451
sS's'
I12
sS'm'
(lp875
I1
aI3
aI1
aI7
aI29
aI49
aI31
aI71
aI489
aI527
aI1763
aI2909
asS'd'
I438
sa(dp876
S'a'
I1480
sS's'
I12
sS'm'
(lp877
I1
aI1
aI5
aI11
aI5
aI5
aI73
aI189
aI321
aI57
aI1191
aI3685
asS'd'
I439
sa(dp878
S'a'
I1491
sS's'
I12
sS'm'
(lp879
I1
aI1
aI5
aI15
aI13
aI45
aI125
aI207
aI371
aI415
aI315
aI983
asS'd'
I440
sa(dp880
S'a'
I1503
sS's'
I12
sS'm'
(lp881
I1
aI3
aI3
aI5
aI25
aI59
aI33
aI31
aI239
aI919
aI1859
aI2709
asS'd'
I441
sa(dp882
S'a'
I1504
sS's'
I12
sS'm'
(lp883
I1
aI3
aI5
aI13
aI27
aI61
aI23
aI115
aI61
aI413
aI1275
aI3559
asS'd'
I442
sa(dp884
S'a'
I1513
sS's'
I12
sS'm'
(lp885
I1
aI3
aI7
aI15
aI5
aI59
aI101
aI81
aI47
aI967
aI809
aI3189
asS'd'
I443
sa(dp886
S'a'
I1538
sS's'
I12
sS'm'
(lp887
I1
aI1
aI5
aI11
aI31
aI15
aI39
aI25
aI173
aI505
aI809
aI2677
asS'd'
I444
sa(dp888
S'a'
I1544
sS's'
I12
sS'm'
(lp889
I1
aI1
aI5
aI9
aI19
aI13
aI95
aI89
aI511
aI127
aI1395
aI2935
asS'd'
I445
sa(dp890
S'a'
I1547
sS's'
I12
sS'm'
(lp891
I1
aI1
aI5
aI5
aI31
aI45
aI9
aI57
aI91
aI303
aI1295
aI3215
asS'd'
I446
sa(dp892
S'a'
I1555
sS's'
I12
sS'm'
(lp893
I1
aI3
aI3
aI3
aI19
aI15
aI113
aI187
aI217
aI489
aI1285
aI1803
asS'd'
I447
sa(dp894
S'a'
I1574
sS's'
I12
sS'm'
(lp895
I1
aI1
aI3
aI1
aI13
aI29
aI57
aI139
aI255
aI197
aI537
aI2183
asS'd'
I448
sa(dp896
S'a'
I1603
sS's'
I12
sS'm'
(lp897
I1
aI3
aI1
aI15
aI11
aI7
aI53
aI255
aI467
aI9
aI757
aI3167
asS'd'
I449
sa(dp898
S'a'
I1615
sS's'
I12
sS'm'
(lp899
I1
aI3
aI3
aI15
aI21
aI13
aI9
aI189
aI359
aI323
aI49
aI333
asS'd'
I450
sa(dp900
S'a'
I1618
sS's'
I12
sS'm'
(lp901
I1
aI3
aI7
aI11
aI7
aI37
aI21
aI119
aI401
aI157
aI1659
aI1069
asS'd'
I451
sa(dp902
S'a'
I1629
sS's'
I12
sS'm'
(lp903
I1
aI1
aI5
aI7
aI17
aI33
aI115
aI229
aI149
aI151
aI2027
aI279
asS'd'
I452
sa(dp904
S'a'
I1634
sS's'
I12
sS'm'
(lp905
I1
aI1
aI5
aI15
aI5
aI49
aI77
aI155
aI383
aI385
aI1985
aI945
asS'd'
I453
sa(dp906
S'a'
I1636
sS's'
I12
sS'm'
(lp907
I1
aI3
aI7
aI3
aI7
aI55
aI85
aI41
aI357
aI527
aI1715
aI1619
asS'd'
I454
sa(dp908
S'a'
I1639
sS's'
I12
sS'm'
(lp909
I1
aI1
aI3
aI1
aI21
aI45
aI115
aI21
aI199
aI967
aI1581
aI3807
asS'd'
I455
sa(dp910
S'a'
I1657
sS's'
I12
sS'm'
(lp911
I1
aI1
aI3
aI7
aI21
aI39
aI117
aI191
aI169
aI73
aI413
aI3417
asS'd'
I456
sa(dp912
S'a'
I1667
sS's'
I12
sS'm'
(lp913
I1
aI1
aI1
aI13
aI1
aI31
aI57
aI195
aI231
aI321
aI367
aI1027
asS'd'
I457
sa(dp914
S'a'
I1681
sS's'
I12
sS'm'
(lp915
I1
aI3
aI7
aI3
aI11
aI29
aI47
aI161
aI71
aI419
aI1721
aI437
asS'd'
I458
sa(dp916
S'a'
I1697
sS's'
I12
sS'm'
(lp917
I1
aI1
aI7
aI3
aI11
aI9
aI43
aI65
aI157
aI1
aI1851
aI823
asS'd'
I459
sa(dp918
S'a'
I1704
sS's'
I12
sS'm'
(lp919
I1
aI1
aI1
aI5
aI21
aI15
aI31
aI101
aI293
aI299
aI127
aI1321
asS'd'
I460
sa(dp920
S'a'
I1709
sS's'
I12
sS'm'
(lp921
I1
aI1
aI7
aI1
aI27
aI1
aI11
aI229
aI241
aI705
aI43
aI1475
asS'd'
I461
sa(dp922
S'a'
I1722
sS's'
I12
sS'm'
(lp923
I1
aI3
aI7
aI1
aI5
aI15
aI73
aI183
aI193
aI55
aI1345
aI49
asS'd'
I462
sa(dp924
S'a'
I1730
sS's'
I12
sS'm'
(lp925
I1
aI3
aI3
aI3
aI19
aI3
aI55
aI21
aI169
aI663
aI1675
aI137
asS'd'
I463
sa(dp926
S'a'
I1732
sS's'
I12
sS'm'
(lp927
I1
aI1
aI1
aI13
aI7
aI21
aI69
aI67
aI373
aI965
aI1273
aI2279
asS'd'
I464
sa(dp928
S'a'
I1802
sS's'
I12
sS'm'
(lp929
I1
aI1
aI7
aI7
aI21
aI23
aI17
aI43
aI341
aI845
aI465
aI3355
asS'd'
I465
sa(dp930
S'a'
I1804
sS's'
I12
sS'm'
(lp931
I1
aI3
aI5
aI5
aI25
aI5
aI81
aI101
aI233
aI139
aI359
aI2057
asS'd'
I466
sa(dp932
S'a'
I1815
sS's'
I12
sS'm'
(lp933
I1
aI1
aI3
aI11
aI15
aI39
aI55
aI3
aI471
aI765
aI1143
aI3941
asS'd'
I467
sa(dp934
S'a'
I1826
sS's'
I12
sS'm'
(lp935
I1
aI1
aI7
aI15
aI9
aI57
aI81
aI79
aI215
aI433
aI333
aI3855
asS'd'
I468
sa(dp936
S'a'
I1832
sS's'
I12
sS'm'
(lp937
I1
aI1
aI5
aI5
aI19
aI45
aI83
aI31
aI209
aI363
aI701
aI1303
asS'd'
I469
sa(dp938
S'a'
I1843
sS's'
I12
sS'm'
(lp939
I1
aI3
aI7
aI5
aI1
aI13
aI55
aI163
aI435
aI807
aI287
aI2031
asS'd'
I470
sa(dp940
S'a'
I1849
sS's'
I12
sS'm'
(lp941
I1
aI3
aI3
aI7
aI3
aI3
aI17
aI197
aI39
aI169
aI489
aI1769
asS'd'
I471
sa(dp942
S'a'
I1863
sS's'
I12
sS'm'
(lp943
I1
aI1
aI3
aI5
aI29
aI43
aI87
aI161
aI289
aI339
aI1233
aI2353
asS'd'
I472
sa(dp944
S'a'
I1905
sS's'
I12
sS'm'
(lp945
I1
aI3
aI3
aI9
aI21
aI9
aI77
aI1
aI453
aI167
aI1643
aI2227
asS'd'
I473
sa(dp946
S'a'
I1928
sS's'
I12
sS'm'
(lp947
I1
aI1
aI7
aI1
aI15
aI7
aI67
aI33
aI193
aI241
aI1031
aI2339
asS'd'
I474
sa(dp948
S'a'
I1933
sS's'
I12
sS'm'
(lp949
I1
aI3
aI1
aI11
aI1
aI63
aI45
aI65
aI265
aI661
aI849
aI1979
asS'd'
I475
sa(dp950
S'a'
I1939
sS's'
I12
sS'm'
(lp951
I1
aI3
aI1
aI13
aI19
aI49
aI3
aI11
aI159
aI213
aI659
aI2839
asS'd'
I476
sa(dp952
S'a'
I1976
sS's'
I12
sS'm'
(lp953
I1
aI3
aI5
aI11
aI9
aI29
aI27
aI227
aI253
aI449
aI1403
aI3427
asS'd'
I477
sa(dp954
S'a'
I1996
sS's'
I12
sS'm'
(lp955
I1
aI1
aI3
aI1
aI7
aI3
aI77
aI143
aI277
aI779
aI1499
aI475
asS'd'
I478
sa(dp956
S'a'
I2013
sS's'
I12
sS'm'
(lp957
I1
aI1
aI1
aI5
aI11
aI23
aI87
aI131
aI393
aI849
aI193
aI3189
asS'd'
I479
sa(dp958
S'a'
I2014
sS's'
I12
sS'm'
(lp959
I1
aI3
aI5
aI11
aI3
aI3
aI89
aI9
aI449
aI243
aI1501
aI1739
asS'd'
I480
sa(dp960
S'a'
I2020
sS's'
I12
sS'm'
(lp961
I1
aI3
aI1
aI9
aI29
aI29
aI113
aI15
aI65
aI611
aI135
aI3687
asS'd'
I481
sa(dp962
S'a'
I13
sS's'
I13
sS'm'
(lp963
I1
aI1
aI1
aI9
aI21
aI19
aI39
aI151
aI395
aI501
aI1339
aI959
aI2725
asS'd'
I482
sa(dp964
S'a'
I19
sS's'
I13
sS'm'
(lp965
I1
aI3
aI7
aI1
aI7
aI35
aI45
aI33
aI119
aI225
aI1631
aI1695
aI1459
asS'd'
I483
sa(dp966
S'a'
I26
sS's'
I13
sS'm'
(lp967
I1
aI1
aI1
aI3
aI25
aI55
aI37
aI79
aI167
aI907
aI1075
aI271
aI4059
asS'd'
I484
sa(dp968
S'a'
I41
sS's'
I13
sS'm'
(lp969
I1
aI3
aI5
aI13
aI5
aI13
aI53
aI165
aI437
aI67
aI1705
aI3177
aI8095
asS'd'
I485
sa(dp970
S'a'
I50
sS's'
I13
sS'm'
(lp971
I1
aI3
aI3
aI13
aI27
aI57
aI95
aI55
aI443
aI245
aI1945
aI1725
aI1929
asS'd'
I486
sa(dp972
S'a'
I55
sS's'
I13
sS'm'
(lp973
I1
aI3
aI1
aI9
aI5
aI33
aI109
aI35
aI99
aI827
aI341
aI2401
aI2411
asS'd'
I487
sa(dp974
S'a'
I69
sS's'
I13
sS'm'
(lp975
I1
aI1
aI5
aI9
aI7
aI33
aI43
aI39
aI87
aI799
aI635
aI3481
aI7159
asS'd'
I488
sa(dp976
S'a'
I70
sS's'
I13
sS'm'
(lp977
I1
aI3
aI1
aI1
aI31
aI15
aI45
aI27
aI337
aI113
aI987
aI2065
aI2529
asS'd'
I489
sa(dp978
S'a'
I79
sS's'
I13
sS'm'
(lp979
I1
aI1
aI5
aI9
aI5
aI15
aI105
aI123
aI479
aI289
aI1609
aI2177
aI4629
asS'd'
I490
sa(dp980
S'a'
I82
sS's'
I13
sS'm'
(lp981
I1
aI3
aI5
aI11
aI31
aI47
aI97
aI87
aI385
aI195
aI1041
aI651
aI3271
asS'd'
I491
sa(dp982
S'a'
I87
sS's'
I13
sS'm'
(lp983
I1
aI1
aI3
aI7
aI17
aI3
aI101
aI55
aI87
aI629
aI1687
aI1387
aI2745
asS'd'
I492
sa(dp984
S'a'
I93
sS's'
I13
sS'm'
(lp985
I1
aI3
aI5
aI5
aI7
aI21
aI9
aI237
aI313
aI549
aI1107
aI117
aI6183
asS'd'
I493
sa(dp986
S'a'
I94
sS's'
I13
sS'm'
(lp987
I1
aI1
aI3
aI9
aI9
aI5
aI55
aI201
aI487
aI851
aI1103
aI2993
aI4055
asS'd'
I494
sa(dp988
S'a'
I97
sS's'
I13
sS'm'
(lp989
I1
aI1
aI5
aI9
aI31
aI19
aI59
aI7
aI363
aI381
aI1167
aI2057
aI5715
asS'd'
I495
sa(dp990
S'a'
I100
sS's'
I13
sS'm'
(lp991
I1
aI3
aI3
aI15
aI23
aI63
aI19
aI227
aI387
aI827
aI487
aI1049
aI7471
asS'd'
I496
sa(dp992
S'a'
I112
sS's'
I13
sS'm'
(lp993
I1
aI3
aI1
aI5
aI23
aI25
aI61
aI245
aI363
aI863
aI963
aI3583
aI6475
asS'd'
I497
sa(dp994
S'a'
I121
sS's'
I13
sS'm'
(lp995
I1
aI1
aI5
aI1
aI5
aI27
aI81
aI85
aI275
aI49
aI235
aI3291
aI1195
asS'd'
I498
sa(dp996
S'a'
I134
sS's'
I13
sS'm'
(lp997
I1
aI1
aI5
aI7
aI23
aI53
aI85
aI107
aI511
aI779
aI1265
aI1093
aI7859
asS'd'
I499
sa(dp998
S'a'
I138
sS's'
I13
sS'm'
(lp999
I1
aI3
aI3
aI1
aI9
aI21
aI75
aI219
aI59
aI485
aI1739
aI3845
aI1109
asS'd'
I500
sa(dp1000
S'a'
I148
sS's'
I13
sS'm'
(lp1001
I1
aI3
aI5
aI1
aI13
aI41
aI19
aI143
aI293
aI391
aI2023
aI1791
aI4399
asS'd'
I501
sa(dp1002
S'a'
I151
sS's'
I13
sS'm'
(lp1003
I1
aI3
aI7
aI15
aI21
aI13
aI21
aI195
aI215
aI413
aI523
aI2099
aI2341
asS'd'
I502
sa(dp1004
S'a'
I157
sS's'
I13
sS'm'
(lp1005
I1
aI1
aI1
aI3
aI29
aI51
aI47
aI57
aI135
aI575
aI943
aI1673
aI541
asS'd'
I503
sa(dp1006
S'a'
I161
sS's'
I13
sS'm'
(lp1007
I1
aI3
aI5
aI1
aI9
aI13
aI113
aI175
aI447
aI115
aI657
aI4077
aI5973
asS'd'
I504
sa(dp1008
S'a'
I179
sS's'
I13
sS'm'
(lp1009
I1
aI1
aI1
aI11
aI17
aI41
aI37
aI95
aI297
aI579
aI911
aI2207
aI2387
asS'd'
I505
sa(dp1010
S'a'
I181
sS's'
I13
sS'm'
(lp1011
I1
aI3
aI5
aI3
aI23
aI11
aI23
aI231
aI93
aI667
aI711
aI1563
aI7961
asS'd'
I506
sa(dp1012
S'a'
I188
sS's'
I13
sS'm'
(lp1013
I1
aI1
aI7
aI3
aI17
aI59
aI13
aI181
aI141
aI991
aI1817
aI457
aI1711
asS'd'
I507
sa(dp1014
S'a'
I196
sS's'
I13
sS'm'
(lp1015
I1
aI3
aI3
aI5
aI31
aI59
aI81
aI205
aI245
aI537
aI1049
aI997
aI1815
asS'd'
I508
sa(dp1016
S'a'
I203
sS's'
I13
sS'm'
(lp1017
I1
aI3
aI7
aI5
aI17
aI13
aI9
aI79
aI17
aI185
aI5
aI2211
aI6263
asS'd'
I509
sa(dp1018
S'a'
I206
sS's'
I13
sS'm'
(lp1019
I1
aI3
aI7
aI13
aI7
aI53
aI61
aI145
aI13
aI285
aI1203
aI947
aI2933
asS'd'
I510
sa(dp1020
S'a'
I223
sS's'
I13
sS'm'
(lp1021
I1
aI1
aI7
aI3
aI31
aI19
aI69
aI217
aI47
aI441
aI1893
aI673
aI4451
asS'd'
I511
sa(dp1022
S'a'
I224
sS's'
I13
sS'm'
(lp1023
I1
aI1
aI1
aI1
aI25
aI9
aI23
aI225
aI385
aI629
aI603
aI3747
aI4241
asS'd'
I512
sa(dp1024
S'a'
I227
sS's'
I13
sS'm'
(lp1025
I1
aI3
aI1
aI9
aI5
aI37
aI31
aI237
aI431
aI79
aI1521
aI459
aI2523
asS'd'
I513
sa(dp1026
S'a'
I230
sS's'
I13
sS'm'
(lp1027
I1
aI3
aI7
aI3
aI9
aI43
aI105
aI179
aI5
aI225
aI799
aI1777
aI4893
asS'd'
I514
sa(dp1028
S'a'
I239
sS's'
I13
sS'm'
(lp1029
I1
aI1
aI3
aI1
aI29
aI45
aI29
aI159
aI267
aI247
aI455
aI847
aI3909
asS'd'
I515
sa(dp1030
S'a'
I241
sS's'
I13
sS'm'
(lp1031
I1
aI1
aI3
aI7
aI25
aI21
aI121
aI57
aI467
aI275
aI719
aI1521
aI7319
asS'd'
I516
sa(dp1032
S'a'
I248
sS's'
I13
sS'm'
(lp1033
I1
aI3
aI1
aI3
aI11
aI35
aI119
aI123
aI81
aI979
aI1187
aI3623
aI4293
asS'd'
I517
sa(dp1034
S'a'
I253
sS's'
I13
sS'm'
(lp1035
I1
aI1
aI1
aI7
aI15
aI25
aI121
aI235
aI25
aI487
aI873
aI1787
aI1977
asS'd'
I518
sa(dp1036
S'a'
I268
sS's'
I13
sS'm'
(lp1037
I1
aI1
aI1
aI11
aI3
aI7
aI17
aI135
aI345
aI353
aI383
aI4011
aI2573
asS'd'
I519
sa(dp1038
S'a'
I274
sS's'
I13
sS'm'
(lp1039
I1
aI3
aI7
aI15
aI27
aI13
aI97
aI123
aI65
aI675
aI951
aI1285
aI6559
asS'd'
I520
sa(dp1040
S'a'
I283
sS's'
I13
sS'm'
(lp1041
I1
aI3
aI7
aI3
aI7
aI1
aI71
aI19
aI325
aI765
aI337
aI1197
aI2697
asS'd'
I521
sa(dp1042
S'a'
I286
sS's'
I13
sS'm'
(lp1043
I1
aI3
aI5
aI1
aI31
aI37
aI11
aI71
aI169
aI283
aI83
aI3801
aI7083
asS'd'
I522
sa(dp1044
S'a'
I289
sS's'
I13
sS'm'
(lp1045
I1
aI1
aI3
aI15
aI17
aI29
aI83
aI65
aI275
aI679
aI1749
aI4007
aI7749
asS'd'
I523
sa(dp1046
S'a'
I301
sS's'
I13
sS'm'
(lp1047
I1
aI1
aI3
aI1
aI21
aI11
aI41
aI95
aI237
aI361
aI1819
aI2783
aI2383
asS'd'
I524
sa(dp1048
S'a'
I302
sS's'
I13
sS'm'
(lp1049
I1
aI3
aI7
aI11
aI29
aI57
aI111
aI187
aI465
aI145
aI605
aI1987
aI8109
asS'd'
I525
sa(dp1050
S'a'
I316
sS's'
I13
sS'm'
(lp1051
I1
aI1
aI3
aI3
aI19
aI15
aI55
aI83
aI357
aI1001
aI643
aI1517
aI6529
asS'd'
I526
sa(dp1052
S'a'
I319
sS's'
I13
sS'm'
(lp1053
I1
aI3
aI1
aI5
aI29
aI35
aI73
aI23
aI77
aI619
aI1523
aI1725
aI8145
asS'd'
I527
sa(dp1054
S'a'
I324
sS's'
I13
sS'm'
(lp1055
I1
aI1
aI5
aI5
aI19
aI23
aI7
aI197
aI449
aI337
aI717
aI2921
aI315
asS'd'
I528
sa(dp1056
S'a'
I331
sS's'
I13
sS'm'
(lp1057
I1
aI3
aI5
aI9
aI7
aI63
aI117
aI97
aI97
aI813
aI1925
aI2817
aI1579
asS'd'
I529
sa(dp1058
S'a'
I333
sS's'
I13
sS'm'
(lp1059
I1
aI1
aI1
aI11
aI31
aI7
aI25
aI235
aI231
aI133
aI1007
aI1371
aI1553
asS'd'
I530
sa(dp1060
S'a'
I345
sS's'
I13
sS'm'
(lp1061
I1
aI1
aI7
aI5
aI19
aI7
aI47
aI171
aI267
aI243
aI1331
aI567
aI6033
asS'd'
I531
sa(dp1062
S'a'
I351
sS's'
I13
sS'm'
(lp1063
I1
aI1
aI5
aI1
aI7
aI49
aI55
aI89
aI109
aI735
aI1455
aI3193
aI6239
asS'd'
I532
sa(dp1064
S'a'
I358
sS's'
I13
sS'm'
(lp1065
I1
aI1
aI1
aI7
aI1
aI61
aI9
aI103
aI3
aI929
aI1481
aI2927
aI2957
asS'd'
I533
sa(dp1066
S'a'
I375
sS's'
I13
sS'm'
(lp1067
I1
aI1
aI5
aI13
aI17
aI21
aI75
aI49
aI255
aI1019
aI1161
aI2133
aI1177
asS'd'
I534
sa(dp1068
S'a'
I379
sS's'
I13
sS'm'
(lp1069
I1
aI3
aI1
aI3
aI13
aI15
aI41
aI247
aI211
aI409
aI1163
aI523
aI2635
asS'd'
I535
sa(dp1070
S'a'
I381
sS's'
I13
sS'm'
(lp1071
I1
aI3
aI7
aI7
aI21
aI59
aI91
aI149
aI479
aI391
aI681
aI2311
aI6249
asS'd'
I536
sa(dp1072
S'a'
I386
sS's'
I13
sS'm'
(lp1073
I1
aI1
aI5
aI11
aI27
aI53
aI21
aI211
aI197
aI815
aI719
aI1605
aI255
asS'd'
I537
sa(dp1074
S'a'
I403
sS's'
I13
sS'm'
(lp1075
I1
aI1
aI3
aI3
aI9
aI33
aI59
aI3
aI323
aI1
aI101
aI1135
aI8105
asS'd'
I538
sa(dp1076
S'a'
I405
sS's'
I13
sS'm'
(lp1077
I1
aI3
aI3
aI1
aI29
aI5
aI17
aI141
aI51
aI991
aI841
aI327
aI3859
asS'd'
I539
sa(dp1078
S'a'
I419
sS's'
I13
sS'm'
(lp1079
I1
aI3
aI1
aI5
aI11
aI19
aI23
aI89
aI175
aI173
aI165
aI2881
aI1881
asS'd'
I540
sa(dp1080
S'a'
I426
sS's'
I13
sS'm'
(lp1081
I1
aI1
aI1
aI15
aI13
aI51
aI87
aI39
aI495
aI611
aI1341
aI1531
aI7029
asS'd'
I541
sa(dp1082
S'a'
I428
sS's'
I13
sS'm'
(lp1083
I1
aI1
aI3
aI11
aI13
aI55
aI75
aI185
aI57
aI61
aI1917
aI2051
aI5965
asS'd'
I542
sa(dp1084
S'a'
I439
sS's'
I13
sS'm'
(lp1085
I1
aI1
aI5
aI5
aI7
aI53
aI11
aI217
aI213
aI933
aI921
aI3607
aI5175
asS'd'
I543
sa(dp1086
S'a'
I440
sS's'
I13
sS'm'
(lp1087
I1
aI3
aI3
aI5
aI17
aI53
aI103
aI251
aI369
aI781
aI1319
aI3717
aI4439
asS'd'
I544
sa(dp1088
S'a'
I446
sS's'
I13
sS'm'
(lp1089
I1
aI3
aI5
aI13
aI1
aI39
aI25
aI235
aI321
aI773
aI251
aI3111
aI6397
asS'd'
I545
sa(dp1090
S'a'
I451
sS's'
I13
sS'm'
(lp1091
I1
aI1
aI7
aI3
aI31
aI5
aI25
aI29
aI325
aI385
aI1313
aI127
aI4705
asS'd'
I546
sa(dp1092
S'a'
I454
sS's'
I13
sS'm'
(lp1093
I1
aI1
aI5
aI15
aI15
aI27
aI15
aI85
aI239
aI243
aI1633
aI3473
aI2621
asS'd'
I547
sa(dp1094
S'a'
I458
sS's'
I13
sS'm'
(lp1095
I1
aI3
aI3
aI3
aI9
aI19
aI113
aI13
aI137
aI165
aI25
aI2957
aI7549
asS'd'
I548
sa(dp1096
S'a'
I465
sS's'
I13
sS'm'
(lp1097
I1
aI3
aI1
aI3
aI11
aI21
aI3
aI97
aI417
aI183
aI1205
aI1437
aI247
asS'd'
I549
sa(dp1098
S'a'
I468
sS's'
I13
sS'm'
(lp1099
I1
aI1
aI7
aI3
aI17
aI21
aI125
aI55
aI67
aI387
aI385
aI2323
aI887
asS'd'
I550
sa(dp1100
S'a'
I472
sS's'
I13
sS'm'
(lp1101
I1
aI3
aI5
aI5
aI29
aI11
aI103
aI223
aI233
aI641
aI133
aI415
aI1297
asS'd'
I551
sa(dp1102
S'a'
I475
sS's'
I13
sS'm'
(lp1103
I1
aI3
aI3
aI11
aI1
aI9
aI5
aI189
aI235
aI1007
aI1363
aI3985
aI889
asS'd'
I552
sa(dp1104
S'a'
I477
sS's'
I13
sS'm'
(lp1105
I1
aI3
aI7
aI9
aI23
aI19
aI19
aI183
aI269
aI403
aI1643
aI3559
aI5189
asS'd'
I553
sa(dp1106
S'a'
I496
sS's'
I13
sS'm'
(lp1107
I1
aI3
aI7
aI3
aI29
aI45
aI17
aI69
aI475
aI149
aI1291
aI2689
aI7625
asS'd'
I554
sa(dp1108
S'a'
I502
sS's'
I13
sS'm'
(lp1109
I1
aI3
aI7
aI3
aI27
aI37
aI41
aI73
aI253
aI1001
aI431
aI1111
aI7887
asS'd'
I555
sa(dp1110
S'a'
I508
sS's'
I13
sS'm'
(lp1111
I1
aI1
aI7
aI5
aI3
aI7
aI87
aI143
aI289
aI495
aI631
aI3011
aI6151
asS'd'
I556
sa(dp1112
S'a'
I517
sS's'
I13
sS'm'
(lp1113
I1
aI1
aI1
aI13
aI5
aI45
aI17
aI167
aI23
aI975
aI801
aI1975
aI6833
asS'd'
I557
sa(dp1114
S'a'
I521
sS's'
I13
sS'm'
(lp1115
I1
aI3
aI1
aI11
aI7
aI21
aI39
aI23
aI213
aI429
aI1301
aI2059
aI197
asS'd'
I558
sa(dp1116
S'a'
I527
sS's'
I13
sS'm'
(lp1117
I1
aI3
aI3
aI15
aI3
aI57
aI121
aI133
aI29
aI711
aI1961
aI2497
aI189
asS'd'
I559
sa(dp1118
S'a'
I530
sS's'
I13
sS'm'
(lp1119
I1
aI1
aI3
aI5
aI11
aI55
aI115
aI137
aI233
aI673
aI985
aI2849
aI5911
asS'd'
I560
sa(dp1120
S'a'
I532
sS's'
I13
sS'm'
(lp1121
I1
aI1
aI7
aI15
aI29
aI45
aI1
aI241
aI329
aI323
aI925
aI2821
aI3331
asS'd'
I561
sa(dp1122
S'a'
I542
sS's'
I13
sS'm'
(lp1123
I1
aI1
aI5
aI7
aI13
aI31
aI81
aI105
aI199
aI145
aI195
aI1365
aI5119
asS'd'
I562
sa(dp1124
S'a'
I552
sS's'
I13
sS'm'
(lp1125
I1
aI3
aI7
aI11
aI3
aI55
aI11
aI31
aI117
aI343
aI1265
aI1837
aI2451
asS'd'
I563
sa(dp1126
S'a'
I555
sS's'
I13
sS'm'
(lp1127
I1
aI1
aI3
aI7
aI29
aI57
aI61
aI179
aI429
aI591
aI177
aI1945
aI2159
asS'd'
I564
sa(dp1128
S'a'
I560
sS's'
I13
sS'm'
(lp1129
I1
aI3
aI5
aI11
aI23
aI49
aI101
aI137
aI339
aI323
aI1035
aI1749
aI7737
asS'd'
I565
sa(dp1130
S'a'
I566
sS's'
I13
sS'm'
(lp1131
I1
aI3
aI1
aI13
aI21
aI35
aI55
aI79
aI19
aI269
aI1055
aI2651
aI7083
asS'd'
I566
sa(dp1132
S'a'
I575
sS's'
I13
sS'm'
(lp1133
I1
aI3
aI3
aI11
aI9
aI9
aI95
aI167
aI437
aI361
aI1185
aI4083
aI603
asS'd'
I567
sa(dp1134
S'a'
I577
sS's'
I13
sS'm'
(lp1135
I1
aI1
aI1
aI7
aI31
aI61
aI77
aI65
aI489
aI657
aI691
aI2423
aI4147
asS'd'
I568
sa(dp1136
S'a'
I589
sS's'
I13
sS'm'
(lp1137
I1
aI3
aI5
aI7
aI21
aI37
aI87
aI191
aI311
aI453
aI2013
aI829
aI2619
asS'd'
I569
sa(dp1138
S'a'
I590
sS's'
I13
sS'm'
(lp1139
I1
aI1
aI5
aI9
aI17
aI47
aI35
aI101
aI5
aI813
aI1157
aI1279
aI7365
asS'd'
I570
sa(dp1140
S'a'
I602
sS's'
I13
sS'm'
(lp1141
I1
aI1
aI5
aI3
aI11
aI35
aI113
aI199
aI369
aI721
aI901
aI1471
aI7801
asS'd'
I571
sa(dp1142
S'a'
I607
sS's'
I13
sS'm'
(lp1143
I1
aI3
aI1
aI5
aI9
aI61
aI83
aI157
aI391
aI739
aI1957
aI2123
aI4341
asS'd'
I572
sa(dp1144
S'a'
I608
sS's'
I13
sS'm'
(lp1145
I1
aI3
aI5
aI11
aI19
aI19
aI111
aI225
aI383
aI219
aI997
aI717
aI7505
asS'd'
I573
sa(dp1146
S'a'
I611
sS's'
I13
sS'm'
(lp1147
I1
aI3
aI1
aI11
aI13
aI63
aI35
aI127
aI209
aI831
aI501
aI3017
aI3507
asS'd'
I574
sa(dp1148
S'a'
I613
sS's'
I13
sS'm'
(lp1149
I1
aI3
aI7
aI9
aI29
aI7
aI11
aI163
aI81
aI563
aI1445
aI3215
aI6377
asS'd'
I575
sa(dp1150
S'a'
I625
sS's'
I13
sS'm'
(lp1151
I1
aI3
aI7
aI11
aI25
aI3
aI39
aI195
aI491
aI45
aI839
aI4021
aI4899
asS'd'
I576
sa(dp1152
S'a'
I644
sS's'
I13
sS'm'
(lp1153
I1
aI3
aI7
aI15
aI13
aI5
aI67
aI143
aI117
aI505
aI1281
aI3679
aI5695
asS'd'
I577
sa(dp1154
S'a'
I651
sS's'
I13
sS'm'
(lp1155
I1
aI3
aI7
aI9
aI9
aI19
aI21
aI221
aI147
aI763
aI683
aI2211
aI589
asS'd'
I578
sa(dp1156
S'a'
I654
sS's'
I13
sS'm'
(lp1157
I1
aI1
aI3
aI5
aI21
aI47
aI53
aI109
aI299
aI807
aI1153
aI1209
aI7961
asS'd'
I579
sa(dp1158
S'a'
I656
sS's'
I13
sS'm'
(lp1159
I1
aI3
aI7
aI11
aI9
aI31
aI45
aI43
aI505
aI647
aI1127
aI2681
aI4917
asS'd'
I580
sa(dp1160
S'a'
I662
sS's'
I13
sS'm'
(lp1161
I1
aI1
aI5
aI15
aI31
aI41
aI63
aI113
aI399
aI727
aI673
aI2587
aI5259
asS'd'
I581
sa(dp1162
S'a'
I668
sS's'
I13
sS'm'
(lp1163
I1
aI1
aI1
aI13
aI17
aI53
aI35
aI99
aI57
aI243
aI1447
aI1919
aI2831
asS'd'
I582
sa(dp1164
S'a'
I681
sS's'
I13
sS'm'
(lp1165
I1
aI3
aI7
aI11
aI23
aI51
aI13
aI9
aI49
aI449
aI997
aI3073
aI4407
asS'd'
I583
sa(dp1166
S'a'
I682
sS's'
I13
sS'm'
(lp1167
I1
aI3
aI5
aI7
aI23
aI33
aI89
aI41
aI415
aI53
aI697
aI1113
aI1489
asS'd'
I584
sa(dp1168
S'a'
I689
sS's'
I13
sS'm'
(lp1169
I1
aI1
aI3
aI7
aI1
aI13
aI29
aI13
aI255
aI749
aI77
aI3463
aI1761
asS'd'
I585
sa(dp1170
S'a'
I696
sS's'
I13
sS'm'
(lp1171
I1
aI3
aI3
aI7
aI13
aI15
aI93
aI191
aI309
aI869
aI739
aI1041
aI3053
asS'd'
I586
sa(dp1172
S'a'
I699
sS's'
I13
sS'm'
(lp1173
I1
aI3
aI5
aI13
aI5
aI19
aI109
aI211
aI347
aI839
aI893
aI2947
aI7735
asS'd'
I587
sa(dp1174
S'a'
I707
sS's'
I13
sS'm'
(lp1175
I1
aI3
aI1
aI13
aI27
aI3
aI119
aI157
aI485
aI99
aI1703
aI3895
aI573
asS'd'
I588
sa(dp1176
S'a'
I709
sS's'
I13
sS'm'
(lp1177
I1
aI3
aI7
aI11
aI1
aI23
aI123
aI105
aI31
aI359
aI275
aI1775
aI3685
asS'd'
I589
sa(dp1178
S'a'
I714
sS's'
I13
sS'm'
(lp1179
I1
aI3
aI3
aI5
aI27
aI11
aI125
aI3
aI413
aI199
aI2043
aI2895
aI2945
asS'd'
I590
sa(dp1180
S'a'
I716
sS's'
I13
sS'm'
(lp1181
I1
aI3
aI3
aI3
aI15
aI49
aI121
aI159
aI233
aI543
aI193
aI4007
aI321
asS'd'
I591
sa(dp1182
S'a'
I719
sS's'
I13
sS'm'
(lp1183
I1
aI1
aI3
aI5
aI9
aI47
aI87
aI1
aI51
aI1011
aI1595
aI2239
aI6467
asS'd'
I592
sa(dp1184
S'a'
I727
sS's'
I13
sS'm'
(lp1185
I1
aI3
aI7
aI9
aI1
aI33
aI87
aI137
aI469
aI749
aI1413
aI805
aI6817
asS'd'
I593
sa(dp1186
S'a'
I734
sS's'
I13
sS'm'
(lp1187
I1
aI3
aI1
aI13
aI19
aI45
aI95
aI227
aI29
aI677
aI1275
aI3395
aI4451
asS'd'
I594
sa(dp1188
S'a'
I738
sS's'
I13
sS'm'
(lp1189
I1
aI1
aI7
aI5
aI7
aI63
aI33
aI71
aI443
aI561
aI1311
aI3069
aI6943
asS'd'
I595
sa(dp1190
S'a'
I743
sS's'
I13
sS'm'
(lp1191
I1
aI1
aI1
aI13
aI9
aI37
aI23
aI69
aI13
aI415
aI1479
aI1197
aI861
asS'd'
I596
sa(dp1192
S'a'
I747
sS's'
I13
sS'm'
(lp1193
I1
aI3
aI3
aI13
aI27
aI21
aI13
aI233
aI105
aI777
aI345
aI2443
aI1105
asS'd'
I597
sa(dp1194
S'a'
I757
sS's'
I13
sS'm'
(lp1195
I1
aI1
aI7
aI11
aI23
aI13
aI21
aI147
aI221
aI549
aI73
aI2729
aI6279
asS'd'
I598
sa(dp1196
S'a'
I769
sS's'
I13
sS'm'
(lp1197
I1
aI1
aI7
aI7
aI25
aI27
aI15
aI45
aI227
aI39
aI75
aI1191
aI3563
asS'd'
I599
sa(dp1198
S'a'
I770
sS's'
I13
sS'm'
(lp1199
I1
aI1
aI5
aI7
aI13
aI49
aI99
aI167
aI227
aI13
aI353
aI1047
aI8075
asS'd'
I600
sa(dp1200
S'a'
I776
sS's'
I13
sS'm'
(lp1201
I1
aI1
aI3
aI13
aI31
aI9
aI27
aI7
aI461
aI737
aI1559
aI3243
aI53
asS'd'
I601
sa(dp1202
S'a'
I790
sS's'
I13
sS'm'
(lp1203
I1
aI3
aI1
aI1
aI21
aI41
aI97
aI165
aI171
aI821
aI587
aI2137
aI2293
asS'd'
I602
sa(dp1204
S'a'
I799
sS's'
I13
sS'm'
(lp1205
I1
aI3
aI1
aI11
aI17
aI41
aI29
aI187
aI87
aI599
aI1467
aI1395
aI5931
asS'd'
I603
sa(dp1206
S'a'
I805
sS's'
I13
sS'm'
(lp1207
I1
aI1
aI1
aI9
aI9
aI49
aI89
aI205
aI409
aI453
aI61
aI1923
aI1257
asS'd'
I604
sa(dp1208
S'a'
I809
sS's'
I13
sS'm'
(lp1209
I1
aI3
aI7
aI3
aI9
aI43
aI89
aI143
aI431
aI83
aI1243
aI1795
aI3599
asS'd'
I605
sa(dp1210
S'a'
I812
sS's'
I13
sS'm'
(lp1211
I1
aI3
aI5
aI13
aI3
aI25
aI59
aI219
aI43
aI223
aI797
aI2651
aI6015
asS'd'
I606
sa(dp1212
S'a'
I820
sS's'
I13
sS'm'
(lp1213
I1
aI1
aI5
aI15
aI7
aI55
aI65
aI207
aI213
aI311
aI1287
aI1269
aI6467
asS'd'
I607
sa(dp1214
S'a'
I827
sS's'
I13
sS'm'
(lp1215
I1
aI3
aI7
aI11
aI21
aI57
aI31
aI183
aI351
aI857
aI911
aI1683
aI7155
asS'd'
I608
sa(dp1216
S'a'
I829
sS's'
I13
sS'm'
(lp1217
I1
aI3
aI5
aI11
aI27
aI1
aI21
aI47
aI387
aI383
aI1593
aI115
aI3805
asS'd'
I609
sa(dp1218
S'a'
I835
sS's'
I13
sS'm'
(lp1219
I1
aI3
aI1
aI1
aI13
aI23
aI87
aI173
aI181
aI619
aI1653
aI3931
aI6073
asS'd'
I610
sa(dp1220
S'a'
I841
sS's'
I13
sS'm'
(lp1221
I1
aI1
aI7
aI5
aI17
aI43
aI37
aI61
aI307
aI621
aI1785
aI55
aI115
asS'd'
I611
sa(dp1222
S'a'
I844
sS's'
I13
sS'm'
(lp1223
I1
aI3
aI7
aI15
aI25
aI61
aI123
aI15
aI237
aI671
aI1473
aI467
aI1907
asS'd'
I612
sa(dp1224
S'a'
I856
sS's'
I13
sS'm'
(lp1225
I1
aI1
aI7
aI5
aI29
aI57
aI75
aI237
aI85
aI699
aI159
aI3577
aI4771
asS'd'
I613
sa(dp1226
S'a'
I859
sS's'
I13
sS'm'
(lp1227
I1
aI1
aI1
aI11
aI25
aI19
aI51
aI1
aI147
aI31
aI895
aI2617
aI625
asS'd'
I614
sa(dp1228
S'a'
I862
sS's'
I13
sS'm'
(lp1229
I1
aI3
aI7
aI5
aI29
aI15
aI115
aI175
aI395
aI391
aI1141
aI1827
aI1181
asS'd'
I615
sa(dp1230
S'a'
I865
sS's'
I13
sS'm'
(lp1231
I1
aI3
aI5
aI7
aI17
aI7
aI11
aI193
aI89
aI243
aI561
aI3787
aI4551
asS'd'
I616
sa(dp1232
S'a'
I885
sS's'
I13
sS'm'
(lp1233
I1
aI3
aI1
aI11
aI7
aI57
aI7
aI125
aI403
aI947
aI1261
aI409
aI8083
asS'd'
I617
sa(dp1234
S'a'
I890
sS's'
I13
sS'm'
(lp1235
I1
aI1
aI5
aI13
aI21
aI63
aI115
aI233
aI231
aI921
aI1747
aI3635
aI2519
asS'd'
I618
sa(dp1236
S'a'
I905
sS's'
I13
sS'm'
(lp1237
I1
aI1
aI5
aI11
aI3
aI27
aI15
aI91
aI505
aI591
aI1451
aI3881
aI2997
asS'd'
I619
sa(dp1238
S'a'
I916
sS's'
I13
sS'm'
(lp1239
I1
aI1
aI3
aI11
aI21
aI9
aI109
aI153
aI317
aI533
aI593
aI3967
aI2797
asS'd'
I620
sa(dp1240
S'a'
I925
sS's'
I13
sS'm'
(lp1241
I1
aI3
aI3
aI13
aI9
aI57
aI121
aI245
aI219
aI867
aI967
aI791
aI7095
asS'd'
I621
sa(dp1242
S'a'
I935
sS's'
I13
sS'm'
(lp1243
I1
aI1
aI1
aI9
aI29
aI21
aI99
aI35
aI375
aI959
aI329
aI4087
aI7171
asS'd'
I622
sa(dp1244
S'a'
I939
sS's'
I13
sS'm'
(lp1245
I1
aI1
aI1
aI9
aI11
aI17
aI17
aI97
aI89
aI135
aI631
aI3809
aI3253
asS'd'
I623
sa(dp1246
S'a'
I942
sS's'
I13
sS'm'
(lp1247
I1
aI1
aI1
aI15
aI21
aI51
aI91
aI249
aI459
aI801
aI757
aI2353
aI2033
asS'd'
I624
sa(dp1248
S'a'
I949
sS's'
I13
sS'm'
(lp1249
I1
aI3
aI5
aI9
aI23
aI29
aI77
aI53
aI399
aI767
aI1817
aI2171
aI1629
asS'd'
I625
sa(dp1250
S'a'
I953
sS's'
I13
sS'm'
(lp1251
I1
aI1
aI3
aI5
aI29
aI5
aI43
aI121
aI17
aI859
aI1479
aI3785
aI6641
asS'd'
I626
sa(dp1252
S'a'
I956
sS's'
I13
sS'm'
(lp1253
I1
aI1
aI3
aI7
aI7
aI61
aI45
aI109
aI371
aI833
aI91
aI153
aI4553
asS'd'
I627
sa(dp1254
S'a'
I961
sS's'
I13
sS'm'
(lp1255
I1
aI1
aI3
aI11
aI7
aI55
aI81
aI123
aI389
aI139
aI1933
aI891
aI1789
asS'd'
I628
sa(dp1256
S'a'
I968
sS's'
I13
sS'm'
(lp1257
I1
aI3
aI7
aI15
aI25
aI17
aI93
aI165
aI503
aI717
aI1553
aI1475
aI1627
asS'd'
I629
sa(dp1258
S'a'
I976
sS's'
I13
sS'm'
(lp1259
I1
aI1
aI1
aI13
aI13
aI63
aI13
aI225
aI357
aI571
aI33
aI4073
aI3795
asS'd'
I630
sa(dp1260
S'a'
I988
sS's'
I13
sS'm'
(lp1261
I1
aI1
aI3
aI11
aI1
aI31
aI107
aI145
aI407
aI961
aI501
aI2987
aI103
asS'd'
I631
sa(dp1262
S'a'
I995
sS's'
I13
sS'm'
(lp1263
I1
aI1
aI7
aI1
aI23
aI63
aI49
aI193
aI173
aI281
aI25
aI2465
aI5927
asS'd'
I632
sa(dp1264
S'a'
I997
sS's'
I13
sS'm'
(lp1265
I1
aI1
aI7
aI1
aI1
aI1
aI85
aI77
aI273
aI693
aI349
aI1239
aI4503
asS'd'
I633
sa(dp1266
S'a'
I1007
sS's'
I13
sS'm'
(lp1267
I1
aI1
aI5
aI11
aI7
aI61
aI9
aI121
aI25
aI357
aI1443
aI405
aI7827
asS'd'
I634
sa(dp1268
S'a'
I1015
sS's'
I13
sS'm'
(lp1269
I1
aI1
aI7
aI13
aI11
aI53
aI11
aI207
aI145
aI211
aI1703
aI1081
aI2117
asS'd'
I635
sa(dp1270
S'a'
I1016
sS's'
I13
sS'm'
(lp1271
I1
aI1
aI3
aI11
aI27
aI23
aI19
aI9
aI297
aI279
aI1481
aI2273
aI6387
asS'd'
I636
sa(dp1272
S'a'
I1027
sS's'
I13
sS'm'
(lp1273
I1
aI3
aI3
aI5
aI15
aI45
aI3
aI41
aI305
aI87
aI1815
aI3461
aI5349
asS'd'
I637
sa(dp1274
S'a'
I1036
sS's'
I13
sS'm'
(lp1275
I1
aI3
aI3
aI13
aI9
aI37
aI79
aI125
aI259
aI561
aI1087
aI4091
aI793
asS'd'
I638
sa(dp1276
S'a'
I1039
sS's'
I13
sS'm'
(lp1277
I1
aI3
aI5
aI7
aI31
aI55
aI7
aI145
aI347
aI929
aI589
aI2783
aI5905
asS'd'
I639
sa(dp1278
S'a'
I1041
sS's'
I13
sS'm'
(lp1279
I1
aI1
aI7
aI15
aI3
aI25
aI1
aI181
aI13
aI243
aI653
aI2235
aI7445
asS'd'
I640
sa(dp1280
S'a'
I1048
sS's'
I13
sS'm'
(lp1281
I1
aI3
aI5
aI5
aI17
aI53
aI65
aI7
aI33
aI583
aI1363
aI1313
aI2319
asS'd'
I641
sa(dp1282
S'a'
I1053
sS's'
I13
sS'm'
(lp1283
I1
aI3
aI3
aI7
aI27
aI47
aI97
aI201
aI187
aI321
aI63
aI1515
aI7917
asS'd'
I642
sa(dp1284
S'a'
I1054
sS's'
I13
sS'm'
(lp1285
I1
aI1
aI3
aI5
aI23
aI9
aI3
aI165
aI61
aI19
aI1789
aI3783
aI3037
asS'd'
I643
sa(dp1286
S'a'
I1058
sS's'
I13
sS'm'
(lp1287
I1
aI3
aI1
aI13
aI15
aI43
aI125
aI191
aI67
aI273
aI1551
aI2227
aI5253
asS'd'
I644
sa(dp1288
S'a'
I1075
sS's'
I13
sS'm'
(lp1289
I1
aI1
aI1
aI13
aI25
aI53
aI107
aI33
aI299
aI249
aI1475
aI2233
aI907
asS'd'
I645
sa(dp1290
S'a'
I1082
sS's'
I13
sS'm'
(lp1291
I1
aI3
aI5
aI1
aI23
aI37
aI85
aI17
aI207
aI643
aI665
aI2933
aI5199
asS'd'
I646
sa(dp1292
S'a'
I1090
sS's'
I13
sS'm'
(lp1293
I1
aI1
aI7
aI7
aI25
aI57
aI59
aI41
aI15
aI751
aI751
aI1749
aI7053
asS'd'
I647
sa(dp1294
S'a'
I1109
sS's'
I13
sS'm'
(lp1295
I1
aI3
aI3
aI1
aI13
aI25
aI127
aI93
aI281
aI613
aI875
aI2223
aI6345
asS'd'
I648
sa(dp1296
S'a'
I1110
sS's'
I13
sS'm'
(lp1297
I1
aI1
aI5
aI3
aI29
aI55
aI79
aI249
aI43
aI317
aI533
aI995
aI1991
asS'd'
I649
sa(dp1298
S'a'
I1119
sS's'
I13
sS'm'
(lp1299
I1
aI3
aI3
aI15
aI17
aI49
aI79
aI31
aI193
aI233
aI1437
aI2615
aI819
asS'd'
I650
sa(dp1300
S'a'
I1126
sS's'
I13
sS'm'
(lp1301
I1
aI1
aI5
aI15
aI25
aI3
aI123
aI145
aI377
aI9
aI455
aI1191
aI3953
asS'd'
I651
sa(dp1302
S'a'
I1130
sS's'
I13
sS'm'
(lp1303
I1
aI3
aI5
aI3
aI15
aI19
aI41
aI231
aI81
aI393
aI3
aI19
aI2409
asS'd'
I652
sa(dp1304
S'a'
I1135
sS's'
I13
sS'm'
(lp1305
I1
aI1
aI3
aI1
aI27
aI43
aI113
aI179
aI7
aI853
aI947
aI2731
aI297
asS'd'
I653
sa(dp1306
S'a'
I1137
sS's'
I13
sS'm'
(lp1307
I1
aI1
aI1
aI11
aI29
aI39
aI53
aI191
aI443
aI689
aI529
aI3329
aI7431
asS'd'
I654
sa(dp1308
S'a'
I1140
sS's'
I13
sS'm'
(lp1309
I1
aI3
aI7
aI5
aI3
aI29
aI19
aI67
aI441
aI113
aI949
aI2769
aI4169
asS'd'
I655
sa(dp1310
S'a'
I1149
sS's'
I13
sS'm'
(lp1311
I1
aI3
aI5
aI11
aI11
aI55
aI85
aI169
aI215
aI815
aI803
aI2345
aI3967
asS'd'
I656
sa(dp1312
S'a'
I1156
sS's'
I13
sS'm'
(lp1313
I1
aI1
aI7
aI9
aI5
aI45
aI111
aI5
aI419
aI375
aI303
aI1725
aI4489
asS'd'
I657
sa(dp1314
S'a'
I1159
sS's'
I13
sS'm'
(lp1315
I1
aI3
aI5
aI15
aI29
aI43
aI79
aI19
aI23
aI417
aI381
aI541
aI4923
asS'd'
I658
sa(dp1316
S'a'
I1160
sS's'
I13
sS'm'
(lp1317
I1
aI1
aI3
aI15
aI3
aI31
aI117
aI39
aI117
aI305
aI1227
aI1223
aI143
asS'd'
I659
sa(dp1318
S'a'
I1165
sS's'
I13
sS'm'
(lp1319
I1
aI1
aI5
aI9
aI5
aI47
aI87
aI239
aI181
aI353
aI1561
aI3313
aI1921
asS'd'
I660
sa(dp1320
S'a'
I1173
sS's'
I13
sS'm'
(lp1321
I1
aI3
aI3
aI1
aI3
aI15
aI53
aI221
aI441
aI987
aI1997
aI2529
aI8059
asS'd'
I661
sa(dp1322
S'a'
I1178
sS's'
I13
sS'm'
(lp1323
I1
aI1
aI7
aI11
aI15
aI57
aI111
aI139
aI137
aI883
aI1881
aI2823
aI5661
asS'd'
I662
sa(dp1324
S'a'
I1183
sS's'
I13
sS'm'
(lp1325
I1
aI3
aI5
aI5
aI21
aI11
aI5
aI13
aI27
aI973
aI587
aI1331
aI1373
asS'd'
I663
sa(dp1326
S'a'
I1184
sS's'
I13
sS'm'
(lp1327
I1
aI1
aI7
aI11
aI29
aI51
aI93
aI29
aI217
aI221
aI55
aI2477
aI1979
asS'd'
I664
sa(dp1328
S'a'
I1189
sS's'
I13
sS'm'
(lp1329
I1
aI3
aI3
aI13
aI3
aI11
aI49
aI75
aI379
aI371
aI1441
aI793
aI7633
asS'd'
I665
sa(dp1330
S'a'
I1194
sS's'
I13
sS'm'
(lp1331
I1
aI1
aI1
aI13
aI19
aI45
aI89
aI249
aI91
aI649
aI1695
aI915
aI5619
asS'd'
I666
sa(dp1332
S'a'
I1211
sS's'
I13
sS'm'
(lp1333
I1
aI3
aI1
aI7
aI7
aI29
aI1
aI77
aI313
aI895
aI519
aI771
aI295
asS'd'
I667
sa(dp1334
S'a'
I1214
sS's'
I13
sS'm'
(lp1335
I1
aI3
aI1
aI15
aI5
aI3
aI1
aI57
aI331
aI109
aI485
aI2853
aI6831
asS'd'
I668
sa(dp1336
S'a'
I1216
sS's'
I13
sS'm'
(lp1337
I1
aI1
aI1
aI15
aI17
aI3
aI35
aI99
aI245
aI971
aI839
aI2509
aI2803
asS'd'
I669
sa(dp1338
S'a'
I1225
sS's'
I13
sS'm'
(lp1339
I1
aI3
aI3
aI3
aI9
aI37
aI57
aI251
aI325
aI317
aI529
aI1313
aI6379
asS'd'
I670
sa(dp1340
S'a'
I1231
sS's'
I13
sS'm'
(lp1341
I1
aI1
aI1
aI15
aI25
aI59
aI1
aI119
aI95
aI15
aI795
aI2375
aI6463
asS'd'
I671
sa(dp1342
S'a'
I1239
sS's'
I13
sS'm'
(lp1343
I1
aI3
aI1
aI5
aI1
aI49
aI117
aI21
aI47
aI179
aI863
aI85
aI1669
asS'd'
I672
sa(dp1344
S'a'
I1243
sS's'
I13
sS'm'
(lp1345
I1
aI3
aI7
aI3
aI9
aI37
aI19
aI221
aI455
aI973
aI571
aI1427
aI817
asS'd'
I673
sa(dp1346
S'a'
I1246
sS's'
I13
sS'm'
(lp1347
I1
aI1
aI1
aI15
aI17
aI9
aI67
aI213
aI127
aI887
aI1299
aI2913
aI7451
asS'd'
I674
sa(dp1348
S'a'
I1249
sS's'
I13
sS'm'
(lp1349
I1
aI3
aI1
aI13
aI27
aI27
aI41
aI43
aI171
aI623
aI691
aI391
aI4885
asS'd'
I675
sa(dp1350
S'a'
I1259
sS's'
I13
sS'm'
(lp1351
I1
aI3
aI1
aI13
aI17
aI17
aI123
aI239
aI143
aI227
aI1151
aI519
aI6543
asS'd'
I676
sa(dp1352
S'a'
I1273
sS's'
I13
sS'm'
(lp1353
I1
aI3
aI7
aI5
aI7
aI63
aI97
aI39
aI101
aI555
aI1057
aI381
aI7891
asS'd'
I677
sa(dp1354
S'a'
I1274
sS's'
I13
sS'm'
(lp1355
I1
aI3
aI5
aI1
aI3
aI27
aI85
aI129
aI161
aI875
aI1945
aI3541
aI695
asS'd'
I678
sa(dp1356
S'a'
I1281
sS's'
I13
sS'm'
(lp1357
I1
aI3
aI3
aI5
aI21
aI59
aI25
aI183
aI35
aI25
aI987
aI1459
aI181
asS'd'
I679
sa(dp1358
S'a'
I1287
sS's'
I13
sS'm'
(lp1359
I1
aI3
aI5
aI13
aI1
aI15
aI127
aI237
aI349
aI337
aI1491
aI2383
aI7811
asS'd'
I680
sa(dp1360
S'a'
I1294
sS's'
I13
sS'm'
(lp1361
I1
aI3
aI5
aI5
aI31
aI5
aI109
aI51
aI409
aI733
aI1395
aI3207
aI6049
asS'd'
I681
sa(dp1362
S'a'
I1296
sS's'
I13
sS'm'
(lp1363
I1
aI1
aI5
aI7
aI13
aI35
aI113
aI25
aI263
aI389
aI299
aI2521
aI1783
asS'd'
I682
sa(dp1364
S'a'
I1305
sS's'
I13
sS'm'
(lp1365
I1
aI3
aI7
aI11
aI15
aI47
aI97
aI73
aI55
aI75
aI113
aI2695
aI1023
asS'd'
I683
sa(dp1366
S'a'
I1306
sS's'
I13
sS'm'
(lp1367
I1
aI3
aI1
aI1
aI3
aI13
aI69
aI211
aI289
aI483
aI1335
aI787
aI677
asS'd'
I684
sa(dp1368
S'a'
I1318
sS's'
I13
sS'm'
(lp1369
I1
aI1
aI3
aI3
aI17
aI7
aI37
aI77
aI505
aI137
aI1113
aI345
aI2975
asS'd'
I685
sa(dp1370
S'a'
I1332
sS's'
I13
sS'm'
(lp1371
I1
aI1
aI1
aI13
aI3
aI11
aI95
aI199
aI453
aI109
aI479
aI3725
aI239
asS'd'
I686
sa(dp1372
S'a'
I1335
sS's'
I13
sS'm'
(lp1373
I1
aI1
aI7
aI15
aI19
aI53
aI3
aI145
aI359
aI863
aI347
aI3833
aI3043
asS'd'
I687
sa(dp1374
S'a'
I1336
sS's'
I13
sS'm'
(lp1375
I1
aI1
aI7
aI15
aI25
aI63
aI127
aI129
aI125
aI195
aI155
aI2211
aI8153
asS'd'
I688
sa(dp1376
S'a'
I1341
sS's'
I13
sS'm'
(lp1377
I1
aI1
aI7
aI13
aI9
aI49
aI121
aI115
aI73
aI119
aI1851
aI727
aI47
asS'd'
I689
sa(dp1378
S'a'
I1342
sS's'
I13
sS'm'
(lp1379
I1
aI3
aI3
aI13
aI13
aI11
aI71
aI7
aI45
aI591
aI133
aI2407
aI5563
asS'd'
I690
sa(dp1380
S'a'
I1362
sS's'
I13
sS'm'
(lp1381
I1
aI1
aI1
aI13
aI23
aI29
aI87
aI89
aI501
aI71
aI1759
aI1119
aI687
asS'd'
I691
sa(dp1382
S'a'
I1364
sS's'
I13
sS'm'
(lp1383
I1
aI1
aI7
aI7
aI13
aI7
aI13
aI183
aI53
aI951
aI1877
aI3991
aI6771
asS'd'
I692
sa(dp1384
S'a'
I1368
sS's'
I13
sS'm'
(lp1385
I1
aI3
aI7
aI11
aI7
aI1
aI27
aI47
aI61
aI21
aI919
aI961
aI1091
asS'd'
I693
sa(dp1386
S'a'
I1378
sS's'
I13
sS'm'
(lp1387
I1
aI3
aI5
aI5
aI1
aI27
aI1
aI5
aI63
aI157
aI1297
aI1049
aI5893
asS'd'
I694
sa(dp1388
S'a'
I1387
sS's'
I13
sS'm'
(lp1389
I1
aI3
aI7
aI9
aI19
aI33
aI17
aI133
aI425
aI797
aI1721
aI153
aI119
asS'd'
I695
sa(dp1390
S'a'
I1389
sS's'
I13
sS'm'
(lp1391
I1
aI3
aI3
aI7
aI13
aI37
aI1
aI215
aI509
aI1003
aI61
aI2353
aI7511
asS'd'
I696
sa(dp1392
S'a'
I1397
sS's'
I13
sS'm'
(lp1393
I1
aI1
aI7
aI1
aI29
aI19
aI31
aI79
aI199
aI555
aI1209
aI1603
aI6089
asS'd'
I697
sa(dp1394
S'a'
I1401
sS's'
I13
sS'm'
(lp1395
I1
aI3
aI1
aI1
aI5
aI31
aI111
aI127
aI333
aI429
aI1863
aI3925
aI5411
asS'd'
I698
sa(dp1396
S'a'
I1408
sS's'
I13
sS'm'
(lp1397
I1
aI1
aI7
aI5
aI5
aI5
aI123
aI191
aI47
aI993
aI269
aI4051
aI2111
asS'd'
I699
sa(dp1398
S'a'
I1418
sS's'
I13
sS'm'
(lp1399
I1
aI1
aI5
aI15
aI1
aI9
aI87
aI5
aI47
aI463
aI865
aI1813
aI7357
asS'd'
I700
sa(dp1400
S'a'
I1425
sS's'
I13
sS'm'
(lp1401
I1
aI3
aI1
aI3
aI23
aI63
aI123
aI83
aI511
aI777
aI63
aI1285
aI4537
asS'd'
I701
sa(dp1402
S'a'
I1426
sS's'
I13
sS'm'
(lp1403
I1
aI3
aI3
aI7
aI27
aI25
aI31
aI65
aI441
aI529
aI1815
aI1893
aI323
asS'd'
I702
sa(dp1404
S'a'
I1431
sS's'
I13
sS'm'
(lp1405
I1
aI3
aI7
aI5
aI11
aI19
aI7
aI5
aI397
aI811
aI755
aI2883
aI4217
asS'd'
I703
sa(dp1406
S'a'
I1435
sS's'
I13
sS'm'
(lp1407
I1
aI3
aI1
aI13
aI9
aI21
aI13
aI7
aI271
aI539
aI1769
aI3243
aI5325
asS'd'
I704
sa(dp1408
S'a'
I1441
sS's'
I13
sS'm'
(lp1409
I1
aI1
aI7
aI1
aI31
aI13
aI47
aI131
aI181
aI457
aI1559
aI2663
aI6653
asS'd'
I705
sa(dp1410
S'a'
I1444
sS's'
I13
sS'm'
(lp1411
I1
aI3
aI3
aI7
aI29
aI55
aI25
aI203
aI419
aI91
aI437
aI1159
aI5691
asS'd'
I706
sa(dp1412
S'a'
I1462
sS's'
I13
sS'm'
(lp1413
I1
aI1
aI3
aI13
aI29
aI19
aI71
aI217
aI337
aI329
aI501
aI939
aI2205
asS'd'
I707
sa(dp1414
S'a'
I1471
sS's'
I13
sS'm'
(lp1415
I1
aI1
aI3
aI1
aI1
aI27
aI17
aI201
aI97
aI285
aI1269
aI4043
aI2207
asS'd'
I708
sa(dp1416
S'a'
I1474
sS's'
I13
sS'm'
(lp1417
I1
aI1
aI1
aI1
aI3
aI41
aI13
aI199
aI141
aI129
aI1515
aI3129
aI5969
asS'd'
I709
sa(dp1418
S'a'
I1483
sS's'
I13
sS'm'
(lp1419
I1
aI3
aI3
aI9
aI3
aI17
aI119
aI41
aI271
aI933
aI877
aI701
aI2197
asS'd'
I710
sa(dp1420
S'a'
I1485
sS's'
I13
sS'm'
(lp1421
I1
aI1
aI1
aI7
aI15
aI47
aI3
aI195
aI115
aI821
aI725
aI843
aI6071
asS'd'
I711
sa(dp1422
S'a'
I1494
sS's'
I13
sS'm'
(lp1423
I1
aI3
aI5
aI15
aI17
aI33
aI85
aI65
aI297
aI571
aI1123
aI2743
aI5727
asS'd'
I712
sa(dp1424
S'a'
I1497
sS's'
I13
sS'm'
(lp1425
I1
aI1
aI5
aI11
aI27
aI15
aI37
aI235
aI415
aI293
aI1439
aI2739
aI4171
asS'd'
I713
sa(dp1426
S'a'
I1516
sS's'
I13
sS'm'
(lp1427
I1
aI3
aI7
aI7
aI1
aI55
aI71
aI35
aI307
aI11
aI401
aI1881
aI933
asS'd'
I714
sa(dp1428
S'a'
I1522
sS's'
I13
sS'm'
(lp1429
I1
aI3
aI1
aI11
aI21
aI37
aI3
aI177
aI119
aI339
aI559
aI3991
aI3437
asS'd'
I715
sa(dp1430
S'a'
I1534
sS's'
I13
sS'm'
(lp1431
I1
aI3
aI3
aI9
aI17
aI17
aI97
aI119
aI301
aI169
aI157
aI3267
aI2261
asS'd'
I716
sa(dp1432
S'a'
I1543
sS's'
I13
sS'm'
(lp1433
I1
aI3
aI3
aI9
aI29
aI3
aI111
aI101
aI355
aI869
aI375
aI2609
aI7377
asS'd'
I717
sa(dp1434
S'a'
I1552
sS's'
I13
sS'm'
(lp1435
I1
aI3
aI5
aI9
aI7
aI21
aI123
aI99
aI343
aI693
aI1927
aI1605
aI4923
asS'd'
I718
sa(dp1436
S'a'
I1557
sS's'
I13
sS'm'
(lp1437
I1
aI1
aI3
aI5
aI13
aI31
aI99
aI17
aI75
aI385
aI1539
aI1553
aI7077
asS'd'
I719
sa(dp1438
S'a'
I1558
sS's'
I13
sS'm'
(lp1439
I1
aI3
aI3
aI5
aI31
aI35
aI107
aI11
aI407
aI1019
aI1317
aI3593
aI7203
asS'd'
I720
sa(dp1440
S'a'
I1567
sS's'
I13
sS'm'
(lp1441
I1
aI3
aI3
aI13
aI17
aI33
aI99
aI245
aI401
aI957
aI157
aI1949
aI1571
asS'd'
I721
sa(dp1442
S'a'
I1568
sS's'
I13
sS'm'
(lp1443
I1
aI3
aI1
aI11
aI27
aI15
aI11
aI109
aI429
aI307
aI1911
aI2701
aI861
asS'd'
I722
sa(dp1444
S'a'
I1574
sS's'
I13
sS'm'
(lp1445
I1
aI1
aI5
aI13
aI13
aI35
aI55
aI255
aI311
aI957
aI1803
aI2673
aI5195
asS'd'
I723
sa(dp1446
S'a'
I1592
sS's'
I13
sS'm'
(lp1447
I1
aI1
aI1
aI11
aI19
aI3
aI89
aI37
aI211
aI783
aI1355
aI3567
aI7135
asS'd'
I724
sa(dp1448
S'a'
I1605
sS's'
I13
sS'm'
(lp1449
I1
aI1
aI5
aI5
aI21
aI49
aI79
aI17
aI509
aI331
aI183
aI3831
aI855
asS'd'
I725
sa(dp1450
S'a'
I1606
sS's'
I13
sS'm'
(lp1451
I1
aI3
aI7
aI5
aI29
aI19
aI85
aI109
aI105
aI523
aI845
aI3385
aI7477
asS'd'
I726
sa(dp1452
S'a'
I1610
sS's'
I13
sS'm'
(lp1453
I1
aI1
aI1
aI7
aI25
aI17
aI125
aI131
aI53
aI757
aI253
aI2989
aI2939
asS'd'
I727
sa(dp1454
S'a'
I1617
sS's'
I13
sS'm'
(lp1455
I1
aI3
aI3
aI9
aI19
aI23
aI105
aI39
aI351
aI677
aI211
aI401
aI8103
asS'd'
I728
sa(dp1456
S'a'
I1623
sS's'
I13
sS'm'
(lp1457
I1
aI3
aI5
aI1
aI5
aI11
aI17
aI3
aI405
aI469
aI1569
aI2865
aI3133
asS'd'
I729
sa(dp1458
S'a'
I1630
sS's'
I13
sS'm'
(lp1459
I1
aI1
aI3
aI13
aI15
aI5
aI117
aI179
aI139
aI145
aI477
aI1137
aI2537
asS'd'
I730
sa(dp1460
S'a'
I1634
sS's'
I13
sS'm'
(lp1461
I1
aI1
aI7
aI9
aI5
aI21
aI9
aI93
aI211
aI963
aI1207
aI3343
aI4911
asS'd'
I731
sa(dp1462
S'a'
I1640
sS's'
I13
sS'm'
(lp1463
I1
aI1
aI1
aI9
aI13
aI43
aI17
aI53
aI81
aI793
aI1571
aI2523
aI3683
asS'd'
I732
sa(dp1464
S'a'
I1643
sS's'
I13
sS'm'
(lp1465
I1
aI3
aI3
aI13
aI25
aI21
aI5
aI59
aI489
aI987
aI1941
aI171
aI6009
asS'd'
I733
sa(dp1466
S'a'
I1648
sS's'
I13
sS'm'
(lp1467
I1
aI3
aI3
aI7
aI1
aI39
aI89
aI171
aI403
aI467
aI1767
aI3423
aI2791
asS'd'
I734
sa(dp1468
S'a'
I1651
sS's'
I13
sS'm'
(lp1469
I1
aI1
aI3
aI9
aI19
aI49
aI91
aI125
aI163
aI1013
aI89
aI2849
aI6785
asS'd'
I735
sa(dp1470
S'a'
I1653
sS's'
I13
sS'm'
(lp1471
I1
aI1
aI5
aI9
aI9
aI11
aI15
aI241
aI43
aI297
aI1719
aI1541
aI1821
asS'd'
I736
sa(dp1472
S'a'
I1670
sS's'
I13
sS'm'
(lp1473
I1
aI3
aI7
aI15
aI29
aI23
aI103
aI239
aI191
aI33
aI1043
aI3649
aI6579
asS'd'
I737
sa(dp1474
S'a'
I1676
sS's'
I13
sS'm'
(lp1475
I1
aI3
aI3
aI9
aI21
aI51
aI123
aI55
aI223
aI645
aI1463
aI4021
aI5891
asS'd'
I738
sa(dp1476
S'a'
I1684
sS's'
I13
sS'm'
(lp1477
I1
aI1
aI5
aI7
aI3
aI41
aI27
aI235
aI391
aI303
aI2021
aI3187
aI7607
asS'd'
I739
sa(dp1478
S'a'
I1687
sS's'
I13
sS'm'
(lp1479
I1
aI1
aI1
aI9
aI5
aI49
aI49
aI29
aI377
aI251
aI1887
aI1017
aI1301
asS'd'
I740
sa(dp1480
S'a'
I1691
sS's'
I13
sS'm'
(lp1481
I1
aI1
aI3
aI3
aI13
aI41
aI27
aI47
aI223
aI23
aI517
aI3227
aI6731
asS'd'
I741
sa(dp1482
S'a'
I1693
sS's'
I13
sS'm'
(lp1483
I1
aI1
aI7
aI1
aI31
aI25
aI47
aI9
aI511
aI623
aI2047
aI1263
aI1511
asS'd'
I742
sa(dp1484
S'a'
I1698
sS's'
I13
sS'm'
(lp1485
I1
aI1
aI3
aI15
aI15
aI23
aI53
aI1
aI261
aI595
aI85
aI241
aI7047
asS'd'
I743
sa(dp1486
S'a'
I1709
sS's'
I13
sS'm'
(lp1487
I1
aI3
aI3
aI11
aI17
aI5
aI81
aI73
aI149
aI781
aI2035
aI3163
aI4247
asS'd'
I744
sa(dp1488
S'a'
I1715
sS's'
I13
sS'm'
(lp1489
I1
aI3
aI7
aI7
aI29
aI59
aI49
aI79
aI397
aI901
aI1105
aI2191
aI6277
asS'd'
I745
sa(dp1490
S'a'
I1722
sS's'
I13
sS'm'
(lp1491
I1
aI3
aI3
aI11
aI13
aI27
aI25
aI173
aI107
aI73
aI1265
aI585
aI5251
asS'd'
I746
sa(dp1492
S'a'
I1732
sS's'
I13
sS'm'
(lp1493
I1
aI1
aI7
aI15
aI29
aI23
aI73
aI229
aI235
aI887
aI1469
aI4073
aI2591
asS'd'
I747
sa(dp1494
S'a'
I1735
sS's'
I13
sS'm'
(lp1495
I1
aI1
aI3
aI9
aI17
aI15
aI83
aI173
aI207
aI879
aI1701
aI1509
aI11
asS'd'
I748
sa(dp1496
S'a'
I1747
sS's'
I13
sS'm'
(lp1497
I1
aI1
aI3
aI5
aI5
aI37
aI65
aI161
aI39
aI421
aI1153
aI2007
aI5355
asS'd'
I749
sa(dp1498
S'a'
I1749
sS's'
I13
sS'm'
(lp1499
I1
aI1
aI7
aI11
aI23
aI37
aI5
aI11
aI9
aI499
aI17
aI157
aI5747
asS'd'
I750
sa(dp1500
S'a'
I1754
sS's'
I13
sS'm'
(lp1501
I1
aI3
aI7
aI13
aI25
aI9
aI49
aI7
aI39
aI945
aI1349
aI1759
aI1441
asS'd'
I751
sa(dp1502
S'a'
I1777
sS's'
I13
sS'm'
(lp1503
I1
aI1
aI5
aI3
aI21
aI15
aI113
aI81
aI265
aI837
aI333
aI3625
aI6133
asS'd'
I752
sa(dp1504
S'a'
I1784
sS's'
I13
sS'm'
(lp1505
I1
aI3
aI1
aI11
aI13
aI27
aI73
aI109
aI297
aI327
aI299
aI3253
aI6957
asS'd'
I753
sa(dp1506
S'a'
I1790
sS's'
I13
sS'm'
(lp1507
I1
aI1
aI3
aI13
aI19
aI39
aI123
aI73
aI65
aI5
aI1061
aI2187
aI5055
asS'd'
I754
sa(dp1508
S'a'
I1795
sS's'
I13
sS'm'
(lp1509
I1
aI1
aI3
aI1
aI11
aI31
aI21
aI115
aI453
aI857
aI711
aI495
aI549
asS'd'
I755
sa(dp1510
S'a'
I1801
sS's'
I13
sS'm'
(lp1511
I1
aI3
aI7
aI7
aI15
aI29
aI79
aI103
aI47
aI713
aI1735
aI3121
aI6321
asS'd'
I756
sa(dp1512
S'a'
I1802
sS's'
I13
sS'm'
(lp1513
I1
aI1
aI5
aI5
aI29
aI9
aI97
aI33
aI471
aI705
aI329
aI1501
aI1349
asS'd'
I757
sa(dp1514
S'a'
I1812
sS's'
I13
sS'm'
(lp1515
I1
aI3
aI3
aI1
aI21
aI9
aI111
aI209
aI71
aI47
aI491
aI2143
aI1797
asS'd'
I758
sa(dp1516
S'a'
I1828
sS's'
I13
sS'm'
(lp1517
I1
aI3
aI3
aI3
aI11
aI39
aI21
aI135
aI445
aI259
aI607
aI3811
aI5449
asS'd'
I759
sa(dp1518
S'a'
I1831
sS's'
I13
sS'm'
(lp1519
I1
aI1
aI7
aI9
aI11
aI25
aI113
aI251
aI395
aI317
aI317
aI91
aI1979
asS'd'
I760
sa(dp1520
S'a'
I1837
sS's'
I13
sS'm'
(lp1521
I1
aI3
aI1
aI9
aI3
aI21
aI103
aI133
aI389
aI943
aI1235
aI1749
aI7063
asS'd'
I761
sa(dp1522
S'a'
I1838
sS's'
I13
sS'm'
(lp1523
I1
aI1
aI3
aI7
aI1
aI11
aI5
aI15
aI497
aI477
aI479
aI3079
aI6969
asS'd'
I762
sa(dp1524
S'a'
I1840
sS's'
I13
sS'm'
(lp1525
I1
aI1
aI3
aI3
aI15
aI39
aI105
aI131
aI475
aI465
aI181
aI865
aI3813
asS'd'
I763
sa(dp1526
S'a'
I1845
sS's'
I13
sS'm'
(lp1527
I1
aI1
aI7
aI9
aI19
aI63
aI123
aI131
aI415
aI525
aI457
aI2471
aI3135
asS'd'
I764
sa(dp1528
S'a'
I1863
sS's'
I13
sS'm'
(lp1529
I1
aI3
aI7
aI15
aI25
aI35
aI123
aI45
aI341
aI805
aI485
aI4049
aI7065
asS'd'
I765
sa(dp1530
S'a'
I1864
sS's'
I13
sS'm'
(lp1531
I1
aI1
aI1
aI5
aI29
aI9
aI47
aI227
aI51
aI867
aI1873
aI1593
aI2271
asS'd'
I766
sa(dp1532
S'a'
I1867
sS's'
I13
sS'm'
(lp1533
I1
aI1
aI7
aI15
aI31
aI9
aI71
aI117
aI285
aI711
aI837
aI1435
aI6275
asS'd'
I767
sa(dp1534
S'a'
I1870
sS's'
I13
sS'm'
(lp1535
I1
aI3
aI1
aI1
aI5
aI19
aI79
aI25
aI301
aI415
aI1871
aI645
aI3251
asS'd'
I768
sa(dp1536
S'a'
I1877
sS's'
I13
sS'm'
(lp1537
I1
aI3
aI1
aI3
aI17
aI51
aI99
aI185
aI447
aI43
aI523
aI219
aI429
asS'd'
I769
sa(dp1538
S'a'
I1881
sS's'
I13
sS'm'
(lp1539
I1
aI3
aI1
aI13
aI29
aI13
aI51
aI93
aI7
aI995
aI757
aI3017
aI6865
asS'd'
I770
sa(dp1540
S'a'
I1884
sS's'
I13
sS'm'
(lp1541
I1
aI1
aI3
aI15
aI7
aI25
aI75
aI17
aI155
aI981
aI1231
aI1229
aI1995
asS'd'
I771
sa(dp1542
S'a'
I1903
sS's'
I13
sS'm'
(lp1543
I1
aI3
aI5
aI3
aI27
aI45
aI71
aI73
aI225
aI763
aI377
aI1139
aI2863
asS'd'
I772
sa(dp1544
S'a'
I1917
sS's'
I13
sS'm'
(lp1545
I1
aI1
aI3
aI1
aI1
aI39
aI69
aI113
aI29
aI371
aI1051
aI793
aI3749
asS'd'
I773
sa(dp1546
S'a'
I1918
sS's'
I13
sS'm'
(lp1547
I1
aI1
aI3
aI13
aI23
aI61
aI27
aI183
aI307
aI431
aI1345
aI2757
aI4031
asS'd'
I774
sa(dp1548
S'a'
I1922
sS's'
I13
sS'm'
(lp1549
I1
aI3
aI7
aI5
aI5
aI59
aI117
aI197
aI303
aI721
aI877
aI723
aI1601
asS'd'
I775
sa(dp1550
S'a'
I1924
sS's'
I13
sS'm'
(lp1551
I1
aI3
aI5
aI1
aI27
aI33
aI99
aI237
aI485
aI711
aI665
aI3077
aI5105
asS'd'
I776
sa(dp1552
S'a'
I1928
sS's'
I13
sS'm'
(lp1553
I1
aI1
aI3
aI1
aI13
aI9
aI103
aI201
aI23
aI951
aI2029
aI165
aI2093
asS'd'
I777
sa(dp1554
S'a'
I1931
sS's'
I13
sS'm'
(lp1555
I1
aI3
aI5
aI13
aI5
aI29
aI55
aI85
aI221
aI677
aI611
aI3613
aI4567
asS'd'
I778
sa(dp1556
S'a'
I1951
sS's'
I13
sS'm'
(lp1557
I1
aI1
aI1
aI1
aI7
aI61
aI9
aI233
aI261
aI561
aI953
aI4023
aI2443
asS'd'
I779
sa(dp1558
S'a'
I1952
sS's'
I13
sS'm'
(lp1559
I1
aI3
aI3
aI13
aI1
aI17
aI103
aI71
aI223
aI213
aI833
aI1747
aI6999
asS'd'
I780
sa(dp1560
S'a'
I1957
sS's'
I13
sS'm'
(lp1561
I1
aI3
aI5
aI15
aI25
aI53
aI57
aI187
aI25
aI695
aI1207
aI4089
aI2877
asS'd'
I781
sa(dp1562
S'a'
I1958
sS's'
I13
sS'm'
(lp1563
I1
aI1
aI7
aI1
aI7
aI31
aI87
aI129
aI493
aI519
aI1555
aI1155
aI4637
asS'd'
I782
sa(dp1564
S'a'
I1964
sS's'
I13
sS'm'
(lp1565
I1
aI1
aI1
aI15
aI21
aI17
aI23
aI29
aI19
aI255
aI927
aI1791
aI3093
asS'd'
I783
sa(dp1566
S'a'
I1967
sS's'
I13
sS'm'
(lp1567
I1
aI1
aI3
aI9
aI17
aI33
aI95
aI129
aI175
aI461
aI287
aI2633
aI2325
asS'd'
I784
sa(dp1568
S'a'
I1970
sS's'
I13
sS'm'
(lp1569
I1
aI3
aI5
aI7
aI23
aI19
aI63
aI209
aI249
aI583
aI1373
aI2039
aI2225
asS'd'
I785
sa(dp1570
S'a'
I1972
sS's'
I13
sS'm'
(lp1571
I1
aI3
aI3
aI5
aI5
aI19
aI79
aI241
aI459
aI355
aI1455
aI3313
aI3639
asS'd'
I786
sa(dp1572
S'a'
I1994
sS's'
I13
sS'm'
(lp1573
I1
aI1
aI7
aI9
aI21
aI41
aI97
aI119
aI129
aI769
aI1541
aI3495
aI7741
asS'd'
I787
sa(dp1574
S'a'
I2002
sS's'
I13
sS'm'
(lp1575
I1
aI1
aI7
aI11
aI9
aI29
aI35
aI255
aI141
aI937
aI1763
aI41
aI1393
asS'd'
I788
sa(dp1576
S'a'
I2007
sS's'
I13
sS'm'
(lp1577
I1
aI3
aI7
aI1
aI13
aI51
aI61
aI157
aI177
aI847
aI1829
aI3539
aI285
asS'd'
I789
sa(dp1578
S'a'
I2008
sS's'
I13
sS'm'
(lp1579
I1
aI1
aI1
aI15
aI21
aI13
aI9
aI55
aI397
aI19
aI1495
aI1255
aI7235
asS'd'
I790
sa(dp1580
S'a'
I2023
sS's'
I13
sS'm'
(lp1581
I1
aI1
aI7
aI7
aI25
aI37
aI53
aI237
aI319
aI197
aI269
aI1205
aI1485
asS'd'
I791
sa(dp1582
S'a'
I2030
sS's'
I13
sS'm'
(lp1583
I1
aI1
aI5
aI15
aI23
aI17
aI35
aI247
aI323
aI807
aI233
aI3681
aI4407
asS'd'
I792
sa(dp1584
S'a'
I2035
sS's'
I13
sS'm'
(lp1585
I1
aI1
aI3
aI7
aI9
aI59
aI85
aI105
aI493
aI763
aI1639
aI391
aI1451
asS'd'
I793
sa(dp1586
S'a'
I2038
sS's'
I13
sS'm'
(lp1587
I1
aI3
aI3
aI9
aI15
aI33
aI5
aI253
aI129
aI625
aI1527
aI2793
aI6057
asS'd'
I794
sa(dp1588
S'a'
I2042
sS's'
I13
sS'm'
(lp1589
I1
aI3
aI1
aI1
aI7
aI47
aI21
aI161
aI235
aI83
aI397
aI3563
aI5953
asS'd'
I795
sa(dp1590
S'a'
I2047
sS's'
I13
sS'm'
(lp1591
I1
aI3
aI7
aI11
aI3
aI41
aI25
aI117
aI375
aI779
aI1297
aI3715
aI8117
asS'd'
I796
sa(dp1592
S'a'
I2051
sS's'
I13
sS'm'
(lp1593
I1
aI1
aI3
aI7
aI31
aI19
aI103
aI173
aI475
aI189
aI2035
aI2921
aI1107
asS'd'
I797
sa(dp1594
S'a'
I2058
sS's'
I13
sS'm'
(lp1595
I1
aI1
aI7
aI3
aI25
aI7
aI93
aI255
aI307
aI113
aI1893
aI2233
aI6919
asS'd'
I798
sa(dp1596
S'a'
I2060
sS's'
I13
sS'm'
(lp1597
I1
aI3
aI5
aI15
aI9
aI57
aI79
aI143
aI165
aI5
aI1389
aI193
aI693
asS'd'
I799
sa(dp1598
S'a'
I2071
sS's'
I13
sS'm'
(lp1599
I1
aI3
aI5
aI1
aI29
aI45
aI91
aI49
aI189
aI461
aI439
aI1283
aI7835
asS'd'
I800
sa(dp1600
S'a'
I2084
sS's'
I13
sS'm'
(lp1601
I1
aI1
aI3
aI13
aI11
aI61
aI41
aI231
aI373
aI695
aI395
aI915
aI5393
asS'd'
I801
sa(dp1602
S'a'
I2087
sS's'
I13
sS'm'
(lp1603
I1
aI3
aI7
aI11
aI5
aI51
aI67
aI53
aI483
aI95
aI1943
aI247
aI5653
asS'd'
I802
sa(dp1604
S'a'
I2099
sS's'
I13
sS'm'
(lp1605
I1
aI3
aI7
aI5
aI5
aI57
aI45
aI235
aI137
aI793
aI1069
aI1661
aI1557
asS'd'
I803
sa(dp1606
S'a'
I2108
sS's'
I13
sS'm'
(lp1607
I1
aI3
aI5
aI3
aI25
aI55
aI103
aI177
aI81
aI861
aI1151
aI143
aI7655
asS'd'
I804
sa(dp1608
S'a'
I2111
sS's'
I13
sS'm'
(lp1609
I1
aI1
aI3
aI1
aI21
aI41
aI67
aI131
aI253
aI431
aI1269
aI3181
aI3429
asS'd'
I805
sa(dp1610
S'a'
I2120
sS's'
I13
sS'm'
(lp1611
I1
aI3
aI1
aI1
aI21
aI7
aI77
aI221
aI257
aI663
aI71
aI2949
aI2481
asS'd'
I806
sa(dp1612
S'a'
I2128
sS's'
I13
sS'm'
(lp1613
I1
aI3
aI5
aI3
aI3
aI23
aI45
aI107
aI299
aI739
aI1013
aI3
aI3165
asS'd'
I807
sa(dp1614
S'a'
I2138
sS's'
I13
sS'm'
(lp1615
I1
aI1
aI5
aI1
aI3
aI37
aI109
aI37
aI243
aI983
aI1221
aI1691
aI3869
asS'd'
I808
sa(dp1616
S'a'
I2143
sS's'
I13
sS'm'
(lp1617
I1
aI1
aI5
aI5
aI31
aI7
aI5
aI193
aI397
aI867
aI1495
aI3435
aI7441
asS'd'
I809
sa(dp1618
S'a'
I2144
sS's'
I13
sS'm'
(lp1619
I1
aI1
aI1
aI1
aI17
aI59
aI97
aI233
aI389
aI597
aI1013
aI1631
aI483
asS'd'
I810
sa(dp1620
S'a'
I2153
sS's'
I13
sS'm'
(lp1621
I1
aI1
aI1
aI11
aI7
aI41
aI107
aI53
aI111
aI125
aI1513
aI1921
aI7647
asS'd'
I811
sa(dp1622
S'a'
I2156
sS's'
I13
sS'm'
(lp1623
I1
aI3
aI3
aI3
aI31
aI29
aI117
aI3
aI365
aI971
aI1139
aI2123
aI5913
asS'd'
I812
sa(dp1624
S'a'
I2162
sS's'
I13
sS'm'
(lp1625
I1
aI1
aI1
aI13
aI23
aI3
aI1
aI167
aI475
aI639
aI1811
aI3841
aI3081
asS'd'
I813
sa(dp1626
S'a'
I2167
sS's'
I13
sS'm'
(lp1627
I1
aI1
aI5
aI3
aI5
aI47
aI65
aI123
aI275
aI783
aI95
aI119
aI7591
asS'd'
I814
sa(dp1628
S'a'
I2178
sS's'
I13
sS'm'
(lp1629
I1
aI3
aI1
aI15
aI13
aI33
aI93
aI237
aI467
aI431
aI705
aI4013
aI4035
asS'd'
I815
sa(dp1630
S'a'
I2183
sS's'
I13
sS'm'
(lp1631
I1
aI3
aI5
aI1
aI19
aI7
aI101
aI231
aI155
aI737
aI1381
aI3343
aI2051
asS'd'
I816
sa(dp1632
S'a'
I2202
sS's'
I13
sS'm'
(lp1633
I1
aI1
aI5
aI9
aI15
aI49
aI45
aI163
aI433
aI765
aI2031
aI201
aI2589
asS'd'
I817
sa(dp1634
S'a'
I2211
sS's'
I13
sS'm'
(lp1635
I1
aI3
aI7
aI9
aI19
aI41
aI31
aI89
aI93
aI623
aI105
aI745
aI4409
asS'd'
I818
sa(dp1636
S'a'
I2214
sS's'
I13
sS'm'
(lp1637
I1
aI1
aI5
aI1
aI11
aI45
aI127
aI85
aI389
aI439
aI829
aI477
aI7965
asS'd'
I819
sa(dp1638
S'a'
I2223
sS's'
I13
sS'm'
(lp1639
I1
aI3
aI3
aI15
aI13
aI41
aI1
aI207
aI435
aI585
aI311
aI1725
aI2737
asS'd'
I820
sa(dp1640
S'a'
I2225
sS's'
I13
sS'm'
(lp1641
I1
aI3
aI3
aI3
aI13
aI49
aI21
aI31
aI197
aI799
aI1411
aI2959
aI7133
asS'd'
I821
sa(dp1642
S'a'
I2232
sS's'
I13
sS'm'
(lp1643
I1
aI3
aI1
aI3
aI7
aI43
aI9
aI141
aI133
aI579
aI1059
aI93
aI957
asS'd'
I822
sa(dp1644
S'a'
I2237
sS's'
I13
sS'm'
(lp1645
I1
aI3
aI7
aI1
aI15
aI51
aI23
aI213
aI381
aI851
aI699
aI2261
aI3419
asS'd'
I823
sa(dp1646
S'a'
I2257
sS's'
I13
sS'm'
(lp1647
I1
aI3
aI5
aI9
aI25
aI35
aI67
aI141
aI35
aI409
aI1423
aI365
aI1645
asS'd'
I824
sa(dp1648
S'a'
I2260
sS's'
I13
sS'm'
(lp1649
I1
aI3
aI3
aI11
aI15
aI33
aI27
aI181
aI93
aI87
aI1761
aI3511
aI1353
asS'd'
I825
sa(dp1650
S'a'
I2267
sS's'
I13
sS'm'
(lp1651
I1
aI3
aI5
aI3
aI25
aI63
aI111
aI137
aI321
aI819
aI705
aI1547
aI7271
asS'd'
I826
sa(dp1652
S'a'
I2274
sS's'
I13
sS'm'
(lp1653
I1
aI3
aI1
aI1
aI5
aI57
aI99
aI59
aI411
aI757
aI1371
aI3953
aI3695
asS'd'
I827
sa(dp1654
S'a'
I2276
sS's'
I13
sS'm'
(lp1655
I1
aI3
aI5
aI11
aI11
aI21
aI25
aI147
aI239
aI455
aI709
aI953
aI7175
asS'd'
I828
sa(dp1656
S'a'
I2285
sS's'
I13
sS'm'
(lp1657
I1
aI3
aI3
aI15
aI5
aI53
aI91
aI205
aI341
aI63
aI723
aI1565
aI7135
asS'd'
I829
sa(dp1658
S'a'
I2288
sS's'
I13
sS'm'
(lp1659
I1
aI1
aI7
aI15
aI11
aI21
aI99
aI79
aI63
aI593
aI2007
aI3629
aI5271
asS'd'
I830
sa(dp1660
S'a'
I2293
sS's'
I13
sS'm'
(lp1661
I1
aI3
aI3
aI1
aI9
aI21
aI45
aI175
aI453
aI435
aI1855
aI2649
aI6959
asS'd'
I831
sa(dp1662
S'a'
I2294
sS's'
I13
sS'm'
(lp1663
I1
aI1
aI3
aI15
aI15
aI33
aI121
aI121
aI251
aI431
aI1127
aI3305
aI4199
asS'd'
I832
sa(dp1664
S'a'
I2297
sS's'
I13
sS'm'
(lp1665
I1
aI1
aI1
aI9
aI31
aI15
aI71
aI29
aI345
aI391
aI1159
aI2809
aI345
asS'd'
I833
sa(dp1666
S'a'
I2303
sS's'
I13
sS'm'
(lp1667
I1
aI3
aI7
aI1
aI23
aI29
aI95
aI151
aI327
aI727
aI647
aI1623
aI2971
asS'd'
I834
sa(dp1668
S'a'
I2308
sS's'
I13
sS'm'
(lp1669
I1
aI1
aI7
aI7
aI9
aI29
aI79
aI91
aI127
aI909
aI1293
aI1315
aI5315
asS'd'
I835
sa(dp1670
S'a'
I2311
sS's'
I13
sS'm'
(lp1671
I1
aI1
aI5
aI11
aI13
aI37
aI89
aI73
aI149
aI477
aI1909
aI3343
aI525
asS'd'
I836
sa(dp1672
S'a'
I2318
sS's'
I13
sS'm'
(lp1673
I1
aI3
aI5
aI7
aI5
aI59
aI55
aI255
aI223
aI459
aI2027
aI237
aI4205
asS'd'
I837
sa(dp1674
S'a'
I2323
sS's'
I13
sS'm'
(lp1675
I1
aI1
aI1
aI7
aI27
aI11
aI95
aI65
aI325
aI835
aI907
aI3801
aI3787
asS'd'
I838
sa(dp1676
S'a'
I2332
sS's'
I13
sS'm'
(lp1677
I1
aI1
aI1
aI11
aI27
aI33
aI99
aI175
aI51
aI913
aI331
aI1851
aI4133
asS'd'
I839
sa(dp1678
S'a'
I2341
sS's'
I13
sS'm'
(lp1679
I1
aI3
aI5
aI5
aI13
aI37
aI31
aI99
aI273
aI409
aI1827
aI3845
aI5491
asS'd'
I840
sa(dp1680
S'a'
I2345
sS's'
I13
sS'm'
(lp1681
I1
aI1
aI3
aI7
aI23
aI19
aI107
aI85
aI283
aI523
aI509
aI451
aI421
asS'd'
I841
sa(dp1682
S'a'
I2348
sS's'
I13
sS'm'
(lp1683
I1
aI3
aI5
aI7
aI13
aI9
aI51
aI81
aI87
aI619
aI61
aI2803
aI5271
asS'd'
I842
sa(dp1684
S'a'
I2354
sS's'
I13
sS'm'
(lp1685
I1
aI1
aI1
aI15
aI9
aI45
aI35
aI219
aI401
aI271
aI953
aI649
aI6847
asS'd'
I843
sa(dp1686
S'a'
I2368
sS's'
I13
sS'm'
(lp1687
I1
aI1
aI7
aI11
aI9
aI45
aI17
aI219
aI169
aI837
aI1483
aI1605
aI2901
asS'd'
I844
sa(dp1688
S'a'
I2377
sS's'
I13
sS'm'
(lp1689
I1
aI1
aI7
aI7
aI21
aI43
aI37
aI33
aI291
aI359
aI71
aI2899
aI7037
asS'd'
I845
sa(dp1690
S'a'
I2380
sS's'
I13
sS'm'
(lp1691
I1
aI3
aI3
aI13
aI31
aI53
aI37
aI15
aI149
aI949
aI551
aI3445
aI5455
asS'd'
I846
sa(dp1692
S'a'
I2383
sS's'
I13
sS'm'
(lp1693
I1
aI3
aI1
aI5
aI19
aI45
aI81
aI223
aI193
aI439
aI2047
aI3879
aI789
asS'd'
I847
sa(dp1694
S'a'
I2388
sS's'
I13
sS'm'
(lp1695
I1
aI1
aI7
aI3
aI11
aI63
aI35
aI61
aI255
aI563
aI459
aI2991
aI3359
asS'd'
I848
sa(dp1696
S'a'
I2395
sS's'
I13
sS'm'
(lp1697
I1
aI1
aI5
aI9
aI13
aI49
aI47
aI185
aI239
aI221
aI1533
aI3635
aI2045
asS'd'
I849
sa(dp1698
S'a'
I2397
sS's'
I13
sS'm'
(lp1699
I1
aI3
aI7
aI3
aI25
aI37
aI127
aI223
aI51
aI357
aI483
aI3837
aI6873
asS'd'
I850
sa(dp1700
S'a'
I2401
sS's'
I13
sS'm'
(lp1701
I1
aI1
aI7
aI9
aI31
aI37
aI113
aI31
aI387
aI833
aI1243
aI1543
aI5535
asS'd'
I851
sa(dp1702
S'a'
I2411
sS's'
I13
sS'm'
(lp1703
I1
aI3
aI1
aI9
aI23
aI59
aI119
aI221
aI73
aI185
aI2007
aI2885
aI2563
asS'd'
I852
sa(dp1704
S'a'
I2413
sS's'
I13
sS'm'
(lp1705
I1
aI1
aI1
aI13
aI7
aI33
aI53
aI179
aI67
aI185
aI1541
aI1807
aI4659
asS'd'
I853
sa(dp1706
S'a'
I2419
sS's'
I13
sS'm'
(lp1707
I1
aI3
aI1
aI11
aI31
aI37
aI23
aI215
aI269
aI357
aI207
aI645
aI4219
asS'd'
I854
sa(dp1708
S'a'
I2435
sS's'
I13
sS'm'
(lp1709
I1
aI3
aI3
aI13
aI19
aI27
aI107
aI55
aI91
aI71
aI1695
aI1815
aI89
asS'd'
I855
sa(dp1710
S'a'
I2442
sS's'
I13
sS'm'
(lp1711
I1
aI1
aI3
aI15
aI3
aI19
aI35
aI247
aI49
aI529
aI1523
aI3317
aI6151
asS'd'
I856
sa(dp1712
S'a'
I2455
sS's'
I13
sS'm'
(lp1713
I1
aI1
aI7
aI7
aI23
aI25
aI107
aI139
aI483
aI503
aI1277
aI243
aI7879
asS'd'
I857
sa(dp1714
S'a'
I2472
sS's'
I13
sS'm'
(lp1715
I1
aI3
aI3
aI13
aI3
aI15
aI11
aI197
aI135
aI839
aI985
aI275
aI5527
asS'd'
I858
sa(dp1716
S'a'
I2478
sS's'
I13
sS'm'
(lp1717
I1
aI3
aI5
aI3
aI25
aI47
aI95
aI21
aI113
aI307
aI1001
aI3065
aI295
asS'd'
I859
sa(dp1718
S'a'
I2490
sS's'
I13
sS'm'
(lp1719
I1
aI1
aI3
aI9
aI19
aI19
aI99
aI213
aI363
aI449
aI735
aI2851
aI2521
asS'd'
I860
sa(dp1720
S'a'
I2507
sS's'
I13
sS'm'
(lp1721
I1
aI1
aI3
aI9
aI5
aI49
aI63
aI61
aI157
aI857
aI497
aI2801
aI6987
asS'd'
I861
sa(dp1722
S'a'
I2509
sS's'
I13
sS'm'
(lp1723
I1
aI1
aI1
aI9
aI1
aI41
aI109
aI119
aI499
aI939
aI867
aI3675
aI8023
asS'd'
I862
sa(dp1724
S'a'
I2517
sS's'
I13
sS'm'
(lp1725
I1
aI3
aI1
aI1
aI13
aI33
aI109
aI123
aI289
aI3
aI1271
aI2773
aI4265
asS'd'
I863
sa(dp1726
S'a'
I2524
sS's'
I13
sS'm'
(lp1727
I1
aI3
aI1
aI11
aI9
aI57
aI83
aI221
aI95
aI43
aI1189
aI457
aI7133
asS'd'
I864
sa(dp1728
S'a'
I2528
sS's'
I13
sS'm'
(lp1729
I1
aI1
aI7
aI3
aI11
aI49
aI33
aI219
aI229
aI289
aI685
aI3359
aI4495
asS'd'
I865
sa(dp1730
S'a'
I2531
sS's'
I13
sS'm'
(lp1731
I1
aI3
aI1
aI3
aI19
aI43
aI67
aI193
aI41
aI771
aI407
aI81
aI3891
asS'd'
I866
sa(dp1732
S'a'
I2538
sS's'
I13
sS'm'
(lp1733
I1
aI1
aI7
aI11
aI5
aI29
aI51
aI175
aI297
aI539
aI1
aI2245
aI6439
asS'd'
I867
sa(dp1734
S'a'
I2545
sS's'
I13
sS'm'
(lp1735
I1
aI3
aI7
aI15
aI21
aI33
aI117
aI183
aI511
aI489
aI1283
aI3281
aI5979
asS'd'
I868
sa(dp1736
S'a'
I2546
sS's'
I13
sS'm'
(lp1737
I1
aI3
aI7
aI5
aI9
aI3
aI125
aI147
aI359
aI549
aI369
aI3049
aI2405
asS'd'
I869
sa(dp1738
S'a'
I2555
sS's'
I13
sS'm'
(lp1739
I1
aI3
aI5
aI7
aI19
aI5
aI65
aI97
aI483
aI377
aI1523
aI1457
aI2995
asS'd'
I870
sa(dp1740
S'a'
I2557
sS's'
I13
sS'm'
(lp1741
I1
aI1
aI5
aI1
aI11
aI21
aI41
aI113
aI277
aI131
aI1475
aI1043
aI2367
asS'd'
I871
sa(dp1742
S'a'
I2564
sS's'
I13
sS'm'
(lp1743
I1
aI3
aI3
aI1
aI15
aI17
aI101
aI69
aI443
aI865
aI817
aI1421
aI5231
asS'd'
I872
sa(dp1744
S'a'
I2573
sS's'
I13
sS'm'
(lp1745
I1
aI1
aI3
aI3
aI3
aI55
aI95
aI99
aI75
aI195
aI1929
aI3931
aI5855
asS'd'
I873
sa(dp1746
S'a'
I2579
sS's'
I13
sS'm'
(lp1747
I1
aI3
aI1
aI3
aI19
aI23
aI93
aI213
aI241
aI551
aI1307
aI585
aI7729
asS'd'
I874
sa(dp1748
S'a'
I2592
sS's'
I13
sS'm'
(lp1749
I1
aI3
aI1
aI11
aI23
aI15
aI53
aI249
aI467
aI519
aI95
aI741
aI409
asS'd'
I875
sa(dp1750
S'a'
I2598
sS's'
I13
sS'm'
(lp1751
I1
aI1
aI1
aI15
aI29
aI37
aI43
aI203
aI233
aI877
aI77
aI1933
aI2729
asS'd'
I876
sa(dp1752
S'a'
I2607
sS's'
I13
sS'm'
(lp1753
I1
aI3
aI7
aI11
aI27
aI39
aI43
aI161
aI255
aI15
aI1463
aI833
aI495
asS'd'
I877
sa(dp1754
S'a'
I2612
sS's'
I13
sS'm'
(lp1755
I1
aI1
aI7
aI11
aI3
aI53
aI81
aI67
aI375
aI823
aI1903
aI3061
aI395
asS'd'
I878
sa(dp1756
S'a'
I2619
sS's'
I13
sS'm'
(lp1757
I1
aI1
aI1
aI1
aI15
aI37
aI93
aI233
aI247
aI501
aI1321
aI3275
aI5409
asS'd'
I879
sa(dp1758
S'a'
I2621
sS's'
I13
sS'm'
(lp1759
I1
aI3
aI3
aI7
aI7
aI11
aI5
aI105
aI139
aI983
aI1239
aI531
aI3881
asS'd'
I880
sa(dp1760
S'a'
I2627
sS's'
I13
sS'm'
(lp1761
I1
aI1
aI5
aI3
aI19
aI49
aI107
aI227
aI361
aI101
aI355
aI2649
aI7383
asS'd'
I881
sa(dp1762
S'a'
I2633
sS's'
I13
sS'm'
(lp1763
I1
aI1
aI7
aI5
aI25
aI41
aI101
aI121
aI209
aI293
aI1937
aI2259
aI5557
asS'd'
I882
sa(dp1764
S'a'
I2636
sS's'
I13
sS'm'
(lp1765
I1
aI1
aI3
aI7
aI7
aI1
aI9
aI13
aI463
aI1019
aI995
aI3159
aI107
asS'd'
I883
sa(dp1766
S'a'
I2642
sS's'
I13
sS'm'
(lp1767
I1
aI3
aI5
aI11
aI5
aI35
aI127
aI97
aI261
aI789
aI807
aI807
aI6257
asS'd'
I884
sa(dp1768
S'a'
I2654
sS's'
I13
sS'm'
(lp1769
I1
aI1
aI7
aI5
aI11
aI13
aI45
aI91
aI417
aI101
aI1973
aI3645
aI2107
asS'd'
I885
sa(dp1770
S'a'
I2660
sS's'
I13
sS'm'
(lp1771
I1
aI1
aI3
aI7
aI5
aI63
aI57
aI49
aI203
aI157
aI115
aI1393
aI8117
asS'd'
I886
sa(dp1772
S'a'
I2669
sS's'
I13
sS'm'
(lp1773
I1
aI3
aI5
aI5
aI3
aI43
aI15
aI155
aI127
aI489
aI1165
aI3701
aI4867
asS'd'
I887
sa(dp1774
S'a'
I2675
sS's'
I13
sS'm'
(lp1775
I1
aI1
aI7
aI7
aI29
aI29
aI69
aI215
aI415
aI367
aI371
aI1901
aI6075
asS'd'
I888
sa(dp1776
S'a'
I2684
sS's'
I13
sS'm'
(lp1777
I1
aI1
aI1
aI3
aI11
aI33
aI89
aI149
aI433
aI705
aI1437
aI1597
aI505
asS'd'
I889
sa(dp1778
S'a'
I2694
sS's'
I13
sS'm'
(lp1779
I1
aI3
aI5
aI1
aI13
aI37
aI19
aI119
aI5
aI581
aI2037
aI1633
aI2099
asS'd'
I890
sa(dp1780
S'a'
I2703
sS's'
I13
sS'm'
(lp1781
I1
aI3
aI7
aI13
aI5
aI49
aI103
aI245
aI215
aI515
aI133
aI2007
aI1933
asS'd'
I891
sa(dp1782
S'a'
I2706
sS's'
I13
sS'm'
(lp1783
I1
aI3
aI1
aI9
aI1
aI3
aI25
aI197
aI253
aI387
aI1683
aI2267
aI221
asS'd'
I892
sa(dp1784
S'a'
I2712
sS's'
I13
sS'm'
(lp1785
I1
aI3
aI5
aI15
aI21
aI9
aI73
aI201
aI405
aI999
aI437
aI3877
aI6045
asS'd'
I893
sa(dp1786
S'a'
I2715
sS's'
I13
sS'm'
(lp1787
I1
aI1
aI3
aI1
aI31
aI55
aI25
aI83
aI421
aI395
aI1807
aI2129
aI7797
asS'd'
I894
sa(dp1788
S'a'
I2722
sS's'
I13
sS'm'
(lp1789
I1
aI1
aI3
aI1
aI23
aI21
aI121
aI183
aI125
aI347
aI143
aI3685
aI4317
asS'd'
I895
sa(dp1790
S'a'
I2727
sS's'
I13
sS'm'
(lp1791
I1
aI3
aI3
aI3
aI17
aI45
aI17
aI223
aI267
aI795
aI1815
aI1309
aI155
asS'd'
I896
sa(dp1792
S'a'
I2734
sS's'
I13
sS'm'
(lp1793
I1
aI1
aI1
aI15
aI17
aI59
aI5
aI133
aI15
aI715
aI1503
aI153
aI2887
asS'd'
I897
sa(dp1794
S'a'
I2742
sS's'
I13
sS'm'
(lp1795
I1
aI1
aI1
aI1
aI27
aI13
aI119
aI77
aI243
aI995
aI1851
aI3719
aI4695
asS'd'
I898
sa(dp1796
S'a'
I2745
sS's'
I13
sS'm'
(lp1797
I1
aI3
aI1
aI5
aI31
aI49
aI43
aI165
aI49
aI609
aI1265
aI1141
aI505
asS'd'
I899
sa(dp1798
S'a'
I2751
sS's'
I13
sS'm'
(lp1799
I1
aI1
aI7
aI13
aI11
aI63
aI21
aI253
aI229
aI585
aI1543
aI3719
aI4141
asS'd'
I900
sa(dp1800
S'a'
I2766
sS's'
I13
sS'm'
(lp1801
I1
aI3
aI7
aI11
aI23
aI27
aI17
aI131
aI295
aI895
aI1493
aI1411
aI3247
asS'd'
I901
sa(dp1802
S'a'
I2768
sS's'
I13
sS'm'
(lp1803
I1
aI1
aI5
aI9
aI29
aI7
aI97
aI15
aI113
aI445
aI859
aI1483
aI1121
asS'd'
I902
sa(dp1804
S'a'
I2780
sS's'
I13
sS'm'
(lp1805
I1
aI3
aI1
aI9
aI13
aI49
aI99
aI107
aI323
aI201
aI681
aI3071
aI5281
asS'd'
I903
sa(dp1806
S'a'
I2790
sS's'
I13
sS'm'
(lp1807
I1
aI1
aI1
aI15
aI9
aI19
aI61
aI161
aI7
aI87
aI587
aI2199
aI2811
asS'd'
I904
sa(dp1808
S'a'
I2794
sS's'
I13
sS'm'
(lp1809
I1
aI3
aI3
aI15
aI15
aI19
aI95
aI45
aI299
aI829
aI981
aI3479
aI487
asS'd'
I905
sa(dp1810
S'a'
I2796
sS's'
I13
sS'm'
(lp1811
I1
aI1
aI1
aI9
aI3
aI37
aI7
aI19
aI227
aI13
aI397
aI513
aI1257
asS'd'
I906
sa(dp1812
S'a'
I2801
sS's'
I13
sS'm'
(lp1813
I1
aI1
aI5
aI15
aI15
aI13
aI17
aI111
aI135
aI929
aI1145
aI811
aI1801
asS'd'
I907
sa(dp1814
S'a'
I2804
sS's'
I13
sS'm'
(lp1815
I1
aI3
aI1
aI3
aI27
aI57
aI31
aI19
aI279
aI103
aI693
aI631
aI3409
asS'd'
I908
sa(dp1816
S'a'
I2807
sS's'
I13
sS'm'
(lp1817
I1
aI1
aI1
aI1
aI15
aI13
aI67
aI83
aI23
aI799
aI1735
aI2063
aI3363
asS'd'
I909
sa(dp1818
S'a'
I2816
sS's'
I13
sS'm'
(lp1819
I1
aI3
aI3
aI7
aI3
aI1
aI61
aI31
aI41
aI533
aI2025
aI4067
aI6963
asS'd'
I910
sa(dp1820
S'a'
I2821
sS's'
I13
sS'm'
(lp1821
I1
aI1
aI5
aI7
aI17
aI27
aI81
aI79
aI107
aI205
aI29
aI97
aI4883
asS'd'
I911
sa(dp1822
S'a'
I2831
sS's'
I13
sS'm'
(lp1823
I1
aI1
aI1
aI5
aI19
aI49
aI91
aI201
aI283
aI949
aI651
aI3819
aI5073
asS'd'
I912
sa(dp1824
S'a'
I2834
sS's'
I13
sS'm'
(lp1825
I1
aI1
aI7
aI9
aI11
aI13
aI73
aI197
aI37
aI219
aI1931
aI3369
aI6017
asS'd'
I913
sa(dp1826
S'a'
I2839
sS's'
I13
sS'm'
(lp1827
I1
aI1
aI7
aI15
aI11
aI7
aI75
aI205
aI7
aI819
aI399
aI661
aI6487
asS'd'
I914
sa(dp1828
S'a'
I2845
sS's'
I13
sS'm'
(lp1829
I1
aI3
aI3
aI3
aI27
aI37
aI95
aI41
aI307
aI165
aI1077
aI3485
aI563
asS'd'
I915
sa(dp1830
S'a'
I2852
sS's'
I13
sS'm'
(lp1831
I1
aI3
aI5
aI3
aI21
aI49
aI57
aI179
aI109
aI627
aI1789
aI431
aI2941
asS'd'
I916
sa(dp1832
S'a'
I2856
sS's'
I13
sS'm'
(lp1833
I1
aI1
aI7
aI5
aI11
aI19
aI43
aI137
aI149
aI679
aI1543
aI245
aI1381
asS'd'
I917
sa(dp1834
S'a'
I2861
sS's'
I13
sS'm'
(lp1835
I1
aI3
aI5
aI5
aI15
aI3
aI69
aI81
aI135
aI159
aI1363
aI3401
aI6355
asS'd'
I918
sa(dp1836
S'a'
I2873
sS's'
I13
sS'm'
(lp1837
I1
aI3
aI5
aI1
aI9
aI61
aI49
aI53
aI319
aI25
aI1647
aI1297
aI615
asS'd'
I919
sa(dp1838
S'a'
I2874
sS's'
I13
sS'm'
(lp1839
I1
aI3
aI5
aI11
aI31
aI43
aI9
aI101
aI71
aI919
aI335
aI3147
aI5823
asS'd'
I920
sa(dp1840
S'a'
I2888
sS's'
I13
sS'm'
(lp1841
I1
aI3
aI1
aI1
aI15
aI5
aI29
aI109
aI511
aI945
aI867
aI3677
aI6915
asS'd'
I921
sa(dp1842
S'a'
I2893
sS's'
I13
sS'm'
(lp1843
I1
aI3
aI3
aI15
aI17
aI49
aI91
aI111
aI215
aI29
aI1879
aI97
aI2505
asS'd'
I922
sa(dp1844
S'a'
I2894
sS's'
I13
sS'm'
(lp1845
I1
aI3
aI1
aI13
aI19
aI61
aI11
aI111
aI163
aI777
aI533
aI1113
aI5339
asS'd'
I923
sa(dp1846
S'a'
I2902
sS's'
I13
sS'm'
(lp1847
I1
aI1
aI7
aI9
aI17
aI55
aI117
aI91
aI455
aI289
aI557
aI913
aI4455
asS'd'
I924
sa(dp1848
S'a'
I2917
sS's'
I13
sS'm'
(lp1849
I1
aI3
aI1
aI7
aI25
aI19
aI123
aI37
aI1
aI277
aI717
aI2965
aI4469
asS'd'
I925
sa(dp1850
S'a'
I2921
sS's'
I13
sS'm'
(lp1851
I1
aI3
aI7
aI3
aI19
aI23
aI87
aI235
aI209
aI457
aI2041
aI2893
aI1805
asS'd'
I926
sa(dp1852
S'a'
I2922
sS's'
I13
sS'm'
(lp1853
I1
aI3
aI3
aI5
aI5
aI43
aI23
aI61
aI351
aI791
aI59
aI2009
aI2909
asS'd'
I927
sa(dp1854
S'a'
I2929
sS's'
I13
sS'm'
(lp1855
I1
aI1
aI3
aI7
aI5
aI1
aI27
aI231
aI385
aI257
aI1261
aI2701
aI1807
asS'd'
I928
sa(dp1856
S'a'
I2935
sS's'
I13
sS'm'
(lp1857
I1
aI3
aI1
aI1
aI27
aI19
aI87
aI253
aI131
aI685
aI1743
aI3983
aI2651
asS'd'
I929
sa(dp1858
S'a'
I2946
sS's'
I13
sS'm'
(lp1859
I1
aI3
aI7
aI11
aI21
aI17
aI11
aI81
aI191
aI641
aI1821
aI3005
aI7251
asS'd'
I930
sa(dp1860
S'a'
I2951
sS's'
I13
sS'm'
(lp1861
I1
aI3
aI3
aI5
aI15
aI31
aI41
aI213
aI55
aI931
aI1953
aI49
aI6037
asS'd'
I931
sa(dp1862
S'a'
I2957
sS's'
I13
sS'm'
(lp1863
I1
aI1
aI7
aI15
aI7
aI27
aI65
aI223
aI113
aI79
aI1875
aI911
aI5445
asS'd'
I932
sa(dp1864
S'a'
I2960
sS's'
I13
sS'm'
(lp1865
I1
aI3
aI7
aI7
aI23
aI55
aI51
aI167
aI495
aI25
aI1585
aI3447
aI799
asS'd'
I933
sa(dp1866
S'a'
I2966
sS's'
I13
sS'm'
(lp1867
I1
aI1
aI3
aI7
aI27
aI15
aI95
aI193
aI337
aI415
aI975
aI3085
aI967
asS'd'
I934
sa(dp1868
S'a'
I2972
sS's'
I13
sS'm'
(lp1869
I1
aI1
aI7
aI15
aI19
aI7
aI93
aI41
aI433
aI551
aI401
aI3169
aI3971
asS'd'
I935
sa(dp1870
S'a'
I2976
sS's'
I13
sS'm'
(lp1871
I1
aI1
aI7
aI11
aI13
aI15
aI53
aI69
aI433
aI59
aI1117
aI3359
aI6231
asS'd'
I936
sa(dp1872
S'a'
I2979
sS's'
I13
sS'm'
(lp1873
I1
aI1
aI7
aI3
aI23
aI5
aI115
aI201
aI225
aI109
aI1903
aI3897
aI6265
asS'd'
I937
sa(dp1874
S'a'
I2985
sS's'
I13
sS'm'
(lp1875
I1
aI1
aI1
aI11
aI17
aI1
aI39
aI143
aI361
aI659
aI1105
aI23
aI4923
asS'd'
I938
sa(dp1876
S'a'
I3000
sS's'
I13
sS'm'
(lp1877
I1
aI1
aI1
aI9
aI27
aI57
aI85
aI227
aI261
aI119
aI1881
aI3965
aI6999
asS'd'
I939
sa(dp1878
S'a'
I3003
sS's'
I13
sS'm'
(lp1879
I1
aI3
aI7
aI7
aI15
aI7
aI107
aI17
aI315
aI49
aI1591
aI905
aI7789
asS'd'
I940
sa(dp1880
S'a'
I3013
sS's'
I13
sS'm'
(lp1881
I1
aI3
aI1
aI7
aI29
aI3
aI47
aI237
aI157
aI769
aI839
aI3199
aI3195
asS'd'
I941
sa(dp1882
S'a'
I3018
sS's'
I13
sS'm'
(lp1883
I1
aI1
aI3
aI15
aI25
aI39
aI63
aI15
aI111
aI857
aI881
aI1505
aI7671
asS'd'
I942
sa(dp1884
S'a'
I3020
sS's'
I13
sS'm'
(lp1885
I1
aI1
aI7
aI1
aI3
aI35
aI41
aI215
aI99
aI895
aI1025
aI1483
aI4707
asS'd'
I943
sa(dp1886
S'a'
I3025
sS's'
I13
sS'm'
(lp1887
I1
aI3
aI5
aI1
aI1
aI31
aI25
aI247
aI113
aI841
aI397
aI1825
aI6969
asS'd'
I944
sa(dp1888
S'a'
I3042
sS's'
I13
sS'm'
(lp1889
I1
aI1
aI3
aI5
aI19
aI41
aI49
aI243
aI225
aI973
aI241
aI175
aI1041
asS'd'
I945
sa(dp1890
S'a'
I3047
sS's'
I13
sS'm'
(lp1891
I1
aI1
aI1
aI7
aI15
aI15
aI105
aI141
aI83
aI75
aI1675
aI3523
aI5219
asS'd'
I946
sa(dp1892
S'a'
I3048
sS's'
I13
sS'm'
(lp1893
I1
aI1
aI7
aI5
aI13
aI27
aI47
aI199
aI445
aI841
aI959
aI1157
aI2209
asS'd'
I947
sa(dp1894
S'a'
I3051
sS's'
I13
sS'm'
(lp1895
I1
aI3
aI5
aI15
aI23
aI31
aI31
aI81
aI85
aI33
aI785
aI2639
aI7799
asS'd'
I948
sa(dp1896
S'a'
I3054
sS's'
I13
sS'm'
(lp1897
I1
aI1
aI5
aI13
aI21
aI3
aI47
aI99
aI235
aI943
aI1731
aI2467
aI7891
asS'd'
I949
sa(dp1898
S'a'
I3056
sS's'
I13
sS'm'
(lp1899
I1
aI1
aI1
aI3
aI17
aI53
aI85
aI219
aI73
aI131
aI1339
aI875
aI1191
asS'd'
I950
sa(dp1900
S'a'
I3065
sS's'
I13
sS'm'
(lp1901
I1
aI1
aI5
aI7
aI17
aI63
aI113
aI7
aI185
aI557
aI749
aI3563
aI4973
asS'd'
I951
sa(dp1902
S'a'
I3073
sS's'
I13
sS'm'
(lp1903
I1
aI3
aI3
aI15
aI15
aI21
aI43
aI111
aI155
aI689
aI345
aI423
aI3597
asS'd'
I952
sa(dp1904
S'a'
I3074
sS's'
I13
sS'm'
(lp1905
I1
aI1
aI5
aI1
aI15
aI29
aI93
aI5
aI361
aI713
aI695
aI3937
aI425
asS'd'
I953
sa(dp1906
S'a'
I3083
sS's'
I13
sS'm'
(lp1907
I1
aI3
aI7
aI7
aI13
aI41
aI115
aI175
aI315
aI937
aI123
aI2841
aI4457
asS'd'
I954
sa(dp1908
S'a'
I3086
sS's'
I13
sS'm'
(lp1909
I1
aI1
aI3
aI11
aI25
aI5
aI103
aI53
aI423
aI811
aI657
aI399
aI7257
asS'd'
I955
sa(dp1910
S'a'
I3091
sS's'
I13
sS'm'
(lp1911
I1
aI1
aI1
aI1
aI1
aI13
aI101
aI211
aI383
aI325
aI97
aI1703
aI4429
asS'd'
I956
sa(dp1912
S'a'
I3097
sS's'
I13
sS'm'
(lp1913
I1
aI3
aI7
aI9
aI31
aI45
aI83
aI157
aI509
aI701
aI841
aI1105
aI3643
asS'd'
I957
sa(dp1914
S'a'
I3109
sS's'
I13
sS'm'
(lp1915
I1
aI1
aI1
aI7
aI1
aI9
aI69
aI17
aI129
aI281
aI1161
aI2945
aI7693
asS'd'
I958
sa(dp1916
S'a'
I3116
sS's'
I13
sS'm'
(lp1917
I1
aI3
aI7
aI1
aI11
aI29
aI51
aI143
aI77
aI433
aI1723
aI2317
aI5641
asS'd'
I959
sa(dp1918
S'a'
I3124
sS's'
I13
sS'm'
(lp1919
I1
aI1
aI1
aI1
aI21
aI43
aI13
aI67
aI177
aI505
aI1629
aI1267
aI4885
asS'd'
I960
sa(dp1920
S'a'
I3128
sS's'
I13
sS'm'
(lp1921
I1
aI1
aI3
aI11
aI27
aI63
aI111
aI47
aI233
aI781
aI453
aI1679
aI3209
asS'd'
I961
sa(dp1922
S'a'
I3153
sS's'
I13
sS'm'
(lp1923
I1
aI1
aI3
aI13
aI29
aI27
aI119
aI141
aI493
aI971
aI461
aI1159
aI633
asS'd'
I962
sa(dp1924
S'a'
I3160
sS's'
I13
sS'm'
(lp1925
I1
aI1
aI3
aI15
aI23
aI5
aI79
aI215
aI163
aI149
aI1805
aI2399
aI61
asS'd'
I963
sa(dp1926
S'a'
I3165
sS's'
I13
sS'm'
(lp1927
I1
aI3
aI5
aI13
aI19
aI5
aI1
aI39
aI409
aI561
aI709
aI829
aI1357
asS'd'
I964
sa(dp1928
S'a'
I3172
sS's'
I13
sS'm'
(lp1929
I1
aI3
aI3
aI13
aI19
aI43
aI9
aI177
aI449
aI447
aI73
aI2107
aI5669
asS'd'
I965
sa(dp1930
S'a'
I3175
sS's'
I13
sS'm'
(lp1931
I1
aI3
aI5
aI1
aI23
aI13
aI63
aI109
aI203
aI593
aI829
aI4017
aI6881
asS'd'
I966
sa(dp1932
S'a'
I3184
sS's'
I13
sS'm'
(lp1933
I1
aI1
aI5
aI7
aI3
aI9
aI53
aI175
aI391
aI169
aI1283
aI3793
aI4451
asS'd'
I967
sa(dp1934
S'a'
I3193
sS's'
I13
sS'm'
(lp1935
I1
aI1
aI5
aI7
aI29
aI43
aI9
aI5
aI209
aI77
aI927
aI2941
aI8145
asS'd'
I968
sa(dp1936
S'a'
I3196
sS's'
I13
sS'm'
(lp1937
I1
aI3
aI5
aI15
aI17
aI49
aI5
aI143
aI131
aI771
aI1685
aI925
aI2175
asS'd'
I969
sa(dp1938
S'a'
I3200
sS's'
I13
sS'm'
(lp1939
I1
aI1
aI3
aI11
aI27
aI27
aI27
aI159
aI161
aI1015
aI1587
aI4049
aI1983
asS'd'
I970
sa(dp1940
S'a'
I3203
sS's'
I13
sS'm'
(lp1941
I1
aI3
aI1
aI3
aI23
aI57
aI119
aI67
aI481
aI577
aI389
aI3319
aI5325
asS'd'
I971
sa(dp1942
S'a'
I3205
sS's'
I13
sS'm'
(lp1943
I1
aI3
aI5
aI1
aI19
aI39
aI87
aI61
aI329
aI657
aI1773
aI31
aI1707
asS'd'
I972
sa(dp1944
S'a'
I3209
sS's'
I13
sS'm'
(lp1945
I1
aI1
aI3
aI1
aI5
aI25
aI15
aI241
aI131
aI815
aI1751
aI3029
aI8039
asS'd'
I973
sa(dp1946
S'a'
I3224
sS's'
I13
sS'm'
(lp1947
I1
aI3
aI3
aI13
aI27
aI13
aI77
aI87
aI437
aI57
aI621
aI1031
aI7891
asS'd'
I974
sa(dp1948
S'a'
I3239
sS's'
I13
sS'm'
(lp1949
I1
aI3
aI1
aI13
aI23
aI51
aI117
aI37
aI331
aI745
aI605
aI3179
aI4713
asS'd'
I975
sa(dp1950
S'a'
I3251
sS's'
I13
sS'm'
(lp1951
I1
aI1
aI5
aI5
aI19
aI17
aI99
aI167
aI87
aI721
aI737
aI789
aI2165
asS'd'
I976
sa(dp1952
S'a'
I3254
sS's'
I13
sS'm'
(lp1953
I1
aI3
aI5
aI13
aI1
aI51
aI119
aI211
aI165
aI299
aI1327
aI3053
aI3343
asS'd'
I977
sa(dp1954
S'a'
I3265
sS's'
I13
sS'm'
(lp1955
I1
aI1
aI5
aI15
aI29
aI45
aI17
aI129
aI67
aI345
aI1553
aI2705
aI7369
asS'd'
I978
sa(dp1956
S'a'
I3266
sS's'
I13
sS'm'
(lp1957
I1
aI1
aI1
aI9
aI23
aI7
aI13
aI209
aI7
aI407
aI317
aI3077
aI7287
asS'd'
I979
sa(dp1958
S'a'
I3275
sS's'
I13
sS'm'
(lp1959
I1
aI1
aI1
aI5
aI9
aI59
aI89
aI3
aI487
aI451
aI505
aI2499
aI7563
asS'd'
I980
sa(dp1960
S'a'
I3280
sS's'
I13
sS'm'
(lp1961
I1
aI3
aI1
aI7
aI21
aI1
aI21
aI203
aI101
aI417
aI1389
aI2751
aI1397
asS'd'
I981
sa(dp1962
S'a'
I3283
sS's'
I13
sS'm'
(lp1963
I1
aI3
aI7
aI13
aI7
aI31
aI3
aI247
aI349
aI485
aI1259
aI549
aI6321
asS'd'
I982
sa(dp1964
S'a'
I3286
sS's'
I13
sS'm'
(lp1965
I1
aI1
aI7
aI7
aI27
aI33
aI107
aI197
aI293
aI729
aI1753
aI2571
aI103
asS'd'
I983
sa(dp1966
S'a'
I3301
sS's'
I13
sS'm'
(lp1967
I1
aI3
aI5
aI9
aI25
aI35
aI5
aI253
aI137
aI213
aI2041
aI3387
aI1809
asS'd'
I984
sa(dp1968
S'a'
I3302
sS's'
I13
sS'm'
(lp1969
I1
aI1
aI7
aI13
aI15
aI35
aI67
aI83
aI295
aI175
aI839
aI2831
aI839
asS'd'
I985
sa(dp1970
S'a'
I3305
sS's'
I13
sS'm'
(lp1971
I1
aI3
aI3
aI11
aI3
aI17
aI55
aI141
aI247
aI991
aI117
aI3799
aI1221
asS'd'
I986
sa(dp1972
S'a'
I3319
sS's'
I13
sS'm'
(lp1973
I1
aI1
aI5
aI1
aI11
aI37
aI87
aI233
aI457
aI653
aI899
aI2933
aI3105
asS'd'
I987
sa(dp1974
S'a'
I3323
sS's'
I13
sS'm'
(lp1975
I1
aI1
aI3
aI15
aI3
aI31
aI67
aI167
aI437
aI9
aI651
aI1109
aI1139
asS'd'
I988
sa(dp1976
S'a'
I3326
sS's'
I13
sS'm'
(lp1977
I1
aI1
aI3
aI1
aI7
aI63
aI67
aI17
aI11
aI883
aI1855
aI1941
aI4751
asS'd'
I989
sa(dp1978
S'a'
I3331
sS's'
I13
sS'm'
(lp1979
I1
aI3
aI7
aI9
aI19
aI33
aI113
aI117
aI495
aI39
aI1795
aI2561
aI5519
asS'd'
I990
sa(dp1980
S'a'
I3348
sS's'
I13
sS'm'
(lp1981
I1
aI1
aI7
aI5
aI1
aI3
aI103
aI37
aI201
aI223
aI1101
aI877
aI6483
asS'd'
I991
sa(dp1982
S'a'
I3351
sS's'
I13
sS'm'
(lp1983
I1
aI1
aI5
aI9
aI29
aI49
aI51
aI33
aI439
aI917
aI861
aI1321
aI2135
asS'd'
I992
sa(dp1984
S'a'
I3358
sS's'
I13
sS'm'
(lp1985
I1
aI1
aI3
aI3
aI1
aI5
aI17
aI93
aI217
aI619
aI613
aI1357
aI6095
asS'd'
I993
sa(dp1986
S'a'
I3368
sS's'
I13
sS'm'
(lp1987
I1
aI3
aI1
aI11
aI3
aI21
aI5
aI41
aI15
aI175
aI843
aI2937
aI6849
asS'd'
I994
sa(dp1988
S'a'
I3374
sS's'
I13
sS'm'
(lp1989
I1
aI3
aI3
aI7
aI9
aI57
aI55
aI127
aI79
aI287
aI445
aI2205
aI7989
asS'd'
I995
sa(dp1990
S'a'
I3376
sS's'
I13
sS'm'
(lp1991
I1
aI1
aI7
aI13
aI23
aI17
aI93
aI129
aI157
aI135
aI1747
aI1813
aI4183
asS'd'
I996
sa(dp1992
S'a'
I3379
sS's'
I13
sS'm'
(lp1993
I1
aI1
aI1
aI5
aI31
aI59
aI99
aI33
aI425
aI329
aI887
aI367
aI1761
asS'd'
I997
sa(dp1994
S'a'
I3385
sS's'
I13
sS'm'
(lp1995
I1
aI1
aI7
aI9
aI17
aI53
aI77
aI139
aI435
aI387
aI49
aI3649
aI1773
asS'd'
I998
sa(dp1996
S'a'
I3386
sS's'
I13
sS'm'
(lp1997
I1
aI3
aI3
aI15
aI21
aI57
aI45
aI161
aI331
aI719
aI273
aI3479
aI4173
asS'd'
I999
sa(dp1998
S'a'
I3396
sS's'
I13
sS'm'
(lp1999
I1
aI1
aI3
aI9
aI3
aI3
aI105
aI201
aI373
aI877
aI919
aI1263
aI6649
asS'd'
I1000
sa(dp2000
S'a'
I3420
sS's'
I13
sS'm'
(lp2001
I1
aI3
aI1
aI15
aI13
aI43
aI13
aI99
aI73
aI163
aI353
aI3569
aI5601
asS'd'
I1001
sa(dp2002
S'a'
I3423
sS's'
I13
sS'm'
(lp2003
I1
aI3
aI7
aI3
aI5
aI9
aI69
aI177
aI449
aI47
aI781
aI1125
aI4245
asS'd'
I1002
sa(dp2004
S'a'
I3430
sS's'
I13
sS'm'
(lp2005
I1
aI1
aI1
aI5
aI3
aI45
aI1
aI123
aI409
aI903
aI205
aI2057
aI7637
asS'd'
I1003
sa(dp2006
S'a'
I3433
sS's'
I13
sS'm'
(lp2007
I1
aI3
aI5
aI9
aI19
aI47
aI87
aI135
aI481
aI799
aI101
aI3409
aI2241
asS'd'
I1004
sa(dp2008
S'a'
I3434
sS's'
I13
sS'm'
(lp2009
I1
aI3
aI1
aI13
aI3
aI25
aI15
aI27
aI181
aI967
aI669
aI2577
aI7249
asS'd'
I1005
sa(dp2010
S'a'
I3439
sS's'
I13
sS'm'
(lp2011
I1
aI1
aI7
aI3
aI31
aI5
aI103
aI53
aI1
aI911
aI1209
aI3697
aI6685
asS'd'
I1006
sa(dp2012
S'a'
I3442
sS's'
I13
sS'm'
(lp2013
I1
aI1
aI3
aI1
aI5
aI5
aI49
aI135
aI281
aI747
aI761
aI2973
aI7963
asS'd'
I1007
sa(dp2014
S'a'
I3444
sS's'
I13
sS'm'
(lp2015
I1
aI3
aI3
aI5
aI19
aI61
aI125
aI199
aI299
aI515
aI1365
aI369
aI7027
asS'd'
I1008
sa(dp2016
S'a'
I3453
sS's'
I13
sS'm'
(lp2017
I1
aI3
aI1
aI7
aI5
aI41
aI63
aI229
aI283
aI571
aI147
aI447
aI657
asS'd'
I1009
sa(dp2018
S'a'
I3464
sS's'
I13
sS'm'
(lp2019
I1
aI3
aI1
aI11
aI5
aI15
aI55
aI7
aI259
aI61
aI27
aI1429
aI5631
asS'd'
I1010
sa(dp2020
S'a'
I3477
sS's'
I13
sS'm'
(lp2021
I1
aI1
aI5
aI1
aI3
aI53
aI51
aI253
aI155
aI553
aI1293
aI3735
aI6567
asS'd'
I1011
sa(dp2022
S'a'
I3478
sS's'
I13
sS'm'
(lp2023
I1
aI3
aI5
aI9
aI5
aI41
aI21
aI159
aI101
aI785
aI1981
aI3799
aI7693
asS'd'
I1012
sa(dp2024
S'a'
I3482
sS's'
I13
sS'm'
(lp2025
I1
aI3
aI7
aI7
aI9
aI3
aI95
aI105
aI129
aI213
aI1215
aI1027
aI5699
asS'd'
I1013
sa(dp2026
S'a'
I3487
sS's'
I13
sS'm'
(lp2027
I1
aI1
aI3
aI3
aI29
aI13
aI9
aI253
aI449
aI321
aI341
aI2879
aI171
asS'd'
I1014
sa(dp2028
S'a'
I3497
sS's'
I13
sS'm'
(lp2029
I1
aI3
aI7
aI11
aI21
aI11
aI75
aI35
aI43
aI965
aI675
aI2217
aI7175
asS'd'
I1015
sa(dp2030
S'a'
I3500
sS's'
I13
sS'm'
(lp2031
I1
aI1
aI5
aI15
aI31
aI5
aI29
aI137
aI311
aI751
aI47
aI1367
aI5921
asS'd'
I1016
sa(dp2032
S'a'
I3505
sS's'
I13
sS'm'
(lp2033
I1
aI1
aI3
aI15
aI17
aI1
aI45
aI69
aI55
aI649
aI835
aI569
aI7615
asS'd'
I1017
sa(dp2034
S'a'
I3506
sS's'
I13
sS'm'
(lp2035
I1
aI3
aI1
aI13
aI31
aI7
aI23
aI15
aI391
aI145
aI1845
aI1825
aI1403
asS'd'
I1018
sa(dp2036
S'a'
I3511
sS's'
I13
sS'm'
(lp2037
I1
aI1
aI3
aI15
aI5
aI9
aI79
aI77
aI105
aI399
aI1933
aI2503
aI4781
asS'd'
I1019
sa(dp2038
S'a'
I3512
sS's'
I13
sS'm'
(lp2039
I1
aI3
aI1
aI3
aI17
aI47
aI19
aI13
aI107
aI475
aI759
aI2933
aI3761
asS'd'
I1020
sa(dp2040
S'a'
I3515
sS's'
I13
sS'm'
(lp2041
I1
aI1
aI7
aI11
aI3
aI7
aI121
aI209
aI397
aI877
aI293
aI847
aI7039
asS'd'
I1021
sa(dp2042
S'a'
I3525
sS's'
I13
sS'm'
(lp2043
I1
aI1
aI1
aI15
aI29
aI45
aI5
aI109
aI335
aI461
aI143
aI931
aI4045
asS'd'
I1022
sa(dp2044
S'a'
I3532
sS's'
I13
sS'm'
(lp2045
I1
aI3
aI1
aI7
aI11
aI57
aI73
aI89
aI201
aI173
aI803
aI3953
aI5205
asS'd'
I1023
sa(dp2046
S'a'
I3538
sS's'
I13
sS'm'
(lp2047
I1
aI1
aI5
aI11
aI11
aI33
aI37
aI29
aI263
aI1019
aI657
aI1453
aI7807
asS'd'
I1024
sa(dp2048
S'a'
I3540
sS's'
I13
sS'm'
(lp2049
I1
aI3
aI3
aI13
aI31
aI25
aI37
aI47
aI261
aI607
aI1703
aI2603
aI417
asS'd'
I1025
sa(dp2050
S'a'
I3547
sS's'
I13
sS'm'
(lp2051
I1
aI1
aI1
aI1
aI31
aI61
aI45
aI115
aI275
aI239
aI1989
aI1897
aI4329
asS'd'
I1026
sa(dp2052
S'a'
I3549
sS's'
I13
sS'm'
(lp2053
I1
aI3
aI5
aI3
aI31
aI3
aI11
aI173
aI335
aI579
aI1193
aI2219
aI7875
asS'd'
I1027
sa(dp2054
S'a'
I3560
sS's'
I13
sS'm'
(lp2055
I1
aI1
aI7
aI9
aI29
aI45
aI13
aI67
aI399
aI177
aI1293
aI3865
aI2225
asS'd'
I1028
sa(dp2056
S'a'
I3571
sS's'
I13
sS'm'
(lp2057
I1
aI1
aI7
aI11
aI11
aI51
aI121
aI227
aI469
aI905
aI929
aI2635
aI4165
asS'd'
I1029
sa(dp2058
S'a'
I3577
sS's'
I13
sS'm'
(lp2059
I1
aI3
aI7
aI9
aI13
aI39
aI55
aI167
aI23
aI147
aI1603
aI2083
aI4645
asS'd'
I1030
sa(dp2060
S'a'
I3583
sS's'
I13
sS'm'
(lp2061
I1
aI1
aI3
aI15
aI27
aI53
aI11
aI155
aI157
aI629
aI259
aI3009
aI4605
asS'd'
I1031
sa(dp2062
S'a'
I3590
sS's'
I13
sS'm'
(lp2063
I1
aI3
aI1
aI7
aI15
aI47
aI51
aI1
aI259
aI603
aI887
aI2833
aI6581
asS'd'
I1032
sa(dp2064
S'a'
I3593
sS's'
I13
sS'm'
(lp2065
I1
aI3
aI5
aI3
aI1
aI47
aI91
aI43
aI361
aI571
aI29
aI1453
aI4269
asS'd'
I1033
sa(dp2066
S'a'
I3594
sS's'
I13
sS'm'
(lp2067
I1
aI1
aI3
aI9
aI11
aI51
aI55
aI23
aI415
aI277
aI1423
aI3475
aI1527
asS'd'
I1034
sa(dp2068
S'a'
I3599
sS's'
I13
sS'm'
(lp2069
I1
aI1
aI3
aI11
aI29
aI49
aI101
aI75
aI299
aI709
aI805
aI4037
aI4389
asS'd'
I1035
sa(dp2070
S'a'
I3601
sS's'
I13
sS'm'
(lp2071
I1
aI1
aI7
aI3
aI23
aI1
aI37
aI51
aI379
aI771
aI1301
aI3717
aI6673
asS'd'
I1036
sa(dp2072
S'a'
I3602
sS's'
I13
sS'm'
(lp2073
I1
aI1
aI5
aI3
aI23
aI11
aI125
aI177
aI375
aI665
aI951
aI1577
aI2603
asS'd'
I1037
sa(dp2074
S'a'
I3613
sS's'
I13
sS'm'
(lp2075
I1
aI1
aI1
aI1
aI1
aI5
aI71
aI255
aI21
aI459
aI467
aI2083
aI5415
asS'd'
I1038
sa(dp2076
S'a'
I3623
sS's'
I13
sS'm'
(lp2077
I1
aI1
aI5
aI13
aI23
aI29
aI109
aI157
aI363
aI971
aI549
aI647
aI1177
asS'd'
I1039
sa(dp2078
S'a'
I3630
sS's'
I13
sS'm'
(lp2079
I1
aI1
aI3
aI9
aI7
aI15
aI101
aI3
aI365
aI213
aI745
aI1155
aI6099
asS'd'
I1040
sa(dp2080
S'a'
I3638
sS's'
I13
sS'm'
(lp2081
I1
aI3
aI5
aI15
aI15
aI19
aI47
aI179
aI303
aI521
aI1279
aI219
aI2415
asS'd'
I1041
sa(dp2082
S'a'
I3649
sS's'
I13
sS'm'
(lp2083
I1
aI3
aI3
aI13
aI27
aI11
aI83
aI165
aI369
aI989
aI261
aI3933
aI4809
asS'd'
I1042
sa(dp2084
S'a'
I3655
sS's'
I13
sS'm'
(lp2085
I1
aI1
aI3
aI11
aI31
aI59
aI1
aI185
aI53
aI703
aI1471
aI2935
aI1107
asS'd'
I1043
sa(dp2086
S'a'
I3662
sS's'
I13
sS'm'
(lp2087
I1
aI3
aI3
aI7
aI25
aI3
aI81
aI27
aI93
aI521
aI433
aI2859
aI5861
asS'd'
I1044
sa(dp2088
S'a'
I3667
sS's'
I13
sS'm'
(lp2089
I1
aI3
aI3
aI11
aI29
aI15
aI49
aI167
aI315
aI927
aI543
aI3473
aI4307
asS'd'
I1045
sa(dp2090
S'a'
I3669
sS's'
I13
sS'm'
(lp2091
I1
aI3
aI1
aI3
aI29
aI33
aI53
aI15
aI183
aI691
aI703
aI1311
aI3393
asS'd'
I1046
sa(dp2092
S'a'
I3676
sS's'
I13
sS'm'
(lp2093
I1
aI3
aI5
aI13
aI23
aI49
aI3
aI11
aI1
aI357
aI1407
aI415
aI7211
asS'd'
I1047
sa(dp2094
S'a'
I3683
sS's'
I13
sS'm'
(lp2095
I1
aI3
aI7
aI15
aI1
aI25
aI91
aI113
aI323
aI371
aI189
aI925
aI1181
asS'd'
I1048
sa(dp2096
S'a'
I3700
sS's'
I13
sS'm'
(lp2097
I1
aI3
aI3
aI3
aI17
aI59
aI119
aI199
aI115
aI223
aI877
aI2193
aI193
asS'd'
I1049
sa(dp2098
S'a'
I3709
sS's'
I13
sS'm'
(lp2099
I1
aI1
aI1
aI5
aI5
aI35
aI31
aI59
aI437
aI411
aI37
aI2405
aI3797
asS'd'
I1050
sa(dp2100
S'a'
I3710
sS's'
I13
sS'm'
(lp2101
I1
aI3
aI1
aI13
aI9
aI37
aI1
aI241
aI59
aI157
aI1785
aI1223
aI563
asS'd'
I1051
sa(dp2102
S'a'
I3713
sS's'
I13
sS'm'
(lp2103
I1
aI3
aI5
aI13
aI3
aI21
aI25
aI95
aI15
aI745
aI85
aI701
aI5361
asS'd'
I1052
sa(dp2104
S'a'
I3723
sS's'
I13
sS'm'
(lp2105
I1
aI3
aI7
aI1
aI31
aI33
aI111
aI195
aI35
aI913
aI2013
aI2951
aI6611
asS'd'
I1053
sa(dp2106
S'a'
I3725
sS's'
I13
sS'm'
(lp2107
I1
aI3
aI5
aI1
aI19
aI3
aI75
aI119
aI111
aI409
aI951
aI1457
aI4957
asS'd'
I1054
sa(dp2108
S'a'
I3728
sS's'
I13
sS'm'
(lp2109
I1
aI3
aI1
aI15
aI19
aI59
aI3
aI155
aI237
aI657
aI1967
aI3323
aI6235
asS'd'
I1055
sa(dp2110
S'a'
I3734
sS's'
I13
sS'm'
(lp2111
I1
aI1
aI5
aI1
aI3
aI19
aI45
aI105
aI377
aI881
aI167
aI2255
aI4483
asS'd'
I1056
sa(dp2112
S'a'
I3737
sS's'
I13
sS'm'
(lp2113
I1
aI1
aI7
aI7
aI13
aI13
aI99
aI89
aI201
aI279
aI161
aI2483
aI6001
asS'd'
I1057
sa(dp2114
S'a'
I3738
sS's'
I13
sS'm'
(lp2115
I1
aI1
aI7
aI3
aI13
aI17
aI97
aI129
aI137
aI377
aI1519
aI183
aI3725
asS'd'
I1058
sa(dp2116
S'a'
I3744
sS's'
I13
sS'm'
(lp2117
I1
aI1
aI7
aI9
aI9
aI5
aI45
aI135
aI115
aI181
aI1685
aI3505
aI4387
asS'd'
I1059
sa(dp2118
S'a'
I3750
sS's'
I13
sS'm'
(lp2119
I1
aI1
aI1
aI1
aI19
aI35
aI69
aI113
aI305
aI419
aI949
aI2969
aI247
asS'd'
I1060
sa(dp2120
S'a'
I3762
sS's'
I13
sS'm'
(lp2121
I1
aI1
aI5
aI13
aI23
aI61
aI13
aI139
aI501
aI811
aI67
aI1501
aI6493
asS'd'
I1061
sa(dp2122
S'a'
I3764
sS's'
I13
sS'm'
(lp2123
I1
aI1
aI3
aI13
aI15
aI41
aI27
aI217
aI293
aI13
aI145
aI2631
aI6991
asS'd'
I1062
sa(dp2124
S'a'
I3774
sS's'
I13
sS'm'
(lp2125
I1
aI3
aI3
aI13
aI15
aI37
aI71
aI123
aI285
aI49
aI627
aI1283
aI5993
asS'd'
I1063
sa(dp2126
S'a'
I3776
sS's'
I13
sS'm'
(lp2127
I1
aI3
aI3
aI11
aI9
aI25
aI11
aI1
aI203
aI353
aI1347
aI1999
aI2799
asS'd'
I1064
sa(dp2128
S'a'
I3786
sS's'
I13
sS'm'
(lp2129
I1
aI3
aI5
aI1
aI7
aI49
aI101
aI231
aI499
aI63
aI1977
aI2207
aI7829
asS'd'
I1065
sa(dp2130
S'a'
I3800
sS's'
I13
sS'm'
(lp2131
I1
aI1
aI7
aI1
aI17
aI15
aI115
aI139
aI381
aI943
aI623
aI4037
aI2971
asS'd'
I1066
sa(dp2132
S'a'
I3803
sS's'
I13
sS'm'
(lp2133
I1
aI1
aI3
aI5
aI13
aI55
aI23
aI87
aI139
aI795
aI1669
aI1375
aI1185
asS'd'
I1067
sa(dp2134
S'a'
I3809
sS's'
I13
sS'm'
(lp2135
I1
aI3
aI3
aI5
aI5
aI45
aI97
aI253
aI241
aI333
aI645
aI555
aI7867
asS'd'
I1068
sa(dp2136
S'a'
I3816
sS's'
I13
sS'm'
(lp2137
I1
aI3
aI5
aI1
aI1
aI1
aI89
aI27
aI407
aI509
aI1433
aI609
aI2355
asS'd'
I1069
sa(dp2138
S'a'
I3821
sS's'
I13
sS'm'
(lp2139
I1
aI3
aI7
aI1
aI27
aI29
aI5
aI157
aI495
aI811
aI1293
aI1143
aI827
asS'd'
I1070
sa(dp2140
S'a'
I3827
sS's'
I13
sS'm'
(lp2141
I1
aI1
aI3
aI3
aI25
aI49
aI127
aI111
aI191
aI3
aI845
aI1383
aI2521
asS'd'
I1071
sa(dp2142
S'a'
I3829
sS's'
I13
sS'm'
(lp2143
I1
aI1
aI5
aI7
aI5
aI51
aI101
aI155
aI237
aI461
aI831
aI3091
aI3851
asS'd'
I1072
sa(dp2144
S'a'
I3836
sS's'
I13
sS'm'
(lp2145
I1
aI3
aI7
aI1
aI29
aI35
aI105
aI91
aI285
aI705
aI131
aI395
aI6011
asS'd'
I1073
sa(dp2146
S'a'
I3842
sS's'
I13
sS'm'
(lp2147
I1
aI3
aI5
aI3
aI13
aI21
aI83
aI173
aI221
aI827
aI1775
aI1931
aI6727
asS'd'
I1074
sa(dp2148
S'a'
I3844
sS's'
I13
sS'm'
(lp2149
I1
aI1
aI3
aI5
aI3
aI25
aI95
aI115
aI205
aI569
aI1447
aI933
aI6425
asS'd'
I1075
sa(dp2150
S'a'
I3847
sS's'
I13
sS'm'
(lp2151
I1
aI1
aI7
aI9
aI31
aI3
aI17
aI175
aI145
aI447
aI1321
aI1069
aI6527
asS'd'
I1076
sa(dp2152
S'a'
I3853
sS's'
I13
sS'm'
(lp2153
I1
aI1
aI3
aI3
aI23
aI1
aI79
aI51
aI421
aI419
aI873
aI3939
aI1801
asS'd'
I1077
sa(dp2154
S'a'
I3861
sS's'
I13
sS'm'
(lp2155
I1
aI1
aI5
aI1
aI3
aI39
aI15
aI85
aI169
aI669
aI919
aI397
aI5579
asS'd'
I1078
sa(dp2156
S'a'
I3871
sS's'
I13
sS'm'
(lp2157
I1
aI3
aI5
aI1
aI21
aI61
aI87
aI217
aI251
aI619
aI1091
aI4009
aI229
asS'd'
I1079
sa(dp2158
S'a'
I3872
sS's'
I13
sS'm'
(lp2159
I1
aI1
aI1
aI11
aI23
aI55
aI85
aI121
aI363
aI867
aI315
aI447
aI3373
asS'd'
I1080
sa(dp2160
S'a'
I3881
sS's'
I13
sS'm'
(lp2161
I1
aI3
aI3
aI13
aI29
aI19
aI89
aI85
aI137
aI469
aI1873
aI2765
aI3975
asS'd'
I1081
sa(dp2162
S'a'
I3890
sS's'
I13
sS'm'
(lp2163
I1
aI3
aI7
aI13
aI19
aI63
aI61
aI77
aI67
aI361
aI11
aI1787
aI4703
asS'd'
I1082
sa(dp2164
S'a'
I3892
sS's'
I13
sS'm'
(lp2165
I1
aI1
aI3
aI11
aI7
aI15
aI127
aI105
aI179
aI857
aI1671
aI3647
aI3389
asS'd'
I1083
sa(dp2166
S'a'
I3909
sS's'
I13
sS'm'
(lp2167
I1
aI1
aI1
aI7
aI19
aI21
aI99
aI161
aI499
aI519
aI1287
aI2973
aI479
asS'd'
I1084
sa(dp2168
S'a'
I3921
sS's'
I13
sS'm'
(lp2169
I1
aI1
aI3
aI13
aI29
aI51
aI95
aI251
aI55
aI519
aI1955
aI2881
aI5951
asS'd'
I1085
sa(dp2170
S'a'
I3934
sS's'
I13
sS'm'
(lp2171
I1
aI1
aI3
aI11
aI23
aI63
aI121
aI237
aI175
aI311
aI701
aI1539
aI2383
asS'd'
I1086
sa(dp2172
S'a'
I3938
sS's'
I13
sS'm'
(lp2173
I1
aI1
aI7
aI5
aI5
aI45
aI73
aI97
aI5
aI153
aI715
aI2037
aI3353
asS'd'
I1087
sa(dp2174
S'a'
I3947
sS's'
I13
sS'm'
(lp2175
I1
aI1
aI1
aI3
aI13
aI7
aI67
aI173
aI425
aI843
aI1497
aI2729
aI5193
asS'd'
I1088
sa(dp2176
S'a'
I3950
sS's'
I13
sS'm'
(lp2177
I1
aI1
aI7
aI1
aI23
aI3
aI119
aI11
aI77
aI141
aI1905
aI2269
aI4269
asS'd'
I1089
sa(dp2178
S'a'
I3952
sS's'
I13
sS'm'
(lp2179
I1
aI1
aI7
aI15
aI1
aI23
aI79
aI251
aI439
aI603
aI405
aI2449
aI6383
asS'd'
I1090
sa(dp2180
S'a'
I3964
sS's'
I13
sS'm'
(lp2181
I1
aI3
aI7
aI11
aI29
aI27
aI47
aI255
aI47
aI661
aI1967
aI1007
aI3689
asS'd'
I1091
sa(dp2182
S'a'
I3974
sS's'
I13
sS'm'
(lp2183
I1
aI3
aI7
aI5
aI19
aI39
aI35
aI115
aI417
aI373
aI291
aI329
aI603
asS'd'
I1092
sa(dp2184
S'a'
I3980
sS's'
I13
sS'm'
(lp2185
I1
aI3
aI1
aI9
aI11
aI33
aI27
aI193
aI207
aI423
aI1311
aI1369
aI7307
asS'd'
I1093
sa(dp2186
S'a'
I3983
sS's'
I13
sS'm'
(lp2187
I1
aI1
aI3
aI11
aI9
aI29
aI83
aI17
aI497
aI493
aI329
aI3141
aI5935
asS'd'
I1094
sa(dp2188
S'a'
I3986
sS's'
I13
sS'm'
(lp2189
I1
aI3
aI1
aI5
aI31
aI51
aI29
aI171
aI51
aI493
aI1621
aI3501
aI4091
asS'd'
I1095
sa(dp2190
S'a'
I3995
sS's'
I13
sS'm'
(lp2191
I1
aI1
aI5
aI9
aI21
aI43
aI105
aI207
aI245
aI363
aI1191
aI699
aI1139
asS'd'
I1096
sa(dp2192
S'a'
I3998
sS's'
I13
sS'm'
(lp2193
I1
aI1
aI3
aI11
aI19
aI5
aI81
aI119
aI247
aI169
aI1337
aI45
aI6565
asS'd'
I1097
sa(dp2194
S'a'
I4001
sS's'
I13
sS'm'
(lp2195
I1
aI3
aI1
aI11
aI3
aI51
aI3
aI101
aI159
aI11
aI253
aI299
aI5043
asS'd'
I1098
sa(dp2196
S'a'
I4002
sS's'
I13
sS'm'
(lp2197
I1
aI3
aI1
aI5
aI11
aI53
aI85
aI39
aI57
aI645
aI2007
aI1039
aI3627
asS'd'
I1099
sa(dp2198
S'a'
I4004
sS's'
I13
sS'm'
(lp2199
I1
aI3
aI5
aI3
aI17
aI61
aI97
aI165
aI415
aI357
aI283
aI601
aI5505
asS'd'
I1100
sa(dp2200
S'a'
I4008
sS's'
I13
sS'm'
(lp2201
I1
aI3
aI7
aI3
aI9
aI51
aI49
aI85
aI3
aI227
aI137
aI309
aI243
asS'd'
I1101
sa(dp2202
S'a'
I4011
sS's'
I13
sS'm'
(lp2203
I1
aI1
aI5
aI3
aI11
aI59
aI11
aI131
aI409
aI703
aI455
aI123
aI6727
asS'd'
I1102
sa(dp2204
S'a'
I4016
sS's'
I13
sS'm'
(lp2205
I1
aI3
aI7
aI9
aI25
aI49
aI21
aI171
aI287
aI379
aI667
aI313
aI713
asS'd'
I1103
sa(dp2206
S'a'
I4033
sS's'
I13
sS'm'
(lp2207
I1
aI1
aI3
aI9
aI7
aI35
aI47
aI3
aI367
aI581
aI1627
aI1665
aI3905
asS'd'
I1104
sa(dp2208
S'a'
I4036
sS's'
I13
sS'm'
(lp2209
I1
aI3
aI1
aI1
aI29
aI57
aI35
aI55
aI255
aI653
aI823
aI2197
aI6179
asS'd'
I1105
sa(dp2210
S'a'
I4040
sS's'
I13
sS'm'
(lp2211
I1
aI3
aI7
aI15
aI17
aI15
aI117
aI83
aI359
aI163
aI115
aI2999
aI5373
asS'd'
I1106
sa(dp2212
S'a'
I4053
sS's'
I13
sS'm'
(lp2213
I1
aI1
aI5
aI3
aI21
aI61
aI35
aI97
aI71
aI687
aI207
aI2917
aI1049
asS'd'
I1107
sa(dp2214
S'a'
I4058
sS's'
I13
sS'm'
(lp2215
I1
aI1
aI1
aI15
aI13
aI15
aI125
aI81
aI263
aI661
aI417
aI3243
aI1669
asS'd'
I1108
sa(dp2216
S'a'
I4081
sS's'
I13
sS'm'
(lp2217
I1
aI1
aI7
aI3
aI3
aI19
aI111
aI193
aI443
aI339
aI659
aI1211
aI1557
asS'd'
I1109
sa(dp2218
S'a'
I4091
sS's'
I13
sS'm'
(lp2219
I1
aI3
aI1
aI3
aI27
aI3
aI3
aI173
aI391
aI213
aI803
aI3281
aI3207
asS'd'
I1110
sa(dp2220
S'a'
I4094
sS's'
I13
sS'm'
(lp2221
I1
aI1
aI5
aI15
aI19
aI1
aI7
aI211
aI157
aI603
aI403
aI1387
aI1583
asS'd'
I1111
sa(dp2222
S'a'
I21
sS's'
I14
sS'm'
(lp2223
I1
aI3
aI5
aI13
aI17
aI53
aI125
aI13
aI339
aI723
aI521
aI413
aI5801
aI10451
asS'd'
I1112
sa(dp2224
S'a'
I28
sS's'
I14
sS'm'
(lp2225
I1
aI1
aI3
aI13
aI29
aI9
aI99
aI77
aI141
aI609
aI1533
aI983
aI2039
aI51
asS'd'
I1113
sa(dp2226
S'a'
I41
sS's'
I14
sS'm'
(lp2227
I1
aI1
aI3
aI11
aI21
aI55
aI5
aI51
aI423
aI309
aI525
aI3715
aI3025
aI15055
asS'd'
I1114
sa(dp2228
S'a'
I47
sS's'
I14
sS'm'
(lp2229
I1
aI1
aI3
aI7
aI9
aI21
aI77
aI171
aI239
aI341
aI1653
aI1093
aI2273
aI10723
asS'd'
I1115
sa(dp2230
S'a'
I61
sS's'
I14
sS'm'
(lp2231
I1
aI1
aI1
aI15
aI31
aI15
aI23
aI35
aI317
aI869
aI1917
aI1415
aI4313
aI3687
asS'd'
I1116
sa(dp2232
S'a'
I84
sS's'
I14
sS'm'
(lp2233
I1
aI1
aI1
aI5
aI21
aI25
aI99
aI167
aI439
aI453
aI473
aI431
aI6665
aI4989
asS'd'
I1117
sa(dp2234
S'a'
I87
sS's'
I14
sS'm'
(lp2235
I1
aI1
aI7
aI9
aI31
aI47
aI81
aI83
aI345
aI43
aI1363
aI1885
aI3155
aI3185
asS'd'
I1118
sa(dp2236
S'a'
I93
sS's'
I14
sS'm'
(lp2237
I1
aI3
aI7
aI1
aI31
aI17
aI61
aI185
aI341
aI129
aI547
aI717
aI2071
aI9991
asS'd'
I1119
sa(dp2238
S'a'
I94
sS's'
I14
sS'm'
(lp2239
I1
aI3
aI1
aI13
aI23
aI61
aI77
aI217
aI455
aI77
aI1263
aI1601
aI3501
aI14953
asS'd'
I1120
sa(dp2240
S'a'
I103
sS's'
I14
sS'm'
(lp2241
I1
aI1
aI7
aI7
aI19
aI19
aI1
aI229
aI431
aI943
aI1069
aI1949
aI1289
aI15729
asS'd'
I1121
sa(dp2242
S'a'
I117
sS's'
I14
sS'm'
(lp2243
I1
aI1
aI3
aI5
aI1
aI35
aI97
aI251
aI487
aI459
aI1265
aI1739
aI165
aI10365
asS'd'
I1122
sa(dp2244
S'a'
I121
sS's'
I14
sS'm'
(lp2245
I1
aI3
aI5
aI3
aI11
aI25
aI79
aI175
aI383
aI545
aI187
aI197
aI4329
aI3363
asS'd'
I1123
sa(dp2246
S'a'
I134
sS's'
I14
sS'm'
(lp2247
I1
aI1
aI3
aI3
aI29
aI9
aI63
aI55
aI175
aI277
aI431
aI2549
aI2629
aI6409
asS'd'
I1124
sa(dp2248
S'a'
I137
sS's'
I14
sS'm'
(lp2249
I1
aI1
aI3
aI15
aI17
aI21
aI79
aI139
aI99
aI135
aI1763
aI1805
aI3471
aI5439
asS'd'
I1125
sa(dp2250
S'a'
I157
sS's'
I14
sS'm'
(lp2251
I1
aI1
aI3
aI9
aI9
aI15
aI35
aI119
aI289
aI835
aI769
aI3843
aI4119
aI4421
asS'd'
I1126
sa(dp2252
S'a'
I161
sS's'
I14
sS'm'
(lp2253
I1
aI1
aI1
aI5
aI19
aI19
aI67
aI199
aI307
aI815
aI1367
aI1231
aI3927
aI6593
asS'd'
I1127
sa(dp2254
S'a'
I205
sS's'
I14
sS'm'
(lp2255
I1
aI1
aI3
aI1
aI29
aI51
aI121
aI209
aI431
aI47
aI1115
aI907
aI2535
aI9755
asS'd'
I1128
sa(dp2256
S'a'
I206
sS's'
I14
sS'm'
(lp2257
I1
aI1
aI3
aI5
aI17
aI1
aI5
aI119
aI121
aI223
aI1719
aI1291
aI3947
aI15891
asS'd'
I1129
sa(dp2258
S'a'
I211
sS's'
I14
sS'm'
(lp2259
I1
aI3
aI1
aI15
aI29
aI25
aI3
aI131
aI373
aI307
aI645
aI3513
aI1289
aI1987
asS'd'
I1130
sa(dp2260
S'a'
I214
sS's'
I14
sS'm'
(lp2261
I1
aI3
aI3
aI11
aI29
aI45
aI105
aI179
aI331
aI465
aI891
aI1315
aI403
aI3057
asS'd'
I1131
sa(dp2262
S'a'
I218
sS's'
I14
sS'm'
(lp2263
I1
aI1
aI5
aI13
aI17
aI59
aI77
aI127
aI485
aI855
aI1147
aI3093
aI891
aI9869
asS'd'
I1132
sa(dp2264
S'a'
I234
sS's'
I14
sS'm'
(lp2265
I1
aI1
aI1
aI7
aI23
aI27
aI31
aI203
aI285
aI463
aI827
aI685
aI1349
aI15051
asS'd'
I1133
sa(dp2266
S'a'
I236
sS's'
I14
sS'm'
(lp2267
I1
aI1
aI1
aI5
aI29
aI5
aI107
aI195
aI31
aI425
aI19
aI2865
aI3869
aI11153
asS'd'
I1134
sa(dp2268
S'a'
I248
sS's'
I14
sS'm'
(lp2269
I1
aI1
aI7
aI5
aI7
aI47
aI1
aI73
aI307
aI347
aI393
aI2205
aI7709
aI15121
asS'd'
I1135
sa(dp2270
S'a'
I262
sS's'
I14
sS'm'
(lp2271
I1
aI1
aI1
aI13
aI15
aI61
aI25
aI131
aI113
aI369
aI1995
aI2527
aI4475
aI1745
asS'd'
I1136
sa(dp2272
S'a'
I299
sS's'
I14
sS'm'
(lp2273
I1
aI1
aI1
aI1
aI31
aI63
aI21
aI253
aI307
aI281
aI859
aI3319
aI6721
aI2891
asS'd'
I1137
sa(dp2274
S'a'
I304
sS's'
I14
sS'm'
(lp2275
I1
aI1
aI3
aI11
aI1
aI17
aI5
aI183
aI301
aI979
aI651
aI1685
aI6343
aI10067
asS'd'
I1138
sa(dp2276
S'a'
I319
sS's'
I14
sS'm'
(lp2277
I1
aI1
aI5
aI15
aI23
aI45
aI99
aI145
aI263
aI507
aI1381
aI3425
aI2215
aI1815
asS'd'
I1139
sa(dp2278
S'a'
I322
sS's'
I14
sS'm'
(lp2279
I1
aI3
aI1
aI5
aI11
aI63
aI85
aI203
aI411
aI881
aI1369
aI1237
aI4657
aI6541
asS'd'
I1140
sa(dp2280
S'a'
I334
sS's'
I14
sS'm'
(lp2281
I1
aI3
aI3
aI13
aI17
aI53
aI121
aI201
aI269
aI983
aI215
aI3187
aI7121
aI6111
asS'd'
I1141
sa(dp2282
S'a'
I355
sS's'
I14
sS'm'
(lp2283
I1
aI3
aI5
aI15
aI15
aI5
aI13
aI143
aI3
aI313
aI1677
aI1093
aI3295
aI3387
asS'd'
I1142
sa(dp2284
S'a'
I357
sS's'
I14
sS'm'
(lp2285
I1
aI1
aI3
aI13
aI3
aI23
aI73
aI17
aI257
aI965
aI239
aI1271
aI2803
aI7327
asS'd'
I1143
sa(dp2286
S'a'
I358
sS's'
I14
sS'm'
(lp2287
I1
aI3
aI5
aI13
aI9
aI57
aI115
aI37
aI41
aI467
aI135
aI1403
aI3811
aI4741
asS'd'
I1144
sa(dp2288
S'a'
I369
sS's'
I14
sS'm'
(lp2289
I1
aI3
aI7
aI15
aI9
aI33
aI39
aI203
aI351
aI367
aI1355
aI1403
aI3685
aI4757
asS'd'
I1145
sa(dp2290
S'a'
I372
sS's'
I14
sS'm'
(lp2291
I1
aI3
aI5
aI11
aI31
aI3
aI113
aI123
aI203
aI421
aI1821
aI3151
aI2375
aI4419
asS'd'
I1146
sa(dp2292
S'a'
I375
sS's'
I14
sS'm'
(lp2293
I1
aI1
aI1
aI7
aI21
aI63
aI99
aI23
aI133
aI79
aI991
aI1755
aI4989
aI4709
asS'd'
I1147
sa(dp2294
S'a'
I388
sS's'
I14
sS'm'
(lp2295
I1
aI3
aI5
aI1
aI25
aI63
aI113
aI239
aI49
aI443
aI173
aI1261
aI3201
aI10599
asS'd'
I1148
sa(dp2296
S'a'
I400
sS's'
I14
sS'm'
(lp2297
I1
aI3
aI3
aI13
aI3
aI25
aI101
aI169
aI23
aI585
aI327
aI1327
aI111
aI10059
asS'd'
I1149
sa(dp2298
S'a'
I415
sS's'
I14
sS'm'
(lp2299
I1
aI3
aI3
aI5
aI19
aI1
aI33
aI89
aI437
aI213
aI1933
aI1741
aI2603
aI5625
asS'd'
I1150
sa(dp2300
S'a'
I446
sS's'
I14
sS'm'
(lp2301
I1
aI3
aI1
aI3
aI15
aI15
aI25
aI139
aI73
aI335
aI237
aI2461
aI3101
aI14951
asS'd'
I1151
sa(dp2302
S'a'
I451
sS's'
I14
sS'm'
(lp2303
I1
aI3
aI5
aI1
aI31
aI15
aI31
aI187
aI177
aI659
aI1339
aI3767
aI4975
aI7123
asS'd'
I1152
sa(dp2304
S'a'
I458
sS's'
I14
sS'm'
(lp2305
I1
aI3
aI1
aI3
aI25
aI19
aI47
aI89
aI107
aI107
aI649
aI683
aI3123
aI11085
asS'd'
I1153
sa(dp2306
S'a'
I471
sS's'
I14
sS'm'
(lp2307
I1
aI3
aI7
aI9
aI15
aI21
aI101
aI25
aI11
aI625
aI1555
aI675
aI3893
aI5805
asS'd'
I1154
sa(dp2308
S'a'
I484
sS's'
I14
sS'm'
(lp2309
I1
aI1
aI1
aI5
aI7
aI49
aI123
aI21
aI439
aI369
aI207
aI535
aI4619
aI14665
asS'd'
I1155
sa(dp2310
S'a'
I501
sS's'
I14
sS'm'
(lp2311
I1
aI1
aI5
aI7
aI1
aI25
aI103
aI185
aI99
aI239
aI1093
aI1561
aI6177
aI4039
asS'd'
I1156
sa(dp2312
S'a'
I502
sS's'
I14
sS'm'
(lp2313
I1
aI3
aI7
aI5
aI29
aI21
aI43
aI103
aI343
aI973
aI1561
aI2975
aI7467
aI7947
asS'd'
I1157
sa(dp2314
S'a'
I517
sS's'
I14
sS'm'
(lp2315
I1
aI1
aI7
aI9
aI19
aI3
aI13
aI23
aI461
aI813
aI1191
aI985
aI559
aI3317
asS'd'
I1158
sa(dp2316
S'a'
I545
sS's'
I14
sS'm'
(lp2317
I1
aI3
aI5
aI5
aI27
aI31
aI79
aI15
aI365
aI901
aI1949
aI117
aI3619
aI13311
asS'd'
I1159
sa(dp2318
S'a'
I569
sS's'
I14
sS'm'
(lp2319
I1
aI3
aI5
aI7
aI5
aI33
aI67
aI199
aI425
aI189
aI1691
aI3099
aI815
aI1677
asS'd'
I1160
sa(dp2320
S'a'
I617
sS's'
I14
sS'm'
(lp2321
I1
aI1
aI7
aI11
aI13
aI29
aI73
aI137
aI265
aI601
aI445
aI3893
aI2511
aI8047
asS'd'
I1161
sa(dp2322
S'a'
I618
sS's'
I14
sS'm'
(lp2323
I1
aI1
aI3
aI1
aI13
aI5
aI57
aI101
aI357
aI391
aI335
aI601
aI1359
aI1065
asS'd'
I1162
sa(dp2324
S'a'
I623
sS's'
I14
sS'm'
(lp2325
I1
aI1
aI1
aI1
aI25
aI57
aI27
aI115
aI31
aI873
aI611
aI2125
aI447
aI13585
asS'd'
I1163
sa(dp2326
S'a'
I625
sS's'
I14
sS'm'
(lp2327
I1
aI3
aI3
aI13
aI27
aI17
aI73
aI11
aI359
aI33
aI1153
aI271
aI4537
aI15141
asS'd'
I1164
sa(dp2328
S'a'
I637
sS's'
I14
sS'm'
(lp2329
I1
aI3
aI7
aI3
aI11
aI63
aI103
aI61
aI59
aI629
aI1629
aI3279
aI3919
aI3177
asS'd'
I1165
sa(dp2330
S'a'
I661
sS's'
I14
sS'm'
(lp2331
I1
aI1
aI5
aI15
aI3
aI63
aI85
aI193
aI381
aI165
aI175
aI3247
aI2501
aI4209
asS'd'
I1166
sa(dp2332
S'a'
I668
sS's'
I14
sS'm'
(lp2333
I1
aI1
aI5
aI15
aI1
aI33
aI59
aI219
aI487
aI193
aI1557
aI703
aI2907
aI7953
asS'd'
I1167
sa(dp2334
S'a'
I684
sS's'
I14
sS'm'
(lp2335
I1
aI1
aI7
aI3
aI9
aI3
aI105
aI95
aI389
aI991
aI21
aI3841
aI6983
aI285
asS'd'
I1168
sa(dp2336
S'a'
I695
sS's'
I14
sS'm'
(lp2337
I1
aI1
aI1
aI1
aI1
aI31
aI25
aI137
aI117
aI67
aI1283
aI1963
aI6591
aI15541
asS'd'
I1169
sa(dp2338
S'a'
I716
sS's'
I14
sS'm'
(lp2339
I1
aI3
aI5
aI11
aI7
aI15
aI127
aI89
aI453
aI777
aI1827
aI2311
aI7661
aI11833
asS'd'
I1170
sa(dp2340
S'a'
I719
sS's'
I14
sS'm'
(lp2341
I1
aI1
aI7
aI13
aI19
aI29
aI79
aI165
aI223
aI453
aI2039
aI3961
aI6467
aI5481
asS'd'
I1171
sa(dp2342
S'a'
I722
sS's'
I14
sS'm'
(lp2343
I1
aI3
aI3
aI7
aI17
aI41
aI43
aI157
aI323
aI3
aI1001
aI2109
aI4513
aI12127
asS'd'
I1172
sa(dp2344
S'a'
I731
sS's'
I14
sS'm'
(lp2345
I1
aI1
aI5
aI9
aI31
aI57
aI3
aI217
aI113
aI271
aI1663
aI1367
aI6949
aI8165
asS'd'
I1173
sa(dp2346
S'a'
I738
sS's'
I14
sS'm'
(lp2347
I1
aI1
aI7
aI15
aI27
aI35
aI81
aI235
aI61
aI205
aI525
aI311
aI6357
aI2527
asS'd'
I1174
sa(dp2348
S'a'
I747
sS's'
I14
sS'm'
(lp2349
I1
aI3
aI1
aI9
aI19
aI29
aI71
aI207
aI321
aI1011
aI1615
aI1333
aI3459
aI6681
asS'd'
I1175
sa(dp2350
S'a'
I755
sS's'
I14
sS'm'
(lp2351
I1
aI3
aI7
aI7
aI3
aI57
aI41
aI19
aI25
aI397
aI565
aI1837
aI7625
aI11813
asS'd'
I1176
sa(dp2352
S'a'
I761
sS's'
I14
sS'm'
(lp2353
I1
aI3
aI3
aI1
aI27
aI47
aI31
aI79
aI441
aI961
aI1255
aI423
aI2405
aI913
asS'd'
I1177
sa(dp2354
S'a'
I767
sS's'
I14
sS'm'
(lp2355
I1
aI3
aI3
aI13
aI3
aI29
aI69
aI227
aI85
aI201
aI395
aI3199
aI3869
aI13099
asS'd'
I1178
sa(dp2356
S'a'
I775
sS's'
I14
sS'm'
(lp2357
I1
aI3
aI3
aI7
aI29
aI61
aI99
aI7
aI27
aI227
aI945
aI873
aI475
aI4363
asS'd'
I1179
sa(dp2358
S'a'
I782
sS's'
I14
sS'm'
(lp2359
I1
aI3
aI5
aI13
aI19
aI21
aI57
aI149
aI217
aI443
aI565
aI453
aI5487
aI10981
asS'd'
I1180
sa(dp2360
S'a'
I787
sS's'
I14
sS'm'
(lp2361
I1
aI3
aI3
aI1
aI9
aI27
aI47
aI191
aI35
aI395
aI1429
aI4079
aI6871
aI8013
asS'd'
I1181
sa(dp2362
S'a'
I794
sS's'
I14
sS'm'
(lp2363
I1
aI3
aI5
aI15
aI5
aI43
aI9
aI79
aI279
aI563
aI1125
aI985
aI8117
aI4099
asS'd'
I1182
sa(dp2364
S'a'
I803
sS's'
I14
sS'm'
(lp2365
I1
aI3
aI5
aI1
aI13
aI41
aI21
aI117
aI287
aI667
aI701
aI1483
aI8167
aI13283
asS'd'
I1183
sa(dp2366
S'a'
I812
sS's'
I14
sS'm'
(lp2367
I1
aI3
aI1
aI3
aI15
aI15
aI59
aI5
aI383
aI509
aI1657
aI3977
aI7697
aI10941
asS'd'
I1184
sa(dp2368
S'a'
I817
sS's'
I14
sS'm'
(lp2369
I1
aI3
aI1
aI1
aI17
aI29
aI19
aI23
aI377
aI45
aI981
aI1631
aI3557
aI6749
asS'd'
I1185
sa(dp2370
S'a'
I824
sS's'
I14
sS'm'
(lp2371
I1
aI3
aI3
aI9
aI9
aI51
aI9
aI193
aI345
aI361
aI1679
aI3333
aI713
aI5387
asS'd'
I1186
sa(dp2372
S'a'
I829
sS's'
I14
sS'm'
(lp2373
I1
aI3
aI5
aI5
aI17
aI45
aI97
aI17
aI385
aI349
aI105
aI2245
aI7295
aI14393
asS'd'
I1187
sa(dp2374
S'a'
I850
sS's'
I14
sS'm'
(lp2375
I1
aI3
aI7
aI3
aI19
aI51
aI35
aI99
aI79
aI301
aI1563
aI399
aI5879
aI14675
asS'd'
I1188
sa(dp2376
S'a'
I866
sS's'
I14
sS'm'
(lp2377
I1
aI1
aI7
aI15
aI13
aI53
aI55
aI203
aI417
aI161
aI2033
aI1845
aI6763
aI3585
asS'd'
I1189
sa(dp2378
S'a'
I871
sS's'
I14
sS'm'
(lp2379
I1
aI1
aI3
aI3
aI7
aI23
aI11
aI43
aI241
aI309
aI1453
aI3147
aI2619
aI3163
asS'd'
I1190
sa(dp2380
S'a'
I877
sS's'
I14
sS'm'
(lp2381
I1
aI1
aI1
aI11
aI17
aI1
aI17
aI137
aI443
aI465
aI993
aI3217
aI7879
aI14607
asS'd'
I1191
sa(dp2382
S'a'
I920
sS's'
I14
sS'm'
(lp2383
I1
aI1
aI7
aI13
aI29
aI49
aI71
aI217
aI291
aI793
aI135
aI21
aI2503
aI11091
asS'd'
I1192
sa(dp2384
S'a'
I935
sS's'
I14
sS'm'
(lp2385
I1
aI3
aI1
aI11
aI31
aI51
aI121
aI227
aI377
aI157
aI1457
aI1317
aI5625
aI6217
asS'd'
I1193
sa(dp2386
S'a'
I959
sS's'
I14
sS'm'
(lp2387
I1
aI1
aI3
aI7
aI23
aI61
aI47
aI93
aI79
aI617
aI1805
aI2403
aI5513
aI16335
asS'd'
I1194
sa(dp2388
S'a'
I979
sS's'
I14
sS'm'
(lp2389
I1
aI3
aI5
aI11
aI23
aI25
aI41
aI11
aI495
aI587
aI1223
aI3107
aI1469
aI15223
asS'd'
I1195
sa(dp2390
S'a'
I992
sS's'
I14
sS'm'
(lp2391
I1
aI3
aI7
aI7
aI9
aI1
aI1
aI49
aI23
aI723
aI1761
aI3717
aI7375
aI10875
asS'd'
I1196
sa(dp2392
S'a'
I1010
sS's'
I14
sS'm'
(lp2393
I1
aI3
aI3
aI11
aI25
aI37
aI57
aI63
aI309
aI603
aI183
aI285
aI1663
aI5627
asS'd'
I1197
sa(dp2394
S'a'
I1012
sS's'
I14
sS'm'
(lp2395
I1
aI3
aI7
aI11
aI19
aI25
aI25
aI201
aI391
aI257
aI529
aI1645
aI1
aI15111
asS'd'
I1198
sa(dp2396
S'a'
I1015
sS's'
I14
sS'm'
(lp2397
I1
aI3
aI3
aI9
aI11
aI43
aI91
aI65
aI5
aI959
aI301
aI1015
aI6343
aI3453
asS'd'
I1199
sa(dp2398
S'a'
I1033
sS's'
I14
sS'm'
(lp2399
I1
aI3
aI3
aI11
aI17
aI17
aI103
aI37
aI77
aI973
aI575
aI439
aI49
aI3639
asS'd'
I1200
sa(dp2400
S'a'
I1036
sS's'
I14
sS'm'
(lp2401
I1
aI1
aI5
aI7
aI1
aI15
aI107
aI237
aI231
aI967
aI923
aI1101
aI6715
aI1713
asS'd'
I1201
sa(dp2402
S'a'
I1053
sS's'
I14
sS'm'
(lp2403
I1
aI3
aI1
aI15
aI9
aI33
aI29
aI211
aI245
aI601
aI1783
aI887
aI1209
aI11785
asS'd'
I1202
sa(dp2404
S'a'
I1057
sS's'
I14
sS'm'
(lp2405
I1
aI3
aI3
aI7
aI21
aI43
aI27
aI89
aI27
aI141
aI865
aI367
aI1379
aI4063
asS'd'
I1203
sa(dp2406
S'a'
I1069
sS's'
I14
sS'm'
(lp2407
I1
aI3
aI7
aI7
aI15
aI17
aI15
aI15
aI131
aI649
aI1955
aI3289
aI3983
aI10689
asS'd'
I1204
sa(dp2408
S'a'
I1072
sS's'
I14
sS'm'
(lp2409
I1
aI3
aI1
aI5
aI17
aI7
aI125
aI69
aI359
aI981
aI1345
aI933
aI5281
aI7113
asS'd'
I1205
sa(dp2410
S'a'
I1075
sS's'
I14
sS'm'
(lp2411
I1
aI1
aI5
aI9
aI17
aI7
aI41
aI207
aI497
aI1015
aI493
aI891
aI3563
aI3541
asS'd'
I1206
sa(dp2412
S'a'
I1087
sS's'
I14
sS'm'
(lp2413
I1
aI3
aI5
aI11
aI27
aI3
aI47
aI31
aI303
aI1007
aI2047
aI2203
aI6257
aI8369
asS'd'
I1207
sa(dp2414
S'a'
I1089
sS's'
I14
sS'm'
(lp2415
I1
aI1
aI1
aI15
aI25
aI15
aI89
aI51
aI217
aI357
aI1133
aI1917
aI213
aI3365
asS'd'
I1208
sa(dp2416
S'a'
I1137
sS's'
I14
sS'm'
(lp2417
I1
aI1
aI5
aI13
aI29
aI23
aI123
aI207
aI429
aI805
aI819
aI2357
aI6313
aI11019
asS'd'
I1209
sa(dp2418
S'a'
I1166
sS's'
I14
sS'm'
(lp2419
I1
aI1
aI3
aI7
aI19
aI15
aI41
aI73
aI279
aI11
aI1089
aI3107
aI7737
aI15953
asS'd'
I1210
sa(dp2420
S'a'
I1174
sS's'
I14
sS'm'
(lp2421
I1
aI3
aI5
aI7
aI7
aI15
aI41
aI73
aI493
aI457
aI1731
aI1139
aI2513
aI12373
asS'd'
I1211
sa(dp2422
S'a'
I1180
sS's'
I14
sS'm'
(lp2423
I1
aI3
aI5
aI9
aI17
aI5
aI55
aI155
aI173
aI1005
aI529
aI3175
aI7667
aI4747
asS'd'
I1212
sa(dp2424
S'a'
I1204
sS's'
I14
sS'm'
(lp2425
I1
aI1
aI7
aI7
aI5
aI21
aI105
aI31
aI205
aI847
aI1033
aI3167
aI2347
aI8499
asS'd'
I1213
sa(dp2426
S'a'
I1211
sS's'
I14
sS'm'
(lp2427
I1
aI3
aI5
aI3
aI11
aI17
aI59
aI189
aI179
aI1007
aI33
aI3287
aI4813
aI8177
asS'd'
I1214
sa(dp2428
S'a'
I1219
sS's'
I14
sS'm'
(lp2429
I1
aI3
aI3
aI13
aI27
aI47
aI47
aI171
aI413
aI875
aI1081
aI1259
aI7139
aI8645
asS'd'
I1215
sa(dp2430
S'a'
I1236
sS's'
I14
sS'm'
(lp2431
I1
aI3
aI5
aI7
aI25
aI21
aI51
aI29
aI361
aI293
aI51
aI1119
aI1453
aI5283
asS'd'
I1216
sa(dp2432
S'a'
I1255
sS's'
I14
sS'm'
(lp2433
I1
aI3
aI7
aI7
aI29
aI55
aI103
aI199
aI511
aI341
aI1957
aI3987
aI2855
aI1279
asS'd'
I1217
sa(dp2434
S'a'
I1264
sS's'
I14
sS'm'
(lp2435
I1
aI1
aI1
aI9
aI23
aI51
aI61
aI63
aI391
aI37
aI55
aI3771
aI6517
aI15913
asS'd'
I1218
sa(dp2436
S'a'
I1306
sS's'
I14
sS'm'
(lp2437
I1
aI1
aI1
aI9
aI3
aI19
aI13
aI147
aI453
aI855
aI1321
aI189
aI5043
aI11215
asS'd'
I1219
sa(dp2438
S'a'
I1330
sS's'
I14
sS'm'
(lp2439
I1
aI3
aI3
aI13
aI23
aI3
aI87
aI155
aI401
aI981
aI607
aI3413
aI995
aI6473
asS'd'
I1220
sa(dp2440
S'a'
I1341
sS's'
I14
sS'm'
(lp2441
I1
aI3
aI1
aI9
aI29
aI47
aI95
aI123
aI421
aI353
aI1867
aI2609
aI2569
aI14083
asS'd'
I1221
sa(dp2442
S'a'
I1344
sS's'
I14
sS'm'
(lp2443
I1
aI1
aI5
aI13
aI25
aI39
aI29
aI111
aI125
aI545
aI1493
aI2371
aI6361
aI6307
asS'd'
I1222
sa(dp2444
S'a'
I1347
sS's'
I14
sS'm'
(lp2445
I1
aI3
aI3
aI11
aI13
aI31
aI87
aI75
aI27
aI393
aI921
aI3655
aI3343
aI16349
asS'd'
I1223
sa(dp2446
S'a'
I1349
sS's'
I14
sS'm'
(lp2447
I1
aI1
aI5
aI9
aI19
aI19
aI7
aI129
aI223
aI715
aI433
aI1627
aI4463
aI2951
asS'd'
I1224
sa(dp2448
S'a'
I1361
sS's'
I14
sS'm'
(lp2449
I1
aI1
aI7
aI1
aI31
aI13
aI49
aI33
aI89
aI43
aI1529
aI725
aI3809
aI3427
asS'd'
I1225
sa(dp2450
S'a'
I1380
sS's'
I14
sS'm'
(lp2451
I1
aI1
aI7
aI3
aI1
aI27
aI45
aI9
aI309
aI875
aI659
aI2661
aI553
aI7069
asS'd'
I1226
sa(dp2452
S'a'
I1390
sS's'
I14
sS'm'
(lp2453
I1
aI1
aI7
aI15
aI13
aI37
aI61
aI19
aI125
aI683
aI1227
aI2255
aI1455
aI9339
asS'd'
I1227
sa(dp2454
S'a'
I1404
sS's'
I14
sS'm'
(lp2455
I1
aI3
aI5
aI7
aI19
aI7
aI71
aI21
aI465
aI645
aI1885
aI873
aI7405
aI1913
asS'd'
I1228
sa(dp2456
S'a'
I1435
sS's'
I14
sS'm'
(lp2457
I1
aI3
aI1
aI11
aI11
aI35
aI79
aI61
aI79
aI57
aI1603
aI3719
aI6323
aI16371
asS'd'
I1229
sa(dp2458
S'a'
I1444
sS's'
I14
sS'm'
(lp2459
I1
aI1
aI7
aI1
aI29
aI57
aI85
aI21
aI205
aI37
aI2045
aI683
aI4901
aI8223
asS'd'
I1230
sa(dp2460
S'a'
I1453
sS's'
I14
sS'm'
(lp2461
I1
aI1
aI5
aI13
aI31
aI31
aI65
aI131
aI259
aI535
aI967
aI3943
aI2605
aI2089
asS'd'
I1231
sa(dp2462
S'a'
I1461
sS's'
I14
sS'm'
(lp2463
I1
aI1
aI7
aI9
aI27
aI61
aI39
aI243
aI207
aI41
aI1909
aI3279
aI1331
aI4635
asS'd'
I1232
sa(dp2464
S'a'
I1462
sS's'
I14
sS'm'
(lp2465
I1
aI3
aI3
aI5
aI11
aI63
aI105
aI19
aI169
aI95
aI773
aI3175
aI1869
aI1797
asS'd'
I1233
sa(dp2466
S'a'
I1465
sS's'
I14
sS'm'
(lp2467
I1
aI3
aI3
aI15
aI13
aI33
aI107
aI197
aI153
aI795
aI1477
aI105
aI4965
aI991
asS'd'
I1234
sa(dp2468
S'a'
I1468
sS's'
I14
sS'm'
(lp2469
I1
aI3
aI7
aI11
aI11
aI37
aI23
aI149
aI197
aI3
aI1035
aI3857
aI553
aI1059
asS'd'
I1235
sa(dp2470
S'a'
I1474
sS's'
I14
sS'm'
(lp2471
I1
aI3
aI1
aI3
aI17
aI29
aI89
aI189
aI193
aI59
aI1477
aI3517
aI2565
aI7739
asS'd'
I1236
sa(dp2472
S'a'
I1483
sS's'
I14
sS'm'
(lp2473
I1
aI1
aI1
aI9
aI23
aI3
aI25
aI163
aI469
aI305
aI1791
aI3393
aI6141
aI8119
asS'd'
I1237
sa(dp2474
S'a'
I1488
sS's'
I14
sS'm'
(lp2475
I1
aI3
aI5
aI7
aI7
aI41
aI19
aI101
aI179
aI487
aI1071
aI2761
aI8043
aI5103
asS'd'
I1238
sa(dp2476
S'a'
I1493
sS's'
I14
sS'm'
(lp2477
I1
aI1
aI7
aI9
aI1
aI21
aI101
aI103
aI349
aI85
aI1841
aI1033
aI4473
aI3563
asS'd'
I1239
sa(dp2478
S'a'
I1500
sS's'
I14
sS'm'
(lp2479
I1
aI1
aI3
aI13
aI23
aI61
aI39
aI27
aI479
aI13
aI45
aI1371
aI7897
aI10637
asS'd'
I1240
sa(dp2480
S'a'
I1509
sS's'
I14
sS'm'
(lp2481
I1
aI1
aI5
aI9
aI17
aI61
aI71
aI55
aI355
aI99
aI1695
aI3053
aI839
aI959
asS'd'
I1241
sa(dp2482
S'a'
I1510
sS's'
I14
sS'm'
(lp2483
I1
aI1
aI3
aI1
aI7
aI27
aI87
aI221
aI327
aI241
aI461
aI3177
aI5933
aI8299
asS'd'
I1242
sa(dp2484
S'a'
I1514
sS's'
I14
sS'm'
(lp2485
I1
aI3
aI7
aI9
aI5
aI41
aI111
aI245
aI447
aI263
aI1363
aI1767
aI6331
aI3355
asS'd'
I1243
sa(dp2486
S'a'
I1519
sS's'
I14
sS'm'
(lp2487
I1
aI3
aI3
aI13
aI15
aI11
aI15
aI169
aI429
aI149
aI1965
aI2477
aI7733
aI2499
asS'd'
I1244
sa(dp2488
S'a'
I1528
sS's'
I14
sS'm'
(lp2489
I1
aI1
aI5
aI15
aI15
aI47
aI25
aI33
aI469
aI701
aI773
aI2747
aI1533
aI14633
asS'd'
I1245
sa(dp2490
S'a'
I1533
sS's'
I14
sS'm'
(lp2491
I1
aI3
aI1
aI5
aI19
aI57
aI37
aI75
aI423
aI11
aI685
aI2487
aI1779
aI8797
asS'd'
I1246
sa(dp2492
S'a'
I1540
sS's'
I14
sS'm'
(lp2493
I1
aI3
aI1
aI5
aI19
aI41
aI67
aI99
aI333
aI991
aI953
aI3221
aI939
aI4197
asS'd'
I1247
sa(dp2494
S'a'
I1550
sS's'
I14
sS'm'
(lp2495
I1
aI3
aI1
aI15
aI11
aI39
aI25
aI1
aI159
aI679
aI465
aI1611
aI5799
aI2537
asS'd'
I1248
sa(dp2496
S'a'
I1567
sS's'
I14
sS'm'
(lp2497
I1
aI1
aI5
aI11
aI5
aI37
aI37
aI7
aI101
aI703
aI235
aI23
aI2209
aI12799
asS'd'
I1249
sa(dp2498
S'a'
I1571
sS's'
I14
sS'm'
(lp2499
I1
aI1
aI7
aI3
aI11
aI23
aI71
aI215
aI45
aI269
aI1539
aI3625
aI5773
aI6889
asS'd'
I1250
sa(dp2500
S'a'
I1573
sS's'
I14
sS'm'
(lp2501
I1
aI3
aI5
aI15
aI27
aI33
aI105
aI109
aI205
aI653
aI821
aI435
aI1087
aI2495
asS'd'
I1251
sa(dp2502
S'a'
I1578
sS's'
I14
sS'm'
(lp2503
I1
aI1
aI3
aI5
aI11
aI39
aI53
aI213
aI41
aI385
aI1425
aI25
aI5553
aI12523
asS'd'
I1252
sa(dp2504
S'a'
I1598
sS's'
I14
sS'm'
(lp2505
I1
aI3
aI5
aI15
aI29
aI49
aI13
aI253
aI505
aI407
aI985
aI2569
aI6727
aI4761
asS'd'
I1253
sa(dp2506
S'a'
I1606
sS's'
I14
sS'm'
(lp2507
I1
aI1
aI1
aI3
aI29
aI17
aI69
aI47
aI25
aI819
aI1145
aI2479
aI1183
aI3343
asS'd'
I1254
sa(dp2508
S'a'
I1618
sS's'
I14
sS'm'
(lp2509
I1
aI3
aI1
aI15
aI25
aI61
aI43
aI55
aI279
aI579
aI361
aI355
aI6101
aI3143
asS'd'
I1255
sa(dp2510
S'a'
I1630
sS's'
I14
sS'm'
(lp2511
I1
aI3
aI5
aI11
aI3
aI59
aI125
aI101
aI451
aI495
aI1711
aI3443
aI3625
aI15579
asS'd'
I1256
sa(dp2512
S'a'
I1634
sS's'
I14
sS'm'
(lp2513
I1
aI3
aI1
aI11
aI25
aI61
aI49
aI219
aI23
aI795
aI481
aI3609
aI3691
aI15419
asS'd'
I1257
sa(dp2514
S'a'
I1640
sS's'
I14
sS'm'
(lp2515
I1
aI3
aI7
aI5
aI9
aI59
aI49
aI233
aI345
aI143
aI181
aI3587
aI3041
aI1219
asS'd'
I1258
sa(dp2516
S'a'
I1643
sS's'
I14
sS'm'
(lp2517
I1
aI3
aI7
aI13
aI9
aI31
aI39
aI137
aI261
aI919
aI1367
aI3145
aI4659
aI5875
asS'd'
I1259
sa(dp2518
S'a'
I1654
sS's'
I14
sS'm'
(lp2519
I1
aI1
aI3
aI3
aI27
aI43
aI95
aI65
aI301
aI915
aI31
aI451
aI7743
aI7277
asS'd'
I1260
sa(dp2520
S'a'
I1679
sS's'
I14
sS'm'
(lp2521
I1
aI3
aI1
aI5
aI23
aI37
aI53
aI31
aI203
aI453
aI71
aI1585
aI6011
aI16369
asS'd'
I1261
sa(dp2522
S'a'
I1688
sS's'
I14
sS'm'
(lp2523
I1
aI1
aI5
aI1
aI15
aI47
aI91
aI227
aI297
aI45
aI1415
aI3647
aI7811
aI14015
asS'd'
I1262
sa(dp2524
S'a'
I1698
sS's'
I14
sS'm'
(lp2525
I1
aI1
aI1
aI1
aI29
aI27
aI93
aI121
aI169
aI69
aI1361
aI2907
aI1867
aI7017
asS'd'
I1263
sa(dp2526
S'a'
I1703
sS's'
I14
sS'm'
(lp2527
I1
aI3
aI1
aI7
aI23
aI53
aI77
aI41
aI25
aI873
aI1333
aI3889
aI3239
aI1771
asS'd'
I1264
sa(dp2528
S'a'
I1704
sS's'
I14
sS'm'
(lp2529
I1
aI1
aI1
aI7
aI31
aI27
aI87
aI81
aI167
aI343
aI1981
aI2499
aI7749
aI15747
asS'd'
I1265
sa(dp2530
S'a'
I1722
sS's'
I14
sS'm'
(lp2531
I1
aI3
aI5
aI13
aI1
aI17
aI97
aI37
aI81
aI645
aI1167
aI3547
aI7769
aI10731
asS'd'
I1266
sa(dp2532
S'a'
I1735
sS's'
I14
sS'm'
(lp2533
I1
aI1
aI7
aI5
aI9
aI17
aI31
aI55
aI151
aI463
aI1041
aI2303
aI4015
aI3737
asS'd'
I1267
sa(dp2534
S'a'
I1750
sS's'
I14
sS'm'
(lp2535
I1
aI1
aI3
aI11
aI31
aI9
aI81
aI213
aI95
aI215
aI2031
aI2129
aI4299
aI3021
asS'd'
I1268
sa(dp2536
S'a'
I1753
sS's'
I14
sS'm'
(lp2537
I1
aI1
aI1
aI3
aI25
aI25
aI115
aI229
aI101
aI441
aI783
aI1729
aI7905
aI2375
asS'd'
I1269
sa(dp2538
S'a'
I1760
sS's'
I14
sS'm'
(lp2539
I1
aI1
aI5
aI9
aI3
aI19
aI73
aI35
aI379
aI493
aI1333
aI1647
aI13
aI197
asS'd'
I1270
sa(dp2540
S'a'
I1789
sS's'
I14
sS'm'
(lp2541
I1
aI1
aI7
aI9
aI3
aI55
aI99
aI43
aI281
aI9
aI73
aI2477
aI8183
aI11055
asS'd'
I1271
sa(dp2542
S'a'
I1792
sS's'
I14
sS'm'
(lp2543
I1
aI3
aI7
aI13
aI25
aI19
aI27
aI195
aI469
aI175
aI355
aI1861
aI7255
aI15377
asS'd'
I1272
sa(dp2544
S'a'
I1802
sS's'
I14
sS'm'
(lp2545
I1
aI1
aI3
aI11
aI15
aI19
aI115
aI31
aI413
aI835
aI697
aI879
aI6515
aI13465
asS'd'
I1273
sa(dp2546
S'a'
I1819
sS's'
I14
sS'm'
(lp2547
I1
aI3
aI3
aI15
aI3
aI61
aI105
aI201
aI151
aI739
aI49
aI3963
aI2573
aI3303
asS'd'
I1274
sa(dp2548
S'a'
I1825
sS's'
I14
sS'm'
(lp2549
I1
aI3
aI5
aI7
aI23
aI5
aI11
aI215
aI19
aI591
aI509
aI2887
aI1631
aI4391
asS'd'
I1275
sa(dp2550
S'a'
I1828
sS's'
I14
sS'm'
(lp2551
I1
aI3
aI3
aI3
aI25
aI1
aI109
aI5
aI363
aI545
aI1745
aI503
aI827
aI4677
asS'd'
I1276
sa(dp2552
S'a'
I1832
sS's'
I14
sS'm'
(lp2553
I1
aI1
aI3
aI15
aI1
aI45
aI121
aI141
aI497
aI745
aI1825
aI2041
aI2561
aI8153
asS'd'
I1277
sa(dp2554
S'a'
I1845
sS's'
I14
sS'm'
(lp2555
I1
aI3
aI1
aI11
aI29
aI7
aI71
aI241
aI7
aI39
aI1379
aI2479
aI7483
aI7195
asS'd'
I1278
sa(dp2556
S'a'
I1846
sS's'
I14
sS'm'
(lp2557
I1
aI1
aI7
aI11
aI3
aI27
aI39
aI97
aI339
aI217
aI1409
aI1569
aI4761
aI1567
asS'd'
I1279
sa(dp2558
S'a'
I1857
sS's'
I14
sS'm'
(lp2559
I1
aI1
aI5
aI15
aI11
aI53
aI87
aI213
aI297
aI923
aI393
aI717
aI3297
aI16123
asS'd'
I1280
sa(dp2560
S'a'
I1869
sS's'
I14
sS'm'
(lp2561
I1
aI1
aI1
aI11
aI27
aI41
aI121
aI49
aI225
aI379
aI1305
aI319
aI2461
aI5445
asS'd'
I1281
sa(dp2562
S'a'
I1897
sS's'
I14
sS'm'
(lp2563
I1
aI1
aI5
aI5
aI25
aI3
aI121
aI23
aI47
aI843
aI1679
aI1427
aI6393
aI4199
asS'd'
I1282
sa(dp2564
S'a'
I1906
sS's'
I14
sS'm'
(lp2565
I1
aI1
aI5
aI13
aI17
aI3
aI17
aI25
aI161
aI487
aI121
aI361
aI1375
aI10745
asS'd'
I1283
sa(dp2566
S'a'
I1908
sS's'
I14
sS'm'
(lp2567
I1
aI1
aI7
aI3
aI3
aI37
aI7
aI245
aI107
aI107
aI745
aI2415
aI2131
aI11419
asS'd'
I1284
sa(dp2568
S'a'
I1911
sS's'
I14
sS'm'
(lp2569
I1
aI1
aI5
aI3
aI3
aI23
aI67
aI91
aI281
aI387
aI465
aI905
aI883
aI9775
asS'd'
I1285
sa(dp2570
S'a'
I1934
sS's'
I14
sS'm'
(lp2571
I1
aI3
aI7
aI15
aI25
aI55
aI123
aI49
aI23
aI983
aI1903
aI2589
aI2073
aI7823
asS'd'
I1286
sa(dp2572
S'a'
I1962
sS's'
I14
sS'm'
(lp2573
I1
aI1
aI5
aI11
aI25
aI17
aI63
aI229
aI267
aI175
aI1759
aI1947
aI479
aI11089
asS'd'
I1287
sa(dp2574
S'a'
I1967
sS's'
I14
sS'm'
(lp2575
I1
aI3
aI7
aI3
aI11
aI37
aI83
aI95
aI415
aI1003
aI1175
aI2361
aI2117
aI9809
asS'd'
I1288
sa(dp2576
S'a'
I1972
sS's'
I14
sS'm'
(lp2577
I1
aI3
aI1
aI9
aI5
aI39
aI51
aI129
aI249
aI161
aI1981
aI2755
aI8057
aI13641
asS'd'
I1289
sa(dp2578
S'a'
I1975
sS's'
I14
sS'm'
(lp2579
I1
aI1
aI7
aI1
aI15
aI47
aI9
aI197
aI199
aI549
aI1091
aI2853
aI2331
aI4535
asS'd'
I1290
sa(dp2580
S'a'
I1999
sS's'
I14
sS'm'
(lp2581
I1
aI3
aI3
aI13
aI15
aI21
aI23
aI111
aI463
aI719
aI1667
aI377
aI5039
aI10723
asS'd'
I1291
sa(dp2582
S'a'
I2004
sS's'
I14
sS'm'
(lp2583
I1
aI1
aI3
aI7
aI23
aI47
aI39
aI47
aI307
aI949
aI1651
aI2525
aI5835
aI1425
asS'd'
I1292
sa(dp2584
S'a'
I2011
sS's'
I14
sS'm'
(lp2585
I1
aI3
aI3
aI9
aI23
aI47
aI111
aI39
aI251
aI1001
aI179
aI3985
aI535
aI15435
asS'd'
I1293
sa(dp2586
S'a'
I2013
sS's'
I14
sS'm'
(lp2587
I1
aI1
aI3
aI13
aI5
aI45
aI51
aI123
aI205
aI651
aI1583
aI1691
aI1631
aI11975
asS'd'
I1294
sa(dp2588
S'a'
I2037
sS's'
I14
sS'm'
(lp2589
I1
aI1
aI7
aI9
aI1
aI29
aI59
aI27
aI389
aI497
aI1459
aI1633
aI521
aI14037
asS'd'
I1295
sa(dp2590
S'a'
I2051
sS's'
I14
sS'm'
(lp2591
I1
aI1
aI3
aI3
aI3
aI23
aI35
aI247
aI371
aI729
aI931
aI681
aI1777
aI8353
asS'd'
I1296
sa(dp2592
S'a'
I2063
sS's'
I14
sS'm'
(lp2593
I1
aI3
aI3
aI1
aI19
aI15
aI17
aI191
aI495
aI643
aI319
aI37
aI5691
aI7049
asS'd'
I1297
sa(dp2594
S'a'
I2066
sS's'
I14
sS'm'
(lp2595
I1
aI3
aI5
aI11
aI5
aI31
aI123
aI243
aI335
aI573
aI113
aI209
aI4825
aI7783
asS'd'
I1298
sa(dp2596
S'a'
I2094
sS's'
I14
sS'm'
(lp2597
I1
aI3
aI7
aI7
aI29
aI19
aI25
aI191
aI89
aI515
aI55
aI3013
aI4523
aI12913
asS'd'
I1299
sa(dp2598
S'a'
I2128
sS's'
I14
sS'm'
(lp2599
I1
aI1
aI3
aI3
aI15
aI3
aI35
aI37
aI339
aI7
aI697
aI359
aI4553
aI1431
asS'd'
I1300
sa(dp2600
S'a'
I2154
sS's'
I14
sS'm'
(lp2601
I1
aI3
aI1
aI1
aI9
aI15
aI33
aI77
aI161
aI13
aI255
aI1187
aI6587
aI11715
asS'd'
I1301
sa(dp2602
S'a'
I2164
sS's'
I14
sS'm'
(lp2603
I1
aI3
aI7
aI7
aI25
aI57
aI61
aI171
aI231
aI43
aI1219
aI903
aI5623
aI4781
asS'd'
I1302
sa(dp2604
S'a'
I2198
sS's'
I14
sS'm'
(lp2605
I1
aI1
aI5
aI15
aI29
aI47
aI117
aI23
aI213
aI907
aI1423
aI369
aI4529
aI9651
asS'd'
I1303
sa(dp2606
S'a'
I2217
sS's'
I14
sS'm'
(lp2607
I1
aI1
aI5
aI7
aI15
aI55
aI105
aI249
aI401
aI37
aI1885
aI3779
aI3441
aI9057
asS'd'
I1304
sa(dp2608
S'a'
I2220
sS's'
I14
sS'm'
(lp2609
I1
aI1
aI5
aI3
aI3
aI27
aI49
aI89
aI335
aI561
aI1235
aI3251
aI2731
aI12711
asS'd'
I1305
sa(dp2610
S'a'
I2223
sS's'
I14
sS'm'
(lp2611
I1
aI1
aI1
aI15
aI29
aI49
aI37
aI173
aI25
aI743
aI1321
aI821
aI5899
aI9213
asS'd'
I1306
sa(dp2612
S'a'
I2238
sS's'
I14
sS'm'
(lp2613
I1
aI1
aI7
aI3
aI1
aI41
aI61
aI209
aI275
aI925
aI521
aI3029
aI1569
aI9277
asS'd'
I1307
sa(dp2614
S'a'
I2245
sS's'
I14
sS'm'
(lp2615
I1
aI3
aI5
aI13
aI17
aI1
aI11
aI171
aI441
aI119
aI1589
aI299
aI157
aI11439
asS'd'
I1308
sa(dp2616
S'a'
I2252
sS's'
I14
sS'm'
(lp2617
I1
aI1
aI5
aI9
aI13
aI33
aI27
aI77
aI363
aI939
aI1103
aI2135
aI1759
aI5429
asS'd'
I1309
sa(dp2618
S'a'
I2258
sS's'
I14
sS'm'
(lp2619
I1
aI3
aI7
aI1
aI17
aI39
aI49
aI201
aI49
aI803
aI2003
aI1193
aI7415
aI13847
asS'd'
I1310
sa(dp2620
S'a'
I2264
sS's'
I14
sS'm'
(lp2621
I1
aI1
aI5
aI5
aI17
aI49
aI39
aI19
aI311
aI801
aI1441
aI3263
aI7973
aI14181
asS'd'
I1311
sa(dp2622
S'a'
I2280
sS's'
I14
sS'm'
(lp2623
I1
aI1
aI3
aI9
aI9
aI27
aI59
aI89
aI81
aI473
aI1369
aI3121
aI7929
aI10905
asS'd'
I1312
sa(dp2624
S'a'
I2285
sS's'
I14
sS'm'
(lp2625
I1
aI3
aI3
aI5
aI17
aI35
aI35
aI239
aI379
aI431
aI501
aI3561
aI2059
aI9679
asS'd'
I1313
sa(dp2626
S'a'
I2293
sS's'
I14
sS'm'
(lp2627
I1
aI3
aI5
aI15
aI25
aI29
aI113
aI179
aI269
aI891
aI301
aI2017
aI7513
aI9379
asS'd'
I1314
sa(dp2628
S'a'
I2294
sS's'
I14
sS'm'
(lp2629
I1
aI3
aI1
aI11
aI17
aI35
aI49
aI149
aI135
aI661
aI1691
aI3169
aI3765
aI9003
asS'd'
I1315
sa(dp2630
S'a'
I2298
sS's'
I14
sS'm'
(lp2631
I1
aI3
aI7
aI15
aI5
aI21
aI53
aI241
aI475
aI271
aI683
aI2351
aI2181
aI6333
asS'd'
I1316
sa(dp2632
S'a'
I2303
sS's'
I14
sS'm'
(lp2633
I1
aI1
aI7
aI13
aI25
aI33
aI71
aI153
aI221
aI507
aI2017
aI2401
aI7545
aI8489
asS'd'
I1317
sa(dp2634
S'a'
I2306
sS's'
I14
sS'm'
(lp2635
I1
aI1
aI7
aI5
aI1
aI49
aI87
aI1
aI179
aI331
aI1597
aI3713
aI809
aI11109
asS'd'
I1318
sa(dp2636
S'a'
I2311
sS's'
I14
sS'm'
(lp2637
I1
aI3
aI1
aI5
aI5
aI61
aI93
aI39
aI479
aI977
aI1099
aI1291
aI7049
aI2797
asS'd'
I1319
sa(dp2638
S'a'
I2326
sS's'
I14
sS'm'
(lp2639
I1
aI3
aI1
aI13
aI19
aI41
aI57
aI77
aI5
aI117
aI125
aI115
aI3969
aI1345
asS'd'
I1320
sa(dp2640
S'a'
I2354
sS's'
I14
sS'm'
(lp2641
I1
aI1
aI1
aI9
aI15
aI9
aI57
aI7
aI219
aI41
aI767
aI23
aI5771
aI14175
asS'd'
I1321
sa(dp2642
S'a'
I2373
sS's'
I14
sS'm'
(lp2643
I1
aI3
aI7
aI9
aI17
aI61
aI1
aI59
aI227
aI349
aI63
aI189
aI3871
aI7919
asS'd'
I1322
sa(dp2644
S'a'
I2380
sS's'
I14
sS'm'
(lp2645
I1
aI3
aI5
aI5
aI9
aI29
aI33
aI203
aI413
aI701
aI1129
aI2103
aI1889
aI8377
asS'd'
I1323
sa(dp2646
S'a'
I2385
sS's'
I14
sS'm'
(lp2647
I1
aI1
aI3
aI1
aI9
aI17
aI69
aI115
aI123
aI1001
aI1
aI2893
aI3957
aI8593
asS'd'
I1324
sa(dp2648
S'a'
I2392
sS's'
I14
sS'm'
(lp2649
I1
aI1
aI3
aI1
aI31
aI41
aI83
aI91
aI113
aI195
aI1121
aI2665
aI6815
aI1189
asS'd'
I1325
sa(dp2650
S'a'
I2401
sS's'
I14
sS'm'
(lp2651
I1
aI1
aI1
aI13
aI3
aI59
aI13
aI123
aI95
aI103
aI1689
aI2809
aI5049
aI4055
asS'd'
I1326
sa(dp2652
S'a'
I2402
sS's'
I14
sS'm'
(lp2653
I1
aI1
aI1
aI15
aI21
aI41
aI11
aI167
aI375
aI589
aI207
aI1631
aI1597
aI8091
asS'd'
I1327
sa(dp2654
S'a'
I2408
sS's'
I14
sS'm'
(lp2655
I1
aI3
aI5
aI5
aI1
aI33
aI57
aI89
aI157
aI921
aI1353
aI2777
aI461
aI14567
asS'd'
I1328
sa(dp2656
S'a'
I2419
sS's'
I14
sS'm'
(lp2657
I1
aI1
aI5
aI1
aI25
aI5
aI51
aI247
aI1
aI577
aI463
aI3741
aI303
aI16059
asS'd'
I1329
sa(dp2658
S'a'
I2450
sS's'
I14
sS'm'
(lp2659
I1
aI1
aI7
aI5
aI13
aI7
aI17
aI87
aI51
aI987
aI835
aI93
aI5203
aI3973
asS'd'
I1330
sa(dp2660
S'a'
I2452
sS's'
I14
sS'm'
(lp2661
I1
aI1
aI7
aI7
aI3
aI27
aI7
aI1
aI135
aI171
aI231
aI3349
aI4459
aI2925
asS'd'
I1331
sa(dp2662
S'a'
I2477
sS's'
I14
sS'm'
(lp2663
I1
aI1
aI5
aI5
aI9
aI51
aI71
aI153
aI115
aI315
aI265
aI2207
aI4127
aI12631
asS'd'
I1332
sa(dp2664
S'a'
I2509
sS's'
I14
sS'm'
(lp2665
I1
aI1
aI3
aI15
aI23
aI59
aI35
aI121
aI425
aI921
aI1255
aI2123
aI5811
aI15937
asS'd'
I1333
sa(dp2666
S'a'
I2510
sS's'
I14
sS'm'
(lp2667
I1
aI3
aI7
aI7
aI11
aI21
aI45
aI57
aI269
aI395
aI555
aI783
aI6677
aI2889
asS'd'
I1334
sa(dp2668
S'a'
I2515
sS's'
I14
sS'm'
(lp2669
I1
aI3
aI5
aI7
aI31
aI19
aI73
aI35
aI465
aI349
aI1429
aI863
aI4707
aI6121
asS'd'
I1335
sa(dp2670
S'a'
I2524
sS's'
I14
sS'm'
(lp2671
I1
aI3
aI3
aI9
aI25
aI27
aI119
aI159
aI195
aI949
aI19
aI73
aI4511
aI15711
asS'd'
I1336
sa(dp2672
S'a'
I2527
sS's'
I14
sS'm'
(lp2673
I1
aI3
aI3
aI7
aI9
aI59
aI47
aI57
aI91
aI749
aI1579
aI1297
aI2445
aI5167
asS'd'
I1337
sa(dp2674
S'a'
I2531
sS's'
I14
sS'm'
(lp2675
I1
aI3
aI3
aI3
aI31
aI57
aI19
aI203
aI61
aI927
aI1477
aI2863
aI1305
aI11673
asS'd'
I1338
sa(dp2676
S'a'
I2552
sS's'
I14
sS'm'
(lp2677
I1
aI3
aI7
aI11
aI29
aI13
aI3
aI111
aI351
aI79
aI1863
aI2213
aI3273
aI7049
asS'd'
I1339
sa(dp2678
S'a'
I2561
sS's'
I14
sS'm'
(lp2679
I1
aI3
aI3
aI9
aI7
aI23
aI47
aI237
aI121
aI877
aI441
aI119
aI2723
aI3989
asS'd'
I1340
sa(dp2680
S'a'
I2567
sS's'
I14
sS'm'
(lp2681
I1
aI3
aI3
aI11
aI17
aI23
aI63
aI177
aI231
aI363
aI1451
aI33
aI2169
aI7251
asS'd'
I1341
sa(dp2682
S'a'
I2571
sS's'
I14
sS'm'
(lp2683
I1
aI1
aI5
aI11
aI31
aI41
aI93
aI229
aI39
aI1009
aI1061
aI433
aI2393
aI15401
asS'd'
I1342
sa(dp2684
S'a'
I2586
sS's'
I14
sS'm'
(lp2685
I1
aI1
aI5
aI15
aI31
aI37
aI25
aI135
aI135
aI897
aI33
aI3713
aI7663
aI8079
asS'd'
I1343
sa(dp2686
S'a'
I2588
sS's'
I14
sS'm'
(lp2687
I1
aI1
aI5
aI7
aI17
aI49
aI43
aI89
aI411
aI731
aI1431
aI3893
aI1635
aI7063
asS'd'
I1344
sa(dp2688
S'a'
I2595
sS's'
I14
sS'm'
(lp2689
I1
aI1
aI1
aI13
aI29
aI27
aI5
aI77
aI283
aI913
aI789
aI817
aI3309
aI475
asS'd'
I1345
sa(dp2690
S'a'
I2607
sS's'
I14
sS'm'
(lp2691
I1
aI1
aI3
aI1
aI19
aI21
aI67
aI77
aI423
aI551
aI5
aI1057
aI5469
aI7859
asS'd'
I1346
sa(dp2692
S'a'
I2621
sS's'
I14
sS'm'
(lp2693
I1
aI1
aI5
aI1
aI1
aI21
aI99
aI237
aI215
aI759
aI1505
aI1983
aI1517
aI8923
asS'd'
I1347
sa(dp2694
S'a'
I2630
sS's'
I14
sS'm'
(lp2695
I1
aI3
aI5
aI7
aI19
aI61
aI73
aI215
aI165
aI127
aI205
aI259
aI7755
aI15395
asS'd'
I1348
sa(dp2696
S'a'
I2639
sS's'
I14
sS'm'
(lp2697
I1
aI1
aI5
aI9
aI15
aI23
aI17
aI111
aI471
aI751
aI1923
aI775
aI6901
aI13095
asS'd'
I1349
sa(dp2698
S'a'
I2653
sS's'
I14
sS'm'
(lp2699
I1
aI1
aI7
aI1
aI25
aI5
aI63
aI141
aI461
aI687
aI1589
aI1559
aI7719
aI11349
asS'd'
I1350
sa(dp2700
S'a'
I2670
sS's'
I14
sS'm'
(lp2701
I1
aI1
aI1
aI3
aI11
aI63
aI11
aI27
aI253
aI439
aI297
aI1315
aI829
aI3765
asS'd'
I1351
sa(dp2702
S'a'
I2672
sS's'
I14
sS'm'
(lp2703
I1
aI3
aI1
aI1
aI9
aI47
aI127
aI179
aI173
aI809
aI241
aI35
aI7355
aI5049
asS'd'
I1352
sa(dp2704
S'a'
I2700
sS's'
I14
sS'm'
(lp2705
I1
aI3
aI1
aI11
aI19
aI63
aI93
aI1
aI205
aI977
aI303
aI3409
aI6529
aI10927
asS'd'
I1353
sa(dp2706
S'a'
I2711
sS's'
I14
sS'm'
(lp2707
I1
aI3
aI7
aI9
aI31
aI63
aI41
aI79
aI477
aI91
aI1801
aI3487
aI6885
aI13341
asS'd'
I1354
sa(dp2708
S'a'
I2715
sS's'
I14
sS'm'
(lp2709
I1
aI1
aI3
aI7
aI15
aI59
aI9
aI101
aI459
aI247
aI549
aI2855
aI5765
aI7785
asS'd'
I1355
sa(dp2710
S'a'
I2748
sS's'
I14
sS'm'
(lp2711
I1
aI1
aI7
aI3
aI13
aI59
aI71
aI123
aI93
aI517
aI1453
aI2389
aI4429
aI5053
asS'd'
I1356
sa(dp2712
S'a'
I2751
sS's'
I14
sS'm'
(lp2713
I1
aI1
aI5
aI3
aI19
aI21
aI77
aI53
aI81
aI879
aI1653
aI1637
aI3667
aI2623
asS'd'
I1357
sa(dp2714
S'a'
I2753
sS's'
I14
sS'm'
(lp2715
I1
aI1
aI1
aI15
aI17
aI57
aI65
aI53
aI407
aI765
aI417
aI497
aI5009
aI2175
asS'd'
I1358
sa(dp2716
S'a'
I2754
sS's'
I14
sS'm'
(lp2717
I1
aI3
aI3
aI7
aI31
aI13
aI5
aI203
aI263
aI17
aI119
aI1607
aI6773
aI11195
asS'd'
I1359
sa(dp2718
S'a'
I2760
sS's'
I14
sS'm'
(lp2719
I1
aI3
aI3
aI13
aI19
aI13
aI13
aI147
aI93
aI735
aI689
aI781
aI655
aI6853
asS'd'
I1360
sa(dp2720
S'a'
I2774
sS's'
I14
sS'm'
(lp2721
I1
aI1
aI1
aI1
aI1
aI25
aI63
aI159
aI493
aI987
aI71
aI1249
aI5859
aI11717
asS'd'
I1361
sa(dp2722
S'a'
I2784
sS's'
I14
sS'm'
(lp2723
I1
aI1
aI1
aI15
aI13
aI23
aI61
aI61
aI5
aI947
aI1853
aI3331
aI467
aI8081
asS'd'
I1362
sa(dp2724
S'a'
I2793
sS's'
I14
sS'm'
(lp2725
I1
aI1
aI3
aI9
aI19
aI61
aI65
aI189
aI95
aI309
aI283
aI1725
aI5683
aI15463
asS'd'
I1363
sa(dp2726
S'a'
I2804
sS's'
I14
sS'm'
(lp2727
I1
aI1
aI7
aI5
aI9
aI33
aI35
aI75
aI475
aI831
aI1445
aI1485
aI5047
aI9631
asS'd'
I1364
sa(dp2728
S'a'
I2811
sS's'
I14
sS'm'
(lp2729
I1
aI1
aI3
aI15
aI11
aI23
aI59
aI87
aI433
aI221
aI685
aI3113
aI4095
aI13819
asS'd'
I1365
sa(dp2730
S'a'
I2822
sS's'
I14
sS'm'
(lp2731
I1
aI1
aI7
aI15
aI25
aI29
aI67
aI17
aI349
aI353
aI1321
aI563
aI57
aI533
asS'd'
I1366
sa(dp2732
S'a'
I2826
sS's'
I14
sS'm'
(lp2733
I1
aI3
aI3
aI3
aI5
aI43
aI109
aI217
aI15
aI185
aI1895
aI1015
aI1831
aI10623
asS'd'
I1367
sa(dp2734
S'a'
I2836
sS's'
I14
sS'm'
(lp2735
I1
aI1
aI7
aI1
aI1
aI47
aI81
aI185
aI59
aI691
aI191
aI3709
aI1535
aI13347
asS'd'
I1368
sa(dp2736
S'a'
I2839
sS's'
I14
sS'm'
(lp2737
I1
aI1
aI5
aI1
aI23
aI57
aI83
aI217
aI457
aI771
aI1877
aI2789
aI8143
aI4797
asS'd'
I1369
sa(dp2738
S'a'
I2840
sS's'
I14
sS'm'
(lp2739
I1
aI1
aI3
aI7
aI23
aI35
aI79
aI49
aI227
aI205
aI1523
aI3873
aI4843
aI10505
asS'd'
I1370
sa(dp2740
S'a'
I2893
sS's'
I14
sS'm'
(lp2741
I1
aI1
aI1
aI1
aI17
aI43
aI121
aI95
aI205
aI35
aI189
aI2061
aI1693
aI13273
asS'd'
I1371
sa(dp2742
S'a'
I2901
sS's'
I14
sS'm'
(lp2743
I1
aI1
aI1
aI15
aI31
aI49
aI83
aI249
aI433
aI497
aI1949
aI1845
aI5215
aI5971
asS'd'
I1372
sa(dp2744
S'a'
I2902
sS's'
I14
sS'm'
(lp2745
I1
aI3
aI1
aI1
aI21
aI53
aI73
aI211
aI265
aI929
aI923
aI279
aI3621
aI9469
asS'd'
I1373
sa(dp2746
S'a'
I2905
sS's'
I14
sS'm'
(lp2747
I1
aI3
aI7
aI7
aI1
aI57
aI13
aI45
aI467
aI705
aI371
aI1345
aI1647
aI3411
asS'd'
I1374
sa(dp2748
S'a'
I2912
sS's'
I14
sS'm'
(lp2749
I1
aI3
aI1
aI11
aI27
aI29
aI117
aI163
aI143
aI669
aI489
aI3913
aI7891
aI9031
asS'd'
I1375
sa(dp2750
S'a'
I2915
sS's'
I14
sS'm'
(lp2751
I1
aI3
aI7
aI15
aI27
aI15
aI77
aI217
aI107
aI839
aI1517
aI1543
aI357
aI10365
asS'd'
I1376
sa(dp2752
S'a'
I2918
sS's'
I14
sS'm'
(lp2753
I1
aI1
aI1
aI5
aI31
aI17
aI107
aI245
aI345
aI939
aI1453
aI3645
aI6865
aI16173
asS'd'
I1377
sa(dp2754
S'a'
I2939
sS's'
I14
sS'm'
(lp2755
I1
aI3
aI5
aI5
aI9
aI61
aI43
aI97
aI453
aI917
aI945
aI2143
aI5473
aI5611
asS'd'
I1378
sa(dp2756
S'a'
I2965
sS's'
I14
sS'm'
(lp2757
I1
aI1
aI5
aI11
aI3
aI33
aI71
aI97
aI137
aI549
aI1605
aI3839
aI4883
aI2677
asS'd'
I1379
sa(dp2758
S'a'
I2966
sS's'
I14
sS'm'
(lp2759
I1
aI3
aI1
aI11
aI29
aI23
aI85
aI47
aI225
aI633
aI1613
aI1297
aI1415
aI15813
asS'd'
I1380
sa(dp2760
S'a'
I2975
sS's'
I14
sS'm'
(lp2761
I1
aI1
aI3
aI3
aI9
aI19
aI57
aI107
aI79
aI449
aI1951
aI753
aI6317
aI10377
asS'd'
I1381
sa(dp2762
S'a'
I2988
sS's'
I14
sS'm'
(lp2763
I1
aI1
aI1
aI5
aI21
aI3
aI39
aI187
aI299
aI517
aI1313
aI741
aI7259
aI4197
asS'd'
I1382
sa(dp2764
S'a'
I2993
sS's'
I14
sS'm'
(lp2765
I1
aI1
aI5
aI13
aI1
aI39
aI39
aI41
aI381
aI123
aI1257
aI3185
aI493
aI3723
asS'd'
I1383
sa(dp2766
S'a'
I3006
sS's'
I14
sS'm'
(lp2767
I1
aI3
aI7
aI7
aI3
aI37
aI15
aI161
aI129
aI169
aI555
aI3605
aI4287
aI15831
asS'd'
I1384
sa(dp2768
S'a'
I3017
sS's'
I14
sS'm'
(lp2769
I1
aI3
aI7
aI15
aI15
aI23
aI81
aI163
aI257
aI791
aI505
aI1903
aI2703
aI11919
asS'd'
I1385
sa(dp2770
S'a'
I3031
sS's'
I14
sS'm'
(lp2771
I1
aI3
aI7
aI7
aI27
aI63
aI17
aI147
aI111
aI851
aI1533
aI1365
aI5359
aI3315
asS'd'
I1386
sa(dp2772
S'a'
I3038
sS's'
I14
sS'm'
(lp2773
I1
aI3
aI7
aI1
aI15
aI5
aI61
aI143
aI385
aI261
aI1019
aI1705
aI1737
aI14485
asS'd'
I1387
sa(dp2774
S'a'
I3041
sS's'
I14
sS'm'
(lp2775
I1
aI3
aI5
aI5
aI25
aI17
aI49
aI229
aI431
aI567
aI1613
aI3431
aI2139
aI2981
asS'd'
I1388
sa(dp2776
S'a'
I3042
sS's'
I14
sS'm'
(lp2777
I1
aI3
aI5
aI11
aI17
aI57
aI71
aI241
aI31
aI1007
aI1695
aI2965
aI149
aI14125
asS'd'
I1389
sa(dp2778
S'a'
I3051
sS's'
I14
sS'm'
(lp2779
I1
aI1
aI3
aI11
aI7
aI49
aI39
aI101
aI5
aI501
aI1491
aI3061
aI225
aI12255
asS'd'
I1390
sa(dp2780
S'a'
I3073
sS's'
I14
sS'm'
(lp2781
I1
aI3
aI5
aI7
aI17
aI35
aI37
aI97
aI415
aI15
aI1349
aI997
aI2949
aI4511
asS'd'
I1391
sa(dp2782
S'a'
I3088
sS's'
I14
sS'm'
(lp2783
I1
aI3
aI1
aI5
aI25
aI35
aI99
aI183
aI161
aI59
aI1363
aI515
aI3767
aI3641
asS'd'
I1392
sa(dp2784
S'a'
I3097
sS's'
I14
sS'm'
(lp2785
I1
aI1
aI7
aI15
aI7
aI15
aI127
aI137
aI281
aI67
aI139
aI2315
aI3517
aI13371
asS'd'
I1393
sa(dp2786
S'a'
I3098
sS's'
I14
sS'm'
(lp2787
I1
aI1
aI5
aI15
aI23
aI49
aI19
aI79
aI425
aI805
aI1035
aI429
aI7707
aI14195
asS'd'
I1394
sa(dp2788
S'a'
I3103
sS's'
I14
sS'm'
(lp2789
I1
aI3
aI5
aI3
aI21
aI25
aI123
aI11
aI425
aI475
aI961
aI2995
aI7405
aI5449
asS'd'
I1395
sa(dp2790
S'a'
I3104
sS's'
I14
sS'm'
(lp2791
I1
aI1
aI7
aI1
aI21
aI1
aI75
aI231
aI451
aI595
aI719
aI2369
aI5907
aI1227
asS'd'
I1396
sa(dp2792
S'a'
I3146
sS's'
I14
sS'm'
(lp2793
I1
aI1
aI1
aI9
aI21
aI57
aI45
aI255
aI19
aI79
aI481
aI3363
aI3451
aI8399
asS'd'
I1397
sa(dp2794
S'a'
I3148
sS's'
I14
sS'm'
(lp2795
I1
aI1
aI7
aI13
aI31
aI49
aI95
aI69
aI483
aI427
aI37
aI4047
aI7057
aI9111
asS'd'
I1398
sa(dp2796
S'a'
I3153
sS's'
I14
sS'm'
(lp2797
I1
aI3
aI3
aI11
aI3
aI61
aI87
aI79
aI499
aI91
aI771
aI1987
aI2017
aI3381
asS'd'
I1399
sa(dp2798
S'a'
I3159
sS's'
I14
sS'm'
(lp2799
I1
aI3
aI1
aI7
aI5
aI57
aI1
aI121
aI155
aI225
aI501
aI477
aI6555
aI9863
asS'd'
I1400
sa(dp2800
S'a'
I3182
sS's'
I14
sS'm'
(lp2801
I1
aI3
aI7
aI11
aI27
aI49
aI83
aI213
aI61
aI283
aI1599
aI3205
aI2525
aI8553
asS'd'
I1401
sa(dp2802
S'a'
I3187
sS's'
I14
sS'm'
(lp2803
I1
aI1
aI1
aI9
aI9
aI49
aI3
aI51
aI141
aI33
aI301
aI2167
aI587
aI15067
asS'd'
I1402
sa(dp2804
S'a'
I3189
sS's'
I14
sS'm'
(lp2805
I1
aI1
aI1
aI11
aI7
aI55
aI99
aI81
aI191
aI553
aI953
aI3753
aI6731
aI1093
asS'd'
I1403
sa(dp2806
S'a'
I3199
sS's'
I14
sS'm'
(lp2807
I1
aI1
aI3
aI3
aI11
aI59
aI57
aI235
aI297
aI197
aI853
aI1411
aI3799
aI7527
asS'd'
I1404
sa(dp2808
S'a'
I3239
sS's'
I14
sS'm'
(lp2809
I1
aI3
aI5
aI3
aI7
aI7
aI5
aI201
aI393
aI95
aI91
aI3273
aI6285
aI10661
asS'd'
I1405
sa(dp2810
S'a'
I3263
sS's'
I14
sS'm'
(lp2811
I1
aI1
aI5
aI7
aI17
aI57
aI87
aI3
aI413
aI915
aI659
aI369
aI3593
aI14429
asS'd'
I1406
sa(dp2812
S'a'
I3271
sS's'
I14
sS'm'
(lp2813
I1
aI3
aI7
aI1
aI31
aI31
aI45
aI115
aI417
aI427
aI745
aI4087
aI953
aI1119
asS'd'
I1407
sa(dp2814
S'a'
I3275
sS's'
I14
sS'm'
(lp2815
I1
aI3
aI7
aI3
aI29
aI43
aI45
aI221
aI41
aI641
aI451
aI173
aI2999
aI12103
asS'd'
I1408
sa(dp2816
S'a'
I3278
sS's'
I14
sS'm'
(lp2817
I1
aI1
aI3
aI11
aI25
aI57
aI117
aI201
aI135
aI787
aI1525
aI3879
aI3247
aI8907
asS'd'
I1409
sa(dp2818
S'a'
I3280
sS's'
I14
sS'm'
(lp2819
I1
aI1
aI7
aI11
aI3
aI35
aI69
aI157
aI331
aI615
aI573
aI2169
aI3575
aI289
asS'd'
I1410
sa(dp2820
S'a'
I3283
sS's'
I14
sS'm'
(lp2821
I1
aI3
aI3
aI13
aI15
aI51
aI67
aI127
aI265
aI495
aI103
aI3145
aI2685
aI15919
asS'd'
I1411
sa(dp2822
S'a'
I3290
sS's'
I14
sS'm'
(lp2823
I1
aI3
aI5
aI11
aI31
aI27
aI65
aI57
aI153
aI465
aI1163
aI467
aI4103
aI4713
asS'd'
I1412
sa(dp2824
S'a'
I3311
sS's'
I14
sS'm'
(lp2825
I1
aI3
aI7
aI3
aI23
aI31
aI9
aI51
aI239
aI417
aI1597
aI229
aI2865
aI15199
asS'd'
I1413
sa(dp2826
S'a'
I3316
sS's'
I14
sS'm'
(lp2827
I1
aI3
aI5
aI3
aI11
aI45
aI123
aI217
aI31
aI765
aI1009
aI2001
aI3645
aI9407
asS'd'
I1414
sa(dp2828
S'a'
I3343
sS's'
I14
sS'm'
(lp2829
I1
aI3
aI3
aI9
aI5
aI23
aI117
aI83
aI237
aI1017
aI251
aI1187
aI2631
aI5151
asS'd'
I1415
sa(dp2830
S'a'
I3346
sS's'
I14
sS'm'
(lp2831
I1
aI1
aI1
aI7
aI23
aI55
aI97
aI141
aI501
aI305
aI467
aI4061
aI2369
aI15973
asS'd'
I1416
sa(dp2832
S'a'
I3357
sS's'
I14
sS'm'
(lp2833
I1
aI1
aI7
aI5
aI31
aI51
aI125
aI191
aI219
aI495
aI37
aI3337
aI813
aI241
asS'd'
I1417
sa(dp2834
S'a'
I3358
sS's'
I14
sS'm'
(lp2835
I1
aI3
aI1
aI1
aI11
aI39
aI93
aI109
aI285
aI147
aI1297
aI737
aI4051
aI7223
asS'd'
I1418
sa(dp2836
S'a'
I3361
sS's'
I14
sS'm'
(lp2837
I1
aI3
aI1
aI15
aI13
aI17
aI57
aI171
aI463
aI163
aI609
aI1681
aI7583
aI9231
asS'd'
I1419
sa(dp2838
S'a'
I3362
sS's'
I14
sS'm'
(lp2839
I1
aI3
aI1
aI1
aI23
aI5
aI51
aI5
aI205
aI415
aI419
aI989
aI4239
aI10943
asS'd'
I1420
sa(dp2840
S'a'
I3364
sS's'
I14
sS'm'
(lp2841
I1
aI1
aI3
aI15
aI3
aI13
aI65
aI145
aI387
aI59
aI395
aI1067
aI4143
aI5649
asS'd'
I1421
sa(dp2842
S'a'
I3386
sS's'
I14
sS'm'
(lp2843
I1
aI3
aI1
aI13
aI9
aI59
aI121
aI127
aI95
aI71
aI1541
aI1423
aI1753
aI8041
asS'd'
I1422
sa(dp2844
S'a'
I3418
sS's'
I14
sS'm'
(lp2845
I1
aI1
aI3
aI15
aI7
aI5
aI69
aI167
aI181
aI991
aI1189
aI4017
aI5935
aI6669
asS'd'
I1423
sa(dp2846
S'a'
I3424
sS's'
I14
sS'm'
(lp2847
I1
aI3
aI5
aI7
aI23
aI41
aI53
aI21
aI47
aI261
aI1231
aI2011
aI133
aI2247
asS'd'
I1424
sa(dp2848
S'a'
I3433
sS's'
I14
sS'm'
(lp2849
I1
aI1
aI1
aI5
aI17
aI47
aI77
aI19
aI331
aI609
aI1893
aI3965
aI3123
aI9093
asS'd'
I1425
sa(dp2850
S'a'
I3434
sS's'
I14
sS'm'
(lp2851
I1
aI3
aI1
aI3
aI9
aI39
aI103
aI231
aI249
aI75
aI373
aI107
aI1823
aI10801
asS'd'
I1426
sa(dp2852
S'a'
I3436
sS's'
I14
sS'm'
(lp2853
I1
aI3
aI3
aI7
aI1
aI51
aI35
aI111
aI137
aI879
aI1221
aI225
aI4285
aI2287
asS'd'
I1427
sa(dp2854
S'a'
I3463
sS's'
I14
sS'm'
(lp2855
I1
aI1
aI7
aI9
aI23
aI17
aI75
aI245
aI409
aI163
aI395
aI3731
aI7111
aI6845
asS'd'
I1428
sa(dp2856
S'a'
I3467
sS's'
I14
sS'm'
(lp2857
I1
aI1
aI3
aI13
aI29
aI47
aI75
aI153
aI497
aI621
aI1691
aI3187
aI2125
aI10533
asS'd'
I1429
sa(dp2858
S'a'
I3477
sS's'
I14
sS'm'
(lp2859
I1
aI1
aI7
aI7
aI9
aI7
aI55
aI159
aI255
aI417
aI1335
aI643
aI3843
aI3733
asS'd'
I1430
sa(dp2860
S'a'
I3484
sS's'
I14
sS'm'
(lp2861
I1
aI3
aI3
aI1
aI21
aI41
aI7
aI21
aI5
aI679
aI1655
aI95
aI5699
aI5785
asS'd'
I1431
sa(dp2862
S'a'
I3505
sS's'
I14
sS'm'
(lp2863
I1
aI1
aI1
aI13
aI19
aI7
aI85
aI7
aI195
aI357
aI1097
aI2893
aI2913
aI9635
asS'd'
I1432
sa(dp2864
S'a'
I3508
sS's'
I14
sS'm'
(lp2865
I1
aI1
aI5
aI9
aI25
aI33
aI41
aI155
aI39
aI655
aI1993
aI3117
aI3639
aI7977
asS'd'
I1433
sa(dp2866
S'a'
I3515
sS's'
I14
sS'm'
(lp2867
I1
aI1
aI1
aI13
aI3
aI63
aI121
aI247
aI151
aI673
aI609
aI285
aI2299
aI7663
asS'd'
I1434
sa(dp2868
S'a'
I3532
sS's'
I14
sS'm'
(lp2869
I1
aI3
aI7
aI11
aI17
aI13
aI49
aI253
aI245
aI21
aI273
aI993
aI911
aI863
asS'd'
I1435
sa(dp2870
S'a'
I3553
sS's'
I14
sS'm'
(lp2871
I1
aI1
aI5
aI5
aI23
aI1
aI121
aI95
aI225
aI9
aI1237
aI1183
aI6461
aI559
asS'd'
I1436
sa(dp2872
S'a'
I3554
sS's'
I14
sS'm'
(lp2873
I1
aI3
aI7
aI13
aI3
aI7
aI121
aI151
aI233
aI561
aI281
aI3583
aI897
aI1767
asS'd'
I1437
sa(dp2874
S'a'
I3568
sS's'
I14
sS'm'
(lp2875
I1
aI1
aI7
aI7
aI9
aI47
aI107
aI41
aI25
aI569
aI1697
aI2299
aI6231
aI12209
asS'd'
I1438
sa(dp2876
S'a'
I3573
sS's'
I14
sS'm'
(lp2877
I1
aI3
aI7
aI7
aI27
aI43
aI59
aI37
aI31
aI51
aI503
aI149
aI4043
aI11847
asS'd'
I1439
sa(dp2878
S'a'
I3587
sS's'
I14
sS'm'
(lp2879
I1
aI3
aI3
aI11
aI5
aI1
aI119
aI181
aI47
aI641
aI685
aI4017
aI637
aI16251
asS'd'
I1440
sa(dp2880
S'a'
I3589
sS's'
I14
sS'm'
(lp2881
I1
aI3
aI3
aI7
aI11
aI1
aI101
aI7
aI239
aI747
aI307
aI1721
aI5979
aI4367
asS'd'
I1441
sa(dp2882
S'a'
I3596
sS's'
I14
sS'm'
(lp2883
I1
aI3
aI5
aI7
aI1
aI63
aI19
aI151
aI469
aI333
aI1587
aI2453
aI897
aI4711
asS'd'
I1442
sa(dp2884
S'a'
I3608
sS's'
I14
sS'm'
(lp2885
I1
aI3
aI1
aI5
aI11
aI61
aI21
aI253
aI91
aI993
aI1347
aI1993
aI5607
aI13031
asS'd'
I1443
sa(dp2886
S'a'
I3620
sS's'
I14
sS'm'
(lp2887
I1
aI3
aI5
aI5
aI1
aI39
aI65
aI71
aI189
aI389
aI1437
aI1055
aI6439
aI3989
asS'd'
I1444
sa(dp2888
S'a'
I3630
sS's'
I14
sS'm'
(lp2889
I1
aI1
aI3
aI3
aI19
aI15
aI93
aI3
aI339
aI165
aI1675
aI3953
aI2145
aI12113
asS'd'
I1445
sa(dp2890
S'a'
I3644
sS's'
I14
sS'm'
(lp2891
I1
aI1
aI3
aI13
aI13
aI45
aI5
aI175
aI211
aI993
aI705
aI2761
aI3023
aI13633
asS'd'
I1446
sa(dp2892
S'a'
I3649
sS's'
I14
sS'm'
(lp2893
I1
aI1
aI3
aI1
aI19
aI39
aI121
aI29
aI287
aI87
aI281
aI3491
aI7107
aI13007
asS'd'
I1447
sa(dp2894
S'a'
I3664
sS's'
I14
sS'm'
(lp2895
I1
aI1
aI7
aI1
aI29
aI49
aI103
aI187
aI39
aI923
aI51
aI1533
aI3249
aI4399
asS'd'
I1448
sa(dp2896
S'a'
I3679
sS's'
I14
sS'm'
(lp2897
I1
aI1
aI5
aI5
aI5
aI43
aI25
aI107
aI453
aI955
aI115
aI57
aI4589
aI14573
asS'd'
I1449
sa(dp2898
S'a'
I3680
sS's'
I14
sS'm'
(lp2899
I1
aI1
aI3
aI5
aI21
aI45
aI103
aI99
aI183
aI987
aI1207
aI1697
aI8033
aI13703
asS'd'
I1450
sa(dp2900
S'a'
I3685
sS's'
I14
sS'm'
(lp2901
I1
aI1
aI5
aI7
aI11
aI23
aI9
aI17
aI261
aI749
aI1957
aI935
aI6283
aI8625
asS'd'
I1451
sa(dp2902
S'a'
I3686
sS's'
I14
sS'm'
(lp2903
I1
aI1
aI1
aI9
aI9
aI51
aI69
aI225
aI265
aI323
aI1161
aI2993
aI7305
aI2249
asS'd'
I1452
sa(dp2904
S'a'
I3698
sS's'
I14
sS'm'
(lp2905
I1
aI3
aI1
aI9
aI23
aI19
aI57
aI205
aI503
aI489
aI1499
aI3277
aI817
aI11931
asS'd'
I1453
sa(dp2906
S'a'
I3714
sS's'
I14
sS'm'
(lp2907
I1
aI3
aI3
aI5
aI1
aI7
aI49
aI1
aI313
aI123
aI643
aI2027
aI1469
aI3585
asS'd'
I1454
sa(dp2908
S'a'
I3726
sS's'
I14
sS'm'
(lp2909
I1
aI3
aI7
aI11
aI27
aI47
aI95
aI111
aI27
aI213
aI465
aI3693
aI3661
aI7531
asS'd'
I1455
sa(dp2910
S'a'
I3737
sS's'
I14
sS'm'
(lp2911
I1
aI1
aI7
aI9
aI3
aI37
aI115
aI189
aI31
aI613
aI1393
aI1229
aI4767
aI12425
asS'd'
I1456
sa(dp2912
S'a'
I3767
sS's'
I14
sS'm'
(lp2913
I1
aI1
aI3
aI3
aI25
aI17
aI99
aI47
aI161
aI931
aI959
aI1293
aI7095
aI8325
asS'd'
I1457
sa(dp2914
S'a'
I3782
sS's'
I14
sS'm'
(lp2915
I1
aI1
aI1
aI7
aI23
aI9
aI11
aI51
aI205
aI419
aI479
aI1497
aI2493
aI13921
asS'd'
I1458
sa(dp2916
S'a'
I3786
sS's'
I14
sS'm'
(lp2917
I1
aI3
aI1
aI9
aI17
aI29
aI51
aI79
aI159
aI435
aI477
aI413
aI3815
aI5589
asS'd'
I1459
sa(dp2918
S'a'
I3793
sS's'
I14
sS'm'
(lp2919
I1
aI3
aI7
aI5
aI7
aI23
aI99
aI43
aI169
aI665
aI403
aI1163
aI4337
aI1335
asS'd'
I1460
sa(dp2920
S'a'
I3796
sS's'
I14
sS'm'
(lp2921
I1
aI3
aI1
aI5
aI25
aI27
aI125
aI249
aI421
aI267
aI1259
aI4089
aI59
aI9377
asS'd'
I1461
sa(dp2922
S'a'
I3805
sS's'
I14
sS'm'
(lp2923
I1
aI3
aI3
aI1
aI27
aI37
aI91
aI17
aI123
aI597
aI1749
aI3449
aI6503
aI11043
asS'd'
I1462
sa(dp2924
S'a'
I3815
sS's'
I14
sS'm'
(lp2925
I1
aI3
aI7
aI7
aI23
aI41
aI19
aI245
aI109
aI569
aI547
aI1917
aI7943
aI2697
asS'd'
I1463
sa(dp2926
S'a'
I3841
sS's'
I14
sS'm'
(lp2927
I1
aI3
aI7
aI7
aI9
aI1
aI123
aI105
aI329
aI435
aI2013
aI2745
aI347
aI11045
asS'd'
I1464
sa(dp2928
S'a'
I3847
sS's'
I14
sS'm'
(lp2929
I1
aI1
aI1
aI13
aI29
aI53
aI51
aI67
aI105
aI89
aI1887
aI3543
aI963
aI8159
asS'd'
I1465
sa(dp2930
S'a'
I3853
sS's'
I14
sS'm'
(lp2931
I1
aI1
aI5
aI3
aI5
aI27
aI41
aI67
aI67
aI883
aI973
aI1785
aI901
aI14969
asS'd'
I1466
sa(dp2932
S'a'
I3862
sS's'
I14
sS'm'
(lp2933
I1
aI3
aI3
aI13
aI17
aI11
aI117
aI115
aI163
aI939
aI79
aI641
aI4365
aI2267
asS'd'
I1467
sa(dp2934
S'a'
I3875
sS's'
I14
sS'm'
(lp2935
I1
aI1
aI3
aI3
aI9
aI5
aI41
aI123
aI149
aI9
aI1533
aI3939
aI5995
aI12701
asS'd'
I1468
sa(dp2936
S'a'
I3902
sS's'
I14
sS'm'
(lp2937
I1
aI1
aI1
aI15
aI31
aI1
aI101
aI229
aI191
aI965
aI61
aI2671
aI4177
aI15779
asS'd'
I1469
sa(dp2938
S'a'
I3904
sS's'
I14
sS'm'
(lp2939
I1
aI1
aI5
aI15
aI1
aI25
aI49
aI185
aI33
aI697
aI1043
aI2639
aI7819
aI3171
asS'd'
I1470
sa(dp2940
S'a'
I3916
sS's'
I14
sS'm'
(lp2941
I1
aI3
aI5
aI13
aI19
aI9
aI111
aI49
aI47
aI847
aI1865
aI717
aI5287
aI13417
asS'd'
I1471
sa(dp2942
S'a'
I3947
sS's'
I14
sS'm'
(lp2943
I1
aI3
aI7
aI11
aI5
aI61
aI63
aI111
aI171
aI735
aI2003
aI73
aI5701
aI647
asS'd'
I1472
sa(dp2944
S'a'
I3949
sS's'
I14
sS'm'
(lp2945
I1
aI3
aI1
aI11
aI1
aI49
aI121
aI79
aI431
aI671
aI1241
aI1161
aI2057
aI263
asS'd'
I1473
sa(dp2946
S'a'
I3955
sS's'
I14
sS'm'
(lp2947
I1
aI3
aI3
aI1
aI1
aI23
aI75
aI15
aI117
aI641
aI313
aI1525
aI2041
aI1409
asS'd'
I1474
sa(dp2948
S'a'
I3962
sS's'
I14
sS'm'
(lp2949
I1
aI3
aI5
aI11
aI15
aI57
aI13
aI67
aI139
aI131
aI1339
aI2419
aI7945
aI11877
asS'd'
I1475
sa(dp2950
S'a'
I3971
sS's'
I14
sS'm'
(lp2951
I1
aI3
aI1
aI1
aI19
aI39
aI97
aI83
aI297
aI595
aI1611
aI5
aI4753
aI3435
asS'd'
I1476
sa(dp2952
S'a'
I3980
sS's'
I14
sS'm'
(lp2953
I1
aI3
aI1
aI9
aI7
aI49
aI125
aI101
aI383
aI717
aI63
aI2295
aI3873
aI13461
asS'd'
I1477
sa(dp2954
S'a'
I3985
sS's'
I14
sS'm'
(lp2955
I1
aI1
aI3
aI3
aI15
aI29
aI89
aI77
aI269
aI689
aI229
aI1207
aI7311
aI8663
asS'd'
I1478
sa(dp2956
S'a'
I3998
sS's'
I14
sS'm'
(lp2957
I1
aI1
aI1
aI1
aI1
aI61
aI25
aI255
aI203
aI233
aI271
aI987
aI2277
aI8735
asS'd'
I1479
sa(dp2958
S'a'
I4001
sS's'
I14
sS'm'
(lp2959
I1
aI1
aI5
aI7
aI21
aI27
aI63
aI79
aI337
aI133
aI1453
aI3633
aI6157
aI15875
asS'd'
I1480
sa(dp2960
S'a'
I4002
sS's'
I14
sS'm'
(lp2961
I1
aI3
aI1
aI7
aI7
aI55
aI31
aI81
aI203
aI709
aI1743
aI1677
aI4247
aI11411
asS'd'
I1481
sa(dp2962
S'a'
I4016
sS's'
I14
sS'm'
(lp2963
I1
aI1
aI3
aI3
aI29
aI51
aI37
aI17
aI487
aI325
aI1393
aI1433
aI3467
aI2851
asS'd'
I1482
sa(dp2964
S'a'
I4021
sS's'
I14
sS'm'
(lp2965
I1
aI1
aI7
aI9
aI3
aI41
aI99
aI177
aI241
aI869
aI739
aI2729
aI4585
aI14801
asS'd'
I1483
sa(dp2966
S'a'
I4026
sS's'
I14
sS'm'
(lp2967
I1
aI1
aI3
aI1
aI9
aI43
aI97
aI65
aI99
aI295
aI1693
aI2083
aI3241
aI4073
asS'd'
I1484
sa(dp2968
S'a'
I4043
sS's'
I14
sS'm'
(lp2969
I1
aI1
aI1
aI9
aI5
aI39
aI67
aI119
aI235
aI543
aI795
aI2773
aI3195
aI6273
asS'd'
I1485
sa(dp2970
S'a'
I4079
sS's'
I14
sS'm'
(lp2971
I1
aI1
aI5
aI5
aI21
aI41
aI89
aI1
aI85
aI81
aI57
aI2951
aI1531
aI10101
asS'd'
I1486
sa(dp2972
S'a'
I4102
sS's'
I14
sS'm'
(lp2973
I1
aI1
aI1
aI7
aI3
aI35
aI127
aI69
aI39
aI265
aI1643
aI2973
aI267
aI12365
asS'd'
I1487
sa(dp2974
S'a'
I4106
sS's'
I14
sS'm'
(lp2975
I1
aI3
aI1
aI1
aI21
aI57
aI99
aI205
aI119
aI477
aI1771
aI1989
aI2761
aI12573
asS'd'
I1488
sa(dp2976
S'a'
I4119
sS's'
I14
sS'm'
(lp2977
I1
aI1
aI3
aI13
aI1
aI59
aI93
aI125
aI279
aI935
aI1877
aI2061
aI4845
aI7835
asS'd'
I1489
sa(dp2978
S'a'
I4126
sS's'
I14
sS'm'
(lp2979
I1
aI1
aI7
aI9
aI7
aI45
aI69
aI99
aI273
aI35
aI1579
aI2137
aI7175
aI6999
asS'd'
I1490
sa(dp2980
S'a'
I4147
sS's'
I14
sS'm'
(lp2981
I1
aI1
aI7
aI7
aI29
aI21
aI127
aI91
aI9
aI383
aI787
aI1783
aI601
aI5047
asS'd'
I1491
sa(dp2982
S'a'
I4149
sS's'
I14
sS'm'
(lp2983
I1
aI1
aI7
aI13
aI7
aI29
aI35
aI219
aI43
aI581
aI2043
aI2211
aI6169
aI12173
asS'd'
I1492
sa(dp2984
S'a'
I4164
sS's'
I14
sS'm'
(lp2985
I1
aI3
aI5
aI13
aI29
aI29
aI39
aI63
aI411
aI645
aI415
aI2383
aI1989
aI11411
asS'd'
I1493
sa(dp2986
S'a'
I4174
sS's'
I14
sS'm'
(lp2987
I1
aI1
aI7
aI9
aI15
aI9
aI87
aI95
aI321
aI709
aI543
aI3831
aI2453
aI4167
asS'd'
I1494
sa(dp2988
S'a'
I4181
sS's'
I14
sS'm'
(lp2989
I1
aI3
aI1
aI5
aI31
aI25
aI5
aI85
aI239
aI487
aI1613
aI3937
aI4661
aI3535
asS'd'
I1495
sa(dp2990
S'a'
I4185
sS's'
I14
sS'm'
(lp2991
I1
aI3
aI5
aI11
aI27
aI41
aI3
aI201
aI39
aI91
aI1997
aI237
aI5639
aI14703
asS'd'
I1496
sa(dp2992
S'a'
I4188
sS's'
I14
sS'm'
(lp2993
I1
aI1
aI7
aI3
aI27
aI49
aI87
aI71
aI473
aI247
aI1007
aI47
aI475
aI5413
asS'd'
I1497
sa(dp2994
S'a'
I4202
sS's'
I14
sS'm'
(lp2995
I1
aI3
aI7
aI15
aI9
aI57
aI81
aI149
aI287
aI333
aI1911
aI3417
aI1081
aI8995
asS'd'
I1498
sa(dp2996
S'a'
I4228
sS's'
I14
sS'm'
(lp2997
I1
aI1
aI5
aI1
aI3
aI63
aI43
aI151
aI97
aI431
aI961
aI1019
aI5153
aI2407
asS'd'
I1499
sa(dp2998
S'a'
I4232
sS's'
I14
sS'm'
(lp2999
I1
aI1
aI5
aI5
aI27
aI21
aI127
aI161
aI507
aI311
aI129
aI3489
aI1133
aI3241
asS'd'
I1500
sa(dp3000
S'a'
I4246
sS's'
I14
sS'm'
(lp3001
I1
aI3
aI7
aI15
aI21
aI33
aI117
aI83
aI497
aI667
aI1399
aI931
aI1695
aI8171
asS'd'
I1501
sa(dp3002
S'a'
I4252
sS's'
I14
sS'm'
(lp3003
I1
aI1
aI1
aI13
aI3
aI39
aI53
aI27
aI193
aI993
aI671
aI1871
aI7579
aI11457
asS'd'
I1502
sa(dp3004
S'a'
I4256
sS's'
I14
sS'm'
(lp3005
I1
aI1
aI5
aI11
aI7
aI39
aI81
aI107
aI195
aI387
aI849
aI395
aI1317
aI6487
asS'd'
I1503
sa(dp3006
S'a'
I4286
sS's'
I14
sS'm'
(lp3007
I1
aI3
aI3
aI3
aI3
aI15
aI45
aI127
aI279
aI111
aI331
aI357
aI4637
aI4697
asS'd'
I1504
sa(dp3008
S'a'
I4303
sS's'
I14
sS'm'
(lp3009
I1
aI1
aI3
aI9
aI21
aI49
aI47
aI97
aI61
aI101
aI181
aI1867
aI1201
aI14099
asS'd'
I1505
sa(dp3010
S'a'
I4306
sS's'
I14
sS'm'
(lp3011
I1
aI1
aI5
aI11
aI25
aI19
aI51
aI51
aI101
aI451
aI545
aI101
aI7497
aI9141
asS'd'
I1506
sa(dp3012
S'a'
I4311
sS's'
I14
sS'm'
(lp3013
I1
aI1
aI1
aI3
aI13
aI53
aI119
aI81
aI377
aI245
aI765
aI251
aI3757
aI16045
asS'd'
I1507
sa(dp3014
S'a'
I4317
sS's'
I14
sS'm'
(lp3015
I1
aI1
aI1
aI3
aI5
aI61
aI65
aI37
aI331
aI925
aI1439
aI3219
aI2843
aI11397
asS'd'
I1508
sa(dp3016
S'a'
I4342
sS's'
I14
sS'm'
(lp3017
I1
aI3
aI5
aI9
aI23
aI31
aI95
aI155
aI83
aI641
aI1129
aI135
aI477
aI1623
asS'd'
I1509
sa(dp3018
S'a'
I4346
sS's'
I14
sS'm'
(lp3019
I1
aI1
aI3
aI9
aI9
aI61
aI93
aI11
aI331
aI585
aI799
aI1417
aI1533
aI463
asS'd'
I1510
sa(dp3020
S'a'
I4377
sS's'
I14
sS'm'
(lp3021
I1
aI1
aI7
aI7
aI21
aI51
aI61
aI29
aI467
aI935
aI11
aI3357
aI1087
aI12337
asS'd'
I1511
sa(dp3022
S'a'
I4401
sS's'
I14
sS'm'
(lp3023
I1
aI3
aI3
aI11
aI1
aI39
aI103
aI153
aI351
aI893
aI1823
aI835
aI2149
aI4203
asS'd'
I1512
sa(dp3024
S'a'
I4407
sS's'
I14
sS'm'
(lp3025
I1
aI1
aI1
aI9
aI31
aI13
aI61
aI235
aI369
aI359
aI835
aI2067
aI2697
aI15289
asS'd'
I1513
sa(dp3026
S'a'
I4414
sS's'
I14
sS'm'
(lp3027
I1
aI1
aI7
aI1
aI15
aI1
aI107
aI27
aI201
aI451
aI1521
aI313
aI3195
aI3847
asS'd'
I1514
sa(dp3028
S'a'
I4422
sS's'
I14
sS'm'
(lp3029
I1
aI1
aI5
aI13
aI1
aI27
aI63
aI137
aI355
aI489
aI2039
aI1015
aI2519
aI13797
asS'd'
I1515
sa(dp3030
S'a'
I4431
sS's'
I14
sS'm'
(lp3031
I1
aI1
aI7
aI9
aI29
aI33
aI23
aI197
aI49
aI555
aI1087
aI3447
aI7299
aI15513
asS'd'
I1516
sa(dp3032
S'a'
I4434
sS's'
I14
sS'm'
(lp3033
I1
aI3
aI5
aI11
aI7
aI37
aI55
aI63
aI443
aI573
aI1715
aI631
aI3405
aI6155
asS'd'
I1517
sa(dp3034
S'a'
I4436
sS's'
I14
sS'm'
(lp3035
I1
aI3
aI3
aI3
aI31
aI35
aI51
aI167
aI225
aI617
aI2007
aI2555
aI6819
aI12709
asS'd'
I1518
sa(dp3036
S'a'
I4443
sS's'
I14
sS'm'
(lp3037
I1
aI1
aI1
aI13
aI15
aI5
aI73
aI85
aI109
aI43
aI1067
aI3941
aI1125
aI10269
asS'd'
I1519
sa(dp3038
S'a'
I4459
sS's'
I14
sS'm'
(lp3039
I1
aI1
aI7
aI11
aI17
aI3
aI127
aI145
aI279
aI19
aI1007
aI3287
aI4751
aI12507
asS'd'
I1520
sa(dp3040
S'a'
I4461
sS's'
I14
sS'm'
(lp3041
I1
aI3
aI7
aI3
aI19
aI1
aI117
aI111
aI193
aI435
aI47
aI1801
aI529
aI8547
asS'd'
I1521
sa(dp3042
S'a'
I4462
sS's'
I14
sS'm'
(lp3043
I1
aI3
aI3
aI13
aI1
aI19
aI101
aI19
aI469
aI187
aI207
aI1243
aI8153
aI3273
asS'd'
I1522
sa(dp3044
S'a'
I4473
sS's'
I14
sS'm'
(lp3045
I1
aI3
aI1
aI5
aI11
aI51
aI69
aI189
aI453
aI775
aI241
aI3331
aI4067
aI14759
asS'd'
I1523
sa(dp3046
S'a'
I4497
sS's'
I14
sS'm'
(lp3047
I1
aI1
aI1
aI1
aI23
aI55
aI113
aI133
aI497
aI731
aI391
aI2777
aI3529
aI955
asS'd'
I1524
sa(dp3048
S'a'
I4504
sS's'
I14
sS'm'
(lp3049
I1
aI3
aI1
aI11
aI5
aI49
aI59
aI35
aI261
aI949
aI325
aI3595
aI7433
aI11099
asS'd'
I1525
sa(dp3050
S'a'
I4507
sS's'
I14
sS'm'
(lp3051
I1
aI3
aI5
aI9
aI13
aI37
aI103
aI219
aI329
aI865
aI1787
aI2497
aI7249
aI9877
asS'd'
I1526
sa(dp3052
S'a'
I4525
sS's'
I14
sS'm'
(lp3053
I1
aI3
aI7
aI9
aI11
aI33
aI19
aI255
aI191
aI935
aI1115
aI1901
aI1577
aI9623
asS'd'
I1527
sa(dp3054
S'a'
I4534
sS's'
I14
sS'm'
(lp3055
I1
aI1
aI5
aI7
aI29
aI23
aI77
aI43
aI283
aI143
aI1211
aI73
aI2835
aI10235
asS'd'
I1528
sa(dp3056
S'a'
I4538
sS's'
I14
sS'm'
(lp3057
I1
aI1
aI7
aI3
aI3
aI27
aI35
aI173
aI453
aI425
aI1225
aI3023
aI2159
aI8433
asS'd'
I1529
sa(dp3058
S'a'
I4548
sS's'
I14
sS'm'
(lp3059
I1
aI1
aI1
aI5
aI27
aI21
aI35
aI25
aI71
aI145
aI1545
aI523
aI4527
aI7655
asS'd'
I1530
sa(dp3060
S'a'
I4552
sS's'
I14
sS'm'
(lp3061
I1
aI1
aI5
aI3
aI13
aI49
aI61
aI157
aI113
aI775
aI763
aI1785
aI225
aI11851
asS'd'
I1531
sa(dp3062
S'a'
I4560
sS's'
I14
sS'm'
(lp3063
I1
aI1
aI3
aI1
aI31
aI57
aI97
aI229
aI291
aI777
aI213
aI4067
aI921
aI8203
asS'd'
I1532
sa(dp3064
S'a'
I4575
sS's'
I14
sS'm'
(lp3065
I1
aI1
aI5
aI1
aI25
aI13
aI125
aI123
aI263
aI207
aI119
aI3111
aI3841
aI843
asS'd'
I1533
sa(dp3066
S'a'
I4599
sS's'
I14
sS'm'
(lp3067
I1
aI1
aI7
aI7
aI25
aI57
aI81
aI129
aI31
aI133
aI1869
aI2949
aI5563
aI14965
asS'd'
I1534
sa(dp3068
S'a'
I4612
sS's'
I14
sS'm'
(lp3069
I1
aI3
aI3
aI7
aI3
aI51
aI33
aI127
aI281
aI425
aI1253
aI405
aI7941
aI8799
asS'd'
I1535
sa(dp3070
S'a'
I4619
sS's'
I14
sS'm'
(lp3071
I1
aI1
aI3
aI9
aI3
aI63
aI93
aI173
aI255
aI609
aI49
aI111
aI7785
aI15865
asS'd'
I1536
sa(dp3072
S'a'
I4640
sS's'
I14
sS'm'
(lp3073
I1
aI1
aI1
aI3
aI17
aI59
aI113
aI55
aI155
aI789
aI1335
aI177
aI3071
aI1851
asS'd'
I1537
sa(dp3074
S'a'
I4643
sS's'
I14
sS'm'
(lp3075
I1
aI3
aI7
aI15
aI15
aI23
aI35
aI35
aI131
aI623
aI47
aI437
aI1337
aI9891
asS'd'
I1538
sa(dp3076
S'a'
I4677
sS's'
I14
sS'm'
(lp3077
I1
aI3
aI7
aI5
aI29
aI57
aI39
aI31
aI111
aI271
aI59
aI1473
aI949
aI3899
asS'd'
I1539
sa(dp3078
S'a'
I4687
sS's'
I14
sS'm'
(lp3079
I1
aI1
aI3
aI11
aI17
aI19
aI41
aI229
aI259
aI691
aI1455
aI3023
aI7455
aI9711
asS'd'
I1540
sa(dp3080
S'a'
I4723
sS's'
I14
sS'm'
(lp3081
I1
aI3
aI5
aI11
aI29
aI13
aI9
aI165
aI499
aI355
aI1415
aI1395
aI7595
aI15571
asS'd'
I1541
sa(dp3082
S'a'
I4730
sS's'
I14
sS'm'
(lp3083
I1
aI3
aI1
aI9
aI5
aI5
aI25
aI247
aI185
aI241
aI1325
aI3133
aI7471
aI2649
asS'd'
I1542
sa(dp3084
S'a'
I4736
sS's'
I14
sS'm'
(lp3085
I1
aI3
aI3
aI11
aI17
aI29
aI57
aI61
aI51
aI203
aI993
aI1837
aI3785
aI15163
asS'd'
I1543
sa(dp3086
S'a'
I4741
sS's'
I14
sS'm'
(lp3087
I1
aI1
aI7
aI7
aI21
aI57
aI79
aI165
aI277
aI133
aI93
aI1055
aI7169
aI15685
asS'd'
I1544
sa(dp3088
S'a'
I4763
sS's'
I14
sS'm'
(lp3089
I1
aI1
aI5
aI3
aI5
aI17
aI25
aI177
aI95
aI323
aI367
aI1359
aI4915
aI6409
asS'd'
I1545
sa(dp3090
S'a'
I4765
sS's'
I14
sS'm'
(lp3091
I1
aI1
aI1
aI1
aI11
aI25
aI115
aI45
aI373
aI221
aI1483
aI591
aI6561
aI4527
asS'd'
I1546
sa(dp3092
S'a'
I4770
sS's'
I14
sS'm'
(lp3093
I1
aI3
aI5
aI3
aI5
aI23
aI69
aI77
aI313
aI473
aI1037
aI4045
aI3969
aI5445
asS'd'
I1547
sa(dp3094
S'a'
I4781
sS's'
I14
sS'm'
(lp3095
I1
aI3
aI1
aI5
aI1
aI15
aI73
aI83
aI439
aI463
aI203
aI361
aI6835
aI1061
asS'd'
I1548
sa(dp3096
S'a'
I4808
sS's'
I14
sS'm'
(lp3097
I1
aI1
aI3
aI11
aI21
aI5
aI89
aI233
aI405
aI253
aI773
aI3901
aI6085
aI5677
asS'd'
I1549
sa(dp3098
S'a'
I4822
sS's'
I14
sS'm'
(lp3099
I1
aI1
aI3
aI9
aI15
aI53
aI71
aI29
aI101
aI599
aI1073
aI705
aI4507
aI12779
asS'd'
I1550
sa(dp3100
S'a'
I4828
sS's'
I14
sS'm'
(lp3101
I1
aI1
aI3
aI1
aI3
aI9
aI27
aI97
aI207
aI859
aI417
aI735
aI2179
aI5071
asS'd'
I1551
sa(dp3102
S'a'
I4831
sS's'
I14
sS'm'
(lp3103
I1
aI1
aI1
aI3
aI13
aI63
aI65
aI125
aI195
aI611
aI649
aI2221
aI3143
aI143
asS'd'
I1552
sa(dp3104
S'a'
I4842
sS's'
I14
sS'm'
(lp3105
I1
aI3
aI3
aI15
aI17
aI57
aI99
aI119
aI243
aI407
aI1229
aI813
aI5245
aI1893
asS'd'
I1553
sa(dp3106
S'a'
I4855
sS's'
I14
sS'm'
(lp3107
I1
aI1
aI1
aI5
aI27
aI27
aI49
aI13
aI313
aI287
aI473
aI2629
aI3509
aI11371
asS'd'
I1554
sa(dp3108
S'a'
I4859
sS's'
I14
sS'm'
(lp3109
I1
aI1
aI7
aI7
aI23
aI3
aI75
aI59
aI245
aI689
aI1215
aI2375
aI3325
aI1593
asS'd'
I1555
sa(dp3110
S'a'
I4867
sS's'
I14
sS'm'
(lp3111
I1
aI3
aI1
aI5
aI21
aI51
aI43
aI107
aI91
aI611
aI1405
aI677
aI2087
aI9565
asS'd'
I1556
sa(dp3112
S'a'
I4870
sS's'
I14
sS'm'
(lp3113
I1
aI3
aI7
aI11
aI9
aI27
aI81
aI101
aI449
aI201
aI1507
aI2217
aI6767
aI8059
asS'd'
I1557
sa(dp3114
S'a'
I4881
sS's'
I14
sS'm'
(lp3115
I1
aI1
aI3
aI9
aI13
aI41
aI21
aI195
aI421
aI315
aI347
aI2621
aI2359
aI9247
asS'd'
I1558
sa(dp3116
S'a'
I4893
sS's'
I14
sS'm'
(lp3117
I1
aI1
aI5
aI7
aI31
aI45
aI77
aI229
aI455
aI575
aI1087
aI1147
aI2273
aI13773
asS'd'
I1559
sa(dp3118
S'a'
I4910
sS's'
I14
sS'm'
(lp3119
I1
aI1
aI1
aI1
aI9
aI5
aI87
aI19
aI207
aI545
aI1435
aI495
aI1299
aI4947
asS'd'
I1560
sa(dp3120
S'a'
I4917
sS's'
I14
sS'm'
(lp3121
I1
aI1
aI3
aI3
aI15
aI9
aI63
aI67
aI219
aI735
aI1911
aI2361
aI6503
aI11977
asS'd'
I1561
sa(dp3122
S'a'
I4929
sS's'
I14
sS'm'
(lp3123
I1
aI3
aI1
aI9
aI31
aI27
aI103
aI153
aI81
aI939
aI461
aI2753
aI697
aI537
asS'd'
I1562
sa(dp3124
S'a'
I4939
sS's'
I14
sS'm'
(lp3125
I1
aI3
aI3
aI9
aI21
aI53
aI49
aI211
aI415
aI817
aI321
aI3775
aI2921
aI9473
asS'd'
I1563
sa(dp3126
S'a'
I4947
sS's'
I14
sS'm'
(lp3127
I1
aI1
aI7
aI3
aI23
aI55
aI15
aI51
aI435
aI1013
aI73
aI3967
aI4575
aI13099
asS'd'
I1564
sa(dp3128
S'a'
I4949
sS's'
I14
sS'm'
(lp3129
I1
aI1
aI3
aI7
aI5
aI27
aI43
aI225
aI267
aI21
aI1261
aI603
aI6913
aI4421
asS'd'
I1565
sa(dp3130
S'a'
I4954
sS's'
I14
sS'm'
(lp3131
I1
aI1
aI7
aI13
aI25
aI31
aI101
aI109
aI237
aI91
aI1587
aI1987
aI2795
aI6991
asS'd'
I1566
sa(dp3132
S'a'
I4972
sS's'
I14
sS'm'
(lp3133
I1
aI1
aI3
aI13
aI23
aI51
aI91
aI89
aI287
aI39
aI1513
aI463
aI6135
aI10469
asS'd'
I1567
sa(dp3134
S'a'
I4975
sS's'
I14
sS'm'
(lp3135
I1
aI3
aI3
aI1
aI9
aI43
aI125
aI157
aI369
aI495
aI1849
aI785
aI6357
aI6557
asS'd'
I1568
sa(dp3136
S'a'
I5000
sS's'
I14
sS'm'
(lp3137
I1
aI3
aI1
aI13
aI5
aI25
aI107
aI139
aI367
aI239
aI1671
aI1239
aI7027
aI5291
asS'd'
I1569
sa(dp3138
S'a'
I5005
sS's'
I14
sS'm'
(lp3139
I1
aI3
aI5
aI13
aI11
aI13
aI35
aI177
aI45
aI939
aI251
aI59
aI333
aI13105
asS'd'
I1570
sa(dp3140
S'a'
I5029
sS's'
I14
sS'm'
(lp3141
I1
aI3
aI5
aI7
aI29
aI57
aI109
aI227
aI435
aI739
aI423
aI1941
aI3345
aI12731
asS'd'
I1571
sa(dp3142
S'a'
I5039
sS's'
I14
sS'm'
(lp3143
I1
aI3
aI3
aI9
aI23
aI51
aI19
aI207
aI69
aI99
aI955
aI519
aI7305
aI2415
asS'd'
I1572
sa(dp3144
S'a'
I5044
sS's'
I14
sS'm'
(lp3145
I1
aI1
aI5
aI13
aI17
aI1
aI67
aI201
aI61
aI403
aI1059
aI2915
aI2419
aI12773
asS'd'
I1573
sa(dp3146
S'a'
I5051
sS's'
I14
sS'm'
(lp3147
I1
aI3
aI1
aI11
aI17
aI19
aI25
aI27
aI207
aI299
aI143
aI1955
aI5669
aI2301
asS'd'
I1574
sa(dp3148
S'a'
I5056
sS's'
I14
sS'm'
(lp3149
I1
aI1
aI5
aI3
aI25
aI57
aI45
aI255
aI489
aI1011
aI1699
aI2637
aI5279
aI12211
asS'd'
I1575
sa(dp3150
S'a'
I5073
sS's'
I14
sS'm'
(lp3151
I1
aI3
aI3
aI15
aI7
aI47
aI113
aI33
aI511
aI907
aI1815
aI1741
aI2091
aI13857
asS'd'
I1576
sa(dp3152
S'a'
I5096
sS's'
I14
sS'm'
(lp3153
I1
aI3
aI3
aI5
aI5
aI27
aI95
aI3
aI353
aI253
aI947
aI393
aI1815
aI14551
asS'd'
I1577
sa(dp3154
S'a'
I5128
sS's'
I14
sS'm'
(lp3155
I1
aI1
aI5
aI11
aI29
aI19
aI63
aI117
aI293
aI861
aI2039
aI9
aI5999
aI6909
asS'd'
I1578
sa(dp3156
S'a'
I5134
sS's'
I14
sS'm'
(lp3157
I1
aI3
aI7
aI3
aI15
aI63
aI107
aI173
aI509
aI817
aI99
aI2825
aI131
aI7917
asS'd'
I1579
sa(dp3158
S'a'
I5161
sS's'
I14
sS'm'
(lp3159
I1
aI3
aI1
aI1
aI29
aI49
aI33
aI153
aI119
aI777
aI1315
aI3581
aI5675
aI4043
asS'd'
I1580
sa(dp3160
S'a'
I5179
sS's'
I14
sS'm'
(lp3161
I1
aI3
aI5
aI15
aI13
aI11
aI17
aI147
aI327
aI305
aI367
aI3237
aI5423
aI13757
asS'd'
I1581
sa(dp3162
S'a'
I5193
sS's'
I14
sS'm'
(lp3163
I1
aI1
aI5
aI13
aI1
aI39
aI35
aI29
aI25
aI751
aI1365
aI2243
aI8181
aI7063
asS'd'
I1582
sa(dp3164
S'a'
I5199
sS's'
I14
sS'm'
(lp3165
I1
aI3
aI7
aI11
aI25
aI53
aI11
aI111
aI289
aI755
aI1201
aI691
aI3679
aI3725
asS'd'
I1583
sa(dp3166
S'a'
I5202
sS's'
I14
sS'm'
(lp3167
I1
aI1
aI1
aI11
aI11
aI37
aI33
aI211
aI395
aI691
aI1817
aI861
aI6485
aI12077
asS'd'
I1584
sa(dp3168
S'a'
I5204
sS's'
I14
sS'm'
(lp3169
I1
aI3
aI3
aI11
aI21
aI3
aI111
aI171
aI305
aI561
aI1501
aI2011
aI7841
aI10931
asS'd'
I1585
sa(dp3170
S'a'
I5218
sS's'
I14
sS'm'
(lp3171
I1
aI3
aI7
aI9
aI9
aI59
aI109
aI113
aI31
aI915
aI103
aI1861
aI2779
aI10619
asS'd'
I1586
sa(dp3172
S'a'
I5247
sS's'
I14
sS'm'
(lp3173
I1
aI1
aI1
aI1
aI7
aI25
aI61
aI97
aI103
aI723
aI1145
aI3105
aI371
aI339
asS'd'
I1587
sa(dp3174
S'a'
I5260
sS's'
I14
sS'm'
(lp3175
I1
aI1
aI7
aI13
aI17
aI9
aI113
aI51
aI233
aI209
aI1117
aI211
aI6969
aI2347
asS'd'
I1588
sa(dp3176
S'a'
I5271
sS's'
I14
sS'm'
(lp3177
I1
aI1
aI5
aI9
aI25
aI43
aI21
aI217
aI327
aI735
aI197
aI1063
aI799
aI801
asS'd'
I1589
sa(dp3178
S'a'
I5301
sS's'
I14
sS'm'
(lp3179
I1
aI1
aI7
aI13
aI9
aI13
aI73
aI33
aI415
aI923
aI863
aI1999
aI5383
aI8119
asS'd'
I1590
sa(dp3180
S'a'
I5305
sS's'
I14
sS'm'
(lp3181
I1
aI3
aI1
aI5
aI7
aI33
aI51
aI185
aI289
aI967
aI1277
aI1011
aI767
aI15505
asS'd'
I1591
sa(dp3182
S'a'
I5319
sS's'
I14
sS'm'
(lp3183
I1
aI3
aI3
aI13
aI21
aI11
aI105
aI235
aI343
aI1021
aI2009
aI2251
aI3865
aI6923
asS'd'
I1592
sa(dp3184
S'a'
I5326
sS's'
I14
sS'm'
(lp3185
I1
aI3
aI5
aI9
aI29
aI11
aI33
aI17
aI149
aI155
aI1739
aI3039
aI7015
aI2401
asS'd'
I1593
sa(dp3186
S'a'
I5328
sS's'
I14
sS'm'
(lp3187
I1
aI3
aI7
aI7
aI17
aI13
aI89
aI177
aI297
aI267
aI545
aI3861
aI329
aI13267
asS'd'
I1594
sa(dp3188
S'a'
I5333
sS's'
I14
sS'm'
(lp3189
I1
aI3
aI5
aI15
aI27
aI33
aI1
aI231
aI181
aI557
aI447
aI379
aI7845
aI1295
asS'd'
I1595
sa(dp3190
S'a'
I5364
sS's'
I14
sS'm'
(lp3191
I1
aI1
aI5
aI13
aI3
aI63
aI59
aI33
aI263
aI877
aI1867
aI1383
aI641
aI7139
asS'd'
I1596
sa(dp3192
S'a'
I5376
sS's'
I14
sS'm'
(lp3193
I1
aI3
aI7
aI5
aI13
aI51
aI9
aI113
aI223
aI605
aI1189
aI4063
aI6925
aI9563
asS'd'
I1597
sa(dp3194
S'a'
I5399
sS's'
I14
sS'm'
(lp3195
I1
aI1
aI1
aI13
aI5
aI35
aI83
aI107
aI295
aI231
aI265
aI5
aI4087
aI6407
asS'd'
I1598
sa(dp3196
S'a'
I5416
sS's'
I14
sS'm'
(lp3197
I1
aI1
aI5
aI1
aI7
aI25
aI95
aI137
aI97
aI987
aI1753
aI2781
aI1369
aI6903
asS'd'
I1599
sa(dp3198
S'a'
I5421
sS's'
I14
sS'm'
(lp3199
I1
aI1
aI5
aI13
aI19
aI61
aI77
aI229
aI193
aI165
aI811
aI249
aI79
aI10719
asS'd'
I1600
sa(dp3200
S'a'
I5427
sS's'
I14
sS'm'
(lp3201
I1
aI3
aI7
aI7
aI27
aI9
aI119
aI193
aI459
aI43
aI1989
aI2959
aI3595
aI6341
asS'd'
I1601
sa(dp3202
S'a'
I5429
sS's'
I14
sS'm'
(lp3203
I1
aI1
aI5
aI11
aI5
aI43
aI35
aI33
aI25
aI581
aI897
aI351
aI4201
aI3971
asS'd'
I1602
sa(dp3204
S'a'
I5430
sS's'
I14
sS'm'
(lp3205
I1
aI1
aI7
aI11
aI21
aI29
aI53
aI45
aI359
aI197
aI313
aI3825
aI6717
aI4077
asS'd'
I1603
sa(dp3206
S'a'
I5434
sS's'
I14
sS'm'
(lp3207
I1
aI1
aI1
aI15
aI3
aI45
aI99
aI133
aI357
aI315
aI1159
aI241
aI2463
aI11253
asS'd'
I1604
sa(dp3208
S'a'
I5441
sS's'
I14
sS'm'
(lp3209
I1
aI1
aI7
aI11
aI9
aI33
aI111
aI85
aI443
aI601
aI447
aI337
aI6471
aI7029
asS'd'
I1605
sa(dp3210
S'a'
I5451
sS's'
I14
sS'm'
(lp3211
I1
aI3
aI7
aI9
aI13
aI33
aI25
aI31
aI9
aI729
aI1763
aI4077
aI7575
aI7877
asS'd'
I1606
sa(dp3212
S'a'
I5465
sS's'
I14
sS'm'
(lp3213
I1
aI3
aI5
aI13
aI13
aI37
aI29
aI103
aI53
aI229
aI591
aI1073
aI1323
aI14405
asS'd'
I1607
sa(dp3214
S'a'
I5466
sS's'
I14
sS'm'
(lp3215
I1
aI1
aI5
aI1
aI17
aI33
aI15
aI183
aI473
aI297
aI2003
aI93
aI4955
aI1787
asS'd'
I1608
sa(dp3216
S'a'
I5471
sS's'
I14
sS'm'
(lp3217
I1
aI1
aI5
aI13
aI5
aI29
aI113
aI161
aI267
aI451
aI1193
aI149
aI273
aI11809
asS'd'
I1609
sa(dp3218
S'a'
I5477
sS's'
I14
sS'm'
(lp3219
I1
aI1
aI1
aI9
aI17
aI39
aI47
aI233
aI165
aI373
aI955
aI2891
aI7523
aI7235
asS'd'
I1610
sa(dp3220
S'a'
I5492
sS's'
I14
sS'm'
(lp3221
I1
aI1
aI1
aI3
aI7
aI21
aI115
aI205
aI153
aI449
aI339
aI2073
aI1077
aI5749
asS'd'
I1611
sa(dp3222
S'a'
I5495
sS's'
I14
sS'm'
(lp3223
I1
aI1
aI7
aI13
aI9
aI39
aI117
aI187
aI37
aI753
aI227
aI3519
aI7391
aI5751
asS'd'
I1612
sa(dp3224
S'a'
I5505
sS's'
I14
sS'm'
(lp3225
I1
aI1
aI1
aI9
aI5
aI19
aI41
aI161
aI141
aI195
aI1719
aI3321
aI5
aI12877
asS'd'
I1613
sa(dp3226
S'a'
I5515
sS's'
I14
sS'm'
(lp3227
I1
aI3
aI7
aI11
aI21
aI13
aI83
aI55
aI299
aI75
aI1905
aI3765
aI4685
aI12297
asS'd'
I1614
sa(dp3228
S'a'
I5525
sS's'
I14
sS'm'
(lp3229
I1
aI1
aI7
aI3
aI3
aI23
aI111
aI243
aI187
aI297
aI1061
aI2515
aI977
aI9555
asS'd'
I1615
sa(dp3230
S'a'
I5529
sS's'
I14
sS'm'
(lp3231
I1
aI3
aI7
aI3
aI29
aI11
aI103
aI177
aI225
aI875
aI1649
aI1401
aI6383
aI8309
asS'd'
I1616
sa(dp3232
S'a'
I5532
sS's'
I14
sS'm'
(lp3233
I1
aI3
aI5
aI3
aI3
aI41
aI71
aI3
aI373
aI757
aI701
aI2825
aI1521
aI13217
asS'd'
I1617
sa(dp3234
S'a'
I5539
sS's'
I14
sS'm'
(lp3235
I1
aI1
aI5
aI3
aI11
aI5
aI103
aI227
aI209
aI723
aI1543
aI3895
aI6345
aI7901
asS'd'
I1618
sa(dp3236
S'a'
I5541
sS's'
I14
sS'm'
(lp3237
I1
aI1
aI5
aI1
aI9
aI51
aI77
aI67
aI359
aI937
aI557
aI993
aI3871
aI3577
asS'd'
I1619
sa(dp3238
S'a'
I5556
sS's'
I14
sS'm'
(lp3239
I1
aI3
aI7
aI1
aI1
aI15
aI121
aI239
aI29
aI113
aI1123
aI3877
aI6941
aI14129
asS'd'
I1620
sa(dp3240
S'a'
I5566
sS's'
I14
sS'm'
(lp3241
I1
aI1
aI5
aI1
aI27
aI61
aI83
aI113
aI185
aI601
aI947
aI3933
aI381
aI13869
asS'd'
I1621
sa(dp3242
S'a'
I5568
sS's'
I14
sS'm'
(lp3243
I1
aI1
aI5
aI3
aI5
aI37
aI97
aI31
aI81
aI367
aI747
aI1811
aI5313
aI14151
asS'd'
I1622
sa(dp3244
S'a'
I5574
sS's'
I14
sS'm'
(lp3245
I1
aI3
aI5
aI9
aI27
aI61
aI87
aI31
aI185
aI521
aI837
aI959
aI5001
aI3957
asS'd'
I1623
sa(dp3246
S'a'
I5595
sS's'
I14
sS'm'
(lp3247
I1
aI3
aI5
aI3
aI11
aI61
aI37
aI19
aI107
aI749
aI1345
aI3829
aI6701
aI4315
asS'd'
I1624
sa(dp3248
S'a'
I5602
sS's'
I14
sS'm'
(lp3249
I1
aI3
aI1
aI15
aI13
aI45
aI101
aI113
aI243
aI963
aI1861
aI3283
aI1419
aI12131
asS'd'
I1625
sa(dp3250
S'a'
I5611
sS's'
I14
sS'm'
(lp3251
I1
aI1
aI7
aI1
aI11
aI63
aI17
aI117
aI271
aI819
aI677
aI669
aI1991
aI12511
asS'd'
I1626
sa(dp3252
S'a'
I5616
sS's'
I14
sS'm'
(lp3253
I1
aI1
aI1
aI13
aI13
aI33
aI41
aI73
aI187
aI537
aI993
aI3147
aI1013
aI16063
asS'd'
I1627
sa(dp3254
S'a'
I5622
sS's'
I14
sS'm'
(lp3255
I1
aI3
aI1
aI1
aI25
aI21
aI107
aI81
aI117
aI917
aI113
aI349
aI4475
aI9149
asS'd'
I1628
sa(dp3256
S'a'
I5628
sS's'
I14
sS'm'
(lp3257
I1
aI1
aI1
aI11
aI21
aI21
aI29
aI251
aI125
aI681
aI141
aI2893
aI5843
aI14359
asS'd'
I1629
sa(dp3258
S'a'
I5655
sS's'
I14
sS'm'
(lp3259
I1
aI3
aI3
aI1
aI5
aI41
aI85
aI163
aI387
aI29
aI1593
aI221
aI2769
aI10809
asS'd'
I1630
sa(dp3260
S'a'
I5662
sS's'
I14
sS'm'
(lp3261
I1
aI3
aI5
aI11
aI1
aI17
aI69
aI127
aI273
aI449
aI1855
aI2971
aI7031
aI10583
asS'd'
I1631
sa(dp3262
S'a'
I5675
sS's'
I14
sS'm'
(lp3263
I1
aI1
aI5
aI7
aI1
aI61
aI9
aI211
aI123
aI563
aI111
aI1883
aI5801
aI2191
asS'd'
I1632
sa(dp3264
S'a'
I5689
sS's'
I14
sS'm'
(lp3265
I1
aI1
aI3
aI11
aI11
aI51
aI1
aI81
aI405
aI803
aI2017
aI161
aI5429
aI731
asS'd'
I1633
sa(dp3266
S'a'
I5722
sS's'
I14
sS'm'
(lp3267
I1
aI1
aI7
aI9
aI15
aI55
aI65
aI51
aI459
aI485
aI1539
aI3135
aI2929
aI7867
asS'd'
I1634
sa(dp3268
S'a'
I5724
sS's'
I14
sS'm'
(lp3269
I1
aI1
aI7
aI11
aI3
aI45
aI15
aI7
aI331
aI417
aI1813
aI4009
aI1341
aI10965
asS'd'
I1635
sa(dp3270
S'a'
I5728
sS's'
I14
sS'm'
(lp3271
I1
aI1
aI1
aI5
aI9
aI29
aI89
aI121
aI277
aI509
aI1989
aI1293
aI4787
aI16097
asS'd'
I1636
sa(dp3272
S'a'
I5731
sS's'
I14
sS'm'
(lp3273
I1
aI1
aI3
aI9
aI17
aI45
aI97
aI197
aI339
aI943
aI1377
aI2947
aI5833
aI7
asS'd'
I1637
sa(dp3274
S'a'
I5746
sS's'
I14
sS'm'
(lp3275
I1
aI1
aI7
aI9
aI15
aI61
aI75
aI233
aI401
aI705
aI825
aI2521
aI3787
aI14387
asS'd'
I1638
sa(dp3276
S'a'
I5764
sS's'
I14
sS'm'
(lp3277
I1
aI1
aI7
aI15
aI25
aI57
aI3
aI43
aI361
aI459
aI1551
aI1859
aI6787
aI2293
asS'd'
I1639
sa(dp3278
S'a'
I5771
sS's'
I14
sS'm'
(lp3279
I1
aI3
aI3
aI11
aI11
aI35
aI91
aI65
aI43
aI509
aI1829
aI1149
aI4801
aI4109
asS'd'
I1640
sa(dp3280
S'a'
I5781
sS's'
I14
sS'm'
(lp3281
I1
aI3
aI5
aI9
aI15
aI3
aI81
aI109
aI231
aI481
aI417
aI2505
aI315
aI6693
asS'd'
I1641
sa(dp3282
S'a'
I5801
sS's'
I14
sS'm'
(lp3283
I1
aI1
aI3
aI9
aI3
aI7
aI107
aI221
aI297
aI543
aI149
aI579
aI927
aI79
asS'd'
I1642
sa(dp3284
S'a'
I5809
sS's'
I14
sS'm'
(lp3285
I1
aI3
aI1
aI11
aI17
aI3
aI81
aI137
aI157
aI587
aI741
aI1277
aI2631
aI3953
asS'd'
I1643
sa(dp3286
S'a'
I5810
sS's'
I14
sS'm'
(lp3287
I1
aI1
aI7
aI5
aI13
aI43
aI117
aI19
aI495
aI185
aI1105
aI605
aI5249
aI11099
asS'd'
I1644
sa(dp3288
S'a'
I5812
sS's'
I14
sS'm'
(lp3289
I1
aI1
aI7
aI9
aI23
aI55
aI91
aI213
aI21
aI779
aI857
aI2047
aI7813
aI10053
asS'd'
I1645
sa(dp3290
S'a'
I5841
sS's'
I14
sS'm'
(lp3291
I1
aI1
aI1
aI1
aI27
aI7
aI39
aI181
aI63
aI519
aI1073
aI3147
aI4111
aI363
asS'd'
I1646
sa(dp3292
S'a'
I5848
sS's'
I14
sS'm'
(lp3293
I1
aI3
aI7
aI9
aI15
aI61
aI7
aI139
aI495
aI805
aI1545
aI3789
aI2411
aI3989
asS'd'
I1647
sa(dp3294
S'a'
I5853
sS's'
I14
sS'm'
(lp3295
I1
aI1
aI3
aI1
aI25
aI11
aI23
aI241
aI167
aI607
aI479
aI153
aI7787
aI13929
asS'd'
I1648
sa(dp3296
S'a'
I5854
sS's'
I14
sS'm'
(lp3297
I1
aI3
aI5
aI15
aI29
aI35
aI45
aI71
aI457
aI297
aI883
aI3021
aI5361
aI15427
asS'd'
I1649
sa(dp3298
S'a'
I5858
sS's'
I14
sS'm'
(lp3299
I1
aI3
aI1
aI7
aI29
aI27
aI93
aI241
aI427
aI89
aI1185
aI37
aI3863
aI14095
asS'd'
I1650
sa(dp3300
S'a'
I5892
sS's'
I14
sS'm'
(lp3301
I1
aI3
aI1
aI5
aI5
aI45
aI51
aI15
aI235
aI889
aI1649
aI2331
aI2713
aI10943
asS'd'
I1651
sa(dp3302
S'a'
I5907
sS's'
I14
sS'm'
(lp3303
I1
aI1
aI3
aI11
aI11
aI15
aI71
aI85
aI135
aI163
aI139
aI1147
aI1043
aI3195
asS'd'
I1652
sa(dp3304
S'a'
I5910
sS's'
I14
sS'm'
(lp3305
I1
aI3
aI5
aI13
aI3
aI43
aI71
aI131
aI473
aI933
aI569
aI2491
aI7751
aI1865
asS'd'
I1653
sa(dp3306
S'a'
I5913
sS's'
I14
sS'm'
(lp3307
I1
aI1
aI7
aI9
aI21
aI37
aI105
aI227
aI329
aI509
aI1319
aI307
aI1557
aI14625
asS'd'
I1654
sa(dp3308
S'a'
I5920
sS's'
I14
sS'm'
(lp3309
I1
aI1
aI3
aI13
aI15
aI1
aI25
aI93
aI335
aI953
aI769
aI4039
aI369
aI10727
asS'd'
I1655
sa(dp3310
S'a'
I5929
sS's'
I14
sS'm'
(lp3311
I1
aI3
aI7
aI5
aI17
aI21
aI59
aI89
aI437
aI679
aI437
aI1543
aI7663
aI5005
asS'd'
I1656
sa(dp3312
S'a'
I5949
sS's'
I14
sS'm'
(lp3313
I1
aI1
aI7
aI15
aI27
aI49
aI125
aI13
aI397
aI877
aI1087
aI2191
aI4711
aI9065
asS'd'
I1657
sa(dp3314
S'a'
I5952
sS's'
I14
sS'm'
(lp3315
I1
aI1
aI7
aI5
aI15
aI47
aI115
aI125
aI187
aI31
aI1003
aI2575
aI5397
aI3883
asS'd'
I1658
sa(dp3316
S'a'
I5955
sS's'
I14
sS'm'
(lp3317
I1
aI1
aI7
aI11
aI15
aI1
aI127
aI207
aI383
aI707
aI183
aI1053
aI3123
aI14071
asS'd'
I1659
sa(dp3318
S'a'
I5962
sS's'
I14
sS'm'
(lp3319
I1
aI3
aI3
aI1
aI31
aI53
aI15
aI19
aI477
aI245
aI777
aI1613
aI5813
aI7443
asS'd'
I1660
sa(dp3320
S'a'
I5975
sS's'
I14
sS'm'
(lp3321
I1
aI3
aI1
aI11
aI23
aI59
aI65
aI23
aI493
aI157
aI1389
aI2833
aI4535
aI3907
asS'd'
I1661
sa(dp3322
S'a'
I5985
sS's'
I14
sS'm'
(lp3323
I1
aI1
aI7
aI1
aI19
aI7
aI51
aI135
aI327
aI441
aI1841
aI3091
aI3451
aI14381
asS'd'
I1662
sa(dp3324
S'a'
I5997
sS's'
I14
sS'm'
(lp3325
I1
aI1
aI7
aI7
aI3
aI37
aI29
aI249
aI437
aI319
aI1693
aI945
aI7639
aI5923
asS'd'
I1663
sa(dp3326
S'a'
I5998
sS's'
I14
sS'm'
(lp3327
I1
aI3
aI7
aI15
aI7
aI61
aI81
aI127
aI383
aI99
aI23
aI3833
aI3973
aI7651
asS'd'
I1664
sa(dp3328
S'a'
I6012
sS's'
I14
sS'm'
(lp3329
I1
aI3
aI1
aI7
aI7
aI21
aI119
aI185
aI243
aI619
aI1363
aI2033
aI4835
aI5089
asS'd'
I1665
sa(dp3330
S'a'
I6016
sS's'
I14
sS'm'
(lp3331
I1
aI3
aI1
aI1
aI3
aI27
aI63
aI145
aI271
aI735
aI695
aI3981
aI3049
aI5433
asS'd'
I1666
sa(dp3332
S'a'
I6026
sS's'
I14
sS'm'
(lp3333
I1
aI3
aI3
aI1
aI3
aI29
aI79
aI211
aI279
aI819
aI501
aI3665
aI1455
aI10455
asS'd'
I1667
sa(dp3334
S'a'
I6036
sS's'
I14
sS'm'
(lp3335
I1
aI1
aI3
aI3
aI31
aI61
aI113
aI5
aI411
aI91
aI489
aI3257
aI5939
aI6715
asS'd'
I1668
sa(dp3336
S'a'
I6040
sS's'
I14
sS'm'
(lp3337
I1
aI1
aI5
aI1
aI23
aI11
aI103
aI89
aI377
aI441
aI43
aI967
aI3383
aI8717
asS'd'
I1669
sa(dp3338
S'a'
I6045
sS's'
I14
sS'm'
(lp3339
I1
aI1
aI5
aI13
aI29
aI39
aI97
aI189
aI197
aI621
aI1755
aI333
aI6783
aI9711
asS'd'
I1670
sa(dp3340
S'a'
I6055
sS's'
I14
sS'm'
(lp3341
I1
aI1
aI5
aI13
aI27
aI17
aI97
aI197
aI351
aI799
aI335
aI765
aI5329
aI12549
asS'd'
I1671
sa(dp3342
S'a'
I6059
sS's'
I14
sS'm'
(lp3343
I1
aI1
aI5
aI11
aI29
aI17
aI9
aI211
aI127
aI633
aI1187
aI3965
aI4145
aI12667
asS'd'
I1672
sa(dp3344
S'a'
I6088
sS's'
I14
sS'm'
(lp3345
I1
aI1
aI7
aI5
aI27
aI29
aI65
aI115
aI287
aI325
aI461
aI5
aI899
aI2027
asS'd'
I1673
sa(dp3346
S'a'
I6115
sS's'
I14
sS'm'
(lp3347
I1
aI1
aI1
aI5
aI27
aI17
aI31
aI13
aI231
aI627
aI1163
aI649
aI1693
aI9975
asS'd'
I1674
sa(dp3348
S'a'
I6124
sS's'
I14
sS'm'
(lp3349
I1
aI3
aI1
aI15
aI7
aI49
aI113
aI123
aI427
aI603
aI347
aI2785
aI7129
aI4645
asS'd'
I1675
sa(dp3350
S'a'
I6127
sS's'
I14
sS'm'
(lp3351
I1
aI1
aI3
aI7
aI1
aI33
aI113
aI105
aI411
aI939
aI205
aI3965
aI4361
aI4649
asS'd'
I1676
sa(dp3352
S'a'
I6132
sS's'
I14
sS'm'
(lp3353
I1
aI1
aI1
aI1
aI5
aI21
aI35
aI159
aI275
aI929
aI1193
aI3205
aI4787
aI3515
asS'd'
I1677
sa(dp3354
S'a'
I6146
sS's'
I14
sS'm'
(lp3355
I1
aI1
aI1
aI5
aI1
aI21
aI29
aI191
aI275
aI233
aI1239
aI515
aI4349
aI14989
asS'd'
I1678
sa(dp3356
S'a'
I6158
sS's'
I14
sS'm'
(lp3357
I1
aI1
aI5
aI11
aI27
aI43
aI111
aI83
aI153
aI577
aI1537
aI149
aI231
aI839
asS'd'
I1679
sa(dp3358
S'a'
I6169
sS's'
I14
sS'm'
(lp3359
I1
aI3
aI5
aI13
aI21
aI19
aI57
aI69
aI87
aI163
aI271
aI3535
aI1057
aI8517
asS'd'
I1680
sa(dp3360
S'a'
I6206
sS's'
I14
sS'm'
(lp3361
I1
aI3
aI3
aI13
aI17
aI49
aI65
aI45
aI457
aI241
aI391
aI2033
aI2507
aI7771
asS'd'
I1681
sa(dp3362
S'a'
I6228
sS's'
I14
sS'm'
(lp3363
I1
aI1
aI5
aI7
aI11
aI19
aI79
aI133
aI341
aI761
aI27
aI3905
aI4137
aI14363
asS'd'
I1682
sa(dp3364
S'a'
I6237
sS's'
I14
sS'm'
(lp3365
I1
aI3
aI3
aI13
aI19
aI1
aI11
aI139
aI249
aI245
aI1393
aI2151
aI2857
aI1665
asS'd'
I1683
sa(dp3366
S'a'
I6244
sS's'
I14
sS'm'
(lp3367
I1
aI1
aI3
aI15
aI11
aI7
aI127
aI47
aI385
aI1007
aI713
aI2235
aI5489
aI8755
asS'd'
I1684
sa(dp3368
S'a'
I6247
sS's'
I14
sS'm'
(lp3369
I1
aI3
aI5
aI13
aI19
aI21
aI21
aI167
aI405
aI655
aI1653
aI889
aI7367
aI4177
asS'd'
I1685
sa(dp3370
S'a'
I6256
sS's'
I14
sS'm'
(lp3371
I1
aI1
aI5
aI3
aI19
aI63
aI99
aI39
aI89
aI415
aI951
aI2863
aI6569
aI3797
asS'd'
I1686
sa(dp3372
S'a'
I6281
sS's'
I14
sS'm'
(lp3373
I1
aI1
aI1
aI13
aI31
aI29
aI119
aI35
aI311
aI839
aI1749
aI941
aI7487
aI2385
asS'd'
I1687
sa(dp3374
S'a'
I6282
sS's'
I14
sS'm'
(lp3375
I1
aI3
aI7
aI3
aI17
aI3
aI97
aI143
aI465
aI345
aI1457
aI2201
aI5329
aI359
asS'd'
I1688
sa(dp3376
S'a'
I6284
sS's'
I14
sS'm'
(lp3377
I1
aI3
aI7
aI11
aI1
aI15
aI3
aI115
aI335
aI567
aI1749
aI1811
aI3491
aI15939
asS'd'
I1689
sa(dp3378
S'a'
I6296
sS's'
I14
sS'm'
(lp3379
I1
aI1
aI3
aI13
aI3
aI21
aI7
aI141
aI149
aI571
aI1877
aI473
aI2143
aI9569
asS'd'
I1690
sa(dp3380
S'a'
I6299
sS's'
I14
sS'm'
(lp3381
I1
aI3
aI3
aI11
aI23
aI61
aI47
aI179
aI297
aI453
aI181
aI3405
aI2981
aI13409
asS'd'
I1691
sa(dp3382
S'a'
I6302
sS's'
I14
sS'm'
(lp3383
I1
aI3
aI1
aI13
aI1
aI43
aI5
aI201
aI371
aI1003
aI367
aI2709
aI7675
aI14973
asS'd'
I1692
sa(dp3384
S'a'
I6325
sS's'
I14
sS'm'
(lp3385
I1
aI3
aI3
aI15
aI29
aI17
aI19
aI241
aI495
aI317
aI1135
aI2227
aI6457
aI4783
asS'd'
I1693
sa(dp3386
S'a'
I6349
sS's'
I14
sS'm'
(lp3387
I1
aI3
aI3
aI7
aI29
aI9
aI57
aI95
aI261
aI531
aI1717
aI3389
aI7991
aI3793
asS'd'
I1694
sa(dp3388
S'a'
I6352
sS's'
I14
sS'm'
(lp3389
I1
aI1
aI1
aI5
aI31
aI43
aI73
aI119
aI499
aI589
aI1529
aI3337
aI4097
aI15641
asS'd'
I1695
sa(dp3390
S'a'
I6362
sS's'
I14
sS'm'
(lp3391
I1
aI1
aI7
aI9
aI29
aI43
aI127
aI91
aI243
aI979
aI1325
aI2835
aI2787
aI9445
asS'd'
I1696
sa(dp3392
S'a'
I6383
sS's'
I14
sS'm'
(lp3393
I1
aI1
aI7
aI5
aI9
aI3
aI115
aI199
aI219
aI901
aI747
aI1077
aI3197
aI2443
asS'd'
I1697
sa(dp3394
S'a'
I6386
sS's'
I14
sS'm'
(lp3395
I1
aI3
aI5
aI1
aI3
aI43
aI7
aI117
aI297
aI313
aI1043
aI1579
aI5099
aI13289
asS'd'
I1698
sa(dp3396
S'a'
I6395
sS's'
I14
sS'm'
(lp3397
I1
aI1
aI7
aI11
aI29
aI33
aI15
aI121
aI131
aI579
aI317
aI1871
aI1121
aI11653
asS'd'
I1699
sa(dp3398
S'a'
I6397
sS's'
I14
sS'm'
(lp3399
I1
aI1
aI5
aI9
aI25
aI25
aI43
aI89
aI355
aI1011
aI1385
aI2901
aI6387
aI1653
asS'd'
I1700
sa(dp3400
S'a'
I6415
sS's'
I14
sS'm'
(lp3401
I1
aI1
aI1
aI9
aI5
aI47
aI61
aI165
aI85
aI757
aI1397
aI1177
aI1657
aI4899
asS'd'
I1701
sa(dp3402
S'a'
I6424
sS's'
I14
sS'm'
(lp3403
I1
aI1
aI3
aI9
aI11
aI49
aI15
aI139
aI261
aI613
aI931
aI1299
aI2777
aI2835
asS'd'
I1702
sa(dp3404
S'a'
I6429
sS's'
I14
sS'm'
(lp3405
I1
aI1
aI1
aI5
aI3
aI55
aI83
aI227
aI125
aI581
aI1607
aI1171
aI6681
aI14463
asS'd'
I1703
sa(dp3406
S'a'
I6439
sS's'
I14
sS'm'
(lp3407
I1
aI3
aI5
aI13
aI5
aI55
aI3
aI247
aI493
aI155
aI1073
aI3743
aI5719
aI4019
asS'd'
I1704
sa(dp3408
S'a'
I6451
sS's'
I14
sS'm'
(lp3409
I1
aI1
aI7
aI1
aI11
aI23
aI13
aI75
aI399
aI847
aI499
aI1643
aI6977
aI3699
asS'd'
I1705
sa(dp3410
S'a'
I6489
sS's'
I14
sS'm'
(lp3411
I1
aI3
aI1
aI9
aI11
aI41
aI47
aI131
aI313
aI627
aI481
aI2469
aI3281
aI979
asS'd'
I1706
sa(dp3412
S'a'
I6496
sS's'
I14
sS'm'
(lp3413
I1
aI3
aI5
aI13
aI29
aI3
aI65
aI101
aI11
aI29
aI1807
aI153
aI1487
aI16109
asS'd'
I1707
sa(dp3414
S'a'
I6502
sS's'
I14
sS'm'
(lp3415
I1
aI1
aI5
aI9
aI13
aI31
aI83
aI195
aI351
aI355
aI467
aI3871
aI3085
aI4441
asS'd'
I1708
sa(dp3416
S'a'
I6511
sS's'
I14
sS'm'
(lp3417
I1
aI3
aI5
aI3
aI19
aI21
aI111
aI179
aI143
aI361
aI1619
aI1547
aI3409
aI6905
asS'd'
I1709
sa(dp3418
S'a'
I6514
sS's'
I14
sS'm'
(lp3419
I1
aI1
aI5
aI9
aI31
aI1
aI93
aI199
aI491
aI135
aI1627
aI2559
aI1389
aI14561
asS'd'
I1710
sa(dp3420
S'a'
I6520
sS's'
I14
sS'm'
(lp3421
I1
aI3
aI3
aI9
aI25
aI53
aI3
aI105
aI39
aI445
aI259
aI1045
aI1129
aI9153
asS'd'
I1711
sa(dp3422
S'a'
I6523
sS's'
I14
sS'm'
(lp3423
I1
aI1
aI5
aI9
aI19
aI63
aI71
aI9
aI73
aI435
aI1377
aI4015
aI1821
aI6453
asS'd'
I1712
sa(dp3424
S'a'
I6529
sS's'
I14
sS'm'
(lp3425
I1
aI3
aI7
aI13
aI19
aI13
aI37
aI247
aI391
aI23
aI1491
aI1257
aI6395
aI237
asS'd'
I1713
sa(dp3426
S'a'
I6532
sS's'
I14
sS'm'
(lp3427
I1
aI1
aI3
aI3
aI19
aI55
aI109
aI23
aI227
aI747
aI729
aI2221
aI727
aI2209
asS'd'
I1714
sa(dp3428
S'a'
I6547
sS's'
I14
sS'm'
(lp3429
I1
aI1
aI5
aI11
aI25
aI21
aI75
aI37
aI219
aI355
aI1005
aI1895
aI7039
aI5225
asS'd'
I1715
sa(dp3430
S'a'
I6549
sS's'
I14
sS'm'
(lp3431
I1
aI3
aI5
aI13
aI11
aI43
aI9
aI67
aI87
aI797
aI1077
aI245
aI4521
aI11845
asS'd'
I1716
sa(dp3432
S'a'
I6598
sS's'
I14
sS'm'
(lp3433
I1
aI3
aI5
aI3
aI15
aI29
aI127
aI237
aI277
aI373
aI1859
aI3083
aI587
aI1123
asS'd'
I1717
sa(dp3434
S'a'
I6601
sS's'
I14
sS'm'
(lp3435
I1
aI1
aI7
aI15
aI13
aI7
aI103
aI53
aI13
aI965
aI1497
aI775
aI3439
aI1501
asS'd'
I1718
sa(dp3436
S'a'
I6610
sS's'
I14
sS'm'
(lp3437
I1
aI3
aI3
aI15
aI17
aI13
aI97
aI169
aI67
aI953
aI189
aI2739
aI1459
aI10543
asS'd'
I1719
sa(dp3438
S'a'
I6622
sS's'
I14
sS'm'
(lp3439
I1
aI1
aI5
aI1
aI17
aI39
aI15
aI127
aI327
aI989
aI1471
aI3235
aI2801
aI15311
asS'd'
I1720
sa(dp3440
S'a'
I6632
sS's'
I14
sS'm'
(lp3441
I1
aI1
aI1
aI15
aI5
aI37
aI55
aI155
aI47
aI463
aI1851
aI3467
aI2765
aI9359
asS'd'
I1721
sa(dp3442
S'a'
I6655
sS's'
I14
sS'm'
(lp3443
I1
aI3
aI3
aI15
aI1
aI13
aI93
aI239
aI291
aI115
aI365
aI61
aI395
aI15853
asS'd'
I1722
sa(dp3444
S'a'
I6665
sS's'
I14
sS'm'
(lp3445
I1
aI1
aI5
aI1
aI19
aI27
aI61
aI95
aI105
aI369
aI1557
aI961
aI6917
aI3621
asS'd'
I1723
sa(dp3446
S'a'
I6666
sS's'
I14
sS'm'
(lp3447
I1
aI3
aI3
aI9
aI7
aI35
aI115
aI53
aI111
aI345
aI1145
aI1687
aI3401
aI12107
asS'd'
I1724
sa(dp3448
S'a'
I6695
sS's'
I14
sS'm'
(lp3449
I1
aI1
aI1
aI5
aI7
aI31
aI63
aI19
aI373
aI79
aI1369
aI3037
aI2835
aI4439
asS'd'
I1725
sa(dp3450
S'a'
I6701
sS's'
I14
sS'm'
(lp3451
I1
aI3
aI7
aI9
aI11
aI17
aI29
aI33
aI331
aI447
aI1981
aI3333
aI6535
aI6557
asS'd'
I1726
sa(dp3452
S'a'
I6709
sS's'
I14
sS'm'
(lp3453
I1
aI3
aI3
aI5
aI11
aI41
aI29
aI43
aI365
aI279
aI1919
aI945
aI179
aI1987
asS'd'
I1727
sa(dp3454
S'a'
I6710
sS's'
I14
sS'm'
(lp3455
I1
aI3
aI1
aI13
aI7
aI7
aI25
aI33
aI103
aI367
aI1267
aI763
aI5691
aI8643
asS'd'
I1728
sa(dp3456
S'a'
I6741
sS's'
I14
sS'm'
(lp3457
I1
aI3
aI1
aI5
aI11
aI15
aI3
aI213
aI511
aI211
aI1069
aI4047
aI3335
aI12729
asS'd'
I1729
sa(dp3458
S'a'
I6745
sS's'
I14
sS'm'
(lp3459
I1
aI1
aI3
aI1
aI5
aI11
aI27
aI201
aI361
aI537
aI679
aI3709
aI293
aI2997
asS'd'
I1730
sa(dp3460
S'a'
I6758
sS's'
I14
sS'm'
(lp3461
I1
aI1
aI3
aI1
aI25
aI15
aI19
aI185
aI369
aI577
aI1625
aI655
aI2363
aI3861
asS'd'
I1731
sa(dp3462
S'a'
I6767
sS's'
I14
sS'm'
(lp3463
I1
aI1
aI5
aI5
aI1
aI47
aI61
aI45
aI411
aI597
aI955
aI1007
aI3775
aI5809
asS'd'
I1732
sa(dp3464
S'a'
I6772
sS's'
I14
sS'm'
(lp3465
I1
aI1
aI5
aI3
aI27
aI51
aI101
aI167
aI429
aI333
aI1703
aI3541
aI2947
aI3695
asS'd'
I1733
sa(dp3466
S'a'
I6782
sS's'
I14
sS'm'
(lp3467
I1
aI3
aI5
aI5
aI1
aI53
aI17
aI63
aI141
aI215
aI1223
aI3129
aI635
aI15919
asS'd'
I1734
sa(dp3468
S'a'
I6797
sS's'
I14
sS'm'
(lp3469
I1
aI3
aI3
aI1
aI23
aI31
aI25
aI11
aI195
aI241
aI995
aI3941
aI573
aI13855
asS'd'
I1735
sa(dp3470
S'a'
I6800
sS's'
I14
sS'm'
(lp3471
I1
aI3
aI3
aI7
aI17
aI13
aI71
aI203
aI465
aI479
aI1857
aI1493
aI8067
aI7113
asS'd'
I1736
sa(dp3472
S'a'
I6843
sS's'
I14
sS'm'
(lp3473
I1
aI1
aI5
aI3
aI11
aI57
aI9
aI59
aI225
aI691
aI425
aI2423
aI6031
aI6631
asS'd'
I1737
sa(dp3474
S'a'
I6845
sS's'
I14
sS'm'
(lp3475
I1
aI3
aI7
aI1
aI29
aI57
aI103
aI123
aI401
aI807
aI471
aI2759
aI5113
aI15937
asS'd'
I1738
sa(dp3476
S'a'
I6860
sS's'
I14
sS'm'
(lp3477
I1
aI3
aI1
aI1
aI3
aI1
aI67
aI123
aI157
aI655
aI519
aI323
aI1853
aI15041
asS'd'
I1739
sa(dp3478
S'a'
I6865
sS's'
I14
sS'm'
(lp3479
I1
aI1
aI7
aI5
aI11
aI11
aI105
aI135
aI247
aI689
aI1141
aI2347
aI7113
aI9347
asS'd'
I1740
sa(dp3480
S'a'
I6878
sS's'
I14
sS'm'
(lp3481
I1
aI1
aI3
aI11
aI15
aI37
aI87
aI3
aI209
aI575
aI1521
aI3863
aI3893
aI211
asS'd'
I1741
sa(dp3482
S'a'
I6887
sS's'
I14
sS'm'
(lp3483
I1
aI3
aI1
aI3
aI29
aI55
aI115
aI31
aI19
aI195
aI985
aI3275
aI363
aI9801
asS'd'
I1742
sa(dp3484
S'a'
I6888
sS's'
I14
sS'm'
(lp3485
I1
aI1
aI3
aI9
aI13
aI31
aI57
aI251
aI201
aI275
aI1751
aI389
aI1463
aI13159
asS'd'
I1743
sa(dp3486
S'a'
I6901
sS's'
I14
sS'm'
(lp3487
I1
aI3
aI5
aI15
aI19
aI51
aI127
aI255
aI397
aI243
aI29
aI3007
aI7845
aI4687
asS'd'
I1744
sa(dp3488
S'a'
I6906
sS's'
I14
sS'm'
(lp3489
I1
aI1
aI7
aI15
aI9
aI37
aI39
aI217
aI509
aI137
aI1123
aI3361
aI6323
aI5323
asS'd'
I1745
sa(dp3490
S'a'
I6940
sS's'
I14
sS'm'
(lp3491
I1
aI3
aI7
aI5
aI25
aI3
aI93
aI203
aI345
aI581
aI261
aI2811
aI4829
aI6977
asS'd'
I1746
sa(dp3492
S'a'
I6947
sS's'
I14
sS'm'
(lp3493
I1
aI1
aI7
aI1
aI15
aI41
aI51
aI227
aI447
aI893
aI1209
aI3865
aI5229
aI4277
asS'd'
I1747
sa(dp3494
S'a'
I6953
sS's'
I14
sS'm'
(lp3495
I1
aI1
aI1
aI5
aI31
aI19
aI23
aI195
aI359
aI853
aI595
aI337
aI2503
aI16371
asS'd'
I1748
sa(dp3496
S'a'
I6954
sS's'
I14
sS'm'
(lp3497
I1
aI3
aI7
aI5
aI5
aI13
aI89
aI157
aI351
aI777
aI151
aI3565
aI4219
aI7423
asS'd'
I1749
sa(dp3498
S'a'
I6959
sS's'
I14
sS'm'
(lp3499
I1
aI1
aI1
aI5
aI7
aI1
aI9
aI89
aI175
aI909
aI1523
aI2295
aI7949
aI6739
asS'd'
I1750
sa(dp3500
S'a'
I6961
sS's'
I14
sS'm'
(lp3501
I1
aI3
aI5
aI15
aI27
aI17
aI11
aI235
aI19
aI105
aI457
aI465
aI3819
aI11335
asS'd'
I1751
sa(dp3502
S'a'
I6964
sS's'
I14
sS'm'
(lp3503
I1
aI3
aI1
aI13
aI3
aI41
aI85
aI221
aI451
aI613
aI543
aI2265
aI6831
aI1725
asS'd'
I1752
sa(dp3504
S'a'
I6991
sS's'
I14
sS'm'
(lp3505
I1
aI1
aI7
aI7
aI3
aI29
aI9
aI197
aI455
aI665
aI343
aI1811
aI5395
aI393
asS'd'
I1753
sa(dp3506
S'a'
I6993
sS's'
I14
sS'm'
(lp3507
I1
aI1
aI3
aI13
aI29
aI55
aI71
aI95
aI475
aI615
aI2029
aI123
aI413
aI16127
asS'd'
I1754
sa(dp3508
S'a'
I6999
sS's'
I14
sS'm'
(lp3509
I1
aI1
aI5
aI9
aI15
aI61
aI9
aI51
aI105
aI271
aI511
aI2801
aI693
aI11839
asS'd'
I1755
sa(dp3510
S'a'
I7016
sS's'
I14
sS'm'
(lp3511
I1
aI1
aI7
aI13
aI29
aI9
aI105
aI59
aI377
aI635
aI717
aI4033
aI6963
aI10541
asS'd'
I1756
sa(dp3512
S'a'
I7029
sS's'
I14
sS'm'
(lp3513
I1
aI1
aI1
aI13
aI7
aI13
aI59
aI17
aI335
aI355
aI77
aI3665
aI7003
aI9521
asS'd'
I1757
sa(dp3514
S'a'
I7036
sS's'
I14
sS'm'
(lp3515
I1
aI3
aI1
aI1
aI23
aI43
aI51
aI209
aI151
aI365
aI1021
aI2859
aI3937
aI2899
asS'd'
I1758
sa(dp3516
S'a'
I7045
sS's'
I14
sS'm'
(lp3517
I1
aI1
aI3
aI3
aI31
aI41
aI111
aI107
aI171
aI433
aI1233
aI505
aI2971
aI6927
asS'd'
I1759
sa(dp3518
S'a'
I7076
sS's'
I14
sS'm'
(lp3519
I1
aI3
aI7
aI13
aI17
aI25
aI127
aI195
aI257
aI551
aI1867
aI2145
aI3695
aI14567
asS'd'
I1760
sa(dp3520
S'a'
I7083
sS's'
I14
sS'm'
(lp3521
I1
aI1
aI5
aI13
aI13
aI45
aI39
aI195
aI55
aI991
aI1981
aI1043
aI5875
aI581
asS'd'
I1761
sa(dp3522
S'a'
I7094
sS's'
I14
sS'm'
(lp3523
I1
aI3
aI3
aI11
aI25
aI31
aI91
aI153
aI415
aI449
aI1301
aI563
aI7755
aI10671
asS'd'
I1762
sa(dp3524
S'a'
I7097
sS's'
I14
sS'm'
(lp3525
I1
aI1
aI3
aI5
aI31
aI63
aI1
aI157
aI229
aI949
aI971
aI137
aI6589
aI8387
asS'd'
I1763
sa(dp3526
S'a'
I7123
sS's'
I14
sS'm'
(lp3527
I1
aI3
aI7
aI15
aI25
aI7
aI89
aI133
aI73
aI497
aI1361
aI613
aI455
aI1005
asS'd'
I1764
sa(dp3528
S'a'
I7130
sS's'
I14
sS'm'
(lp3529
I1
aI3
aI3
aI1
aI13
aI5
aI119
aI93
aI175
aI511
aI1923
aI763
aI7573
aI7545
asS'd'
I1765
sa(dp3530
S'a'
I7139
sS's'
I14
sS'm'
(lp3531
I1
aI1
aI3
aI15
aI27
aI59
aI49
aI205
aI497
aI485
aI117
aI2523
aI4495
aI15153
asS'd'
I1766
sa(dp3532
S'a'
I7145
sS's'
I14
sS'm'
(lp3533
I1
aI3
aI7
aI9
aI15
aI47
aI111
aI31
aI363
aI11
aI475
aI2931
aI6813
aI1259
asS'd'
I1767
sa(dp3534
S'a'
I7146
sS's'
I14
sS'm'
(lp3535
I1
aI1
aI5
aI5
aI1
aI35
aI95
aI225
aI17
aI991
aI809
aI2601
aI6455
aI13803
asS'd'
I1768
sa(dp3536
S'a'
I7178
sS's'
I14
sS'm'
(lp3537
I1
aI1
aI5
aI5
aI15
aI1
aI1
aI171
aI433
aI887
aI1813
aI3431
aI2471
aI7803
asS'd'
I1769
sa(dp3538
S'a'
I7186
sS's'
I14
sS'm'
(lp3539
I1
aI3
aI3
aI15
aI1
aI15
aI43
aI179
aI15
aI949
aI1881
aI1027
aI6989
aI8955
asS'd'
I1770
sa(dp3540
S'a'
I7192
sS's'
I14
sS'm'
(lp3541
I1
aI3
aI7
aI13
aI1
aI3
aI49
aI183
aI373
aI175
aI1733
aI913
aI929
aI1065
asS'd'
I1771
sa(dp3542
S'a'
I7198
sS's'
I14
sS'm'
(lp3543
I1
aI3
aI5
aI7
aI15
aI51
aI107
aI115
aI323
aI357
aI167
aI2069
aI7541
aI9601
asS'd'
I1772
sa(dp3544
S'a'
I7222
sS's'
I14
sS'm'
(lp3545
I1
aI1
aI3
aI5
aI5
aI21
aI31
aI107
aI21
aI299
aI1937
aI43
aI3673
aI8155
asS'd'
I1773
sa(dp3546
S'a'
I7269
sS's'
I14
sS'm'
(lp3547
I1
aI3
aI5
aI11
aI9
aI55
aI35
aI113
aI29
aI99
aI161
aI1607
aI8141
aI4951
asS'd'
I1774
sa(dp3548
S'a'
I7270
sS's'
I14
sS'm'
(lp3549
I1
aI3
aI7
aI15
aI25
aI7
aI113
aI179
aI213
aI19
aI1717
aI1027
aI2021
aI11263
asS'd'
I1775
sa(dp3550
S'a'
I7276
sS's'
I14
sS'm'
(lp3551
I1
aI1
aI5
aI1
aI31
aI33
aI85
aI111
aI67
aI95
aI2013
aI2217
aI871
aI5329
asS'd'
I1776
sa(dp3552
S'a'
I7287
sS's'
I14
sS'm'
(lp3553
I1
aI1
aI1
aI7
aI7
aI63
aI67
aI145
aI495
aI419
aI1945
aI3437
aI6255
aI151
asS'd'
I1777
sa(dp3554
S'a'
I7307
sS's'
I14
sS'm'
(lp3555
I1
aI3
aI5
aI7
aI17
aI37
aI97
aI187
aI215
aI399
aI1603
aI2195
aI5923
aI769
asS'd'
I1778
sa(dp3556
S'a'
I7315
sS's'
I14
sS'm'
(lp3557
I1
aI1
aI3
aI9
aI25
aI1
aI119
aI193
aI385
aI861
aI2005
aI2769
aI675
aI767
asS'd'
I1779
sa(dp3558
S'a'
I7334
sS's'
I14
sS'm'
(lp3559
I1
aI3
aI1
aI15
aI19
aI7
aI5
aI227
aI173
aI383
aI289
aI461
aI579
aI3689
asS'd'
I1780
sa(dp3560
S'a'
I7340
sS's'
I14
sS'm'
(lp3561
I1
aI3
aI1
aI11
aI1
aI37
aI93
aI239
aI465
aI891
aI1479
aI921
aI4439
aI15265
asS'd'
I1781
sa(dp3562
S'a'
I7351
sS's'
I14
sS'm'
(lp3563
I1
aI1
aI1
aI13
aI27
aI61
aI99
aI69
aI279
aI655
aI1853
aI1593
aI6319
aI9003
asS'd'
I1782
sa(dp3564
S'a'
I7352
sS's'
I14
sS'm'
(lp3565
I1
aI1
aI1
aI11
aI5
aI7
aI19
aI7
aI387
aI303
aI321
aI931
aI5809
aI16029
asS'd'
I1783
sa(dp3566
S'a'
I7357
sS's'
I14
sS'm'
(lp3567
I1
aI1
aI1
aI15
aI21
aI55
aI43
aI107
aI217
aI687
aI19
aI3225
aI3419
aI9991
asS'd'
I1784
sa(dp3568
S'a'
I7360
sS's'
I14
sS'm'
(lp3569
I1
aI1
aI7
aI5
aI7
aI55
aI79
aI41
aI317
aI357
aI859
aI1205
aI191
aI9395
asS'd'
I1785
sa(dp3570
S'a'
I7363
sS's'
I14
sS'm'
(lp3571
I1
aI1
aI3
aI11
aI3
aI43
aI7
aI133
aI115
aI995
aI1205
aI1055
aI4153
aI10481
asS'd'
I1786
sa(dp3572
S'a'
I7384
sS's'
I14
sS'm'
(lp3573
I1
aI1
aI7
aI11
aI31
aI57
aI53
aI9
aI459
aI223
aI1969
aI3513
aI7033
aI8505
asS'd'
I1787
sa(dp3574
S'a'
I7396
sS's'
I14
sS'm'
(lp3575
I1
aI1
aI3
aI7
aI17
aI11
aI115
aI255
aI281
aI97
aI1685
aI2039
aI2845
aI11637
asS'd'
I1788
sa(dp3576
S'a'
I7403
sS's'
I14
sS'm'
(lp3577
I1
aI3
aI7
aI1
aI23
aI41
aI69
aI199
aI53
aI105
aI657
aI1453
aI4429
aI1101
asS'd'
I1789
sa(dp3578
S'a'
I7406
sS's'
I14
sS'm'
(lp3579
I1
aI3
aI1
aI5
aI11
aI33
aI91
aI131
aI191
aI73
aI823
aI117
aI1053
aI127
asS'd'
I1790
sa(dp3580
S'a'
I7425
sS's'
I14
sS'm'
(lp3581
I1
aI3
aI7
aI11
aI7
aI3
aI21
aI65
aI187
aI103
aI1393
aI1797
aI6673
aI1409
asS'd'
I1791
sa(dp3582
S'a'
I7437
sS's'
I14
sS'm'
(lp3583
I1
aI3
aI7
aI1
aI31
aI25
aI25
aI161
aI299
aI275
aI417
aI2267
aI6861
aI1255
asS'd'
I1792
sa(dp3584
S'a'
I7445
sS's'
I14
sS'm'
(lp3585
I1
aI3
aI5
aI13
aI5
aI11
aI61
aI155
aI115
aI1001
aI747
aI889
aI3235
aI5709
asS'd'
I1793
sa(dp3586
S'a'
I7450
sS's'
I14
sS'm'
(lp3587
I1
aI3
aI7
aI7
aI7
aI1
aI97
aI177
aI507
aI273
aI1781
aI3455
aI5123
aI15607
asS'd'
I1794
sa(dp3588
S'a'
I7455
sS's'
I14
sS'm'
(lp3589
I1
aI1
aI7
aI5
aI1
aI7
aI59
aI49
aI147
aI343
aI97
aI3517
aI5611
aI8705
asS'd'
I1795
sa(dp3590
S'a'
I7461
sS's'
I14
sS'm'
(lp3591
I1
aI1
aI5
aI13
aI21
aI29
aI13
aI21
aI503
aI515
aI1217
aI3905
aI5513
aI15849
asS'd'
I1796
sa(dp3592
S'a'
I7466
sS's'
I14
sS'm'
(lp3593
I1
aI3
aI1
aI9
aI9
aI39
aI65
aI111
aI385
aI757
aI583
aI2225
aI2039
aI2817
asS'd'
I1797
sa(dp3594
S'a'
I7488
sS's'
I14
sS'm'
(lp3595
I1
aI3
aI3
aI15
aI23
aI17
aI63
aI169
aI503
aI949
aI849
aI461
aI6799
aI669
asS'd'
I1798
sa(dp3596
S'a'
I7494
sS's'
I14
sS'm'
(lp3597
I1
aI1
aI1
aI3
aI1
aI41
aI63
aI159
aI251
aI457
aI521
aI1653
aI623
aI3287
asS'd'
I1799
sa(dp3598
S'a'
I7515
sS's'
I14
sS'm'
(lp3599
I1
aI1
aI7
aI3
aI9
aI1
aI41
aI37
aI441
aI921
aI1415
aI2955
aI5841
aI1451
asS'd'
I1800
sa(dp3600
S'a'
I7517
sS's'
I14
sS'm'
(lp3601
I1
aI1
aI5
aI11
aI23
aI29
aI89
aI185
aI413
aI357
aI1131
aI2369
aI3835
aI6233
asS'd'
I1801
sa(dp3602
S'a'
I7521
sS's'
I14
sS'm'
(lp3603
I1
aI1
aI5
aI15
aI27
aI35
aI17
aI73
aI315
aI911
aI1761
aI797
aI5349
aI3219
asS'd'
I1802
sa(dp3604
S'a'
I7536
sS's'
I14
sS'm'
(lp3605
I1
aI3
aI7
aI11
aI21
aI9
aI119
aI233
aI249
aI901
aI189
aI3625
aI2691
aI16201
asS'd'
I1803
sa(dp3606
S'a'
I7546
sS's'
I14
sS'm'
(lp3607
I1
aI3
aI3
aI13
aI29
aI61
aI105
aI145
aI187
aI79
aI609
aI321
aI4289
aI3933
asS'd'
I1804
sa(dp3608
S'a'
I7569
sS's'
I14
sS'm'
(lp3609
I1
aI3
aI1
aI15
aI19
aI63
aI13
aI185
aI115
aI219
aI1021
aI1205
aI4273
aI11521
asS'd'
I1805
sa(dp3610
S'a'
I7591
sS's'
I14
sS'm'
(lp3611
I1
aI1
aI3
aI3
aI23
aI31
aI93
aI153
aI87
aI947
aI1039
aI469
aI4047
aI8869
asS'd'
I1806
sa(dp3612
S'a'
I7592
sS's'
I14
sS'm'
(lp3613
I1
aI1
aI1
aI1
aI9
aI1
aI85
aI3
aI15
aI995
aI455
aI2769
aI6781
aI16203
asS'd'
I1807
sa(dp3614
S'a'
I7598
sS's'
I14
sS'm'
(lp3615
I1
aI1
aI3
aI3
aI13
aI7
aI55
aI215
aI185
aI367
aI765
aI441
aI4497
aI1521
asS'd'
I1808
sa(dp3616
S'a'
I7612
sS's'
I14
sS'm'
(lp3617
I1
aI1
aI1
aI5
aI1
aI31
aI13
aI95
aI417
aI735
aI975
aI3407
aI4871
aI16133
asS'd'
I1809
sa(dp3618
S'a'
I7623
sS's'
I14
sS'm'
(lp3619
I1
aI1
aI3
aI3
aI5
aI43
aI111
aI107
aI419
aI515
aI1075
aI3597
aI1187
aI4143
asS'd'
I1810
sa(dp3620
S'a'
I7632
sS's'
I14
sS'm'
(lp3621
I1
aI1
aI3
aI13
aI31
aI51
aI83
aI163
aI489
aI887
aI863
aI599
aI9
aI13861
asS'd'
I1811
sa(dp3622
S'a'
I7637
sS's'
I14
sS'm'
(lp3623
I1
aI3
aI3
aI3
aI19
aI27
aI91
aI115
aI103
aI969
aI593
aI3667
aI1867
aI15433
asS'd'
I1812
sa(dp3624
S'a'
I7644
sS's'
I14
sS'm'
(lp3625
I1
aI3
aI3
aI13
aI7
aI25
aI47
aI141
aI57
aI553
aI1785
aI1709
aI7453
aI2209
asS'd'
I1813
sa(dp3626
S'a'
I7657
sS's'
I14
sS'm'
(lp3627
I1
aI3
aI1
aI13
aI11
aI13
aI71
aI219
aI5
aI451
aI2043
aI1605
aI6439
aI12203
asS'd'
I1814
sa(dp3628
S'a'
I7665
sS's'
I14
sS'm'
(lp3629
I1
aI3
aI1
aI13
aI5
aI57
aI61
aI223
aI401
aI413
aI321
aI1365
aI619
aI12477
asS'd'
I1815
sa(dp3630
S'a'
I7672
sS's'
I14
sS'm'
(lp3631
I1
aI3
aI1
aI5
aI25
aI57
aI89
aI211
aI195
aI455
aI1165
aI3979
aI6313
aI5751
asS'd'
I1816
sa(dp3632
S'a'
I7682
sS's'
I14
sS'm'
(lp3633
I1
aI1
aI1
aI9
aI31
aI23
aI71
aI145
aI89
aI285
aI1593
aI1171
aI5685
aI15459
asS'd'
I1817
sa(dp3634
S'a'
I7699
sS's'
I14
sS'm'
(lp3635
I1
aI3
aI7
aI7
aI9
aI41
aI65
aI251
aI65
aI137
aI1577
aI3027
aI5555
aI2865
asS'd'
I1818
sa(dp3636
S'a'
I7702
sS's'
I14
sS'm'
(lp3637
I1
aI1
aI5
aI13
aI27
aI5
aI125
aI21
aI171
aI647
aI983
aI2921
aI6623
aI5695
asS'd'
I1819
sa(dp3638
S'a'
I7724
sS's'
I14
sS'm'
(lp3639
I1
aI1
aI1
aI13
aI15
aI9
aI117
aI197
aI123
aI953
aI1191
aI3657
aI5757
aI15957
asS'd'
I1820
sa(dp3640
S'a'
I7749
sS's'
I14
sS'm'
(lp3641
I1
aI1
aI3
aI7
aI29
aI13
aI5
aI175
aI395
aI127
aI679
aI255
aI6055
aI7639
asS'd'
I1821
sa(dp3642
S'a'
I7753
sS's'
I14
sS'm'
(lp3643
I1
aI3
aI7
aI15
aI15
aI51
aI77
aI147
aI319
aI147
aI1775
aI3983
aI3175
aI5723
asS'd'
I1822
sa(dp3644
S'a'
I7754
sS's'
I14
sS'm'
(lp3645
I1
aI3
aI3
aI3
aI7
aI11
aI119
aI41
aI43
aI153
aI975
aI679
aI3081
aI10359
asS'd'
I1823
sa(dp3646
S'a'
I7761
sS's'
I14
sS'm'
(lp3647
I1
aI1
aI5
aI13
aI3
aI7
aI65
aI67
aI63
aI399
aI1561
aI2789
aI2083
aI12289
asS'd'
I1824
sa(dp3648
S'a'
I7771
sS's'
I14
sS'm'
(lp3649
I1
aI1
aI7
aI3
aI19
aI53
aI103
aI67
aI35
aI865
aI161
aI93
aI2533
aI3851
asS'd'
I1825
sa(dp3650
S'a'
I7777
sS's'
I14
sS'm'
(lp3651
I1
aI1
aI1
aI11
aI31
aI9
aI29
aI189
aI199
aI817
aI1571
aI395
aI345
aI3777
asS'd'
I1826
sa(dp3652
S'a'
I7784
sS's'
I14
sS'm'
(lp3653
I1
aI3
aI5
aI11
aI31
aI3
aI9
aI67
aI277
aI735
aI181
aI2777
aI3009
aI7233
asS'd'
I1827
sa(dp3654
S'a'
I7804
sS's'
I14
sS'm'
(lp3655
I1
aI1
aI3
aI3
aI17
aI7
aI17
aI3
aI375
aI933
aI237
aI3919
aI5409
aI3355
asS'd'
I1828
sa(dp3656
S'a'
I7807
sS's'
I14
sS'm'
(lp3657
I1
aI3
aI3
aI5
aI9
aI27
aI19
aI77
aI221
aI3
aI1965
aI309
aI3001
aI15977
asS'd'
I1829
sa(dp3658
S'a'
I7808
sS's'
I14
sS'm'
(lp3659
I1
aI1
aI5
aI1
aI3
aI33
aI35
aI133
aI37
aI709
aI627
aI1705
aI2525
aI4307
asS'd'
I1830
sa(dp3660
S'a'
I7818
sS's'
I14
sS'm'
(lp3661
I1
aI1
aI7
aI3
aI25
aI21
aI105
aI55
aI375
aI681
aI881
aI1299
aI5879
aI459
asS'd'
I1831
sa(dp3662
S'a'
I7835
sS's'
I14
sS'm'
(lp3663
I1
aI3
aI7
aI1
aI13
aI7
aI113
aI103
aI313
aI515
aI1041
aI3683
aI4619
aI5093
asS'd'
I1832
sa(dp3664
S'a'
I7842
sS's'
I14
sS'm'
(lp3665
I1
aI1
aI3
aI7
aI19
aI43
aI83
aI37
aI39
aI133
aI1759
aI1171
aI1521
aI13717
asS'd'
I1833
sa(dp3666
S'a'
I7865
sS's'
I14
sS'm'
(lp3667
I1
aI1
aI7
aI13
aI7
aI35
aI15
aI155
aI293
aI1001
aI157
aI3883
aI405
aI1797
asS'd'
I1834
sa(dp3668
S'a'
I7868
sS's'
I14
sS'm'
(lp3669
I1
aI1
aI3
aI3
aI13
aI19
aI125
aI49
aI333
aI387
aI339
aI1815
aI4503
aI7359
asS'd'
I1835
sa(dp3670
S'a'
I7880
sS's'
I14
sS'm'
(lp3671
I1
aI1
aI3
aI13
aI19
aI19
aI105
aI225
aI151
aI27
aI1251
aI885
aI4815
aI7863
asS'd'
I1836
sa(dp3672
S'a'
I7883
sS's'
I14
sS'm'
(lp3673
I1
aI1
aI1
aI5
aI7
aI59
aI17
aI145
aI77
aI117
aI1355
aI1429
aI2301
aI16177
asS'd'
I1837
sa(dp3674
S'a'
I7891
sS's'
I14
sS'm'
(lp3675
I1
aI3
aI3
aI13
aI5
aI31
aI119
aI167
aI459
aI727
aI1799
aI2537
aI695
aI13637
asS'd'
I1838
sa(dp3676
S'a'
I7897
sS's'
I14
sS'm'
(lp3677
I1
aI3
aI3
aI3
aI27
aI51
aI107
aI85
aI267
aI57
aI1279
aI823
aI6247
aI3603
asS'd'
I1839
sa(dp3678
S'a'
I7907
sS's'
I14
sS'm'
(lp3679
I1
aI1
aI7
aI15
aI29
aI17
aI67
aI197
aI215
aI465
aI109
aI3461
aI5269
aI15287
asS'd'
I1840
sa(dp3680
S'a'
I7910
sS's'
I14
sS'm'
(lp3681
I1
aI1
aI3
aI5
aI11
aI15
aI123
aI53
aI293
aI797
aI1105
aI1777
aI6509
aI217
asS'd'
I1841
sa(dp3682
S'a'
I7924
sS's'
I14
sS'm'
(lp3683
I1
aI3
aI3
aI13
aI3
aI5
aI109
aI53
aI203
aI693
aI871
aI135
aI369
aI11149
asS'd'
I1842
sa(dp3684
S'a'
I7933
sS's'
I14
sS'm'
(lp3685
I1
aI3
aI5
aI15
aI17
aI43
aI81
aI235
aI119
aI817
aI1777
aI261
aI8049
aI4251
asS'd'
I1843
sa(dp3686
S'a'
I7934
sS's'
I14
sS'm'
(lp3687
I1
aI1
aI3
aI7
aI7
aI13
aI87
aI99
aI481
aI931
aI1507
aI651
aI5267
aI8281
asS'd'
I1844
sa(dp3688
S'a'
I7942
sS's'
I14
sS'm'
(lp3689
I1
aI3
aI1
aI13
aI27
aI43
aI77
aI225
aI341
aI163
aI933
aI429
aI4943
aI7781
asS'd'
I1845
sa(dp3690
S'a'
I7948
sS's'
I14
sS'm'
(lp3691
I1
aI1
aI7
aI1
aI1
aI49
aI85
aI211
aI449
aI479
aI1395
aI787
aI5653
aI14891
asS'd'
I1846
sa(dp3692
S'a'
I7959
sS's'
I14
sS'm'
(lp3693
I1
aI1
aI5
aI9
aI25
aI13
aI49
aI85
aI125
aI85
aI1281
aI3365
aI4305
aI11791
asS'd'
I1847
sa(dp3694
S'a'
I7984
sS's'
I14
sS'm'
(lp3695
I1
aI3
aI1
aI13
aI3
aI31
aI117
aI39
aI43
aI151
aI663
aI669
aI1571
aI5207
asS'd'
I1848
sa(dp3696
S'a'
I7994
sS's'
I14
sS'm'
(lp3697
I1
aI3
aI7
aI15
aI17
aI7
aI79
aI163
aI37
aI841
aI1799
aI1787
aI4501
aI3785
asS'd'
I1849
sa(dp3698
S'a'
I7999
sS's'
I14
sS'm'
(lp3699
I1
aI1
aI3
aI9
aI1
aI23
aI67
aI191
aI449
aI931
aI1521
aI2705
aI887
aI7037
asS'd'
I1850
sa(dp3700
S'a'
I8014
sS's'
I14
sS'm'
(lp3701
I1
aI1
aI1
aI1
aI5
aI13
aI55
aI161
aI419
aI577
aI1703
aI2589
aI2651
aI2873
asS'd'
I1851
sa(dp3702
S'a'
I8021
sS's'
I14
sS'm'
(lp3703
I1
aI3
aI3
aI3
aI5
aI19
aI37
aI169
aI69
aI1003
aI1755
aI3101
aI1469
aI8583
asS'd'
I1852
sa(dp3704
S'a'
I8041
sS's'
I14
sS'm'
(lp3705
I1
aI1
aI1
aI1
aI11
aI33
aI105
aI79
aI283
aI91
aI299
aI835
aI3193
aI5593
asS'd'
I1853
sa(dp3706
S'a'
I8049
sS's'
I14
sS'm'
(lp3707
I1
aI3
aI3
aI13
aI25
aI21
aI81
aI213
aI465
aI475
aI331
aI457
aI61
aI9511
asS'd'
I1854
sa(dp3708
S'a'
I8050
sS's'
I14
sS'm'
(lp3709
I1
aI1
aI3
aI11
aI1
aI11
aI77
aI95
aI455
aI949
aI1999
aI1833
aI1275
aI5631
asS'd'
I1855
sa(dp3710
S'a'
I8068
sS's'
I14
sS'm'
(lp3711
I1
aI1
aI1
aI1
aI15
aI25
aI51
aI137
aI275
aI451
aI1179
aI3595
aI5177
aI7105
asS'd'
I1856
sa(dp3712
S'a'
I8080
sS's'
I14
sS'm'
(lp3713
I1
aI3
aI3
aI3
aI3
aI59
aI79
aI143
aI393
aI583
aI349
aI3039
aI7079
aI14245
asS'd'
I1857
sa(dp3714
S'a'
I8095
sS's'
I14
sS'm'
(lp3715
I1
aI1
aI7
aI9
aI21
aI11
aI123
aI105
aI53
aI297
aI803
aI4025
aI5421
aI14527
asS'd'
I1858
sa(dp3716
S'a'
I8102
sS's'
I14
sS'm'
(lp3717
I1
aI3
aI7
aI11
aI21
aI15
aI103
aI109
aI311
aI321
aI1217
aI2777
aI5457
aI1823
asS'd'
I1859
sa(dp3718
S'a'
I8106
sS's'
I14
sS'm'
(lp3719
I1
aI3
aI5
aI11
aI19
aI31
aI79
aI89
aI295
aI413
aI817
aI499
aI3699
aI14411
asS'd'
I1860
sa(dp3720
S'a'
I8120
sS's'
I14
sS'm'
(lp3721
I1
aI1
aI1
aI5
aI11
aI3
aI81
aI13
aI315
aI841
aI1543
aI411
aI6883
aI6347
asS'd'
I1861
sa(dp3722
S'a'
I8133
sS's'
I14
sS'm'
(lp3723
I1
aI3
aI3
aI11
aI23
aI43
aI23
aI131
aI17
aI517
aI995
aI2687
aI7443
aI15085
asS'd'
I1862
sa(dp3724
S'a'
I8134
sS's'
I14
sS'm'
(lp3725
I1
aI1
aI1
aI1
aI11
aI57
aI73
aI9
aI123
aI905
aI1763
aI1789
aI3701
aI7131
asS'd'
I1863
sa(dp3726
S'a'
I8143
sS's'
I14
sS'm'
(lp3727
I1
aI1
aI3
aI5
aI9
aI53
aI99
aI229
aI43
aI207
aI625
aI1583
aI6727
aI15249
asS'd'
I1864
sa(dp3728
S'a'
I8162
sS's'
I14
sS'm'
(lp3729
I1
aI1
aI7
aI7
aI17
aI39
aI91
aI1
aI297
aI711
aI225
aI513
aI7391
aI291
asS'd'
I1865
sa(dp3730
S'a'
I8168
sS's'
I14
sS'm'
(lp3731
I1
aI1
aI7
aI11
aI7
aI55
aI111
aI129
aI423
aI521
aI1807
aI3015
aI1449
aI12321
asS'd'
I1866
sa(dp3732
S'a'
I8179
sS's'
I14
sS'm'
(lp3733
I1
aI3
aI7
aI3
aI13
aI9
aI125
aI187
aI11
aI485
aI647
aI275
aI3495
aI11989
asS'd'
I1867
sa(dp3734
S'a'
I1
sS's'
I15
sS'm'
(lp3735
I1
aI1
aI3
aI11
aI11
aI25
aI49
aI33
aI361
aI105
aI271
aI3841
aI4837
aI2437
aI30181
asS'd'
I1868
sa(dp3736
S'a'
I8
sS's'
I15
sS'm'
(lp3737
I1
aI3
aI5
aI1
aI27
aI15
aI119
aI35
aI159
aI273
aI1489
aI3157
aI5433
aI3337
aI26859
asS'd'
I1869
sa(dp3738
S'a'
I11
sS's'
I15
sS'm'
(lp3739
I1
aI3
aI5
aI13
aI23
aI31
aI97
aI145
aI41
aI605
aI1455
aI59
aI5389
aI5527
aI14447
asS'd'
I1870
sa(dp3740
S'a'
I22
sS's'
I15
sS'm'
(lp3741
I1
aI1
aI7
aI9
aI7
aI41
aI61
aI193
aI353
aI879
aI1805
aI581
aI5447
aI11177
aI7331
asS'd'
I1871
sa(dp3742
S'a'
I26
sS's'
I15
sS'm'
(lp3743
I1
aI1
aI7
aI11
aI29
aI19
aI55
aI207
aI361
aI759
aI63
aI2255
aI2119
aI14671
aI21783
asS'd'
I1872
sa(dp3744
S'a'
I47
sS's'
I15
sS'm'
(lp3745
I1
aI3
aI1
aI13
aI17
aI7
aI73
aI179
aI103
aI23
aI917
aI1205
aI4925
aI1691
aI5419
asS'd'
I1873
sa(dp3746
S'a'
I59
sS's'
I15
sS'm'
(lp3747
I1
aI3
aI5
aI3
aI15
aI3
aI9
aI109
aI227
aI861
aI867
aI3529
aI1535
aI489
aI22873
asS'd'
I1874
sa(dp3748
S'a'
I64
sS's'
I15
sS'm'
(lp3749
I1
aI3
aI3
aI9
aI15
aI15
aI95
aI193
aI385
aI997
aI1525
aI1865
aI1425
aI4079
aI14771
asS'd'
I1875
sa(dp3750
S'a'
I67
sS's'
I15
sS'm'
(lp3751
I1
aI1
aI3
aI5
aI5
aI29
aI49
aI171
aI171
aI623
aI1167
aI3743
aI1809
aI12009
aI7043
asS'd'
I1876
sa(dp3752
S'a'
I73
sS's'
I15
sS'm'
(lp3753
I1
aI3
aI7
aI5
aI23
aI11
aI87
aI183
aI299
aI555
aI1857
aI489
aI3505
aI9161
aI28763
asS'd'
I1877
sa(dp3754
S'a'
I82
sS's'
I15
sS'm'
(lp3755
I1
aI3
aI5
aI9
aI19
aI21
aI85
aI127
aI337
aI439
aI1183
aI1891
aI1877
aI4373
aI10451
asS'd'
I1878
sa(dp3756
S'a'
I97
sS's'
I15
sS'm'
(lp3757
I1
aI3
aI7
aI13
aI27
aI17
aI29
aI83
aI463
aI385
aI1167
aI3453
aI4523
aI4759
aI9321
asS'd'
I1879
sa(dp3758
S'a'
I103
sS's'
I15
sS'm'
(lp3759
I1
aI1
aI3
aI7
aI21
aI59
aI65
aI83
aI177
aI763
aI317
aI2913
aI7527
aI5967
aI17167
asS'd'
I1880
sa(dp3760
S'a'
I110
sS's'
I15
sS'm'
(lp3761
I1
aI1
aI7
aI15
aI13
aI27
aI49
aI35
aI253
aI101
aI1699
aI355
aI2181
aI10859
aI24221
asS'd'
I1881
sa(dp3762
S'a'
I115
sS's'
I15
sS'm'
(lp3763
I1
aI1
aI5
aI1
aI17
aI17
aI81
aI91
aI349
aI655
aI1373
aI2225
aI945
aI899
aI31801
asS'd'
I1882
sa(dp3764
S'a'
I122
sS's'
I15
sS'm'
(lp3765
I1
aI3
aI7
aI11
aI5
aI1
aI81
aI53
aI215
aI587
aI167
aI4045
aI5671
aI5597
aI13529
asS'd'
I1883
sa(dp3766
S'a'
I128
sS's'
I15
sS'm'
(lp3767
I1
aI3
aI5
aI15
aI1
aI9
aI59
aI235
aI315
aI195
aI909
aI2237
aI505
aI10415
aI28145
asS'd'
I1884
sa(dp3768
S'a'
I138
sS's'
I15
sS'm'
(lp3769
I1
aI1
aI1
aI3
aI9
aI31
aI41
aI43
aI275
aI921
aI25
aI671
aI5737
aI11241
aI4193
asS'd'
I1885
sa(dp3770
S'a'
I146
sS's'
I15
sS'm'
(lp3771
I1
aI3
aI3
aI13
aI29
aI13
aI95
aI213
aI317
aI995
aI1489
aI3779
aI3043
aI8569
aI28823
asS'd'
I1886
sa(dp3772
S'a'
I171
sS's'
I15
sS'm'
(lp3773
I1
aI1
aI7
aI5
aI9
aI49
aI125
aI241
aI87
aI153
aI1673
aI3849
aI7253
aI1715
aI11627
asS'd'
I1887
sa(dp3774
S'a'
I174
sS's'
I15
sS'm'
(lp3775
I1
aI1
aI3
aI9
aI27
aI27
aI19
aI223
aI63
aI463
aI1095
aI1395
aI6643
aI11589
aI2145
asS'd'
I1888
sa(dp3776
S'a'
I176
sS's'
I15
sS'm'
(lp3777
I1
aI1
aI3
aI15
aI21
aI17
aI45
aI23
aI357
aI11
aI1307
aI1791
aI2481
aI2123
aI24341
asS'd'
I1889
sa(dp3778
S'a'
I182
sS's'
I15
sS'm'
(lp3779
I1
aI3
aI5
aI15
aI31
aI53
aI117
aI51
aI433
aI193
aI1239
aI3329
aI2403
aI12745
aI32219
asS'd'
I1890
sa(dp3780
S'a'
I194
sS's'
I15
sS'm'
(lp3781
I1
aI1
aI5
aI9
aI7
aI27
aI9
aI115
aI417
aI579
aI83
aI173
aI4717
aI15665
aI27463
asS'd'
I1891
sa(dp3782
S'a'
I208
sS's'
I15
sS'm'
(lp3783
I1
aI3
aI5
aI7
aI9
aI9
aI31
aI35
aI249
aI567
aI331
aI905
aI5101
aI14817
aI14255
asS'd'
I1892
sa(dp3784
S'a'
I211
sS's'
I15
sS'm'
(lp3785
I1
aI3
aI7
aI3
aI1
aI61
aI29
aI129
aI119
aI421
aI1597
aI2987
aI3041
aI7629
aI23451
asS'd'
I1893
sa(dp3786
S'a'
I220
sS's'
I15
sS'm'
(lp3787
I1
aI1
aI7
aI9
aI13
aI1
aI99
aI105
aI107
aI509
aI989
aI2259
aI1009
aI6827
aI8903
asS'd'
I1894
sa(dp3788
S'a'
I229
sS's'
I15
sS'm'
(lp3789
I1
aI3
aI5
aI15
aI11
aI29
aI85
aI29
aI265
aI105
aI2035
aI3349
aI3543
aI13903
aI10213
asS'd'
I1895
sa(dp3790
S'a'
I230
sS's'
I15
sS'm'
(lp3791
I1
aI3
aI1
aI1
aI25
aI19
aI53
aI139
aI467
aI485
aI491
aI3067
aI7353
aI13861
aI25819
asS'd'
I1896
sa(dp3792
S'a'
I239
sS's'
I15
sS'm'
(lp3793
I1
aI1
aI5
aI3
aI3
aI43
aI41
aI185
aI45
aI463
aI351
aI2823
aI2519
aI6705
aI11395
asS'd'
I1897
sa(dp3794
S'a'
I254
sS's'
I15
sS'm'
(lp3795
I1
aI3
aI7
aI13
aI11
aI15
aI87
aI221
aI427
aI673
aI1631
aI599
aI3259
aI10691
aI31283
asS'd'
I1898
sa(dp3796
S'a'
I265
sS's'
I15
sS'm'
(lp3797
I1
aI3
aI5
aI11
aI9
aI9
aI15
aI49
aI275
aI335
aI1613
aI3587
aI5309
aI14849
aI26475
asS'd'
I1899
sa(dp3798
S'a'
I285
sS's'
I15
sS'm'
(lp3799
I1
aI3
aI7
aI9
aI29
aI13
aI79
aI225
aI381
aI781
aI1411
aI2761
aI7157
aI14983
aI19717
asS'd'
I1900
sa(dp3800
S'a'
I290
sS's'
I15
sS'm'
(lp3801
I1
aI1
aI7
aI11
aI29
aI25
aI117
aI183
aI101
aI651
aI653
aI3157
aI445
aI14389
aI23293
asS'd'
I1901
sa(dp3802
S'a'
I319
sS's'
I15
sS'm'
(lp3803
I1
aI1
aI1
aI3
aI5
aI33
aI73
aI155
aI473
aI387
aI591
aI2045
aI5965
aI16299
aI31499
asS'd'
I1902
sa(dp3804
S'a'
I324
sS's'
I15
sS'm'
(lp3805
I1
aI3
aI1
aI7
aI11
aI33
aI29
aI21
aI491
aI937
aI729
aI4075
aI975
aI2461
aI18991
asS'd'
I1903
sa(dp3806
S'a'
I327
sS's'
I15
sS'm'
(lp3807
I1
aI3
aI7
aI15
aI29
aI39
aI105
aI111
aI173
aI943
aI69
aI295
aI8175
aI13037
aI26131
asS'd'
I1904
sa(dp3808
S'a'
I333
sS's'
I15
sS'm'
(lp3809
I1
aI1
aI5
aI15
aI7
aI5
aI97
aI147
aI105
aI887
aI443
aI2595
aI5889
aI10753
aI1619
asS'd'
I1905
sa(dp3810
S'a'
I357
sS's'
I15
sS'm'
(lp3811
I1
aI3
aI3
aI15
aI11
aI45
aI87
aI207
aI353
aI909
aI1847
aI323
aI2283
aI12885
aI16415
asS'd'
I1906
sa(dp3812
S'a'
I364
sS's'
I15
sS'm'
(lp3813
I1
aI1
aI5
aI3
aI19
aI33
aI43
aI79
aI115
aI653
aI359
aI2873
aI4609
aI12439
aI6339
asS'd'
I1907
sa(dp3814
S'a'
I395
sS's'
I15
sS'm'
(lp3815
I1
aI3
aI7
aI9
aI17
aI61
aI49
aI227
aI291
aI69
aI1753
aI3899
aI483
aI3187
aI29041
asS'd'
I1908
sa(dp3816
S'a'
I397
sS's'
I15
sS'm'
(lp3817
I1
aI3
aI5
aI3
aI25
aI35
aI61
aI211
aI393
aI199
aI691
aI1779
aI6295
aI13371
aI15817
asS'd'
I1909
sa(dp3818
S'a'
I405
sS's'
I15
sS'm'
(lp3819
I1
aI3
aI7
aI5
aI7
aI23
aI37
aI91
aI245
aI915
aI579
aI867
aI6193
aI1063
aI17363
asS'd'
I1910
sa(dp3820
S'a'
I409
sS's'
I15
sS'm'
(lp3821
I1
aI3
aI7
aI7
aI23
aI51
aI41
aI63
aI375
aI3
aI159
aI1889
aI4419
aI1687
aI17977
asS'd'
I1911
sa(dp3822
S'a'
I419
sS's'
I15
sS'm'
(lp3823
I1
aI1
aI1
aI7
aI13
aI11
aI53
aI43
aI317
aI325
aI1749
aI2423
aI4123
aI8595
aI20773
asS'd'
I1912
sa(dp3824
S'a'
I422
sS's'
I15
sS'm'
(lp3825
I1
aI1
aI7
aI7
aI9
aI9
aI61
aI113
aI437
aI213
aI1407
aI645
aI4345
aI807
aI30411
asS'd'
I1913
sa(dp3826
S'a'
I431
sS's'
I15
sS'm'
(lp3827
I1
aI3
aI3
aI11
aI17
aI39
aI17
aI113
aI391
aI385
aI581
aI2023
aI7449
aI10153
aI22033
asS'd'
I1914
sa(dp3828
S'a'
I433
sS's'
I15
sS'm'
(lp3829
I1
aI1
aI3
aI5
aI29
aI31
aI101
aI215
aI379
aI377
aI1113
aI2855
aI7147
aI14377
aI25515
asS'd'
I1915
sa(dp3830
S'a'
I436
sS's'
I15
sS'm'
(lp3831
I1
aI3
aI5
aI5
aI13
aI3
aI121
aI125
aI227
aI969
aI11
aI1115
aI5657
aI9209
aI6117
asS'd'
I1916
sa(dp3832
S'a'
I440
sS's'
I15
sS'm'
(lp3833
I1
aI3
aI7
aI15
aI29
aI17
aI33
aI123
aI317
aI301
aI749
aI1365
aI5619
aI605
aI1613
asS'd'
I1917
sa(dp3834
S'a'
I453
sS's'
I15
sS'm'
(lp3835
I1
aI3
aI1
aI15
aI7
aI53
aI125
aI249
aI219
aI655
aI105
aI2825
aI1649
aI12783
aI19777
asS'd'
I1918
sa(dp3836
S'a'
I460
sS's'
I15
sS'm'
(lp3837
I1
aI1
aI7
aI1
aI25
aI53
aI19
aI53
aI157
aI373
aI1855
aI495
aI5065
aI9465
aI2313
asS'd'
I1919
sa(dp3838
S'a'
I471
sS's'
I15
sS'm'
(lp3839
I1
aI3
aI5
aI13
aI3
aI57
aI57
aI161
aI431
aI415
aI1859
aI1033
aI6349
aI1577
aI31579
asS'd'
I1920
sa(dp3840
S'a'
I478
sS's'
I15
sS'm'
(lp3841
I1
aI1
aI7
aI5
aI23
aI63
aI29
aI221
aI13
aI965
aI1997
aI2265
aI1583
aI10491
aI9551
asS'd'
I1921
sa(dp3842
S'a'
I482
sS's'
I15
sS'm'
(lp3843
I1
aI1
aI3
aI13
aI31
aI25
aI23
aI61
aI285
aI5
aI2005
aI879
aI795
aI13299
aI19685
asS'd'
I1922
sa(dp3844
S'a'
I488
sS's'
I15
sS'm'
(lp3845
I1
aI1
aI7
aI1
aI21
aI45
aI121
aI89
aI263
aI543
aI1333
aI2711
aI219
aI10823
aI26139
asS'd'
I1923
sa(dp3846
S'a'
I524
sS's'
I15
sS'm'
(lp3847
I1
aI1
aI3
aI3
aI27
aI13
aI19
aI117
aI161
aI457
aI1541
aI295
aI4953
aI12125
aI14503
asS'd'
I1924
sa(dp3848
S'a'
I529
sS's'
I15
sS'm'
(lp3849
I1
aI3
aI5
aI3
aI7
aI63
aI13
aI247
aI439
aI681
aI977
aI2537
aI6923
aI10323
aI7349
asS'd'
I1925
sa(dp3850
S'a'
I535
sS's'
I15
sS'm'
(lp3851
I1
aI3
aI5
aI9
aI3
aI51
aI81
aI251
aI349
aI983
aI581
aI2515
aI2281
aI2849
aI31915
asS'd'
I1926
sa(dp3852
S'a'
I536
sS's'
I15
sS'm'
(lp3853
I1
aI3
aI5
aI3
aI11
aI63
aI47
aI137
aI303
aI627
aI91
aI2269
aI7097
aI2145
aI31059
asS'd'
I1927
sa(dp3854
S'a'
I539
sS's'
I15
sS'm'
(lp3855
I1
aI1
aI3
aI15
aI13
aI17
aI53
aI27
aI133
aI13
aI117
aI1837
aI4103
aI5843
aI29153
asS'd'
I1928
sa(dp3856
S'a'
I563
sS's'
I15
sS'm'
(lp3857
I1
aI1
aI5
aI13
aI21
aI33
aI37
aI253
aI465
aI209
aI309
aI49
aI3209
aI15677
aI14569
asS'd'
I1929
sa(dp3858
S'a'
I566
sS's'
I15
sS'm'
(lp3859
I1
aI1
aI7
aI15
aI13
aI21
aI33
aI203
aI499
aI141
aI1155
aI3893
aI1663
aI2115
aI27459
asS'd'
I1930
sa(dp3860
S'a'
I572
sS's'
I15
sS'm'
(lp3861
I1
aI3
aI5
aI11
aI21
aI9
aI39
aI157
aI257
aI273
aI1257
aI1831
aI515
aI7969
aI20133
asS'd'
I1931
sa(dp3862
S'a'
I577
sS's'
I15
sS'm'
(lp3863
I1
aI1
aI3
aI13
aI19
aI29
aI15
aI189
aI103
aI219
aI1395
aI517
aI7425
aI6585
aI15865
asS'd'
I1932
sa(dp3864
S'a'
I587
sS's'
I15
sS'm'
(lp3865
I1
aI1
aI5
aI11
aI21
aI31
aI49
aI151
aI39
aI537
aI1783
aI3449
aI6915
aI223
aI11587
asS'd'
I1933
sa(dp3866
S'a'
I592
sS's'
I15
sS'm'
(lp3867
I1
aI3
aI3
aI11
aI7
aI63
aI69
aI31
aI27
aI911
aI1903
aI2821
aI7977
aI12949
aI32257
asS'd'
I1934
sa(dp3868
S'a'
I602
sS's'
I15
sS'm'
(lp3869
I1
aI1
aI7
aI9
aI25
aI45
aI23
aI233
aI511
aI595
aI1383
aI1721
aI6789
aI12055
aI21179
asS'd'
I1935
sa(dp3870
S'a'
I623
sS's'
I15
sS'm'
(lp3871
I1
aI1
aI7
aI13
aI1
aI27
aI123
aI49
aI439
aI683
aI501
aI641
aI1947
aI6111
aI25423
asS'd'
I1936
sa(dp3872
S'a'
I635
sS's'
I15
sS'm'
(lp3873
I1
aI3
aI3
aI5
aI1
aI23
aI57
aI241
aI243
aI593
aI2039
aI1617
aI2209
aI5171
aI9675
asS'd'
I1937
sa(dp3874
S'a'
I638
sS's'
I15
sS'm'
(lp3875
I1
aI1
aI1
aI7
aI5
aI19
aI83
aI55
aI481
aI125
aI177
aI1021
aI1139
aI11403
aI23099
asS'd'
I1938
sa(dp3876
S'a'
I654
sS's'
I15
sS'm'
(lp3877
I1
aI1
aI3
aI5
aI29
aI39
aI33
aI217
aI461
aI907
aI733
aI3795
aI4811
aI12939
aI27715
asS'd'
I1939
sa(dp3878
S'a'
I656
sS's'
I15
sS'm'
(lp3879
I1
aI3
aI7
aI3
aI7
aI11
aI39
aI165
aI495
aI147
aI999
aI1827
aI817
aI603
aI9293
asS'd'
I1940
sa(dp3880
S'a'
I659
sS's'
I15
sS'm'
(lp3881
I1
aI3
aI7
aI15
aI25
aI53
aI35
aI15
aI431
aI733
aI1213
aI2907
aI8087
aI3939
aI27363
asS'd'
I1941
sa(dp3882
S'a'
I665
sS's'
I15
sS'm'
(lp3883
I1
aI3
aI7
aI13
aI13
aI9
aI33
aI27
aI485
aI183
aI455
aI3341
aI2555
aI4985
aI8793
asS'd'
I1942
sa(dp3884
S'a'
I675
sS's'
I15
sS'm'
(lp3885
I1
aI1
aI1
aI15
aI25
aI47
aI75
aI21
aI205
aI15
aI1639
aI3067
aI1295
aI11693
aI16903
asS'd'
I1943
sa(dp3886
S'a'
I677
sS's'
I15
sS'm'
(lp3887
I1
aI1
aI1
aI15
aI3
aI31
aI93
aI57
aI43
aI185
aI251
aI1899
aI7885
aI10829
aI3609
asS'd'
I1944
sa(dp3888
S'a'
I687
sS's'
I15
sS'm'
(lp3889
I1
aI1
aI3
aI1
aI29
aI9
aI69
aI223
aI221
aI537
aI365
aI3411
aI5771
aI15279
aI5309
asS'd'
I1945
sa(dp3890
S'a'
I696
sS's'
I15
sS'm'
(lp3891
I1
aI1
aI7
aI5
aI1
aI5
aI125
aI243
aI213
aI1003
aI1571
aI3355
aI3981
aI8781
aI25993
asS'd'
I1946
sa(dp3892
S'a'
I701
sS's'
I15
sS'm'
(lp3893
I1
aI1
aI1
aI13
aI7
aI19
aI53
aI243
aI301
aI75
aI1183
aI2723
aI6687
aI13
aI16581
asS'd'
I1947
sa(dp3894
S'a'
I704
sS's'
I15
sS'm'
(lp3895
I1
aI3
aI1
aI13
aI17
aI51
aI91
aI239
aI437
aI191
aI1065
aI2495
aI5755
aI3405
aI8299
asS'd'
I1948
sa(dp3896
S'a'
I710
sS's'
I15
sS'm'
(lp3897
I1
aI1
aI5
aI5
aI11
aI59
aI21
aI169
aI299
aI123
aI1845
aI2199
aI2157
aI14461
aI10327
asS'd'
I1949
sa(dp3898
S'a'
I721
sS's'
I15
sS'm'
(lp3899
I1
aI3
aI7
aI7
aI19
aI47
aI51
aI179
aI41
aI19
aI1347
aI2325
aI8063
aI5993
aI15653
asS'd'
I1950
sa(dp3900
S'a'
I728
sS's'
I15
sS'm'
(lp3901
I1
aI1
aI1
aI9
aI25
aI27
aI7
aI133
aI223
aI533
aI719
aI353
aI7093
aI8285
aI10375
asS'd'
I1951
sa(dp3902
S'a'
I738
sS's'
I15
sS'm'
(lp3903
I1
aI3
aI5
aI15
aI31
aI5
aI67
aI39
aI441
aI495
aI977
aI3699
aI1435
aI11385
aI14567
asS'd'
I1952
sa(dp3904
S'a'
I740
sS's'
I15
sS'm'
(lp3905
I1
aI1
aI3
aI15
aI15
aI39
aI25
aI33
aI91
aI523
aI249
aI4035
aI769
aI5181
aI9691
asS'd'
I1953
sa(dp3906
S'a'
I749
sS's'
I15
sS'm'
(lp3907
I1
aI1
aI3
aI3
aI3
aI57
aI83
aI187
aI423
aI165
aI161
aI3453
aI2241
aI981
aI8429
asS'd'
I1954
sa(dp3908
S'a'
I758
sS's'
I15
sS'm'
(lp3909
I1
aI1
aI7
aI15
aI1
aI17
aI57
aI189
aI283
aI11
aI823
aI3505
aI7025
aI11879
aI15441
asS'd'
I1955
sa(dp3910
S'a'
I761
sS's'
I15
sS'm'
(lp3911
I1
aI1
aI3
aI11
aI1
aI41
aI7
aI255
aI385
aI339
aI607
aI1405
aI1473
aI13697
aI9491
asS'd'
I1956
sa(dp3912
S'a'
I772
sS's'
I15
sS'm'
(lp3913
I1
aI1
aI7
aI15
aI5
aI9
aI91
aI99
aI211
aI233
aI51
aI2663
aI1165
aI9283
aI18495
asS'd'
I1957
sa(dp3914
S'a'
I776
sS's'
I15
sS'm'
(lp3915
I1
aI1
aI3
aI7
aI21
aI37
aI13
aI91
aI39
aI27
aI1021
aI2813
aI5937
aI6645
aI3403
asS'd'
I1958
sa(dp3916
S'a'
I782
sS's'
I15
sS'm'
(lp3917
I1
aI3
aI1
aI1
aI29
aI29
aI5
aI69
aI399
aI665
aI1407
aI3921
aI2653
aI11753
aI18925
asS'd'
I1959
sa(dp3918
S'a'
I789
sS's'
I15
sS'm'
(lp3919
I1
aI3
aI7
aI15
aI13
aI41
aI39
aI1
aI437
aI549
aI161
aI2315
aI5631
aI8335
aI22661
asS'd'
I1960
sa(dp3920
S'a'
I810
sS's'
I15
sS'm'
(lp3921
I1
aI1
aI3
aI1
aI7
aI17
aI115
aI61
aI69
aI955
aI475
aI3763
aI8035
aI927
aI17893
asS'd'
I1961
sa(dp3922
S'a'
I812
sS's'
I15
sS'm'
(lp3923
I1
aI3
aI1
aI13
aI21
aI59
aI81
aI145
aI463
aI145
aI1941
aI2777
aI7453
aI14229
aI11281
asS'd'
I1962
sa(dp3924
S'a'
I818
sS's'
I15
sS'm'
(lp3925
I1
aI1
aI1
aI15
aI15
aI11
aI27
aI165
aI461
aI395
aI1645
aI3611
aI7463
aI12379
aI26787
asS'd'
I1963
sa(dp3926
S'a'
I830
sS's'
I15
sS'm'
(lp3927
I1
aI1
aI7
aI9
aI29
aI19
aI27
aI123
aI21
aI149
aI1643
aI4001
aI7207
aI6769
aI4647
asS'd'
I1964
sa(dp3928
S'a'
I832
sS's'
I15
sS'm'
(lp3929
I1
aI1
aI1
aI11
aI13
aI9
aI103
aI139
aI185
aI587
aI591
aI1113
aI2223
aI11667
aI32671
asS'd'
I1965
sa(dp3930
S'a'
I852
sS's'
I15
sS'm'
(lp3931
I1
aI3
aI1
aI1
aI31
aI13
aI19
aI93
aI229
aI125
aI1471
aI2369
aI3055
aI10277
aI28563
asS'd'
I1966
sa(dp3932
S'a'
I855
sS's'
I15
sS'm'
(lp3933
I1
aI3
aI7
aI5
aI7
aI53
aI99
aI175
aI161
aI851
aI617
aI4027
aI2357
aI11199
aI1931
asS'd'
I1967
sa(dp3934
S'a'
I859
sS's'
I15
sS'm'
(lp3935
I1
aI3
aI5
aI11
aI3
aI31
aI111
aI179
aI237
aI845
aI539
aI1057
aI259
aI3417
aI26637
asS'd'
I1968
sa(dp3936
S'a'
I865
sS's'
I15
sS'm'
(lp3937
I1
aI1
aI5
aI3
aI21
aI49
aI125
aI119
aI463
aI403
aI737
aI1811
aI3941
aI13015
aI29081
asS'd'
I1969
sa(dp3938
S'a'
I877
sS's'
I15
sS'm'
(lp3939
I1
aI3
aI5
aI13
aI5
aI29
aI69
aI251
aI313
aI357
aI663
aI1097
aI3307
aI12845
aI28495
asS'd'
I1970
sa(dp3940
S'a'
I895
sS's'
I15
sS'm'
(lp3941
I1
aI3
aI3
aI5
aI29
aI17
aI89
aI15
aI411
aI409
aI2013
aI757
aI4085
aI12521
aI11131
asS'd'
I1971
sa(dp3942
S'a'
I901
sS's'
I15
sS'm'
(lp3943
I1
aI1
aI1
aI15
aI7
aI51
aI3
aI193
aI493
aI133
aI381
aI2027
aI227
aI6635
aI12931
asS'd'
I1972
sa(dp3944
S'a'
I902
sS's'
I15
sS'm'
(lp3945
I1
aI1
aI1
aI15
aI7
aI23
aI99
aI203
aI323
aI1007
aI1465
aI2887
aI2215
aI1787
aI22069
asS'd'
I1973
sa(dp3946
S'a'
I906
sS's'
I15
sS'm'
(lp3947
I1
aI1
aI5
aI9
aI29
aI59
aI77
aI151
aI509
aI313
aI415
aI3977
aI5431
aI8019
aI8571
asS'd'
I1974
sa(dp3948
S'a'
I916
sS's'
I15
sS'm'
(lp3949
I1
aI3
aI1
aI15
aI19
aI13
aI57
aI217
aI87
aI119
aI25
aI1149
aI5667
aI3765
aI6959
asS'd'
I1975
sa(dp3950
S'a'
I920
sS's'
I15
sS'm'
(lp3951
I1
aI3
aI7
aI13
aI19
aI31
aI119
aI3
aI457
aI117
aI905
aI361
aI1483
aI12405
aI27005
asS'd'
I1976
sa(dp3952
S'a'
I949
sS's'
I15
sS'm'
(lp3953
I1
aI3
aI5
aI11
aI15
aI35
aI61
aI77
aI119
aI51
aI1753
aI2765
aI1091
aI10573
aI23595
asS'd'
I1977
sa(dp3954
S'a'
I962
sS's'
I15
sS'm'
(lp3955
I1
aI3
aI3
aI7
aI1
aI35
aI17
aI93
aI197
aI511
aI1253
aI3031
aI2739
aI15127
aI15147
asS'd'
I1978
sa(dp3956
S'a'
I964
sS's'
I15
sS'm'
(lp3957
I1
aI3
aI3
aI1
aI11
aI55
aI55
aI107
aI161
aI75
aI129
aI2195
aI2023
aI4877
aI25797
asS'd'
I1979
sa(dp3958
S'a'
I967
sS's'
I15
sS'm'
(lp3959
I1
aI3
aI5
aI7
aI23
aI19
aI113
aI167
aI167
aI271
aI1303
aI125
aI5057
aI1323
aI5165
asS'd'
I1980
sa(dp3960
S'a'
I981
sS's'
I15
sS'm'
(lp3961
I1
aI1
aI5
aI3
aI21
aI31
aI11
aI119
aI215
aI483
aI1535
aI407
aI6485
aI15401
aI30297
asS'd'
I1981
sa(dp3962
S'a'
I982
sS's'
I15
sS'm'
(lp3963
I1
aI3
aI5
aI9
aI21
aI5
aI77
aI95
aI443
aI247
aI913
aI605
aI365
aI7465
aI19707
asS'd'
I1982
sa(dp3964
S'a'
I985
sS's'
I15
sS'm'
(lp3965
I1
aI3
aI1
aI7
aI17
aI59
aI9
aI35
aI391
aI767
aI1493
aI475
aI4725
aI7529
aI31579
asS'd'
I1983
sa(dp3966
S'a'
I991
sS's'
I15
sS'm'
(lp3967
I1
aI3
aI3
aI7
aI31
aI21
aI61
aI31
aI421
aI179
aI273
aI771
aI5745
aI10575
aI32765
asS'd'
I1984
sa(dp3968
S'a'
I1007
sS's'
I15
sS'm'
(lp3969
I1
aI3
aI5
aI15
aI27
aI13
aI125
aI55
aI423
aI1021
aI497
aI3521
aI6903
aI15111
aI8285
asS'd'
I1985
sa(dp3970
S'a'
I1016
sS's'
I15
sS'm'
(lp3971
I1
aI1
aI5
aI9
aI13
aI31
aI105
aI93
aI421
aI709
aI643
aI1079
aI1533
aI9149
aI10799
asS'd'
I1986
sa(dp3972
S'a'
I1024
sS's'
I15
sS'm'
(lp3973
I1
aI3
aI1
aI11
aI19
aI29
aI53
aI199
aI319
aI247
aI655
aI3039
aI6411
aI12267
aI14245
asS'd'
I1987
sa(dp3974
S'a'
I1051
sS's'
I15
sS'm'
(lp3975
I1
aI3
aI1
aI11
aI9
aI57
aI5
aI91
aI469
aI149
aI259
aI329
aI5433
aI6941
aI15093
asS'd'
I1988
sa(dp3976
S'a'
I1060
sS's'
I15
sS'm'
(lp3977
I1
aI3
aI1
aI5
aI5
aI51
aI59
aI25
aI455
aI367
aI1623
aI441
aI3155
aI11695
aI20767
asS'd'
I1989
sa(dp3978
S'a'
I1070
sS's'
I15
sS'm'
(lp3979
I1
aI3
aI7
aI7
aI11
aI49
aI113
aI95
aI91
aI389
aI605
aI1973
aI2051
aI2315
aI22229
asS'd'
I1990
sa(dp3980
S'a'
I1072
sS's'
I15
sS'm'
(lp3981
I1
aI3
aI5
aI3
aI19
aI11
aI99
aI135
aI433
aI781
aI1473
aI885
aI1105
aI3573
aI3739
asS'd'
I1991
sa(dp3982
S'a'
I1084
sS's'
I15
sS'm'
(lp3983
I1
aI3
aI1
aI11
aI3
aI25
aI9
aI227
aI433
aI723
aI317
aI139
aI6627
aI8067
aI28439
asS'd'
I1992
sa(dp3984
S'a'
I1089
sS's'
I15
sS'm'
(lp3985
I1
aI1
aI1
aI9
aI9
aI9
aI5
aI63
aI241
aI215
aI1991
aI2949
aI3943
aI775
aI31511
asS'd'
I1993
sa(dp3986
S'a'
I1095
sS's'
I15
sS'm'
(lp3987
I1
aI1
aI3
aI7
aI17
aI49
aI35
aI167
aI131
aI107
aI1295
aI2465
aI4577
aI11147
aI29833
asS'd'
I1994
sa(dp3988
S'a'
I1114
sS's'
I15
sS'm'
(lp3989
I1
aI1
aI5
aI1
aI5
aI25
aI119
aI129
aI391
aI743
aI1069
aI2957
aI349
aI6891
aI13635
asS'd'
I1995
sa(dp3990
S'a'
I1123
sS's'
I15
sS'm'
(lp3991
I1
aI3
aI1
aI7
aI9
aI31
aI63
aI253
aI215
aI51
aI1347
aI2361
aI3125
aI13049
aI28461
asS'd'
I1996
sa(dp3992
S'a'
I1132
sS's'
I15
sS'm'
(lp3993
I1
aI1
aI7
aI9
aI3
aI31
aI21
aI163
aI255
aI47
aI259
aI535
aI5461
aI3349
aI30649
asS'd'
I1997
sa(dp3994
S'a'
I1154
sS's'
I15
sS'm'
(lp3995
I1
aI3
aI3
aI13
aI17
aI33
aI87
aI47
aI243
aI709
aI929
aI3943
aI3107
aI3421
aI13721
asS'd'
I1998
sa(dp3996
S'a'
I1156
sS's'
I15
sS'm'
(lp3997
I1
aI3
aI5
aI11
aI25
aI61
aI61
aI173
aI397
aI735
aI2005
aI3355
aI8121
aI11593
aI27697
asS'd'
I1999
sa(dp3998
S'a'
I1163
sS's'
I15
sS'm'
(lp3999
I1
aI3
aI5
aI15
aI17
aI43
aI63
aI231
aI275
aI311
aI1277
aI2669
aI7307
aI2099
aI9755
asS'd'
I2000
sa(dp4000
S'a'
I1171
sS's'
I15
sS'm'
(lp4001
I1
aI3
aI5
aI3
aI25
aI43
aI71
aI191
aI9
aI121
aI1873
aI3747
aI7491
aI14055
aI24293
asS'd'
I2001
sa(dp4002
S'a'
I1202
sS's'
I15
sS'm'
(lp4003
I1
aI3
aI5
aI13
aI17
aI35
aI113
aI113
aI385
aI941
aI39
aI2705
aI1225
aI5167
aI1373
asS'd'
I2002
sa(dp4004
S'a'
I1228
sS's'
I15
sS'm'
(lp4005
I1
aI3
aI5
aI5
aI7
aI35
aI19
aI105
aI487
aI71
aI139
aI627
aI4187
aI3183
aI713
asS'd'
I2003
sa(dp4006
S'a'
I1239
sS's'
I15
sS'm'
(lp4007
I1
aI1
aI5
aI13
aI29
aI29
aI103
aI5
aI157
aI869
aI1675
aI423
aI6689
aI10697
aI5303
asS'd'
I2004
sa(dp4008
S'a'
I1255
sS's'
I15
sS'm'
(lp4009
I1
aI1
aI5
aI1
aI29
aI31
aI61
aI111
aI473
aI963
aI685
aI1483
aI2383
aI8109
aI8495
asS'd'
I2005
sa(dp4010
S'a'
I1256
sS's'
I15
sS'm'
(lp4011
I1
aI1
aI5
aI3
aI19
aI13
aI95
aI113
aI217
aI59
aI1353
aI1647
aI3617
aI3271
aI2321
asS'd'
I2006
sa(dp4012
S'a'
I1262
sS's'
I15
sS'm'
(lp4013
I1
aI3
aI5
aI7
aI25
aI35
aI59
aI131
aI309
aI445
aI415
aI93
aI1453
aI8789
aI30201
asS'd'
I2007
sa(dp4014
S'a'
I1270
sS's'
I15
sS'm'
(lp4015
I1
aI1
aI5
aI1
aI5
aI43
aI71
aI241
aI123
aI189
aI831
aI3469
aI8093
aI6187
aI32721
asS'd'
I2008
sa(dp4016
S'a'
I1279
sS's'
I15
sS'm'
(lp4017
I1
aI3
aI7
aI5
aI25
aI31
aI123
aI171
aI319
aI379
aI889
aI2365
aI4881
aI12225
aI16609
asS'd'
I2009
sa(dp4018
S'a'
I1308
sS's'
I15
sS'm'
(lp4019
I1
aI3
aI1
aI11
aI27
aI43
aI121
aI63
aI291
aI591
aI811
aI1995
aI4777
aI2083
aI31385
asS'd'
I2010
sa(dp4020
S'a'
I1322
sS's'
I15
sS'm'
(lp4021
I1
aI1
aI5
aI11
aI27
aI53
aI85
aI187
aI461
aI823
aI703
aI399
aI6925
aI11517
aI28697
asS'd'
I2011
sa(dp4022
S'a'
I1329
sS's'
I15
sS'm'
(lp4023
I1
aI1
aI3
aI5
aI13
aI11
aI33
aI121
aI93
aI717
aI1275
aI3877
aI4247
aI5845
aI26909
asS'd'
I2012
sa(dp4024
S'a'
I1330
sS's'
I15
sS'm'
(lp4025
I1
aI3
aI1
aI9
aI7
aI5
aI47
aI199
aI367
aI561
aI185
aI2855
aI5997
aI2699
aI7581
asS'd'
I2013
sa(dp4026
S'a'
I1336
sS's'
I15
sS'm'
(lp4027
I1
aI1
aI5
aI9
aI23
aI11
aI71
aI201
aI61
aI729
aI1011
aI3529
aI663
aI1413
aI25675
asS'd'
I2014
sa(dp4028
S'a'
I1341
sS's'
I15
sS'm'
(lp4029
I1
aI3
aI7
aI13
aI27
aI21
aI11
aI127
aI281
aI487
aI1217
aI3129
aI5541
aI3129
aI17783
asS'd'
I2015
sa(dp4030
S'a'
I1347
sS's'
I15
sS'm'
(lp4031
I1
aI1
aI5
aI9
aI1
aI29
aI85
aI193
aI213
aI743
aI1473
aI611
aI391
aI9405
aI21137
asS'd'
I2016
sa(dp4032
S'a'
I1349
sS's'
I15
sS'm'
(lp4033
I1
aI3
aI3
aI3
aI31
aI63
aI37
aI147
aI39
aI351
aI79
aI3069
aI2441
aI8901
aI8777
asS'd'
I2017
sa(dp4034
S'a'
I1359
sS's'
I15
sS'm'
(lp4035
I1
aI1
aI7
aI7
aI25
aI49
aI55
aI47
aI441
aI343
aI1267
aI1123
aI5917
aI14395
aI10579
asS'd'
I2018
sa(dp4036
S'a'
I1367
sS's'
I15
sS'm'
(lp4037
I1
aI1
aI7
aI1
aI13
aI55
aI55
aI123
aI103
aI773
aI125
aI2145
aI4743
aI13347
aI2589
asS'd'
I2019
sa(dp4038
S'a'
I1368
sS's'
I15
sS'm'
(lp4039
I1
aI3
aI7
aI3
aI9
aI33
aI25
aI183
aI469
aI213
aI291
aI75
aI6725
aI6847
aI26745
asS'd'
I2020
sa(dp4040
S'a'
I1390
sS's'
I15
sS'm'
(lp4041
I1
aI3
aI3
aI7
aI15
aI43
aI7
aI79
aI171
aI21
aI1767
aI2537
aI4285
aI12007
aI24039
asS'd'
I2021
sa(dp4042
S'a'
I1413
sS's'
I15
sS'm'
(lp4043
I1
aI3
aI7
aI13
aI9
aI61
aI125
aI23
aI227
aI879
aI215
aI1635
aI2835
aI883
aI15939
asS'd'
I2022
sa(dp4044
S'a'
I1414
sS's'
I15
sS'm'
(lp4045
I1
aI1
aI5
aI13
aI25
aI45
aI63
aI43
aI183
aI829
aI149
aI989
aI987
aI3819
aI12181
asS'd'
I2023
sa(dp4046
S'a'
I1437
sS's'
I15
sS'm'
(lp4047
I1
aI1
aI3
aI7
aI19
aI27
aI35
aI83
aI135
aI459
aI785
aI131
aI2655
aI3329
aI3009
asS'd'
I2024
sa(dp4048
S'a'
I1441
sS's'
I15
sS'm'
(lp4049
I1
aI1
aI7
aI5
aI11
aI41
aI9
aI219
aI475
aI985
aI1329
aI3787
aI1975
aI4679
aI8627
asS'd'
I2025
sa(dp4050
S'a'
I1462
sS's'
I15
sS'm'
(lp4051
I1
aI1
aI7
aI3
aI1
aI17
aI91
aI155
aI3
aI763
aI1879
aI233
aI215
aI2955
aI25993
asS'd'
I2026
sa(dp4052
S'a'
I1465
sS's'
I15
sS'm'
(lp4053
I1
aI1
aI1
aI11
aI25
aI11
aI23
aI227
aI453
aI775
aI1935
aI3833
aI4583
aI269
aI705
asS'd'
I2027
sa(dp4054
S'a'
I1480
sS's'
I15
sS'm'
(lp4055
I1
aI3
aI3
aI11
aI7
aI25
aI105
aI21
aI449
aI555
aI1275
aI3475
aI5503
aI15617
aI813
asS'd'
I2028
sa(dp4056
S'a'
I1486
sS's'
I15
sS'm'
(lp4057
I1
aI3
aI7
aI13
aI31
aI37
aI25
aI255
aI233
aI663
aI1155
aI1563
aI4775
aI7449
aI29949
asS'd'
I2029
sa(dp4058
S'a'
I1504
sS's'
I15
sS'm'
(lp4059
I1
aI1
aI3
aI1
aI23
aI51
aI51
aI137
aI63
aI809
aI349
aI2789
aI6953
aI10605
aI18959
asS'd'
I2030
sa(dp4060
S'a'
I1509
sS's'
I15
sS'm'
(lp4061
I1
aI3
aI3
aI13
aI21
aI45
aI15
aI161
aI393
aI229
aI437
aI2967
aI4019
aI3893
aI21305
asS'd'
I2031
sa(dp4062
S'a'
I1514
sS's'
I15
sS'm'
(lp4063
I1
aI1
aI3
aI7
aI5
aI11
aI15
aI211
aI287
aI131
aI1847
aI2569
aI7881
aI15669
aI31037
asS'd'
I2032
sa(dp4064
S'a'
I1522
sS's'
I15
sS'm'
(lp4065
I1
aI3
aI3
aI15
aI27
aI19
aI85
aI251
aI221
aI639
aI665
aI3729
aI5771
aI7873
aI28005
asS'd'
I2033
sa(dp4066
S'a'
I1528
sS's'
I15
sS'm'
(lp4067
I1
aI3
aI7
aI15
aI15
aI47
aI93
aI215
aI343
aI85
aI1401
aI1375
aI2949
aI13661
aI25453
asS'd'
I2034
sa(dp4068
S'a'
I1552
sS's'
I15
sS'm'
(lp4069
I1
aI1
aI1
aI9
aI7
aI51
aI53
aI217
aI471
aI389
aI551
aI1141
aI1767
aI2237
aI17797
asS'd'
I2035
sa(dp4070
S'a'
I1555
sS's'
I15
sS'm'
(lp4071
I1
aI1
aI7
aI9
aI3
aI29
aI65
aI29
aI223
aI591
aI1719
aI1049
aI7643
aI3853
aI29867
asS'd'
I2036
sa(dp4072
S'a'
I1571
sS's'
I15
sS'm'
(lp4073
I1
aI1
aI1
aI11
aI13
aI41
aI85
aI29
aI451
aI387
aI1783
aI3733
aI8033
aI4711
aI31643
asS'd'
I2037
sa(dp4074
S'a'
I1578
sS's'
I15
sS'm'
(lp4075
I1
aI3
aI1
aI11
aI11
aI57
aI75
aI153
aI7
aI373
aI2011
aI271
aI469
aI3267
aI18969
asS'd'
I2038
sa(dp4076
S'a'
I1585
sS's'
I15
sS'm'
(lp4077
I1
aI1
aI5
aI3
aI19
aI43
aI7
aI243
aI385
aI293
aI923
aI843
aI4895
aI469
aI8421
asS'd'
I2039
sa(dp4078
S'a'
I1588
sS's'
I15
sS'm'
(lp4079
I1
aI3
aI1
aI15
aI29
aI47
aI17
aI125
aI471
aI927
aI349
aI3859
aI3059
aI11483
aI14791
asS'd'
I2040
sa(dp4080
S'a'
I1603
sS's'
I15
sS'm'
(lp4081
I1
aI3
aI1
aI11
aI17
aI17
aI111
aI109
aI9
aI213
aI1313
aI3903
aI4411
aI4329
aI28277
asS'd'
I2041
sa(dp4082
S'a'
I1609
sS's'
I15
sS'm'
(lp4083
I1
aI3
aI3
aI15
aI1
aI55
aI47
aI69
aI143
aI789
aI1149
aI3833
aI5053
aI6949
aI10569
asS'd'
I2042
sa(dp4084
S'a'
I1617
sS's'
I15
sS'm'
(lp4085
I1
aI3
aI5
aI7
aI11
aI15
aI79
aI83
aI123
aI937
aI1115
aI2775
aI3041
aI11869
aI21167
asS'd'
I2043
sa(dp4086
S'a'
I1620
sS's'
I15
sS'm'
(lp4087
I1
aI3
aI7
aI13
aI9
aI47
aI45
aI221
aI139
aI923
aI1661
aI1379
aI2485
aI7233
aI6035
asS'd'
I2044
sa(dp4088
S'a'
I1629
sS's'
I15
sS'm'
(lp4089
I1
aI1
aI3
aI3
aI11
aI55
aI77
aI3
aI87
aI693
aI1991
aI1145
aI2783
aI16207
aI24569
asS'd'
I2045
sa(dp4090
S'a'
I1636
sS's'
I15
sS'm'
(lp4091
I1
aI1
aI5
aI11
aI3
aI35
aI91
aI9
aI391
aI927
aI101
aI1839
aI3755
aI10345
aI16907
asS'd'
I2046
sa(dp4092
S'a'
I1648
sS's'
I15
sS'm'
(lp4093
I1
aI3
aI5
aI3
aI5
aI49
aI79
aI91
aI205
aI443
aI1369
aI197
aI2537
aI11219
aI17765
asS'd'
I2047
sa(dp4094
S'a'
I1667
sS's'
I15
sS'm'
(lp4095
I1
aI1
aI3
aI15
aI9
aI7
aI25
aI25
aI357
aI247
aI477
aI421
aI7679
aI5987
aI30079
asS'd'
I2048
sa(dp4096
S'a'
I1669
sS's'
I15
sS'm'
(lp4097
I1
aI1
aI5
aI3
aI29
aI5
aI89
aI117
aI481
aI491
aI371
aI389
aI7101
aI2253
aI23617
asS'd'
I2049
sa(dp4098
S'a'
I1682
sS's'
I15
sS'm'
(lp4099
I1
aI1
aI5
aI13
aI29
aI59
aI17
aI181
aI511
aI291
aI1991
aI3499
aI8177
aI5559
aI30045
asS'd'
I2050
sa(dp4100
S'a'
I1697
sS's'
I15
sS'm'
(lp4101
I1
aI3
aI3
aI11
aI23
aI31
aI117
aI217
aI241
aI115
aI749
aI945
aI1897
aI12253
aI8473
asS'd'
I2051
sa(dp4102
S'a'
I1704
sS's'
I15
sS'm'
(lp4103
I1
aI1
aI7
aI15
aI25
aI47
aI31
aI1
aI165
aI311
aI635
aI3629
aI1593
aI8305
aI30033
asS'd'
I2052
sa(dp4104
S'a'
I1709
sS's'
I15
sS'm'
(lp4105
I1
aI3
aI5
aI9
aI3
aI17
aI101
aI237
aI379
aI503
aI49
aI929
aI1687
aI3865
aI26723
asS'd'
I2053
sa(dp4106
S'a'
I1727
sS's'
I15
sS'm'
(lp4107
I1
aI3
aI5
aI5
aI15
aI41
aI1
aI239
aI53
aI215
aI1733
aI827
aI579
aI4089
aI6579
asS'd'
I2054
sa(dp4108
S'a'
I1730
sS's'
I15
sS'm'
(lp4109
I1
aI3
aI1
aI15
aI15
aI21
aI35
aI21
aI403
aI257
aI1475
aI2403
aI4705
aI11553
aI203
asS'd'
I2055
sa(dp4110
S'a'
I1732
sS's'
I15
sS'm'
(lp4111
I1
aI3
aI5
aI11
aI9
aI53
aI113
aI9
aI447
aI511
aI543
aI3141
aI7389
aI11249
aI431
asS'd'
I2056
sa(dp4112
S'a'
I1741
sS's'
I15
sS'm'
(lp4113
I1
aI3
aI5
aI9
aI9
aI11
aI55
aI93
aI325
aI411
aI305
aI2573
aI6871
aI12339
aI6435
asS'd'
I2057
sa(dp4114
S'a'
I1744
sS's'
I15
sS'm'
(lp4115
I1
aI3
aI3
aI7
aI31
aI27
aI21
aI113
aI99
aI853
aI365
aI589
aI3731
aI10875
aI12767
asS'd'
I2058
sa(dp4116
S'a'
I1759
sS's'
I15
sS'm'
(lp4117
I1
aI3
aI1
aI7
aI15
aI27
aI31
aI17
aI275
aI93
aI1161
aI2619
aI1329
aI7307
aI587
asS'd'
I2059
sa(dp4118
S'a'
I1765
sS's'
I15
sS'm'
(lp4119
I1
aI3
aI5
aI9
aI17
aI47
aI49
aI237
aI27
aI193
aI1237
aI591
aI5151
aI5521
aI31583
asS'd'
I2060
sa(dp4120
S'a'
I1766
sS's'
I15
sS'm'
(lp4121
I1
aI3
aI5
aI3
aI13
aI1
aI27
aI87
aI43
aI977
aI305
aI3293
aI2475
aI14571
aI18321
asS'd'
I2061
sa(dp4122
S'a'
I1778
sS's'
I15
sS'm'
(lp4123
I1
aI1
aI5
aI7
aI15
aI13
aI101
aI1
aI291
aI807
aI1711
aI2277
aI5573
aI11051
aI13133
asS'd'
I2062
sa(dp4124
S'a'
I1780
sS's'
I15
sS'm'
(lp4125
I1
aI3
aI3
aI1
aI9
aI3
aI65
aI81
aI415
aI733
aI1527
aI2747
aI6069
aI159
aI7095
asS'd'
I2063
sa(dp4126
S'a'
I1783
sS's'
I15
sS'm'
(lp4127
I1
aI3
aI3
aI15
aI27
aI1
aI71
aI49
aI231
aI851
aI2039
aI613
aI1899
aI2537
aI14511
asS'd'
I2064
sa(dp4128
S'a'
I1797
sS's'
I15
sS'm'
(lp4129
I1
aI1
aI1
aI11
aI3
aI41
aI55
aI23
aI247
aI1011
aI581
aI2363
aI2745
aI1337
aI20931
asS'd'
I2065
sa(dp4130
S'a'
I1807
sS's'
I15
sS'm'
(lp4131
I1
aI1
aI3
aI11
aI17
aI61
aI67
aI255
aI143
aI357
aI945
aI3407
aI5817
aI4155
aI23851
asS'd'
I2066
sa(dp4132
S'a'
I1821
sS's'
I15
sS'm'
(lp4133
I1
aI3
aI5
aI3
aI23
aI1
aI75
aI247
aI265
aI413
aI1899
aI2565
aI6629
aI15655
aI16117
asS'd'
I2067
sa(dp4134
S'a'
I1832
sS's'
I15
sS'm'
(lp4135
I1
aI1
aI1
aI9
aI11
aI49
aI11
aI189
aI223
aI177
aI1457
aI1931
aI163
aI15905
aI17297
asS'd'
I2068
sa(dp4136
S'a'
I1835
sS's'
I15
sS'm'
(lp4137
I1
aI3
aI7
aI13
aI17
aI1
aI111
aI189
aI343
aI961
aI427
aI2507
aI2393
aI8653
aI6353
asS'd'
I2069
sa(dp4138
S'a'
I1849
sS's'
I15
sS'm'
(lp4139
I1
aI3
aI7
aI13
aI23
aI61
aI59
aI51
aI313
aI963
aI791
aI3681
aI5637
aI3965
aI9263
asS'd'
I2070
sa(dp4140
S'a'
I1850
sS's'
I15
sS'm'
(lp4141
I1
aI3
aI7
aI7
aI21
aI53
aI127
aI141
aI499
aI859
aI337
aI2835
aI3195
aI4351
aI32369
asS'd'
I2071
sa(dp4142
S'a'
I1863
sS's'
I15
sS'm'
(lp4143
I1
aI1
aI7
aI5
aI1
aI5
aI53
aI63
aI497
aI535
aI35
aI305
aI4395
aI9757
aI13193
asS'd'
I2072
sa(dp4144
S'a'
I1867
sS's'
I15
sS'm'
(lp4145
I1
aI1
aI5
aI13
aI13
aI31
aI59
aI229
aI211
aI745
aI1453
aI3677
aI3005
aI7703
aI23907
asS'd'
I2073
sa(dp4146
S'a'
I1869
sS's'
I15
sS'm'
(lp4147
I1
aI3
aI5
aI5
aI7
aI63
aI17
aI197
aI493
aI861
aI499
aI3015
aI6349
aI1815
aI7437
asS'd'
I2074
sa(dp4148
S'a'
I1872
sS's'
I15
sS'm'
(lp4149
I1
aI1
aI1
aI13
aI13
aI37
aI29
aI189
aI253
aI1017
aI321
aI3145
aI407
aI7547
aI17099
asS'd'
I2075
sa(dp4150
S'a'
I1887
sS's'
I15
sS'm'
(lp4151
I1
aI3
aI3
aI3
aI23
aI53
aI69
aI77
aI175
aI17
aI1831
aI841
aI3851
aI1295
aI32107
asS'd'
I2076
sa(dp4152
S'a'
I1888
sS's'
I15
sS'm'
(lp4153
I1
aI3
aI7
aI13
aI13
aI39
aI107
aI237
aI389
aI729
aI635
aI3717
aI3041
aI3169
aI14987
asS'd'
I2077
sa(dp4154
S'a'
I1897
sS's'
I15
sS'm'
(lp4155
I1
aI1
aI3
aI1
aI25
aI7
aI69
aI35
aI495
aI49
aI659
aI2783
aI6051
aI13875
aI23927
asS'd'
I2078
sa(dp4156
S'a'
I1906
sS's'
I15
sS'm'
(lp4157
I1
aI3
aI7
aI5
aI5
aI25
aI49
aI7
aI193
aI493
aI93
aI657
aI1515
aI13975
aI14155
asS'd'
I2079
sa(dp4158
S'a'
I1917
sS's'
I15
sS'm'
(lp4159
I1
aI3
aI1
aI1
aI11
aI15
aI113
aI45
aI21
aI595
aI731
aI3397
aI4117
aI9711
aI16625
asS'd'
I2080
sa(dp4160
S'a'
I1927
sS's'
I15
sS'm'
(lp4161
I1
aI3
aI3
aI9
aI19
aI19
aI59
aI7
aI105
aI579
aI599
aI2859
aI97
aI14717
aI15361
asS'd'
I2081
sa(dp4162
S'a'
I1939
sS's'
I15
sS'm'
(lp4163
I1
aI1
aI1
aI5
aI27
aI49
aI113
aI5
aI367
aI563
aI1397
aI2805
aI3021
aI3111
aI20671
asS'd'
I2082
sa(dp4164
S'a'
I1941
sS's'
I15
sS'm'
(lp4165
I1
aI3
aI3
aI15
aI27
aI51
aI99
aI167
aI109
aI365
aI1959
aI1523
aI6959
aI14405
aI18191
asS'd'
I2083
sa(dp4166
S'a'
I1948
sS's'
I15
sS'm'
(lp4167
I1
aI3
aI1
aI5
aI21
aI51
aI125
aI67
aI123
aI45
aI1657
aI51
aI4825
aI14081
aI31049
asS'd'
I2084
sa(dp4168
S'a'
I1970
sS's'
I15
sS'm'
(lp4169
I1
aI1
aI5
aI7
aI21
aI59
aI21
aI249
aI77
aI793
aI1687
aI2561
aI2241
aI4321
aI7477
asS'd'
I2085
sa(dp4170
S'a'
I1979
sS's'
I15
sS'm'
(lp4171
I1
aI1
aI1
aI7
aI15
aI35
aI71
aI29
aI267
aI611
aI1813
aI1823
aI7039
aI3299
aI9919
asS'd'
I2086
sa(dp4172
S'a'
I1982
sS's'
I15
sS'm'
(lp4173
I1
aI3
aI7
aI11
aI21
aI59
aI109
aI213
aI371
aI785
aI659
aI1687
aI4827
aI6017
aI19619
asS'd'
I2087
sa(dp4174
S'a'
I2002
sS's'
I15
sS'm'
(lp4175
I1
aI1
aI3
aI11
aI27
aI17
aI1
aI55
aI367
aI939
aI333
aI127
aI5105
aI2405
aI28139
asS'd'
I2088
sa(dp4176
S'a'
I2020
sS's'
I15
sS'm'
(lp4177
I1
aI1
aI7
aI13
aI5
aI35
aI59
aI133
aI509
aI573
aI625
aI3857
aI7935
aI5279
aI3727
asS'd'
I2089
sa(dp4178
S'a'
I2024
sS's'
I15
sS'm'
(lp4179
I1
aI1
aI1
aI7
aI11
aI47
aI127
aI157
aI19
aI403
aI151
aI1143
aI7407
aI8985
aI32521
asS'd'
I2090
sa(dp4180
S'a'
I2032
sS's'
I15
sS'm'
(lp4181
I1
aI3
aI1
aI1
aI5
aI13
aI105
aI123
aI63
aI139
aI1569
aI1983
aI563
aI7175
aI27705
asS'd'
I2091
sa(dp4182
S'a'
I2053
sS's'
I15
sS'm'
(lp4183
I1
aI1
aI3
aI13
aI9
aI35
aI105
aI227
aI145
aI21
aI1369
aI57
aI393
aI2921
aI18511
asS'd'
I2092
sa(dp4184
S'a'
I2060
sS's'
I15
sS'm'
(lp4185
I1
aI3
aI1
aI7
aI17
aI61
aI99
aI187
aI261
aI281
aI437
aI2219
aI5999
aI1857
aI18001
asS'd'
I2093
sa(dp4186
S'a'
I2063
sS's'
I15
sS'm'
(lp4187
I1
aI3
aI3
aI5
aI1
aI59
aI67
aI45
aI451
aI439
aI2005
aI3607
aI3
aI7167
aI14227
asS'd'
I2094
sa(dp4188
S'a'
I2066
sS's'
I15
sS'm'
(lp4189
I1
aI3
aI3
aI3
aI29
aI19
aI25
aI251
aI275
aI733
aI1749
aI4021
aI871
aI3227
aI13701
asS'd'
I2095
sa(dp4190
S'a'
I2075
sS's'
I15
sS'm'
(lp4191
I1
aI3
aI3
aI13
aI27
aI53
aI57
aI243
aI491
aI521
aI1921
aI1037
aI5013
aI5703
aI15261
asS'd'
I2096
sa(dp4192
S'a'
I2078
sS's'
I15
sS'm'
(lp4193
I1
aI3
aI1
aI11
aI13
aI57
aI1
aI15
aI123
aI533
aI785
aI335
aI1423
aI14269
aI3483
asS'd'
I2097
sa(dp4194
S'a'
I2081
sS's'
I15
sS'm'
(lp4195
I1
aI3
aI7
aI13
aI15
aI55
aI5
aI139
aI385
aI47
aI1981
aI1291
aI7397
aI12925
aI29445
asS'd'
I2098
sa(dp4196
S'a'
I2091
sS's'
I15
sS'm'
(lp4197
I1
aI1
aI7
aI1
aI23
aI23
aI59
aI93
aI117
aI57
aI63
aI3047
aI4849
aI11637
aI25311
asS'd'
I2099
sa(dp4198
S'a'
I2096
sS's'
I15
sS'm'
(lp4199
I1
aI1
aI7
aI13
aI19
aI37
aI25
aI203
aI477
aI447
aI1345
aI3485
aI2099
aI13347
aI11621
asS'd'
I2100
sa(dp4200
S'a'
I2102
sS's'
I15
sS'm'
(lp4201
I1
aI1
aI7
aI3
aI11
aI23
aI81
aI17
aI41
aI735
aI1149
aI3253
aI7665
aI8291
aI22293
asS'd'
I2101
sa(dp4202
S'a'
I2106
sS's'
I15
sS'm'
(lp4203
I1
aI1
aI5
aI3
aI15
aI9
aI57
aI167
aI463
aI493
aI747
aI1947
aI6471
aI1111
aI31619
asS'd'
I2102
sa(dp4204
S'a'
I2116
sS's'
I15
sS'm'
(lp4205
I1
aI1
aI5
aI15
aI7
aI15
aI107
aI205
aI325
aI167
aI1749
aI927
aI3589
aI6127
aI7617
asS'd'
I2103
sa(dp4206
S'a'
I2120
sS's'
I15
sS'm'
(lp4207
I1
aI1
aI1
aI13
aI21
aI25
aI83
aI147
aI411
aI399
aI1423
aI2279
aI3661
aI7591
aI17429
asS'd'
I2104
sa(dp4208
S'a'
I2125
sS's'
I15
sS'm'
(lp4209
I1
aI1
aI1
aI9
aI5
aI17
aI69
aI205
aI243
aI647
aI473
aI1717
aI1977
aI10725
aI2913
asS'd'
I2105
sa(dp4210
S'a'
I2134
sS's'
I15
sS'm'
(lp4211
I1
aI1
aI3
aI5
aI5
aI37
aI103
aI15
aI485
aI641
aI1761
aI3755
aI6997
aI10985
aI11773
asS'd'
I2106
sa(dp4212
S'a'
I2178
sS's'
I15
sS'm'
(lp4213
I1
aI1
aI5
aI13
aI9
aI51
aI87
aI195
aI97
aI807
aI1801
aI961
aI6341
aI4307
aI29105
asS'd'
I2107
sa(dp4214
S'a'
I2180
sS's'
I15
sS'm'
(lp4215
I1
aI3
aI1
aI13
aI9
aI35
aI83
aI61
aI387
aI817
aI951
aI3993
aI7831
aI8479
aI23941
asS'd'
I2108
sa(dp4216
S'a'
I2187
sS's'
I15
sS'm'
(lp4217
I1
aI1
aI7
aI11
aI19
aI47
aI75
aI37
aI91
aI337
aI953
aI1169
aI163
aI2259
aI24713
asS'd'
I2109
sa(dp4218
S'a'
I2189
sS's'
I15
sS'm'
(lp4219
I1
aI1
aI1
aI11
aI13
aI15
aI83
aI171
aI159
aI87
aI619
aI2973
aI2653
aI13725
aI12499
asS'd'
I2110
sa(dp4220
S'a'
I2190
sS's'
I15
sS'm'
(lp4221
I1
aI3
aI5
aI3
aI5
aI63
aI119
aI25
aI343
aI269
aI553
aI2183
aI959
aI3825
aI22189
asS'd'
I2111
sa(dp4222
S'a'
I2208
sS's'
I15
sS'm'
(lp4223
I1
aI1
aI5
aI15
aI5
aI37
aI89
aI109
aI497
aI1013
aI265
aI669
aI1859
aI2647
aI3445
asS'd'
I2112
sa(dp4224
S'a'
I2214
sS's'
I15
sS'm'
(lp4225
I1
aI3
aI3
aI9
aI21
aI21
aI15
aI245
aI107
aI649
aI367
aI1601
aI7279
aI15783
aI4943
asS'd'
I2113
sa(dp4226
S'a'
I2237
sS's'
I15
sS'm'
(lp4227
I1
aI3
aI3
aI15
aI5
aI41
aI125
aI113
aI159
aI161
aI1191
aI3491
aI3531
aI55
aI20857
asS'd'
I2114
sa(dp4228
S'a'
I2252
sS's'
I15
sS'm'
(lp4229
I1
aI3
aI5
aI9
aI21
aI57
aI21
aI195
aI99
aI193
aI1915
aI2923
aI6349
aI15085
aI24929
asS'd'
I2115
sa(dp4230
S'a'
I2257
sS's'
I15
sS'm'
(lp4231
I1
aI1
aI1
aI11
aI31
aI11
aI73
aI141
aI361
aI621
aI1021
aI2067
aI5115
aI12665
aI26845
asS'd'
I2116
sa(dp4232
S'a'
I2260
sS's'
I15
sS'm'
(lp4233
I1
aI1
aI1
aI3
aI29
aI11
aI43
aI61
aI209
aI923
aI1753
aI1937
aI843
aI205
aI8367
asS'd'
I2117
sa(dp4234
S'a'
I2264
sS's'
I15
sS'm'
(lp4235
I1
aI1
aI1
aI5
aI15
aI33
aI119
aI209
aI215
aI973
aI1775
aI815
aI6693
aI7957
aI14517
asS'd'
I2118
sa(dp4236
S'a'
I2270
sS's'
I15
sS'm'
(lp4237
I1
aI1
aI1
aI5
aI17
aI57
aI27
aI147
aI489
aI59
aI1439
aI2279
aI445
aI11791
aI19739
asS'd'
I2119
sa(dp4238
S'a'
I2279
sS's'
I15
sS'm'
(lp4239
I1
aI3
aI1
aI7
aI11
aI55
aI1
aI83
aI305
aI17
aI1909
aI405
aI2325
aI5293
aI28559
asS'd'
I2120
sa(dp4240
S'a'
I2288
sS's'
I15
sS'm'
(lp4241
I1
aI3
aI3
aI7
aI11
aI27
aI103
aI157
aI455
aI1005
aI2033
aI3145
aI1919
aI15723
aI25197
asS'd'
I2121
sa(dp4242
S'a'
I2305
sS's'
I15
sS'm'
(lp4243
I1
aI1
aI5
aI11
aI15
aI51
aI37
aI131
aI503
aI1007
aI1795
aI2421
aI1335
aI7413
aI21741
asS'd'
I2122
sa(dp4244
S'a'
I2312
sS's'
I15
sS'm'
(lp4245
I1
aI1
aI3
aI1
aI23
aI63
aI69
aI83
aI419
aI283
aI583
aI123
aI7725
aI2243
aI8403
asS'd'
I2123
sa(dp4246
S'a'
I2317
sS's'
I15
sS'm'
(lp4247
I1
aI1
aI5
aI5
aI27
aI45
aI109
aI17
aI299
aI65
aI351
aI947
aI1165
aI10723
aI2053
asS'd'
I2124
sa(dp4248
S'a'
I2323
sS's'
I15
sS'm'
(lp4249
I1
aI1
aI3
aI3
aI23
aI61
aI115
aI253
aI1
aI931
aI1481
aI3187
aI441
aI14735
aI27207
asS'd'
I2125
sa(dp4250
S'a'
I2329
sS's'
I15
sS'm'
(lp4251
I1
aI1
aI5
aI3
aI25
aI11
aI83
aI141
aI359
aI343
aI901
aI1629
aI731
aI12841
aI14357
asS'd'
I2126
sa(dp4252
S'a'
I2335
sS's'
I15
sS'm'
(lp4253
I1
aI1
aI3
aI9
aI7
aI45
aI97
aI3
aI299
aI217
aI663
aI1527
aI6379
aI4527
aI26147
asS'd'
I2127
sa(dp4254
S'a'
I2342
sS's'
I15
sS'm'
(lp4255
I1
aI1
aI7
aI9
aI11
aI53
aI9
aI203
aI337
aI713
aI1517
aI719
aI4587
aI11443
aI26905
asS'd'
I2128
sa(dp4256
S'a'
I2345
sS's'
I15
sS'm'
(lp4257
I1
aI1
aI7
aI9
aI11
aI41
aI125
aI213
aI237
aI377
aI361
aI3231
aI4223
aI3263
aI12655
asS'd'
I2129
sa(dp4258
S'a'
I2365
sS's'
I15
sS'm'
(lp4259
I1
aI3
aI7
aI7
aI7
aI33
aI99
aI19
aI117
aI273
aI985
aI107
aI3831
aI10135
aI19423
asS'd'
I2130
sa(dp4260
S'a'
I2371
sS's'
I15
sS'm'
(lp4261
I1
aI1
aI5
aI15
aI25
aI41
aI13
aI125
aI449
aI169
aI1149
aI4021
aI5663
aI3077
aI19163
asS'd'
I2131
sa(dp4262
S'a'
I2378
sS's'
I15
sS'm'
(lp4263
I1
aI3
aI5
aI9
aI25
aI57
aI47
aI103
aI269
aI51
aI1805
aI2503
aI6687
aI8065
aI12045
asS'd'
I2132
sa(dp4264
S'a'
I2385
sS's'
I15
sS'm'
(lp4265
I1
aI3
aI5
aI7
aI3
aI35
aI87
aI225
aI189
aI229
aI931
aI3293
aI1347
aI1427
aI3269
asS'd'
I2133
sa(dp4266
S'a'
I2395
sS's'
I15
sS'm'
(lp4267
I1
aI1
aI1
aI3
aI5
aI31
aI61
aI19
aI247
aI9
aI1667
aI343
aI559
aI2703
aI3763
asS'd'
I2134
sa(dp4268
S'a'
I2404
sS's'
I15
sS'm'
(lp4269
I1
aI3
aI5
aI15
aI31
aI19
aI57
aI187
aI109
aI121
aI1287
aI2269
aI659
aI16235
aI1273
asS'd'
I2135
sa(dp4270
S'a'
I2414
sS's'
I15
sS'm'
(lp4271
I1
aI1
aI1
aI3
aI5
aI47
aI59
aI243
aI255
aI97
aI1959
aI1723
aI1347
aI3019
aI26989
asS'd'
I2136
sa(dp4272
S'a'
I2426
sS's'
I15
sS'm'
(lp4273
I1
aI3
aI3
aI15
aI29
aI35
aI75
aI67
aI497
aI731
aI193
aI3307
aI3579
aI12005
aI7209
asS'd'
I2137
sa(dp4274
S'a'
I2428
sS's'
I15
sS'm'
(lp4275
I1
aI1
aI5
aI9
aI13
aI35
aI79
aI213
aI51
aI983
aI1927
aI1793
aI5037
aI5463
aI965
asS'd'
I2138
sa(dp4276
S'a'
I2441
sS's'
I15
sS'm'
(lp4277
I1
aI1
aI7
aI11
aI5
aI41
aI7
aI83
aI15
aI411
aI1775
aI3515
aI6755
aI3249
aI16425
asS'd'
I2139
sa(dp4278
S'a'
I2456
sS's'
I15
sS'm'
(lp4279
I1
aI3
aI5
aI1
aI19
aI61
aI3
aI19
aI395
aI819
aI1331
aI179
aI5225
aI5333
aI3601
asS'd'
I2140
sa(dp4280
S'a'
I2466
sS's'
I15
sS'm'
(lp4281
I1
aI1
aI3
aI9
aI7
aI5
aI87
aI15
aI387
aI609
aI1465
aI277
aI987
aI8377
aI903
asS'd'
I2141
sa(dp4282
S'a'
I2468
sS's'
I15
sS'm'
(lp4283
I1
aI1
aI1
aI3
aI15
aI11
aI123
aI107
aI355
aI333
aI285
aI1801
aI6989
aI1549
aI25791
asS'd'
I2142
sa(dp4284
S'a'
I2475
sS's'
I15
sS'm'
(lp4285
I1
aI1
aI7
aI13
aI27
aI13
aI73
aI111
aI481
aI227
aI1091
aI365
aI5713
aI5087
aI27217
asS'd'
I2143
sa(dp4286
S'a'
I2489
sS's'
I15
sS'm'
(lp4287
I1
aI3
aI3
aI15
aI1
aI55
aI95
aI213
aI377
aI405
aI139
aI1867
aI2175
aI4217
aI28813
asS'd'
I2144
sa(dp4288
S'a'
I2495
sS's'
I15
sS'm'
(lp4289
I1
aI3
aI5
aI11
aI21
aI43
aI109
aI155
aI181
aI901
aI1951
aI507
aI4389
aI10815
aI3141
asS'd'
I2145
sa(dp4290
S'a'
I2497
sS's'
I15
sS'm'
(lp4291
I1
aI1
aI1
aI15
aI17
aI11
aI43
aI215
aI501
aI19
aI259
aI3479
aI6381
aI6927
aI31247
asS'd'
I2146
sa(dp4292
S'a'
I2510
sS's'
I15
sS'm'
(lp4293
I1
aI3
aI5
aI15
aI19
aI61
aI75
aI41
aI391
aI95
aI865
aI1441
aI7993
aI13979
aI24663
asS'd'
I2147
sa(dp4294
S'a'
I2512
sS's'
I15
sS'm'
(lp4295
I1
aI3
aI1
aI3
aI21
aI15
aI115
aI213
aI1
aI645
aI777
aI1517
aI2543
aI11223
aI3633
asS'd'
I2148
sa(dp4296
S'a'
I2522
sS's'
I15
sS'm'
(lp4297
I1
aI3
aI5
aI3
aI9
aI57
aI39
aI211
aI407
aI65
aI1795
aI2805
aI2799
aI8691
aI1987
asS'd'
I2149
sa(dp4298
S'a'
I2533
sS's'
I15
sS'm'
(lp4299
I1
aI1
aI3
aI13
aI17
aI55
aI47
aI113
aI29
aI139
aI1301
aI3303
aI1129
aI13947
aI29821
asS'd'
I2150
sa(dp4300
S'a'
I2543
sS's'
I15
sS'm'
(lp4301
I1
aI1
aI3
aI13
aI5
aI35
aI97
aI151
aI477
aI409
aI1397
aI3399
aI4421
aI15929
aI6163
asS'd'
I2151
sa(dp4302
S'a'
I2551
sS's'
I15
sS'm'
(lp4303
I1
aI3
aI1
aI9
aI21
aI51
aI99
aI133
aI149
aI763
aI623
aI173
aI4311
aI11081
aI1095
asS'd'
I2152
sa(dp4304
S'a'
I2552
sS's'
I15
sS'm'
(lp4305
I1
aI3
aI7
aI15
aI13
aI3
aI99
aI3
aI195
aI907
aI1335
aI1355
aI7977
aI5773
aI32383
asS'd'
I2153
sa(dp4306
S'a'
I2557
sS's'
I15
sS'm'
(lp4307
I1
aI1
aI3
aI9
aI17
aI43
aI43
aI217
aI475
aI917
aI1373
aI1677
aI4871
aI9619
aI16657
asS'd'
I2154
sa(dp4308
S'a'
I2567
sS's'
I15
sS'm'
(lp4309
I1
aI3
aI3
aI7
aI31
aI31
aI55
aI11
aI73
aI693
aI25
aI417
aI1195
aI6225
aI32279
asS'd'
I2155
sa(dp4310
S'a'
I2581
sS's'
I15
sS'm'
(lp4311
I1
aI3
aI5
aI9
aI21
aI57
aI127
aI149
aI79
aI379
aI1609
aI2543
aI6473
aI16033
aI27191
asS'd'
I2156
sa(dp4312
S'a'
I2586
sS's'
I15
sS'm'
(lp4313
I1
aI1
aI5
aI1
aI13
aI9
aI81
aI153
aI297
aI789
aI1749
aI2819
aI3961
aI11231
aI24927
asS'd'
I2157
sa(dp4314
S'a'
I2597
sS's'
I15
sS'm'
(lp4315
I1
aI3
aI5
aI3
aI23
aI61
aI45
aI43
aI43
aI133
aI1481
aI1543
aI2991
aI13739
aI10287
asS'd'
I2158
sa(dp4316
S'a'
I2601
sS's'
I15
sS'm'
(lp4317
I1
aI1
aI3
aI9
aI25
aI43
aI31
aI177
aI337
aI193
aI1083
aI1
aI991
aI9725
aI8379
asS'd'
I2159
sa(dp4318
S'a'
I2622
sS's'
I15
sS'm'
(lp4319
I1
aI3
aI5
aI11
aI13
aI33
aI65
aI83
aI421
aI149
aI409
aI2443
aI7423
aI8847
aI29599
asS'd'
I2160
sa(dp4320
S'a'
I2633
sS's'
I15
sS'm'
(lp4321
I1
aI1
aI5
aI11
aI11
aI1
aI23
aI225
aI77
aI585
aI1505
aI2525
aI739
aI10915
aI25733
asS'd'
I2161
sa(dp4322
S'a'
I2636
sS's'
I15
sS'm'
(lp4323
I1
aI3
aI7
aI13
aI7
aI55
aI3
aI223
aI415
aI521
aI1865
aI2349
aI5663
aI7455
aI16569
asS'd'
I2162
sa(dp4324
S'a'
I2642
sS's'
I15
sS'm'
(lp4325
I1
aI1
aI7
aI13
aI1
aI45
aI121
aI49
aI463
aI99
aI1061
aI2559
aI5087
aI13389
aI11035
asS'd'
I2163
sa(dp4326
S'a'
I2644
sS's'
I15
sS'm'
(lp4327
I1
aI3
aI7
aI11
aI31
aI51
aI35
aI235
aI385
aI1023
aI1771
aI2013
aI5437
aI4877
aI22119
asS'd'
I2164
sa(dp4328
S'a'
I2653
sS's'
I15
sS'm'
(lp4329
I1
aI3
aI3
aI11
aI21
aI3
aI11
aI119
aI81
aI737
aI1093
aI2377
aI4055
aI1121
aI15767
asS'd'
I2165
sa(dp4330
S'a'
I2667
sS's'
I15
sS'm'
(lp4331
I1
aI1
aI5
aI13
aI9
aI3
aI83
aI217
aI387
aI249
aI1047
aI1861
aI4103
aI15367
aI24545
asS'd'
I2166
sa(dp4332
S'a'
I2669
sS's'
I15
sS'm'
(lp4333
I1
aI3
aI3
aI1
aI5
aI37
aI43
aI183
aI383
aI463
aI937
aI1165
aI1481
aI959
aI17047
asS'd'
I2167
sa(dp4334
S'a'
I2672
sS's'
I15
sS'm'
(lp4335
I1
aI1
aI3
aI5
aI7
aI43
aI127
aI243
aI81
aI1021
aI165
aI753
aI4711
aI12965
aI22049
asS'd'
I2168
sa(dp4336
S'a'
I2675
sS's'
I15
sS'm'
(lp4337
I1
aI1
aI5
aI5
aI3
aI61
aI65
aI53
aI425
aI89
aI5
aI1467
aI1395
aI9579
aI8961
asS'd'
I2169
sa(dp4338
S'a'
I2682
sS's'
I15
sS'm'
(lp4339
I1
aI3
aI7
aI13
aI11
aI35
aI123
aI21
aI83
aI689
aI667
aI1203
aI5959
aI15697
aI26885
asS'd'
I2170
sa(dp4340
S'a'
I2687
sS's'
I15
sS'm'
(lp4341
I1
aI1
aI5
aI13
aI9
aI49
aI41
aI101
aI291
aI339
aI1067
aI657
aI4453
aI1137
aI21131
asS'd'
I2171
sa(dp4342
S'a'
I2691
sS's'
I15
sS'm'
(lp4343
I1
aI3
aI3
aI3
aI17
aI61
aI11
aI213
aI27
aI805
aI1691
aI1057
aI6011
aI11941
aI18883
asS'd'
I2172
sa(dp4344
S'a'
I2698
sS's'
I15
sS'm'
(lp4345
I1
aI3
aI1
aI7
aI3
aI51
aI5
aI63
aI121
aI3
aI245
aI2631
aI3737
aI16121
aI26803
asS'd'
I2173
sa(dp4346
S'a'
I2708
sS's'
I15
sS'm'
(lp4347
I1
aI3
aI1
aI1
aI23
aI51
aI79
aI19
aI161
aI107
aI609
aI3489
aI3389
aI4035
aI2427
asS'd'
I2174
sa(dp4348
S'a'
I2712
sS's'
I15
sS'm'
(lp4349
I1
aI3
aI1
aI1
aI17
aI11
aI101
aI101
aI373
aI63
aI1641
aI285
aI1333
aI165
aI14025
asS'd'
I2175
sa(dp4350
S'a'
I2718
sS's'
I15
sS'm'
(lp4351
I1
aI1
aI1
aI5
aI1
aI51
aI83
aI137
aI45
aI1019
aI821
aI867
aI6055
aI10443
aI9857
asS'd'
I2176
sa(dp4352
S'a'
I2722
sS's'
I15
sS'm'
(lp4353
I1
aI3
aI1
aI5
aI17
aI23
aI25
aI181
aI429
aI495
aI317
aI3219
aI5963
aI13945
aI9969
asS'd'
I2177
sa(dp4354
S'a'
I2736
sS's'
I15
sS'm'
(lp4355
I1
aI3
aI7
aI3
aI3
aI15
aI123
aI191
aI369
aI177
aI1697
aI2113
aI3889
aI5201
aI21839
asS'd'
I2178
sa(dp4356
S'a'
I2741
sS's'
I15
sS'm'
(lp4357
I1
aI3
aI1
aI11
aI21
aI39
aI51
aI139
aI271
aI605
aI1007
aI3513
aI3365
aI3781
aI6799
asS'd'
I2179
sa(dp4358
S'a'
I2756
sS's'
I15
sS'm'
(lp4359
I1
aI1
aI7
aI5
aI13
aI19
aI47
aI165
aI249
aI405
aI255
aI1295
aI4513
aI14395
aI5587
asS'd'
I2180
sa(dp4360
S'a'
I2765
sS's'
I15
sS'm'
(lp4361
I1
aI1
aI3
aI7
aI5
aI17
aI99
aI1
aI393
aI31
aI621
aI797
aI6113
aI16003
aI32043
asS'd'
I2181
sa(dp4362
S'a'
I2774
sS's'
I15
sS'm'
(lp4363
I1
aI3
aI5
aI13
aI11
aI21
aI65
aI81
aI147
aI443
aI775
aI3671
aI7029
aI11749
aI3339
asS'd'
I2182
sa(dp4364
S'a'
I2799
sS's'
I15
sS'm'
(lp4365
I1
aI3
aI7
aI1
aI23
aI33
aI99
aI177
aI161
aI577
aI1729
aI617
aI3465
aI11787
aI17577
asS'd'
I2183
sa(dp4366
S'a'
I2804
sS's'
I15
sS'm'
(lp4367
I1
aI1
aI5
aI7
aI15
aI15
aI53
aI193
aI97
aI255
aI1223
aI545
aI5153
aI873
aI24525
asS'd'
I2184
sa(dp4368
S'a'
I2825
sS's'
I15
sS'm'
(lp4369
I1
aI3
aI5
aI1
aI7
aI57
aI47
aI121
aI383
aI835
aI1709
aI2363
aI4731
aI12163
aI7001
asS'd'
I2185
sa(dp4370
S'a'
I2826
sS's'
I15
sS'm'
(lp4371
I1
aI3
aI3
aI11
aI19
aI33
aI63
aI99
aI387
aI95
aI783
aI1009
aI6373
aI4021
aI7685
asS'd'
I2186
sa(dp4372
S'a'
I2840
sS's'
I15
sS'm'
(lp4373
I1
aI1
aI1
aI15
aI25
aI33
aI73
aI135
aI335
aI785
aI935
aI1927
aI5847
aI10501
aI7719
asS'd'
I2187
sa(dp4374
S'a'
I2843
sS's'
I15
sS'm'
(lp4375
I1
aI1
aI5
aI3
aI27
aI45
aI71
aI215
aI489
aI157
aI1189
aI2577
aI6901
aI10219
aI3025
asS'd'
I2188
sa(dp4376
S'a'
I2846
sS's'
I15
sS'm'
(lp4377
I1
aI1
aI7
aI7
aI21
aI3
aI97
aI225
aI101
aI159
aI293
aI2789
aI7955
aI14829
aI1209
asS'd'
I2189
sa(dp4378
S'a'
I2849
sS's'
I15
sS'm'
(lp4379
I1
aI3
aI1
aI5
aI23
aI41
aI83
aI63
aI361
aI195
aI1707
aI2081
aI5363
aI6327
aI179
asS'd'
I2190
sa(dp4380
S'a'
I2867
sS's'
I15
sS'm'
(lp4381
I1
aI1
aI3
aI1
aI21
aI51
aI59
aI67
aI175
aI363
aI825
aI2971
aI3321
aI8837
aI11805
asS'd'
I2191
sa(dp4382
S'a'
I2876
sS's'
I15
sS'm'
(lp4383
I1
aI3
aI7
aI1
aI19
aI3
aI15
aI21
aI429
aI675
aI1589
aI2615
aI2575
aI1537
aI7139
asS'd'
I2192
sa(dp4384
S'a'
I2891
sS's'
I15
sS'm'
(lp4385
I1
aI3
aI3
aI5
aI21
aI29
aI17
aI115
aI345
aI397
aI523
aI1699
aI7043
aI11173
aI3023
asS'd'
I2193
sa(dp4386
S'a'
I2902
sS's'
I15
sS'm'
(lp4387
I1
aI1
aI5
aI7
aI19
aI63
aI99
aI175
aI91
aI433
aI153
aI3749
aI517
aI13667
aI7423
asS'd'
I2194
sa(dp4388
S'a'
I2912
sS's'
I15
sS'm'
(lp4389
I1
aI3
aI7
aI3
aI25
aI23
aI53
aI149
aI65
aI551
aI1231
aI365
aI6637
aI15137
aI16319
asS'd'
I2195
sa(dp4390
S'a'
I2917
sS's'
I15
sS'm'
(lp4391
I1
aI3
aI7
aI13
aI5
aI45
aI11
aI151
aI323
aI31
aI1749
aI409
aI6753
aI10503
aI14991
asS'd'
I2196
sa(dp4392
S'a'
I2927
sS's'
I15
sS'm'
(lp4393
I1
aI3
aI7
aI3
aI5
aI21
aI29
aI117
aI321
aI341
aI1811
aI3619
aI4337
aI12255
aI8629
asS'd'
I2197
sa(dp4394
S'a'
I2941
sS's'
I15
sS'm'
(lp4395
I1
aI3
aI7
aI3
aI7
aI3
aI5
aI221
aI407
aI671
aI1763
aI3669
aI2353
aI8175
aI23489
asS'd'
I2198
sa(dp4396
S'a'
I2965
sS's'
I15
sS'm'
(lp4397
I1
aI1
aI3
aI7
aI11
aI55
aI53
aI185
aI247
aI35
aI1823
aI513
aI1379
aI11827
aI20069
asS'd'
I2199
sa(dp4398
S'a'
I2970
sS's'
I15
sS'm'
(lp4399
I1
aI3
aI3
aI5
aI29
aI51
aI73
aI191
aI185
aI961
aI881
aI2019
aI5651
aI1019
aI15587
asS'd'
I2200
sa(dp4400
S'a'
I2982
sS's'
I15
sS'm'
(lp4401
I1
aI3
aI7
aI13
aI7
aI55
aI59
aI5
aI417
aI829
aI453
aI2339
aI587
aI13283
aI797
asS'd'
I2201
sa(dp4402
S'a'
I2993
sS's'
I15
sS'm'
(lp4403
I1
aI3
aI7
aI3
aI11
aI41
aI75
aI85
aI65
aI149
aI1583
aI529
aI2707
aI11479
aI7109
asS'd'
I2202
sa(dp4404
S'a'
I3018
sS's'
I15
sS'm'
(lp4405
I1
aI3
aI7
aI9
aI13
aI57
aI37
aI243
aI91
aI613
aI665
aI171
aI1631
aI13737
aI2377
asS'd'
I2203
sa(dp4406
S'a'
I3023
sS's'
I15
sS'm'
(lp4407
I1
aI1
aI3
aI7
aI5
aI43
aI97
aI53
aI477
aI793
aI999
aI3647
aI2555
aI7371
aI19295
asS'd'
I2204
sa(dp4408
S'a'
I3025
sS's'
I15
sS'm'
(lp4409
I1
aI1
aI7
aI1
aI1
aI9
aI99
aI253
aI317
aI817
aI1559
aI2081
aI2529
aI14611
aI15997
asS'd'
I2205
sa(dp4410
S'a'
I3026
sS's'
I15
sS'm'
(lp4411
I1
aI3
aI3
aI1
aI5
aI41
aI57
aI121
aI387
aI441
aI709
aI1511
aI7045
aI8409
aI13297
asS'd'
I2206
sa(dp4412
S'a'
I3028
sS's'
I15
sS'm'
(lp4413
I1
aI1
aI1
aI13
aI29
aI57
aI63
aI183
aI327
aI473
aI1943
aI213
aI3973
aI16289
aI2739
asS'd'
I2207
sa(dp4414
S'a'
I3032
sS's'
I15
sS'm'
(lp4415
I1
aI3
aI7
aI9
aI25
aI15
aI75
aI185
aI335
aI881
aI1041
aI3339
aI4471
aI6823
aI21121
asS'd'
I2208
sa(dp4416
S'a'
I3053
sS's'
I15
sS'm'
(lp4417
I1
aI3
aI3
aI13
aI23
aI3
aI57
aI117
aI511
aI927
aI771
aI3229
aI949
aI15487
aI11963
asS'd'
I2209
sa(dp4418
S'a'
I3054
sS's'
I15
sS'm'
(lp4419
I1
aI1
aI3
aI7
aI27
aI19
aI55
aI207
aI331
aI705
aI1945
aI797
aI7125
aI10493
aI16585
asS'd'
I2210
sa(dp4420
S'a'
I3065
sS's'
I15
sS'm'
(lp4421
I1
aI3
aI1
aI1
aI29
aI7
aI91
aI93
aI459
aI93
aI1501
aI1927
aI6415
aI16255
aI9823
asS'd'
I2211
sa(dp4422
S'a'
I3071
sS's'
I15
sS'm'
(lp4423
I1
aI1
aI5
aI5
aI31
aI11
aI97
aI179
aI505
aI807
aI877
aI4003
aI4377
aI8851
aI4239
asS'd'
I2212
sa(dp4424
S'a'
I3076
sS's'
I15
sS'm'
(lp4425
I1
aI1
aI3
aI5
aI11
aI25
aI17
aI131
aI23
aI95
aI311
aI1429
aI2029
aI13091
aI23739
asS'd'
I2213
sa(dp4426
S'a'
I3088
sS's'
I15
sS'm'
(lp4427
I1
aI1
aI3
aI11
aI13
aI27
aI33
aI127
aI481
aI117
aI1127
aI1619
aI6493
aI8507
aI6615
asS'd'
I2214
sa(dp4428
S'a'
I3107
sS's'
I15
sS'm'
(lp4429
I1
aI3
aI1
aI13
aI19
aI27
aI89
aI101
aI27
aI235
aI1579
aI1701
aI4421
aI16037
aI16239
asS'd'
I2215
sa(dp4430
S'a'
I3146
sS's'
I15
sS'm'
(lp4431
I1
aI3
aI1
aI15
aI1
aI15
aI3
aI117
aI317
aI475
aI1691
aI2423
aI5519
aI1703
aI2969
asS'd'
I2216
sa(dp4432
S'a'
I3148
sS's'
I15
sS'm'
(lp4433
I1
aI1
aI3
aI1
aI13
aI15
aI19
aI37
aI237
aI467
aI1321
aI453
aI2169
aI13313
aI31499
asS'd'
I2217
sa(dp4434
S'a'
I3159
sS's'
I15
sS'm'
(lp4435
I1
aI1
aI3
aI15
aI29
aI55
aI31
aI199
aI85
aI285
aI967
aI367
aI3941
aI151
aI20587
asS'd'
I2218
sa(dp4436
S'a'
I3165
sS's'
I15
sS'm'
(lp4437
I1
aI3
aI7
aI15
aI7
aI13
aI31
aI35
aI117
aI543
aI1179
aI3441
aI3039
aI11225
aI30229
asS'd'
I2219
sa(dp4438
S'a'
I3170
sS's'
I15
sS'm'
(lp4439
I1
aI1
aI3
aI15
aI3
aI43
aI1
aI63
aI353
aI395
aI1775
aI3493
aI5175
aI13193
aI25343
asS'd'
I2220
sa(dp4440
S'a'
I3179
sS's'
I15
sS'm'
(lp4441
I1
aI3
aI3
aI15
aI17
aI25
aI57
aI205
aI411
aI83
aI1877
aI2093
aI5599
aI12115
aI8751
asS'd'
I2221
sa(dp4442
S'a'
I3182
sS's'
I15
sS'm'
(lp4443
I1
aI1
aI1
aI11
aI15
aI9
aI115
aI99
aI85
aI887
aI987
aI4015
aI7077
aI3739
aI21505
asS'd'
I2222
sa(dp4444
S'a'
I3205
sS's'
I15
sS'm'
(lp4445
I1
aI3
aI1
aI11
aI25
aI39
aI127
aI37
aI329
aI273
aI1531
aI3211
aI7115
aI15501
aI26575
asS'd'
I2223
sa(dp4446
S'a'
I3212
sS's'
I15
sS'm'
(lp4447
I1
aI1
aI5
aI13
aI15
aI3
aI3
aI101
aI431
aI645
aI493
aI723
aI8083
aI1423
aI14879
asS'd'
I2224
sa(dp4448
S'a'
I3218
sS's'
I15
sS'm'
(lp4449
I1
aI3
aI3
aI5
aI31
aI35
aI37
aI131
aI259
aI849
aI325
aI3403
aI3627
aI3295
aI30885
asS'd'
I2225
sa(dp4450
S'a'
I3220
sS's'
I15
sS'm'
(lp4451
I1
aI3
aI7
aI1
aI9
aI3
aI31
aI201
aI379
aI907
aI1005
aI3333
aI7457
aI2533
aI30357
asS'd'
I2226
sa(dp4452
S'a'
I3223
sS's'
I15
sS'm'
(lp4453
I1
aI3
aI1
aI9
aI7
aI7
aI95
aI103
aI121
aI157
aI895
aI2683
aI5839
aI12403
aI14327
asS'd'
I2227
sa(dp4454
S'a'
I3227
sS's'
I15
sS'm'
(lp4455
I1
aI3
aI7
aI3
aI13
aI5
aI55
aI233
aI3
aI855
aI859
aI1115
aI3883
aI8041
aI3353
asS'd'
I2228
sa(dp4456
S'a'
I3233
sS's'
I15
sS'm'
(lp4457
I1
aI1
aI5
aI9
aI3
aI55
aI99
aI79
aI263
aI831
aI1579
aI205
aI5673
aI1999
aI14879
asS'd'
I2229
sa(dp4458
S'a'
I3234
sS's'
I15
sS'm'
(lp4459
I1
aI3
aI1
aI5
aI17
aI25
aI85
aI19
aI189
aI141
aI877
aI667
aI4461
aI11915
aI23247
asS'd'
I2230
sa(dp4460
S'a'
I3254
sS's'
I15
sS'm'
(lp4461
I1
aI1
aI5
aI5
aI1
aI35
aI15
aI219
aI469
aI725
aI1793
aI3683
aI3661
aI15627
aI30197
asS'd'
I2231
sa(dp4462
S'a'
I3263
sS's'
I15
sS'm'
(lp4463
I1
aI1
aI7
aI5
aI27
aI3
aI41
aI153
aI431
aI487
aI759
aI1345
aI6735
aI9937
aI26277
asS'd'
I2232
sa(dp4464
S'a'
I3268
sS's'
I15
sS'm'
(lp4465
I1
aI1
aI1
aI11
aI11
aI13
aI41
aI121
aI265
aI465
aI1447
aI5
aI3407
aI1907
aI10037
asS'd'
I2233
sa(dp4466
S'a'
I3272
sS's'
I15
sS'm'
(lp4467
I1
aI3
aI5
aI9
aI15
aI63
aI5
aI7
aI407
aI83
aI365
aI3687
aI7721
aI6973
aI16967
asS'd'
I2234
sa(dp4468
S'a'
I3277
sS's'
I15
sS'm'
(lp4469
I1
aI1
aI7
aI7
aI5
aI41
aI75
aI155
aI417
aI565
aI1199
aI1111
aI2823
aI10703
aI22561
asS'd'
I2235
sa(dp4470
S'a'
I3292
sS's'
I15
sS'm'
(lp4471
I1
aI3
aI7
aI5
aI7
aI43
aI39
aI185
aI105
aI327
aI1977
aI1137
aI3261
aI10583
aI11661
asS'd'
I2236
sa(dp4472
S'a'
I3295
sS's'
I15
sS'm'
(lp4473
I1
aI3
aI7
aI7
aI19
aI19
aI103
aI137
aI169
aI273
aI1357
aI3413
aI7647
aI10531
aI32489
asS'd'
I2237
sa(dp4474
S'a'
I3296
sS's'
I15
sS'm'
(lp4475
I1
aI1
aI3
aI13
aI13
aI3
aI81
aI23
aI161
aI295
aI735
aI2031
aI1027
aI15513
aI20165
asS'd'
I2238
sa(dp4476
S'a'
I3301
sS's'
I15
sS'm'
(lp4477
I1
aI1
aI5
aI1
aI15
aI1
aI91
aI35
aI375
aI207
aI1417
aI1115
aI2237
aI11749
aI8509
asS'd'
I2239
sa(dp4478
S'a'
I3306
sS's'
I15
sS'm'
(lp4479
I1
aI3
aI7
aI3
aI25
aI51
aI49
aI219
aI195
aI417
aI1523
aI3953
aI5739
aI7499
aI27071
asS'd'
I2240
sa(dp4480
S'a'
I3313
sS's'
I15
sS'm'
(lp4481
I1
aI1
aI3
aI11
aI23
aI29
aI19
aI81
aI421
aI633
aI513
aI547
aI7545
aI29
aI11909
asS'd'
I2241
sa(dp4482
S'a'
I3346
sS's'
I15
sS'm'
(lp4483
I1
aI1
aI1
aI7
aI13
aI61
aI33
aI243
aI221
aI231
aI111
aI879
aI2861
aI1795
aI27531
asS'd'
I2242
sa(dp4484
S'a'
I3367
sS's'
I15
sS'm'
(lp4485
I1
aI3
aI7
aI3
aI19
aI21
aI1
aI141
aI159
aI605
aI969
aI3013
aI6583
aI2447
aI19919
asS'd'
I2243
sa(dp4486
S'a'
I3371
sS's'
I15
sS'm'
(lp4487
I1
aI3
aI7
aI3
aI31
aI9
aI91
aI83
aI29
aI873
aI929
aI43
aI2253
aI12539
aI23951
asS'd'
I2244
sa(dp4488
S'a'
I3373
sS's'
I15
sS'm'
(lp4489
I1
aI1
aI5
aI3
aI31
aI15
aI87
aI105
aI319
aI973
aI1489
aI3417
aI3377
aI15749
aI2357
asS'd'
I2245
sa(dp4490
S'a'
I3374
sS's'
I15
sS'm'
(lp4491
I1
aI1
aI3
aI15
aI7
aI23
aI3
aI81
aI383
aI419
aI713
aI997
aI6873
aI593
aI285
asS'd'
I2246
sa(dp4492
S'a'
I3376
sS's'
I15
sS'm'
(lp4493
I1
aI3
aI3
aI1
aI29
aI13
aI29
aI101
aI441
aI693
aI2039
aI2951
aI5921
aI12129
aI12053
asS'd'
I2247
sa(dp4494
S'a'
I3382
sS's'
I15
sS'm'
(lp4495
I1
aI1
aI3
aI15
aI9
aI29
aI97
aI117
aI421
aI433
aI1017
aI125
aI3607
aI9415
aI6843
asS'd'
I2248
sa(dp4496
S'a'
I3388
sS's'
I15
sS'm'
(lp4497
I1
aI3
aI5
aI9
aI11
aI13
aI75
aI155
aI413
aI75
aI109
aI1599
aI6161
aI16115
aI12621
asS'd'
I2249
sa(dp4498
S'a'
I3391
sS's'
I15
sS'm'
(lp4499
I1
aI3
aI3
aI3
aI11
aI13
aI49
aI225
aI401
aI599
aI1815
aI1643
aI7853
aI13305
aI25195
asS'd'
I2250
sa(dp4500
S'a'
I3403
sS's'
I15
sS'm'
(lp4501
I1
aI3
aI7
aI5
aI15
aI11
aI27
aI95
aI387
aI931
aI549
aI2179
aI3397
aI15883
aI16563
asS'd'
I2251
sa(dp4502
S'a'
I3406
sS's'
I15
sS'm'
(lp4503
I1
aI1
aI7
aI3
aI9
aI39
aI121
aI5
aI453
aI27
aI1747
aI657
aI2593
aI1289
aI12577
asS'd'
I2252
sa(dp4504
S'a'
I3413
sS's'
I15
sS'm'
(lp4505
I1
aI3
aI7
aI5
aI25
aI25
aI109
aI49
aI185
aI985
aI631
aI803
aI3865
aI8955
aI17901
asS'd'
I2253
sa(dp4506
S'a'
I3420
sS's'
I15
sS'm'
(lp4507
I1
aI1
aI3
aI13
aI3
aI59
aI47
aI49
aI139
aI275
aI1471
aI2995
aI5593
aI14011
aI18741
asS'd'
I2254
sa(dp4508
S'a'
I3427
sS's'
I15
sS'm'
(lp4509
I1
aI1
aI5
aI15
aI29
aI11
aI97
aI225
aI245
aI291
aI1873
aI2365
aI767
aI3419
aI14943
asS'd'
I2255
sa(dp4510
S'a'
I3441
sS's'
I15
sS'm'
(lp4511
I1
aI3
aI3
aI5
aI15
aI17
aI19
aI209
aI359
aI891
aI1375
aI2003
aI7247
aI5299
aI28841
asS'd'
I2256
sa(dp4512
S'a'
I3453
sS's'
I15
sS'm'
(lp4513
I1
aI3
aI7
aI7
aI9
aI55
aI105
aI35
aI77
aI47
aI1023
aI13
aI2901
aI847
aI10265
asS'd'
I2257
sa(dp4514
S'a'
I3464
sS's'
I15
sS'm'
(lp4515
I1
aI3
aI7
aI7
aI7
aI5
aI65
aI233
aI141
aI277
aI1333
aI2357
aI443
aI7257
aI21979
asS'd'
I2258
sa(dp4516
S'a'
I3469
sS's'
I15
sS'm'
(lp4517
I1
aI3
aI5
aI11
aI13
aI63
aI41
aI87
aI193
aI737
aI1085
aI2317
aI7869
aI10149
aI12163
asS'd'
I2259
sa(dp4518
S'a'
I3481
sS's'
I15
sS'm'
(lp4519
I1
aI3
aI1
aI1
aI7
aI57
aI75
aI235
aI461
aI857
aI155
aI2679
aI5925
aI2565
aI10881
asS'd'
I2260
sa(dp4520
S'a'
I3488
sS's'
I15
sS'm'
(lp4521
I1
aI1
aI7
aI15
aI13
aI41
aI63
aI135
aI433
aI387
aI1943
aI2249
aI5469
aI11679
aI28661
asS'd'
I2261
sa(dp4522
S'a'
I3497
sS's'
I15
sS'm'
(lp4523
I1
aI3
aI3
aI13
aI5
aI3
aI103
aI161
aI367
aI649
aI789
aI1179
aI4163
aI5699
aI16787
asS'd'
I2262
sa(dp4524
S'a'
I3503
sS's'
I15
sS'm'
(lp4525
I1
aI3
aI7
aI7
aI31
aI13
aI45
aI141
aI113
aI769
aI1035
aI457
aI6709
aI14989
aI27311
asS'd'
I2263
sa(dp4526
S'a'
I3511
sS's'
I15
sS'm'
(lp4527
I1
aI1
aI3
aI1
aI1
aI43
aI119
aI145
aI111
aI593
aI1139
aI417
aI637
aI4437
aI17285
asS'd'
I2264
sa(dp4528
S'a'
I3515
sS's'
I15
sS'm'
(lp4529
I1
aI3
aI5
aI9
aI9
aI33
aI19
aI99
aI201
aI685
aI1793
aI2621
aI6857
aI8769
aI5623
asS'd'
I2265
sa(dp4530
S'a'
I3525
sS's'
I15
sS'm'
(lp4531
I1
aI3
aI5
aI5
aI23
aI43
aI27
aI189
aI325
aI415
aI215
aI1253
aI3599
aI1215
aI10093
asS'd'
I2266
sa(dp4532
S'a'
I3529
sS's'
I15
sS'm'
(lp4533
I1
aI1
aI3
aI13
aI11
aI35
aI113
aI173
aI503
aI19
aI1459
aI503
aI5363
aI3967
aI13945
asS'd'
I2267
sa(dp4534
S'a'
I3547
sS's'
I15
sS'm'
(lp4535
I1
aI1
aI5
aI11
aI31
aI49
aI13
aI173
aI199
aI623
aI1231
aI2495
aI6581
aI7957
aI25321
asS'd'
I2268
sa(dp4536
S'a'
I3550
sS's'
I15
sS'm'
(lp4537
I1
aI3
aI1
aI9
aI23
aI3
aI79
aI149
aI505
aI937
aI1839
aI3701
aI1673
aI8589
aI8031
asS'd'
I2269
sa(dp4538
S'a'
I3573
sS's'
I15
sS'm'
(lp4539
I1
aI3
aI3
aI5
aI21
aI27
aI107
aI11
aI505
aI407
aI177
aI3593
aI4729
aI12773
aI11685
asS'd'
I2270
sa(dp4540
S'a'
I3583
sS's'
I15
sS'm'
(lp4541
I1
aI3
aI1
aI11
aI29
aI49
aI79
aI53
aI61
aI895
aI2035
aI563
aI5613
aI6065
aI6207
asS'd'
I2271
sa(dp4542
S'a'
I3594
sS's'
I15
sS'm'
(lp4543
I1
aI1
aI3
aI7
aI1
aI53
aI3
aI215
aI99
aI865
aI1749
aI3533
aI4305
aI1243
aI28463
asS'd'
I2272
sa(dp4544
S'a'
I3607
sS's'
I15
sS'm'
(lp4545
I1
aI1
aI1
aI13
aI31
aI59
aI115
aI53
aI403
aI909
aI847
aI103
aI4967
aI10623
aI30073
asS'd'
I2273
sa(dp4546
S'a'
I3613
sS's'
I15
sS'm'
(lp4547
I1
aI1
aI7
aI5
aI27
aI1
aI119
aI83
aI457
aI81
aI395
aI811
aI6221
aI14337
aI541
asS'd'
I2274
sa(dp4548
S'a'
I3624
sS's'
I15
sS'm'
(lp4549
I1
aI1
aI5
aI5
aI5
aI53
aI83
aI117
aI269
aI327
aI875
aI101
aI3343
aI715
aI26339
asS'd'
I2275
sa(dp4550
S'a'
I3630
sS's'
I15
sS'm'
(lp4551
I1
aI1
aI1
aI11
aI31
aI39
aI121
aI147
aI305
aI383
aI1211
aI1897
aI7647
aI11687
aI18907
asS'd'
I2276
sa(dp4552
S'a'
I3635
sS's'
I15
sS'm'
(lp4553
I1
aI3
aI3
aI15
aI23
aI53
aI17
aI85
aI395
aI503
aI61
aI1745
aI4713
aI4641
aI13787
asS'd'
I2277
sa(dp4554
S'a'
I3642
sS's'
I15
sS'm'
(lp4555
I1
aI1
aI7
aI7
aI27
aI1
aI105
aI29
aI287
aI37
aI959
aI975
aI4427
aI4705
aI10175
asS'd'
I2278
sa(dp4556
S'a'
I3644
sS's'
I15
sS'm'
(lp4557
I1
aI3
aI3
aI5
aI7
aI63
aI57
aI199
aI27
aI107
aI1095
aI3923
aI6969
aI713
aI11619
asS'd'
I2279
sa(dp4558
S'a'
I3650
sS's'
I15
sS'm'
(lp4559
I1
aI3
aI5
aI1
aI5
aI49
aI85
aI45
aI449
aI45
aI49
aI3419
aI1109
aI455
aI15917
asS'd'
I2280
sa(dp4560
S'a'
I3679
sS's'
I15
sS'm'
(lp4561
I1
aI1
aI1
aI5
aI13
aI15
aI39
aI27
aI467
aI85
aI1537
aI3055
aI1977
aI8829
aI25231
asS'd'
I2281
sa(dp4562
S'a'
I3690
sS's'
I15
sS'm'
(lp4563
I1
aI1
aI1
aI15
aI1
aI47
aI23
aI121
aI147
aI547
aI1865
aI1491
aI779
aI3515
aI12667
asS'd'
I2282
sa(dp4564
S'a'
I3698
sS's'
I15
sS'm'
(lp4565
I1
aI3
aI3
aI1
aI19
aI5
aI77
aI101
aI1
aI721
aI1149
aI2967
aI4925
aI11889
aI16655
asS'd'
I2283
sa(dp4566
S'a'
I3704
sS's'
I15
sS'm'
(lp4567
I1
aI1
aI1
aI7
aI1
aI35
aI95
aI239
aI127
aI855
aI1031
aI455
aI7631
aI6039
aI21983
asS'd'
I2284
sa(dp4568
S'a'
I3707
sS's'
I15
sS'm'
(lp4569
I1
aI3
aI7
aI9
aI23
aI43
aI75
aI105
aI335
aI223
aI1825
aI3217
aI413
aI7473
aI30005
asS'd'
I2285
sa(dp4570
S'a'
I3713
sS's'
I15
sS'm'
(lp4571
I1
aI1
aI5
aI15
aI29
aI9
aI43
aI145
aI223
aI523
aI511
aI323
aI5955
aI11141
aI22533
asS'd'
I2286
sa(dp4572
S'a'
I3754
sS's'
I15
sS'm'
(lp4573
I1
aI1
aI3
aI1
aI13
aI61
aI93
aI133
aI461
aI233
aI383
aI693
aI7347
aI3165
aI27493
asS'd'
I2287
sa(dp4574
S'a'
I3756
sS's'
I15
sS'm'
(lp4575
I1
aI3
aI7
aI1
aI13
aI45
aI113
aI207
aI53
aI1007
aI815
aI1145
aI2937
aI289
aI22195
asS'd'
I2288
sa(dp4576
S'a'
I3761
sS's'
I15
sS'm'
(lp4577
I1
aI3
aI5
aI5
aI19
aI17
aI113
aI89
aI19
aI1023
aI1625
aI3277
aI697
aI5187
aI15433
asS'd'
I2289
sa(dp4578
S'a'
I3776
sS's'
I15
sS'm'
(lp4579
I1
aI1
aI3
aI13
aI21
aI15
aI15
aI197
aI409
aI391
aI1993
aI2475
aI3189
aI4431
aI29585
asS'd'
I2290
sa(dp4580
S'a'
I3781
sS's'
I15
sS'm'
(lp4581
I1
aI1
aI5
aI5
aI31
aI7
aI111
aI231
aI187
aI543
aI45
aI3863
aI3811
aI4573
aI4437
asS'd'
I2291
sa(dp4582
S'a'
I3788
sS's'
I15
sS'm'
(lp4583
I1
aI3
aI3
aI7
aI19
aI7
aI123
aI23
aI79
aI513
aI189
aI3663
aI1291
aI13257
aI8949
asS'd'
I2292
sa(dp4584
S'a'
I3791
sS's'
I15
sS'm'
(lp4585
I1
aI1
aI5
aI13
aI3
aI53
aI109
aI133
aI157
aI223
aI651
aI3059
aI6055
aI14455
aI26903
asS'd'
I2293
sa(dp4586
S'a'
I3794
sS's'
I15
sS'm'
(lp4587
I1
aI1
aI7
aI1
aI23
aI63
aI59
aI229
aI17
aI199
aI643
aI637
aI7631
aI13647
aI7399
asS'd'
I2294
sa(dp4588
S'a'
I3806
sS's'
I15
sS'm'
(lp4589
I1
aI1
aI1
aI3
aI1
aI51
aI119
aI67
aI335
aI543
aI913
aI3565
aI4795
aI13405
aI7463
asS'd'
I2295
sa(dp4590
S'a'
I3841
sS's'
I15
sS'm'
(lp4591
I1
aI1
aI5
aI3
aI31
aI5
aI91
aI97
aI23
aI223
aI837
aI1353
aI1929
aI12043
aI10039
asS'd'
I2296
sa(dp4592
S'a'
I3848
sS's'
I15
sS'm'
(lp4593
I1
aI3
aI5
aI7
aI19
aI3
aI79
aI171
aI301
aI687
aI1545
aI355
aI4709
aI12965
aI16797
asS'd'
I2297
sa(dp4594
S'a'
I3851
sS's'
I15
sS'm'
(lp4595
I1
aI3
aI5
aI11
aI11
aI49
aI111
aI123
aI251
aI569
aI1605
aI401
aI5439
aI13519
aI8847
asS'd'
I2298
sa(dp4596
S'a'
I3856
sS's'
I15
sS'm'
(lp4597
I1
aI3
aI1
aI3
aI3
aI53
aI7
aI55
aI369
aI633
aI181
aI4037
aI2993
aI15815
aI8661
asS'd'
I2299
sa(dp4598
S'a'
I3868
sS's'
I15
sS'm'
(lp4599
I1
aI1
aI1
aI13
aI31
aI29
aI75
aI167
aI279
aI597
aI539
aI1791
aI8013
aI4387
aI9717
asS'd'
I2300
sa(dp4600
S'a'
I3875
sS's'
I15
sS'm'
(lp4601
I1
aI1
aI5
aI7
aI17
aI15
aI99
aI183
aI211
aI49
aI225
aI3143
aI4537
aI13141
aI23375
asS'd'
I2301
sa(dp4602
S'a'
I3882
sS's'
I15
sS'm'
(lp4603
I1
aI1
aI3
aI5
aI3
aI59
aI25
aI149
aI467
aI69
aI1939
aI1007
aI2765
aI4693
aI29815
asS'd'
I2302
sa(dp4604
S'a'
I3884
sS's'
I15
sS'm'
(lp4605
I1
aI3
aI1
aI3
aI17
aI33
aI119
aI189
aI447
aI251
aI879
aI177
aI5395
aI13487
aI9587
asS'd'
I2303
sa(dp4606
S'a'
I3889
sS's'
I15
sS'm'
(lp4607
I1
aI3
aI3
aI7
aI15
aI31
aI115
aI3
aI21
aI817
aI475
aI1849
aI6041
aI12541
aI18701
asS'd'
I2304
sa(dp4608
S'a'
I3892
sS's'
I15
sS'm'
(lp4609
I1
aI1
aI5
aI13
aI31
aI33
aI7
aI115
aI361
aI587
aI1919
aI1007
aI3537
aI7493
aI19357
asS'd'
I2305
sa(dp4610
S'a'
I3919
sS's'
I15
sS'm'
(lp4611
I1
aI3
aI7
aI13
aI23
aI35
aI15
aI111
aI123
aI633
aI805
aI1983
aI2109
aI14477
aI4985
asS'd'
I2306
sa(dp4612
S'a'
I3921
sS's'
I15
sS'm'
(lp4613
I1
aI3
aI3
aI11
aI25
aI13
aI11
aI205
aI97
aI893
aI927
aI1291
aI4007
aI13593
aI29693
asS'd'
I2307
sa(dp4614
S'a'
I3958
sS's'
I15
sS'm'
(lp4615
I1
aI3
aI5
aI15
aI9
aI13
aI121
aI89
aI215
aI823
aI1389
aI1581
aI8095
aI4707
aI16061
asS'd'
I2308
sa(dp4616
S'a'
I3961
sS's'
I15
sS'm'
(lp4617
I1
aI3
aI1
aI3
aI23
aI39
aI83
aI23
aI47
aI941
aI1419
aI2389
aI5699
aI7519
aI5829
asS'd'
I2309
sa(dp4618
S'a'
I3973
sS's'
I15
sS'm'
(lp4619
I1
aI3
aI1
aI9
aI23
aI43
aI79
aI237
aI93
aI203
aI695
aI225
aI5645
aI3591
aI16775
asS'd'
I2310
sa(dp4620
S'a'
I3977
sS's'
I15
sS'm'
(lp4621
I1
aI3
aI5
aI3
aI15
aI19
aI89
aI129
aI375
aI125
aI225
aI1323
aI2267
aI11607
aI17937
asS'd'
I2311
sa(dp4622
S'a'
I3985
sS's'
I15
sS'm'
(lp4623
I1
aI3
aI3
aI1
aI31
aI37
aI93
aI133
aI377
aI959
aI707
aI621
aI7179
aI15493
aI30287
asS'd'
I2312
sa(dp4624
S'a'
I3991
sS's'
I15
sS'm'
(lp4625
I1
aI3
aI7
aI13
aI5
aI13
aI15
aI1
aI37
aI525
aI1641
aI2829
aI6139
aI4069
aI19187
asS'd'
I2313
sa(dp4626
S'a'
I4004
sS's'
I15
sS'm'
(lp4627
I1
aI3
aI3
aI9
aI17
aI3
aI67
aI97
aI375
aI845
aI403
aI973
aI3919
aI2275
aI31627
asS'd'
I2314
sa(dp4628
S'a'
I4007
sS's'
I15
sS'm'
(lp4629
I1
aI1
aI3
aI3
aI25
aI7
aI91
aI67
aI271
aI465
aI481
aI3477
aI5229
aI241
aI8411
asS'd'
I2315
sa(dp4630
S'a'
I4019
sS's'
I15
sS'm'
(lp4631
I1
aI1
aI1
aI11
aI1
aI41
aI109
aI115
aI75
aI787
aI309
aI2887
aI179
aI9073
aI13895
asS'd'
I2316
sa(dp4632
S'a'
I4045
sS's'
I15
sS'm'
(lp4633
I1
aI3
aI3
aI15
aI11
aI31
aI113
aI91
aI303
aI907
aI1933
aI2167
aI7799
aI11821
aI20659
asS'd'
I2317
sa(dp4634
S'a'
I4054
sS's'
I15
sS'm'
(lp4635
I1
aI3
aI1
aI15
aI27
aI17
aI21
aI41
aI99
aI137
aI1397
aI929
aI5819
aI11977
aI6201
asS'd'
I2318
sa(dp4636
S'a'
I4057
sS's'
I15
sS'm'
(lp4637
I1
aI1
aI7
aI13
aI21
aI29
aI47
aI239
aI287
aI305
aI899
aI2711
aI1723
aI3315
aI199
asS'd'
I2319
sa(dp4638
S'a'
I4058
sS's'
I15
sS'm'
(lp4639
I1
aI1
aI1
aI3
aI31
aI21
aI101
aI149
aI107
aI761
aI1197
aI1703
aI4803
aI8411
aI10649
asS'd'
I2320
sa(dp4640
S'a'
I4070
sS's'
I15
sS'm'
(lp4641
I1
aI1
aI5
aI15
aI23
aI45
aI109
aI221
aI85
aI619
aI169
aI1013
aI3305
aI9451
aI26189
asS'd'
I2321
sa(dp4642
S'a'
I4101
sS's'
I15
sS'm'
(lp4643
I1
aI3
aI5
aI13
aI7
aI57
aI19
aI153
aI231
aI627
aI565
aI1595
aI6309
aI5037
aI25505
asS'd'
I2322
sa(dp4644
S'a'
I4113
sS's'
I15
sS'm'
(lp4645
I1
aI1
aI7
aI7
aI1
aI45
aI43
aI79
aI271
aI59
aI219
aI2255
aI1785
aI7919
aI24061
asS'd'
I2323
sa(dp4646
S'a'
I4114
sS's'
I15
sS'm'
(lp4647
I1
aI3
aI7
aI5
aI31
aI57
aI57
aI231
aI33
aI227
aI531
aI679
aI1141
aI85
aI19777
asS'd'
I2324
sa(dp4648
S'a'
I4119
sS's'
I15
sS'm'
(lp4649
I1
aI1
aI3
aI15
aI11
aI59
aI59
aI169
aI459
aI693
aI907
aI1191
aI3783
aI12809
aI6263
asS'd'
I2325
sa(dp4650
S'a'
I4129
sS's'
I15
sS'm'
(lp4651
I1
aI1
aI7
aI13
aI19
aI21
aI105
aI65
aI267
aI141
aI1547
aI781
aI7295
aI13565
aI17775
asS'd'
I2326
sa(dp4652
S'a'
I4141
sS's'
I15
sS'm'
(lp4653
I1
aI3
aI3
aI5
aI31
aI63
aI97
aI155
aI477
aI661
aI329
aI797
aI2539
aI4061
aI10537
asS'd'
I2327
sa(dp4654
S'a'
I4142
sS's'
I15
sS'm'
(lp4655
I1
aI3
aI3
aI7
aI11
aI17
aI119
aI89
aI71
aI103
aI1043
aI413
aI6035
aI12829
aI11559
asS'd'
I2328
sa(dp4656
S'a'
I4147
sS's'
I15
sS'm'
(lp4657
I1
aI3
aI1
aI9
aI5
aI19
aI53
aI185
aI103
aI629
aI2015
aI1257
aI5163
aI10581
aI13449
asS'd'
I2329
sa(dp4658
S'a'
I4149
sS's'
I15
sS'm'
(lp4659
I1
aI1
aI1
aI5
aI23
aI35
aI25
aI129
aI179
aI959
aI677
aI2249
aI6315
aI12151
aI3459
asS'd'
I2330
sa(dp4660
S'a'
I4150
sS's'
I15
sS'm'
(lp4661
I1
aI1
aI1
aI1
aI9
aI47
aI93
aI45
aI35
aI45
aI265
aI2065
aI6225
aI25
aI27135
asS'd'
I2331
sa(dp4662
S'a'
I4164
sS's'
I15
sS'm'
(lp4663
I1
aI3
aI1
aI11
aI21
aI53
aI127
aI163
aI311
aI667
aI597
aI1561
aI4515
aI23
aI9551
asS'd'
I2332
sa(dp4664
S'a'
I4168
sS's'
I15
sS'm'
(lp4665
I1
aI1
aI3
aI3
aI7
aI47
aI105
aI211
aI241
aI95
aI389
aI899
aI6001
aI8129
aI19889
asS'd'
I2333
sa(dp4666
S'a'
I4186
sS's'
I15
sS'm'
(lp4667
I1
aI1
aI3
aI15
aI29
aI45
aI9
aI27
aI483
aI799
aI269
aI1811
aI4493
aI7109
aI22149
asS'd'
I2334
sa(dp4668
S'a'
I4198
sS's'
I15
sS'm'
(lp4669
I1
aI1
aI3
aI3
aI29
aI5
aI57
aI205
aI187
aI615
aI1677
aI3987
aI4577
aI8799
aI16311
asS'd'
I2335
sa(dp4670
S'a'
I4207
sS's'
I15
sS'm'
(lp4671
I1
aI1
aI5
aI3
aI15
aI5
aI91
aI101
aI319
aI445
aI1261
aI2039
aI4071
aI8249
aI11611
asS'd'
I2336
sa(dp4672
S'a'
I4221
sS's'
I15
sS'm'
(lp4673
I1
aI3
aI7
aI11
aI19
aI17
aI1
aI185
aI153
aI579
aI1001
aI2031
aI2295
aI16335
aI24771
asS'd'
I2337
sa(dp4674
S'a'
I4225
sS's'
I15
sS'm'
(lp4675
I1
aI3
aI3
aI15
aI13
aI45
aI93
aI185
aI319
aI667
aI1085
aI93
aI577
aI11551
aI11355
asS'd'
I2338
sa(dp4676
S'a'
I4231
sS's'
I15
sS'm'
(lp4677
I1
aI1
aI7
aI13
aI3
aI61
aI45
aI191
aI51
aI981
aI1151
aI2715
aI2503
aI4147
aI4587
asS'd'
I2339
sa(dp4678
S'a'
I4238
sS's'
I15
sS'm'
(lp4679
I1
aI1
aI3
aI3
aI27
aI17
aI71
aI141
aI57
aI981
aI1033
aI333
aI4639
aI15885
aI1039
asS'd'
I2340
sa(dp4680
S'a'
I4243
sS's'
I15
sS'm'
(lp4681
I1
aI3
aI3
aI15
aI21
aI55
aI33
aI123
aI357
aI893
aI829
aI4045
aI5027
aI11727
aI13357
asS'd'
I2341
sa(dp4682
S'a'
I4249
sS's'
I15
sS'm'
(lp4683
I1
aI1
aI1
aI9
aI31
aI47
aI27
aI223
aI311
aI205
aI179
aI3411
aI4019
aI10997
aI28115
asS'd'
I2342
sa(dp4684
S'a'
I4250
sS's'
I15
sS'm'
(lp4685
I1
aI3
aI5
aI1
aI3
aI39
aI15
aI7
aI501
aI641
aI735
aI295
aI2005
aI12641
aI19779
asS'd'
I2343
sa(dp4686
S'a'
I4252
sS's'
I15
sS'm'
(lp4687
I1
aI3
aI3
aI1
aI15
aI1
aI75
aI243
aI329
aI267
aI1323
aI2285
aI5389
aI11881
aI15737
asS'd'
I2344
sa(dp4688
S'a'
I4259
sS's'
I15
sS'm'
(lp4689
I1
aI1
aI3
aI3
aI13
aI17
aI101
aI99
aI209
aI939
aI1147
aI3221
aI5159
aI3435
aI183
asS'd'
I2345
sa(dp4690
S'a'
I4279
sS's'
I15
sS'm'
(lp4691
I1
aI1
aI1
aI1
aI27
aI43
aI29
aI179
aI179
aI659
aI807
aI313
aI4165
aI963
aI11317
asS'd'
I2346
sa(dp4692
S'a'
I4285
sS's'
I15
sS'm'
(lp4693
I1
aI1
aI3
aI13
aI9
aI51
aI125
aI245
aI381
aI555
aI1383
aI3887
aI2045
aI12829
aI12029
asS'd'
I2347
sa(dp4694
S'a'
I4288
sS's'
I15
sS'm'
(lp4695
I1
aI1
aI1
aI9
aI29
aI39
aI55
aI127
aI235
aI617
aI1553
aI3133
aI7735
aI14725
aI16733
asS'd'
I2348
sa(dp4696
S'a'
I4303
sS's'
I15
sS'm'
(lp4697
I1
aI1
aI3
aI5
aI15
aI9
aI47
aI217
aI89
aI987
aI1083
aI1045
aI4745
aI12915
aI13719
asS'd'
I2349
sa(dp4698
S'a'
I4312
sS's'
I15
sS'm'
(lp4699
I1
aI3
aI3
aI7
aI23
aI3
aI35
aI79
aI45
aI435
aI1549
aI2645
aI2831
aI10359
aI10041
asS'd'
I2350
sa(dp4700
S'a'
I4322
sS's'
I15
sS'm'
(lp4701
I1
aI1
aI7
aI15
aI31
aI61
aI25
aI223
aI511
aI319
aI487
aI1677
aI739
aI7097
aI18417
asS'd'
I2351
sa(dp4702
S'a'
I4327
sS's'
I15
sS'm'
(lp4703
I1
aI1
aI7
aI5
aI19
aI21
aI123
aI237
aI299
aI367
aI1341
aI1449
aI2949
aI8629
aI11051
asS'd'
I2352
sa(dp4704
S'a'
I4336
sS's'
I15
sS'm'
(lp4705
I1
aI3
aI7
aI7
aI31
aI53
aI125
aI33
aI257
aI719
aI1297
aI895
aI5095
aI10237
aI12309
asS'd'
I2353
sa(dp4706
S'a'
I4359
sS's'
I15
sS'm'
(lp4707
I1
aI3
aI1
aI5
aI31
aI59
aI73
aI211
aI97
aI209
aI1289
aI4033
aI6143
aI14275
aI7997
asS'd'
I2354
sa(dp4708
S'a'
I4384
sS's'
I15
sS'm'
(lp4709
I1
aI1
aI5
aI7
aI31
aI5
aI75
aI105
aI389
aI985
aI9
aI4033
aI1185
aI7821
aI19083
asS'd'
I2355
sa(dp4710
S'a'
I4387
sS's'
I15
sS'm'
(lp4711
I1
aI1
aI1
aI15
aI11
aI39
aI73
aI253
aI275
aI813
aI25
aI3441
aI2493
aI5873
aI3739
asS'd'
I2356
sa(dp4712
S'a'
I4401
sS's'
I15
sS'm'
(lp4713
I1
aI3
aI7
aI1
aI31
aI19
aI119
aI5
aI109
aI397
aI1329
aI3347
aI5941
aI12449
aI2533
asS'd'
I2357
sa(dp4714
S'a'
I4407
sS's'
I15
sS'm'
(lp4715
I1
aI1
aI1
aI1
aI5
aI59
aI61
aI175
aI435
aI985
aI65
aI3781
aI5425
aI15073
aI16361
asS'd'
I2358
sa(dp4716
S'a'
I4428
sS's'
I15
sS'm'
(lp4717
I1
aI3
aI5
aI7
aI31
aI13
aI53
aI87
aI69
aI305
aI1455
aI273
aI2197
aI4277
aI24423
asS'd'
I2359
sa(dp4718
S'a'
I4436
sS's'
I15
sS'm'
(lp4719
I1
aI3
aI3
aI15
aI13
aI13
aI91
aI171
aI71
aI583
aI15
aI3599
aI6801
aI10041
aI26097
asS'd'
I2360
sa(dp4720
S'a'
I4450
sS's'
I15
sS'm'
(lp4721
I1
aI3
aI3
aI5
aI5
aI13
aI91
aI225
aI63
aI69
aI1795
aI341
aI461
aI5015
aI9471
asS'd'
I2361
sa(dp4722
S'a'
I4452
sS's'
I15
sS'm'
(lp4723
I1
aI3
aI7
aI5
aI21
aI55
aI109
aI39
aI459
aI925
aI229
aI2855
aI5807
aI2117
aI31739
asS'd'
I2362
sa(dp4724
S'a'
I4459
sS's'
I15
sS'm'
(lp4725
I1
aI1
aI3
aI3
aI1
aI5
aI17
aI177
aI401
aI727
aI1555
aI3097
aI1243
aI5933
aI14579
asS'd'
I2363
sa(dp4726
S'a'
I4461
sS's'
I15
sS'm'
(lp4727
I1
aI1
aI7
aI3
aI19
aI19
aI37
aI87
aI105
aI73
aI197
aI4067
aI6237
aI10553
aI9207
asS'd'
I2364
sa(dp4728
S'a'
I4470
sS's'
I15
sS'm'
(lp4729
I1
aI1
aI3
aI15
aI1
aI55
aI119
aI115
aI441
aI3
aI1003
aI1631
aI197
aI12929
aI25385
asS'd'
I2365
sa(dp4730
S'a'
I4483
sS's'
I15
sS'm'
(lp4731
I1
aI3
aI7
aI11
aI31
aI1
aI119
aI49
aI467
aI647
aI685
aI2771
aI3689
aI11049
aI26787
asS'd'
I2366
sa(dp4732
S'a'
I4485
sS's'
I15
sS'm'
(lp4733
I1
aI1
aI1
aI11
aI19
aI19
aI21
aI73
aI459
aI935
aI615
aI371
aI1099
aI14407
aI10375
asS'd'
I2367
sa(dp4734
S'a'
I4486
sS's'
I15
sS'm'
(lp4735
I1
aI3
aI5
aI13
aI15
aI3
aI107
aI179
aI259
aI677
aI1101
aI315
aI7673
aI14639
aI11241
asS'd'
I2368
sa(dp4736
S'a'
I4492
sS's'
I15
sS'm'
(lp4737
I1
aI1
aI7
aI9
aI15
aI21
aI93
aI25
aI349
aI23
aI1087
aI27
aI5691
aI12997
aI29301
asS'd'
I2369
sa(dp4738
S'a'
I4497
sS's'
I15
sS'm'
(lp4739
I1
aI3
aI3
aI5
aI7
aI43
aI1
aI195
aI69
aI753
aI1315
aI2629
aI3259
aI5595
aI19439
asS'd'
I2370
sa(dp4740
S'a'
I4514
sS's'
I15
sS'm'
(lp4741
I1
aI3
aI5
aI5
aI31
aI9
aI75
aI217
aI217
aI197
aI1925
aI2033
aI3585
aI15219
aI20251
asS'd'
I2371
sa(dp4742
S'a'
I4533
sS's'
I15
sS'm'
(lp4743
I1
aI1
aI5
aI11
aI17
aI31
aI3
aI209
aI315
aI49
aI949
aI2267
aI4611
aI4375
aI16431
asS'd'
I2372
sa(dp4744
S'a'
I4537
sS's'
I15
sS'm'
(lp4745
I1
aI1
aI7
aI9
aI17
aI35
aI13
aI115
aI119
aI553
aI1527
aI2857
aI3599
aI391
aI25101
asS'd'
I2373
sa(dp4746
S'a'
I4546
sS's'
I15
sS'm'
(lp4747
I1
aI3
aI3
aI15
aI13
aI59
aI17
aI177
aI301
aI719
aI909
aI1663
aI5033
aI1129
aI529
asS'd'
I2374
sa(dp4748
S'a'
I4551
sS's'
I15
sS'm'
(lp4749
I1
aI1
aI7
aI5
aI15
aI13
aI99
aI157
aI379
aI975
aI1019
aI2251
aI3807
aI10621
aI351
asS'd'
I2375
sa(dp4750
S'a'
I4555
sS's'
I15
sS'm'
(lp4751
I1
aI3
aI3
aI13
aI5
aI57
aI5
aI31
aI361
aI981
aI883
aI3723
aI2259
aI5151
aI11783
asS'd'
I2376
sa(dp4752
S'a'
I4560
sS's'
I15
sS'm'
(lp4753
I1
aI1
aI1
aI13
aI1
aI43
aI125
aI19
aI77
aI509
aI1817
aI3795
aI1863
aI8401
aI27253
asS'd'
I2377
sa(dp4754
S'a'
I4569
sS's'
I15
sS'm'
(lp4755
I1
aI1
aI5
aI7
aI19
aI41
aI21
aI151
aI89
aI189
aI769
aI1937
aI4497
aI13607
aI24691
asS'd'
I2378
sa(dp4756
S'a'
I4576
sS's'
I15
sS'm'
(lp4757
I1
aI1
aI1
aI9
aI21
aI9
aI1
aI195
aI31
aI907
aI1719
aI1549
aI809
aI13629
aI16597
asS'd'
I2379
sa(dp4758
S'a'
I4582
sS's'
I15
sS'm'
(lp4759
I1
aI1
aI1
aI3
aI21
aI61
aI103
aI219
aI311
aI849
aI523
aI21
aI4533
aI6367
aI3935
asS'd'
I2380
sa(dp4760
S'a'
I4586
sS's'
I15
sS'm'
(lp4761
I1
aI1
aI7
aI9
aI7
aI33
aI77
aI19
aI489
aI933
aI1729
aI1813
aI6741
aI10701
aI7
asS'd'
I2381
sa(dp4762
S'a'
I4609
sS's'
I15
sS'm'
(lp4763
I1
aI1
aI1
aI5
aI23
aI53
aI43
aI63
aI453
aI209
aI1313
aI2847
aI2641
aI13783
aI14983
asS'd'
I2382
sa(dp4764
S'a'
I4610
sS's'
I15
sS'm'
(lp4765
I1
aI3
aI7
aI7
aI15
aI45
aI83
aI241
aI509
aI659
aI213
aI221
aI5205
aI6019
aI18945
asS'd'
I2383
sa(dp4766
S'a'
I4612
sS's'
I15
sS'm'
(lp4767
I1
aI1
aI5
aI9
aI25
aI43
aI37
aI9
aI191
aI505
aI765
aI295
aI953
aI1045
aI11203
asS'd'
I2384
sa(dp4768
S'a'
I4649
sS's'
I15
sS'm'
(lp4769
I1
aI3
aI7
aI11
aI5
aI49
aI45
aI177
aI379
aI695
aI355
aI1711
aI7747
aI497
aI7597
asS'd'
I2385
sa(dp4770
S'a'
I4652
sS's'
I15
sS'm'
(lp4771
I1
aI1
aI5
aI13
aI23
aI47
aI101
aI145
aI301
aI207
aI195
aI2225
aI8093
aI15345
aI14097
asS'd'
I2386
sa(dp4772
S'a'
I4672
sS's'
I15
sS'm'
(lp4773
I1
aI3
aI7
aI13
aI9
aI9
aI55
aI223
aI343
aI921
aI1825
aI3281
aI2627
aI855
aI27651
asS'd'
I2387
sa(dp4774
S'a'
I4677
sS's'
I15
sS'm'
(lp4775
I1
aI1
aI7
aI1
aI21
aI1
aI67
aI149
aI433
aI111
aI577
aI3675
aI495
aI9043
aI23613
asS'd'
I2388
sa(dp4776
S'a'
I4684
sS's'
I15
sS'm'
(lp4777
I1
aI3
aI1
aI13
aI9
aI39
aI37
aI73
aI117
aI559
aI1131
aI2511
aI7599
aI8393
aI24747
asS'd'
I2389
sa(dp4778
S'a'
I4690
sS's'
I15
sS'm'
(lp4779
I1
aI3
aI3
aI7
aI11
aI15
aI85
aI229
aI7
aI21
aI1649
aI739
aI375
aI13991
aI27053
asS'd'
I2390
sa(dp4780
S'a'
I4695
sS's'
I15
sS'm'
(lp4781
I1
aI1
aI5
aI5
aI15
aI41
aI49
aI117
aI173
aI825
aI1343
aI377
aI1789
aI12519
aI30667
asS'd'
I2391
sa(dp4782
S'a'
I4696
sS's'
I15
sS'm'
(lp4783
I1
aI1
aI7
aI15
aI9
aI11
aI97
aI99
aI347
aI729
aI9
aI1703
aI1177
aI5189
aI9061
asS'd'
I2392
sa(dp4784
S'a'
I4702
sS's'
I15
sS'm'
(lp4785
I1
aI1
aI5
aI11
aI15
aI25
aI99
aI63
aI89
aI675
aI561
aI215
aI8111
aI3955
aI24635
asS'd'
I2393
sa(dp4786
S'a'
I4705
sS's'
I15
sS'm'
(lp4787
I1
aI1
aI1
aI1
aI7
aI53
aI99
aI193
aI233
aI731
aI733
aI1883
aI7783
aI14413
aI14003
asS'd'
I2394
sa(dp4788
S'a'
I4717
sS's'
I15
sS'm'
(lp4789
I1
aI3
aI5
aI7
aI31
aI23
aI45
aI153
aI337
aI293
aI443
aI2301
aI5135
aI7455
aI13123
asS'd'
I2395
sa(dp4790
S'a'
I4726
sS's'
I15
sS'm'
(lp4791
I1
aI3
aI1
aI3
aI23
aI53
aI23
aI165
aI53
aI875
aI1543
aI1035
aI4247
aI5101
aI28445
asS'd'
I2396
sa(dp4792
S'a'
I4736
sS's'
I15
sS'm'
(lp4793
I1
aI1
aI1
aI15
aI13
aI41
aI77
aI93
aI205
aI743
aI1101
aI1413
aI2371
aI7183
aI12337
asS'd'
I2397
sa(dp4794
S'a'
I4753
sS's'
I15
sS'm'
(lp4795
I1
aI1
aI3
aI15
aI17
aI63
aI25
aI101
aI147
aI149
aI1207
aI3525
aI2661
aI9539
aI11145
asS'd'
I2398
sa(dp4796
S'a'
I4754
sS's'
I15
sS'm'
(lp4797
I1
aI3
aI1
aI9
aI17
aI5
aI3
aI35
aI389
aI909
aI1017
aI2803
aI5243
aI13025
aI8851
asS'd'
I2399
sa(dp4798
S'a'
I4756
sS's'
I15
sS'm'
(lp4799
I1
aI1
aI7
aI15
aI19
aI27
aI69
aI91
aI71
aI547
aI1421
aI831
aI6969
aI5517
aI28233
asS'd'
I2400
sa(dp4800
S'a'
I4775
sS's'
I15
sS'm'
(lp4801
I1
aI1
aI3
aI3
aI17
aI45
aI55
aI63
aI263
aI819
aI1211
aI2739
aI655
aI13269
aI22281
asS'd'
I2401
sa(dp4802
S'a'
I4801
sS's'
I15
sS'm'
(lp4803
I1
aI3
aI1
aI5
aI23
aI13
aI81
aI251
aI83
aI551
aI491
aI1029
aI3561
aI357
aI23393
asS'd'
I2402
sa(dp4804
S'a'
I4819
sS's'
I15
sS'm'
(lp4805
I1
aI3
aI1
aI13
aI25
aI27
aI93
aI143
aI407
aI403
aI1395
aI1733
aI3187
aI1917
aI31453
asS'd'
I2403
sa(dp4806
S'a'
I4828
sS's'
I15
sS'm'
(lp4807
I1
aI1
aI7
aI13
aI3
aI21
aI85
aI113
aI483
aI461
aI1343
aI561
aI2081
aI10857
aI24253
asS'd'
I2404
sa(dp4808
S'a'
I4838
sS's'
I15
sS'm'
(lp4809
I1
aI1
aI1
aI1
aI11
aI11
aI53
aI135
aI25
aI163
aI1729
aI617
aI1533
aI10881
aI16041
asS'd'
I2405
sa(dp4810
S'a'
I4852
sS's'
I15
sS'm'
(lp4811
I1
aI1
aI5
aI1
aI3
aI49
aI125
aI139
aI77
aI891
aI815
aI3431
aI4875
aI12513
aI4595
asS'd'
I2406
sa(dp4812
S'a'
I4856
sS's'
I15
sS'm'
(lp4813
I1
aI1
aI1
aI1
aI27
aI63
aI111
aI109
aI421
aI425
aI345
aI1613
aI5447
aI1357
aI32413
asS'd'
I2407
sa(dp4814
S'a'
I4873
sS's'
I15
sS'm'
(lp4815
I1
aI3
aI5
aI3
aI17
aI5
aI37
aI171
aI259
aI281
aI1003
aI2901
aI3241
aI15557
aI21415
asS'd'
I2408
sa(dp4816
S'a'
I4887
sS's'
I15
sS'm'
(lp4817
I1
aI1
aI5
aI11
aI15
aI55
aI75
aI199
aI493
aI215
aI1625
aI2345
aI7873
aI2325
aI11003
asS'd'
I2409
sa(dp4818
S'a'
I4891
sS's'
I15
sS'm'
(lp4819
I1
aI3
aI7
aI1
aI21
aI33
aI23
aI5
aI495
aI941
aI1185
aI475
aI5799
aI15161
aI10677
asS'd'
I2410
sa(dp4820
S'a'
I4904
sS's'
I15
sS'm'
(lp4821
I1
aI1
aI5
aI9
aI31
aI37
aI37
aI29
aI217
aI389
aI297
aI3097
aI7319
aI2601
aI15307
asS'd'
I2411
sa(dp4822
S'a'
I4912
sS's'
I15
sS'm'
(lp4823
I1
aI3
aI7
aI5
aI7
aI45
aI111
aI167
aI297
aI275
aI1669
aI2489
aI1511
aI15753
aI1289
asS'd'
I2412
sa(dp4824
S'a'
I4921
sS's'
I15
sS'm'
(lp4825
I1
aI3
aI1
aI7
aI3
aI45
aI19
aI11
aI189
aI199
aI1227
aI2647
aI1897
aI9077
aI17189
asS'd'
I2413
sa(dp4826
S'a'
I4936
sS's'
I15
sS'm'
(lp4827
I1
aI1
aI1
aI13
aI15
aI39
aI19
aI179
aI147
aI341
aI283
aI3029
aI7599
aI8937
aI18761
asS'd'
I2414
sa(dp4828
S'a'
I4941
sS's'
I15
sS'm'
(lp4829
I1
aI3
aI3
aI9
aI3
aI11
aI41
aI255
aI365
aI835
aI921
aI389
aI919
aI15223
aI14541
asS'd'
I2415
sa(dp4830
S'a'
I4942
sS's'
I15
sS'm'
(lp4831
I1
aI1
aI3
aI3
aI5
aI37
aI29
aI203
aI313
aI271
aI1207
aI487
aI3711
aI3811
aI26757
asS'd'
I2416
sa(dp4832
S'a'
I4963
sS's'
I15
sS'm'
(lp4833
I1
aI3
aI7
aI9
aI19
aI53
aI49
aI139
aI351
aI537
aI1681
aI1595
aI5399
aI13839
aI28685
asS'd'
I2417
sa(dp4834
S'a'
I4984
sS's'
I15
sS'm'
(lp4835
I1
aI3
aI1
aI1
aI15
aI35
aI21
aI37
aI247
aI891
aI1855
aI1243
aI3137
aI10381
aI30379
asS'd'
I2418
sa(dp4836
S'a'
I4990
sS's'
I15
sS'm'
(lp4837
I1
aI3
aI7
aI5
aI9
aI47
aI91
aI25
aI479
aI337
aI781
aI3545
aI1045
aI9491
aI22853
asS'd'
I2419
sa(dp4838
S'a'
I5005
sS's'
I15
sS'm'
(lp4839
I1
aI1
aI5
aI15
aI19
aI31
aI81
aI5
aI117
aI923
aI565
aI2443
aI7383
aI1795
aI11685
asS'd'
I2420
sa(dp4840
S'a'
I5013
sS's'
I15
sS'm'
(lp4841
I1
aI3
aI3
aI5
aI17
aI15
aI21
aI245
aI489
aI889
aI2047
aI2737
aI7445
aI14785
aI13401
asS'd'
I2421
sa(dp4842
S'a'
I5020
sS's'
I15
sS'm'
(lp4843
I1
aI1
aI1
aI15
aI19
aI45
aI67
aI117
aI299
aI607
aI953
aI743
aI6863
aI12123
aI6701
asS'd'
I2422
sa(dp4844
S'a'
I5039
sS's'
I15
sS'm'
(lp4845
I1
aI1
aI3
aI1
aI1
aI43
aI19
aI129
aI345
aI861
aI209
aI2387
aI7205
aI7131
aI8235
asS'd'
I2423
sa(dp4846
S'a'
I5048
sS's'
I15
sS'm'
(lp4847
I1
aI3
aI5
aI1
aI1
aI13
aI75
aI99
aI333
aI157
aI23
aI1217
aI1857
aI15479
aI16031
asS'd'
I2424
sa(dp4848
S'a'
I5062
sS's'
I15
sS'm'
(lp4849
I1
aI3
aI3
aI11
aI7
aI61
aI119
aI89
aI491
aI401
aI227
aI1739
aI3807
aI16003
aI2875
asS'd'
I2425
sa(dp4850
S'a'
I5080
sS's'
I15
sS'm'
(lp4851
I1
aI3
aI7
aI15
aI13
aI55
aI3
aI159
aI405
aI593
aI975
aI361
aI2563
aI6061
aI28087
asS'd'
I2426
sa(dp4852
S'a'
I5085
sS's'
I15
sS'm'
(lp4853
I1
aI1
aI3
aI13
aI19
aI5
aI5
aI9
aI119
aI41
aI33
aI1111
aI4443
aI4663
aI28841
asS'd'
I2427
sa(dp4854
S'a'
I5086
sS's'
I15
sS'm'
(lp4855
I1
aI1
aI7
aI7
aI25
aI59
aI125
aI255
aI49
aI947
aI1673
aI2947
aI6369
aI2267
aI8813
asS'd'
I2428
sa(dp4856
S'a'
I5095
sS's'
I15
sS'm'
(lp4857
I1
aI1
aI5
aI15
aI25
aI25
aI111
aI193
aI217
aI193
aI821
aI2779
aI69
aI2957
aI27043
asS'd'
I2429
sa(dp4858
S'a'
I5096
sS's'
I15
sS'm'
(lp4859
I1
aI3
aI5
aI7
aI21
aI19
aI51
aI157
aI203
aI487
aI1745
aI1875
aI911
aI14071
aI7557
asS'd'
I2430
sa(dp4860
S'a'
I5102
sS's'
I15
sS'm'
(lp4861
I1
aI1
aI5
aI9
aI3
aI15
aI55
aI73
aI313
aI245
aI1061
aI1929
aI3035
aI607
aI11563
asS'd'
I2431
sa(dp4862
S'a'
I5107
sS's'
I15
sS'm'
(lp4863
I1
aI1
aI5
aI7
aI3
aI57
aI105
aI121
aI461
aI43
aI803
aI1801
aI4059
aI2157
aI17547
asS'd'
I2432
sa(dp4864
S'a'
I5141
sS's'
I15
sS'm'
(lp4865
I1
aI3
aI7
aI7
aI19
aI11
aI1
aI121
aI499
aI841
aI601
aI3515
aI2969
aI13697
aI8917
asS'd'
I2433
sa(dp4866
S'a'
I5145
sS's'
I15
sS'm'
(lp4867
I1
aI3
aI3
aI3
aI13
aI35
aI113
aI231
aI391
aI689
aI697
aI2871
aI7387
aI715
aI27005
asS'd'
I2434
sa(dp4868
S'a'
I5148
sS's'
I15
sS'm'
(lp4869
I1
aI1
aI1
aI13
aI19
aI5
aI17
aI43
aI175
aI291
aI987
aI1917
aI7635
aI15655
aI10689
asS'd'
I2435
sa(dp4870
S'a'
I5157
sS's'
I15
sS'm'
(lp4871
I1
aI1
aI7
aI15
aI19
aI37
aI121
aI243
aI125
aI623
aI1231
aI29
aI2325
aI5147
aI21435
asS'd'
I2436
sa(dp4872
S'a'
I5158
sS's'
I15
sS'm'
(lp4873
I1
aI3
aI5
aI15
aI25
aI27
aI57
aI187
aI77
aI401
aI1489
aI2977
aI5415
aI3381
aI2551
asS'd'
I2437
sa(dp4874
S'a'
I5162
sS's'
I15
sS'm'
(lp4875
I1
aI1
aI1
aI7
aI1
aI1
aI85
aI27
aI115
aI559
aI9
aI2365
aI711
aI5733
aI2819
asS'd'
I2438
sa(dp4876
S'a'
I5172
sS's'
I15
sS'm'
(lp4877
I1
aI3
aI1
aI15
aI9
aI29
aI61
aI113
aI169
aI349
aI591
aI1061
aI6041
aI7613
aI23691
asS'd'
I2439
sa(dp4878
S'a'
I5182
sS's'
I15
sS'm'
(lp4879
I1
aI1
aI5
aI1
aI13
aI45
aI49
aI227
aI345
aI563
aI87
aI3597
aI3961
aI7205
aI8441
asS'd'
I2440
sa(dp4880
S'a'
I5184
sS's'
I15
sS'm'
(lp4881
I1
aI1
aI1
aI5
aI3
aI21
aI121
aI183
aI463
aI83
aI1365
aI539
aI1485
aI10063
aI24867
asS'd'
I2441
sa(dp4882
S'a'
I5193
sS's'
I15
sS'm'
(lp4883
I1
aI3
aI5
aI5
aI3
aI61
aI101
aI237
aI41
aI147
aI1907
aI3049
aI7583
aI8283
aI6099
asS'd'
I2442
sa(dp4884
S'a'
I5199
sS's'
I15
sS'm'
(lp4885
I1
aI3
aI1
aI15
aI31
aI57
aI19
aI155
aI445
aI805
aI1793
aI207
aI1975
aI3357
aI14281
asS'd'
I2443
sa(dp4886
S'a'
I5201
sS's'
I15
sS'm'
(lp4887
I1
aI1
aI7
aI13
aI9
aI39
aI27
aI73
aI165
aI345
aI543
aI4095
aI133
aI10469
aI11573
asS'd'
I2444
sa(dp4888
S'a'
I5204
sS's'
I15
sS'm'
(lp4889
I1
aI1
aI7
aI15
aI17
aI57
aI99
aI81
aI359
aI367
aI1057
aI1173
aI4225
aI15127
aI2615
asS'd'
I2445
sa(dp4890
S'a'
I5211
sS's'
I15
sS'm'
(lp4891
I1
aI3
aI5
aI3
aI31
aI23
aI113
aI111
aI495
aI947
aI1625
aI1195
aI2053
aI1509
aI1347
asS'd'
I2446
sa(dp4892
S'a'
I5223
sS's'
I15
sS'm'
(lp4893
I1
aI1
aI5
aI5
aI9
aI47
aI25
aI63
aI455
aI107
aI771
aI3815
aI3827
aI16287
aI11615
asS'd'
I2447
sa(dp4894
S'a'
I5230
sS's'
I15
sS'm'
(lp4895
I1
aI1
aI7
aI9
aI17
aI61
aI51
aI215
aI63
aI123
aI1253
aI3927
aI721
aI9647
aI3283
asS'd'
I2448
sa(dp4896
S'a'
I5232
sS's'
I15
sS'm'
(lp4897
I1
aI1
aI5
aI15
aI11
aI17
aI83
aI255
aI473
aI107
aI681
aI763
aI7855
aI8043
aI31503
asS'd'
I2449
sa(dp4898
S'a'
I5253
sS's'
I15
sS'm'
(lp4899
I1
aI3
aI1
aI7
aI7
aI31
aI37
aI5
aI253
aI155
aI2017
aI609
aI1421
aI14927
aI25241
asS'd'
I2450
sa(dp4900
S'a'
I5257
sS's'
I15
sS'm'
(lp4901
I1
aI3
aI3
aI13
aI31
aI25
aI21
aI241
aI431
aI193
aI681
aI2265
aI5091
aI11479
aI21443
asS'd'
I2451
sa(dp4902
S'a'
I5260
sS's'
I15
sS'm'
(lp4903
I1
aI3
aI5
aI5
aI15
aI9
aI49
aI255
aI157
aI995
aI631
aI1995
aI3605
aI9085
aI24245
asS'd'
I2452
sa(dp4904
S'a'
I5284
sS's'
I15
sS'm'
(lp4905
I1
aI3
aI3
aI7
aI19
aI31
aI85
aI153
aI493
aI951
aI451
aI1587
aI6609
aI3681
aI13205
asS'd'
I2453
sa(dp4906
S'a'
I5306
sS's'
I15
sS'm'
(lp4907
I1
aI1
aI5
aI1
aI17
aI41
aI107
aI231
aI307
aI361
aI575
aI3239
aI3443
aI16159
aI20625
asS'd'
I2454
sa(dp4908
S'a'
I5331
sS's'
I15
sS'm'
(lp4909
I1
aI1
aI7
aI9
aI31
aI49
aI93
aI79
aI181
aI117
aI1241
aI3645
aI4901
aI12599
aI13247
asS'd'
I2455
sa(dp4910
S'a'
I5334
sS's'
I15
sS'm'
(lp4911
I1
aI3
aI3
aI9
aI7
aI31
aI127
aI201
aI11
aI199
aI1851
aI23
aI5667
aI8159
aI20951
asS'd'
I2456
sa(dp4912
S'a'
I5364
sS's'
I15
sS'm'
(lp4913
I1
aI3
aI3
aI7
aI3
aI37
aI29
aI189
aI65
aI461
aI769
aI321
aI6577
aI16223
aI16865
asS'd'
I2457
sa(dp4914
S'a'
I5367
sS's'
I15
sS'm'
(lp4915
I1
aI1
aI5
aI11
aI1
aI13
aI91
aI167
aI33
aI111
aI1445
aI1047
aI2479
aI12623
aI22893
asS'd'
I2458
sa(dp4916
S'a'
I5371
sS's'
I15
sS'm'
(lp4917
I1
aI1
aI3
aI1
aI3
aI1
aI47
aI185
aI329
aI903
aI1651
aI3005
aI907
aI1255
aI8303
asS'd'
I2459
sa(dp4918
S'a'
I5382
sS's'
I15
sS'm'
(lp4919
I1
aI3
aI5
aI13
aI19
aI31
aI5
aI233
aI265
aI769
aI1303
aI2503
aI2229
aI14019
aI20257
asS'd'
I2460
sa(dp4920
S'a'
I5386
sS's'
I15
sS'm'
(lp4921
I1
aI3
aI7
aI3
aI27
aI11
aI67
aI195
aI5
aI661
aI125
aI3761
aI7211
aI16043
aI7267
asS'd'
I2461
sa(dp4922
S'a'
I5399
sS's'
I15
sS'm'
(lp4923
I1
aI1
aI1
aI3
aI27
aI13
aI115
aI25
aI473
aI417
aI1751
aI2223
aI2099
aI5913
aI14273
asS'd'
I2462
sa(dp4924
S'a'
I5400
sS's'
I15
sS'm'
(lp4925
I1
aI3
aI7
aI15
aI13
aI53
aI99
aI115
aI225
aI737
aI1621
aI539
aI4131
aI471
aI31865
asS'd'
I2463
sa(dp4926
S'a'
I5409
sS's'
I15
sS'm'
(lp4927
I1
aI1
aI5
aI5
aI25
aI19
aI39
aI207
aI153
aI569
aI1755
aI2477
aI3065
aI7383
aI29919
asS'd'
I2464
sa(dp4928
S'a'
I5415
sS's'
I15
sS'm'
(lp4929
I1
aI3
aI5
aI11
aI13
aI59
aI33
aI3
aI435
aI273
aI701
aI3819
aI7291
aI11803
aI26111
asS'd'
I2465
sa(dp4930
S'a'
I5416
sS's'
I15
sS'm'
(lp4931
I1
aI1
aI3
aI9
aI29
aI19
aI71
aI59
aI93
aI1019
aI887
aI83
aI4675
aI7541
aI26821
asS'd'
I2466
sa(dp4932
S'a'
I5424
sS's'
I15
sS'm'
(lp4933
I1
aI3
aI1
aI3
aI21
aI53
aI71
aI73
aI43
aI321
aI1581
aI1399
aI4043
aI12995
aI16825
asS'd'
I2467
sa(dp4934
S'a'
I5436
sS's'
I15
sS'm'
(lp4935
I1
aI3
aI7
aI15
aI3
aI13
aI37
aI11
aI93
aI873
aI1193
aI3481
aI451
aI15869
aI17879
asS'd'
I2468
sa(dp4936
S'a'
I5454
sS's'
I15
sS'm'
(lp4937
I1
aI3
aI1
aI11
aI31
aI19
aI101
aI57
aI129
aI753
aI853
aI463
aI6757
aI11083
aI8667
asS'd'
I2469
sa(dp4938
S'a'
I5462
sS's'
I15
sS'm'
(lp4939
I1
aI3
aI5
aI15
aI25
aI41
aI25
aI197
aI235
aI609
aI905
aI993
aI3233
aI1935
aI24661
asS'd'
I2470
sa(dp4940
S'a'
I5468
sS's'
I15
sS'm'
(lp4941
I1
aI3
aI1
aI5
aI21
aI7
aI53
aI107
aI473
aI77
aI1135
aI1045
aI4933
aI5615
aI15931
asS'd'
I2471
sa(dp4942
S'a'
I5481
sS's'
I15
sS'm'
(lp4943
I1
aI3
aI7
aI11
aI3
aI9
aI105
aI183
aI151
aI527
aI425
aI975
aI4073
aI913
aI2793
asS'd'
I2472
sa(dp4944
S'a'
I5505
sS's'
I15
sS'm'
(lp4945
I1
aI1
aI7
aI13
aI19
aI61
aI81
aI9
aI413
aI851
aI1723
aI1113
aI1453
aI8635
aI3353
asS'd'
I2473
sa(dp4946
S'a'
I5511
sS's'
I15
sS'm'
(lp4947
I1
aI3
aI7
aI15
aI19
aI53
aI83
aI31
aI441
aI343
aI575
aI935
aI4543
aI1303
aI12567
asS'd'
I2474
sa(dp4948
S'a'
I5518
sS's'
I15
sS'm'
(lp4949
I1
aI1
aI1
aI5
aI29
aI19
aI119
aI75
aI3
aI591
aI845
aI649
aI1717
aI13695
aI26905
asS'd'
I2475
sa(dp4950
S'a'
I5530
sS's'
I15
sS'm'
(lp4951
I1
aI1
aI7
aI9
aI5
aI53
aI127
aI191
aI15
aI773
aI1433
aI2899
aI21
aI4977
aI17839
asS'd'
I2476
sa(dp4952
S'a'
I5532
sS's'
I15
sS'm'
(lp4953
I1
aI1
aI5
aI9
aI21
aI9
aI99
aI115
aI397
aI99
aI725
aI3835
aI973
aI1219
aI21159
asS'd'
I2477
sa(dp4954
S'a'
I5539
sS's'
I15
sS'm'
(lp4955
I1
aI3
aI5
aI3
aI7
aI39
aI29
aI93
aI303
aI913
aI981
aI3549
aI5225
aI10907
aI393
asS'd'
I2478
sa(dp4956
S'a'
I5553
sS's'
I15
sS'm'
(lp4957
I1
aI3
aI3
aI11
aI9
aI25
aI105
aI101
aI1
aI867
aI389
aI2241
aI773
aI14123
aI10015
asS'd'
I2479
sa(dp4958
S'a'
I5573
sS's'
I15
sS'm'
(lp4959
I1
aI1
aI5
aI1
aI1
aI37
aI117
aI213
aI391
aI779
aI1851
aI1485
aI1277
aI5607
aI819
asS'd'
I2480
sa(dp4960
S'a'
I5580
sS's'
I15
sS'm'
(lp4961
I1
aI3
aI7
aI1
aI3
aI5
aI43
aI47
aI483
aI367
aI749
aI1693
aI4961
aI15257
aI3775
asS'd'
I2481
sa(dp4962
S'a'
I5597
sS's'
I15
sS'm'
(lp4963
I1
aI3
aI3
aI1
aI27
aI11
aI21
aI83
aI437
aI379
aI1041
aI393
aI5611
aI2421
aI31739
asS'd'
I2482
sa(dp4964
S'a'
I5602
sS's'
I15
sS'm'
(lp4965
I1
aI3
aI5
aI7
aI19
aI1
aI79
aI63
aI53
aI201
aI1159
aI2501
aI6327
aI11317
aI9537
asS'd'
I2483
sa(dp4966
S'a'
I5608
sS's'
I15
sS'm'
(lp4967
I1
aI3
aI5
aI13
aI9
aI37
aI61
aI217
aI427
aI913
aI1311
aI3503
aI5473
aI10583
aI19723
asS'd'
I2484
sa(dp4968
S'a'
I5611
sS's'
I15
sS'm'
(lp4969
I1
aI1
aI3
aI9
aI11
aI29
aI121
aI175
aI141
aI515
aI925
aI837
aI6011
aI10419
aI32157
asS'd'
I2485
sa(dp4970
S'a'
I5613
sS's'
I15
sS'm'
(lp4971
I1
aI3
aI5
aI9
aI27
aI57
aI97
aI175
aI365
aI367
aI1737
aI3845
aI1257
aI12243
aI2201
asS'd'
I2486
sa(dp4972
S'a'
I5625
sS's'
I15
sS'm'
(lp4973
I1
aI3
aI3
aI9
aI23
aI1
aI53
aI123
aI127
aI333
aI1335
aI707
aI5747
aI6541
aI9809
asS'd'
I2487
sa(dp4974
S'a'
I5632
sS's'
I15
sS'm'
(lp4975
I1
aI3
aI1
aI9
aI17
aI37
aI101
aI41
aI91
aI61
aI433
aI979
aI4345
aI12351
aI10829
asS'd'
I2488
sa(dp4976
S'a'
I5635
sS's'
I15
sS'm'
(lp4977
I1
aI3
aI3
aI13
aI3
aI21
aI15
aI49
aI257
aI99
aI1793
aI2987
aI5233
aI11625
aI28069
asS'd'
I2489
sa(dp4978
S'a'
I5638
sS's'
I15
sS'm'
(lp4979
I1
aI1
aI7
aI11
aI21
aI13
aI89
aI11
aI135
aI153
aI783
aI2893
aI6815
aI12007
aI15605
asS'd'
I2490
sa(dp4980
S'a'
I5652
sS's'
I15
sS'm'
(lp4981
I1
aI3
aI7
aI13
aI5
aI61
aI73
aI5
aI269
aI699
aI925
aI2925
aI5919
aI5841
aI24875
asS'd'
I2491
sa(dp4982
S'a'
I5659
sS's'
I15
sS'm'
(lp4983
I1
aI3
aI5
aI5
aI25
aI45
aI43
aI93
aI15
aI927
aI1253
aI319
aI1173
aI14559
aI20221
asS'd'
I2492
sa(dp4984
S'a'
I5677
sS's'
I15
sS'm'
(lp4985
I1
aI1
aI3
aI3
aI27
aI45
aI9
aI103
aI447
aI627
aI1239
aI3869
aI2169
aI49
aI17917
asS'd'
I2493
sa(dp4986
S'a'
I5686
sS's'
I15
sS'm'
(lp4987
I1
aI3
aI7
aI7
aI11
aI9
aI1
aI1
aI1
aI527
aI825
aI3295
aI623
aI2095
aI10537
asS'd'
I2494
sa(dp4988
S'a'
I5689
sS's'
I15
sS'm'
(lp4989
I1
aI3
aI3
aI11
aI21
aI11
aI59
aI165
aI33
aI743
aI1461
aI1535
aI6393
aI1301
aI17823
asS'd'
I2495
sa(dp4990
S'a'
I5698
sS's'
I15
sS'm'
(lp4991
I1
aI1
aI7
aI3
aI19
aI43
aI47
aI245
aI469
aI551
aI1447
aI1963
aI169
aI1481
aI31925
asS'd'
I2496
sa(dp4992
S'a'
I5703
sS's'
I15
sS'm'
(lp4993
I1
aI1
aI3
aI1
aI11
aI21
aI51
aI7
aI251
aI199
aI1153
aI767
aI6417
aI3417
aI30171
asS'd'
I2497
sa(dp4994
S'a'
I5707
sS's'
I15
sS'm'
(lp4995
I1
aI3
aI7
aI1
aI31
aI5
aI41
aI103
aI447
aI263
aI211
aI2029
aI8021
aI4705
aI10579
asS'd'
I2498
sa(dp4996
S'a'
I5731
sS's'
I15
sS'm'
(lp4997
I1
aI1
aI3
aI5
aI17
aI25
aI55
aI75
aI393
aI107
aI2017
aI2389
aI1685
aI14021
aI9161
asS'd'
I2499
sa(dp4998
S'a'
I5738
sS's'
I15
sS'm'
(lp4999
I1
aI1
aI1
aI9
aI13
aI1
aI75
aI237
aI205
aI461
aI689
aI2531
aI2839
aI13925
aI23351
asS'd'
I2500
sa(dp5000
S'a'
I5743
sS's'
I15
sS'm'
(lp5001
I1
aI3
aI7
aI1
aI23
aI39
aI33
aI189
aI157
aI571
aI239
aI1053
aI1559
aI1685
aI23059
asS'd'
I2501
sa(dp5002
S'a'
I5748
sS's'
I15
sS'm'
(lp5003
I1
aI3
aI3
aI3
aI27
aI61
aI71
aI121
aI49
aI157
aI1341
aI1707
aI2417
aI11689
aI26507
asS'd'
I2502
sa(dp5004
S'a'
I5758
sS's'
I15
sS'm'
(lp5005
I1
aI3
aI7
aI7
aI19
aI63
aI47
aI53
aI95
aI791
aI1467
aI1273
aI2045
aI755
aI8555
asS'd'
I2503
sa(dp5006
S'a'
I5762
sS's'
I15
sS'm'
(lp5007
I1
aI1
aI3
aI15
aI27
aI33
aI21
aI253
aI317
aI153
aI1509
aI1765
aI3809
aI601
aI5907
asS'd'
I2504
sa(dp5008
S'a'
I5768
sS's'
I15
sS'm'
(lp5009
I1
aI3
aI5
aI15
aI11
aI17
aI97
aI91
aI165
aI199
aI1751
aI2135
aI1315
aI3077
aI29995
asS'd'
I2505
sa(dp5010
S'a'
I5773
sS's'
I15
sS'm'
(lp5011
I1
aI3
aI1
aI5
aI3
aI33
aI93
aI49
aI39
aI743
aI341
aI2549
aI7603
aI3369
aI30889
asS'd'
I2506
sa(dp5012
S'a'
I5776
sS's'
I15
sS'm'
(lp5013
I1
aI1
aI3
aI13
aI3
aI5
aI87
aI63
aI293
aI785
aI1591
aI675
aI3915
aI2209
aI18201
asS'd'
I2507
sa(dp5014
S'a'
I5815
sS's'
I15
sS'm'
(lp5015
I1
aI3
aI3
aI11
aI3
aI15
aI69
aI231
aI241
aI127
aI429
aI2201
aI8173
aI12549
aI25745
asS'd'
I2508
sa(dp5016
S'a'
I5841
sS's'
I15
sS'm'
(lp5017
I1
aI1
aI5
aI11
aI15
aI39
aI3
aI29
aI125
aI685
aI643
aI1385
aI829
aI7347
aI28793
asS'd'
I2509
sa(dp5018
S'a'
I5847
sS's'
I15
sS'm'
(lp5019
I1
aI1
aI7
aI15
aI27
aI15
aI59
aI237
aI299
aI773
aI1097
aI3875
aI6503
aI7129
aI28495
asS'd'
I2510
sa(dp5020
S'a'
I5860
sS's'
I15
sS'm'
(lp5021
I1
aI3
aI5
aI13
aI9
aI17
aI31
aI227
aI69
aI443
aI1633
aI525
aI1659
aI14681
aI15209
asS'd'
I2511
sa(dp5022
S'a'
I5870
sS's'
I15
sS'm'
(lp5023
I1
aI3
aI5
aI5
aI13
aI51
aI69
aI173
aI111
aI433
aI279
aI2145
aI2091
aI9741
aI24881
asS'd'
I2512
sa(dp5024
S'a'
I5875
sS's'
I15
sS'm'
(lp5025
I1
aI3
aI1
aI7
aI7
aI35
aI55
aI51
aI357
aI99
aI1789
aI333
aI2073
aI10151
aI14527
asS'd'
I2513
sa(dp5026
S'a'
I5877
sS's'
I15
sS'm'
(lp5027
I1
aI3
aI3
aI7
aI13
aI41
aI101
aI87
aI425
aI701
aI1143
aI2733
aI6473
aI8667
aI17419
asS'd'
I2514
sa(dp5028
S'a'
I5884
sS's'
I15
sS'm'
(lp5029
I1
aI1
aI5
aI5
aI25
aI29
aI63
aI31
aI385
aI537
aI563
aI607
aI6723
aI9251
aI6531
asS'd'
I2515
sa(dp5030
S'a'
I5892
sS's'
I15
sS'm'
(lp5031
I1
aI3
aI5
aI5
aI9
aI63
aI111
aI131
aI239
aI723
aI705
aI2805
aI6579
aI12691
aI17521
asS'd'
I2516
sa(dp5032
S'a'
I5902
sS's'
I15
sS'm'
(lp5033
I1
aI3
aI1
aI7
aI31
aI55
aI101
aI225
aI477
aI271
aI611
aI3179
aI7859
aI9835
aI2165
asS'd'
I2517
sa(dp5034
S'a'
I5910
sS's'
I15
sS'm'
(lp5035
I1
aI1
aI3
aI3
aI5
aI15
aI81
aI127
aI391
aI333
aI419
aI1091
aI5997
aI12315
aI31521
asS'd'
I2518
sa(dp5036
S'a'
I5916
sS's'
I15
sS'm'
(lp5037
I1
aI3
aI5
aI15
aI23
aI7
aI35
aI109
aI181
aI295
aI825
aI419
aI969
aI15753
aI9365
asS'd'
I2519
sa(dp5038
S'a'
I5919
sS's'
I15
sS'm'
(lp5039
I1
aI3
aI5
aI5
aI25
aI23
aI69
aI177
aI325
aI359
aI1577
aI619
aI6233
aI11753
aI8103
asS'd'
I2520
sa(dp5040
S'a'
I5935
sS's'
I15
sS'm'
(lp5041
I1
aI3
aI5
aI11
aI31
aI13
aI79
aI61
aI241
aI1011
aI1961
aI949
aI6211
aI497
aI7099
asS'd'
I2521
sa(dp5042
S'a'
I5937
sS's'
I15
sS'm'
(lp5043
I1
aI3
aI5
aI3
aI25
aI19
aI67
aI235
aI337
aI1015
aI1485
aI355
aI3653
aI12735
aI14503
asS'd'
I2522
sa(dp5044
S'a'
I5944
sS's'
I15
sS'm'
(lp5045
I1
aI3
aI5
aI7
aI31
aI23
aI35
aI231
aI147
aI15
aI263
aI1995
aI431
aI5941
aI18931
asS'd'
I2523
sa(dp5046
S'a'
I5947
sS's'
I15
sS'm'
(lp5047
I1
aI3
aI3
aI7
aI1
aI35
aI37
aI7
aI85
aI443
aI715
aI743
aI2189
aI12537
aI17427
asS'd'
I2524
sa(dp5048
S'a'
I5958
sS's'
I15
sS'm'
(lp5049
I1
aI1
aI3
aI1
aI7
aI41
aI1
aI209
aI121
aI929
aI661
aI3999
aI955
aI5123
aI31115
asS'd'
I2525
sa(dp5050
S'a'
I5962
sS's'
I15
sS'm'
(lp5051
I1
aI1
aI3
aI5
aI11
aI43
aI127
aI125
aI107
aI293
aI273
aI2071
aI3003
aI11631
aI7769
asS'd'
I2526
sa(dp5052
S'a'
I5969
sS's'
I15
sS'm'
(lp5053
I1
aI1
aI1
aI13
aI13
aI29
aI39
aI217
aI111
aI779
aI1287
aI1675
aI4201
aI4869
aI20403
asS'd'
I2527
sa(dp5054
S'a'
I5981
sS's'
I15
sS'm'
(lp5055
I1
aI1
aI3
aI15
aI25
aI53
aI25
aI135
aI389
aI925
aI1971
aI663
aI7545
aI2673
aI7725
asS'd'
I2528
sa(dp5056
S'a'
I5995
sS's'
I15
sS'm'
(lp5057
I1
aI1
aI5
aI13
aI3
aI59
aI97
aI91
aI357
aI45
aI947
aI3031
aI8095
aI6269
aI13975
asS'd'
I2529
sa(dp5058
S'a'
I5998
sS's'
I15
sS'm'
(lp5059
I1
aI1
aI5
aI15
aI25
aI31
aI1
aI171
aI375
aI939
aI507
aI3591
aI1089
aI13605
aI2813
asS'd'
I2530
sa(dp5060
S'a'
I6003
sS's'
I15
sS'm'
(lp5061
I1
aI1
aI3
aI7
aI25
aI21
aI41
aI131
aI147
aI737
aI9
aI1603
aI1859
aI11573
aI28397
asS'd'
I2531
sa(dp5062
S'a'
I6010
sS's'
I15
sS'm'
(lp5063
I1
aI3
aI3
aI9
aI21
aI9
aI59
aI27
aI169
aI875
aI711
aI1389
aI2899
aI7937
aI4173
asS'd'
I2532
sa(dp5064
S'a'
I6016
sS's'
I15
sS'm'
(lp5065
I1
aI1
aI5
aI9
aI13
aI29
aI71
aI39
aI51
aI337
aI1067
aI2661
aI1203
aI5967
aI19249
asS'd'
I2533
sa(dp5066
S'a'
I6025
sS's'
I15
sS'm'
(lp5067
I1
aI3
aI7
aI1
aI17
aI21
aI43
aI79
aI181
aI741
aI1901
aI3445
aI7171
aI2109
aI1589
asS'd'
I2534
sa(dp5068
S'a'
I6031
sS's'
I15
sS'm'
(lp5069
I1
aI1
aI3
aI9
aI23
aI37
aI105
aI51
aI227
aI775
aI1265
aI2987
aI2197
aI13903
aI28891
asS'd'
I2535
sa(dp5070
S'a'
I6036
sS's'
I15
sS'm'
(lp5071
I1
aI1
aI1
aI13
aI23
aI47
aI111
aI41
aI93
aI261
aI75
aI2155
aI4301
aI11517
aI16101
asS'd'
I2536
sa(dp5072
S'a'
I6039
sS's'
I15
sS'm'
(lp5073
I1
aI1
aI3
aI3
aI27
aI27
aI123
aI125
aI501
aI775
aI413
aI1065
aI7607
aI15065
aI26013
asS'd'
I2537
sa(dp5074
S'a'
I6045
sS's'
I15
sS'm'
(lp5075
I1
aI3
aI7
aI3
aI27
aI11
aI59
aI87
aI207
aI743
aI1765
aI2969
aI913
aI8101
aI11583
asS'd'
I2538
sa(dp5076
S'a'
I6049
sS's'
I15
sS'm'
(lp5077
I1
aI3
aI3
aI1
aI23
aI7
aI113
aI17
aI285
aI993
aI695
aI2399
aI5019
aI4779
aI28917
asS'd'
I2539
sa(dp5078
S'a'
I6052
sS's'
I15
sS'm'
(lp5079
I1
aI3
aI1
aI5
aI11
aI51
aI49
aI139
aI213
aI435
aI1475
aI2209
aI6695
aI12981
aI9851
asS'd'
I2540
sa(dp5080
S'a'
I6067
sS's'
I15
sS'm'
(lp5081
I1
aI3
aI5
aI7
aI1
aI63
aI31
aI151
aI173
aI767
aI1453
aI1497
aI6911
aI9597
aI25551
asS'd'
I2541
sa(dp5082
S'a'
I6074
sS's'
I15
sS'm'
(lp5083
I1
aI1
aI7
aI7
aI21
aI53
aI39
aI159
aI389
aI231
aI309
aI359
aI7701
aI14819
aI5175
asS'd'
I2542
sa(dp5084
S'a'
I6087
sS's'
I15
sS'm'
(lp5085
I1
aI1
aI1
aI1
aI11
aI47
aI83
aI29
aI247
aI89
aI369
aI2727
aI3103
aI14421
aI17369
asS'd'
I2543
sa(dp5086
S'a'
I6101
sS's'
I15
sS'm'
(lp5087
I1
aI3
aI1
aI5
aI25
aI25
aI111
aI245
aI239
aI755
aI113
aI1765
aI3583
aI917
aI403
asS'd'
I2544
sa(dp5088
S'a'
I6121
sS's'
I15
sS'm'
(lp5089
I1
aI3
aI3
aI3
aI5
aI59
aI85
aI151
aI463
aI591
aI743
aI3767
aI121
aI2927
aI11031
asS'd'
I2545
sa(dp5090
S'a'
I6129
sS's'
I15
sS'm'
(lp5091
I1
aI3
aI5
aI9
aI11
aI39
aI77
aI161
aI275
aI233
aI1991
aI2683
aI6545
aI2423
aI32113
asS'd'
I2546
sa(dp5092
S'a'
I6142
sS's'
I15
sS'm'
(lp5093
I1
aI3
aI5
aI11
aI5
aI57
aI13
aI229
aI329
aI757
aI1863
aI3959
aI4243
aI7265
aI15599
asS'd'
I2547
sa(dp5094
S'a'
I6151
sS's'
I15
sS'm'
(lp5095
I1
aI1
aI1
aI1
aI1
aI23
aI19
aI67
aI453
aI593
aI2011
aI1813
aI4695
aI8903
aI9623
asS'd'
I2548
sa(dp5096
S'a'
I6157
sS's'
I15
sS'm'
(lp5097
I1
aI3
aI3
aI7
aI1
aI29
aI103
aI255
aI493
aI647
aI1709
aI4065
aI4199
aI949
aI28829
asS'd'
I2549
sa(dp5098
S'a'
I6166
sS's'
I15
sS'm'
(lp5099
I1
aI1
aI7
aI9
aI3
aI55
aI53
aI33
aI5
aI223
aI423
aI3347
aI7647
aI7211
aI25157
asS'd'
I2550
sa(dp5100
S'a'
I6170
sS's'
I15
sS'm'
(lp5101
I1
aI3
aI5
aI13
aI3
aI43
aI79
aI255
aI471
aI573
aI1007
aI2119
aI6731
aI10047
aI23179
asS'd'
I2551
sa(dp5102
S'a'
I6175
sS's'
I15
sS'm'
(lp5103
I1
aI1
aI1
aI3
aI7
aI39
aI55
aI61
aI53
aI377
aI435
aI401
aI3307
aI12621
aI14299
asS'd'
I2552
sa(dp5104
S'a'
I6186
sS's'
I15
sS'm'
(lp5105
I1
aI3
aI3
aI7
aI21
aI31
aI67
aI17
aI243
aI425
aI747
aI2995
aI1389
aI2557
aI18415
asS'd'
I2553
sa(dp5106
S'a'
I6203
sS's'
I15
sS'm'
(lp5107
I1
aI3
aI1
aI3
aI3
aI39
aI75
aI11
aI447
aI249
aI1135
aI1011
aI1657
aI10767
aI19501
asS'd'
I2554
sa(dp5108
S'a'
I6217
sS's'
I15
sS'm'
(lp5109
I1
aI3
aI1
aI11
aI17
aI51
aI117
aI129
aI17
aI143
aI785
aI103
aI5049
aI14703
aI28479
asS'd'
I2555
sa(dp5110
S'a'
I6231
sS's'
I15
sS'm'
(lp5111
I1
aI3
aI7
aI5
aI13
aI17
aI75
aI255
aI75
aI661
aI1175
aI477
aI1811
aI1479
aI15783
asS'd'
I2556
sa(dp5112
S'a'
I6241
sS's'
I15
sS'm'
(lp5113
I1
aI3
aI7
aI9
aI11
aI57
aI101
aI77
aI431
aI247
aI997
aI3657
aI5117
aI6815
aI3841
asS'd'
I2557
sa(dp5114
S'a'
I6242
sS's'
I15
sS'm'
(lp5115
I1
aI1
aI5
aI1
aI17
aI21
aI101
aI183
aI209
aI69
aI299
aI1585
aI6381
aI12983
aI10053
asS'd'
I2558
sa(dp5116
S'a'
I6248
sS's'
I15
sS'm'
(lp5117
I1
aI1
aI7
aI3
aI5
aI13
aI21
aI63
aI83
aI857
aI749
aI1251
aI5363
aI9629
aI16563
asS'd'
I2559
sa(dp5118
S'a'
I6256
sS's'
I15
sS'm'
(lp5119
I1
aI3
aI3
aI9
aI3
aI59
aI9
aI45
aI55
aI489
aI137
aI2423
aI2661
aI12111
aI4375
asS'd'
I2560
sa(dp5120
S'a'
I6265
sS's'
I15
sS'm'
(lp5121
I1
aI1
aI5
aI9
aI23
aI9
aI41
aI177
aI447
aI671
aI1631
aI3115
aI4215
aI14435
aI8743
asS'd'
I2561
sa(dp5122
S'a'
I6275
sS's'
I15
sS'm'
(lp5123
I1
aI3
aI7
aI11
aI19
aI23
aI15
aI221
aI413
aI783
aI1247
aI2343
aI4397
aI3145
aI32043
asS'd'
I2562
sa(dp5124
S'a'
I6277
sS's'
I15
sS'm'
(lp5125
I1
aI3
aI3
aI1
aI31
aI55
aI31
aI87
aI333
aI849
aI1777
aI343
aI5199
aI1507
aI11621
asS'd'
I2563
sa(dp5126
S'a'
I6302
sS's'
I15
sS'm'
(lp5127
I1
aI3
aI7
aI3
aI17
aI57
aI63
aI63
aI111
aI977
aI631
aI3019
aI2953
aI14273
aI29209
asS'd'
I2564
sa(dp5128
S'a'
I6315
sS's'
I15
sS'm'
(lp5129
I1
aI3
aI1
aI13
aI9
aI39
aI87
aI15
aI397
aI185
aI701
aI1487
aI3807
aI13727
aI19883
asS'd'
I2565
sa(dp5130
S'a'
I6318
sS's'
I15
sS'm'
(lp5131
I1
aI3
aI7
aI1
aI17
aI57
aI57
aI157
aI119
aI181
aI899
aI353
aI3603
aI15041
aI7421
asS'd'
I2566
sa(dp5132
S'a'
I6330
sS's'
I15
sS'm'
(lp5133
I1
aI1
aI7
aI3
aI29
aI13
aI29
aI191
aI105
aI373
aI961
aI1991
aI5531
aI6793
aI29497
asS'd'
I2567
sa(dp5134
S'a'
I6343
sS's'
I15
sS'm'
(lp5135
I1
aI3
aI3
aI11
aI7
aI61
aI65
aI39
aI215
aI187
aI191
aI1651
aI2481
aI3951
aI24965
asS'd'
I2568
sa(dp5136
S'a'
I6347
sS's'
I15
sS'm'
(lp5137
I1
aI1
aI7
aI5
aI25
aI11
aI105
aI23
aI257
aI771
aI1359
aI2837
aI7821
aI12223
aI28033
asS'd'
I2569
sa(dp5138
S'a'
I6350
sS's'
I15
sS'm'
(lp5139
I1
aI3
aI5
aI11
aI3
aI3
aI23
aI139
aI407
aI885
aI1679
aI2979
aI8149
aI14281
aI12487
asS'd'
I2570
sa(dp5140
S'a'
I6352
sS's'
I15
sS'm'
(lp5141
I1
aI3
aI7
aI3
aI21
aI45
aI13
aI85
aI249
aI1015
aI2023
aI1429
aI965
aI7091
aI31721
asS'd'
I2571
sa(dp5142
S'a'
I6371
sS's'
I15
sS'm'
(lp5143
I1
aI1
aI1
aI13
aI19
aI5
aI119
aI47
aI91
aI285
aI211
aI2607
aI4287
aI9197
aI455
asS'd'
I2572
sa(dp5144
S'a'
I6383
sS's'
I15
sS'm'
(lp5145
I1
aI3
aI1
aI1
aI9
aI59
aI25
aI137
aI121
aI287
aI577
aI3325
aI2365
aI8823
aI5033
asS'd'
I2573
sa(dp5146
S'a'
I6386
sS's'
I15
sS'm'
(lp5147
I1
aI3
aI3
aI13
aI25
aI63
aI99
aI43
aI15
aI855
aI245
aI3189
aI59
aI5181
aI21299
asS'd'
I2574
sa(dp5148
S'a'
I6405
sS's'
I15
sS'm'
(lp5149
I1
aI3
aI5
aI11
aI7
aI9
aI41
aI157
aI359
aI773
aI1347
aI2049
aI4589
aI13731
aI32133
asS'd'
I2575
sa(dp5150
S'a'
I6409
sS's'
I15
sS'm'
(lp5151
I1
aI1
aI7
aI11
aI31
aI37
aI83
aI105
aI183
aI375
aI79
aI1821
aI1989
aI15199
aI22207
asS'd'
I2576
sa(dp5152
S'a'
I6410
sS's'
I15
sS'm'
(lp5153
I1
aI1
aI5
aI3
aI23
aI37
aI127
aI9
aI467
aI651
aI993
aI69
aI6943
aI4093
aI20871
asS'd'
I2577
sa(dp5154
S'a'
I6433
sS's'
I15
sS'm'
(lp5155
I1
aI1
aI3
aI15
aI31
aI49
aI123
aI149
aI211
aI371
aI1825
aI3011
aI485
aI1251
aI17343
asS'd'
I2578
sa(dp5156
S'a'
I6436
sS's'
I15
sS'm'
(lp5157
I1
aI1
aI1
aI15
aI11
aI33
aI127
aI251
aI89
aI317
aI1869
aI219
aI2275
aI14201
aI27063
asS'd'
I2579
sa(dp5158
S'a'
I6439
sS's'
I15
sS'm'
(lp5159
I1
aI1
aI5
aI5
aI19
aI5
aI81
aI35
aI233
aI95
aI9
aI863
aI725
aI11095
aI16217
asS'd'
I2580
sa(dp5160
S'a'
I6463
sS's'
I15
sS'm'
(lp5161
I1
aI1
aI1
aI15
aI23
aI47
aI51
aI43
aI169
aI637
aI865
aI57
aI1509
aI1683
aI7587
asS'd'
I2581
sa(dp5162
S'a'
I6468
sS's'
I15
sS'm'
(lp5163
I1
aI3
aI1
aI3
aI7
aI7
aI117
aI187
aI273
aI303
aI717
aI3091
aI2083
aI3315
aI647
asS'd'
I2582
sa(dp5164
S'a'
I6477
sS's'
I15
sS'm'
(lp5165
I1
aI1
aI5
aI15
aI13
aI27
aI23
aI227
aI145
aI547
aI1783
aI987
aI6895
aI7135
aI11023
asS'd'
I2583
sa(dp5166
S'a'
I6496
sS's'
I15
sS'm'
(lp5167
I1
aI1
aI5
aI11
aI21
aI39
aI57
aI203
aI477
aI17
aI985
aI1729
aI4297
aI7483
aI13263
asS'd'
I2584
sa(dp5168
S'a'
I6511
sS's'
I15
sS'm'
(lp5169
I1
aI3
aI7
aI9
aI3
aI49
aI71
aI45
aI143
aI967
aI39
aI583
aI2123
aI5165
aI17437
asS'd'
I2585
sa(dp5170
S'a'
I6516
sS's'
I15
sS'm'
(lp5171
I1
aI1
aI1
aI9
aI21
aI51
aI71
aI163
aI441
aI709
aI397
aI445
aI6167
aI7753
aI11513
asS'd'
I2586
sa(dp5172
S'a'
I6519
sS's'
I15
sS'm'
(lp5173
I1
aI1
aI7
aI7
aI27
aI35
aI5
aI181
aI449
aI53
aI621
aI3401
aI5263
aI4557
aI9141
asS'd'
I2587
sa(dp5174
S'a'
I6523
sS's'
I15
sS'm'
(lp5175
I1
aI1
aI5
aI7
aI7
aI37
aI83
aI111
aI485
aI881
aI465
aI3371
aI5603
aI371
aI29393
asS'd'
I2588
sa(dp5176
S'a'
I6530
sS's'
I15
sS'm'
(lp5177
I1
aI3
aI1
aI15
aI7
aI47
aI41
aI245
aI377
aI823
aI309
aI3929
aI2159
aI13917
aI13365
asS'd'
I2589
sa(dp5178
S'a'
I6539
sS's'
I15
sS'm'
(lp5179
I1
aI3
aI7
aI7
aI7
aI29
aI25
aI141
aI19
aI611
aI79
aI2689
aI109
aI12321
aI8345
asS'd'
I2590
sa(dp5180
S'a'
I6547
sS's'
I15
sS'm'
(lp5181
I1
aI1
aI1
aI13
aI3
aI53
aI113
aI151
aI381
aI791
aI137
aI3185
aI3567
aI211
aI597
asS'd'
I2591
sa(dp5182
S'a'
I6589
sS's'
I15
sS'm'
(lp5183
I1
aI1
aI3
aI9
aI7
aI53
aI87
aI89
aI491
aI861
aI467
aI3763
aI2025
aI4187
aI9637
asS'd'
I2592
sa(dp5184
S'a'
I6592
sS's'
I15
sS'm'
(lp5185
I1
aI1
aI7
aI1
aI27
aI33
aI71
aI41
aI63
aI1011
aI741
aI1135
aI175
aI3739
aI21493
asS'd'
I2593
sa(dp5186
S'a'
I6601
sS's'
I15
sS'm'
(lp5187
I1
aI3
aI3
aI5
aI9
aI19
aI55
aI175
aI325
aI55
aI1193
aI1423
aI2049
aI9633
aI17515
asS'd'
I2594
sa(dp5188
S'a'
I6610
sS's'
I15
sS'm'
(lp5189
I1
aI1
aI3
aI1
aI27
aI55
aI69
aI103
aI401
aI707
aI825
aI399
aI6799
aI13199
aI6295
asS'd'
I2595
sa(dp5190
S'a'
I6616
sS's'
I15
sS'm'
(lp5191
I1
aI3
aI7
aI3
aI19
aI63
aI25
aI151
aI17
aI159
aI1673
aI615
aI6317
aI13261
aI26267
asS'd'
I2596
sa(dp5192
S'a'
I6619
sS's'
I15
sS'm'
(lp5193
I1
aI3
aI7
aI9
aI27
aI1
aI77
aI129
aI423
aI647
aI707
aI2579
aI3525
aI6723
aI31615
asS'd'
I2597
sa(dp5194
S'a'
I6626
sS's'
I15
sS'm'
(lp5195
I1
aI3
aI3
aI7
aI7
aI31
aI35
aI241
aI309
aI369
aI895
aI3683
aI4795
aI11319
aI451
asS'd'
I2598
sa(dp5196
S'a'
I6635
sS's'
I15
sS'm'
(lp5197
I1
aI3
aI5
aI7
aI17
aI7
aI117
aI141
aI267
aI713
aI569
aI1915
aI4369
aI7793
aI30853
asS'd'
I2599
sa(dp5198
S'a'
I6637
sS's'
I15
sS'm'
(lp5199
I1
aI3
aI7
aI1
aI29
aI61
aI81
aI73
aI413
aI13
aI1977
aI3229
aI5853
aI8451
aI15539
asS'd'
I2600
sa(dp5200
S'a'
I6638
sS's'
I15
sS'm'
(lp5201
I1
aI3
aI7
aI1
aI5
aI45
aI109
aI21
aI431
aI487
aI2019
aI2647
aI927
aI16015
aI10711
asS'd'
I2601
sa(dp5202
S'a'
I6652
sS's'
I15
sS'm'
(lp5203
I1
aI3
aI1
aI3
aI11
aI19
aI37
aI183
aI451
aI377
aI269
aI3993
aI3229
aI4899
aI26561
asS'd'
I2602
sa(dp5204
S'a'
I6656
sS's'
I15
sS'm'
(lp5205
I1
aI3
aI1
aI11
aI5
aI19
aI121
aI55
aI57
aI117
aI687
aI83
aI3047
aI1367
aI17595
asS'd'
I2603
sa(dp5206
S'a'
I6662
sS's'
I15
sS'm'
(lp5207
I1
aI3
aI1
aI7
aI17
aI31
aI41
aI219
aI239
aI963
aI199
aI2895
aI5599
aI7639
aI17201
asS'd'
I2604
sa(dp5208
S'a'
I6689
sS's'
I15
sS'm'
(lp5209
I1
aI3
aI3
aI5
aI27
aI53
aI71
aI183
aI509
aI771
aI1809
aI1539
aI2229
aI4893
aI17115
asS'd'
I2605
sa(dp5210
S'a'
I6699
sS's'
I15
sS'm'
(lp5211
I1
aI1
aI3
aI9
aI9
aI9
aI13
aI49
aI265
aI643
aI1929
aI859
aI497
aI9797
aI27771
asS'd'
I2606
sa(dp5212
S'a'
I6710
sS's'
I15
sS'm'
(lp5213
I1
aI3
aI7
aI11
aI19
aI39
aI115
aI139
aI207
aI903
aI963
aI1849
aI4403
aI6229
aI10021
asS'd'
I2607
sa(dp5214
S'a'
I6714
sS's'
I15
sS'm'
(lp5215
I1
aI3
aI7
aI13
aI3
aI57
aI99
aI223
aI503
aI423
aI1755
aI807
aI1885
aI213
aI18723
asS'd'
I2608
sa(dp5216
S'a'
I6719
sS's'
I15
sS'm'
(lp5217
I1
aI3
aI7
aI15
aI11
aI15
aI111
aI193
aI243
aI599
aI593
aI3385
aI5393
aI15073
aI17777
asS'd'
I2609
sa(dp5218
S'a'
I6739
sS's'
I15
sS'm'
(lp5219
I1
aI1
aI5
aI3
aI19
aI63
aI121
aI207
aI99
aI435
aI1961
aI2747
aI6405
aI3971
aI23481
asS'd'
I2610
sa(dp5220
S'a'
I6751
sS's'
I15
sS'm'
(lp5221
I1
aI3
aI5
aI13
aI9
aI29
aI79
aI131
aI415
aI49
aI229
aI1003
aI3263
aI12975
aI15987
asS'd'
I2611
sa(dp5222
S'a'
I6775
sS's'
I15
sS'm'
(lp5223
I1
aI1
aI3
aI7
aI1
aI41
aI127
aI155
aI29
aI73
aI963
aI659
aI2741
aI3465
aI2595
asS'd'
I2612
sa(dp5224
S'a'
I6779
sS's'
I15
sS'm'
(lp5225
I1
aI1
aI3
aI5
aI23
aI23
aI93
aI233
aI113
aI521
aI427
aI1557
aI6917
aI12953
aI22441
asS'd'
I2613
sa(dp5226
S'a'
I6788
sS's'
I15
sS'm'
(lp5227
I1
aI1
aI5
aI13
aI5
aI25
aI85
aI191
aI387
aI69
aI955
aI243
aI4473
aI9813
aI21711
asS'd'
I2614
sa(dp5228
S'a'
I6798
sS's'
I15
sS'm'
(lp5229
I1
aI3
aI3
aI7
aI1
aI53
aI95
aI65
aI231
aI995
aI539
aI2103
aI5513
aI14087
aI28655
asS'd'
I2615
sa(dp5230
S'a'
I6815
sS's'
I15
sS'm'
(lp5231
I1
aI3
aI5
aI3
aI17
aI13
aI19
aI227
aI197
aI91
aI1437
aI1121
aI3307
aI6903
aI3297
asS'd'
I2616
sa(dp5232
S'a'
I6819
sS's'
I15
sS'm'
(lp5233
I1
aI1
aI5
aI11
aI31
aI29
aI109
aI171
aI257
aI783
aI861
aI9
aI4895
aI1859
aI10909
asS'd'
I2617
sa(dp5234
S'a'
I6825
sS's'
I15
sS'm'
(lp5235
I1
aI1
aI7
aI13
aI5
aI47
aI61
aI5
aI363
aI351
aI1525
aI823
aI2883
aI12435
aI17629
asS'd'
I2618
sa(dp5236
S'a'
I6826
sS's'
I15
sS'm'
(lp5237
I1
aI1
aI5
aI11
aI9
aI3
aI69
aI159
aI371
aI477
aI1223
aI1973
aI2757
aI413
aI31223
asS'd'
I2619
sa(dp5238
S'a'
I6836
sS's'
I15
sS'm'
(lp5239
I1
aI1
aI3
aI5
aI23
aI45
aI43
aI195
aI423
aI829
aI1673
aI1563
aI6633
aI14775
aI21097
asS'd'
I2620
sa(dp5240
S'a'
I6843
sS's'
I15
sS'm'
(lp5241
I1
aI1
aI3
aI3
aI13
aI9
aI107
aI209
aI49
aI609
aI1047
aI3691
aI7483
aI4269
aI7557
asS'd'
I2621
sa(dp5242
S'a'
I6845
sS's'
I15
sS'm'
(lp5243
I1
aI1
aI3
aI15
aI3
aI43
aI73
aI161
aI53
aI813
aI325
aI3439
aI7009
aI8691
aI11711
asS'd'
I2622
sa(dp5244
S'a'
I6858
sS's'
I15
sS'm'
(lp5245
I1
aI1
aI3
aI3
aI23
aI45
aI99
aI61
aI407
aI15
aI1515
aI1557
aI953
aI8567
aI13729
asS'd'
I2623
sa(dp5246
S'a'
I6868
sS's'
I15
sS'm'
(lp5247
I1
aI1
aI5
aI9
aI31
aI35
aI117
aI57
aI227
aI923
aI1373
aI1811
aI3405
aI11979
aI10149
asS'd'
I2624
sa(dp5248
S'a'
I6877
sS's'
I15
sS'm'
(lp5249
I1
aI1
aI3
aI9
aI15
aI53
aI105
aI209
aI153
aI67
aI1477
aI667
aI3077
aI4911
aI3871
asS'd'
I2625
sa(dp5250
S'a'
I6881
sS's'
I15
sS'm'
(lp5251
I1
aI1
aI3
aI3
aI21
aI53
aI93
aI101
aI183
aI1023
aI3
aI3041
aI5815
aI9043
aI5801
asS'd'
I2626
sa(dp5252
S'a'
I6891
sS's'
I15
sS'm'
(lp5253
I1
aI3
aI3
aI5
aI17
aI49
aI127
aI161
aI321
aI869
aI1369
aI923
aI3235
aI711
aI30007
asS'd'
I2627
sa(dp5254
S'a'
I6896
sS's'
I15
sS'm'
(lp5255
I1
aI1
aI3
aI3
aI15
aI17
aI97
aI229
aI389
aI159
aI1075
aI2001
aI7905
aI15191
aI14693
asS'd'
I2628
sa(dp5256
S'a'
I6899
sS's'
I15
sS'm'
(lp5257
I1
aI1
aI5
aI11
aI5
aI5
aI121
aI173
aI95
aI173
aI1883
aI3915
aI1439
aI9981
aI24375
asS'd'
I2629
sa(dp5258
S'a'
I6901
sS's'
I15
sS'm'
(lp5259
I1
aI3
aI3
aI1
aI31
aI53
aI29
aI189
aI37
aI623
aI217
aI949
aI3959
aI7189
aI25427
asS'd'
I2630
sa(dp5260
S'a'
I6908
sS's'
I15
sS'm'
(lp5261
I1
aI3
aI5
aI9
aI21
aI45
aI101
aI23
aI355
aI729
aI797
aI2317
aI2931
aI7433
aI29175
asS'd'
I2631
sa(dp5262
S'a'
I6914
sS's'
I15
sS'm'
(lp5263
I1
aI3
aI7
aI1
aI1
aI63
aI63
aI155
aI237
aI865
aI1169
aI43
aI7335
aI6445
aI7979
asS'd'
I2632
sa(dp5264
S'a'
I6916
sS's'
I15
sS'm'
(lp5265
I1
aI3
aI7
aI7
aI11
aI51
aI37
aI199
aI503
aI991
aI319
aI3013
aI7885
aI12837
aI32419
asS'd'
I2633
sa(dp5266
S'a'
I6923
sS's'
I15
sS'm'
(lp5267
I1
aI3
aI7
aI7
aI27
aI31
aI101
aI243
aI37
aI811
aI1909
aI109
aI6455
aI7903
aI11821
asS'd'
I2634
sa(dp5268
S'a'
I6925
sS's'
I15
sS'm'
(lp5269
I1
aI1
aI3
aI13
aI23
aI21
aI89
aI99
aI243
aI605
aI1017
aI1871
aI1101
aI12825
aI8227
asS'd'
I2635
sa(dp5270
S'a'
I6928
sS's'
I15
sS'm'
(lp5271
I1
aI3
aI3
aI13
aI19
aI3
aI51
aI59
aI501
aI605
aI385
aI2189
aI3229
aI7981
aI31407
asS'd'
I2636
sa(dp5272
S'a'
I6931
sS's'
I15
sS'm'
(lp5273
I1
aI1
aI1
aI1
aI25
aI11
aI127
aI215
aI295
aI237
aI1245
aI3657
aI7803
aI3897
aI655
asS'd'
I2637
sa(dp5274
S'a'
I6934
sS's'
I15
sS'm'
(lp5275
I1
aI1
aI7
aI7
aI5
aI9
aI63
aI129
aI143
aI417
aI795
aI3409
aI2847
aI5887
aI3093
asS'd'
I2638
sa(dp5276
S'a'
I6937
sS's'
I15
sS'm'
(lp5277
I1
aI3
aI3
aI13
aI7
aI57
aI67
aI57
aI5
aI847
aI1185
aI3349
aI4841
aI11457
aI8857
asS'd'
I2639
sa(dp5278
S'a'
I6938
sS's'
I15
sS'm'
(lp5279
I1
aI1
aI3
aI3
aI9
aI53
aI51
aI43
aI85
aI437
aI13
aI2543
aI3651
aI15493
aI767
asS'd'
I2640
sa(dp5280
S'a'
I6949
sS's'
I15
sS'm'
(lp5281
I1
aI1
aI7
aI9
aI1
aI49
aI97
aI115
aI133
aI1011
aI1399
aI2653
aI7765
aI13999
aI12097
asS'd'
I2641
sa(dp5282
S'a'
I6956
sS's'
I15
sS'm'
(lp5283
I1
aI1
aI5
aI1
aI3
aI27
aI123
aI107
aI389
aI401
aI1759
aI1333
aI1371
aI5277
aI14865
asS'd'
I2642
sa(dp5284
S'a'
I6973
sS's'
I15
sS'm'
(lp5285
I1
aI1
aI5
aI1
aI13
aI23
aI3
aI123
aI137
aI821
aI399
aI1671
aI3095
aI3121
aI31387
asS'd'
I2643
sa(dp5286
S'a'
I6976
sS's'
I15
sS'm'
(lp5287
I1
aI1
aI5
aI3
aI7
aI35
aI57
aI237
aI509
aI753
aI1783
aI2815
aI6495
aI13283
aI7091
asS'd'
I2644
sa(dp5288
S'a'
I6981
sS's'
I15
sS'm'
(lp5289
I1
aI1
aI7
aI11
aI5
aI37
aI77
aI109
aI7
aI969
aI1087
aI3705
aI1695
aI14223
aI28959
asS'd'
I2645
sa(dp5290
S'a'
I6988
sS's'
I15
sS'm'
(lp5291
I1
aI3
aI1
aI11
aI25
aI5
aI25
aI163
aI179
aI185
aI671
aI1031
aI4537
aI11601
aI9323
asS'd'
I2646
sa(dp5292
S'a'
I6999
sS's'
I15
sS'm'
(lp5293
I1
aI1
aI3
aI7
aI17
aI25
aI49
aI221
aI183
aI619
aI1953
aI343
aI4523
aI14883
aI6833
asS'd'
I2647
sa(dp5294
S'a'
I7016
sS's'
I15
sS'm'
(lp5295
I1
aI3
aI7
aI5
aI27
aI19
aI59
aI153
aI11
aI807
aI513
aI3019
aI6875
aI5307
aI8405
asS'd'
I2648
sa(dp5296
S'a'
I7027
sS's'
I15
sS'm'
(lp5297
I1
aI1
aI1
aI13
aI25
aI41
aI21
aI109
aI321
aI135
aI497
aI1235
aI5177
aI5167
aI18609
asS'd'
I2649
sa(dp5298
S'a'
I7029
sS's'
I15
sS'm'
(lp5299
I1
aI1
aI7
aI5
aI21
aI53
aI25
aI197
aI411
aI503
aI1009
aI1921
aI4305
aI2633
aI31415
asS'd'
I2650
sa(dp5300
S'a'
I7055
sS's'
I15
sS'm'
(lp5301
I1
aI3
aI5
aI1
aI25
aI45
aI27
aI227
aI271
aI903
aI639
aI3805
aI657
aI8683
aI29585
asS'd'
I2651
sa(dp5302
S'a'
I7058
sS's'
I15
sS'm'
(lp5303
I1
aI1
aI5
aI3
aI9
aI49
aI37
aI35
aI351
aI491
aI851
aI2983
aI31
aI5619
aI6919
asS'd'
I2652
sa(dp5304
S'a'
I7074
sS's'
I15
sS'm'
(lp5305
I1
aI1
aI5
aI3
aI11
aI49
aI33
aI153
aI393
aI1017
aI1561
aI2795
aI4435
aI12589
aI22349
asS'd'
I2653
sa(dp5306
S'a'
I7083
sS's'
I15
sS'm'
(lp5307
I1
aI1
aI1
aI15
aI17
aI29
aI49
aI245
aI217
aI359
aI1133
aI393
aI3317
aI415
aI16407
asS'd'
I2654
sa(dp5308
S'a'
I7093
sS's'
I15
sS'm'
(lp5309
I1
aI1
aI3
aI5
aI3
aI9
aI95
aI63
aI319
aI319
aI1009
aI19
aI6453
aI16279
aI6975
asS'd'
I2655
sa(dp5310
S'a'
I7100
sS's'
I15
sS'm'
(lp5311
I1
aI1
aI5
aI9
aI3
aI25
aI67
aI95
aI369
aI237
aI285
aI2409
aI671
aI5143
aI121
asS'd'
I2656
sa(dp5312
S'a'
I7105
sS's'
I15
sS'm'
(lp5313
I1
aI1
aI3
aI1
aI9
aI49
aI35
aI87
aI317
aI185
aI445
aI2263
aI7923
aI10183
aI26615
asS'd'
I2657
sa(dp5314
S'a'
I7112
sS's'
I15
sS'm'
(lp5315
I1
aI3
aI3
aI11
aI9
aI59
aI29
aI135
aI129
aI477
aI353
aI3571
aI1057
aI16329
aI23523
asS'd'
I2658
sa(dp5316
S'a'
I7118
sS's'
I15
sS'm'
(lp5317
I1
aI1
aI1
aI15
aI13
aI11
aI19
aI5
aI133
aI827
aI1799
aI1893
aI1939
aI1101
aI12147
asS'd'
I2659
sa(dp5318
S'a'
I7120
sS's'
I15
sS'm'
(lp5319
I1
aI1
aI3
aI3
aI15
aI49
aI33
aI185
aI511
aI1013
aI41
aI3499
aI6235
aI7643
aI16725
asS'd'
I2660
sa(dp5320
S'a'
I7129
sS's'
I15
sS'm'
(lp5321
I1
aI1
aI5
aI11
aI27
aI45
aI89
aI157
aI63
aI137
aI2047
aI1275
aI4995
aI625
aI6111
asS'd'
I2661
sa(dp5322
S'a'
I7166
sS's'
I15
sS'm'
(lp5323
I1
aI3
aI7
aI11
aI3
aI1
aI121
aI1
aI341
aI33
aI1895
aI3033
aI3929
aI10257
aI21037
asS'd'
I2662
sa(dp5324
S'a'
I7207
sS's'
I15
sS'm'
(lp5325
I1
aI3
aI3
aI11
aI7
aI11
aI117
aI5
aI115
aI287
aI335
aI3415
aI5397
aI15065
aI19121
asS'd'
I2663
sa(dp5326
S'a'
I7216
sS's'
I15
sS'm'
(lp5327
I1
aI3
aI3
aI13
aI21
aI25
aI15
aI125
aI277
aI125
aI801
aI3761
aI2623
aI11333
aI16867
asS'd'
I2664
sa(dp5328
S'a'
I7226
sS's'
I15
sS'm'
(lp5329
I1
aI3
aI5
aI11
aI19
aI33
aI21
aI71
aI499
aI747
aI1515
aI185
aI1759
aI14623
aI895
asS'd'
I2665
sa(dp5330
S'a'
I7234
sS's'
I15
sS'm'
(lp5331
I1
aI3
aI7
aI1
aI29
aI35
aI9
aI203
aI277
aI299
aI1509
aI2017
aI2897
aI14175
aI1643
asS'd'
I2666
sa(dp5332
S'a'
I7236
sS's'
I15
sS'm'
(lp5333
I1
aI3
aI5
aI11
aI7
aI47
aI111
aI197
aI459
aI941
aI1619
aI2119
aI2191
aI11049
aI6811
asS'd'
I2667
sa(dp5334
S'a'
I7246
sS's'
I15
sS'm'
(lp5335
I1
aI1
aI5
aI9
aI7
aI43
aI103
aI115
aI87
aI269
aI1235
aI77
aI5887
aI1611
aI29041
asS'd'
I2668
sa(dp5336
S'a'
I7248
sS's'
I15
sS'm'
(lp5337
I1
aI1
aI5
aI7
aI1
aI61
aI83
aI225
aI179
aI81
aI1145
aI2403
aI1485
aI8967
aI20607
asS'd'
I2669
sa(dp5338
S'a'
I7254
sS's'
I15
sS'm'
(lp5339
I1
aI3
aI3
aI1
aI25
aI47
aI27
aI157
aI359
aI803
aI1683
aI1995
aI6445
aI13113
aI17899
asS'd'
I2670
sa(dp5340
S'a'
I7263
sS's'
I15
sS'm'
(lp5341
I1
aI3
aI1
aI7
aI21
aI37
aI43
aI119
aI245
aI49
aI1581
aI2275
aI3311
aI4087
aI29765
asS'd'
I2671
sa(dp5342
S'a'
I7273
sS's'
I15
sS'm'
(lp5343
I1
aI1
aI3
aI13
aI5
aI33
aI49
aI191
aI455
aI105
aI665
aI3855
aI3207
aI2671
aI32203
asS'd'
I2672
sa(dp5344
S'a'
I7274
sS's'
I15
sS'm'
(lp5345
I1
aI3
aI1
aI1
aI25
aI63
aI19
aI217
aI17
aI353
aI947
aI1951
aI4097
aI9041
aI11921
asS'd'
I2673
sa(dp5346
S'a'
I7293
sS's'
I15
sS'm'
(lp5347
I1
aI3
aI1
aI7
aI21
aI31
aI113
aI97
aI347
aI993
aI1799
aI3831
aI3711
aI6193
aI1235
asS'd'
I2674
sa(dp5348
S'a'
I7297
sS's'
I15
sS'm'
(lp5349
I1
aI1
aI1
aI5
aI3
aI63
aI11
aI203
aI425
aI445
aI1361
aI531
aI1265
aI1755
aI11685
asS'd'
I2675
sa(dp5350
S'a'
I7310
sS's'
I15
sS'm'
(lp5351
I1
aI3
aI1
aI7
aI13
aI29
aI23
aI85
aI57
aI467
aI1835
aI133
aI7961
aI4175
aI2445
asS'd'
I2676
sa(dp5352
S'a'
I7315
sS's'
I15
sS'm'
(lp5353
I1
aI1
aI1
aI15
aI23
aI27
aI37
aI5
aI123
aI913
aI1293
aI1633
aI3113
aI5413
aI26407
asS'd'
I2677
sa(dp5354
S'a'
I7317
sS's'
I15
sS'm'
(lp5355
I1
aI1
aI5
aI13
aI27
aI1
aI121
aI151
aI303
aI931
aI375
aI3679
aI1863
aI12301
aI30907
asS'd'
I2678
sa(dp5356
S'a'
I7331
sS's'
I15
sS'm'
(lp5357
I1
aI3
aI1
aI9
aI31
aI9
aI49
aI203
aI177
aI937
aI1503
aI933
aI5867
aI12533
aI13621
asS'd'
I2679
sa(dp5358
S'a'
I7338
sS's'
I15
sS'm'
(lp5359
I1
aI3
aI3
aI15
aI1
aI41
aI23
aI191
aI191
aI931
aI837
aI3553
aI2611
aI4735
aI18105
asS'd'
I2680
sa(dp5360
S'a'
I7340
sS's'
I15
sS'm'
(lp5361
I1
aI1
aI5
aI7
aI27
aI49
aI51
aI111
aI435
aI195
aI1229
aI711
aI7145
aI14571
aI31707
asS'd'
I2681
sa(dp5362
S'a'
I7346
sS's'
I15
sS'm'
(lp5363
I1
aI1
aI7
aI7
aI3
aI41
aI59
aI203
aI291
aI903
aI1727
aI2757
aI1463
aI6287
aI31535
asS'd'
I2682
sa(dp5364
S'a'
I7355
sS's'
I15
sS'm'
(lp5365
I1
aI1
aI7
aI13
aI23
aI5
aI75
aI3
aI207
aI525
aI411
aI2133
aI2231
aI477
aI7155
asS'd'
I2683
sa(dp5366
S'a'
I7366
sS's'
I15
sS'm'
(lp5367
I1
aI3
aI5
aI7
aI13
aI19
aI111
aI225
aI489
aI83
aI1177
aI4077
aI4617
aI14413
aI7133
asS'd'
I2684
sa(dp5368
S'a'
I7383
sS's'
I15
sS'm'
(lp5369
I1
aI3
aI1
aI7
aI9
aI59
aI3
aI113
aI379
aI803
aI1289
aI3347
aI4127
aI6669
aI14867
asS'd'
I2685
sa(dp5370
S'a'
I7389
sS's'
I15
sS'm'
(lp5371
I1
aI3
aI7
aI3
aI31
aI37
aI87
aI79
aI399
aI749
aI995
aI1611
aI3137
aI12543
aI31955
asS'd'
I2686
sa(dp5372
S'a'
I7393
sS's'
I15
sS'm'
(lp5373
I1
aI1
aI5
aI7
aI21
aI59
aI49
aI45
aI511
aI639
aI1033
aI2169
aI3265
aI15001
aI10745
asS'd'
I2687
sa(dp5374
S'a'
I7396
sS's'
I15
sS'm'
(lp5375
I1
aI1
aI5
aI1
aI25
aI19
aI23
aI203
aI11
aI883
aI1031
aI4087
aI5059
aI11321
aI21675
asS'd'
I2688
sa(dp5376
S'a'
I7400
sS's'
I15
sS'm'
(lp5377
I1
aI3
aI7
aI5
aI11
aI27
aI33
aI205
aI163
aI289
aI501
aI3505
aI1515
aI1895
aI15889
asS'd'
I2689
sa(dp5378
S'a'
I7414
sS's'
I15
sS'm'
(lp5379
I1
aI3
aI1
aI1
aI23
aI7
aI39
aI239
aI29
aI119
aI1499
aI2071
aI6495
aI12107
aI5339
asS'd'
I2690
sa(dp5380
S'a'
I7417
sS's'
I15
sS'm'
(lp5381
I1
aI3
aI1
aI1
aI23
aI29
aI55
aI181
aI327
aI905
aI427
aI1033
aI427
aI3687
aI5367
asS'd'
I2691
sa(dp5382
S'a'
I7426
sS's'
I15
sS'm'
(lp5383
I1
aI3
aI3
aI7
aI21
aI27
aI115
aI127
aI393
aI855
aI1291
aI2121
aI381
aI9995
aI29757
asS'd'
I2692
sa(dp5384
S'a'
I7432
sS's'
I15
sS'm'
(lp5385
I1
aI3
aI5
aI1
aI25
aI13
aI15
aI183
aI269
aI1005
aI1531
aI3451
aI3975
aI9479
aI23695
asS'd'
I2693
sa(dp5386
S'a'
I7452
sS's'
I15
sS'm'
(lp5387
I1
aI3
aI7
aI7
aI19
aI31
aI111
aI97
aI33
aI179
aI1343
aI2069
aI977
aI5043
aI9129
asS'd'
I2694
sa(dp5388
S'a'
I7468
sS's'
I15
sS'm'
(lp5389
I1
aI3
aI1
aI5
aI17
aI57
aI99
aI129
aI379
aI829
aI837
aI1845
aI3613
aI7351
aI19291
asS'd'
I2695
sa(dp5390
S'a'
I7488
sS's'
I15
sS'm'
(lp5391
I1
aI3
aI3
aI5
aI31
aI23
aI119
aI229
aI135
aI389
aI9
aI705
aI6697
aI15441
aI5303
asS'd'
I2696
sa(dp5392
S'a'
I7491
sS's'
I15
sS'm'
(lp5393
I1
aI1
aI1
aI11
aI25
aI31
aI105
aI95
aI5
aI931
aI789
aI375
aI7543
aI9957
aI28627
asS'd'
I2697
sa(dp5394
S'a'
I7494
sS's'
I15
sS'm'
(lp5395
I1
aI1
aI7
aI15
aI21
aI17
aI19
aI103
aI389
aI545
aI1725
aI2867
aI4251
aI3829
aI6907
asS'd'
I2698
sa(dp5396
S'a'
I7497
sS's'
I15
sS'm'
(lp5397
I1
aI3
aI7
aI7
aI15
aI37
aI97
aI65
aI337
aI409
aI1649
aI2869
aI7929
aI8905
aI21989
asS'd'
I2699
sa(dp5398
S'a'
I7515
sS's'
I15
sS'm'
(lp5399
I1
aI3
aI5
aI3
aI11
aI15
aI69
aI29
aI353
aI207
aI233
aI411
aI2047
aI10303
aI31655
asS'd'
I2700
sa(dp5400
S'a'
I7531
sS's'
I15
sS'm'
(lp5401
I1
aI3
aI3
aI7
aI27
aI43
aI125
aI107
aI69
aI981
aI215
aI1955
aI3589
aI597
aI12703
asS'd'
I2701
sa(dp5402
S'a'
I7552
sS's'
I15
sS'm'
(lp5403
I1
aI1
aI7
aI9
aI25
aI13
aI109
aI73
aI227
aI663
aI1115
aI285
aI471
aI3359
aI15787
asS'd'
I2702
sa(dp5404
S'a'
I7562
sS's'
I15
sS'm'
(lp5405
I1
aI3
aI7
aI5
aI1
aI45
aI7
aI79
aI441
aI149
aI701
aI1457
aI6595
aI14829
aI20865
asS'd'
I2703
sa(dp5406
S'a'
I7564
sS's'
I15
sS'm'
(lp5407
I1
aI3
aI7
aI15
aI15
aI47
aI83
aI239
aI295
aI23
aI1085
aI813
aI1209
aI3573
aI2855
asS'd'
I2704
sa(dp5408
S'a'
I7569
sS's'
I15
sS'm'
(lp5409
I1
aI1
aI3
aI15
aI13
aI7
aI59
aI67
aI255
aI537
aI1841
aI3857
aI6821
aI15175
aI13997
asS'd'
I2705
sa(dp5410
S'a'
I7582
sS's'
I15
sS'm'
(lp5411
I1
aI3
aI1
aI1
aI9
aI57
aI59
aI21
aI21
aI41
aI1693
aI2805
aI7953
aI1389
aI14105
asS'd'
I2706
sa(dp5412
S'a'
I7585
sS's'
I15
sS'm'
(lp5413
I1
aI3
aI5
aI15
aI19
aI49
aI107
aI117
aI99
aI607
aI145
aI53
aI1863
aI9383
aI12029
asS'd'
I2707
sa(dp5414
S'a'
I7588
sS's'
I15
sS'm'
(lp5415
I1
aI3
aI3
aI13
aI1
aI39
aI5
aI141
aI503
aI265
aI281
aI1785
aI2673
aI6597
aI6333
asS'd'
I2708
sa(dp5416
S'a'
I7592
sS's'
I15
sS'm'
(lp5417
I1
aI1
aI5
aI3
aI3
aI19
aI3
aI181
aI169
aI269
aI955
aI2399
aI3157
aI11053
aI8563
asS'd'
I2709
sa(dp5418
S'a'
I7597
sS's'
I15
sS'm'
(lp5419
I1
aI3
aI3
aI13
aI11
aI1
aI95
aI43
aI179
aI507
aI443
aI209
aI3239
aI14239
aI21829
asS'd'
I2710
sa(dp5420
S'a'
I7603
sS's'
I15
sS'm'
(lp5421
I1
aI1
aI7
aI9
aI3
aI17
aI99
aI179
aI445
aI479
aI1897
aI1507
aI5753
aI4757
aI2135
asS'd'
I2711
sa(dp5422
S'a'
I7610
sS's'
I15
sS'm'
(lp5423
I1
aI3
aI3
aI1
aI9
aI51
aI29
aI13
aI295
aI291
aI927
aI85
aI5707
aI7447
aI32319
asS'd'
I2712
sa(dp5424
S'a'
I7624
sS's'
I15
sS'm'
(lp5425
I1
aI1
aI1
aI3
aI13
aI11
aI21
aI157
aI213
aI327
aI1071
aI591
aI2639
aI15405
aI6617
asS'd'
I2713
sa(dp5426
S'a'
I7642
sS's'
I15
sS'm'
(lp5427
I1
aI3
aI5
aI1
aI7
aI25
aI55
aI47
aI495
aI681
aI727
aI2707
aI2955
aI705
aI7489
asS'd'
I2714
sa(dp5428
S'a'
I7647
sS's'
I15
sS'm'
(lp5429
I1
aI1
aI3
aI9
aI17
aI3
aI73
aI67
aI465
aI367
aI1473
aI3195
aI7825
aI5299
aI1817
asS'd'
I2715
sa(dp5430
S'a'
I7653
sS's'
I15
sS'm'
(lp5431
I1
aI1
aI1
aI1
aI19
aI31
aI77
aI253
aI71
aI599
aI1601
aI871
aI2243
aI6699
aI13013
asS'd'
I2716
sa(dp5432
S'a'
I7654
sS's'
I15
sS'm'
(lp5433
I1
aI1
aI7
aI9
aI21
aI1
aI71
aI115
aI5
aI65
aI767
aI925
aI7901
aI10761
aI19431
asS'd'
I2717
sa(dp5434
S'a'
I7666
sS's'
I15
sS'm'
(lp5435
I1
aI3
aI1
aI7
aI23
aI31
aI31
aI15
aI105
aI391
aI585
aI2995
aI2635
aI10607
aI24951
asS'd'
I2718
sa(dp5436
S'a'
I7668
sS's'
I15
sS'm'
(lp5437
I1
aI3
aI3
aI1
aI19
aI25
aI71
aI211
aI41
aI197
aI787
aI225
aI6781
aI813
aI10117
asS'd'
I2719
sa(dp5438
S'a'
I7684
sS's'
I15
sS'm'
(lp5439
I1
aI3
aI3
aI3
aI17
aI29
aI3
aI153
aI231
aI643
aI1151
aI447
aI3699
aI9625
aI26677
asS'd'
I2720
sa(dp5440
S'a'
I7705
sS's'
I15
sS'm'
(lp5441
I1
aI1
aI5
aI9
aI1
aI25
aI71
aI21
aI395
aI297
aI557
aI3841
aI233
aI1877
aI4569
asS'd'
I2721
sa(dp5442
S'a'
I7732
sS's'
I15
sS'm'
(lp5443
I1
aI1
aI3
aI13
aI1
aI45
aI115
aI61
aI5
aI937
aI173
aI2109
aI2927
aI9599
aI9155
asS'd'
I2722
sa(dp5444
S'a'
I7741
sS's'
I15
sS'm'
(lp5445
I1
aI1
aI3
aI3
aI15
aI21
aI61
aI121
aI253
aI285
aI1083
aI3545
aI5537
aI6773
aI2629
asS'd'
I2723
sa(dp5446
S'a'
I7749
sS's'
I15
sS'm'
(lp5447
I1
aI3
aI3
aI15
aI13
aI63
aI33
aI77
aI49
aI849
aI1795
aI2771
aI5481
aI9833
aI603
asS'd'
I2724
sa(dp5448
S'a'
I7750
sS's'
I15
sS'm'
(lp5449
I1
aI1
aI7
aI5
aI1
aI39
aI113
aI237
aI225
aI1005
aI1687
aI2297
aI3213
aI2605
aI14669
asS'd'
I2725
sa(dp5450
S'a'
I7759
sS's'
I15
sS'm'
(lp5451
I1
aI1
aI3
aI1
aI11
aI1
aI39
aI23
aI67
aI441
aI1235
aI2545
aI3139
aI15901
aI29243
asS'd'
I2726
sa(dp5452
S'a'
I7764
sS's'
I15
sS'm'
(lp5453
I1
aI3
aI1
aI3
aI15
aI49
aI39
aI57
aI311
aI345
aI525
aI223
aI4923
aI6311
aI25275
asS'd'
I2727
sa(dp5454
S'a'
I7777
sS's'
I15
sS'm'
(lp5455
I1
aI1
aI5
aI7
aI9
aI13
aI69
aI11
aI349
aI423
aI1773
aI1055
aI1001
aI9359
aI17025
asS'd'
I2728
sa(dp5456
S'a'
I7790
sS's'
I15
sS'm'
(lp5457
I1
aI1
aI1
aI13
aI15
aI63
aI89
aI207
aI335
aI591
aI1223
aI2701
aI55
aI12471
aI13127
asS'd'
I2729
sa(dp5458
S'a'
I7817
sS's'
I15
sS'm'
(lp5459
I1
aI1
aI3
aI5
aI15
aI19
aI83
aI67
aI407
aI113
aI1961
aI779
aI5803
aI12417
aI21751
asS'd'
I2730
sa(dp5460
S'a'
I7826
sS's'
I15
sS'm'
(lp5461
I1
aI3
aI3
aI1
aI21
aI53
aI81
aI95
aI405
aI427
aI1047
aI2443
aI4153
aI5843
aI22511
asS'd'
I2731
sa(dp5462
S'a'
I7831
sS's'
I15
sS'm'
(lp5463
I1
aI1
aI7
aI7
aI7
aI25
aI115
aI155
aI453
aI537
aI741
aI2379
aI2343
aI16035
aI19587
asS'd'
I2732
sa(dp5464
S'a'
I7859
sS's'
I15
sS'm'
(lp5465
I1
aI3
aI3
aI11
aI27
aI21
aI111
aI121
aI503
aI437
aI803
aI3399
aI5303
aI10163
aI18199
asS'd'
I2733
sa(dp5466
S'a'
I7871
sS's'
I15
sS'm'
(lp5467
I1
aI1
aI5
aI13
aI19
aI27
aI7
aI81
aI259
aI545
aI965
aI743
aI4533
aI8813
aI21253
asS'd'
I2734
sa(dp5468
S'a'
I7873
sS's'
I15
sS'm'
(lp5469
I1
aI1
aI5
aI5
aI1
aI59
aI37
aI11
aI105
aI343
aI75
aI1319
aI6317
aI9593
aI1699
asS'd'
I2735
sa(dp5470
S'a'
I7876
sS's'
I15
sS'm'
(lp5471
I1
aI3
aI1
aI9
aI13
aI9
aI115
aI131
aI387
aI1023
aI253
aI693
aI5191
aI12777
aI10565
asS'd'
I2736
sa(dp5472
S'a'
I7900
sS's'
I15
sS'm'
(lp5473
I1
aI3
aI1
aI15
aI7
aI35
aI111
aI195
aI287
aI305
aI533
aI1901
aI3363
aI10085
aI30791
asS'd'
I2737
sa(dp5474
S'a'
I7904
sS's'
I15
sS'm'
(lp5475
I1
aI1
aI3
aI9
aI27
aI51
aI21
aI77
aI413
aI925
aI717
aI791
aI4147
aI585
aI5649
asS'd'
I2738
sa(dp5476
S'a'
I7913
sS's'
I15
sS'm'
(lp5477
I1
aI3
aI3
aI5
aI25
aI59
aI79
aI249
aI185
aI567
aI71
aI1997
aI7373
aI2327
aI18637
asS'd'
I2739
sa(dp5478
S'a'
I7916
sS's'
I15
sS'm'
(lp5479
I1
aI3
aI3
aI11
aI15
aI21
aI97
aI99
aI391
aI57
aI1973
aI29
aI7451
aI2529
aI25737
asS'd'
I2740
sa(dp5480
S'a'
I7922
sS's'
I15
sS'm'
(lp5481
I1
aI3
aI7
aI5
aI7
aI59
aI93
aI5
aI287
aI469
aI1639
aI3637
aI5465
aI14431
aI32265
asS'd'
I2741
sa(dp5482
S'a'
I7946
sS's'
I15
sS'm'
(lp5483
I1
aI1
aI3
aI11
aI3
aI1
aI71
aI75
aI427
aI299
aI811
aI3697
aI3529
aI5433
aI26957
asS'd'
I2742
sa(dp5484
S'a'
I7953
sS's'
I15
sS'm'
(lp5485
I1
aI3
aI1
aI9
aI19
aI59
aI37
aI255
aI165
aI1005
aI19
aI2851
aI4309
aI455
aI9485
asS'd'
I2743
sa(dp5486
S'a'
I7956
sS's'
I15
sS'm'
(lp5487
I1
aI1
aI1
aI5
aI1
aI55
aI15
aI233
aI133
aI47
aI1831
aI713
aI2601
aI1017
aI3201
asS'd'
I2744
sa(dp5488
S'a'
I7963
sS's'
I15
sS'm'
(lp5489
I1
aI1
aI5
aI5
aI21
aI55
aI127
aI69
aI377
aI41
aI25
aI2295
aI7595
aI4733
aI11615
asS'd'
I2745
sa(dp5490
S'a'
I7979
sS's'
I15
sS'm'
(lp5491
I1
aI1
aI5
aI3
aI23
aI5
aI7
aI181
aI161
aI775
aI1095
aI2271
aI6637
aI14489
aI6873
asS'd'
I2746
sa(dp5492
S'a'
I7981
sS's'
I15
sS'm'
(lp5493
I1
aI3
aI5
aI9
aI9
aI15
aI5
aI133
aI357
aI21
aI127
aI2685
aI6299
aI4363
aI17573
asS'd'
I2747
sa(dp5494
S'a'
I7984
sS's'
I15
sS'm'
(lp5495
I1
aI3
aI3
aI9
aI13
aI39
aI51
aI223
aI201
aI401
aI1839
aI2461
aI7633
aI6039
aI10445
asS'd'
I2748
sa(dp5496
S'a'
I7989
sS's'
I15
sS'm'
(lp5497
I1
aI1
aI5
aI1
aI9
aI21
aI19
aI249
aI227
aI359
aI255
aI2895
aI4117
aI2073
aI27687
asS'd'
I2749
sa(dp5498
S'a'
I7999
sS's'
I15
sS'm'
(lp5499
I1
aI1
aI5
aI15
aI5
aI61
aI113
aI161
aI95
aI3
aI877
aI2775
aI293
aI6655
aI4023
asS'd'
I2750
sa(dp5500
S'a'
I8001
sS's'
I15
sS'm'
(lp5501
I1
aI3
aI7
aI1
aI7
aI55
aI73
aI39
aI295
aI403
aI985
aI2315
aI1667
aI13525
aI1453
asS'd'
I2751
sa(dp5502
S'a'
I8021
sS's'
I15
sS'm'
(lp5503
I1
aI1
aI5
aI1
aI27
aI1
aI85
aI195
aI11
aI713
aI1841
aI3895
aI3131
aI2193
aI17607
asS'd'
I2752
sa(dp5504
S'a'
I8056
sS's'
I15
sS'm'
(lp5505
I1
aI3
aI5
aI13
aI25
aI1
aI119
aI97
aI239
aI167
aI1393
aI1753
aI6989
aI12155
aI12509
asS'd'
I2753
sa(dp5506
S'a'
I8080
sS's'
I15
sS'm'
(lp5507
I1
aI1
aI7
aI15
aI31
aI21
aI41
aI255
aI425
aI445
aI165
aI2097
aI5627
aI4971
aI13207
asS'd'
I2754
sa(dp5508
S'a'
I8083
sS's'
I15
sS'm'
(lp5509
I1
aI1
aI1
aI15
aI13
aI33
aI81
aI105
aI453
aI197
aI13
aI1547
aI7381
aI8709
aI15103
asS'd'
I2755
sa(dp5510
S'a'
I8089
sS's'
I15
sS'm'
(lp5511
I1
aI1
aI3
aI11
aI11
aI33
aI107
aI123
aI483
aI367
aI121
aI995
aI1911
aI8205
aI22577
asS'd'
I2756
sa(dp5512
S'a'
I8090
sS's'
I15
sS'm'
(lp5513
I1
aI1
aI1
aI9
aI9
aI43
aI71
aI49
aI273
aI431
aI1705
aI3313
aI4259
aI16291
aI14345
asS'd'
I2757
sa(dp5514
S'a'
I8114
sS's'
I15
sS'm'
(lp5515
I1
aI1
aI1
aI7
aI3
aI1
aI43
aI213
aI97
aI547
aI1559
aI1149
aI2791
aI3751
aI887
asS'd'
I2758
sa(dp5516
S'a'
I8128
sS's'
I15
sS'm'
(lp5517
I1
aI1
aI3
aI15
aI25
aI47
aI49
aI251
aI425
aI35
aI295
aI3767
aI6305
aI9633
aI5045
asS'd'
I2759
sa(dp5518
S'a'
I8133
sS's'
I15
sS'm'
(lp5519
I1
aI3
aI3
aI1
aI5
aI55
aI91
aI245
aI27
aI981
aI331
aI555
aI6553
aI11017
aI15289
asS'd'
I2760
sa(dp5520
S'a'
I8145
sS's'
I15
sS'm'
(lp5521
I1
aI1
aI3
aI7
aI1
aI23
aI23
aI155
aI223
aI565
aI1005
aI3211
aI3847
aI7479
aI3643
asS'd'
I2761
sa(dp5522
S'a'
I8155
sS's'
I15
sS'm'
(lp5523
I1
aI1
aI5
aI1
aI17
aI7
aI47
aI95
aI35
aI779
aI1685
aI2099
aI7505
aI15425
aI18089
asS'd'
I2762
sa(dp5524
S'a'
I8161
sS's'
I15
sS'm'
(lp5525
I1
aI3
aI3
aI7
aI3
aI63
aI83
aI151
aI211
aI147
aI611
aI1171
aI1681
aI7687
aI13423
asS'd'
I2763
sa(dp5526
S'a'
I8182
sS's'
I15
sS'm'
(lp5527
I1
aI3
aI3
aI1
aI3
aI27
aI107
aI117
aI497
aI537
aI195
aI3075
aI2753
aI1665
aI19399
asS'd'
I2764
sa(dp5528
S'a'
I8186
sS's'
I15
sS'm'
(lp5529
I1
aI1
aI1
aI7
aI23
aI5
aI103
aI209
aI117
aI845
aI1243
aI1283
aI4253
aI9723
aI20937
asS'd'
I2765
sa(dp5530
S'a'
I8191
sS's'
I15
sS'm'
(lp5531
I1
aI3
aI1
aI1
aI5
aI49
aI7
aI13
aI419
aI125
aI287
aI1599
aI8161
aI1275
aI24661
asS'd'
I2766
sa(dp5532
S'a'
I8192
sS's'
I15
sS'm'
(lp5533
I1
aI3
aI3
aI3
aI13
aI63
aI23
aI183
aI39
aI979
aI1301
aI2349
aI905
aI15805
aI30151
asS'd'
I2767
sa(dp5534
S'a'
I8195
sS's'
I15
sS'm'
(lp5535
I1
aI1
aI3
aI9
aI17
aI11
aI97
aI189
aI189
aI511
aI1779
aI2077
aI6891
aI11623
aI23949
asS'd'
I2768
sa(dp5536
S'a'
I8201
sS's'
I15
sS'm'
(lp5537
I1
aI1
aI7
aI11
aI13
aI45
aI15
aI37
aI11
aI853
aI915
aI1569
aI6103
aI10633
aI3137
asS'd'
I2769
sa(dp5538
S'a'
I8207
sS's'
I15
sS'm'
(lp5539
I1
aI3
aI3
aI5
aI15
aI61
aI91
aI255
aI131
aI821
aI1755
aI1501
aI2663
aI1747
aI941
asS'd'
I2770
sa(dp5540
S'a'
I8210
sS's'
I15
sS'm'
(lp5541
I1
aI1
aI3
aI7
aI19
aI19
aI65
aI95
aI499
aI239
aI2023
aI3185
aI4649
aI3861
aI3767
asS'd'
I2771
sa(dp5542
S'a'
I8228
sS's'
I15
sS'm'
(lp5543
I1
aI3
aI5
aI15
aI15
aI63
aI55
aI93
aI127
aI303
aI171
aI1763
aI4991
aI9479
aI9917
asS'd'
I2772
sa(dp5544
S'a'
I8249
sS's'
I15
sS'm'
(lp5545
I1
aI3
aI7
aI5
aI31
aI53
aI111
aI35
aI433
aI163
aI1903
aI3991
aI3585
aI643
aI21941
asS'd'
I2773
sa(dp5546
S'a'
I8252
sS's'
I15
sS'm'
(lp5547
I1
aI3
aI1
aI9
aI27
aI39
aI67
aI89
aI487
aI349
aI587
aI1723
aI4311
aI11321
aI25785
asS'd'
I2774
sa(dp5548
S'a'
I8258
sS's'
I15
sS'm'
(lp5549
I1
aI3
aI5
aI7
aI1
aI63
aI23
aI237
aI507
aI689
aI1341
aI441
aI1721
aI843
aI20335
asS'd'
I2775
sa(dp5550
S'a'
I8267
sS's'
I15
sS'm'
(lp5551
I1
aI1
aI3
aI3
aI31
aI63
aI83
aI103
aI25
aI799
aI1379
aI1817
aI3809
aI12285
aI16673
asS'd'
I2776
sa(dp5552
S'a'
I8270
sS's'
I15
sS'm'
(lp5553
I1
aI1
aI5
aI3
aI25
aI29
aI99
aI193
aI21
aI549
aI33
aI3109
aI4135
aI10071
aI32355
asS'd'
I2777
sa(dp5554
S'a'
I8275
sS's'
I15
sS'm'
(lp5555
I1
aI3
aI1
aI7
aI13
aI27
aI83
aI189
aI121
aI167
aI379
aI1503
aI7955
aI13189
aI313
asS'd'
I2778
sa(dp5556
S'a'
I8284
sS's'
I15
sS'm'
(lp5557
I1
aI3
aI5
aI15
aI25
aI19
aI83
aI87
aI257
aI237
aI709
aI1169
aI1561
aI7117
aI4785
asS'd'
I2779
sa(dp5558
S'a'
I8293
sS's'
I15
sS'm'
(lp5559
I1
aI1
aI1
aI7
aI9
aI55
aI21
aI5
aI439
aI367
aI403
aI2311
aI6243
aI8349
aI13127
asS'd'
I2780
sa(dp5560
S'a'
I8298
sS's'
I15
sS'm'
(lp5561
I1
aI3
aI7
aI3
aI5
aI35
aI51
aI67
aI453
aI767
aI29
aI3293
aI6665
aI11459
aI2799
asS'd'
I2781
sa(dp5562
S'a'
I8305
sS's'
I15
sS'm'
(lp5563
I1
aI3
aI3
aI3
aI5
aI19
aI59
aI7
aI367
aI683
aI783
aI1317
aI7119
aI6129
aI19525
asS'd'
I2782
sa(dp5564
S'a'
I8317
sS's'
I15
sS'm'
(lp5565
I1
aI1
aI5
aI5
aI5
aI19
aI61
aI67
aI381
aI291
aI875
aI2179
aI2481
aI9325
aI11253
asS'd'
I2783
sa(dp5566
S'a'
I8328
sS's'
I15
sS'm'
(lp5567
I1
aI3
aI5
aI5
aI7
aI47
aI107
aI9
aI141
aI667
aI1989
aI821
aI3909
aI1733
aI10187
asS'd'
I2784
sa(dp5568
S'a'
I8336
sS's'
I15
sS'm'
(lp5569
I1
aI1
aI7
aI7
aI31
aI61
aI1
aI71
aI477
aI689
aI1539
aI3617
aI8105
aI6535
aI3293
asS'd'
I2785
sa(dp5570
S'a'
I8345
sS's'
I15
sS'm'
(lp5571
I1
aI1
aI5
aI5
aI23
aI9
aI103
aI197
aI241
aI249
aI297
aI3607
aI6217
aI1673
aI30103
asS'd'
I2786
sa(dp5572
S'a'
I8351
sS's'
I15
sS'm'
(lp5573
I1
aI3
aI1
aI5
aI23
aI15
aI115
aI105
aI365
aI51
aI825
aI2687
aI359
aI16325
aI15083
asS'd'
I2787
sa(dp5574
S'a'
I8367
sS's'
I15
sS'm'
(lp5575
I1
aI1
aI3
aI11
aI29
aI45
aI65
aI251
aI169
aI189
aI1243
aI2345
aI1345
aI14471
aI25631
asS'd'
I2788
sa(dp5576
S'a'
I8379
sS's'
I15
sS'm'
(lp5577
I1
aI1
aI5
aI9
aI7
aI63
aI81
aI167
aI309
aI539
aI1169
aI3949
aI4193
aI12047
aI1491
asS'd'
I2789
sa(dp5578
S'a'
I8381
sS's'
I15
sS'm'
(lp5579
I1
aI3
aI1
aI9
aI29
aI33
aI89
aI167
aI67
aI73
aI1885
aI477
aI5745
aI13365
aI6819
asS'd'
I2790
sa(dp5580
S'a'
I8382
sS's'
I15
sS'm'
(lp5581
I1
aI3
aI7
aI9
aI9
aI49
aI95
aI13
aI157
aI997
aI1725
aI935
aI7543
aI6349
aI18277
asS'd'
I2791
sa(dp5582
S'a'
I8393
sS's'
I15
sS'm'
(lp5583
I1
aI1
aI5
aI5
aI11
aI59
aI97
aI17
aI303
aI469
aI93
aI2761
aI7395
aI9021
aI24299
asS'd'
I2792
sa(dp5584
S'a'
I8402
sS's'
I15
sS'm'
(lp5585
I1
aI1
aI7
aI3
aI27
aI63
aI71
aI99
aI407
aI139
aI711
aI2589
aI4715
aI5405
aI3277
asS'd'
I2793
sa(dp5586
S'a'
I8414
sS's'
I15
sS'm'
(lp5587
I1
aI3
aI7
aI3
aI11
aI15
aI49
aI57
aI271
aI493
aI1165
aI2839
aI8191
aI2609
aI14759
asS'd'
I2794
sa(dp5588
S'a'
I8417
sS's'
I15
sS'm'
(lp5589
I1
aI1
aI1
aI7
aI21
aI15
aI71
aI245
aI413
aI473
aI1321
aI1165
aI1027
aI6983
aI12867
asS'd'
I2795
sa(dp5590
S'a'
I8420
sS's'
I15
sS'm'
(lp5591
I1
aI1
aI5
aI3
aI15
aI21
aI19
aI197
aI401
aI627
aI2047
aI2761
aI5807
aI5751
aI28025
asS'd'
I2796
sa(dp5592
S'a'
I8429
sS's'
I15
sS'm'
(lp5593
I1
aI1
aI3
aI3
aI5
aI57
aI19
aI209
aI341
aI165
aI489
aI455
aI231
aI14385
aI12457
asS'd'
I2797
sa(dp5594
S'a'
I8435
sS's'
I15
sS'm'
(lp5595
I1
aI3
aI3
aI11
aI13
aI63
aI79
aI129
aI17
aI315
aI1881
aI1069
aI177
aI12013
aI29567
asS'd'
I2798
sa(dp5596
S'a'
I8438
sS's'
I15
sS'm'
(lp5597
I1
aI1
aI3
aI7
aI31
aI29
aI51
aI235
aI475
aI375
aI617
aI437
aI6379
aI8505
aI23079
asS'd'
I2799
sa(dp5598
S'a'
I8450
sS's'
I15
sS'm'
(lp5599
I1
aI1
aI3
aI7
aI27
aI3
aI3
aI137
aI203
aI959
aI363
aI371
aI2899
aI13491
aI22979
asS'd'
I2800
sa(dp5600
S'a'
I8452
sS's'
I15
sS'm'
(lp5601
I1
aI3
aI3
aI3
aI9
aI1
aI57
aI7
aI363
aI537
aI713
aI2417
aI509
aI7747
aI22135
asS'd'
I2801
sa(dp5602
S'a'
I8459
sS's'
I15
sS'm'
(lp5603
I1
aI3
aI3
aI3
aI13
aI21
aI79
aI121
aI487
aI921
aI113
aI281
aI2853
aI14855
aI19747
asS'd'
I2802
sa(dp5604
S'a'
I8470
sS's'
I15
sS'm'
(lp5605
I1
aI1
aI1
aI11
aI3
aI53
aI89
aI123
aI307
aI585
aI567
aI1925
aI505
aI15935
aI20419
asS'd'
I2803
sa(dp5606
S'a'
I8486
sS's'
I15
sS'm'
(lp5607
I1
aI1
aI3
aI3
aI15
aI45
aI77
aI197
aI499
aI683
aI1405
aI3573
aI981
aI14135
aI19763
asS'd'
I2804
sa(dp5608
S'a'
I8490
sS's'
I15
sS'm'
(lp5609
I1
aI1
aI1
aI11
aI27
aI31
aI61
aI191
aI29
aI601
aI373
aI2011
aI6193
aI3599
aI4387
asS'd'
I2805
sa(dp5610
S'a'
I8500
sS's'
I15
sS'm'
(lp5611
I1
aI3
aI5
aI9
aI7
aI13
aI1
aI193
aI469
aI603
aI1315
aI3329
aI3761
aI8355
aI10425
asS'd'
I2806
sa(dp5612
S'a'
I8524
sS's'
I15
sS'm'
(lp5613
I1
aI1
aI3
aI9
aI29
aI61
aI103
aI17
aI117
aI251
aI2029
aI2963
aI3763
aI16117
aI6627
asS'd'
I2807
sa(dp5614
S'a'
I8536
sS's'
I15
sS'm'
(lp5615
I1
aI3
aI1
aI3
aI7
aI51
aI91
aI145
aI497
aI657
aI871
aI3707
aI5905
aI10449
aI14901
asS'd'
I2808
sa(dp5616
S'a'
I8552
sS's'
I15
sS'm'
(lp5617
I1
aI1
aI3
aI1
aI3
aI53
aI23
aI149
aI461
aI333
aI1809
aI1315
aI1815
aI8223
aI13297
asS'd'
I2809
sa(dp5618
S'a'
I8558
sS's'
I15
sS'm'
(lp5619
I1
aI1
aI1
aI7
aI15
aI31
aI3
aI47
aI443
aI829
aI1305
aI893
aI4191
aI9681
aI32661
asS'd'
I2810
sa(dp5620
S'a'
I8570
sS's'
I15
sS'm'
(lp5621
I1
aI3
aI1
aI3
aI27
aI43
aI51
aI221
aI295
aI825
aI649
aI2953
aI6203
aI8237
aI20253
asS'd'
I2811
sa(dp5622
S'a'
I8576
sS's'
I15
sS'm'
(lp5623
I1
aI3
aI1
aI3
aI9
aI35
aI41
aI195
aI249
aI225
aI387
aI3789
aI1499
aI2559
aI28413
asS'd'
I2812
sa(dp5624
S'a'
I8582
sS's'
I15
sS'm'
(lp5625
I1
aI1
aI5
aI15
aI19
aI29
aI13
aI115
aI333
aI787
aI787
aI723
aI2987
aI6227
aI10865
asS'd'
I2813
sa(dp5626
S'a'
I8594
sS's'
I15
sS'm'
(lp5627
I1
aI3
aI5
aI13
aI5
aI59
aI5
aI251
aI79
aI387
aI11
aI3167
aI6619
aI13317
aI18979
asS'd'
I2814
sa(dp5628
S'a'
I8606
sS's'
I15
sS'm'
(lp5629
I1
aI1
aI7
aI11
aI31
aI51
aI43
aI1
aI189
aI519
aI1945
aI2129
aI4365
aI14059
aI3139
asS'd'
I2815
sa(dp5630
S'a'
I8619
sS's'
I15
sS'm'
(lp5631
I1
aI1
aI7
aI5
aI31
aI9
aI43
aI19
aI151
aI533
aI1061
aI3849
aI6871
aI6941
aI14935
asS'd'
I2816
sa(dp5632
S'a'
I8621
sS's'
I15
sS'm'
(lp5633
I1
aI3
aI7
aI5
aI19
aI57
aI7
aI129
aI25
aI353
aI17
aI1739
aI6513
aI399
aI28835
asS'd'
I2817
sa(dp5634
S'a'
I8624
sS's'
I15
sS'm'
(lp5635
I1
aI3
aI5
aI15
aI25
aI15
aI37
aI125
aI39
aI239
aI271
aI65
aI2189
aI10449
aI11815
asS'd'
I2818
sa(dp5636
S'a'
I8633
sS's'
I15
sS'm'
(lp5637
I1
aI3
aI7
aI15
aI19
aI57
aI47
aI245
aI509
aI945
aI385
aI3987
aI3585
aI14711
aI9655
asS'd'
I2819
sa(dp5638
S'a'
I8641
sS's'
I15
sS'm'
(lp5639
I1
aI1
aI3
aI13
aI21
aI31
aI13
aI81
aI9
aI489
aI1321
aI63
aI1363
aI2219
aI19541
asS'd'
I2820
sa(dp5640
S'a'
I8653
sS's'
I15
sS'm'
(lp5641
I1
aI1
aI5
aI7
aI3
aI57
aI25
aI147
aI23
aI553
aI889
aI307
aI6429
aI15807
aI12861
asS'd'
I2821
sa(dp5642
S'a'
I8654
sS's'
I15
sS'm'
(lp5643
I1
aI1
aI3
aI15
aI29
aI21
aI99
aI237
aI151
aI881
aI675
aI3625
aI1159
aI11759
aI21347
asS'd'
I2822
sa(dp5644
S'a'
I8662
sS's'
I15
sS'm'
(lp5645
I1
aI1
aI7
aI1
aI9
aI13
aI111
aI239
aI235
aI609
aI1569
aI3271
aI2837
aI13807
aI7301
asS'd'
I2823
sa(dp5646
S'a'
I8675
sS's'
I15
sS'm'
(lp5647
I1
aI3
aI1
aI15
aI7
aI59
aI27
aI81
aI129
aI9
aI647
aI3595
aI1877
aI1067
aI1859
asS'd'
I2824
sa(dp5648
S'a'
I8689
sS's'
I15
sS'm'
(lp5649
I1
aI3
aI7
aI1
aI3
aI25
aI119
aI57
aI145
aI441
aI1045
aI789
aI215
aI1265
aI9369
asS'd'
I2825
sa(dp5650
S'a'
I8695
sS's'
I15
sS'm'
(lp5651
I1
aI3
aI7
aI3
aI17
aI25
aI87
aI211
aI441
aI229
aI223
aI2795
aI7241
aI7007
aI20575
asS'd'
I2826
sa(dp5652
S'a'
I8702
sS's'
I15
sS'm'
(lp5653
I1
aI1
aI3
aI1
aI13
aI1
aI55
aI227
aI389
aI141
aI1097
aI2487
aI7603
aI4161
aI5025
asS'd'
I2827
sa(dp5654
S'a'
I8706
sS's'
I15
sS'm'
(lp5655
I1
aI1
aI3
aI5
aI15
aI29
aI29
aI145
aI233
aI209
aI891
aI89
aI8097
aI2897
aI26685
asS'd'
I2828
sa(dp5656
S'a'
I8720
sS's'
I15
sS'm'
(lp5657
I1
aI1
aI3
aI1
aI29
aI53
aI19
aI95
aI161
aI359
aI435
aI3313
aI4955
aI7965
aI21015
asS'd'
I2829
sa(dp5658
S'a'
I8729
sS's'
I15
sS'm'
(lp5659
I1
aI3
aI5
aI9
aI19
aI3
aI109
aI77
aI29
aI937
aI1663
aI125
aI2453
aI1069
aI20639
asS'd'
I2830
sa(dp5660
S'a'
I8739
sS's'
I15
sS'm'
(lp5661
I1
aI3
aI7
aI13
aI5
aI23
aI43
aI231
aI347
aI591
aI1963
aI2491
aI4045
aI16029
aI8149
asS'd'
I2831
sa(dp5662
S'a'
I8741
sS's'
I15
sS'm'
(lp5663
I1
aI1
aI5
aI1
aI13
aI3
aI75
aI211
aI419
aI929
aI901
aI3453
aI8121
aI799
aI8897
asS'd'
I2832
sa(dp5664
S'a'
I8751
sS's'
I15
sS'm'
(lp5665
I1
aI1
aI7
aI15
aI11
aI11
aI123
aI111
aI309
aI415
aI1071
aI975
aI2009
aI12945
aI19617
asS'd'
I2833
sa(dp5666
S'a'
I8759
sS's'
I15
sS'm'
(lp5667
I1
aI1
aI1
aI7
aI31
aI35
aI81
aI255
aI89
aI643
aI451
aI513
aI497
aI11751
aI24215
asS'd'
I2834
sa(dp5668
S'a'
I8766
sS's'
I15
sS'm'
(lp5669
I1
aI3
aI5
aI5
aI25
aI17
aI5
aI165
aI139
aI929
aI1927
aI1353
aI7427
aI9719
aI17087
asS'd'
I2835
sa(dp5670
S'a'
I8777
sS's'
I15
sS'm'
(lp5671
I1
aI3
aI5
aI1
aI21
aI55
aI79
aI85
aI333
aI847
aI1305
aI851
aI5057
aI8361
aI18269
asS'd'
I2836
sa(dp5672
S'a'
I8783
sS's'
I15
sS'm'
(lp5673
I1
aI3
aI7
aI15
aI27
aI17
aI55
aI125
aI395
aI223
aI271
aI781
aI1639
aI10569
aI11143
asS'd'
I2837
sa(dp5674
S'a'
I8786
sS's'
I15
sS'm'
(lp5675
I1
aI1
aI7
aI9
aI7
aI33
aI127
aI85
aI209
aI339
aI483
aI241
aI2523
aI14951
aI6855
asS'd'
I2838
sa(dp5676
S'a'
I8795
sS's'
I15
sS'm'
(lp5677
I1
aI1
aI3
aI9
aI5
aI19
aI9
aI183
aI435
aI343
aI1105
aI3139
aI7617
aI1311
aI267
asS'd'
I2839
sa(dp5678
S'a'
I8802
sS's'
I15
sS'm'
(lp5679
I1
aI1
aI5
aI1
aI15
aI53
aI11
aI63
aI113
aI241
aI855
aI3123
aI4777
aI3495
aI23345
asS'd'
I2840
sa(dp5680
S'a'
I8814
sS's'
I15
sS'm'
(lp5681
I1
aI3
aI1
aI5
aI19
aI29
aI119
aI205
aI167
aI683
aI289
aI1629
aI4977
aI8981
aI6867
asS'd'
I2841
sa(dp5682
S'a'
I8821
sS's'
I15
sS'm'
(lp5683
I1
aI3
aI1
aI1
aI31
aI63
aI95
aI159
aI267
aI231
aI863
aI3385
aI5315
aI7267
aI13757
asS'd'
I2842
sa(dp5684
S'a'
I8828
sS's'
I15
sS'm'
(lp5685
I1
aI3
aI5
aI11
aI19
aI21
aI53
aI41
aI125
aI179
aI533
aI1279
aI3759
aI7073
aI13905
asS'd'
I2843
sa(dp5686
S'a'
I8831
sS's'
I15
sS'm'
(lp5687
I1
aI3
aI5
aI9
aI17
aI7
aI27
aI67
aI97
aI809
aI1423
aI2743
aI2859
aI16121
aI329
asS'd'
I2844
sa(dp5688
S'a'
I8837
sS's'
I15
sS'm'
(lp5689
I1
aI3
aI1
aI15
aI1
aI41
aI59
aI155
aI509
aI51
aI1827
aI3739
aI3879
aI13369
aI30821
asS'd'
I2845
sa(dp5690
S'a'
I8842
sS's'
I15
sS'm'
(lp5691
I1
aI3
aI3
aI7
aI21
aI31
aI7
aI13
aI347
aI919
aI1225
aI497
aI5051
aI3769
aI20211
asS'd'
I2846
sa(dp5692
S'a'
I8855
sS's'
I15
sS'm'
(lp5693
I1
aI3
aI7
aI13
aI31
aI9
aI127
aI195
aI123
aI387
aI3
aI3593
aI6623
aI9827
aI29319
asS'd'
I2847
sa(dp5694
S'a'
I8856
sS's'
I15
sS'm'
(lp5695
I1
aI1
aI3
aI9
aI7
aI27
aI95
aI211
aI287
aI189
aI1683
aI1999
aI7641
aI14983
aI4699
asS'd'
I2848
sa(dp5696
S'a'
I8868
sS's'
I15
sS'm'
(lp5697
I1
aI1
aI5
aI3
aI7
aI21
aI29
aI189
aI101
aI423
aI885
aI3275
aI6569
aI11023
aI22265
asS'd'
I2849
sa(dp5698
S'a'
I8877
sS's'
I15
sS'm'
(lp5699
I1
aI3
aI5
aI3
aI9
aI33
aI79
aI75
aI327
aI975
aI287
aI3025
aI2157
aI7301
aI24447
asS'd'
I2850
sa(dp5700
S'a'
I8890
sS's'
I15
sS'm'
(lp5701
I1
aI3
aI3
aI15
aI31
aI27
aI63
aI1
aI71
aI119
aI1151
aI517
aI6131
aI11055
aI179
asS'd'
I2851
sa(dp5702
S'a'
I8892
sS's'
I15
sS'm'
(lp5703
I1
aI3
aI7
aI11
aI23
aI15
aI101
aI247
aI349
aI735
aI673
aI997
aI6451
aI229
aI32103
asS'd'
I2852
sa(dp5704
S'a'
I8900
sS's'
I15
sS'm'
(lp5705
I1
aI3
aI5
aI15
aI7
aI1
aI51
aI135
aI207
aI741
aI1831
aI1235
aI4747
aI11915
aI22009
asS'd'
I2853
sa(dp5706
S'a'
I8909
sS's'
I15
sS'm'
(lp5707
I1
aI3
aI1
aI13
aI9
aI31
aI19
aI221
aI465
aI681
aI627
aI2595
aI5617
aI14201
aI30355
asS'd'
I2854
sa(dp5708
S'a'
I8912
sS's'
I15
sS'm'
(lp5709
I1
aI1
aI3
aI1
aI13
aI49
aI55
aI155
aI11
aI885
aI1275
aI3591
aI2217
aI6659
aI30885
asS'd'
I2855
sa(dp5710
S'a'
I8921
sS's'
I15
sS'm'
(lp5711
I1
aI1
aI7
aI11
aI27
aI57
aI93
aI95
aI243
aI63
aI1405
aI2049
aI7689
aI15943
aI18503
asS'd'
I2856
sa(dp5712
S'a'
I8922
sS's'
I15
sS'm'
(lp5713
I1
aI1
aI7
aI7
aI5
aI11
aI47
aI189
aI467
aI631
aI1665
aI2717
aI4285
aI2087
aI1435
asS'd'
I2857
sa(dp5714
S'a'
I8927
sS's'
I15
sS'm'
(lp5715
I1
aI1
aI3
aI11
aI7
aI27
aI127
aI3
aI231
aI757
aI435
aI2545
aI3537
aI9127
aI19915
asS'd'
I2858
sa(dp5716
S'a'
I8943
sS's'
I15
sS'm'
(lp5717
I1
aI1
aI5
aI13
aI5
aI29
aI85
aI127
aI339
aI875
aI497
aI1573
aI6553
aI11983
aI18029
asS'd'
I2859
sa(dp5718
S'a'
I8948
sS's'
I15
sS'm'
(lp5719
I1
aI3
aI1
aI1
aI21
aI3
aI15
aI91
aI231
aI683
aI1529
aI2651
aI4147
aI13437
aI23861
asS'd'
I2860
sa(dp5720
S'a'
I8951
sS's'
I15
sS'm'
(lp5721
I1
aI3
aI1
aI7
aI27
aI17
aI19
aI179
aI243
aI223
aI1037
aI1501
aI5935
aI2259
aI25185
asS'd'
I2861
sa(dp5722
S'a'
I8958
sS's'
I15
sS'm'
(lp5723
I1
aI1
aI3
aI15
aI11
aI19
aI127
aI27
aI483
aI219
aI583
aI2555
aI531
aI3451
aI17875
asS'd'
I2862
sa(dp5724
S'a'
I8984
sS's'
I15
sS'm'
(lp5725
I1
aI1
aI1
aI13
aI31
aI39
aI89
aI149
aI363
aI741
aI1355
aI4067
aI3171
aI6783
aI1799
asS'd'
I2863
sa(dp5726
S'a'
I8994
sS's'
I15
sS'm'
(lp5727
I1
aI1
aI3
aI11
aI25
aI51
aI45
aI235
aI379
aI123
aI1701
aI725
aI1991
aI7471
aI9833
asS'd'
I2864
sa(dp5728
S'a'
I9000
sS's'
I15
sS'm'
(lp5729
I1
aI1
aI5
aI13
aI15
aI47
aI13
aI201
aI263
aI57
aI375
aI2963
aI7475
aI15929
aI13775
asS'd'
I2865
sa(dp5730
S'a'
I9013
sS's'
I15
sS'm'
(lp5731
I1
aI1
aI3
aI1
aI29
aI29
aI11
aI161
aI345
aI253
aI97
aI255
aI7267
aI2379
aI3933
asS'd'
I2866
sa(dp5732
S'a'
I9018
sS's'
I15
sS'm'
(lp5733
I1
aI3
aI1
aI15
aI3
aI47
aI11
aI69
aI347
aI747
aI795
aI2401
aI3367
aI2383
aI6125
asS'd'
I2867
sa(dp5734
S'a'
I9020
sS's'
I15
sS'm'
(lp5735
I1
aI1
aI7
aI3
aI1
aI49
aI101
aI47
aI71
aI761
aI1503
aI2619
aI191
aI8895
aI873
asS'd'
I2868
sa(dp5736
S'a'
I9031
sS's'
I15
sS'm'
(lp5737
I1
aI3
aI3
aI5
aI25
aI41
aI93
aI85
aI427
aI109
aI1675
aI2409
aI4317
aI9233
aI30283
asS'd'
I2869
sa(dp5738
S'a'
I9035
sS's'
I15
sS'm'
(lp5739
I1
aI1
aI3
aI9
aI11
aI3
aI67
aI159
aI425
aI751
aI887
aI1415
aI403
aI15977
aI10739
asS'd'
I2870
sa(dp5740
S'a'
I9045
sS's'
I15
sS'm'
(lp5741
I1
aI1
aI5
aI13
aI9
aI1
aI9
aI103
aI481
aI601
aI931
aI1957
aI5763
aI7095
aI27141
asS'd'
I2871
sa(dp5742
S'a'
I9052
sS's'
I15
sS'm'
(lp5743
I1
aI1
aI3
aI15
aI29
aI13
aI43
aI33
aI297
aI269
aI1041
aI1411
aI3461
aI12043
aI10045
asS'd'
I2872
sa(dp5744
S'a'
I9056
sS's'
I15
sS'm'
(lp5745
I1
aI3
aI5
aI3
aI3
aI3
aI5
aI7
aI185
aI753
aI133
aI1561
aI5595
aI13777
aI25795
asS'd'
I2873
sa(dp5746
S'a'
I9059
sS's'
I15
sS'm'
(lp5747
I1
aI3
aI5
aI5
aI1
aI19
aI29
aI145
aI163
aI149
aI619
aI2603
aI7757
aI10035
aI10189
asS'd'
I2874
sa(dp5748
S'a'
I9066
sS's'
I15
sS'm'
(lp5749
I1
aI3
aI7
aI15
aI27
aI15
aI111
aI173
aI135
aI117
aI157
aI2601
aI7919
aI12111
aI22795
asS'd'
I2875
sa(dp5750
S'a'
I9076
sS's'
I15
sS'm'
(lp5751
I1
aI3
aI1
aI1
aI29
aI27
aI65
aI31
aI101
aI715
aI289
aI3643
aI2335
aI6789
aI23397
asS'd'
I2876
sa(dp5752
S'a'
I9089
sS's'
I15
sS'm'
(lp5753
I1
aI3
aI1
aI3
aI11
aI45
aI71
aI109
aI321
aI423
aI1695
aI169
aI3075
aI12423
aI11391
asS'd'
I2877
sa(dp5754
S'a'
I9129
sS's'
I15
sS'm'
(lp5755
I1
aI1
aI3
aI9
aI13
aI51
aI35
aI121
aI203
aI279
aI433
aI2725
aI7951
aI2105
aI27333
asS'd'
I2878
sa(dp5756
S'a'
I9132
sS's'
I15
sS'm'
(lp5757
I1
aI1
aI1
aI15
aI23
aI31
aI25
aI105
aI501
aI441
aI1511
aI3133
aI2811
aI10595
aI21779
asS'd'
I2879
sa(dp5758
S'a'
I9147
sS's'
I15
sS'm'
(lp5759
I1
aI1
aI5
aI13
aI7
aI1
aI97
aI193
aI121
aI993
aI1347
aI1903
aI1883
aI6583
aI24535
asS'd'
I2880
sa(dp5760
S'a'
I9164
sS's'
I15
sS'm'
(lp5761
I1
aI1
aI7
aI9
aI7
aI29
aI17
aI41
aI101
aI447
aI1289
aI387
aI1891
aI2723
aI26091
asS'd'
I2881
sa(dp5762
S'a'
I9167
sS's'
I15
sS'm'
(lp5763
I1
aI1
aI3
aI3
aI3
aI53
aI81
aI81
aI177
aI165
aI195
aI3413
aI8177
aI3817
aI8453
asS'd'
I2882
sa(dp5764
S'a'
I9185
sS's'
I15
sS'm'
(lp5765
I1
aI3
aI7
aI15
aI15
aI31
aI23
aI31
aI337
aI439
aI1773
aI63
aI5351
aI5491
aI1767
asS'd'
I2883
sa(dp5766
S'a'
I9195
sS's'
I15
sS'm'
(lp5767
I1
aI3
aI1
aI11
aI5
aI15
aI23
aI75
aI437
aI553
aI429
aI2705
aI3625
aI13851
aI19865
asS'd'
I2884
sa(dp5768
S'a'
I9197
sS's'
I15
sS'm'
(lp5769
I1
aI3
aI3
aI9
aI13
aI15
aI33
aI235
aI215
aI415
aI1737
aI1409
aI2101
aI14623
aI14717
asS'd'
I2885
sa(dp5770
S'a'
I9210
sS's'
I15
sS'm'
(lp5771
I1
aI3
aI7
aI7
aI13
aI51
aI101
aI217
aI175
aI813
aI1639
aI4009
aI1625
aI4991
aI17525
asS'd'
I2886
sa(dp5772
S'a'
I9217
sS's'
I15
sS'm'
(lp5773
I1
aI1
aI5
aI13
aI23
aI33
aI29
aI175
aI39
aI673
aI557
aI3239
aI5129
aI11049
aI27227
asS'd'
I2887
sa(dp5774
S'a'
I9229
sS's'
I15
sS'm'
(lp5775
I1
aI3
aI7
aI13
aI1
aI37
aI33
aI139
aI493
aI891
aI1883
aI2525
aI5741
aI15795
aI5875
asS'd'
I2888
sa(dp5776
S'a'
I9248
sS's'
I15
sS'm'
(lp5777
I1
aI3
aI1
aI15
aI15
aI27
aI127
aI111
aI147
aI363
aI725
aI3077
aI4341
aI9131
aI24635
asS'd'
I2889
sa(dp5778
S'a'
I9254
sS's'
I15
sS'm'
(lp5779
I1
aI1
aI7
aI3
aI17
aI25
aI59
aI135
aI177
aI635
aI73
aI3455
aI3083
aI6009
aI13033
asS'd'
I2890
sa(dp5780
S'a'
I9263
sS's'
I15
sS'm'
(lp5781
I1
aI1
aI1
aI5
aI15
aI53
aI93
aI161
aI215
aI459
aI1087
aI179
aI2235
aI8885
aI15309
asS'd'
I2891
sa(dp5782
S'a'
I9266
sS's'
I15
sS'm'
(lp5783
I1
aI1
aI7
aI13
aI7
aI17
aI75
aI173
aI449
aI855
aI103
aI2739
aI3421
aI11811
aI18805
asS'd'
I2892
sa(dp5784
S'a'
I9268
sS's'
I15
sS'm'
(lp5785
I1
aI1
aI7
aI9
aI5
aI11
aI53
aI75
aI247
aI249
aI1201
aI953
aI2455
aI4589
aI6027
asS'd'
I2893
sa(dp5786
S'a'
I9290
sS's'
I15
sS'm'
(lp5787
I1
aI1
aI5
aI13
aI27
aI51
aI119
aI39
aI137
aI11
aI1435
aI3773
aI3889
aI6081
aI11829
asS'd'
I2894
sa(dp5788
S'a'
I9310
sS's'
I15
sS'm'
(lp5789
I1
aI1
aI5
aI5
aI5
aI35
aI1
aI197
aI501
aI185
aI1039
aI1563
aI6421
aI14373
aI25655
asS'd'
I2895
sa(dp5790
S'a'
I9316
sS's'
I15
sS'm'
(lp5791
I1
aI1
aI3
aI13
aI31
aI55
aI115
aI183
aI483
aI655
aI1351
aI3203
aI725
aI3299
aI22579
asS'd'
I2896
sa(dp5792
S'a'
I9338
sS's'
I15
sS'm'
(lp5793
I1
aI3
aI5
aI11
aI31
aI31
aI83
aI59
aI395
aI21
aI1881
aI2821
aI2251
aI11781
aI26265
asS'd'
I2897
sa(dp5794
S'a'
I9340
sS's'
I15
sS'm'
(lp5795
I1
aI3
aI7
aI13
aI21
aI19
aI103
aI21
aI403
aI443
aI1951
aI55
aI985
aI15983
aI15087
asS'd'
I2898
sa(dp5796
S'a'
I9343
sS's'
I15
sS'm'
(lp5797
I1
aI1
aI5
aI15
aI29
aI11
aI51
aI53
aI255
aI183
aI1475
aI1491
aI259
aI387
aI10303
asS'd'
I2899
sa(dp5798
S'a'
I9344
sS's'
I15
sS'm'
(lp5799
I1
aI3
aI5
aI7
aI21
aI37
aI45
aI39
aI479
aI637
aI1325
aI3753
aI3319
aI7403
aI31759
asS'd'
I2900
sa(dp5800
S'a'
I9350
sS's'
I15
sS'm'
(lp5801
I1
aI1
aI3
aI5
aI7
aI43
aI89
aI53
aI269
aI187
aI995
aI141
aI119
aI8139
aI29699
asS'd'
I2901
sa(dp5802
S'a'
I9354
sS's'
I15
sS'm'
(lp5803
I1
aI1
aI1
aI5
aI1
aI53
aI3
aI23
aI379
aI223
aI1889
aI4035
aI1437
aI12425
aI9051
asS'd'
I2902
sa(dp5804
S'a'
I9359
sS's'
I15
sS'm'
(lp5805
I1
aI3
aI1
aI13
aI3
aI31
aI61
aI43
aI249
aI449
aI901
aI1921
aI3495
aI8599
aI5263
asS'd'
I2903
sa(dp5806
S'a'
I9361
sS's'
I15
sS'm'
(lp5807
I1
aI1
aI3
aI5
aI3
aI25
aI35
aI133
aI25
aI597
aI915
aI3663
aI5147
aI11831
aI24269
asS'd'
I2904
sa(dp5808
S'a'
I9364
sS's'
I15
sS'm'
(lp5809
I1
aI1
aI1
aI9
aI21
aI27
aI93
aI93
aI217
aI299
aI1881
aI3647
aI4825
aI7989
aI24121
asS'd'
I2905
sa(dp5810
S'a'
I9368
sS's'
I15
sS'm'
(lp5811
I1
aI3
aI1
aI15
aI5
aI15
aI49
aI129
aI315
aI631
aI2037
aI1567
aI4043
aI15589
aI30905
asS'd'
I2906
sa(dp5812
S'a'
I9371
sS's'
I15
sS'm'
(lp5813
I1
aI3
aI3
aI7
aI25
aI5
aI123
aI51
aI47
aI471
aI1563
aI3947
aI7975
aI3681
aI9611
asS'd'
I2907
sa(dp5814
S'a'
I9373
sS's'
I15
sS'm'
(lp5815
I1
aI3
aI7
aI15
aI1
aI17
aI73
aI245
aI465
aI95
aI95
aI1159
aI1319
aI4675
aI8841
asS'd'
I2908
sa(dp5816
S'a'
I9389
sS's'
I15
sS'm'
(lp5817
I1
aI1
aI3
aI15
aI5
aI51
aI35
aI71
aI423
aI651
aI753
aI173
aI2131
aI15799
aI29601
asS'd'
I2909
sa(dp5818
S'a'
I9390
sS's'
I15
sS'm'
(lp5819
I1
aI1
aI1
aI1
aI3
aI53
aI83
aI187
aI445
aI827
aI1549
aI979
aI5363
aI1701
aI2149
asS'd'
I2910
sa(dp5820
S'a'
I9409
sS's'
I15
sS'm'
(lp5821
I1
aI1
aI7
aI9
aI3
aI15
aI65
aI161
aI37
aI233
aI771
aI3749
aI727
aI6857
aI17175
asS'd'
I2911
sa(dp5822
S'a'
I9443
sS's'
I15
sS'm'
(lp5823
I1
aI1
aI7
aI7
aI27
aI29
aI107
aI247
aI249
aI353
aI773
aI3677
aI7273
aI5419
aI29397
asS'd'
I2912
sa(dp5824
S'a'
I9445
sS's'
I15
sS'm'
(lp5825
I1
aI3
aI3
aI7
aI31
aI49
aI87
aI159
aI145
aI497
aI1715
aI2115
aI5035
aI6431
aI7245
asS'd'
I2913
sa(dp5826
S'a'
I9446
sS's'
I15
sS'm'
(lp5827
I1
aI3
aI3
aI5
aI7
aI31
aI51
aI117
aI101
aI617
aI557
aI2551
aI6589
aI13295
aI31975
asS'd'
I2914
sa(dp5828
S'a'
I9452
sS's'
I15
sS'm'
(lp5829
I1
aI1
aI3
aI3
aI15
aI27
aI125
aI163
aI169
aI893
aI1771
aI25
aI5787
aI10267
aI10297
asS'd'
I2915
sa(dp5830
S'a'
I9490
sS's'
I15
sS'm'
(lp5831
I1
aI1
aI1
aI5
aI9
aI47
aI85
aI65
aI289
aI783
aI1105
aI4035
aI4111
aI2589
aI24575
asS'd'
I2916
sa(dp5832
S'a'
I9492
sS's'
I15
sS'm'
(lp5833
I1
aI3
aI3
aI13
aI23
aI33
aI7
aI49
aI301
aI531
aI1713
aI2755
aI5543
aI8153
aI24099
asS'd'
I2917
sa(dp5834
S'a'
I9495
sS's'
I15
sS'm'
(lp5835
I1
aI1
aI5
aI9
aI7
aI39
aI101
aI67
aI417
aI923
aI757
aI1537
aI5553
aI12233
aI20881
asS'd'
I2918
sa(dp5836
S'a'
I9508
sS's'
I15
sS'm'
(lp5837
I1
aI1
aI5
aI1
aI19
aI7
aI25
aI123
aI125
aI183
aI573
aI3317
aI6867
aI871
aI17631
asS'd'
I2919
sa(dp5838
S'a'
I9523
sS's'
I15
sS'm'
(lp5839
I1
aI1
aI3
aI15
aI19
aI13
aI117
aI41
aI129
aI715
aI1525
aI2257
aI2179
aI10807
aI23271
asS'd'
I2920
sa(dp5840
S'a'
I9543
sS's'
I15
sS'm'
(lp5841
I1
aI3
aI1
aI5
aI25
aI53
aI19
aI169
aI289
aI569
aI1135
aI1967
aI7001
aI15883
aI15113
asS'd'
I2921
sa(dp5842
S'a'
I9558
sS's'
I15
sS'm'
(lp5843
I1
aI3
aI7
aI15
aI7
aI37
aI127
aI147
aI415
aI313
aI1541
aI1889
aI3763
aI16199
aI12681
asS'd'
I2922
sa(dp5844
S'a'
I9567
sS's'
I15
sS'm'
(lp5845
I1
aI1
aI3
aI9
aI1
aI35
aI95
aI137
aI237
aI951
aI899
aI3177
aI6073
aI10655
aI31687
asS'd'
I2923
sa(dp5846
S'a'
I9580
sS's'
I15
sS'm'
(lp5847
I1
aI1
aI5
aI5
aI29
aI57
aI45
aI253
aI297
aI529
aI1553
aI467
aI8035
aI15675
aI21691
asS'd'
I2924
sa(dp5848
S'a'
I9585
sS's'
I15
sS'm'
(lp5849
I1
aI3
aI7
aI15
aI25
aI41
aI59
aI81
aI87
aI985
aI1001
aI2369
aI661
aI7551
aI11829
asS'd'
I2925
sa(dp5850
S'a'
I9591
sS's'
I15
sS'm'
(lp5851
I1
aI1
aI7
aI9
aI27
aI21
aI7
aI233
aI309
aI67
aI701
aI2737
aI4261
aI2467
aI15691
asS'd'
I2926
sa(dp5852
S'a'
I9611
sS's'
I15
sS'm'
(lp5853
I1
aI3
aI7
aI1
aI19
aI55
aI47
aI155
aI333
aI101
aI517
aI1991
aI4619
aI10435
aI27241
asS'd'
I2927
sa(dp5854
S'a'
I9613
sS's'
I15
sS'm'
(lp5855
I1
aI1
aI7
aI3
aI23
aI35
aI7
aI125
aI157
aI537
aI933
aI3281
aI4975
aI8969
aI27581
asS'd'
I2928
sa(dp5856
S'a'
I9614
sS's'
I15
sS'm'
(lp5857
I1
aI1
aI3
aI7
aI19
aI53
aI81
aI103
aI461
aI435
aI777
aI335
aI5261
aI12249
aI9695
asS'd'
I2929
sa(dp5858
S'a'
I9621
sS's'
I15
sS'm'
(lp5859
I1
aI3
aI1
aI7
aI19
aI9
aI75
aI245
aI355
aI37
aI1855
aI1339
aI3107
aI7251
aI16543
asS'd'
I2930
sa(dp5860
S'a'
I9631
sS's'
I15
sS'm'
(lp5861
I1
aI1
aI1
aI3
aI5
aI35
aI39
aI223
aI113
aI423
aI1423
aI713
aI6113
aI349
aI24147
asS'd'
I2931
sa(dp5862
S'a'
I9642
sS's'
I15
sS'm'
(lp5863
I1
aI3
aI1
aI1
aI15
aI31
aI11
aI75
aI499
aI345
aI1253
aI2629
aI2551
aI7483
aI25395
asS'd'
I2932
sa(dp5864
S'a'
I9656
sS's'
I15
sS'm'
(lp5865
I1
aI1
aI3
aI11
aI25
aI25
aI3
aI211
aI185
aI45
aI1865
aI1805
aI3303
aI11091
aI529
asS'd'
I2933
sa(dp5866
S'a'
I9661
sS's'
I15
sS'm'
(lp5867
I1
aI3
aI1
aI1
aI9
aI21
aI7
aI165
aI107
aI641
aI1083
aI2805
aI2099
aI5855
aI18477
asS'd'
I2934
sa(dp5868
S'a'
I9667
sS's'
I15
sS'm'
(lp5869
I1
aI3
aI5
aI3
aI9
aI21
aI77
aI103
aI505
aI277
aI335
aI797
aI3869
aI2957
aI1979
asS'd'
I2935
sa(dp5870
S'a'
I9694
sS's'
I15
sS'm'
(lp5871
I1
aI3
aI5
aI15
aI31
aI23
aI77
aI247
aI303
aI891
aI1261
aI3233
aI3495
aI13111
aI13185
asS'd'
I2936
sa(dp5872
S'a'
I9715
sS's'
I15
sS'm'
(lp5873
I1
aI3
aI5
aI11
aI11
aI35
aI49
aI229
aI149
aI931
aI881
aI775
aI2949
aI3141
aI29157
asS'd'
I2937
sa(dp5874
S'a'
I9722
sS's'
I15
sS'm'
(lp5875
I1
aI1
aI3
aI5
aI19
aI57
aI23
aI95
aI347
aI221
aI195
aI3561
aI1481
aI2063
aI3979
asS'd'
I2938
sa(dp5876
S'a'
I9738
sS's'
I15
sS'm'
(lp5877
I1
aI3
aI5
aI3
aI13
aI1
aI23
aI173
aI431
aI29
aI421
aI3235
aI2751
aI4447
aI28283
asS'd'
I2939
sa(dp5878
S'a'
I9745
sS's'
I15
sS'm'
(lp5879
I1
aI1
aI5
aI13
aI23
aI3
aI1
aI9
aI327
aI855
aI1251
aI2997
aI6129
aI4223
aI11555
asS'd'
I2940
sa(dp5880
S'a'
I9758
sS's'
I15
sS'm'
(lp5881
I1
aI3
aI7
aI13
aI29
aI21
aI37
aI229
aI217
aI353
aI1239
aI3955
aI491
aI12183
aI14777
asS'd'
I2941
sa(dp5882
S'a'
I9764
sS's'
I15
sS'm'
(lp5883
I1
aI1
aI5
aI5
aI1
aI33
aI103
aI187
aI183
aI939
aI1873
aI2633
aI6143
aI15405
aI17353
asS'd'
I2942
sa(dp5884
S'a'
I9782
sS's'
I15
sS'm'
(lp5885
I1
aI1
aI1
aI9
aI21
aI27
aI71
aI129
aI499
aI279
aI1181
aI4053
aI2485
aI1961
aI30603
asS'd'
I2943
sa(dp5886
S'a'
I9791
sS's'
I15
sS'm'
(lp5887
I1
aI1
aI3
aI15
aI21
aI37
aI45
aI201
aI221
aI187
aI727
aI1241
aI6171
aI1383
aI22277
asS'd'
I2944
sa(dp5888
S'a'
I9793
sS's'
I15
sS'm'
(lp5889
I1
aI3
aI7
aI5
aI21
aI17
aI67
aI177
aI323
aI601
aI633
aI865
aI6131
aI10329
aI8689
asS'd'
I2945
sa(dp5890
S'a'
I9794
sS's'
I15
sS'm'
(lp5891
I1
aI3
aI5
aI9
aI15
aI45
aI71
aI43
aI359
aI651
aI103
aI403
aI3249
aI11769
aI6567
asS'd'
I2946
sa(dp5892
S'a'
I9805
sS's'
I15
sS'm'
(lp5893
I1
aI3
aI3
aI13
aI3
aI23
aI101
aI145
aI367
aI999
aI1489
aI3673
aI2959
aI10855
aI16029
asS'd'
I2947
sa(dp5894
S'a'
I9808
sS's'
I15
sS'm'
(lp5895
I1
aI3
aI7
aI3
aI13
aI43
aI123
aI87
aI55
aI1015
aI141
aI2917
aI6567
aI16025
aI25555
asS'd'
I2948
sa(dp5896
S'a'
I9811
sS's'
I15
sS'm'
(lp5897
I1
aI3
aI1
aI3
aI17
aI7
aI21
aI161
aI41
aI889
aI1315
aI1897
aI639
aI15451
aI3049
asS'd'
I2949
sa(dp5898
S'a'
I9817
sS's'
I15
sS'm'
(lp5899
I1
aI3
aI5
aI15
aI27
aI33
aI55
aI17
aI81
aI431
aI325
aI909
aI3547
aI10121
aI17815
asS'd'
I2950
sa(dp5900
S'a'
I9824
sS's'
I15
sS'm'
(lp5901
I1
aI1
aI3
aI1
aI15
aI37
aI43
aI137
aI203
aI191
aI1129
aI1585
aI435
aI3177
aI769
asS'd'
I2951
sa(dp5902
S'a'
I9836
sS's'
I15
sS'm'
(lp5903
I1
aI3
aI7
aI11
aI21
aI23
aI125
aI41
aI17
aI951
aI465
aI3691
aI3465
aI13247
aI13779
asS'd'
I2952
sa(dp5904
S'a'
I9851
sS's'
I15
sS'm'
(lp5905
I1
aI3
aI3
aI1
aI31
aI23
aI43
aI101
aI405
aI739
aI1061
aI2955
aI5643
aI16137
aI8763
asS'd'
I2953
sa(dp5906
S'a'
I9853
sS's'
I15
sS'm'
(lp5907
I1
aI1
aI5
aI1
aI19
aI33
aI99
aI109
aI203
aI65
aI395
aI2775
aI1373
aI2557
aI5875
asS'd'
I2954
sa(dp5908
S'a'
I9854
sS's'
I15
sS'm'
(lp5909
I1
aI3
aI3
aI3
aI27
aI51
aI79
aI63
aI331
aI365
aI1071
aI1661
aI4549
aI8561
aI1719
asS'd'
I2955
sa(dp5910
S'a'
I9877
sS's'
I15
sS'm'
(lp5911
I1
aI3
aI3
aI9
aI3
aI17
aI53
aI161
aI141
aI489
aI1325
aI1709
aI1381
aI5093
aI171
asS'd'
I2956
sa(dp5912
S'a'
I9881
sS's'
I15
sS'm'
(lp5913
I1
aI1
aI7
aI15
aI9
aI3
aI95
aI237
aI197
aI949
aI7
aI1837
aI729
aI10111
aI6637
asS'd'
I2957
sa(dp5914
S'a'
I9923
sS's'
I15
sS'm'
(lp5915
I1
aI1
aI3
aI3
aI19
aI31
aI57
aI173
aI483
aI861
aI1001
aI1919
aI3389
aI11777
aI20693
asS'd'
I2958
sa(dp5916
S'a'
I9935
sS's'
I15
sS'm'
(lp5917
I1
aI3
aI1
aI9
aI27
aI13
aI113
aI177
aI75
aI925
aI949
aI119
aI4759
aI7775
aI23033
asS'd'
I2959
sa(dp5918
S'a'
I9937
sS's'
I15
sS'm'
(lp5919
I1
aI1
aI7
aI15
aI23
aI15
aI65
aI61
aI137
aI653
aI1843
aI323
aI379
aI15157
aI29885
asS'd'
I2960
sa(dp5920
S'a'
I9954
sS's'
I15
sS'm'
(lp5921
I1
aI3
aI3
aI7
aI29
aI3
aI11
aI205
aI347
aI745
aI1477
aI3929
aI5749
aI4735
aI29435
asS'd'
I2961
sa(dp5922
S'a'
I9959
sS's'
I15
sS'm'
(lp5923
I1
aI3
aI5
aI9
aI1
aI11
aI111
aI15
aI7
aI69
aI45
aI3607
aI1099
aI9203
aI21301
asS'd'
I2962
sa(dp5924
S'a'
I9963
sS's'
I15
sS'm'
(lp5925
I1
aI3
aI3
aI3
aI23
aI3
aI83
aI173
aI73
aI485
aI681
aI1867
aI3839
aI11823
aI13339
asS'd'
I2963
sa(dp5926
S'a'
I9968
sS's'
I15
sS'm'
(lp5927
I1
aI1
aI3
aI11
aI31
aI43
aI107
aI127
aI465
aI389
aI1595
aI427
aI1571
aI5885
aI29569
asS'd'
I2964
sa(dp5928
S'a'
I9973
sS's'
I15
sS'm'
(lp5929
I1
aI1
aI7
aI9
aI27
aI25
aI117
aI27
aI287
aI391
aI279
aI3247
aI35
aI12973
aI5483
asS'd'
I2965
sa(dp5930
S'a'
I9974
sS's'
I15
sS'm'
(lp5931
I1
aI3
aI7
aI11
aI19
aI55
aI45
aI127
aI245
aI945
aI305
aI3907
aI2455
aI3163
aI31
asS'd'
I2966
sa(dp5932
S'a'
I9980
sS's'
I15
sS'm'
(lp5933
I1
aI1
aI7
aI11
aI15
aI17
aI65
aI15
aI37
aI207
aI1447
aI3027
aI2281
aI6557
aI16717
asS'd'
I2967
sa(dp5934
S'a'
I9983
sS's'
I15
sS'm'
(lp5935
I1
aI1
aI1
aI13
aI5
aI27
aI33
aI213
aI29
aI603
aI1171
aI3235
aI2255
aI2017
aI30999
asS'd'
I2968
sa(dp5936
S'a'
I9985
sS's'
I15
sS'm'
(lp5937
I1
aI3
aI1
aI5
aI11
aI1
aI73
aI233
aI69
aI125
aI397
aI297
aI3337
aI6191
aI31055
asS'd'
I2969
sa(dp5938
S'a'
I10003
sS's'
I15
sS'm'
(lp5939
I1
aI1
aI1
aI15
aI1
aI1
aI65
aI145
aI201
aI917
aI1891
aI2999
aI4069
aI10413
aI15819
asS'd'
I2970
sa(dp5940
S'a'
I10010
sS's'
I15
sS'm'
(lp5941
I1
aI3
aI5
aI13
aI15
aI51
aI115
aI167
aI311
aI375
aI1069
aI2595
aI3337
aI753
aI11903
asS'd'
I2971
sa(dp5942
S'a'
I10034
sS's'
I15
sS'm'
(lp5943
I1
aI1
aI3
aI1
aI1
aI23
aI69
aI125
aI147
aI915
aI1945
aI411
aI979
aI13863
aI30443
asS'd'
I2972
sa(dp5944
S'a'
I10040
sS's'
I15
sS'm'
(lp5945
I1
aI3
aI1
aI11
aI5
aI1
aI93
aI23
aI135
aI93
aI1689
aI23
aI3519
aI4491
aI24673
asS'd'
I2973
sa(dp5946
S'a'
I10063
sS's'
I15
sS'm'
(lp5947
I1
aI1
aI7
aI3
aI11
aI59
aI93
aI153
aI487
aI475
aI1191
aI1455
aI5963
aI8259
aI18811
asS'd'
I2974
sa(dp5948
S'a'
I10077
sS's'
I15
sS'm'
(lp5949
I1
aI1
aI3
aI1
aI13
aI15
aI55
aI71
aI433
aI33
aI491
aI1835
aI5695
aI10509
aI347
asS'd'
I2975
sa(dp5950
S'a'
I10081
sS's'
I15
sS'm'
(lp5951
I1
aI1
aI1
aI15
aI19
aI1
aI23
aI47
aI235
aI101
aI1057
aI901
aI5477
aI7079
aI30885
asS'd'
I2976
sa(dp5952
S'a'
I10082
sS's'
I15
sS'm'
(lp5953
I1
aI1
aI5
aI13
aI11
aI43
aI119
aI77
aI441
aI121
aI783
aI827
aI1757
aI12751
aI31593
asS'd'
I2977
sa(dp5954
S'a'
I10084
sS's'
I15
sS'm'
(lp5955
I1
aI3
aI7
aI11
aI19
aI17
aI37
aI225
aI329
aI231
aI515
aI1541
aI7371
aI6355
aI10905
asS'd'
I2978
sa(dp5956
S'a'
I10088
sS's'
I15
sS'm'
(lp5957
I1
aI1
aI5
aI13
aI7
aI11
aI35
aI215
aI345
aI577
aI147
aI2803
aI3291
aI4631
aI5329
asS'd'
I2979
sa(dp5958
S'a'
I10091
sS's'
I15
sS'm'
(lp5959
I1
aI1
aI3
aI9
aI21
aI55
aI113
aI251
aI25
aI221
aI1445
aI3385
aI1589
aI4109
aI29897
asS'd'
I2980
sa(dp5960
S'a'
I10105
sS's'
I15
sS'm'
(lp5961
I1
aI1
aI5
aI7
aI9
aI45
aI5
aI33
aI331
aI285
aI1101
aI3131
aI2713
aI5653
aI3823
asS'd'
I2981
sa(dp5962
S'a'
I10111
sS's'
I15
sS'm'
(lp5963
I1
aI3
aI7
aI7
aI5
aI39
aI43
aI167
aI481
aI629
aI777
aI1827
aI4653
aI403
aI4781
asS'd'
I2982
sa(dp5964
S'a'
I10118
sS's'
I15
sS'm'
(lp5965
I1
aI3
aI3
aI7
aI31
aI33
aI31
aI159
aI313
aI673
aI1425
aI663
aI5819
aI1297
aI26627
asS'd'
I2983
sa(dp5966
S'a'
I10127
sS's'
I15
sS'm'
(lp5967
I1
aI3
aI3
aI1
aI19
aI61
aI117
aI93
aI373
aI491
aI1031
aI757
aI4185
aI771
aI7265
asS'd'
I2984
sa(dp5968
S'a'
I10135
sS's'
I15
sS'm'
(lp5969
I1
aI1
aI7
aI9
aI3
aI45
aI65
aI223
aI437
aI41
aI1139
aI2733
aI5963
aI2709
aI25429
asS'd'
I2985
sa(dp5970
S'a'
I10169
sS's'
I15
sS'm'
(lp5971
I1
aI3
aI5
aI11
aI21
aI27
aI31
aI127
aI255
aI761
aI1865
aI1319
aI6583
aI9235
aI10717
asS'd'
I2986
sa(dp5972
S'a'
I10172
sS's'
I15
sS'm'
(lp5973
I1
aI1
aI1
aI5
aI21
aI1
aI63
aI43
aI413
aI557
aI567
aI2893
aI8017
aI2307
aI29525
asS'd'
I2987
sa(dp5974
S'a'
I10183
sS's'
I15
sS'm'
(lp5975
I1
aI1
aI7
aI3
aI31
aI1
aI15
aI235
aI215
aI395
aI1971
aI469
aI5275
aI431
aI5349
asS'd'
I2988
sa(dp5976
S'a'
I10190
sS's'
I15
sS'm'
(lp5977
I1
aI1
aI1
aI13
aI25
aI59
aI71
aI245
aI389
aI279
aI1293
aI89
aI6551
aI10285
aI14495
asS'd'
I2989
sa(dp5978
S'a'
I10192
sS's'
I15
sS'm'
(lp5979
I1
aI1
aI5
aI5
aI9
aI63
aI17
aI229
aI425
aI939
aI877
aI3689
aI7229
aI6707
aI30771
asS'd'
I2990
sa(dp5980
S'a'
I10211
sS's'
I15
sS'm'
(lp5981
I1
aI3
aI7
aI7
aI11
aI29
aI43
aI41
aI25
aI237
aI1585
aI3735
aI2617
aI7541
aI26243
asS'd'
I2991
sa(dp5982
S'a'
I10218
sS's'
I15
sS'm'
(lp5983
I1
aI1
aI7
aI9
aI21
aI5
aI69
aI231
aI361
aI39
aI1695
aI3043
aI2973
aI5487
aI12857
asS'd'
I2992
sa(dp5984
S'a'
I10228
sS's'
I15
sS'm'
(lp5985
I1
aI1
aI5
aI3
aI17
aI63
aI91
aI217
aI407
aI133
aI1373
aI4021
aI1737
aI10043
aI4561
asS'd'
I2993
sa(dp5986
S'a'
I10235
sS's'
I15
sS'm'
(lp5987
I1
aI3
aI7
aI9
aI31
aI13
aI101
aI231
aI175
aI457
aI89
aI2167
aI2725
aI8505
aI375
asS'd'
I2994
sa(dp5988
S'a'
I10242
sS's'
I15
sS'm'
(lp5989
I1
aI1
aI3
aI15
aI31
aI11
aI27
aI211
aI347
aI291
aI1881
aI3091
aI3307
aI5117
aI13341
asS'd'
I2995
sa(dp5990
S'a'
I10244
sS's'
I15
sS'm'
(lp5991
I1
aI3
aI5
aI5
aI13
aI25
aI5
aI197
aI237
aI135
aI635
aI1175
aI5085
aI14737
aI10807
asS'd'
I2996
sa(dp5992
S'a'
I10271
sS's'
I15
sS'm'
(lp5993
I1
aI3
aI3
aI9
aI7
aI63
aI107
aI127
aI147
aI477
aI1813
aI2619
aI8089
aI2651
aI26549
asS'd'
I2997
sa(dp5994
S'a'
I10278
sS's'
I15
sS'm'
(lp5995
I1
aI1
aI5
aI11
aI15
aI45
aI27
aI133
aI45
aI621
aI707
aI2679
aI5929
aI19
aI9599
asS'd'
I2998
sa(dp5996
S'a'
I10296
sS's'
I15
sS'm'
(lp5997
I1
aI3
aI7
aI9
aI21
aI37
aI41
aI255
aI69
aI1009
aI1999
aI367
aI6177
aI10017
aI3549
asS'd'
I2999
sa(dp5998
S'a'
I10299
sS's'
I15
sS'm'
(lp5999
I1
aI1
aI1
aI15
aI19
aI55
aI73
aI189
aI423
aI983
aI1811
aI2551
aI4765
aI12077
aI18205
asS'd'
I3000
sa(dp6000
S'a'
I10307
sS's'
I15
sS'm'
(lp6001
I1
aI1
aI5
aI7
aI17
aI13
aI25
aI225
aI463
aI471
aI631
aI1811
aI5797
aI3235
aI32253
asS'd'
I3001
sa(dp6002
S'a'
I10309
sS's'
I15
sS'm'
(lp6003
I1
aI3
aI7
aI1
aI29
aI7
aI123
aI187
aI331
aI735
aI1757
aI1115
aI2077
aI15725
aI2183
asS'd'
I3002
sa(dp6004
S'a'
I10310
sS's'
I15
sS'm'
(lp6005
I1
aI3
aI7
aI9
aI17
aI61
aI111
aI93
aI21
aI1003
aI1905
aI3719
aI2111
aI11845
aI6427
asS'd'
I3003
sa(dp6006
S'a'
I10314
sS's'
I15
sS'm'
(lp6007
I1
aI3
aI7
aI7
aI17
aI21
aI51
aI59
aI115
aI723
aI2039
aI2833
aI5969
aI5737
aI18407
asS'd'
I3004
sa(dp6008
S'a'
I10316
sS's'
I15
sS'm'
(lp6009
I1
aI3
aI3
aI13
aI9
aI47
aI95
aI233
aI13
aI281
aI1049
aI619
aI405
aI16205
aI20097
asS'd'
I3005
sa(dp6010
S'a'
I10321
sS's'
I15
sS'm'
(lp6011
I1
aI3
aI7
aI13
aI9
aI41
aI11
aI171
aI453
aI713
aI587
aI1669
aI2489
aI10277
aI18599
asS'd'
I3006
sa(dp6012
S'a'
I10328
sS's'
I15
sS'm'
(lp6013
I1
aI3
aI3
aI13
aI21
aI41
aI123
aI173
aI511
aI399
aI859
aI1515
aI5773
aI12535
aI26289
asS'd'
I3007
sa(dp6014
S'a'
I10338
sS's'
I15
sS'm'
(lp6015
I1
aI1
aI7
aI15
aI11
aI3
aI113
aI111
aI73
aI7
aI1191
aI2573
aI7713
aI465
aI27615
asS'd'
I3008
sa(dp6016
S'a'
I10343
sS's'
I15
sS'm'
(lp6017
I1
aI1
aI7
aI15
aI5
aI5
aI39
aI11
aI489
aI13
aI1041
aI1639
aI7879
aI11899
aI6899
asS'd'
I3009
sa(dp6018
S'a'
I10344
sS's'
I15
sS'm'
(lp6019
I1
aI1
aI5
aI9
aI27
aI31
aI15
aI237
aI401
aI795
aI1675
aI2361
aI7333
aI12507
aI14627
asS'd'
I3010
sa(dp6020
S'a'
I10347
sS's'
I15
sS'm'
(lp6021
I1
aI3
aI1
aI7
aI21
aI53
aI31
aI81
aI189
aI683
aI1283
aI419
aI7585
aI9207
aI15053
asS'd'
I3011
sa(dp6022
S'a'
I10352
sS's'
I15
sS'm'
(lp6023
I1
aI3
aI5
aI11
aI21
aI1
aI49
aI251
aI403
aI489
aI1235
aI429
aI4855
aI4081
aI17575
asS'd'
I3012
sa(dp6024
S'a'
I10364
sS's'
I15
sS'm'
(lp6025
I1
aI3
aI1
aI15
aI29
aI33
aI77
aI53
aI105
aI731
aI749
aI2677
aI3967
aI7967
aI18723
asS'd'
I3013
sa(dp6026
S'a'
I10373
sS's'
I15
sS'm'
(lp6027
I1
aI3
aI3
aI11
aI9
aI47
aI11
aI95
aI137
aI923
aI599
aI1585
aI1969
aI9625
aI19171
asS'd'
I3014
sa(dp6028
S'a'
I10386
sS's'
I15
sS'm'
(lp6029
I1
aI1
aI1
aI5
aI7
aI7
aI85
aI49
aI339
aI883
aI261
aI2125
aI3831
aI9797
aI16395
asS'd'
I3015
sa(dp6030
S'a'
I10391
sS's'
I15
sS'm'
(lp6031
I1
aI3
aI3
aI3
aI5
aI9
aI33
aI99
aI75
aI889
aI101
aI2099
aI6635
aI11511
aI21573
asS'd'
I3016
sa(dp6032
S'a'
I10398
sS's'
I15
sS'm'
(lp6033
I1
aI1
aI5
aI11
aI1
aI11
aI79
aI49
aI7
aI131
aI471
aI1235
aI3287
aI14777
aI12053
asS'd'
I3017
sa(dp6034
S'a'
I10408
sS's'
I15
sS'm'
(lp6035
I1
aI1
aI5
aI15
aI9
aI9
aI83
aI15
aI21
aI899
aI1785
aI2349
aI3471
aI6723
aI1405
asS'd'
I3018
sa(dp6036
S'a'
I10413
sS's'
I15
sS'm'
(lp6037
I1
aI3
aI5
aI11
aI1
aI7
aI121
aI223
aI509
aI859
aI1037
aI491
aI5529
aI481
aI17029
asS'd'
I3019
sa(dp6038
S'a'
I10422
sS's'
I15
sS'm'
(lp6039
I1
aI1
aI7
aI5
aI17
aI35
aI91
aI171
aI113
aI65
aI423
aI2371
aI5105
aI12827
aI31087
asS'd'
I3020
sa(dp6040
S'a'
I10445
sS's'
I15
sS'm'
(lp6041
I1
aI1
aI3
aI3
aI21
aI47
aI55
aI11
aI159
aI51
aI263
aI2113
aI661
aI9147
aI28929
asS'd'
I3021
sa(dp6042
S'a'
I10460
sS's'
I15
sS'm'
(lp6043
I1
aI1
aI1
aI9
aI19
aI7
aI43
aI223
aI207
aI787
aI543
aI2141
aI4247
aI7369
aI29031
asS'd'
I3022
sa(dp6044
S'a'
I10463
sS's'
I15
sS'm'
(lp6045
I1
aI1
aI7
aI11
aI11
aI51
aI121
aI9
aI211
aI905
aI687
aI889
aI1827
aI13395
aI3507
asS'd'
I3023
sa(dp6046
S'a'
I10464
sS's'
I15
sS'm'
(lp6047
I1
aI3
aI1
aI7
aI15
aI23
aI5
aI139
aI469
aI569
aI33
aI3477
aI5391
aI13665
aI17011
asS'd'
I3024
sa(dp6048
S'a'
I10474
sS's'
I15
sS'm'
(lp6049
I1
aI1
aI1
aI15
aI29
aI29
aI29
aI201
aI63
aI1019
aI97
aI1671
aI9
aI4617
aI19833
asS'd'
I3025
sa(dp6050
S'a'
I10476
sS's'
I15
sS'm'
(lp6051
I1
aI1
aI5
aI15
aI25
aI5
aI67
aI225
aI189
aI919
aI1471
aI1451
aI5017
aI16189
aI31555
asS'd'
I3026
sa(dp6052
S'a'
I10487
sS's'
I15
sS'm'
(lp6053
I1
aI3
aI5
aI5
aI15
aI51
aI89
aI221
aI149
aI863
aI43
aI2381
aI1767
aI8037
aI5319
asS'd'
I3027
sa(dp6054
S'a'
I10494
sS's'
I15
sS'm'
(lp6055
I1
aI3
aI3
aI1
aI15
aI17
aI5
aI77
aI69
aI27
aI1883
aI63
aI5987
aI1497
aI3723
asS'd'
I3028
sa(dp6056
S'a'
I10499
sS's'
I15
sS'm'
(lp6057
I1
aI3
aI7
aI11
aI7
aI5
aI113
aI229
aI123
aI709
aI1531
aI641
aI6655
aI14923
aI22947
asS'd'
I3029
sa(dp6058
S'a'
I10506
sS's'
I15
sS'm'
(lp6059
I1
aI3
aI1
aI13
aI21
aI15
aI45
aI175
aI81
aI499
aI1113
aI587
aI7573
aI11689
aI15651
asS'd'
I3030
sa(dp6060
S'a'
I10513
sS's'
I15
sS'm'
(lp6061
I1
aI3
aI1
aI1
aI29
aI43
aI101
aI37
aI131
aI757
aI465
aI743
aI2737
aI8063
aI23967
asS'd'
I3031
sa(dp6062
S'a'
I10516
sS's'
I15
sS'm'
(lp6063
I1
aI1
aI7
aI13
aI9
aI21
aI39
aI177
aI51
aI691
aI2047
aI1519
aI6137
aI5271
aI8703
asS'd'
I3032
sa(dp6064
S'a'
I10523
sS's'
I15
sS'm'
(lp6065
I1
aI1
aI3
aI3
aI5
aI55
aI63
aI21
aI3
aI317
aI461
aI527
aI2673
aI16211
aI6721
asS'd'
I3033
sa(dp6066
S'a'
I10539
sS's'
I15
sS'm'
(lp6067
I1
aI3
aI5
aI5
aI5
aI47
aI7
aI241
aI387
aI589
aI323
aI203
aI7241
aI14761
aI13287
asS'd'
I3034
sa(dp6068
S'a'
I10549
sS's'
I15
sS'm'
(lp6069
I1
aI3
aI5
aI3
aI23
aI63
aI55
aI61
aI231
aI1023
aI1315
aI1181
aI243
aI7389
aI25639
asS'd'
I3035
sa(dp6070
S'a'
I10550
sS's'
I15
sS'm'
(lp6071
I1
aI1
aI7
aI13
aI31
aI43
aI41
aI81
aI127
aI887
aI1513
aI4055
aI1361
aI2443
aI6963
asS'd'
I3036
sa(dp6072
S'a'
I10567
sS's'
I15
sS'm'
(lp6073
I1
aI1
aI1
aI5
aI7
aI43
aI43
aI33
aI323
aI911
aI1373
aI3053
aI6503
aI513
aI6457
asS'd'
I3037
sa(dp6074
S'a'
I10576
sS's'
I15
sS'm'
(lp6075
I1
aI1
aI7
aI11
aI25
aI61
aI21
aI149
aI205
aI349
aI1433
aI1587
aI149
aI7275
aI5465
asS'd'
I3038
sa(dp6076
S'a'
I10625
sS's'
I15
sS'm'
(lp6077
I1
aI3
aI5
aI5
aI11
aI9
aI31
aI217
aI119
aI511
aI209
aI3325
aI2023
aI2877
aI463
asS'd'
I3039
sa(dp6078
S'a'
I10635
sS's'
I15
sS'm'
(lp6079
I1
aI3
aI5
aI15
aI21
aI47
aI89
aI41
aI347
aI849
aI1375
aI3311
aI807
aI11443
aI27643
asS'd'
I3040
sa(dp6080
S'a'
I10643
sS's'
I15
sS'm'
(lp6081
I1
aI1
aI5
aI7
aI29
aI43
aI123
aI191
aI321
aI373
aI447
aI2145
aI1221
aI2071
aI12689
asS'd'
I3041
sa(dp6082
S'a'
I10656
sS's'
I15
sS'm'
(lp6083
I1
aI3
aI5
aI15
aI1
aI21
aI43
aI141
aI461
aI779
aI1109
aI2915
aI909
aI8585
aI19859
asS'd'
I3042
sa(dp6084
S'a'
I10671
sS's'
I15
sS'm'
(lp6085
I1
aI3
aI3
aI11
aI5
aI17
aI57
aI13
aI393
aI661
aI1761
aI2455
aI43
aI8593
aI20505
asS'd'
I3043
sa(dp6086
S'a'
I10676
sS's'
I15
sS'm'
(lp6087
I1
aI3
aI5
aI1
aI31
aI47
aI65
aI249
aI77
aI513
aI851
aI2381
aI3447
aI693
aI7729
asS'd'
I3044
sa(dp6088
S'a'
I10683
sS's'
I15
sS'm'
(lp6089
I1
aI3
aI5
aI15
aI31
aI19
aI83
aI47
aI369
aI697
aI1815
aI819
aI7573
aI9245
aI8013
asS'd'
I3045
sa(dp6090
S'a'
I10685
sS's'
I15
sS'm'
(lp6091
I1
aI3
aI5
aI5
aI11
aI25
aI27
aI151
aI107
aI339
aI299
aI3869
aI3393
aI5661
aI2947
asS'd'
I3046
sa(dp6092
S'a'
I10688
sS's'
I15
sS'm'
(lp6093
I1
aI1
aI3
aI1
aI1
aI59
aI85
aI57
aI175
aI465
aI239
aI3115
aI7157
aI7035
aI11463
asS'd'
I3047
sa(dp6094
S'a'
I10697
sS's'
I15
sS'm'
(lp6095
I1
aI1
aI7
aI5
aI31
aI41
aI53
aI149
aI121
aI743
aI189
aI159
aI5289
aI2945
aI1179
asS'd'
I3048
sa(dp6096
S'a'
I10700
sS's'
I15
sS'm'
(lp6097
I1
aI3
aI3
aI15
aI23
aI51
aI83
aI25
aI159
aI163
aI61
aI713
aI4529
aI5253
aI1603
asS'd'
I3049
sa(dp6098
S'a'
I10712
sS's'
I15
sS'm'
(lp6099
I1
aI3
aI5
aI11
aI7
aI29
aI15
aI177
aI507
aI695
aI1305
aI1863
aI7525
aI3063
aI27433
asS'd'
I3050
sa(dp6100
S'a'
I10724
sS's'
I15
sS'm'
(lp6101
I1
aI1
aI3
aI11
aI5
aI41
aI115
aI227
aI409
aI951
aI591
aI4003
aI7717
aI4369
aI15637
asS'd'
I3051
sa(dp6102
S'a'
I10728
sS's'
I15
sS'm'
(lp6103
I1
aI1
aI7
aI11
aI23
aI55
aI71
aI135
aI51
aI411
aI2003
aI2375
aI6823
aI1711
aI4443
asS'd'
I3052
sa(dp6104
S'a'
I10734
sS's'
I15
sS'm'
(lp6105
I1
aI3
aI1
aI3
aI31
aI47
aI31
aI233
aI243
aI3
aI313
aI1649
aI6955
aI13679
aI32327
asS'd'
I3053
sa(dp6106
S'a'
I10739
sS's'
I15
sS'm'
(lp6107
I1
aI1
aI3
aI11
aI29
aI9
aI1
aI79
aI247
aI677
aI685
aI3107
aI5987
aI9675
aI29523
asS'd'
I3054
sa(dp6108
S'a'
I10762
sS's'
I15
sS'm'
(lp6109
I1
aI1
aI1
aI7
aI25
aI31
aI39
aI241
aI483
aI839
aI1143
aI437
aI2317
aI2437
aI173
asS'd'
I3055
sa(dp6110
S'a'
I10772
sS's'
I15
sS'm'
(lp6111
I1
aI1
aI5
aI1
aI17
aI19
aI83
aI57
aI39
aI479
aI715
aI1911
aI1091
aI10937
aI22145
asS'd'
I3056
sa(dp6112
S'a'
I10781
sS's'
I15
sS'm'
(lp6113
I1
aI1
aI7
aI1
aI27
aI45
aI35
aI55
aI477
aI719
aI217
aI3349
aI7717
aI6853
aI9699
asS'd'
I3057
sa(dp6114
S'a'
I10800
sS's'
I15
sS'm'
(lp6115
I1
aI3
aI1
aI11
aI9
aI39
aI25
aI223
aI303
aI713
aI151
aI2611
aI4629
aI5855
aI31729
asS'd'
I3058
sa(dp6116
S'a'
I10805
sS's'
I15
sS'm'
(lp6117
I1
aI1
aI1
aI11
aI13
aI35
aI53
aI39
aI167
aI779
aI1673
aI1221
aI6281
aI15113
aI32027
asS'd'
I3059
sa(dp6118
S'a'
I10827
sS's'
I15
sS'm'
(lp6119
I1
aI1
aI5
aI9
aI19
aI63
aI89
aI113
aI199
aI107
aI1015
aI835
aI2879
aI9499
aI25597
asS'd'
I3060
sa(dp6120
S'a'
I10830
sS's'
I15
sS'm'
(lp6121
I1
aI1
aI7
aI3
aI19
aI37
aI15
aI23
aI449
aI641
aI1811
aI3407
aI6775
aI6283
aI31157
asS'd'
I3061
sa(dp6122
S'a'
I10837
sS's'
I15
sS'm'
(lp6123
I1
aI1
aI3
aI1
aI19
aI15
aI31
aI99
aI511
aI897
aI1693
aI2093
aI955
aI15897
aI26693
asS'd'
I3062
sa(dp6124
S'a'
I10841
sS's'
I15
sS'm'
(lp6125
I1
aI1
aI5
aI1
aI5
aI15
aI47
aI19
aI441
aI541
aI1621
aI3877
aI6407
aI15991
aI1931
asS'd'
I3063
sa(dp6126
S'a'
I10847
sS's'
I15
sS'm'
(lp6127
I1
aI3
aI5
aI9
aI21
aI61
aI15
aI77
aI265
aI351
aI879
aI3835
aI6555
aI2349
aI23235
asS'd'
I3064
sa(dp6128
S'a'
I10848
sS's'
I15
sS'm'
(lp6129
I1
aI1
aI5
aI11
aI25
aI37
aI29
aI181
aI341
aI641
aI1213
aI1319
aI6359
aI6231
aI32573
asS'd'
I3065
sa(dp6130
S'a'
I10857
sS's'
I15
sS'm'
(lp6131
I1
aI1
aI1
aI7
aI1
aI37
aI87
aI123
aI33
aI913
aI111
aI2613
aI4895
aI12595
aI26633
asS'd'
I3066
sa(dp6132
S'a'
I10866
sS's'
I15
sS'm'
(lp6133
I1
aI3
aI5
aI3
aI27
aI11
aI45
aI89
aI183
aI241
aI1355
aI783
aI3343
aI239
aI8643
asS'd'
I3067
sa(dp6134
S'a'
I10868
sS's'
I15
sS'm'
(lp6135
I1
aI3
aI7
aI7
aI9
aI35
aI67
aI187
aI233
aI577
aI1445
aI3063
aI6039
aI16233
aI1453
asS'd'
I3068
sa(dp6136
S'a'
I10872
sS's'
I15
sS'm'
(lp6137
I1
aI1
aI3
aI13
aI27
aI11
aI23
aI15
aI95
aI63
aI1931
aI911
aI8149
aI6833
aI3051
asS'd'
I3069
sa(dp6138
S'a'
I10887
sS's'
I15
sS'm'
(lp6139
I1
aI3
aI3
aI5
aI29
aI49
aI125
aI117
aI47
aI143
aI423
aI3215
aI3605
aI3677
aI17155
asS'd'
I3070
sa(dp6140
S'a'
I10899
sS's'
I15
sS'm'
(lp6141
I1
aI3
aI1
aI1
aI31
aI1
aI123
aI195
aI83
aI893
aI1947
aI339
aI2927
aI7183
aI15443
asS'd'
I3071
sa(dp6142
S'a'
I10901
sS's'
I15
sS'm'
(lp6143
I1
aI1
aI7
aI13
aI31
aI15
aI91
aI207
aI39
aI275
aI439
aI2617
aI3093
aI11041
aI24997
asS'd'
I3072
sa(dp6144
S'a'
I10915
sS's'
I15
sS'm'
(lp6145
I1
aI1
aI5
aI3
aI3
aI41
aI13
aI67
aI361
aI497
aI25
aI3807
aI3551
aI9681
aI21043
asS'd'
I3073
sa(dp6146
S'a'
I10924
sS's'
I15
sS'm'
(lp6147
I1
aI3
aI3
aI3
aI11
aI27
aI103
aI59
aI427
aI327
aI1705
aI29
aI8127
aI1641
aI20847
asS'd'
I3074
sa(dp6148
S'a'
I10929
sS's'
I15
sS'm'
(lp6149
I1
aI3
aI7
aI5
aI3
aI37
aI81
aI137
aI225
aI101
aI187
aI3067
aI2491
aI12687
aI16227
asS'd'
I3075
sa(dp6150
S'a'
I10942
sS's'
I15
sS'm'
(lp6151
I1
aI3
aI5
aI15
aI15
aI33
aI69
aI223
aI225
aI771
aI1917
aI2293
aI2889
aI12083
aI23971
asS'd'
I3076
sa(dp6152
S'a'
I10971
sS's'
I15
sS'm'
(lp6153
I1
aI1
aI3
aI5
aI11
aI9
aI121
aI81
aI203
aI425
aI1189
aI2011
aI3041
aI3247
aI739
asS'd'
I3077
sa(dp6154
S'a'
I10992
sS's'
I15
sS'm'
(lp6155
I1
aI3
aI1
aI1
aI13
aI9
aI39
aI169
aI437
aI571
aI1481
aI3355
aI3895
aI8975
aI31031
asS'd'
I3078
sa(dp6156
S'a'
I10995
sS's'
I15
sS'm'
(lp6157
I1
aI3
aI1
aI11
aI1
aI43
aI35
aI35
aI293
aI11
aI657
aI1415
aI5021
aI14463
aI17945
asS'd'
I3079
sa(dp6158
S'a'
I11002
sS's'
I15
sS'm'
(lp6159
I1
aI1
aI5
aI5
aI13
aI47
aI91
aI15
aI159
aI23
aI971
aI3575
aI757
aI13477
aI31757
asS'd'
I3080
sa(dp6160
S'a'
I11010
sS's'
I15
sS'm'
(lp6161
I1
aI1
aI7
aI1
aI5
aI63
aI69
aI27
aI71
aI129
aI123
aI3767
aI89
aI7865
aI1189
asS'd'
I3081
sa(dp6162
S'a'
I11027
sS's'
I15
sS'm'
(lp6163
I1
aI3
aI3
aI5
aI23
aI1
aI83
aI3
aI487
aI491
aI217
aI2583
aI3889
aI15009
aI9227
asS'd'
I3082
sa(dp6164
S'a'
I11029
sS's'
I15
sS'm'
(lp6165
I1
aI3
aI5
aI15
aI25
aI1
aI73
aI107
aI245
aI191
aI1449
aI571
aI1403
aI6953
aI17457
asS'd'
I3083
sa(dp6166
S'a'
I11045
sS's'
I15
sS'm'
(lp6167
I1
aI3
aI3
aI3
aI27
aI19
aI25
aI105
aI207
aI857
aI1161
aI3657
aI2107
aI7955
aI517
asS'd'
I3084
sa(dp6168
S'a'
I11057
sS's'
I15
sS'm'
(lp6169
I1
aI3
aI3
aI9
aI21
aI29
aI5
aI103
aI219
aI35
aI3
aI1635
aI4815
aI15797
aI29839
asS'd'
I3085
sa(dp6170
S'a'
I11070
sS's'
I15
sS'm'
(lp6171
I1
aI1
aI7
aI7
aI3
aI63
aI75
aI77
aI13
aI57
aI603
aI2333
aI7761
aI14397
aI10875
asS'd'
I3086
sa(dp6172
S'a'
I11092
sS's'
I15
sS'm'
(lp6173
I1
aI3
aI7
aI13
aI3
aI11
aI5
aI255
aI1
aI947
aI1695
aI1927
aI7447
aI7407
aI20797
asS'd'
I3087
sa(dp6174
S'a'
I11099
sS's'
I15
sS'm'
(lp6175
I1
aI1
aI5
aI1
aI1
aI21
aI105
aI73
aI429
aI973
aI1801
aI3943
aI6161
aI1309
aI3359
asS'd'
I3088
sa(dp6176
S'a'
I11106
sS's'
I15
sS'm'
(lp6177
I1
aI1
aI3
aI15
aI27
aI9
aI9
aI129
aI117
aI545
aI9
aI1983
aI6351
aI10925
aI27337
asS'd'
I3089
sa(dp6178
S'a'
I11115
sS's'
I15
sS'm'
(lp6179
I1
aI3
aI3
aI5
aI5
aI5
aI13
aI155
aI503
aI875
aI1243
aI2259
aI3445
aI11953
aI6517
asS'd'
I3090
sa(dp6180
S'a'
I11120
sS's'
I15
sS'm'
(lp6181
I1
aI1
aI7
aI3
aI29
aI21
aI121
aI147
aI413
aI423
aI1887
aI2429
aI2765
aI16335
aI3071
asS'd'
I3091
sa(dp6182
S'a'
I11126
sS's'
I15
sS'm'
(lp6183
I1
aI1
aI7
aI9
aI5
aI53
aI41
aI137
aI463
aI583
aI1627
aI1731
aI6675
aI3703
aI8177
asS'd'
I3092
sa(dp6184
S'a'
I11153
sS's'
I15
sS'm'
(lp6185
I1
aI3
aI5
aI11
aI31
aI29
aI67
aI159
aI425
aI25
aI1457
aI139
aI5019
aI701
aI7357
asS'd'
I3093
sa(dp6186
S'a'
I11190
sS's'
I15
sS'm'
(lp6187
I1
aI3
aI1
aI5
aI25
aI15
aI123
aI123
aI245
aI859
aI249
aI2175
aI2137
aI5765
aI4873
asS'd'
I3094
sa(dp6188
S'a'
I11199
sS's'
I15
sS'm'
(lp6189
I1
aI1
aI3
aI5
aI23
aI1
aI111
aI111
aI111
aI469
aI1473
aI1777
aI3579
aI13503
aI2569
asS'd'
I3095
sa(dp6190
S'a'
I11222
sS's'
I15
sS'm'
(lp6191
I1
aI1
aI7
aI3
aI17
aI23
aI51
aI23
aI499
aI135
aI713
aI3317
aI807
aI9589
aI11349
asS'd'
I3096
sa(dp6192
S'a'
I11225
sS's'
I15
sS'm'
(lp6193
I1
aI1
aI1
aI15
aI9
aI51
aI75
aI159
aI359
aI773
aI1521
aI2913
aI5901
aI3047
aI14649
asS'd'
I3097
sa(dp6194
S'a'
I11226
sS's'
I15
sS'm'
(lp6195
I1
aI1
aI3
aI1
aI13
aI61
aI117
aI195
aI49
aI267
aI57
aI1769
aI3621
aI9415
aI29443
asS'd'
I3098
sa(dp6196
S'a'
I11231
sS's'
I15
sS'm'
(lp6197
I1
aI3
aI7
aI11
aI3
aI25
aI33
aI31
aI315
aI191
aI359
aI3399
aI2481
aI13831
aI20205
asS'd'
I3099
sa(dp6198
S'a'
I11244
sS's'
I15
sS'm'
(lp6199
I1
aI3
aI3
aI5
aI31
aI43
aI35
aI125
aI291
aI51
aI1469
aI3857
aI1707
aI2641
aI32137
asS'd'
I3100
sa(dp6200
S'a'
I11259
sS's'
I15
sS'm'
(lp6201
I1
aI3
aI5
aI1
aI25
aI11
aI113
aI137
aI211
aI159
aI1667
aI939
aI6439
aI5337
aI32059
asS'd'
I3101
sa(dp6202
S'a'
I11261
sS's'
I15
sS'm'
(lp6203
I1
aI3
aI3
aI11
aI31
aI61
aI99
aI49
aI383
aI343
aI395
aI51
aI6931
aI16039
aI5901
asS'd'
I3102
sa(dp6204
S'a'
I11270
sS's'
I15
sS'm'
(lp6205
I1
aI1
aI3
aI5
aI9
aI63
aI63
aI49
aI405
aI915
aI1505
aI2141
aI6749
aI7799
aI17313
asS'd'
I3103
sa(dp6206
S'a'
I11273
sS's'
I15
sS'm'
(lp6207
I1
aI3
aI7
aI11
aI15
aI11
aI49
aI161
aI155
aI869
aI121
aI301
aI6561
aI4279
aI15233
asS'd'
I3104
sa(dp6208
S'a'
I11300
sS's'
I15
sS'm'
(lp6209
I1
aI1
aI5
aI13
aI19
aI13
aI103
aI59
aI503
aI293
aI701
aI2527
aI5327
aI13927
aI5025
asS'd'
I3105
sa(dp6210
S'a'
I11307
sS's'
I15
sS'm'
(lp6211
I1
aI1
aI7
aI1
aI1
aI37
aI55
aI155
aI485
aI399
aI855
aI2677
aI5927
aI9657
aI2795
asS'd'
I3106
sa(dp6212
S'a'
I11318
sS's'
I15
sS'm'
(lp6213
I1
aI1
aI1
aI5
aI19
aI15
aI121
aI69
aI385
aI75
aI1567
aI2649
aI5601
aI12981
aI15903
asS'd'
I3107
sa(dp6214
S'a'
I11332
sS's'
I15
sS'm'
(lp6215
I1
aI1
aI1
aI11
aI19
aI21
aI45
aI59
aI505
aI737
aI15
aI1383
aI1177
aI8375
aI15557
asS'd'
I3108
sa(dp6216
S'a'
I11335
sS's'
I15
sS'm'
(lp6217
I1
aI1
aI7
aI13
aI29
aI19
aI123
aI127
aI469
aI773
aI733
aI3289
aI8087
aI5803
aI27897
asS'd'
I3109
sa(dp6218
S'a'
I11341
sS's'
I15
sS'm'
(lp6219
I1
aI3
aI3
aI11
aI19
aI55
aI101
aI67
aI485
aI939
aI607
aI1521
aI6161
aI12235
aI16499
asS'd'
I3110
sa(dp6220
S'a'
I11347
sS's'
I15
sS'm'
(lp6221
I1
aI3
aI5
aI13
aI29
aI31
aI31
aI9
aI453
aI151
aI1055
aI3873
aI405
aI12877
aI29829
asS'd'
I3111
sa(dp6222
S'a'
I11354
sS's'
I15
sS'm'
(lp6223
I1
aI3
aI5
aI1
aI17
aI1
aI17
aI131
aI107
aI1003
aI1749
aI1849
aI6207
aI2153
aI21275
asS'd'
I3112
sa(dp6224
S'a'
I11360
sS's'
I15
sS'm'
(lp6225
I1
aI3
aI7
aI15
aI7
aI25
aI51
aI143
aI51
aI517
aI1841
aI1771
aI5389
aI4633
aI11123
asS'd'
I3113
sa(dp6226
S'a'
I11369
sS's'
I15
sS'm'
(lp6227
I1
aI3
aI7
aI11
aI23
aI7
aI89
aI95
aI403
aI361
aI835
aI585
aI2783
aI8091
aI5141
asS'd'
I3114
sa(dp6228
S'a'
I11372
sS's'
I15
sS'm'
(lp6229
I1
aI3
aI1
aI9
aI1
aI53
aI115
aI11
aI493
aI587
aI305
aI3605
aI1711
aI4169
aI20013
asS'd'
I3115
sa(dp6230
S'a'
I11378
sS's'
I15
sS'm'
(lp6231
I1
aI3
aI7
aI3
aI17
aI59
aI55
aI251
aI49
aI759
aI1227
aI3685
aI7765
aI1445
aI20385
asS'd'
I3116
sa(dp6232
S'a'
I11396
sS's'
I15
sS'm'
(lp6233
I1
aI1
aI5
aI7
aI29
aI55
aI19
aI157
aI129
aI927
aI893
aI1235
aI1955
aI8153
aI2865
asS'd'
I3117
sa(dp6234
S'a'
I11405
sS's'
I15
sS'm'
(lp6235
I1
aI3
aI1
aI15
aI21
aI35
aI81
aI53
aI175
aI939
aI1635
aI3597
aI747
aI14011
aI12867
asS'd'
I3118
sa(dp6236
S'a'
I11417
sS's'
I15
sS'm'
(lp6237
I1
aI3
aI7
aI1
aI27
aI61
aI91
aI73
aI405
aI677
aI603
aI2715
aI7099
aI941
aI24523
asS'd'
I3119
sa(dp6238
S'a'
I11424
sS's'
I15
sS'm'
(lp6239
I1
aI3
aI5
aI9
aI13
aI45
aI35
aI167
aI57
aI483
aI735
aI2777
aI7847
aI6257
aI13109
asS'd'
I3120
sa(dp6240
S'a'
I11427
sS's'
I15
sS'm'
(lp6241
I1
aI3
aI5
aI7
aI1
aI3
aI97
aI13
aI159
aI533
aI1791
aI1061
aI981
aI10795
aI26165
asS'd'
I3121
sa(dp6242
S'a'
I11430
sS's'
I15
sS'm'
(lp6243
I1
aI1
aI5
aI13
aI27
aI5
aI125
aI25
aI251
aI221
aI1909
aI197
aI6987
aI11537
aI15287
asS'd'
I3122
sa(dp6244
S'a'
I11439
sS's'
I15
sS'm'
(lp6245
I1
aI3
aI5
aI5
aI27
aI15
aI1
aI131
aI375
aI923
aI81
aI3153
aI6071
aI2515
aI23729
asS'd'
I3123
sa(dp6246
S'a'
I11442
sS's'
I15
sS'm'
(lp6247
I1
aI3
aI3
aI9
aI9
aI23
aI71
aI13
aI465
aI261
aI937
aI1549
aI5993
aI11325
aI15065
asS'd'
I3124
sa(dp6248
S'a'
I11448
sS's'
I15
sS'm'
(lp6249
I1
aI3
aI1
aI3
aI7
aI63
aI17
aI129
aI435
aI23
aI215
aI2251
aI1561
aI9703
aI26483
asS'd'
I3125
sa(dp6250
S'a'
I11461
sS's'
I15
sS'm'
(lp6251
I1
aI1
aI3
aI1
aI5
aI53
aI77
aI109
aI9
aI605
aI371
aI2081
aI6023
aI7145
aI15837
asS'd'
I3126
sa(dp6252
S'a'
I11468
sS's'
I15
sS'm'
(lp6253
I1
aI3
aI7
aI11
aI27
aI39
aI115
aI47
aI259
aI337
aI1857
aI3465
aI1549
aI7747
aI8525
asS'd'
I3127
sa(dp6254
S'a'
I11471
sS's'
I15
sS'm'
(lp6255
I1
aI3
aI7
aI7
aI29
aI29
aI75
aI77
aI29
aI661
aI899
aI3137
aI2661
aI15271
aI28093
asS'd'
I3128
sa(dp6256
S'a'
I11473
sS's'
I15
sS'm'
(lp6257
I1
aI1
aI1
aI3
aI3
aI3
aI11
aI219
aI39
aI757
aI1465
aI249
aI7445
aI7013
aI15187
asS'd'
I3129
sa(dp6258
S'a'
I11476
sS's'
I15
sS'm'
(lp6259
I1
aI3
aI7
aI15
aI15
aI1
aI39
aI245
aI427
aI1003
aI1493
aI1913
aI6435
aI14787
aI13481
asS'd'
I3130
sa(dp6260
S'a'
I11480
sS's'
I15
sS'm'
(lp6261
I1
aI1
aI7
aI3
aI3
aI37
aI5
aI97
aI343
aI833
aI1379
aI1551
aI5403
aI1843
aI5877
asS'd'
I3131
sa(dp6262
S'a'
I11489
sS's'
I15
sS'm'
(lp6263
I1
aI3
aI1
aI1
aI3
aI17
aI17
aI163
aI339
aI691
aI1707
aI1845
aI5941
aI4259
aI24531
asS'd'
I3132
sa(dp6264
S'a'
I11499
sS's'
I15
sS'm'
(lp6265
I1
aI1
aI1
aI1
aI27
aI21
aI85
aI221
aI71
aI949
aI1753
aI391
aI6349
aI15901
aI20811
asS'd'
I3133
sa(dp6266
S'a'
I11516
sS's'
I15
sS'm'
(lp6267
I1
aI1
aI1
aI5
aI31
aI19
aI45
aI99
aI469
aI783
aI1747
aI3807
aI5889
aI9485
aI13715
asS'd'
I3134
sa(dp6268
S'a'
I11522
sS's'
I15
sS'm'
(lp6269
I1
aI3
aI1
aI9
aI23
aI21
aI39
aI25
aI421
aI713
aI461
aI2857
aI5023
aI5341
aI6409
asS'd'
I3135
sa(dp6270
S'a'
I11531
sS's'
I15
sS'm'
(lp6271
I1
aI3
aI7
aI5
aI25
aI19
aI59
aI147
aI387
aI857
aI375
aI3103
aI1261
aI13697
aI25675
asS'd'
I3136
sa(dp6272
S'a'
I11539
sS's'
I15
sS'm'
(lp6273
I1
aI3
aI5
aI5
aI31
aI21
aI49
aI251
aI463
aI441
aI473
aI3487
aI3915
aI11151
aI17721
asS'd'
I3137
sa(dp6274
S'a'
I11546
sS's'
I15
sS'm'
(lp6275
I1
aI1
aI3
aI9
aI15
aI47
aI81
aI219
aI143
aI141
aI81
aI1705
aI5847
aI3437
aI30521
asS'd'
I3138
sa(dp6276
S'a'
I11551
sS's'
I15
sS'm'
(lp6277
I1
aI1
aI7
aI3
aI25
aI19
aI97
aI41
aI77
aI105
aI1337
aI695
aI7589
aI8587
aI7509
asS'd'
I3139
sa(dp6278
S'a'
I11564
sS's'
I15
sS'm'
(lp6279
I1
aI1
aI5
aI13
aI3
aI11
aI61
aI19
aI139
aI667
aI963
aI1567
aI5715
aI7079
aI15967
asS'd'
I3140
sa(dp6280
S'a'
I11582
sS's'
I15
sS'm'
(lp6281
I1
aI1
aI5
aI5
aI5
aI29
aI67
aI57
aI477
aI173
aI1163
aI727
aI823
aI15635
aI17705
asS'd'
I3141
sa(dp6282
S'a'
I11589
sS's'
I15
sS'm'
(lp6283
I1
aI3
aI7
aI11
aI13
aI39
aI57
aI193
aI73
aI617
aI535
aI1623
aI4581
aI4451
aI2589
asS'd'
I3142
sa(dp6284
S'a'
I11593
sS's'
I15
sS'm'
(lp6285
I1
aI1
aI5
aI5
aI9
aI27
aI75
aI127
aI325
aI413
aI1669
aI1749
aI8045
aI16199
aI12237
asS'd'
I3143
sa(dp6286
S'a'
I11601
sS's'
I15
sS'm'
(lp6287
I1
aI1
aI3
aI1
aI17
aI23
aI27
aI189
aI319
aI953
aI347
aI909
aI4401
aI12791
aI25077
asS'd'
I3144
sa(dp6288
S'a'
I11608
sS's'
I15
sS'm'
(lp6289
I1
aI1
aI3
aI3
aI17
aI51
aI37
aI79
aI301
aI607
aI885
aI1169
aI3275
aI3327
aI20013
asS'd'
I3145
sa(dp6290
S'a'
I11617
sS's'
I15
sS'm'
(lp6291
I1
aI3
aI5
aI3
aI21
aI9
aI99
aI213
aI387
aI889
aI575
aI3591
aI5377
aI2981
aI23989
asS'd'
I3146
sa(dp6292
S'a'
I11630
sS's'
I15
sS'm'
(lp6293
I1
aI3
aI3
aI13
aI11
aI7
aI23
aI255
aI279
aI853
aI453
aI2377
aI8123
aI15393
aI9669
asS'd'
I3147
sa(dp6294
S'a'
I11663
sS's'
I15
sS'm'
(lp6295
I1
aI3
aI1
aI7
aI11
aI9
aI109
aI35
aI405
aI449
aI1967
aI2943
aI3485
aI5031
aI14273
asS'd'
I3148
sa(dp6296
S'a'
I11666
sS's'
I15
sS'm'
(lp6297
I1
aI3
aI3
aI1
aI25
aI27
aI43
aI115
aI435
aI675
aI1937
aI1477
aI4831
aI9417
aI7017
asS'd'
I3149
sa(dp6298
S'a'
I11668
sS's'
I15
sS'm'
(lp6299
I1
aI1
aI7
aI13
aI19
aI45
aI83
aI241
aI487
aI215
aI1453
aI209
aI4061
aI1765
aI15623
asS'd'
I3150
sa(dp6300
S'a'
I11677
sS's'
I15
sS'm'
(lp6301
I1
aI1
aI7
aI7
aI21
aI31
aI95
aI9
aI287
aI1005
aI1933
aI3405
aI6913
aI7733
aI18975
asS'd'
I3151
sa(dp6302
S'a'
I11682
sS's'
I15
sS'm'
(lp6303
I1
aI1
aI1
aI11
aI13
aI11
aI25
aI39
aI283
aI57
aI255
aI2809
aI5767
aI6127
aI6705
asS'd'
I3152
sa(dp6304
S'a'
I11687
sS's'
I15
sS'm'
(lp6305
I1
aI3
aI1
aI11
aI1
aI51
aI73
aI181
aI261
aI215
aI385
aI2777
aI5169
aI12431
aI23563
asS'd'
I3153
sa(dp6306
S'a'
I11696
sS's'
I15
sS'm'
(lp6307
I1
aI3
aI3
aI9
aI9
aI39
aI123
aI197
aI501
aI679
aI109
aI3369
aI4817
aI8855
aI7997
asS'd'
I3154
sa(dp6308
S'a'
I11713
sS's'
I15
sS'm'
(lp6309
I1
aI1
aI5
aI1
aI29
aI61
aI15
aI183
aI453
aI999
aI1211
aI3217
aI8035
aI5153
aI19975
asS'd'
I3155
sa(dp6310
S'a'
I11728
sS's'
I15
sS'm'
(lp6311
I1
aI3
aI7
aI11
aI11
aI21
aI51
aI45
aI379
aI793
aI289
aI309
aI1229
aI7159
aI581
asS'd'
I3156
sa(dp6312
S'a'
I11747
sS's'
I15
sS'm'
(lp6313
I1
aI1
aI3
aI9
aI17
aI11
aI75
aI67
aI289
aI191
aI1083
aI2949
aI6063
aI6611
aI21595
asS'd'
I3157
sa(dp6314
S'a'
I11750
sS's'
I15
sS'm'
(lp6315
I1
aI3
aI7
aI3
aI27
aI45
aI59
aI193
aI485
aI277
aI27
aI1219
aI2389
aI15875
aI6273
asS'd'
I3158
sa(dp6316
S'a'
I11754
sS's'
I15
sS'm'
(lp6317
I1
aI1
aI5
aI3
aI31
aI29
aI65
aI197
aI115
aI163
aI9
aI783
aI5573
aI2833
aI12603
asS'd'
I3159
sa(dp6318
S'a'
I11759
sS's'
I15
sS'm'
(lp6319
I1
aI1
aI3
aI7
aI5
aI53
aI115
aI181
aI175
aI749
aI1335
aI1151
aI2131
aI12467
aI15811
asS'd'
I3160
sa(dp6320
S'a'
I11761
sS's'
I15
sS'm'
(lp6321
I1
aI1
aI1
aI9
aI27
aI39
aI11
aI1
aI443
aI677
aI777
aI1857
aI7459
aI3177
aI3875
asS'd'
I3161
sa(dp6322
S'a'
I11764
sS's'
I15
sS'm'
(lp6323
I1
aI1
aI7
aI7
aI17
aI3
aI23
aI161
aI105
aI603
aI1991
aI3845
aI465
aI11467
aI2077
asS'd'
I3162
sa(dp6324
S'a'
I11767
sS's'
I15
sS'm'
(lp6325
I1
aI1
aI3
aI13
aI5
aI23
aI39
aI35
aI399
aI795
aI265
aI207
aI1823
aI15069
aI31839
asS'd'
I3163
sa(dp6326
S'a'
I11797
sS's'
I15
sS'm'
(lp6327
I1
aI1
aI1
aI1
aI29
aI61
aI89
aI193
aI41
aI99
aI315
aI1021
aI6109
aI12507
aI19973
asS'd'
I3164
sa(dp6328
S'a'
I11804
sS's'
I15
sS'm'
(lp6329
I1
aI1
aI5
aI3
aI13
aI57
aI119
aI251
aI215
aI695
aI1521
aI4081
aI2481
aI657
aI855
asS'd'
I3165
sa(dp6330
S'a'
I11808
sS's'
I15
sS'm'
(lp6331
I1
aI1
aI7
aI3
aI25
aI5
aI3
aI133
aI111
aI385
aI773
aI1027
aI4327
aI3031
aI3537
asS'd'
I3166
sa(dp6332
S'a'
I11831
sS's'
I15
sS'm'
(lp6333
I1
aI3
aI7
aI5
aI5
aI27
aI43
aI117
aI479
aI83
aI1421
aI2791
aI6551
aI6231
aI10353
asS'd'
I3167
sa(dp6334
S'a'
I11832
sS's'
I15
sS'm'
(lp6335
I1
aI1
aI1
aI13
aI3
aI29
aI35
aI71
aI85
aI821
aI1671
aI3057
aI797
aI13683
aI7025
asS'd'
I3168
sa(dp6336
S'a'
I11849
sS's'
I15
sS'm'
(lp6337
I1
aI3
aI7
aI1
aI1
aI47
aI117
aI233
aI141
aI993
aI1381
aI2551
aI1031
aI11765
aI18429
asS'd'
I3169
sa(dp6338
S'a'
I11855
sS's'
I15
sS'm'
(lp6339
I1
aI3
aI1
aI3
aI13
aI3
aI77
aI29
aI35
aI807
aI1109
aI695
aI5605
aI5477
aI449
asS'd'
I3170
sa(dp6340
S'a'
I11863
sS's'
I15
sS'm'
(lp6341
I1
aI1
aI1
aI15
aI21
aI37
aI117
aI105
aI273
aI311
aI1287
aI1415
aI1221
aI1847
aI19487
asS'd'
I3171
sa(dp6342
S'a'
I11880
sS's'
I15
sS'm'
(lp6343
I1
aI3
aI1
aI11
aI21
aI61
aI107
aI225
aI335
aI501
aI1995
aI2399
aI5475
aI12613
aI18877
asS'd'
I3172
sa(dp6344
S'a'
I11883
sS's'
I15
sS'm'
(lp6345
I1
aI3
aI3
aI1
aI31
aI41
aI27
aI205
aI103
aI837
aI639
aI2007
aI2759
aI12471
aI1457
asS'd'
I3173
sa(dp6346
S'a'
I11885
sS's'
I15
sS'm'
(lp6347
I1
aI1
aI7
aI13
aI29
aI39
aI71
aI245
aI105
aI235
aI1277
aI1515
aI6129
aI15947
aI26653
asS'd'
I3174
sa(dp6348
S'a'
I11898
sS's'
I15
sS'm'
(lp6349
I1
aI1
aI7
aI5
aI7
aI13
aI87
aI251
aI315
aI1017
aI587
aI2917
aI5911
aI2919
aI29715
asS'd'
I3175
sa(dp6350
S'a'
I11916
sS's'
I15
sS'm'
(lp6351
I1
aI1
aI1
aI3
aI7
aI19
aI81
aI243
aI177
aI917
aI2023
aI2365
aI7465
aI4901
aI29841
asS'd'
I3176
sa(dp6352
S'a'
I11924
sS's'
I15
sS'm'
(lp6353
I1
aI3
aI5
aI15
aI1
aI31
aI15
aI147
aI285
aI1003
aI1757
aI47
aI6925
aI1197
aI19633
asS'd'
I3177
sa(dp6354
S'a'
I11928
sS's'
I15
sS'm'
(lp6355
I1
aI1
aI5
aI7
aI27
aI25
aI47
aI209
aI85
aI403
aI1399
aI2331
aI3663
aI595
aI13407
asS'd'
I3178
sa(dp6356
S'a'
I11947
sS's'
I15
sS'm'
(lp6357
I1
aI3
aI5
aI9
aI7
aI25
aI7
aI139
aI389
aI817
aI1153
aI1421
aI5735
aI9577
aI10269
asS'd'
I3179
sa(dp6358
S'a'
I11955
sS's'
I15
sS'm'
(lp6359
I1
aI1
aI1
aI9
aI5
aI61
aI49
aI117
aI389
aI541
aI433
aI1405
aI2575
aI223
aI7265
asS'd'
I3180
sa(dp6360
S'a'
I11961
sS's'
I15
sS'm'
(lp6361
I1
aI1
aI5
aI7
aI15
aI1
aI81
aI207
aI435
aI843
aI835
aI3797
aI7637
aI5333
aI31115
asS'd'
I3181
sa(dp6362
S'a'
I11962
sS's'
I15
sS'm'
(lp6363
I1
aI3
aI7
aI11
aI13
aI3
aI47
aI249
aI301
aI715
aI2015
aI3049
aI8155
aI10989
aI26051
asS'd'
I3182
sa(dp6364
S'a'
I11982
sS's'
I15
sS'm'
(lp6365
I1
aI1
aI7
aI7
aI3
aI33
aI119
aI113
aI381
aI575
aI367
aI41
aI3317
aI11727
aI4351
asS'd'
I3183
sa(dp6366
S'a'
I11990
sS's'
I15
sS'm'
(lp6367
I1
aI3
aI3
aI13
aI9
aI3
aI51
aI37
aI173
aI137
aI533
aI1827
aI631
aI10047
aI6267
asS'd'
I3184
sa(dp6368
S'a'
I12010
sS's'
I15
sS'm'
(lp6369
I1
aI3
aI3
aI11
aI17
aI39
aI61
aI245
aI13
aI139
aI1281
aI1319
aI1233
aI13629
aI32269
asS'd'
I3185
sa(dp6370
S'a'
I12018
sS's'
I15
sS'm'
(lp6371
I1
aI1
aI1
aI7
aI15
aI17
aI91
aI109
aI163
aI609
aI11
aI3251
aI7653
aI14035
aI31755
asS'd'
I3186
sa(dp6372
S'a'
I12027
sS's'
I15
sS'm'
(lp6373
I1
aI3
aI3
aI15
aI13
aI21
aI55
aI231
aI385
aI133
aI1833
aI2637
aI6935
aI14303
aI26745
asS'd'
I3187
sa(dp6374
S'a'
I12029
sS's'
I15
sS'm'
(lp6375
I1
aI1
aI1
aI7
aI17
aI41
aI125
aI141
aI89
aI823
aI1411
aI3637
aI6211
aI13323
aI6111
asS'd'
I3188
sa(dp6376
S'a'
I12035
sS's'
I15
sS'm'
(lp6377
I1
aI1
aI1
aI11
aI1
aI21
aI9
aI43
aI97
aI685
aI1223
aI1491
aI121
aI1793
aI2397
asS'd'
I3189
sa(dp6378
S'a'
I12055
sS's'
I15
sS'm'
(lp6379
I1
aI3
aI5
aI5
aI17
aI17
aI5
aI223
aI129
aI865
aI1839
aI1137
aI6391
aI4377
aI9233
asS'd'
I3190
sa(dp6380
S'a'
I12062
sS's'
I15
sS'm'
(lp6381
I1
aI3
aI7
aI15
aI21
aI55
aI5
aI77
aI341
aI637
aI1853
aI1435
aI1195
aI9283
aI21257
asS'd'
I3191
sa(dp6382
S'a'
I12068
sS's'
I15
sS'm'
(lp6383
I1
aI3
aI5
aI1
aI9
aI49
aI43
aI211
aI127
aI655
aI1397
aI1235
aI5279
aI2351
aI24229
asS'd'
I3192
sa(dp6384
S'a'
I12071
sS's'
I15
sS'm'
(lp6385
I1
aI3
aI5
aI3
aI25
aI29
aI13
aI229
aI25
aI675
aI837
aI2753
aI2125
aI9863
aI11293
asS'd'
I3193
sa(dp6386
S'a'
I12072
sS's'
I15
sS'm'
(lp6387
I1
aI3
aI5
aI7
aI23
aI43
aI127
aI1
aI163
aI237
aI337
aI3019
aI7747
aI16227
aI2881
asS'd'
I3194
sa(dp6388
S'a'
I12086
sS's'
I15
sS'm'
(lp6389
I1
aI3
aI5
aI5
aI25
aI9
aI43
aI171
aI421
aI521
aI1885
aI337
aI7873
aI6347
aI13181
asS'd'
I3195
sa(dp6390
S'a'
I12097
sS's'
I15
sS'm'
(lp6391
I1
aI3
aI5
aI5
aI7
aI47
aI107
aI173
aI163
aI191
aI625
aI3389
aI2833
aI7945
aI24787
asS'd'
I3196
sa(dp6392
S'a'
I12098
sS's'
I15
sS'm'
(lp6393
I1
aI1
aI7
aI3
aI27
aI57
aI27
aI209
aI253
aI815
aI301
aI1633
aI3945
aI5051
aI28851
asS'd'
I3197
sa(dp6394
S'a'
I12100
sS's'
I15
sS'm'
(lp6395
I1
aI3
aI7
aI9
aI9
aI51
aI103
aI213
aI437
aI189
aI1857
aI1331
aI5551
aI10641
aI27405
asS'd'
I3198
sa(dp6396
S'a'
I12112
sS's'
I15
sS'm'
(lp6397
I1
aI1
aI5
aI5
aI15
aI1
aI25
aI105
aI117
aI347
aI161
aI3369
aI3589
aI12903
aI23559
asS'd'
I3199
sa(dp6398
S'a'
I12118
sS's'
I15
sS'm'
(lp6399
I1
aI1
aI1
aI5
aI3
aI61
aI93
aI51
aI81
aI281
aI1383
aI745
aI4137
aI2005
aI3635
asS'd'
I3200
sa(dp6400
S'a'
I12133
sS's'
I15
sS'm'
(lp6401
I1
aI3
aI7
aI5
aI13
aI57
aI111
aI211
aI303
aI477
aI359
aI4019
aI6779
aI5129
aI22035
asS'd'
I3201
sa(dp6402
S'a'
I12134
sS's'
I15
sS'm'
(lp6403
I1
aI1
aI1
aI7
aI17
aI29
aI113
aI113
aI201
aI531
aI749
aI739
aI2879
aI3315
aI18733
asS'd'
I3202
sa(dp6404
S'a'
I12137
sS's'
I15
sS'm'
(lp6405
I1
aI3
aI7
aI13
aI21
aI55
aI21
aI183
aI359
aI75
aI377
aI2211
aI4281
aI14317
aI28307
asS'd'
I3203
sa(dp6406
S'a'
I12161
sS's'
I15
sS'm'
(lp6407
I1
aI3
aI7
aI1
aI21
aI1
aI49
aI213
aI317
aI75
aI113
aI1741
aI7963
aI12785
aI11571
asS'd'
I3204
sa(dp6408
S'a'
I12162
sS's'
I15
sS'm'
(lp6409
I1
aI3
aI7
aI9
aI11
aI31
aI29
aI101
aI261
aI141
aI715
aI2727
aI8187
aI2075
aI32433
asS'd'
I3205
sa(dp6410
S'a'
I12171
sS's'
I15
sS'm'
(lp6411
I1
aI3
aI7
aI3
aI23
aI9
aI17
aI143
aI385
aI211
aI593
aI241
aI6567
aI10777
aI6677
asS'd'
I3206
sa(dp6412
S'a'
I12174
sS's'
I15
sS'm'
(lp6413
I1
aI1
aI3
aI15
aI3
aI17
aI117
aI99
aI91
aI793
aI989
aI2421
aI5643
aI16103
aI9759
asS'd'
I3207
sa(dp6414
S'a'
I12185
sS's'
I15
sS'm'
(lp6415
I1
aI3
aI7
aI11
aI23
aI43
aI107
aI35
aI421
aI431
aI743
aI853
aI7939
aI12169
aI4199
asS'd'
I3208
sa(dp6416
S'a'
I12204
sS's'
I15
sS'm'
(lp6417
I1
aI1
aI1
aI11
aI21
aI53
aI17
aI203
aI123
aI395
aI59
aI929
aI255
aI7585
aI10945
asS'd'
I3209
sa(dp6418
S'a'
I12212
sS's'
I15
sS'm'
(lp6419
I1
aI3
aI3
aI15
aI17
aI57
aI57
aI133
aI67
aI71
aI1685
aI903
aI4079
aI15423
aI26495
asS'd'
I3210
sa(dp6420
S'a'
I12215
sS's'
I15
sS'm'
(lp6421
I1
aI1
aI1
aI15
aI3
aI47
aI95
aI39
aI405
aI407
aI1557
aI3001
aI6225
aI15187
aI5663
asS'd'
I3211
sa(dp6422
S'a'
I12216
sS's'
I15
sS'm'
(lp6423
I1
aI3
aI5
aI5
aI13
aI47
aI33
aI61
aI375
aI1023
aI1981
aI2773
aI2375
aI11321
aI17731
asS'd'
I3212
sa(dp6424
S'a'
I12253
sS's'
I15
sS'm'
(lp6425
I1
aI3
aI5
aI9
aI17
aI59
aI117
aI95
aI493
aI149
aI1269
aI2865
aI369
aI2109
aI24601
asS'd'
I3213
sa(dp6426
S'a'
I12260
sS's'
I15
sS'm'
(lp6427
I1
aI3
aI5
aI13
aI17
aI63
aI67
aI247
aI95
aI721
aI67
aI305
aI6179
aI15399
aI32559
asS'd'
I3214
sa(dp6428
S'a'
I12277
sS's'
I15
sS'm'
(lp6429
I1
aI1
aI5
aI1
aI3
aI21
aI41
aI15
aI453
aI475
aI2017
aI3193
aI5903
aI897
aI4237
asS'd'
I3215
sa(dp6430
S'a'
I12289
sS's'
I15
sS'm'
(lp6431
I1
aI1
aI5
aI3
aI15
aI41
aI1
aI141
aI441
aI575
aI155
aI3791
aI7711
aI11231
aI24611
asS'd'
I3216
sa(dp6432
S'a'
I12295
sS's'
I15
sS'm'
(lp6433
I1
aI3
aI7
aI1
aI17
aI53
aI27
aI169
aI31
aI437
aI963
aI1793
aI7777
aI1917
aI29311
asS'd'
I3217
sa(dp6434
S'a'
I12314
sS's'
I15
sS'm'
(lp6435
I1
aI3
aI3
aI13
aI9
aI27
aI77
aI87
aI329
aI885
aI749
aI1713
aI6013
aI6921
aI629
asS'd'
I3218
sa(dp6436
S'a'
I12323
sS's'
I15
sS'm'
(lp6437
I1
aI3
aI5
aI13
aI3
aI7
aI53
aI27
aI353
aI267
aI925
aI2141
aI439
aI15175
aI30851
asS'd'
I3219
sa(dp6438
S'a'
I12325
sS's'
I15
sS'm'
(lp6439
I1
aI3
aI3
aI13
aI17
aI57
aI35
aI101
aI265
aI901
aI1825
aI2159
aI6149
aI5967
aI24023
asS'd'
I3220
sa(dp6440
S'a'
I12335
sS's'
I15
sS'm'
(lp6441
I1
aI1
aI5
aI11
aI13
aI51
aI99
aI111
aI193
aI415
aI1541
aI2929
aI5045
aI3147
aI12587
asS'd'
I3221
sa(dp6442
S'a'
I12349
sS's'
I15
sS'm'
(lp6443
I1
aI3
aI7
aI11
aI15
aI9
aI33
aI17
aI511
aI815
aI299
aI1077
aI6171
aI10451
aI15039
asS'd'
I3222
sa(dp6444
S'a'
I12358
sS's'
I15
sS'm'
(lp6445
I1
aI1
aI1
aI15
aI25
aI63
aI51
aI137
aI449
aI951
aI1051
aI1101
aI4725
aI2463
aI7355
asS'd'
I3223
sa(dp6446
S'a'
I12372
sS's'
I15
sS'm'
(lp6447
I1
aI1
aI1
aI7
aI27
aI63
aI29
aI179
aI317
aI521
aI1459
aI827
aI6599
aI13459
aI15439
asS'd'
I3224
sa(dp6448
S'a'
I12376
sS's'
I15
sS'm'
(lp6449
I1
aI3
aI3
aI15
aI17
aI31
aI37
aI191
aI229
aI245
aI181
aI941
aI5761
aI1849
aI31599
asS'd'
I3225
sa(dp6450
S'a'
I12379
sS's'
I15
sS'm'
(lp6451
I1
aI1
aI1
aI9
aI27
aI45
aI67
aI239
aI481
aI615
aI1667
aI3751
aI8141
aI10013
aI2125
asS'd'
I3226
sa(dp6452
S'a'
I12386
sS's'
I15
sS'm'
(lp6453
I1
aI1
aI1
aI1
aI13
aI51
aI117
aI135
aI73
aI151
aI1291
aI2541
aI1411
aI3767
aI26949
asS'd'
I3227
sa(dp6454
S'a'
I12395
sS's'
I15
sS'm'
(lp6455
I1
aI3
aI1
aI9
aI7
aI11
aI21
aI187
aI243
aI857
aI1951
aI865
aI7273
aI2041
aI8155
asS'd'
I3228
sa(dp6456
S'a'
I12416
sS's'
I15
sS'm'
(lp6457
I1
aI1
aI3
aI3
aI19
aI33
aI89
aI115
aI455
aI137
aI707
aI1867
aI4221
aI2433
aI9119
asS'd'
I3229
sa(dp6458
S'a'
I12421
sS's'
I15
sS'm'
(lp6459
I1
aI1
aI3
aI11
aI5
aI3
aI121
aI1
aI71
aI951
aI603
aI3873
aI723
aI3285
aI19289
asS'd'
I3230
sa(dp6460
S'a'
I12440
sS's'
I15
sS'm'
(lp6461
I1
aI3
aI7
aI15
aI21
aI1
aI117
aI17
aI455
aI519
aI731
aI3003
aI5741
aI9557
aI29163
asS'd'
I3231
sa(dp6462
S'a'
I12452
sS's'
I15
sS'm'
(lp6463
I1
aI1
aI3
aI13
aI25
aI5
aI43
aI147
aI209
aI895
aI255
aI1231
aI241
aI487
aI15593
asS'd'
I3232
sa(dp6464
S'a'
I12455
sS's'
I15
sS'm'
(lp6465
I1
aI1
aI3
aI13
aI7
aI1
aI89
aI187
aI217
aI927
aI2029
aI3521
aI2777
aI8103
aI22819
asS'd'
I3233
sa(dp6466
S'a'
I12456
sS's'
I15
sS'm'
(lp6467
I1
aI1
aI7
aI11
aI7
aI33
aI3
aI73
aI5
aI489
aI227
aI2259
aI7031
aI6425
aI26135
asS'd'
I3234
sa(dp6468
S'a'
I12462
sS's'
I15
sS'm'
(lp6469
I1
aI3
aI3
aI7
aI31
aI19
aI97
aI201
aI455
aI819
aI945
aI2771
aI8083
aI8711
aI2835
asS'd'
I3235
sa(dp6470
S'a'
I12467
sS's'
I15
sS'm'
(lp6471
I1
aI1
aI1
aI5
aI15
aI45
aI43
aI157
aI245
aI967
aI877
aI2289
aI4499
aI9891
aI18827
asS'd'
I3236
sa(dp6472
S'a'
I12479
sS's'
I15
sS'm'
(lp6473
I1
aI3
aI1
aI7
aI21
aI59
aI123
aI63
aI231
aI485
aI1781
aI1211
aI4597
aI5269
aI1607
asS'd'
I3237
sa(dp6474
S'a'
I12505
sS's'
I15
sS'm'
(lp6475
I1
aI1
aI1
aI13
aI23
aI39
aI105
aI55
aI249
aI991
aI1625
aI3089
aI3825
aI4275
aI29139
asS'd'
I3238
sa(dp6476
S'a'
I12521
sS's'
I15
sS'm'
(lp6477
I1
aI3
aI3
aI1
aI29
aI29
aI55
aI169
aI13
aI895
aI1355
aI1101
aI6063
aI12935
aI23215
asS'd'
I3239
sa(dp6478
S'a'
I12535
sS's'
I15
sS'm'
(lp6479
I1
aI1
aI5
aI5
aI31
aI49
aI99
aI137
aI209
aI1017
aI1179
aI3931
aI637
aI14131
aI19285
asS'd'
I3240
sa(dp6480
S'a'
I12547
sS's'
I15
sS'm'
(lp6481
I1
aI1
aI1
aI1
aI3
aI11
aI119
aI11
aI215
aI337
aI243
aI3883
aI3807
aI7335
aI11901
asS'd'
I3241
sa(dp6482
S'a'
I12556
sS's'
I15
sS'm'
(lp6483
I1
aI3
aI7
aI3
aI7
aI27
aI71
aI225
aI219
aI367
aI1213
aI2739
aI1185
aI10175
aI21313
asS'd'
I3242
sa(dp6484
S'a'
I12561
sS's'
I15
sS'm'
(lp6485
I1
aI3
aI7
aI13
aI7
aI49
aI23
aI223
aI61
aI1011
aI797
aI1335
aI6711
aI5961
aI5605
asS'd'
I3243
sa(dp6486
S'a'
I12568
sS's'
I15
sS'm'
(lp6487
I1
aI3
aI3
aI11
aI19
aI37
aI1
aI149
aI39
aI661
aI929
aI2125
aI2299
aI5181
aI28083
asS'd'
I3244
sa(dp6488
S'a'
I12578
sS's'
I15
sS'm'
(lp6489
I1
aI3
aI3
aI13
aI13
aI9
aI67
aI21
aI463
aI279
aI529
aI523
aI6705
aI11011
aI31695
asS'd'
I3245
sa(dp6490
S'a'
I12583
sS's'
I15
sS'm'
(lp6491
I1
aI3
aI1
aI5
aI13
aI1
aI123
aI3
aI291
aI625
aI1949
aI2713
aI5917
aI10343
aI13627
asS'd'
I3246
sa(dp6492
S'a'
I12595
sS's'
I15
sS'm'
(lp6493
I1
aI1
aI3
aI9
aI27
aI41
aI3
aI207
aI103
aI265
aI811
aI549
aI6109
aI313
aI8889
asS'd'
I3247
sa(dp6494
S'a'
I12604
sS's'
I15
sS'm'
(lp6495
I1
aI3
aI3
aI13
aI23
aI43
aI99
aI33
aI279
aI463
aI955
aI793
aI4113
aI10615
aI16957
asS'd'
I3248
sa(dp6496
S'a'
I12610
sS's'
I15
sS'm'
(lp6497
I1
aI1
aI5
aI7
aI11
aI49
aI79
aI45
aI17
aI937
aI359
aI1037
aI1099
aI3003
aI31561
asS'd'
I3249
sa(dp6498
S'a'
I12621
sS's'
I15
sS'm'
(lp6499
I1
aI1
aI1
aI7
aI3
aI45
aI111
aI35
aI109
aI983
aI53
aI4057
aI7349
aI3599
aI2209
asS'd'
I3250
sa(dp6500
S'a'
I12622
sS's'
I15
sS'm'
(lp6501
I1
aI3
aI7
aI11
aI9
aI43
aI27
aI9
aI85
aI529
aI1497
aI347
aI759
aI12449
aI11373
asS'd'
I3251
sa(dp6502
S'a'
I12624
sS's'
I15
sS'm'
(lp6503
I1
aI1
aI3
aI9
aI17
aI1
aI49
aI31
aI367
aI813
aI1385
aI2025
aI773
aI4679
aI4543
asS'd'
I3252
sa(dp6504
S'a'
I12629
sS's'
I15
sS'm'
(lp6505
I1
aI1
aI5
aI15
aI15
aI9
aI43
aI97
aI239
aI995
aI1037
aI841
aI4167
aI12113
aI23765
asS'd'
I3253
sa(dp6506
S'a'
I12630
sS's'
I15
sS'm'
(lp6507
I1
aI3
aI5
aI9
aI29
aI53
aI123
aI49
aI221
aI113
aI1157
aI73
aI6087
aI1363
aI11029
asS'd'
I3254
sa(dp6508
S'a'
I12639
sS's'
I15
sS'm'
(lp6509
I1
aI3
aI1
aI13
aI3
aI15
aI69
aI199
aI279
aI919
aI5
aI161
aI4817
aI15031
aI121
asS'd'
I3255
sa(dp6510
S'a'
I12640
sS's'
I15
sS'm'
(lp6511
I1
aI3
aI1
aI9
aI3
aI31
aI117
aI77
aI393
aI241
aI645
aI3181
aI1067
aI15879
aI2037
asS'd'
I3256
sa(dp6512
S'a'
I12650
sS's'
I15
sS'm'
(lp6513
I1
aI3
aI3
aI15
aI3
aI63
aI57
aI33
aI117
aI789
aI941
aI1301
aI5865
aI12693
aI3523
asS'd'
I3257
sa(dp6514
S'a'
I12679
sS's'
I15
sS'm'
(lp6515
I1
aI1
aI5
aI13
aI3
aI61
aI51
aI151
aI175
aI305
aI95
aI1557
aI6567
aI7841
aI13903
asS'd'
I3258
sa(dp6516
S'a'
I12680
sS's'
I15
sS'm'
(lp6517
I1
aI3
aI3
aI5
aI15
aI25
aI127
aI79
aI245
aI767
aI645
aI3933
aI1357
aI12579
aI4067
asS'd'
I3259
sa(dp6518
S'a'
I12698
sS's'
I15
sS'm'
(lp6519
I1
aI3
aI5
aI11
aI21
aI31
aI13
aI251
aI127
aI231
aI1795
aI2627
aI1191
aI3363
aI23543
asS'd'
I3260
sa(dp6520
S'a'
I12716
sS's'
I15
sS'm'
(lp6521
I1
aI1
aI3
aI5
aI7
aI49
aI121
aI57
aI131
aI481
aI1879
aI525
aI5225
aI337
aI1957
asS'd'
I3261
sa(dp6522
S'a'
I12721
sS's'
I15
sS'm'
(lp6523
I1
aI1
aI5
aI13
aI9
aI55
aI27
aI37
aI211
aI125
aI119
aI3373
aI251
aI12357
aI13975
asS'd'
I3262
sa(dp6524
S'a'
I12722
sS's'
I15
sS'm'
(lp6525
I1
aI3
aI3
aI15
aI1
aI51
aI91
aI119
aI233
aI993
aI203
aI1635
aI1167
aI6327
aI29119
asS'd'
I3263
sa(dp6526
S'a'
I12731
sS's'
I15
sS'm'
(lp6527
I1
aI1
aI7
aI1
aI13
aI5
aI23
aI253
aI121
aI989
aI1105
aI3321
aI3221
aI6073
aI21185
asS'd'
I3264
sa(dp6528
S'a'
I12742
sS's'
I15
sS'm'
(lp6529
I1
aI1
aI3
aI15
aI13
aI49
aI121
aI247
aI247
aI133
aI485
aI1067
aI7875
aI411
aI7647
asS'd'
I3265
sa(dp6530
S'a'
I12745
sS's'
I15
sS'm'
(lp6531
I1
aI3
aI7
aI13
aI31
aI37
aI127
aI241
aI145
aI133
aI53
aI267
aI2029
aI3703
aI16123
asS'd'
I3266
sa(dp6532
S'a'
I12751
sS's'
I15
sS'm'
(lp6533
I1
aI3
aI1
aI15
aI15
aI9
aI15
aI89
aI35
aI367
aI401
aI61
aI1953
aI7873
aI17861
asS'd'
I3267
sa(dp6534
S'a'
I12759
sS's'
I15
sS'm'
(lp6535
I1
aI1
aI1
aI1
aI1
aI41
aI71
aI249
aI213
aI779
aI1385
aI1767
aI999
aI15151
aI16647
asS'd'
I3268
sa(dp6536
S'a'
I12763
sS's'
I15
sS'm'
(lp6537
I1
aI3
aI7
aI13
aI31
aI23
aI123
aI235
aI343
aI949
aI309
aI3777
aI3587
aI853
aI19779
asS'd'
I3269
sa(dp6538
S'a'
I12769
sS's'
I15
sS'm'
(lp6539
I1
aI1
aI3
aI13
aI29
aI35
aI5
aI37
aI63
aI757
aI303
aI1579
aI3443
aI243
aI11873
asS'd'
I3270
sa(dp6540
S'a'
I12781
sS's'
I15
sS'm'
(lp6541
I1
aI3
aI1
aI9
aI19
aI49
aI81
aI53
aI11
aI901
aI1857
aI147
aI3103
aI14019
aI21
asS'd'
I3271
sa(dp6542
S'a'
I12793
sS's'
I15
sS'm'
(lp6543
I1
aI3
aI7
aI13
aI3
aI39
aI99
aI99
aI45
aI91
aI1567
aI551
aI3129
aI4809
aI29057
asS'd'
I3272
sa(dp6544
S'a'
I12799
sS's'
I15
sS'm'
(lp6545
I1
aI3
aI7
aI3
aI3
aI27
aI17
aI231
aI377
aI381
aI1479
aI2525
aI2595
aI2799
aI25737
asS'd'
I3273
sa(dp6546
S'a'
I12815
sS's'
I15
sS'm'
(lp6547
I1
aI3
aI5
aI15
aI15
aI25
aI103
aI215
aI301
aI59
aI1417
aI981
aI7579
aI12731
aI22329
asS'd'
I3274
sa(dp6548
S'a'
I12824
sS's'
I15
sS'm'
(lp6549
I1
aI1
aI1
aI13
aI5
aI31
aI61
aI31
aI349
aI925
aI1301
aI685
aI435
aI11567
aI10715
asS'd'
I3275
sa(dp6550
S'a'
I12836
sS's'
I15
sS'm'
(lp6551
I1
aI1
aI7
aI9
aI19
aI57
aI109
aI1
aI37
aI377
aI1015
aI2273
aI6741
aI3191
aI15949
asS'd'
I3276
sa(dp6552
S'a'
I12845
sS's'
I15
sS'm'
(lp6553
I1
aI3
aI3
aI13
aI3
aI23
aI103
aI127
aI11
aI59
aI1847
aI1175
aI425
aI3423
aI20643
asS'd'
I3277
sa(dp6554
S'a'
I12853
sS's'
I15
sS'm'
(lp6555
I1
aI3
aI3
aI7
aI3
aI11
aI105
aI141
aI55
aI217
aI1427
aI477
aI667
aI9403
aI11905
asS'd'
I3278
sa(dp6556
S'a'
I12854
sS's'
I15
sS'm'
(lp6557
I1
aI3
aI3
aI5
aI3
aI27
aI11
aI187
aI495
aI907
aI1925
aI445
aI6639
aI8159
aI15225
asS'd'
I3279
sa(dp6558
S'a'
I12857
sS's'
I15
sS'm'
(lp6559
I1
aI3
aI1
aI5
aI27
aI31
aI77
aI213
aI73
aI343
aI1123
aI3609
aI2431
aI15329
aI32165
asS'd'
I3280
sa(dp6560
S'a'
I12866
sS's'
I15
sS'm'
(lp6561
I1
aI1
aI7
aI5
aI1
aI11
aI105
aI139
aI485
aI1007
aI709
aI3509
aI5231
aI11717
aI31433
asS'd'
I3281
sa(dp6562
S'a'
I12872
sS's'
I15
sS'm'
(lp6563
I1
aI1
aI3
aI15
aI23
aI49
aI95
aI169
aI399
aI1019
aI19
aI2013
aI5311
aI7951
aI22609
asS'd'
I3282
sa(dp6564
S'a'
I12875
sS's'
I15
sS'm'
(lp6565
I1
aI3
aI1
aI7
aI13
aI3
aI29
aI203
aI209
aI701
aI1791
aI2615
aI5351
aI4237
aI12565
asS'd'
I3283
sa(dp6566
S'a'
I12878
sS's'
I15
sS'm'
(lp6567
I1
aI3
aI1
aI15
aI27
aI11
aI91
aI31
aI205
aI205
aI1683
aI901
aI5129
aI6049
aI11865
asS'd'
I3284
sa(dp6568
S'a'
I12880
sS's'
I15
sS'm'
(lp6569
I1
aI1
aI7
aI7
aI27
aI59
aI21
aI3
aI209
aI79
aI769
aI4013
aI2041
aI2645
aI11561
asS'd'
I3285
sa(dp6570
S'a'
I12885
sS's'
I15
sS'm'
(lp6571
I1
aI3
aI7
aI11
aI5
aI45
aI39
aI243
aI185
aI871
aI795
aI1845
aI8043
aI6285
aI20991
asS'd'
I3286
sa(dp6572
S'a'
I12901
sS's'
I15
sS'm'
(lp6573
I1
aI1
aI5
aI7
aI13
aI7
aI15
aI165
aI349
aI179
aI789
aI1269
aI3787
aI5429
aI26567
asS'd'
I3287
sa(dp6574
S'a'
I12902
sS's'
I15
sS'm'
(lp6575
I1
aI3
aI3
aI13
aI31
aI23
aI75
aI41
aI177
aI735
aI1889
aI4039
aI3519
aI15003
aI965
asS'd'
I3288
sa(dp6576
S'a'
I12920
sS's'
I15
sS'm'
(lp6577
I1
aI3
aI1
aI7
aI15
aI57
aI15
aI139
aI27
aI469
aI1003
aI691
aI7893
aI9643
aI30983
asS'd'
I3289
sa(dp6578
S'a'
I12926
sS's'
I15
sS'm'
(lp6579
I1
aI3
aI1
aI13
aI23
aI27
aI3
aI237
aI233
aI875
aI99
aI883
aI6167
aI5463
aI6245
asS'd'
I3290
sa(dp6580
S'a'
I12929
sS's'
I15
sS'm'
(lp6581
I1
aI1
aI5
aI13
aI25
aI57
aI79
aI51
aI147
aI619
aI1147
aI279
aI6583
aI1939
aI477
asS'd'
I3291
sa(dp6582
S'a'
I12939
sS's'
I15
sS'm'
(lp6583
I1
aI3
aI5
aI5
aI31
aI61
aI125
aI163
aI213
aI699
aI583
aI3865
aI615
aI9707
aI11651
asS'd'
I3292
sa(dp6584
S'a'
I12941
sS's'
I15
sS'm'
(lp6585
I1
aI1
aI5
aI1
aI5
aI21
aI93
aI239
aI31
aI641
aI777
aI27
aI5247
aI8993
aI21053
asS'd'
I3293
sa(dp6586
S'a'
I12950
sS's'
I15
sS'm'
(lp6587
I1
aI3
aI7
aI9
aI1
aI13
aI61
aI57
aI503
aI453
aI83
aI3271
aI2845
aI1121
aI18639
asS'd'
I3294
sa(dp6588
S'a'
I12953
sS's'
I15
sS'm'
(lp6589
I1
aI1
aI7
aI5
aI29
aI53
aI13
aI219
aI379
aI441
aI821
aI3179
aI4877
aI2535
aI7557
asS'd'
I3295
sa(dp6590
S'a'
I12992
sS's'
I15
sS'm'
(lp6591
I1
aI1
aI7
aI13
aI9
aI53
aI17
aI183
aI265
aI393
aI587
aI2753
aI6453
aI7135
aI24737
asS'd'
I3296
sa(dp6592
S'a'
I13002
sS's'
I15
sS'm'
(lp6593
I1
aI1
aI1
aI13
aI11
aI23
aI73
aI109
aI393
aI1013
aI559
aI755
aI7291
aI6631
aI26509
asS'd'
I3297
sa(dp6594
S'a'
I13010
sS's'
I15
sS'm'
(lp6595
I1
aI3
aI1
aI5
aI5
aI15
aI107
aI103
aI355
aI307
aI1559
aI837
aI5413
aI5285
aI17489
asS'd'
I3298
sa(dp6596
S'a'
I13058
sS's'
I15
sS'm'
(lp6597
I1
aI1
aI5
aI7
aI17
aI21
aI21
aI23
aI109
aI709
aI1947
aI3585
aI3629
aI4669
aI949
asS'd'
I3299
sa(dp6598
S'a'
I13072
sS's'
I15
sS'm'
(lp6599
I1
aI3
aI7
aI1
aI9
aI33
aI85
aI147
aI467
aI259
aI1913
aI199
aI7399
aI9551
aI22387
asS'd'
I3300
sa(dp6600
S'a'
I13084
sS's'
I15
sS'm'
(lp6601
I1
aI3
aI5
aI11
aI15
aI53
aI23
aI41
aI249
aI515
aI1161
aI2467
aI1299
aI7449
aI2613
asS'd'
I3301
sa(dp6602
S'a'
I13087
sS's'
I15
sS'm'
(lp6603
I1
aI1
aI5
aI5
aI5
aI29
aI91
aI139
aI487
aI545
aI321
aI3035
aI4545
aI6747
aI21673
asS'd'
I3302
sa(dp6604
S'a'
I13091
sS's'
I15
sS'm'
(lp6605
I1
aI1
aI3
aI13
aI23
aI49
aI95
aI103
aI25
aI119
aI469
aI2515
aI2551
aI841
aI25089
asS'd'
I3303
sa(dp6606
S'a'
I13097
sS's'
I15
sS'm'
(lp6607
I1
aI1
aI5
aI7
aI11
aI31
aI31
aI197
aI245
aI715
aI257
aI4043
aI8099
aI11531
aI5617
asS'd'
I3304
sa(dp6608
S'a'
I13108
sS's'
I15
sS'm'
(lp6609
I1
aI1
aI3
aI3
aI19
aI7
aI9
aI179
aI103
aI995
aI191
aI179
aI3843
aI5215
aI27639
asS'd'
I3305
sa(dp6610
S'a'
I13123
sS's'
I15
sS'm'
(lp6611
I1
aI3
aI1
aI7
aI23
aI59
aI25
aI65
aI399
aI211
aI1453
aI3511
aI7203
aI16015
aI32197
asS'd'
I3306
sa(dp6612
S'a'
I13149
sS's'
I15
sS'm'
(lp6613
I1
aI3
aI3
aI5
aI9
aI35
aI109
aI67
aI197
aI449
aI643
aI519
aI5751
aI15551
aI11331
asS'd'
I3307
sa(dp6614
S'a'
I13150
sS's'
I15
sS'm'
(lp6615
I1
aI3
aI5
aI3
aI1
aI17
aI53
aI201
aI265
aI351
aI467
aI911
aI1117
aI7183
aI20371
asS'd'
I3308
sa(dp6616
S'a'
I13163
sS's'
I15
sS'm'
(lp6617
I1
aI1
aI7
aI7
aI27
aI17
aI93
aI81
aI227
aI619
aI1191
aI735
aI815
aI12615
aI2719
asS'd'
I3309
sa(dp6618
S'a'
I13166
sS's'
I15
sS'm'
(lp6619
I1
aI3
aI1
aI15
aI19
aI3
aI83
aI75
aI343
aI297
aI1019
aI3469
aI4383
aI13299
aI29755
asS'd'
I3310
sa(dp6620
S'a'
I13178
sS's'
I15
sS'm'
(lp6621
I1
aI1
aI5
aI3
aI13
aI55
aI119
aI169
aI85
aI595
aI299
aI2469
aI5625
aI2877
aI16117
asS'd'
I3311
sa(dp6622
S'a'
I13180
sS's'
I15
sS'm'
(lp6623
I1
aI1
aI3
aI5
aI15
aI17
aI61
aI161
aI47
aI393
aI143
aI867
aI5517
aI9495
aI12795
asS'd'
I3312
sa(dp6624
S'a'
I13184
sS's'
I15
sS'm'
(lp6625
I1
aI3
aI5
aI1
aI27
aI31
aI113
aI125
aI251
aI687
aI969
aI1473
aI2245
aI6355
aI13655
asS'd'
I3313
sa(dp6626
S'a'
I13204
sS's'
I15
sS'm'
(lp6627
I1
aI1
aI1
aI5
aI5
aI37
aI29
aI133
aI443
aI899
aI277
aI2353
aI7223
aI4459
aI19159
asS'd'
I3314
sa(dp6628
S'a'
I13238
sS's'
I15
sS'm'
(lp6629
I1
aI1
aI3
aI9
aI19
aI27
aI53
aI145
aI195
aI167
aI2045
aI447
aI1803
aI1895
aI8431
asS'd'
I3315
sa(dp6630
S'a'
I13242
sS's'
I15
sS'm'
(lp6631
I1
aI1
aI3
aI9
aI5
aI27
aI91
aI147
aI233
aI451
aI475
aI27
aI4629
aI16181
aI16437
asS'd'
I3316
sa(dp6632
S'a'
I13249
sS's'
I15
sS'm'
(lp6633
I1
aI3
aI5
aI3
aI29
aI17
aI53
aI167
aI433
aI689
aI1131
aI2985
aI1553
aI11697
aI6993
asS'd'
I3317
sa(dp6634
S'a'
I13255
sS's'
I15
sS'm'
(lp6635
I1
aI3
aI3
aI13
aI21
aI43
aI69
aI229
aI399
aI525
aI179
aI237
aI7017
aI5703
aI17653
asS'd'
I3318
sa(dp6636
S'a'
I13269
sS's'
I15
sS'm'
(lp6637
I1
aI1
aI3
aI15
aI13
aI39
aI75
aI163
aI229
aI875
aI197
aI477
aI3667
aI15501
aI15801
asS'd'
I3319
sa(dp6638
S'a'
I13270
sS's'
I15
sS'm'
(lp6639
I1
aI1
aI7
aI15
aI15
aI51
aI81
aI187
aI487
aI673
aI865
aI1915
aI1009
aI5935
aI8097
asS'd'
I3320
sa(dp6640
S'a'
I13274
sS's'
I15
sS'm'
(lp6641
I1
aI3
aI5
aI5
aI7
aI3
aI63
aI77
aI495
aI815
aI391
aI2321
aI1007
aI15661
aI30715
asS'd'
I3321
sa(dp6642
S'a'
I13285
sS's'
I15
sS'm'
(lp6643
I1
aI1
aI7
aI3
aI17
aI25
aI83
aI173
aI173
aI911
aI1373
aI2957
aI4549
aI15977
aI17695
asS'd'
I3322
sa(dp6644
S'a'
I13289
sS's'
I15
sS'm'
(lp6645
I1
aI1
aI7
aI13
aI13
aI23
aI77
aI147
aI497
aI1003
aI1687
aI1795
aI1393
aI1881
aI8479
asS'd'
I3323
sa(dp6646
S'a'
I13298
sS's'
I15
sS'm'
(lp6647
I1
aI3
aI7
aI11
aI27
aI43
aI97
aI25
aI61
aI457
aI203
aI2573
aI5943
aI15021
aI4003
asS'd'
I3324
sa(dp6648
S'a'
I13307
sS's'
I15
sS'm'
(lp6649
I1
aI3
aI3
aI13
aI9
aI37
aI37
aI25
aI219
aI889
aI1535
aI2791
aI4531
aI13679
aI12663
asS'd'
I3325
sa(dp6650
S'a'
I13312
sS's'
I15
sS'm'
(lp6651
I1
aI1
aI3
aI1
aI17
aI7
aI51
aI123
aI89
aI887
aI1467
aI1645
aI3767
aI6383
aI30837
asS'd'
I3326
sa(dp6652
S'a'
I13335
sS's'
I15
sS'm'
(lp6653
I1
aI3
aI3
aI1
aI21
aI47
aI5
aI151
aI83
aI257
aI569
aI2711
aI637
aI12569
aI16893
asS'd'
I3327
sa(dp6654
S'a'
I13345
sS's'
I15
sS'm'
(lp6655
I1
aI3
aI7
aI1
aI31
aI37
aI73
aI3
aI115
aI919
aI1817
aI2483
aI4811
aI15245
aI4375
asS'd'
I3328
sa(dp6656
S'a'
I13357
sS's'
I15
sS'm'
(lp6657
I1
aI1
aI1
aI5
aI1
aI39
aI39
aI231
aI9
aI733
aI455
aI3383
aI4777
aI7235
aI12631
asS'd'
I3329
sa(dp6658
S'a'
I13366
sS's'
I15
sS'm'
(lp6659
I1
aI1
aI7
aI9
aI13
aI25
aI55
aI25
aI73
aI59
aI1699
aI929
aI755
aI1279
aI5583
asS'd'
I3330
sa(dp6660
S'a'
I13372
sS's'
I15
sS'm'
(lp6661
I1
aI3
aI5
aI3
aI9
aI49
aI79
aI55
aI479
aI179
aI1159
aI4079
aI3503
aI11603
aI12361
asS'd'
I3331
sa(dp6662
S'a'
I13380
sS's'
I15
sS'm'
(lp6663
I1
aI1
aI5
aI9
aI21
aI45
aI31
aI163
aI377
aI817
aI219
aI147
aI2581
aI12769
aI30783
asS'd'
I3332
sa(dp6664
S'a'
I13384
sS's'
I15
sS'm'
(lp6665
I1
aI3
aI1
aI7
aI15
aI27
aI39
aI189
aI493
aI259
aI1663
aI1213
aI961
aI11089
aI16079
asS'd'
I3333
sa(dp6666
S'a'
I13395
sS's'
I15
sS'm'
(lp6667
I1
aI1
aI5
aI9
aI5
aI41
aI13
aI153
aI313
aI337
aI1027
aI1267
aI4249
aI13071
aI27043
asS'd'
I3334
sa(dp6668
S'a'
I13407
sS's'
I15
sS'm'
(lp6669
I1
aI3
aI7
aI3
aI13
aI11
aI23
aI255
aI51
aI527
aI317
aI3217
aI5037
aI12723
aI17411
asS'd'
I3335
sa(dp6670
S'a'
I13408
sS's'
I15
sS'm'
(lp6671
I1
aI1
aI5
aI1
aI25
aI57
aI83
aI97
aI233
aI513
aI1283
aI2675
aI4111
aI4111
aI32141
asS'd'
I3336
sa(dp6672
S'a'
I13413
sS's'
I15
sS'm'
(lp6673
I1
aI3
aI3
aI15
aI25
aI33
aI103
aI81
aI155
aI189
aI139
aI1179
aI2691
aI15119
aI13959
asS'd'
I3337
sa(dp6674
S'a'
I13414
sS's'
I15
sS'm'
(lp6675
I1
aI3
aI3
aI1
aI25
aI55
aI67
aI19
aI19
aI9
aI437
aI579
aI4273
aI10733
aI7125
asS'd'
I3338
sa(dp6676
S'a'
I13417
sS's'
I15
sS'm'
(lp6677
I1
aI1
aI1
aI7
aI23
aI41
aI47
aI5
aI435
aI749
aI595
aI199
aI3941
aI7199
aI4795
asS'd'
I3339
sa(dp6678
S'a'
I13437
sS's'
I15
sS'm'
(lp6679
I1
aI3
aI1
aI15
aI5
aI49
aI35
aI9
aI199
aI703
aI1769
aI3269
aI5689
aI13063
aI22771
asS'd'
I3340
sa(dp6680
S'a'
I13441
sS's'
I15
sS'm'
(lp6681
I1
aI1
aI5
aI5
aI21
aI55
aI125
aI55
aI63
aI149
aI1167
aI3577
aI1051
aI3921
aI20545
asS'd'
I3341
sa(dp6682
S'a'
I13447
sS's'
I15
sS'm'
(lp6683
I1
aI3
aI7
aI13
aI29
aI53
aI107
aI193
aI163
aI339
aI1335
aI1573
aI5809
aI5681
aI29487
asS'd'
I3342
sa(dp6684
S'a'
I13456
sS's'
I15
sS'm'
(lp6685
I1
aI1
aI1
aI9
aI17
aI9
aI91
aI177
aI211
aI453
aI1807
aI1881
aI6051
aI225
aI6021
asS'd'
I3343
sa(dp6686
S'a'
I13459
sS's'
I15
sS'm'
(lp6687
I1
aI1
aI1
aI13
aI15
aI1
aI29
aI43
aI181
aI105
aI1945
aI2313
aI6429
aI2901
aI6221
asS'd'
I3344
sa(dp6688
S'a'
I13461
sS's'
I15
sS'm'
(lp6689
I1
aI3
aI5
aI11
aI29
aI55
aI115
aI115
aI187
aI1013
aI697
aI1885
aI121
aI12387
aI32443
asS'd'
I3345
sa(dp6690
S'a'
I13466
sS's'
I15
sS'm'
(lp6691
I1
aI1
aI1
aI7
aI19
aI51
aI21
aI107
aI55
aI125
aI1655
aI2281
aI3293
aI15749
aI27521
asS'd'
I3346
sa(dp6692
S'a'
I13484
sS's'
I15
sS'm'
(lp6693
I1
aI1
aI7
aI9
aI19
aI9
aI81
aI93
aI139
aI401
aI193
aI73
aI5159
aI9323
aI6019
asS'd'
I3347
sa(dp6694
S'a'
I13487
sS's'
I15
sS'm'
(lp6695
I1
aI1
aI7
aI9
aI15
aI51
aI115
aI69
aI247
aI599
aI1163
aI2251
aI1211
aI8827
aI15581
asS'd'
I3348
sa(dp6696
S'a'
I13489
sS's'
I15
sS'm'
(lp6697
I1
aI1
aI7
aI9
aI5
aI39
aI75
aI185
aI321
aI911
aI849
aI843
aI6791
aI10407
aI10513
asS'd'
I3349
sa(dp6698
S'a'
I13492
sS's'
I15
sS'm'
(lp6699
I1
aI1
aI5
aI5
aI15
aI9
aI21
aI47
aI459
aI681
aI2001
aI1545
aI5939
aI7073
aI29043
asS'd'
I3350
sa(dp6700
S'a'
I13496
sS's'
I15
sS'm'
(lp6701
I1
aI3
aI1
aI11
aI13
aI25
aI53
aI97
aI487
aI797
aI567
aI3757
aI5597
aI6313
aI18531
asS'd'
I3351
sa(dp6702
S'a'
I13510
sS's'
I15
sS'm'
(lp6703
I1
aI1
aI3
aI3
aI29
aI55
aI11
aI219
aI325
aI591
aI2015
aI383
aI2595
aI11855
aI22501
asS'd'
I3352
sa(dp6704
S'a'
I13531
sS's'
I15
sS'm'
(lp6705
I1
aI1
aI1
aI5
aI15
aI57
aI33
aI125
aI323
aI749
aI1843
aI4019
aI2075
aI6673
aI6957
asS'd'
I3353
sa(dp6706
S'a'
I13538
sS's'
I15
sS'm'
(lp6707
I1
aI1
aI5
aI7
aI19
aI7
aI47
aI239
aI51
aI107
aI1081
aI467
aI5493
aI7617
aI10355
asS'd'
I3354
sa(dp6708
S'a'
I13543
sS's'
I15
sS'm'
(lp6709
I1
aI3
aI1
aI1
aI11
aI3
aI67
aI199
aI175
aI421
aI935
aI309
aI4449
aI6363
aI9183
asS'd'
I3355
sa(dp6710
S'a'
I13547
sS's'
I15
sS'm'
(lp6711
I1
aI1
aI1
aI7
aI9
aI33
aI3
aI219
aI481
aI513
aI417
aI1267
aI2863
aI765
aI18431
asS'd'
I3356
sa(dp6712
S'a'
I13572
sS's'
I15
sS'm'
(lp6713
I1
aI3
aI1
aI1
aI19
aI1
aI89
aI109
aI415
aI105
aI487
aI3241
aI7465
aI9233
aI16307
asS'd'
I3357
sa(dp6714
S'a'
I13581
sS's'
I15
sS'm'
(lp6715
I1
aI1
aI3
aI13
aI9
aI43
aI25
aI231
aI383
aI789
aI1855
aI691
aI3465
aI2387
aI11715
asS'd'
I3358
sa(dp6716
S'a'
I13590
sS's'
I15
sS'm'
(lp6717
I1
aI3
aI3
aI3
aI13
aI39
aI63
aI107
aI33
aI265
aI437
aI117
aI3179
aI5543
aI28179
asS'd'
I3359
sa(dp6718
S'a'
I13605
sS's'
I15
sS'm'
(lp6719
I1
aI3
aI3
aI13
aI21
aI5
aI31
aI111
aI321
aI425
aI253
aI3501
aI3209
aI15429
aI18383
asS'd'
I3360
sa(dp6720
S'a'
I13612
sS's'
I15
sS'm'
(lp6721
I1
aI3
aI5
aI9
aI1
aI27
aI117
aI187
aI433
aI459
aI1999
aI1069
aI4857
aI8591
aI26343
asS'd'
I3361
sa(dp6722
S'a'
I13624
sS's'
I15
sS'm'
(lp6723
I1
aI1
aI7
aI3
aI15
aI43
aI11
aI193
aI391
aI341
aI1203
aI1259
aI7265
aI1783
aI13161
asS'd'
I3362
sa(dp6724
S'a'
I13630
sS's'
I15
sS'm'
(lp6725
I1
aI1
aI7
aI1
aI5
aI15
aI45
aI143
aI193
aI985
aI1105
aI3483
aI2421
aI9687
aI22347
asS'd'
I3363
sa(dp6726
S'a'
I13632
sS's'
I15
sS'm'
(lp6727
I1
aI3
aI7
aI13
aI21
aI17
aI79
aI231
aI487
aI663
aI1101
aI1025
aI5779
aI14681
aI29181
asS'd'
I3364
sa(dp6728
S'a'
I13638
sS's'
I15
sS'm'
(lp6729
I1
aI1
aI3
aI9
aI15
aI19
aI55
aI219
aI27
aI963
aI1513
aI1017
aI3907
aI12279
aI32655
asS'd'
I3365
sa(dp6730
S'a'
I13661
sS's'
I15
sS'm'
(lp6731
I1
aI3
aI7
aI3
aI31
aI27
aI17
aI1
aI51
aI861
aI529
aI1225
aI6395
aI15787
aI5231
asS'd'
I3366
sa(dp6732
S'a'
I13665
sS's'
I15
sS'm'
(lp6733
I1
aI3
aI3
aI11
aI27
aI7
aI101
aI143
aI21
aI191
aI1437
aI2393
aI4097
aI14319
aI6977
asS'd'
I3367
sa(dp6734
S'a'
I13668
sS's'
I15
sS'm'
(lp6735
I1
aI3
aI3
aI3
aI25
aI35
aI105
aI141
aI433
aI269
aI1469
aI2939
aI5387
aI7373
aI7863
asS'd'
I3368
sa(dp6736
S'a'
I13686
sS's'
I15
sS'm'
(lp6737
I1
aI3
aI7
aI5
aI5
aI21
aI23
aI11
aI217
aI357
aI1847
aI101
aI1161
aI5297
aI14879
asS'd'
I3369
sa(dp6738
S'a'
I13699
sS's'
I15
sS'm'
(lp6739
I1
aI3
aI1
aI3
aI25
aI23
aI81
aI217
aI505
aI161
aI1667
aI1343
aI1751
aI2463
aI26431
asS'd'
I3370
sa(dp6740
S'a'
I13701
sS's'
I15
sS'm'
(lp6741
I1
aI1
aI3
aI1
aI17
aI51
aI125
aI205
aI385
aI351
aI731
aI2911
aI2749
aI2689
aI27031
asS'd'
I3371
sa(dp6742
S'a'
I13708
sS's'
I15
sS'm'
(lp6743
I1
aI1
aI5
aI5
aI5
aI17
aI31
aI171
aI477
aI671
aI167
aI1797
aI8047
aI10123
aI4325
asS'd'
I3372
sa(dp6744
S'a'
I13716
sS's'
I15
sS'm'
(lp6745
I1
aI1
aI7
aI1
aI11
aI23
aI123
aI161
aI99
aI1007
aI765
aI1965
aI5395
aI16193
aI17751
asS'd'
I3373
sa(dp6746
S'a'
I13725
sS's'
I15
sS'm'
(lp6747
I1
aI3
aI1
aI9
aI13
aI11
aI111
aI217
aI31
aI753
aI377
aI2267
aI7893
aI7195
aI24999
asS'd'
I3374
sa(dp6748
S'a'
I13730
sS's'
I15
sS'm'
(lp6749
I1
aI3
aI1
aI9
aI21
aI53
aI127
aI121
aI151
aI395
aI1447
aI1411
aI5179
aI12043
aI27607
asS'd'
I3375
sa(dp6750
S'a'
I13742
sS's'
I15
sS'm'
(lp6751
I1
aI1
aI5
aI3
aI11
aI37
aI97
aI139
aI113
aI835
aI229
aI3741
aI827
aI5527
aI5779
asS'd'
I3376
sa(dp6752
S'a'
I13747
sS's'
I15
sS'm'
(lp6753
I1
aI1
aI7
aI7
aI27
aI55
aI11
aI55
aI429
aI269
aI1179
aI233
aI1053
aI10225
aI16703
asS'd'
I3377
sa(dp6754
S'a'
I13749
sS's'
I15
sS'm'
(lp6755
I1
aI1
aI1
aI3
aI15
aI9
aI67
aI119
aI95
aI753
aI511
aI2507
aI3953
aI6403
aI27635
asS'd'
I3378
sa(dp6756
S'a'
I13753
sS's'
I15
sS'm'
(lp6757
I1
aI3
aI3
aI7
aI27
aI57
aI25
aI27
aI249
aI515
aI193
aI4043
aI2017
aI751
aI10949
asS'd'
I3379
sa(dp6758
S'a'
I13754
sS's'
I15
sS'm'
(lp6759
I1
aI3
aI1
aI9
aI31
aI57
aI67
aI21
aI177
aI573
aI1835
aI2015
aI6201
aI2383
aI31087
asS'd'
I3380
sa(dp6760
S'a'
I13771
sS's'
I15
sS'm'
(lp6761
I1
aI1
aI5
aI1
aI19
aI3
aI89
aI243
aI69
aI387
aI1905
aI3465
aI2775
aI7713
aI30081
asS'd'
I3381
sa(dp6762
S'a'
I13773
sS's'
I15
sS'm'
(lp6763
I1
aI1
aI3
aI3
aI9
aI59
aI15
aI89
aI85
aI605
aI881
aI263
aI2551
aI797
aI16541
asS'd'
I3382
sa(dp6764
S'a'
I13782
sS's'
I15
sS'm'
(lp6765
I1
aI3
aI7
aI11
aI25
aI41
aI59
aI139
aI405
aI571
aI1147
aI2963
aI4175
aI12901
aI6309
asS'd'
I3383
sa(dp6766
S'a'
I13785
sS's'
I15
sS'm'
(lp6767
I1
aI3
aI1
aI5
aI29
aI29
aI11
aI243
aI183
aI281
aI807
aI1
aI7079
aI10079
aI13865
asS'd'
I3384
sa(dp6768
S'a'
I13798
sS's'
I15
sS'm'
(lp6769
I1
aI3
aI7
aI5
aI5
aI1
aI89
aI55
aI423
aI157
aI675
aI1849
aI241
aI6467
aI15459
asS'd'
I3385
sa(dp6770
S'a'
I13802
sS's'
I15
sS'm'
(lp6771
I1
aI1
aI7
aI11
aI15
aI63
aI89
aI109
aI501
aI549
aI317
aI3043
aI1151
aI3895
aI19851
asS'd'
I3386
sa(dp6772
S'a'
I13809
sS's'
I15
sS'm'
(lp6773
I1
aI3
aI1
aI15
aI7
aI23
aI97
aI97
aI225
aI737
aI1117
aI3325
aI209
aI14169
aI10813
asS'd'
I3387
sa(dp6774
S'a'
I13828
sS's'
I15
sS'm'
(lp6775
I1
aI3
aI7
aI13
aI13
aI39
aI91
aI153
aI395
aI879
aI1041
aI3753
aI5577
aI1985
aI25247
asS'd'
I3388
sa(dp6776
S'a'
I13832
sS's'
I15
sS'm'
(lp6777
I1
aI1
aI1
aI3
aI17
aI15
aI113
aI143
aI101
aI901
aI1119
aI1819
aI3577
aI3441
aI31511
asS'd'
I3389
sa(dp6778
S'a'
I13840
sS's'
I15
sS'm'
(lp6779
I1
aI3
aI1
aI11
aI15
aI27
aI21
aI37
aI287
aI121
aI451
aI1353
aI2173
aI299
aI18791
asS'd'
I3390
sa(dp6780
S'a'
I13850
sS's'
I15
sS'm'
(lp6781
I1
aI3
aI3
aI5
aI23
aI1
aI49
aI145
aI315
aI769
aI99
aI1385
aI5961
aI9121
aI1465
asS'd'
I3391
sa(dp6782
S'a'
I13861
sS's'
I15
sS'm'
(lp6783
I1
aI3
aI3
aI13
aI21
aI39
aI39
aI233
aI271
aI113
aI615
aI2257
aI3765
aI5921
aI313
asS'd'
I3392
sa(dp6784
S'a'
I13874
sS's'
I15
sS'm'
(lp6785
I1
aI3
aI7
aI7
aI25
aI45
aI11
aI237
aI83
aI203
aI929
aI1819
aI2679
aI11583
aI30091
asS'd'
I3393
sa(dp6786
S'a'
I13876
sS's'
I15
sS'm'
(lp6787
I1
aI1
aI1
aI7
aI21
aI63
aI85
aI251
aI133
aI991
aI1515
aI2547
aI6051
aI7279
aI3569
asS'd'
I3394
sa(dp6788
S'a'
I13886
sS's'
I15
sS'm'
(lp6789
I1
aI3
aI7
aI15
aI11
aI19
aI87
aI17
aI313
aI283
aI1021
aI2743
aI4855
aI13741
aI17955
asS'd'
I3395
sa(dp6790
S'a'
I13897
sS's'
I15
sS'm'
(lp6791
I1
aI1
aI7
aI13
aI29
aI13
aI61
aI93
aI81
aI91
aI995
aI907
aI4847
aI2599
aI20041
asS'd'
I3396
sa(dp6792
S'a'
I13900
sS's'
I15
sS'm'
(lp6793
I1
aI1
aI3
aI11
aI13
aI45
aI103
aI33
aI243
aI109
aI2029
aI121
aI231
aI16179
aI13741
asS'd'
I3397
sa(dp6794
S'a'
I13915
sS's'
I15
sS'm'
(lp6795
I1
aI3
aI5
aI9
aI9
aI5
aI73
aI225
aI199
aI723
aI611
aI1909
aI2345
aI10257
aI9909
asS'd'
I3398
sa(dp6796
S'a'
I13927
sS's'
I15
sS'm'
(lp6797
I1
aI1
aI3
aI11
aI7
aI5
aI33
aI89
aI431
aI899
aI803
aI3173
aI6131
aI16097
aI20561
asS'd'
I3399
sa(dp6798
S'a'
I13951
sS's'
I15
sS'm'
(lp6799
I1
aI3
aI3
aI7
aI7
aI47
aI23
aI47
aI411
aI69
aI239
aI661
aI5591
aI10457
aI24245
asS'd'
I3400
sa(dp6800
S'a'
I13955
sS's'
I15
sS'm'
(lp6801
I1
aI1
aI5
aI15
aI25
aI35
aI87
aI23
aI115
aI939
aI1579
aI119
aI4001
aI13791
aI9729
asS'd'
I3401
sa(dp6802
S'a'
I13962
sS's'
I15
sS'm'
(lp6803
I1
aI3
aI5
aI11
aI25
aI45
aI29
aI195
aI369
aI237
aI735
aI155
aI123
aI4415
aI32255
asS'd'
I3402
sa(dp6804
S'a'
I13969
sS's'
I15
sS'm'
(lp6805
I1
aI3
aI3
aI9
aI13
aI53
aI15
aI77
aI313
aI75
aI529
aI925
aI5679
aI14585
aI19889
asS'd'
I3403
sa(dp6806
S'a'
I13979
sS's'
I15
sS'm'
(lp6807
I1
aI1
aI7
aI15
aI15
aI27
aI105
aI13
aI31
aI669
aI563
aI1809
aI4321
aI7797
aI4177
asS'd'
I3404
sa(dp6808
S'a'
I13988
sS's'
I15
sS'm'
(lp6809
I1
aI1
aI5
aI9
aI3
aI29
aI111
aI177
aI33
aI235
aI1951
aI1561
aI141
aI4803
aI16327
asS'd'
I3405
sa(dp6810
S'a'
I13998
sS's'
I15
sS'm'
(lp6811
I1
aI1
aI1
aI7
aI9
aI41
aI1
aI149
aI95
aI933
aI115
aI1619
aI771
aI8189
aI8781
asS'd'
I3406
sa(dp6812
S'a'
I14000
sS's'
I15
sS'm'
(lp6813
I1
aI1
aI5
aI3
aI13
aI41
aI33
aI159
aI355
aI159
aI1243
aI1439
aI6571
aI14397
aI31321
asS'd'
I3407
sa(dp6814
S'a'
I14005
sS's'
I15
sS'm'
(lp6815
I1
aI1
aI7
aI11
aI9
aI15
aI91
aI145
aI457
aI255
aI1449
aI611
aI1449
aI2521
aI28949
asS'd'
I3408
sa(dp6816
S'a'
I14027
sS's'
I15
sS'm'
(lp6817
I1
aI3
aI7
aI5
aI27
aI57
aI35
aI99
aI447
aI287
aI743
aI1163
aI4379
aI7361
aI3831
asS'd'
I3409
sa(dp6818
S'a'
I14037
sS's'
I15
sS'm'
(lp6819
I1
aI3
aI3
aI7
aI15
aI53
aI41
aI83
aI133
aI571
aI1739
aI531
aI2921
aI11527
aI21941
asS'd'
I3410
sa(dp6820
S'a'
I14051
sS's'
I15
sS'm'
(lp6821
I1
aI1
aI1
aI13
aI9
aI27
aI39
aI113
aI429
aI447
aI595
aI3171
aI5245
aI4095
aI14847
asS'd'
I3411
sa(dp6822
S'a'
I14054
sS's'
I15
sS'm'
(lp6823
I1
aI1
aI3
aI7
aI19
aI19
aI21
aI101
aI489
aI1011
aI265
aI3899
aI3225
aI11701
aI5193
asS'd'
I3412
sa(dp6824
S'a'
I14060
sS's'
I15
sS'm'
(lp6825
I1
aI3
aI7
aI3
aI15
aI25
aI103
aI213
aI441
aI215
aI1483
aI263
aI3561
aI7915
aI7969
asS'd'
I3413
sa(dp6826
S'a'
I14063
sS's'
I15
sS'm'
(lp6827
I1
aI3
aI3
aI3
aI11
aI47
aI97
aI29
aI489
aI867
aI1347
aI2155
aI4871
aI8001
aI18305
asS'd'
I3414
sa(dp6828
S'a'
I14071
sS's'
I15
sS'm'
(lp6829
I1
aI3
aI1
aI9
aI25
aI15
aI61
aI17
aI343
aI775
aI1765
aI3803
aI4577
aI8437
aI12605
asS'd'
I3415
sa(dp6830
S'a'
I14078
sS's'
I15
sS'm'
(lp6831
I1
aI1
aI5
aI3
aI11
aI39
aI69
aI23
aI23
aI65
aI1967
aI2429
aI1703
aI6671
aI14981
asS'd'
I3416
sa(dp6832
S'a'
I14080
sS's'
I15
sS'm'
(lp6833
I1
aI1
aI5
aI15
aI23
aI59
aI125
aI51
aI225
aI439
aI2019
aI2589
aI7781
aI13111
aI2911
asS'd'
I3417
sa(dp6834
S'a'
I14085
sS's'
I15
sS'm'
(lp6835
I1
aI1
aI1
aI3
aI1
aI31
aI37
aI245
aI203
aI305
aI821
aI367
aI5211
aI9791
aI21777
asS'd'
I3418
sa(dp6836
S'a'
I14086
sS's'
I15
sS'm'
(lp6837
I1
aI1
aI5
aI9
aI9
aI31
aI97
aI25
aI271
aI83
aI343
aI2461
aI1805
aI14383
aI10059
asS'd'
I3419
sa(dp6838
S'a'
I14095
sS's'
I15
sS'm'
(lp6839
I1
aI1
aI5
aI13
aI15
aI33
aI127
aI109
aI137
aI963
aI961
aI1647
aI7881
aI8133
aI22359
asS'd'
I3420
sa(dp6840
S'a'
I14138
sS's'
I15
sS'm'
(lp6841
I1
aI1
aI3
aI7
aI25
aI31
aI123
aI241
aI283
aI1
aI1781
aI23
aI971
aI6485
aI127
asS'd'
I3421
sa(dp6842
S'a'
I14145
sS's'
I15
sS'm'
(lp6843
I1
aI1
aI5
aI15
aI15
aI27
aI25
aI145
aI395
aI679
aI979
aI571
aI1585
aI14787
aI7465
asS'd'
I3422
sa(dp6844
S'a'
I14158
sS's'
I15
sS'm'
(lp6845
I1
aI1
aI5
aI7
aI13
aI11
aI7
aI131
aI511
aI597
aI379
aI1513
aI6267
aI16039
aI1503
asS'd'
I3423
sa(dp6846
S'a'
I14166
sS's'
I15
sS'm'
(lp6847
I1
aI1
aI1
aI13
aI15
aI49
aI73
aI217
aI353
aI577
aI1913
aI1127
aI961
aI11557
aI24993
asS'd'
I3424
sa(dp6848
S'a'
I14179
sS's'
I15
sS'm'
(lp6849
I1
aI3
aI3
aI9
aI7
aI3
aI105
aI141
aI377
aI687
aI1917
aI485
aI983
aI11149
aI23303
asS'd'
I3425
sa(dp6850
S'a'
I14181
sS's'
I15
sS'm'
(lp6851
I1
aI1
aI3
aI15
aI11
aI7
aI117
aI179
aI505
aI67
aI1817
aI913
aI5757
aI1981
aI1637
asS'd'
I3426
sa(dp6852
S'a'
I14188
sS's'
I15
sS'm'
(lp6853
I1
aI1
aI1
aI7
aI5
aI29
aI3
aI43
aI223
aI295
aI1895
aI3425
aI5355
aI5155
aI17197
asS'd'
I3427
sa(dp6854
S'a'
I14193
sS's'
I15
sS'm'
(lp6855
I1
aI1
aI7
aI9
aI21
aI59
aI121
aI245
aI73
aI233
aI1527
aI869
aI4145
aI7995
aI6473
asS'd'
I3428
sa(dp6856
S'a'
I14200
sS's'
I15
sS'm'
(lp6857
I1
aI1
aI5
aI13
aI17
aI21
aI89
aI179
aI495
aI669
aI453
aI2603
aI5969
aI6161
aI4743
asS'd'
I3429
sa(dp6858
S'a'
I14203
sS's'
I15
sS'm'
(lp6859
I1
aI1
aI7
aI11
aI25
aI21
aI103
aI131
aI391
aI249
aI1633
aI2603
aI2207
aI8987
aI15487
asS'd'
I3430
sa(dp6860
S'a'
I14215
sS's'
I15
sS'm'
(lp6861
I1
aI3
aI7
aI9
aI13
aI45
aI99
aI251
aI115
aI597
aI1505
aI2421
aI1231
aI10015
aI24295
asS'd'
I3431
sa(dp6862
S'a'
I14224
sS's'
I15
sS'm'
(lp6863
I1
aI1
aI5
aI5
aI31
aI49
aI17
aI67
aI463
aI813
aI1491
aI3309
aI7881
aI8109
aI7289
asS'd'
I3432
sa(dp6864
S'a'
I14230
sS's'
I15
sS'm'
(lp6865
I1
aI3
aI1
aI15
aI23
aI35
aI123
aI21
aI169
aI499
aI95
aI603
aI1829
aI7865
aI26313
asS'd'
I3433
sa(dp6866
S'a'
I14233
sS's'
I15
sS'm'
(lp6867
I1
aI1
aI7
aI1
aI9
aI29
aI45
aI65
aI95
aI97
aI673
aI3673
aI2969
aI2317
aI22209
asS'd'
I3434
sa(dp6868
S'a'
I14236
sS's'
I15
sS'm'
(lp6869
I1
aI1
aI3
aI7
aI29
aI33
aI121
aI17
aI331
aI487
aI1901
aI1951
aI5383
aI9375
aI4029
asS'd'
I3435
sa(dp6870
S'a'
I14246
sS's'
I15
sS'm'
(lp6871
I1
aI3
aI7
aI9
aI25
aI43
aI91
aI147
aI141
aI401
aI1647
aI2697
aI4645
aI7179
aI31857
asS'd'
I3436
sa(dp6872
S'a'
I14267
sS's'
I15
sS'm'
(lp6873
I1
aI3
aI5
aI11
aI9
aI31
aI127
aI105
aI39
aI883
aI1635
aI919
aI5069
aI2875
aI24519
asS'd'
I3437
sa(dp6874
S'a'
I14282
sS's'
I15
sS'm'
(lp6875
I1
aI1
aI5
aI9
aI1
aI63
aI73
aI135
aI95
aI503
aI385
aI3903
aI545
aI12635
aI27569
asS'd'
I3438
sa(dp6876
S'a'
I14287
sS's'
I15
sS'm'
(lp6877
I1
aI1
aI3
aI11
aI27
aI31
aI47
aI173
aI55
aI339
aI1255
aI1947
aI793
aI14133
aI13963
asS'd'
I3439
sa(dp6878
S'a'
I14301
sS's'
I15
sS'm'
(lp6879
I1
aI1
aI3
aI15
aI17
aI33
aI113
aI249
aI401
aI743
aI1307
aI3123
aI627
aI1253
aI13285
asS'd'
I3440
sa(dp6880
S'a'
I14323
sS's'
I15
sS'm'
(lp6881
I1
aI1
aI3
aI1
aI9
aI7
aI39
aI65
aI281
aI107
aI833
aI193
aI2987
aI12267
aI31335
asS'd'
I3441
sa(dp6882
S'a'
I14325
sS's'
I15
sS'm'
(lp6883
I1
aI1
aI7
aI3
aI15
aI21
aI99
aI211
aI39
aI179
aI587
aI1169
aI6455
aI8225
aI2049
asS'd'
I3442
sa(dp6884
S'a'
I14329
sS's'
I15
sS'm'
(lp6885
I1
aI3
aI5
aI13
aI5
aI13
aI123
aI1
aI223
aI273
aI731
aI2967
aI4793
aI4229
aI26031
asS'd'
I3443
sa(dp6886
S'a'
I14339
sS's'
I15
sS'm'
(lp6887
I1
aI1
aI1
aI1
aI3
aI17
aI7
aI23
aI225
aI757
aI743
aI1257
aI2047
aI12509
aI25467
asS'd'
I3444
sa(dp6888
S'a'
I14342
sS's'
I15
sS'm'
(lp6889
I1
aI1
aI7
aI15
aI29
aI3
aI15
aI113
aI227
aI675
aI1295
aI2777
aI2921
aI5485
aI2577
asS'd'
I3445
sa(dp6890
S'a'
I14351
sS's'
I15
sS'm'
(lp6891
I1
aI3
aI7
aI13
aI19
aI21
aI85
aI129
aI45
aI599
aI317
aI1513
aI4953
aI10383
aI25253
asS'd'
I3446
sa(dp6892
S'a'
I14356
sS's'
I15
sS'm'
(lp6893
I1
aI1
aI7
aI11
aI13
aI47
aI127
aI67
aI219
aI131
aI905
aI2005
aI851
aI15243
aI5777
asS'd'
I3447
sa(dp6894
S'a'
I14359
sS's'
I15
sS'm'
(lp6895
I1
aI1
aI5
aI3
aI23
aI57
aI57
aI189
aI153
aI37
aI955
aI2049
aI1295
aI15119
aI27213
asS'd'
I3448
sa(dp6896
S'a'
I14370
sS's'
I15
sS'm'
(lp6897
I1
aI3
aI7
aI11
aI13
aI61
aI3
aI241
aI269
aI789
aI1595
aI2369
aI4843
aI11347
aI21543
asS'd'
I3449
sa(dp6898
S'a'
I14402
sS's'
I15
sS'm'
(lp6899
I1
aI1
aI5
aI5
aI25
aI21
aI19
aI237
aI3
aI605
aI1343
aI3965
aI3511
aI7889
aI27759
asS'd'
I3450
sa(dp6900
S'a'
I14411
sS's'
I15
sS'm'
(lp6901
I1
aI3
aI1
aI15
aI21
aI15
aI123
aI5
aI345
aI945
aI283
aI1313
aI335
aI2085
aI19505
asS'd'
I3451
sa(dp6902
S'a'
I14421
sS's'
I15
sS'm'
(lp6903
I1
aI1
aI3
aI3
aI5
aI21
aI123
aI89
aI67
aI11
aI1247
aI1155
aI287
aI13455
aI5693
asS'd'
I3452
sa(dp6904
S'a'
I14431
sS's'
I15
sS'm'
(lp6905
I1
aI3
aI3
aI13
aI1
aI53
aI101
aI27
aI387
aI379
aI19
aI751
aI2445
aI11737
aI975
asS'd'
I3453
sa(dp6906
S'a'
I14435
sS's'
I15
sS'm'
(lp6907
I1
aI3
aI3
aI3
aI9
aI29
aI81
aI117
aI443
aI145
aI1619
aI1813
aI8125
aI5829
aI28617
asS'd'
I3454
sa(dp6908
S'a'
I14442
sS's'
I15
sS'm'
(lp6909
I1
aI1
aI5
aI15
aI27
aI15
aI83
aI83
aI61
aI715
aI1655
aI1631
aI3457
aI2727
aI2163
asS'd'
I3455
sa(dp6910
S'a'
I14447
sS's'
I15
sS'm'
(lp6911
I1
aI3
aI1
aI5
aI11
aI11
aI121
aI7
aI135
aI883
aI927
aI1817
aI6839
aI12361
aI24119
asS'd'
I3456
sa(dp6912
S'a'
I14456
sS's'
I15
sS'm'
(lp6913
I1
aI3
aI7
aI11
aI23
aI59
aI39
aI165
aI109
aI355
aI1303
aI381
aI5697
aI275
aI3771
asS'd'
I3457
sa(dp6914
S'a'
I14459
sS's'
I15
sS'm'
(lp6915
I1
aI3
aI5
aI11
aI11
aI5
aI81
aI157
aI55
aI435
aI613
aI127
aI4087
aI3791
aI21627
asS'd'
I3458
sa(dp6916
S'a'
I14472
sS's'
I15
sS'm'
(lp6917
I1
aI3
aI7
aI15
aI13
aI37
aI83
aI195
aI207
aI771
aI51
aI3685
aI6389
aI1229
aI11101
asS'd'
I3459
sa(dp6918
S'a'
I14477
sS's'
I15
sS'm'
(lp6919
I1
aI3
aI7
aI13
aI31
aI3
aI9
aI13
aI487
aI95
aI77
aI809
aI5809
aI12887
aI29933
asS'd'
I3460
sa(dp6920
S'a'
I14490
sS's'
I15
sS'm'
(lp6921
I1
aI1
aI3
aI7
aI25
aI9
aI13
aI29
aI353
aI659
aI1785
aI3825
aI3729
aI13109
aI12973
asS'd'
I3461
sa(dp6922
S'a'
I14496
sS's'
I15
sS'm'
(lp6923
I1
aI1
aI1
aI5
aI21
aI3
aI97
aI1
aI245
aI917
aI29
aI1429
aI8141
aI7569
aI32493
asS'd'
I3462
sa(dp6924
S'a'
I14501
sS's'
I15
sS'm'
(lp6925
I1
aI3
aI1
aI9
aI19
aI13
aI13
aI109
aI377
aI1007
aI1737
aI1939
aI1419
aI1145
aI5065
asS'd'
I3463
sa(dp6926
S'a'
I14505
sS's'
I15
sS'm'
(lp6927
I1
aI1
aI7
aI9
aI27
aI57
aI53
aI69
aI423
aI43
aI1629
aI1003
aI1473
aI10809
aI5659
asS'd'
I3464
sa(dp6928
S'a'
I14513
sS's'
I15
sS'm'
(lp6929
I1
aI1
aI1
aI9
aI1
aI45
aI11
aI231
aI351
aI155
aI663
aI2783
aI3491
aI5725
aI25207
asS'd'
I3465
sa(dp6930
S'a'
I14520
sS's'
I15
sS'm'
(lp6931
I1
aI1
aI1
aI3
aI15
aI25
aI77
aI89
aI231
aI813
aI657
aI2603
aI4885
aI1383
aI14499
asS'd'
I3466
sa(dp6932
S'a'
I14534
sS's'
I15
sS'm'
(lp6933
I1
aI3
aI5
aI5
aI9
aI21
aI101
aI181
aI449
aI491
aI737
aI803
aI659
aI11771
aI545
asS'd'
I3467
sa(dp6934
S'a'
I14562
sS's'
I15
sS'm'
(lp6935
I1
aI3
aI7
aI9
aI7
aI19
aI27
aI199
aI265
aI329
aI1031
aI1235
aI3191
aI10071
aI16281
asS'd'
I3468
sa(dp6936
S'a'
I14576
sS's'
I15
sS'm'
(lp6937
I1
aI1
aI7
aI11
aI27
aI55
aI3
aI127
aI503
aI1003
aI1041
aI1953
aI5835
aI4851
aI13485
asS'd'
I3469
sa(dp6938
S'a'
I14579
sS's'
I15
sS'm'
(lp6939
I1
aI1
aI7
aI15
aI5
aI45
aI97
aI61
aI221
aI497
aI1949
aI3163
aI4707
aI8441
aI1437
asS'd'
I3470
sa(dp6940
S'a'
I14585
sS's'
I15
sS'm'
(lp6941
I1
aI3
aI5
aI1
aI3
aI35
aI107
aI9
aI473
aI971
aI227
aI2225
aI3999
aI3095
aI18879
asS'd'
I3471
sa(dp6942
S'a'
I14586
sS's'
I15
sS'm'
(lp6943
I1
aI1
aI1
aI9
aI21
aI59
aI21
aI1
aI41
aI435
aI575
aI491
aI1839
aI1095
aI9727
asS'd'
I3472
sa(dp6944
S'a'
I14606
sS's'
I15
sS'm'
(lp6945
I1
aI3
aI5
aI9
aI13
aI29
aI123
aI251
aI465
aI701
aI1105
aI829
aI573
aI11503
aI11861
asS'd'
I3473
sa(dp6946
S'a'
I14627
sS's'
I15
sS'm'
(lp6947
I1
aI3
aI3
aI13
aI27
aI59
aI29
aI111
aI225
aI973
aI561
aI1481
aI835
aI9261
aI13831
asS'd'
I3474
sa(dp6948
S'a'
I14630
sS's'
I15
sS'm'
(lp6949
I1
aI1
aI1
aI7
aI17
aI3
aI97
aI211
aI333
aI315
aI571
aI3523
aI7305
aI6461
aI20139
asS'd'
I3475
sa(dp6950
S'a'
I14634
sS's'
I15
sS'm'
(lp6951
I1
aI3
aI7
aI11
aI31
aI21
aI105
aI247
aI113
aI863
aI1767
aI381
aI4623
aI8935
aI7911
asS'd'
I3476
sa(dp6952
S'a'
I14636
sS's'
I15
sS'm'
(lp6953
I1
aI1
aI5
aI7
aI29
aI45
aI17
aI155
aI69
aI17
aI655
aI1983
aI6385
aI6177
aI7961
asS'd'
I3477
sa(dp6954
S'a'
I14647
sS's'
I15
sS'm'
(lp6955
I1
aI3
aI3
aI15
aI31
aI15
aI63
aI81
aI309
aI115
aI393
aI3445
aI689
aI13963
aI18887
asS'd'
I3478
sa(dp6956
S'a'
I14653
sS's'
I15
sS'm'
(lp6957
I1
aI1
aI5
aI1
aI19
aI39
aI127
aI61
aI357
aI53
aI195
aI2745
aI7853
aI5753
aI3669
asS'd'
I3479
sa(dp6958
S'a'
I14659
sS's'
I15
sS'm'
(lp6959
I1
aI3
aI7
aI7
aI17
aI51
aI57
aI145
aI451
aI365
aI1517
aI909
aI4265
aI10737
aI9579
asS'd'
I3480
sa(dp6960
S'a'
I14671
sS's'
I15
sS'm'
(lp6961
I1
aI1
aI3
aI13
aI3
aI37
aI121
aI103
aI257
aI47
aI1685
aI2951
aI5753
aI15379
aI8899
asS'd'
I3481
sa(dp6962
S'a'
I14674
sS's'
I15
sS'm'
(lp6963
I1
aI1
aI5
aI7
aI31
aI63
aI61
aI197
aI97
aI773
aI133
aI1517
aI3093
aI14879
aI22941
asS'd'
I3482
sa(dp6964
S'a'
I14701
sS's'
I15
sS'm'
(lp6965
I1
aI1
aI5
aI1
aI3
aI9
aI27
aI53
aI97
aI663
aI1915
aI409
aI471
aI1391
aI24853
asS'd'
I3483
sa(dp6966
S'a'
I14716
sS's'
I15
sS'm'
(lp6967
I1
aI1
aI1
aI7
aI21
aI53
aI69
aI5
aI187
aI571
aI2023
aI997
aI323
aI12059
aI7071
asS'd'
I3484
sa(dp6968
S'a'
I14719
sS's'
I15
sS'm'
(lp6969
I1
aI3
aI3
aI1
aI7
aI59
aI55
aI157
aI101
aI123
aI1301
aI3709
aI4673
aI3897
aI28791
asS'd'
I3485
sa(dp6970
S'a'
I14720
sS's'
I15
sS'm'
(lp6971
I1
aI3
aI7
aI5
aI5
aI23
aI39
aI139
aI365
aI415
aI1481
aI3415
aI6323
aI11109
aI5719
asS'd'
I3486
sa(dp6972
S'a'
I14725
sS's'
I15
sS'm'
(lp6973
I1
aI3
aI5
aI3
aI5
aI11
aI23
aI143
aI243
aI229
aI183
aI3367
aI3187
aI8151
aI28351
asS'd'
I3487
sa(dp6974
S'a'
I14730
sS's'
I15
sS'm'
(lp6975
I1
aI3
aI7
aI9
aI5
aI37
aI29
aI23
aI437
aI827
aI985
aI2879
aI7611
aI1391
aI19087
asS'd'
I3488
sa(dp6976
S'a'
I14743
sS's'
I15
sS'm'
(lp6977
I1
aI3
aI3
aI5
aI7
aI9
aI5
aI143
aI217
aI757
aI1697
aI2459
aI453
aI8679
aI4513
asS'd'
I3489
sa(dp6978
S'a'
I14747
sS's'
I15
sS'm'
(lp6979
I1
aI3
aI5
aI5
aI11
aI33
aI3
aI143
aI293
aI921
aI185
aI2461
aI5547
aI12247
aI28591
asS'd'
I3490
sa(dp6980
S'a'
I14786
sS's'
I15
sS'm'
(lp6981
I1
aI3
aI7
aI5
aI3
aI53
aI43
aI179
aI235
aI417
aI1307
aI1367
aI3695
aI12809
aI1807
asS'd'
I3491
sa(dp6982
S'a'
I14788
sS's'
I15
sS'm'
(lp6983
I1
aI3
aI1
aI11
aI15
aI43
aI115
aI229
aI157
aI25
aI687
aI3347
aI271
aI5777
aI8557
asS'd'
I3492
sa(dp6984
S'a'
I14792
sS's'
I15
sS'm'
(lp6985
I1
aI3
aI7
aI5
aI27
aI37
aI55
aI135
aI209
aI47
aI1603
aI957
aI5785
aI11141
aI10407
asS'd'
I3493
sa(dp6986
S'a'
I14795
sS's'
I15
sS'm'
(lp6987
I1
aI1
aI1
aI15
aI17
aI17
aI103
aI29
aI489
aI493
aI119
aI1707
aI3463
aI1815
aI32055
asS'd'
I3494
sa(dp6988
S'a'
I14809
sS's'
I15
sS'm'
(lp6989
I1
aI3
aI7
aI11
aI17
aI13
aI115
aI145
aI77
aI515
aI1911
aI477
aI5997
aI8731
aI3143
asS'd'
I3495
sa(dp6990
S'a'
I14831
sS's'
I15
sS'm'
(lp6991
I1
aI3
aI1
aI13
aI31
aI41
aI73
aI91
aI231
aI1
aI455
aI2023
aI4691
aI3613
aI16329
asS'd'
I3496
sa(dp6992
S'a'
I14834
sS's'
I15
sS'm'
(lp6993
I1
aI1
aI5
aI15
aI15
aI39
aI17
aI117
aI131
aI657
aI1939
aI2245
aI2575
aI195
aI25209
asS'd'
I3497
sa(dp6994
S'a'
I14850
sS's'
I15
sS'm'
(lp6995
I1
aI3
aI7
aI15
aI5
aI51
aI69
aI141
aI499
aI931
aI1165
aI2119
aI1703
aI10867
aI28443
asS'd'
I3498
sa(dp6996
S'a'
I14855
sS's'
I15
sS'm'
(lp6997
I1
aI1
aI1
aI15
aI13
aI45
aI45
aI103
aI115
aI177
aI651
aI2545
aI1417
aI5349
aI3385
asS'd'
I3499
sa(dp6998
S'a'
I14859
sS's'
I15
sS'm'
(lp6999
I1
aI3
aI3
aI1
aI1
aI41
aI117
aI15
aI225
aI861
aI843
aI2775
aI4543
aI6275
aI14671
asS'd'
I3500
sa(dp7000
S'a'
I14864
sS's'
I15
sS'm'
(lp7001
I1
aI3
aI5
aI15
aI5
aI35
aI87
aI193
aI341
aI55
aI1131
aI945
aI6865
aI11271
aI18705
asS'd'
I3501
sa(dp7002
S'a'
I14876
sS's'
I15
sS'm'
(lp7003
I1
aI3
aI5
aI9
aI13
aI35
aI71
aI197
aI79
aI351
aI3
aI3939
aI1105
aI12455
aI28921
asS'd'
I3502
sa(dp7004
S'a'
I14889
sS's'
I15
sS'm'
(lp7005
I1
aI3
aI1
aI13
aI9
aI23
aI89
aI165
aI59
aI257
aI1369
aI161
aI6255
aI2997
aI19175
asS'd'
I3503
sa(dp7006
S'a'
I14890
sS's'
I15
sS'm'
(lp7007
I1
aI3
aI5
aI3
aI5
aI41
aI107
aI231
aI111
aI207
aI1865
aI2079
aI5891
aI2487
aI5863
asS'd'
I3504
sa(dp7008
S'a'
I14898
sS's'
I15
sS'm'
(lp7009
I1
aI3
aI7
aI15
aI3
aI3
aI105
aI235
aI263
aI991
aI367
aI1885
aI1769
aI7805
aI11909
asS'd'
I3505
sa(dp7010
S'a'
I14909
sS's'
I15
sS'm'
(lp7011
I1
aI3
aI3
aI5
aI15
aI59
aI67
aI247
aI77
aI367
aI1641
aI1959
aI1921
aI5939
aI17355
asS'd'
I3506
sa(dp7012
S'a'
I14917
sS's'
I15
sS'm'
(lp7013
I1
aI1
aI7
aI1
aI3
aI53
aI37
aI5
aI221
aI779
aI1353
aI1633
aI2769
aI6355
aI8505
asS'd'
I3507
sa(dp7014
S'a'
I14924
sS's'
I15
sS'm'
(lp7015
I1
aI1
aI7
aI13
aI11
aI13
aI73
aI227
aI115
aI523
aI355
aI3127
aI7545
aI8409
aI22335
asS'd'
I3508
sa(dp7016
S'a'
I14929
sS's'
I15
sS'm'
(lp7017
I1
aI1
aI5
aI11
aI21
aI15
aI91
aI115
aI427
aI683
aI461
aI2433
aI6313
aI4595
aI24401
asS'd'
I3509
sa(dp7018
S'a'
I14942
sS's'
I15
sS'm'
(lp7019
I1
aI3
aI7
aI5
aI29
aI21
aI57
aI215
aI423
aI717
aI1455
aI705
aI6835
aI4503
aI26077
asS'd'
I3510
sa(dp7020
S'a'
I14951
sS's'
I15
sS'm'
(lp7021
I1
aI1
aI1
aI15
aI3
aI33
aI25
aI227
aI381
aI477
aI1023
aI2751
aI2229
aI631
aI16903
asS'd'
I3511
sa(dp7022
S'a'
I14969
sS's'
I15
sS'm'
(lp7023
I1
aI3
aI1
aI11
aI9
aI17
aI59
aI73
aI53
aI671
aI251
aI1729
aI7593
aI12473
aI22533
asS'd'
I3512
sa(dp7024
S'a'
I14970
sS's'
I15
sS'm'
(lp7025
I1
aI3
aI3
aI1
aI3
aI35
aI37
aI173
aI459
aI143
aI135
aI3871
aI2689
aI8007
aI4379
asS'd'
I3513
sa(dp7026
S'a'
I14972
sS's'
I15
sS'm'
(lp7027
I1
aI3
aI5
aI9
aI23
aI19
aI43
aI45
aI493
aI509
aI1851
aI1615
aI5675
aI13793
aI6973
asS'd'
I3514
sa(dp7028
S'a'
I14982
sS's'
I15
sS'm'
(lp7029
I1
aI3
aI3
aI15
aI5
aI17
aI77
aI85
aI451
aI753
aI579
aI1057
aI4851
aI6017
aI4195
asS'd'
I3515
sa(dp7030
S'a'
I14988
sS's'
I15
sS'm'
(lp7031
I1
aI3
aI3
aI5
aI31
aI29
aI81
aI159
aI103
aI391
aI15
aI899
aI4623
aI5957
aI31961
asS'd'
I3516
sa(dp7032
S'a'
I14994
sS's'
I15
sS'm'
(lp7033
I1
aI1
aI1
aI7
aI17
aI57
aI81
aI17
aI177
aI633
aI49
aI2793
aI5229
aI5995
aI9491
asS'd'
I3517
sa(dp7034
S'a'
I15005
sS's'
I15
sS'm'
(lp7035
I1
aI1
aI7
aI15
aI17
aI19
aI65
aI57
aI189
aI239
aI1229
aI929
aI2681
aI12845
aI29311
asS'd'
I3518
sa(dp7036
S'a'
I15016
sS's'
I15
sS'm'
(lp7037
I1
aI3
aI1
aI11
aI13
aI47
aI61
aI203
aI383
aI875
aI943
aI139
aI4217
aI8279
aI1047
asS'd'
I3519
sa(dp7038
S'a'
I15024
sS's'
I15
sS'm'
(lp7039
I1
aI3
aI7
aI13
aI23
aI7
aI1
aI69
aI47
aI537
aI1325
aI3101
aI685
aI14057
aI19953
asS'd'
I3520
sa(dp7040
S'a'
I15030
sS's'
I15
sS'm'
(lp7041
I1
aI3
aI3
aI1
aI1
aI7
aI39
aI77
aI47
aI755
aI527
aI2985
aI5433
aI15095
aI27741
asS'd'
I3521
sa(dp7042
S'a'
I15048
sS's'
I15
sS'm'
(lp7043
I1
aI1
aI7
aI5
aI23
aI57
aI79
aI155
aI81
aI937
aI1071
aI3929
aI1655
aI3831
aI17351
asS'd'
I3522
sa(dp7044
S'a'
I15054
sS's'
I15
sS'm'
(lp7045
I1
aI3
aI7
aI1
aI3
aI41
aI13
aI235
aI207
aI487
aI1883
aI2247
aI1231
aI2751
aI15615
asS'd'
I3523
sa(dp7046
S'a'
I15066
sS's'
I15
sS'm'
(lp7047
I1
aI1
aI7
aI1
aI21
aI57
aI95
aI191
aI119
aI483
aI283
aI2221
aI5665
aI14819
aI26097
asS'd'
I3524
sa(dp7048
S'a'
I15071
sS's'
I15
sS'm'
(lp7049
I1
aI3
aI1
aI1
aI9
aI59
aI27
aI51
aI393
aI31
aI925
aI715
aI7705
aI14885
aI28767
asS'd'
I3525
sa(dp7050
S'a'
I15072
sS's'
I15
sS'm'
(lp7051
I1
aI1
aI3
aI3
aI3
aI61
aI109
aI131
aI113
aI249
aI1331
aI2521
aI2973
aI6375
aI20093
asS'd'
I3526
sa(dp7052
S'a'
I15075
sS's'
I15
sS'm'
(lp7053
I1
aI3
aI7
aI9
aI31
aI37
aI125
aI245
aI237
aI245
aI111
aI379
aI7495
aI15531
aI2325
asS'd'
I3527
sa(dp7054
S'a'
I15119
sS's'
I15
sS'm'
(lp7055
I1
aI3
aI7
aI13
aI21
aI21
aI57
aI21
aI449
aI969
aI417
aI2999
aI509
aI639
aI7797
asS'd'
I3528
sa(dp7056
S'a'
I15121
sS's'
I15
sS'm'
(lp7057
I1
aI3
aI7
aI7
aI7
aI29
aI11
aI175
aI55
aI705
aI891
aI863
aI3021
aI10071
aI10267
asS'd'
I3529
sa(dp7058
S'a'
I15133
sS's'
I15
sS'm'
(lp7059
I1
aI1
aI3
aI13
aI19
aI17
aI127
aI57
aI449
aI579
aI337
aI899
aI1235
aI11269
aI4245
asS'd'
I3530
sa(dp7060
S'a'
I15138
sS's'
I15
sS'm'
(lp7061
I1
aI1
aI1
aI11
aI29
aI61
aI35
aI75
aI249
aI683
aI287
aI45
aI3277
aI7521
aI2073
asS'd'
I3531
sa(dp7062
S'a'
I15143
sS's'
I15
sS'm'
(lp7063
I1
aI3
aI5
aI5
aI15
aI25
aI77
aI63
aI63
aI801
aI1387
aI1533
aI2185
aI10899
aI28381
asS'd'
I3532
sa(dp7064
S'a'
I15170
sS's'
I15
sS'm'
(lp7065
I1
aI3
aI1
aI1
aI21
aI49
aI3
aI249
aI419
aI575
aI87
aI3749
aI2523
aI16125
aI9483
asS'd'
I3533
sa(dp7066
S'a'
I15194
sS's'
I15
sS'm'
(lp7067
I1
aI1
aI1
aI11
aI21
aI43
aI85
aI211
aI449
aI439
aI1495
aI1841
aI4765
aI15253
aI1467
asS'd'
I3534
sa(dp7068
S'a'
I15212
sS's'
I15
sS'm'
(lp7069
I1
aI3
aI3
aI15
aI3
aI37
aI31
aI243
aI187
aI995
aI1103
aI2723
aI1523
aI15967
aI28649
asS'd'
I3535
sa(dp7070
S'a'
I15223
sS's'
I15
sS'm'
(lp7071
I1
aI1
aI5
aI11
aI9
aI11
aI17
aI87
aI335
aI125
aI1079
aI1657
aI1237
aI8059
aI29833
asS'd'
I3536
sa(dp7072
S'a'
I15229
sS's'
I15
sS'm'
(lp7073
I1
aI3
aI1
aI3
aI3
aI41
aI35
aI37
aI33
aI61
aI505
aI3203
aI5
aI101
aI8571
asS'd'
I3537
sa(dp7074
S'a'
I15254
sS's'
I15
sS'm'
(lp7075
I1
aI1
aI3
aI11
aI9
aI11
aI85
aI235
aI261
aI473
aI109
aI2127
aI5745
aI6389
aI7431
asS'd'
I3538
sa(dp7076
S'a'
I15263
sS's'
I15
sS'm'
(lp7077
I1
aI1
aI5
aI15
aI3
aI55
aI77
aI97
aI17
aI193
aI1267
aI3063
aI6531
aI9797
aI8639
asS'd'
I3539
sa(dp7078
S'a'
I15270
sS's'
I15
sS'm'
(lp7079
I1
aI1
aI5
aI5
aI25
aI41
aI79
aI83
aI485
aI697
aI149
aI1023
aI89
aI6115
aI15227
asS'd'
I3540
sa(dp7080
S'a'
I15273
sS's'
I15
sS'm'
(lp7081
I1
aI1
aI3
aI15
aI1
aI9
aI73
aI251
aI33
aI599
aI1017
aI353
aI4305
aI16033
aI29663
asS'd'
I3541
sa(dp7082
S'a'
I15287
sS's'
I15
sS'm'
(lp7083
I1
aI3
aI7
aI15
aI3
aI1
aI89
aI39
aI125
aI337
aI1445
aI3131
aI3685
aI9849
aI25829
asS'd'
I3542
sa(dp7084
S'a'
I15299
sS's'
I15
sS'm'
(lp7085
I1
aI3
aI7
aI3
aI19
aI1
aI63
aI179
aI349
aI135
aI185
aI2977
aI2527
aI15087
aI18133
asS'd'
I3543
sa(dp7086
S'a'
I15301
sS's'
I15
sS'm'
(lp7087
I1
aI1
aI3
aI3
aI23
aI7
aI91
aI221
aI325
aI723
aI345
aI81
aI8077
aI5501
aI8453
asS'd'
I3544
sa(dp7088
S'a'
I15306
sS's'
I15
sS'm'
(lp7089
I1
aI1
aI3
aI9
aI7
aI3
aI13
aI173
aI479
aI161
aI1989
aI3255
aI2069
aI6717
aI559
asS'd'
I3545
sa(dp7090
S'a'
I15313
sS's'
I15
sS'm'
(lp7091
I1
aI3
aI3
aI5
aI9
aI61
aI93
aI203
aI277
aI367
aI1141
aI981
aI4745
aI12625
aI21003
asS'd'
I3546
sa(dp7092
S'a'
I15320
sS's'
I15
sS'm'
(lp7093
I1
aI3
aI5
aI5
aI27
aI17
aI5
aI211
aI403
aI701
aI5
aI3091
aI4611
aI5615
aI23667
asS'd'
I3547
sa(dp7094
S'a'
I15323
sS's'
I15
sS'm'
(lp7095
I1
aI1
aI3
aI1
aI21
aI61
aI125
aI77
aI57
aI463
aI1499
aI791
aI2087
aI2805
aI18829
asS'd'
I3548
sa(dp7096
S'a'
I15329
sS's'
I15
sS'm'
(lp7097
I1
aI3
aI5
aI3
aI11
aI41
aI125
aI231
aI119
aI837
aI831
aI1331
aI7439
aI2381
aI3759
asS'd'
I3549
sa(dp7098
S'a'
I15332
sS's'
I15
sS'm'
(lp7099
I1
aI3
aI1
aI11
aI19
aI59
aI117
aI107
aI443
aI699
aI315
aI1491
aI2581
aI15871
aI17159
asS'd'
I3550
sa(dp7100
S'a'
I15341
sS's'
I15
sS'm'
(lp7101
I1
aI3
aI5
aI11
aI5
aI9
aI121
aI35
aI209
aI877
aI527
aI3493
aI4657
aI16093
aI17589
asS'd'
I3551
sa(dp7102
S'a'
I15359
sS's'
I15
sS'm'
(lp7103
I1
aI1
aI7
aI15
aI9
aI43
aI119
aI29
aI381
aI479
aI1443
aI3171
aI5053
aI9625
aI21161
asS'd'
I3552
sa(dp7104
S'a'
I15361
sS's'
I15
sS'm'
(lp7105
I1
aI1
aI3
aI5
aI15
aI21
aI31
aI223
aI83
aI399
aI1529
aI3605
aI6343
aI10469
aI10099
asS'd'
I3553
sa(dp7106
S'a'
I15364
sS's'
I15
sS'm'
(lp7107
I1
aI1
aI3
aI5
aI5
aI45
aI23
aI123
aI353
aI971
aI85
aI3069
aI3245
aI6569
aI13241
asS'd'
I3554
sa(dp7108
S'a'
I15367
sS's'
I15
sS'm'
(lp7109
I1
aI1
aI1
aI3
aI25
aI49
aI5
aI77
aI491
aI881
aI993
aI1195
aI7677
aI5709
aI10807
asS'd'
I3555
sa(dp7110
S'a'
I15379
sS's'
I15
sS'm'
(lp7111
I1
aI3
aI3
aI3
aI5
aI49
aI127
aI255
aI183
aI583
aI1599
aI987
aI7281
aI7149
aI28507
asS'd'
I3556
sa(dp7112
S'a'
I15391
sS's'
I15
sS'm'
(lp7113
I1
aI1
aI5
aI1
aI13
aI55
aI55
aI157
aI197
aI25
aI1971
aI3161
aI3903
aI8919
aI13563
asS'd'
I3557
sa(dp7114
S'a'
I15415
sS's'
I15
sS'm'
(lp7115
I1
aI3
aI7
aI9
aI3
aI37
aI79
aI193
aI25
aI103
aI843
aI2651
aI6341
aI2653
aI24337
asS'd'
I3558
sa(dp7116
S'a'
I15416
sS's'
I15
sS'm'
(lp7117
I1
aI1
aI7
aI3
aI25
aI49
aI99
aI139
aI45
aI211
aI2033
aI2331
aI7037
aI7177
aI1755
asS'd'
I3559
sa(dp7118
S'a'
I15419
sS's'
I15
sS'm'
(lp7119
I1
aI3
aI7
aI3
aI5
aI19
aI127
aI135
aI403
aI221
aI141
aI1065
aI3935
aI2745
aI25979
asS'd'
I3560
sa(dp7120
S'a'
I15433
sS's'
I15
sS'm'
(lp7121
I1
aI1
aI3
aI3
aI31
aI23
aI111
aI37
aI261
aI7
aI835
aI2379
aI7927
aI8181
aI23751
asS'd'
I3561
sa(dp7122
S'a'
I15469
sS's'
I15
sS'm'
(lp7123
I1
aI3
aI7
aI15
aI1
aI39
aI79
aI3
aI103
aI427
aI1917
aI809
aI5039
aI689
aI1939
asS'd'
I3562
sa(dp7124
S'a'
I15478
sS's'
I15
sS'm'
(lp7125
I1
aI1
aI1
aI15
aI29
aI37
aI39
aI243
aI149
aI353
aI763
aI3405
aI5751
aI9441
aI6653
asS'd'
I3563
sa(dp7126
S'a'
I15481
sS's'
I15
sS'm'
(lp7127
I1
aI3
aI3
aI11
aI1
aI57
aI125
aI151
aI445
aI423
aI841
aI2265
aI5017
aI15863
aI13057
asS'd'
I3564
sa(dp7128
S'a'
I15482
sS's'
I15
sS'm'
(lp7129
I1
aI3
aI5
aI13
aI11
aI49
aI61
aI159
aI211
aI917
aI561
aI1903
aI3985
aI11117
aI28969
asS'd'
I3565
sa(dp7130
S'a'
I15498
sS's'
I15
sS'm'
(lp7131
I1
aI3
aI5
aI13
aI29
aI5
aI35
aI51
aI91
aI291
aI9
aI3713
aI3341
aI4551
aI12085
asS'd'
I3566
sa(dp7132
S'a'
I15505
sS's'
I15
sS'm'
(lp7133
I1
aI3
aI3
aI1
aI1
aI39
aI111
aI141
aI319
aI179
aI1709
aI1605
aI5063
aI13279
aI10003
asS'd'
I3567
sa(dp7134
S'a'
I15517
sS's'
I15
sS'm'
(lp7135
I1
aI1
aI3
aI9
aI7
aI59
aI91
aI41
aI343
aI475
aI1669
aI2311
aI5141
aI12661
aI25847
asS'd'
I3568
sa(dp7136
S'a'
I15518
sS's'
I15
sS'm'
(lp7137
I1
aI3
aI5
aI9
aI9
aI11
aI49
aI221
aI1
aI243
aI791
aI229
aI503
aI373
aI19189
asS'd'
I3569
sa(dp7138
S'a'
I15527
sS's'
I15
sS'm'
(lp7139
I1
aI1
aI5
aI11
aI17
aI13
aI45
aI57
aI215
aI491
aI1601
aI2183
aI3713
aI429
aI22007
asS'd'
I3570
sa(dp7140
S'a'
I15528
sS's'
I15
sS'm'
(lp7141
I1
aI1
aI3
aI11
aI31
aI61
aI23
aI237
aI261
aI955
aI1085
aI1541
aI2601
aI909
aI7749
asS'd'
I3571
sa(dp7142
S'a'
I15545
sS's'
I15
sS'm'
(lp7143
I1
aI1
aI3
aI9
aI13
aI11
aI121
aI173
aI177
aI551
aI1757
aI2745
aI2265
aI4611
aI743
asS'd'
I3572
sa(dp7144
S'a'
I15548
sS's'
I15
sS'm'
(lp7145
I1
aI1
aI3
aI15
aI23
aI43
aI107
aI239
aI463
aI369
aI1857
aI1073
aI1247
aI1029
aI22557
asS'd'
I3573
sa(dp7146
S'a'
I15554
sS's'
I15
sS'm'
(lp7147
I1
aI1
aI3
aI11
aI23
aI35
aI89
aI93
aI41
aI941
aI1141
aI2339
aI1423
aI8007
aI28685
asS'd'
I3574
sa(dp7148
S'a'
I15565
sS's'
I15
sS'm'
(lp7149
I1
aI3
aI5
aI13
aI29
aI7
aI79
aI15
aI59
aI145
aI1237
aI2215
aI1257
aI12621
aI31101
asS'd'
I3575
sa(dp7150
S'a'
I15577
sS's'
I15
sS'm'
(lp7151
I1
aI1
aI3
aI7
aI13
aI55
aI57
aI229
aI205
aI1009
aI341
aI3901
aI5189
aI957
aI32587
asS'd'
I3576
sa(dp7152
S'a'
I15580
sS's'
I15
sS'm'
(lp7153
I1
aI3
aI7
aI11
aI1
aI1
aI41
aI7
aI365
aI407
aI1609
aI1423
aI6483
aI5171
aI32519
asS'd'
I3577
sa(dp7154
S'a'
I15587
sS's'
I15
sS'm'
(lp7155
I1
aI3
aI7
aI3
aI17
aI31
aI125
aI27
aI125
aI335
aI1395
aI2639
aI329
aI2549
aI14449
asS'd'
I3578
sa(dp7156
S'a'
I15601
sS's'
I15
sS'm'
(lp7157
I1
aI3
aI3
aI7
aI19
aI45
aI11
aI73
aI123
aI179
aI1685
aI3385
aI2379
aI3387
aI16793
asS'd'
I3579
sa(dp7158
S'a'
I15604
sS's'
I15
sS'm'
(lp7159
I1
aI3
aI7
aI5
aI31
aI25
aI47
aI153
aI121
aI453
aI935
aI3953
aI2081
aI12145
aI24979
asS'd'
I3580
sa(dp7160
S'a'
I15611
sS's'
I15
sS'm'
(lp7161
I1
aI1
aI7
aI13
aI25
aI11
aI65
aI3
aI277
aI237
aI1129
aI1801
aI4165
aI9065
aI18747
asS'd'
I3581
sa(dp7162
S'a'
I15616
sS's'
I15
sS'm'
(lp7163
I1
aI1
aI7
aI7
aI13
aI5
aI37
aI253
aI507
aI645
aI1355
aI3401
aI6707
aI6329
aI11237
asS'd'
I3582
sa(dp7164
S'a'
I15619
sS's'
I15
sS'm'
(lp7165
I1
aI1
aI3
aI15
aI17
aI49
aI3
aI233
aI407
aI451
aI69
aI3859
aI3171
aI12303
aI21031
asS'd'
I3583
sa(dp7166
S'a'
I15625
sS's'
I15
sS'm'
(lp7167
I1
aI1
aI3
aI3
aI9
aI53
aI119
aI117
aI401
aI903
aI1449
aI3639
aI4083
aI2095
aI22085
asS'd'
I3584
sa(dp7168
S'a'
I15633
sS's'
I15
sS'm'
(lp7169
I1
aI3
aI7
aI15
aI5
aI61
aI117
aI193
aI137
aI431
aI195
aI4019
aI3047
aI5049
aI14281
asS'd'
I3585
sa(dp7170
S'a'
I15674
sS's'
I15
sS'm'
(lp7171
I1
aI1
aI1
aI15
aI17
aI19
aI29
aI83
aI449
aI257
aI1105
aI1949
aI1749
aI3459
aI6343
asS'd'
I3586
sa(dp7172
S'a'
I15681
sS's'
I15
sS'm'
(lp7173
I1
aI1
aI1
aI15
aI23
aI39
aI61
aI219
aI109
aI365
aI863
aI1813
aI6673
aI15999
aI5101
asS'd'
I3587
sa(dp7174
S'a'
I15691
sS's'
I15
sS'm'
(lp7175
I1
aI1
aI5
aI5
aI13
aI11
aI37
aI151
aI365
aI719
aI1233
aI2425
aI1285
aI1721
aI1205
asS'd'
I3588
sa(dp7176
S'a'
I15693
sS's'
I15
sS'm'
(lp7177
I1
aI3
aI3
aI3
aI7
aI53
aI109
aI153
aI45
aI425
aI1741
aI1229
aI4405
aI8071
aI25155
asS'd'
I3589
sa(dp7178
S'a'
I15696
sS's'
I15
sS'm'
(lp7179
I1
aI3
aI1
aI1
aI1
aI13
aI39
aI49
aI413
aI77
aI1367
aI2553
aI5563
aI7659
aI3467
asS'd'
I3590
sa(dp7180
S'a'
I15712
sS's'
I15
sS'm'
(lp7181
I1
aI1
aI5
aI9
aI3
aI49
aI23
aI11
aI445
aI121
aI1505
aI877
aI4137
aI1809
aI2429
asS'd'
I3591
sa(dp7182
S'a'
I15717
sS's'
I15
sS'm'
(lp7183
I1
aI1
aI1
aI11
aI21
aI13
aI93
aI33
aI493
aI805
aI775
aI2939
aI2961
aI13625
aI31879
asS'd'
I3592
sa(dp7184
S'a'
I15724
sS's'
I15
sS'm'
(lp7185
I1
aI1
aI7
aI5
aI1
aI59
aI63
aI131
aI373
aI23
aI337
aI2107
aI5315
aI4889
aI22851
asS'd'
I3593
sa(dp7186
S'a'
I15727
sS's'
I15
sS'm'
(lp7187
I1
aI1
aI3
aI13
aI21
aI47
aI15
aI131
aI353
aI793
aI1891
aI1757
aI5793
aI1147
aI23697
asS'd'
I3594
sa(dp7188
S'a'
I15730
sS's'
I15
sS'm'
(lp7189
I1
aI3
aI5
aI13
aI7
aI59
aI25
aI135
aI259
aI109
aI1835
aI429
aI8153
aI7355
aI145
asS'd'
I3595
sa(dp7190
S'a'
I15746
sS's'
I15
sS'm'
(lp7191
I1
aI3
aI3
aI13
aI9
aI47
aI121
aI89
aI89
aI635
aI1079
aI2353
aI4803
aI11369
aI12653
asS'd'
I3596
sa(dp7192
S'a'
I15751
sS's'
I15
sS'm'
(lp7193
I1
aI3
aI5
aI9
aI23
aI39
aI49
aI231
aI105
aI603
aI613
aI2021
aI6073
aI11819
aI10595
asS'd'
I3597
sa(dp7194
S'a'
I15760
sS's'
I15
sS'm'
(lp7195
I1
aI3
aI7
aI7
aI7
aI19
aI19
aI155
aI347
aI387
aI1459
aI3793
aI619
aI14437
aI2455
asS'd'
I3598
sa(dp7196
S'a'
I15770
sS's'
I15
sS'm'
(lp7197
I1
aI1
aI1
aI15
aI21
aI35
aI19
aI185
aI483
aI425
aI479
aI3429
aI5403
aI10791
aI14219
asS'd'
I3599
sa(dp7198
S'a'
I15782
sS's'
I15
sS'm'
(lp7199
I1
aI1
aI3
aI11
aI5
aI51
aI105
aI63
aI493
aI677
aI1457
aI2865
aI5619
aI9321
aI19583
asS'd'
I3600
sa(dp7200
S'a'
I15791
sS's'
I15
sS'm'
(lp7201
I1
aI1
aI3
aI3
aI23
aI1
aI77
aI177
aI263
aI289
aI1567
aI3837
aI5359
aI3269
aI16023
asS'd'
I3601
sa(dp7202
S'a'
I15796
sS's'
I15
sS'm'
(lp7203
I1
aI1
aI7
aI3
aI13
aI61
aI79
aI77
aI51
aI953
aI1417
aI795
aI4467
aI2981
aI25131
asS'd'
I3602
sa(dp7204
S'a'
I15808
sS's'
I15
sS'm'
(lp7205
I1
aI1
aI5
aI13
aI23
aI13
aI29
aI185
aI337
aI7
aI149
aI3609
aI8119
aI9545
aI16579
asS'd'
I3603
sa(dp7206
S'a'
I15814
sS's'
I15
sS'm'
(lp7207
I1
aI3
aI1
aI5
aI23
aI9
aI123
aI15
aI99
aI55
aI1021
aI3709
aI1521
aI15189
aI22193
asS'd'
I3604
sa(dp7208
S'a'
I15825
sS's'
I15
sS'm'
(lp7209
I1
aI3
aI7
aI9
aI13
aI41
aI39
aI45
aI49
aI181
aI1587
aI3213
aI1037
aI14775
aI3333
asS'd'
I3605
sa(dp7210
S'a'
I15828
sS's'
I15
sS'm'
(lp7211
I1
aI1
aI1
aI7
aI29
aI55
aI59
aI31
aI411
aI601
aI191
aI283
aI3211
aI7951
aI7919
asS'd'
I3606
sa(dp7212
S'a'
I15835
sS's'
I15
sS'm'
(lp7213
I1
aI1
aI7
aI7
aI21
aI47
aI7
aI193
aI343
aI831
aI1267
aI3289
aI1015
aI13093
aI2717
asS'd'
I3607
sa(dp7214
S'a'
I15844
sS's'
I15
sS'm'
(lp7215
I1
aI3
aI7
aI1
aI17
aI9
aI97
aI19
aI279
aI827
aI1699
aI3573
aI3137
aI3535
aI17791
asS'd'
I3608
sa(dp7216
S'a'
I15847
sS's'
I15
sS'm'
(lp7217
I1
aI1
aI5
aI11
aI27
aI15
aI103
aI135
aI35
aI625
aI1575
aI97
aI7013
aI13353
aI19333
asS'd'
I3609
sa(dp7218
S'a'
I15853
sS's'
I15
sS'm'
(lp7219
I1
aI3
aI3
aI7
aI17
aI13
aI49
aI135
aI435
aI743
aI1799
aI2655
aI4839
aI2893
aI31153
asS'd'
I3610
sa(dp7220
S'a'
I15856
sS's'
I15
sS'm'
(lp7221
I1
aI1
aI5
aI1
aI3
aI41
aI1
aI195
aI53
aI803
aI1575
aI2939
aI3873
aI10495
aI5211
asS'd'
I3611
sa(dp7222
S'a'
I15877
sS's'
I15
sS'm'
(lp7223
I1
aI3
aI1
aI15
aI19
aI19
aI37
aI59
aI355
aI483
aI685
aI3899
aI4645
aI15127
aI3479
asS'd'
I3612
sa(dp7224
S'a'
I15878
sS's'
I15
sS'm'
(lp7225
I1
aI1
aI5
aI3
aI25
aI9
aI9
aI229
aI101
aI631
aI1165
aI4091
aI3723
aI10655
aI9463
asS'd'
I3613
sa(dp7226
S'a'
I15887
sS's'
I15
sS'm'
(lp7227
I1
aI3
aI5
aI15
aI5
aI13
aI91
aI61
aI19
aI469
aI1675
aI3331
aI3121
aI3435
aI4111
asS'd'
I3614
sa(dp7228
S'a'
I15908
sS's'
I15
sS'm'
(lp7229
I1
aI1
aI7
aI1
aI31
aI61
aI23
aI83
aI165
aI551
aI1097
aI3825
aI5385
aI4723
aI3635
asS'd'
I3615
sa(dp7230
S'a'
I15917
sS's'
I15
sS'm'
(lp7231
I1
aI3
aI7
aI15
aI9
aI31
aI11
aI121
aI503
aI855
aI561
aI1647
aI1229
aI1147
aI15997
asS'd'
I3616
sa(dp7232
S'a'
I15923
sS's'
I15
sS'm'
(lp7233
I1
aI3
aI7
aI13
aI21
aI47
aI41
aI195
aI197
aI719
aI1263
aI3609
aI7515
aI2659
aI30713
asS'd'
I3617
sa(dp7234
S'a'
I15930
sS's'
I15
sS'm'
(lp7235
I1
aI1
aI1
aI7
aI31
aI61
aI101
aI101
aI479
aI571
aI605
aI301
aI6633
aI15587
aI23665
asS'd'
I3618
sa(dp7236
S'a'
I15937
sS's'
I15
sS'm'
(lp7237
I1
aI3
aI7
aI3
aI25
aI39
aI35
aI225
aI135
aI463
aI53
aI709
aI5129
aI4135
aI10421
asS'd'
I3619
sa(dp7238
S'a'
I15958
sS's'
I15
sS'm'
(lp7239
I1
aI1
aI5
aI13
aI19
aI55
aI107
aI15
aI163
aI287
aI673
aI899
aI5197
aI4619
aI3465
asS'd'
I3620
sa(dp7240
S'a'
I15977
sS's'
I15
sS'm'
(lp7241
I1
aI3
aI3
aI5
aI21
aI49
aI15
aI105
aI283
aI877
aI1875
aI1079
aI3431
aI13053
aI26599
asS'd'
I3621
sa(dp7242
S'a'
I15991
sS's'
I15
sS'm'
(lp7243
I1
aI1
aI7
aI1
aI1
aI1
aI95
aI113
aI119
aI575
aI1159
aI2325
aI6895
aI12177
aI4369
asS'd'
I3622
sa(dp7244
S'a'
I16007
sS's'
I15
sS'm'
(lp7245
I1
aI1
aI1
aI11
aI25
aI25
aI83
aI207
aI301
aI729
aI1947
aI2321
aI3621
aI15707
aI11303
asS'd'
I3623
sa(dp7246
S'a'
I16011
sS's'
I15
sS'm'
(lp7247
I1
aI1
aI5
aI5
aI7
aI63
aI83
aI105
aI211
aI175
aI1817
aI2883
aI5385
aI7437
aI24865
asS'd'
I3624
sa(dp7248
S'a'
I16014
sS's'
I15
sS'm'
(lp7249
I1
aI3
aI7
aI5
aI23
aI39
aI19
aI211
aI151
aI295
aI573
aI223
aI5065
aI6345
aI23187
asS'd'
I3625
sa(dp7250
S'a'
I16021
sS's'
I15
sS'm'
(lp7251
I1
aI1
aI7
aI11
aI15
aI31
aI89
aI123
aI57
aI695
aI685
aI1799
aI659
aI9929
aI22933
asS'd'
I3626
sa(dp7252
S'a'
I16022
sS's'
I15
sS'm'
(lp7253
I1
aI1
aI7
aI7
aI19
aI17
aI27
aI137
aI117
aI141
aI1481
aI869
aI7061
aI3073
aI19671
asS'd'
I3627
sa(dp7254
S'a'
I16028
sS's'
I15
sS'm'
(lp7255
I1
aI3
aI3
aI11
aI9
aI19
aI123
aI93
aI39
aI517
aI883
aI3769
aI2267
aI8089
aI6617
asS'd'
I3628
sa(dp7256
S'a'
I16035
sS's'
I15
sS'm'
(lp7257
I1
aI3
aI1
aI7
aI9
aI61
aI51
aI241
aI319
aI853
aI1239
aI899
aI105
aI1677
aI29351
asS'd'
I3629
sa(dp7258
S'a'
I16041
sS's'
I15
sS'm'
(lp7259
I1
aI1
aI7
aI15
aI13
aI59
aI85
aI175
aI223
aI87
aI905
aI3175
aI3405
aI3489
aI18475
asS'd'
I3630
sa(dp7260
S'a'
I16056
sS's'
I15
sS'm'
(lp7261
I1
aI1
aI1
aI15
aI1
aI55
aI79
aI97
aI315
aI605
aI851
aI4015
aI3689
aI9371
aI31523
asS'd'
I3631
sa(dp7262
S'a'
I16069
sS's'
I15
sS'm'
(lp7263
I1
aI1
aI5
aI15
aI1
aI39
aI91
aI27
aI211
aI881
aI1375
aI2307
aI5791
aI10185
aI23093
asS'd'
I3632
sa(dp7264
S'a'
I16076
sS's'
I15
sS'm'
(lp7265
I1
aI3
aI1
aI5
aI3
aI17
aI59
aI219
aI105
aI623
aI21
aI2843
aI3427
aI4799
aI3793
asS'd'
I3633
sa(dp7266
S'a'
I16081
sS's'
I15
sS'm'
(lp7267
I1
aI3
aI3
aI7
aI21
aI55
aI17
aI29
aI397
aI93
aI1981
aI4047
aI935
aI5971
aI14589
asS'd'
I3634
sa(dp7268
S'a'
I16087
sS's'
I15
sS'm'
(lp7269
I1
aI1
aI3
aI9
aI5
aI57
aI63
aI27
aI373
aI815
aI167
aI205
aI367
aI4945
aI30041
asS'd'
I3635
sa(dp7270
S'a'
I16088
sS's'
I15
sS'm'
(lp7271
I1
aI1
aI5
aI9
aI7
aI3
aI69
aI35
aI197
aI309
aI1729
aI3735
aI1523
aI10427
aI26253
asS'd'
I3636
sa(dp7272
S'a'
I16110
sS's'
I15
sS'm'
(lp7273
I1
aI1
aI3
aI7
aI7
aI49
aI35
aI189
aI297
aI311
aI2025
aI305
aI3863
aI14393
aI2533
asS'd'
I3637
sa(dp7274
S'a'
I16112
sS's'
I15
sS'm'
(lp7275
I1
aI3
aI3
aI9
aI17
aI31
aI5
aI17
aI167
aI601
aI909
aI3149
aI2533
aI12123
aI25325
asS'd'
I3638
sa(dp7276
S'a'
I16117
sS's'
I15
sS'm'
(lp7277
I1
aI3
aI5
aI3
aI11
aI41
aI69
aI199
aI79
aI611
aI133
aI3519
aI5955
aI4609
aI27403
asS'd'
I3639
sa(dp7278
S'a'
I16150
sS's'
I15
sS'm'
(lp7279
I1
aI3
aI3
aI13
aI3
aI17
aI53
aI165
aI361
aI797
aI1447
aI869
aI6707
aI6541
aI32249
asS'd'
I3640
sa(dp7280
S'a'
I16153
sS's'
I15
sS'm'
(lp7281
I1
aI3
aI1
aI1
aI29
aI47
aI17
aI45
aI473
aI199
aI1595
aI3095
aI3635
aI6965
aI21859
asS'd'
I3641
sa(dp7282
S'a'
I16160
sS's'
I15
sS'm'
(lp7283
I1
aI1
aI3
aI9
aI1
aI15
aI59
aI163
aI91
aI811
aI1087
aI1707
aI6743
aI12643
aI29901
asS'd'
I3642
sa(dp7284
S'a'
I16166
sS's'
I15
sS'm'
(lp7285
I1
aI1
aI1
aI3
aI19
aI21
aI7
aI209
aI121
aI821
aI709
aI1085
aI5333
aI7689
aI28355
asS'd'
I3643
sa(dp7286
S'a'
I16172
sS's'
I15
sS'm'
(lp7287
I1
aI3
aI1
aI15
aI5
aI27
aI115
aI31
aI37
aI79
aI1347
aI155
aI3709
aI13251
aI32151
asS'd'
I3644
sa(dp7288
S'a'
I16190
sS's'
I15
sS'm'
(lp7289
I1
aI3
aI7
aI15
aI27
aI27
aI127
aI231
aI137
aI205
aI1665
aI1461
aI299
aI2797
aI879
asS'd'
I3645
sa(dp7290
S'a'
I16195
sS's'
I15
sS'm'
(lp7291
I1
aI1
aI1
aI7
aI13
aI3
aI127
aI13
aI253
aI481
aI1435
aI1895
aI2665
aI7611
aI17761
asS'd'
I3646
sa(dp7292
S'a'
I16204
sS's'
I15
sS'm'
(lp7293
I1
aI1
aI3
aI7
aI7
aI21
aI71
aI247
aI301
aI183
aI1785
aI331
aI4835
aI2251
aI4493
asS'd'
I3647
sa(dp7294
S'a'
I16216
sS's'
I15
sS'm'
(lp7295
I1
aI3
aI7
aI9
aI9
aI1
aI77
aI169
aI103
aI647
aI1959
aI1847
aI5803
aI3421
aI15915
asS'd'
I3648
sa(dp7296
S'a'
I16222
sS's'
I15
sS'm'
(lp7297
I1
aI3
aI1
aI7
aI19
aI17
aI81
aI45
aI263
aI549
aI1607
aI2177
aI1117
aI14427
aI16451
asS'd'
I3649
sa(dp7298
S'a'
I16228
sS's'
I15
sS'm'
(lp7299
I1
aI1
aI7
aI15
aI27
aI25
aI27
aI27
aI33
aI813
aI1667
aI253
aI2749
aI927
aI29707
asS'd'
I3650
sa(dp7300
S'a'
I16245
sS's'
I15
sS'm'
(lp7301
I1
aI1
aI7
aI3
aI17
aI29
aI13
aI67
aI417
aI303
aI19
aI3809
aI7225
aI12775
aI3933
asS'd'
I3651
sa(dp7302
S'a'
I16255
sS's'
I15
sS'm'
(lp7303
I1
aI1
aI1
aI11
aI13
aI41
aI77
aI217
aI281
aI659
aI1099
aI3047
aI1619
aI525
aI4313
asS'd'
I3652
sa(dp7304
S'a'
I16265
sS's'
I15
sS'm'
(lp7305
I1
aI3
aI3
aI9
aI23
aI47
aI5
aI33
aI219
aI531
aI77
aI2307
aI1893
aI8335
aI8281
asS'd'
I3653
sa(dp7306
S'a'
I16273
sS's'
I15
sS'm'
(lp7307
I1
aI3
aI7
aI3
aI3
aI35
aI27
aI249
aI159
aI495
aI431
aI3001
aI1475
aI11505
aI15693
asS'd'
I3654
sa(dp7308
S'a'
I16276
sS's'
I15
sS'm'
(lp7309
I1
aI1
aI5
aI9
aI21
aI49
aI43
aI159
aI465
aI959
aI179
aI993
aI121
aI11569
aI21027
asS'd'
I3655
sa(dp7310
S'a'
I16283
sS's'
I15
sS'm'
(lp7311
I1
aI3
aI1
aI5
aI1
aI61
aI9
aI221
aI231
aI55
aI191
aI2829
aI3331
aI8911
aI15109
asS'd'
I3656
sa(dp7312
S'a'
I16295
sS's'
I15
sS'm'
(lp7313
I1
aI1
aI7
aI1
aI7
aI35
aI67
aI97
aI159
aI191
aI935
aI3151
aI6397
aI10751
aI1835
asS'd'
I3657
sa(dp7314
S'a'
I16304
sS's'
I15
sS'm'
(lp7315
I1
aI1
aI1
aI7
aI15
aI39
aI127
aI163
aI437
aI333
aI829
aI753
aI8151
aI13239
aI523
asS'd'
I3658
sa(dp7316
S'a'
I16313
sS's'
I15
sS'm'
(lp7317
I1
aI1
aI3
aI13
aI9
aI25
aI73
aI155
aI445
aI239
aI2035
aI15
aI5243
aI15531
aI1733
asS'd'
I3659
sa(dp7318
S'a'
I16319
sS's'
I15
sS'm'
(lp7319
I1
aI3
aI7
aI15
aI5
aI25
aI3
aI55
aI117
aI57
aI783
aI1509
aI7043
aI13159
aI8557
asS'd'
I3660
sa(dp7320
S'a'
I16328
sS's'
I15
sS'm'
(lp7321
I1
aI3
aI5
aI1
aI21
aI55
aI89
aI119
aI199
aI79
aI161
aI1597
aI3263
aI3335
aI5757
asS'd'
I3661
sa(dp7322
S'a'
I16345
sS's'
I15
sS'm'
(lp7323
I1
aI3
aI7
aI5
aI27
aI23
aI85
aI113
aI111
aI211
aI389
aI1513
aI2759
aI7945
aI931
asS'd'
I3662
sa(dp7324
S'a'
I16355
sS's'
I15
sS'm'
(lp7325
I1
aI1
aI1
aI7
aI1
aI5
aI17
aI177
aI357
aI619
aI5
aI2583
aI621
aI2973
aI28845
asS'd'
I3663
sa(dp7326
S'a'
I16364
sS's'
I15
sS'm'
(lp7327
I1
aI3
aI7
aI13
aI11
aI21
aI47
aI99
aI421
aI279
aI1541
aI1305
aI4571
aI6127
aI20735
asS'd'
I3664
sa(dp7328
S'a'
I16372
sS's'
I15
sS'm'
(lp7329
I1
aI3
aI5
aI5
aI23
aI43
aI19
aI137
aI425
aI409
aI1625
aI2671
aI4385
aI3197
aI25753
asS'd'
I3665
sa(dp7330
S'a'
I16375
sS's'
I15
sS'm'
(lp7331
I1
aI1
aI7
aI5
aI27
aI17
aI57
aI15
aI383
aI181
aI951
aI2115
aI5237
aI1495
aI9671
asS'd'
I3666
sa(dp7332
S'a'
I16382
sS's'
I15
sS'm'
(lp7333
I1
aI3
aI3
aI11
aI9
aI1
aI53
aI127
aI375
aI499
aI1487
aI121
aI1465
aI3175
aI24337
asS'd'
I3667
sa(dp7334
S'a'
I22
sS's'
I16
sS'm'
(lp7335
I1
aI3
aI7
aI11
aI29
aI35
aI67
aI129
aI221
aI439
aI1159
aI3501
aI7741
aI8885
aI11381
aI20707
asS'd'
I3668
sa(dp7336
S'a'
I28
sS's'
I16
sS'm'
(lp7337
I1
aI3
aI5
aI11
aI29
aI59
aI23
aI117
aI343
aI637
aI1825
aI1687
aI2823
aI11641
aI3311
aI23603
asS'd'
I3669
sa(dp7338
S'a'
I31
sS's'
I16
sS'm'
(lp7339
I1
aI1
aI5
aI11
aI1
aI35
aI103
aI155
aI233
aI575
aI1761
aI503
aI4175
aI6105
aI29737
aI32681
asS'd'
I3670
sa(dp7340
S'a'
I41
sS's'
I16
sS'm'
(lp7341
I1
aI3
aI3
aI1
aI5
aI63
aI27
aI71
aI245
aI433
aI1779
aI2475
aI5479
aI4705
aI10795
aI34247
asS'd'
I3671
sa(dp7342
S'a'
I94
sS's'
I16
sS'm'
(lp7343
I1
aI3
aI5
aI7
aI29
aI45
aI117
aI5
aI393
aI849
aI843
aI3131
aI6995
aI9979
aI28907
aI30115
asS'd'
I3672
sa(dp7344
S'a'
I107
sS's'
I16
sS'm'
(lp7345
I1
aI3
aI5
aI9
aI27
aI29
aI69
aI5
aI395
aI561
aI1531
aI409
aI2779
aI8785
aI16405
aI27315
asS'd'
I3673
sa(dp7346
S'a'
I151
sS's'
I16
sS'm'
(lp7347
I1
aI3
aI1
aI9
aI15
aI29
aI85
aI3
aI331
aI19
aI1941
aI567
aI6957
aI747
aI1627
aI11347
asS'd'
I3674
sa(dp7348
S'a'
I158
sS's'
I16
sS'm'
(lp7349
I1
aI1
aI3
aI9
aI27
aI45
aI47
aI127
aI133
aI921
aI1817
aI2231
aI6333
aI14371
aI12799
aI9831
asS'd'
I3675
sa(dp7350
S'a'
I167
sS's'
I16
sS'm'
(lp7351
I1
aI1
aI5
aI15
aI31
aI7
aI125
aI13
aI455
aI159
aI331
aI3629
aI4705
aI11261
aI3657
aI36307
asS'd'
I3676
sa(dp7352
S'a'
I174
sS's'
I16
sS'm'
(lp7353
I1
aI1
aI5
aI9
aI11
aI53
aI51
aI35
aI87
aI885
aI1975
aI3899
aI1013
aI7667
aI32385
aI33635
asS'd'
I3677
sa(dp7354
S'a'
I203
sS's'
I16
sS'm'
(lp7355
I1
aI1
aI1
aI3
aI7
aI45
aI107
aI177
aI193
aI765
aI731
aI139
aI5563
aI623
aI16485
aI54999
asS'd'
I3678
sa(dp7356
S'a'
I208
sS's'
I16
sS'm'
(lp7357
I1
aI1
aI5
aI9
aI17
aI53
aI117
aI69
aI385
aI587
aI1483
aI149
aI2769
aI3013
aI18183
aI10173
asS'd'
I3679
sa(dp7358
S'a'
I214
sS's'
I16
sS'm'
(lp7359
I1
aI1
aI5
aI11
aI5
aI3
aI25
aI153
aI351
aI749
aI801
aI3077
aI3209
aI11189
aI25241
aI14115
asS'd'
I3680
sa(dp7360
S'a'
I223
sS's'
I16
sS'm'
(lp7361
I1
aI1
aI7
aI9
aI1
aI47
aI41
aI247
aI135
aI163
aI899
aI1517
aI5647
aI10595
aI32531
aI12497
asS'd'
I3681
sa(dp7362
S'a'
I227
sS's'
I16
sS'm'
(lp7363
I1
aI3
aI5
aI11
aI5
aI61
aI111
aI215
aI251
aI279
aI825
aI2155
aI3527
aI173
aI10973
aI59257
asS'd'
I3682
sa(dp7364
S'a'
I266
sS's'
I16
sS'm'
(lp7365
I1
aI3
aI5
aI11
aI25
aI15
aI71
aI83
aI135
aI231
aI1415
aI3761
aI7513
aI8337
aI28979
aI43615
asS'd'
I3683
sa(dp7366
S'a'
I268
sS's'
I16
sS'm'
(lp7367
I1
aI3
aI5
aI13
aI19
aI5
aI55
aI165
aI141
aI119
aI1891
aI2255
aI4735
aI16217
aI26195
aI50527
asS'd'
I3684
sa(dp7368
S'a'
I274
sS's'
I16
sS'm'
(lp7369
I1
aI1
aI7
aI15
aI23
aI59
aI59
aI191
aI1
aI855
aI453
aI2619
aI5013
aI14749
aI24335
aI44339
asS'd'
I3685
sa(dp7370
S'a'
I279
sS's'
I16
sS'm'
(lp7371
I1
aI1
aI1
aI13
aI15
aI41
aI51
aI147
aI229
aI495
aI1191
aI867
aI1525
aI581
aI29713
aI26391
asS'd'
I3686
sa(dp7372
S'a'
I302
sS's'
I16
sS'm'
(lp7373
I1
aI1
aI1
aI9
aI29
aI5
aI59
aI127
aI105
aI417
aI301
aI2249
aI6335
aI3513
aI17373
aI52977
asS'd'
I3687
sa(dp7374
S'a'
I310
sS's'
I16
sS'm'
(lp7375
I1
aI1
aI3
aI7
aI21
aI27
aI109
aI143
aI63
aI347
aI1429
aI2889
aI2597
aI10243
aI9913
aI22687
asS'd'
I3688
sa(dp7376
S'a'
I322
sS's'
I16
sS'm'
(lp7377
I1
aI3
aI5
aI5
aI7
aI3
aI125
aI147
aI313
aI351
aI1163
aI415
aI5615
aI5571
aI7089
aI55621
asS'd'
I3689
sa(dp7378
S'a'
I328
sS's'
I16
sS'm'
(lp7379
I1
aI3
aI3
aI3
aI31
aI43
aI101
aI93
aI9
aI671
aI135
aI333
aI2169
aI11169
aI7403
aI50707
asS'd'
I3690
sa(dp7380
S'a'
I336
sS's'
I16
sS'm'
(lp7381
I1
aI1
aI7
aI13
aI15
aI33
aI125
aI155
aI227
aI827
aI1047
aI2441
aI3007
aI10881
aI19969
aI63805
asS'd'
I3691
sa(dp7382
S'a'
I370
sS's'
I16
sS'm'
(lp7383
I1
aI3
aI3
aI5
aI31
aI33
aI29
aI249
aI159
aI797
aI1475
aI841
aI6933
aI6417
aI25629
aI61865
asS'd'
I3692
sa(dp7384
S'a'
I398
sS's'
I16
sS'm'
(lp7385
I1
aI3
aI3
aI15
aI11
aI55
aI11
aI117
aI149
aI911
aI1589
aI3133
aI6477
aI6123
aI10471
aI41099
asS'd'
I3693
sa(dp7386
S'a'
I421
sS's'
I16
sS'm'
(lp7387
I1
aI3
aI3
aI9
aI27
aI37
aI1
aI119
aI509
aI969
aI831
aI3771
aI2093
aI13621
aI31737
aI43269
asS'd'
I3694
sa(dp7388
S'a'
I436
sS's'
I16
sS'm'
(lp7389
I1
aI1
aI1
aI1
aI9
aI23
aI119
aI109
aI487
aI753
aI1673
aI2163
aI3349
aI4741
aI29971
aI3407
asS'd'
I3695
sa(dp7390
S'a'
I440
sS's'
I16
sS'm'
(lp7391
I1
aI3
aI3
aI7
aI25
aI7
aI67
aI9
aI461
aI631
aI651
aI2271
aI5663
aI2621
aI3953
aI20975
asS'd'
I3696
sa(dp7392
S'a'
I451
sS's'
I16
sS'm'
(lp7393
I1
aI1
aI5
aI11
aI13
aI31
aI29
aI255
aI371
aI517
aI845
aI3649
aI1187
aI10061
aI22887
aI58417
asS'd'
I3697
sa(dp7394
S'a'
I454
sS's'
I16
sS'm'
(lp7395
I1
aI3
aI5
aI13
aI29
aI1
aI11
aI137
aI151
aI249
aI167
aI1243
aI997
aI11023
aI11875
aI42315
asS'd'
I3698
sa(dp7396
S'a'
I463
sS's'
I16
sS'm'
(lp7397
I1
aI1
aI5
aI5
aI5
aI55
aI103
aI71
aI255
aI1023
aI209
aI1005
aI2147
aI11527
aI17863
aI6661
asS'd'
I3699
sa(dp7398
S'a'
I465
sS's'
I16
sS'm'
(lp7399
I1
aI1
aI3
aI3
aI31
aI39
aI7
aI151
aI353
aI775
aI1313
aI1257
aI4197
aI2625
aI9571
aI27269
asS'd'
I3700
sa(dp7400
S'a'
I494
sS's'
I16
sS'm'
(lp7401
I1
aI1
aI1
aI3
aI7
aI17
aI3
aI127
aI501
aI503
aI1879
aI2329
aI3049
aI10603
aI2111
aI33189
asS'd'
I3701
sa(dp7402
S'a'
I508
sS's'
I16
sS'm'
(lp7403
I1
aI3
aI3
aI7
aI13
aI59
aI93
aI13
aI375
aI483
aI1991
aI2257
aI3003
aI1699
aI4339
aI51827
asS'd'
I3702
sa(dp7404
S'a'
I532
sS's'
I16
sS'm'
(lp7405
I1
aI3
aI7
aI15
aI27
aI41
aI59
aI225
aI405
aI807
aI1545
aI2581
aI1173
aI14137
aI3413
aI39299
asS'd'
I3703
sa(dp7406
S'a'
I555
sS's'
I16
sS'm'
(lp7407
I1
aI1
aI1
aI3
aI9
aI23
aI37
aI123
aI465
aI1023
aI1065
aI1455
aI5107
aI3839
aI20451
aI11461
asS'd'
I3704
sa(dp7408
S'a'
I563
sS's'
I16
sS'm'
(lp7409
I1
aI1
aI1
aI11
aI19
aI55
aI91
aI121
aI317
aI199
aI215
aI3031
aI7223
aI11891
aI21463
aI64921
asS'd'
I3705
sa(dp7410
S'a'
I577
sS's'
I16
sS'm'
(lp7411
I1
aI3
aI7
aI11
aI19
aI5
aI5
aI115
aI399
aI219
aI71
aI1465
aI281
aI14451
aI26807
aI42541
asS'd'
I3706
sa(dp7412
S'a'
I580
sS's'
I16
sS'm'
(lp7413
I1
aI3
aI5
aI13
aI3
aI33
aI75
aI35
aI19
aI559
aI761
aI947
aI7479
aI15325
aI31453
aI20561
asS'd'
I3707
sa(dp7414
S'a'
I584
sS's'
I16
sS'm'
(lp7415
I1
aI3
aI3
aI13
aI23
aI47
aI99
aI73
aI331
aI353
aI401
aI1737
aI6235
aI13781
aI5547
aI56443
asS'd'
I3708
sa(dp7416
S'a'
I607
sS's'
I16
sS'm'
(lp7417
I1
aI3
aI3
aI13
aI21
aI37
aI41
aI205
aI87
aI399
aI51
aI3175
aI7403
aI12875
aI21129
aI7079
asS'd'
I3709
sa(dp7418
S'a'
I608
sS's'
I16
sS'm'
(lp7419
I1
aI3
aI5
aI11
aI15
aI47
aI33
aI39
aI465
aI871
aI277
aI2351
aI695
aI1953
aI24293
aI20595
asS'd'
I3710
sa(dp7420
S'a'
I665
sS's'
I16
sS'm'
(lp7421
I1
aI1
aI7
aI11
aI13
aI15
aI115
aI59
aI469
aI715
aI191
aI1927
aI905
aI13463
aI29005
aI46789
asS'd'
I3711
sa(dp7422
S'a'
I675
sS's'
I16
sS'm'
(lp7423
I1
aI3
aI5
aI9
aI13
aI55
aI79
aI17
aI265
aI887
aI905
aI3985
aI6907
aI3379
aI20055
aI58569
asS'd'
I3712
sa(dp7424
S'a'
I692
sS's'
I16
sS'm'
(lp7425
I1
aI1
aI7
aI11
aI21
aI29
aI23
aI109
aI17
aI427
aI1623
aI2219
aI3857
aI3709
aI25033
aI63823
asS'd'
I3713
sa(dp7426
S'a'
I707
sS's'
I16
sS'm'
(lp7427
I1
aI3
aI5
aI15
aI19
aI27
aI113
aI15
aI25
aI63
aI1885
aI2693
aI5301
aI9385
aI14137
aI26097
asS'd'
I3714
sa(dp7428
S'a'
I737
sS's'
I16
sS'm'
(lp7429
I1
aI3
aI3
aI11
aI17
aI5
aI73
aI143
aI79
aI957
aI461
aI1709
aI4909
aI2285
aI18113
aI8401
asS'd'
I3715
sa(dp7430
S'a'
I750
sS's'
I16
sS'm'
(lp7431
I1
aI1
aI3
aI7
aI9
aI9
aI101
aI127
aI137
aI755
aI1359
aI1965
aI83
aI13335
aI27763
aI7941
asS'd'
I3716
sa(dp7432
S'a'
I757
sS's'
I16
sS'm'
(lp7433
I1
aI1
aI1
aI3
aI13
aI61
aI95
aI61
aI295
aI615
aI555
aI2163
aI8155
aI14043
aI21465
aI46741
asS'd'
I3717
sa(dp7434
S'a'
I800
sS's'
I16
sS'm'
(lp7435
I1
aI1
aI1
aI13
aI29
aI19
aI111
aI17
aI373
aI153
aI1703
aI2199
aI7209
aI15845
aI1879
aI7493
asS'd'
I3718
sa(dp7436
S'a'
I805
sS's'
I16
sS'm'
(lp7437
I1
aI3
aI1
aI13
aI21
aI51
aI49
aI51
aI255
aI151
aI207
aI1915
aI7629
aI2705
aI8739
aI7467
asS'd'
I3719
sa(dp7438
S'a'
I809
sS's'
I16
sS'm'
(lp7439
I1
aI3
aI7
aI5
aI21
aI21
aI23
aI193
aI467
aI739
aI519
aI2315
aI2953
aI10633
aI9163
aI6007
asS'd'
I3720
sa(dp7440
S'a'
I837
sS's'
I16
sS'm'
(lp7441
I1
aI3
aI1
aI5
aI23
aI19
aI23
aI247
aI93
aI297
aI1089
aI2349
aI4683
aI13609
aI7615
aI18647
asS'd'
I3721
sa(dp7442
S'a'
I865
sS's'
I16
sS'm'
(lp7443
I1
aI1
aI3
aI3
aI21
aI39
aI19
aI71
aI93
aI1
aI133
aI3531
aI7503
aI2819
aI24211
aI1739
asS'd'
I3722
sa(dp7444
S'a'
I949
sS's'
I16
sS'm'
(lp7445
I1
aI3
aI5
aI13
aI9
aI43
aI31
aI111
aI493
aI739
aI705
aI2715
aI3613
aI11877
aI27945
aI46053
asS'd'
I3723
sa(dp7446
S'a'
I950
sS's'
I16
sS'm'
(lp7447
I1
aI1
aI7
aI13
aI27
aI59
aI103
aI129
aI53
aI531
aI1379
aI1441
aI5341
aI14937
aI5079
aI39881
asS'd'
I3724
sa(dp7448
S'a'
I956
sS's'
I16
sS'm'
(lp7449
I1
aI1
aI3
aI3
aI11
aI63
aI91
aI95
aI433
aI393
aI715
aI809
aI591
aI4141
aI17417
aI54107
asS'd'
I3725
sa(dp7450
S'a'
I961
sS's'
I16
sS'm'
(lp7451
I1
aI3
aI5
aI1
aI7
aI25
aI25
aI175
aI205
aI803
aI183
aI1441
aI1279
aI2753
aI20001
aI56677
asS'd'
I3726
sa(dp7452
S'a'
I1016
sS's'
I16
sS'm'
(lp7453
I1
aI1
aI5
aI3
aI13
aI23
aI77
aI25
aI133
aI137
aI1907
aI1313
aI2463
aI14339
aI13
aI57757
asS'd'
I3727
sa(dp7454
S'a'
I1030
sS's'
I16
sS'm'
(lp7455
I1
aI1
aI5
aI9
aI23
aI35
aI1
aI119
aI111
aI61
aI403
aI1815
aI1985
aI5651
aI10883
aI55943
asS'd'
I3728
sa(dp7456
S'a'
I1072
sS's'
I16
sS'm'
(lp7457
I1
aI3
aI1
aI7
aI21
aI43
aI115
aI7
aI107
aI719
aI759
aI1521
aI467
aI8735
aI29785
aI63821
asS'd'
I3729
sa(dp7458
S'a'
I1119
sS's'
I16
sS'm'
(lp7459
I1
aI1
aI3
aI13
aI19
aI17
aI51
aI141
aI399
aI569
aI703
aI2221
aI2809
aI13355
aI1907
aI15837
asS'd'
I3730
sa(dp7460
S'a'
I1130
sS's'
I16
sS'm'
(lp7461
I1
aI1
aI5
aI15
aI15
aI53
aI57
aI31
aI481
aI69
aI1439
aI4049
aI6727
aI11307
aI20683
aI63517
asS'd'
I3731
sa(dp7462
S'a'
I1135
sS's'
I16
sS'm'
(lp7463
I1
aI1
aI1
aI3
aI13
aI27
aI9
aI255
aI363
aI131
aI1745
aI2489
aI6451
aI6585
aI12873
aI35405
asS'd'
I3732
sa(dp7464
S'a'
I1137
sS's'
I16
sS'm'
(lp7465
I1
aI3
aI5
aI1
aI17
aI31
aI113
aI135
aI449
aI915
aI1017
aI2317
aI6821
aI5483
aI30707
aI45279
asS'd'
I3733
sa(dp7466
S'a'
I1144
sS's'
I16
sS'm'
(lp7467
I1
aI3
aI5
aI1
aI13
aI47
aI25
aI53
aI413
aI545
aI1777
aI3049
aI7527
aI9689
aI25935
aI9919
asS'd'
I3734
sa(dp7468
S'a'
I1149
sS's'
I16
sS'm'
(lp7469
I1
aI3
aI7
aI11
aI17
aI39
aI13
aI131
aI295
aI517
aI1755
aI2977
aI6267
aI12351
aI8957
aI17765
asS'd'
I3735
sa(dp7470
S'a'
I1180
sS's'
I16
sS'm'
(lp7471
I1
aI1
aI7
aI5
aI27
aI57
aI47
aI21
aI125
aI429
aI1169
aI1717
aI5455
aI16359
aI29065
aI6671
asS'd'
I3736
sa(dp7472
S'a'
I1214
sS's'
I16
sS'm'
(lp7473
I1
aI1
aI5
aI5
aI21
aI15
aI79
aI241
aI83
aI515
aI859
aI2351
aI3125
aI7465
aI30475
aI19759
asS'd'
I3737
sa(dp7474
S'a'
I1221
sS's'
I16
sS'm'
(lp7475
I1
aI3
aI1
aI9
aI11
aI5
aI81
aI11
aI7
aI221
aI141
aI3329
aI3435
aI323
aI18999
aI54735
asS'd'
I3738
sa(dp7476
S'a'
I1234
sS's'
I16
sS'm'
(lp7477
I1
aI1
aI1
aI15
aI7
aI57
aI87
aI251
aI63
aI561
aI929
aI1367
aI2511
aI14527
aI9335
aI38775
asS'd'
I3739
sa(dp7478
S'a'
I1239
sS's'
I16
sS'm'
(lp7479
I1
aI3
aI3
aI9
aI23
aI37
aI59
aI105
aI179
aI515
aI235
aI2445
aI433
aI13039
aI27005
aI48829
asS'd'
I3740
sa(dp7480
S'a'
I1249
sS's'
I16
sS'm'
(lp7481
I1
aI1
aI1
aI1
aI23
aI37
aI103
aI31
aI89
aI921
aI1687
aI831
aI387
aI10237
aI1241
aI19295
asS'd'
I3741
sa(dp7482
S'a'
I1250
sS's'
I16
sS'm'
(lp7483
I1
aI3
aI3
aI7
aI25
aI23
aI57
aI251
aI309
aI579
aI603
aI807
aI7383
aI8579
aI4025
aI16757
asS'd'
I3742
sa(dp7484
S'a'
I1267
sS's'
I16
sS'm'
(lp7485
I1
aI1
aI3
aI15
aI23
aI59
aI29
aI33
aI467
aI641
aI1271
aI2915
aI2549
aI14767
aI26557
aI43483
asS'd'
I3743
sa(dp7486
S'a'
I1273
sS's'
I16
sS'm'
(lp7487
I1
aI1
aI7
aI13
aI1
aI57
aI23
aI129
aI321
aI75
aI189
aI4087
aI5011
aI4355
aI25759
aI37153
asS'd'
I3744
sa(dp7488
S'a'
I1342
sS's'
I16
sS'm'
(lp7489
I1
aI1
aI5
aI1
aI21
aI57
aI25
aI183
aI37
aI669
aI259
aI1381
aI877
aI10245
aI16643
aI61035
asS'd'
I3745
sa(dp7490
S'a'
I1344
sS's'
I16
sS'm'
(lp7491
I1
aI1
aI7
aI5
aI11
aI11
aI85
aI141
aI393
aI957
aI1745
aI2243
aI1681
aI5583
aI16527
aI12017
asS'd'
I3746
sa(dp7492
S'a'
I1373
sS's'
I16
sS'm'
(lp7493
I1
aI1
aI5
aI15
aI23
aI31
aI5
aI169
aI287
aI527
aI1831
aI2937
aI7533
aI9739
aI24305
aI2239
asS'd'
I3747
sa(dp7494
S'a'
I1378
sS's'
I16
sS'm'
(lp7495
I1
aI1
aI7
aI1
aI7
aI13
aI3
aI243
aI189
aI309
aI607
aI3659
aI6369
aI7649
aI24255
aI55373
asS'd'
I3748
sa(dp7496
S'a'
I1408
sS's'
I16
sS'm'
(lp7497
I1
aI1
aI1
aI3
aI3
aI59
aI103
aI209
aI287
aI913
aI1223
aI1063
aI7715
aI6073
aI26697
aI25671
asS'd'
I3749
sa(dp7498
S'a'
I1417
sS's'
I16
sS'm'
(lp7499
I1
aI3
aI7
aI5
aI19
aI19
aI117
aI191
aI275
aI637
aI991
aI2199
aI2921
aI10553
aI21211
aI25981
asS'd'
I3750
sa(dp7500
S'a'
I1418
sS's'
I16
sS'm'
(lp7501
I1
aI3
aI3
aI5
aI29
aI59
aI17
aI13
aI127
aI57
aI1405
aI3181
aI2237
aI1795
aI21419
aI43421
asS'd'
I3751
sa(dp7502
S'a'
I1448
sS's'
I16
sS'm'
(lp7503
I1
aI1
aI1
aI15
aI25
aI41
aI11
aI117
aI463
aI425
aI305
aI1441
aI4307
aI7967
aI17529
aI4043
asS'd'
I3752
sa(dp7504
S'a'
I1454
sS's'
I16
sS'm'
(lp7505
I1
aI3
aI5
aI5
aI19
aI53
aI69
aI73
aI453
aI611
aI1583
aI1721
aI6303
aI10561
aI18527
aI48973
asS'd'
I3753
sa(dp7506
S'a'
I1510
sS's'
I16
sS'm'
(lp7507
I1
aI1
aI7
aI11
aI15
aI61
aI87
aI69
aI463
aI771
aI819
aI469
aI8165
aI8897
aI29657
aI55161
asS'd'
I3754
sa(dp7508
S'a'
I1513
sS's'
I16
sS'm'
(lp7509
I1
aI1
aI5
aI1
aI15
aI25
aI23
aI47
aI287
aI457
aI1219
aI473
aI4127
aI3349
aI9425
aI41541
asS'd'
I3755
sa(dp7510
S'a'
I1522
sS's'
I16
sS'm'
(lp7511
I1
aI3
aI7
aI5
aI17
aI17
aI33
aI161
aI239
aI231
aI241
aI1297
aI4879
aI12761
aI20939
aI65261
asS'd'
I3756
sa(dp7512
S'a'
I1543
sS's'
I16
sS'm'
(lp7513
I1
aI3
aI3
aI9
aI19
aI53
aI95
aI89
aI117
aI333
aI1815
aI2217
aI7779
aI8213
aI4667
aI58395
asS'd'
I3757
sa(dp7514
S'a'
I1550
sS's'
I16
sS'm'
(lp7515
I1
aI3
aI3
aI9
aI17
aI7
aI41
aI99
aI371
aI797
aI729
aI2851
aI2003
aI4463
aI20793
aI54315
asS'd'
I3758
sa(dp7516
S'a'
I1552
sS's'
I16
sS'm'
(lp7517
I1
aI3
aI5
aI5
aI23
aI39
aI19
aI235
aI163
aI365
aI141
aI791
aI455
aI2761
aI9115
aI53351
asS'd'
I3759
sa(dp7518
S'a'
I1588
sS's'
I16
sS'm'
(lp7519
I1
aI3
aI3
aI3
aI9
aI27
aI29
aI139
aI165
aI867
aI2023
aI1333
aI3771
aI10451
aI9141
aI41177
asS'd'
I3760
sa(dp7520
S'a'
I1592
sS's'
I16
sS'm'
(lp7521
I1
aI1
aI3
aI7
aI3
aI11
aI125
aI157
aI355
aI519
aI187
aI3381
aI1151
aI1629
aI25247
aI42797
asS'd'
I3761
sa(dp7522
S'a'
I1597
sS's'
I16
sS'm'
(lp7523
I1
aI3
aI3
aI3
aI21
aI25
aI37
aI155
aI257
aI311
aI961
aI1945
aI1383
aI5679
aI7857
aI7183
asS'd'
I3762
sa(dp7524
S'a'
I1606
sS's'
I16
sS'm'
(lp7525
I1
aI3
aI3
aI5
aI29
aI11
aI49
aI125
aI171
aI605
aI1923
aI2781
aI2555
aI5063
aI5075
aI43301
asS'd'
I3763
sa(dp7526
S'a'
I1610
sS's'
I16
sS'm'
(lp7527
I1
aI3
aI5
aI9
aI27
aI1
aI27
aI149
aI253
aI205
aI1299
aI2901
aI2891
aI975
aI7641
aI8115
asS'd'
I3764
sa(dp7528
S'a'
I1617
sS's'
I16
sS'm'
(lp7529
I1
aI3
aI5
aI3
aI31
aI7
aI49
aI215
aI81
aI791
aI1485
aI837
aI5051
aI1947
aI7521
aI25723
asS'd'
I3765
sa(dp7530
S'a'
I1623
sS's'
I16
sS'm'
(lp7531
I1
aI3
aI5
aI7
aI23
aI25
aI69
aI13
aI3
aI859
aI441
aI3577
aI1687
aI6559
aI8687
aI46757
asS'd'
I3766
sa(dp7532
S'a'
I1657
sS's'
I16
sS'm'
(lp7533
I1
aI1
aI1
aI9
aI1
aI59
aI3
aI31
aI251
aI187
aI617
aI2607
aI4635
aI6121
aI8565
aI8871
asS'd'
I3767
sa(dp7534
S'a'
I1697
sS's'
I16
sS'm'
(lp7535
I1
aI3
aI3
aI9
aI29
aI37
aI127
aI87
aI153
aI633
aI1691
aI2729
aI3167
aI3219
aI21237
aI25573
asS'd'
I3768
sa(dp7536
S'a'
I1729
sS's'
I16
sS'm'
(lp7537
I1
aI1
aI5
aI13
aI19
aI63
aI93
aI235
aI299
aI621
aI405
aI663
aI6639
aI12265
aI9303
aI42719
asS'd'
I3769
sa(dp7538
S'a'
I1735
sS's'
I16
sS'm'
(lp7539
I1
aI1
aI3
aI9
aI25
aI11
aI9
aI231
aI101
aI335
aI1793
aI1497
aI7069
aI4171
aI30199
aI63
asS'd'
I3770
sa(dp7540
S'a'
I1769
sS's'
I16
sS'm'
(lp7541
I1
aI1
aI1
aI1
aI5
aI19
aI17
aI217
aI165
aI413
aI925
aI1409
aI6559
aI14537
aI22057
aI44331
asS'd'
I3771
sa(dp7542
S'a'
I1778
sS's'
I16
sS'm'
(lp7543
I1
aI1
aI3
aI7
aI11
aI51
aI45
aI217
aI57
aI795
aI951
aI2933
aI6705
aI137
aI30525
aI9679
asS'd'
I3772
sa(dp7544
S'a'
I1826
sS's'
I16
sS'm'
(lp7545
I1
aI1
aI3
aI15
aI27
aI47
aI35
aI125
aI363
aI619
aI1027
aI2861
aI3923
aI10459
aI16789
aI27277
asS'd'
I3773
sa(dp7546
S'a'
I1858
sS's'
I16
sS'm'
(lp7547
I1
aI1
aI7
aI7
aI13
aI37
aI33
aI29
aI385
aI851
aI143
aI119
aI7345
aI4251
aI25121
aI31609
asS'd'
I3774
sa(dp7548
S'a'
I1870
sS's'
I16
sS'm'
(lp7549
I1
aI3
aI1
aI1
aI17
aI25
aI119
aI7
aI365
aI397
aI601
aI2087
aI6903
aI15345
aI14671
aI37889
asS'd'
I3775
sa(dp7550
S'a'
I1875
sS's'
I16
sS'm'
(lp7551
I1
aI3
aI1
aI13
aI19
aI51
aI41
aI139
aI133
aI723
aI25
aI2621
aI1257
aI7037
aI9527
aI50037
asS'd'
I3776
sa(dp7552
S'a'
I1922
sS's'
I16
sS'm'
(lp7553
I1
aI1
aI5
aI11
aI5
aI59
aI119
aI75
aI397
aI545
aI1095
aI585
aI3271
aI1049
aI123
aI33029
asS'd'
I3777
sa(dp7554
S'a'
I1924
sS's'
I16
sS'm'
(lp7555
I1
aI1
aI7
aI11
aI9
aI27
aI21
aI197
aI177
aI31
aI453
aI2457
aI2733
aI7787
aI1923
aI24639
asS'd'
I3778
sa(dp7556
S'a'
I1933
sS's'
I16
sS'm'
(lp7557
I1
aI1
aI7
aI13
aI29
aI13
aI91
aI91
aI243
aI279
aI601
aI1699
aI7169
aI4727
aI7815
aI29099
asS'd'
I3779
sa(dp7558
S'a'
I1972
sS's'
I16
sS'm'
(lp7559
I1
aI3
aI7
aI5
aI1
aI35
aI27
aI235
aI163
aI913
aI1479
aI769
aI7179
aI1983
aI25977
aI55373
asS'd'
I3780
sa(dp7560
S'a'
I1979
sS's'
I16
sS'm'
(lp7561
I1
aI3
aI5
aI11
aI9
aI33
aI99
aI141
aI301
aI109
aI1785
aI129
aI1707
aI5181
aI4797
aI9979
asS'd'
I3781
sa(dp7562
S'a'
I1987
sS's'
I16
sS'm'
(lp7563
I1
aI1
aI1
aI13
aI3
aI47
aI89
aI43
aI293
aI87
aI1689
aI3885
aI7747
aI5607
aI477
aI31887
asS'd'
I3782
sa(dp7564
S'a'
I1994
sS's'
I16
sS'm'
(lp7565
I1
aI1
aI5
aI1
aI9
aI21
aI73
aI37
aI45
aI621
aI1855
aI3691
aI4899
aI2191
aI13459
aI23543
asS'd'
I3783
sa(dp7566
S'a'
I2008
sS's'
I16
sS'm'
(lp7567
I1
aI1
aI1
aI1
aI7
aI39
aI61
aI125
aI341
aI905
aI213
aI1755
aI241
aI13407
aI8791
aI10165
asS'd'
I3784
sa(dp7568
S'a'
I2023
sS's'
I16
sS'm'
(lp7569
I1
aI1
aI1
aI1
aI19
aI31
aI79
aI19
aI55
aI875
aI1017
aI1787
aI4879
aI533
aI15029
aI52295
asS'd'
I3785
sa(dp7570
S'a'
I2029
sS's'
I16
sS'm'
(lp7571
I1
aI3
aI1
aI1
aI9
aI59
aI113
aI71
aI113
aI649
aI561
aI71
aI5253
aI783
aI7389
aI19361
asS'd'
I3786
sa(dp7572
S'a'
I2053
sS's'
I16
sS'm'
(lp7573
I1
aI1
aI1
aI11
aI5
aI39
aI61
aI225
aI291
aI907
aI795
aI1099
aI597
aI11829
aI15137
aI42865
asS'd'
I3787
sa(dp7574
S'a'
I2081
sS's'
I16
sS'm'
(lp7575
I1
aI3
aI1
aI5
aI25
aI11
aI71
aI155
aI271
aI309
aI1981
aI1253
aI463
aI1133
aI20833
aI48625
asS'd'
I3788
sa(dp7576
S'a'
I2087
sS's'
I16
sS'm'
(lp7577
I1
aI3
aI5
aI9
aI7
aI41
aI87
aI241
aI457
aI899
aI1493
aI3675
aI3025
aI10607
aI22569
aI52813
asS'd'
I3789
sa(dp7578
S'a'
I2094
sS's'
I16
sS'm'
(lp7579
I1
aI3
aI7
aI13
aI7
aI37
aI37
aI103
aI281
aI915
aI1259
aI4049
aI559
aI173
aI4123
aI63767
asS'd'
I3790
sa(dp7580
S'a'
I2111
sS's'
I16
sS'm'
(lp7581
I1
aI3
aI7
aI15
aI13
aI57
aI9
aI51
aI39
aI549
aI1431
aI2887
aI1081
aI4643
aI16331
aI14221
asS'd'
I3791
sa(dp7582
S'a'
I2113
sS's'
I16
sS'm'
(lp7583
I1
aI3
aI5
aI7
aI13
aI1
aI101
aI125
aI25
aI713
aI1423
aI513
aI3323
aI9951
aI7163
aI20969
asS'd'
I3792
sa(dp7584
S'a'
I2114
sS's'
I16
sS'm'
(lp7585
I1
aI1
aI7
aI15
aI11
aI25
aI25
aI3
aI47
aI531
aI1529
aI471
aI6191
aI10051
aI29671
aI49085
asS'd'
I3793
sa(dp7586
S'a'
I2123
sS's'
I16
sS'm'
(lp7587
I1
aI1
aI3
aI5
aI23
aI51
aI117
aI141
aI55
aI275
aI761
aI1923
aI6267
aI2291
aI3701
aI26615
asS'd'
I3794
sa(dp7588
S'a'
I2190
sS's'
I16
sS'm'
(lp7589
I1
aI1
aI7
aI9
aI15
aI19
aI111
aI65
aI137
aI373
aI1753
aI3591
aI1137
aI11639
aI28591
aI27265
asS'd'
I3795
sa(dp7590
S'a'
I2231
sS's'
I16
sS'm'
(lp7591
I1
aI3
aI1
aI15
aI29
aI5
aI67
aI13
aI425
aI961
aI453
aI2481
aI1407
aI3479
aI23303
aI30407
asS'd'
I3796
sa(dp7592
S'a'
I2276
sS's'
I16
sS'm'
(lp7593
I1
aI1
aI5
aI3
aI19
aI39
aI39
aI123
aI351
aI77
aI1339
aI1765
aI3767
aI1907
aI13627
aI23877
asS'd'
I3797
sa(dp7594
S'a'
I2285
sS's'
I16
sS'm'
(lp7595
I1
aI3
aI5
aI9
aI23
aI7
aI103
aI177
aI221
aI197
aI561
aI2121
aI7231
aI12053
aI30127
aI29849
asS'd'
I3798
sa(dp7596
S'a'
I2297
sS's'
I16
sS'm'
(lp7597
I1
aI1
aI5
aI7
aI15
aI1
aI3
aI123
aI197
aI493
aI171
aI2425
aI3865
aI4061
aI31883
aI2491
asS'd'
I3799
sa(dp7598
S'a'
I2336
sS's'
I16
sS'm'
(lp7599
I1
aI1
aI3
aI13
aI29
aI33
aI99
aI67
aI327
aI969
aI1793
aI1871
aI1839
aI13059
aI7605
aI16797
asS'd'
I3800
sa(dp7600
S'a'
I2345
sS's'
I16
sS'm'
(lp7601
I1
aI3
aI5
aI11
aI25
aI53
aI25
aI93
aI303
aI623
aI1889
aI1471
aI1213
aI14459
aI8527
aI25095
asS'd'
I3801
sa(dp7602
S'a'
I2353
sS's'
I16
sS'm'
(lp7603
I1
aI1
aI1
aI13
aI15
aI3
aI115
aI3
aI289
aI743
aI1855
aI359
aI2375
aI13765
aI19711
aI40765
asS'd'
I3802
sa(dp7604
S'a'
I2363
sS's'
I16
sS'm'
(lp7605
I1
aI1
aI7
aI11
aI27
aI51
aI85
aI163
aI219
aI871
aI637
aI2011
aI5981
aI587
aI17521
aI17333
asS'd'
I3803
sa(dp7606
S'a'
I2368
sS's'
I16
sS'm'
(lp7607
I1
aI3
aI5
aI1
aI21
aI59
aI49
aI39
aI305
aI513
aI2017
aI285
aI5817
aI13123
aI27765
aI46741
asS'd'
I3804
sa(dp7608
S'a'
I2373
sS's'
I16
sS'm'
(lp7609
I1
aI3
aI3
aI7
aI21
aI39
aI71
aI163
aI423
aI845
aI783
aI397
aI7319
aI10677
aI13407
aI47471
asS'd'
I3805
sa(dp7610
S'a'
I2391
sS's'
I16
sS'm'
(lp7611
I1
aI3
aI7
aI5
aI21
aI59
aI99
aI179
aI473
aI687
aI1393
aI723
aI2245
aI2933
aI25943
aI7769
asS'd'
I3806
sa(dp7612
S'a'
I2402
sS's'
I16
sS'm'
(lp7613
I1
aI1
aI5
aI9
aI5
aI45
aI71
aI189
aI165
aI555
aI643
aI2289
aI3133
aI12319
aI22209
aI1533
asS'd'
I3807
sa(dp7614
S'a'
I2413
sS's'
I16
sS'm'
(lp7615
I1
aI1
aI3
aI9
aI7
aI43
aI1
aI155
aI323
aI169
aI339
aI2561
aI4049
aI4953
aI5289
aI8783
asS'd'
I3808
sa(dp7616
S'a'
I2422
sS's'
I16
sS'm'
(lp7617
I1
aI3
aI1
aI11
aI15
aI5
aI25
aI201
aI267
aI891
aI561
aI501
aI575
aI15147
aI1743
aI45237
asS'd'
I3809
sa(dp7618
S'a'
I2425
sS's'
I16
sS'm'
(lp7619
I1
aI3
aI5
aI13
aI25
aI27
aI105
aI205
aI165
aI795
aI975
aI943
aI7413
aI10299
aI14839
aI54895
asS'd'
I3810
sa(dp7620
S'a'
I2461
sS's'
I16
sS'm'
(lp7621
I1
aI1
aI5
aI1
aI17
aI43
aI69
aI103
aI449
aI917
aI103
aI945
aI513
aI709
aI11647
aI28065
asS'd'
I3811
sa(dp7622
S'a'
I2462
sS's'
I16
sS'm'
(lp7623
I1
aI1
aI3
aI15
aI23
aI51
aI23
aI7
aI159
aI743
aI177
aI3457
aI415
aI1775
aI25353
aI36385
asS'd'
I3812
sa(dp7624
S'a'
I2490
sS's'
I16
sS'm'
(lp7625
I1
aI3
aI5
aI13
aI9
aI63
aI121
aI19
aI165
aI449
aI1523
aI1959
aI6901
aI12281
aI29149
aI45999
asS'd'
I3813
sa(dp7626
S'a'
I2492
sS's'
I16
sS'm'
(lp7627
I1
aI3
aI7
aI11
aI17
aI19
aI9
aI155
aI373
aI753
aI1313
aI2205
aI3571
aI16317
aI16151
aI15325
asS'd'
I3814
sa(dp7628
S'a'
I2510
sS's'
I16
sS'm'
(lp7629
I1
aI3
aI3
aI7
aI15
aI43
aI65
aI183
aI407
aI123
aI1151
aI375
aI3461
aI6673
aI12985
aI21005
asS'd'
I3815
sa(dp7630
S'a'
I2564
sS's'
I16
sS'm'
(lp7631
I1
aI3
aI7
aI7
aI9
aI1
aI87
aI247
aI489
aI123
aI1677
aI1947
aI7961
aI13497
aI27919
aI28993
asS'd'
I3816
sa(dp7632
S'a'
I2573
sS's'
I16
sS'm'
(lp7633
I1
aI3
aI3
aI7
aI19
aI21
aI95
aI227
aI217
aI133
aI69
aI1535
aI699
aI3521
aI29255
aI34733
asS'd'
I3817
sa(dp7634
S'a'
I2598
sS's'
I16
sS'm'
(lp7635
I1
aI3
aI5
aI3
aI7
aI57
aI45
aI251
aI407
aI81
aI1259
aI2425
aI2217
aI13097
aI12773
aI14643
asS'd'
I3818
sa(dp7636
S'a'
I2627
sS's'
I16
sS'm'
(lp7637
I1
aI1
aI1
aI11
aI23
aI37
aI13
aI229
aI467
aI591
aI1521
aI469
aI3763
aI2289
aI14233
aI24053
asS'd'
I3819
sa(dp7638
S'a'
I2633
sS's'
I16
sS'm'
(lp7639
I1
aI3
aI5
aI1
aI27
aI53
aI105
aI5
aI85
aI765
aI1973
aI2597
aI5725
aI1063
aI18145
aI961
asS'd'
I3820
sa(dp7640
S'a'
I2647
sS's'
I16
sS'm'
(lp7641
I1
aI3
aI7
aI1
aI21
aI47
aI115
aI95
aI403
aI3
aI1593
aI3379
aI7371
aI15553
aI12503
aI57979
asS'd'
I3821
sa(dp7642
S'a'
I2660
sS's'
I16
sS'm'
(lp7643
I1
aI1
aI3
aI1
aI1
aI35
aI121
aI29
aI379
aI245
aI919
aI2673
aI3503
aI14197
aI31193
aI8355
asS'd'
I3822
sa(dp7644
S'a'
I2664
sS's'
I16
sS'm'
(lp7645
I1
aI3
aI5
aI11
aI19
aI49
aI97
aI7
aI195
aI1013
aI1671
aI3415
aI2009
aI13389
aI4837
aI27453
asS'd'
I3823
sa(dp7646
S'a'
I2678
sS's'
I16
sS'm'
(lp7647
I1
aI1
aI5
aI13
aI9
aI15
aI115
aI97
aI463
aI449
aI303
aI2681
aI1215
aI12559
aI15685
aI21321
asS'd'
I3824
sa(dp7648
S'a'
I2684
sS's'
I16
sS'm'
(lp7649
I1
aI3
aI5
aI13
aI23
aI5
aI113
aI193
aI419
aI301
aI1121
aI317
aI5503
aI4683
aI25519
aI65
asS'd'
I3825
sa(dp7650
S'a'
I2691
sS's'
I16
sS'm'
(lp7651
I1
aI3
aI3
aI7
aI15
aI29
aI45
aI97
aI323
aI475
aI143
aI1173
aI4033
aI8939
aI31849
aI3575
asS'd'
I3826
sa(dp7652
S'a'
I2759
sS's'
I16
sS'm'
(lp7653
I1
aI1
aI7
aI7
aI21
aI1
aI101
aI143
aI197
aI409
aI855
aI1753
aI5211
aI3763
aI11139
aI37309
asS'd'
I3827
sa(dp7654
S'a'
I2768
sS's'
I16
sS'm'
(lp7655
I1
aI1
aI3
aI13
aI25
aI33
aI55
aI45
aI381
aI349
aI991
aI535
aI4823
aI3701
aI31629
aI48037
asS'd'
I3828
sa(dp7656
S'a'
I2773
sS's'
I16
sS'm'
(lp7657
I1
aI3
aI1
aI11
aI17
aI51
aI27
aI57
aI409
aI551
aI949
aI365
aI8093
aI10831
aI19697
aI39437
asS'd'
I3829
sa(dp7658
S'a'
I2794
sS's'
I16
sS'm'
(lp7659
I1
aI3
aI5
aI3
aI31
aI33
aI81
aI49
aI91
aI865
aI469
aI2115
aI377
aI8237
aI31907
aI38239
asS'd'
I3830
sa(dp7660
S'a'
I2813
sS's'
I16
sS'm'
(lp7661
I1
aI1
aI3
aI7
aI29
aI59
aI57
aI17
aI121
aI889
aI1557
aI1797
aI5001
aI14209
aI21355
aI59739
asS'd'
I3831
sa(dp7662
S'a'
I2831
sS's'
I16
sS'm'
(lp7663
I1
aI1
aI5
aI9
aI11
aI45
aI89
aI87
aI397
aI785
aI525
aI1593
aI5251
aI12449
aI23579
aI54265
asS'd'
I3832
sa(dp7664
S'a'
I2843
sS's'
I16
sS'm'
(lp7665
I1
aI3
aI5
aI11
aI5
aI31
aI19
aI47
aI207
aI331
aI91
aI1691
aI5171
aI53
aI15945
aI33349
asS'd'
I3833
sa(dp7666
S'a'
I2846
sS's'
I16
sS'm'
(lp7667
I1
aI1
aI1
aI15
aI11
aI41
aI91
aI177
aI505
aI871
aI815
aI3673
aI5631
aI9915
aI1133
aI37861
asS'd'
I3834
sa(dp7668
S'a'
I2849
sS's'
I16
sS'm'
(lp7669
I1
aI3
aI5
aI5
aI25
aI63
aI53
aI231
aI55
aI51
aI481
aI303
aI1859
aI11973
aI28557
aI22045
asS'd'
I3835
sa(dp7670
S'a'
I2856
sS's'
I16
sS'm'
(lp7671
I1
aI1
aI5
aI3
aI27
aI11
aI37
aI91
aI363
aI411
aI1131
aI3369
aI377
aI6585
aI7353
aI42949
asS'd'
I3836
sa(dp7672
S'a'
I2893
sS's'
I16
sS'm'
(lp7673
I1
aI3
aI1
aI9
aI31
aI63
aI83
aI23
aI405
aI941
aI119
aI1471
aI2509
aI15507
aI29239
aI49613
asS'd'
I3837
sa(dp7674
S'a'
I2901
sS's'
I16
sS'm'
(lp7675
I1
aI1
aI5
aI1
aI11
aI63
aI117
aI237
aI407
aI231
aI1425
aI71
aI8005
aI4023
aI9029
aI59819
asS'd'
I3838
sa(dp7676
S'a'
I2924
sS's'
I16
sS'm'
(lp7677
I1
aI1
aI5
aI7
aI1
aI9
aI43
aI87
aI351
aI63
aI1075
aI3381
aI5447
aI2437
aI24983
aI26905
asS'd'
I3839
sa(dp7678
S'a'
I2942
sS's'
I16
sS'm'
(lp7679
I1
aI3
aI7
aI5
aI5
aI35
aI33
aI89
aI251
aI819
aI1735
aI2625
aI6363
aI6837
aI27603
aI26669
asS'd'
I3840
sa(dp7680
S'a'
I2975
sS's'
I16
sS'm'
(lp7681
I1
aI3
aI7
aI13
aI29
aI63
aI51
aI245
aI371
aI791
aI907
aI3499
aI3033
aI8443
aI20023
aI1805
asS'd'
I3841
sa(dp7682
S'a'
I2979
sS's'
I16
sS'm'
(lp7683
I1
aI1
aI5
aI7
aI13
aI15
aI109
aI197
aI451
aI709
aI929
aI3193
aI5727
aI11185
aI29479
aI1671
asS'd'
I3842
sa(dp7684
S'a'
I2985
sS's'
I16
sS'm'
(lp7685
I1
aI1
aI7
aI13
aI19
aI23
aI97
aI9
aI359
aI635
aI777
aI39
aI893
aI2531
aI13563
aI19295
asS'd'
I3843
sa(dp7686
S'a'
I3020
sS's'
I16
sS'm'
(lp7687
I1
aI1
aI5
aI1
aI31
aI63
aI55
aI7
aI157
aI877
aI991
aI1317
aI1595
aI2019
aI21435
aI52255
asS'd'
I3844
sa(dp7688
S'a'
I3025
sS's'
I16
sS'm'
(lp7689
I1
aI1
aI5
aI3
aI19
aI37
aI23
aI13
aI335
aI431
aI483
aI615
aI2431
aI505
aI26245
aI63323
asS'd'
I3845
sa(dp7690
S'a'
I3028
sS's'
I16
sS'm'
(lp7691
I1
aI3
aI7
aI5
aI5
aI9
aI37
aI65
aI303
aI423
aI1907
aI2661
aI7213
aI2975
aI29045
aI16243
asS'd'
I3846
sa(dp7692
S'a'
I3051
sS's'
I16
sS'm'
(lp7693
I1
aI3
aI1
aI5
aI13
aI37
aI115
aI217
aI227
aI159
aI707
aI1387
aI943
aI4935
aI5503
aI35171
asS'd'
I3847
sa(dp7694
S'a'
I3127
sS's'
I16
sS'm'
(lp7695
I1
aI3
aI7
aI9
aI19
aI15
aI87
aI233
aI453
aI159
aI169
aI1077
aI2129
aI413
aI19773
aI629
asS'd'
I3848
sa(dp7696
S'a'
I3142
sS's'
I16
sS'm'
(lp7697
I1
aI1
aI5
aI15
aI29
aI39
aI37
aI243
aI233
aI365
aI1843
aI2219
aI1255
aI15287
aI603
aI13511
asS'd'
I3849
sa(dp7698
S'a'
I3145
sS's'
I16
sS'm'
(lp7699
I1
aI1
aI3
aI3
aI31
aI53
aI33
aI125
aI497
aI597
aI127
aI1829
aI3905
aI2611
aI4263
aI40971
asS'd'
I3850
sa(dp7700
S'a'
I3156
sS's'
I16
sS'm'
(lp7701
I1
aI3
aI5
aI9
aI11
aI47
aI71
aI215
aI383
aI321
aI1445
aI135
aI5953
aI8791
aI22073
aI16537
asS'd'
I3851
sa(dp7702
S'a'
I3165
sS's'
I16
sS'm'
(lp7703
I1
aI3
aI3
aI13
aI15
aI7
aI7
aI133
aI401
aI459
aI1117
aI3165
aI4105
aI11943
aI22431
aI56821
asS'd'
I3852
sa(dp7704
S'a'
I3196
sS's'
I16
sS'm'
(lp7705
I1
aI1
aI7
aI9
aI31
aI39
aI19
aI7
aI19
aI401
aI79
aI3641
aI6815
aI1489
aI7537
aI49467
asS'd'
I3853
sa(dp7706
S'a'
I3199
sS's'
I16
sS'm'
(lp7707
I1
aI3
aI7
aI7
aI17
aI11
aI91
aI205
aI251
aI321
aI515
aI3521
aI311
aI3169
aI271
aI34749
asS'd'
I3854
sa(dp7708
S'a'
I3217
sS's'
I16
sS'm'
(lp7709
I1
aI3
aI3
aI7
aI29
aI15
aI5
aI153
aI83
aI603
aI1373
aI997
aI4939
aI9811
aI243
aI5375
asS'd'
I3855
sa(dp7710
S'a'
I3218
sS's'
I16
sS'm'
(lp7711
I1
aI1
aI3
aI11
aI21
aI47
aI25
aI221
aI237
aI177
aI535
aI2819
aI6213
aI7877
aI26795
aI36609
asS'd'
I3856
sa(dp7712
S'a'
I3253
sS's'
I16
sS'm'
(lp7713
I1
aI3
aI7
aI3
aI31
aI1
aI69
aI73
aI47
aI653
aI139
aI1649
aI7183
aI1293
aI26507
aI38415
asS'd'
I3857
sa(dp7714
S'a'
I3258
sS's'
I16
sS'm'
(lp7715
I1
aI1
aI1
aI13
aI17
aI41
aI23
aI73
aI115
aI509
aI787
aI3733
aI1871
aI171
aI29967
aI39941
asS'd'
I3858
sa(dp7716
S'a'
I3260
sS's'
I16
sS'm'
(lp7717
I1
aI3
aI5
aI1
aI9
aI7
aI61
aI23
aI105
aI381
aI1421
aI2887
aI3717
aI643
aI26375
aI57991
asS'd'
I3859
sa(dp7718
S'a'
I3289
sS's'
I16
sS'm'
(lp7719
I1
aI3
aI3
aI3
aI19
aI3
aI101
aI117
aI393
aI83
aI1255
aI3331
aI6481
aI8661
aI20855
aI28875
asS'd'
I3860
sa(dp7720
S'a'
I3314
sS's'
I16
sS'm'
(lp7721
I1
aI3
aI5
aI11
aI21
aI13
aI111
aI193
aI51
aI899
aI159
aI1989
aI7931
aI10511
aI3933
aI447
asS'd'
I3861
sa(dp7722
S'a'
I3326
sS's'
I16
sS'm'
(lp7723
I1
aI1
aI5
aI15
aI23
aI35
aI49
aI139
aI397
aI145
aI597
aI1847
aI7077
aI715
aI20227
aI42183
asS'd'
I3862
sa(dp7724
S'a'
I3331
sS's'
I16
sS'm'
(lp7725
I1
aI3
aI3
aI3
aI17
aI3
aI87
aI233
aI35
aI317
aI337
aI237
aI6901
aI3439
aI20033
aI10307
asS'd'
I3863
sa(dp7726
S'a'
I3371
sS's'
I16
sS'm'
(lp7727
I1
aI3
aI5
aI3
aI11
aI35
aI13
aI171
aI7
aI963
aI1443
aI1501
aI7617
aI963
aI25453
aI62589
asS'd'
I3864
sa(dp7728
S'a'
I3381
sS's'
I16
sS'm'
(lp7729
I1
aI1
aI1
aI5
aI11
aI9
aI39
aI175
aI409
aI411
aI1407
aI2743
aI4255
aI989
aI15823
aI1707
asS'd'
I3865
sa(dp7730
S'a'
I3396
sS's'
I16
sS'm'
(lp7731
I1
aI1
aI7
aI13
aI27
aI55
aI63
aI239
aI355
aI417
aI2007
aI2299
aI2921
aI1637
aI10687
aI60615
asS'd'
I3866
sa(dp7732
S'a'
I3441
sS's'
I16
sS'm'
(lp7733
I1
aI1
aI7
aI9
aI5
aI61
aI57
aI73
aI263
aI307
aI2003
aI1763
aI639
aI5885
aI14709
aI16985
asS'd'
I3867
sa(dp7734
S'a'
I3442
sS's'
I16
sS'm'
(lp7735
I1
aI1
aI3
aI3
aI21
aI55
aI19
aI249
aI509
aI533
aI1361
aI1397
aI2777
aI15523
aI4389
aI13339
asS'd'
I3868
sa(dp7736
S'a'
I3460
sS's'
I16
sS'm'
(lp7737
I1
aI3
aI5
aI15
aI9
aI3
aI91
aI237
aI451
aI299
aI1541
aI4083
aI879
aI7859
aI21585
aI14833
asS'd'
I3869
sa(dp7738
S'a'
I3477
sS's'
I16
sS'm'
(lp7739
I1
aI1
aI7
aI3
aI31
aI47
aI49
aI231
aI123
aI391
aI1633
aI2567
aI5577
aI1631
aI27951
aI22913
asS'd'
I3870
sa(dp7740
S'a'
I3491
sS's'
I16
sS'm'
(lp7741
I1
aI3
aI7
aI13
aI11
aI13
aI1
aI111
aI183
aI87
aI839
aI1915
aI5523
aI3677
aI13065
aI38225
asS'd'
I3871
sa(dp7742
S'a'
I3493
sS's'
I16
sS'm'
(lp7743
I1
aI1
aI3
aI7
aI15
aI15
aI63
aI241
aI167
aI345
aI653
aI701
aI4725
aI12911
aI11545
aI24475
asS'd'
I3872
sa(dp7744
S'a'
I3543
sS's'
I16
sS'm'
(lp7745
I1
aI1
aI3
aI7
aI25
aI15
aI49
aI235
aI331
aI639
aI965
aI1117
aI7147
aI3789
aI3309
aI20255
asS'd'
I3873
sa(dp7746
S'a'
I3549
sS's'
I16
sS'm'
(lp7747
I1
aI3
aI5
aI7
aI7
aI63
aI93
aI241
aI253
aI31
aI951
aI3723
aI3359
aI7303
aI191
aI36427
asS'd'
I3874
sa(dp7748
S'a'
I3550
sS's'
I16
sS'm'
(lp7749
I1
aI3
aI7
aI9
aI9
aI59
aI5
aI107
aI181
aI413
aI1269
aI3121
aI1929
aI11921
aI8931
aI47459
asS'd'
I3875
sa(dp7750
S'a'
I3553
sS's'
I16
sS'm'
(lp7751
I1
aI3
aI1
aI15
aI25
aI27
aI13
aI47
aI295
aI111
aI1287
aI2551
aI4887
aI4145
aI17063
aI42037
asS'd'
I3876
sa(dp7752
S'a'
I3563
sS's'
I16
sS'm'
(lp7753
I1
aI3
aI3
aI13
aI17
aI17
aI21
aI17
aI491
aI845
aI1463
aI1305
aI1375
aI16149
aI19331
aI25043
asS'd'
I3877
sa(dp7754
S'a'
I3568
sS's'
I16
sS'm'
(lp7755
I1
aI3
aI5
aI1
aI27
aI5
aI93
aI139
aI283
aI711
aI1141
aI1743
aI5001
aI8851
aI19351
aI12275
asS'd'
I3878
sa(dp7756
S'a'
I3604
sS's'
I16
sS'm'
(lp7757
I1
aI1
aI1
aI1
aI23
aI25
aI51
aI63
aI429
aI735
aI201
aI3785
aI6677
aI16375
aI19681
aI17857
asS'd'
I3879
sa(dp7758
S'a'
I3632
sS's'
I16
sS'm'
(lp7759
I1
aI3
aI3
aI3
aI9
aI63
aI71
aI147
aI463
aI465
aI1163
aI1045
aI6967
aI12537
aI31853
aI38391
asS'd'
I3880
sa(dp7760
S'a'
I3650
sS's'
I16
sS'm'
(lp7761
I1
aI3
aI7
aI1
aI5
aI51
aI79
aI239
aI389
aI3
aI601
aI3787
aI7635
aI16295
aI1681
aI63971
asS'd'
I3881
sa(dp7762
S'a'
I3662
sS's'
I16
sS'm'
(lp7763
I1
aI3
aI1
aI3
aI5
aI31
aI103
aI89
aI321
aI971
aI783
aI3685
aI1155
aI10353
aI2167
aI35423
asS'd'
I3882
sa(dp7764
S'a'
I3674
sS's'
I16
sS'm'
(lp7765
I1
aI1
aI5
aI15
aI25
aI19
aI93
aI59
aI361
aI217
aI1141
aI597
aI5877
aI15961
aI1593
aI22925
asS'd'
I3883
sa(dp7766
S'a'
I3685
sS's'
I16
sS'm'
(lp7767
I1
aI3
aI1
aI9
aI25
aI59
aI69
aI89
aI477
aI89
aI487
aI237
aI5625
aI9579
aI30421
aI21883
asS'd'
I3884
sa(dp7768
S'a'
I3686
sS's'
I16
sS'm'
(lp7769
I1
aI1
aI3
aI7
aI1
aI5
aI13
aI225
aI9
aI981
aI1081
aI1407
aI6855
aI15215
aI21713
aI62313
asS'd'
I3885
sa(dp7770
S'a'
I3700
sS's'
I16
sS'm'
(lp7771
I1
aI1
aI7
aI15
aI11
aI13
aI119
aI109
aI151
aI245
aI1195
aI3741
aI755
aI8047
aI15431
aI21001
asS'd'
I3886
sa(dp7772
S'a'
I3703
sS's'
I16
sS'm'
(lp7773
I1
aI3
aI7
aI3
aI17
aI47
aI107
aI137
aI99
aI255
aI1597
aI3281
aI5779
aI13487
aI15061
aI19199
asS'd'
I3887
sa(dp7774
S'a'
I3704
sS's'
I16
sS'm'
(lp7775
I1
aI1
aI3
aI3
aI9
aI39
aI77
aI227
aI511
aI839
aI1375
aI3887
aI25
aI14763
aI13259
aI217
asS'd'
I3888
sa(dp7776
S'a'
I3723
sS's'
I16
sS'm'
(lp7777
I1
aI3
aI5
aI7
aI17
aI3
aI87
aI61
aI439
aI287
aI709
aI4085
aI4251
aI8945
aI28203
aI24011
asS'd'
I3889
sa(dp7778
S'a'
I3743
sS's'
I16
sS'm'
(lp7779
I1
aI3
aI1
aI1
aI29
aI25
aI49
aI101
aI209
aI359
aI285
aI1593
aI4161
aI2943
aI23225
aI6381
asS'd'
I3890
sa(dp7780
S'a'
I3753
sS's'
I16
sS'm'
(lp7781
I1
aI1
aI3
aI13
aI1
aI45
aI87
aI7
aI491
aI399
aI905
aI1403
aI4791
aI7419
aI14355
aI47767
asS'd'
I3891
sa(dp7782
S'a'
I3756
sS's'
I16
sS'm'
(lp7783
I1
aI1
aI7
aI15
aI13
aI25
aI111
aI197
aI297
aI301
aI499
aI4007
aI2235
aI7681
aI4641
aI32447
asS'd'
I3892
sa(dp7784
S'a'
I3759
sS's'
I16
sS'm'
(lp7785
I1
aI1
aI3
aI3
aI27
aI41
aI97
aI83
aI405
aI353
aI1609
aI201
aI1503
aI10673
aI29377
aI20445
asS'd'
I3893
sa(dp7786
S'a'
I3762
sS's'
I16
sS'm'
(lp7787
I1
aI1
aI7
aI3
aI9
aI47
aI65
aI191
aI207
aI545
aI377
aI3011
aI7361
aI3467
aI14073
aI46769
asS'd'
I3894
sa(dp7788
S'a'
I3771
sS's'
I16
sS'm'
(lp7789
I1
aI1
aI7
aI5
aI7
aI39
aI9
aI91
aI187
aI949
aI1829
aI161
aI3689
aI4145
aI32675
aI23263
asS'd'
I3895
sa(dp7790
S'a'
I3776
sS's'
I16
sS'm'
(lp7791
I1
aI1
aI5
aI9
aI29
aI9
aI83
aI113
aI77
aI673
aI613
aI3645
aI6671
aI8583
aI27701
aI18615
asS'd'
I3896
sa(dp7792
S'a'
I3779
sS's'
I16
sS'm'
(lp7793
I1
aI3
aI5
aI9
aI29
aI13
aI127
aI247
aI285
aI845
aI463
aI539
aI4441
aI1867
aI12469
aI16213
asS'd'
I3897
sa(dp7794
S'a'
I3839
sS's'
I16
sS'm'
(lp7795
I1
aI3
aI7
aI15
aI1
aI29
aI47
aI157
aI239
aI595
aI563
aI1103
aI3431
aI2849
aI28125
aI19969
asS'd'
I3898
sa(dp7796
S'a'
I3856
sS's'
I16
sS'm'
(lp7797
I1
aI1
aI1
aI15
aI25
aI13
aI1
aI131
aI57
aI257
aI2021
aI169
aI7603
aI10721
aI21675
aI63171
asS'd'
I3899
sa(dp7798
S'a'
I3871
sS's'
I16
sS'm'
(lp7799
I1
aI3
aI5
aI3
aI5
aI19
aI31
aI57
aI275
aI381
aI775
aI681
aI1145
aI12237
aI5141
aI29375
asS'd'
I3900
sa(dp7800
S'a'
I3887
sS's'
I16
sS'm'
(lp7801
I1
aI3
aI5
aI13
aI27
aI13
aI47
aI201
aI267
aI581
aI1563
aI3845
aI951
aI7209
aI27253
aI19755
asS'd'
I3901
sa(dp7802
S'a'
I3896
sS's'
I16
sS'm'
(lp7803
I1
aI3
aI5
aI15
aI19
aI35
aI57
aI17
aI61
aI273
aI967
aI3029
aI1747
aI1753
aI31321
aI23711
asS'd'
I3902
sa(dp7804
S'a'
I3901
sS's'
I16
sS'm'
(lp7805
I1
aI1
aI1
aI5
aI13
aI13
aI7
aI177
aI335
aI393
aI1401
aI1411
aI4703
aI8259
aI1281
aI39835
asS'd'
I3903
sa(dp7806
S'a'
I3916
sS's'
I16
sS'm'
(lp7807
I1
aI1
aI3
aI15
aI25
aI27
aI27
aI121
aI183
aI105
aI663
aI1375
aI6987
aI7151
aI13763
aI39323
asS'd'
I3904
sa(dp7808
S'a'
I3919
sS's'
I16
sS'm'
(lp7809
I1
aI3
aI7
aI5
aI15
aI1
aI81
aI129
aI455
aI163
aI675
aI81
aI3735
aI14409
aI7269
aI16425
asS'd'
I3905
sa(dp7810
S'a'
I3937
sS's'
I16
sS'm'
(lp7811
I1
aI3
aI3
aI11
aI13
aI7
aI79
aI157
aI165
aI663
aI229
aI3539
aI1837
aI6485
aI30729
aI42157
asS'd'
I3906
sa(dp7812
S'a'
I3943
sS's'
I16
sS'm'
(lp7813
I1
aI1
aI5
aI15
aI9
aI9
aI9
aI47
aI133
aI863
aI43
aI1461
aI511
aI13991
aI24781
aI19221
asS'd'
I3907
sa(dp7814
S'a'
I3955
sS's'
I16
sS'm'
(lp7815
I1
aI3
aI1
aI7
aI31
aI33
aI103
aI13
aI159
aI689
aI1353
aI4025
aI6051
aI7683
aI1741
aI30047
asS'd'
I3908
sa(dp7816
S'a'
I3961
sS's'
I16
sS'm'
(lp7817
I1
aI1
aI3
aI11
aI5
aI45
aI71
aI219
aI475
aI585
aI1207
aI3163
aI4661
aI4713
aI12729
aI30445
asS'd'
I3909
sa(dp7818
S'a'
I3988
sS's'
I16
sS'm'
(lp7819
I1
aI3
aI7
aI5
aI5
aI53
aI101
aI227
aI129
aI521
aI91
aI1129
aI4683
aI11235
aI24697
aI45055
asS'd'
I3910
sa(dp7820
S'a'
I3997
sS's'
I16
sS'm'
(lp7821
I1
aI1
aI3
aI13
aI1
aI43
aI7
aI1
aI73
aI857
aI1713
aI185
aI1685
aI2369
aI24187
aI40419
asS'd'
I3911
sa(dp7822
S'a'
I4011
sS's'
I16
sS'm'
(lp7823
I1
aI1
aI7
aI7
aI21
aI7
aI13
aI177
aI503
aI1003
aI1091
aI2411
aI1433
aI9063
aI13901
aI3329
asS'd'
I3912
sa(dp7824
S'a'
I4026
sS's'
I16
sS'm'
(lp7825
I1
aI1
aI7
aI1
aI7
aI41
aI99
aI203
aI325
aI249
aI1763
aI545
aI2981
aI14125
aI7815
aI11385
asS'd'
I3913
sa(dp7826
S'a'
I4033
sS's'
I16
sS'm'
(lp7827
I1
aI3
aI7
aI11
aI3
aI11
aI95
aI137
aI325
aI701
aI1177
aI1631
aI4483
aI2955
aI30229
aI25577
asS'd'
I3914
sa(dp7828
S'a'
I4045
sS's'
I16
sS'm'
(lp7829
I1
aI1
aI7
aI7
aI17
aI45
aI77
aI103
aI143
aI97
aI1963
aI3635
aI1539
aI10491
aI23483
aI22767
asS'd'
I3915
sa(dp7830
S'a'
I4060
sS's'
I16
sS'm'
(lp7831
I1
aI1
aI7
aI15
aI7
aI5
aI81
aI63
aI243
aI55
aI39
aI207
aI2315
aI8285
aI8155
aI11631
asS'd'
I3916
sa(dp7832
S'a'
I4063
sS's'
I16
sS'm'
(lp7833
I1
aI3
aI5
aI15
aI23
aI19
aI115
aI9
aI125
aI851
aI161
aI3767
aI3513
aI1855
aI11139
aI1719
asS'd'
I3917
sa(dp7834
S'a'
I4064
sS's'
I16
sS'm'
(lp7835
I1
aI3
aI7
aI11
aI11
aI23
aI15
aI13
aI235
aI5
aI1039
aI1425
aI6485
aI5539
aI8967
aI64809
asS'd'
I3918
sa(dp7836
S'a'
I4126
sS's'
I16
sS'm'
(lp7837
I1
aI3
aI5
aI7
aI19
aI11
aI83
aI135
aI45
aI905
aI1081
aI1857
aI3185
aI13555
aI21365
aI38143
asS'd'
I3919
sa(dp7838
S'a'
I4136
sS's'
I16
sS'm'
(lp7839
I1
aI1
aI5
aI1
aI25
aI27
aI119
aI109
aI167
aI847
aI1539
aI2653
aI797
aI11185
aI23501
aI22389
asS'd'
I3920
sa(dp7840
S'a'
I4167
sS's'
I16
sS'm'
(lp7841
I1
aI1
aI7
aI7
aI11
aI3
aI51
aI97
aI277
aI557
aI207
aI3645
aI825
aI8521
aI26653
aI60071
asS'd'
I3921
sa(dp7842
S'a'
I4173
sS's'
I16
sS'm'
(lp7843
I1
aI3
aI3
aI15
aI17
aI35
aI57
aI7
aI267
aI549
aI97
aI243
aI1137
aI10311
aI6737
aI19077
asS'd'
I3922
sa(dp7844
S'a'
I4188
sS's'
I16
sS'm'
(lp7845
I1
aI1
aI1
aI15
aI23
aI33
aI27
aI203
aI415
aI1023
aI1145
aI1881
aI7715
aI4413
aI3727
aI5185
asS'd'
I3923
sa(dp7846
S'a'
I4195
sS's'
I16
sS'm'
(lp7847
I1
aI1
aI3
aI3
aI13
aI47
aI63
aI13
aI75
aI505
aI595
aI2911
aI4029
aI14187
aI23151
aI42877
asS'd'
I3924
sa(dp7848
S'a'
I4226
sS's'
I16
sS'm'
(lp7849
I1
aI1
aI5
aI15
aI23
aI5
aI11
aI65
aI147
aI675
aI1961
aI2177
aI727
aI15077
aI23759
aI10195
asS'd'
I3925
sa(dp7850
S'a'
I4291
sS's'
I16
sS'm'
(lp7851
I1
aI3
aI5
aI9
aI9
aI39
aI69
aI229
aI341
aI627
aI1331
aI3139
aI3921
aI9219
aI14887
aI4659
asS'd'
I3926
sa(dp7852
S'a'
I4298
sS's'
I16
sS'm'
(lp7853
I1
aI1
aI7
aI3
aI1
aI35
aI49
aI71
aI165
aI83
aI719
aI2771
aI6475
aI7821
aI16709
aI4449
asS'd'
I3927
sa(dp7854
S'a'
I4308
sS's'
I16
sS'm'
(lp7855
I1
aI3
aI5
aI5
aI23
aI15
aI3
aI57
aI465
aI77
aI121
aI3767
aI6841
aI13601
aI12035
aI14075
asS'd'
I3928
sa(dp7856
S'a'
I4312
sS's'
I16
sS'm'
(lp7857
I1
aI1
aI7
aI3
aI3
aI23
aI45
aI131
aI287
aI941
aI713
aI415
aI6865
aI14209
aI29555
aI55493
asS'd'
I3929
sa(dp7858
S'a'
I4336
sS's'
I16
sS'm'
(lp7859
I1
aI3
aI5
aI11
aI29
aI35
aI55
aI75
aI225
aI779
aI569
aI1795
aI1377
aI12765
aI19081
aI47287
asS'd'
I3930
sa(dp7860
S'a'
I4371
sS's'
I16
sS'm'
(lp7861
I1
aI3
aI7
aI3
aI31
aI47
aI127
aI89
aI157
aI737
aI1395
aI3615
aI7923
aI14731
aI15797
aI40061
asS'd'
I3931
sa(dp7862
S'a'
I4378
sS's'
I16
sS'm'
(lp7863
I1
aI1
aI1
aI11
aI21
aI37
aI21
aI59
aI9
aI141
aI193
aI3095
aI3435
aI12371
aI26931
aI61861
asS'd'
I3932
sa(dp7864
S'a'
I4384
sS's'
I16
sS'm'
(lp7865
I1
aI1
aI3
aI7
aI13
aI51
aI15
aI153
aI77
aI1013
aI651
aI3949
aI6229
aI14297
aI1039
aI46139
asS'd'
I3933
sa(dp7866
S'a'
I4393
sS's'
I16
sS'm'
(lp7867
I1
aI3
aI3
aI13
aI7
aI43
aI95
aI61
aI217
aI3
aI549
aI739
aI123
aI3661
aI15375
aI13919
asS'd'
I3934
sa(dp7868
S'a'
I4421
sS's'
I16
sS'm'
(lp7869
I1
aI3
aI5
aI9
aI13
aI37
aI101
aI89
aI55
aI413
aI1089
aI775
aI7575
aI13063
aI31393
aI29583
asS'd'
I3935
sa(dp7870
S'a'
I4425
sS's'
I16
sS'm'
(lp7871
I1
aI1
aI3
aI9
aI25
aI63
aI119
aI143
aI499
aI145
aI603
aI2067
aI4713
aI13457
aI14053
aI117
asS'd'
I3936
sa(dp7872
S'a'
I4439
sS's'
I16
sS'm'
(lp7873
I1
aI1
aI5
aI9
aI7
aI23
aI57
aI253
aI115
aI591
aI2003
aI63
aI7615
aI11493
aI28519
aI47087
asS'd'
I3937
sa(dp7874
S'a'
I4440
sS's'
I16
sS'm'
(lp7875
I1
aI1
aI7
aI3
aI7
aI53
aI121
aI33
aI233
aI645
aI1093
aI1697
aI7213
aI2603
aI10743
aI51303
asS'd'
I3938
sa(dp7876
S'a'
I4500
sS's'
I16
sS'm'
(lp7877
I1
aI3
aI5
aI7
aI13
aI31
aI17
aI125
aI93
aI969
aI159
aI1529
aI7165
aI7371
aI8707
aI56953
asS'd'
I3939
sa(dp7878
S'a'
I4514
sS's'
I16
sS'm'
(lp7879
I1
aI3
aI3
aI1
aI13
aI9
aI91
aI25
aI171
aI843
aI1635
aI2043
aI1043
aI15893
aI11409
aI53689
asS'd'
I3940
sa(dp7880
S'a'
I4523
sS's'
I16
sS'm'
(lp7881
I1
aI3
aI5
aI7
aI13
aI19
aI89
aI97
aI203
aI923
aI1109
aI2061
aI463
aI11703
aI8925
aI56015
asS'd'
I3941
sa(dp7882
S'a'
I4534
sS's'
I16
sS'm'
(lp7883
I1
aI3
aI5
aI11
aI5
aI21
aI79
aI237
aI195
aI649
aI717
aI211
aI919
aI12855
aI3045
aI39659
asS'd'
I3942
sa(dp7884
S'a'
I4593
sS's'
I16
sS'm'
(lp7885
I1
aI1
aI1
aI15
aI13
aI19
aI21
aI69
aI393
aI257
aI1263
aI309
aI3209
aI8403
aI24467
aI6467
asS'd'
I3943
sa(dp7886
S'a'
I4615
sS's'
I16
sS'm'
(lp7887
I1
aI1
aI1
aI11
aI7
aI27
aI59
aI117
aI379
aI353
aI943
aI2513
aI3869
aI4567
aI12989
aI13139
asS'd'
I3944
sa(dp7888
S'a'
I4630
sS's'
I16
sS'm'
(lp7889
I1
aI1
aI1
aI3
aI13
aI43
aI11
aI15
aI149
aI237
aI1555
aI71
aI2357
aI15773
aI21419
aI40571
asS'd'
I3945
sa(dp7890
S'a'
I4636
sS's'
I16
sS'm'
(lp7891
I1
aI3
aI1
aI9
aI19
aI23
aI59
aI215
aI15
aI921
aI1729
aI249
aI3785
aI7171
aI1233
aI3449
asS'd'
I3946
sa(dp7892
S'a'
I4645
sS's'
I16
sS'm'
(lp7893
I1
aI1
aI1
aI7
aI7
aI37
aI63
aI205
aI75
aI599
aI951
aI2513
aI3347
aI2497
aI8019
aI5433
asS'd'
I3947
sa(dp7894
S'a'
I4684
sS's'
I16
sS'm'
(lp7895
I1
aI3
aI3
aI15
aI27
aI17
aI25
aI201
aI23
aI699
aI1525
aI465
aI1115
aI12299
aI14747
aI40363
asS'd'
I3948
sa(dp7896
S'a'
I4687
sS's'
I16
sS'm'
(lp7897
I1
aI1
aI1
aI3
aI29
aI59
aI115
aI233
aI107
aI815
aI291
aI3821
aI7325
aI7381
aI21445
aI33917
asS'd'
I3949
sa(dp7898
S'a'
I4723
sS's'
I16
sS'm'
(lp7899
I1
aI3
aI1
aI11
aI11
aI33
aI107
aI171
aI421
aI893
aI587
aI3373
aI4101
aI3885
aI25383
aI12035
asS'd'
I3950
sa(dp7900
S'a'
I4735
sS's'
I16
sS'm'
(lp7901
I1
aI3
aI3
aI7
aI5
aI23
aI43
aI51
aI357
aI77
aI1327
aI2995
aI1321
aI1571
aI26419
aI23603
asS'd'
I3951
sa(dp7902
S'a'
I4746
sS's'
I16
sS'm'
(lp7903
I1
aI3
aI7
aI9
aI27
aI57
aI101
aI51
aI215
aI215
aI469
aI303
aI723
aI2903
aI30569
aI42631
asS'd'
I3952
sa(dp7904
S'a'
I4779
sS's'
I16
sS'm'
(lp7905
I1
aI3
aI3
aI13
aI1
aI7
aI63
aI205
aI143
aI321
aI1439
aI253
aI2667
aI1271
aI11761
aI55631
asS'd'
I3953
sa(dp7906
S'a'
I4782
sS's'
I16
sS'm'
(lp7907
I1
aI1
aI7
aI9
aI3
aI7
aI7
aI15
aI503
aI875
aI1619
aI1715
aI5047
aI5665
aI5503
aI17745
asS'd'
I3954
sa(dp7908
S'a'
I4793
sS's'
I16
sS'm'
(lp7909
I1
aI1
aI7
aI15
aI19
aI49
aI65
aI31
aI245
aI371
aI377
aI2963
aI6185
aI5519
aI10743
aI33231
asS'd'
I3955
sa(dp7910
S'a'
I4796
sS's'
I16
sS'm'
(lp7911
I1
aI1
aI7
aI3
aI25
aI27
aI115
aI51
aI299
aI451
aI285
aI1709
aI6153
aI14881
aI17861
aI22071
asS'd'
I3956
sa(dp7912
S'a'
I4813
sS's'
I16
sS'm'
(lp7913
I1
aI3
aI1
aI5
aI21
aI21
aI127
aI185
aI325
aI995
aI213
aI3279
aI4769
aI15943
aI2589
aI29567
asS'd'
I3957
sa(dp7914
S'a'
I4850
sS's'
I16
sS'm'
(lp7915
I1
aI3
aI7
aI5
aI21
aI9
aI63
aI59
aI159
aI743
aI663
aI2965
aI97
aI8993
aI25633
aI29033
asS'd'
I3958
sa(dp7916
S'a'
I4867
sS's'
I16
sS'm'
(lp7917
I1
aI3
aI7
aI13
aI3
aI35
aI59
aI101
aI21
aI659
aI1531
aI3995
aI795
aI2143
aI21749
aI52715
asS'd'
I3959
sa(dp7918
S'a'
I4874
sS's'
I16
sS'm'
(lp7919
I1
aI3
aI3
aI15
aI27
aI29
aI95
aI1
aI501
aI425
aI417
aI2351
aI7877
aI4127
aI3633
aI23347
asS'd'
I3960
sa(dp7920
S'a'
I4881
sS's'
I16
sS'm'
(lp7921
I1
aI3
aI5
aI7
aI7
aI49
aI55
aI19
aI329
aI467
aI425
aI1609
aI6987
aI16123
aI26879
aI42883
asS'd'
I3961
sa(dp7922
S'a'
I4894
sS's'
I16
sS'm'
(lp7923
I1
aI1
aI1
aI15
aI17
aI21
aI13
aI13
aI85
aI7
aI677
aI3739
aI5491
aI6299
aI29957
aI55765
asS'd'
I3962
sa(dp7924
S'a'
I4904
sS's'
I16
sS'm'
(lp7925
I1
aI1
aI1
aI7
aI31
aI21
aI1
aI5
aI193
aI659
aI979
aI3409
aI3151
aI6615
aI7445
aI8151
asS'd'
I3963
sa(dp7926
S'a'
I4927
sS's'
I16
sS'm'
(lp7927
I1
aI3
aI1
aI1
aI11
aI61
aI27
aI205
aI263
aI805
aI955
aI3469
aI1233
aI1609
aI15329
aI13353
asS'd'
I3964
sa(dp7928
S'a'
I4929
sS's'
I16
sS'm'
(lp7929
I1
aI3
aI3
aI9
aI3
aI29
aI59
aI75
aI149
aI557
aI663
aI3887
aI3369
aI3397
aI10611
aI9511
asS'd'
I3965
sa(dp7930
S'a'
I4989
sS's'
I16
sS'm'
(lp7931
I1
aI1
aI7
aI13
aI29
aI21
aI101
aI139
aI99
aI411
aI569
aI2343
aI6901
aI1685
aI20599
aI49543
asS'd'
I3966
sa(dp7932
S'a'
I5000
sS's'
I16
sS'm'
(lp7933
I1
aI3
aI3
aI15
aI11
aI3
aI87
aI89
aI5
aI293
aI291
aI1405
aI1489
aI9877
aI32505
aI32263
asS'd'
I3967
sa(dp7934
S'a'
I5020
sS's'
I16
sS'm'
(lp7935
I1
aI1
aI5
aI5
aI19
aI45
aI89
aI5
aI381
aI253
aI1339
aI707
aI4645
aI14177
aI29441
aI8965
asS'd'
I3968
sa(dp7936
S'a'
I5036
sS's'
I16
sS'm'
(lp7937
I1
aI3
aI7
aI15
aI27
aI45
aI25
aI177
aI81
aI229
aI1339
aI2143
aI6547
aI6841
aI23449
aI14813
asS'd'
I3969
sa(dp7938
S'a'
I5041
sS's'
I16
sS'm'
(lp7939
I1
aI1
aI1
aI3
aI27
aI23
aI81
aI157
aI53
aI513
aI1435
aI277
aI2353
aI3545
aI21461
aI51479
asS'd'
I3970
sa(dp7940
S'a'
I5059
sS's'
I16
sS'm'
(lp7941
I1
aI3
aI1
aI3
aI3
aI17
aI75
aI139
aI283
aI881
aI1157
aI2081
aI937
aI14549
aI10215
aI13053
asS'd'
I3971
sa(dp7942
S'a'
I5074
sS's'
I16
sS'm'
(lp7943
I1
aI1
aI7
aI9
aI25
aI27
aI27
aI133
aI21
aI559
aI225
aI613
aI6931
aI11785
aI23413
aI35757
asS'd'
I3972
sa(dp7944
S'a'
I5090
sS's'
I16
sS'm'
(lp7945
I1
aI1
aI3
aI13
aI19
aI9
aI65
aI49
aI453
aI779
aI621
aI1151
aI1807
aI13269
aI6515
aI17113
asS'd'
I3973
sa(dp7946
S'a'
I5110
sS's'
I16
sS'm'
(lp7947
I1
aI1
aI1
aI13
aI21
aI49
aI39
aI79
aI119
aI401
aI903
aI493
aI3779
aI7389
aI29425
aI28091
asS'd'
I3974
sa(dp7948
S'a'
I5134
sS's'
I16
sS'm'
(lp7949
I1
aI3
aI1
aI3
aI23
aI57
aI59
aI213
aI463
aI839
aI1201
aI1951
aI5197
aI13035
aI29657
aI35517
asS'd'
I3975
sa(dp7950
S'a'
I5152
sS's'
I16
sS'm'
(lp7951
I1
aI3
aI7
aI7
aI3
aI49
aI29
aI181
aI367
aI101
aI1277
aI3329
aI3563
aI10373
aI29757
aI62555
asS'd'
I3976
sa(dp7952
S'a'
I5176
sS's'
I16
sS'm'
(lp7953
I1
aI3
aI1
aI7
aI31
aI31
aI117
aI213
aI373
aI57
aI1095
aI2733
aI3431
aI3915
aI7665
aI44459
asS'd'
I3977
sa(dp7954
S'a'
I5181
sS's'
I16
sS'm'
(lp7955
I1
aI1
aI7
aI5
aI9
aI25
aI47
aI117
aI355
aI495
aI1367
aI2579
aI5617
aI787
aI27655
aI18885
asS'd'
I3978
sa(dp7956
S'a'
I5204
sS's'
I16
sS'm'
(lp7957
I1
aI1
aI1
aI3
aI9
aI39
aI113
aI87
aI107
aI477
aI891
aI2273
aI4239
aI7521
aI147
aI1737
asS'd'
I3979
sa(dp7958
S'a'
I5218
sS's'
I16
sS'm'
(lp7959
I1
aI1
aI1
aI3
aI13
aI61
aI81
aI225
aI113
aI441
aI889
aI1915
aI3897
aI15179
aI4053
aI5925
asS'd'
I3980
sa(dp7960
S'a'
I5242
sS's'
I16
sS'm'
(lp7961
I1
aI1
aI5
aI3
aI15
aI47
aI59
aI187
aI475
aI197
aI1381
aI33
aI4605
aI1487
aI14359
aI33769
asS'd'
I3981
sa(dp7962
S'a'
I5253
sS's'
I16
sS'm'
(lp7963
I1
aI3
aI7
aI15
aI23
aI45
aI53
aI215
aI129
aI465
aI795
aI53
aI7077
aI9517
aI2663
aI55397
asS'd'
I3982
sa(dp7964
S'a'
I5260
sS's'
I16
sS'm'
(lp7965
I1
aI1
aI7
aI13
aI25
aI49
aI105
aI255
aI245
aI153
aI1093
aI2123
aI2823
aI5125
aI17483
aI49003
asS'd'
I3983
sa(dp7966
S'a'
I5281
sS's'
I16
sS'm'
(lp7967
I1
aI1
aI1
aI13
aI31
aI5
aI7
aI243
aI255
aI231
aI1663
aI1007
aI7573
aI405
aI29183
aI11367
asS'd'
I3984
sa(dp7968
S'a'
I5282
sS's'
I16
sS'm'
(lp7969
I1
aI1
aI5
aI13
aI15
aI15
aI115
aI91
aI63
aI1013
aI1713
aI1945
aI6397
aI14213
aI24417
aI40807
asS'd'
I3985
sa(dp7970
S'a'
I5313
sS's'
I16
sS'm'
(lp7971
I1
aI1
aI5
aI3
aI19
aI49
aI91
aI25
aI43
aI601
aI25
aI2405
aI1973
aI629
aI497
aI12625
asS'd'
I3986
sa(dp7972
S'a'
I5316
sS's'
I16
sS'm'
(lp7973
I1
aI1
aI3
aI5
aI13
aI45
aI11
aI81
aI251
aI545
aI1155
aI1409
aI7187
aI847
aI2835
aI32909
asS'd'
I3987
sa(dp7974
S'a'
I5326
sS's'
I16
sS'm'
(lp7975
I1
aI3
aI1
aI13
aI27
aI57
aI1
aI103
aI465
aI809
aI1727
aI3721
aI3347
aI3791
aI17247
aI8377
asS'd'
I3988
sa(dp7976
S'a'
I5340
sS's'
I16
sS'm'
(lp7977
I1
aI3
aI3
aI15
aI25
aI31
aI91
aI91
aI119
aI205
aI1431
aI703
aI5327
aI7323
aI30415
aI61955
asS'd'
I3989
sa(dp7978
S'a'
I5347
sS's'
I16
sS'm'
(lp7979
I1
aI3
aI5
aI11
aI19
aI39
aI79
aI243
aI109
aI463
aI1869
aI2133
aI4139
aI10461
aI14793
aI24025
asS'd'
I3990
sa(dp7980
S'a'
I5354
sS's'
I16
sS'm'
(lp7981
I1
aI3
aI5
aI7
aI23
aI41
aI5
aI7
aI249
aI3
aI1743
aI489
aI4921
aI397
aI30955
aI22207
asS'd'
I3991
sa(dp7982
S'a'
I5368
sS's'
I16
sS'm'
(lp7983
I1
aI3
aI5
aI15
aI3
aI7
aI115
aI19
aI217
aI231
aI1183
aI3723
aI5055
aI12967
aI7855
aI5067
asS'd'
I3992
sa(dp7984
S'a'
I5394
sS's'
I16
sS'm'
(lp7985
I1
aI3
aI3
aI3
aI11
aI31
aI113
aI41
aI429
aI797
aI557
aI1199
aI1819
aI1933
aI9917
aI32229
asS'd'
I3993
sa(dp7986
S'a'
I5396
sS's'
I16
sS'm'
(lp7987
I1
aI1
aI5
aI3
aI13
aI63
aI31
aI183
aI465
aI915
aI723
aI3227
aI4125
aI2813
aI26313
aI34287
asS'd'
I3994
sa(dp7988
S'a'
I5400
sS's'
I16
sS'm'
(lp7989
I1
aI1
aI7
aI5
aI31
aI55
aI117
aI243
aI37
aI885
aI85
aI1067
aI3895
aI15655
aI28527
aI32109
asS'd'
I3995
sa(dp7990
S'a'
I5405
sS's'
I16
sS'm'
(lp7991
I1
aI3
aI7
aI15
aI17
aI43
aI43
aI173
aI119
aI187
aI1161
aI599
aI4595
aI1681
aI11981
aI681
asS'd'
I3996
sa(dp7992
S'a'
I5439
sS's'
I16
sS'm'
(lp7993
I1
aI1
aI7
aI7
aI29
aI47
aI25
aI93
aI411
aI103
aI783
aI1029
aI1927
aI3569
aI27647
aI8281
asS'd'
I3997
sa(dp7994
S'a'
I5442
sS's'
I16
sS'm'
(lp7995
I1
aI3
aI3
aI9
aI19
aI57
aI31
aI183
aI141
aI889
aI157
aI2267
aI5701
aI6273
aI25739
aI34039
asS'd'
I3998
sa(dp7996
S'a'
I5459
sS's'
I16
sS'm'
(lp7997
I1
aI3
aI5
aI1
aI29
aI35
aI105
aI165
aI505
aI299
aI1149
aI2397
aI2013
aI11591
aI15917
aI4791
asS'd'
I3999
sa(dp7998
S'a'
I5478
sS's'
I16
sS'm'
(lp7999
I1
aI3
aI3
aI9
aI7
aI35
aI47
aI77
aI69
aI335
aI585
aI1131
aI531
aI8597
aI307
aI55985
asS'd'
I4000
sa(dp8000
S'a'
I5484
sS's'
I16
sS'm'
(lp8001
I1
aI3
aI7
aI1
aI29
aI9
aI25
aI155
aI149
aI845
aI567
aI3735
aI3501
aI9365
aI12025
aI19131
asS'd'
I4001
sa(dp8002
S'a'
I5508
sS's'
I16
sS'm'
(lp8003
I1
aI3
aI5
aI3
aI11
aI31
aI25
aI9
aI411
aI519
aI1611
aI1441
aI1487
aI10049
aI14373
aI24605
asS'd'
I4002
sa(dp8004
S'a'
I5523
sS's'
I16
sS'm'
(lp8005
I1
aI3
aI3
aI5
aI3
aI7
aI101
aI107
aI339
aI155
aI1843
aI2529
aI443
aI8177
aI28655
aI8151
asS'd'
I4003
sa(dp8006
S'a'
I5545
sS's'
I16
sS'm'
(lp8007
I1
aI1
aI7
aI5
aI29
aI37
aI73
aI131
aI125
aI451
aI947
aI797
aI5053
aI10155
aI30801
aI27235
asS'd'
I4004
sa(dp8008
S'a'
I5565
sS's'
I16
sS'm'
(lp8009
I1
aI1
aI7
aI13
aI19
aI47
aI101
aI45
aI495
aI457
aI1293
aI1971
aI5495
aI12737
aI17687
aI26123
asS'd'
I4005
sa(dp8010
S'a'
I5566
sS's'
I16
sS'm'
(lp8011
I1
aI1
aI7
aI7
aI11
aI11
aI75
aI177
aI251
aI553
aI1883
aI3379
aI1429
aI12227
aI10301
aI16467
asS'd'
I4006
sa(dp8012
S'a'
I5580
sS's'
I16
sS'm'
(lp8013
I1
aI3
aI3
aI9
aI3
aI61
aI95
aI35
aI97
aI551
aI233
aI2045
aI4873
aI9109
aI10019
aI64523
asS'd'
I4007
sa(dp8014
S'a'
I5608
sS's'
I16
sS'm'
(lp8015
I1
aI3
aI1
aI5
aI11
aI3
aI15
aI177
aI301
aI573
aI2029
aI191
aI5551
aI12083
aI27287
aI57235
asS'd'
I4008
sa(dp8016
S'a'
I5613
sS's'
I16
sS'm'
(lp8017
I1
aI3
aI5
aI1
aI21
aI9
aI121
aI169
aI347
aI187
aI57
aI3163
aI5609
aI1921
aI17581
aI28351
asS'd'
I4009
sa(dp8018
S'a'
I5647
sS's'
I16
sS'm'
(lp8019
I1
aI3
aI3
aI1
aI31
aI51
aI15
aI45
aI429
aI245
aI573
aI1595
aI5343
aI7519
aI17009
aI1299
asS'd'
I4010
sa(dp8020
S'a'
I5661
sS's'
I16
sS'm'
(lp8021
I1
aI1
aI7
aI3
aI13
aI47
aI109
aI65
aI389
aI867
aI963
aI145
aI1089
aI9749
aI19625
aI43121
asS'd'
I4011
sa(dp8022
S'a'
I5671
sS's'
I16
sS'm'
(lp8023
I1
aI3
aI1
aI7
aI21
aI61
aI77
aI67
aI143
aI579
aI625
aI2007
aI6343
aI4259
aI21233
aI237
asS'd'
I4012
sa(dp8024
S'a'
I5678
sS's'
I16
sS'm'
(lp8025
I1
aI3
aI5
aI9
aI27
aI15
aI107
aI91
aI399
aI895
aI645
aI2301
aI439
aI6789
aI18299
aI47285
asS'd'
I4013
sa(dp8026
S'a'
I5680
sS's'
I16
sS'm'
(lp8027
I1
aI3
aI3
aI5
aI13
aI11
aI43
aI199
aI505
aI409
aI25
aI2057
aI479
aI6031
aI9561
aI51613
asS'd'
I4014
sa(dp8028
S'a'
I5685
sS's'
I16
sS'm'
(lp8029
I1
aI1
aI7
aI13
aI15
aI55
aI105
aI53
aI171
aI925
aI1849
aI2881
aI6951
aI13069
aI865
aI41019
asS'd'
I4015
sa(dp8030
S'a'
I5689
sS's'
I16
sS'm'
(lp8031
I1
aI3
aI1
aI9
aI17
aI31
aI45
aI23
aI411
aI185
aI189
aI2123
aI2583
aI12713
aI12681
aI2231
asS'd'
I4016
sa(dp8032
S'a'
I5692
sS's'
I16
sS'm'
(lp8033
I1
aI3
aI7
aI9
aI3
aI63
aI11
aI253
aI177
aI127
aI545
aI3293
aI4449
aI15995
aI10223
aI33529
asS'd'
I4017
sa(dp8034
S'a'
I5724
sS's'
I16
sS'm'
(lp8035
I1
aI1
aI5
aI11
aI13
aI7
aI53
aI161
aI421
aI551
aI697
aI627
aI3879
aI1639
aI24419
aI3337
asS'd'
I4018
sa(dp8036
S'a'
I5745
sS's'
I16
sS'm'
(lp8037
I1
aI1
aI7
aI7
aI27
aI7
aI37
aI205
aI429
aI407
aI1133
aI3563
aI2921
aI6173
aI11173
aI3009
asS'd'
I4019
sa(dp8038
S'a'
I5755
sS's'
I16
sS'm'
(lp8039
I1
aI3
aI3
aI15
aI31
aI39
aI117
aI81
aI337
aI729
aI567
aI2299
aI1481
aI3189
aI1795
aI40299
asS'd'
I4020
sa(dp8040
S'a'
I5757
sS's'
I16
sS'm'
(lp8041
I1
aI3
aI5
aI15
aI15
aI47
aI91
aI127
aI275
aI55
aI951
aI3423
aI2879
aI6115
aI1549
aI2287
asS'd'
I4021
sa(dp8042
S'a'
I5786
sS's'
I16
sS'm'
(lp8043
I1
aI3
aI3
aI11
aI17
aI3
aI127
aI207
aI141
aI889
aI185
aI1095
aI4567
aI1371
aI30545
aI54445
asS'd'
I4022
sa(dp8044
S'a'
I5792
sS's'
I16
sS'm'
(lp8045
I1
aI1
aI7
aI3
aI25
aI11
aI11
aI139
aI43
aI1
aI1977
aI397
aI5775
aI6913
aI13249
aI46767
asS'd'
I4023
sa(dp8046
S'a'
I5810
sS's'
I16
sS'm'
(lp8047
I1
aI1
aI7
aI7
aI27
aI13
aI31
aI251
aI191
aI1015
aI161
aI3719
aI5321
aI13013
aI25187
aI51881
asS'd'
I4024
sa(dp8048
S'a'
I5824
sS's'
I16
sS'm'
(lp8049
I1
aI1
aI1
aI1
aI3
aI3
aI13
aI19
aI423
aI349
aI1955
aI2103
aI6395
aI3315
aI23809
aI25925
asS'd'
I4025
sa(dp8050
S'a'
I5869
sS's'
I16
sS'm'
(lp8051
I1
aI3
aI5
aI13
aI3
aI59
aI41
aI167
aI423
aI93
aI1299
aI2623
aI5829
aI8537
aI8701
aI43757
asS'd'
I4026
sa(dp8052
S'a'
I5872
sS's'
I16
sS'm'
(lp8053
I1
aI3
aI5
aI11
aI9
aI19
aI127
aI119
aI329
aI819
aI7
aI3809
aI5305
aI3643
aI27369
aI61827
asS'd'
I4027
sa(dp8054
S'a'
I5895
sS's'
I16
sS'm'
(lp8055
I1
aI3
aI1
aI15
aI25
aI43
aI55
aI159
aI205
aI911
aI983
aI2825
aI3751
aI7845
aI12023
aI18431
asS'd'
I4028
sa(dp8056
S'a'
I5923
sS's'
I16
sS'm'
(lp8057
I1
aI3
aI3
aI13
aI11
aI1
aI65
aI133
aI479
aI181
aI679
aI981
aI3317
aI6241
aI11693
aI9619
asS'd'
I4029
sa(dp8058
S'a'
I5925
sS's'
I16
sS'm'
(lp8059
I1
aI3
aI3
aI3
aI21
aI25
aI117
aI183
aI127
aI73
aI703
aI1469
aI257
aI11229
aI10167
aI50847
asS'd'
I4030
sa(dp8060
S'a'
I5926
sS's'
I16
sS'm'
(lp8061
I1
aI1
aI5
aI13
aI5
aI5
aI113
aI15
aI231
aI269
aI989
aI465
aI3267
aI15239
aI29503
aI42855
asS'd'
I4031
sa(dp8062
S'a'
I5944
sS's'
I16
sS'm'
(lp8063
I1
aI3
aI3
aI15
aI9
aI63
aI79
aI27
aI21
aI709
aI1969
aI3761
aI1015
aI13619
aI4205
aI40591
asS'd'
I4032
sa(dp8064
S'a'
I5986
sS's'
I16
sS'm'
(lp8065
I1
aI1
aI7
aI11
aI29
aI3
aI5
aI45
aI107
aI131
aI1287
aI3551
aI849
aI2003
aI27451
aI47103
asS'd'
I4033
sa(dp8066
S'a'
I6012
sS's'
I16
sS'm'
(lp8067
I1
aI3
aI5
aI11
aI3
aI47
aI59
aI53
aI369
aI249
aI915
aI2475
aI7539
aI763
aI7063
aI63971
asS'd'
I4034
sa(dp8068
S'a'
I6015
sS's'
I16
sS'm'
(lp8069
I1
aI1
aI5
aI1
aI7
aI53
aI45
aI127
aI321
aI341
aI635
aI2277
aI1383
aI10951
aI29055
aI45087
asS'd'
I4035
sa(dp8070
S'a'
I6046
sS's'
I16
sS'm'
(lp8071
I1
aI3
aI7
aI3
aI5
aI1
aI119
aI79
aI487
aI93
aI25
aI491
aI4085
aI6403
aI27779
aI8753
asS'd'
I4036
sa(dp8072
S'a'
I6049
sS's'
I16
sS'm'
(lp8073
I1
aI1
aI1
aI3
aI9
aI59
aI49
aI141
aI323
aI703
aI1175
aI423
aI4323
aI10083
aI4289
aI28931
asS'd'
I4037
sa(dp8074
S'a'
I6061
sS's'
I16
sS'm'
(lp8075
I1
aI3
aI3
aI15
aI31
aI31
aI73
aI15
aI187
aI653
aI91
aI1707
aI1431
aI9861
aI19071
aI8137
asS'd'
I4038
sa(dp8076
S'a'
I6067
sS's'
I16
sS'm'
(lp8077
I1
aI1
aI1
aI5
aI27
aI63
aI93
aI1
aI329
aI353
aI863
aI473
aI7681
aI10653
aI15819
aI8495
asS'd'
I4039
sa(dp8078
S'a'
I6099
sS's'
I16
sS'm'
(lp8079
I1
aI1
aI1
aI5
aI25
aI57
aI119
aI167
aI219
aI319
aI231
aI1065
aI6217
aI5131
aI1513
aI49281
asS'd'
I4040
sa(dp8080
S'a'
I6121
sS's'
I16
sS'm'
(lp8081
I1
aI3
aI7
aI3
aI17
aI3
aI113
aI91
aI201
aI179
aI1907
aI3423
aI821
aI12927
aI24827
aI49403
asS'd'
I4041
sa(dp8082
S'a'
I6155
sS's'
I16
sS'm'
(lp8083
I1
aI1
aI5
aI7
aI19
aI63
aI75
aI151
aI387
aI489
aI777
aI2049
aI1151
aI1351
aI25687
aI4143
asS'd'
I4042
sa(dp8084
S'a'
I6163
sS's'
I16
sS'm'
(lp8085
I1
aI3
aI5
aI7
aI9
aI37
aI9
aI3
aI87
aI385
aI1667
aI2139
aI7527
aI16133
aI30023
aI28783
asS'd'
I4043
sa(dp8086
S'a'
I6203
sS's'
I16
sS'm'
(lp8087
I1
aI1
aI5
aI9
aI11
aI55
aI95
aI73
aI413
aI867
aI589
aI2901
aI3021
aI413
aI5955
aI38921
asS'd'
I4044
sa(dp8088
S'a'
I6208
sS's'
I16
sS'm'
(lp8089
I1
aI3
aI5
aI15
aI1
aI17
aI17
aI7
aI485
aI323
aI519
aI2325
aI2255
aI4211
aI20661
aI28931
asS'd'
I4045
sa(dp8090
S'a'
I6231
sS's'
I16
sS'm'
(lp8091
I1
aI1
aI5
aI13
aI21
aI19
aI85
aI189
aI167
aI645
aI1475
aI3095
aI7123
aI3351
aI7961
aI20967
asS'd'
I4046
sa(dp8092
S'a'
I6241
sS's'
I16
sS'm'
(lp8093
I1
aI1
aI7
aI13
aI3
aI47
aI13
aI213
aI237
aI291
aI285
aI1465
aI1765
aI12361
aI32651
aI54205
asS'd'
I4047
sa(dp8094
S'a'
I6254
sS's'
I16
sS'm'
(lp8095
I1
aI3
aI7
aI13
aI13
aI27
aI71
aI35
aI67
aI117
aI647
aI2359
aI3295
aI8445
aI24761
aI29379
asS'd'
I4048
sa(dp8096
S'a'
I6262
sS's'
I16
sS'm'
(lp8097
I1
aI1
aI1
aI3
aI3
aI19
aI23
aI37
aI5
aI1019
aI5
aI1605
aI2291
aI14015
aI9311
aI39751
asS'd'
I4049
sa(dp8098
S'a'
I6266
sS's'
I16
sS'm'
(lp8099
I1
aI3
aI3
aI3
aI31
aI1
aI65
aI159
aI221
aI675
aI1061
aI971
aI2333
aI8265
aI14361
aI3263
asS'd'
I4050
sa(dp8100
S'a'
I6275
sS's'
I16
sS'm'
(lp8101
I1
aI1
aI3
aI7
aI3
aI5
aI81
aI17
aI101
aI991
aI753
aI2883
aI4977
aI4409
aI1757
aI26803
asS'd'
I4051
sa(dp8102
S'a'
I6278
sS's'
I16
sS'm'
(lp8103
I1
aI1
aI5
aI9
aI13
aI25
aI45
aI43
aI199
aI967
aI829
aI713
aI4547
aI7223
aI6497
aI53895
asS'd'
I4052
sa(dp8104
S'a'
I6292
sS's'
I16
sS'm'
(lp8105
I1
aI1
aI7
aI5
aI23
aI15
aI89
aI179
aI509
aI147
aI315
aI133
aI111
aI15577
aI23427
aI5229
asS'd'
I4053
sa(dp8106
S'a'
I6329
sS's'
I16
sS'm'
(lp8107
I1
aI3
aI3
aI7
aI27
aI7
aI113
aI65
aI315
aI135
aI1309
aI1179
aI5755
aI7513
aI6815
aI5137
asS'd'
I4054
sa(dp8108
S'a'
I6355
sS's'
I16
sS'm'
(lp8109
I1
aI1
aI3
aI7
aI1
aI13
aI29
aI155
aI477
aI721
aI71
aI865
aI3897
aI3331
aI30641
aI65471
asS'd'
I4055
sa(dp8110
S'a'
I6357
sS's'
I16
sS'm'
(lp8111
I1
aI1
aI7
aI3
aI29
aI45
aI83
aI3
aI461
aI109
aI1545
aI1365
aI6633
aI16137
aI23859
aI5995
asS'd'
I4056
sa(dp8112
S'a'
I6374
sS's'
I16
sS'm'
(lp8113
I1
aI3
aI1
aI1
aI3
aI33
aI77
aI83
aI459
aI187
aI879
aI3731
aI6939
aI6707
aI23409
aI24245
asS'd'
I4057
sa(dp8114
S'a'
I6380
sS's'
I16
sS'm'
(lp8115
I1
aI3
aI5
aI5
aI13
aI43
aI127
aI41
aI29
aI735
aI1391
aI2947
aI4873
aI4627
aI15
aI41719
asS'd'
I4058
sa(dp8116
S'a'
I6423
sS's'
I16
sS'm'
(lp8117
I1
aI3
aI1
aI3
aI17
aI17
aI51
aI93
aI189
aI227
aI449
aI2809
aI825
aI2009
aI9563
aI41435
asS'd'
I4059
sa(dp8118
S'a'
I6427
sS's'
I16
sS'm'
(lp8119
I1
aI3
aI3
aI11
aI25
aI47
aI113
aI173
aI141
aI919
aI677
aI117
aI5293
aI815
aI23749
aI19789
asS'd'
I4060
sa(dp8120
S'a'
I6430
sS's'
I16
sS'm'
(lp8121
I1
aI1
aI1
aI13
aI15
aI61
aI121
aI223
aI53
aI389
aI489
aI1527
aI4771
aI8975
aI8005
aI14275
asS'd'
I4061
sa(dp8122
S'a'
I6436
sS's'
I16
sS'm'
(lp8123
I1
aI1
aI3
aI15
aI31
aI57
aI111
aI145
aI279
aI991
aI489
aI3239
aI7647
aI473
aI31279
aI33447
asS'd'
I4062
sa(dp8124
S'a'
I6443
sS's'
I16
sS'm'
(lp8125
I1
aI1
aI7
aI5
aI31
aI13
aI75
aI185
aI395
aI611
aI609
aI159
aI7931
aI9887
aI4115
aI53121
asS'd'
I4063
sa(dp8126
S'a'
I6445
sS's'
I16
sS'm'
(lp8127
I1
aI3
aI5
aI5
aI13
aI39
aI103
aI237
aI77
aI913
aI511
aI1583
aI6763
aI14523
aI4247
aI63403
asS'd'
I4064
sa(dp8128
S'a'
I6458
sS's'
I16
sS'm'
(lp8129
I1
aI1
aI1
aI15
aI11
aI5
aI43
aI43
aI297
aI827
aI747
aI5
aI3785
aI15021
aI11291
aI36743
asS'd'
I4065
sa(dp8130
S'a'
I6478
sS's'
I16
sS'm'
(lp8131
I1
aI1
aI7
aI9
aI9
aI53
aI113
aI95
aI403
aI53
aI1335
aI4033
aI8147
aI11963
aI6523
aI23675
asS'd'
I4066
sa(dp8132
S'a'
I6490
sS's'
I16
sS'm'
(lp8133
I1
aI1
aI5
aI9
aI27
aI29
aI69
aI79
aI327
aI409
aI1147
aI1579
aI2625
aI12227
aI30933
aI9057
asS'd'
I4067
sa(dp8134
S'a'
I6508
sS's'
I16
sS'm'
(lp8135
I1
aI1
aI1
aI7
aI1
aI33
aI29
aI173
aI5
aI517
aI437
aI2035
aI57
aI12825
aI22095
aI65519
asS'd'
I4068
sa(dp8136
S'a'
I6519
sS's'
I16
sS'm'
(lp8137
I1
aI1
aI3
aI7
aI27
aI29
aI53
aI79
aI429
aI707
aI589
aI2605
aI339
aI7039
aI19319
aI17649
asS'd'
I4069
sa(dp8138
S'a'
I6520
sS's'
I16
sS'm'
(lp8139
I1
aI3
aI3
aI11
aI9
aI57
aI43
aI117
aI39
aI193
aI1427
aI2553
aI6877
aI7653
aI29041
aI44865
asS'd'
I4070
sa(dp8140
S'a'
I6530
sS's'
I16
sS'm'
(lp8141
I1
aI3
aI3
aI7
aI23
aI45
aI29
aI151
aI265
aI739
aI365
aI3565
aI6447
aI9761
aI24021
aI679
asS'd'
I4071
sa(dp8142
S'a'
I6541
sS's'
I16
sS'm'
(lp8143
I1
aI3
aI5
aI1
aI1
aI43
aI73
aI55
aI131
aI175
aI959
aI659
aI7315
aI15145
aI18301
aI14865
asS'd'
I4072
sa(dp8144
S'a'
I6556
sS's'
I16
sS'm'
(lp8145
I1
aI1
aI3
aI5
aI15
aI15
aI91
aI113
aI359
aI241
aI1627
aI1069
aI1761
aI211
aI32671
aI58833
asS'd'
I4073
sa(dp8146
S'a'
I6607
sS's'
I16
sS'm'
(lp8147
I1
aI3
aI3
aI7
aI29
aI27
aI79
aI53
aI409
aI81
aI693
aI3137
aI7385
aI11007
aI28459
aI28621
asS'd'
I4074
sa(dp8148
S'a'
I6612
sS's'
I16
sS'm'
(lp8149
I1
aI1
aI7
aI5
aI29
aI7
aI67
aI195
aI77
aI773
aI1361
aI2153
aI4459
aI7301
aI5129
aI17797
asS'd'
I4075
sa(dp8150
S'a'
I6626
sS's'
I16
sS'm'
(lp8151
I1
aI3
aI3
aI7
aI25
aI27
aI91
aI223
aI115
aI91
aI1645
aI2167
aI1955
aI9601
aI22127
aI13055
asS'd'
I4076
sa(dp8152
S'a'
I6632
sS's'
I16
sS'm'
(lp8153
I1
aI3
aI7
aI3
aI27
aI53
aI67
aI249
aI51
aI151
aI663
aI3231
aI895
aI6777
aI3037
aI56755
asS'd'
I4077
sa(dp8154
S'a'
I6649
sS's'
I16
sS'm'
(lp8155
I1
aI1
aI5
aI1
aI25
aI63
aI71
aI179
aI375
aI301
aI1127
aI2125
aI783
aI14481
aI7293
aI47883
asS'd'
I4078
sa(dp8156
S'a'
I6666
sS's'
I16
sS'm'
(lp8157
I1
aI1
aI3
aI9
aI25
aI3
aI39
aI69
aI1
aI85
aI1271
aI1571
aI1953
aI5077
aI20369
aI44827
asS'd'
I4079
sa(dp8158
S'a'
I6674
sS's'
I16
sS'm'
(lp8159
I1
aI3
aI1
aI13
aI11
aI61
aI77
aI59
aI127
aI475
aI1609
aI3553
aI2553
aI15589
aI9351
aI59787
asS'd'
I4080
sa(dp8160
S'a'
I6733
sS's'
I16
sS'm'
(lp8161
I1
aI3
aI1
aI5
aI21
aI7
aI61
aI27
aI199
aI653
aI1243
aI2481
aI5369
aI12903
aI30229
aI39453
asS'd'
I4081
sa(dp8162
S'a'
I6782
sS's'
I16
sS'm'
(lp8163
I1
aI3
aI7
aI3
aI13
aI15
aI107
aI153
aI501
aI573
aI863
aI3179
aI6019
aI15177
aI16075
aI43767
asS'd'
I4082
sa(dp8164
S'a'
I6786
sS's'
I16
sS'm'
(lp8165
I1
aI1
aI7
aI1
aI23
aI55
aI17
aI35
aI5
aI137
aI1707
aI1377
aI6857
aI15361
aI27299
aI61871
asS'd'
I4083
sa(dp8166
S'a'
I6798
sS's'
I16
sS'm'
(lp8167
I1
aI3
aI5
aI7
aI27
aI17
aI87
aI213
aI95
aI125
aI1239
aI3923
aI4193
aI11049
aI12783
aI45017
asS'd'
I4084
sa(dp8168
S'a'
I6821
sS's'
I16
sS'm'
(lp8169
I1
aI1
aI5
aI15
aI9
aI55
aI11
aI217
aI7
aI249
aI369
aI205
aI4251
aI13785
aI24781
aI48929
asS'd'
I4085
sa(dp8170
S'a'
I6840
sS's'
I16
sS'm'
(lp8171
I1
aI3
aI7
aI1
aI15
aI35
aI33
aI107
aI503
aI563
aI1591
aI3487
aI7495
aI1767
aI24791
aI31281
asS'd'
I4086
sa(dp8172
S'a'
I6846
sS's'
I16
sS'm'
(lp8173
I1
aI3
aI1
aI11
aI3
aI15
aI47
aI193
aI289
aI253
aI909
aI1105
aI5119
aI1415
aI7737
aI4341
asS'd'
I4087
sa(dp8174
S'a'
I6865
sS's'
I16
sS'm'
(lp8175
I1
aI1
aI1
aI3
aI23
aI33
aI53
aI187
aI469
aI573
aI835
aI2049
aI791
aI1177
aI31051
aI30955
asS'd'
I4088
sa(dp8176
S'a'
I6884
sS's'
I16
sS'm'
(lp8177
I1
aI3
aI3
aI11
aI15
aI51
aI77
aI143
aI369
aI991
aI1103
aI1293
aI6019
aI6361
aI26301
aI20741
asS'd'
I4089
sa(dp8178
S'a'
I6891
sS's'
I16
sS'm'
(lp8179
I1
aI1
aI1
aI5
aI17
aI19
aI85
aI135
aI113
aI593
aI579
aI1063
aI7173
aI2491
aI9355
aI19223
asS'd'
I4090
sa(dp8180
S'a'
I6925
sS's'
I16
sS'm'
(lp8181
I1
aI1
aI5
aI15
aI25
aI51
aI107
aI193
aI31
aI1
aI1693
aI125
aI6223
aI14619
aI22683
aI26321
asS'd'
I4091
sa(dp8182
S'a'
I6938
sS's'
I16
sS'm'
(lp8183
I1
aI1
aI7
aI1
aI17
aI45
aI87
aI39
aI87
aI499
aI1185
aI2763
aI3989
aI2863
aI24555
aI33817
asS'd'
I4092
sa(dp8184
S'a'
I6967
sS's'
I16
sS'm'
(lp8185
I1
aI3
aI1
aI11
aI31
aI5
aI121
aI231
aI185
aI793
aI255
aI2785
aI5261
aI3687
aI21711
aI3941
asS'd'
I4093
sa(dp8186
S'a'
I6988
sS's'
I16
sS'm'
(lp8187
I1
aI3
aI7
aI15
aI5
aI59
aI73
aI227
aI365
aI937
aI893
aI2155
aI4385
aI14345
aI6813
aI28461
asS'd'
I4094
sa(dp8188
S'a'
I6996
sS's'
I16
sS'm'
(lp8189
I1
aI1
aI5
aI7
aI7
aI23
aI7
aI239
aI431
aI45
aI1015
aI1663
aI1893
aI5035
aI24075
aI2119
asS'd'
I4095
sa(dp8190
S'a'
I7009
sS's'
I16
sS'm'
(lp8191
I1
aI3
aI5
aI1
aI3
aI15
aI63
aI103
aI119
aI801
aI1681
aI3463
aI6083
aI6453
aI11379
aI8205
asS'd'
I4096
sa(dp8192
S'a'
I7016
sS's'
I16
sS'm'
(lp8193
I1
aI3
aI7
aI9
aI21
aI61
aI9
aI9
aI433
aI541
aI603
aI3905
aI3787
aI10187
aI3643
aI21319
asS'd'
I4097
sa(dp8194
S'a'
I7030
sS's'
I16
sS'm'
(lp8195
I1
aI3
aI5
aI3
aI11
aI1
aI101
aI243
aI363
aI559
aI561
aI1163
aI455
aI4657
aI1147
aI39961
asS'd'
I4098
sa(dp8196
S'a'
I7043
sS's'
I16
sS'm'
(lp8197
I1
aI3
aI5
aI13
aI17
aI37
aI57
aI47
aI483
aI571
aI1579
aI1137
aI8125
aI12271
aI23279
aI1615
asS'd'
I4099
sa(dp8198
S'a'
I7045
sS's'
I16
sS'm'
(lp8199
I1
aI3
aI5
aI1
aI21
aI5
aI13
aI155
aI75
aI705
aI389
aI2855
aI6345
aI2279
aI12627
aI49451
asS'd'
I4100
sa(dp8200
S'a'
I7052
sS's'
I16
sS'm'
(lp8201
I1
aI1
aI3
aI9
aI15
aI51
aI73
aI99
aI445
aI227
aI1705
aI2175
aI8111
aI9381
aI31555
aI48491
asS'd'
I4101
sa(dp8202
S'a'
I7073
sS's'
I16
sS'm'
(lp8203
I1
aI3
aI3
aI5
aI9
aI63
aI107
aI51
aI461
aI979
aI1033
aI421
aI4807
aI11707
aI13261
aI26015
asS'd'
I4102
sa(dp8204
S'a'
I7142
sS's'
I16
sS'm'
(lp8205
I1
aI1
aI5
aI3
aI13
aI53
aI117
aI249
aI57
aI851
aI1391
aI3245
aI35
aI16043
aI24399
aI63667
asS'd'
I4103
sa(dp8206
S'a'
I7153
sS's'
I16
sS'm'
(lp8207
I1
aI3
aI1
aI11
aI23
aI33
aI57
aI125
aI385
aI495
aI871
aI199
aI4269
aI2269
aI22897
aI23597
asS'd'
I4104
sa(dp8208
S'a'
I7168
sS's'
I16
sS'm'
(lp8209
I1
aI3
aI5
aI15
aI29
aI11
aI77
aI21
aI479
aI369
aI723
aI3721
aI1121
aI9463
aI19775
aI54525
asS'd'
I4105
sa(dp8210
S'a'
I7174
sS's'
I16
sS'm'
(lp8211
I1
aI3
aI5
aI7
aI7
aI45
aI29
aI153
aI395
aI223
aI1917
aI3713
aI5087
aI10827
aI1383
aI36823
asS'd'
I4106
sa(dp8212
S'a'
I7183
sS's'
I16
sS'm'
(lp8213
I1
aI3
aI1
aI3
aI31
aI19
aI111
aI55
aI275
aI923
aI917
aI2925
aI673
aI6579
aI18783
aI5137
asS'd'
I4107
sa(dp8214
S'a'
I7195
sS's'
I16
sS'm'
(lp8215
I1
aI3
aI1
aI15
aI25
aI31
aI59
aI255
aI31
aI697
aI1751
aI381
aI299
aI295
aI14037
aI40953
asS'd'
I4108
sa(dp8216
S'a'
I7204
sS's'
I16
sS'm'
(lp8217
I1
aI3
aI1
aI7
aI15
aI23
aI69
aI219
aI351
aI183
aI1083
aI2227
aI6249
aI9385
aI13167
aI2901
asS'd'
I4109
sa(dp8218
S'a'
I7214
sS's'
I16
sS'm'
(lp8219
I1
aI3
aI7
aI1
aI5
aI61
aI117
aI13
aI67
aI521
aI41
aI2929
aI3937
aI1215
aI25593
aI32627
asS'd'
I4110
sa(dp8220
S'a'
I7222
sS's'
I16
sS'm'
(lp8221
I1
aI3
aI5
aI1
aI9
aI47
aI63
aI39
aI371
aI657
aI491
aI2243
aI4049
aI10517
aI12409
aI40597
asS'd'
I4111
sa(dp8222
S'a'
I7267
sS's'
I16
sS'm'
(lp8223
I1
aI3
aI7
aI15
aI17
aI3
aI77
aI13
aI275
aI225
aI487
aI2055
aI1293
aI15927
aI31773
aI18275
asS'd'
I4112
sa(dp8224
S'a'
I7269
sS's'
I16
sS'm'
(lp8225
I1
aI1
aI5
aI13
aI11
aI57
aI113
aI27
aI191
aI363
aI1341
aI3487
aI8031
aI13801
aI7563
aI40675
asS'd'
I4113
sa(dp8226
S'a'
I7279
sS's'
I16
sS'm'
(lp8227
I1
aI1
aI3
aI3
aI27
aI31
aI103
aI143
aI271
aI305
aI2033
aI3623
aI4219
aI9307
aI7501
aI8959
asS'd'
I4114
sa(dp8228
S'a'
I7293
sS's'
I16
sS'm'
(lp8229
I1
aI1
aI1
aI13
aI1
aI3
aI27
aI97
aI475
aI143
aI333
aI2997
aI1807
aI4231
aI27437
aI59717
asS'd'
I4115
sa(dp8230
S'a'
I7312
sS's'
I16
sS'm'
(lp8231
I1
aI3
aI7
aI5
aI5
aI3
aI69
aI233
aI309
aI511
aI1429
aI1887
aI2745
aI4969
aI17595
aI5451
asS'd'
I4116
sa(dp8232
S'a'
I7327
sS's'
I16
sS'm'
(lp8233
I1
aI1
aI7
aI3
aI23
aI17
aI115
aI89
aI265
aI467
aI257
aI2027
aI5331
aI1195
aI4451
aI8621
asS'd'
I4117
sa(dp8234
S'a'
I7334
sS's'
I16
sS'm'
(lp8235
I1
aI1
aI7
aI13
aI29
aI35
aI117
aI155
aI99
aI327
aI853
aI3595
aI2997
aI10745
aI21619
aI26549
asS'd'
I4118
sa(dp8236
S'a'
I7337
sS's'
I16
sS'm'
(lp8237
I1
aI3
aI3
aI13
aI1
aI13
aI75
aI151
aI67
aI271
aI1609
aI1117
aI4293
aI4645
aI12005
aI55983
asS'd'
I4119
sa(dp8238
S'a'
I7343
sS's'
I16
sS'm'
(lp8239
I1
aI1
aI1
aI13
aI1
aI61
aI101
aI63
aI161
aI261
aI1759
aI567
aI665
aI2339
aI9157
aI55603
asS'd'
I4120
sa(dp8240
S'a'
I7346
sS's'
I16
sS'm'
(lp8241
I1
aI1
aI7
aI11
aI25
aI19
aI71
aI27
aI255
aI435
aI227
aI4087
aI4309
aI14903
aI14513
aI30207
asS'd'
I4121
sa(dp8242
S'a'
I7372
sS's'
I16
sS'm'
(lp8243
I1
aI3
aI3
aI3
aI11
aI41
aI1
aI67
aI383
aI403
aI45
aI1521
aI1535
aI3353
aI27361
aI7549
asS'd'
I4122
sa(dp8244
S'a'
I7387
sS's'
I16
sS'm'
(lp8245
I1
aI1
aI1
aI1
aI13
aI51
aI31
aI137
aI147
aI907
aI19
aI3639
aI3739
aI877
aI15005
aI60357
asS'd'
I4123
sa(dp8246
S'a'
I7390
sS's'
I16
sS'm'
(lp8247
I1
aI1
aI3
aI11
aI11
aI23
aI29
aI173
aI105
aI873
aI1727
aI2761
aI2015
aI7491
aI17491
aI41065
asS'd'
I4124
sa(dp8248
S'a'
I7396
sS's'
I16
sS'm'
(lp8249
I1
aI1
aI5
aI3
aI31
aI1
aI119
aI53
aI11
aI731
aI393
aI4031
aI4421
aI15715
aI6431
aI18089
asS'd'
I4125
sa(dp8250
S'a'
I7423
sS's'
I16
sS'm'
(lp8251
I1
aI1
aI3
aI5
aI15
aI37
aI55
aI147
aI307
aI521
aI711
aI3085
aI5989
aI8081
aI23351
aI35259
asS'd'
I4126
sa(dp8252
S'a'
I7428
sS's'
I16
sS'm'
(lp8253
I1
aI3
aI5
aI13
aI21
aI19
aI47
aI107
aI447
aI713
aI1655
aI2809
aI4741
aI2105
aI9255
aI103
asS'd'
I4127
sa(dp8254
S'a'
I7437
sS's'
I16
sS'm'
(lp8255
I1
aI3
aI1
aI3
aI17
aI47
aI63
aI125
aI343
aI763
aI1777
aI607
aI5625
aI9517
aI7221
aI27257
asS'd'
I4128
sa(dp8256
S'a'
I7466
sS's'
I16
sS'm'
(lp8257
I1
aI1
aI7
aI5
aI31
aI13
aI67
aI255
aI41
aI947
aI99
aI721
aI7367
aI11427
aI1357
aI12383
asS'd'
I4129
sa(dp8258
S'a'
I7474
sS's'
I16
sS'm'
(lp8259
I1
aI1
aI7
aI3
aI23
aI27
aI73
aI185
aI189
aI545
aI1877
aI3169
aI5419
aI15873
aI29059
aI23983
asS'd'
I4130
sa(dp8260
S'a'
I7476
sS's'
I16
sS'm'
(lp8261
I1
aI1
aI3
aI1
aI5
aI13
aI81
aI45
aI79
aI717
aI819
aI3539
aI7561
aI7319
aI5113
aI27273
asS'd'
I4131
sa(dp8262
S'a'
I7483
sS's'
I16
sS'm'
(lp8263
I1
aI3
aI7
aI9
aI21
aI25
aI71
aI247
aI41
aI583
aI771
aI3745
aI1883
aI5717
aI755
aI14549
asS'd'
I4132
sa(dp8264
S'a'
I7518
sS's'
I16
sS'm'
(lp8265
I1
aI1
aI3
aI7
aI23
aI25
aI87
aI143
aI499
aI515
aI1753
aI1229
aI173
aI10629
aI30579
aI29643
asS'd'
I4133
sa(dp8266
S'a'
I7527
sS's'
I16
sS'm'
(lp8267
I1
aI3
aI1
aI13
aI29
aI33
aI31
aI69
aI503
aI117
aI1717
aI101
aI7647
aI1567
aI28677
aI3079
asS'd'
I4134
sa(dp8268
S'a'
I7545
sS's'
I16
sS'm'
(lp8269
I1
aI3
aI1
aI15
aI29
aI45
aI57
aI81
aI171
aI813
aI505
aI3647
aI3913
aI5557
aI2477
aI42429
asS'd'
I4135
sa(dp8270
S'a'
I7572
sS's'
I16
sS'm'
(lp8271
I1
aI1
aI5
aI13
aI21
aI13
aI81
aI5
aI471
aI221
aI1563
aI1741
aI7269
aI16327
aI22687
aI5291
asS'd'
I4136
sa(dp8272
S'a'
I7586
sS's'
I16
sS'm'
(lp8273
I1
aI3
aI5
aI3
aI23
aI41
aI107
aI61
aI95
aI79
aI467
aI1533
aI739
aI6791
aI26911
aI20309
asS'd'
I4137
sa(dp8274
S'a'
I7597
sS's'
I16
sS'm'
(lp8275
I1
aI3
aI7
aI7
aI3
aI53
aI71
aI163
aI459
aI975
aI687
aI1115
aI5241
aI299
aI26361
aI38583
asS'd'
I4138
sa(dp8276
S'a'
I7630
sS's'
I16
sS'm'
(lp8277
I1
aI3
aI1
aI9
aI3
aI63
aI7
aI133
aI421
aI325
aI823
aI1175
aI6201
aI5707
aI31539
aI34645
asS'd'
I4139
sa(dp8278
S'a'
I7653
sS's'
I16
sS'm'
(lp8279
I1
aI3
aI7
aI5
aI27
aI27
aI107
aI239
aI247
aI257
aI1367
aI3685
aI7839
aI11693
aI3237
aI13085
asS'd'
I4140
sa(dp8280
S'a'
I7657
sS's'
I16
sS'm'
(lp8281
I1
aI1
aI1
aI3
aI27
aI41
aI51
aI69
aI271
aI809
aI1453
aI519
aI1301
aI2367
aI637
aI5267
asS'd'
I4141
sa(dp8282
S'a'
I7671
sS's'
I16
sS'm'
(lp8283
I1
aI3
aI7
aI13
aI19
aI17
aI3
aI69
aI369
aI447
aI1685
aI4075
aI6143
aI11387
aI5411
aI29825
asS'd'
I4142
sa(dp8284
S'a'
I7672
sS's'
I16
sS'm'
(lp8285
I1
aI1
aI3
aI1
aI25
aI61
aI79
aI163
aI1
aI905
aI1969
aI2735
aI7709
aI16121
aI20441
aI4543
asS'd'
I4143
sa(dp8286
S'a'
I7715
sS's'
I16
sS'm'
(lp8287
I1
aI3
aI7
aI5
aI15
aI29
aI7
aI245
aI343
aI803
aI1719
aI3993
aI983
aI2925
aI10393
aI6053
asS'd'
I4144
sa(dp8288
S'a'
I7717
sS's'
I16
sS'm'
(lp8289
I1
aI3
aI1
aI7
aI17
aI55
aI63
aI29
aI441
aI387
aI885
aI3269
aI1977
aI1927
aI3657
aI47043
asS'd'
I4145
sa(dp8290
S'a'
I7732
sS's'
I16
sS'm'
(lp8291
I1
aI1
aI3
aI1
aI17
aI59
aI51
aI9
aI173
aI327
aI1185
aI3241
aI3785
aI10907
aI19429
aI22209
asS'd'
I4146
sa(dp8292
S'a'
I7735
sS's'
I16
sS'm'
(lp8293
I1
aI1
aI3
aI13
aI21
aI57
aI125
aI5
aI359
aI437
aI1231
aI2441
aI5813
aI9991
aI283
aI52555
asS'd'
I4147
sa(dp8294
S'a'
I7753
sS's'
I16
sS'm'
(lp8295
I1
aI3
aI1
aI7
aI15
aI19
aI39
aI125
aI405
aI381
aI1757
aI4075
aI5565
aI2065
aI295
aI8867
asS'd'
I4148
sa(dp8296
S'a'
I7811
sS's'
I16
sS'm'
(lp8297
I1
aI3
aI3
aI11
aI7
aI33
aI95
aI19
aI253
aI141
aI1093
aI1787
aI7495
aI5229
aI15923
aI44157
asS'd'
I4149
sa(dp8298
S'a'
I7817
sS's'
I16
sS'm'
(lp8299
I1
aI3
aI7
aI15
aI1
aI49
aI69
aI163
aI85
aI345
aI901
aI2329
aI8003
aI9915
aI2209
aI33979
asS'd'
I4150
sa(dp8300
S'a'
I7825
sS's'
I16
sS'm'
(lp8301
I1
aI1
aI1
aI9
aI23
aI51
aI125
aI163
aI257
aI681
aI565
aI945
aI6375
aI8679
aI5985
aI28919
asS'd'
I4151
sa(dp8302
S'a'
I7832
sS's'
I16
sS'm'
(lp8303
I1
aI3
aI5
aI7
aI11
aI23
aI123
aI231
aI377
aI121
aI1519
aI2061
aI2957
aI14537
aI17625
aI37773
asS'd'
I4152
sa(dp8304
S'a'
I7838
sS's'
I16
sS'm'
(lp8305
I1
aI3
aI5
aI1
aI17
aI43
aI89
aI119
aI455
aI279
aI1857
aI3405
aI5225
aI13035
aI6055
aI30861
asS'd'
I4153
sa(dp8306
S'a'
I7841
sS's'
I16
sS'm'
(lp8307
I1
aI3
aI7
aI15
aI31
aI63
aI25
aI225
aI3
aI527
aI355
aI1435
aI5763
aI15203
aI26579
aI45659
asS'd'
I4154
sa(dp8308
S'a'
I7844
sS's'
I16
sS'm'
(lp8309
I1
aI1
aI1
aI3
aI27
aI43
aI71
aI193
aI135
aI5
aI683
aI925
aI7023
aI7711
aI2807
aI56003
asS'd'
I4155
sa(dp8310
S'a'
I7859
sS's'
I16
sS'm'
(lp8311
I1
aI1
aI1
aI11
aI3
aI3
aI109
aI29
aI109
aI683
aI419
aI3295
aI1961
aI5953
aI8887
aI1523
asS'd'
I4156
sa(dp8312
S'a'
I7861
sS's'
I16
sS'm'
(lp8313
I1
aI3
aI3
aI11
aI17
aI39
aI19
aI225
aI103
aI249
aI81
aI3875
aI4515
aI3969
aI24745
aI60031
asS'd'
I4157
sa(dp8314
S'a'
I7873
sS's'
I16
sS'm'
(lp8315
I1
aI1
aI3
aI3
aI3
aI23
aI15
aI149
aI305
aI399
aI1347
aI1001
aI597
aI10003
aI22123
aI29919
asS'd'
I4158
sa(dp8316
S'a'
I7880
sS's'
I16
sS'm'
(lp8317
I1
aI3
aI5
aI1
aI23
aI35
aI123
aI7
aI283
aI283
aI759
aI3061
aI2011
aI7771
aI32201
aI40667
asS'd'
I4159
sa(dp8318
S'a'
I7897
sS's'
I16
sS'm'
(lp8319
I1
aI3
aI7
aI15
aI23
aI5
aI81
aI51
aI357
aI79
aI133
aI285
aI425
aI7743
aI13499
aI18983
asS'd'
I4160
sa(dp8320
S'a'
I7904
sS's'
I16
sS'm'
(lp8321
I1
aI3
aI3
aI5
aI17
aI37
aI75
aI221
aI473
aI111
aI335
aI683
aI7353
aI2283
aI13457
aI61171
asS'd'
I4161
sa(dp8322
S'a'
I7910
sS's'
I16
sS'm'
(lp8323
I1
aI3
aI1
aI7
aI13
aI45
aI13
aI223
aI149
aI209
aI727
aI3553
aI2573
aI177
aI855
aI44139
asS'd'
I4162
sa(dp8324
S'a'
I7960
sS's'
I16
sS'm'
(lp8325
I1
aI1
aI3
aI15
aI23
aI5
aI103
aI139
aI17
aI29
aI1961
aI3021
aI5745
aI12963
aI30669
aI44171
asS'd'
I4163
sa(dp8326
S'a'
I7969
sS's'
I16
sS'm'
(lp8327
I1
aI3
aI1
aI1
aI3
aI33
aI67
aI203
aI29
aI785
aI71
aI1693
aI4487
aI10221
aI24523
aI51223
asS'd'
I4164
sa(dp8328
S'a'
I7970
sS's'
I16
sS'm'
(lp8329
I1
aI1
aI5
aI7
aI7
aI27
aI17
aI183
aI229
aI669
aI1675
aI3751
aI3233
aI10677
aI7509
aI52313
asS'd'
I4165
sa(dp8330
S'a'
I7976
sS's'
I16
sS'm'
(lp8331
I1
aI1
aI5
aI5
aI25
aI35
aI21
aI163
aI483
aI399
aI195
aI3465
aI6349
aI545
aI18861
aI31759
asS'd'
I4166
sa(dp8332
S'a'
I7979
sS's'
I16
sS'm'
(lp8333
I1
aI3
aI1
aI5
aI15
aI39
aI13
aI157
aI71
aI841
aI447
aI3625
aI53
aI12745
aI2719
aI27617
asS'd'
I4167
sa(dp8334
S'a'
I8007
sS's'
I16
sS'm'
(lp8335
I1
aI1
aI5
aI5
aI3
aI45
aI113
aI121
aI263
aI887
aI709
aI2189
aI3811
aI1409
aI10451
aI48777
asS'd'
I4168
sa(dp8336
S'a'
I8041
sS's'
I16
sS'm'
(lp8337
I1
aI1
aI5
aI15
aI9
aI41
aI31
aI95
aI377
aI215
aI437
aI3633
aI433
aI11935
aI15283
aI55451
asS'd'
I4169
sa(dp8338
S'a'
I8047
sS's'
I16
sS'm'
(lp8339
I1
aI1
aI7
aI7
aI13
aI23
aI79
aI3
aI451
aI409
aI1103
aI1771
aI553
aI3175
aI28703
aI49357
asS'd'
I4170
sa(dp8340
S'a'
I8052
sS's'
I16
sS'm'
(lp8341
I1
aI3
aI1
aI1
aI13
aI33
aI95
aI133
aI419
aI851
aI955
aI3985
aI5869
aI14219
aI253
aI46883
asS'd'
I4171
sa(dp8342
S'a'
I8061
sS's'
I16
sS'm'
(lp8343
I1
aI3
aI3
aI3
aI23
aI55
aI91
aI207
aI281
aI355
aI361
aI261
aI6837
aI4401
aI25455
aI25313
asS'd'
I4172
sa(dp8344
S'a'
I8078
sS's'
I16
sS'm'
(lp8345
I1
aI3
aI5
aI9
aI27
aI9
aI107
aI51
aI69
aI555
aI835
aI3541
aI1827
aI5737
aI31225
aI55619
asS'd'
I4173
sa(dp8346
S'a'
I8128
sS's'
I16
sS'm'
(lp8347
I1
aI1
aI1
aI11
aI27
aI51
aI79
aI85
aI447
aI447
aI9
aI2803
aI377
aI4347
aI2183
aI61257
asS'd'
I4174
sa(dp8348
S'a'
I8146
sS's'
I16
sS'm'
(lp8349
I1
aI1
aI1
aI3
aI23
aI21
aI51
aI217
aI297
aI135
aI573
aI689
aI4947
aI14143
aI26247
aI43061
asS'd'
I4175
sa(dp8350
S'a'
I8162
sS's'
I16
sS'm'
(lp8351
I1
aI3
aI1
aI7
aI15
aI13
aI27
aI151
aI463
aI707
aI43
aI3641
aI4999
aI3783
aI9083
aI22085
asS'd'
I4176
sa(dp8352
S'a'
I8250
sS's'
I16
sS'm'
(lp8353
I1
aI3
aI3
aI5
aI3
aI1
aI15
aI119
aI343
aI605
aI105
aI2939
aI2259
aI889
aI21759
aI34073
asS'd'
I4177
sa(dp8354
S'a'
I8270
sS's'
I16
sS'm'
(lp8355
I1
aI1
aI1
aI7
aI3
aI63
aI103
aI1
aI235
aI317
aI263
aI2701
aI7331
aI15921
aI17295
aI613
asS'd'
I4178
sa(dp8356
S'a'
I8294
sS's'
I16
sS'm'
(lp8357
I1
aI1
aI7
aI3
aI19
aI3
aI5
aI17
aI105
aI491
aI755
aI233
aI5397
aI12365
aI16325
aI59377
asS'd'
I4179
sa(dp8358
S'a'
I8324
sS's'
I16
sS'm'
(lp8359
I1
aI3
aI3
aI15
aI15
aI27
aI37
aI151
aI481
aI949
aI1473
aI233
aI1925
aI15471
aI24957
aI3241
asS'd'
I4180
sa(dp8360
S'a'
I8351
sS's'
I16
sS'm'
(lp8361
I1
aI1
aI7
aI5
aI9
aI61
aI49
aI91
aI169
aI179
aI701
aI3957
aI473
aI15087
aI6109
aI25083
asS'd'
I4181
sa(dp8362
S'a'
I8357
sS's'
I16
sS'm'
(lp8363
I1
aI3
aI3
aI11
aI27
aI43
aI5
aI33
aI69
aI705
aI733
aI2675
aI2677
aI4235
aI11109
aI15557
asS'd'
I4182
sa(dp8364
S'a'
I8361
sS's'
I16
sS'm'
(lp8365
I1
aI3
aI1
aI3
aI17
aI19
aI101
aI119
aI289
aI341
aI1395
aI563
aI6859
aI10359
aI10077
aI26905
asS'd'
I4183
sa(dp8366
S'a'
I8364
sS's'
I16
sS'm'
(lp8367
I1
aI1
aI1
aI15
aI21
aI47
aI41
aI145
aI439
aI583
aI1755
aI1977
aI5235
aI15961
aI21315
aI60577
asS'd'
I4184
sa(dp8368
S'a'
I8393
sS's'
I16
sS'm'
(lp8369
I1
aI1
aI5
aI3
aI9
aI59
aI71
aI143
aI27
aI1007
aI313
aI1567
aI1685
aI11063
aI28889
aI44253
asS'd'
I4185
sa(dp8370
S'a'
I8396
sS's'
I16
sS'm'
(lp8371
I1
aI1
aI5
aI5
aI29
aI29
aI43
aI127
aI13
aI585
aI1245
aI187
aI2697
aI8791
aI19561
aI6463
asS'd'
I4186
sa(dp8372
S'a'
I8407
sS's'
I16
sS'm'
(lp8373
I1
aI1
aI3
aI11
aI29
aI39
aI127
aI75
aI23
aI521
aI421
aI3115
aI139
aI5429
aI23341
aI58035
asS'd'
I4187
sa(dp8374
S'a'
I8413
sS's'
I16
sS'm'
(lp8375
I1
aI1
aI3
aI1
aI27
aI35
aI27
aI9
aI185
aI653
aI887
aI2715
aI3775
aI1753
aI22105
aI62095
asS'd'
I4188
sa(dp8376
S'a'
I8414
sS's'
I16
sS'm'
(lp8377
I1
aI1
aI5
aI5
aI5
aI63
aI23
aI31
aI317
aI1001
aI1751
aI1185
aI7933
aI525
aI30501
aI15565
asS'd'
I4189
sa(dp8378
S'a'
I8432
sS's'
I16
sS'm'
(lp8379
I1
aI1
aI1
aI5
aI9
aI27
aI79
aI91
aI453
aI995
aI1041
aI3213
aI8027
aI5855
aI7435
aI64079
asS'd'
I4190
sa(dp8380
S'a'
I8435
sS's'
I16
sS'm'
(lp8381
I1
aI1
aI3
aI11
aI1
aI51
aI27
aI195
aI139
aI41
aI1891
aI1437
aI1033
aI11671
aI3235
aI35083
asS'd'
I4191
sa(dp8382
S'a'
I8441
sS's'
I16
sS'm'
(lp8383
I1
aI3
aI1
aI3
aI11
aI25
aI33
aI249
aI373
aI497
aI1631
aI293
aI3657
aI10741
aI15831
aI59545
asS'd'
I4192
sa(dp8384
S'a'
I8447
sS's'
I16
sS'm'
(lp8385
I1
aI1
aI1
aI1
aI15
aI63
aI13
aI165
aI113
aI559
aI1615
aI3579
aI1993
aI1907
aI22335
aI47791
asS'd'
I4193
sa(dp8386
S'a'
I8449
sS's'
I16
sS'm'
(lp8387
I1
aI1
aI3
aI15
aI13
aI49
aI63
aI235
aI31
aI811
aI1729
aI221
aI5143
aI11547
aI30029
aI52003
asS'd'
I4194
sa(dp8388
S'a'
I8456
sS's'
I16
sS'm'
(lp8389
I1
aI1
aI5
aI13
aI29
aI61
aI25
aI221
aI421
aI221
aI583
aI373
aI2341
aI7493
aI13981
aI54141
asS'd'
I4195
sa(dp8390
S'a'
I8485
sS's'
I16
sS'm'
(lp8391
I1
aI1
aI5
aI11
aI21
aI49
aI71
aI249
aI237
aI369
aI1273
aI1067
aI4411
aI409
aI7219
aI52933
asS'd'
I4196
sa(dp8392
S'a'
I8504
sS's'
I16
sS'm'
(lp8393
I1
aI3
aI1
aI1
aI13
aI23
aI53
aI15
aI137
aI553
aI401
aI2247
aI5591
aI14021
aI445
aI20433
asS'd'
I4197
sa(dp8394
S'a'
I8512
sS's'
I16
sS'm'
(lp8395
I1
aI1
aI7
aI7
aI7
aI23
aI19
aI189
aI119
aI643
aI847
aI2123
aI5343
aI11839
aI4575
aI60461
asS'd'
I4198
sa(dp8396
S'a'
I8532
sS's'
I16
sS'm'
(lp8397
I1
aI1
aI5
aI5
aI11
aI19
aI111
aI219
aI185
aI499
aI595
aI723
aI3519
aI10891
aI27603
aI29261
asS'd'
I4199
sa(dp8398
S'a'
I8551
sS's'
I16
sS'm'
(lp8399
I1
aI3
aI3
aI1
aI9
aI13
aI95
aI227
aI459
aI133
aI1535
aI3481
aI7187
aI14797
aI16511
aI6737
asS'd'
I4200
sa(dp8400
S'a'
I8560
sS's'
I16
sS'm'
(lp8401
I1
aI1
aI7
aI7
aI19
aI57
aI117
aI7
aI455
aI941
aI455
aI797
aI6313
aI10071
aI18651
aI25013
asS'd'
I4201
sa(dp8402
S'a'
I8566
sS's'
I16
sS'm'
(lp8403
I1
aI3
aI7
aI3
aI25
aI25
aI79
aI19
aI383
aI971
aI1625
aI2803
aI2461
aI633
aI32317
aI48407
asS'd'
I4202
sa(dp8404
S'a'
I8581
sS's'
I16
sS'm'
(lp8405
I1
aI1
aI7
aI7
aI25
aI43
aI93
aI135
aI155
aI685
aI2001
aI3007
aI7315
aI15555
aI30401
aI36291
asS'd'
I4203
sa(dp8406
S'a'
I8609
sS's'
I16
sS'm'
(lp8407
I1
aI1
aI1
aI5
aI13
aI33
aI123
aI221
aI341
aI105
aI1075
aI3125
aI5323
aI14293
aI29875
aI52215
asS'd'
I4204
sa(dp8408
S'a'
I8639
sS's'
I16
sS'm'
(lp8409
I1
aI1
aI3
aI9
aI29
aI51
aI25
aI59
aI171
aI563
aI1695
aI2845
aI6067
aI10671
aI2909
aI33977
asS'd'
I4205
sa(dp8410
S'a'
I8641
sS's'
I16
sS'm'
(lp8411
I1
aI3
aI3
aI7
aI25
aI21
aI39
aI65
aI485
aI949
aI1773
aI2775
aI6019
aI14587
aI6715
aI54793
asS'd'
I4206
sa(dp8412
S'a'
I8671
sS's'
I16
sS'm'
(lp8413
I1
aI1
aI7
aI11
aI17
aI57
aI125
aI17
aI111
aI167
aI289
aI3939
aI7357
aI2289
aI1717
aI45225
asS'd'
I4207
sa(dp8414
S'a'
I8695
sS's'
I16
sS'm'
(lp8415
I1
aI1
aI7
aI7
aI21
aI55
aI3
aI139
aI471
aI747
aI1437
aI1353
aI7657
aI13133
aI14193
aI38191
asS'd'
I4208
sa(dp8416
S'a'
I8701
sS's'
I16
sS'm'
(lp8417
I1
aI3
aI5
aI7
aI25
aI57
aI55
aI17
aI315
aI159
aI437
aI933
aI7493
aI6025
aI2775
aI24287
asS'd'
I4209
sa(dp8418
S'a'
I8711
sS's'
I16
sS'm'
(lp8419
I1
aI1
aI7
aI1
aI15
aI45
aI67
aI191
aI355
aI681
aI1763
aI1237
aI105
aI1425
aI26089
aI42879
asS'd'
I4210
sa(dp8420
S'a'
I8739
sS's'
I16
sS'm'
(lp8421
I1
aI1
aI5
aI13
aI13
aI53
aI25
aI127
aI103
aI155
aI935
aI2561
aI475
aI4341
aI30541
aI43835
asS'd'
I4211
sa(dp8422
S'a'
I8763
sS's'
I16
sS'm'
(lp8423
I1
aI1
aI5
aI15
aI27
aI59
aI99
aI173
aI189
aI41
aI105
aI3287
aI4071
aI15005
aI18301
aI34487
asS'd'
I4212
sa(dp8424
S'a'
I8778
sS's'
I16
sS'm'
(lp8425
I1
aI1
aI5
aI11
aI21
aI9
aI57
aI115
aI495
aI561
aI579
aI397
aI3049
aI2007
aI5041
aI37345
asS'd'
I4213
sa(dp8426
S'a'
I8783
sS's'
I16
sS'm'
(lp8427
I1
aI1
aI5
aI11
aI15
aI11
aI101
aI241
aI69
aI483
aI1007
aI4069
aI5221
aI5323
aI20177
aI24397
asS'd'
I4214
sa(dp8428
S'a'
I8785
sS's'
I16
sS'm'
(lp8429
I1
aI1
aI1
aI7
aI29
aI15
aI119
aI161
aI21
aI517
aI847
aI2217
aI4487
aI4817
aI24053
aI21683
asS'd'
I4215
sa(dp8430
S'a'
I8797
sS's'
I16
sS'm'
(lp8431
I1
aI3
aI1
aI3
aI3
aI51
aI85
aI63
aI345
aI799
aI1699
aI3961
aI7109
aI3931
aI28173
aI46851
asS'd'
I4216
sa(dp8432
S'a'
I8802
sS's'
I16
sS'm'
(lp8433
I1
aI1
aI5
aI7
aI15
aI25
aI97
aI139
aI331
aI969
aI1129
aI2451
aI3107
aI12235
aI12949
aI29837
asS'd'
I4217
sa(dp8434
S'a'
I8816
sS's'
I16
sS'm'
(lp8435
I1
aI3
aI7
aI1
aI19
aI21
aI51
aI155
aI295
aI565
aI29
aI2107
aI341
aI14611
aI15281
aI50727
asS'd'
I4218
sa(dp8436
S'a'
I8828
sS's'
I16
sS'm'
(lp8437
I1
aI3
aI1
aI11
aI3
aI37
aI13
aI217
aI429
aI217
aI301
aI1217
aI5655
aI13845
aI32465
aI23559
asS'd'
I4219
sa(dp8438
S'a'
I8837
sS's'
I16
sS'm'
(lp8439
I1
aI3
aI3
aI9
aI9
aI57
aI79
aI231
aI433
aI703
aI699
aI3813
aI7035
aI5885
aI19185
aI52551
asS'd'
I4220
sa(dp8440
S'a'
I8852
sS's'
I16
sS'm'
(lp8441
I1
aI1
aI1
aI5
aI19
aI17
aI31
aI209
aI49
aI515
aI279
aI909
aI5881
aI2673
aI23635
aI23101
asS'd'
I4221
sa(dp8442
S'a'
I8859
sS's'
I16
sS'm'
(lp8443
I1
aI1
aI5
aI7
aI3
aI3
aI119
aI139
aI245
aI775
aI1009
aI1157
aI1405
aI9737
aI17671
aI62981
asS'd'
I4222
sa(dp8444
S'a'
I8889
sS's'
I16
sS'm'
(lp8445
I1
aI3
aI7
aI11
aI17
aI21
aI105
aI21
aI67
aI871
aI233
aI3607
aI571
aI9141
aI9751
aI28093
asS'd'
I4223
sa(dp8446
S'a'
I8900
sS's'
I16
sS'm'
(lp8447
I1
aI1
aI3
aI13
aI5
aI53
aI91
aI181
aI143
aI375
aI1113
aI705
aI837
aI10505
aI11459
aI57753
asS'd'
I4224
sa(dp8448
S'a'
I8903
sS's'
I16
sS'm'
(lp8449
I1
aI3
aI5
aI11
aI9
aI19
aI107
aI229
aI305
aI107
aI1027
aI691
aI4677
aI8987
aI7931
aI951
asS'd'
I4225
sa(dp8450
S'a'
I8909
sS's'
I16
sS'm'
(lp8451
I1
aI1
aI7
aI9
aI9
aI17
aI39
aI195
aI103
aI315
aI517
aI123
aI7167
aI7039
aI3571
aI40469
asS'd'
I4226
sa(dp8452
S'a'
I8910
sS's'
I16
sS'm'
(lp8453
I1
aI1
aI1
aI5
aI1
aI21
aI121
aI53
aI427
aI111
aI717
aI1065
aI2843
aI5873
aI28411
aI42443
asS'd'
I4227
sa(dp8454
S'a'
I8922
sS's'
I16
sS'm'
(lp8455
I1
aI1
aI3
aI11
aI27
aI7
aI37
aI255
aI429
aI363
aI997
aI2429
aI6871
aI1271
aI29375
aI62897
asS'd'
I4228
sa(dp8456
S'a'
I8924
sS's'
I16
sS'm'
(lp8457
I1
aI3
aI3
aI13
aI23
aI23
aI123
aI119
aI213
aI51
aI771
aI1603
aI1621
aI1497
aI23667
aI13443
asS'd'
I4229
sa(dp8458
S'a'
I8955
sS's'
I16
sS'm'
(lp8459
I1
aI1
aI3
aI13
aI17
aI21
aI81
aI17
aI211
aI815
aI1751
aI3875
aI4001
aI3927
aI6185
aI28753
asS'd'
I4230
sa(dp8460
S'a'
I8958
sS's'
I16
sS'm'
(lp8461
I1
aI3
aI1
aI5
aI13
aI41
aI23
aI187
aI475
aI353
aI1307
aI543
aI5077
aI3459
aI20553
aI29119
asS'd'
I4231
sa(dp8462
S'a'
I8980
sS's'
I16
sS'm'
(lp8463
I1
aI1
aI1
aI7
aI1
aI39
aI3
aI247
aI375
aI101
aI81
aI1515
aI1079
aI15307
aI18865
aI63115
asS'd'
I4232
sa(dp8464
S'a'
I8994
sS's'
I16
sS'm'
(lp8465
I1
aI1
aI5
aI9
aI23
aI7
aI61
aI45
aI379
aI553
aI435
aI1805
aI4147
aI2289
aI22081
aI40041
asS'd'
I4233
sa(dp8466
S'a'
I9006
sS's'
I16
sS'm'
(lp8467
I1
aI1
aI7
aI3
aI17
aI13
aI107
aI169
aI119
aI981
aI1825
aI3329
aI7779
aI12245
aI28367
aI6749
asS'd'
I4234
sa(dp8468
S'a'
I9017
sS's'
I16
sS'm'
(lp8469
I1
aI3
aI3
aI13
aI29
aI13
aI93
aI155
aI331
aI507
aI1073
aI279
aI6615
aI14077
aI3005
aI10171
asS'd'
I4235
sa(dp8470
S'a'
I9032
sS's'
I16
sS'm'
(lp8471
I1
aI1
aI5
aI7
aI29
aI23
aI81
aI181
aI321
aI921
aI1531
aI2607
aI7291
aI1255
aI29889
aI30095
asS'd'
I4236
sa(dp8472
S'a'
I9040
sS's'
I16
sS'm'
(lp8473
I1
aI1
aI1
aI5
aI7
aI1
aI9
aI231
aI203
aI559
aI243
aI3999
aI3649
aI7939
aI14729
aI34771
asS'd'
I4237
sa(dp8474
S'a'
I9061
sS's'
I16
sS'm'
(lp8475
I1
aI3
aI7
aI3
aI17
aI29
aI79
aI251
aI305
aI641
aI1125
aI1155
aI7139
aI6721
aI43
aI34927
asS'd'
I4238
sa(dp8476
S'a'
I9066
sS's'
I16
sS'm'
(lp8477
I1
aI1
aI5
aI13
aI21
aI39
aI55
aI103
aI143
aI487
aI849
aI1111
aI1105
aI8483
aI5417
aI10521
asS'd'
I4239
sa(dp8478
S'a'
I9071
sS's'
I16
sS'm'
(lp8479
I1
aI1
aI5
aI5
aI19
aI5
aI111
aI49
aI95
aI917
aI843
aI2539
aI6831
aI9019
aI16045
aI59363
asS'd'
I4240
sa(dp8480
S'a'
I9076
sS's'
I16
sS'm'
(lp8481
I1
aI3
aI3
aI11
aI7
aI45
aI87
aI51
aI49
aI615
aI603
aI1623
aI5351
aI11939
aI21945
aI40539
asS'd'
I4241
sa(dp8482
S'a'
I9086
sS's'
I16
sS'm'
(lp8483
I1
aI1
aI5
aI9
aI9
aI33
aI113
aI37
aI163
aI853
aI1313
aI4021
aI965
aI11465
aI8573
aI24425
asS'd'
I4242
sa(dp8484
S'a'
I9104
sS's'
I16
sS'm'
(lp8485
I1
aI3
aI3
aI9
aI17
aI19
aI121
aI95
aI93
aI441
aI1951
aI3335
aI6279
aI16087
aI14763
aI60771
asS'd'
I4243
sa(dp8486
S'a'
I9150
sS's'
I16
sS'm'
(lp8487
I1
aI3
aI3
aI9
aI13
aI15
aI19
aI129
aI257
aI641
aI533
aI1667
aI5959
aI2259
aI10439
aI29745
asS'd'
I4244
sa(dp8488
S'a'
I9161
sS's'
I16
sS'm'
(lp8489
I1
aI1
aI7
aI7
aI11
aI31
aI45
aI247
aI233
aI101
aI899
aI2033
aI7803
aI11423
aI30645
aI7723
asS'd'
I4245
sa(dp8490
S'a'
I9164
sS's'
I16
sS'm'
(lp8491
I1
aI3
aI5
aI11
aI23
aI3
aI69
aI79
aI319
aI125
aI1463
aI2047
aI7311
aI5819
aI445
aI13725
asS'd'
I4246
sa(dp8492
S'a'
I9185
sS's'
I16
sS'm'
(lp8493
I1
aI1
aI1
aI3
aI7
aI43
aI83
aI89
aI467
aI709
aI1993
aI3773
aI7805
aI305
aI15701
aI51101
asS'd'
I4247
sa(dp8494
S'a'
I9188
sS's'
I16
sS'm'
(lp8495
I1
aI1
aI7
aI5
aI19
aI53
aI77
aI237
aI27
aI853
aI1003
aI2041
aI5739
aI4663
aI9783
aI23761
asS'd'
I4248
sa(dp8496
S'a'
I9212
sS's'
I16
sS'm'
(lp8497
I1
aI1
aI3
aI7
aI19
aI31
aI71
aI33
aI479
aI693
aI1503
aI9
aI2779
aI1481
aI9413
aI36227
asS'd'
I4249
sa(dp8498
S'a'
I9230
sS's'
I16
sS'm'
(lp8499
I1
aI3
aI1
aI11
aI9
aI23
aI1
aI99
aI247
aI33
aI1987
aI1577
aI8029
aI7785
aI29947
aI38751
asS'd'
I4250
sa(dp8500
S'a'
I9242
sS's'
I16
sS'm'
(lp8501
I1
aI1
aI1
aI3
aI15
aI57
aI23
aI53
aI431
aI553
aI1433
aI2447
aI1871
aI10701
aI30961
aI12281
asS'd'
I4251
sa(dp8502
S'a'
I9247
sS's'
I16
sS'm'
(lp8503
I1
aI3
aI5
aI9
aI11
aI49
aI123
aI91
aI87
aI155
aI301
aI3339
aI6183
aI15895
aI17309
aI45927
asS'd'
I4252
sa(dp8504
S'a'
I9260
sS's'
I16
sS'm'
(lp8505
I1
aI1
aI1
aI9
aI9
aI41
aI79
aI123
aI261
aI547
aI1931
aI2553
aI7405
aI14431
aI24079
aI20769
asS'd'
I4253
sa(dp8506
S'a'
I9280
sS's'
I16
sS'm'
(lp8507
I1
aI3
aI1
aI3
aI31
aI17
aI17
aI137
aI419
aI591
aI1693
aI3977
aI861
aI16025
aI189
aI60995
asS'd'
I4254
sa(dp8508
S'a'
I9300
sS's'
I16
sS'm'
(lp8509
I1
aI3
aI7
aI11
aI19
aI47
aI107
aI243
aI87
aI135
aI507
aI189
aI1397
aI3841
aI22999
aI50781
asS'd'
I4255
sa(dp8510
S'a'
I9313
sS's'
I16
sS'm'
(lp8511
I1
aI3
aI5
aI5
aI15
aI11
aI19
aI239
aI133
aI295
aI673
aI2389
aI4753
aI4363
aI21669
aI25579
asS'd'
I4256
sa(dp8512
S'a'
I9325
sS's'
I16
sS'm'
(lp8513
I1
aI3
aI5
aI7
aI19
aI43
aI55
aI129
aI239
aI89
aI1731
aI1381
aI5483
aI11773
aI9201
aI17745
asS'd'
I4257
sa(dp8514
S'a'
I9343
sS's'
I16
sS'm'
(lp8515
I1
aI3
aI1
aI13
aI3
aI15
aI77
aI131
aI417
aI845
aI1953
aI2871
aI1789
aI10343
aI11363
aI20699
asS'd'
I4258
sa(dp8516
S'a'
I9349
sS's'
I16
sS'm'
(lp8517
I1
aI3
aI7
aI1
aI9
aI43
aI55
aI223
aI239
aI317
aI1951
aI2725
aI209
aI3853
aI2201
aI6839
asS'd'
I4259
sa(dp8518
S'a'
I9354
sS's'
I16
sS'm'
(lp8519
I1
aI3
aI1
aI3
aI7
aI35
aI29
aI21
aI149
aI779
aI467
aI65
aI811
aI4859
aI14021
aI38429
asS'd'
I4260
sa(dp8520
S'a'
I9367
sS's'
I16
sS'm'
(lp8521
I1
aI3
aI7
aI1
aI19
aI31
aI97
aI9
aI11
aI415
aI689
aI1513
aI2475
aI5039
aI5669
aI65103
asS'd'
I4261
sa(dp8522
S'a'
I9368
sS's'
I16
sS'm'
(lp8523
I1
aI3
aI3
aI3
aI11
aI25
aI37
aI247
aI189
aI911
aI429
aI2395
aI3653
aI79
aI28115
aI23513
asS'd'
I4262
sa(dp8524
S'a'
I9455
sS's'
I16
sS'm'
(lp8525
I1
aI1
aI5
aI5
aI5
aI27
aI25
aI45
aI291
aI455
aI741
aI2259
aI8131
aI11779
aI14693
aI58729
asS'd'
I4263
sa(dp8526
S'a'
I9458
sS's'
I16
sS'm'
(lp8527
I1
aI3
aI3
aI7
aI21
aI33
aI67
aI49
aI153
aI491
aI1811
aI1955
aI925
aI15555
aI13801
aI48717
asS'd'
I4264
sa(dp8528
S'a'
I9469
sS's'
I16
sS'm'
(lp8529
I1
aI3
aI1
aI3
aI11
aI53
aI105
aI129
aI457
aI225
aI497
aI1123
aI973
aI2901
aI26655
aI3643
asS'd'
I4265
sa(dp8530
S'a'
I9481
sS's'
I16
sS'm'
(lp8531
I1
aI1
aI7
aI13
aI29
aI49
aI71
aI37
aI321
aI865
aI735
aI357
aI7629
aI6011
aI28221
aI39041
asS'd'
I4266
sa(dp8532
S'a'
I9482
sS's'
I16
sS'm'
(lp8533
I1
aI3
aI5
aI3
aI19
aI59
aI65
aI199
aI69
aI391
aI1735
aI3075
aI287
aI16213
aI3211
aI22425
asS'd'
I4267
sa(dp8534
S'a'
I9495
sS's'
I16
sS'm'
(lp8535
I1
aI1
aI1
aI5
aI15
aI61
aI67
aI255
aI5
aI689
aI759
aI155
aI7245
aI5881
aI21685
aI3863
asS'd'
I4268
sa(dp8536
S'a'
I9526
sS's'
I16
sS'm'
(lp8537
I1
aI1
aI3
aI11
aI23
aI63
aI69
aI241
aI359
aI735
aI371
aI603
aI2707
aI15833
aI31795
aI14901
asS'd'
I4269
sa(dp8538
S'a'
I9530
sS's'
I16
sS'm'
(lp8539
I1
aI1
aI1
aI7
aI19
aI33
aI83
aI19
aI13
aI667
aI317
aI3891
aI5497
aI8213
aI4913
aI22387
asS'd'
I4270
sa(dp8540
S'a'
I9558
sS's'
I16
sS'm'
(lp8541
I1
aI3
aI5
aI9
aI13
aI21
aI11
aI187
aI249
aI647
aI349
aI605
aI2199
aI5033
aI29773
aI48953
asS'd'
I4271
sa(dp8542
S'a'
I9562
sS's'
I16
sS'm'
(lp8543
I1
aI3
aI3
aI11
aI3
aI3
aI93
aI235
aI441
aI933
aI383
aI2007
aI4015
aI4175
aI3937
aI20623
asS'd'
I4272
sa(dp8544
S'a'
I9573
sS's'
I16
sS'm'
(lp8545
I1
aI3
aI7
aI13
aI3
aI61
aI87
aI219
aI263
aI651
aI1343
aI2709
aI31
aI16249
aI4749
aI28909
asS'd'
I4273
sa(dp8546
S'a'
I9583
sS's'
I16
sS'm'
(lp8547
I1
aI3
aI1
aI1
aI17
aI19
aI101
aI107
aI499
aI127
aI13
aI2123
aI5597
aI3751
aI14527
aI12009
asS'd'
I4274
sa(dp8548
S'a'
I9595
sS's'
I16
sS'm'
(lp8549
I1
aI3
aI5
aI13
aI27
aI57
aI1
aI195
aI107
aI947
aI1475
aI2831
aI6449
aI16117
aI20555
aI61513
asS'd'
I4275
sa(dp8550
S'a'
I9597
sS's'
I16
sS'm'
(lp8551
I1
aI3
aI1
aI9
aI9
aI47
aI89
aI187
aI401
aI299
aI637
aI197
aI1235
aI12655
aI25025
aI24443
asS'd'
I4276
sa(dp8552
S'a'
I9616
sS's'
I16
sS'm'
(lp8553
I1
aI1
aI3
aI5
aI9
aI57
aI33
aI41
aI451
aI983
aI391
aI2707
aI705
aI13913
aI28843
aI34091
asS'd'
I4277
sa(dp8554
S'a'
I9638
sS's'
I16
sS'm'
(lp8555
I1
aI3
aI5
aI3
aI29
aI19
aI61
aI31
aI349
aI253
aI1267
aI3345
aI2151
aI11309
aI19623
aI62407
asS'd'
I4278
sa(dp8556
S'a'
I9649
sS's'
I16
sS'm'
(lp8557
I1
aI1
aI1
aI3
aI11
aI37
aI31
aI59
aI1
aI253
aI103
aI2811
aI1871
aI4239
aI26311
aI32507
asS'd'
I4279
sa(dp8558
S'a'
I9662
sS's'
I16
sS'm'
(lp8559
I1
aI1
aI5
aI7
aI7
aI7
aI69
aI63
aI281
aI901
aI1785
aI2131
aI4265
aI253
aI13997
aI30177
asS'd'
I4280
sa(dp8560
S'a'
I9691
sS's'
I16
sS'm'
(lp8561
I1
aI3
aI1
aI11
aI3
aI27
aI111
aI67
aI411
aI751
aI241
aI293
aI6271
aI4187
aI22119
aI63737
asS'd'
I4281
sa(dp8562
S'a'
I9700
sS's'
I16
sS'm'
(lp8563
I1
aI3
aI5
aI5
aI27
aI19
aI45
aI203
aI81
aI59
aI1839
aI935
aI4847
aI1869
aI12037
aI30971
asS'd'
I4282
sa(dp8564
S'a'
I9703
sS's'
I16
sS'm'
(lp8565
I1
aI1
aI3
aI9
aI19
aI25
aI9
aI27
aI373
aI875
aI1735
aI689
aI2957
aI7873
aI29771
aI4093
asS'd'
I4283
sa(dp8566
S'a'
I9710
sS's'
I16
sS'm'
(lp8567
I1
aI1
aI7
aI11
aI13
aI39
aI11
aI129
aI53
aI433
aI1731
aI899
aI5855
aI10221
aI24165
aI50205
asS'd'
I4284
sa(dp8568
S'a'
I9721
sS's'
I16
sS'm'
(lp8569
I1
aI3
aI1
aI15
aI25
aI31
aI85
aI49
aI325
aI299
aI183
aI287
aI2425
aI15353
aI25999
aI59129
asS'd'
I4285
sa(dp8570
S'a'
I9724
sS's'
I16
sS'm'
(lp8571
I1
aI1
aI5
aI5
aI17
aI25
aI23
aI5
aI287
aI677
aI591
aI981
aI429
aI15297
aI14573
aI61095
asS'd'
I4286
sa(dp8572
S'a'
I9727
sS's'
I16
sS'm'
(lp8573
I1
aI1
aI5
aI15
aI5
aI15
aI67
aI195
aI209
aI341
aI1621
aI1379
aI3031
aI5469
aI31563
aI49291
asS'd'
I4287
sa(dp8574
S'a'
I9743
sS's'
I16
sS'm'
(lp8575
I1
aI1
aI1
aI1
aI25
aI33
aI61
aI201
aI15
aI125
aI969
aI1965
aI2021
aI10253
aI23801
aI28165
asS'd'
I4288
sa(dp8576
S'a'
I9779
sS's'
I16
sS'm'
(lp8577
I1
aI1
aI5
aI5
aI13
aI17
aI5
aI245
aI11
aI133
aI287
aI1929
aI4331
aI15919
aI29663
aI10243
asS'd'
I4289
sa(dp8578
S'a'
I9785
sS's'
I16
sS'm'
(lp8579
I1
aI1
aI7
aI9
aI19
aI33
aI39
aI191
aI489
aI631
aI69
aI1883
aI2183
aI14993
aI32715
aI62217
asS'd'
I4290
sa(dp8580
S'a'
I9811
sS's'
I16
sS'm'
(lp8581
I1
aI3
aI3
aI13
aI23
aI61
aI103
aI193
aI501
aI431
aI437
aI417
aI6557
aI11701
aI27577
aI42943
asS'd'
I4291
sa(dp8582
S'a'
I9820
sS's'
I16
sS'm'
(lp8583
I1
aI3
aI3
aI9
aI9
aI27
aI69
aI247
aI469
aI841
aI733
aI813
aI2673
aI7145
aI5385
aI44917
asS'd'
I4292
sa(dp8584
S'a'
I9827
sS's'
I16
sS'm'
(lp8585
I1
aI1
aI7
aI9
aI5
aI13
aI19
aI71
aI323
aI923
aI1885
aI3031
aI4507
aI13787
aI14149
aI1483
asS'd'
I4293
sa(dp8586
S'a'
I9851
sS's'
I16
sS'm'
(lp8587
I1
aI3
aI1
aI5
aI1
aI15
aI89
aI229
aI301
aI733
aI1343
aI2415
aI6463
aI1875
aI9293
aI6037
asS'd'
I4294
sa(dp8588
S'a'
I9854
sS's'
I16
sS'm'
(lp8589
I1
aI3
aI1
aI5
aI29
aI57
aI67
aI121
aI311
aI441
aI1079
aI1963
aI7137
aI6745
aI9893
aI22811
asS'd'
I4295
sa(dp8590
S'a'
I9863
sS's'
I16
sS'm'
(lp8591
I1
aI1
aI7
aI7
aI25
aI13
aI27
aI61
aI331
aI361
aI481
aI3783
aI3061
aI7827
aI18885
aI27583
asS'd'
I4296
sa(dp8592
S'a'
I9884
sS's'
I16
sS'm'
(lp8593
I1
aI3
aI1
aI5
aI17
aI47
aI19
aI83
aI309
aI65
aI1573
aI3437
aI5623
aI12691
aI21075
aI27789
asS'd'
I4297
sa(dp8594
S'a'
I9894
sS's'
I16
sS'm'
(lp8595
I1
aI1
aI7
aI9
aI13
aI51
aI7
aI209
aI131
aI111
aI1143
aI53
aI7277
aI9297
aI20869
aI33121
asS'd'
I4298
sa(dp8596
S'a'
I9903
sS's'
I16
sS'm'
(lp8597
I1
aI1
aI3
aI9
aI13
aI17
aI57
aI89
aI239
aI281
aI775
aI333
aI5631
aI2805
aI10195
aI9665
asS'd'
I4299
sa(dp8598
S'a'
I9908
sS's'
I16
sS'm'
(lp8599
I1
aI1
aI3
aI7
aI19
aI39
aI71
aI79
aI63
aI551
aI103
aI3169
aI2761
aI13929
aI20751
aI18951
asS'd'
I4300
sa(dp8600
S'a'
I9912
sS's'
I16
sS'm'
(lp8601
I1
aI1
aI7
aI11
aI5
aI23
aI7
aI249
aI447
aI483
aI433
aI3117
aI1421
aI14991
aI5397
aI19813
asS'd'
I4301
sa(dp8602
S'a'
I9925
sS's'
I16
sS'm'
(lp8603
I1
aI3
aI1
aI13
aI3
aI9
aI109
aI241
aI181
aI33
aI853
aI3939
aI3751
aI2151
aI28375
aI64443
asS'd'
I4302
sa(dp8604
S'a'
I9926
sS's'
I16
sS'm'
(lp8605
I1
aI1
aI7
aI7
aI3
aI33
aI65
aI211
aI251
aI631
aI1819
aI3913
aI3353
aI12975
aI19117
aI51515
asS'd'
I4303
sa(dp8606
S'a'
I9971
sS's'
I16
sS'm'
(lp8607
I1
aI1
aI1
aI13
aI3
aI21
aI9
aI203
aI223
aI229
aI1399
aI117
aI6297
aI11535
aI16383
aI12541
asS'd'
I4304
sa(dp8608
S'a'
I9973
sS's'
I16
sS'm'
(lp8609
I1
aI1
aI5
aI7
aI25
aI9
aI53
aI27
aI497
aI103
aI1915
aI2179
aI3679
aI11375
aI18907
aI63385
asS'd'
I4305
sa(dp8610
S'a'
I9977
sS's'
I16
sS'm'
(lp8611
I1
aI3
aI1
aI5
aI1
aI53
aI91
aI169
aI87
aI387
aI377
aI1121
aI7241
aI5133
aI1949
aI13433
asS'd'
I4306
sa(dp8612
S'a'
I10021
sS's'
I16
sS'm'
(lp8613
I1
aI1
aI1
aI3
aI27
aI35
aI61
aI189
aI241
aI445
aI287
aI325
aI127
aI2363
aI30663
aI43557
asS'd'
I4307
sa(dp8614
S'a'
I10039
sS's'
I16
sS'm'
(lp8615
I1
aI3
aI1
aI3
aI17
aI47
aI59
aI237
aI213
aI979
aI807
aI85
aI4621
aI9915
aI13631
aI55657
asS'd'
I4308
sa(dp8616
S'a'
I10048
sS's'
I16
sS'm'
(lp8617
I1
aI3
aI5
aI7
aI27
aI5
aI101
aI89
aI495
aI675
aI1181
aI2295
aI1913
aI3731
aI32639
aI58297
asS'd'
I4309
sa(dp8618
S'a'
I10051
sS's'
I16
sS'm'
(lp8619
I1
aI3
aI3
aI11
aI5
aI41
aI49
aI229
aI93
aI659
aI927
aI3425
aI4083
aI11859
aI10603
aI20631
asS'd'
I4310
sa(dp8620
S'a'
I10065
sS's'
I16
sS'm'
(lp8621
I1
aI3
aI5
aI11
aI31
aI51
aI67
aI69
aI253
aI497
aI1665
aI1985
aI5439
aI15999
aI4175
aI62175
asS'd'
I4311
sa(dp8622
S'a'
I10071
sS's'
I16
sS'm'
(lp8623
I1
aI1
aI7
aI11
aI1
aI21
aI95
aI19
aI205
aI513
aI1329
aI1821
aI1251
aI2381
aI32623
aI23923
asS'd'
I4312
sa(dp8624
S'a'
I10072
sS's'
I16
sS'm'
(lp8625
I1
aI1
aI5
aI13
aI3
aI1
aI29
aI175
aI315
aI865
aI599
aI1695
aI1391
aI2313
aI31035
aI19159
asS'd'
I4313
sa(dp8626
S'a'
I10101
sS's'
I16
sS'm'
(lp8627
I1
aI3
aI3
aI1
aI3
aI45
aI109
aI93
aI481
aI285
aI869
aI3441
aI3715
aI1355
aI9581
aI50173
asS'd'
I4314
sa(dp8628
S'a'
I10106
sS's'
I16
sS'm'
(lp8629
I1
aI3
aI7
aI7
aI15
aI35
aI107
aI107
aI315
aI213
aI281
aI2073
aI4689
aI5877
aI31
aI40967
asS'd'
I4315
sa(dp8630
S'a'
I10130
sS's'
I16
sS'm'
(lp8631
I1
aI1
aI3
aI11
aI11
aI3
aI73
aI41
aI79
aI37
aI481
aI1993
aI931
aI7677
aI11321
aI45735
asS'd'
I4316
sa(dp8632
S'a'
I10136
sS's'
I16
sS'm'
(lp8633
I1
aI1
aI7
aI1
aI15
aI21
aI65
aI237
aI263
aI849
aI863
aI4039
aI3171
aI13381
aI30373
aI51639
asS'd'
I4317
sa(dp8634
S'a'
I10148
sS's'
I16
sS'm'
(lp8635
I1
aI1
aI1
aI3
aI21
aI57
aI113
aI3
aI487
aI41
aI1277
aI633
aI2839
aI4977
aI14537
aI31749
asS'd'
I4318
sa(dp8636
S'a'
I10155
sS's'
I16
sS'm'
(lp8637
I1
aI1
aI5
aI1
aI1
aI55
aI71
aI181
aI147
aI895
aI1839
aI2157
aI3187
aI6403
aI30367
aI48915
asS'd'
I4319
sa(dp8638
S'a'
I10157
sS's'
I16
sS'm'
(lp8639
I1
aI1
aI5
aI3
aI9
aI17
aI19
aI127
aI115
aI875
aI831
aI2439
aI7475
aI1921
aI18657
aI27709
asS'd'
I4320
sa(dp8640
S'a'
I10160
sS's'
I16
sS'm'
(lp8641
I1
aI3
aI3
aI13
aI29
aI11
aI35
aI81
aI291
aI483
aI625
aI3957
aI6017
aI12543
aI18773
aI2471
asS'd'
I4321
sa(dp8642
S'a'
I10166
sS's'
I16
sS'm'
(lp8643
I1
aI3
aI3
aI13
aI11
aI39
aI7
aI85
aI493
aI209
aI819
aI3277
aI4275
aI8997
aI22943
aI33199
asS'd'
I4322
sa(dp8644
S'a'
I10183
sS's'
I16
sS'm'
(lp8645
I1
aI1
aI1
aI7
aI11
aI25
aI1
aI57
aI505
aI135
aI1713
aI3051
aI5893
aI10711
aI10681
aI12235
asS'd'
I4323
sa(dp8646
S'a'
I10192
sS's'
I16
sS'm'
(lp8647
I1
aI3
aI5
aI11
aI23
aI33
aI13
aI107
aI289
aI251
aI235
aI1747
aI4097
aI12237
aI17559
aI5063
asS'd'
I4324
sa(dp8648
S'a'
I10225
sS's'
I16
sS'm'
(lp8649
I1
aI3
aI3
aI9
aI19
aI17
aI23
aI45
aI297
aI147
aI1301
aI2057
aI7871
aI9971
aI1965
aI62449
asS'd'
I4325
sa(dp8650
S'a'
I10241
sS's'
I16
sS'm'
(lp8651
I1
aI1
aI7
aI3
aI17
aI21
aI19
aI203
aI289
aI897
aI1967
aI3519
aI3261
aI8199
aI24181
aI23273
asS'd'
I4326
sa(dp8652
S'a'
I10247
sS's'
I16
sS'm'
(lp8653
I1
aI1
aI7
aI11
aI1
aI17
aI25
aI63
aI509
aI969
aI47
aI1329
aI701
aI5227
aI419
aI14839
asS'd'
I4327
sa(dp8654
S'a'
I10284
sS's'
I16
sS'm'
(lp8655
I1
aI3
aI5
aI11
aI1
aI41
aI81
aI157
aI395
aI97
aI1919
aI3043
aI6015
aI15
aI23733
aI55485
asS'd'
I4328
sa(dp8656
S'a'
I10304
sS's'
I16
sS'm'
(lp8657
I1
aI1
aI3
aI11
aI17
aI37
aI17
aI181
aI179
aI297
aI1007
aI1559
aI6275
aI11645
aI7535
aI28941
asS'd'
I4329
sa(dp8658
S'a'
I10322
sS's'
I16
sS'm'
(lp8659
I1
aI3
aI7
aI15
aI5
aI47
aI31
aI237
aI215
aI563
aI207
aI1867
aI6635
aI6857
aI11837
aI22865
asS'd'
I4330
sa(dp8660
S'a'
I10331
sS's'
I16
sS'm'
(lp8661
I1
aI3
aI1
aI7
aI7
aI39
aI51
aI179
aI57
aI987
aI893
aI2715
aI1045
aI5799
aI19805
aI54275
asS'd'
I4331
sa(dp8662
S'a'
I10333
sS's'
I16
sS'm'
(lp8663
I1
aI1
aI7
aI15
aI25
aI9
aI127
aI243
aI323
aI1013
aI929
aI2891
aI7549
aI1071
aI17663
aI15247
asS'd'
I4332
sa(dp8664
S'a'
I10340
sS's'
I16
sS'm'
(lp8665
I1
aI1
aI1
aI5
aI25
aI23
aI101
aI9
aI371
aI89
aI1749
aI3559
aI8071
aI13887
aI14807
aI42825
asS'd'
I4333
sa(dp8666
S'a'
I10347
sS's'
I16
sS'm'
(lp8667
I1
aI3
aI3
aI11
aI21
aI41
aI3
aI77
aI3
aI709
aI1745
aI3615
aI4141
aI5275
aI28329
aI59739
asS'd'
I4334
sa(dp8668
S'a'
I10357
sS's'
I16
sS'm'
(lp8669
I1
aI1
aI7
aI13
aI1
aI11
aI73
aI183
aI363
aI241
aI863
aI3983
aI3235
aI293
aI649
aI441
asS'd'
I4335
sa(dp8670
S'a'
I10371
sS's'
I16
sS'm'
(lp8671
I1
aI1
aI5
aI3
aI13
aI27
aI13
aI191
aI57
aI639
aI1803
aI2353
aI3143
aI12763
aI5771
aI36155
asS'd'
I4336
sa(dp8672
S'a'
I10386
sS's'
I16
sS'm'
(lp8673
I1
aI1
aI5
aI3
aI1
aI53
aI85
aI45
aI283
aI823
aI1213
aI3261
aI4599
aI13267
aI4613
aI12657
asS'd'
I4337
sa(dp8674
S'a'
I10404
sS's'
I16
sS'm'
(lp8675
I1
aI3
aI5
aI15
aI27
aI49
aI59
aI123
aI357
aI527
aI337
aI2751
aI3999
aI8525
aI12501
aI40621
asS'd'
I4338
sa(dp8676
S'a'
I10414
sS's'
I16
sS'm'
(lp8677
I1
aI1
aI1
aI7
aI1
aI55
aI85
aI17
aI447
aI599
aI1315
aI2313
aI1649
aI907
aI25647
aI3251
asS'd'
I4339
sa(dp8678
S'a'
I10422
sS's'
I16
sS'm'
(lp8679
I1
aI3
aI5
aI13
aI9
aI1
aI37
aI121
aI143
aI1
aI631
aI2273
aI1673
aI3649
aI27533
aI28731
asS'd'
I4340
sa(dp8680
S'a'
I10448
sS's'
I16
sS'm'
(lp8681
I1
aI1
aI7
aI13
aI9
aI31
aI57
aI249
aI397
aI815
aI501
aI895
aI1217
aI11387
aI8635
aI65193
asS'd'
I4341
sa(dp8682
S'a'
I10451
sS's'
I16
sS'm'
(lp8683
I1
aI1
aI5
aI5
aI9
aI35
aI95
aI193
aI133
aI513
aI1929
aI3841
aI3063
aI2383
aI24413
aI51185
asS'd'
I4342
sa(dp8684
S'a'
I10473
sS's'
I16
sS'm'
(lp8685
I1
aI1
aI1
aI13
aI3
aI49
aI45
aI191
aI15
aI181
aI1819
aI3741
aI1227
aI12775
aI9461
aI44951
asS'd'
I4343
sa(dp8686
S'a'
I10479
sS's'
I16
sS'm'
(lp8687
I1
aI1
aI1
aI1
aI3
aI45
aI121
aI19
aI269
aI829
aI517
aI3913
aI183
aI11019
aI24969
aI21973
asS'd'
I4344
sa(dp8688
S'a'
I10501
sS's'
I16
sS'm'
(lp8689
I1
aI1
aI5
aI11
aI31
aI39
aI125
aI189
aI401
aI57
aI1557
aI1727
aI1415
aI4025
aI30353
aI36589
asS'd'
I4345
sa(dp8690
S'a'
I10530
sS's'
I16
sS'm'
(lp8691
I1
aI1
aI3
aI9
aI3
aI55
aI125
aI187
aI409
aI499
aI1853
aI2781
aI4323
aI16159
aI16345
aI34659
asS'd'
I4346
sa(dp8692
S'a'
I10542
sS's'
I16
sS'm'
(lp8693
I1
aI3
aI5
aI11
aI31
aI5
aI125
aI137
aI197
aI475
aI2003
aI617
aI1289
aI8365
aI28981
aI57537
asS'd'
I4347
sa(dp8694
S'a'
I10544
sS's'
I16
sS'm'
(lp8695
I1
aI1
aI1
aI5
aI19
aI29
aI83
aI127
aI311
aI177
aI1635
aI2187
aI5377
aI12841
aI7591
aI6095
asS'd'
I4348
sa(dp8696
S'a'
I10571
sS's'
I16
sS'm'
(lp8697
I1
aI3
aI5
aI5
aI21
aI39
aI65
aI197
aI115
aI411
aI1457
aI3011
aI7021
aI14119
aI61
aI21107
asS'd'
I4349
sa(dp8698
S'a'
I10628
sS's'
I16
sS'm'
(lp8699
I1
aI3
aI3
aI9
aI19
aI57
aI99
aI23
aI451
aI507
aI973
aI415
aI7123
aI11367
aI29767
aI23763
asS'd'
I4350
sa(dp8700
S'a'
I10643
sS's'
I16
sS'm'
(lp8701
I1
aI1
aI5
aI7
aI29
aI23
aI47
aI11
aI267
aI873
aI591
aI373
aI7097
aI3783
aI23289
aI5547
asS'd'
I4351
sa(dp8702
S'a'
I10673
sS's'
I16
sS'm'
(lp8703
I1
aI1
aI5
aI15
aI7
aI7
aI7
aI91
aI389
aI841
aI1995
aI459
aI7013
aI3109
aI23615
aI21519
asS'd'
I4352
sa(dp8704
S'a'
I10683
sS's'
I16
sS'm'
(lp8705
I1
aI3
aI1
aI1
aI13
aI25
aI87
aI235
aI193
aI201
aI713
aI1633
aI3117
aI13609
aI17211
aI573
asS'd'
I4353
sa(dp8706
S'a'
I10736
sS's'
I16
sS'm'
(lp8707
I1
aI1
aI1
aI9
aI27
aI53
aI105
aI39
aI217
aI781
aI997
aI661
aI6243
aI6427
aI17739
aI62239
asS'd'
I4354
sa(dp8708
S'a'
I10795
sS's'
I16
sS'm'
(lp8709
I1
aI1
aI7
aI3
aI3
aI49
aI75
aI125
aI239
aI195
aI215
aI2983
aI1185
aI4743
aI12069
aI55509
asS'd'
I4355
sa(dp8710
S'a'
I10797
sS's'
I16
sS'm'
(lp8711
I1
aI1
aI5
aI15
aI31
aI11
aI9
aI91
aI253
aI361
aI571
aI1589
aI2425
aI4279
aI3765
aI46519
asS'd'
I4356
sa(dp8712
S'a'
I10815
sS's'
I16
sS'm'
(lp8713
I1
aI1
aI3
aI3
aI21
aI49
aI49
aI213
aI399
aI253
aI1565
aI2447
aI453
aI7957
aI24799
aI58503
asS'd'
I4357
sa(dp8714
S'a'
I10817
sS's'
I16
sS'm'
(lp8715
I1
aI1
aI7
aI1
aI5
aI59
aI81
aI97
aI209
aI477
aI17
aI3085
aI3655
aI599
aI24011
aI14981
asS'd'
I4358
sa(dp8716
S'a'
I10842
sS's'
I16
sS'm'
(lp8717
I1
aI3
aI3
aI13
aI19
aI49
aI7
aI35
aI111
aI169
aI629
aI1587
aI5345
aI13699
aI21187
aI30199
asS'd'
I4359
sa(dp8718
S'a'
I10844
sS's'
I16
sS'm'
(lp8719
I1
aI1
aI3
aI13
aI19
aI59
aI73
aI127
aI475
aI509
aI9
aI2661
aI711
aI15835
aI31429
aI33885
asS'd'
I4360
sa(dp8720
S'a'
I10863
sS's'
I16
sS'm'
(lp8721
I1
aI3
aI5
aI3
aI31
aI15
aI43
aI185
aI29
aI897
aI1041
aI1057
aI6285
aI13925
aI4023
aI25741
asS'd'
I4361
sa(dp8722
S'a'
I10899
sS's'
I16
sS'm'
(lp8723
I1
aI3
aI5
aI11
aI1
aI33
aI63
aI155
aI175
aI501
aI1147
aI3395
aI3285
aI16237
aI22315
aI50945
asS'd'
I4362
sa(dp8724
S'a'
I10902
sS's'
I16
sS'm'
(lp8725
I1
aI3
aI3
aI3
aI5
aI13
aI77
aI227
aI85
aI139
aI139
aI1
aI7147
aI2023
aI32705
aI38753
asS'd'
I4363
sa(dp8726
S'a'
I10917
sS's'
I16
sS'm'
(lp8727
I1
aI3
aI5
aI11
aI31
aI59
aI35
aI179
aI489
aI537
aI1537
aI2877
aI4937
aI10825
aI2445
aI34907
asS'd'
I4364
sa(dp8728
S'a'
I10953
sS's'
I16
sS'm'
(lp8729
I1
aI3
aI7
aI11
aI17
aI17
aI95
aI223
aI165
aI925
aI829
aI3971
aI1
aI7393
aI8825
aI25705
asS'd'
I4365
sa(dp8730
S'a'
I10967
sS's'
I16
sS'm'
(lp8731
I1
aI1
aI1
aI1
aI25
aI17
aI25
aI143
aI385
aI907
aI1381
aI1823
aI3819
aI8475
aI5321
aI12037
asS'd'
I4366
sa(dp8732
S'a'
I11002
sS's'
I16
sS'm'
(lp8733
I1
aI1
aI5
aI11
aI7
aI47
aI97
aI85
aI105
aI23
aI263
aI1329
aI1905
aI12121
aI29635
aI41249
asS'd'
I4367
sa(dp8734
S'a'
I11024
sS's'
I16
sS'm'
(lp8735
I1
aI1
aI7
aI11
aI1
aI51
aI13
aI13
aI5
aI143
aI83
aI3831
aI959
aI6083
aI16997
aI59887
asS'd'
I4368
sa(dp8736
S'a'
I11029
sS's'
I16
sS'm'
(lp8737
I1
aI3
aI3
aI13
aI25
aI9
aI31
aI5
aI215
aI791
aI767
aI1733
aI2715
aI14283
aI25795
aI54249
asS'd'
I4369
sa(dp8738
S'a'
I11030
sS's'
I16
sS'm'
(lp8739
I1
aI3
aI7
aI5
aI11
aI19
aI125
aI81
aI71
aI131
aI1869
aI1111
aI6763
aI5275
aI23095
aI1139
asS'd'
I4370
sa(dp8740
S'a'
I11043
sS's'
I16
sS'm'
(lp8741
I1
aI3
aI3
aI9
aI25
aI43
aI119
aI49
aI133
aI217
aI521
aI1367
aI5867
aI6829
aI29871
aI60383
asS'd'
I4371
sa(dp8742
S'a'
I11087
sS's'
I16
sS'm'
(lp8743
I1
aI1
aI7
aI9
aI5
aI11
aI59
aI157
aI279
aI301
aI481
aI3273
aI7943
aI3273
aI27783
aI17271
asS'd'
I4372
sa(dp8744
S'a'
I11106
sS's'
I16
sS'm'
(lp8745
I1
aI3
aI3
aI13
aI11
aI57
aI13
aI5
aI435
aI169
aI541
aI517
aI2359
aI9121
aI27911
aI15679
asS'd'
I4373
sa(dp8746
S'a'
I11130
sS's'
I16
sS'm'
(lp8747
I1
aI1
aI3
aI9
aI9
aI55
aI65
aI113
aI21
aI807
aI373
aI2825
aI1527
aI15565
aI8339
aI7227
asS'd'
I4374
sa(dp8748
S'a'
I11156
sS's'
I16
sS'm'
(lp8749
I1
aI3
aI5
aI9
aI1
aI1
aI49
aI255
aI477
aI821
aI1647
aI713
aI6841
aI3187
aI22649
aI51469
asS'd'
I4375
sa(dp8750
S'a'
I11176
sS's'
I16
sS'm'
(lp8751
I1
aI3
aI3
aI11
aI13
aI43
aI63
aI139
aI71
aI809
aI993
aI2429
aI833
aI6545
aI10987
aI39567
asS'd'
I4376
sa(dp8752
S'a'
I11221
sS's'
I16
sS'm'
(lp8753
I1
aI1
aI1
aI9
aI19
aI23
aI47
aI199
aI191
aI827
aI391
aI837
aI1209
aI2493
aI24071
aI46589
asS'd'
I4377
sa(dp8754
S'a'
I11267
sS's'
I16
sS'm'
(lp8755
I1
aI1
aI5
aI13
aI25
aI51
aI39
aI43
aI103
aI899
aI1729
aI2389
aI2965
aI189
aI3063
aI50609
asS'd'
I4378
sa(dp8756
S'a'
I11282
sS's'
I16
sS'm'
(lp8757
I1
aI1
aI3
aI1
aI5
aI29
aI105
aI201
aI503
aI199
aI507
aI205
aI2389
aI695
aI15233
aI50353
asS'd'
I4379
sa(dp8758
S'a'
I11294
sS's'
I16
sS'm'
(lp8759
I1
aI3
aI1
aI7
aI19
aI53
aI45
aI21
aI89
aI545
aI1885
aI765
aI6673
aI13485
aI9987
aI2609
asS'd'
I4380
sa(dp8760
S'a'
I11332
sS's'
I16
sS'm'
(lp8761
I1
aI3
aI7
aI13
aI17
aI7
aI59
aI23
aI159
aI309
aI1407
aI2483
aI1807
aI15733
aI5603
aI52989
asS'd'
I4381
sa(dp8762
S'a'
I11353
sS's'
I16
sS'm'
(lp8763
I1
aI1
aI1
aI11
aI13
aI19
aI123
aI185
aI413
aI745
aI361
aI2391
aI6697
aI2281
aI11999
aI13175
asS'd'
I4382
sa(dp8764
S'a'
I11369
sS's'
I16
sS'm'
(lp8765
I1
aI3
aI5
aI5
aI11
aI49
aI123
aI173
aI325
aI667
aI895
aI1067
aI8121
aI10577
aI30561
aI17391
asS'd'
I4383
sa(dp8766
S'a'
I11372
sS's'
I16
sS'm'
(lp8767
I1
aI1
aI5
aI5
aI23
aI21
aI77
aI223
aI281
aI161
aI141
aI345
aI3879
aI11393
aI26907
aI53805
asS'd'
I4384
sa(dp8768
S'a'
I11375
sS's'
I16
sS'm'
(lp8769
I1
aI3
aI3
aI5
aI3
aI47
aI17
aI109
aI185
aI139
aI957
aI1417
aI4553
aI6101
aI29537
aI34821
asS'd'
I4385
sa(dp8770
S'a'
I11413
sS's'
I16
sS'm'
(lp8771
I1
aI1
aI5
aI3
aI29
aI49
aI89
aI61
aI45
aI593
aI269
aI1483
aI2971
aI991
aI21239
aI29301
asS'd'
I4386
sa(dp8772
S'a'
I11429
sS's'
I16
sS'm'
(lp8773
I1
aI1
aI3
aI13
aI29
aI45
aI33
aI253
aI291
aI783
aI737
aI2363
aI2651
aI8627
aI21785
aI58419
asS'd'
I4387
sa(dp8774
S'a'
I11430
sS's'
I16
sS'm'
(lp8775
I1
aI3
aI7
aI15
aI29
aI15
aI81
aI185
aI363
aI681
aI1737
aI3789
aI4365
aI3295
aI23205
aI4457
asS'd'
I4388
sa(dp8776
S'a'
I11444
sS's'
I16
sS'm'
(lp8777
I1
aI3
aI3
aI11
aI15
aI39
aI67
aI167
aI197
aI357
aI871
aI2201
aI5377
aI6299
aI20873
aI59283
asS'd'
I4389
sa(dp8778
S'a'
I11462
sS's'
I16
sS'm'
(lp8779
I1
aI3
aI3
aI5
aI9
aI15
aI127
aI49
aI21
aI719
aI357
aI425
aI5507
aI9639
aI275
aI47503
asS'd'
I4390
sa(dp8780
S'a'
I11465
sS's'
I16
sS'm'
(lp8781
I1
aI1
aI7
aI1
aI13
aI63
aI111
aI121
aI21
aI481
aI247
aI3147
aI5783
aI8947
aI20809
aI42039
asS'd'
I4391
sa(dp8782
S'a'
I11471
sS's'
I16
sS'm'
(lp8783
I1
aI1
aI3
aI3
aI31
aI33
aI9
aI69
aI187
aI517
aI2029
aI2205
aI7661
aI4757
aI27525
aI9665
asS'd'
I4392
sa(dp8784
S'a'
I11473
sS's'
I16
sS'm'
(lp8785
I1
aI1
aI1
aI13
aI7
aI41
aI103
aI161
aI233
aI221
aI693
aI213
aI4609
aI7771
aI28703
aI17407
asS'd'
I4393
sa(dp8786
S'a'
I11495
sS's'
I16
sS'm'
(lp8787
I1
aI3
aI7
aI15
aI31
aI47
aI27
aI111
aI479
aI1003
aI1687
aI347
aI2179
aI11861
aI8169
aI31941
asS'd'
I4394
sa(dp8788
S'a'
I11499
sS's'
I16
sS'm'
(lp8789
I1
aI1
aI3
aI5
aI5
aI63
aI25
aI125
aI199
aI147
aI589
aI3565
aI3449
aI8331
aI8923
aI31207
asS'd'
I4395
sa(dp8790
S'a'
I11519
sS's'
I16
sS'm'
(lp8791
I1
aI1
aI3
aI11
aI19
aI25
aI77
aI99
aI299
aI19
aI1641
aI2193
aI4299
aI3635
aI20621
aI267
asS'd'
I4396
sa(dp8792
S'a'
I11562
sS's'
I16
sS'm'
(lp8793
I1
aI3
aI7
aI11
aI9
aI59
aI7
aI167
aI1
aI775
aI29
aI1935
aI3723
aI11835
aI2887
aI65285
asS'd'
I4397
sa(dp8794
S'a'
I11576
sS's'
I16
sS'm'
(lp8795
I1
aI1
aI7
aI13
aI5
aI47
aI101
aI155
aI235
aI93
aI465
aI3581
aI1837
aI7675
aI10789
aI45167
asS'd'
I4398
sa(dp8796
S'a'
I11582
sS's'
I16
sS'm'
(lp8797
I1
aI1
aI3
aI5
aI9
aI59
aI121
aI109
aI59
aI821
aI1625
aI343
aI1591
aI3875
aI13729
aI56381
asS'd'
I4399
sa(dp8798
S'a'
I11596
sS's'
I16
sS'm'
(lp8799
I1
aI1
aI1
aI9
aI27
aI53
aI93
aI215
aI133
aI561
aI39
aI2591
aI1041
aI11913
aI24493
aI37921
asS'd'
I4400
sa(dp8800
S'a'
I11602
sS's'
I16
sS'm'
(lp8801
I1
aI1
aI1
aI7
aI23
aI7
aI81
aI107
aI219
aI943
aI563
aI1083
aI5803
aI5687
aI32559
aI62727
asS'd'
I4401
sa(dp8802
S'a'
I11611
sS's'
I16
sS'm'
(lp8803
I1
aI3
aI7
aI7
aI21
aI53
aI3
aI5
aI231
aI601
aI1561
aI103
aI3837
aI2675
aI5263
aI23375
asS'd'
I4402
sa(dp8804
S'a'
I11627
sS's'
I16
sS'm'
(lp8805
I1
aI1
aI3
aI13
aI15
aI27
aI89
aI7
aI251
aI887
aI951
aI3001
aI5687
aI4921
aI5091
aI59337
asS'd'
I4403
sa(dp8806
S'a'
I11682
sS's'
I16
sS'm'
(lp8807
I1
aI3
aI7
aI15
aI25
aI53
aI19
aI155
aI185
aI503
aI547
aI1917
aI7633
aI15167
aI14177
aI46761
asS'd'
I4404
sa(dp8808
S'a'
I11687
sS's'
I16
sS'm'
(lp8809
I1
aI3
aI5
aI15
aI21
aI49
aI13
aI163
aI471
aI281
aI1141
aI3013
aI6847
aI9261
aI15955
aI9397
asS'd'
I4405
sa(dp8810
S'a'
I11691
sS's'
I16
sS'm'
(lp8811
I1
aI3
aI3
aI3
aI1
aI21
aI19
aI239
aI479
aI609
aI65
aI2735
aI5337
aI6293
aI19351
aI34135
asS'd'
I4406
sa(dp8812
S'a'
I11733
sS's'
I16
sS'm'
(lp8813
I1
aI1
aI7
aI1
aI9
aI1
aI119
aI23
aI411
aI535
aI101
aI1597
aI2379
aI2421
aI31471
aI36473
asS'd'
I4407
sa(dp8814
S'a'
I11747
sS's'
I16
sS'm'
(lp8815
I1
aI3
aI1
aI11
aI31
aI63
aI17
aI225
aI45
aI409
aI59
aI3943
aI8043
aI11759
aI10667
aI58793
asS'd'
I4408
sa(dp8816
S'a'
I11759
sS's'
I16
sS'm'
(lp8817
I1
aI1
aI3
aI3
aI9
aI49
aI61
aI239
aI245
aI765
aI1945
aI3567
aI5355
aI14799
aI7141
aI59511
asS'd'
I4409
sa(dp8818
S'a'
I11778
sS's'
I16
sS'm'
(lp8819
I1
aI3
aI7
aI9
aI3
aI33
aI103
aI99
aI35
aI799
aI1347
aI2253
aI8189
aI14177
aI13479
aI13749
asS'd'
I4410
sa(dp8820
S'a'
I11852
sS's'
I16
sS'm'
(lp8821
I1
aI3
aI1
aI15
aI15
aI51
aI7
aI179
aI471
aI265
aI1571
aI2983
aI701
aI15133
aI7885
aI29977
asS'd'
I4411
sa(dp8822
S'a'
I11857
sS's'
I16
sS'm'
(lp8823
I1
aI1
aI5
aI9
aI15
aI37
aI49
aI213
aI113
aI729
aI1115
aI2727
aI2635
aI8433
aI11145
aI46779
asS'd'
I4412
sa(dp8824
S'a'
I11858
sS's'
I16
sS'm'
(lp8825
I1
aI3
aI5
aI11
aI7
aI3
aI115
aI151
aI133
aI723
aI7
aI4063
aI5807
aI9845
aI25829
aI29315
asS'd'
I4413
sa(dp8826
S'a'
I11893
sS's'
I16
sS'm'
(lp8827
I1
aI3
aI5
aI9
aI25
aI55
aI17
aI135
aI145
aI379
aI1603
aI3459
aI5773
aI6545
aI17509
aI25847
asS'd'
I4414
sa(dp8828
S'a'
I11907
sS's'
I16
sS'm'
(lp8829
I1
aI1
aI7
aI11
aI1
aI61
aI113
aI147
aI489
aI775
aI1347
aI2199
aI299
aI9589
aI19931
aI1335
asS'd'
I4415
sa(dp8830
S'a'
I11928
sS's'
I16
sS'm'
(lp8831
I1
aI3
aI1
aI3
aI1
aI7
aI27
aI243
aI355
aI425
aI1215
aI3723
aI3489
aI9285
aI12739
aI24797
asS'd'
I4416
sa(dp8832
S'a'
I11931
sS's'
I16
sS'm'
(lp8833
I1
aI3
aI5
aI11
aI15
aI25
aI57
aI221
aI247
aI647
aI259
aI1665
aI7055
aI2835
aI16411
aI42999
asS'd'
I4417
sa(dp8834
S'a'
I11933
sS's'
I16
sS'm'
(lp8835
I1
aI1
aI3
aI7
aI9
aI25
aI61
aI233
aI73
aI235
aI1539
aI1865
aI5671
aI1329
aI5767
aI43039
asS'd'
I4418
sa(dp8836
S'a'
I11967
sS's'
I16
sS'm'
(lp8837
I1
aI1
aI7
aI9
aI21
aI11
aI123
aI7
aI41
aI407
aI1485
aI1723
aI585
aI10597
aI16215
aI63403
asS'd'
I4419
sa(dp8838
S'a'
I11976
sS's'
I16
sS'm'
(lp8839
I1
aI1
aI5
aI7
aI27
aI9
aI123
aI101
aI273
aI673
aI1141
aI3841
aI4041
aI13169
aI8221
aI12915
asS'd'
I4420
sa(dp8840
S'a'
I11989
sS's'
I16
sS'm'
(lp8841
I1
aI3
aI1
aI13
aI3
aI17
aI105
aI17
aI237
aI321
aI855
aI2821
aI2467
aI3503
aI15187
aI1689
asS'd'
I4421
sa(dp8842
S'a'
I12003
sS's'
I16
sS'm'
(lp8843
I1
aI1
aI5
aI5
aI19
aI23
aI9
aI205
aI87
aI153
aI1543
aI1193
aI6619
aI6845
aI8459
aI37533
asS'd'
I4422
sa(dp8844
S'a'
I12024
sS's'
I16
sS'm'
(lp8845
I1
aI1
aI7
aI15
aI13
aI29
aI79
aI9
aI203
aI211
aI239
aI2349
aI3447
aI7797
aI19311
aI58071
asS'd'
I4423
sa(dp8846
S'a'
I12030
sS's'
I16
sS'm'
(lp8847
I1
aI3
aI5
aI11
aI5
aI49
aI71
aI151
aI333
aI895
aI1095
aI2471
aI2477
aI14493
aI16711
aI14393
asS'd'
I4424
sa(dp8848
S'a'
I12037
sS's'
I16
sS'm'
(lp8849
I1
aI1
aI5
aI13
aI17
aI19
aI25
aI149
aI111
aI631
aI763
aI2535
aI3631
aI1809
aI8163
aI18037
asS'd'
I4425
sa(dp8850
S'a'
I12044
sS's'
I16
sS'm'
(lp8851
I1
aI3
aI3
aI13
aI23
aI61
aI25
aI179
aI351
aI247
aI1021
aI2413
aI2585
aI3731
aI5435
aI52723
asS'd'
I4426
sa(dp8852
S'a'
I12052
sS's'
I16
sS'm'
(lp8853
I1
aI1
aI5
aI11
aI1
aI39
aI65
aI59
aI21
aI927
aI1989
aI2823
aI4857
aI15521
aI30495
aI16067
asS'd'
I4427
sa(dp8854
S'a'
I12059
sS's'
I16
sS'm'
(lp8855
I1
aI3
aI3
aI7
aI17
aI5
aI17
aI125
aI379
aI875
aI1565
aI2435
aI933
aI6809
aI20129
aI26339
asS'd'
I4428
sa(dp8856
S'a'
I12075
sS's'
I16
sS'm'
(lp8857
I1
aI1
aI7
aI5
aI3
aI57
aI51
aI213
aI455
aI663
aI2007
aI3995
aI4889
aI9527
aI23507
aI3261
asS'd'
I4429
sa(dp8858
S'a'
I12083
sS's'
I16
sS'm'
(lp8859
I1
aI3
aI7
aI5
aI1
aI29
aI85
aI151
aI165
aI123
aI1425
aI2851
aI4427
aI7683
aI21085
aI28925
asS'd'
I4430
sa(dp8860
S'a'
I12117
sS's'
I16
sS'm'
(lp8861
I1
aI1
aI5
aI3
aI11
aI33
aI127
aI3
aI41
aI591
aI1539
aI3823
aI125
aI421
aI9051
aI55025
asS'd'
I4431
sa(dp8862
S'a'
I12138
sS's'
I16
sS'm'
(lp8863
I1
aI3
aI1
aI5
aI7
aI9
aI69
aI35
aI59
aI477
aI1207
aI1245
aI6423
aI11329
aI26535
aI37621
asS'd'
I4432
sa(dp8864
S'a'
I12146
sS's'
I16
sS'm'
(lp8865
I1
aI3
aI1
aI1
aI15
aI35
aI17
aI123
aI303
aI193
aI1489
aI2587
aI1883
aI4063
aI1921
aI60413
asS'd'
I4433
sa(dp8866
S'a'
I12202
sS's'
I16
sS'm'
(lp8867
I1
aI1
aI5
aI1
aI7
aI61
aI39
aI247
aI139
aI1015
aI757
aI823
aI4757
aI9307
aI32287
aI37241
asS'd'
I4434
sa(dp8868
S'a'
I12230
sS's'
I16
sS'm'
(lp8869
I1
aI1
aI7
aI15
aI3
aI5
aI85
aI93
aI457
aI999
aI1331
aI919
aI5271
aI11687
aI24233
aI38803
asS'd'
I4435
sa(dp8870
S'a'
I12254
sS's'
I16
sS'm'
(lp8871
I1
aI3
aI3
aI9
aI5
aI43
aI37
aI13
aI505
aI603
aI1857
aI2675
aI2017
aI9481
aI10873
aI54755
asS'd'
I4436
sa(dp8872
S'a'
I12304
sS's'
I16
sS'm'
(lp8873
I1
aI1
aI5
aI15
aI13
aI3
aI7
aI239
aI471
aI835
aI553
aI413
aI4029
aI8613
aI15533
aI58987
asS'd'
I4437
sa(dp8874
S'a'
I12316
sS's'
I16
sS'm'
(lp8875
I1
aI3
aI1
aI5
aI19
aI27
aI21
aI43
aI57
aI755
aI1245
aI2805
aI3799
aI2013
aI21145
aI10317
asS'd'
I4438
sa(dp8876
S'a'
I12337
sS's'
I16
sS'm'
(lp8877
I1
aI3
aI5
aI13
aI9
aI23
aI35
aI5
aI315
aI169
aI399
aI2641
aI1525
aI10561
aI11917
aI33009
asS'd'
I4439
sa(dp8878
S'a'
I12347
sS's'
I16
sS'm'
(lp8879
I1
aI3
aI5
aI7
aI23
aI53
aI101
aI105
aI451
aI207
aI1271
aI3067
aI6725
aI15525
aI7951
aI1283
asS'd'
I4440
sa(dp8880
S'a'
I12367
sS's'
I16
sS'm'
(lp8881
I1
aI1
aI5
aI5
aI27
aI21
aI77
aI143
aI213
aI443
aI149
aI2667
aI5269
aI10359
aI25287
aI5843
asS'd'
I4441
sa(dp8882
S'a'
I12398
sS's'
I16
sS'm'
(lp8883
I1
aI3
aI5
aI5
aI25
aI27
aI109
aI157
aI459
aI767
aI765
aI2667
aI1833
aI3781
aI9077
aI64321
asS'd'
I4442
sa(dp8884
S'a'
I12421
sS's'
I16
sS'm'
(lp8885
I1
aI3
aI3
aI13
aI31
aI25
aI97
aI237
aI279
aI431
aI1713
aI809
aI1989
aI10431
aI5867
aI42197
asS'd'
I4443
sa(dp8886
S'a'
I12428
sS's'
I16
sS'm'
(lp8887
I1
aI3
aI7
aI3
aI9
aI5
aI5
aI191
aI73
aI521
aI787
aI903
aI3073
aI2067
aI24741
aI57029
asS'd'
I4444
sa(dp8888
S'a'
I12446
sS's'
I16
sS'm'
(lp8889
I1
aI3
aI3
aI1
aI3
aI41
aI125
aI53
aI125
aI781
aI865
aI3677
aI6279
aI357
aI10667
aI1127
asS'd'
I4445
sa(dp8890
S'a'
I12449
sS's'
I16
sS'm'
(lp8891
I1
aI1
aI5
aI11
aI25
aI19
aI99
aI121
aI359
aI73
aI607
aI2149
aI5739
aI15669
aI29457
aI57549
asS'd'
I4446
sa(dp8892
S'a'
I12467
sS's'
I16
sS'm'
(lp8893
I1
aI1
aI5
aI3
aI23
aI5
aI35
aI55
aI369
aI239
aI329
aI3057
aI3757
aI12523
aI5017
aI52185
asS'd'
I4447
sa(dp8894
S'a'
I12479
sS's'
I16
sS'm'
(lp8895
I1
aI3
aI3
aI13
aI17
aI61
aI69
aI165
aI267
aI323
aI2007
aI2025
aI4423
aI15975
aI31897
aI37013
asS'd'
I4448
sa(dp8896
S'a'
I12502
sS's'
I16
sS'm'
(lp8897
I1
aI3
aI7
aI13
aI19
aI19
aI87
aI111
aI389
aI611
aI1523
aI963
aI4671
aI12569
aI21839
aI10919
asS'd'
I4449
sa(dp8898
S'a'
I12521
sS's'
I16
sS'm'
(lp8899
I1
aI1
aI1
aI3
aI1
aI27
aI13
aI227
aI29
aI457
aI221
aI127
aI5335
aI16247
aI19559
aI19185
asS'd'
I4450
sa(dp8900
S'a'
I12553
sS's'
I16
sS'm'
(lp8901
I1
aI3
aI5
aI7
aI29
aI21
aI23
aI157
aI197
aI87
aI1591
aI1829
aI407
aI15885
aI14933
aI1997
asS'd'
I4451
sa(dp8902
S'a'
I12568
sS's'
I16
sS'm'
(lp8903
I1
aI1
aI1
aI9
aI3
aI35
aI43
aI187
aI195
aI325
aI197
aI2905
aI7323
aI1563
aI7659
aI45185
asS'd'
I4452
sa(dp8904
S'a'
I12573
sS's'
I16
sS'm'
(lp8905
I1
aI1
aI1
aI15
aI3
aI23
aI105
aI33
aI73
aI495
aI1409
aI2583
aI1099
aI1041
aI28955
aI60293
asS'd'
I4453
sa(dp8906
S'a'
I12592
sS's'
I16
sS'm'
(lp8907
I1
aI3
aI7
aI13
aI25
aI19
aI99
aI137
aI139
aI719
aI529
aI1147
aI5813
aI11551
aI30183
aI14593
asS'd'
I4454
sa(dp8908
S'a'
I12597
sS's'
I16
sS'm'
(lp8909
I1
aI3
aI3
aI5
aI17
aI25
aI73
aI193
aI309
aI887
aI1655
aI1641
aI2091
aI12087
aI21881
aI1145
asS'd'
I4455
sa(dp8910
S'a'
I12601
sS's'
I16
sS'm'
(lp8911
I1
aI3
aI1
aI1
aI27
aI41
aI13
aI151
aI83
aI645
aI327
aI1795
aI2717
aI12433
aI22751
aI9823
asS'd'
I4456
sa(dp8912
S'a'
I12615
sS's'
I16
sS'm'
(lp8913
I1
aI1
aI3
aI7
aI1
aI43
aI77
aI229
aI59
aI499
aI1883
aI135
aI3461
aI9821
aI219
aI6111
asS'd'
I4457
sa(dp8914
S'a'
I12619
sS's'
I16
sS'm'
(lp8915
I1
aI3
aI3
aI3
aI23
aI7
aI17
aI67
aI361
aI565
aI1621
aI3253
aI7973
aI281
aI3209
aI48215
asS'd'
I4458
sa(dp8916
S'a'
I12694
sS's'
I16
sS'm'
(lp8917
I1
aI1
aI3
aI7
aI31
aI15
aI97
aI141
aI197
aI883
aI1689
aI269
aI7487
aI10403
aI18903
aI58147
asS'd'
I4459
sa(dp8918
S'a'
I12697
sS's'
I16
sS'm'
(lp8919
I1
aI3
aI3
aI3
aI23
aI9
aI87
aI125
aI359
aI527
aI1251
aI637
aI1145
aI12721
aI14693
aI6021
asS'd'
I4460
sa(dp8920
S'a'
I12698
sS's'
I16
sS'm'
(lp8921
I1
aI1
aI3
aI5
aI13
aI43
aI105
aI173
aI205
aI859
aI1237
aI1227
aI1409
aI15513
aI25317
aI30745
asS'd'
I4461
sa(dp8922
S'a'
I12713
sS's'
I16
sS'm'
(lp8923
I1
aI3
aI3
aI15
aI31
aI43
aI125
aI149
aI145
aI109
aI975
aI1167
aI7629
aI8373
aI5923
aI64117
asS'd'
I4462
sa(dp8924
S'a'
I12722
sS's'
I16
sS'm'
(lp8925
I1
aI3
aI1
aI15
aI3
aI33
aI89
aI173
aI379
aI615
aI1401
aI1567
aI4453
aI7461
aI32555
aI64369
asS'd'
I4463
sa(dp8926
S'a'
I12733
sS's'
I16
sS'm'
(lp8927
I1
aI3
aI7
aI11
aI27
aI23
aI45
aI7
aI15
aI21
aI1663
aI3327
aI5611
aI8535
aI27669
aI25525
asS'd'
I4464
sa(dp8928
S'a'
I12736
sS's'
I16
sS'm'
(lp8929
I1
aI1
aI3
aI15
aI15
aI61
aI127
aI145
aI151
aI41
aI629
aI767
aI5801
aI3395
aI1157
aI30033
asS'd'
I4465
sa(dp8930
S'a'
I12790
sS's'
I16
sS'm'
(lp8931
I1
aI1
aI1
aI5
aI9
aI63
aI73
aI9
aI299
aI237
aI369
aI1295
aI4869
aI6821
aI19961
aI32129
asS'd'
I4466
sa(dp8932
S'a'
I12794
sS's'
I16
sS'm'
(lp8933
I1
aI1
aI5
aI13
aI19
aI7
aI119
aI35
aI319
aI405
aI1255
aI1299
aI4311
aI14999
aI24567
aI4001
asS'd'
I4467
sa(dp8934
S'a'
I12803
sS's'
I16
sS'm'
(lp8935
I1
aI1
aI1
aI13
aI9
aI39
aI13
aI207
aI355
aI691
aI37
aI3137
aI6073
aI16179
aI28661
aI41
asS'd'
I4468
sa(dp8936
S'a'
I12815
sS's'
I16
sS'm'
(lp8937
I1
aI1
aI3
aI7
aI21
aI3
aI123
aI27
aI187
aI183
aI769
aI2367
aI2957
aI7065
aI17855
aI60805
asS'd'
I4469
sa(dp8938
S'a'
I12829
sS's'
I16
sS'm'
(lp8939
I1
aI1
aI1
aI1
aI19
aI31
aI71
aI85
aI303
aI617
aI1007
aI283
aI8087
aI11079
aI11463
aI65295
asS'd'
I4470
sa(dp8940
S'a'
I12833
sS's'
I16
sS'm'
(lp8941
I1
aI1
aI3
aI13
aI25
aI63
aI61
aI187
aI401
aI465
aI1485
aI801
aI3649
aI7763
aI8495
aI54479
asS'd'
I4471
sa(dp8942
S'a'
I12840
sS's'
I16
sS'm'
(lp8943
I1
aI3
aI7
aI3
aI13
aI51
aI41
aI119
aI311
aI699
aI1113
aI3631
aI1981
aI3259
aI25871
aI45659
asS'd'
I4472
sa(dp8944
S'a'
I12875
sS's'
I16
sS'm'
(lp8945
I1
aI3
aI7
aI13
aI31
aI27
aI57
aI181
aI325
aI107
aI1745
aI635
aI3941
aI3305
aI14563
aI29855
asS'd'
I4473
sa(dp8946
S'a'
I12877
sS's'
I16
sS'm'
(lp8947
I1
aI3
aI7
aI15
aI5
aI55
aI5
aI147
aI365
aI485
aI1841
aI3673
aI6513
aI11121
aI5725
aI18027
asS'd'
I4474
sa(dp8948
S'a'
I12916
sS's'
I16
sS'm'
(lp8949
I1
aI3
aI5
aI11
aI13
aI45
aI35
aI79
aI109
aI683
aI1171
aI3015
aI2163
aI4823
aI4365
aI42931
asS'd'
I4475
sa(dp8950
S'a'
I12923
sS's'
I16
sS'm'
(lp8951
I1
aI1
aI7
aI7
aI13
aI45
aI57
aI39
aI297
aI985
aI1559
aI487
aI5071
aI2657
aI9401
aI41889
asS'd'
I4476
sa(dp8952
S'a'
I12935
sS's'
I16
sS'm'
(lp8953
I1
aI1
aI5
aI9
aI29
aI33
aI79
aI237
aI509
aI537
aI549
aI3657
aI7141
aI15189
aI30853
aI36089
asS'd'
I4477
sa(dp8954
S'a'
I12949
sS's'
I16
sS'm'
(lp8955
I1
aI3
aI5
aI7
aI31
aI15
aI75
aI73
aI237
aI273
aI865
aI743
aI2607
aI7611
aI18441
aI12703
asS'd'
I4478
sa(dp8956
S'a'
I12954
sS's'
I16
sS'm'
(lp8957
I1
aI1
aI1
aI9
aI27
aI9
aI35
aI137
aI265
aI181
aI1341
aI1945
aI5615
aI161
aI18369
aI4791
asS'd'
I4479
sa(dp8958
S'a'
I12956
sS's'
I16
sS'm'
(lp8959
I1
aI3
aI7
aI11
aI27
aI29
aI29
aI43
aI489
aI177
aI1489
aI2927
aI623
aI14571
aI22447
aI46905
asS'd'
I4480
sa(dp8960
S'a'
I12959
sS's'
I16
sS'm'
(lp8961
I1
aI3
aI3
aI3
aI19
aI41
aI107
aI53
aI239
aI263
aI1433
aI1655
aI7991
aI7405
aI2735
aI25519
asS'd'
I4481
sa(dp8962
S'a'
I12978
sS's'
I16
sS'm'
(lp8963
I1
aI3
aI5
aI7
aI19
aI37
aI73
aI243
aI215
aI445
aI1781
aI3223
aI187
aI8443
aI18185
aI45093
asS'd'
I4482
sa(dp8964
S'a'
I13001
sS's'
I16
sS'm'
(lp8965
I1
aI1
aI3
aI13
aI9
aI57
aI111
aI111
aI221
aI505
aI1261
aI3045
aI1655
aI16247
aI21033
aI17993
asS'd'
I4483
sa(dp8966
S'a'
I13010
sS's'
I16
sS'm'
(lp8967
I1
aI1
aI7
aI5
aI7
aI55
aI77
aI5
aI131
aI969
aI1481
aI2883
aI2645
aI3003
aI5601
aI37063
asS'd'
I4484
sa(dp8968
S'a'
I13043
sS's'
I16
sS'm'
(lp8969
I1
aI1
aI5
aI15
aI29
aI55
aI39
aI197
aI349
aI29
aI341
aI67
aI1977
aI425
aI4063
aI42705
asS'd'
I4485
sa(dp8970
S'a'
I13049
sS's'
I16
sS'm'
(lp8971
I1
aI1
aI7
aI13
aI1
aI57
aI81
aI81
aI129
aI681
aI1407
aI2465
aI3627
aI2325
aI31649
aI18449
asS'd'
I4486
sa(dp8972
S'a'
I13058
sS's'
I16
sS'm'
(lp8973
I1
aI3
aI5
aI5
aI23
aI59
aI35
aI217
aI393
aI155
aI1315
aI105
aI2285
aI5167
aI27997
aI55193
asS'd'
I4487
sa(dp8974
S'a'
I13112
sS's'
I16
sS'm'
(lp8975
I1
aI1
aI7
aI3
aI11
aI59
aI53
aI179
aI319
aI819
aI947
aI3881
aI765
aI4219
aI16405
aI48055
asS'd'
I4488
sa(dp8976
S'a'
I13140
sS's'
I16
sS'm'
(lp8977
I1
aI1
aI3
aI9
aI23
aI9
aI67
aI67
aI137
aI523
aI1585
aI2441
aI167
aI5217
aI12031
aI14297
asS'd'
I4489
sa(dp8978
S'a'
I13149
sS's'
I16
sS'm'
(lp8979
I1
aI1
aI5
aI13
aI31
aI63
aI121
aI91
aI439
aI917
aI203
aI1519
aI4363
aI2391
aI25755
aI26677
asS'd'
I4490
sa(dp8980
S'a'
I13163
sS's'
I16
sS'm'
(lp8981
I1
aI1
aI3
aI5
aI25
aI31
aI11
aI95
aI339
aI817
aI35
aI3923
aI7365
aI10537
aI5521
aI54579
asS'd'
I4491
sa(dp8982
S'a'
I13187
sS's'
I16
sS'm'
(lp8983
I1
aI3
aI7
aI1
aI3
aI33
aI47
aI13
aI139
aI445
aI1357
aI3907
aI7495
aI8789
aI26589
aI43479
asS'd'
I4492
sa(dp8984
S'a'
I13196
sS's'
I16
sS'm'
(lp8985
I1
aI1
aI1
aI11
aI5
aI45
aI61
aI13
aI167
aI287
aI931
aI881
aI5713
aI12937
aI12951
aI21597
asS'd'
I4493
sa(dp8986
S'a'
I13237
sS's'
I16
sS'm'
(lp8987
I1
aI3
aI5
aI1
aI29
aI23
aI7
aI117
aI503
aI897
aI733
aI1113
aI7205
aI11507
aI561
aI34011
asS'd'
I4494
sa(dp8988
S'a'
I13244
sS's'
I16
sS'm'
(lp8989
I1
aI3
aI5
aI7
aI3
aI51
aI21
aI147
aI35
aI259
aI689
aI3801
aI2481
aI9673
aI4065
aI595
asS'd'
I4495
sa(dp8990
S'a'
I13264
sS's'
I16
sS'm'
(lp8991
I1
aI3
aI3
aI9
aI9
aI29
aI5
aI191
aI393
aI979
aI1627
aI937
aI75
aI2339
aI24007
aI30555
asS'd'
I4496
sa(dp8992
S'a'
I13279
sS's'
I16
sS'm'
(lp8993
I1
aI1
aI5
aI7
aI9
aI35
aI111
aI23
aI113
aI563
aI1689
aI1575
aI6127
aI9919
aI2555
aI52261
asS'd'
I4497
sa(dp8994
S'a'
I13292
sS's'
I16
sS'm'
(lp8995
I1
aI3
aI1
aI5
aI31
aI21
aI117
aI159
aI473
aI279
aI1281
aI311
aI159
aI3343
aI27761
aI59989
asS'd'
I4498
sa(dp8996
S'a'
I13295
sS's'
I16
sS'm'
(lp8997
I1
aI1
aI5
aI1
aI23
aI31
aI67
aI241
aI401
aI69
aI933
aI777
aI267
aI12411
aI23767
aI9047
asS'd'
I4499
sa(dp8998
S'a'
I13307
sS's'
I16
sS'm'
(lp8999
I1
aI1
aI5
aI1
aI15
aI49
aI99
aI15
aI267
aI913
aI1581
aI3713
aI651
aI14275
aI10103
aI57619
asS'd'
I4500
sa(dp9000
S'a'
I13312
sS's'
I16
sS'm'
(lp9001
I1
aI3
aI5
aI9
aI19
aI23
aI95
aI5
aI449
aI153
aI1195
aI1315
aI2347
aI12683
aI10865
aI50703
asS'd'
I4501
sa(dp9002
S'a'
I13317
sS's'
I16
sS'm'
(lp9003
I1
aI1
aI1
aI13
aI17
aI17
aI115
aI31
aI135
aI725
aI795
aI1695
aI6199
aI4179
aI5223
aI48457
asS'd'
I4502
sa(dp9004
S'a'
I13327
sS's'
I16
sS'm'
(lp9005
I1
aI3
aI5
aI15
aI31
aI15
aI3
aI247
aI385
aI269
aI1665
aI581
aI2809
aI6333
aI7459
aI14815
asS'd'
I4503
sa(dp9006
S'a'
I13348
sS's'
I16
sS'm'
(lp9007
I1
aI3
aI7
aI5
aI15
aI35
aI117
aI85
aI11
aI621
aI1523
aI981
aI511
aI14113
aI4673
aI22683
asS'd'
I4504
sa(dp9008
S'a'
I13390
sS's'
I16
sS'm'
(lp9009
I1
aI1
aI7
aI1
aI27
aI61
aI119
aI249
aI431
aI147
aI173
aI423
aI1353
aI4747
aI12761
aI32947
asS'd'
I4505
sa(dp9010
S'a'
I13413
sS's'
I16
sS'm'
(lp9011
I1
aI3
aI7
aI1
aI23
aI39
aI15
aI153
aI219
aI359
aI1233
aI169
aI2817
aI11043
aI12435
aI30135
asS'd'
I4506
sa(dp9012
S'a'
I13417
sS's'
I16
sS'm'
(lp9013
I1
aI1
aI5
aI1
aI1
aI55
aI39
aI1
aI151
aI865
aI125
aI2351
aI6315
aI1165
aI20163
aI43991
asS'd'
I4507
sa(dp9014
S'a'
I13418
sS's'
I16
sS'm'
(lp9015
I1
aI1
aI3
aI9
aI25
aI41
aI115
aI221
aI129
aI265
aI1887
aI4057
aI7523
aI13591
aI5735
aI59645
asS'd'
I4508
sa(dp9016
S'a'
I13451
sS's'
I16
sS'm'
(lp9017
I1
aI1
aI5
aI5
aI19
aI15
aI9
aI77
aI511
aI627
aI153
aI2015
aI247
aI15949
aI9715
aI45411
asS'd'
I4509
sa(dp9018
S'a'
I13475
sS's'
I16
sS'm'
(lp9019
I1
aI1
aI7
aI7
aI17
aI17
aI107
aI183
aI39
aI647
aI1339
aI3915
aI4937
aI537
aI27011
aI58937
asS'd'
I4510
sa(dp9020
S'a'
I13482
sS's'
I16
sS'm'
(lp9021
I1
aI3
aI3
aI13
aI3
aI3
aI69
aI201
aI431
aI65
aI683
aI121
aI7017
aI2791
aI16713
aI62555
asS'd'
I4511
sa(dp9022
S'a'
I13510
sS's'
I16
sS'm'
(lp9023
I1
aI3
aI7
aI3
aI7
aI41
aI117
aI237
aI23
aI757
aI545
aI3899
aI1837
aI5555
aI7891
aI29151
asS'd'
I4512
sa(dp9024
S'a'
I13527
sS's'
I16
sS'm'
(lp9025
I1
aI1
aI1
aI3
aI27
aI15
aI39
aI195
aI353
aI299
aI1431
aI2279
aI1795
aI13773
aI24915
aI49701
asS'd'
I4513
sa(dp9026
S'a'
I13543
sS's'
I16
sS'm'
(lp9027
I1
aI1
aI5
aI5
aI7
aI7
aI125
aI5
aI401
aI523
aI1967
aI2471
aI7279
aI7559
aI12025
aI60599
asS'd'
I4514
sa(dp9028
S'a'
I13547
sS's'
I16
sS'm'
(lp9029
I1
aI1
aI1
aI13
aI13
aI59
aI13
aI249
aI465
aI847
aI1483
aI975
aI7729
aI2773
aI15745
aI51237
asS'd'
I4515
sa(dp9030
S'a'
I13627
sS's'
I16
sS'm'
(lp9031
I1
aI1
aI7
aI9
aI31
aI41
aI115
aI141
aI247
aI355
aI1109
aI3239
aI6495
aI4515
aI30145
aI47705
asS'd'
I4516
sa(dp9032
S'a'
I13650
sS's'
I16
sS'm'
(lp9033
I1
aI1
aI3
aI13
aI29
aI41
aI69
aI179
aI45
aI271
aI1909
aI3095
aI7199
aI14049
aI9903
aI33383
asS'd'
I4517
sa(dp9034
S'a'
I13683
sS's'
I16
sS'm'
(lp9035
I1
aI1
aI3
aI13
aI7
aI45
aI105
aI105
aI243
aI121
aI67
aI3169
aI237
aI137
aI4175
aI54325
asS'd'
I4518
sa(dp9036
S'a'
I13696
sS's'
I16
sS'm'
(lp9037
I1
aI3
aI3
aI11
aI19
aI51
aI93
aI155
aI79
aI579
aI1621
aI1251
aI7639
aI15875
aI25815
aI56063
asS'd'
I4519
sa(dp9038
S'a'
I13702
sS's'
I16
sS'm'
(lp9039
I1
aI3
aI3
aI9
aI27
aI27
aI77
aI45
aI217
aI965
aI1045
aI875
aI4515
aI11261
aI27859
aI757
asS'd'
I4520
sa(dp9040
S'a'
I13713
sS's'
I16
sS'm'
(lp9041
I1
aI1
aI3
aI11
aI17
aI7
aI81
aI37
aI299
aI765
aI977
aI3371
aI3163
aI13267
aI18345
aI9239
asS'd'
I4521
sa(dp9042
S'a'
I13739
sS's'
I16
sS'm'
(lp9043
I1
aI1
aI1
aI3
aI15
aI23
aI115
aI11
aI183
aI511
aI557
aI3253
aI153
aI8465
aI22659
aI42143
asS'd'
I4522
sa(dp9044
S'a'
I13749
sS's'
I16
sS'm'
(lp9045
I1
aI1
aI5
aI13
aI17
aI61
aI127
aI219
aI225
aI981
aI615
aI731
aI4069
aI12637
aI11601
aI38257
asS'd'
I4523
sa(dp9046
S'a'
I13767
sS's'
I16
sS'm'
(lp9047
I1
aI1
aI5
aI3
aI29
aI3
aI73
aI79
aI393
aI779
aI823
aI2473
aI3811
aI4417
aI9399
aI50011
asS'd'
I4524
sa(dp9048
S'a'
I13774
sS's'
I16
sS'm'
(lp9049
I1
aI3
aI3
aI9
aI21
aI35
aI61
aI99
aI115
aI657
aI629
aI1129
aI2355
aI12665
aI459
aI40831
asS'd'
I4525
sa(dp9050
S'a'
I13781
sS's'
I16
sS'm'
(lp9051
I1
aI3
aI1
aI7
aI25
aI61
aI53
aI249
aI15
aI649
aI665
aI595
aI1441
aI8035
aI5381
aI7147
asS'd'
I4526
sa(dp9052
S'a'
I13795
sS's'
I16
sS'm'
(lp9053
I1
aI3
aI1
aI7
aI19
aI9
aI27
aI207
aI205
aI461
aI1069
aI4039
aI6549
aI2333
aI29
aI50067
asS'd'
I4527
sa(dp9054
S'a'
I13821
sS's'
I16
sS'm'
(lp9055
I1
aI1
aI5
aI3
aI15
aI7
aI115
aI205
aI71
aI73
aI53
aI71
aI6049
aI15293
aI17041
aI20313
asS'd'
I4528
sa(dp9056
S'a'
I13825
sS's'
I16
sS'm'
(lp9057
I1
aI1
aI7
aI7
aI9
aI7
aI119
aI99
aI357
aI729
aI2041
aI3355
aI5333
aI1263
aI30521
aI64867
asS'd'
I4529
sa(dp9058
S'a'
I13838
sS's'
I16
sS'm'
(lp9059
I1
aI1
aI1
aI7
aI31
aI63
aI63
aI159
aI281
aI295
aI913
aI2161
aI8033
aI13789
aI17711
aI14915
asS'd'
I4530
sa(dp9060
S'a'
I13852
sS's'
I16
sS'm'
(lp9061
I1
aI1
aI7
aI9
aI29
aI55
aI69
aI129
aI453
aI361
aI1883
aI17
aI1765
aI111
aI10311
aI55019
asS'd'
I4531
sa(dp9062
S'a'
I13879
sS's'
I16
sS'm'
(lp9063
I1
aI1
aI5
aI9
aI21
aI15
aI31
aI57
aI291
aI915
aI945
aI1775
aI5905
aI9073
aI3271
aI15571
asS'd'
I4532
sa(dp9064
S'a'
I13888
sS's'
I16
sS'm'
(lp9065
I1
aI1
aI1
aI7
aI21
aI11
aI1
aI9
aI167
aI143
aI1535
aI1267
aI6675
aI13037
aI19513
aI52027
asS'd'
I4533
sa(dp9066
S'a'
I13897
sS's'
I16
sS'm'
(lp9067
I1
aI3
aI3
aI7
aI31
aI45
aI57
aI105
aI63
aI121
aI631
aI679
aI3873
aI16333
aI32069
aI64725
asS'd'
I4534
sa(dp9068
S'a'
I13906
sS's'
I16
sS'm'
(lp9069
I1
aI1
aI1
aI9
aI23
aI41
aI29
aI207
aI489
aI319
aI905
aI3147
aI4195
aI2697
aI5281
aI1771
asS'd'
I4535
sa(dp9070
S'a'
I13939
sS's'
I16
sS'm'
(lp9071
I1
aI1
aI1
aI9
aI25
aI33
aI57
aI201
aI405
aI111
aI407
aI3489
aI449
aI8601
aI1273
aI42105
asS'd'
I4536
sa(dp9072
S'a'
I13962
sS's'
I16
sS'm'
(lp9073
I1
aI1
aI1
aI3
aI19
aI45
aI123
aI159
aI237
aI173
aI781
aI787
aI7537
aI15453
aI25567
aI53729
asS'd'
I4537
sa(dp9074
S'a'
I13981
sS's'
I16
sS'm'
(lp9075
I1
aI1
aI7
aI3
aI29
aI9
aI89
aI207
aI345
aI685
aI1701
aI2859
aI8065
aI9289
aI2459
aI28367
asS'd'
I4538
sa(dp9076
S'a'
I13985
sS's'
I16
sS'm'
(lp9077
I1
aI3
aI1
aI3
aI31
aI41
aI55
aI241
aI81
aI1001
aI595
aI1725
aI853
aI11427
aI20617
aI1717
asS'd'
I4539
sa(dp9078
S'a'
I14020
sS's'
I16
sS'm'
(lp9079
I1
aI1
aI3
aI3
aI9
aI45
aI121
aI69
aI177
aI1017
aI211
aI2753
aI6923
aI1387
aI32063
aI45337
asS'd'
I4540
sa(dp9080
S'a'
I14030
sS's'
I16
sS'm'
(lp9081
I1
aI1
aI5
aI15
aI21
aI23
aI95
aI61
aI485
aI403
aI619
aI3035
aI4881
aI13423
aI17815
aI35221
asS'd'
I4541
sa(dp9082
S'a'
I14041
sS's'
I16
sS'm'
(lp9083
I1
aI1
aI3
aI3
aI3
aI59
aI23
aI69
aI77
aI309
aI227
aI2877
aI3739
aI3539
aI20083
aI23415
asS'd'
I4542
sa(dp9084
S'a'
I14047
sS's'
I16
sS'm'
(lp9085
I1
aI3
aI7
aI3
aI17
aI43
aI95
aI239
aI223
aI353
aI1237
aI3239
aI1369
aI7245
aI32401
aI63889
asS'd'
I4543
sa(dp9086
S'a'
I14048
sS's'
I16
sS'm'
(lp9087
I1
aI1
aI1
aI5
aI25
aI63
aI123
aI3
aI291
aI819
aI793
aI241
aI5619
aI8871
aI18341
aI2757
asS'd'
I4544
sa(dp9088
S'a'
I14066
sS's'
I16
sS'm'
(lp9089
I1
aI3
aI7
aI15
aI3
aI21
aI17
aI97
aI395
aI333
aI909
aI3783
aI3635
aI751
aI24227
aI44281
asS'd'
I4545
sa(dp9090
S'a'
I14092
sS's'
I16
sS'm'
(lp9091
I1
aI3
aI7
aI13
aI29
aI49
aI123
aI195
aI191
aI159
aI211
aI1887
aI3047
aI4871
aI2607
aI32425
asS'd'
I4546
sa(dp9092
S'a'
I14120
sS's'
I16
sS'm'
(lp9093
I1
aI1
aI7
aI7
aI15
aI57
aI91
aI255
aI267
aI897
aI441
aI1581
aI953
aI6951
aI17275
aI50229
asS'd'
I4547
sa(dp9094
S'a'
I14126
sS's'
I16
sS'm'
(lp9095
I1
aI3
aI7
aI15
aI1
aI35
aI91
aI219
aI7
aI117
aI119
aI2687
aI1215
aI9517
aI10849
aI28347
asS'd'
I4548
sa(dp9096
S'a'
I14131
sS's'
I16
sS'm'
(lp9097
I1
aI1
aI1
aI1
aI21
aI55
aI67
aI221
aI503
aI883
aI1037
aI2965
aI1485
aI8557
aI28091
aI25459
asS'd'
I4549
sa(dp9098
S'a'
I14143
sS's'
I16
sS'm'
(lp9099
I1
aI3
aI5
aI9
aI19
aI3
aI73
aI123
aI95
aI307
aI1339
aI3669
aI5077
aI12049
aI523
aI21457
asS'd'
I4550
sa(dp9100
S'a'
I14146
sS's'
I16
sS'm'
(lp9101
I1
aI3
aI1
aI13
aI3
aI1
aI111
aI97
aI371
aI697
aI1989
aI3715
aI7875
aI6841
aI7009
aI17809
asS'd'
I4551
sa(dp9102
S'a'
I14152
sS's'
I16
sS'm'
(lp9103
I1
aI1
aI1
aI9
aI25
aI21
aI19
aI43
aI329
aI531
aI491
aI1147
aI1469
aI12841
aI29651
aI29517
asS'd'
I4552
sa(dp9104
S'a'
I14155
sS's'
I16
sS'm'
(lp9105
I1
aI1
aI5
aI1
aI15
aI63
aI101
aI197
aI477
aI245
aI341
aI61
aI3803
aI10001
aI5519
aI19083
asS'd'
I4553
sa(dp9106
S'a'
I14157
sS's'
I16
sS'm'
(lp9107
I1
aI3
aI7
aI5
aI13
aI43
aI7
aI143
aI291
aI531
aI1727
aI871
aI7091
aI5737
aI24285
aI51017
asS'd'
I4554
sa(dp9108
S'a'
I14188
sS's'
I16
sS'm'
(lp9109
I1
aI3
aI5
aI1
aI17
aI13
aI15
aI85
aI361
aI153
aI989
aI1367
aI4705
aI3599
aI4441
aI52471
asS'd'
I4555
sa(dp9110
S'a'
I14206
sS's'
I16
sS'm'
(lp9111
I1
aI1
aI7
aI13
aI21
aI43
aI111
aI29
aI299
aI439
aI1929
aI283
aI5901
aI14113
aI3929
aI55843
asS'd'
I4556
sa(dp9112
S'a'
I14243
sS's'
I16
sS'm'
(lp9113
I1
aI3
aI3
aI9
aI31
aI59
aI63
aI43
aI91
aI171
aI733
aI1359
aI425
aI8505
aI19777
aI54723
asS'd'
I4557
sa(dp9114
S'a'
I14278
sS's'
I16
sS'm'
(lp9115
I1
aI1
aI5
aI7
aI31
aI1
aI97
aI253
aI331
aI307
aI1749
aI2115
aI2535
aI9945
aI11013
aI14231
asS'd'
I4558
sa(dp9116
S'a'
I14308
sS's'
I16
sS'm'
(lp9117
I1
aI1
aI5
aI11
aI13
aI7
aI109
aI237
aI301
aI383
aI683
aI1603
aI6393
aI2437
aI12101
aI1767
asS'd'
I4559
sa(dp9118
S'a'
I14317
sS's'
I16
sS'm'
(lp9119
I1
aI1
aI3
aI11
aI15
aI61
aI119
aI199
aI109
aI265
aI1431
aI1729
aI3409
aI10129
aI16945
aI34681
asS'd'
I4560
sa(dp9120
S'a'
I14335
sS's'
I16
sS'm'
(lp9121
I1
aI3
aI7
aI9
aI13
aI59
aI121
aI73
aI29
aI513
aI279
aI457
aI7985
aI15199
aI10185
aI33621
asS'd'
I4561
sa(dp9122
S'a'
I14354
sS's'
I16
sS'm'
(lp9123
I1
aI3
aI7
aI7
aI23
aI17
aI27
aI65
aI387
aI487
aI1803
aI2789
aI461
aI11201
aI7001
aI40229
asS'd'
I4562
sa(dp9124
S'a'
I14356
sS's'
I16
sS'm'
(lp9125
I1
aI1
aI3
aI15
aI9
aI13
aI55
aI127
aI33
aI513
aI1055
aI643
aI505
aI3005
aI6121
aI64147
asS'd'
I4563
sa(dp9126
S'a'
I14379
sS's'
I16
sS'm'
(lp9127
I1
aI3
aI5
aI15
aI5
aI11
aI77
aI233
aI175
aI691
aI171
aI2491
aI6915
aI14195
aI7845
aI36499
asS'd'
I4564
sa(dp9128
S'a'
I14381
sS's'
I16
sS'm'
(lp9129
I1
aI1
aI5
aI11
aI19
aI45
aI103
aI207
aI99
aI645
aI1739
aI1517
aI5907
aI16035
aI14559
aI44007
asS'd'
I4565
sa(dp9130
S'a'
I14384
sS's'
I16
sS'm'
(lp9131
I1
aI3
aI7
aI15
aI21
aI27
aI53
aI107
aI89
aI291
aI983
aI3527
aI1025
aI2985
aI13747
aI32715
asS'd'
I4566
sa(dp9132
S'a'
I14389
sS's'
I16
sS'm'
(lp9133
I1
aI1
aI3
aI15
aI23
aI17
aI27
aI209
aI77
aI959
aI813
aI3597
aI5809
aI693
aI10325
aI16855
asS'd'
I4567
sa(dp9134
S'a'
I14390
sS's'
I16
sS'm'
(lp9135
I1
aI1
aI7
aI11
aI23
aI53
aI123
aI99
aI211
aI935
aI1371
aI1657
aI4699
aI2683
aI27933
aI21451
asS'd'
I4568
sa(dp9136
S'a'
I14414
sS's'
I16
sS'm'
(lp9137
I1
aI3
aI3
aI3
aI17
aI21
aI93
aI201
aI423
aI351
aI1639
aI227
aI5719
aI13111
aI5993
aI44615
asS'd'
I4569
sa(dp9138
S'a'
I14425
sS's'
I16
sS'm'
(lp9139
I1
aI1
aI7
aI3
aI13
aI49
aI59
aI255
aI213
aI147
aI1441
aI3593
aI6419
aI15657
aI1947
aI62713
asS'd'
I4570
sa(dp9140
S'a'
I14462
sS's'
I16
sS'm'
(lp9141
I1
aI3
aI1
aI7
aI7
aI41
aI79
aI135
aI275
aI585
aI925
aI3139
aI4351
aI1827
aI23533
aI63031
asS'd'
I4571
sa(dp9142
S'a'
I14472
sS's'
I16
sS'm'
(lp9143
I1
aI1
aI7
aI3
aI11
aI1
aI13
aI149
aI29
aI897
aI1043
aI2393
aI3931
aI6741
aI19973
aI46303
asS'd'
I4572
sa(dp9144
S'a'
I14508
sS's'
I16
sS'm'
(lp9145
I1
aI1
aI1
aI13
aI13
aI57
aI9
aI253
aI149
aI67
aI1531
aI4049
aI3013
aI13947
aI3371
aI35317
asS'd'
I4573
sa(dp9146
S'a'
I14511
sS's'
I16
sS'm'
(lp9147
I1
aI3
aI1
aI1
aI15
aI19
aI11
aI125
aI179
aI383
aI1273
aI1551
aI6441
aI6579
aI19659
aI31005
asS'd'
I4574
sa(dp9148
S'a'
I14519
sS's'
I16
sS'm'
(lp9149
I1
aI1
aI3
aI15
aI29
aI37
aI11
aI199
aI273
aI663
aI549
aI3167
aI2049
aI8815
aI30719
aI47905
asS'd'
I4575
sa(dp9150
S'a'
I14528
sS's'
I16
sS'm'
(lp9151
I1
aI1
aI1
aI15
aI7
aI9
aI113
aI231
aI155
aI27
aI419
aI1303
aI4493
aI5633
aI5743
aI51335
asS'd'
I4576
sa(dp9152
S'a'
I14561
sS's'
I16
sS'm'
(lp9153
I1
aI3
aI5
aI13
aI21
aI35
aI7
aI63
aI391
aI637
aI2011
aI841
aI5933
aI10563
aI7593
aI34767
asS'd'
I4577
sa(dp9154
S'a'
I14574
sS's'
I16
sS'm'
(lp9155
I1
aI3
aI1
aI15
aI19
aI13
aI89
aI127
aI507
aI715
aI1305
aI2989
aI7551
aI1953
aI26101
aI54913
asS'd'
I4578
sa(dp9156
S'a'
I14588
sS's'
I16
sS'm'
(lp9157
I1
aI1
aI5
aI1
aI1
aI33
aI101
aI211
aI173
aI761
aI177
aI2721
aI6949
aI15801
aI6639
aI21405
asS'd'
I4579
sa(dp9158
S'a'
I14594
sS's'
I16
sS'm'
(lp9159
I1
aI3
aI1
aI13
aI15
aI23
aI113
aI177
aI43
aI743
aI57
aI3875
aI7833
aI13619
aI17637
aI5547
asS'd'
I4580
sa(dp9160
S'a'
I14606
sS's'
I16
sS'm'
(lp9161
I1
aI1
aI3
aI13
aI21
aI7
aI123
aI83
aI391
aI731
aI597
aI2595
aI1949
aI14619
aI17141
aI60595
asS'd'
I4581
sa(dp9162
S'a'
I14614
sS's'
I16
sS'm'
(lp9163
I1
aI3
aI7
aI13
aI31
aI55
aI15
aI43
aI17
aI855
aI233
aI1411
aI1063
aI12977
aI22159
aI59185
asS'd'
I4582
sa(dp9164
S'a'
I14639
sS's'
I16
sS'm'
(lp9165
I1
aI3
aI7
aI7
aI17
aI53
aI67
aI37
aI245
aI941
aI1213
aI1965
aI6635
aI10189
aI12979
aI63503
asS'd'
I4583
sa(dp9166
S'a'
I14641
sS's'
I16
sS'm'
(lp9167
I1
aI1
aI5
aI15
aI31
aI23
aI15
aI175
aI177
aI643
aI1705
aI3541
aI2009
aI12005
aI27281
aI16821
asS'd'
I4584
sa(dp9168
S'a'
I14648
sS's'
I16
sS'm'
(lp9169
I1
aI3
aI1
aI13
aI27
aI37
aI1
aI171
aI255
aI445
aI171
aI3555
aI8169
aI399
aI20975
aI36195
asS'd'
I4585
sa(dp9170
S'a'
I14668
sS's'
I16
sS'm'
(lp9171
I1
aI3
aI7
aI11
aI13
aI15
aI123
aI65
aI173
aI317
aI1991
aI2093
aI8073
aI12831
aI15455
aI30175
asS'd'
I4586
sa(dp9172
S'a'
I14673
sS's'
I16
sS'm'
(lp9173
I1
aI3
aI1
aI7
aI5
aI53
aI59
aI219
aI407
aI501
aI875
aI2627
aI1335
aI14387
aI25523
aI28337
asS'd'
I4587
sa(dp9174
S'a'
I14679
sS's'
I16
sS'm'
(lp9175
I1
aI3
aI3
aI13
aI13
aI41
aI93
aI125
aI41
aI461
aI887
aI1085
aI3393
aI11945
aI16329
aI43531
asS'd'
I4588
sa(dp9176
S'a'
I14695
sS's'
I16
sS'm'
(lp9177
I1
aI3
aI1
aI11
aI21
aI39
aI1
aI185
aI429
aI285
aI443
aI1165
aI451
aI10903
aI31511
aI50555
asS'd'
I4589
sa(dp9178
S'a'
I14702
sS's'
I16
sS'm'
(lp9179
I1
aI1
aI7
aI5
aI11
aI25
aI61
aI171
aI493
aI733
aI215
aI1871
aI7783
aI14113
aI2061
aI58961
asS'd'
I4590
sa(dp9180
S'a'
I14704
sS's'
I16
sS'm'
(lp9181
I1
aI1
aI7
aI7
aI27
aI23
aI53
aI45
aI131
aI301
aI275
aI3855
aI4883
aI6303
aI25269
aI12697
asS'd'
I4591
sa(dp9182
S'a'
I14740
sS's'
I16
sS'm'
(lp9183
I1
aI3
aI5
aI7
aI11
aI15
aI71
aI101
aI377
aI803
aI1369
aI3741
aI633
aI10571
aI30659
aI31101
asS'd'
I4592
sa(dp9184
S'a'
I14754
sS's'
I16
sS'm'
(lp9185
I1
aI1
aI5
aI15
aI19
aI53
aI3
aI153
aI411
aI685
aI1405
aI109
aI5755
aI13311
aI3713
aI24579
asS'd'
I4593
sa(dp9186
S'a'
I14774
sS's'
I16
sS'm'
(lp9187
I1
aI1
aI3
aI3
aI27
aI7
aI89
aI39
aI5
aI853
aI1757
aI2927
aI2889
aI9735
aI17959
aI39301
asS'd'
I4594
sa(dp9188
S'a'
I14792
sS's'
I16
sS'm'
(lp9189
I1
aI3
aI1
aI3
aI13
aI41
aI57
aI71
aI187
aI285
aI825
aI1807
aI7261
aI2645
aI21861
aI1839
asS'd'
I4595
sa(dp9190
S'a'
I14797
sS's'
I16
sS'm'
(lp9191
I1
aI3
aI3
aI5
aI15
aI21
aI23
aI7
aI341
aI981
aI891
aI721
aI7221
aI3137
aI28725
aI54993
asS'd'
I4596
sa(dp9192
S'a'
I14806
sS's'
I16
sS'm'
(lp9193
I1
aI1
aI5
aI3
aI3
aI61
aI59
aI97
aI205
aI359
aI185
aI3361
aI7637
aI15473
aI6351
aI62097
asS'd'
I4597
sa(dp9194
S'a'
I14812
sS's'
I16
sS'm'
(lp9195
I1
aI1
aI1
aI9
aI13
aI11
aI123
aI71
aI199
aI251
aI535
aI371
aI1605
aI12107
aI13833
aI2019
asS'd'
I4598
sa(dp9196
S'a'
I14856
sS's'
I16
sS'm'
(lp9197
I1
aI1
aI7
aI13
aI27
aI1
aI43
aI73
aI409
aI601
aI1481
aI649
aI3293
aI12257
aI23377
aI17225
asS'd'
I4599
sa(dp9198
S'a'
I14876
sS's'
I16
sS'm'
(lp9199
I1
aI1
aI7
aI11
aI15
aI55
aI99
aI45
aI261
aI461
aI1507
aI3575
aI4425
aI9895
aI20191
aI61863
asS'd'
I4600
sa(dp9200
S'a'
I14900
sS's'
I16
sS'm'
(lp9201
I1
aI3
aI7
aI1
aI3
aI7
aI19
aI85
aI139
aI691
aI1685
aI137
aI7547
aI16143
aI14193
aI52479
asS'd'
I4601
sa(dp9202
S'a'
I14910
sS's'
I16
sS'm'
(lp9203
I1
aI3
aI5
aI9
aI17
aI61
aI7
aI189
aI31
aI641
aI1381
aI3999
aI4909
aI8463
aI31761
aI54493
asS'd'
I4602
sa(dp9204
S'a'
I14912
sS's'
I16
sS'm'
(lp9205
I1
aI1
aI5
aI15
aI17
aI15
aI69
aI111
aI55
aI1011
aI1859
aI2643
aI6043
aI5125
aI15875
aI611
asS'd'
I4603
sa(dp9206
S'a'
I14915
sS's'
I16
sS'm'
(lp9207
I1
aI1
aI3
aI5
aI3
aI33
aI73
aI227
aI327
aI369
aI189
aI1841
aI5625
aI1179
aI18651
aI34951
asS'd'
I4604
sa(dp9208
S'a'
I14922
sS's'
I16
sS'm'
(lp9209
I1
aI3
aI7
aI13
aI1
aI17
aI109
aI149
aI89
aI889
aI799
aI3423
aI2599
aI14525
aI12763
aI23855
asS'd'
I4605
sa(dp9210
S'a'
I14939
sS's'
I16
sS'm'
(lp9211
I1
aI1
aI3
aI15
aI5
aI63
aI87
aI7
aI63
aI171
aI1215
aI557
aI3009
aI16305
aI23517
aI40101
asS'd'
I4606
sa(dp9212
S'a'
I14946
sS's'
I16
sS'm'
(lp9213
I1
aI1
aI3
aI3
aI29
aI31
aI79
aI183
aI401
aI813
aI41
aI1111
aI5669
aI15697
aI7813
aI10215
asS'd'
I4607
sa(dp9214
S'a'
I14955
sS's'
I16
sS'm'
(lp9215
I1
aI1
aI5
aI7
aI9
aI25
aI25
aI57
aI343
aI375
aI535
aI3405
aI1909
aI3201
aI2417
aI52285
asS'd'
I4608
sa(dp9216
S'a'
I14966
sS's'
I16
sS'm'
(lp9217
I1
aI1
aI5
aI9
aI25
aI19
aI33
aI183
aI29
aI991
aI1045
aI2249
aI2933
aI12525
aI13943
aI10423
asS'd'
I4609
sa(dp9218
S'a'
I14976
sS's'
I16
sS'm'
(lp9219
I1
aI3
aI1
aI7
aI3
aI45
aI49
aI37
aI429
aI67
aI821
aI1289
aI7311
aI16165
aI25861
aI57715
asS'd'
I4610
sa(dp9220
S'a'
I14986
sS's'
I16
sS'm'
(lp9221
I1
aI1
aI7
aI3
aI19
aI3
aI33
aI153
aI505
aI683
aI611
aI1691
aI6421
aI15517
aI19161
aI49013
asS'd'
I4611
sa(dp9222
S'a'
I14993
sS's'
I16
sS'm'
(lp9223
I1
aI3
aI7
aI7
aI21
aI21
aI85
aI55
aI293
aI199
aI1671
aI1881
aI7147
aI8241
aI16173
aI51873
asS'd'
I4612
sa(dp9224
S'a'
I15012
sS's'
I16
sS'm'
(lp9225
I1
aI3
aI1
aI15
aI3
aI61
aI97
aI191
aI435
aI511
aI1599
aI2705
aI1897
aI2607
aI1801
aI48583
asS'd'
I4613
sa(dp9226
S'a'
I15041
sS's'
I16
sS'm'
(lp9227
I1
aI1
aI5
aI3
aI9
aI23
aI23
aI185
aI401
aI947
aI33
aI385
aI7491
aI14129
aI14561
aI13759
asS'd'
I4614
sa(dp9228
S'a'
I15053
sS's'
I16
sS'm'
(lp9229
I1
aI3
aI5
aI15
aI19
aI21
aI37
aI233
aI149
aI673
aI29
aI1315
aI3487
aI6705
aI28283
aI43135
asS'd'
I4615
sa(dp9230
S'a'
I15056
sS's'
I16
sS'm'
(lp9231
I1
aI1
aI1
aI11
aI9
aI35
aI101
aI255
aI345
aI829
aI689
aI2747
aI2145
aI2101
aI24863
aI35529
asS'd'
I4616
sa(dp9232
S'a'
I15059
sS's'
I16
sS'm'
(lp9233
I1
aI3
aI7
aI9
aI5
aI5
aI53
aI247
aI157
aI729
aI1621
aI129
aI2485
aI5371
aI11115
aI47771
asS'd'
I4617
sa(dp9234
S'a'
I15110
sS's'
I16
sS'm'
(lp9235
I1
aI1
aI3
aI9
aI29
aI29
aI13
aI229
aI87
aI281
aI1119
aI1085
aI4423
aI1667
aI27067
aI50397
asS'd'
I4618
sa(dp9236
S'a'
I15116
sS's'
I16
sS'm'
(lp9237
I1
aI1
aI3
aI7
aI11
aI29
aI77
aI85
aI121
aI495
aI501
aI3209
aI3531
aI2307
aI11367
aI34135
asS'd'
I4619
sa(dp9238
S'a'
I15133
sS's'
I16
sS'm'
(lp9239
I1
aI1
aI7
aI9
aI3
aI37
aI33
aI209
aI493
aI869
aI367
aI3221
aI1643
aI3353
aI22851
aI4313
asS'd'
I4620
sa(dp9240
S'a'
I15134
sS's'
I16
sS'm'
(lp9241
I1
aI1
aI1
aI7
aI15
aI53
aI27
aI17
aI29
aI345
aI821
aI1831
aI1963
aI10089
aI5101
aI32689
asS'd'
I4621
sa(dp9242
S'a'
I15137
sS's'
I16
sS'm'
(lp9243
I1
aI1
aI3
aI9
aI9
aI61
aI31
aI215
aI497
aI591
aI1301
aI157
aI3329
aI13877
aI9017
aI34673
asS'd'
I4622
sa(dp9244
S'a'
I15147
sS's'
I16
sS'm'
(lp9245
I1
aI1
aI5
aI1
aI29
aI49
aI93
aI139
aI279
aI167
aI143
aI279
aI6145
aI6247
aI519
aI8869
asS'd'
I4623
sa(dp9246
S'a'
I15182
sS's'
I16
sS'm'
(lp9247
I1
aI3
aI3
aI1
aI25
aI41
aI81
aI219
aI505
aI335
aI1941
aI2963
aI413
aI775
aI4181
aI55269
asS'd'
I4624
sa(dp9248
S'a'
I15203
sS's'
I16
sS'm'
(lp9249
I1
aI1
aI1
aI11
aI27
aI23
aI91
aI9
aI497
aI811
aI1469
aI1999
aI5377
aI2943
aI17635
aI11151
asS'd'
I4625
sa(dp9250
S'a'
I15215
sS's'
I16
sS'm'
(lp9251
I1
aI1
aI5
aI15
aI17
aI23
aI15
aI235
aI271
aI749
aI1873
aI3805
aI5405
aI7421
aI24339
aI14351
asS'd'
I4626
sa(dp9252
S'a'
I15245
sS's'
I16
sS'm'
(lp9253
I1
aI3
aI7
aI1
aI5
aI61
aI81
aI9
aI269
aI43
aI1391
aI2711
aI6973
aI11299
aI2263
aI8715
asS'd'
I4627
sa(dp9254
S'a'
I15246
sS's'
I16
sS'm'
(lp9255
I1
aI1
aI5
aI13
aI11
aI1
aI69
aI235
aI25
aI227
aI63
aI2591
aI913
aI12555
aI6263
aI38981
asS'd'
I4628
sa(dp9256
S'a'
I15264
sS's'
I16
sS'm'
(lp9257
I1
aI3
aI1
aI7
aI17
aI7
aI97
aI251
aI149
aI959
aI1895
aI1179
aI4031
aI15975
aI20313
aI64067
asS'd'
I4629
sa(dp9258
S'a'
I15269
sS's'
I16
sS'm'
(lp9259
I1
aI3
aI7
aI15
aI3
aI17
aI85
aI229
aI149
aI925
aI585
aI3755
aI2359
aI13131
aI12665
aI28861
asS'd'
I4630
sa(dp9260
S'a'
I15296
sS's'
I16
sS'm'
(lp9261
I1
aI3
aI3
aI9
aI5
aI31
aI107
aI93
aI347
aI851
aI1249
aI2161
aI6095
aI8315
aI20259
aI39527
asS'd'
I4631
sa(dp9262
S'a'
I15314
sS's'
I16
sS'm'
(lp9263
I1
aI3
aI7
aI7
aI21
aI63
aI85
aI241
aI501
aI627
aI1211
aI1713
aI6907
aI4229
aI7557
aI55719
asS'd'
I4632
sa(dp9264
S'a'
I15323
sS's'
I16
sS'm'
(lp9265
I1
aI1
aI1
aI13
aI19
aI43
aI21
aI177
aI13
aI353
aI679
aI511
aI5565
aI993
aI25345
aI25087
asS'd'
I4633
sa(dp9266
S'a'
I15364
sS's'
I16
sS'm'
(lp9267
I1
aI3
aI3
aI15
aI21
aI15
aI87
aI83
aI381
aI547
aI1429
aI2417
aI2425
aI2097
aI20889
aI12353
asS'd'
I4634
sa(dp9268
S'a'
I15386
sS's'
I16
sS'm'
(lp9269
I1
aI3
aI1
aI11
aI23
aI21
aI69
aI147
aI427
aI271
aI1829
aI525
aI2951
aI10773
aI32425
aI17685
asS'd'
I4635
sa(dp9270
S'a'
I15391
sS's'
I16
sS'm'
(lp9271
I1
aI3
aI1
aI7
aI15
aI55
aI21
aI131
aI195
aI927
aI635
aI3505
aI3817
aI14883
aI1149
aI10079
asS'd'
I4636
sa(dp9272
S'a'
I15436
sS's'
I16
sS'm'
(lp9273
I1
aI3
aI3
aI9
aI23
aI15
aI45
aI147
aI249
aI87
aI377
aI1551
aI4343
aI15373
aI2895
aI44973
asS'd'
I4637
sa(dp9274
S'a'
I15460
sS's'
I16
sS'm'
(lp9275
I1
aI3
aI1
aI7
aI31
aI63
aI67
aI107
aI109
aI1019
aI815
aI231
aI8135
aI559
aI8175
aI21735
asS'd'
I4638
sa(dp9276
S'a'
I15464
sS's'
I16
sS'm'
(lp9277
I1
aI1
aI5
aI7
aI7
aI63
aI103
aI133
aI167
aI883
aI1647
aI2827
aI6015
aI8541
aI16963
aI37129
asS'd'
I4639
sa(dp9278
S'a'
I15469
sS's'
I16
sS'm'
(lp9279
I1
aI3
aI5
aI9
aI27
aI25
aI59
aI147
aI29
aI943
aI865
aI1233
aI2165
aI15259
aI2235
aI25831
asS'd'
I4640
sa(dp9280
S'a'
I15470
sS's'
I16
sS'm'
(lp9281
I1
aI1
aI5
aI7
aI25
aI5
aI67
aI89
aI493
aI111
aI359
aI1115
aI7963
aI6545
aI7749
aI29179
asS'd'
I4641
sa(dp9282
S'a'
I15477
sS's'
I16
sS'm'
(lp9283
I1
aI3
aI7
aI5
aI19
aI17
aI89
aI195
aI337
aI115
aI1417
aI3837
aI4761
aI1959
aI16205
aI61597
asS'd'
I4642
sa(dp9284
S'a'
I15488
sS's'
I16
sS'm'
(lp9285
I1
aI1
aI5
aI11
aI25
aI43
aI3
aI111
aI491
aI897
aI1541
aI909
aI4751
aI739
aI7827
aI64485
asS'd'
I4643
sa(dp9286
S'a'
I15494
sS's'
I16
sS'm'
(lp9287
I1
aI1
aI5
aI15
aI19
aI61
aI39
aI111
aI451
aI419
aI1657
aI2427
aI4589
aI5577
aI23967
aI19259
asS'd'
I4644
sa(dp9288
S'a'
I15548
sS's'
I16
sS'm'
(lp9289
I1
aI3
aI3
aI1
aI31
aI15
aI7
aI131
aI329
aI847
aI537
aI1775
aI3833
aI5683
aI17267
aI16389
asS'd'
I4645
sa(dp9290
S'a'
I15551
sS's'
I16
sS'm'
(lp9291
I1
aI1
aI7
aI1
aI9
aI29
aI13
aI25
aI409
aI513
aI1695
aI2175
aI5099
aI727
aI5723
aI43547
asS'd'
I4646
sa(dp9292
S'a'
I15560
sS's'
I16
sS'm'
(lp9293
I1
aI1
aI5
aI7
aI13
aI11
aI29
aI123
aI127
aI193
aI1647
aI3157
aI2149
aI16209
aI19909
aI14473
asS'd'
I4647
sa(dp9294
S'a'
I15563
sS's'
I16
sS'm'
(lp9295
I1
aI1
aI1
aI15
aI15
aI37
aI125
aI157
aI487
aI143
aI1891
aI2895
aI4397
aI10685
aI1463
aI55027
asS'd'
I4648
sa(dp9296
S'a'
I15604
sS's'
I16
sS'm'
(lp9297
I1
aI3
aI7
aI1
aI1
aI15
aI115
aI105
aI479
aI529
aI1095
aI2687
aI4483
aI15027
aI15487
aI7113
asS'd'
I4649
sa(dp9298
S'a'
I15607
sS's'
I16
sS'm'
(lp9299
I1
aI1
aI3
aI9
aI23
aI63
aI113
aI211
aI155
aI931
aI175
aI3037
aI2359
aI10413
aI24561
aI21099
asS'd'
I4650
sa(dp9300
S'a'
I15616
sS's'
I16
sS'm'
(lp9301
I1
aI3
aI3
aI11
aI5
aI15
aI13
aI37
aI257
aI447
aI203
aI545
aI2467
aI9979
aI17543
aI62703
asS'd'
I4651
sa(dp9302
S'a'
I15631
sS's'
I16
sS'm'
(lp9303
I1
aI1
aI3
aI7
aI17
aI31
aI83
aI91
aI79
aI265
aI1415
aI2347
aI5337
aI7615
aI27739
aI33841
asS'd'
I4652
sa(dp9304
S'a'
I15699
sS's'
I16
sS'm'
(lp9305
I1
aI3
aI5
aI7
aI17
aI63
aI37
aI153
aI347
aI909
aI1945
aI7
aI2071
aI15195
aI32083
aI26713
asS'd'
I4653
sa(dp9306
S'a'
I15701
sS's'
I16
sS'm'
(lp9307
I1
aI1
aI3
aI11
aI19
aI51
aI69
aI21
aI323
aI635
aI443
aI1685
aI6275
aI13787
aI20921
aI45553
asS'd'
I4654
sa(dp9308
S'a'
I15708
sS's'
I16
sS'm'
(lp9309
I1
aI3
aI3
aI7
aI15
aI35
aI67
aI247
aI257
aI429
aI2029
aI523
aI3219
aI3893
aI26677
aI53273
asS'd'
I4655
sa(dp9310
S'a'
I15739
sS's'
I16
sS'm'
(lp9311
I1
aI1
aI7
aI9
aI9
aI3
aI119
aI121
aI445
aI149
aI1539
aI1887
aI2995
aI14867
aI809
aI48065
asS'd'
I4656
sa(dp9312
S'a'
I15746
sS's'
I16
sS'm'
(lp9313
I1
aI3
aI5
aI13
aI15
aI27
aI75
aI9
aI217
aI35
aI1363
aI2383
aI4357
aI1153
aI20565
aI62277
asS'd'
I4657
sa(dp9314
S'a'
I15772
sS's'
I16
sS'm'
(lp9315
I1
aI1
aI7
aI1
aI21
aI1
aI11
aI53
aI331
aI765
aI407
aI453
aI2725
aI11199
aI645
aI14915
asS'd'
I4658
sa(dp9316
S'a'
I15793
sS's'
I16
sS'm'
(lp9317
I1
aI1
aI5
aI1
aI29
aI11
aI5
aI159
aI395
aI53
aI323
aI1347
aI5529
aI8525
aI24003
aI20535
asS'd'
I4659
sa(dp9318
S'a'
I15832
sS's'
I16
sS'm'
(lp9319
I1
aI3
aI3
aI15
aI9
aI19
aI87
aI181
aI391
aI639
aI703
aI611
aI997
aI359
aI2471
aI58465
asS'd'
I4660
sa(dp9320
S'a'
I15837
sS's'
I16
sS'm'
(lp9321
I1
aI3
aI5
aI9
aI27
aI9
aI117
aI47
aI223
aI509
aI1537
aI1235
aI3885
aI6767
aI17131
aI63421
asS'd'
I4661
sa(dp9322
S'a'
I15866
sS's'
I16
sS'm'
(lp9323
I1
aI1
aI5
aI1
aI15
aI15
aI113
aI67
aI477
aI597
aI1795
aI3065
aI4565
aI3609
aI16419
aI19667
asS'd'
I4662
sa(dp9324
S'a'
I15899
sS's'
I16
sS'm'
(lp9325
I1
aI1
aI7
aI11
aI1
aI63
aI33
aI211
aI271
aI773
aI499
aI2309
aI1303
aI14015
aI30377
aI53195
asS'd'
I4663
sa(dp9326
S'a'
I15911
sS's'
I16
sS'm'
(lp9327
I1
aI1
aI7
aI11
aI5
aI23
aI17
aI183
aI321
aI315
aI203
aI3371
aI907
aI9331
aI21031
aI33765
asS'd'
I4664
sa(dp9328
S'a'
I15918
sS's'
I16
sS'm'
(lp9329
I1
aI3
aI3
aI7
aI7
aI53
aI111
aI69
aI441
aI283
aI195
aI2415
aI7293
aI7659
aI2731
aI5417
asS'd'
I4665
sa(dp9330
S'a'
I15952
sS's'
I16
sS'm'
(lp9331
I1
aI3
aI5
aI15
aI3
aI61
aI5
aI241
aI427
aI463
aI1729
aI653
aI7671
aI10979
aI7247
aI36931
asS'd'
I4666
sa(dp9332
S'a'
I15962
sS's'
I16
sS'm'
(lp9333
I1
aI3
aI1
aI9
aI3
aI5
aI105
aI117
aI465
aI853
aI2005
aI3925
aI4419
aI4441
aI3701
aI50747
asS'd'
I4667
sa(dp9334
S'a'
I15967
sS's'
I16
sS'm'
(lp9335
I1
aI1
aI3
aI7
aI1
aI3
aI3
aI149
aI65
aI405
aI299
aI99
aI481
aI14323
aI11565
aI6227
asS'd'
I4668
sa(dp9336
S'a'
I15973
sS's'
I16
sS'm'
(lp9337
I1
aI3
aI7
aI5
aI29
aI3
aI19
aI3
aI253
aI895
aI879
aI2435
aI2151
aI10673
aI11013
aI43055
asS'd'
I4669
sa(dp9338
S'a'
I15977
sS's'
I16
sS'm'
(lp9339
I1
aI3
aI1
aI11
aI15
aI57
aI127
aI197
aI319
aI913
aI1039
aI811
aI7767
aI5791
aI31725
aI8733
asS'd'
I4670
sa(dp9340
S'a'
I16016
sS's'
I16
sS'm'
(lp9341
I1
aI1
aI7
aI3
aI13
aI25
aI25
aI81
aI229
aI185
aI39
aI2789
aI579
aI4973
aI28617
aI60871
asS'd'
I4671
sa(dp9342
S'a'
I16035
sS's'
I16
sS'm'
(lp9343
I1
aI1
aI7
aI3
aI25
aI17
aI41
aI7
aI103
aI269
aI55
aI2651
aI7579
aI10935
aI8917
aI14323
asS'd'
I4672
sa(dp9344
S'a'
I16044
sS's'
I16
sS'm'
(lp9345
I1
aI3
aI7
aI7
aI13
aI7
aI99
aI205
aI293
aI877
aI1893
aI3013
aI2389
aI6021
aI2645
aI18175
asS'd'
I4673
sa(dp9346
S'a'
I16067
sS's'
I16
sS'm'
(lp9347
I1
aI1
aI3
aI7
aI9
aI39
aI59
aI187
aI191
aI761
aI339
aI3381
aI2921
aI5197
aI16963
aI43019
asS'd'
I4674
sa(dp9348
S'a'
I16082
sS's'
I16
sS'm'
(lp9349
I1
aI3
aI3
aI13
aI7
aI23
aI41
aI203
aI311
aI981
aI323
aI1675
aI6689
aI579
aI3885
aI64475
asS'd'
I4675
sa(dp9350
S'a'
I16084
sS's'
I16
sS'm'
(lp9351
I1
aI3
aI5
aI15
aI21
aI39
aI35
aI193
aI167
aI1009
aI493
aI829
aI6571
aI1299
aI13061
aI1163
asS'd'
I4676
sa(dp9352
S'a'
I16098
sS's'
I16
sS'm'
(lp9353
I1
aI1
aI3
aI5
aI3
aI19
aI123
aI123
aI111
aI599
aI193
aI3419
aI7173
aI5595
aI12449
aI52247
asS'd'
I4677
sa(dp9354
S'a'
I16107
sS's'
I16
sS'm'
(lp9355
I1
aI3
aI5
aI11
aI9
aI25
aI65
aI49
aI239
aI953
aI481
aI3455
aI4335
aI305
aI22469
aI11949
asS'd'
I4678
sa(dp9356
S'a'
I16142
sS's'
I16
sS'm'
(lp9357
I1
aI1
aI3
aI7
aI15
aI1
aI13
aI77
aI147
aI49
aI1445
aI931
aI3405
aI15951
aI15215
aI26451
asS'd'
I4679
sa(dp9358
S'a'
I16149
sS's'
I16
sS'm'
(lp9359
I1
aI3
aI1
aI1
aI21
aI53
aI17
aI7
aI247
aI243
aI805
aI795
aI489
aI13351
aI13493
aI30937
asS'd'
I4680
sa(dp9360
S'a'
I16165
sS's'
I16
sS'm'
(lp9361
I1
aI3
aI7
aI5
aI5
aI13
aI39
aI115
aI397
aI757
aI423
aI2559
aI1677
aI9449
aI24563
aI869
asS'd'
I4681
sa(dp9362
S'a'
I16172
sS's'
I16
sS'm'
(lp9363
I1
aI1
aI3
aI11
aI23
aI9
aI27
aI233
aI165
aI853
aI1721
aI599
aI551
aI11657
aI27623
aI40119
asS'd'
I4682
sa(dp9364
S'a'
I16178
sS's'
I16
sS'm'
(lp9365
I1
aI1
aI3
aI1
aI3
aI47
aI75
aI207
aI113
aI417
aI1317
aI2215
aI2395
aI1841
aI23125
aI50401
asS'd'
I4683
sa(dp9366
S'a'
I16197
sS's'
I16
sS'm'
(lp9367
I1
aI3
aI3
aI1
aI13
aI55
aI103
aI55
aI351
aI785
aI1665
aI3603
aI7705
aI4811
aI21129
aI38115
asS'd'
I4684
sa(dp9368
S'a'
I16201
sS's'
I16
sS'm'
(lp9369
I1
aI1
aI1
aI5
aI5
aI49
aI93
aI189
aI317
aI701
aI1545
aI1017
aI4133
aI7655
aI19827
aI52155
asS'd'
I4685
sa(dp9370
S'a'
I16215
sS's'
I16
sS'm'
(lp9371
I1
aI3
aI3
aI13
aI17
aI37
aI7
aI249
aI139
aI529
aI235
aI3801
aI7871
aI459
aI15127
aI13231
asS'd'
I4686
sa(dp9372
S'a'
I16221
sS's'
I16
sS'm'
(lp9373
I1
aI3
aI7
aI5
aI7
aI63
aI99
aI241
aI131
aI455
aI1287
aI3539
aI8029
aI12661
aI23313
aI54029
asS'd'
I4687
sa(dp9374
S'a'
I16226
sS's'
I16
sS'm'
(lp9375
I1
aI3
aI1
aI3
aI29
aI63
aI51
aI227
aI497
aI685
aI1351
aI449
aI7851
aI10815
aI17379
aI62097
asS'd'
I4688
sa(dp9376
S'a'
I16232
sS's'
I16
sS'm'
(lp9377
I1
aI3
aI1
aI11
aI25
aI61
aI73
aI29
aI467
aI533
aI855
aI853
aI5557
aI10953
aI18307
aI27135
asS'd'
I4689
sa(dp9378
S'a'
I16246
sS's'
I16
sS'm'
(lp9379
I1
aI1
aI7
aI3
aI13
aI49
aI63
aI171
aI177
aI283
aI1365
aI3087
aI5445
aI15109
aI12523
aI25193
asS'd'
I4690
sa(dp9380
S'a'
I16261
sS's'
I16
sS'm'
(lp9381
I1
aI3
aI5
aI15
aI9
aI39
aI95
aI81
aI417
aI199
aI1653
aI3673
aI2663
aI8101
aI12199
aI22759
asS'd'
I4691
sa(dp9382
S'a'
I16279
sS's'
I16
sS'm'
(lp9383
I1
aI1
aI3
aI9
aI29
aI15
aI29
aI215
aI21
aI721
aI245
aI1197
aI7251
aI5721
aI6735
aI7751
asS'd'
I4692
sa(dp9384
S'a'
I16280
sS's'
I16
sS'm'
(lp9385
I1
aI3
aI5
aI5
aI21
aI7
aI81
aI61
aI157
aI707
aI819
aI1689
aI4203
aI5559
aI25483
aI43325
asS'd'
I4693
sa(dp9386
S'a'
I16290
sS's'
I16
sS'm'
(lp9387
I1
aI1
aI7
aI13
aI15
aI51
aI47
aI197
aI269
aI921
aI353
aI2865
aI6227
aI537
aI20015
aI53823
asS'd'
I4694
sa(dp9388
S'a'
I16314
sS's'
I16
sS'm'
(lp9389
I1
aI1
aI3
aI5
aI13
aI25
aI91
aI221
aI111
aI587
aI1119
aI2343
aI4651
aI4641
aI15915
aI36323
asS'd'
I4695
sa(dp9390
S'a'
I16345
sS's'
I16
sS'm'
(lp9391
I1
aI1
aI7
aI11
aI1
aI45
aI7
aI215
aI483
aI545
aI731
aI3041
aI3121
aI8681
aI20651
aI4069
asS'd'
I4696
sa(dp9392
S'a'
I16355
sS's'
I16
sS'm'
(lp9393
I1
aI3
aI7
aI13
aI13
aI27
aI109
aI65
aI103
aI805
aI299
aI2069
aI6017
aI14565
aI20505
aI16161
asS'd'
I4697
sa(dp9394
S'a'
I16361
sS's'
I16
sS'm'
(lp9395
I1
aI1
aI7
aI5
aI11
aI33
aI105
aI213
aI237
aI583
aI1033
aI2333
aI845
aI6493
aI505
aI2563
asS'd'
I4698
sa(dp9396
S'a'
I16393
sS's'
I16
sS'm'
(lp9397
I1
aI1
aI5
aI7
aI3
aI5
aI11
aI173
aI373
aI341
aI269
aI177
aI3175
aI9413
aI601
aI13591
asS'd'
I4699
sa(dp9398
S'a'
I16394
sS's'
I16
sS'm'
(lp9399
I1
aI1
aI5
aI13
aI7
aI57
aI61
aI187
aI121
aI405
aI665
aI111
aI7535
aI3355
aI14051
aI511
asS'd'
I4700
sa(dp9400
S'a'
I16438
sS's'
I16
sS'm'
(lp9401
I1
aI1
aI1
aI3
aI3
aI29
aI15
aI253
aI227
aI123
aI333
aI1343
aI7313
aI1955
aI17741
aI30831
asS'd'
I4701
sa(dp9402
S'a'
I16450
sS's'
I16
sS'm'
(lp9403
I1
aI3
aI5
aI1
aI5
aI47
aI65
aI183
aI199
aI839
aI925
aI2711
aI4609
aI201
aI15177
aI29817
asS'd'
I4702
sa(dp9404
S'a'
I16507
sS's'
I16
sS'm'
(lp9405
I1
aI3
aI7
aI9
aI21
aI63
aI5
aI163
aI265
aI581
aI1795
aI3937
aI4641
aI2073
aI32225
aI60831
asS'd'
I4703
sa(dp9406
S'a'
I16523
sS's'
I16
sS'm'
(lp9407
I1
aI1
aI1
aI5
aI7
aI47
aI125
aI203
aI511
aI841
aI1937
aI3431
aI1495
aI12827
aI5893
aI19265
asS'd'
I4704
sa(dp9408
S'a'
I16533
sS's'
I16
sS'm'
(lp9409
I1
aI1
aI5
aI5
aI9
aI49
aI17
aI247
aI391
aI241
aI3
aI2253
aI6319
aI89
aI4449
aI6371
asS'd'
I4705
sa(dp9410
S'a'
I16603
sS's'
I16
sS'm'
(lp9411
I1
aI3
aI1
aI1
aI31
aI7
aI51
aI61
aI461
aI391
aI273
aI1609
aI5825
aI16029
aI3851
aI39213
asS'd'
I4706
sa(dp9412
S'a'
I16648
sS's'
I16
sS'm'
(lp9413
I1
aI3
aI3
aI7
aI29
aI21
aI65
aI75
aI317
aI925
aI1319
aI3827
aI965
aI5685
aI17007
aI64365
asS'd'
I4707
sa(dp9414
S'a'
I16653
sS's'
I16
sS'm'
(lp9415
I1
aI3
aI1
aI5
aI23
aI23
aI109
aI59
aI31
aI659
aI635
aI2209
aI857
aI9847
aI2507
aI18325
asS'd'
I4708
sa(dp9416
S'a'
I16672
sS's'
I16
sS'm'
(lp9417
I1
aI1
aI1
aI1
aI17
aI51
aI53
aI77
aI461
aI147
aI229
aI2821
aI2405
aI1259
aI1121
aI17429
asS'd'
I4709
sa(dp9418
S'a'
I16682
sS's'
I16
sS'm'
(lp9419
I1
aI3
aI5
aI3
aI31
aI3
aI57
aI157
aI321
aI731
aI1609
aI2139
aI899
aI12599
aI19803
aI51083
asS'd'
I4710
sa(dp9420
S'a'
I16709
sS's'
I16
sS'm'
(lp9421
I1
aI1
aI3
aI11
aI27
aI43
aI73
aI209
aI431
aI587
aI1247
aI2803
aI3593
aI1351
aI18701
aI33423
asS'd'
I4711
sa(dp9422
S'a'
I16713
sS's'
I16
sS'm'
(lp9423
I1
aI3
aI5
aI13
aI27
aI19
aI67
aI245
aI339
aI879
aI2035
aI1801
aI5845
aI3883
aI22057
aI15771
asS'd'
I4712
sa(dp9424
S'a'
I16719
sS's'
I16
sS'm'
(lp9425
I1
aI1
aI3
aI11
aI11
aI55
aI93
aI51
aI57
aI127
aI1325
aI3975
aI3989
aI2347
aI18831
aI2979
asS'd'
I4713
sa(dp9426
S'a'
I16733
sS's'
I16
sS'm'
(lp9427
I1
aI1
aI1
aI13
aI17
aI1
aI17
aI103
aI303
aI777
aI1973
aI2943
aI7439
aI8953
aI27227
aI10229
asS'd'
I4714
sa(dp9428
S'a'
I16740
sS's'
I16
sS'm'
(lp9429
I1
aI3
aI3
aI15
aI1
aI41
aI53
aI219
aI415
aI399
aI995
aI205
aI7719
aI10937
aI31879
aI755
asS'd'
I4715
sa(dp9430
S'a'
I16761
sS's'
I16
sS'm'
(lp9431
I1
aI3
aI7
aI9
aI13
aI7
aI99
aI93
aI419
aI1019
aI1605
aI161
aI3831
aI9147
aI7877
aI1333
asS'd'
I4716
sa(dp9432
S'a'
I16767
sS's'
I16
sS'm'
(lp9433
I1
aI3
aI7
aI15
aI5
aI51
aI37
aI115
aI259
aI549
aI353
aI2067
aI7657
aI1283
aI20333
aI2325
asS'd'
I4717
sa(dp9434
S'a'
I16771
sS's'
I16
sS'm'
(lp9435
I1
aI1
aI3
aI3
aI23
aI33
aI63
aI233
aI363
aI719
aI1099
aI471
aI3079
aI10577
aI19063
aI31535
asS'd'
I4718
sa(dp9436
S'a'
I16788
sS's'
I16
sS'm'
(lp9437
I1
aI3
aI7
aI15
aI23
aI19
aI109
aI105
aI497
aI881
aI1055
aI537
aI4607
aI239
aI22785
aI60201
asS'd'
I4719
sa(dp9438
S'a'
I16811
sS's'
I16
sS'm'
(lp9439
I1
aI3
aI3
aI5
aI19
aI11
aI1
aI207
aI163
aI437
aI713
aI667
aI1427
aI7505
aI28055
aI43101
asS'd'
I4720
sa(dp9440
S'a'
I16814
sS's'
I16
sS'm'
(lp9441
I1
aI3
aI5
aI5
aI25
aI45
aI75
aI9
aI109
aI545
aI573
aI2685
aI1013
aI2973
aI18619
aI55945
asS'd'
I4721
sa(dp9442
S'a'
I16816
sS's'
I16
sS'm'
(lp9443
I1
aI1
aI1
aI3
aI27
aI27
aI39
aI33
aI285
aI453
aI613
aI2707
aI2155
aI4355
aI21105
aI7969
asS'd'
I4722
sa(dp9444
S'a'
I16828
sS's'
I16
sS'm'
(lp9445
I1
aI3
aI3
aI9
aI1
aI31
aI71
aI101
aI491
aI409
aI65
aI1479
aI5743
aI525
aI2863
aI53657
asS'd'
I4723
sa(dp9446
S'a'
I16834
sS's'
I16
sS'm'
(lp9447
I1
aI1
aI3
aI1
aI17
aI63
aI55
aI11
aI125
aI447
aI275
aI2243
aI6827
aI5753
aI32401
aI13819
asS'd'
I4724
sa(dp9448
S'a'
I16863
sS's'
I16
sS'm'
(lp9449
I1
aI1
aI3
aI9
aI21
aI47
aI5
aI127
aI285
aI471
aI1681
aI945
aI6141
aI5651
aI10273
aI30811
asS'd'
I4725
sa(dp9450
S'a'
I16864
sS's'
I16
sS'm'
(lp9451
I1
aI3
aI3
aI1
aI13
aI53
aI91
aI3
aI255
aI429
aI107
aI2937
aI2971
aI10251
aI15009
aI37477
asS'd'
I4726
sa(dp9452
S'a'
I16879
sS's'
I16
sS'm'
(lp9453
I1
aI1
aI7
aI13
aI21
aI63
aI73
aI3
aI63
aI491
aI101
aI1981
aI7457
aI879
aI6243
aI22275
asS'd'
I4727
sa(dp9454
S'a'
I16888
sS's'
I16
sS'm'
(lp9455
I1
aI3
aI1
aI1
aI11
aI43
aI121
aI101
aI293
aI639
aI645
aI2723
aI2075
aI3629
aI22105
aI18199
asS'd'
I4728
sa(dp9456
S'a'
I16904
sS's'
I16
sS'm'
(lp9457
I1
aI1
aI3
aI1
aI31
aI9
aI69
aI97
aI511
aI663
aI1147
aI1237
aI389
aI255
aI8661
aI38533
asS'd'
I4729
sa(dp9458
S'a'
I16909
sS's'
I16
sS'm'
(lp9459
I1
aI3
aI3
aI7
aI3
aI13
aI23
aI71
aI197
aI439
aI2003
aI1771
aI8073
aI1549
aI29089
aI5409
asS'd'
I4730
sa(dp9460
S'a'
I16921
sS's'
I16
sS'm'
(lp9461
I1
aI3
aI1
aI1
aI9
aI23
aI1
aI221
aI159
aI699
aI593
aI3385
aI3869
aI10105
aI22097
aI34753
asS'd'
I4731
sa(dp9462
S'a'
I16934
sS's'
I16
sS'm'
(lp9463
I1
aI1
aI7
aI1
aI29
aI47
aI41
aI137
aI333
aI357
aI325
aI3151
aI6641
aI3823
aI8763
aI28327
asS'd'
I4732
sa(dp9464
S'a'
I16951
sS's'
I16
sS'm'
(lp9465
I1
aI3
aI1
aI7
aI19
aI19
aI39
aI225
aI477
aI619
aI583
aI229
aI6177
aI9615
aI1203
aI13711
asS'd'
I4733
sa(dp9466
S'a'
I16983
sS's'
I16
sS'm'
(lp9467
I1
aI1
aI3
aI13
aI9
aI41
aI127
aI147
aI13
aI301
aI861
aI2019
aI3517
aI1147
aI21587
aI42387
asS'd'
I4734
sa(dp9468
S'a'
I16999
sS's'
I16
sS'm'
(lp9469
I1
aI1
aI5
aI11
aI9
aI63
aI11
aI121
aI251
aI199
aI483
aI2287
aI4667
aI3887
aI10611
aI6019
asS'd'
I4735
sa(dp9470
S'a'
I17000
sS's'
I16
sS'm'
(lp9471
I1
aI1
aI3
aI13
aI23
aI19
aI89
aI73
aI355
aI399
aI749
aI687
aI5707
aI11443
aI817
aI38967
asS'd'
I4736
sa(dp9472
S'a'
I17006
sS's'
I16
sS'm'
(lp9473
I1
aI3
aI5
aI9
aI3
aI23
aI115
aI207
aI373
aI541
aI73
aI1285
aI7245
aI12505
aI5787
aI61207
asS'd'
I4737
sa(dp9474
S'a'
I17020
sS's'
I16
sS'm'
(lp9475
I1
aI3
aI5
aI15
aI27
aI37
aI115
aI203
aI195
aI793
aI1577
aI1283
aI7299
aI4025
aI5319
aI5375
asS'd'
I4738
sa(dp9476
S'a'
I17030
sS's'
I16
sS'm'
(lp9477
I1
aI1
aI7
aI15
aI25
aI19
aI61
aI11
aI215
aI771
aI1057
aI451
aI1965
aI13693
aI25617
aI42657
asS'd'
I4739
sa(dp9478
S'a'
I17033
sS's'
I16
sS'm'
(lp9479
I1
aI3
aI3
aI7
aI1
aI19
aI23
aI217
aI175
aI901
aI2009
aI2989
aI5111
aI5027
aI4805
aI37843
asS'd'
I4740
sa(dp9480
S'a'
I17044
sS's'
I16
sS'm'
(lp9481
I1
aI3
aI1
aI11
aI11
aI37
aI3
aI131
aI459
aI769
aI201
aI3979
aI3009
aI8691
aI27005
aI32175
asS'd'
I4741
sa(dp9482
S'a'
I17051
sS's'
I16
sS'm'
(lp9483
I1
aI3
aI5
aI7
aI27
aI27
aI117
aI23
aI403
aI1003
aI1501
aI785
aI6313
aI10187
aI5085
aI30751
asS'd'
I4742
sa(dp9484
S'a'
I17072
sS's'
I16
sS'm'
(lp9485
I1
aI1
aI7
aI3
aI11
aI41
aI73
aI151
aI19
aI657
aI131
aI1901
aI3879
aI14995
aI24085
aI56621
asS'd'
I4743
sa(dp9486
S'a'
I17078
sS's'
I16
sS'm'
(lp9487
I1
aI3
aI7
aI15
aI23
aI3
aI61
aI199
aI67
aI483
aI1961
aI3583
aI5937
aI5749
aI16685
aI11831
asS'd'
I4744
sa(dp9488
S'a'
I17084
sS's'
I16
sS'm'
(lp9489
I1
aI1
aI3
aI15
aI25
aI15
aI97
aI9
aI299
aI641
aI883
aI2901
aI123
aI1523
aI7055
aI15609
asS'd'
I4745
sa(dp9490
S'a'
I17087
sS's'
I16
sS'm'
(lp9491
I1
aI3
aI5
aI5
aI31
aI55
aI19
aI45
aI239
aI543
aI2005
aI1041
aI1711
aI11059
aI19927
aI17475
asS'd'
I4746
sa(dp9492
S'a'
I17090
sS's'
I16
sS'm'
(lp9493
I1
aI1
aI3
aI9
aI5
aI59
aI105
aI209
aI323
aI613
aI1963
aI2227
aI2947
aI11761
aI21375
aI13265
asS'd'
I4747
sa(dp9494
S'a'
I17123
sS's'
I16
sS'm'
(lp9495
I1
aI3
aI3
aI15
aI1
aI5
aI117
aI37
aI93
aI243
aI305
aI2299
aI5163
aI9205
aI28761
aI35987
asS'd'
I4748
sa(dp9496
S'a'
I17132
sS's'
I16
sS'm'
(lp9497
I1
aI1
aI1
aI5
aI5
aI29
aI13
aI147
aI457
aI187
aI1729
aI1547
aI7745
aI13155
aI7833
aI57449
asS'd'
I4749
sa(dp9498
S'a'
I17140
sS's'
I16
sS'm'
(lp9499
I1
aI3
aI3
aI13
aI1
aI51
aI49
aI253
aI23
aI389
aI1611
aI3045
aI5909
aI3947
aI25105
aI3327
asS'd'
I4750
sa(dp9500
S'a'
I17149
sS's'
I16
sS'm'
(lp9501
I1
aI3
aI1
aI11
aI15
aI47
aI19
aI15
aI231
aI57
aI763
aI1879
aI1765
aI14861
aI22893
aI19437
asS'd'
I4751
sa(dp9502
S'a'
I17157
sS's'
I16
sS'm'
(lp9503
I1
aI1
aI3
aI15
aI1
aI19
aI85
aI65
aI139
aI629
aI361
aI3513
aI3807
aI799
aI8349
aI29247
asS'd'
I4752
sa(dp9504
S'a'
I17164
sS's'
I16
sS'm'
(lp9505
I1
aI3
aI3
aI13
aI9
aI11
aI65
aI201
aI471
aI89
aI355
aI121
aI3947
aI10775
aI3599
aI6041
asS'd'
I4753
sa(dp9506
S'a'
I17170
sS's'
I16
sS'm'
(lp9507
I1
aI3
aI7
aI3
aI5
aI53
aI33
aI167
aI431
aI129
aI1449
aI3263
aI7691
aI12569
aI7551
aI41101
asS'd'
I4754
sa(dp9508
S'a'
I17179
sS's'
I16
sS'm'
(lp9509
I1
aI1
aI3
aI15
aI9
aI41
aI5
aI239
aI361
aI773
aI955
aI3663
aI6051
aI12889
aI5841
aI59615
asS'd'
I4755
sa(dp9510
S'a'
I17237
sS's'
I16
sS'm'
(lp9511
I1
aI1
aI7
aI5
aI5
aI33
aI97
aI9
aI495
aI845
aI1953
aI3533
aI5715
aI15055
aI25211
aI9351
asS'd'
I4756
sa(dp9512
S'a'
I17248
sS's'
I16
sS'm'
(lp9513
I1
aI3
aI1
aI11
aI25
aI37
aI83
aI153
aI289
aI739
aI353
aI1121
aI7641
aI2081
aI28439
aI38085
asS'd'
I4757
sa(dp9514
S'a'
I17260
sS's'
I16
sS'm'
(lp9515
I1
aI3
aI1
aI1
aI27
aI9
aI63
aI135
aI395
aI641
aI1759
aI3727
aI4371
aI5193
aI2783
aI12389
asS'd'
I4758
sa(dp9516
S'a'
I17272
sS's'
I16
sS'm'
(lp9517
I1
aI3
aI3
aI15
aI3
aI9
aI5
aI153
aI111
aI675
aI1957
aI3817
aI4269
aI10787
aI3413
aI34199
asS'd'
I4759
sa(dp9518
S'a'
I17275
sS's'
I16
sS'm'
(lp9519
I1
aI3
aI5
aI9
aI23
aI23
aI97
aI137
aI255
aI249
aI333
aI2329
aI1055
aI13769
aI13109
aI33443
asS'd'
I4760
sa(dp9520
S'a'
I17287
sS's'
I16
sS'm'
(lp9521
I1
aI1
aI1
aI15
aI7
aI37
aI99
aI219
aI483
aI755
aI263
aI3523
aI6179
aI4817
aI29873
aI12771
asS'd'
I4761
sa(dp9522
S'a'
I17294
sS's'
I16
sS'm'
(lp9523
I1
aI1
aI3
aI5
aI23
aI7
aI77
aI97
aI105
aI631
aI175
aI1911
aI7271
aI1009
aI24081
aI61207
asS'd'
I4762
sa(dp9524
S'a'
I17296
sS's'
I16
sS'm'
(lp9525
I1
aI3
aI5
aI3
aI1
aI31
aI71
aI91
aI265
aI669
aI1839
aI3989
aI8127
aI15001
aI1419
aI8895
asS'd'
I4763
sa(dp9526
S'a'
I17305
sS's'
I16
sS'm'
(lp9527
I1
aI3
aI1
aI13
aI27
aI51
aI93
aI155
aI49
aI991
aI653
aI203
aI3863
aI5363
aI31969
aI36083
asS'd'
I4764
sa(dp9528
S'a'
I17312
sS's'
I16
sS'm'
(lp9529
I1
aI3
aI3
aI7
aI31
aI27
aI21
aI73
aI21
aI675
aI407
aI1215
aI2963
aI6799
aI15259
aI13125
asS'd'
I4765
sa(dp9530
S'a'
I17321
sS's'
I16
sS'm'
(lp9531
I1
aI3
aI5
aI13
aI5
aI53
aI19
aI215
aI243
aI487
aI689
aI2519
aI6393
aI3987
aI30847
aI37919
asS'd'
I4766
sa(dp9532
S'a'
I17367
sS's'
I16
sS'm'
(lp9533
I1
aI3
aI3
aI7
aI5
aI31
aI115
aI231
aI255
aI955
aI2023
aI1487
aI6575
aI9873
aI22585
aI29951
asS'd'
I4767
sa(dp9534
S'a'
I17368
sS's'
I16
sS'm'
(lp9535
I1
aI3
aI5
aI11
aI11
aI57
aI109
aI203
aI417
aI29
aI1311
aI893
aI1047
aI2413
aI9305
aI38219
asS'd'
I4768
sa(dp9536
S'a'
I17378
sS's'
I16
sS'm'
(lp9537
I1
aI3
aI1
aI7
aI23
aI51
aI113
aI3
aI105
aI915
aI1145
aI3431
aI7331
aI3323
aI31669
aI21485
asS'd'
I4769
sa(dp9538
S'a'
I17433
sS's'
I16
sS'm'
(lp9539
I1
aI1
aI7
aI13
aI9
aI29
aI119
aI205
aI403
aI1023
aI257
aI863
aI2983
aI1895
aI16539
aI23385
asS'd'
I4770
sa(dp9540
S'a'
I17455
sS's'
I16
sS'm'
(lp9541
I1
aI1
aI7
aI13
aI27
aI21
aI47
aI139
aI341
aI509
aI1107
aI2197
aI3649
aI14301
aI30789
aI48783
asS'd'
I4771
sa(dp9542
S'a'
I17457
sS's'
I16
sS'm'
(lp9543
I1
aI3
aI3
aI7
aI25
aI19
aI99
aI11
aI309
aI919
aI1809
aI3015
aI1587
aI3779
aI1289
aI30207
asS'd'
I4772
sa(dp9544
S'a'
I17508
sS's'
I16
sS'm'
(lp9545
I1
aI3
aI5
aI11
aI9
aI43
aI57
aI171
aI9
aI151
aI173
aI2301
aI7723
aI2083
aI319
aI52883
asS'd'
I4773
sa(dp9546
S'a'
I17559
sS's'
I16
sS'm'
(lp9547
I1
aI1
aI3
aI1
aI3
aI13
aI63
aI11
aI231
aI117
aI1257
aI237
aI581
aI13871
aI15501
aI8741
asS'd'
I4774
sa(dp9548
S'a'
I17560
sS's'
I16
sS'm'
(lp9549
I1
aI3
aI5
aI9
aI13
aI63
aI55
aI155
aI291
aI721
aI123
aI929
aI3351
aI11651
aI12513
aI1779
asS'd'
I4775
sa(dp9550
S'a'
I17582
sS's'
I16
sS'm'
(lp9551
I1
aI3
aI7
aI3
aI31
aI5
aI61
aI81
aI465
aI639
aI1363
aI3157
aI2401
aI9513
aI32559
aI34477
asS'd'
I4776
sa(dp9552
S'a'
I17596
sS's'
I16
sS'm'
(lp9553
I1
aI3
aI1
aI15
aI27
aI25
aI3
aI117
aI277
aI13
aI707
aI3825
aI7287
aI10181
aI30127
aI57247
asS'd'
I4777
sa(dp9554
S'a'
I17599
sS's'
I16
sS'm'
(lp9555
I1
aI1
aI7
aI11
aI21
aI21
aI53
aI17
aI407
aI851
aI1191
aI285
aI6815
aI1595
aI25507
aI8099
asS'd'
I4778
sa(dp9556
S'a'
I17613
sS's'
I16
sS'm'
(lp9557
I1
aI3
aI5
aI9
aI9
aI61
aI83
aI61
aI65
aI671
aI63
aI311
aI6897
aI15327
aI29809
aI4899
asS'd'
I4779
sa(dp9558
S'a'
I17619
sS's'
I16
sS'm'
(lp9559
I1
aI1
aI7
aI1
aI21
aI45
aI99
aI235
aI477
aI461
aI1119
aI4087
aI2057
aI14861
aI31969
aI24357
asS'd'
I4780
sa(dp9560
S'a'
I17622
sS's'
I16
sS'm'
(lp9561
I1
aI1
aI7
aI9
aI31
aI9
aI65
aI123
aI281
aI273
aI1059
aI1625
aI6265
aI9635
aI11563
aI45053
asS'd'
I4781
sa(dp9562
S'a'
I17655
sS's'
I16
sS'm'
(lp9563
I1
aI3
aI3
aI7
aI1
aI41
aI15
aI23
aI43
aI727
aI1271
aI1741
aI765
aI13265
aI4145
aI1335
asS'd'
I4782
sa(dp9564
S'a'
I17661
sS's'
I16
sS'm'
(lp9565
I1
aI1
aI3
aI7
aI17
aI55
aI107
aI231
aI263
aI197
aI659
aI3621
aI2789
aI5171
aI28635
aI13595
asS'd'
I4783
sa(dp9566
S'a'
I17698
sS's'
I16
sS'm'
(lp9567
I1
aI1
aI5
aI1
aI27
aI23
aI13
aI83
aI431
aI507
aI1571
aI1573
aI1733
aI12171
aI8181
aI30843
asS'd'
I4784
sa(dp9568
S'a'
I17712
sS's'
I16
sS'm'
(lp9569
I1
aI3
aI7
aI11
aI1
aI53
aI107
aI39
aI497
aI579
aI453
aI1339
aI1415
aI10317
aI2741
aI34599
asS'd'
I4785
sa(dp9570
S'a'
I17715
sS's'
I16
sS'm'
(lp9571
I1
aI3
aI3
aI5
aI31
aI41
aI49
aI41
aI33
aI665
aI1783
aI87
aI317
aI6603
aI22603
aI22879
asS'd'
I4786
sa(dp9572
S'a'
I17721
sS's'
I16
sS'm'
(lp9573
I1
aI3
aI1
aI15
aI5
aI47
aI41
aI87
aI231
aI853
aI1615
aI2299
aI4643
aI9249
aI15641
aI14323
asS'd'
I4787
sa(dp9574
S'a'
I17722
sS's'
I16
sS'm'
(lp9575
I1
aI3
aI7
aI9
aI5
aI45
aI55
aI153
aI31
aI247
aI67
aI2335
aI6057
aI4379
aI27579
aI38437
asS'd'
I4788
sa(dp9576
S'a'
I17742
sS's'
I16
sS'm'
(lp9577
I1
aI1
aI5
aI7
aI9
aI3
aI73
aI81
aI479
aI909
aI1097
aI3945
aI4485
aI7815
aI22855
aI20825
asS'd'
I4789
sa(dp9578
S'a'
I17778
sS's'
I16
sS'm'
(lp9579
I1
aI3
aI1
aI15
aI19
aI43
aI97
aI57
aI339
aI167
aI135
aI1777
aI7681
aI9715
aI13863
aI6347
asS'd'
I4790
sa(dp9580
S'a'
I17818
sS's'
I16
sS'm'
(lp9581
I1
aI1
aI1
aI1
aI5
aI53
aI33
aI123
aI449
aI165
aI1283
aI2977
aI5919
aI12929
aI32073
aI61851
asS'd'
I4791
sa(dp9582
S'a'
I17836
sS's'
I16
sS'm'
(lp9583
I1
aI1
aI5
aI15
aI27
aI27
aI19
aI157
aI267
aI651
aI1319
aI3841
aI7739
aI9947
aI16801
aI41325
asS'd'
I4792
sa(dp9584
S'a'
I17841
sS's'
I16
sS'm'
(lp9585
I1
aI3
aI7
aI9
aI19
aI7
aI83
aI95
aI401
aI293
aI437
aI1983
aI119
aI7553
aI11097
aI11925
asS'd'
I4793
sa(dp9586
S'a'
I17856
sS's'
I16
sS'm'
(lp9587
I1
aI1
aI3
aI5
aI21
aI1
aI53
aI201
aI385
aI843
aI1801
aI99
aI2697
aI9659
aI28789
aI31417
asS'd'
I4794
sa(dp9588
S'a'
I17883
sS's'
I16
sS'm'
(lp9589
I1
aI1
aI5
aI5
aI29
aI57
aI103
aI89
aI77
aI597
aI1849
aI3433
aI4267
aI11167
aI3841
aI44023
asS'd'
I4795
sa(dp9590
S'a'
I17896
sS's'
I16
sS'm'
(lp9591
I1
aI1
aI7
aI1
aI21
aI47
aI113
aI253
aI249
aI431
aI1899
aI2859
aI7345
aI5725
aI14805
aI19999
asS'd'
I4796
sa(dp9592
S'a'
I17901
sS's'
I16
sS'm'
(lp9593
I1
aI3
aI3
aI5
aI1
aI11
aI77
aI213
aI359
aI665
aI1855
aI2743
aI2407
aI14677
aI17957
aI63257
asS'd'
I4797
sa(dp9594
S'a'
I17926
sS's'
I16
sS'm'
(lp9595
I1
aI3
aI7
aI13
aI23
aI29
aI127
aI183
aI275
aI849
aI1005
aI3159
aI3867
aI13029
aI7527
aI13035
asS'd'
I4798
sa(dp9596
S'a'
I17937
sS's'
I16
sS'm'
(lp9597
I1
aI1
aI1
aI15
aI29
aI47
aI81
aI225
aI77
aI879
aI1279
aI1929
aI1457
aI2025
aI32229
aI2847
asS'd'
I4799
sa(dp9598
S'a'
I17992
sS's'
I16
sS'm'
(lp9599
I1
aI1
aI1
aI3
aI29
aI45
aI37
aI189
aI217
aI53
aI281
aI1785
aI1929
aI763
aI5875
aI34303
asS'd'
I4800
sa(dp9600
S'a'
I17995
sS's'
I16
sS'm'
(lp9601
I1
aI3
aI1
aI9
aI21
aI61
aI21
aI149
aI215
aI13
aI1221
aI5
aI7153
aI14089
aI3119
aI33115
asS'd'
I4801
sa(dp9602
S'a'
I17998
sS's'
I16
sS'm'
(lp9603
I1
aI3
aI7
aI11
aI7
aI57
aI89
aI185
aI485
aI649
aI1765
aI747
aI2983
aI6791
aI25015
aI13627
asS'd'
I4802
sa(dp9604
S'a'
I18021
sS's'
I16
sS'm'
(lp9605
I1
aI1
aI1
aI9
aI11
aI53
aI77
aI203
aI425
aI237
aI487
aI2317
aI1047
aI8277
aI23405
aI30445
asS'd'
I4803
sa(dp9606
S'a'
I18039
sS's'
I16
sS'm'
(lp9607
I1
aI1
aI3
aI5
aI7
aI29
aI39
aI195
aI109
aI381
aI655
aI931
aI4469
aI16215
aI10627
aI64171
asS'd'
I4804
sa(dp9608
S'a'
I18067
sS's'
I16
sS'm'
(lp9609
I1
aI3
aI1
aI3
aI5
aI9
aI89
aI131
aI509
aI275
aI489
aI3161
aI3701
aI11951
aI6579
aI42839
asS'd'
I4805
sa(dp9610
S'a'
I18122
sS's'
I16
sS'm'
(lp9611
I1
aI3
aI7
aI13
aI15
aI37
aI65
aI91
aI305
aI433
aI1815
aI169
aI3117
aI47
aI30331
aI34863
asS'd'
I4806
sa(dp9612
S'a'
I18129
sS's'
I16
sS'm'
(lp9613
I1
aI3
aI3
aI9
aI21
aI3
aI21
aI113
aI25
aI833
aI1579
aI4021
aI3481
aI55
aI20833
aI46277
asS'd'
I4807
sa(dp9614
S'a'
I18130
sS's'
I16
sS'm'
(lp9615
I1
aI1
aI1
aI5
aI19
aI37
aI61
aI229
aI61
aI363
aI817
aI1235
aI6235
aI7261
aI29917
aI43057
asS'd'
I4808
sa(dp9616
S'a'
I18142
sS's'
I16
sS'm'
(lp9617
I1
aI3
aI1
aI9
aI7
aI59
aI119
aI189
aI341
aI945
aI633
aI3683
aI2589
aI15453
aI4989
aI40055
asS'd'
I4809
sa(dp9618
S'a'
I18148
sS's'
I16
sS'm'
(lp9619
I1
aI1
aI1
aI5
aI25
aI63
aI61
aI73
aI207
aI205
aI1011
aI2857
aI8137
aI2121
aI26731
aI46011
asS'd'
I4810
sa(dp9620
S'a'
I18163
sS's'
I16
sS'm'
(lp9621
I1
aI3
aI7
aI11
aI13
aI59
aI107
aI57
aI49
aI555
aI441
aI1771
aI4939
aI12107
aI8263
aI16243
asS'd'
I4811
sa(dp9622
S'a'
I18192
sS's'
I16
sS'm'
(lp9623
I1
aI3
aI5
aI13
aI15
aI49
aI89
aI217
aI83
aI225
aI2001
aI2727
aI4651
aI619
aI16473
aI47525
asS'd'
I4812
sa(dp9624
S'a'
I18211
sS's'
I16
sS'm'
(lp9625
I1
aI3
aI5
aI9
aI5
aI63
aI115
aI91
aI337
aI757
aI703
aI559
aI1683
aI14875
aI30769
aI30331
asS'd'
I4813
sa(dp9626
S'a'
I18228
sS's'
I16
sS'm'
(lp9627
I1
aI3
aI1
aI15
aI3
aI3
aI119
aI79
aI487
aI519
aI523
aI1177
aI7105
aI12343
aI24671
aI31711
asS'd'
I4814
sa(dp9628
S'a'
I18264
sS's'
I16
sS'm'
(lp9629
I1
aI1
aI7
aI15
aI25
aI63
aI87
aI23
aI59
aI277
aI849
aI953
aI4567
aI11309
aI26181
aI749
asS'd'
I4815
sa(dp9630
S'a'
I18347
sS's'
I16
sS'm'
(lp9631
I1
aI3
aI7
aI15
aI5
aI33
aI21
aI127
aI3
aI239
aI839
aI997
aI7253
aI8183
aI19779
aI4185
asS'd'
I4816
sa(dp9632
S'a'
I18372
sS's'
I16
sS'm'
(lp9633
I1
aI1
aI5
aI15
aI25
aI37
aI99
aI51
aI465
aI137
aI1339
aI541
aI4319
aI9883
aI17425
aI30743
asS'd'
I4817
sa(dp9634
S'a'
I18409
sS's'
I16
sS'm'
(lp9635
I1
aI3
aI3
aI5
aI13
aI7
aI3
aI249
aI365
aI749
aI527
aI3675
aI3005
aI12905
aI9621
aI899
asS'd'
I4818
sa(dp9636
S'a'
I18412
sS's'
I16
sS'm'
(lp9637
I1
aI3
aI3
aI7
aI29
aI31
aI69
aI21
aI365
aI1021
aI1329
aI2623
aI3549
aI5491
aI21293
aI63771
asS'd'
I4819
sa(dp9638
S'a'
I18418
sS's'
I16
sS'm'
(lp9639
I1
aI1
aI5
aI9
aI5
aI35
aI53
aI247
aI193
aI17
aI227
aI381
aI5233
aI821
aI3991
aI4439
asS'd'
I4820
sa(dp9640
S'a'
I18423
sS's'
I16
sS'm'
(lp9641
I1
aI1
aI7
aI15
aI5
aI59
aI27
aI167
aI489
aI335
aI1565
aI2999
aI1777
aI5009
aI519
aI57967
asS'd'
I4821
sa(dp9642
S'a'
I18433
sS's'
I16
sS'm'
(lp9643
I1
aI1
aI1
aI11
aI25
aI47
aI23
aI155
aI419
aI863
aI1905
aI355
aI1089
aI5871
aI10149
aI53341
asS'd'
I4822
sa(dp9644
S'a'
I18439
sS's'
I16
sS'm'
(lp9645
I1
aI1
aI7
aI7
aI29
aI55
aI127
aI83
aI33
aI309
aI2017
aI1021
aI5987
aI1101
aI13657
aI60587
asS'd'
I4823
sa(dp9646
S'a'
I18445
sS's'
I16
sS'm'
(lp9647
I1
aI1
aI1
aI7
aI3
aI1
aI9
aI75
aI257
aI407
aI659
aI529
aI2087
aI14759
aI14483
aI36425
asS'd'
I4824
sa(dp9648
S'a'
I18451
sS's'
I16
sS'm'
(lp9649
I1
aI3
aI7
aI3
aI11
aI29
aI113
aI255
aI301
aI799
aI1171
aI3721
aI135
aI3467
aI7109
aI50339
asS'd'
I4825
sa(dp9650
S'a'
I18467
sS's'
I16
sS'm'
(lp9651
I1
aI1
aI1
aI7
aI21
aI25
aI15
aI31
aI61
aI491
aI57
aI189
aI2033
aI4363
aI31295
aI16313
asS'd'
I4826
sa(dp9652
S'a'
I18502
sS's'
I16
sS'm'
(lp9653
I1
aI1
aI5
aI1
aI5
aI17
aI33
aI77
aI483
aI469
aI355
aI2245
aI4165
aI459
aI30411
aI29507
asS'd'
I4827
sa(dp9654
S'a'
I18514
sS's'
I16
sS'm'
(lp9655
I1
aI1
aI3
aI13
aI1
aI27
aI29
aI85
aI491
aI787
aI1157
aI1299
aI4743
aI14795
aI32587
aI12807
asS'd'
I4828
sa(dp9656
S'a'
I18547
sS's'
I16
sS'm'
(lp9657
I1
aI3
aI3
aI1
aI23
aI45
aI35
aI129
aI3
aI55
aI969
aI2387
aI3929
aI10397
aI19879
aI2723
asS'd'
I4829
sa(dp9658
S'a'
I18575
sS's'
I16
sS'm'
(lp9659
I1
aI1
aI1
aI7
aI19
aI3
aI9
aI23
aI497
aI347
aI2039
aI913
aI5925
aI7965
aI5789
aI40949
asS'd'
I4830
sa(dp9660
S'a'
I18593
sS's'
I16
sS'm'
(lp9661
I1
aI1
aI7
aI1
aI29
aI61
aI89
aI3
aI133
aI647
aI1585
aI2661
aI1875
aI1859
aI3937
aI12707
asS'd'
I4831
sa(dp9662
S'a'
I18613
sS's'
I16
sS'm'
(lp9663
I1
aI3
aI3
aI7
aI25
aI11
aI41
aI149
aI427
aI463
aI901
aI2433
aI2617
aI13511
aI3443
aI39867
asS'd'
I4832
sa(dp9664
S'a'
I18620
sS's'
I16
sS'm'
(lp9665
I1
aI1
aI1
aI5
aI27
aI33
aI103
aI251
aI201
aI1023
aI1979
aI3745
aI6365
aI14945
aI22153
aI55535
asS'd'
I4833
sa(dp9666
S'a'
I18637
sS's'
I16
sS'm'
(lp9667
I1
aI3
aI1
aI15
aI23
aI25
aI57
aI215
aI111
aI181
aI385
aI1123
aI3095
aI7085
aI31863
aI37393
asS'd'
I4834
sa(dp9668
S'a'
I18640
sS's'
I16
sS'm'
(lp9669
I1
aI3
aI5
aI13
aI17
aI35
aI27
aI159
aI255
aI241
aI413
aI1823
aI5329
aI1825
aI28489
aI58763
asS'd'
I4835
sa(dp9670
S'a'
I18712
sS's'
I16
sS'm'
(lp9671
I1
aI3
aI1
aI15
aI3
aI33
aI97
aI27
aI409
aI889
aI409
aI2315
aI4743
aI14827
aI3037
aI57149
asS'd'
I4836
sa(dp9672
S'a'
I18728
sS's'
I16
sS'm'
(lp9673
I1
aI1
aI3
aI5
aI19
aI63
aI93
aI51
aI233
aI715
aI1571
aI1101
aI2751
aI14805
aI25683
aI13323
asS'd'
I4837
sa(dp9674
S'a'
I18742
sS's'
I16
sS'm'
(lp9675
I1
aI3
aI7
aI3
aI21
aI15
aI117
aI179
aI263
aI229
aI199
aI2597
aI3999
aI3037
aI3801
aI4775
asS'd'
I4838
sa(dp9676
S'a'
I18748
sS's'
I16
sS'm'
(lp9677
I1
aI3
aI3
aI15
aI1
aI15
aI49
aI91
aI383
aI21
aI1955
aI773
aI1213
aI5971
aI19525
aI8699
asS'd'
I4839
sa(dp9678
S'a'
I18753
sS's'
I16
sS'm'
(lp9679
I1
aI3
aI1
aI15
aI29
aI49
aI11
aI101
aI261
aI761
aI709
aI3811
aI4055
aI15675
aI32305
aI15173
asS'd'
I4840
sa(dp9680
S'a'
I18756
sS's'
I16
sS'm'
(lp9681
I1
aI1
aI1
aI3
aI9
aI41
aI127
aI23
aI413
aI461
aI237
aI1595
aI2257
aI2971
aI31845
aI61485
asS'd'
I4841
sa(dp9682
S'a'
I18771
sS's'
I16
sS'm'
(lp9683
I1
aI1
aI1
aI11
aI23
aI13
aI63
aI21
aI23
aI209
aI1317
aI1071
aI3619
aI7275
aI9343
aI21455
asS'd'
I4842
sa(dp9684
S'a'
I18814
sS's'
I16
sS'm'
(lp9685
I1
aI1
aI5
aI9
aI31
aI35
aI41
aI249
aI477
aI569
aI1175
aI1571
aI4679
aI10337
aI3507
aI23415
asS'd'
I4843
sa(dp9686
S'a'
I18818
sS's'
I16
sS'm'
(lp9687
I1
aI3
aI5
aI11
aI29
aI3
aI117
aI65
aI301
aI913
aI1709
aI1765
aI1801
aI15513
aI31495
aI38131
asS'd'
I4844
sa(dp9688
S'a'
I18827
sS's'
I16
sS'm'
(lp9689
I1
aI3
aI5
aI11
aI27
aI3
aI71
aI195
aI81
aI313
aI505
aI3941
aI3797
aI2031
aI24151
aI65085
asS'd'
I4845
sa(dp9690
S'a'
I18835
sS's'
I16
sS'm'
(lp9691
I1
aI1
aI1
aI5
aI13
aI17
aI59
aI151
aI191
aI489
aI1267
aI3207
aI4495
aI15145
aI12789
aI46313
asS'd'
I4846
sa(dp9692
S'a'
I18842
sS's'
I16
sS'm'
(lp9693
I1
aI3
aI1
aI7
aI29
aI9
aI25
aI31
aI33
aI527
aI1939
aI4089
aI333
aI757
aI8895
aI25331
asS'd'
I4847
sa(dp9694
S'a'
I18854
sS's'
I16
sS'm'
(lp9695
I1
aI1
aI1
aI1
aI9
aI27
aI11
aI205
aI211
aI141
aI1171
aI1881
aI4029
aI10587
aI30103
aI39661
asS'd'
I4848
sa(dp9696
S'a'
I18858
sS's'
I16
sS'm'
(lp9697
I1
aI1
aI3
aI3
aI23
aI3
aI23
aI175
aI355
aI753
aI183
aI1331
aI6403
aI3369
aI29891
aI11109
asS'd'
I4849
sa(dp9698
S'a'
I18895
sS's'
I16
sS'm'
(lp9699
I1
aI3
aI7
aI3
aI17
aI25
aI95
aI145
aI135
aI525
aI1073
aI1841
aI3951
aI2027
aI23053
aI19699
asS'd'
I4850
sa(dp9700
S'a'
I18914
sS's'
I16
sS'm'
(lp9701
I1
aI1
aI5
aI3
aI11
aI43
aI117
aI67
aI299
aI885
aI1095
aI2777
aI8185
aI14331
aI29543
aI655
asS'd'
I4851
sa(dp9702
S'a'
I18925
sS's'
I16
sS'm'
(lp9703
I1
aI3
aI7
aI7
aI3
aI59
aI127
aI147
aI323
aI713
aI99
aI1179
aI6395
aI1821
aI12673
aI35587
asS'd'
I4852
sa(dp9704
S'a'
I18933
sS's'
I16
sS'm'
(lp9705
I1
aI3
aI5
aI3
aI7
aI11
aI33
aI87
aI99
aI967
aI1443
aI1585
aI6215
aI15125
aI30747
aI21513
asS'd'
I4853
sa(dp9706
S'a'
I18937
sS's'
I16
sS'm'
(lp9707
I1
aI3
aI7
aI11
aI23
aI5
aI91
aI171
aI229
aI601
aI833
aI3157
aI1691
aI15081
aI20607
aI7643
asS'd'
I4854
sa(dp9708
S'a'
I18944
sS's'
I16
sS'm'
(lp9709
I1
aI1
aI3
aI1
aI5
aI1
aI39
aI59
aI157
aI7
aI601
aI2079
aI3045
aI3693
aI26511
aI13245
asS'd'
I4855
sa(dp9710
S'a'
I18973
sS's'
I16
sS'm'
(lp9711
I1
aI3
aI5
aI9
aI9
aI27
aI83
aI135
aI185
aI379
aI2027
aI1407
aI7409
aI7363
aI26471
aI35907
asS'd'
I4856
sa(dp9712
S'a'
I19001
sS's'
I16
sS'm'
(lp9713
I1
aI3
aI7
aI15
aI29
aI49
aI1
aI69
aI383
aI915
aI183
aI3809
aI4511
aI8751
aI4715
aI7033
asS'd'
I4857
sa(dp9714
S'a'
I19012
sS's'
I16
sS'm'
(lp9715
I1
aI3
aI3
aI3
aI1
aI17
aI71
aI233
aI243
aI933
aI1165
aI3089
aI3565
aI6521
aI13203
aI13065
asS'd'
I4858
sa(dp9716
S'a'
I19016
sS's'
I16
sS'm'
(lp9717
I1
aI1
aI5
aI9
aI9
aI55
aI53
aI129
aI331
aI943
aI587
aI2573
aI2247
aI15101
aI4987
aI62983
asS'd'
I4859
sa(dp9718
S'a'
I19027
sS's'
I16
sS'm'
(lp9719
I1
aI3
aI1
aI13
aI11
aI43
aI45
aI127
aI129
aI857
aI669
aI321
aI3915
aI4477
aI26973
aI19911
asS'd'
I4860
sa(dp9720
S'a'
I19040
sS's'
I16
sS'm'
(lp9721
I1
aI3
aI1
aI13
aI15
aI3
aI83
aI23
aI13
aI441
aI953
aI2373
aI3539
aI4895
aI26327
aI61961
asS'd'
I4861
sa(dp9722
S'a'
I19074
sS's'
I16
sS'm'
(lp9723
I1
aI1
aI5
aI13
aI23
aI11
aI125
aI83
aI339
aI901
aI1809
aI2691
aI3789
aI15007
aI10569
aI65399
asS'd'
I4862
sa(dp9724
S'a'
I19085
sS's'
I16
sS'm'
(lp9725
I1
aI3
aI1
aI1
aI17
aI27
aI105
aI199
aI435
aI245
aI1227
aI3029
aI3911
aI1021
aI2931
aI24957
asS'd'
I4863
sa(dp9726
S'a'
I19093
sS's'
I16
sS'm'
(lp9727
I1
aI3
aI1
aI11
aI17
aI5
aI123
aI39
aI413
aI627
aI1149
aI3925
aI6635
aI8959
aI31387
aI65047
asS'd'
I4864
sa(dp9728
S'a'
I19100
sS's'
I16
sS'm'
(lp9729
I1
aI3
aI5
aI1
aI23
aI41
aI93
aI217
aI21
aI115
aI1311
aI3901
aI2559
aI2925
aI30755
aI7575
asS'd'
I4865
sa(dp9730
S'a'
I19107
sS's'
I16
sS'm'
(lp9731
I1
aI1
aI3
aI9
aI13
aI11
aI63
aI171
aI135
aI983
aI1679
aI395
aI7349
aI5153
aI26405
aI40589
asS'd'
I4866
sa(dp9732
S'a'
I19128
sS's'
I16
sS'm'
(lp9733
I1
aI3
aI7
aI13
aI27
aI47
aI53
aI169
aI85
aI871
aI1087
aI619
aI7399
aI9719
aI6349
aI59211
asS'd'
I4867
sa(dp9734
S'a'
I19141
sS's'
I16
sS'm'
(lp9735
I1
aI3
aI3
aI15
aI31
aI45
aI3
aI33
aI11
aI879
aI929
aI1977
aI1939
aI1049
aI16159
aI41515
asS'd'
I4868
sa(dp9736
S'a'
I19142
sS's'
I16
sS'm'
(lp9737
I1
aI3
aI5
aI11
aI9
aI27
aI13
aI23
aI127
aI747
aI1121
aI2497
aI8141
aI8601
aI12431
aI3243
asS'd'
I4869
sa(dp9738
S'a'
I19156
sS's'
I16
sS'm'
(lp9739
I1
aI3
aI7
aI15
aI23
aI43
aI23
aI225
aI283
aI13
aI1837
aI2089
aI6435
aI10121
aI22307
aI58767
asS'd'
I4870
sa(dp9740
S'a'
I19169
sS's'
I16
sS'm'
(lp9741
I1
aI1
aI5
aI11
aI17
aI3
aI41
aI221
aI143
aI669
aI261
aI1367
aI7813
aI15265
aI32751
aI62007
asS'd'
I4871
sa(dp9742
S'a'
I19176
sS's'
I16
sS'm'
(lp9743
I1
aI1
aI1
aI1
aI5
aI45
aI41
aI161
aI327
aI185
aI1403
aI485
aI2831
aI10025
aI12555
aI47661
asS'd'
I4872
sa(dp9744
S'a'
I19222
sS's'
I16
sS'm'
(lp9745
I1
aI3
aI7
aI1
aI31
aI55
aI87
aI83
aI439
aI929
aI653
aI4095
aI5443
aI7361
aI27801
aI12979
asS'd'
I4873
sa(dp9746
S'a'
I19226
sS's'
I16
sS'm'
(lp9747
I1
aI3
aI1
aI7
aI1
aI57
aI11
aI145
aI55
aI269
aI711
aI1889
aI8023
aI7171
aI3247
aI35691
asS'd'
I4874
sa(dp9748
S'a'
I19247
sS's'
I16
sS'm'
(lp9749
I1
aI1
aI1
aI3
aI15
aI1
aI15
aI131
aI479
aI797
aI815
aI2343
aI1603
aI10775
aI21341
aI20825
asS'd'
I4875
sa(dp9750
S'a'
I19259
sS's'
I16
sS'm'
(lp9751
I1
aI3
aI5
aI9
aI3
aI27
aI31
aI117
aI441
aI177
aI215
aI3991
aI3197
aI8397
aI19195
aI3883
asS'd'
I4876
sa(dp9752
S'a'
I19262
sS's'
I16
sS'm'
(lp9753
I1
aI1
aI5
aI13
aI1
aI19
aI13
aI27
aI157
aI1001
aI43
aI251
aI7997
aI7495
aI16515
aI44287
asS'd'
I4877
sa(dp9754
S'a'
I19291
sS's'
I16
sS'm'
(lp9755
I1
aI1
aI3
aI5
aI17
aI57
aI117
aI53
aI413
aI905
aI551
aI149
aI7531
aI14885
aI32493
aI34961
asS'd'
I4878
sa(dp9756
S'a'
I19309
sS's'
I16
sS'm'
(lp9757
I1
aI3
aI3
aI7
aI27
aI1
aI7
aI13
aI259
aI21
aI189
aI451
aI6171
aI3603
aI12053
aI45619
asS'd'
I4879
sa(dp9758
S'a'
I19324
sS's'
I16
sS'm'
(lp9759
I1
aI1
aI7
aI11
aI5
aI41
aI59
aI119
aI419
aI93
aI1399
aI629
aI1269
aI3789
aI17035
aI61583
asS'd'
I4880
sa(dp9760
S'a'
I19334
sS's'
I16
sS'm'
(lp9761
I1
aI3
aI5
aI11
aI1
aI11
aI59
aI83
aI473
aI273
aI839
aI3111
aI3081
aI10159
aI6143
aI16297
asS'd'
I4881
sa(dp9762
S'a'
I19338
sS's'
I16
sS'm'
(lp9763
I1
aI1
aI5
aI15
aI25
aI15
aI17
aI63
aI275
aI927
aI189
aI89
aI6595
aI4399
aI27201
aI57205
asS'd'
I4882
sa(dp9764
S'a'
I19343
sS's'
I16
sS'm'
(lp9765
I1
aI1
aI7
aI3
aI31
aI17
aI63
aI203
aI321
aI655
aI1751
aI2985
aI3291
aI11567
aI15135
aI49747
asS'd'
I4883
sa(dp9766
S'a'
I19376
sS's'
I16
sS'm'
(lp9767
I1
aI3
aI5
aI13
aI27
aI25
aI89
aI39
aI299
aI833
aI1269
aI271
aI6481
aI3045
aI7203
aI20279
asS'd'
I4884
sa(dp9768
S'a'
I19408
sS's'
I16
sS'm'
(lp9769
I1
aI3
aI1
aI1
aI29
aI13
aI13
aI37
aI33
aI563
aI1825
aI3257
aI3153
aI963
aI25801
aI15861
asS'd'
I4885
sa(dp9770
S'a'
I19413
sS's'
I16
sS'm'
(lp9771
I1
aI3
aI5
aI11
aI15
aI7
aI49
aI117
aI479
aI221
aI579
aI2995
aI1673
aI14927
aI2869
aI35547
asS'd'
I4886
sa(dp9772
S'a'
I19420
sS's'
I16
sS'm'
(lp9773
I1
aI3
aI1
aI11
aI31
aI11
aI77
aI161
aI183
aI187
aI1967
aI3037
aI4441
aI10547
aI1443
aI8619
asS'd'
I4887
sa(dp9774
S'a'
I19441
sS's'
I16
sS'm'
(lp9775
I1
aI1
aI3
aI11
aI27
aI41
aI83
aI179
aI293
aI421
aI1395
aI1673
aI6375
aI9801
aI14265
aI18117
asS'd'
I4888
sa(dp9776
S'a'
I19444
sS's'
I16
sS'm'
(lp9777
I1
aI1
aI3
aI7
aI9
aI19
aI55
aI235
aI499
aI637
aI1121
aI3583
aI8007
aI3749
aI20903
aI6179
asS'd'
I4889
sa(dp9778
S'a'
I19454
sS's'
I16
sS'm'
(lp9779
I1
aI3
aI7
aI13
aI9
aI55
aI125
aI77
aI395
aI9
aI2005
aI2247
aI1609
aI6805
aI13099
aI26367
asS'd'
I4890
sa(dp9780
S'a'
I19461
sS's'
I16
sS'm'
(lp9781
I1
aI3
aI5
aI13
aI9
aI41
aI49
aI133
aI443
aI995
aI209
aI341
aI8013
aI11037
aI29663
aI21161
asS'd'
I4891
sa(dp9782
S'a'
I19473
sS's'
I16
sS'm'
(lp9783
I1
aI3
aI1
aI1
aI1
aI47
aI45
aI243
aI161
aI433
aI129
aI39
aI4885
aI8777
aI6395
aI16953
asS'd'
I4892
sa(dp9784
S'a'
I19479
sS's'
I16
sS'm'
(lp9785
I1
aI3
aI3
aI15
aI11
aI13
aI41
aI113
aI279
aI657
aI763
aI2411
aI7115
aI463
aI10759
aI50493
asS'd'
I4893
sa(dp9786
S'a'
I19489
sS's'
I16
sS'm'
(lp9787
I1
aI1
aI5
aI5
aI31
aI5
aI25
aI181
aI385
aI445
aI625
aI313
aI4983
aI11763
aI6065
aI63835
asS'd'
I4894
sa(dp9788
S'a'
I19591
sS's'
I16
sS'm'
(lp9789
I1
aI3
aI3
aI15
aI13
aI25
aI103
aI5
aI205
aI223
aI1327
aI73
aI677
aI6751
aI2071
aI24963
asS'd'
I4895
sa(dp9790
S'a'
I19605
sS's'
I16
sS'm'
(lp9791
I1
aI1
aI7
aI15
aI21
aI61
aI21
aI11
aI47
aI775
aI113
aI991
aI1943
aI1459
aI18049
aI45025
asS'd'
I4896
sa(dp9792
S'a'
I19616
sS's'
I16
sS'm'
(lp9793
I1
aI3
aI3
aI1
aI11
aI43
aI27
aI89
aI49
aI711
aI173
aI181
aI1261
aI2751
aI21321
aI5467
asS'd'
I4897
sa(dp9794
S'a'
I19619
sS's'
I16
sS'm'
(lp9795
I1
aI3
aI3
aI7
aI17
aI7
aI57
aI61
aI87
aI973
aI985
aI1849
aI559
aI7319
aI11457
aI46071
asS'd'
I4898
sa(dp9796
S'a'
I19653
sS's'
I16
sS'm'
(lp9797
I1
aI1
aI1
aI1
aI9
aI37
aI99
aI157
aI423
aI189
aI1355
aI3983
aI6357
aI10825
aI26517
aI45815
asS'd'
I4899
sa(dp9798
S'a'
I19654
sS's'
I16
sS'm'
(lp9799
I1
aI1
aI3
aI11
aI23
aI33
aI57
aI55
aI59
aI831
aI339
aI725
aI359
aI14859
aI17523
aI36149
asS'd'
I4900
sa(dp9800
S'a'
I19681
sS's'
I16
sS'm'
(lp9801
I1
aI1
aI5
aI5
aI27
aI29
aI47
aI147
aI153
aI801
aI1737
aI3617
aI5447
aI8011
aI30631
aI7921
asS'd'
I4901
sa(dp9802
S'a'
I19711
sS's'
I16
sS'm'
(lp9803
I1
aI1
aI5
aI1
aI11
aI43
aI35
aI105
aI69
aI453
aI1023
aI875
aI6755
aI6015
aI12449
aI50235
asS'd'
I4902
sa(dp9804
S'a'
I19719
sS's'
I16
sS'm'
(lp9805
I1
aI3
aI1
aI5
aI29
aI31
aI43
aI89
aI369
aI891
aI1447
aI353
aI8103
aI2555
aI1197
aI64005
asS'd'
I4903
sa(dp9806
S'a'
I19726
sS's'
I16
sS'm'
(lp9807
I1
aI3
aI3
aI9
aI21
aI33
aI117
aI205
aI473
aI289
aI1699
aI2361
aI7083
aI13001
aI24127
aI48611
asS'd'
I4904
sa(dp9808
S'a'
I19738
sS's'
I16
sS'm'
(lp9809
I1
aI3
aI3
aI15
aI3
aI23
aI79
aI139
aI475
aI511
aI181
aI1331
aI6137
aI2653
aI14071
aI16947
asS'd'
I4905
sa(dp9810
S'a'
I19767
sS's'
I16
sS'm'
(lp9811
I1
aI3
aI3
aI1
aI11
aI47
aI51
aI217
aI305
aI599
aI1609
aI237
aI4537
aI5377
aI717
aI13269
asS'd'
I4906
sa(dp9812
S'a'
I19819
sS's'
I16
sS'm'
(lp9813
I1
aI1
aI7
aI3
aI19
aI31
aI1
aI173
aI509
aI817
aI785
aI1223
aI5585
aI8911
aI643
aI44575
asS'd'
I4907
sa(dp9814
S'a'
I19864
sS's'
I16
sS'm'
(lp9815
I1
aI1
aI3
aI11
aI5
aI11
aI31
aI129
aI269
aI369
aI1833
aI2885
aI441
aI11517
aI2323
aI26735
asS'd'
I4908
sa(dp9816
S'a'
I19867
sS's'
I16
sS'm'
(lp9817
I1
aI1
aI5
aI9
aI7
aI51
aI31
aI21
aI5
aI157
aI541
aI2939
aI4569
aI1679
aI17467
aI27995
asS'd'
I4909
sa(dp9818
S'a'
I19885
sS's'
I16
sS'm'
(lp9819
I1
aI1
aI7
aI3
aI21
aI33
aI85
aI213
aI41
aI851
aI1947
aI3205
aI5065
aI6079
aI30789
aI63677
asS'd'
I4910
sa(dp9820
S'a'
I19894
sS's'
I16
sS'm'
(lp9821
I1
aI1
aI5
aI3
aI9
aI53
aI3
aI179
aI157
aI407
aI537
aI1193
aI4645
aI13791
aI28153
aI11349
asS'd'
I4911
sa(dp9822
S'a'
I19900
sS's'
I16
sS'm'
(lp9823
I1
aI1
aI7
aI13
aI25
aI61
aI9
aI151
aI263
aI143
aI1583
aI2859
aI6617
aI8861
aI4163
aI40695
asS'd'
I4912
sa(dp9824
S'a'
I19903
sS's'
I16
sS'm'
(lp9825
I1
aI1
aI1
aI1
aI7
aI25
aI19
aI207
aI335
aI1019
aI1919
aI3893
aI831
aI12421
aI4667
aI38967
asS'd'
I4913
sa(dp9826
S'a'
I19941
sS's'
I16
sS'm'
(lp9827
I1
aI3
aI5
aI3
aI5
aI51
aI81
aI9
aI425
aI333
aI451
aI2569
aI2771
aI12145
aI26065
aI14385
asS'd'
I4914
sa(dp9828
S'a'
I19951
sS's'
I16
sS'm'
(lp9829
I1
aI1
aI3
aI7
aI3
aI49
aI17
aI147
aI327
aI331
aI1197
aI7
aI3703
aI15247
aI9723
aI52819
asS'd'
I4915
sa(dp9830
S'a'
I19959
sS's'
I16
sS'm'
(lp9831
I1
aI3
aI3
aI7
aI27
aI21
aI117
aI229
aI255
aI603
aI195
aI1049
aI6243
aI13593
aI14553
aI8267
asS'd'
I4916
sa(dp9832
S'a'
I19966
sS's'
I16
sS'm'
(lp9833
I1
aI1
aI5
aI15
aI9
aI53
aI1
aI187
aI79
aI151
aI321
aI1883
aI6105
aI13879
aI8201
aI53213
asS'd'
I4917
sa(dp9834
S'a'
I20009
sS's'
I16
sS'm'
(lp9835
I1
aI1
aI1
aI7
aI21
aI27
aI103
aI147
aI351
aI901
aI1927
aI2145
aI2685
aI453
aI15173
aI7371
asS'd'
I4918
sa(dp9836
S'a'
I20018
sS's'
I16
sS'm'
(lp9837
I1
aI1
aI3
aI5
aI21
aI27
aI125
aI77
aI395
aI27
aI827
aI3617
aI6033
aI1511
aI29461
aI18907
asS'd'
I4919
sa(dp9838
S'a'
I20020
sS's'
I16
sS'm'
(lp9839
I1
aI3
aI5
aI3
aI3
aI27
aI75
aI129
aI441
aI831
aI1213
aI2499
aI5829
aI12181
aI7991
aI58167
asS'd'
I4920
sa(dp9840
S'a'
I20038
sS's'
I16
sS'm'
(lp9841
I1
aI1
aI1
aI9
aI3
aI33
aI85
aI135
aI45
aI405
aI1731
aI551
aI1251
aI7895
aI3975
aI41621
asS'd'
I4921
sa(dp9842
S'a'
I20049
sS's'
I16
sS'm'
(lp9843
I1
aI3
aI5
aI7
aI19
aI19
aI25
aI7
aI477
aI569
aI1089
aI2705
aI6157
aI10279
aI14029
aI36229
asS'd'
I4922
sa(dp9844
S'a'
I20066
sS's'
I16
sS'm'
(lp9845
I1
aI3
aI7
aI3
aI5
aI19
aI99
aI137
aI67
aI361
aI2021
aI2891
aI1957
aI14961
aI22673
aI45707
asS'd'
I4923
sa(dp9846
S'a'
I20108
sS's'
I16
sS'm'
(lp9847
I1
aI3
aI7
aI1
aI21
aI11
aI81
aI225
aI151
aI235
aI1761
aI3875
aI7427
aI11213
aI27023
aI17945
asS'd'
I4924
sa(dp9848
S'a'
I20130
sS's'
I16
sS'm'
(lp9849
I1
aI1
aI7
aI1
aI3
aI1
aI3
aI123
aI39
aI769
aI1467
aI1907
aI7833
aI2099
aI19
aI54653
asS'd'
I4925
sa(dp9850
S'a'
I20132
sS's'
I16
sS'm'
(lp9851
I1
aI1
aI1
aI3
aI25
aI35
aI33
aI111
aI407
aI497
aI1527
aI3999
aI3083
aI6221
aI8319
aI56959
asS'd'
I4926
sa(dp9852
S'a'
I20167
sS's'
I16
sS'm'
(lp9853
I1
aI1
aI3
aI15
aI21
aI49
aI113
aI11
aI191
aI801
aI1835
aI3413
aI3379
aI13129
aI3655
aI23885
asS'd'
I4927
sa(dp9854
S'a'
I20219
sS's'
I16
sS'm'
(lp9855
I1
aI3
aI1
aI5
aI21
aI57
aI87
aI133
aI409
aI325
aI569
aI2099
aI8143
aI315
aI23287
aI21905
asS'd'
I4928
sa(dp9856
S'a'
I20227
sS's'
I16
sS'm'
(lp9857
I1
aI1
aI5
aI5
aI21
aI43
aI25
aI169
aI265
aI123
aI81
aI2683
aI6137
aI7341
aI16383
aI26435
asS'd'
I4929
sa(dp9858
S'a'
I20263
sS's'
I16
sS'm'
(lp9859
I1
aI3
aI1
aI5
aI23
aI17
aI125
aI173
aI3
aI829
aI693
aI751
aI8021
aI3701
aI32643
aI35405
asS'd'
I4930
sa(dp9860
S'a'
I20267
sS's'
I16
sS'm'
(lp9861
I1
aI1
aI3
aI1
aI31
aI13
aI1
aI195
aI435
aI487
aI961
aI1681
aI1233
aI6001
aI3113
aI29515
asS'd'
I4931
sa(dp9862
S'a'
I20272
sS's'
I16
sS'm'
(lp9863
I1
aI1
aI7
aI5
aI9
aI41
aI81
aI137
aI257
aI337
aI1837
aI145
aI4191
aI6313
aI9991
aI25541
asS'd'
I4932
sa(dp9864
S'a'
I20289
sS's'
I16
sS'm'
(lp9865
I1
aI1
aI5
aI13
aI29
aI13
aI1
aI117
aI501
aI991
aI571
aI793
aI1433
aI6005
aI19
aI61135
asS'd'
I4933
sa(dp9866
S'a'
I20296
sS's'
I16
sS'm'
(lp9867
I1
aI1
aI7
aI1
aI9
aI43
aI65
aI69
aI297
aI331
aI1777
aI1843
aI6477
aI13943
aI1301
aI51001
asS'd'
I4934
sa(dp9868
S'a'
I20307
sS's'
I16
sS'm'
(lp9869
I1
aI1
aI1
aI3
aI7
aI35
aI23
aI211
aI33
aI649
aI255
aI1831
aI635
aI9965
aI16679
aI14531
asS'd'
I4935
sa(dp9870
S'a'
I20316
sS's'
I16
sS'm'
(lp9871
I1
aI1
aI1
aI13
aI23
aI57
aI113
aI247
aI321
aI401
aI1761
aI4001
aI1823
aI14457
aI5251
aI4965
asS'd'
I4936
sa(dp9872
S'a'
I20323
sS's'
I16
sS'm'
(lp9873
I1
aI1
aI5
aI5
aI31
aI5
aI53
aI103
aI297
aI575
aI1577
aI2217
aI977
aI14415
aI16953
aI14793
asS'd'
I4937
sa(dp9874
S'a'
I20335
sS's'
I16
sS'm'
(lp9875
I1
aI1
aI5
aI7
aI9
aI19
aI25
aI29
aI121
aI563
aI1707
aI901
aI6167
aI10799
aI11897
aI31187
asS'd'
I4938
sa(dp9876
S'a'
I20344
sS's'
I16
sS'm'
(lp9877
I1
aI1
aI5
aI9
aI17
aI39
aI89
aI29
aI251
aI259
aI411
aI819
aI6037
aI4601
aI11481
aI46141
asS'd'
I4939
sa(dp9878
S'a'
I20354
sS's'
I16
sS'm'
(lp9879
I1
aI1
aI1
aI15
aI23
aI9
aI65
aI95
aI189
aI93
aI1485
aI2417
aI5183
aI5513
aI26623
aI42637
asS'd'
I4940
sa(dp9880
S'a'
I20360
sS's'
I16
sS'm'
(lp9881
I1
aI1
aI5
aI5
aI3
aI3
aI113
aI161
aI463
aI225
aI1089
aI2023
aI2341
aI14931
aI28097
aI56365
asS'd'
I4941
sa(dp9882
S'a'
I20368
sS's'
I16
sS'm'
(lp9883
I1
aI1
aI5
aI9
aI9
aI3
aI109
aI141
aI27
aI473
aI107
aI4023
aI3279
aI7595
aI13289
aI35649
asS'd'
I4942
sa(dp9884
S'a'
I20390
sS's'
I16
sS'm'
(lp9885
I1
aI1
aI5
aI3
aI9
aI37
aI73
aI153
aI487
aI57
aI2035
aI3583
aI239
aI2183
aI10903
aI171
asS'd'
I4943
sa(dp9886
S'a'
I20402
sS's'
I16
sS'm'
(lp9887
I1
aI3
aI3
aI15
aI23
aI39
aI87
aI217
aI451
aI597
aI1817
aI2883
aI145
aI11341
aI16015
aI16765
asS'd'
I4944
sa(dp9888
S'a'
I20413
sS's'
I16
sS'm'
(lp9889
I1
aI1
aI7
aI5
aI19
aI61
aI45
aI37
aI473
aI883
aI277
aI2801
aI13
aI7021
aI3851
aI53223
asS'd'
I4945
sa(dp9890
S'a'
I20425
sS's'
I16
sS'm'
(lp9891
I1
aI3
aI3
aI9
aI1
aI35
aI97
aI237
aI279
aI541
aI1911
aI661
aI7603
aI8183
aI22071
aI37317
asS'd'
I4946
sa(dp9892
S'a'
I20428
sS's'
I16
sS'm'
(lp9893
I1
aI3
aI3
aI11
aI11
aI63
aI101
aI71
aI227
aI259
aI1545
aI2779
aI3859
aI4859
aI18957
aI31749
asS'd'
I4947
sa(dp9894
S'a'
I20434
sS's'
I16
sS'm'
(lp9895
I1
aI3
aI3
aI1
aI27
aI29
aI91
aI215
aI381
aI607
aI1701
aI1709
aI247
aI12403
aI29943
aI59533
asS'd'
I4948
sa(dp9896
S'a'
I20443
sS's'
I16
sS'm'
(lp9897
I1
aI1
aI7
aI1
aI11
aI23
aI47
aI141
aI37
aI881
aI1443
aI3921
aI3281
aI7417
aI1549
aI50653
asS'd'
I4949
sa(dp9898
S'a'
I20488
sS's'
I16
sS'm'
(lp9899
I1
aI1
aI7
aI11
aI23
aI61
aI17
aI39
aI373
aI871
aI1107
aI1875
aI1419
aI3981
aI1333
aI11485
asS'd'
I4950
sa(dp9900
S'a'
I20502
sS's'
I16
sS'm'
(lp9901
I1
aI1
aI7
aI11
aI21
aI51
aI127
aI145
aI129
aI425
aI1263
aI1989
aI699
aI7317
aI24827
aI15049
asS'd'
I4951
sa(dp9902
S'a'
I20505
sS's'
I16
sS'm'
(lp9903
I1
aI1
aI1
aI11
aI9
aI59
aI59
aI67
aI329
aI841
aI905
aI467
aI1905
aI895
aI29711
aI25585
asS'd'
I4952
sa(dp9904
S'a'
I20535
sS's'
I16
sS'm'
(lp9905
I1
aI1
aI1
aI15
aI3
aI39
aI11
aI205
aI297
aI383
aI445
aI2139
aI2935
aI2399
aI22237
aI20355
asS'd'
I4953
sa(dp9906
S'a'
I20541
sS's'
I16
sS'm'
(lp9907
I1
aI3
aI7
aI7
aI17
aI9
aI17
aI205
aI369
aI1019
aI1703
aI755
aI5507
aI14749
aI16461
aI14519
asS'd'
I4954
sa(dp9908
S'a'
I20554
sS's'
I16
sS'm'
(lp9909
I1
aI3
aI7
aI3
aI5
aI31
aI97
aI35
aI43
aI249
aI773
aI4033
aI6085
aI1241
aI24743
aI22415
asS'd'
I4955
sa(dp9910
S'a'
I20577
sS's'
I16
sS'm'
(lp9911
I1
aI3
aI7
aI3
aI17
aI11
aI45
aI203
aI251
aI669
aI1463
aI1897
aI7913
aI2315
aI30307
aI15431
asS'd'
I4956
sa(dp9912
S'a'
I20583
sS's'
I16
sS'm'
(lp9913
I1
aI1
aI5
aI5
aI7
aI53
aI83
aI13
aI1
aI841
aI423
aI1059
aI3951
aI14209
aI11113
aI13931
asS'd'
I4957
sa(dp9914
S'a'
I20602
sS's'
I16
sS'm'
(lp9915
I1
aI3
aI3
aI5
aI5
aI15
aI11
aI71
aI237
aI553
aI829
aI3653
aI4991
aI1003
aI8353
aI52173
asS'd'
I4958
sa(dp9916
S'a'
I20611
sS's'
I16
sS'm'
(lp9917
I1
aI3
aI3
aI9
aI27
aI39
aI83
aI137
aI315
aI883
aI1155
aI3541
aI3815
aI10633
aI13277
aI20269
asS'd'
I4959
sa(dp9918
S'a'
I20614
sS's'
I16
sS'm'
(lp9919
I1
aI3
aI3
aI15
aI13
aI55
aI43
aI19
aI345
aI351
aI1117
aI1747
aI1949
aI3195
aI12241
aI52441
asS'd'
I4960
sa(dp9920
S'a'
I20626
sS's'
I16
sS'm'
(lp9921
I1
aI1
aI3
aI5
aI1
aI11
aI113
aI117
aI37
aI103
aI1681
aI3269
aI1659
aI14779
aI30479
aI31123
asS'd'
I4961
sa(dp9922
S'a'
I20628
sS's'
I16
sS'm'
(lp9923
I1
aI3
aI7
aI13
aI1
aI63
aI9
aI63
aI65
aI737
aI785
aI1713
aI8123
aI10053
aI29871
aI17647
asS'd'
I4962
sa(dp9924
S'a'
I20635
sS's'
I16
sS'm'
(lp9925
I1
aI1
aI3
aI5
aI17
aI45
aI71
aI37
aI283
aI145
aI1927
aI75
aI7355
aI4681
aI2777
aI31465
asS'd'
I4963
sa(dp9926
S'a'
I20642
sS's'
I16
sS'm'
(lp9927
I1
aI1
aI3
aI7
aI21
aI19
aI113
aI89
aI67
aI751
aI99
aI421
aI201
aI6345
aI9473
aI39849
asS'd'
I4964
sa(dp9928
S'a'
I20674
sS's'
I16
sS'm'
(lp9929
I1
aI1
aI5
aI11
aI31
aI57
aI75
aI79
aI393
aI745
aI1435
aI3039
aI1175
aI983
aI923
aI42867
asS'd'
I4965
sa(dp9930
S'a'
I20716
sS's'
I16
sS'm'
(lp9931
I1
aI1
aI5
aI9
aI31
aI47
aI31
aI61
aI85
aI651
aI1733
aI3973
aI1979
aI7223
aI13817
aI30593
asS'd'
I4966
sa(dp9932
S'a'
I20734
sS's'
I16
sS'm'
(lp9933
I1
aI1
aI1
aI11
aI31
aI21
aI23
aI177
aI401
aI55
aI537
aI3775
aI3241
aI15157
aI11849
aI15629
asS'd'
I4967
sa(dp9934
S'a'
I20765
sS's'
I16
sS'm'
(lp9935
I1
aI1
aI1
aI13
aI31
aI53
aI79
aI57
aI35
aI617
aI61
aI89
aI6917
aI6045
aI23879
aI45485
asS'd'
I4968
sa(dp9936
S'a'
I20801
sS's'
I16
sS'm'
(lp9937
I1
aI3
aI7
aI7
aI3
aI43
aI57
aI243
aI107
aI321
aI273
aI2171
aI2069
aI6171
aI29181
aI8281
asS'd'
I4969
sa(dp9938
S'a'
I20804
sS's'
I16
sS'm'
(lp9939
I1
aI1
aI1
aI11
aI3
aI27
aI51
aI57
aI81
aI795
aI1673
aI2601
aI7335
aI16243
aI863
aI20663
asS'd'
I4970
sa(dp9940
S'a'
I20808
sS's'
I16
sS'm'
(lp9941
I1
aI1
aI5
aI9
aI7
aI19
aI31
aI87
aI509
aI899
aI1133
aI1609
aI2163
aI7595
aI10523
aI43181
asS'd'
I4971
sa(dp9942
S'a'
I20831
sS's'
I16
sS'm'
(lp9943
I1
aI1
aI1
aI7
aI21
aI53
aI103
aI43
aI507
aI317
aI685
aI1329
aI7057
aI7275
aI2277
aI28271
asS'd'
I4972
sa(dp9944
S'a'
I20832
sS's'
I16
sS'm'
(lp9945
I1
aI1
aI7
aI7
aI3
aI35
aI81
aI81
aI261
aI587
aI1571
aI2771
aI4653
aI6517
aI25101
aI56753
asS'd'
I4973
sa(dp9946
S'a'
I20862
sS's'
I16
sS'm'
(lp9947
I1
aI3
aI1
aI11
aI17
aI61
aI29
aI137
aI7
aI929
aI393
aI2513
aI2423
aI5457
aI6285
aI12525
asS'd'
I4974
sa(dp9948
S'a'
I20877
sS's'
I16
sS'm'
(lp9949
I1
aI3
aI1
aI9
aI25
aI63
aI17
aI45
aI439
aI591
aI273
aI877
aI7741
aI8381
aI32277
aI24635
asS'd'
I4975
sa(dp9950
S'a'
I20880
sS's'
I16
sS'm'
(lp9951
I1
aI3
aI1
aI5
aI19
aI11
aI17
aI175
aI297
aI77
aI961
aI3331
aI5193
aI14347
aI12713
aI32067
asS'd'
I4976
sa(dp9952
S'a'
I20885
sS's'
I16
sS'm'
(lp9953
I1
aI1
aI5
aI11
aI3
aI17
aI13
aI21
aI219
aI653
aI1279
aI1657
aI7659
aI14459
aI27661
aI38093
asS'd'
I4977
sa(dp9954
S'a'
I20889
sS's'
I16
sS'm'
(lp9955
I1
aI3
aI7
aI7
aI29
aI17
aI67
aI35
aI451
aI91
aI919
aI3163
aI7459
aI14971
aI4317
aI42503
asS'd'
I4978
sa(dp9956
S'a'
I20905
sS's'
I16
sS'm'
(lp9957
I1
aI3
aI3
aI15
aI7
aI61
aI69
aI211
aI349
aI97
aI911
aI503
aI3903
aI12327
aI11049
aI29387
asS'd'
I4979
sa(dp9958
S'a'
I20914
sS's'
I16
sS'm'
(lp9959
I1
aI1
aI7
aI3
aI5
aI7
aI63
aI237
aI387
aI931
aI693
aI1143
aI6545
aI8529
aI25217
aI53087
asS'd'
I4980
sa(dp9960
S'a'
I20967
sS's'
I16
sS'm'
(lp9961
I1
aI1
aI5
aI7
aI1
aI13
aI21
aI169
aI259
aI289
aI437
aI649
aI4905
aI15261
aI29997
aI34043
asS'd'
I4981
sa(dp9962
S'a'
I21028
sS's'
I16
sS'm'
(lp9963
I1
aI1
aI1
aI9
aI25
aI13
aI19
aI229
aI29
aI213
aI1941
aI1501
aI3463
aI15761
aI15635
aI39687
asS'd'
I4982
sa(dp9964
S'a'
I21031
sS's'
I16
sS'm'
(lp9965
I1
aI1
aI5
aI7
aI13
aI29
aI101
aI57
aI483
aI913
aI1025
aI2139
aI4327
aI7847
aI12455
aI41797
asS'd'
I4983
sa(dp9966
S'a'
I21043
sS's'
I16
sS'm'
(lp9967
I1
aI1
aI3
aI11
aI17
aI27
aI97
aI79
aI411
aI9
aI1797
aI3721
aI5291
aI859
aI8889
aI6151
asS'd'
I4984
sa(dp9968
S'a'
I21052
sS's'
I16
sS'm'
(lp9969
I1
aI1
aI1
aI5
aI17
aI61
aI45
aI187
aI157
aI301
aI1017
aI1507
aI6031
aI4271
aI32593
aI23739
asS'd'
I4985
sa(dp9970
S'a'
I21058
sS's'
I16
sS'm'
(lp9971
I1
aI1
aI3
aI11
aI31
aI39
aI7
aI169
aI15
aI799
aI1585
aI2055
aI4683
aI13247
aI23743
aI50399
asS'd'
I4986
sa(dp9972
S'a'
I21087
sS's'
I16
sS'm'
(lp9973
I1
aI1
aI1
aI9
aI25
aI37
aI15
aI39
aI339
aI383
aI1153
aI1211
aI5745
aI15249
aI26021
aI39871
asS'd'
I4987
sa(dp9974
S'a'
I21088
sS's'
I16
sS'm'
(lp9975
I1
aI1
aI3
aI13
aI17
aI51
aI27
aI137
aI231
aI877
aI309
aI3633
aI2575
aI12259
aI2743
aI14781
asS'd'
I4988
sa(dp9976
S'a'
I21093
sS's'
I16
sS'm'
(lp9977
I1
aI1
aI5
aI7
aI5
aI33
aI95
aI19
aI37
aI829
aI1489
aI3525
aI3887
aI8277
aI21867
aI3581
asS'd'
I4989
sa(dp9978
S'a'
I21106
sS's'
I16
sS'm'
(lp9979
I1
aI1
aI1
aI15
aI11
aI33
aI99
aI213
aI365
aI549
aI1603
aI777
aI3787
aI12209
aI14999
aI50607
asS'd'
I4990
sa(dp9980
S'a'
I21108
sS's'
I16
sS'm'
(lp9981
I1
aI3
aI1
aI9
aI23
aI25
aI57
aI147
aI73
aI285
aI1229
aI1763
aI4579
aI14771
aI4003
aI38197
asS'd'
I4991
sa(dp9982
S'a'
I21118
sS's'
I16
sS'm'
(lp9983
I1
aI1
aI5
aI1
aI15
aI55
aI25
aI209
aI135
aI895
aI311
aI139
aI2445
aI6903
aI12129
aI27907
asS'd'
I4992
sa(dp9984
S'a'
I21122
sS's'
I16
sS'm'
(lp9985
I1
aI1
aI5
aI7
aI23
aI29
aI33
aI135
aI325
aI517
aI2021
aI1721
aI4235
aI2363
aI12905
aI18811
asS'd'
I4993
sa(dp9986
S'a'
I21131
sS's'
I16
sS'm'
(lp9987
I1
aI1
aI1
aI9
aI3
aI19
aI69
aI29
aI157
aI787
aI1969
aI3711
aI8179
aI5691
aI4059
aI25541
asS'd'
I4994
sa(dp9988
S'a'
I21139
sS's'
I16
sS'm'
(lp9989
I1
aI1
aI5
aI15
aI1
aI61
aI11
aI195
aI317
aI13
aI923
aI2149
aI4001
aI12843
aI27109
aI30625
asS'd'
I4995
sa(dp9990
S'a'
I21141
sS's'
I16
sS'm'
(lp9991
I1
aI3
aI1
aI7
aI3
aI13
aI45
aI187
aI445
aI859
aI53
aI3227
aI4381
aI8273
aI32699
aI48719
asS'd'
I4996
sa(dp9992
S'a'
I21146
sS's'
I16
sS'm'
(lp9993
I1
aI1
aI7
aI7
aI21
aI19
aI47
aI101
aI119
aI311
aI577
aI3309
aI4585
aI12109
aI15153
aI22915
asS'd'
I4997
sa(dp9994
S'a'
I21162
sS's'
I16
sS'm'
(lp9995
I1
aI3
aI5
aI15
aI21
aI39
aI15
aI211
aI349
aI237
aI1873
aI3333
aI7837
aI12811
aI14321
aI20227
asS'd'
I4998
sa(dp9996
S'a'
I21164
sS's'
I16
sS'm'
(lp9997
I1
aI1
aI5
aI5
aI19
aI47
aI15
aI239
aI23
aI763
aI537
aI1477
aI2231
aI10885
aI19487
aI47385
asS'd'
I4999
sa(dp9998
S'a'
I21184
sS's'
I16
sS'm'
(lp9999
I1
aI1
aI3
aI1
aI19
aI37
aI67
aI85
aI11
aI817
aI869
aI2249
aI4111
aI12411
aI10405
aI20055
asS'd'
I5000
sa(dp10000
S'a'
I21208
sS's'
I16
sS'm'
(lp10001
I1
aI1
aI3
aI3
aI1
aI41
aI85
aI137
aI91
aI369
aI1863
aI759
aI303
aI15859
aI8063
aI12811
asS'd'
I5001
sa(dp10002
S'a'
I21211
sS's'
I16
sS'm'
(lp10003
I1
aI3
aI1
aI11
aI23
aI1
aI11
aI219
aI201
aI573
aI1573
aI619
aI2959
aI6485
aI7483
aI46099
asS'd'
I5002
sa(dp10004
S'a'
I21213
sS's'
I16
sS'm'
(lp10005
I1
aI3
aI3
aI9
aI13
aI9
aI9
aI255
aI47
aI375
aI409
aI1435
aI1665
aI14967
aI3247
aI18193
asS'd'
I5003
sa(dp10006
S'a'
I21230
sS's'
I16
sS'm'
(lp10007
I1
aI1
aI1
aI5
aI9
aI61
aI121
aI173
aI51
aI415
aI1621
aI3947
aI1379
aI847
aI23187
aI39259
asS'd'
I5004
sa(dp10008
S'a'
I21270
sS's'
I16
sS'm'
(lp10009
I1
aI1
aI1
aI7
aI3
aI19
aI95
aI59
aI187
aI453
aI1533
aI445
aI2699
aI4817
aI25983
aI50925
asS'd'
I5005
sa(dp10010
S'a'
I21276
sS's'
I16
sS'm'
(lp10011
I1
aI3
aI5
aI13
aI25
aI25
aI33
aI5
aI497
aI1
aI535
aI1653
aI6541
aI10939
aI17721
aI43829
asS'd'
I5006
sa(dp10012
S'a'
I21285
sS's'
I16
sS'm'
(lp10013
I1
aI3
aI7
aI11
aI9
aI59
aI115
aI127
aI85
aI505
aI541
aI3243
aI5853
aI12673
aI25275
aI39577
asS'd'
I5007
sa(dp10014
S'a'
I21297
sS's'
I16
sS'm'
(lp10015
I1
aI3
aI7
aI1
aI17
aI25
aI83
aI127
aI225
aI295
aI1823
aI2051
aI847
aI4249
aI13763
aI5723
asS'd'
I5008
sa(dp10016
S'a'
I21304
sS's'
I16
sS'm'
(lp10017
I1
aI1
aI1
aI5
aI3
aI63
aI39
aI131
aI191
aI983
aI119
aI3287
aI3335
aI7969
aI31347
aI39439
asS'd'
I5009
sa(dp10018
S'a'
I21310
sS's'
I16
sS'm'
(lp10019
I1
aI3
aI7
aI9
aI19
aI31
aI19
aI91
aI35
aI677
aI1229
aI1371
aI6497
aI3315
aI15239
aI54235
asS'd'
I5010
sa(dp10020
S'a'
I21330
sS's'
I16
sS'm'
(lp10021
I1
aI1
aI1
aI15
aI3
aI49
aI113
aI199
aI135
aI35
aI709
aI385
aI7923
aI3711
aI18351
aI12711
asS'd'
I5011
sa(dp10022
S'a'
I21339
sS's'
I16
sS'm'
(lp10023
I1
aI1
aI3
aI15
aI31
aI13
aI41
aI1
aI183
aI95
aI1625
aI1675
aI7881
aI6607
aI4165
aI27151
asS'd'
I5012
sa(dp10024
S'a'
I21346
sS's'
I16
sS'm'
(lp10025
I1
aI3
aI3
aI15
aI21
aI57
aI81
aI49
aI5
aI297
aI131
aI883
aI1113
aI2497
aI32129
aI39139
asS'd'
I5013
sa(dp10026
S'a'
I21391
sS's'
I16
sS'm'
(lp10027
I1
aI3
aI5
aI7
aI29
aI47
aI101
aI173
aI299
aI879
aI143
aI3341
aI3929
aI6797
aI8753
aI47963
asS'd'
I5014
sa(dp10028
S'a'
I21427
sS's'
I16
sS'm'
(lp10029
I1
aI3
aI3
aI13
aI11
aI39
aI27
aI187
aI27
aI763
aI1515
aI1903
aI5897
aI10061
aI14595
aI12713
asS'd'
I5015
sa(dp10030
S'a'
I21451
sS's'
I16
sS'm'
(lp10031
I1
aI3
aI5
aI11
aI27
aI35
aI37
aI213
aI45
aI867
aI1591
aI3083
aI8123
aI5045
aI31703
aI61487
asS'd'
I5016
sa(dp10032
S'a'
I21465
sS's'
I16
sS'm'
(lp10033
I1
aI1
aI3
aI5
aI3
aI31
aI23
aI89
aI369
aI371
aI1165
aI3673
aI6821
aI333
aI10881
aI4153
asS'd'
I5017
sa(dp10034
S'a'
I21468
sS's'
I16
sS'm'
(lp10035
I1
aI1
aI7
aI13
aI1
aI33
aI49
aI195
aI223
aI197
aI1799
aI2427
aI6171
aI493
aI13503
aI23619
asS'd'
I5018
sa(dp10036
S'a'
I21471
sS's'
I16
sS'm'
(lp10037
I1
aI1
aI3
aI9
aI5
aI59
aI105
aI215
aI449
aI527
aI1661
aI2643
aI309
aI11239
aI11633
aI63459
asS'd'
I5019
sa(dp10038
S'a'
I21533
sS's'
I16
sS'm'
(lp10039
I1
aI1
aI3
aI11
aI13
aI11
aI15
aI99
aI409
aI807
aI1911
aI883
aI1323
aI9037
aI6401
aI545
asS'd'
I5020
sa(dp10040
S'a'
I21610
sS's'
I16
sS'm'
(lp10041
I1
aI1
aI5
aI7
aI7
aI7
aI1
aI167
aI353
aI923
aI1403
aI3109
aI4981
aI3877
aI25451
aI4667
asS'd'
I5021
sa(dp10042
S'a'
I21615
sS's'
I16
sS'm'
(lp10043
I1
aI1
aI5
aI11
aI11
aI25
aI121
aI153
aI111
aI785
aI1301
aI1497
aI6267
aI14919
aI24125
aI52029
asS'd'
I5022
sa(dp10044
S'a'
I21630
sS's'
I16
sS'm'
(lp10045
I1
aI3
aI3
aI5
aI29
aI55
aI63
aI177
aI305
aI41
aI111
aI1065
aI1127
aI113
aI2815
aI12979
asS'd'
I5023
sa(dp10046
S'a'
I21633
sS's'
I16
sS'm'
(lp10047
I1
aI3
aI5
aI7
aI23
aI39
aI17
aI179
aI267
aI917
aI511
aI3923
aI915
aI14173
aI10689
aI50749
asS'd'
I5024
sa(dp10048
S'a'
I21657
sS's'
I16
sS'm'
(lp10049
I1
aI1
aI5
aI3
aI9
aI45
aI15
aI157
aI495
aI625
aI407
aI2769
aI3267
aI7593
aI17957
aI54597
asS'd'
I5025
sa(dp10050
S'a'
I21658
sS's'
I16
sS'm'
(lp10051
I1
aI3
aI3
aI11
aI21
aI13
aI5
aI207
aI107
aI965
aI1803
aI1541
aI3487
aI3483
aI109
aI26923
asS'd'
I5026
sa(dp10052
S'a'
I21669
sS's'
I16
sS'm'
(lp10053
I1
aI1
aI5
aI11
aI25
aI49
aI99
aI135
aI109
aI371
aI1307
aI1815
aI1095
aI2329
aI27101
aI52269
asS'd'
I5027
sa(dp10054
S'a'
I21670
sS's'
I16
sS'm'
(lp10055
I1
aI3
aI5
aI5
aI13
aI9
aI109
aI79
aI151
aI47
aI311
aI3873
aI3645
aI3773
aI1083
aI31599
asS'd'
I5028
sa(dp10056
S'a'
I21673
sS's'
I16
sS'm'
(lp10057
I1
aI3
aI5
aI15
aI1
aI9
aI87
aI21
aI145
aI583
aI159
aI2435
aI587
aI10123
aI24803
aI19993
asS'd'
I5029
sa(dp10058
S'a'
I21701
sS's'
I16
sS'm'
(lp10059
I1
aI3
aI1
aI1
aI23
aI11
aI5
aI45
aI373
aI1011
aI1353
aI277
aI7051
aI3845
aI12391
aI25313
asS'd'
I5030
sa(dp10060
S'a'
I21719
sS's'
I16
sS'm'
(lp10061
I1
aI1
aI1
aI9
aI13
aI13
aI109
aI251
aI97
aI483
aI1723
aI2555
aI813
aI9345
aI11351
aI44705
asS'd'
I5031
sa(dp10062
S'a'
I21720
sS's'
I16
sS'm'
(lp10063
I1
aI3
aI5
aI7
aI21
aI49
aI63
aI13
aI119
aI813
aI1559
aI983
aI499
aI15159
aI24163
aI59903
asS'd'
I5032
sa(dp10064
S'a'
I21747
sS's'
I16
sS'm'
(lp10065
I1
aI1
aI3
aI5
aI27
aI33
aI27
aI165
aI207
aI693
aI1401
aI1357
aI7637
aI337
aI10163
aI43273
asS'd'
I5033
sa(dp10066
S'a'
I21819
sS's'
I16
sS'm'
(lp10067
I1
aI1
aI5
aI13
aI29
aI7
aI71
aI187
aI1
aI655
aI1829
aI2645
aI6969
aI5435
aI28415
aI33199
asS'd'
I5034
sa(dp10068
S'a'
I21839
sS's'
I16
sS'm'
(lp10069
I1
aI1
aI1
aI13
aI17
aI21
aI13
aI141
aI41
aI267
aI1165
aI1893
aI3455
aI10737
aI16693
aI33065
asS'd'
I5035
sa(dp10070
S'a'
I21854
sS's'
I16
sS'm'
(lp10071
I1
aI1
aI5
aI1
aI7
aI27
aI7
aI67
aI107
aI11
aI763
aI2529
aI3023
aI15745
aI17023
aI51911
asS'd'
I5036
sa(dp10072
S'a'
I21857
sS's'
I16
sS'm'
(lp10073
I1
aI3
aI3
aI3
aI7
aI57
aI123
aI249
aI309
aI511
aI1655
aI1379
aI725
aI7325
aI20261
aI65039
asS'd'
I5037
sa(dp10074
S'a'
I21864
sS's'
I16
sS'm'
(lp10075
I1
aI1
aI5
aI11
aI3
aI27
aI23
aI27
aI285
aI771
aI2017
aI1727
aI4847
aI2665
aI30655
aI47625
asS'd'
I5038
sa(dp10076
S'a'
I21882
sS's'
I16
sS'm'
(lp10077
I1
aI3
aI7
aI7
aI17
aI3
aI93
aI133
aI427
aI1021
aI1135
aI341
aI6711
aI11713
aI24157
aI1561
asS'd'
I5039
sa(dp10078
S'a'
I21900
sS's'
I16
sS'm'
(lp10079
I1
aI1
aI3
aI7
aI15
aI55
aI11
aI247
aI65
aI115
aI1967
aI535
aI841
aI15131
aI28381
aI31337
asS'd'
I5040
sa(dp10080
S'a'
I21903
sS's'
I16
sS'm'
(lp10081
I1
aI3
aI1
aI7
aI9
aI45
aI73
aI151
aI127
aI125
aI767
aI2003
aI6893
aI3875
aI451
aI30275
asS'd'
I5041
sa(dp10082
S'a'
I21928
sS's'
I16
sS'm'
(lp10083
I1
aI1
aI7
aI3
aI5
aI55
aI127
aI123
aI163
aI763
aI1165
aI1637
aI6267
aI7215
aI23403
aI20961
asS'd'
I5042
sa(dp10084
S'a'
I21931
sS's'
I16
sS'm'
(lp10085
I1
aI1
aI1
aI13
aI1
aI21
aI65
aI141
aI369
aI413
aI1675
aI27
aI7357
aI6929
aI18083
aI829
asS'd'
I5043
sa(dp10086
S'a'
I21946
sS's'
I16
sS'm'
(lp10087
I1
aI3
aI5
aI13
aI1
aI17
aI97
aI107
aI249
aI931
aI47
aI3537
aI2245
aI9827
aI13673
aI23217
asS'd'
I5044
sa(dp10088
S'a'
I21971
sS's'
I16
sS'm'
(lp10089
I1
aI1
aI1
aI11
aI13
aI19
aI43
aI31
aI51
aI727
aI1001
aI927
aI771
aI11853
aI5761
aI60537
asS'd'
I5045
sa(dp10090
S'a'
I21974
sS's'
I16
sS'm'
(lp10091
I1
aI3
aI1
aI7
aI23
aI27
aI115
aI5
aI201
aI431
aI1021
aI585
aI6069
aI12511
aI6129
aI2105
asS'd'
I5046
sa(dp10092
S'a'
I21978
sS's'
I16
sS'm'
(lp10093
I1
aI1
aI3
aI11
aI3
aI25
aI75
aI129
aI389
aI131
aI223
aI2263
aI5377
aI5561
aI15633
aI39527
asS'd'
I5047
sa(dp10094
S'a'
I21993
sS's'
I16
sS'm'
(lp10095
I1
aI3
aI3
aI1
aI27
aI43
aI101
aI55
aI319
aI549
aI1971
aI2255
aI353
aI93
aI25661
aI59077
asS'd'
I5048
sa(dp10096
S'a'
I21994
sS's'
I16
sS'm'
(lp10097
I1
aI1
aI5
aI11
aI29
aI57
aI27
aI135
aI341
aI913
aI1637
aI1781
aI457
aI11293
aI1013
aI53863
asS'd'
I5049
sa(dp10098
S'a'
I22030
sS's'
I16
sS'm'
(lp10099
I1
aI1
aI1
aI11
aI3
aI51
aI79
aI251
aI443
aI651
aI393
aI3635
aI7397
aI5443
aI4225
aI991
asS'd'
I5050
sa(dp10100
S'a'
I22035
sS's'
I16
sS'm'
(lp10101
I1
aI3
aI7
aI13
aI31
aI9
aI3
aI109
aI427
aI735
aI891
aI2789
aI2169
aI6459
aI355
aI43177
asS'd'
I5051
sa(dp10102
S'a'
I22063
sS's'
I16
sS'm'
(lp10103
I1
aI3
aI3
aI3
aI13
aI7
aI93
aI195
aI293
aI37
aI75
aI2467
aI933
aI8017
aI9925
aI61397
asS'd'
I5052
sa(dp10104
S'a'
I22068
sS's'
I16
sS'm'
(lp10105
I1
aI1
aI5
aI7
aI25
aI15
aI69
aI199
aI161
aI769
aI387
aI1491
aI4553
aI4173
aI25631
aI37089
asS'd'
I5053
sa(dp10106
S'a'
I22086
sS's'
I16
sS'm'
(lp10107
I1
aI3
aI1
aI3
aI7
aI59
aI53
aI93
aI103
aI413
aI531
aI887
aI6149
aI2901
aI22611
aI27135
asS'd'
I5054
sa(dp10108
S'a'
I22104
sS's'
I16
sS'm'
(lp10109
I1
aI1
aI1
aI13
aI31
aI39
aI71
aI215
aI385
aI821
aI1603
aI3043
aI4967
aI10953
aI11543
aI64433
asS'd'
I5055
sa(dp10110
S'a'
I22119
sS's'
I16
sS'm'
(lp10111
I1
aI3
aI1
aI7
aI7
aI63
aI5
aI143
aI1
aI339
aI1165
aI3809
aI4257
aI12879
aI21581
aI21307
asS'd'
I5056
sa(dp10112
S'a'
I22153
sS's'
I16
sS'm'
(lp10113
I1
aI1
aI1
aI15
aI1
aI35
aI67
aI227
aI277
aI879
aI513
aI3423
aI6153
aI11573
aI12809
aI34335
asS'd'
I5057
sa(dp10114
S'a'
I22168
sS's'
I16
sS'm'
(lp10115
I1
aI3
aI7
aI9
aI9
aI39
aI47
aI17
aI101
aI179
aI631
aI1307
aI481
aI871
aI16807
aI39811
asS'd'
I5058
sa(dp10116
S'a'
I22183
sS's'
I16
sS'm'
(lp10117
I1
aI3
aI1
aI1
aI13
aI25
aI53
aI179
aI285
aI267
aI407
aI3781
aI3267
aI3545
aI525
aI15733
asS'd'
I5059
sa(dp10118
S'a'
I22189
sS's'
I16
sS'm'
(lp10119
I1
aI1
aI1
aI13
aI11
aI35
aI45
aI181
aI319
aI767
aI283
aI3021
aI5405
aI403
aI3587
aI7291
asS'd'
I5060
sa(dp10120
S'a'
I22204
sS's'
I16
sS'm'
(lp10121
I1
aI1
aI7
aI3
aI5
aI9
aI67
aI129
aI101
aI117
aI267
aI1925
aI1209
aI13095
aI7123
aI62941
asS'd'
I5061
sa(dp10122
S'a'
I22230
sS's'
I16
sS'm'
(lp10123
I1
aI1
aI7
aI3
aI5
aI63
aI109
aI199
aI95
aI421
aI193
aI3519
aI6551
aI955
aI1679
aI16627
asS'd'
I5062
sa(dp10124
S'a'
I22240
sS's'
I16
sS'm'
(lp10125
I1
aI1
aI5
aI13
aI1
aI47
aI71
aI157
aI343
aI653
aI981
aI1335
aI3737
aI7185
aI28861
aI22883
asS'd'
I5063
sa(dp10126
S'a'
I22246
sS's'
I16
sS'm'
(lp10127
I1
aI1
aI3
aI15
aI7
aI63
aI7
aI81
aI481
aI5
aI1159
aI1361
aI4167
aI2575
aI7437
aI16917
asS'd'
I5064
sa(dp10128
S'a'
I22269
sS's'
I16
sS'm'
(lp10129
I1
aI3
aI7
aI1
aI27
aI17
aI61
aI193
aI317
aI841
aI1149
aI955
aI5161
aI4275
aI1955
aI15665
asS'd'
I5065
sa(dp10130
S'a'
I22282
sS's'
I16
sS'm'
(lp10131
I1
aI1
aI1
aI7
aI19
aI63
aI77
aI57
aI367
aI237
aI579
aI3701
aI5763
aI4951
aI24151
aI45215
asS'd'
I5066
sa(dp10132
S'a'
I22302
sS's'
I16
sS'm'
(lp10133
I1
aI1
aI5
aI11
aI29
aI7
aI119
aI155
aI431
aI999
aI757
aI2433
aI5811
aI3709
aI29573
aI23639
asS'd'
I5067
sa(dp10134
S'a'
I22330
sS's'
I16
sS'm'
(lp10135
I1
aI3
aI3
aI3
aI15
aI35
aI125
aI13
aI275
aI507
aI1719
aI1537
aI2349
aI12909
aI8107
aI9845
asS'd'
I5068
sa(dp10136
S'a'
I22347
sS's'
I16
sS'm'
(lp10137
I1
aI3
aI1
aI13
aI27
aI27
aI11
aI69
aI15
aI1017
aI207
aI625
aI809
aI2921
aI8939
aI30293
asS'd'
I5069
sa(dp10138
S'a'
I22349
sS's'
I16
sS'm'
(lp10139
I1
aI1
aI7
aI11
aI31
aI51
aI113
aI193
aI69
aI845
aI73
aI919
aI3523
aI3987
aI23665
aI36527
asS'd'
I5070
sa(dp10140
S'a'
I22383
sS's'
I16
sS'm'
(lp10141
I1
aI3
aI7
aI11
aI21
aI31
aI103
aI29
aI5
aI81
aI1081
aI3847
aI4697
aI8857
aI30769
aI40053
asS'd'
I5071
sa(dp10142
S'a'
I22386
sS's'
I16
sS'm'
(lp10143
I1
aI1
aI1
aI1
aI5
aI11
aI5
aI169
aI13
aI899
aI769
aI3823
aI5405
aI5991
aI3821
aI27767
asS'd'
I5072
sa(dp10144
S'a'
I22432
sS's'
I16
sS'm'
(lp10145
I1
aI1
aI3
aI15
aI1
aI35
aI9
aI83
aI23
aI701
aI1807
aI1681
aI4009
aI127
aI31751
aI38059
asS'd'
I5073
sa(dp10146
S'a'
I22450
sS's'
I16
sS'm'
(lp10147
I1
aI3
aI3
aI7
aI9
aI61
aI73
aI111
aI193
aI607
aI2001
aI413
aI3751
aI16337
aI16597
aI23959
asS'd'
I5074
sa(dp10148
S'a'
I22473
sS's'
I16
sS'm'
(lp10149
I1
aI1
aI7
aI7
aI21
aI29
aI53
aI253
aI187
aI507
aI1191
aI3521
aI463
aI2167
aI23785
aI19867
asS'd'
I5075
sa(dp10150
S'a'
I22487
sS's'
I16
sS'm'
(lp10151
I1
aI3
aI5
aI3
aI19
aI43
aI101
aI93
aI257
aI61
aI1589
aI3883
aI1975
aI7283
aI5253
aI23257
asS'd'
I5076
sa(dp10152
S'a'
I22527
sS's'
I16
sS'm'
(lp10153
I1
aI3
aI1
aI9
aI1
aI63
aI25
aI101
aI377
aI571
aI1701
aI3385
aI243
aI12051
aI32619
aI10459
asS'd'
I5077
sa(dp10154
S'a'
I22537
sS's'
I16
sS'm'
(lp10155
I1
aI1
aI1
aI5
aI17
aI11
aI37
aI197
aI205
aI879
aI625
aI959
aI7389
aI7857
aI20615
aI20405
asS'd'
I5078
sa(dp10156
S'a'
I22557
sS's'
I16
sS'm'
(lp10157
I1
aI3
aI5
aI5
aI27
aI41
aI9
aI109
aI197
aI623
aI1045
aI63
aI7977
aI11355
aI28613
aI5131
asS'd'
I5079
sa(dp10158
S'a'
I22561
sS's'
I16
sS'm'
(lp10159
I1
aI1
aI5
aI11
aI5
aI29
aI27
aI85
aI131
aI247
aI433
aI3981
aI2677
aI15415
aI869
aI6045
asS'd'
I5080
sa(dp10160
S'a'
I22568
sS's'
I16
sS'm'
(lp10161
I1
aI3
aI1
aI13
aI9
aI49
aI25
aI79
aI135
aI719
aI93
aI631
aI2149
aI5929
aI29833
aI38673
asS'd'
I5081
sa(dp10162
S'a'
I22573
sS's'
I16
sS'm'
(lp10163
I1
aI3
aI7
aI11
aI15
aI13
aI37
aI233
aI227
aI205
aI1579
aI65
aI1429
aI13499
aI26355
aI63821
asS'd'
I5082
sa(dp10164
S'a'
I22591
sS's'
I16
sS'm'
(lp10165
I1
aI1
aI5
aI11
aI21
aI19
aI7
aI183
aI409
aI275
aI899
aI3665
aI6207
aI849
aI24339
aI1617
asS'd'
I5083
sa(dp10166
S'a'
I22593
sS's'
I16
sS'm'
(lp10167
I1
aI3
aI1
aI3
aI21
aI61
aI23
aI125
aI463
aI489
aI1265
aI2975
aI3907
aI11881
aI7533
aI43331
asS'd'
I5084
sa(dp10168
S'a'
I22605
sS's'
I16
sS'm'
(lp10169
I1
aI3
aI1
aI15
aI15
aI51
aI83
aI31
aI175
aI47
aI1791
aI3651
aI6735
aI5013
aI723
aI24141
asS'd'
I5085
sa(dp10170
S'a'
I22620
sS's'
I16
sS'm'
(lp10171
I1
aI3
aI1
aI9
aI17
aI41
aI57
aI43
aI469
aI911
aI1251
aI2105
aI3133
aI3437
aI10097
aI26687
asS'd'
I5086
sa(dp10172
S'a'
I22627
sS's'
I16
sS'm'
(lp10173
I1
aI1
aI3
aI3
aI9
aI9
aI125
aI201
aI141
aI943
aI1509
aI1239
aI6575
aI8707
aI363
aI23309
asS'd'
I5087
sa(dp10174
S'a'
I22636
sS's'
I16
sS'm'
(lp10175
I1
aI1
aI5
aI3
aI19
aI37
aI43
aI141
aI413
aI149
aI1449
aI1003
aI4473
aI12395
aI4101
aI61201
asS'd'
I5088
sa(dp10176
S'a'
I22647
sS's'
I16
sS'm'
(lp10177
I1
aI1
aI5
aI11
aI17
aI37
aI41
aI33
aI57
aI627
aI325
aI1895
aI1773
aI1339
aI24859
aI587
asS'd'
I5089
sa(dp10178
S'a'
I22697
sS's'
I16
sS'm'
(lp10179
I1
aI1
aI1
aI3
aI5
aI49
aI127
aI109
aI361
aI853
aI1437
aI3451
aI4031
aI5379
aI27463
aI47425
asS'd'
I5090
sa(dp10180
S'a'
I22715
sS's'
I16
sS'm'
(lp10181
I1
aI3
aI5
aI7
aI9
aI57
aI71
aI219
aI347
aI791
aI797
aI73
aI6241
aI12717
aI24429
aI40977
asS'd'
I5091
sa(dp10182
S'a'
I22725
sS's'
I16
sS'm'
(lp10183
I1
aI1
aI5
aI9
aI27
aI43
aI43
aI227
aI433
aI413
aI1109
aI2589
aI4535
aI8947
aI8121
aI43479
asS'd'
I5092
sa(dp10184
S'a'
I22732
sS's'
I16
sS'm'
(lp10185
I1
aI3
aI7
aI1
aI9
aI21
aI81
aI23
aI157
aI313
aI197
aI2845
aI8059
aI15957
aI28657
aI28093
asS'd'
I5093
sa(dp10186
S'a'
I22749
sS's'
I16
sS'm'
(lp10187
I1
aI3
aI1
aI11
aI15
aI17
aI115
aI27
aI421
aI743
aI1885
aI2089
aI5011
aI7137
aI7395
aI36853
asS'd'
I5094
sa(dp10188
S'a'
I22766
sS's'
I16
sS'm'
(lp10189
I1
aI1
aI5
aI15
aI5
aI53
aI69
aI255
aI63
aI29
aI1011
aI3201
aI1467
aI15441
aI26255
aI62895
asS'd'
I5095
sa(dp10190
S'a'
I22768
sS's'
I16
sS'm'
(lp10191
I1
aI3
aI1
aI11
aI3
aI47
aI35
aI195
aI149
aI849
aI1317
aI439
aI3539
aI845
aI15295
aI42183
asS'd'
I5096
sa(dp10192
S'a'
I22771
sS's'
I16
sS'm'
(lp10193
I1
aI1
aI5
aI15
aI15
aI37
aI67
aI105
aI495
aI985
aI1777
aI3137
aI8039
aI11795
aI29771
aI35045
asS'd'
I5097
sa(dp10194
S'a'
I22788
sS's'
I16
sS'm'
(lp10195
I1
aI1
aI3
aI1
aI25
aI17
aI67
aI227
aI229
aI419
aI1319
aI3325
aI1293
aI8585
aI28425
aI34013
asS'd'
I5098
sa(dp10196
S'a'
I22797
sS's'
I16
sS'm'
(lp10197
I1
aI1
aI5
aI1
aI27
aI51
aI71
aI197
aI375
aI407
aI259
aI4005
aI3309
aI5475
aI13421
aI60065
asS'd'
I5099
sa(dp10198
S'a'
I22822
sS's'
I16
sS'm'
(lp10199
I1
aI3
aI1
aI5
aI11
aI17
aI89
aI45
aI311
aI425
aI1629
aI773
aI7267
aI8699
aI27547
aI37081
asS'd'
I5100
sa(dp10200
S'a'
I22828
sS's'
I16
sS'm'
(lp10201
I1
aI3
aI1
aI7
aI9
aI25
aI101
aI105
aI489
aI217
aI103
aI1959
aI4049
aI5793
aI31201
aI11947
asS'd'
I5101
sa(dp10202
S'a'
I22851
sS's'
I16
sS'm'
(lp10203
I1
aI1
aI5
aI5
aI19
aI3
aI63
aI55
aI431
aI49
aI273
aI3253
aI5357
aI13801
aI9735
aI21883
asS'd'
I5102
sa(dp10204
S'a'
I22888
sS's'
I16
sS'm'
(lp10205
I1
aI1
aI1
aI11
aI13
aI3
aI75
aI201
aI477
aI201
aI1875
aI657
aI6217
aI8651
aI2207
aI16421
asS'd'
I5103
sa(dp10206
S'a'
I22893
sS's'
I16
sS'm'
(lp10207
I1
aI1
aI5
aI13
aI5
aI31
aI75
aI113
aI25
aI171
aI1147
aI3089
aI7095
aI4405
aI26155
aI42323
asS'd'
I5104
sa(dp10208
S'a'
I22901
sS's'
I16
sS'm'
(lp10209
I1
aI3
aI5
aI5
aI5
aI49
aI99
aI171
aI445
aI1023
aI1793
aI3159
aI5809
aI12509
aI31723
aI60411
asS'd'
I5105
sa(dp10210
S'a'
I22902
sS's'
I16
sS'm'
(lp10211
I1
aI3
aI7
aI3
aI23
aI51
aI111
aI27
aI103
aI159
aI433
aI293
aI1741
aI3599
aI4067
aI40667
asS'd'
I5106
sa(dp10212
S'a'
I22921
sS's'
I16
sS'm'
(lp10213
I1
aI3
aI3
aI13
aI11
aI9
aI11
aI21
aI453
aI35
aI1649
aI1261
aI3539
aI14081
aI5581
aI57105
asS'd'
I5107
sa(dp10214
S'a'
I22929
sS's'
I16
sS'm'
(lp10215
I1
aI3
aI3
aI13
aI7
aI9
aI113
aI87
aI391
aI647
aI223
aI1345
aI4481
aI9855
aI20129
aI10807
asS'd'
I5108
sa(dp10216
S'a'
I22946
sS's'
I16
sS'm'
(lp10217
I1
aI3
aI7
aI15
aI3
aI61
aI15
aI117
aI97
aI495
aI985
aI819
aI181
aI1363
aI13111
aI35857
asS'd'
I5109
sa(dp10218
S'a'
I22948
sS's'
I16
sS'm'
(lp10219
I1
aI3
aI1
aI9
aI3
aI27
aI125
aI151
aI217
aI961
aI707
aI2647
aI5307
aI621
aI12581
aI13941
asS'd'
I5110
sa(dp10220
S'a'
I22951
sS's'
I16
sS'm'
(lp10221
I1
aI3
aI1
aI11
aI17
aI37
aI35
aI211
aI179
aI29
aI627
aI3623
aI6429
aI16237
aI24699
aI14385
asS'd'
I5111
sa(dp10222
S'a'
I22958
sS's'
I16
sS'm'
(lp10223
I1
aI3
aI3
aI9
aI3
aI57
aI35
aI3
aI85
aI1017
aI1739
aI2241
aI7297
aI15637
aI27085
aI41237
asS'd'
I5112
sa(dp10224
S'a'
I22975
sS's'
I16
sS'm'
(lp10225
I1
aI1
aI3
aI7
aI7
aI13
aI5
aI85
aI505
aI51
aI409
aI867
aI677
aI12451
aI13633
aI47883
asS'd'
I5113
sa(dp10226
S'a'
I22983
sS's'
I16
sS'm'
(lp10227
I1
aI3
aI5
aI13
aI5
aI51
aI37
aI79
aI237
aI133
aI1331
aI3263
aI349
aI4971
aI16067
aI62485
asS'd'
I5114
sa(dp10228
S'a'
I22990
sS's'
I16
sS'm'
(lp10229
I1
aI1
aI7
aI11
aI29
aI41
aI101
aI219
aI391
aI1023
aI1681
aI3163
aI4071
aI14665
aI11041
aI14523
asS'd'
I5115
sa(dp10230
S'a'
I23032
sS's'
I16
sS'm'
(lp10231
I1
aI1
aI3
aI3
aI13
aI55
aI37
aI119
aI471
aI665
aI1315
aI3071
aI5993
aI12005
aI13549
aI63425
asS'd'
I5116
sa(dp10232
S'a'
I23047
sS's'
I16
sS'm'
(lp10233
I1
aI3
aI5
aI7
aI5
aI25
aI59
aI71
aI77
aI841
aI91
aI1841
aI6997
aI1139
aI11843
aI52209
asS'd'
I5117
sa(dp10234
S'a'
I23053
sS's'
I16
sS'm'
(lp10235
I1
aI3
aI7
aI15
aI17
aI25
aI85
aI173
aI373
aI459
aI1713
aI1055
aI5337
aI9921
aI15213
aI44235
asS'd'
I5118
sa(dp10236
S'a'
I23054
sS's'
I16
sS'm'
(lp10237
I1
aI1
aI1
aI15
aI7
aI11
aI89
aI237
aI131
aI565
aI745
aI457
aI4447
aI5581
aI11053
aI43819
asS'd'
I5119
sa(dp10238
S'a'
I23082
sS's'
I16
sS'm'
(lp10239
I1
aI3
aI5
aI1
aI29
aI21
aI11
aI7
aI125
aI851
aI2023
aI3321
aI1885
aI67
aI8809
aI43291
asS'd'
I5120
sa(dp10240
S'a'
I23095
sS's'
I16
sS'm'
(lp10241
I1
aI3
aI5
aI11
aI11
aI43
aI41
aI97
aI353
aI813
aI85
aI2453
aI769
aI11709
aI4697
aI2849
asS'd'
I5121
sa(dp10242
S'a'
I23116
sS's'
I16
sS'm'
(lp10243
I1
aI1
aI5
aI5
aI21
aI29
aI87
aI179
aI157
aI981
aI129
aI2139
aI6841
aI5479
aI27111
aI20749
asS'd'
I5122
sa(dp10244
S'a'
I23197
sS's'
I16
sS'm'
(lp10245
I1
aI1
aI3
aI9
aI31
aI59
aI61
aI15
aI423
aI33
aI467
aI1817
aI6535
aI7341
aI31061
aI20937
asS'd'
I5123
sa(dp10246
S'a'
I23221
sS's'
I16
sS'm'
(lp10247
I1
aI1
aI7
aI3
aI1
aI21
aI127
aI135
aI321
aI699
aI727
aI3079
aI753
aI3971
aI5611
aI28345
asS'd'
I5124
sa(dp10248
S'a'
I23257
sS's'
I16
sS'm'
(lp10249
I1
aI3
aI7
aI11
aI27
aI3
aI39
aI63
aI389
aI433
aI43
aI1443
aI6241
aI10769
aI20485
aI58823
asS'd'
I5125
sa(dp10250
S'a'
I23260
sS's'
I16
sS'm'
(lp10251
I1
aI1
aI1
aI11
aI3
aI13
aI5
aI57
aI503
aI707
aI677
aI3355
aI6691
aI8841
aI20041
aI11867
asS'd'
I5126
sa(dp10252
S'a'
I23263
sS's'
I16
sS'm'
(lp10253
I1
aI1
aI3
aI11
aI31
aI39
aI107
aI221
aI81
aI125
aI1439
aI2429
aI2109
aI3931
aI31007
aI10915
asS'd'
I5127
sa(dp10254
S'a'
I23267
sS's'
I16
sS'm'
(lp10255
I1
aI3
aI3
aI9
aI17
aI53
aI13
aI121
aI127
aI15
aI1825
aI1909
aI5951
aI13503
aI31565
aI56163
asS'd'
I5128
sa(dp10256
S'a'
I23282
sS's'
I16
sS'm'
(lp10257
I1
aI1
aI1
aI1
aI19
aI55
aI3
aI153
aI385
aI277
aI1297
aI3655
aI6027
aI3057
aI11341
aI46989
asS'd'
I5129
sa(dp10258
S'a'
I23284
sS's'
I16
sS'm'
(lp10259
I1
aI1
aI5
aI9
aI3
aI55
aI37
aI223
aI353
aI141
aI1917
aI3827
aI2255
aI7617
aI10971
aI10641
asS'd'
I5130
sa(dp10260
S'a'
I23314
sS's'
I16
sS'm'
(lp10261
I1
aI3
aI7
aI9
aI29
aI41
aI71
aI19
aI137
aI243
aI2047
aI395
aI6981
aI15887
aI9479
aI60199
asS'd'
I5131
sa(dp10262
S'a'
I23326
sS's'
I16
sS'm'
(lp10263
I1
aI1
aI1
aI9
aI17
aI43
aI51
aI191
aI405
aI727
aI485
aI987
aI1855
aI15801
aI22529
aI10165
asS'd'
I5132
sa(dp10264
S'a'
I23335
sS's'
I16
sS'm'
(lp10265
I1
aI3
aI1
aI7
aI27
aI31
aI69
aI255
aI153
aI793
aI1353
aI1981
aI83
aI11387
aI6747
aI23379
asS'd'
I5133
sa(dp10266
S'a'
I23368
sS's'
I16
sS'm'
(lp10267
I1
aI1
aI5
aI5
aI31
aI49
aI83
aI157
aI63
aI647
aI1367
aI3995
aI5899
aI8429
aI18317
aI3471
asS'd'
I5134
sa(dp10268
S'a'
I23373
sS's'
I16
sS'm'
(lp10269
I1
aI3
aI5
aI13
aI19
aI15
aI99
aI13
aI143
aI887
aI529
aI2855
aI573
aI9799
aI13585
aI59263
asS'd'
I5135
sa(dp10270
S'a'
I23401
sS's'
I16
sS'm'
(lp10271
I1
aI3
aI5
aI3
aI13
aI47
aI103
aI87
aI11
aI381
aI397
aI899
aI71
aI15539
aI13005
aI38297
asS'd'
I5136
sa(dp10272
S'a'
I23415
sS's'
I16
sS'm'
(lp10273
I1
aI1
aI1
aI3
aI1
aI53
aI45
aI141
aI1
aI941
aI261
aI3291
aI5177
aI9843
aI6309
aI62799
asS'd'
I5137
sa(dp10274
S'a'
I23432
sS's'
I16
sS'm'
(lp10275
I1
aI1
aI5
aI9
aI29
aI31
aI107
aI57
aI135
aI229
aI1147
aI247
aI265
aI12757
aI21365
aI7219
asS'd'
I5138
sa(dp10276
S'a'
I23476
sS's'
I16
sS'm'
(lp10277
I1
aI1
aI1
aI3
aI1
aI49
aI55
aI247
aI495
aI449
aI141
aI1349
aI7393
aI2589
aI23587
aI1097
asS'd'
I5139
sa(dp10278
S'a'
I23479
sS's'
I16
sS'm'
(lp10279
I1
aI3
aI5
aI1
aI9
aI11
aI73
aI153
aI89
aI575
aI1805
aI137
aI435
aI3687
aI32169
aI24275
asS'd'
I5140
sa(dp10280
S'a'
I23497
sS's'
I16
sS'm'
(lp10281
I1
aI1
aI7
aI15
aI25
aI61
aI51
aI21
aI109
aI139
aI611
aI3907
aI6721
aI5081
aI6665
aI51463
asS'd'
I5141
sa(dp10282
S'a'
I23505
sS's'
I16
sS'm'
(lp10283
I1
aI1
aI1
aI9
aI13
aI23
aI59
aI203
aI33
aI1013
aI1533
aI291
aI6171
aI15463
aI8581
aI9497
asS'd'
I5142
sa(dp10284
S'a'
I23508
sS's'
I16
sS'm'
(lp10285
I1
aI3
aI3
aI9
aI7
aI25
aI51
aI189
aI49
aI265
aI1163
aI1141
aI2467
aI7839
aI7083
aI65527
asS'd'
I5143
sa(dp10286
S'a'
I23531
sS's'
I16
sS'm'
(lp10287
I1
aI1
aI3
aI9
aI19
aI33
aI77
aI9
aI181
aI919
aI623
aI1521
aI7853
aI2199
aI24115
aI60607
asS'd'
I5144
sa(dp10288
S'a'
I23536
sS's'
I16
sS'm'
(lp10289
I1
aI1
aI3
aI11
aI9
aI11
aI27
aI57
aI355
aI313
aI151
aI3391
aI4869
aI12541
aI30031
aI29455
asS'd'
I5145
sa(dp10290
S'a'
I23542
sS's'
I16
sS'm'
(lp10291
I1
aI3
aI5
aI9
aI17
aI13
aI91
aI123
aI235
aI841
aI1113
aI1451
aI501
aI3863
aI32483
aI9445
asS'd'
I5146
sa(dp10292
S'a'
I23565
sS's'
I16
sS'm'
(lp10293
I1
aI3
aI7
aI3
aI13
aI25
aI87
aI243
aI449
aI293
aI1279
aI3571
aI2693
aI13459
aI5895
aI38127
asS'd'
I5147
sa(dp10294
S'a'
I23573
sS's'
I16
sS'm'
(lp10295
I1
aI1
aI3
aI15
aI27
aI61
aI7
aI57
aI255
aI971
aI131
aI2585
aI2187
aI7191
aI17779
aI34587
asS'd'
I5148
sa(dp10296
S'a'
I23590
sS's'
I16
sS'm'
(lp10297
I1
aI3
aI3
aI7
aI23
aI29
aI55
aI41
aI463
aI873
aI1781
aI2851
aI4731
aI9819
aI25587
aI32199
asS'd'
I5149
sa(dp10298
S'a'
I23593
sS's'
I16
sS'm'
(lp10299
I1
aI1
aI1
aI9
aI29
aI39
aI25
aI143
aI171
aI141
aI223
aI467
aI4417
aI9575
aI23159
aI33819
asS'd'
I5150
sa(dp10300
S'a'
I23604
sS's'
I16
sS'm'
(lp10301
I1
aI1
aI5
aI13
aI19
aI61
aI19
aI75
aI25
aI361
aI585
aI1627
aI2231
aI1831
aI24885
aI37917
asS'd'
I5151
sa(dp10302
S'a'
I23621
sS's'
I16
sS'm'
(lp10303
I1
aI3
aI7
aI7
aI23
aI19
aI59
aI55
aI323
aI55
aI143
aI131
aI27
aI15363
aI26423
aI43963
asS'd'
I5152
sa(dp10304
S'a'
I23622
sS's'
I16
sS'm'
(lp10305
I1
aI1
aI5
aI5
aI25
aI9
aI33
aI17
aI427
aI481
aI315
aI3999
aI4757
aI4545
aI7695
aI56733
asS'd'
I5153
sa(dp10306
S'a'
I23636
sS's'
I16
sS'm'
(lp10307
I1
aI3
aI5
aI13
aI5
aI59
aI45
aI117
aI115
aI263
aI1441
aI3307
aI1085
aI3875
aI25869
aI19725
asS'd'
I5154
sa(dp10308
S'a'
I23662
sS's'
I16
sS'm'
(lp10309
I1
aI3
aI3
aI15
aI13
aI39
aI31
aI243
aI293
aI345
aI343
aI1911
aI6123
aI12577
aI32081
aI59993
asS'd'
I5155
sa(dp10310
S'a'
I23667
sS's'
I16
sS'm'
(lp10311
I1
aI3
aI5
aI13
aI17
aI37
aI105
aI201
aI205
aI929
aI435
aI1467
aI8063
aI6963
aI13709
aI53275
asS'd'
I5156
sa(dp10312
S'a'
I23703
sS's'
I16
sS'm'
(lp10313
I1
aI3
aI7
aI15
aI31
aI3
aI65
aI221
aI191
aI413
aI427
aI2579
aI377
aI2793
aI26943
aI61299
asS'd'
I5157
sa(dp10314
S'a'
I23725
sS's'
I16
sS'm'
(lp10315
I1
aI3
aI5
aI3
aI11
aI61
aI75
aI107
aI53
aI689
aI1845
aI859
aI333
aI889
aI27607
aI48795
asS'd'
I5158
sa(dp10316
S'a'
I23734
sS's'
I16
sS'm'
(lp10317
I1
aI1
aI5
aI7
aI1
aI11
aI37
aI181
aI323
aI187
aI1511
aI25
aI5517
aI11957
aI7093
aI429
asS'd'
I5159
sa(dp10318
S'a'
I23738
sS's'
I16
sS'm'
(lp10319
I1
aI3
aI3
aI1
aI25
aI31
aI125
aI139
aI433
aI583
aI683
aI587
aI5389
aI1225
aI26047
aI18717
asS'd'
I5160
sa(dp10320
S'a'
I23752
sS's'
I16
sS'm'
(lp10321
I1
aI3
aI1
aI15
aI23
aI33
aI23
aI147
aI279
aI513
aI157
aI4023
aI2669
aI7543
aI2111
aI9191
asS'd'
I5161
sa(dp10322
S'a'
I23800
sS's'
I16
sS'm'
(lp10323
I1
aI3
aI1
aI1
aI5
aI39
aI55
aI127
aI257
aI649
aI347
aI3001
aI2215
aI15579
aI29665
aI10337
asS'd'
I5162
sa(dp10324
S'a'
I23803
sS's'
I16
sS'm'
(lp10325
I1
aI1
aI1
aI15
aI19
aI55
aI105
aI183
aI505
aI1003
aI1311
aI2253
aI1861
aI3561
aI19581
aI46183
asS'd'
I5163
sa(dp10326
S'a'
I23818
sS's'
I16
sS'm'
(lp10327
I1
aI3
aI1
aI9
aI23
aI5
aI127
aI215
aI195
aI817
aI719
aI1285
aI919
aI8627
aI20427
aI2723
asS'd'
I5164
sa(dp10328
S'a'
I23832
sS's'
I16
sS'm'
(lp10329
I1
aI1
aI1
aI5
aI15
aI31
aI43
aI131
aI377
aI57
aI1531
aI915
aI2871
aI1805
aI19765
aI60529
asS'd'
I5165
sa(dp10330
S'a'
I23842
sS's'
I16
sS'm'
(lp10331
I1
aI3
aI3
aI3
aI15
aI1
aI93
aI55
aI477
aI221
aI643
aI1319
aI959
aI475
aI14015
aI48823
asS'd'
I5166
sa(dp10332
S'a'
I23851
sS's'
I16
sS'm'
(lp10333
I1
aI3
aI3
aI7
aI19
aI13
aI13
aI191
aI421
aI751
aI1103
aI2129
aI1973
aI14935
aI26485
aI41269
asS'd'
I5167
sa(dp10334
S'a'
I23873
sS's'
I16
sS'm'
(lp10335
I1
aI1
aI5
aI13
aI17
aI43
aI81
aI83
aI67
aI643
aI1799
aI1157
aI4365
aI2815
aI29871
aI5351
asS'd'
I5168
sa(dp10336
S'a'
I23883
sS's'
I16
sS'm'
(lp10337
I1
aI3
aI1
aI9
aI21
aI31
aI87
aI177
aI25
aI403
aI1357
aI4047
aI959
aI5133
aI7307
aI4333
asS'd'
I5169
sa(dp10338
S'a'
I23888
sS's'
I16
sS'm'
(lp10339
I1
aI1
aI7
aI7
aI29
aI39
aI27
aI139
aI159
aI529
aI1323
aI3823
aI4569
aI12711
aI30263
aI10961
asS'd'
I5170
sa(dp10340
S'a'
I23910
sS's'
I16
sS'm'
(lp10341
I1
aI3
aI1
aI13
aI27
aI15
aI59
aI1
aI107
aI723
aI497
aI43
aI143
aI16119
aI29177
aI5653
asS'd'
I5171
sa(dp10342
S'a'
I23934
sS's'
I16
sS'm'
(lp10343
I1
aI1
aI5
aI9
aI15
aI41
aI23
aI109
aI101
aI639
aI277
aI1687
aI1311
aI1995
aI5403
aI6867
asS'd'
I5172
sa(dp10344
S'a'
I23938
sS's'
I16
sS'm'
(lp10345
I1
aI1
aI5
aI3
aI13
aI1
aI95
aI177
aI379
aI545
aI789
aI3479
aI4135
aI445
aI5869
aI38923
asS'd'
I5173
sa(dp10346
S'a'
I23949
sS's'
I16
sS'm'
(lp10347
I1
aI1
aI3
aI9
aI21
aI31
aI23
aI65
aI209
aI383
aI271
aI367
aI6605
aI1169
aI27267
aI9331
asS'd'
I5174
sa(dp10348
S'a'
I23955
sS's'
I16
sS'm'
(lp10349
I1
aI1
aI1
aI1
aI27
aI39
aI121
aI29
aI155
aI465
aI947
aI2675
aI6753
aI11647
aI29781
aI30251
asS'd'
I5175
sa(dp10350
S'a'
I23967
sS's'
I16
sS'm'
(lp10351
I1
aI3
aI1
aI5
aI7
aI45
aI7
aI21
aI223
aI363
aI1021
aI751
aI2257
aI14423
aI19629
aI64867
asS'd'
I5176
sa(dp10352
S'a'
I23973
sS's'
I16
sS'm'
(lp10353
I1
aI3
aI1
aI9
aI31
aI53
aI13
aI99
aI49
aI389
aI867
aI327
aI4145
aI3265
aI15423
aI14985
asS'd'
I5177
sa(dp10354
S'a'
I23991
sS's'
I16
sS'm'
(lp10355
I1
aI1
aI1
aI15
aI11
aI11
aI27
aI41
aI333
aI161
aI1343
aI2023
aI3789
aI13563
aI16957
aI26849
asS'd'
I5178
sa(dp10356
S'a'
I24024
sS's'
I16
sS'm'
(lp10357
I1
aI3
aI7
aI1
aI7
aI51
aI7
aI163
aI239
aI393
aI231
aI3985
aI309
aI875
aI837
aI24791
asS'd'
I5179
sa(dp10358
S'a'
I24030
sS's'
I16
sS'm'
(lp10359
I1
aI1
aI1
aI7
aI1
aI43
aI105
aI7
aI351
aI755
aI1781
aI1925
aI4961
aI2961
aI13427
aI44317
asS'd'
I5180
sa(dp10360
S'a'
I24039
sS's'
I16
sS'm'
(lp10361
I1
aI3
aI1
aI9
aI17
aI39
aI81
aI75
aI201
aI931
aI1547
aI1857
aI7251
aI11687
aI14437
aI12603
asS'd'
I5181
sa(dp10362
S'a'
I24067
sS's'
I16
sS'm'
(lp10363
I1
aI3
aI3
aI15
aI15
aI23
aI95
aI7
aI193
aI9
aI1749
aI809
aI5083
aI14645
aI24893
aI35979
asS'd'
I5182
sa(dp10364
S'a'
I24069
sS's'
I16
sS'm'
(lp10365
I1
aI1
aI7
aI1
aI9
aI9
aI127
aI149
aI433
aI985
aI1347
aI3379
aI2881
aI681
aI20777
aI30703
asS'd'
I5183
sa(dp10366
S'a'
I24084
sS's'
I16
sS'm'
(lp10367
I1
aI3
aI1
aI11
aI1
aI27
aI121
aI111
aI251
aI45
aI1341
aI1709
aI3733
aI11297
aI29063
aI57119
asS'd'
I5184
sa(dp10368
S'a'
I24112
sS's'
I16
sS'm'
(lp10369
I1
aI3
aI3
aI1
aI19
aI3
aI77
aI127
aI187
aI831
aI1125
aI3119
aI4665
aI9857
aI5301
aI36755
asS'd'
I5185
sa(dp10370
S'a'
I24115
sS's'
I16
sS'm'
(lp10371
I1
aI3
aI3
aI3
aI29
aI29
aI61
aI225
aI257
aI635
aI665
aI1279
aI3019
aI2401
aI8253
aI40251
asS'd'
I5186
sa(dp10372
S'a'
I24184
sS's'
I16
sS'm'
(lp10373
I1
aI1
aI7
aI9
aI29
aI43
aI47
aI95
aI221
aI823
aI257
aI1485
aI5183
aI225
aI27675
aI60479
asS'd'
I5187
sa(dp10374
S'a'
I24187
sS's'
I16
sS'm'
(lp10375
I1
aI1
aI5
aI3
aI15
aI49
aI25
aI69
aI101
aI393
aI901
aI305
aI4917
aI13479
aI30209
aI9199
asS'd'
I5188
sa(dp10376
S'a'
I24262
sS's'
I16
sS'm'
(lp10377
I1
aI1
aI3
aI15
aI1
aI9
aI47
aI243
aI403
aI341
aI143
aI1365
aI1817
aI6017
aI3853
aI58625
asS'd'
I5189
sa(dp10378
S'a'
I24276
sS's'
I16
sS'm'
(lp10379
I1
aI3
aI3
aI11
aI9
aI49
aI93
aI149
aI39
aI177
aI1863
aI1027
aI659
aI9253
aI2131
aI26943
asS'd'
I5190
sa(dp10380
S'a'
I24279
sS's'
I16
sS'm'
(lp10381
I1
aI3
aI1
aI3
aI25
aI1
aI43
aI255
aI217
aI353
aI957
aI39
aI4607
aI15761
aI24291
aI33619
asS'd'
I5191
sa(dp10382
S'a'
I24313
sS's'
I16
sS'm'
(lp10383
I1
aI1
aI1
aI7
aI29
aI51
aI71
aI237
aI1
aI703
aI19
aI213
aI6429
aI11783
aI22049
aI30597
asS'd'
I5192
sa(dp10384
S'a'
I24331
sS's'
I16
sS'm'
(lp10385
I1
aI3
aI1
aI7
aI31
aI63
aI105
aI203
aI473
aI731
aI257
aI91
aI5749
aI4099
aI30125
aI40171
asS'd'
I5193
sa(dp10386
S'a'
I24336
sS's'
I16
sS'm'
(lp10387
I1
aI3
aI7
aI9
aI7
aI29
aI119
aI181
aI225
aI743
aI1031
aI55
aI3997
aI16221
aI11663
aI14847
asS'd'
I5194
sa(dp10388
S'a'
I24348
sS's'
I16
sS'm'
(lp10389
I1
aI3
aI3
aI11
aI5
aI21
aI43
aI17
aI473
aI981
aI125
aI2077
aI6141
aI4757
aI7585
aI29207
asS'd'
I5195
sa(dp10390
S'a'
I24367
sS's'
I16
sS'm'
(lp10391
I1
aI1
aI7
aI1
aI27
aI61
aI27
aI45
aI267
aI483
aI119
aI767
aI957
aI3411
aI2653
aI53967
asS'd'
I5196
sa(dp10392
S'a'
I24372
sS's'
I16
sS'm'
(lp10393
I1
aI1
aI1
aI3
aI9
aI41
aI43
aI17
aI485
aI253
aI825
aI3605
aI5919
aI9285
aI1815
aI6095
asS'd'
I5197
sa(dp10394
S'a'
I24402
sS's'
I16
sS'm'
(lp10395
I1
aI1
aI1
aI11
aI7
aI5
aI53
aI107
aI309
aI609
aI1389
aI2035
aI861
aI15565
aI9375
aI42363
asS'd'
I5198
sa(dp10396
S'a'
I24420
sS's'
I16
sS'm'
(lp10397
I1
aI3
aI5
aI3
aI27
aI57
aI7
aI177
aI183
aI227
aI865
aI183
aI2903
aI6325
aI4393
aI5257
asS'd'
I5199
sa(dp10398
S'a'
I24444
sS's'
I16
sS'm'
(lp10399
I1
aI3
aI1
aI5
aI21
aI29
aI79
aI107
aI365
aI427
aI813
aI3563
aI7713
aI3865
aI4289
aI28555
asS'd'
I5200
sa(dp10400
S'a'
I24465
sS's'
I16
sS'm'
(lp10401
I1
aI3
aI5
aI7
aI11
aI27
aI1
aI197
aI425
aI769
aI1737
aI3627
aI1273
aI4469
aI11967
aI823
asS'd'
I5201
sa(dp10402
S'a'
I24466
sS's'
I16
sS'm'
(lp10403
I1
aI1
aI1
aI13
aI3
aI31
aI127
aI159
aI471
aI367
aI2047
aI949
aI1591
aI12429
aI21497
aI27153
asS'd'
I5202
sa(dp10404
S'a'
I24471
sS's'
I16
sS'm'
(lp10405
I1
aI3
aI1
aI3
aI3
aI31
aI31
aI87
aI243
aI413
aI1777
aI1361
aI4575
aI1147
aI17451
aI33985
asS'd'
I5203
sa(dp10406
S'a'
I24475
sS's'
I16
sS'm'
(lp10407
I1
aI3
aI3
aI5
aI13
aI47
aI45
aI3
aI165
aI329
aI743
aI397
aI2479
aI4999
aI12921
aI26689
asS'd'
I5204
sa(dp10408
S'a'
I24481
sS's'
I16
sS'm'
(lp10409
I1
aI1
aI5
aI7
aI17
aI59
aI25
aI117
aI217
aI601
aI235
aI2691
aI5569
aI7853
aI31063
aI28281
asS'd'
I5205
sa(dp10410
S'a'
I24488
sS's'
I16
sS'm'
(lp10411
I1
aI3
aI1
aI11
aI11
aI39
aI71
aI77
aI481
aI39
aI363
aI1921
aI3263
aI12849
aI13325
aI41803
asS'd'
I5206
sa(dp10412
S'a'
I24501
sS's'
I16
sS'm'
(lp10413
I1
aI3
aI7
aI5
aI19
aI1
aI59
aI197
aI239
aI787
aI1657
aI1449
aI4245
aI1317
aI19609
aI53583
asS'd'
I5207
sa(dp10414
S'a'
I24514
sS's'
I16
sS'm'
(lp10415
I1
aI1
aI7
aI1
aI13
aI33
aI81
aI53
aI223
aI323
aI477
aI2421
aI4045
aI1855
aI5823
aI9661
asS'd'
I5208
sa(dp10416
S'a'
I24531
sS's'
I16
sS'm'
(lp10417
I1
aI3
aI1
aI7
aI1
aI3
aI121
aI19
aI329
aI569
aI481
aI3443
aI499
aI12407
aI19625
aI4627
asS'd'
I5209
sa(dp10418
S'a'
I24534
sS's'
I16
sS'm'
(lp10419
I1
aI1
aI1
aI7
aI3
aI33
aI91
aI241
aI69
aI581
aI1635
aI1025
aI4677
aI14651
aI5229
aI19555
asS'd'
I5210
sa(dp10420
S'a'
I24559
sS's'
I16
sS'm'
(lp10421
I1
aI3
aI1
aI15
aI9
aI11
aI99
aI47
aI489
aI755
aI601
aI1221
aI4251
aI4377
aI20567
aI33839
asS'd'
I5211
sa(dp10422
S'a'
I24602
sS's'
I16
sS'm'
(lp10423
I1
aI3
aI3
aI5
aI21
aI21
aI127
aI151
aI499
aI971
aI1627
aI609
aI2365
aI3183
aI7413
aI697
asS'd'
I5212
sa(dp10424
S'a'
I24614
sS's'
I16
sS'm'
(lp10425
I1
aI3
aI7
aI13
aI5
aI35
aI61
aI121
aI51
aI297
aI29
aI1825
aI495
aI1299
aI12741
aI3253
asS'd'
I5213
sa(dp10426
S'a'
I24637
sS's'
I16
sS'm'
(lp10427
I1
aI1
aI1
aI1
aI13
aI15
aI49
aI205
aI235
aI9
aI849
aI1101
aI4533
aI10221
aI32419
aI50151
asS'd'
I5214
sa(dp10428
S'a'
I24664
sS's'
I16
sS'm'
(lp10429
I1
aI1
aI3
aI13
aI29
aI31
aI57
aI77
aI217
aI195
aI1951
aI189
aI981
aI6169
aI22677
aI64415
asS'd'
I5215
sa(dp10430
S'a'
I24676
sS's'
I16
sS'm'
(lp10431
I1
aI3
aI1
aI5
aI15
aI37
aI25
aI233
aI43
aI843
aI1205
aI153
aI6339
aI3767
aI16725
aI32699
asS'd'
I5216
sa(dp10432
S'a'
I24679
sS's'
I16
sS'm'
(lp10433
I1
aI3
aI3
aI1
aI19
aI37
aI11
aI217
aI461
aI897
aI1181
aI2815
aI295
aI15153
aI25351
aI57211
asS'd'
I5217
sa(dp10434
S'a'
I24697
sS's'
I16
sS'm'
(lp10435
I1
aI3
aI5
aI11
aI15
aI9
aI5
aI179
aI353
aI769
aI1457
aI2919
aI1201
aI14871
aI29549
aI52265
asS'd'
I5218
sa(dp10436
S'a'
I24709
sS's'
I16
sS'm'
(lp10437
I1
aI1
aI3
aI3
aI5
aI51
aI127
aI221
aI329
aI543
aI1537
aI2701
aI2709
aI9311
aI2715
aI42669
asS'd'
I5219
sa(dp10438
S'a'
I24714
sS's'
I16
sS'm'
(lp10439
I1
aI3
aI5
aI15
aI5
aI41
aI79
aI233
aI445
aI265
aI2001
aI935
aI3133
aI3977
aI3601
aI21389
asS'd'
I5220
sa(dp10440
S'a'
I24716
sS's'
I16
sS'm'
(lp10441
I1
aI3
aI5
aI11
aI3
aI7
aI115
aI45
aI311
aI827
aI1897
aI3399
aI4251
aI5341
aI22621
aI25519
asS'd'
I5221
sa(dp10442
S'a'
I24731
sS's'
I16
sS'm'
(lp10443
I1
aI3
aI7
aI1
aI11
aI57
aI117
aI103
aI401
aI505
aI1683
aI2161
aI4363
aI11189
aI20263
aI13065
asS'd'
I5222
sa(dp10444
S'a'
I24744
sS's'
I16
sS'm'
(lp10445
I1
aI1
aI1
aI7
aI23
aI29
aI31
aI77
aI63
aI179
aI195
aI2747
aI579
aI11701
aI5101
aI11497
asS'd'
I5223
sa(dp10446
S'a'
I24762
sS's'
I16
sS'm'
(lp10447
I1
aI3
aI7
aI3
aI9
aI33
aI81
aI165
aI433
aI545
aI1893
aI3709
aI3813
aI6615
aI1485
aI6395
asS'd'
I5224
sa(dp10448
S'a'
I24764
sS's'
I16
sS'm'
(lp10449
I1
aI3
aI1
aI15
aI9
aI5
aI27
aI157
aI389
aI683
aI1919
aI509
aI455
aI3865
aI2303
aI6993
asS'd'
I5225
sa(dp10450
S'a'
I24769
sS's'
I16
sS'm'
(lp10451
I1
aI3
aI3
aI9
aI5
aI5
aI3
aI5
aI299
aI59
aI539
aI1199
aI2443
aI10821
aI3359
aI44345
asS'd'
I5226
sa(dp10452
S'a'
I24806
sS's'
I16
sS'm'
(lp10453
I1
aI3
aI5
aI9
aI21
aI37
aI87
aI35
aI501
aI943
aI1313
aI3929
aI351
aI9851
aI22971
aI35659
asS'd'
I5227
sa(dp10454
S'a'
I24812
sS's'
I16
sS'm'
(lp10455
I1
aI3
aI7
aI11
aI11
aI33
aI77
aI175
aI411
aI315
aI1797
aI2731
aI4611
aI1809
aI22027
aI50595
asS'd'
I5228
sa(dp10456
S'a'
I24872
sS's'
I16
sS'm'
(lp10457
I1
aI3
aI7
aI13
aI15
aI11
aI13
aI189
aI209
aI1015
aI1869
aI1593
aI6887
aI8571
aI7217
aI2641
asS'd'
I5229
sa(dp10458
S'a'
I24878
sS's'
I16
sS'm'
(lp10459
I1
aI1
aI3
aI13
aI29
aI15
aI119
aI127
aI329
aI275
aI865
aI1693
aI225
aI15735
aI11071
aI37127
asS'd'
I5230
sa(dp10460
S'a'
I24883
sS's'
I16
sS'm'
(lp10461
I1
aI3
aI7
aI13
aI9
aI31
aI85
aI25
aI281
aI401
aI1923
aI2391
aI3875
aI2079
aI2055
aI53275
asS'd'
I5231
sa(dp10462
S'a'
I24909
sS's'
I16
sS'm'
(lp10463
I1
aI3
aI1
aI5
aI23
aI57
aI79
aI127
aI209
aI901
aI315
aI1165
aI3393
aI15095
aI1981
aI34993
asS'd'
I5232
sa(dp10464
S'a'
I24943
sS's'
I16
sS'm'
(lp10465
I1
aI3
aI7
aI7
aI25
aI13
aI15
aI223
aI157
aI335
aI1061
aI395
aI6895
aI6283
aI18375
aI4887
asS'd'
I5233
sa(dp10466
S'a'
I24946
sS's'
I16
sS'm'
(lp10467
I1
aI3
aI7
aI13
aI9
aI15
aI99
aI201
aI105
aI643
aI117
aI3027
aI641
aI13353
aI4343
aI11875
asS'd'
I5234
sa(dp10468
S'a'
I24964
sS's'
I16
sS'm'
(lp10469
I1
aI1
aI3
aI11
aI5
aI51
aI5
aI77
aI281
aI207
aI1201
aI1187
aI8107
aI6625
aI7517
aI34377
asS'd'
I5235
sa(dp10470
S'a'
I24992
sS's'
I16
sS'm'
(lp10471
I1
aI1
aI1
aI5
aI29
aI61
aI3
aI181
aI297
aI151
aI565
aI2713
aI6611
aI8665
aI32425
aI50399
asS'd'
I5236
sa(dp10472
S'a'
I25016
sS's'
I16
sS'm'
(lp10473
I1
aI3
aI1
aI5
aI1
aI61
aI41
aI245
aI95
aI647
aI49
aI2195
aI5927
aI15531
aI28547
aI51075
asS'd'
I5237
sa(dp10474
S'a'
I25022
sS's'
I16
sS'm'
(lp10475
I1
aI3
aI3
aI15
aI15
aI63
aI123
aI63
aI77
aI813
aI423
aI715
aI91
aI3793
aI20901
aI54927
asS'd'
I5238
sa(dp10476
S'a'
I25024
sS's'
I16
sS'm'
(lp10477
I1
aI3
aI7
aI9
aI15
aI35
aI31
aI161
aI127
aI13
aI1667
aI1225
aI5279
aI15487
aI18769
aI24887
asS'd'
I5239
sa(dp10478
S'a'
I25039
sS's'
I16
sS'm'
(lp10479
I1
aI1
aI3
aI5
aI27
aI25
aI61
aI21
aI447
aI175
aI1419
aI2691
aI1993
aI13059
aI30809
aI38325
asS'd'
I5240
sa(dp10480
S'a'
I25051
sS's'
I16
sS'm'
(lp10481
I1
aI3
aI1
aI3
aI15
aI21
aI15
aI193
aI233
aI435
aI1993
aI4003
aI4581
aI13837
aI16535
aI43781
asS'd'
I5241
sa(dp10482
S'a'
I25060
sS's'
I16
sS'm'
(lp10483
I1
aI1
aI1
aI5
aI1
aI11
aI21
aI253
aI59
aI59
aI497
aI77
aI2165
aI8197
aI5933
aI25743
asS'd'
I5242
sa(dp10484
S'a'
I25072
sS's'
I16
sS'm'
(lp10485
I1
aI1
aI3
aI9
aI25
aI61
aI29
aI217
aI95
aI269
aI799
aI409
aI801
aI421
aI19065
aI53443
asS'd'
I5243
sa(dp10486
S'a'
I25097
sS's'
I16
sS'm'
(lp10487
I1
aI1
aI7
aI1
aI1
aI41
aI71
aI59
aI191
aI867
aI223
aI1467
aI6679
aI16031
aI4451
aI15313
asS'd'
I5244
sa(dp10488
S'a'
I25122
sS's'
I16
sS'm'
(lp10489
I1
aI1
aI1
aI11
aI13
aI27
aI7
aI241
aI167
aI969
aI1267
aI2953
aI3769
aI2415
aI30065
aI39483
asS'd'
I5245
sa(dp10490
S'a'
I25148
sS's'
I16
sS'm'
(lp10491
I1
aI1
aI1
aI3
aI25
aI61
aI103
aI23
aI53
aI799
aI989
aI3859
aI7299
aI13613
aI12007
aI37535
asS'd'
I5246
sa(dp10492
S'a'
I25194
sS's'
I16
sS'm'
(lp10493
I1
aI1
aI7
aI1
aI29
aI19
aI121
aI57
aI355
aI663
aI643
aI3723
aI1775
aI10363
aI1389
aI16039
asS'd'
I5247
sa(dp10494
S'a'
I25201
sS's'
I16
sS'm'
(lp10495
I1
aI3
aI7
aI3
aI21
aI55
aI51
aI67
aI363
aI843
aI1187
aI1983
aI7757
aI5413
aI26575
aI53329
asS'd'
I5248
sa(dp10496
S'a'
I25204
sS's'
I16
sS'm'
(lp10497
I1
aI3
aI1
aI3
aI31
aI55
aI73
aI55
aI75
aI533
aI197
aI1463
aI2933
aI6033
aI24397
aI41579
asS'd'
I5249
sa(dp10498
S'a'
I25238
sS's'
I16
sS'm'
(lp10499
I1
aI3
aI1
aI11
aI9
aI15
aI107
aI141
aI473
aI825
aI901
aI937
aI7433
aI13119
aI20047
aI6695
asS'd'
I5250
sa(dp10500
S'a'
I25241
sS's'
I16
sS'm'
(lp10501
I1
aI3
aI5
aI7
aI19
aI27
aI3
aI149
aI507
aI137
aI1025
aI1841
aI33
aI3113
aI15101
aI28187
asS'd'
I5251
sa(dp10502
S'a'
I25248
sS's'
I16
sS'm'
(lp10503
I1
aI3
aI5
aI7
aI31
aI27
aI53
aI45
aI177
aI129
aI1241
aI1525
aI991
aI4141
aI17681
aI39435
asS'd'
I5252
sa(dp10504
S'a'
I25257
sS's'
I16
sS'm'
(lp10505
I1
aI1
aI1
aI15
aI31
aI57
aI29
aI137
aI395
aI563
aI101
aI3367
aI1277
aI5431
aI1169
aI44321
asS'd'
I5253
sa(dp10506
S'a'
I25275
sS's'
I16
sS'm'
(lp10507
I1
aI3
aI5
aI7
aI21
aI15
aI123
aI181
aI113
aI313
aI1763
aI1429
aI2985
aI715
aI26129
aI549
asS'd'
I5254
sa(dp10508
S'a'
I25278
sS's'
I16
sS'm'
(lp10509
I1
aI3
aI3
aI1
aI15
aI27
aI27
aI139
aI507
aI79
aI1999
aI2505
aI485
aI7011
aI13369
aI7159
asS'd'
I5255
sa(dp10510
S'a'
I25304
sS's'
I16
sS'm'
(lp10511
I1
aI3
aI3
aI11
aI27
aI53
aI109
aI179
aI399
aI657
aI697
aI421
aI5467
aI4273
aI7837
aI11631
asS'd'
I5256
sa(dp10512
S'a'
I25307
sS's'
I16
sS'm'
(lp10513
I1
aI1
aI1
aI15
aI1
aI57
aI91
aI199
aI443
aI569
aI1945
aI2531
aI6349
aI4851
aI3931
aI20537
asS'd'
I5257
sa(dp10514
S'a'
I25320
sS's'
I16
sS'm'
(lp10515
I1
aI1
aI3
aI13
aI3
aI3
aI107
aI237
aI261
aI377
aI135
aI2809
aI7239
aI1613
aI24035
aI26053
asS'd'
I5258
sa(dp10516
S'a'
I25334
sS's'
I16
sS'm'
(lp10517
I1
aI3
aI3
aI5
aI3
aI59
aI65
aI197
aI411
aI363
aI1729
aI967
aI3963
aI4535
aI111
aI7273
asS'd'
I5259
sa(dp10518
S'a'
I25348
sS's'
I16
sS'm'
(lp10519
I1
aI1
aI7
aI3
aI13
aI39
aI105
aI235
aI203
aI1015
aI1031
aI3127
aI1209
aI10817
aI22177
aI44117
asS'd'
I5260
sa(dp10520
S'a'
I25357
sS's'
I16
sS'm'
(lp10521
I1
aI3
aI3
aI13
aI19
aI21
aI123
aI31
aI59
aI185
aI1007
aI1115
aI1965
aI13087
aI18489
aI34063
asS'd'
I5261
sa(dp10522
S'a'
I25372
sS's'
I16
sS'm'
(lp10523
I1
aI1
aI7
aI5
aI27
aI7
aI33
aI159
aI245
aI57
aI2003
aI3229
aI2095
aI4939
aI31355
aI23121
asS'd'
I5262
sa(dp10524
S'a'
I25394
sS's'
I16
sS'm'
(lp10525
I1
aI3
aI3
aI9
aI9
aI41
aI49
aI203
aI397
aI915
aI821
aI3685
aI2269
aI11159
aI25441
aI46377
asS'd'
I5263
sa(dp10526
S'a'
I25454
sS's'
I16
sS'm'
(lp10527
I1
aI3
aI7
aI5
aI29
aI33
aI29
aI227
aI361
aI961
aI1905
aI1149
aI2799
aI8115
aI28235
aI25685
asS'd'
I5264
sa(dp10528
S'a'
I25465
sS's'
I16
sS'm'
(lp10529
I1
aI3
aI1
aI1
aI19
aI13
aI73
aI103
aI11
aI183
aI853
aI2425
aI3809
aI2391
aI18615
aI32735
asS'd'
I5265
sa(dp10530
S'a'
I25472
sS's'
I16
sS'm'
(lp10531
I1
aI1
aI3
aI3
aI21
aI57
aI47
aI57
aI157
aI43
aI1085
aI3319
aI461
aI11499
aI6809
aI10435
asS'd'
I5266
sa(dp10532
S'a'
I25492
sS's'
I16
sS'm'
(lp10533
I1
aI1
aI5
aI5
aI17
aI21
aI55
aI17
aI421
aI865
aI159
aI1643
aI4523
aI1485
aI11937
aI8287
asS'd'
I5267
sa(dp10534
S'a'
I25505
sS's'
I16
sS'm'
(lp10535
I1
aI1
aI3
aI11
aI7
aI43
aI39
aI37
aI187
aI797
aI1273
aI1227
aI2683
aI1249
aI3375
aI44499
asS'd'
I5268
sa(dp10536
S'a'
I25517
sS's'
I16
sS'm'
(lp10537
I1
aI1
aI5
aI11
aI17
aI35
aI27
aI73
aI97
aI59
aI921
aI2171
aI7577
aI2847
aI93
aI35911
asS'd'
I5269
sa(dp10538
S'a'
I25530
sS's'
I16
sS'm'
(lp10539
I1
aI1
aI5
aI1
aI5
aI17
aI117
aI189
aI357
aI581
aI1945
aI2141
aI1679
aI12019
aI11249
aI6809
asS'd'
I5270
sa(dp10540
S'a'
I25558
sS's'
I16
sS'm'
(lp10541
I1
aI1
aI5
aI7
aI15
aI53
aI9
aI191
aI153
aI257
aI533
aI493
aI2389
aI4657
aI20757
aI57275
asS'd'
I5271
sa(dp10542
S'a'
I25562
sS's'
I16
sS'm'
(lp10543
I1
aI1
aI1
aI11
aI13
aI35
aI85
aI37
aI501
aI525
aI543
aI4057
aI2001
aI6183
aI949
aI18465
asS'd'
I5272
sa(dp10544
S'a'
I25598
sS's'
I16
sS'm'
(lp10545
I1
aI1
aI1
aI3
aI15
aI7
aI39
aI169
aI359
aI671
aI731
aI1523
aI211
aI1233
aI29515
aI57787
asS'd'
I5273
sa(dp10546
S'a'
I25609
sS's'
I16
sS'm'
(lp10547
I1
aI1
aI3
aI7
aI27
aI7
aI41
aI15
aI71
aI733
aI1919
aI401
aI1915
aI4815
aI10571
aI839
asS'd'
I5274
sa(dp10548
S'a'
I25612
sS's'
I16
sS'm'
(lp10549
I1
aI1
aI7
aI13
aI27
aI61
aI5
aI25
aI293
aI779
aI477
aI1537
aI6695
aI7435
aI1281
aI64369
asS'd'
I5275
sa(dp10550
S'a'
I25627
sS's'
I16
sS'm'
(lp10551
I1
aI1
aI7
aI7
aI19
aI11
aI101
aI45
aI449
aI883
aI1181
aI3521
aI6019
aI16305
aI23429
aI43789
asS'd'
I5276
sa(dp10552
S'a'
I25651
sS's'
I16
sS'm'
(lp10553
I1
aI1
aI7
aI5
aI1
aI49
aI121
aI89
aI275
aI367
aI461
aI1717
aI2733
aI4403
aI9123
aI35217
asS'd'
I5277
sa(dp10554
S'a'
I25653
sS's'
I16
sS'm'
(lp10555
I1
aI1
aI7
aI1
aI1
aI37
aI41
aI221
aI281
aI531
aI357
aI3783
aI3561
aI8135
aI18405
aI56045
asS'd'
I5278
sa(dp10556
S'a'
I25668
sS's'
I16
sS'm'
(lp10557
I1
aI3
aI5
aI7
aI29
aI9
aI127
aI37
aI13
aI519
aI1059
aI3973
aI7253
aI15159
aI19337
aI57103
asS'd'
I5279
sa(dp10558
S'a'
I25711
sS's'
I16
sS'm'
(lp10559
I1
aI3
aI3
aI15
aI3
aI41
aI91
aI7
aI63
aI747
aI1649
aI3367
aI5945
aI3603
aI28465
aI511
asS'd'
I5280
sa(dp10560
S'a'
I25765
sS's'
I16
sS'm'
(lp10561
I1
aI3
aI3
aI15
aI27
aI19
aI67
aI179
aI505
aI131
aI149
aI1753
aI3603
aI1135
aI15811
aI5305
asS'd'
I5281
sa(dp10562
S'a'
I25770
sS's'
I16
sS'm'
(lp10563
I1
aI1
aI1
aI5
aI5
aI63
aI71
aI235
aI151
aI651
aI1383
aI249
aI3223
aI14559
aI26809
aI4551
asS'd'
I5282
sa(dp10564
S'a'
I25775
sS's'
I16
sS'm'
(lp10565
I1
aI3
aI3
aI9
aI29
aI41
aI67
aI111
aI175
aI515
aI1123
aI1707
aI6653
aI8233
aI22775
aI61029
asS'd'
I5283
sa(dp10566
S'a'
I25777
sS's'
I16
sS'm'
(lp10567
I1
aI3
aI3
aI9
aI23
aI1
aI75
aI145
aI159
aI785
aI537
aI1995
aI2241
aI8641
aI30709
aI41373
asS'd'
I5284
sa(dp10568
S'a'
I25783
sS's'
I16
sS'm'
(lp10569
I1
aI1
aI5
aI9
aI21
aI1
aI87
aI233
aI401
aI643
aI197
aI3437
aI8163
aI6363
aI6537
aI17283
asS'd'
I5285
sa(dp10570
S'a'
I25801
sS's'
I16
sS'm'
(lp10571
I1
aI3
aI5
aI3
aI23
aI19
aI55
aI243
aI405
aI369
aI1221
aI1959
aI5455
aI11729
aI26117
aI9097
asS'd'
I5286
sa(dp10572
S'a'
I25802
sS's'
I16
sS'm'
(lp10573
I1
aI1
aI3
aI13
aI3
aI57
aI71
aI235
aI125
aI263
aI873
aI1079
aI2627
aI1343
aI1979
aI49519
asS'd'
I5287
sa(dp10574
S'a'
I25812
sS's'
I16
sS'm'
(lp10575
I1
aI3
aI1
aI11
aI21
aI15
aI27
aI7
aI425
aI935
aI305
aI2593
aI4437
aI9517
aI26207
aI4229
asS'd'
I5288
sa(dp10576
S'a'
I25821
sS's'
I16
sS'm'
(lp10577
I1
aI1
aI3
aI13
aI11
aI53
aI1
aI149
aI97
aI939
aI147
aI3365
aI5023
aI607
aI2083
aI8715
asS'd'
I5289
sa(dp10578
S'a'
I25897
sS's'
I16
sS'm'
(lp10579
I1
aI1
aI5
aI3
aI13
aI13
aI113
aI51
aI263
aI837
aI1145
aI3621
aI5697
aI2867
aI7975
aI22839
asS'd'
I5290
sa(dp10580
S'a'
I25906
sS's'
I16
sS'm'
(lp10581
I1
aI1
aI3
aI15
aI31
aI9
aI91
aI231
aI399
aI295
aI1935
aI4021
aI7669
aI3867
aI28015
aI9865
asS'd'
I5291
sa(dp10582
S'a'
I25929
sS's'
I16
sS'm'
(lp10583
I1
aI1
aI1
aI1
aI13
aI59
aI51
aI35
aI407
aI733
aI1629
aI2429
aI291
aI11923
aI32129
aI28847
asS'd'
I5292
sa(dp10584
S'a'
I25940
sS's'
I16
sS'm'
(lp10585
I1
aI3
aI3
aI11
aI25
aI21
aI13
aI117
aI31
aI547
aI327
aI2801
aI2247
aI4051
aI27523
aI4257
asS'd'
I5293
sa(dp10586
S'a'
I25950
sS's'
I16
sS'm'
(lp10587
I1
aI1
aI7
aI7
aI15
aI33
aI15
aI17
aI255
aI363
aI1013
aI1463
aI7537
aI14093
aI21883
aI35389
asS'd'
I5294
sa(dp10588
S'a'
I25968
sS's'
I16
sS'm'
(lp10589
I1
aI1
aI5
aI9
aI11
aI61
aI7
aI161
aI121
aI413
aI515
aI413
aI4439
aI15405
aI30265
aI23939
asS'd'
I5295
sa(dp10590
S'a'
I25971
sS's'
I16
sS'm'
(lp10591
I1
aI3
aI7
aI7
aI11
aI15
aI5
aI181
aI315
aI231
aI1567
aI2985
aI1653
aI12251
aI269
aI37317
asS'd'
I5296
sa(dp10592
S'a'
I25977
sS's'
I16
sS'm'
(lp10593
I1
aI3
aI1
aI11
aI3
aI15
aI91
aI45
aI489
aI571
aI11
aI1239
aI7705
aI4303
aI12535
aI21359
asS'd'
I5297
sa(dp10594
S'a'
I25994
sS's'
I16
sS'm'
(lp10595
I1
aI1
aI7
aI15
aI29
aI43
aI81
aI63
aI483
aI851
aI389
aI1719
aI6111
aI15293
aI2513
aI44397
asS'd'
I5298
sa(dp10596
S'a'
I25996
sS's'
I16
sS'm'
(lp10597
I1
aI1
aI5
aI15
aI25
aI33
aI97
aI131
aI183
aI269
aI1903
aI2733
aI7197
aI4507
aI24471
aI36871
asS'd'
I5299
sa(dp10598
S'a'
I25999
sS's'
I16
sS'm'
(lp10599
I1
aI3
aI3
aI13
aI17
aI33
aI73
aI83
aI247
aI207
aI79
aI1139
aI581
aI12147
aI3539
aI45423
asS'd'
I5300
sa(dp10600
S'a'
I26001
sS's'
I16
sS'm'
(lp10601
I1
aI1
aI1
aI15
aI29
aI49
aI79
aI151
aI267
aI393
aI1995
aI105
aI2873
aI3981
aI19147
aI53987
asS'd'
I5301
sa(dp10602
S'a'
I26030
sS's'
I16
sS'm'
(lp10603
I1
aI1
aI5
aI1
aI31
aI25
aI39
aI203
aI459
aI181
aI661
aI717
aI6235
aI13413
aI1197
aI40665
asS'd'
I5302
sa(dp10604
S'a'
I26069
sS's'
I16
sS'm'
(lp10605
I1
aI1
aI5
aI9
aI19
aI33
aI65
aI239
aI463
aI133
aI461
aI3601
aI7755
aI1771
aI20683
aI7417
asS'd'
I5303
sa(dp10606
S'a'
I26100
sS's'
I16
sS'm'
(lp10607
I1
aI1
aI1
aI1
aI25
aI19
aI25
aI155
aI431
aI33
aI341
aI959
aI5679
aI1205
aI2599
aI37499
asS'd'
I5304
sa(dp10608
S'a'
I26109
sS's'
I16
sS'm'
(lp10609
I1
aI1
aI3
aI5
aI25
aI5
aI83
aI87
aI91
aI991
aI1833
aI4063
aI147
aI14497
aI25725
aI4617
asS'd'
I5305
sa(dp10610
S'a'
I26131
sS's'
I16
sS'm'
(lp10611
I1
aI3
aI5
aI7
aI31
aI7
aI73
aI51
aI339
aI313
aI1593
aI2089
aI7387
aI15759
aI3249
aI7953
asS'd'
I5306
sa(dp10612
S'a'
I26144
sS's'
I16
sS'm'
(lp10613
I1
aI3
aI7
aI1
aI27
aI49
aI35
aI11
aI21
aI45
aI1689
aI1665
aI4591
aI3713
aI12781
aI4805
asS'd'
I5307
sa(dp10614
S'a'
I26149
sS's'
I16
sS'm'
(lp10615
I1
aI1
aI3
aI9
aI29
aI51
aI73
aI51
aI303
aI179
aI67
aI3917
aI7615
aI6131
aI26225
aI55991
asS'd'
I5308
sa(dp10616
S'a'
I26162
sS's'
I16
sS'm'
(lp10617
I1
aI3
aI7
aI11
aI9
aI63
aI29
aI47
aI153
aI883
aI1859
aI1913
aI3563
aI11451
aI6333
aI51367
asS'd'
I5309
sa(dp10618
S'a'
I26167
sS's'
I16
sS'm'
(lp10619
I1
aI3
aI1
aI3
aI5
aI25
aI69
aI87
aI389
aI613
aI989
aI3557
aI1339
aI12503
aI14505
aI63209
asS'd'
I5310
sa(dp10620
S'a'
I26173
sS's'
I16
sS'm'
(lp10621
I1
aI1
aI3
aI1
aI5
aI13
aI37
aI163
aI499
aI163
aI2025
aI1467
aI5059
aI8479
aI2889
aI62957
asS'd'
I5311
sa(dp10622
S'a'
I26179
sS's'
I16
sS'm'
(lp10623
I1
aI1
aI7
aI9
aI23
aI31
aI109
aI49
aI73
aI197
aI337
aI2763
aI4789
aI8983
aI9691
aI32155
asS'd'
I5312
sa(dp10624
S'a'
I26193
sS's'
I16
sS'm'
(lp10625
I1
aI3
aI1
aI3
aI31
aI25
aI121
aI91
aI371
aI339
aI833
aI2217
aI4997
aI9425
aI10685
aI60037
asS'd'
I5313
sa(dp10626
S'a'
I26230
sS's'
I16
sS'm'
(lp10627
I1
aI1
aI7
aI11
aI31
aI3
aI105
aI125
aI255
aI175
aI803
aI1787
aI6231
aI4441
aI5031
aI29737
asS'd'
I5314
sa(dp10628
S'a'
I26234
sS's'
I16
sS'm'
(lp10629
I1
aI1
aI1
aI11
aI21
aI63
aI75
aI209
aI393
aI437
aI495
aI2397
aI4759
aI15703
aI29869
aI62685
asS'd'
I5315
sa(dp10630
S'a'
I26246
sS's'
I16
sS'm'
(lp10631
I1
aI1
aI7
aI7
aI25
aI33
aI117
aI7
aI293
aI623
aI2001
aI2409
aI2487
aI14803
aI3329
aI38277
asS'd'
I5316
sa(dp10632
S'a'
I26267
sS's'
I16
sS'm'
(lp10633
I1
aI1
aI7
aI9
aI31
aI51
aI75
aI151
aI487
aI85
aI639
aI3529
aI4491
aI1957
aI22099
aI20263
asS'd'
I5317
sa(dp10634
S'a'
I26283
sS's'
I16
sS'm'
(lp10635
I1
aI1
aI7
aI5
aI3
aI29
aI11
aI1
aI255
aI555
aI1269
aI779
aI1525
aI7689
aI25847
aI39347
asS'd'
I5318
sa(dp10636
S'a'
I26300
sS's'
I16
sS'm'
(lp10637
I1
aI1
aI7
aI7
aI19
aI21
aI9
aI123
aI3
aI943
aI1627
aI2979
aI919
aI4565
aI31435
aI59789
asS'd'
I5319
sa(dp10638
S'a'
I26341
sS's'
I16
sS'm'
(lp10639
I1
aI3
aI7
aI5
aI29
aI13
aI57
aI221
aI9
aI893
aI1685
aI1879
aI4575
aI7369
aI26191
aI6067
asS'd'
I5320
sa(dp10640
S'a'
I26356
sS's'
I16
sS'm'
(lp10641
I1
aI3
aI7
aI9
aI13
aI11
aI9
aI27
aI313
aI751
aI1377
aI1121
aI3799
aI1673
aI16305
aI30665
asS'd'
I5321
sa(dp10642
S'a'
I26377
sS's'
I16
sS'm'
(lp10643
I1
aI3
aI3
aI13
aI23
aI17
aI59
aI47
aI499
aI525
aI681
aI3195
aI1611
aI7003
aI7325
aI53019
asS'd'
I5322
sa(dp10644
S'a'
I26397
sS's'
I16
sS'm'
(lp10645
I1
aI3
aI1
aI7
aI23
aI51
aI59
aI127
aI67
aI263
aI1305
aI2479
aI637
aI9441
aI6329
aI12857
asS'd'
I5323
sa(dp10646
S'a'
I26404
sS's'
I16
sS'm'
(lp10647
I1
aI1
aI7
aI7
aI9
aI3
aI51
aI193
aI205
aI503
aI19
aI2513
aI7489
aI9241
aI15371
aI20875
asS'd'
I5324
sa(dp10648
S'a'
I26411
sS's'
I16
sS'm'
(lp10649
I1
aI3
aI3
aI1
aI1
aI57
aI17
aI181
aI23
aI549
aI769
aI2325
aI3669
aI7017
aI25601
aI64479
asS'd'
I5325
sa(dp10650
S'a'
I26422
sS's'
I16
sS'm'
(lp10651
I1
aI3
aI1
aI15
aI5
aI55
aI53
aI13
aI327
aI447
aI1031
aI1599
aI3639
aI15305
aI1387
aI16035
asS'd'
I5326
sa(dp10652
S'a'
I26451
sS's'
I16
sS'm'
(lp10653
I1
aI3
aI7
aI15
aI9
aI41
aI53
aI113
aI97
aI99
aI1377
aI4047
aI3713
aI8891
aI5601
aI5853
asS'd'
I5327
sa(dp10654
S'a'
I26454
sS's'
I16
sS'm'
(lp10655
I1
aI1
aI7
aI9
aI9
aI7
aI29
aI249
aI411
aI315
aI181
aI2153
aI6135
aI6947
aI27595
aI15553
asS'd'
I5328
sa(dp10656
S'a'
I26463
sS's'
I16
sS'm'
(lp10657
I1
aI1
aI7
aI11
aI3
aI57
aI35
aI55
aI165
aI313
aI577
aI3049
aI4259
aI4231
aI7225
aI58973
asS'd'
I5329
sa(dp10658
S'a'
I26488
sS's'
I16
sS'm'
(lp10659
I1
aI1
aI1
aI1
aI15
aI43
aI53
aI143
aI157
aI843
aI465
aI3897
aI4797
aI12305
aI28807
aI46381
asS'd'
I5330
sa(dp10660
S'a'
I26498
sS's'
I16
sS'm'
(lp10661
I1
aI3
aI7
aI9
aI17
aI3
aI99
aI61
aI475
aI507
aI831
aI2207
aI367
aI27
aI23205
aI2303
asS'd'
I5331
sa(dp10662
S'a'
I26509
sS's'
I16
sS'm'
(lp10663
I1
aI1
aI3
aI11
aI27
aI29
aI99
aI237
aI43
aI955
aI361
aI3231
aI1863
aI7973
aI8969
aI58663
asS'd'
I5332
sa(dp10664
S'a'
I26512
sS's'
I16
sS'm'
(lp10665
I1
aI3
aI1
aI7
aI15
aI15
aI11
aI251
aI135
aI261
aI675
aI3723
aI7675
aI7993
aI1725
aI41149
asS'd'
I5333
sa(dp10666
S'a'
I26517
sS's'
I16
sS'm'
(lp10667
I1
aI3
aI3
aI9
aI29
aI11
aI105
aI37
aI151
aI215
aI1911
aI4051
aI5427
aI11019
aI9073
aI33129
asS'd'
I5334
sa(dp10668
S'a'
I26534
sS's'
I16
sS'm'
(lp10669
I1
aI3
aI3
aI1
aI19
aI7
aI103
aI81
aI371
aI253
aI1043
aI1231
aI6497
aI10377
aI2349
aI29047
asS'd'
I5335
sa(dp10670
S'a'
I26545
sS's'
I16
sS'm'
(lp10671
I1
aI3
aI7
aI9
aI15
aI11
aI85
aI61
aI507
aI629
aI811
aI3883
aI1435
aI13035
aI31913
aI2153
asS'd'
I5336
sa(dp10672
S'a'
I26546
sS's'
I16
sS'm'
(lp10673
I1
aI1
aI5
aI11
aI13
aI7
aI63
aI147
aI117
aI223
aI1217
aI3317
aI3275
aI6851
aI2917
aI55901
asS'd'
I5337
sa(dp10674
S'a'
I26636
sS's'
I16
sS'm'
(lp10675
I1
aI3
aI3
aI9
aI7
aI21
aI1
aI63
aI117
aI297
aI405
aI797
aI337
aI10173
aI8327
aI29157
asS'd'
I5338
sa(dp10676
S'a'
I26749
sS's'
I16
sS'm'
(lp10677
I1
aI1
aI7
aI11
aI31
aI63
aI97
aI191
aI259
aI421
aI1829
aI2117
aI4203
aI11919
aI18001
aI55791
asS'd'
I5339
sa(dp10678
S'a'
I26753
sS's'
I16
sS'm'
(lp10679
I1
aI3
aI7
aI9
aI21
aI13
aI125
aI247
aI133
aI611
aI463
aI227
aI7815
aI8877
aI17961
aI3641
asS'd'
I5340
sa(dp10680
S'a'
I26759
sS's'
I16
sS'm'
(lp10681
I1
aI1
aI7
aI9
aI27
aI21
aI97
aI165
aI371
aI715
aI491
aI3929
aI3067
aI12501
aI5511
aI18217
asS'd'
I5341
sa(dp10682
S'a'
I26774
sS's'
I16
sS'm'
(lp10683
I1
aI3
aI3
aI15
aI27
aI17
aI81
aI161
aI263
aI273
aI135
aI1159
aI7535
aI4581
aI16065
aI11493
asS'd'
I5342
sa(dp10684
S'a'
I26789
sS's'
I16
sS'm'
(lp10685
I1
aI3
aI3
aI7
aI11
aI59
aI111
aI47
aI381
aI413
aI243
aI2173
aI4957
aI2451
aI15669
aI22071
asS'd'
I5343
sa(dp10686
S'a'
I26808
sS's'
I16
sS'm'
(lp10687
I1
aI3
aI7
aI5
aI3
aI31
aI65
aI131
aI111
aI141
aI1891
aI2983
aI3331
aI769
aI24075
aI40865
asS'd'
I5344
sa(dp10688
S'a'
I26825
sS's'
I16
sS'm'
(lp10689
I1
aI3
aI7
aI11
aI11
aI63
aI7
aI213
aI333
aI111
aI1235
aI961
aI3749
aI9123
aI5067
aI9657
asS'd'
I5345
sa(dp10690
S'a'
I26831
sS's'
I16
sS'm'
(lp10691
I1
aI3
aI3
aI1
aI9
aI33
aI1
aI225
aI37
aI951
aI1995
aI3215
aI3117
aI3723
aI15013
aI64525
asS'd'
I5346
sa(dp10692
S'a'
I26859
sS's'
I16
sS'm'
(lp10693
I1
aI3
aI3
aI13
aI29
aI19
aI103
aI65
aI67
aI423
aI1679
aI3791
aI5551
aI11711
aI195
aI52291
asS'd'
I5347
sa(dp10694
S'a'
I26888
sS's'
I16
sS'm'
(lp10695
I1
aI3
aI1
aI15
aI31
aI7
aI65
aI249
aI489
aI287
aI1385
aI1075
aI1357
aI13593
aI20853
aI46221
asS'd'
I5348
sa(dp10696
S'a'
I26918
sS's'
I16
sS'm'
(lp10697
I1
aI1
aI1
aI13
aI23
aI45
aI29
aI175
aI147
aI101
aI1007
aI1867
aI5387
aI12683
aI29609
aI55861
asS'd'
I5349
sa(dp10698
S'a'
I26929
sS's'
I16
sS'm'
(lp10699
I1
aI3
aI5
aI13
aI21
aI31
aI85
aI187
aI183
aI179
aI1337
aI481
aI71
aI6117
aI2177
aI27915
asS'd'
I5350
sa(dp10700
S'a'
I26950
sS's'
I16
sS'm'
(lp10701
I1
aI3
aI5
aI1
aI15
aI5
aI11
aI141
aI205
aI177
aI891
aI3591
aI4371
aI889
aI12957
aI61083
asS'd'
I5351
sa(dp10702
S'a'
I26954
sS's'
I16
sS'm'
(lp10703
I1
aI3
aI7
aI7
aI11
aI39
aI81
aI241
aI13
aI757
aI521
aI3029
aI2345
aI12385
aI20683
aI64053
asS'd'
I5352
sa(dp10704
S'a'
I26973
sS's'
I16
sS'm'
(lp10705
I1
aI1
aI5
aI13
aI7
aI3
aI77
aI211
aI215
aI97
aI1409
aI1021
aI1267
aI4785
aI27231
aI2877
asS'd'
I5353
sa(dp10706
S'a'
I26997
sS's'
I16
sS'm'
(lp10707
I1
aI3
aI5
aI3
aI11
aI57
aI93
aI39
aI415
aI179
aI1033
aI3267
aI5383
aI10451
aI27117
aI10711
asS'd'
I5354
sa(dp10708
S'a'
I26998
sS's'
I16
sS'm'
(lp10709
I1
aI1
aI1
aI1
aI3
aI45
aI93
aI179
aI453
aI995
aI1423
aI3849
aI4381
aI15789
aI20789
aI18775
asS'd'
I5355
sa(dp10710
S'a'
I27008
sS's'
I16
sS'm'
(lp10711
I1
aI3
aI1
aI3
aI13
aI25
aI33
aI165
aI351
aI887
aI1109
aI195
aI8131
aI3061
aI16743
aI22997
asS'd'
I5356
sa(dp10712
S'a'
I27038
sS's'
I16
sS'm'
(lp10713
I1
aI3
aI1
aI5
aI23
aI35
aI93
aI155
aI333
aI603
aI1571
aI229
aI2979
aI6295
aI20793
aI40901
asS'd'
I5357
sa(dp10714
S'a'
I27109
sS's'
I16
sS'm'
(lp10715
I1
aI3
aI3
aI11
aI29
aI57
aI101
aI61
aI487
aI719
aI1009
aI1933
aI7815
aI15329
aI12489
aI26105
asS'd'
I5358
sa(dp10716
S'a'
I27127
sS's'
I16
sS'm'
(lp10717
I1
aI3
aI3
aI9
aI23
aI59
aI73
aI13
aI141
aI815
aI1819
aI3557
aI2555
aI5901
aI23039
aI62321
asS'd'
I5359
sa(dp10718
S'a'
I27150
sS's'
I16
sS'm'
(lp10719
I1
aI1
aI3
aI5
aI19
aI49
aI27
aI139
aI35
aI995
aI565
aI323
aI6439
aI15679
aI27017
aI30889
asS'd'
I5360
sa(dp10720
S'a'
I27168
sS's'
I16
sS'm'
(lp10721
I1
aI3
aI7
aI3
aI1
aI3
aI27
aI153
aI235
aI59
aI989
aI2763
aI4197
aI3931
aI31227
aI27129
asS'd'
I5361
sa(dp10722
S'a'
I27178
sS's'
I16
sS'm'
(lp10723
I1
aI3
aI1
aI15
aI23
aI45
aI71
aI205
aI465
aI451
aI347
aI1909
aI3287
aI8363
aI9081
aI35641
asS'd'
I5362
sa(dp10724
S'a'
I27212
sS's'
I16
sS'm'
(lp10725
I1
aI1
aI5
aI1
aI25
aI29
aI17
aI201
aI463
aI903
aI1729
aI3435
aI2483
aI10835
aI14315
aI52505
asS'd'
I5363
sa(dp10726
S'a'
I27224
sS's'
I16
sS'm'
(lp10727
I1
aI1
aI1
aI15
aI13
aI23
aI3
aI175
aI273
aI305
aI1945
aI3319
aI7777
aI9411
aI4209
aI4047
asS'd'
I5364
sa(dp10728
S'a'
I27229
sS's'
I16
sS'm'
(lp10729
I1
aI1
aI5
aI15
aI25
aI5
aI71
aI35
aI307
aI89
aI301
aI3465
aI1497
aI13467
aI21911
aI13611
asS'd'
I5365
sa(dp10730
S'a'
I27246
sS's'
I16
sS'm'
(lp10731
I1
aI3
aI1
aI7
aI11
aI7
aI33
aI241
aI349
aI751
aI633
aI3281
aI6733
aI13833
aI23605
aI34307
asS'd'
I5366
sa(dp10732
S'a'
I27251
sS's'
I16
sS'm'
(lp10733
I1
aI1
aI1
aI15
aI17
aI27
aI45
aI13
aI259
aI843
aI1207
aI1735
aI4063
aI6259
aI1751
aI45107
asS'd'
I5367
sa(dp10734
S'a'
I27257
sS's'
I16
sS'm'
(lp10735
I1
aI1
aI5
aI15
aI29
aI51
aI73
aI95
aI5
aI31
aI933
aI423
aI5505
aI2193
aI14919
aI2715
asS'd'
I5368
sa(dp10736
S'a'
I27258
sS's'
I16
sS'm'
(lp10737
I1
aI3
aI1
aI3
aI23
aI5
aI29
aI7
aI271
aI485
aI827
aI1159
aI77
aI16337
aI27933
aI18741
asS'd'
I5369
sa(dp10738
S'a'
I27260
sS's'
I16
sS'm'
(lp10739
I1
aI3
aI7
aI9
aI23
aI33
aI47
aI191
aI59
aI301
aI1277
aI3745
aI7837
aI799
aI2861
aI25853
asS'd'
I5370
sa(dp10740
S'a'
I27287
sS's'
I16
sS'm'
(lp10741
I1
aI3
aI7
aI13
aI13
aI39
aI33
aI91
aI279
aI855
aI1917
aI3601
aI3971
aI6193
aI16951
aI6115
asS'd'
I5371
sa(dp10742
S'a'
I27300
sS's'
I16
sS'm'
(lp10743
I1
aI1
aI3
aI13
aI15
aI59
aI89
aI239
aI313
aI545
aI431
aI3823
aI5741
aI14981
aI2647
aI42813
asS'd'
I5372
sa(dp10744
S'a'
I27315
sS's'
I16
sS'm'
(lp10745
I1
aI1
aI1
aI3
aI17
aI21
aI45
aI37
aI343
aI737
aI1795
aI2659
aI2897
aI7683
aI15191
aI1393
asS'd'
I5373
sa(dp10746
S'a'
I27329
sS's'
I16
sS'm'
(lp10747
I1
aI1
aI3
aI3
aI19
aI55
aI27
aI201
aI459
aI953
aI1531
aI671
aI5667
aI11695
aI149
aI14605
asS'd'
I5374
sa(dp10748
S'a'
I27332
sS's'
I16
sS'm'
(lp10749
I1
aI3
aI7
aI13
aI9
aI63
aI67
aI229
aI69
aI819
aI859
aI2035
aI5725
aI13403
aI24197
aI2599
asS'd'
I5375
sa(dp10750
S'a'
I27349
sS's'
I16
sS'm'
(lp10751
I1
aI1
aI7
aI7
aI1
aI59
aI45
aI105
aI327
aI779
aI59
aI791
aI7593
aI8189
aI28161
aI13339
asS'd'
I5376
sa(dp10752
S'a'
I27350
sS's'
I16
sS'm'
(lp10753
I1
aI3
aI3
aI15
aI25
aI55
aI125
aI189
aI327
aI733
aI115
aI3541
aI5227
aI12143
aI32719
aI15785
asS'd'
I5377
sa(dp10754
S'a'
I27354
sS's'
I16
sS'm'
(lp10755
I1
aI3
aI3
aI7
aI19
aI51
aI35
aI63
aI507
aI89
aI1441
aI2369
aI4927
aI7953
aI10193
aI8331
asS'd'
I5378
sa(dp10756
S'a'
I27359
sS's'
I16
sS'm'
(lp10757
I1
aI1
aI5
aI5
aI21
aI1
aI41
aI49
aI217
aI1001
aI1649
aI2789
aI5267
aI1525
aI3811
aI40785
asS'd'
I5379
sa(dp10758
S'a'
I27377
sS's'
I16
sS'm'
(lp10759
I1
aI1
aI7
aI15
aI31
aI21
aI33
aI183
aI425
aI393
aI367
aI3253
aI3047
aI465
aI28229
aI44743
asS'd'
I5380
sa(dp10760
S'a'
I27378
sS's'
I16
sS'm'
(lp10761
I1
aI3
aI7
aI5
aI1
aI23
aI11
aI71
aI269
aI707
aI5
aI2931
aI1959
aI15089
aI9299
aI43927
asS'd'
I5381
sa(dp10762
S'a'
I27383
sS's'
I16
sS'm'
(lp10763
I1
aI3
aI5
aI15
aI21
aI51
aI31
aI15
aI49
aI481
aI297
aI3871
aI751
aI10661
aI26401
aI62923
asS'd'
I5382
sa(dp10764
S'a'
I27384
sS's'
I16
sS'm'
(lp10765
I1
aI3
aI1
aI7
aI17
aI27
aI35
aI255
aI205
aI865
aI1659
aI1921
aI5457
aI11633
aI2825
aI13549
asS'd'
I5383
sa(dp10766
S'a'
I27387
sS's'
I16
sS'm'
(lp10767
I1
aI1
aI5
aI15
aI17
aI35
aI83
aI237
aI437
aI7
aI2001
aI2225
aI2601
aI10841
aI7953
aI20651
asS'd'
I5384
sa(dp10768
S'a'
I27392
sS's'
I16
sS'm'
(lp10769
I1
aI1
aI1
aI3
aI3
aI37
aI43
aI135
aI451
aI203
aI1319
aI261
aI3889
aI14489
aI9635
aI52545
asS'd'
I5385
sa(dp10770
S'a'
I27402
sS's'
I16
sS'm'
(lp10771
I1
aI3
aI3
aI13
aI15
aI41
aI95
aI207
aI43
aI997
aI207
aI3459
aI5995
aI5187
aI15851
aI28551
asS'd'
I5386
sa(dp10772
S'a'
I27438
sS's'
I16
sS'm'
(lp10773
I1
aI1
aI1
aI5
aI23
aI57
aI49
aI101
aI303
aI921
aI745
aI1407
aI7071
aI2765
aI18703
aI32671
asS'd'
I5387
sa(dp10774
S'a'
I27481
sS's'
I16
sS'm'
(lp10775
I1
aI1
aI7
aI13
aI9
aI59
aI65
aI157
aI209
aI295
aI107
aI3175
aI3401
aI1197
aI1875
aI9033
asS'd'
I5388
sa(dp10776
S'a'
I27482
sS's'
I16
sS'm'
(lp10777
I1
aI1
aI3
aI3
aI17
aI9
aI101
aI75
aI177
aI905
aI1013
aI397
aI3421
aI6475
aI15897
aI11269
asS'd'
I5389
sa(dp10778
S'a'
I27494
sS's'
I16
sS'm'
(lp10779
I1
aI3
aI1
aI5
aI29
aI53
aI105
aI83
aI383
aI137
aI1169
aI1245
aI6973
aI8701
aI317
aI10073
asS'd'
I5390
sa(dp10780
S'a'
I27531
sS's'
I16
sS'm'
(lp10781
I1
aI1
aI1
aI3
aI15
aI55
aI69
aI219
aI507
aI707
aI945
aI397
aI779
aI4157
aI10333
aI7869
asS'd'
I5391
sa(dp10782
S'a'
I27542
sS's'
I16
sS'm'
(lp10783
I1
aI3
aI1
aI3
aI9
aI21
aI125
aI153
aI107
aI969
aI1979
aI651
aI1199
aI11419
aI17043
aI32269
asS'd'
I5392
sa(dp10784
S'a'
I27546
sS's'
I16
sS'm'
(lp10785
I1
aI1
aI1
aI7
aI1
aI29
aI71
aI127
aI209
aI853
aI1515
aI1169
aI5055
aI9981
aI30291
aI29569
asS'd'
I5393
sa(dp10786
S'a'
I27564
sS's'
I16
sS'm'
(lp10787
I1
aI3
aI1
aI11
aI1
aI1
aI109
aI251
aI329
aI633
aI2021
aI1237
aI2147
aI11471
aI26025
aI19649
asS'd'
I5394
sa(dp10788
S'a'
I27567
sS's'
I16
sS'm'
(lp10789
I1
aI1
aI5
aI1
aI5
aI7
aI77
aI175
aI251
aI143
aI711
aI1241
aI2133
aI9993
aI9203
aI24949
asS'd'
I5395
sa(dp10790
S'a'
I27582
sS's'
I16
sS'm'
(lp10791
I1
aI3
aI5
aI11
aI19
aI11
aI101
aI83
aI91
aI595
aI753
aI2389
aI1887
aI11569
aI29759
aI55785
asS'd'
I5396
sa(dp10792
S'a'
I27608
sS's'
I16
sS'm'
(lp10793
I1
aI1
aI1
aI3
aI29
aI47
aI49
aI249
aI495
aI451
aI1887
aI2547
aI543
aI2755
aI17207
aI24379
asS'd'
I5397
sa(dp10794
S'a'
I27611
sS's'
I16
sS'm'
(lp10795
I1
aI3
aI7
aI7
aI19
aI15
aI95
aI143
aI109
aI221
aI2041
aI3593
aI4571
aI14547
aI14217
aI16711
asS'd'
I5398
sa(dp10796
S'a'
I27624
sS's'
I16
sS'm'
(lp10797
I1
aI3
aI5
aI13
aI27
aI55
aI31
aI209
aI39
aI989
aI1435
aI1665
aI7265
aI14127
aI13517
aI37647
asS'd'
I5399
sa(dp10798
S'a'
I27629
sS's'
I16
sS'm'
(lp10799
I1
aI1
aI3
aI7
aI1
aI49
aI63
aI71
aI249
aI371
aI435
aI3591
aI2677
aI143
aI28897
aI38981
asS'd'
I5400
sa(dp10800
S'a'
I27655
sS's'
I16
sS'm'
(lp10801
I1
aI1
aI7
aI7
aI21
aI9
aI53
aI221
aI23
aI515
aI1971
aI3759
aI3511
aI10207
aI12929
aI42021
asS'd'
I5401
sa(dp10802
S'a'
I27667
sS's'
I16
sS'm'
(lp10803
I1
aI3
aI1
aI13
aI25
aI21
aI3
aI85
aI421
aI19
aI663
aI3219
aI3541
aI13021
aI5765
aI39623
asS'd'
I5402
sa(dp10804
S'a'
I27676
sS's'
I16
sS'm'
(lp10805
I1
aI3
aI1
aI7
aI11
aI5
aI125
aI169
aI293
aI715
aI1111
aI2965
aI4815
aI6047
aI27207
aI23093
asS'd'
I5403
sa(dp10806
S'a'
I27710
sS's'
I16
sS'm'
(lp10807
I1
aI1
aI5
aI13
aI11
aI15
aI37
aI201
aI457
aI551
aI821
aI25
aI435
aI14307
aI25855
aI1811
asS'd'
I5404
sa(dp10808
S'a'
I27724
sS's'
I16
sS'm'
(lp10809
I1
aI3
aI3
aI9
aI27
aI15
aI1
aI253
aI217
aI549
aI1013
aI2277
aI4171
aI3813
aI19857
aI8641
asS'd'
I5405
sa(dp10810
S'a'
I27745
sS's'
I16
sS'm'
(lp10811
I1
aI3
aI5
aI5
aI29
aI37
aI71
aI49
aI163
aI949
aI425
aI2459
aI945
aI13125
aI13981
aI21669
asS'd'
I5406
sa(dp10812
S'a'
I27752
sS's'
I16
sS'm'
(lp10813
I1
aI3
aI3
aI9
aI17
aI23
aI53
aI235
aI83
aI887
aI439
aI1939
aI7601
aI15275
aI15739
aI17623
asS'd'
I5407
sa(dp10814
S'a'
I27770
sS's'
I16
sS'm'
(lp10815
I1
aI3
aI5
aI13
aI7
aI51
aI73
aI67
aI167
aI849
aI2021
aI2977
aI6591
aI3721
aI5827
aI40897
asS'd'
I5408
sa(dp10816
S'a'
I27779
sS's'
I16
sS'm'
(lp10817
I1
aI1
aI5
aI11
aI27
aI19
aI81
aI181
aI383
aI23
aI1061
aI3327
aI1671
aI7113
aI7435
aI17591
asS'd'
I5409
sa(dp10818
S'a'
I27781
sS's'
I16
sS'm'
(lp10819
I1
aI3
aI3
aI7
aI25
aI33
aI73
aI23
aI103
aI821
aI917
aI1425
aI4739
aI7227
aI12365
aI29509
asS'd'
I5410
sa(dp10820
S'a'
I27791
sS's'
I16
sS'm'
(lp10821
I1
aI1
aI1
aI7
aI3
aI37
aI81
aI231
aI135
aI663
aI1983
aI399
aI6881
aI14991
aI4957
aI20913
asS'd'
I5411
sa(dp10822
S'a'
I27809
sS's'
I16
sS'm'
(lp10823
I1
aI3
aI7
aI15
aI25
aI41
aI65
aI215
aI301
aI471
aI1669
aI65
aI227
aI9307
aI29867
aI9503
asS'd'
I5412
sa(dp10824
S'a'
I27810
sS's'
I16
sS'm'
(lp10825
I1
aI1
aI7
aI3
aI25
aI47
aI31
aI63
aI53
aI995
aI33
aI1297
aI3423
aI12301
aI16255
aI14467
asS'd'
I5413
sa(dp10826
S'a'
I27815
sS's'
I16
sS'm'
(lp10827
I1
aI3
aI1
aI1
aI31
aI25
aI79
aI131
aI353
aI169
aI1425
aI2257
aI2631
aI1559
aI793
aI48383
asS'd'
I5414
sa(dp10828
S'a'
I27827
sS's'
I16
sS'm'
(lp10829
I1
aI3
aI3
aI5
aI31
aI9
aI93
aI35
aI503
aI243
aI595
aI2939
aI771
aI7333
aI13429
aI59457
asS'd'
I5415
sa(dp10830
S'a'
I27834
sS's'
I16
sS'm'
(lp10831
I1
aI3
aI1
aI7
aI5
aI51
aI21
aI237
aI453
aI743
aI775
aI2207
aI453
aI12077
aI12283
aI9735
asS'd'
I5416
sa(dp10832
S'a'
I27865
sS's'
I16
sS'm'
(lp10833
I1
aI3
aI1
aI15
aI5
aI47
aI59
aI239
aI87
aI97
aI885
aI3191
aI2547
aI13227
aI7433
aI4989
asS'd'
I5417
sa(dp10834
S'a'
I27899
sS's'
I16
sS'm'
(lp10835
I1
aI3
aI5
aI15
aI21
aI33
aI41
aI155
aI509
aI317
aI517
aI1101
aI133
aI1897
aI8235
aI57673
asS'd'
I5418
sa(dp10836
S'a'
I27901
sS's'
I16
sS'm'
(lp10837
I1
aI1
aI5
aI15
aI7
aI9
aI59
aI155
aI415
aI831
aI1173
aI1263
aI5451
aI7181
aI7233
aI51483
asS'd'
I5419
sa(dp10838
S'a'
I27914
sS's'
I16
sS'm'
(lp10839
I1
aI1
aI7
aI3
aI31
aI43
aI71
aI39
aI155
aI529
aI895
aI827
aI3203
aI4625
aI32185
aI53507
asS'd'
I5420
sa(dp10840
S'a'
I27950
sS's'
I16
sS'm'
(lp10841
I1
aI3
aI1
aI11
aI15
aI17
aI85
aI141
aI277
aI439
aI1775
aI4015
aI4457
aI281
aI22455
aI47591
asS'd'
I5421
sa(dp10842
S'a'
I27994
sS's'
I16
sS'm'
(lp10843
I1
aI3
aI5
aI11
aI25
aI41
aI93
aI39
aI51
aI655
aI1347
aI3109
aI2479
aI9057
aI18939
aI26217
asS'd'
I5422
sa(dp10844
S'a'
I28005
sS's'
I16
sS'm'
(lp10845
I1
aI3
aI3
aI11
aI31
aI41
aI7
aI189
aI241
aI443
aI65
aI1723
aI4817
aI13405
aI9641
aI63965
asS'd'
I5423
sa(dp10846
S'a'
I28009
sS's'
I16
sS'm'
(lp10847
I1
aI1
aI5
aI3
aI19
aI29
aI111
aI11
aI399
aI277
aI425
aI1331
aI5365
aI14521
aI16449
aI29411
asS'd'
I5424
sa(dp10848
S'a'
I28033
sS's'
I16
sS'm'
(lp10849
I1
aI1
aI3
aI9
aI25
aI53
aI91
aI175
aI481
aI307
aI1025
aI71
aI7425
aI10667
aI4053
aI25605
asS'd'
I5425
sa(dp10850
S'a'
I28039
sS's'
I16
sS'm'
(lp10851
I1
aI3
aI7
aI7
aI3
aI13
aI75
aI175
aI467
aI363
aI1889
aI1759
aI1155
aI5511
aI13047
aI39637
asS'd'
I5426
sa(dp10852
S'a'
I28060
sS's'
I16
sS'm'
(lp10853
I1
aI3
aI7
aI9
aI5
aI21
aI65
aI43
aI223
aI97
aI835
aI2253
aI3313
aI9817
aI23015
aI55365
asS'd'
I5427
sa(dp10854
S'a'
I28067
sS's'
I16
sS'm'
(lp10855
I1
aI1
aI1
aI13
aI9
aI63
aI95
aI61
aI417
aI713
aI1469
aI1815
aI795
aI13609
aI1567
aI33535
asS'd'
I5428
sa(dp10856
S'a'
I28069
sS's'
I16
sS'm'
(lp10857
I1
aI3
aI7
aI1
aI25
aI45
aI41
aI27
aI53
aI407
aI1633
aI1317
aI6267
aI3293
aI8899
aI45235
asS'd'
I5429
sa(dp10858
S'a'
I28099
sS's'
I16
sS'm'
(lp10859
I1
aI3
aI5
aI11
aI23
aI47
aI91
aI211
aI41
aI775
aI1301
aI1407
aI7931
aI4491
aI7579
aI62321
asS'd'
I5430
sa(dp10860
S'a'
I28113
sS's'
I16
sS'm'
(lp10861
I1
aI1
aI1
aI7
aI23
aI15
aI57
aI31
aI437
aI293
aI1999
aI2589
aI5453
aI2519
aI15533
aI26949
asS'd'
I5431
sa(dp10862
S'a'
I28114
sS's'
I16
sS'm'
(lp10863
I1
aI3
aI1
aI9
aI1
aI27
aI41
aI165
aI129
aI297
aI1887
aI1171
aI201
aI5817
aI24503
aI38911
asS'd'
I5432
sa(dp10864
S'a'
I28139
sS's'
I16
sS'm'
(lp10865
I1
aI3
aI1
aI7
aI1
aI11
aI63
aI225
aI191
aI623
aI1281
aI3275
aI167
aI14697
aI4905
aI47289
asS'd'
I5433
sa(dp10866
S'a'
I28142
sS's'
I16
sS'm'
(lp10867
I1
aI3
aI7
aI7
aI15
aI47
aI87
aI177
aI303
aI391
aI355
aI3997
aI7557
aI6201
aI20531
aI22483
asS'd'
I5434
sa(dp10868
S'a'
I28153
sS's'
I16
sS'm'
(lp10869
I1
aI3
aI3
aI15
aI17
aI31
aI111
aI87
aI61
aI477
aI1581
aI3787
aI5919
aI10511
aI2607
aI62951
asS'd'
I5435
sa(dp10870
S'a'
I28166
sS's'
I16
sS'm'
(lp10871
I1
aI3
aI3
aI9
aI29
aI19
aI63
aI27
aI205
aI915
aI1131
aI3821
aI673
aI2875
aI12703
aI14367
asS'd'
I5436
sa(dp10872
S'a'
I28172
sS's'
I16
sS'm'
(lp10873
I1
aI3
aI7
aI1
aI21
aI19
aI25
aI97
aI281
aI635
aI629
aI181
aI5207
aI11133
aI3687
aI3489
asS'd'
I5437
sa(dp10874
S'a'
I28183
sS's'
I16
sS'm'
(lp10875
I1
aI3
aI3
aI9
aI5
aI63
aI3
aI21
aI385
aI713
aI1805
aI3583
aI2807
aI15455
aI13057
aI39771
asS'd'
I5438
sa(dp10876
S'a'
I28194
sS's'
I16
sS'm'
(lp10877
I1
aI3
aI5
aI9
aI3
aI59
aI1
aI253
aI123
aI405
aI575
aI3911
aI4609
aI11869
aI23593
aI947
asS'd'
I5439
sa(dp10878
S'a'
I28232
sS's'
I16
sS'm'
(lp10879
I1
aI1
aI7
aI5
aI21
aI27
aI101
aI221
aI413
aI153
aI1647
aI3637
aI803
aI5697
aI20761
aI61137
asS'd'
I5440
sa(dp10880
S'a'
I28245
sS's'
I16
sS'm'
(lp10881
I1
aI3
aI7
aI13
aI31
aI35
aI111
aI253
aI187
aI499
aI465
aI157
aI5551
aI10417
aI323
aI34913
asS'd'
I5441
sa(dp10882
S'a'
I28246
sS's'
I16
sS'm'
(lp10883
I1
aI1
aI1
aI7
aI11
aI41
aI29
aI65
aI393
aI69
aI1373
aI2291
aI7807
aI13159
aI13735
aI31001
asS'd'
I5442
sa(dp10884
S'a'
I28252
sS's'
I16
sS'm'
(lp10885
I1
aI3
aI7
aI13
aI31
aI49
aI1
aI35
aI377
aI11
aI427
aI2803
aI1725
aI9165
aI22633
aI63985
asS'd'
I5443
sa(dp10886
S'a'
I28265
sS's'
I16
sS'm'
(lp10887
I1
aI3
aI7
aI13
aI3
aI41
aI27
aI43
aI269
aI599
aI1035
aI3681
aI309
aI6011
aI1065
aI27901
asS'd'
I5444
sa(dp10888
S'a'
I28301
sS's'
I16
sS'm'
(lp10889
I1
aI3
aI5
aI13
aI1
aI19
aI105
aI143
aI425
aI883
aI1669
aI155
aI189
aI8573
aI10759
aI25507
asS'd'
I5445
sa(dp10890
S'a'
I28323
sS's'
I16
sS'm'
(lp10891
I1
aI3
aI5
aI1
aI15
aI37
aI115
aI9
aI149
aI79
aI1733
aI1045
aI1849
aI3289
aI13957
aI63171
asS'd'
I5446
sa(dp10892
S'a'
I28344
sS's'
I16
sS'm'
(lp10893
I1
aI1
aI3
aI9
aI17
aI27
aI49
aI201
aI155
aI901
aI47
aI1585
aI4419
aI8117
aI25425
aI14699
asS'd'
I5447
sa(dp10894
S'a'
I28362
sS's'
I16
sS'm'
(lp10895
I1
aI1
aI7
aI13
aI19
aI55
aI19
aI21
aI427
aI77
aI1295
aI1471
aI6271
aI7985
aI19337
aI12701
asS'd'
I5448
sa(dp10896
S'a'
I28400
sS's'
I16
sS'm'
(lp10897
I1
aI1
aI1
aI1
aI11
aI49
aI101
aI53
aI175
aI157
aI839
aI2713
aI6149
aI6391
aI8089
aI27739
asS'd'
I5449
sa(dp10898
S'a'
I28417
sS's'
I16
sS'm'
(lp10899
I1
aI3
aI1
aI1
aI5
aI21
aI121
aI199
aI107
aI221
aI993
aI1737
aI409
aI2545
aI9287
aI54605
asS'd'
I5450
sa(dp10900
S'a'
I28454
sS's'
I16
sS'm'
(lp10901
I1
aI1
aI1
aI3
aI25
aI25
aI51
aI121
aI371
aI861
aI967
aI3257
aI6221
aI11701
aI27897
aI42509
asS'd'
I5451
sa(dp10902
S'a'
I28466
sS's'
I16
sS'm'
(lp10903
I1
aI1
aI1
aI3
aI17
aI25
aI101
aI191
aI313
aI817
aI815
aI1855
aI7999
aI12649
aI23385
aI26081
asS'd'
I5452
sa(dp10904
S'a'
I28468
sS's'
I16
sS'm'
(lp10905
I1
aI1
aI5
aI1
aI25
aI55
aI51
aI237
aI63
aI943
aI455
aI619
aI2381
aI9773
aI14575
aI34205
asS'd'
I5453
sa(dp10906
S'a'
I28477
sS's'
I16
sS'm'
(lp10907
I1
aI3
aI3
aI3
aI13
aI49
aI101
aI37
aI457
aI727
aI1009
aI2389
aI4841
aI16303
aI9599
aI17773
asS'd'
I5454
sa(dp10908
S'a'
I28498
sS's'
I16
sS'm'
(lp10909
I1
aI1
aI7
aI9
aI19
aI59
aI111
aI205
aI19
aI229
aI1755
aI1169
aI7767
aI13335
aI19669
aI33269
asS'd'
I5455
sa(dp10910
S'a'
I28510
sS's'
I16
sS'm'
(lp10911
I1
aI3
aI1
aI15
aI29
aI1
aI51
aI167
aI7
aI415
aI1295
aI3165
aI1241
aI12859
aI5531
aI20083
asS'd'
I5456
sa(dp10912
S'a'
I28513
sS's'
I16
sS'm'
(lp10913
I1
aI1
aI3
aI7
aI7
aI51
aI31
aI221
aI57
aI643
aI1461
aI3951
aI6237
aI5757
aI1907
aI40471
asS'd'
I5457
sa(dp10914
S'a'
I28571
sS's'
I16
sS'm'
(lp10915
I1
aI3
aI3
aI5
aI23
aI39
aI49
aI177
aI183
aI117
aI1379
aI3803
aI771
aI12723
aI22291
aI32667
asS'd'
I5458
sa(dp10916
S'a'
I28573
sS's'
I16
sS'm'
(lp10917
I1
aI1
aI3
aI13
aI27
aI17
aI39
aI27
aI313
aI141
aI1421
aI2967
aI2213
aI1915
aI23219
aI15113
asS'd'
I5459
sa(dp10918
S'a'
I28578
sS's'
I16
sS'm'
(lp10919
I1
aI1
aI1
aI11
aI5
aI55
aI51
aI55
aI389
aI895
aI57
aI1447
aI1497
aI2799
aI19585
aI11587
asS'd'
I5460
sa(dp10920
S'a'
I28587
sS's'
I16
sS'm'
(lp10921
I1
aI1
aI5
aI13
aI11
aI55
aI91
aI77
aI69
aI131
aI93
aI1383
aI3321
aI10487
aI15087
aI8539
asS'd'
I5461
sa(dp10922
S'a'
I28601
sS's'
I16
sS'm'
(lp10923
I1
aI1
aI3
aI9
aI23
aI49
aI107
aI131
aI363
aI733
aI1189
aI3575
aI7815
aI10071
aI20291
aI7533
asS'd'
I5462
sa(dp10924
S'a'
I28646
sS's'
I16
sS'm'
(lp10925
I1
aI1
aI3
aI15
aI31
aI31
aI73
aI15
aI199
aI17
aI761
aI3271
aI1419
aI12985
aI32717
aI37317
asS'd'
I5463
sa(dp10926
S'a'
I28663
sS's'
I16
sS'm'
(lp10927
I1
aI3
aI1
aI13
aI23
aI9
aI3
aI59
aI109
aI729
aI1321
aI4023
aI7041
aI14445
aI22205
aI8993
asS'd'
I5464
sa(dp10928
S'a'
I28681
sS's'
I16
sS'm'
(lp10929
I1
aI1
aI3
aI15
aI19
aI43
aI99
aI59
aI491
aI479
aI715
aI2235
aI7493
aI889
aI31465
aI1375
asS'd'
I5465
sa(dp10930
S'a'
I28682
sS's'
I16
sS'm'
(lp10931
I1
aI3
aI3
aI15
aI9
aI47
aI35
aI115
aI227
aI615
aI605
aI1143
aI5923
aI10939
aI9497
aI24911
asS'd'
I5466
sa(dp10932
S'a'
I28699
sS's'
I16
sS'm'
(lp10933
I1
aI1
aI3
aI15
aI23
aI53
aI111
aI87
aI423
aI497
aI85
aI3525
aI7341
aI8885
aI21543
aI30083
asS'd'
I5467
sa(dp10934
S'a'
I28706
sS's'
I16
sS'm'
(lp10935
I1
aI1
aI5
aI3
aI21
aI5
aI117
aI157
aI407
aI743
aI715
aI1883
aI4425
aI10187
aI6395
aI43673
asS'd'
I5468
sa(dp10936
S'a'
I28708
sS's'
I16
sS'm'
(lp10937
I1
aI3
aI3
aI3
aI31
aI39
aI119
aI77
aI269
aI891
aI1391
aI3085
aI2881
aI10639
aI3391
aI44911
asS'd'
I5469
sa(dp10938
S'a'
I28717
sS's'
I16
sS'm'
(lp10939
I1
aI3
aI7
aI5
aI7
aI5
aI115
aI91
aI5
aI107
aI1401
aI1409
aI1811
aI737
aI5399
aI9119
asS'd'
I5470
sa(dp10940
S'a'
I28720
sS's'
I16
sS'm'
(lp10941
I1
aI1
aI5
aI9
aI17
aI45
aI107
aI15
aI397
aI299
aI1219
aI1675
aI963
aI10111
aI31679
aI13809
asS'd'
I5471
sa(dp10942
S'a'
I28735
sS's'
I16
sS'm'
(lp10943
I1
aI1
aI3
aI7
aI29
aI17
aI43
aI95
aI261
aI601
aI1091
aI3633
aI1357
aI13461
aI16583
aI12183
asS'd'
I5472
sa(dp10944
S'a'
I28761
sS's'
I16
sS'm'
(lp10945
I1
aI1
aI5
aI1
aI19
aI55
aI5
aI195
aI187
aI427
aI421
aI1717
aI4223
aI2943
aI23147
aI32985
asS'd'
I5473
sa(dp10946
S'a'
I28783
sS's'
I16
sS'm'
(lp10947
I1
aI3
aI1
aI3
aI3
aI23
aI69
aI95
aI347
aI273
aI1223
aI3061
aI1587
aI4707
aI32415
aI53991
asS'd'
I5474
sa(dp10948
S'a'
I28788
sS's'
I16
sS'm'
(lp10949
I1
aI1
aI7
aI13
aI15
aI13
aI29
aI151
aI325
aI949
aI2029
aI813
aI5339
aI14165
aI1159
aI56917
asS'd'
I5475
sa(dp10950
S'a'
I28811
sS's'
I16
sS'm'
(lp10951
I1
aI1
aI1
aI13
aI9
aI33
aI67
aI109
aI215
aI313
aI1407
aI3543
aI2403
aI5051
aI20367
aI13527
asS'd'
I5476
sa(dp10952
S'a'
I28825
sS's'
I16
sS'm'
(lp10953
I1
aI3
aI1
aI9
aI5
aI1
aI9
aI195
aI497
aI495
aI1107
aI745
aI1647
aI10637
aI1933
aI44965
asS'd'
I5477
sa(dp10954
S'a'
I28838
sS's'
I16
sS'm'
(lp10955
I1
aI1
aI3
aI9
aI13
aI19
aI49
aI183
aI497
aI519
aI1433
aI519
aI4317
aI2359
aI10349
aI63789
asS'd'
I5478
sa(dp10956
S'a'
I28850
sS's'
I16
sS'm'
(lp10957
I1
aI3
aI5
aI9
aI29
aI45
aI55
aI163
aI189
aI533
aI275
aI237
aI5453
aI8895
aI6377
aI14117
asS'd'
I5479
sa(dp10958
S'a'
I28891
sS's'
I16
sS'm'
(lp10959
I1
aI3
aI7
aI5
aI25
aI3
aI111
aI241
aI139
aI383
aI689
aI3481
aI2557
aI11163
aI5275
aI14671
asS'd'
I5480
sa(dp10960
S'a'
I28897
sS's'
I16
sS'm'
(lp10961
I1
aI3
aI3
aI9
aI3
aI5
aI5
aI141
aI507
aI331
aI645
aI1957
aI5857
aI2083
aI24717
aI11131
asS'd'
I5481
sa(dp10962
S'a'
I28932
sS's'
I16
sS'm'
(lp10963
I1
aI1
aI5
aI1
aI11
aI49
aI113
aI45
aI491
aI945
aI1467
aI3485
aI6369
aI15983
aI14489
aI12679
asS'd'
I5482
sa(dp10964
S'a'
I28975
sS's'
I16
sS'm'
(lp10965
I1
aI3
aI7
aI9
aI11
aI41
aI77
aI127
aI147
aI635
aI1757
aI587
aI7423
aI4233
aI14875
aI30531
asS'd'
I5483
sa(dp10966
S'a'
I28998
sS's'
I16
sS'm'
(lp10967
I1
aI3
aI3
aI9
aI17
aI29
aI21
aI249
aI155
aI441
aI1443
aI2093
aI1967
aI2117
aI5815
aI3857
asS'd'
I5484
sa(dp10968
S'a'
I29052
sS's'
I16
sS'm'
(lp10969
I1
aI3
aI5
aI3
aI11
aI55
aI75
aI157
aI105
aI507
aI309
aI3737
aI2645
aI7547
aI29373
aI62775
asS'd'
I5485
sa(dp10970
S'a'
I29090
sS's'
I16
sS'm'
(lp10971
I1
aI1
aI3
aI3
aI11
aI29
aI49
aI241
aI21
aI653
aI1273
aI715
aI8123
aI14241
aI25257
aI1681
asS'd'
I5486
sa(dp10972
S'a'
I29096
sS's'
I16
sS'm'
(lp10973
I1
aI1
aI7
aI5
aI11
aI31
aI33
aI215
aI243
aI369
aI247
aI3365
aI4065
aI9389
aI32457
aI58393
asS'd'
I5487
sa(dp10974
S'a'
I29104
sS's'
I16
sS'm'
(lp10975
I1
aI3
aI5
aI3
aI31
aI55
aI51
aI201
aI439
aI835
aI1805
aI25
aI7987
aI10611
aI26893
aI43663
asS'd'
I5488
sa(dp10976
S'a'
I29113
sS's'
I16
sS'm'
(lp10977
I1
aI1
aI5
aI9
aI27
aI51
aI29
aI31
aI17
aI163
aI71
aI603
aI3083
aI12439
aI11043
aI6471
asS'd'
I5489
sa(dp10978
S'a'
I29133
sS's'
I16
sS'm'
(lp10979
I1
aI1
aI5
aI7
aI13
aI1
aI91
aI109
aI213
aI721
aI1345
aI3037
aI3047
aI5209
aI15559
aI17467
asS'd'
I5490
sa(dp10980
S'a'
I29142
sS's'
I16
sS'm'
(lp10981
I1
aI1
aI3
aI9
aI19
aI37
aI93
aI185
aI107
aI859
aI501
aI3843
aI1631
aI4389
aI2215
aI52225
asS'd'
I5491
sa(dp10982
S'a'
I29170
sS's'
I16
sS'm'
(lp10983
I1
aI1
aI3
aI3
aI25
aI5
aI119
aI17
aI33
aI841
aI997
aI439
aI6135
aI7405
aI8445
aI40087
asS'd'
I5492
sa(dp10984
S'a'
I29192
sS's'
I16
sS'm'
(lp10985
I1
aI1
aI7
aI15
aI19
aI17
aI101
aI43
aI423
aI647
aI29
aI1143
aI3259
aI7807
aI15929
aI809
asS'd'
I5493
sa(dp10986
S'a'
I29221
sS's'
I16
sS'm'
(lp10987
I1
aI1
aI7
aI13
aI21
aI57
aI83
aI101
aI183
aI309
aI171
aI3173
aI7919
aI7263
aI29403
aI11055
asS'd'
I5494
sa(dp10988
S'a'
I29236
sS's'
I16
sS'm'
(lp10989
I1
aI1
aI1
aI13
aI3
aI1
aI57
aI15
aI435
aI713
aI459
aI847
aI3115
aI191
aI19809
aI43037
asS'd'
I5495
sa(dp10990
S'a'
I29246
sS's'
I16
sS'm'
(lp10991
I1
aI1
aI7
aI7
aI17
aI45
aI91
aI117
aI157
aI647
aI121
aI4091
aI3611
aI14169
aI19883
aI9069
asS'd'
I5496
sa(dp10992
S'a'
I29293
sS's'
I16
sS'm'
(lp10993
I1
aI1
aI7
aI7
aI1
aI47
aI21
aI253
aI419
aI157
aI549
aI2105
aI4475
aI3127
aI3939
aI5809
asS'd'
I5497
sa(dp10994
S'a'
I29305
sS's'
I16
sS'm'
(lp10995
I1
aI1
aI5
aI7
aI15
aI7
aI71
aI195
aI87
aI757
aI77
aI1391
aI151
aI12995
aI26403
aI17789
asS'd'
I5498
sa(dp10996
S'a'
I29312
sS's'
I16
sS'm'
(lp10997
I1
aI1
aI1
aI15
aI15
aI3
aI79
aI43
aI475
aI263
aI1195
aI2385
aI5955
aI7039
aI15625
aI19263
asS'd'
I5499
sa(dp10998
S'a'
I29339
sS's'
I16
sS'm'
(lp10999
I1
aI1
aI5
aI13
aI13
aI29
aI5
aI29
aI489
aI929
aI2027
aI2771
aI6899
aI14173
aI13747
aI1019
asS'd'
I5500
sa(dp11000
S'a'
I29365
sS's'
I16
sS'm'
(lp11001
I1
aI1
aI7
aI1
aI5
aI45
aI37
aI85
aI221
aI871
aI627
aI3445
aI4853
aI4243
aI21651
aI30201
asS'd'
I5501
sa(dp11002
S'a'
I29389
sS's'
I16
sS'm'
(lp11003
I1
aI3
aI7
aI11
aI9
aI49
aI73
aI245
aI161
aI321
aI579
aI2641
aI6653
aI5513
aI11555
aI53091
asS'd'
I5502
sa(dp11004
S'a'
I29402
sS's'
I16
sS'm'
(lp11005
I1
aI1
aI7
aI7
aI25
aI63
aI101
aI179
aI497
aI113
aI9
aI549
aI5341
aI6097
aI13305
aI52421
asS'd'
I5503
sa(dp11006
S'a'
I29423
sS's'
I16
sS'm'
(lp11007
I1
aI3
aI3
aI7
aI21
aI7
aI89
aI79
aI137
aI651
aI189
aI3025
aI1403
aI4559
aI32611
aI1857
asS'd'
I5504
sa(dp11008
S'a'
I29443
sS's'
I16
sS'm'
(lp11009
I1
aI3
aI1
aI13
aI27
aI55
aI61
aI135
aI81
aI195
aI799
aI3477
aI4873
aI2691
aI29769
aI59033
asS'd'
I5505
sa(dp11010
S'a'
I29445
sS's'
I16
sS'm'
(lp11011
I1
aI3
aI3
aI15
aI29
aI11
aI7
aI11
aI151
aI649
aI1511
aI2327
aI6921
aI12911
aI3571
aI35039
asS'd'
I5506
sa(dp11012
S'a'
I29463
sS's'
I16
sS'm'
(lp11013
I1
aI1
aI5
aI11
aI25
aI19
aI49
aI133
aI455
aI373
aI1827
aI3619
aI2127
aI3365
aI11327
aI52785
asS'd'
I5507
sa(dp11014
S'a'
I29473
sS's'
I16
sS'm'
(lp11015
I1
aI3
aI5
aI1
aI9
aI19
aI107
aI171
aI205
aI93
aI1557
aI2693
aI4297
aI4415
aI20407
aI19221
asS'd'
I5508
sa(dp11016
S'a'
I29493
sS's'
I16
sS'm'
(lp11017
I1
aI3
aI3
aI11
aI15
aI45
aI37
aI143
aI61
aI759
aI2047
aI2465
aI3923
aI9477
aI30831
aI46377
asS'd'
I5509
sa(dp11018
S'a'
I29506
sS's'
I16
sS'm'
(lp11019
I1
aI3
aI7
aI11
aI17
aI51
aI117
aI129
aI77
aI579
aI1167
aI1575
aI1967
aI10099
aI22137
aI31431
asS'd'
I5510
sa(dp11020
S'a'
I29518
sS's'
I16
sS'm'
(lp11021
I1
aI1
aI5
aI13
aI31
aI61
aI67
aI37
aI49
aI283
aI235
aI783
aI7353
aI5149
aI12245
aI18725
asS'd'
I5511
sa(dp11022
S'a'
I29532
sS's'
I16
sS'm'
(lp11023
I1
aI1
aI5
aI3
aI17
aI33
aI35
aI83
aI359
aI253
aI1911
aI913
aI6481
aI4635
aI24223
aI19693
asS'd'
I5512
sa(dp11024
S'a'
I29560
sS's'
I16
sS'm'
(lp11025
I1
aI1
aI1
aI13
aI19
aI15
aI81
aI131
aI417
aI969
aI1911
aI2829
aI3097
aI5333
aI11175
aI52269
asS'd'
I5513
sa(dp11026
S'a'
I29590
sS's'
I16
sS'm'
(lp11027
I1
aI3
aI7
aI15
aI5
aI39
aI19
aI205
aI329
aI83
aI1473
aI3259
aI6409
aI12297
aI30557
aI40917
asS'd'
I5514
sa(dp11028
S'a'
I29594
sS's'
I16
sS'm'
(lp11029
I1
aI3
aI1
aI15
aI17
aI33
aI123
aI185
aI501
aI299
aI621
aI929
aI5797
aI10539
aI12321
aI61043
asS'd'
I5515
sa(dp11030
S'a'
I29637
sS's'
I16
sS'm'
(lp11031
I1
aI3
aI3
aI1
aI7
aI51
aI119
aI19
aI17
aI203
aI373
aI2145
aI2367
aI9965
aI28071
aI50083
asS'd'
I5516
sa(dp11032
S'a'
I29647
sS's'
I16
sS'm'
(lp11033
I1
aI1
aI1
aI5
aI1
aI35
aI43
aI243
aI91
aI793
aI1299
aI2705
aI7987
aI1291
aI10147
aI17863
asS'd'
I5517
sa(dp11034
S'a'
I29650
sS's'
I16
sS'm'
(lp11035
I1
aI3
aI5
aI15
aI27
aI13
aI99
aI33
aI179
aI479
aI897
aI1113
aI1639
aI12321
aI23987
aI36219
asS'd'
I5518
sa(dp11036
S'a'
I29655
sS's'
I16
sS'm'
(lp11037
I1
aI1
aI5
aI9
aI29
aI41
aI85
aI9
aI389
aI583
aI293
aI1727
aI2575
aI13767
aI15443
aI40027
asS'd'
I5519
sa(dp11038
S'a'
I29661
sS's'
I16
sS'm'
(lp11039
I1
aI1
aI7
aI11
aI29
aI33
aI93
aI115
aI51
aI747
aI1569
aI3557
aI869
aI1991
aI29877
aI44131
asS'd'
I5520
sa(dp11040
S'a'
I29680
sS's'
I16
sS'm'
(lp11041
I1
aI3
aI7
aI7
aI29
aI11
aI33
aI137
aI411
aI689
aI1815
aI1789
aI6557
aI5973
aI19445
aI49449
asS'd'
I5521
sa(dp11042
S'a'
I29721
sS's'
I16
sS'm'
(lp11043
I1
aI1
aI5
aI1
aI17
aI3
aI77
aI55
aI351
aI325
aI983
aI3935
aI819
aI14127
aI18893
aI62553
asS'd'
I5522
sa(dp11044
S'a'
I29751
sS's'
I16
sS'm'
(lp11045
I1
aI3
aI3
aI1
aI15
aI33
aI25
aI159
aI135
aI385
aI837
aI3615
aI1649
aI1687
aI3421
aI47579
asS'd'
I5523
sa(dp11046
S'a'
I29755
sS's'
I16
sS'm'
(lp11047
I1
aI3
aI1
aI7
aI17
aI25
aI125
aI169
aI469
aI919
aI1789
aI863
aI2827
aI949
aI21347
aI10517
asS'd'
I5524
sa(dp11048
S'a'
I29760
sS's'
I16
sS'm'
(lp11049
I1
aI3
aI1
aI11
aI27
aI19
aI45
aI255
aI175
aI483
aI1073
aI3779
aI611
aI2809
aI179
aI19767
asS'd'
I5525
sa(dp11050
S'a'
I29772
sS's'
I16
sS'm'
(lp11051
I1
aI1
aI3
aI1
aI21
aI61
aI47
aI171
aI179
aI85
aI61
aI1209
aI4005
aI11439
aI8477
aI27229
asS'd'
I5526
sa(dp11052
S'a'
I29778
sS's'
I16
sS'm'
(lp11053
I1
aI1
aI1
aI1
aI3
aI1
aI43
aI159
aI261
aI411
aI1449
aI1621
aI3681
aI3465
aI24029
aI3493
asS'd'
I5527
sa(dp11054
S'a'
I29799
sS's'
I16
sS'm'
(lp11055
I1
aI3
aI1
aI11
aI5
aI13
aI9
aI23
aI369
aI769
aI363
aI3329
aI409
aI13151
aI30269
aI9621
asS'd'
I5528
sa(dp11056
S'a'
I29824
sS's'
I16
sS'm'
(lp11057
I1
aI1
aI5
aI1
aI13
aI39
aI121
aI39
aI295
aI981
aI1151
aI4039
aI8179
aI5007
aI25527
aI1249
asS'd'
I5529
sa(dp11058
S'a'
I29841
sS's'
I16
sS'm'
(lp11059
I1
aI3
aI7
aI5
aI17
aI21
aI47
aI233
aI211
aI349
aI643
aI109
aI7553
aI11453
aI30967
aI30959
asS'd'
I5530
sa(dp11060
S'a'
I29842
sS's'
I16
sS'm'
(lp11061
I1
aI1
aI5
aI3
aI31
aI39
aI105
aI137
aI487
aI855
aI107
aI1567
aI2385
aI2889
aI25777
aI33709
asS'd'
I5531
sa(dp11062
S'a'
I29853
sS's'
I16
sS'm'
(lp11063
I1
aI1
aI1
aI9
aI1
aI7
aI9
aI69
aI465
aI965
aI355
aI299
aI3327
aI14997
aI14599
aI2241
asS'd'
I5532
sa(dp11064
S'a'
I29867
sS's'
I16
sS'm'
(lp11065
I1
aI3
aI5
aI11
aI5
aI39
aI69
aI203
aI367
aI611
aI199
aI3931
aI5039
aI8683
aI8675
aI49151
asS'd'
I5533
sa(dp11066
S'a'
I29949
sS's'
I16
sS'm'
(lp11067
I1
aI1
aI7
aI13
aI31
aI35
aI101
aI213
aI273
aI827
aI203
aI2773
aI4131
aI1397
aI15311
aI62903
asS'd'
I5534
sa(dp11068
S'a'
I29950
sS's'
I16
sS'm'
(lp11069
I1
aI3
aI7
aI9
aI23
aI41
aI33
aI213
aI411
aI965
aI563
aI3035
aI247
aI15019
aI20429
aI61081
asS'd'
I5535
sa(dp11070
S'a'
I29964
sS's'
I16
sS'm'
(lp11071
I1
aI1
aI5
aI5
aI5
aI1
aI1
aI203
aI27
aI199
aI67
aI1301
aI7831
aI12839
aI2777
aI6325
asS'd'
I5536
sa(dp11072
S'a'
I29967
sS's'
I16
sS'm'
(lp11073
I1
aI3
aI1
aI11
aI27
aI3
aI11
aI173
aI9
aI121
aI1701
aI2741
aI29
aI16319
aI15849
aI11989
asS'd'
I5537
sa(dp11074
S'a'
I29985
sS's'
I16
sS'm'
(lp11075
I1
aI1
aI5
aI13
aI17
aI49
aI125
aI153
aI261
aI603
aI1617
aI3967
aI6083
aI7745
aI19683
aI49885
asS'd'
I5538
sa(dp11076
S'a'
I29992
sS's'
I16
sS'm'
(lp11077
I1
aI3
aI3
aI7
aI23
aI13
aI39
aI169
aI135
aI361
aI579
aI1443
aI7615
aI2389
aI5669
aI651
asS'd'
I5539
sa(dp11078
S'a'
I30000
sS's'
I16
sS'm'
(lp11079
I1
aI3
aI5
aI9
aI31
aI19
aI81
aI83
aI483
aI93
aI1895
aI2285
aI7771
aI8281
aI8353
aI39677
asS'd'
I5540
sa(dp11080
S'a'
I30020
sS's'
I16
sS'm'
(lp11081
I1
aI1
aI7
aI7
aI23
aI51
aI127
aI25
aI101
aI611
aI1095
aI3013
aI2685
aI8153
aI22629
aI53355
asS'd'
I5541
sa(dp11082
S'a'
I30024
sS's'
I16
sS'm'
(lp11083
I1
aI1
aI1
aI11
aI11
aI37
aI35
aI127
aI317
aI877
aI1591
aI401
aI4121
aI9945
aI12121
aI28257
asS'd'
I5542
sa(dp11084
S'a'
I30030
sS's'
I16
sS'm'
(lp11085
I1
aI3
aI5
aI11
aI23
aI9
aI43
aI135
aI37
aI405
aI2009
aI2903
aI3065
aI6591
aI8473
aI58231
asS'd'
I5543
sa(dp11086
S'a'
I30066
sS's'
I16
sS'm'
(lp11087
I1
aI1
aI3
aI11
aI21
aI45
aI21
aI205
aI425
aI891
aI357
aI2609
aI495
aI7541
aI2161
aI37853
asS'd'
I5544
sa(dp11088
S'a'
I30071
sS's'
I16
sS'm'
(lp11089
I1
aI3
aI1
aI1
aI25
aI9
aI113
aI243
aI317
aI491
aI997
aI2023
aI5869
aI13643
aI11483
aI6733
asS'd'
I5545
sa(dp11090
S'a'
I30078
sS's'
I16
sS'm'
(lp11091
I1
aI3
aI1
aI15
aI13
aI3
aI75
aI25
aI409
aI421
aI1817
aI857
aI4575
aI12559
aI1211
aI62177
asS'd'
I5546
sa(dp11092
S'a'
I30082
sS's'
I16
sS'm'
(lp11093
I1
aI1
aI3
aI7
aI17
aI35
aI115
aI195
aI217
aI223
aI1195
aI749
aI5619
aI7265
aI7369
aI46907
asS'd'
I5547
sa(dp11094
S'a'
I30096
sS's'
I16
sS'm'
(lp11095
I1
aI1
aI1
aI13
aI5
aI57
aI117
aI161
aI121
aI533
aI987
aI3959
aI5047
aI15213
aI15811
aI41841
asS'd'
I5548
sa(dp11096
S'a'
I30101
sS's'
I16
sS'm'
(lp11097
I1
aI3
aI7
aI1
aI19
aI55
aI97
aI191
aI217
aI75
aI1881
aI3351
aI3737
aI12179
aI22875
aI28767
asS'd'
I5549
sa(dp11098
S'a'
I30102
sS's'
I16
sS'm'
(lp11099
I1
aI3
aI1
aI9
aI15
aI41
aI9
aI97
aI491
aI31
aI1191
aI963
aI875
aI8259
aI2723
aI9503
asS'd'
I5550
sa(dp11100
S'a'
I30122
sS's'
I16
sS'm'
(lp11101
I1
aI3
aI7
aI9
aI3
aI17
aI21
aI71
aI1
aI523
aI2031
aI3469
aI3181
aI8707
aI6093
aI8837
asS'd'
I5551
sa(dp11102
S'a'
I30141
sS's'
I16
sS'm'
(lp11103
I1
aI3
aI5
aI3
aI5
aI1
aI11
aI91
aI33
aI37
aI643
aI85
aI4325
aI4293
aI8351
aI28311
asS'd'
I5552
sa(dp11104
S'a'
I30159
sS's'
I16
sS'm'
(lp11105
I1
aI3
aI7
aI5
aI15
aI45
aI47
aI183
aI391
aI113
aI493
aI3607
aI2541
aI13521
aI31613
aI36049
asS'd'
I5553
sa(dp11106
S'a'
I30168
sS's'
I16
sS'm'
(lp11107
I1
aI1
aI3
aI9
aI15
aI33
aI115
aI69
aI289
aI217
aI1875
aI1339
aI4995
aI9073
aI6909
aI15977
asS'd'
I5554
sa(dp11108
S'a'
I30177
sS's'
I16
sS'm'
(lp11109
I1
aI1
aI7
aI3
aI9
aI29
aI39
aI219
aI119
aI369
aI893
aI1293
aI4511
aI15703
aI11093
aI30259
asS'd'
I5555
sa(dp11110
S'a'
I30183
sS's'
I16
sS'm'
(lp11111
I1
aI1
aI5
aI13
aI19
aI9
aI17
aI75
aI149
aI415
aI35
aI97
aI563
aI1689
aI18311
aI54291
asS'd'
I5556
sa(dp11112
S'a'
I30197
sS's'
I16
sS'm'
(lp11113
I1
aI1
aI7
aI3
aI17
aI15
aI71
aI29
aI25
aI883
aI1801
aI1675
aI5585
aI9413
aI3813
aI26673
asS'd'
I5557
sa(dp11114
S'a'
I30213
sS's'
I16
sS'm'
(lp11115
I1
aI1
aI3
aI15
aI5
aI13
aI31
aI41
aI311
aI411
aI573
aI281
aI8075
aI7163
aI11817
aI29121
asS'd'
I5558
sa(dp11116
S'a'
I30231
sS's'
I16
sS'm'
(lp11117
I1
aI1
aI7
aI9
aI7
aI57
aI15
aI141
aI337
aI123
aI269
aI3737
aI6455
aI2539
aI13655
aI59809
asS'd'
I5559
sa(dp11118
S'a'
I30232
sS's'
I16
sS'm'
(lp11119
I1
aI3
aI1
aI15
aI15
aI23
aI111
aI51
aI429
aI483
aI1567
aI1317
aI8057
aI1609
aI30181
aI35687
asS'd'
I5560
sa(dp11120
S'a'
I30241
sS's'
I16
sS'm'
(lp11121
I1
aI3
aI7
aI9
aI25
aI43
aI67
aI13
aI319
aI587
aI1827
aI443
aI2031
aI8563
aI16173
aI58667
asS'd'
I5561
sa(dp11122
S'a'
I30253
sS's'
I16
sS'm'
(lp11123
I1
aI3
aI5
aI13
aI11
aI63
aI89
aI105
aI377
aI257
aI7
aI4077
aI5091
aI5125
aI25
aI39033
asS'd'
I5562
sa(dp11124
S'a'
I30259
sS's'
I16
sS'm'
(lp11125
I1
aI3
aI3
aI1
aI9
aI29
aI7
aI87
aI239
aI469
aI1851
aI1711
aI5797
aI7137
aI11405
aI20175
asS'd'
I5563
sa(dp11126
S'a'
I30262
sS's'
I16
sS'm'
(lp11127
I1
aI3
aI3
aI1
aI13
aI17
aI101
aI209
aI301
aI95
aI1181
aI3091
aI4451
aI1241
aI17057
aI335
asS'd'
I5564
sa(dp11128
S'a'
I30268
sS's'
I16
sS'm'
(lp11129
I1
aI1
aI1
aI9
aI31
aI7
aI81
aI161
aI391
aI677
aI115
aI141
aI5375
aI7279
aI1887
aI1645
asS'd'
I5565
sa(dp11130
S'a'
I30297
sS's'
I16
sS'm'
(lp11131
I1
aI1
aI1
aI11
aI27
aI61
aI3
aI195
aI189
aI409
aI1747
aI331
aI2931
aI9535
aI1369
aI47233
asS'd'
I5566
sa(dp11132
S'a'
I30316
sS's'
I16
sS'm'
(lp11133
I1
aI3
aI5
aI15
aI7
aI15
aI105
aI255
aI491
aI689
aI97
aI1131
aI3459
aI7147
aI27541
aI62307
asS'd'
I5567
sa(dp11134
S'a'
I30322
sS's'
I16
sS'm'
(lp11135
I1
aI3
aI5
aI9
aI5
aI23
aI1
aI209
aI233
aI717
aI1919
aI1835
aI5073
aI10403
aI28979
aI1945
asS'd'
I5568
sa(dp11136
S'a'
I30344
sS's'
I16
sS'm'
(lp11137
I1
aI1
aI3
aI9
aI3
aI35
aI107
aI209
aI255
aI447
aI227
aI273
aI443
aI9367
aI24105
aI34095
asS'd'
I5569
sa(dp11138
S'a'
I30350
sS's'
I16
sS'm'
(lp11139
I1
aI1
aI3
aI11
aI3
aI33
aI5
aI165
aI83
aI787
aI1555
aI31
aI4351
aI16301
aI27453
aI56739
asS'd'
I5570
sa(dp11140
S'a'
I30355
sS's'
I16
sS'm'
(lp11141
I1
aI1
aI5
aI5
aI29
aI9
aI127
aI253
aI281
aI487
aI441
aI1129
aI2811
aI9113
aI28855
aI57117
asS'd'
I5571
sa(dp11142
S'a'
I30429
sS's'
I16
sS'm'
(lp11143
I1
aI1
aI1
aI13
aI13
aI1
aI17
aI143
aI121
aI917
aI1571
aI3777
aI2297
aI3691
aI3001
aI42327
asS'd'
I5572
sa(dp11144
S'a'
I30445
sS's'
I16
sS'm'
(lp11145
I1
aI1
aI5
aI1
aI25
aI7
aI41
aI245
aI241
aI929
aI1203
aI3755
aI7113
aI9333
aI22549
aI12253
asS'd'
I5573
sa(dp11146
S'a'
I30477
sS's'
I16
sS'm'
(lp11147
I1
aI3
aI1
aI11
aI1
aI13
aI69
aI73
aI285
aI975
aI1331
aI3411
aI7777
aI3489
aI2763
aI44297
asS'd'
I5574
sa(dp11148
S'a'
I30513
sS's'
I16
sS'm'
(lp11149
I1
aI3
aI5
aI11
aI3
aI37
aI21
aI105
aI153
aI307
aI989
aI627
aI3127
aI6169
aI10573
aI22139
asS'd'
I5575
sa(dp11150
S'a'
I30520
sS's'
I16
sS'm'
(lp11151
I1
aI3
aI5
aI15
aI11
aI11
aI39
aI21
aI355
aI437
aI547
aI2283
aI6443
aI5561
aI6367
aI53899
asS'd'
I5576
sa(dp11152
S'a'
I30540
sS's'
I16
sS'm'
(lp11153
I1
aI1
aI1
aI9
aI25
aI51
aI97
aI175
aI131
aI207
aI1367
aI2561
aI7455
aI8289
aI5877
aI4383
asS'd'
I5577
sa(dp11154
S'a'
I30551
sS's'
I16
sS'm'
(lp11155
I1
aI3
aI7
aI1
aI29
aI17
aI7
aI1
aI43
aI831
aI591
aI2145
aI975
aI909
aI23107
aI43987
asS'd'
I5578
sa(dp11156
S'a'
I30573
sS's'
I16
sS'm'
(lp11157
I1
aI3
aI5
aI13
aI5
aI47
aI65
aI65
aI439
aI807
aI271
aI1615
aI1873
aI10905
aI30537
aI3337
asS'd'
I5579
sa(dp11158
S'a'
I30609
sS's'
I16
sS'm'
(lp11159
I1
aI1
aI1
aI13
aI29
aI1
aI53
aI5
aI307
aI347
aI1059
aI545
aI1129
aI11883
aI5969
aI50433
asS'd'
I5580
sa(dp11160
S'a'
I30622
sS's'
I16
sS'm'
(lp11161
I1
aI1
aI3
aI5
aI31
aI29
aI63
aI201
aI255
aI803
aI677
aI1499
aI1691
aI14037
aI18251
aI6881
asS'd'
I5581
sa(dp11162
S'a'
I30635
sS's'
I16
sS'm'
(lp11163
I1
aI3
aI1
aI5
aI5
aI13
aI13
aI121
aI505
aI855
aI467
aI2803
aI3297
aI4689
aI18443
aI60757
asS'd'
I5582
sa(dp11164
S'a'
I30658
sS's'
I16
sS'm'
(lp11165
I1
aI1
aI5
aI13
aI11
aI19
aI11
aI201
aI101
aI431
aI693
aI1267
aI6909
aI7759
aI2265
aI6125
asS'd'
I5583
sa(dp11166
S'a'
I30667
sS's'
I16
sS'm'
(lp11167
I1
aI1
aI7
aI13
aI9
aI3
aI37
aI209
aI269
aI27
aI1041
aI2587
aI4667
aI11077
aI27009
aI27967
asS'd'
I5584
sa(dp11168
S'a'
I30681
sS's'
I16
sS'm'
(lp11169
I1
aI1
aI5
aI5
aI1
aI5
aI127
aI179
aI463
aI949
aI1525
aI231
aI1201
aI3283
aI9929
aI46677
asS'd'
I5585
sa(dp11170
S'a'
I30684
sS's'
I16
sS'm'
(lp11171
I1
aI3
aI1
aI15
aI9
aI11
aI89
aI129
aI331
aI833
aI1415
aI229
aI2059
aI13145
aI30525
aI33773
asS'd'
I5586
sa(dp11172
S'a'
I30703
sS's'
I16
sS'm'
(lp11173
I1
aI1
aI7
aI15
aI7
aI43
aI95
aI177
aI313
aI989
aI483
aI825
aI1885
aI4535
aI8213
aI39835
asS'd'
I5587
sa(dp11174
S'a'
I30705
sS's'
I16
sS'm'
(lp11175
I1
aI1
aI7
aI3
aI19
aI27
aI45
aI163
aI17
aI523
aI1565
aI3753
aI7433
aI14117
aI8499
aI40177
asS'd'
I5588
sa(dp11176
S'a'
I30715
sS's'
I16
sS'm'
(lp11177
I1
aI3
aI3
aI15
aI23
aI45
aI95
aI31
aI55
aI469
aI383
aI237
aI6287
aI5561
aI20901
aI48259
asS'd'
I5589
sa(dp11178
S'a'
I30742
sS's'
I16
sS'm'
(lp11179
I1
aI1
aI3
aI1
aI23
aI61
aI101
aI185
aI35
aI553
aI463
aI1169
aI2875
aI12491
aI14327
aI47999
asS'd'
I5590
sa(dp11180
S'a'
I30748
sS's'
I16
sS'm'
(lp11181
I1
aI3
aI3
aI13
aI23
aI29
aI77
aI21
aI19
aI3
aI769
aI1943
aI2081
aI9135
aI29767
aI11367
asS'd'
I5591
sa(dp11182
S'a'
I30758
sS's'
I16
sS'm'
(lp11183
I1
aI1
aI5
aI15
aI5
aI11
aI59
aI163
aI355
aI993
aI375
aI3181
aI2675
aI8515
aI17007
aI38467
asS'd'
I5592
sa(dp11184
S'a'
I30767
sS's'
I16
sS'm'
(lp11185
I1
aI1
aI5
aI13
aI19
aI5
aI107
aI83
aI123
aI843
aI413
aI2137
aI7531
aI3833
aI6149
aI55925
asS'd'
I5593
sa(dp11186
S'a'
I30770
sS's'
I16
sS'm'
(lp11187
I1
aI3
aI1
aI13
aI23
aI9
aI41
aI145
aI265
aI591
aI1899
aI3145
aI5255
aI13653
aI12245
aI25367
asS'd'
I5594
sa(dp11188
S'a'
I30772
sS's'
I16
sS'm'
(lp11189
I1
aI1
aI3
aI15
aI1
aI45
aI119
aI79
aI121
aI137
aI1945
aI2041
aI2409
aI1377
aI29501
aI29885
asS'd'
I5595
sa(dp11190
S'a'
I30807
sS's'
I16
sS'm'
(lp11191
I1
aI1
aI7
aI11
aI27
aI57
aI75
aI183
aI341
aI237
aI1909
aI2785
aI5973
aI9965
aI21729
aI45089
asS'd'
I5596
sa(dp11192
S'a'
I30814
sS's'
I16
sS'm'
(lp11193
I1
aI3
aI5
aI7
aI21
aI1
aI41
aI189
aI131
aI1021
aI1375
aI1463
aI5985
aI12499
aI4115
aI9131
asS'd'
I5597
sa(dp11194
S'a'
I30824
sS's'
I16
sS'm'
(lp11195
I1
aI3
aI7
aI15
aI21
aI19
aI59
aI171
aI339
aI841
aI1725
aI2909
aI6437
aI2499
aI17191
aI43275
asS'd'
I5598
sa(dp11196
S'a'
I30857
sS's'
I16
sS'm'
(lp11197
I1
aI3
aI1
aI1
aI15
aI55
aI31
aI199
aI351
aI183
aI1819
aI1873
aI7877
aI12407
aI7881
aI1663
asS'd'
I5599
sa(dp11198
S'a'
I30875
sS's'
I16
sS'm'
(lp11199
I1
aI1
aI3
aI15
aI1
aI61
aI111
aI61
aI115
aI243
aI1281
aI3195
aI1229
aI10973
aI189
aI36049
asS'd'
I5600
sa(dp11200
S'a'
I30931
sS's'
I16
sS'm'
(lp11201
I1
aI1
aI3
aI15
aI13
aI13
aI3
aI49
aI61
aI839
aI1615
aI1853
aI3619
aI7805
aI25441
aI8789
asS'd'
I5601
sa(dp11202
S'a'
I30933
sS's'
I16
sS'm'
(lp11203
I1
aI3
aI1
aI9
aI27
aI43
aI7
aI193
aI397
aI869
aI1079
aI1785
aI6535
aI1801
aI29363
aI59269
asS'd'
I5602
sa(dp11204
S'a'
I30949
sS's'
I16
sS'm'
(lp11205
I1
aI3
aI5
aI5
aI31
aI57
aI37
aI53
aI41
aI871
aI809
aI1235
aI1011
aI12979
aI8749
aI52151
asS'd'
I5603
sa(dp11206
S'a'
I30953
sS's'
I16
sS'm'
(lp11207
I1
aI1
aI7
aI13
aI25
aI59
aI69
aI117
aI463
aI587
aI513
aI297
aI6991
aI5905
aI25737
aI37249
asS'd'
I5604
sa(dp11208
S'a'
I30982
sS's'
I16
sS'm'
(lp11209
I1
aI1
aI5
aI1
aI27
aI19
aI121
aI97
aI349
aI793
aI1971
aI3057
aI4781
aI15841
aI22625
aI58637
asS'd'
I5605
sa(dp11210
S'a'
I31010
sS's'
I16
sS'm'
(lp11211
I1
aI1
aI5
aI5
aI25
aI31
aI11
aI133
aI411
aI239
aI1071
aI3473
aI1733
aI7175
aI31841
aI46665
asS'd'
I5606
sa(dp11212
S'a'
I31012
sS's'
I16
sS'm'
(lp11213
I1
aI3
aI3
aI13
aI19
aI25
aI99
aI175
aI271
aI175
aI1755
aI3597
aI4615
aI15207
aI25573
aI16089
asS'd'
I5607
sa(dp11214
S'a'
I31039
sS's'
I16
sS'm'
(lp11215
I1
aI1
aI7
aI11
aI17
aI19
aI119
aI91
aI505
aI791
aI55
aI2979
aI7463
aI10147
aI23647
aI33283
asS'd'
I5608
sa(dp11216
S'a'
I31041
sS's'
I16
sS'm'
(lp11217
I1
aI3
aI1
aI1
aI21
aI11
aI43
aI173
aI239
aI839
aI1533
aI1559
aI549
aI15621
aI22133
aI46387
asS'd'
I5609
sa(dp11218
S'a'
I31061
sS's'
I16
sS'm'
(lp11219
I1
aI1
aI3
aI13
aI31
aI15
aI73
aI15
aI209
aI267
aI701
aI2899
aI1163
aI10093
aI7727
aI44211
asS'd'
I5610
sa(dp11220
S'a'
I31082
sS's'
I16
sS'm'
(lp11221
I1
aI3
aI1
aI11
aI29
aI21
aI5
aI39
aI421
aI375
aI411
aI3693
aI3901
aI8507
aI10883
aI16189
asS'd'
I5611
sa(dp11222
S'a'
I31087
sS's'
I16
sS'm'
(lp11223
I1
aI3
aI1
aI7
aI13
aI13
aI73
aI167
aI149
aI677
aI1435
aI621
aI2511
aI13813
aI13129
aI55327
asS'd'
I5612
sa(dp11224
S'a'
I31096
sS's'
I16
sS'm'
(lp11225
I1
aI3
aI5
aI15
aI7
aI59
aI83
aI221
aI77
aI357
aI281
aI2689
aI5629
aI5837
aI1701
aI30811
asS'd'
I5613
sa(dp11226
S'a'
I31115
sS's'
I16
sS'm'
(lp11227
I1
aI3
aI3
aI11
aI17
aI1
aI43
aI95
aI473
aI981
aI1487
aI1337
aI905
aI3307
aI22357
aI181
asS'd'
I5614
sa(dp11228
S'a'
I31130
sS's'
I16
sS'm'
(lp11229
I1
aI1
aI3
aI7
aI1
aI27
aI9
aI3
aI489
aI1
aI1265
aI2463
aI539
aI12769
aI825
aI6149
asS'd'
I5615
sa(dp11230
S'a'
I31168
sS's'
I16
sS'm'
(lp11231
I1
aI3
aI3
aI3
aI11
aI27
aI81
aI237
aI411
aI241
aI1613
aI931
aI6397
aI4325
aI29651
aI49003
asS'd'
I5616
sa(dp11232
S'a'
I31171
sS's'
I16
sS'm'
(lp11233
I1
aI3
aI3
aI13
aI1
aI19
aI55
aI73
aI47
aI203
aI1661
aI1245
aI6847
aI2457
aI25427
aI33069
asS'd'
I5617
sa(dp11234
S'a'
I31177
sS's'
I16
sS'm'
(lp11235
I1
aI3
aI7
aI3
aI7
aI47
aI11
aI165
aI391
aI457
aI301
aI1213
aI1913
aI14531
aI7847
aI14811
asS'd'
I5618
sa(dp11236
S'a'
I31180
sS's'
I16
sS'm'
(lp11237
I1
aI3
aI1
aI9
aI1
aI9
aI57
aI203
aI15
aI733
aI1131
aI2751
aI5869
aI3165
aI21497
aI28881
asS'd'
I5619
sa(dp11238
S'a'
I31191
sS's'
I16
sS'm'
(lp11239
I1
aI3
aI1
aI5
aI9
aI7
aI29
aI85
aI71
aI571
aI469
aI2395
aI2819
aI8443
aI2281
aI50489
asS'd'
I5620
sa(dp11240
S'a'
I31207
sS's'
I16
sS'm'
(lp11241
I1
aI3
aI5
aI11
aI13
aI63
aI47
aI47
aI349
aI21
aI861
aI2217
aI2945
aI6967
aI6605
aI16459
asS'd'
I5621
sa(dp11242
S'a'
I31247
sS's'
I16
sS'm'
(lp11243
I1
aI1
aI7
aI5
aI13
aI3
aI41
aI53
aI409
aI289
aI1225
aI2965
aI5283
aI1785
aI14443
aI51755
asS'd'
I5622
sa(dp11244
S'a'
I31249
sS's'
I16
sS'm'
(lp11245
I1
aI3
aI7
aI13
aI19
aI1
aI29
aI191
aI119
aI37
aI697
aI1909
aI481
aI14157
aI13425
aI60581
asS'd'
I5623
sa(dp11246
S'a'
I31285
sS's'
I16
sS'm'
(lp11247
I1
aI3
aI1
aI13
aI1
aI15
aI105
aI79
aI505
aI681
aI1741
aI3683
aI5775
aI7479
aI11387
aI1321
asS'd'
I5624
sa(dp11248
S'a'
I31303
sS's'
I16
sS'm'
(lp11249
I1
aI1
aI1
aI11
aI9
aI35
aI77
aI73
aI351
aI217
aI2029
aI2845
aI5143
aI5677
aI15465
aI33123
asS'd'
I5625
sa(dp11250
S'a'
I31310
sS's'
I16
sS'm'
(lp11251
I1
aI1
aI3
aI3
aI19
aI49
aI63
aI109
aI335
aI743
aI741
aI1673
aI3311
aI3139
aI25197
aI13793
asS'd'
I5626
sa(dp11252
S'a'
I31337
sS's'
I16
sS'm'
(lp11253
I1
aI3
aI1
aI3
aI29
aI63
aI79
aI1
aI493
aI13
aI1487
aI4015
aI6983
aI1433
aI26023
aI55591
asS'd'
I5627
sa(dp11254
S'a'
I31352
sS's'
I16
sS'm'
(lp11255
I1
aI3
aI3
aI11
aI1
aI25
aI57
aI207
aI309
aI201
aI1513
aI1749
aI3785
aI9217
aI11531
aI40597
asS'd'
I5628
sa(dp11256
S'a'
I31357
sS's'
I16
sS'm'
(lp11257
I1
aI3
aI7
aI13
aI3
aI23
aI69
aI253
aI311
aI773
aI807
aI1063
aI745
aI4843
aI25221
aI55885
asS'd'
I5629
sa(dp11258
S'a'
I31374
sS's'
I16
sS'm'
(lp11259
I1
aI1
aI3
aI11
aI29
aI47
aI67
aI183
aI11
aI259
aI5
aI1935
aI2295
aI8105
aI19139
aI11707
asS'd'
I5630
sa(dp11260
S'a'
I31379
sS's'
I16
sS'm'
(lp11261
I1
aI1
aI3
aI3
aI23
aI3
aI53
aI165
aI255
aI501
aI1547
aI3649
aI5631
aI13307
aI8869
aI5595
asS'd'
I5631
sa(dp11262
S'a'
I31388
sS's'
I16
sS'm'
(lp11263
I1
aI1
aI3
aI5
aI7
aI29
aI37
aI223
aI289
aI925
aI959
aI309
aI1479
aI3141
aI18661
aI52123
asS'd'
I5632
sa(dp11264
S'a'
I31410
sS's'
I16
sS'm'
(lp11265
I1
aI3
aI1
aI1
aI7
aI59
aI101
aI219
aI91
aI793
aI1103
aI1485
aI7547
aI12889
aI19097
aI15613
asS'd'
I5633
sa(dp11266
S'a'
I31416
sS's'
I16
sS'm'
(lp11267
I1
aI1
aI5
aI15
aI1
aI17
aI79
aI83
aI131
aI683
aI1611
aI1635
aI5405
aI9621
aI29489
aI4801
asS'd'
I5634
sa(dp11268
S'a'
I31467
sS's'
I16
sS'm'
(lp11269
I1
aI1
aI5
aI7
aI31
aI63
aI59
aI125
aI401
aI261
aI1445
aI33
aI187
aI12913
aI8639
aI48413
asS'd'
I5635
sa(dp11270
S'a'
I31495
sS's'
I16
sS'm'
(lp11271
I1
aI3
aI3
aI13
aI27
aI37
aI27
aI99
aI379
aI851
aI1311
aI4051
aI5483
aI13935
aI29679
aI30905
asS'd'
I5636
sa(dp11272
S'a'
I31504
sS's'
I16
sS'm'
(lp11273
I1
aI1
aI3
aI1
aI7
aI57
aI79
aI23
aI97
aI561
aI1083
aI2327
aI1545
aI5387
aI12119
aI29717
asS'd'
I5637
sa(dp11274
S'a'
I31507
sS's'
I16
sS'm'
(lp11275
I1
aI1
aI7
aI7
aI9
aI41
aI63
aI165
aI315
aI247
aI89
aI2055
aI7399
aI1399
aI2057
aI39851
asS'd'
I5638
sa(dp11276
S'a'
I31509
sS's'
I16
sS'm'
(lp11277
I1
aI1
aI1
aI15
aI9
aI23
aI7
aI15
aI457
aI669
aI661
aI3269
aI915
aI3475
aI15845
aI59769
asS'd'
I5639
sa(dp11278
S'a'
I31514
sS's'
I16
sS'm'
(lp11279
I1
aI3
aI7
aI15
aI17
aI53
aI83
aI5
aI457
aI103
aI1297
aI2413
aI1095
aI7711
aI27935
aI56357
asS'd'
I5640
sa(dp11280
S'a'
I31516
sS's'
I16
sS'm'
(lp11281
I1
aI1
aI3
aI5
aI17
aI3
aI81
aI23
aI165
aI341
aI781
aI3583
aI1751
aI6763
aI13937
aI35331
asS'd'
I5641
sa(dp11282
S'a'
I31530
sS's'
I16
sS'm'
(lp11283
I1
aI1
aI5
aI11
aI31
aI21
aI7
aI63
aI369
aI867
aI573
aI45
aI2781
aI4507
aI21553
aI51933
asS'd'
I5642
sa(dp11284
S'a'
I31555
sS's'
I16
sS'm'
(lp11285
I1
aI1
aI5
aI15
aI1
aI37
aI85
aI133
aI489
aI733
aI1471
aI2089
aI979
aI7723
aI7339
aI59595
asS'd'
I5643
sa(dp11286
S'a'
I31567
sS's'
I16
sS'm'
(lp11287
I1
aI1
aI1
aI1
aI7
aI3
aI3
aI77
aI137
aI1009
aI481
aI1343
aI397
aI15865
aI21701
aI37509
asS'd'
I5644
sa(dp11288
S'a'
I31570
sS's'
I16
sS'm'
(lp11289
I1
aI3
aI7
aI5
aI17
aI57
aI19
aI245
aI249
aI289
aI1847
aI3057
aI4905
aI5905
aI32459
aI41305
asS'd'
I5645
sa(dp11290
S'a'
I31586
sS's'
I16
sS'm'
(lp11291
I1
aI1
aI5
aI1
aI23
aI23
aI1
aI177
aI115
aI337
aI983
aI421
aI3135
aI6319
aI27109
aI59641
asS'd'
I5646
sa(dp11292
S'a'
I31598
sS's'
I16
sS'm'
(lp11293
I1
aI3
aI1
aI5
aI25
aI1
aI63
aI73
aI61
aI967
aI1567
aI2645
aI7347
aI11877
aI28777
aI38507
asS'd'
I5647
sa(dp11294
S'a'
I31605
sS's'
I16
sS'm'
(lp11295
I1
aI1
aI3
aI9
aI5
aI41
aI39
aI101
aI339
aI337
aI1079
aI3861
aI5049
aI5601
aI14377
aI34093
asS'd'
I5648
sa(dp11296
S'a'
I31609
sS's'
I16
sS'm'
(lp11297
I1
aI3
aI7
aI7
aI3
aI47
aI95
aI157
aI167
aI1011
aI1117
aI3669
aI7993
aI11735
aI8505
aI64713
asS'd'
I5649
sa(dp11298
S'a'
I31612
sS's'
I16
sS'm'
(lp11299
I1
aI3
aI1
aI9
aI3
aI33
aI11
aI33
aI65
aI329
aI401
aI2659
aI2851
aI3903
aI29791
aI41613
asS'd'
I5650
sa(dp11300
S'a'
I31626
sS's'
I16
sS'm'
(lp11301
I1
aI1
aI1
aI15
aI15
aI17
aI9
aI69
aI359
aI41
aI1475
aI1919
aI5829
aI2189
aI21295
aI33255
asS'd'
I5651
sa(dp11302
S'a'
I31634
sS's'
I16
sS'm'
(lp11303
I1
aI3
aI1
aI3
aI9
aI23
aI73
aI247
aI399
aI775
aI419
aI3033
aI865
aI12595
aI16345
aI15079
asS'd'
I5652
sa(dp11304
S'a'
I31655
sS's'
I16
sS'm'
(lp11305
I1
aI3
aI1
aI5
aI1
aI17
aI33
aI23
aI419
aI585
aI673
aI929
aI6955
aI10247
aI12647
aI29107
asS'd'
I5653
sa(dp11306
S'a'
I31681
sS's'
I16
sS'm'
(lp11307
I1
aI3
aI3
aI13
aI9
aI33
aI11
aI13
aI127
aI529
aI1219
aI2401
aI6459
aI14745
aI5123
aI53023
asS'd'
I5654
sa(dp11308
S'a'
I31705
sS's'
I16
sS'm'
(lp11309
I1
aI3
aI5
aI11
aI23
aI11
aI5
aI19
aI281
aI121
aI1671
aI2171
aI4545
aI10691
aI24875
aI28849
asS'd'
I5655
sa(dp11310
S'a'
I31706
sS's'
I16
sS'm'
(lp11311
I1
aI3
aI1
aI3
aI13
aI25
aI85
aI131
aI127
aI977
aI1599
aI3319
aI3107
aI3185
aI4879
aI3455
asS'd'
I5656
sa(dp11312
S'a'
I31718
sS's'
I16
sS'm'
(lp11313
I1
aI1
aI5
aI1
aI3
aI13
aI77
aI15
aI133
aI185
aI1319
aI727
aI2181
aI12175
aI28017
aI28023
asS'd'
I5657
sa(dp11314
S'a'
I31735
sS's'
I16
sS'm'
(lp11315
I1
aI3
aI7
aI5
aI29
aI51
aI113
aI203
aI331
aI847
aI1
aI3445
aI3669
aI7711
aI13647
aI58651
asS'd'
I5658
sa(dp11316
S'a'
I31741
sS's'
I16
sS'm'
(lp11317
I1
aI3
aI1
aI3
aI31
aI27
aI35
aI199
aI491
aI839
aI1275
aI3385
aI4743
aI821
aI26259
aI11345
asS'd'
I5659
sa(dp11318
S'a'
I31744
sS's'
I16
sS'm'
(lp11319
I1
aI1
aI7
aI9
aI21
aI47
aI9
aI67
aI119
aI985
aI127
aI1987
aI5451
aI6403
aI26183
aI8349
asS'd'
I5660
sa(dp11320
S'a'
I31762
sS's'
I16
sS'm'
(lp11321
I1
aI3
aI5
aI1
aI19
aI3
aI91
aI217
aI301
aI595
aI1789
aI735
aI4993
aI229
aI18033
aI59625
asS'd'
I5661
sa(dp11322
S'a'
I31774
sS's'
I16
sS'm'
(lp11323
I1
aI3
aI3
aI3
aI11
aI25
aI103
aI211
aI117
aI9
aI773
aI1521
aI2265
aI8277
aI23179
aI22433
asS'd'
I5662
sa(dp11324
S'a'
I31864
sS's'
I16
sS'm'
(lp11325
I1
aI1
aI7
aI9
aI3
aI27
aI63
aI255
aI175
aI699
aI293
aI2409
aI3155
aI285
aI8663
aI53503
asS'd'
I5663
sa(dp11326
S'a'
I31874
sS's'
I16
sS'm'
(lp11327
I1
aI1
aI5
aI7
aI27
aI23
aI63
aI213
aI323
aI697
aI1541
aI3497
aI2985
aI12389
aI11155
aI26217
asS'd'
I5664
sa(dp11328
S'a'
I31900
sS's'
I16
sS'm'
(lp11329
I1
aI3
aI1
aI3
aI31
aI7
aI47
aI207
aI185
aI873
aI1063
aI1055
aI205
aI12469
aI23505
aI56245
asS'd'
I5665
sa(dp11330
S'a'
I31910
sS's'
I16
sS'm'
(lp11331
I1
aI3
aI7
aI13
aI31
aI17
aI47
aI95
aI91
aI483
aI1997
aI3273
aI445
aI2601
aI15219
aI10997
asS'd'
I5666
sa(dp11332
S'a'
I31928
sS's'
I16
sS'm'
(lp11333
I1
aI3
aI3
aI5
aI29
aI45
aI29
aI83
aI457
aI823
aI1395
aI1411
aI1879
aI9409
aI11609
aI32001
asS'd'
I5667
sa(dp11334
S'a'
I31965
sS's'
I16
sS'm'
(lp11335
I1
aI3
aI5
aI11
aI21
aI11
aI43
aI73
aI159
aI137
aI29
aI1957
aI815
aI5077
aI16127
aI42199
asS'd'
I5668
sa(dp11336
S'a'
I31976
sS's'
I16
sS'm'
(lp11337
I1
aI3
aI5
aI13
aI9
aI59
aI47
aI215
aI293
aI807
aI309
aI1951
aI2285
aI9287
aI1019
aI49501
asS'd'
I5669
sa(dp11338
S'a'
I32016
sS's'
I16
sS'm'
(lp11339
I1
aI1
aI7
aI13
aI31
aI7
aI95
aI189
aI233
aI363
aI1039
aI1675
aI1715
aI9049
aI8537
aI31051
asS'd'
I5670
sa(dp11340
S'a'
I32032
sS's'
I16
sS'm'
(lp11341
I1
aI3
aI7
aI9
aI23
aI35
aI125
aI251
aI107
aI401
aI1113
aI3585
aI6331
aI2363
aI27889
aI28877
asS'd'
I5671
sa(dp11342
S'a'
I32037
sS's'
I16
sS'm'
(lp11343
I1
aI1
aI7
aI13
aI9
aI1
aI13
aI69
aI257
aI369
aI547
aI1595
aI1823
aI9553
aI25653
aI31181
asS'd'
I5672
sa(dp11344
S'a'
I32062
sS's'
I16
sS'm'
(lp11345
I1
aI1
aI7
aI11
aI9
aI43
aI3
aI93
aI69
aI1019
aI1935
aI3297
aI47
aI7101
aI1037
aI63473
asS'd'
I5673
sa(dp11346
S'a'
I32069
sS's'
I16
sS'm'
(lp11347
I1
aI1
aI7
aI5
aI21
aI9
aI97
aI105
aI405
aI893
aI1673
aI3783
aI2965
aI7329
aI4549
aI25433
asS'd'
I5674
sa(dp11348
S'a'
I32115
sS's'
I16
sS'm'
(lp11349
I1
aI1
aI5
aI13
aI5
aI17
aI31
aI123
aI415
aI173
aI1333
aI2245
aI1557
aI16011
aI28321
aI4039
asS'd'
I5675
sa(dp11350
S'a'
I32128
sS's'
I16
sS'm'
(lp11351
I1
aI1
aI5
aI9
aI15
aI3
aI27
aI79
aI511
aI39
aI945
aI49
aI3231
aI9199
aI21327
aI11183
asS'd'
I5676
sa(dp11352
S'a'
I32171
sS's'
I16
sS'm'
(lp11353
I1
aI3
aI3
aI9
aI3
aI15
aI115
aI141
aI387
aI341
aI953
aI399
aI6109
aI12037
aI21079
aI26745
asS'd'
I5677
sa(dp11354
S'a'
I32173
sS's'
I16
sS'm'
(lp11355
I1
aI3
aI3
aI1
aI5
aI5
aI31
aI195
aI477
aI755
aI687
aI3811
aI805
aI679
aI20687
aI46299
asS'd'
I5678
sa(dp11356
S'a'
I32182
sS's'
I16
sS'm'
(lp11357
I1
aI1
aI7
aI15
aI1
aI31
aI67
aI159
aI205
aI141
aI1667
aI3077
aI451
aI13161
aI16211
aI6887
asS'd'
I5679
sa(dp11358
S'a'
I32191
sS's'
I16
sS'm'
(lp11359
I1
aI3
aI3
aI1
aI7
aI43
aI87
aI5
aI49
aI205
aI231
aI3957
aI2947
aI13199
aI15743
aI4681
asS'd'
I5680
sa(dp11360
S'a'
I32193
sS's'
I16
sS'm'
(lp11361
I1
aI3
aI3
aI15
aI25
aI37
aI95
aI11
aI439
aI553
aI59
aI1241
aI7407
aI13467
aI22403
aI44441
asS'd'
I5681
sa(dp11362
S'a'
I32194
sS's'
I16
sS'm'
(lp11363
I1
aI1
aI1
aI3
aI21
aI3
aI127
aI239
aI491
aI139
aI1411
aI3417
aI4247
aI6247
aI13809
aI31609
asS'd'
I5682
sa(dp11364
S'a'
I32229
sS's'
I16
sS'm'
(lp11365
I1
aI1
aI5
aI1
aI9
aI13
aI5
aI155
aI109
aI593
aI119
aI4091
aI1911
aI8301
aI4239
aI50081
asS'd'
I5683
sa(dp11366
S'a'
I32230
sS's'
I16
sS'm'
(lp11367
I1
aI3
aI5
aI13
aI27
aI3
aI99
aI225
aI253
aI169
aI801
aI3741
aI1905
aI12073
aI31831
aI17997
asS'd'
I5684
sa(dp11368
S'a'
I32248
sS's'
I16
sS'm'
(lp11369
I1
aI3
aI7
aI15
aI9
aI23
aI93
aI171
aI453
aI983
aI1657
aI1133
aI6381
aI5229
aI32303
aI17439
asS'd'
I5685
sa(dp11370
S'a'
I32263
sS's'
I16
sS'm'
(lp11371
I1
aI1
aI7
aI11
aI7
aI5
aI125
aI141
aI63
aI763
aI1293
aI1007
aI4579
aI1479
aI11977
aI59261
asS'd'
I5686
sa(dp11372
S'a'
I32264
sS's'
I16
sS'm'
(lp11373
I1
aI3
aI1
aI7
aI1
aI15
aI49
aI41
aI367
aI639
aI1933
aI401
aI2335
aI2441
aI13653
aI55555
asS'd'
I5687
sa(dp11374
S'a'
I32269
sS's'
I16
sS'm'
(lp11375
I1
aI3
aI1
aI7
aI15
aI23
aI5
aI213
aI45
aI721
aI543
aI2133
aI4525
aI9719
aI28053
aI54075
asS'd'
I5688
sa(dp11376
S'a'
I32298
sS's'
I16
sS'm'
(lp11377
I1
aI3
aI7
aI3
aI11
aI7
aI23
aI35
aI169
aI829
aI1957
aI2423
aI3583
aI4951
aI28957
aI29753
asS'd'
I5689
sa(dp11378
S'a'
I32335
sS's'
I16
sS'm'
(lp11379
I1
aI1
aI3
aI3
aI1
aI5
aI19
aI235
aI175
aI969
aI229
aI2335
aI7215
aI10195
aI7487
aI64191
asS'd'
I5690
sa(dp11380
S'a'
I32340
sS's'
I16
sS'm'
(lp11381
I1
aI1
aI7
aI3
aI27
aI1
aI73
aI49
aI445
aI863
aI69
aI3555
aI993
aI9553
aI31941
aI29901
asS'd'
I5691
sa(dp11382
S'a'
I32356
sS's'
I16
sS'm'
(lp11383
I1
aI3
aI5
aI11
aI9
aI25
aI59
aI177
aI23
aI997
aI1041
aI1135
aI3879
aI767
aI2263
aI51267
asS'd'
I5692
sa(dp11384
S'a'
I32374
sS's'
I16
sS'm'
(lp11385
I1
aI1
aI7
aI3
aI1
aI63
aI49
aI51
aI237
aI569
aI1293
aI1143
aI3125
aI16315
aI17009
aI24821
asS'd'
I5693
sa(dp11386
S'a'
I32390
sS's'
I16
sS'm'
(lp11387
I1
aI3
aI3
aI15
aI11
aI17
aI121
aI25
aI349
aI833
aI557
aI1975
aI5405
aI15189
aI31243
aI53541
asS'd'
I5694
sa(dp11388
S'a'
I32401
sS's'
I16
sS'm'
(lp11389
I1
aI3
aI7
aI9
aI11
aI15
aI39
aI15
aI75
aI87
aI55
aI2069
aI3291
aI507
aI16925
aI57751
asS'd'
I5695
sa(dp11390
S'a'
I32414
sS's'
I16
sS'm'
(lp11391
I1
aI1
aI3
aI15
aI1
aI21
aI61
aI139
aI357
aI931
aI647
aI947
aI2291
aI15557
aI6739
aI5881
asS'd'
I5696
sa(dp11392
S'a'
I32417
sS's'
I16
sS'm'
(lp11393
I1
aI3
aI1
aI9
aI1
aI47
aI73
aI59
aI115
aI497
aI733
aI1777
aI905
aI16181
aI4351
aI7345
asS'd'
I5697
sa(dp11394
S'a'
I32442
sS's'
I16
sS'm'
(lp11395
I1
aI3
aI3
aI7
aI5
aI21
aI67
aI113
aI71
aI743
aI757
aI1851
aI7899
aI10315
aI15437
aI61803
asS'd'
I5698
sa(dp11396
S'a'
I32450
sS's'
I16
sS'm'
(lp11397
I1
aI3
aI7
aI1
aI9
aI23
aI77
aI131
aI395
aI767
aI1229
aI2629
aI5731
aI11907
aI32217
aI18473
asS'd'
I5699
sa(dp11398
S'a'
I32461
sS's'
I16
sS'm'
(lp11399
I1
aI3
aI5
aI15
aI1
aI23
aI123
aI207
aI291
aI565
aI1211
aI501
aI2111
aI11381
aI5171
aI54841
asS'd'
I5700
sa(dp11400
S'a'
I32473
sS's'
I16
sS'm'
(lp11401
I1
aI1
aI1
aI15
aI21
aI13
aI3
aI175
aI405
aI109
aI1353
aI2495
aI7619
aI14971
aI28179
aI34737
asS'd'
I5701
sa(dp11402
S'a'
I32479
sS's'
I16
sS'm'
(lp11403
I1
aI3
aI5
aI3
aI17
aI25
aI53
aI71
aI229
aI729
aI1953
aI3119
aI7747
aI1551
aI23417
aI35563
asS'd'
I5702
sa(dp11404
S'a'
I32530
sS's'
I16
sS'm'
(lp11405
I1
aI1
aI7
aI7
aI11
aI31
aI81
aI43
aI149
aI537
aI1253
aI2759
aI431
aI4813
aI8375
aI46329
asS'd'
I5703
sa(dp11406
S'a'
I32536
sS's'
I16
sS'm'
(lp11407
I1
aI1
aI1
aI5
aI11
aI27
aI61
aI199
aI239
aI889
aI723
aI2353
aI5663
aI7385
aI28165
aI14675
asS'd'
I5704
sa(dp11408
S'a'
I32548
sS's'
I16
sS'm'
(lp11409
I1
aI3
aI1
aI7
aI3
aI3
aI83
aI247
aI247
aI57
aI579
aI1163
aI2615
aI4051
aI2809
aI46413
asS'd'
I5705
sa(dp11410
S'a'
I32577
sS's'
I16
sS'm'
(lp11411
I1
aI1
aI3
aI11
aI13
aI47
aI11
aI235
aI475
aI35
aI843
aI2329
aI3519
aI8899
aI14533
aI24889
asS'd'
I5706
sa(dp11412
S'a'
I32628
sS's'
I16
sS'm'
(lp11413
I1
aI3
aI1
aI1
aI7
aI31
aI15
aI101
aI327
aI499
aI471
aI1001
aI339
aI11863
aI24787
aI47191
asS'd'
I5707
sa(dp11414
S'a'
I32642
sS's'
I16
sS'm'
(lp11415
I1
aI1
aI7
aI1
aI3
aI55
aI93
aI43
aI11
aI65
aI289
aI1249
aI5325
aI13867
aI29841
aI34333
asS'd'
I5708
sa(dp11416
S'a'
I32665
sS's'
I16
sS'm'
(lp11417
I1
aI3
aI3
aI1
aI25
aI61
aI87
aI113
aI115
aI265
aI1007
aI1129
aI7633
aI6109
aI5733
aI22649
asS'd'
I5709
sa(dp11418
S'a'
I32666
sS's'
I16
sS'm'
(lp11419
I1
aI3
aI1
aI11
aI31
aI59
aI127
aI83
aI33
aI419
aI1037
aI3777
aI6383
aI2711
aI2113
aI17233
asS'd'
I5710
sa(dp11420
S'a'
I32668
sS's'
I16
sS'm'
(lp11421
I1
aI1
aI5
aI13
aI11
aI17
aI73
aI41
aI257
aI223
aI359
aI3821
aI4617
aI1943
aI11331
aI40153
asS'd'
I5711
sa(dp11422
S'a'
I32696
sS's'
I16
sS'm'
(lp11423
I1
aI1
aI1
aI1
aI9
aI25
aI43
aI179
aI17
aI1021
aI1323
aI761
aI5861
aI11547
aI26017
aI5165
asS'd'
I5712
sa(dp11424
S'a'
I32722
sS's'
I16
sS'm'
(lp11425
I1
aI3
aI5
aI3
aI29
aI21
aI53
aI111
aI213
aI717
aI1101
aI3215
aI3021
aI16343
aI23387
aI33439
asS'd'
I5713
sa(dp11426
S'a'
I32757
sS's'
I16
sS'm'
(lp11427
I1
aI3
aI5
aI13
aI29
aI11
aI21
aI89
aI107
aI111
aI1121
aI2785
aI3493
aI9873
aI13
aI40863
asS'd'
I5714
sa(dp11428
S'a'
I32758
sS's'
I16
sS'm'
(lp11429
I1
aI1
aI5
aI13
aI15
aI15
aI111
aI219
aI59
aI43
aI333
aI3581
aI1311
aI2799
aI23987
aI21637
asS'd'
I5715
sa(dp11430
S'a'
I4
sS's'
I17
sS'm'
(lp11431
I1
aI3
aI1
aI11
aI21
aI57
aI115
aI247
aI499
aI525
aI1629
aI3679
aI2109
aI6607
aI27435
aI1745
aI71201
asS'd'
I5716
sa(dp11432
S'a'
I7
sS's'
I17
sS'm'
(lp11433
I1
aI3
aI3
aI3
aI31
aI17
aI113
aI165
aI189
aI361
aI103
aI1775
aI3001
aI3865
aI30591
aI2873
aI17129
asS'd'
I5717
sa(dp11434
S'a'
I16
sS's'
I17
sS'm'
(lp11435
I1
aI1
aI5
aI5
aI15
aI47
aI47
aI85
aI247
aI471
aI713
aI3571
aI2407
aI9811
aI8187
aI32133
aI8541
asS'd'
I5718
sa(dp11436
S'a'
I22
sS's'
I17
sS'm'
(lp11437
I1
aI3
aI3
aI1
aI15
aI1
aI59
aI151
aI469
aI351
aI671
aI2925
aI7207
aI5061
aI28691
aI4363
aI50767
asS'd'
I5719
sa(dp11438
S'a'
I25
sS's'
I17
sS'm'
(lp11439
I1
aI1
aI5
aI7
aI11
aI35
aI67
aI45
aI193
aI3
aI627
aI3333
aI6497
aI12307
aI28807
aI13997
aI108645
asS'd'
I5720
sa(dp11440
S'a'
I31
sS's'
I17
sS'm'
(lp11441
I1
aI3
aI1
aI1
aI17
aI63
aI125
aI185
aI485
aI759
aI717
aI1993
aI6707
aI3993
aI2181
aI8173
aI18057
asS'd'
I5721
sa(dp11442
S'a'
I32
sS's'
I17
sS'm'
(lp11443
I1
aI1
aI3
aI13
aI7
aI15
aI113
aI207
aI103
aI191
aI1895
aI2595
aI3873
aI12021
aI19259
aI12553
aI119119
asS'd'
I5722
sa(dp11444
S'a'
I42
sS's'
I17
sS'm'
(lp11445
I1
aI3
aI7
aI1
aI17
aI11
aI101
aI209
aI315
aI9
aI901
aI2303
aI7623
aI7459
aI26391
aI45143
aI5753
asS'd'
I5723
sa(dp11446
S'a'
I52
sS's'
I17
sS'm'
(lp11447
I1
aI1
aI5
aI15
aI1
aI5
aI71
aI155
aI167
aI89
aI145
aI3483
aI2385
aI15205
aI9193
aI20637
aI58473
asS'd'
I5724
sa(dp11448
S'a'
I61
sS's'
I17
sS'm'
(lp11449
I1
aI1
aI5
aI7
aI25
aI55
aI57
aI51
aI333
aI299
aI1721
aI1667
aI6513
aI10191
aI29405
aI21923
aI76593
asS'd'
I5725
sa(dp11450
S'a'
I70
sS's'
I17
sS'm'
(lp11451
I1
aI1
aI5
aI1
aI7
aI37
aI107
aI91
aI241
aI137
aI627
aI2749
aI5573
aI11243
aI26197
aI4545
aI105599
asS'd'
I5726
sa(dp11452
S'a'
I76
sS's'
I17
sS'm'
(lp11453
I1
aI3
aI1
aI5
aI25
aI37
aI73
aI61
aI57
aI249
aI1953
aI1385
aI6479
aI3701
aI10693
aI617
aI62535
asS'd'
I5727
sa(dp11454
S'a'
I81
sS's'
I17
sS'm'
(lp11455
I1
aI1
aI1
aI15
aI5
aI63
aI41
aI151
aI395
aI681
aI227
aI3027
aI8123
aI15091
aI15475
aI35671
aI21129
asS'd'
I5728
sa(dp11456
S'a'
I87
sS's'
I17
sS'm'
(lp11457
I1
aI3
aI5
aI11
aI29
aI21
aI15
aI233
aI103
aI463
aI1829
aI2257
aI1717
aI2249
aI9599
aI5097
aI55705
asS'd'
I5729
sa(dp11458
S'a'
I93
sS's'
I17
sS'm'
(lp11459
I1
aI3
aI5
aI1
aI29
aI3
aI35
aI151
aI193
aI105
aI1107
aI827
aI7169
aI1843
aI15225
aI29025
aI43165
asS'd'
I5730
sa(dp11460
S'a'
I98
sS's'
I17
sS'm'
(lp11461
I1
aI1
aI7
aI15
aI17
aI51
aI93
aI199
aI205
aI41
aI113
aI1081
aI1571
aI11471
aI11057
aI16149
aI66905
asS'd'
I5731
sa(dp11462
S'a'
I122
sS's'
I17
sS'm'
(lp11463
I1
aI1
aI3
aI11
aI5
aI25
aI107
aI195
aI51
aI675
aI1683
aI3739
aI1653
aI611
aI23249
aI53157
aI127785
asS'd'
I5732
sa(dp11464
S'a'
I133
sS's'
I17
sS'm'
(lp11465
I1
aI1
aI7
aI5
aI7
aI3
aI25
aI145
aI453
aI735
aI441
aI77
aI8171
aI9281
aI22749
aI36973
aI106237
asS'd'
I5733
sa(dp11466
S'a'
I134
sS's'
I17
sS'm'
(lp11467
I1
aI1
aI3
aI13
aI13
aI5
aI95
aI33
aI223
aI369
aI453
aI2031
aI3531
aI6931
aI8977
aI54109
aI115487
asS'd'
I5734
sa(dp11468
S'a'
I140
sS's'
I17
sS'm'
(lp11469
I1
aI1
aI7
aI7
aI1
aI61
aI33
aI183
aI245
aI623
aI529
aI1831
aI1867
aI2845
aI8311
aI10143
aI67897
asS'd'
I5735
sa(dp11470
S'a'
I146
sS's'
I17
sS'm'
(lp11471
I1
aI3
aI7
aI11
aI27
aI23
aI93
aI9
aI61
aI451
aI67
aI1695
aI4227
aI2415
aI19249
aI44765
aI24611
asS'd'
I5736
sa(dp11472
S'a'
I158
sS's'
I17
sS'm'
(lp11473
I1
aI3
aI3
aI11
aI29
aI57
aI65
aI117
aI349
aI149
aI363
aI1095
aI4989
aI3071
aI17519
aI18079
aI7277
asS'd'
I5737
sa(dp11474
S'a'
I171
sS's'
I17
sS'm'
(lp11475
I1
aI3
aI5
aI9
aI1
aI7
aI59
aI87
aI307
aI111
aI1291
aI789
aI7361
aI6477
aI11229
aI36785
aI33303
asS'd'
I5738
sa(dp11476
S'a'
I176
sS's'
I17
sS'm'
(lp11477
I1
aI3
aI5
aI1
aI19
aI47
aI53
aI81
aI127
aI849
aI1479
aI1459
aI1889
aI15087
aI22115
aI20587
aI121005
asS'd'
I5739
sa(dp11478
S'a'
I179
sS's'
I17
sS'm'
(lp11479
I1
aI1
aI7
aI15
aI31
aI31
aI71
aI55
aI253
aI927
aI277
aI2087
aI1313
aI3721
aI22729
aI34709
aI9821
asS'd'
I5740
sa(dp11480
S'a'
I182
sS's'
I17
sS'm'
(lp11481
I1
aI3
aI5
aI13
aI13
aI63
aI73
aI41
aI165
aI315
aI1907
aI2005
aI691
aI725
aI22685
aI8673
aI76011
asS'd'
I5741
sa(dp11482
S'a'
I191
sS's'
I17
sS'm'
(lp11483
I1
aI1
aI5
aI9
aI23
aI61
aI47
aI167
aI279
aI683
aI683
aI1261
aI4037
aI15251
aI9421
aI45359
aI38001
asS'd'
I5742
sa(dp11484
S'a'
I193
sS's'
I17
sS'm'
(lp11485
I1
aI1
aI7
aI3
aI17
aI33
aI69
aI139
aI235
aI709
aI1475
aI2483
aI7559
aI8581
aI23965
aI31153
aI5097
asS'd'
I5743
sa(dp11486
S'a'
I224
sS's'
I17
sS'm'
(lp11487
I1
aI1
aI7
aI15
aI23
aI61
aI43
aI5
aI433
aI531
aI761
aI2749
aI2881
aI5225
aI13491
aI16479
aI50203
asS'd'
I5744
sa(dp11488
S'a'
I227
sS's'
I17
sS'm'
(lp11489
I1
aI1
aI3
aI9
aI29
aI7
aI9
aI23
aI339
aI315
aI1723
aI779
aI2983
aI6571
aI16025
aI63055
aI111103
asS'd'
I5745
sa(dp11490
S'a'
I229
sS's'
I17
sS'm'
(lp11491
I1
aI1
aI7
aI13
aI23
aI55
aI71
aI121
aI297
aI193
aI41
aI3165
aI4419
aI5853
aI28127
aI56151
aI16597
asS'd'
I5746
sa(dp11492
S'a'
I236
sS's'
I17
sS'm'
(lp11493
I1
aI1
aI5
aI7
aI7
aI23
aI93
aI11
aI261
aI297
aI1769
aI1239
aI2579
aI531
aI4423
aI7891
aI21729
asS'd'
I5747
sa(dp11494
S'a'
I248
sS's'
I17
sS'm'
(lp11495
I1
aI3
aI5
aI1
aI13
aI35
aI83
aI85
aI125
aI887
aI161
aI3311
aI7261
aI9557
aI28975
aI28643
aI21479
asS'd'
I5748
sa(dp11496
S'a'
I262
sS's'
I17
sS'm'
(lp11497
I1
aI3
aI5
aI3
aI27
aI5
aI47
aI175
aI287
aI867
aI141
aI3079
aI7583
aI4997
aI18271
aI24097
aI96319
asS'd'
I5749
sa(dp11498
S'a'
I273
sS's'
I17
sS'm'
(lp11499
I1
aI3
aI5
aI1
aI21
aI51
aI47
aI67
aI211
aI281
aI1861
aI1169
aI6403
aI4229
aI3995
aI9921
aI41515
asS'd'
I5750
sa(dp11500
S'a'
I276
sS's'
I17
sS'm'
(lp11501
I1
aI3
aI3
aI11
aI23
aI23
aI81
aI55
aI441
aI211
aI169
aI3197
aI7213
aI7205
aI15
aI11771
aI129091
asS'd'
I5751
sa(dp11502
S'a'
I280
sS's'
I17
sS'm'
(lp11503
I1
aI3
aI7
aI3
aI23
aI39
aI23
aI163
aI253
aI1005
aI1775
aI3393
aI7659
aI8065
aI30021
aI61065
aI35171
asS'd'
I5752
sa(dp11504
S'a'
I283
sS's'
I17
sS'm'
(lp11505
I1
aI3
aI1
aI1
aI29
aI29
aI39
aI143
aI191
aI711
aI1077
aI13
aI4137
aI15425
aI11139
aI1269
aI71915
asS'd'
I5753
sa(dp11506
S'a'
I290
sS's'
I17
sS'm'
(lp11507
I1
aI3
aI3
aI5
aI11
aI41
aI101
aI127
aI301
aI335
aI45
aI2065
aI5835
aI7801
aI2639
aI5735
aI63445
asS'd'
I5754
sa(dp11508
S'a'
I309
sS's'
I17
sS'm'
(lp11509
I1
aI3
aI5
aI9
aI3
aI39
aI51
aI53
aI489
aI663
aI951
aI3931
aI3075
aI753
aI22179
aI20573
aI10775
asS'd'
I5755
sa(dp11510
S'a'
I316
sS's'
I17
sS'm'
(lp11511
I1
aI3
aI3
aI15
aI13
aI31
aI1
aI237
aI79
aI587
aI395
aI591
aI607
aI13105
aI21301
aI26829
aI112181
asS'd'
I5756
sa(dp11512
S'a'
I319
sS's'
I17
sS'm'
(lp11513
I1
aI1
aI7
aI7
aI5
aI55
aI31
aI117
aI247
aI229
aI247
aI307
aI3821
aI6483
aI31317
aI22975
aI40535
asS'd'
I5757
sa(dp11514
S'a'
I321
sS's'
I17
sS'm'
(lp11515
I1
aI3
aI7
aI15
aI15
aI59
aI101
aI17
aI437
aI373
aI1727
aI471
aI2783
aI7825
aI24555
aI58765
aI5097
asS'd'
I5758
sa(dp11516
S'a'
I328
sS's'
I17
sS'm'
(lp11517
I1
aI1
aI3
aI9
aI31
aI27
aI71
aI147
aI71
aI871
aI793
aI2363
aI3213
aI13383
aI29801
aI53187
aI70021
asS'd'
I5759
sa(dp11518
S'a'
I346
sS's'
I17
sS'm'
(lp11519
I1
aI3
aI1
aI1
aI19
aI47
aI121
aI61
aI303
aI565
aI1371
aI3703
aI2201
aI6835
aI26041
aI56039
aI80227
asS'd'
I5760
sa(dp11520
S'a'
I355
sS's'
I17
sS'm'
(lp11521
I1
aI1
aI5
aI5
aI3
aI45
aI91
aI61
aI257
aI947
aI1449
aI4031
aI4925
aI8627
aI11909
aI9529
aI3429
asS'd'
I5761
sa(dp11522
S'a'
I367
sS's'
I17
sS'm'
(lp11523
I1
aI1
aI1
aI7
aI9
aI63
aI69
aI233
aI141
aI361
aI1443
aI2157
aI2877
aI643
aI2779
aI8109
aI126911
asS'd'
I5762
sa(dp11524
S'a'
I369
sS's'
I17
sS'm'
(lp11525
I1
aI1
aI5
aI1
aI5
aI3
aI67
aI157
aI21
aI1
aI361
aI35
aI1475
aI12877
aI22169
aI6653
aI85005
asS'd'
I5763
sa(dp11526
S'a'
I372
sS's'
I17
sS'm'
(lp11527
I1
aI1
aI7
aI9
aI25
aI1
aI7
aI175
aI47
aI963
aI405
aI3955
aI3905
aI8429
aI8483
aI62037
aI11323
asS'd'
I5764
sa(dp11528
S'a'
I382
sS's'
I17
sS'm'
(lp11529
I1
aI1
aI5
aI11
aI29
aI23
aI77
aI211
aI319
aI745
aI1935
aI2429
aI1687
aI2173
aI1571
aI19457
aI117777
asS'd'
I5765
sa(dp11530
S'a'
I388
sS's'
I17
sS'm'
(lp11531
I1
aI1
aI7
aI5
aI15
aI57
aI121
aI189
aI303
aI79
aI527
aI1801
aI71
aI9857
aI14197
aI59007
aI75341
asS'd'
I5766
sa(dp11532
S'a'
I392
sS's'
I17
sS'm'
(lp11533
I1
aI3
aI3
aI5
aI25
aI3
aI19
aI141
aI155
aI157
aI287
aI769
aI5789
aI8443
aI31823
aI1019
aI79111
asS'd'
I5767
sa(dp11534
S'a'
I395
sS's'
I17
sS'm'
(lp11535
I1
aI1
aI5
aI11
aI27
aI27
aI117
aI141
aI355
aI1023
aI869
aI995
aI6311
aI6573
aI11721
aI1565
aI35517
asS'd'
I5768
sa(dp11536
S'a'
I397
sS's'
I17
sS'm'
(lp11537
I1
aI1
aI1
aI9
aI1
aI33
aI107
aI51
aI41
aI889
aI1191
aI1055
aI503
aI14779
aI6641
aI58117
aI74157
asS'd'
I5769
sa(dp11538
S'a'
I403
sS's'
I17
sS'm'
(lp11539
I1
aI1
aI7
aI5
aI13
aI39
aI39
aI33
aI293
aI75
aI963
aI3379
aI1847
aI12371
aI9005
aI38107
aI69753
asS'd'
I5770
sa(dp11540
S'a'
I409
sS's'
I17
sS'm'
(lp11541
I1
aI1
aI5
aI5
aI7
aI37
aI19
aI241
aI427
aI635
aI1711
aI3835
aI773
aI10525
aI17207
aI1675
aI127255
asS'd'
I5771
sa(dp11542
S'a'
I410
sS's'
I17
sS'm'
(lp11543
I1
aI1
aI3
aI7
aI17
aI19
aI11
aI113
aI191
aI947
aI1133
aI3173
aI213
aI10125
aI1373
aI56797
aI111011
asS'd'
I5772
sa(dp11544
S'a'
I425
sS's'
I17
sS'm'
(lp11545
I1
aI3
aI1
aI1
aI29
aI45
aI65
aI237
aI223
aI695
aI697
aI3197
aI6887
aI8079
aI22099
aI12079
aI54847
asS'd'
I5773
sa(dp11546
S'a'
I443
sS's'
I17
sS'm'
(lp11547
I1
aI3
aI3
aI7
aI5
aI47
aI19
aI215
aI341
aI863
aI1879
aI571
aI7113
aI2465
aI23407
aI52555
aI44375
asS'd'
I5774
sa(dp11548
S'a'
I472
sS's'
I17
sS'm'
(lp11549
I1
aI3
aI5
aI11
aI25
aI31
aI109
aI73
aI429
aI553
aI1905
aI1753
aI6733
aI4433
aI13785
aI32041
aI27115
asS'd'
I5775
sa(dp11550
S'a'
I475
sS's'
I17
sS'm'
(lp11551
I1
aI1
aI1
aI3
aI27
aI5
aI97
aI47
aI343
aI977
aI1241
aI721
aI3355
aI3559
aI28349
aI56389
aI63103
asS'd'
I5776
sa(dp11552
S'a'
I481
sS's'
I17
sS'm'
(lp11553
I1
aI3
aI3
aI9
aI21
aI53
aI57
aI211
aI73
aI155
aI1855
aI715
aI3179
aI5963
aI10061
aI35141
aI63131
asS'd'
I5777
sa(dp11554
S'a'
I488
sS's'
I17
sS'm'
(lp11555
I1
aI3
aI1
aI15
aI21
aI25
aI51
aI73
aI31
aI25
aI1385
aI637
aI6585
aI49
aI2105
aI6829
aI9353
asS'd'
I5778
sa(dp11556
S'a'
I493
sS's'
I17
sS'm'
(lp11557
I1
aI1
aI7
aI5
aI11
aI55
aI31
aI69
aI145
aI637
aI1131
aI2175
aI3547
aI13031
aI2131
aI12361
aI74737
asS'd'
I5779
sa(dp11558
S'a'
I501
sS's'
I17
sS'm'
(lp11559
I1
aI3
aI3
aI5
aI31
aI7
aI119
aI119
aI309
aI925
aI895
aI3813
aI1131
aI4765
aI17865
aI48707
aI113577
asS'd'
I5780
sa(dp11560
S'a'
I515
sS's'
I17
sS'm'
(lp11561
I1
aI3
aI3
aI9
aI13
aI33
aI127
aI177
aI323
aI727
aI1881
aI775
aI7329
aI11881
aI28309
aI987
aI116093
asS'd'
I5781
sa(dp11562
S'a'
I522
sS's'
I17
sS'm'
(lp11563
I1
aI1
aI3
aI5
aI31
aI55
aI39
aI41
aI511
aI157
aI1655
aI2991
aI3633
aI8521
aI27049
aI18771
aI54015
asS'd'
I5782
sa(dp11564
S'a'
I524
sS's'
I17
sS'm'
(lp11565
I1
aI3
aI5
aI13
aI11
aI45
aI113
aI185
aI375
aI661
aI1331
aI4013
aI5521
aI1037
aI23365
aI30239
aI76957
asS'd'
I5783
sa(dp11566
S'a'
I527
sS's'
I17
sS'm'
(lp11567
I1
aI3
aI3
aI7
aI19
aI7
aI23
aI17
aI435
aI913
aI1985
aI353
aI6049
aI7549
aI3371
aI60867
aI41099
asS'd'
I5784
sa(dp11568
S'a'
I535
sS's'
I17
sS'm'
(lp11569
I1
aI3
aI3
aI15
aI17
aI9
aI53
aI127
aI149
aI849
aI1181
aI2237
aI1345
aI539
aI19715
aI26277
aI125445
asS'd'
I5785
sa(dp11570
S'a'
I542
sS's'
I17
sS'm'
(lp11571
I1
aI1
aI1
aI3
aI1
aI9
aI67
aI79
aI79
aI795
aI1793
aI3167
aI5917
aI5323
aI22043
aI22007
aI3917
asS'd'
I5786
sa(dp11572
S'a'
I545
sS's'
I17
sS'm'
(lp11573
I1
aI3
aI5
aI9
aI15
aI19
aI59
aI37
aI141
aI145
aI413
aI1095
aI7709
aI669
aI27061
aI40171
aI101499
asS'd'
I5787
sa(dp11574
S'a'
I555
sS's'
I17
sS'm'
(lp11575
I1
aI3
aI1
aI1
aI9
aI49
aI109
aI7
aI119
aI861
aI875
aI1049
aI4125
aI6113
aI15699
aI6105
aI48799
asS'd'
I5788
sa(dp11576
S'a'
I558
sS's'
I17
sS'm'
(lp11577
I1
aI1
aI3
aI9
aI11
aI29
aI43
aI175
aI371
aI357
aI1181
aI3933
aI43
aI4559
aI10333
aI23603
aI83095
asS'd'
I5789
sa(dp11578
S'a'
I560
sS's'
I17
sS'm'
(lp11579
I1
aI3
aI3
aI9
aI9
aI7
aI57
aI61
aI409
aI143
aI591
aI761
aI4107
aI8117
aI1051
aI4471
aI91771
asS'd'
I5790
sa(dp11580
S'a'
I563
sS's'
I17
sS'm'
(lp11581
I1
aI1
aI3
aI11
aI3
aI53
aI119
aI21
aI213
aI219
aI51
aI3491
aI7143
aI937
aI24693
aI3211
aI99463
asS'd'
I5791
sa(dp11582
S'a'
I570
sS's'
I17
sS'm'
(lp11583
I1
aI1
aI3
aI3
aI1
aI47
aI53
aI153
aI211
aI523
aI1637
aI3351
aI3753
aI12489
aI31825
aI27613
aI96431
asS'd'
I5792
sa(dp11584
S'a'
I578
sS's'
I17
sS'm'
(lp11585
I1
aI1
aI5
aI15
aI23
aI57
aI81
aI231
aI147
aI9
aI1043
aI3157
aI1463
aI4835
aI22435
aI57407
aI59615
asS'd'
I5793
sa(dp11586
S'a'
I583
sS's'
I17
sS'm'
(lp11587
I1
aI3
aI3
aI13
aI15
aI63
aI111
aI5
aI449
aI957
aI1175
aI2887
aI7741
aI8975
aI28775
aI4067
aI69283
asS'd'
I5794
sa(dp11588
S'a'
I590
sS's'
I17
sS'm'
(lp11589
I1
aI3
aI1
aI1
aI5
aI61
aI109
aI211
aI349
aI179
aI951
aI153
aI3147
aI7555
aI27037
aI59829
aI16077
asS'd'
I5795
sa(dp11590
S'a'
I597
sS's'
I17
sS'm'
(lp11591
I1
aI3
aI3
aI7
aI15
aI33
aI53
aI61
aI309
aI991
aI227
aI3437
aI3983
aI14559
aI13065
aI46387
aI49105
asS'd'
I5796
sa(dp11592
S'a'
I604
sS's'
I17
sS'm'
(lp11593
I1
aI3
aI5
aI3
aI25
aI23
aI97
aI139
aI315
aI601
aI1179
aI1083
aI6799
aI1813
aI15511
aI60433
aI65641
asS'd'
I5797
sa(dp11594
S'a'
I608
sS's'
I17
sS'm'
(lp11595
I1
aI1
aI7
aI1
aI11
aI43
aI87
aI87
aI173
aI161
aI91
aI3011
aI1869
aI2313
aI13691
aI3509
aI39433
asS'd'
I5798
sa(dp11596
S'a'
I614
sS's'
I17
sS'm'
(lp11597
I1
aI3
aI5
aI7
aI15
aI5
aI39
aI251
aI269
aI819
aI815
aI2283
aI5635
aI6953
aI27017
aI65143
aI45281
asS'd'
I5799
sa(dp11598
S'a'
I635
sS's'
I17
sS'm'
(lp11599
I1
aI3
aI7
aI9
aI1
aI37
aI9
aI57
aI467
aI37
aI1743
aI4031
aI3751
aI8105
aI23789
aI46847
aI21911
asS'd'
I5800
sa(dp11600
S'a'
I637
sS's'
I17
sS'm'
(lp11601
I1
aI1
aI7
aI1
aI23
aI47
aI63
aI99
aI59
aI951
aI1837
aI2829
aI161
aI857
aI4045
aI9945
aI53487
asS'd'
I5801
sa(dp11602
S'a'
I653
sS's'
I17
sS'm'
(lp11603
I1
aI3
aI7
aI7
aI11
aI47
aI43
aI99
aI279
aI945
aI1189
aI2091
aI4597
aI183
aI15527
aI7151
aI112403
asS'd'
I5802
sa(dp11604
S'a'
I654
sS's'
I17
sS'm'
(lp11605
I1
aI3
aI3
aI15
aI9
aI53
aI63
aI135
aI119
aI95
aI131
aI2461
aI157
aI10631
aI20847
aI51699
aI58865
asS'd'
I5803
sa(dp11606
S'a'
I659
sS's'
I17
sS'm'
(lp11607
I1
aI1
aI3
aI1
aI25
aI3
aI115
aI29
aI303
aI361
aI1529
aI3993
aI5899
aI11501
aI4463
aI47121
aI75333
asS'd'
I5804
sa(dp11608
S'a'
I666
sS's'
I17
sS'm'
(lp11609
I1
aI3
aI1
aI15
aI9
aI39
aI31
aI199
aI305
aI279
aI15
aI611
aI561
aI6593
aI3189
aI1863
aI61875
asS'd'
I5805
sa(dp11610
S'a'
I671
sS's'
I17
sS'm'
(lp11611
I1
aI3
aI5
aI15
aI5
aI49
aI87
aI17
aI87
aI5
aI1179
aI1351
aI7647
aI7529
aI15901
aI30351
aI31959
asS'd'
I5806
sa(dp11612
S'a'
I689
sS's'
I17
sS'm'
(lp11613
I1
aI3
aI3
aI9
aI31
aI57
aI127
aI239
aI349
aI773
aI547
aI2649
aI1309
aI8071
aI10741
aI57645
aI14423
asS'd'
I5807
sa(dp11614
S'a'
I690
sS's'
I17
sS'm'
(lp11615
I1
aI1
aI5
aI9
aI5
aI15
aI59
aI185
aI315
aI411
aI1425
aI3905
aI853
aI12393
aI21
aI15195
aI114291
asS'd'
I5808
sa(dp11616
S'a'
I695
sS's'
I17
sS'm'
(lp11617
I1
aI3
aI1
aI5
aI29
aI47
aI19
aI203
aI319
aI673
aI1169
aI2413
aI5295
aI6251
aI19883
aI2725
aI28937
asS'd'
I5809
sa(dp11618
S'a'
I713
sS's'
I17
sS'm'
(lp11619
I1
aI3
aI1
aI5
aI21
aI55
aI19
aI185
aI103
aI827
aI117
aI341
aI3315
aI5625
aI345
aI63845
aI49081
asS'd'
I5810
sa(dp11620
S'a'
I722
sS's'
I17
sS'm'
(lp11621
I1
aI1
aI7
aI9
aI27
aI51
aI105
aI15
aI243
aI735
aI1221
aI1641
aI293
aI14423
aI5363
aI60873
aI66223
asS'd'
I5811
sa(dp11622
S'a'
I733
sS's'
I17
sS'm'
(lp11623
I1
aI1
aI5
aI1
aI19
aI5
aI109
aI131
aI131
aI67
aI231
aI2907
aI4389
aI5079
aI20503
aI59045
aI33625
asS'd'
I5812
sa(dp11624
S'a'
I758
sS's'
I17
sS'm'
(lp11625
I1
aI3
aI1
aI5
aI5
aI15
aI79
aI67
aI287
aI225
aI519
aI1543
aI2389
aI671
aI7767
aI62625
aI61639
asS'd'
I5813
sa(dp11626
S'a'
I770
sS's'
I17
sS'm'
(lp11627
I1
aI1
aI1
aI9
aI25
aI35
aI83
aI15
aI291
aI207
aI1757
aI3691
aI5669
aI11255
aI27939
aI57813
aI46251
asS'd'
I5814
sa(dp11628
S'a'
I782
sS's'
I17
sS'm'
(lp11629
I1
aI3
aI1
aI1
aI29
aI3
aI83
aI109
aI323
aI179
aI1855
aI3205
aI7665
aI16201
aI13863
aI16347
aI98977
asS'd'
I5815
sa(dp11630
S'a'
I784
sS's'
I17
sS'm'
(lp11631
I1
aI3
aI1
aI13
aI17
aI1
aI101
aI183
aI153
aI985
aI125
aI999
aI855
aI15897
aI19491
aI8953
aI23277
asS'd'
I5816
sa(dp11632
S'a'
I793
sS's'
I17
sS'm'
(lp11633
I1
aI1
aI7
aI11
aI9
aI33
aI45
aI229
aI411
aI155
aI537
aI3037
aI1785
aI11719
aI8589
aI16617
aI47339
asS'd'
I5817
sa(dp11634
S'a'
I803
sS's'
I17
sS'm'
(lp11635
I1
aI1
aI5
aI5
aI9
aI11
aI7
aI163
aI305
aI621
aI1647
aI2609
aI7901
aI14421
aI23447
aI1205
aI52681
asS'd'
I5818
sa(dp11636
S'a'
I805
sS's'
I17
sS'm'
(lp11637
I1
aI3
aI3
aI1
aI7
aI29
aI39
aI227
aI419
aI561
aI129
aI3299
aI3123
aI4243
aI18689
aI12335
aI71783
asS'd'
I5819
sa(dp11638
S'a'
I812
sS's'
I17
sS'm'
(lp11639
I1
aI3
aI1
aI9
aI11
aI61
aI65
aI207
aI123
aI763
aI485
aI1943
aI3617
aI415
aI22397
aI58597
aI128017
asS'd'
I5820
sa(dp11640
S'a'
I838
sS's'
I17
sS'm'
(lp11641
I1
aI1
aI5
aI13
aI25
aI43
aI115
aI73
aI269
aI137
aI1765
aI705
aI1705
aI16137
aI22751
aI60021
aI4333
asS'd'
I5821
sa(dp11642
S'a'
I849
sS's'
I17
sS'm'
(lp11643
I1
aI1
aI5
aI13
aI3
aI57
aI9
aI141
aI75
aI695
aI597
aI3435
aI1085
aI4905
aI19625
aI16061
aI12111
asS'd'
I5822
sa(dp11644
S'a'
I875
sS's'
I17
sS'm'
(lp11645
I1
aI1
aI5
aI9
aI29
aI13
aI119
aI251
aI353
aI421
aI1955
aI3503
aI2605
aI2587
aI12503
aI46419
aI128815
asS'd'
I5823
sa(dp11646
S'a'
I877
sS's'
I17
sS'm'
(lp11647
I1
aI3
aI5
aI7
aI7
aI29
aI67
aI25
aI37
aI327
aI1607
aI1899
aI1691
aI5801
aI17441
aI9755
aI24993
asS'd'
I5824
sa(dp11648
S'a'
I880
sS's'
I17
sS'm'
(lp11649
I1
aI1
aI3
aI11
aI17
aI29
aI121
aI201
aI371
aI597
aI213
aI2361
aI6615
aI169
aI24801
aI56175
aI129241
asS'd'
I5825
sa(dp11650
S'a'
I892
sS's'
I17
sS'm'
(lp11651
I1
aI3
aI5
aI1
aI31
aI63
aI85
aI77
aI151
aI599
aI103
aI677
aI4431
aI12897
aI6373
aI40349
aI100819
asS'd'
I5826
sa(dp11652
S'a'
I895
sS's'
I17
sS'm'
(lp11653
I1
aI3
aI5
aI9
aI25
aI9
aI119
aI219
aI379
aI939
aI1907
aI945
aI5819
aI7433
aI32519
aI56493
aI50441
asS'd'
I5827
sa(dp11654
S'a'
I899
sS's'
I17
sS'm'
(lp11655
I1
aI1
aI3
aI9
aI13
aI1
aI63
aI189
aI135
aI839
aI1821
aI2247
aI2547
aI965
aI6847
aI63335
aI32921
asS'd'
I5828
sa(dp11656
S'a'
I919
sS's'
I17
sS'm'
(lp11657
I1
aI3
aI5
aI13
aI21
aI25
aI111
aI37
aI319
aI469
aI1999
aI1637
aI8167
aI2641
aI24615
aI63713
aI115923
asS'd'
I5829
sa(dp11658
S'a'
I920
sS's'
I17
sS'm'
(lp11659
I1
aI3
aI5
aI9
aI9
aI27
aI1
aI63
aI275
aI223
aI1675
aI3833
aI7377
aI9755
aI6279
aI37161
aI108805
asS'd'
I5830
sa(dp11660
S'a'
I932
sS's'
I17
sS'm'
(lp11661
I1
aI3
aI3
aI13
aI29
aI23
aI21
aI73
aI401
aI863
aI701
aI2527
aI4557
aI5549
aI22493
aI6651
aI39167
asS'd'
I5831
sa(dp11662
S'a'
I935
sS's'
I17
sS'm'
(lp11663
I1
aI1
aI3
aI15
aI25
aI21
aI97
aI25
aI83
aI925
aI2029
aI3789
aI3241
aI7617
aI13699
aI31123
aI124619
asS'd'
I5832
sa(dp11664
S'a'
I936
sS's'
I17
sS'm'
(lp11665
I1
aI3
aI7
aI5
aI23
aI7
aI95
aI227
aI123
aI215
aI359
aI2099
aI4505
aI8477
aI32665
aI18211
aI99679
asS'd'
I5833
sa(dp11666
S'a'
I941
sS's'
I17
sS'm'
(lp11667
I1
aI3
aI1
aI9
aI11
aI57
aI75
aI17
aI105
aI175
aI831
aI1033
aI5425
aI8419
aI16163
aI23901
aI33889
asS'd'
I5834
sa(dp11668
S'a'
I950
sS's'
I17
sS'm'
(lp11669
I1
aI1
aI7
aI1
aI17
aI49
aI71
aI23
aI129
aI413
aI333
aI2547
aI4627
aI14961
aI16745
aI53649
aI73059
asS'd'
I5835
sa(dp11670
S'a'
I961
sS's'
I17
sS'm'
(lp11671
I1
aI3
aI5
aI3
aI13
aI33
aI121
aI147
aI443
aI187
aI1949
aI319
aI8141
aI14359
aI11203
aI53569
aI70415
asS'd'
I5836
sa(dp11672
S'a'
I962
sS's'
I17
sS'm'
(lp11673
I1
aI3
aI1
aI11
aI15
aI1
aI23
aI29
aI509
aI985
aI1217
aI3755
aI385
aI3697
aI24631
aI37619
aI62435
asS'd'
I5837
sa(dp11674
S'a'
I971
sS's'
I17
sS'm'
(lp11675
I1
aI3
aI3
aI3
aI17
aI11
aI107
aI37
aI227
aI913
aI259
aI2799
aI3249
aI2347
aI9703
aI52741
aI101187
asS'd'
I5838
sa(dp11676
S'a'
I982
sS's'
I17
sS'm'
(lp11677
I1
aI1
aI5
aI13
aI25
aI25
aI47
aI77
aI405
aI415
aI1947
aI1675
aI5079
aI1333
aI10059
aI32033
aI88975
asS'd'
I5839
sa(dp11678
S'a'
I986
sS's'
I17
sS'm'
(lp11679
I1
aI3
aI5
aI9
aI27
aI7
aI19
aI241
aI445
aI205
aI333
aI285
aI7997
aI6339
aI29643
aI10229
aI29965
asS'd'
I5840
sa(dp11680
S'a'
I1012
sS's'
I17
sS'm'
(lp11681
I1
aI3
aI5
aI11
aI17
aI9
aI91
aI223
aI173
aI1013
aI779
aI3967
aI781
aI5471
aI4309
aI24795
aI99203
asS'd'
I5841
sa(dp11682
S'a'
I1021
sS's'
I17
sS'm'
(lp11683
I1
aI1
aI1
aI3
aI19
aI53
aI7
aI159
aI351
aI515
aI223
aI3375
aI1
aI4985
aI16729
aI43333
aI85917
asS'd'
I5842
sa(dp11684
S'a'
I1024
sS's'
I17
sS'm'
(lp11685
I1
aI3
aI3
aI1
aI19
aI35
aI95
aI69
aI19
aI157
aI1177
aI579
aI7109
aI3499
aI3219
aI26641
aI49491
asS'd'
I5843
sa(dp11686
S'a'
I1029
sS's'
I17
sS'm'
(lp11687
I1
aI3
aI3
aI5
aI25
aI21
aI125
aI5
aI39
aI857
aI615
aI2925
aI2005
aI5503
aI25523
aI36711
aI30939
asS'd'
I5844
sa(dp11688
S'a'
I1030
sS's'
I17
sS'm'
(lp11689
I1
aI3
aI1
aI5
aI11
aI33
aI29
aI5
aI425
aI125
aI939
aI1641
aI321
aI1023
aI12551
aI4587
aI116617
asS'd'
I5845
sa(dp11690
S'a'
I1051
sS's'
I17
sS'm'
(lp11691
I1
aI3
aI3
aI13
aI9
aI59
aI93
aI137
aI103
aI517
aI1555
aI13
aI7965
aI13629
aI14339
aI37425
aI65891
asS'd'
I5846
sa(dp11692
S'a'
I1054
sS's'
I17
sS'm'
(lp11693
I1
aI3
aI7
aI1
aI31
aI31
aI87
aI237
aI365
aI951
aI267
aI2019
aI5085
aI6133
aI29371
aI50319
aI94313
asS'd'
I5847
sa(dp11694
S'a'
I1064
sS's'
I17
sS'm'
(lp11695
I1
aI3
aI5
aI7
aI17
aI19
aI23
aI225
aI501
aI189
aI1291
aI603
aI6873
aI8633
aI11425
aI30565
aI26355
asS'd'
I5848
sa(dp11696
S'a'
I1067
sS's'
I17
sS'm'
(lp11697
I1
aI3
aI7
aI11
aI23
aI17
aI91
aI111
aI415
aI225
aI1287
aI2081
aI4683
aI12069
aI3627
aI32281
aI17995
asS'd'
I5849
sa(dp11698
S'a'
I1082
sS's'
I17
sS'm'
(lp11699
I1
aI1
aI5
aI15
aI25
aI59
aI75
aI203
aI179
aI405
aI1711
aI3147
aI7483
aI5583
aI3729
aI11765
aI61019
asS'd'
I5850
sa(dp11700
S'a'
I1096
sS's'
I17
sS'm'
(lp11701
I1
aI3
aI3
aI9
aI3
aI43
aI65
aI7
aI269
aI33
aI829
aI1789
aI967
aI13119
aI26329
aI16937
aI18533
asS'd'
I5851
sa(dp11702
S'a'
I1116
sS's'
I17
sS'm'
(lp11703
I1
aI1
aI3
aI15
aI11
aI39
aI73
aI11
aI31
aI143
aI1913
aI1227
aI1363
aI11831
aI28687
aI50489
aI106373
asS'd'
I5852
sa(dp11704
S'a'
I1119
sS's'
I17
sS'm'
(lp11705
I1
aI1
aI3
aI3
aI25
aI19
aI15
aI11
aI349
aI1011
aI421
aI3193
aI3665
aI6149
aI20729
aI6997
aI51437
asS'd'
I5853
sa(dp11706
S'a'
I1129
sS's'
I17
sS'm'
(lp11707
I1
aI3
aI5
aI9
aI13
aI63
aI73
aI55
aI417
aI223
aI1753
aI2913
aI4809
aI3947
aI10769
aI5751
aI93867
asS'd'
I5854
sa(dp11708
S'a'
I1130
sS's'
I17
sS'm'
(lp11709
I1
aI3
aI7
aI13
aI31
aI39
aI39
aI133
aI483
aI839
aI1137
aI3303
aI7285
aI4309
aI24079
aI60529
aI103337
asS'd'
I5855
sa(dp11710
S'a'
I1132
sS's'
I17
sS'm'
(lp11711
I1
aI1
aI3
aI7
aI1
aI55
aI3
aI253
aI435
aI589
aI1949
aI1461
aI513
aI381
aI29455
aI4263
aI16831
asS'd'
I5856
sa(dp11712
S'a'
I1137
sS's'
I17
sS'm'
(lp11713
I1
aI1
aI1
aI15
aI25
aI19
aI77
aI101
aI299
aI187
aI1021
aI1533
aI8021
aI4165
aI2277
aI18927
aI110439
asS'd'
I5857
sa(dp11714
S'a'
I1147
sS's'
I17
sS'm'
(lp11715
I1
aI1
aI1
aI11
aI9
aI35
aI71
aI159
aI409
aI527
aI15
aI4073
aI5749
aI8563
aI2503
aI53015
aI111581
asS'd'
I5858
sa(dp11716
S'a'
I1150
sS's'
I17
sS'm'
(lp11717
I1
aI1
aI7
aI5
aI21
aI47
aI113
aI23
aI477
aI559
aI543
aI409
aI4701
aI11479
aI30761
aI8373
aI87777
asS'd'
I5859
sa(dp11718
S'a'
I1154
sS's'
I17
sS'm'
(lp11719
I1
aI3
aI5
aI13
aI9
aI27
aI25
aI137
aI81
aI37
aI799
aI857
aI3539
aI4471
aI15753
aI59015
aI48589
asS'd'
I5860
sa(dp11720
S'a'
I1165
sS's'
I17
sS'm'
(lp11721
I1
aI1
aI3
aI7
aI11
aI57
aI103
aI83
aI209
aI71
aI193
aI3251
aI4839
aI13959
aI32009
aI6471
aI23631
asS'd'
I5861
sa(dp11722
S'a'
I1166
sS's'
I17
sS'm'
(lp11723
I1
aI1
aI7
aI11
aI25
aI33
aI85
aI31
aI371
aI253
aI1667
aI1627
aI6159
aI10039
aI15177
aI52121
aI39475
asS'd'
I5862
sa(dp11724
S'a'
I1174
sS's'
I17
sS'm'
(lp11725
I1
aI1
aI5
aI9
aI13
aI55
aI37
aI13
aI95
aI113
aI1895
aI1525
aI1907
aI6361
aI5863
aI27767
aI108143
asS'd'
I5863
sa(dp11726
S'a'
I1177
sS's'
I17
sS'm'
(lp11727
I1
aI1
aI3
aI13
aI21
aI5
aI53
aI39
aI485
aI171
aI1355
aI2117
aI3127
aI6467
aI31697
aI45343
aI111477
asS'd'
I5864
sa(dp11728
S'a'
I1184
sS's'
I17
sS'm'
(lp11729
I1
aI1
aI7
aI15
aI13
aI57
aI11
aI231
aI329
aI703
aI1823
aI2983
aI215
aI2835
aI19719
aI56637
aI126169
asS'd'
I5865
sa(dp11730
S'a'
I1194
sS's'
I17
sS'm'
(lp11731
I1
aI3
aI5
aI15
aI13
aI51
aI13
aI173
aI301
aI867
aI127
aI2391
aI2795
aI4945
aI13293
aI49947
aI10765
asS'd'
I5866
sa(dp11732
S'a'
I1204
sS's'
I17
sS'm'
(lp11733
I1
aI3
aI3
aI9
aI23
aI5
aI29
aI165
aI467
aI599
aI1181
aI3213
aI4069
aI5473
aI8937
aI51495
aI42611
asS'd'
I5867
sa(dp11734
S'a'
I1208
sS's'
I17
sS'm'
(lp11735
I1
aI1
aI7
aI15
aI5
aI5
aI31
aI125
aI397
aI519
aI1465
aI115
aI7877
aI7025
aI14213
aI50343
aI85827
asS'd'
I5868
sa(dp11736
S'a'
I1213
sS's'
I17
sS'm'
(lp11737
I1
aI3
aI7
aI3
aI25
aI59
aI95
aI103
aI101
aI347
aI95
aI3
aI1251
aI15109
aI12615
aI7511
aI56789
asS'd'
I5869
sa(dp11738
S'a'
I1219
sS's'
I17
sS'm'
(lp11739
I1
aI3
aI5
aI9
aI13
aI59
aI71
aI19
aI107
aI73
aI345
aI3177
aI6519
aI2407
aI18033
aI31075
aI113185
asS'd'
I5870
sa(dp11740
S'a'
I1233
sS's'
I17
sS'm'
(lp11741
I1
aI1
aI1
aI3
aI27
aI37
aI5
aI219
aI169
aI149
aI355
aI549
aI1811
aI11351
aI22627
aI53931
aI88619
asS'd'
I5871
sa(dp11742
S'a'
I1264
sS's'
I17
sS'm'
(lp11743
I1
aI3
aI1
aI3
aI27
aI7
aI9
aI97
aI399
aI947
aI1393
aI3917
aI5439
aI15845
aI19465
aI30123
aI69099
asS'd'
I5872
sa(dp11744
S'a'
I1267
sS's'
I17
sS'm'
(lp11745
I1
aI1
aI7
aI9
aI13
aI25
aI107
aI45
aI111
aI409
aI967
aI3359
aI2499
aI1703
aI20763
aI45187
aI16265
asS'd'
I5873
sa(dp11746
S'a'
I1281
sS's'
I17
sS'm'
(lp11747
I1
aI1
aI1
aI13
aI5
aI49
aI43
aI249
aI49
aI947
aI597
aI1773
aI2387
aI2693
aI15297
aI57969
aI53385
asS'd'
I5874
sa(dp11748
S'a'
I1312
sS's'
I17
sS'm'
(lp11749
I1
aI1
aI7
aI15
aI27
aI25
aI27
aI121
aI421
aI781
aI143
aI817
aI7335
aI14211
aI139
aI55601
aI56671
asS'd'
I5875
sa(dp11750
S'a'
I1321
sS's'
I17
sS'm'
(lp11751
I1
aI3
aI1
aI5
aI29
aI47
aI77
aI23
aI413
aI931
aI785
aI1221
aI769
aI13131
aI26955
aI56441
aI85745
asS'd'
I5876
sa(dp11752
S'a'
I1330
sS's'
I17
sS'm'
(lp11753
I1
aI1
aI1
aI11
aI27
aI3
aI53
aI21
aI467
aI43
aI1533
aI1053
aI691
aI6369
aI8325
aI51087
aI71261
asS'd'
I5877
sa(dp11754
S'a'
I1332
sS's'
I17
sS'm'
(lp11755
I1
aI1
aI3
aI15
aI7
aI9
aI43
aI225
aI293
aI143
aI1049
aI3095
aI6119
aI3165
aI9913
aI26023
aI62657
asS'd'
I5878
sa(dp11756
S'a'
I1335
sS's'
I17
sS'm'
(lp11757
I1
aI3
aI7
aI9
aI11
aI39
aI99
aI193
aI217
aI941
aI259
aI3811
aI6757
aI281
aI10377
aI46961
aI48949
asS'd'
I5879
sa(dp11758
S'a'
I1341
sS's'
I17
sS'm'
(lp11759
I1
aI1
aI1
aI1
aI25
aI1
aI99
aI61
aI495
aI861
aI2013
aI487
aI2821
aI12921
aI30111
aI27213
aI97363
asS'd'
I5880
sa(dp11760
S'a'
I1356
sS's'
I17
sS'm'
(lp11761
I1
aI1
aI5
aI9
aI23
aI33
aI103
aI237
aI161
aI721
aI2021
aI159
aI995
aI475
aI20615
aI30961
aI31767
asS'd'
I5881
sa(dp11762
S'a'
I1371
sS's'
I17
sS'm'
(lp11763
I1
aI3
aI1
aI1
aI5
aI59
aI63
aI139
aI451
aI789
aI1285
aI655
aI5501
aI273
aI21061
aI35937
aI20811
asS'd'
I5882
sa(dp11764
S'a'
I1377
sS's'
I17
sS'm'
(lp11765
I1
aI3
aI3
aI9
aI9
aI15
aI121
aI233
aI287
aI929
aI1605
aI1243
aI417
aI1695
aI29903
aI28699
aI85981
asS'd'
I5883
sa(dp11766
S'a'
I1380
sS's'
I17
sS'm'
(lp11767
I1
aI3
aI3
aI5
aI7
aI25
aI27
aI253
aI469
aI255
aI285
aI2467
aI4897
aI4079
aI29759
aI50351
aI76451
asS'd'
I5884
sa(dp11768
S'a'
I1384
sS's'
I17
sS'm'
(lp11769
I1
aI1
aI3
aI3
aI5
aI33
aI29
aI209
aI291
aI967
aI1429
aI1953
aI5957
aI14065
aI8875
aI32675
aI4629
asS'd'
I5885
sa(dp11770
S'a'
I1395
sS's'
I17
sS'm'
(lp11771
I1
aI3
aI5
aI9
aI7
aI31
aI97
aI21
aI177
aI485
aI1115
aI4051
aI6683
aI7761
aI30181
aI37531
aI51789
asS'd'
I5886
sa(dp11772
S'a'
I1397
sS's'
I17
sS'm'
(lp11773
I1
aI1
aI7
aI3
aI25
aI51
aI23
aI183
aI57
aI699
aI1245
aI2519
aI2783
aI4457
aI6381
aI43199
aI40071
asS'd'
I5887
sa(dp11774
S'a'
I1411
sS's'
I17
sS'm'
(lp11775
I1
aI3
aI5
aI5
aI19
aI55
aI45
aI101
aI299
aI461
aI1009
aI319
aI7335
aI7769
aI5479
aI61113
aI7937
asS'd'
I5888
sa(dp11776
S'a'
I1414
sS's'
I17
sS'm'
(lp11777
I1
aI1
aI7
aI3
aI29
aI21
aI55
aI55
aI437
aI771
aI363
aI683
aI4299
aI15569
aI13813
aI40663
aI86285
asS'd'
I5889
sa(dp11778
S'a'
I1426
sS's'
I17
sS'm'
(lp11779
I1
aI1
aI1
aI13
aI31
aI35
aI93
aI175
aI451
aI387
aI1145
aI3367
aI3833
aI13495
aI11019
aI48925
aI85721
asS'd'
I5890
sa(dp11780
S'a'
I1432
sS's'
I17
sS'm'
(lp11781
I1
aI1
aI7
aI15
aI31
aI21
aI55
aI205
aI117
aI895
aI535
aI2627
aI1473
aI10779
aI24493
aI42999
aI130805
asS'd'
I5891
sa(dp11782
S'a'
I1435
sS's'
I17
sS'm'
(lp11783
I1
aI1
aI3
aI13
aI27
aI11
aI45
aI37
aI193
aI237
aI1505
aI1405
aI3613
aI9565
aI3037
aI53643
aI85211
asS'd'
I5892
sa(dp11784
S'a'
I1437
sS's'
I17
sS'm'
(lp11785
I1
aI1
aI3
aI13
aI9
aI17
aI19
aI27
aI117
aI503
aI65
aI1033
aI7891
aI4005
aI9229
aI20999
aI96601
asS'd'
I5893
sa(dp11786
S'a'
I1442
sS's'
I17
sS'm'
(lp11787
I1
aI3
aI3
aI5
aI17
aI3
aI71
aI79
aI145
aI985
aI935
aI3997
aI6239
aI12511
aI13895
aI65031
aI126383
asS'd'
I5894
sa(dp11788
S'a'
I1454
sS's'
I17
sS'm'
(lp11789
I1
aI1
aI5
aI1
aI23
aI55
aI3
aI105
aI71
aI243
aI1479
aI111
aI7103
aI10753
aI26193
aI35833
aI14583
asS'd'
I5895
sa(dp11790
S'a'
I1468
sS's'
I17
sS'm'
(lp11791
I1
aI3
aI3
aI3
aI15
aI3
aI73
aI125
aI267
aI29
aI1775
aI1437
aI8091
aI10891
aI25731
aI54381
aI12821
asS'd'
I5896
sa(dp11792
S'a'
I1473
sS's'
I17
sS'm'
(lp11793
I1
aI1
aI1
aI3
aI23
aI15
aI67
aI123
aI401
aI347
aI807
aI1097
aI31
aI11209
aI8727
aI58149
aI129099
asS'd'
I5897
sa(dp11794
S'a'
I1488
sS's'
I17
sS'm'
(lp11795
I1
aI3
aI3
aI7
aI7
aI61
aI49
aI129
aI423
aI535
aI135
aI3587
aI233
aI4509
aI23209
aI59203
aI41297
asS'd'
I5898
sa(dp11796
S'a'
I1491
sS's'
I17
sS'm'
(lp11797
I1
aI3
aI1
aI7
aI5
aI29
aI65
aI31
aI335
aI855
aI835
aI1421
aI3081
aI14219
aI16321
aI48269
aI41603
asS'd'
I5899
sa(dp11798
S'a'
I1509
sS's'
I17
sS'm'
(lp11799
I1
aI1
aI1
aI13
aI3
aI21
aI5
aI117
aI163
aI603
aI1519
aI3789
aI7873
aI10981
aI4615
aI9165
aI83929
asS'd'
I5900
sa(dp11800
S'a'
I1524
sS's'
I17
sS'm'
(lp11801
I1
aI3
aI5
aI11
aI15
aI21
aI75
aI151
aI193
aI757
aI647
aI1603
aI333
aI10515
aI22771
aI55459
aI3315
asS'd'
I5901
sa(dp11802
S'a'
I1533
sS's'
I17
sS'm'
(lp11803
I1
aI1
aI7
aI1
aI27
aI3
aI63
aI197
aI271
aI175
aI1599
aI2119
aI1031
aI8671
aI10893
aI35641
aI94535
asS'd'
I5902
sa(dp11804
S'a'
I1555
sS's'
I17
sS'm'
(lp11805
I1
aI1
aI1
aI15
aI1
aI59
aI93
aI17
aI5
aI213
aI1663
aI941
aI435
aI8107
aI1963
aI34951
aI106181
asS'd'
I5903
sa(dp11806
S'a'
I1567
sS's'
I17
sS'm'
(lp11807
I1
aI1
aI5
aI11
aI13
aI35
aI111
aI97
aI267
aI737
aI2023
aI1301
aI7407
aI11249
aI31785
aI31933
aI20673
asS'd'
I5904
sa(dp11808
S'a'
I1571
sS's'
I17
sS'm'
(lp11809
I1
aI3
aI3
aI15
aI5
aI15
aI29
aI63
aI189
aI687
aI27
aI2005
aI7129
aI11377
aI23175
aI42389
aI30933
asS'd'
I5905
sa(dp11810
S'a'
I1586
sS's'
I17
sS'm'
(lp11811
I1
aI1
aI1
aI9
aI13
aI63
aI7
aI155
aI67
aI291
aI1419
aI755
aI2623
aI4749
aI22971
aI7545
aI55711
asS'd'
I5906
sa(dp11812
S'a'
I1592
sS's'
I17
sS'm'
(lp11813
I1
aI3
aI7
aI7
aI23
aI29
aI83
aI151
aI213
aI201
aI157
aI3051
aI6553
aI6401
aI15931
aI47941
aI22869
asS'd'
I5907
sa(dp11814
S'a'
I1595
sS's'
I17
sS'm'
(lp11815
I1
aI3
aI5
aI5
aI7
aI45
aI33
aI155
aI225
aI25
aI49
aI2419
aI4241
aI6835
aI11401
aI50725
aI118343
asS'd'
I5908
sa(dp11816
S'a'
I1600
sS's'
I17
sS'm'
(lp11817
I1
aI1
aI3
aI13
aI31
aI27
aI37
aI41
aI19
aI375
aI1771
aI1789
aI2313
aI2577
aI12615
aI22715
aI22179
asS'd'
I5909
sa(dp11818
S'a'
I1606
sS's'
I17
sS'm'
(lp11819
I1
aI3
aI1
aI11
aI17
aI53
aI55
aI229
aI235
aI837
aI143
aI3583
aI2789
aI5471
aI6515
aI44565
aI8619
asS'd'
I5910
sa(dp11820
S'a'
I1627
sS's'
I17
sS'm'
(lp11821
I1
aI1
aI5
aI15
aI5
aI17
aI23
aI95
aI217
aI551
aI353
aI27
aI3973
aI2547
aI27903
aI50611
aI72277
asS'd'
I5911
sa(dp11822
S'a'
I1648
sS's'
I17
sS'm'
(lp11823
I1
aI1
aI3
aI7
aI5
aI13
aI41
aI111
aI157
aI215
aI1327
aI3073
aI1871
aI11875
aI24239
aI40527
aI97637
asS'd'
I5912
sa(dp11824
S'a'
I1651
sS's'
I17
sS'm'
(lp11825
I1
aI3
aI1
aI1
aI29
aI63
aI111
aI187
aI369
aI395
aI1197
aI3229
aI4353
aI14715
aI29671
aI50503
aI89321
asS'd'
I5913
sa(dp11826
S'a'
I1654
sS's'
I17
sS'm'
(lp11827
I1
aI3
aI1
aI1
aI5
aI63
aI11
aI39
aI171
aI209
aI463
aI3421
aI3451
aI4453
aI14397
aI2219
aI98261
asS'd'
I5914
sa(dp11828
S'a'
I1667
sS's'
I17
sS'm'
(lp11829
I1
aI3
aI3
aI5
aI1
aI1
aI13
aI101
aI67
aI815
aI1521
aI1543
aI7221
aI7337
aI10765
aI30029
aI47881
asS'd'
I5915
sa(dp11830
S'a'
I1669
sS's'
I17
sS'm'
(lp11831
I1
aI1
aI5
aI7
aI9
aI9
aI33
aI197
aI439
aI893
aI961
aI11
aI4319
aI14265
aI24839
aI33581
aI35531
asS'd'
I5916
sa(dp11832
S'a'
I1674
sS's'
I17
sS'm'
(lp11833
I1
aI3
aI3
aI15
aI29
aI35
aI43
aI229
aI313
aI369
aI955
aI1069
aI2939
aI12623
aI20373
aI1533
aI9105
asS'd'
I5917
sa(dp11834
S'a'
I1687
sS's'
I17
sS'm'
(lp11835
I1
aI3
aI1
aI7
aI21
aI7
aI127
aI243
aI103
aI353
aI859
aI3789
aI4369
aI12063
aI22369
aI14531
aI94289
asS'd'
I5918
sa(dp11836
S'a'
I1698
sS's'
I17
sS'm'
(lp11837
I1
aI3
aI5
aI15
aI1
aI27
aI65
aI127
aI229
aI99
aI627
aI2693
aI7173
aI7305
aI29971
aI7097
aI10113
asS'd'
I5919
sa(dp11838
S'a'
I1710
sS's'
I17
sS'm'
(lp11839
I1
aI1
aI5
aI15
aI3
aI47
aI61
aI29
aI155
aI725
aI1727
aI2667
aI7003
aI16277
aI21983
aI21365
aI129365
asS'd'
I5920
sa(dp11840
S'a'
I1717
sS's'
I17
sS'm'
(lp11841
I1
aI1
aI5
aI7
aI27
aI61
aI115
aI133
aI137
aI661
aI1201
aI2151
aI367
aI3567
aI12885
aI62143
aI53955
asS'd'
I5921
sa(dp11842
S'a'
I1722
sS's'
I17
sS'm'
(lp11843
I1
aI1
aI1
aI11
aI9
aI41
aI113
aI103
aI469
aI687
aI1541
aI3679
aI6833
aI10493
aI32747
aI39909
aI121445
asS'd'
I5922
sa(dp11844
S'a'
I1735
sS's'
I17
sS'm'
(lp11845
I1
aI1
aI7
aI5
aI5
aI5
aI91
aI91
aI5
aI405
aI529
aI3999
aI6783
aI2387
aI16621
aI12919
aI8659
asS'd'
I5923
sa(dp11846
S'a'
I1741
sS's'
I17
sS'm'
(lp11847
I1
aI1
aI7
aI13
aI21
aI47
aI125
aI155
aI83
aI913
aI1833
aI4027
aI6657
aI7031
aI31231
aI58201
aI88943
asS'd'
I5924
sa(dp11848
S'a'
I1749
sS's'
I17
sS'm'
(lp11849
I1
aI3
aI7
aI3
aI17
aI55
aI25
aI29
aI181
aI205
aI1173
aI1081
aI6475
aI5037
aI18461
aI22487
aI114131
asS'd'
I5925
sa(dp11850
S'a'
I1750
sS's'
I17
sS'm'
(lp11851
I1
aI1
aI7
aI7
aI25
aI63
aI101
aI103
aI171
aI191
aI1863
aI3441
aI2515
aI14179
aI30123
aI19145
aI31669
asS'd'
I5926
sa(dp11852
S'a'
I1769
sS's'
I17
sS'm'
(lp11853
I1
aI3
aI7
aI11
aI29
aI49
aI73
aI163
aI415
aI821
aI1809
aI723
aI7049
aI14565
aI4829
aI19395
aI61131
asS'd'
I5927
sa(dp11854
S'a'
I1775
sS's'
I17
sS'm'
(lp11855
I1
aI1
aI7
aI9
aI5
aI25
aI103
aI167
aI381
aI757
aI813
aI471
aI3021
aI6619
aI20929
aI38133
aI129505
asS'd'
I5928
sa(dp11856
S'a'
I1777
sS's'
I17
sS'm'
(lp11857
I1
aI1
aI5
aI13
aI25
aI61
aI59
aI199
aI257
aI999
aI169
aI3289
aI7181
aI2049
aI2185
aI39045
aI102703
asS'd'
I5929
sa(dp11858
S'a'
I1778
sS's'
I17
sS'm'
(lp11859
I1
aI1
aI3
aI1
aI21
aI1
aI111
aI125
aI289
aI33
aI701
aI3491
aI5569
aI8055
aI23149
aI26793
aI102563
asS'd'
I5930
sa(dp11860
S'a'
I1792
sS's'
I17
sS'm'
(lp11861
I1
aI1
aI7
aI3
aI25
aI15
aI105
aI235
aI307
aI201
aI1947
aI699
aI2519
aI10615
aI29345
aI17061
aI112949
asS'd'
I5931
sa(dp11862
S'a'
I1797
sS's'
I17
sS'm'
(lp11863
I1
aI3
aI3
aI15
aI19
aI1
aI93
aI173
aI399
aI13
aI269
aI1189
aI523
aI5145
aI32731
aI54087
aI94123
asS'd'
I5932
sa(dp11864
S'a'
I1802
sS's'
I17
sS'm'
(lp11865
I1
aI3
aI1
aI15
aI9
aI41
aI59
aI79
aI217
aI833
aI1993
aI2429
aI3599
aI6919
aI30911
aI12615
aI67947
asS'd'
I5933
sa(dp11866
S'a'
I1822
sS's'
I17
sS'm'
(lp11867
I1
aI3
aI3
aI13
aI31
aI9
aI95
aI37
aI343
aI955
aI1363
aI3851
aI4091
aI13165
aI15241
aI14853
aI35747
asS'd'
I5934
sa(dp11868
S'a'
I1825
sS's'
I17
sS'm'
(lp11869
I1
aI1
aI3
aI5
aI27
aI39
aI37
aI217
aI385
aI473
aI1997
aI2247
aI7353
aI1503
aI9003
aI15055
aI27289
asS'd'
I5935
sa(dp11870
S'a'
I1831
sS's'
I17
sS'm'
(lp11871
I1
aI3
aI7
aI11
aI1
aI13
aI21
aI243
aI375
aI91
aI1295
aI1661
aI203
aI15251
aI15355
aI16065
aI24183
asS'd'
I5936
sa(dp11872
S'a'
I1838
sS's'
I17
sS'm'
(lp11873
I1
aI3
aI1
aI13
aI11
aI45
aI85
aI5
aI275
aI741
aI1395
aI4011
aI7987
aI16087
aI24113
aI50555
aI128147
asS'd'
I5937
sa(dp11874
S'a'
I1852
sS's'
I17
sS'm'
(lp11875
I1
aI1
aI1
aI7
aI3
aI11
aI13
aI189
aI55
aI151
aI395
aI657
aI807
aI11973
aI26297
aI13043
aI109641
asS'd'
I5938
sa(dp11876
S'a'
I1855
sS's'
I17
sS'm'
(lp11877
I1
aI1
aI7
aI13
aI31
aI19
aI33
aI235
aI491
aI647
aI1115
aI2299
aI6381
aI7525
aI2237
aI36197
aI126457
asS'd'
I5939
sa(dp11878
S'a'
I1860
sS's'
I17
sS'm'
(lp11879
I1
aI3
aI5
aI1
aI21
aI15
aI53
aI231
aI77
aI347
aI969
aI141
aI4501
aI9429
aI1815
aI50887
aI74581
asS'd'
I5940
sa(dp11880
S'a'
I1867
sS's'
I17
sS'm'
(lp11881
I1
aI1
aI1
aI9
aI29
aI43
aI47
aI103
aI327
aI131
aI927
aI441
aI7517
aI7277
aI21065
aI409
aI50351
asS'd'
I5941
sa(dp11882
S'a'
I1869
sS's'
I17
sS'm'
(lp11883
I1
aI1
aI5
aI1
aI11
aI13
aI103
aI157
aI239
aI69
aI1347
aI477
aI5017
aI9723
aI28133
aI65135
aI12359
asS'd'
I5942
sa(dp11884
S'a'
I1875
sS's'
I17
sS'm'
(lp11885
I1
aI1
aI1
aI13
aI17
aI63
aI117
aI189
aI323
aI565
aI927
aI1727
aI5337
aI13243
aI5739
aI31241
aI14209
asS'd'
I5943
sa(dp11886
S'a'
I1882
sS's'
I17
sS'm'
(lp11887
I1
aI1
aI3
aI9
aI29
aI9
aI103
aI61
aI467
aI217
aI1367
aI2405
aI5355
aI5743
aI31469
aI30149
aI98775
asS'd'
I5944
sa(dp11888
S'a'
I1903
sS's'
I17
sS'm'
(lp11889
I1
aI1
aI1
aI15
aI23
aI23
aI17
aI229
aI103
aI583
aI179
aI115
aI7081
aI9437
aI32623
aI62639
aI72391
asS'd'
I5945
sa(dp11890
S'a'
I1908
sS's'
I17
sS'm'
(lp11891
I1
aI1
aI5
aI11
aI11
aI39
aI97
aI209
aI115
aI107
aI593
aI2347
aI1445
aI6179
aI32011
aI8435
aI65847
asS'd'
I5946
sa(dp11892
S'a'
I1917
sS's'
I17
sS'm'
(lp11893
I1
aI3
aI7
aI3
aI29
aI27
aI55
aI111
aI27
aI731
aI995
aI1871
aI5017
aI1485
aI11313
aI2559
aI6561
asS'd'
I5947
sa(dp11894
S'a'
I1927
sS's'
I17
sS'm'
(lp11895
I1
aI3
aI1
aI3
aI27
aI9
aI103
aI247
aI83
aI197
aI517
aI1629
aI2189
aI7255
aI183
aI35111
aI15077
asS'd'
I5948
sa(dp11896
S'a'
I1941
sS's'
I17
sS'm'
(lp11897
I1
aI3
aI7
aI5
aI31
aI37
aI87
aI223
aI343
aI331
aI1361
aI3371
aI2007
aI13235
aI10897
aI63839
aI109837
asS'd'
I5949
sa(dp11898
S'a'
I1945
sS's'
I17
sS'm'
(lp11899
I1
aI1
aI7
aI11
aI17
aI5
aI41
aI197
aI489
aI625
aI1595
aI2663
aI5941
aI14029
aI30999
aI16781
aI116001
asS'd'
I5950
sa(dp11900
S'a'
I1948
sS's'
I17
sS'm'
(lp11901
I1
aI3
aI3
aI7
aI19
aI19
aI61
aI175
aI125
aI609
aI1391
aI147
aI3001
aI4189
aI10133
aI24031
aI46219
asS'd'
I5951
sa(dp11902
S'a'
I1962
sS's'
I17
sS'm'
(lp11903
I1
aI1
aI3
aI13
aI13
aI57
aI117
aI181
aI299
aI939
aI583
aI3151
aI829
aI6561
aI30449
aI12211
aI107879
asS'd'
I5952
sa(dp11904
S'a'
I1975
sS's'
I17
sS'm'
(lp11905
I1
aI1
aI5
aI11
aI23
aI45
aI87
aI115
aI259
aI613
aI1001
aI171
aI57
aI13789
aI22173
aI56837
aI26263
asS'd'
I5953
sa(dp11906
S'a'
I1976
sS's'
I17
sS'm'
(lp11907
I1
aI1
aI3
aI3
aI7
aI43
aI45
aI131
aI87
aI251
aI1411
aI2737
aI2739
aI4595
aI12561
aI12043
aI82885
asS'd'
I5954
sa(dp11908
S'a'
I1987
sS's'
I17
sS'm'
(lp11909
I1
aI3
aI3
aI7
aI19
aI39
aI87
aI223
aI461
aI37
aI283
aI3937
aI6193
aI10887
aI11509
aI41131
aI38359
asS'd'
I5955
sa(dp11910
S'a'
I1993
sS's'
I17
sS'm'
(lp11911
I1
aI3
aI1
aI11
aI11
aI37
aI25
aI133
aI105
aI1013
aI925
aI3301
aI239
aI16295
aI4831
aI8649
aI125767
asS'd'
I5956
sa(dp11912
S'a'
I2004
sS's'
I17
sS'm'
(lp11913
I1
aI3
aI3
aI11
aI25
aI11
aI41
aI155
aI1
aI717
aI1587
aI635
aI279
aI1803
aI14817
aI28669
aI88835
asS'd'
I5957
sa(dp11914
S'a'
I2020
sS's'
I17
sS'm'
(lp11915
I1
aI3
aI3
aI11
aI29
aI17
aI39
aI51
aI13
aI871
aI1197
aI2561
aI6671
aI8465
aI22709
aI15933
aI15923
asS'd'
I5958
sa(dp11916
S'a'
I2029
sS's'
I17
sS'm'
(lp11917
I1
aI3
aI7
aI1
aI13
aI17
aI57
aI43
aI267
aI261
aI901
aI241
aI3767
aI15053
aI11017
aI36321
aI72497
asS'd'
I5959
sa(dp11918
S'a'
I2030
sS's'
I17
sS'm'
(lp11919
I1
aI3
aI1
aI15
aI23
aI13
aI17
aI63
aI171
aI919
aI1387
aI2673
aI7605
aI8523
aI14807
aI21187
aI56057
asS'd'
I5960
sa(dp11920
S'a'
I2038
sS's'
I17
sS'm'
(lp11921
I1
aI3
aI7
aI15
aI23
aI41
aI85
aI95
aI53
aI629
aI1877
aI3167
aI2411
aI9619
aI24621
aI31213
aI30069
asS'd'
I5961
sa(dp11922
S'a'
I2041
sS's'
I17
sS'm'
(lp11923
I1
aI1
aI5
aI3
aI3
aI25
aI99
aI39
aI321
aI549
aI599
aI1279
aI2401
aI2335
aI8227
aI59429
aI94549
asS'd'
I5962
sa(dp11924
S'a'
I2048
sS's'
I17
sS'm'
(lp11925
I1
aI3
aI3
aI11
aI9
aI21
aI29
aI55
aI477
aI19
aI1275
aI29
aI2253
aI11421
aI30401
aI57059
aI93219
asS'd'
I5963
sa(dp11926
S'a'
I2054
sS's'
I17
sS'm'
(lp11927
I1
aI1
aI7
aI1
aI27
aI13
aI117
aI249
aI463
aI769
aI281
aI515
aI7467
aI11507
aI1621
aI39765
aI31109
asS'd'
I5964
sa(dp11928
S'a'
I2057
sS's'
I17
sS'm'
(lp11929
I1
aI3
aI5
aI7
aI19
aI7
aI77
aI107
aI23
aI895
aI1013
aI2701
aI3805
aI7327
aI27247
aI6119
aI102395
asS'd'
I5965
sa(dp11930
S'a'
I2058
sS's'
I17
sS'm'
(lp11931
I1
aI1
aI3
aI13
aI21
aI49
aI99
aI15
aI163
aI641
aI1703
aI3061
aI163
aI4265
aI32571
aI13957
aI75005
asS'd'
I5966
sa(dp11932
S'a'
I2068
sS's'
I17
sS'm'
(lp11933
I1
aI1
aI5
aI11
aI27
aI17
aI87
aI169
aI427
aI959
aI361
aI1023
aI5727
aI16279
aI1099
aI39081
aI67215
asS'd'
I5967
sa(dp11934
S'a'
I2072
sS's'
I17
sS'm'
(lp11935
I1
aI3
aI3
aI9
aI23
aI13
aI1
aI91
aI173
aI325
aI1881
aI1385
aI8023
aI935
aI9221
aI19673
aI36949
asS'd'
I5968
sa(dp11936
S'a'
I2075
sS's'
I17
sS'm'
(lp11937
I1
aI3
aI1
aI7
aI7
aI25
aI119
aI189
aI107
aI249
aI811
aI973
aI6499
aI101
aI11281
aI55227
aI32361
asS'd'
I5969
sa(dp11938
S'a'
I2077
sS's'
I17
sS'm'
(lp11939
I1
aI1
aI5
aI13
aI19
aI37
aI117
aI95
aI463
aI587
aI1419
aI445
aI4019
aI7257
aI29757
aI50773
aI52247
asS'd'
I5970
sa(dp11940
S'a'
I2082
sS's'
I17
sS'm'
(lp11941
I1
aI1
aI1
aI1
aI17
aI57
aI81
aI57
aI43
aI789
aI1035
aI625
aI1707
aI9683
aI3681
aI12411
aI110623
asS'd'
I5971
sa(dp11942
S'a'
I2084
sS's'
I17
sS'm'
(lp11943
I1
aI1
aI7
aI5
aI7
aI57
aI49
aI91
aI459
aI513
aI1869
aI3377
aI139
aI10037
aI24091
aI54247
aI41279
asS'd'
I5972
sa(dp11944
S'a'
I2087
sS's'
I17
sS'm'
(lp11945
I1
aI3
aI3
aI9
aI9
aI33
aI29
aI51
aI355
aI415
aI1907
aI809
aI6543
aI349
aI18507
aI12919
aI41667
asS'd'
I5973
sa(dp11946
S'a'
I2101
sS's'
I17
sS'm'
(lp11947
I1
aI1
aI5
aI11
aI3
aI17
aI73
aI201
aI121
aI909
aI1623
aI799
aI3271
aI9051
aI5717
aI15169
aI127861
asS'd'
I5974
sa(dp11948
S'a'
I2111
sS's'
I17
sS'm'
(lp11949
I1
aI1
aI7
aI7
aI23
aI31
aI1
aI155
aI475
aI87
aI2001
aI2459
aI1285
aI5931
aI6803
aI56757
aI71671
asS'd'
I5975
sa(dp11950
S'a'
I2113
sS's'
I17
sS'm'
(lp11951
I1
aI1
aI5
aI13
aI5
aI1
aI21
aI109
aI263
aI841
aI723
aI1539
aI7529
aI433
aI23721
aI33195
aI57001
asS'd'
I5976
sa(dp11952
S'a'
I2119
sS's'
I17
sS'm'
(lp11953
I1
aI3
aI3
aI13
aI29
aI55
aI105
aI231
aI405
aI265
aI671
aI351
aI4693
aI9033
aI21963
aI52073
aI125131
asS'd'
I5977
sa(dp11954
S'a'
I2147
sS's'
I17
sS'm'
(lp11955
I1
aI3
aI1
aI13
aI25
aI51
aI55
aI227
aI245
aI983
aI251
aI2553
aI2017
aI1381
aI31461
aI3953
aI75775
asS'd'
I5978
sa(dp11956
S'a'
I2154
sS's'
I17
sS'm'
(lp11957
I1
aI1
aI1
aI11
aI31
aI11
aI91
aI91
aI287
aI749
aI1019
aI4055
aI3237
aI6965
aI14765
aI1663
aI82987
asS'd'
I5979
sa(dp11958
S'a'
I2161
sS's'
I17
sS'm'
(lp11959
I1
aI1
aI7
aI3
aI11
aI15
aI67
aI161
aI79
aI729
aI1115
aI3713
aI2715
aI9361
aI9365
aI26093
aI63409
asS'd'
I5980
sa(dp11960
S'a'
I2164
sS's'
I17
sS'm'
(lp11961
I1
aI3
aI1
aI7
aI1
aI51
aI125
aI15
aI457
aI433
aI405
aI2329
aI157
aI4817
aI25867
aI38177
aI45319
asS'd'
I5981
sa(dp11962
S'a'
I2177
sS's'
I17
sS'm'
(lp11963
I1
aI3
aI7
aI9
aI25
aI57
aI5
aI233
aI481
aI781
aI1313
aI3179
aI7219
aI8717
aI14825
aI16079
aI127149
asS'd'
I5982
sa(dp11964
S'a'
I2178
sS's'
I17
sS'm'
(lp11965
I1
aI1
aI7
aI15
aI27
aI51
aI5
aI65
aI77
aI313
aI1751
aI1489
aI4307
aI10541
aI11345
aI52577
aI18143
asS'd'
I5983
sa(dp11966
S'a'
I2184
sS's'
I17
sS'm'
(lp11967
I1
aI1
aI1
aI15
aI21
aI5
aI113
aI71
aI411
aI327
aI1681
aI1023
aI5661
aI15815
aI5387
aI10351
aI21121
asS'd'
I5984
sa(dp11968
S'a'
I2198
sS's'
I17
sS'm'
(lp11969
I1
aI1
aI5
aI5
aI29
aI55
aI25
aI255
aI69
aI879
aI501
aI1915
aI3731
aI633
aI12197
aI5249
aI31129
asS'd'
I5985
sa(dp11970
S'a'
I2201
sS's'
I17
sS'm'
(lp11971
I1
aI3
aI5
aI7
aI3
aI23
aI107
aI163
aI485
aI853
aI359
aI3069
aI4353
aI371
aI6027
aI53239
aI105541
asS'd'
I5986
sa(dp11972
S'a'
I2213
sS's'
I17
sS'm'
(lp11973
I1
aI3
aI5
aI15
aI7
aI41
aI9
aI47
aI33
aI327
aI621
aI147
aI577
aI29
aI14623
aI3403
aI9791
asS'd'
I5987
sa(dp11974
S'a'
I2217
sS's'
I17
sS'm'
(lp11975
I1
aI3
aI3
aI15
aI29
aI47
aI41
aI149
aI477
aI127
aI573
aI877
aI3101
aI5963
aI28457
aI14231
aI67425
asS'd'
I5988
sa(dp11976
S'a'
I2228
sS's'
I17
sS'm'
(lp11977
I1
aI1
aI1
aI15
aI31
aI7
aI55
aI191
aI101
aI259
aI1071
aI219
aI2233
aI3583
aI21969
aI32745
aI80529
asS'd'
I5989
sa(dp11978
S'a'
I2240
sS's'
I17
sS'm'
(lp11979
I1
aI3
aI7
aI13
aI17
aI53
aI115
aI69
aI241
aI71
aI1475
aI191
aI509
aI3721
aI15537
aI53773
aI18005
asS'd'
I5990
sa(dp11980
S'a'
I2245
sS's'
I17
sS'm'
(lp11981
I1
aI1
aI3
aI9
aI5
aI57
aI13
aI95
aI103
aI871
aI2043
aI2239
aI7833
aI10727
aI6513
aI55273
aI3781
asS'd'
I5991
sa(dp11982
S'a'
I2250
sS's'
I17
sS'm'
(lp11983
I1
aI1
aI5
aI5
aI9
aI11
aI55
aI151
aI239
aI537
aI135
aI2779
aI7393
aI15393
aI11097
aI58593
aI100745
asS'd'
I5992
sa(dp11984
S'a'
I2263
sS's'
I17
sS'm'
(lp11985
I1
aI1
aI1
aI9
aI15
aI39
aI29
aI105
aI441
aI181
aI1113
aI2125
aI8145
aI11045
aI6589
aI33603
aI83377
asS'd'
I5993
sa(dp11986
S'a'
I2267
sS's'
I17
sS'm'
(lp11987
I1
aI3
aI1
aI1
aI11
aI63
aI69
aI153
aI225
aI845
aI675
aI407
aI4691
aI13383
aI27359
aI38881
aI5509
asS'd'
I5994
sa(dp11988
S'a'
I2285
sS's'
I17
sS'm'
(lp11989
I1
aI3
aI7
aI11
aI23
aI31
aI69
aI3
aI41
aI57
aI683
aI887
aI6861
aI12161
aI14537
aI27293
aI113001
asS'd'
I5995
sa(dp11990
S'a'
I2286
sS's'
I17
sS'm'
(lp11991
I1
aI1
aI1
aI11
aI5
aI1
aI101
aI175
aI437
aI3
aI1477
aI1005
aI6607
aI7429
aI7213
aI4025
aI66479
asS'd'
I5996
sa(dp11992
S'a'
I2291
sS's'
I17
sS'm'
(lp11993
I1
aI1
aI7
aI5
aI19
aI7
aI99
aI131
aI273
aI977
aI1717
aI3831
aI175
aI5673
aI12577
aI36787
aI30945
asS'd'
I5997
sa(dp11994
S'a'
I2298
sS's'
I17
sS'm'
(lp11995
I1
aI3
aI1
aI1
aI15
aI37
aI105
aI195
aI61
aI869
aI255
aI2625
aI7401
aI9361
aI13217
aI52811
aI130811
asS'd'
I5998
sa(dp11996
S'a'
I2306
sS's'
I17
sS'm'
(lp11997
I1
aI3
aI5
aI3
aI29
aI27
aI105
aI23
aI511
aI813
aI1311
aI2859
aI1647
aI1949
aI1329
aI27589
aI125209
asS'd'
I5999
sa(dp11998
S'a'
I2325
sS's'
I17
sS'm'
(lp11999
I1
aI3
aI3
aI1
aI21
aI11
aI119
aI247
aI123
aI401
aI409
aI1845
aI2133
aI10793
aI221
aI43217
aI14069
asS'd'
I6000
sa(dp12000
S'a'
I2329
sS's'
I17
sS'm'
(lp12001
I1
aI1
aI5
aI1
aI29
aI21
aI51
aI73
aI501
aI861
aI725
aI249
aI4249
aI8029
aI15767
aI11985
aI18637
asS'd'
I6001
sa(dp12002
S'a'
I2332
sS's'
I17
sS'm'
(lp12003
I1
aI1
aI5
aI11
aI19
aI39
aI97
aI65
aI13
aI283
aI489
aI2307
aI5239
aI4161
aI18639
aI60035
aI22405
asS'd'
I6002
sa(dp12004
S'a'
I2335
sS's'
I17
sS'm'
(lp12005
I1
aI3
aI5
aI1
aI3
aI7
aI109
aI27
aI429
aI663
aI1569
aI3001
aI3453
aI8627
aI9719
aI23941
aI110451
asS'd'
I6003
sa(dp12006
S'a'
I2339
sS's'
I17
sS'm'
(lp12007
I1
aI3
aI7
aI5
aI17
aI13
aI125
aI209
aI347
aI95
aI1937
aI1419
aI5661
aI7171
aI20607
aI9777
aI68343
asS'd'
I6004
sa(dp12008
S'a'
I2346
sS's'
I17
sS'm'
(lp12009
I1
aI1
aI1
aI1
aI7
aI41
aI43
aI229
aI57
aI49
aI1863
aI2819
aI3735
aI915
aI1571
aI11603
aI116275
asS'd'
I6005
sa(dp12010
S'a'
I2351
sS's'
I17
sS'm'
(lp12011
I1
aI1
aI7
aI9
aI21
aI27
aI5
aI199
aI181
aI521
aI303
aI1097
aI5427
aI8899
aI30325
aI55457
aI16189
asS'd'
I6006
sa(dp12012
S'a'
I2353
sS's'
I17
sS'm'
(lp12013
I1
aI3
aI3
aI7
aI19
aI41
aI3
aI205
aI279
aI223
aI971
aI633
aI2617
aI13191
aI10193
aI23375
aI62563
asS'd'
I6007
sa(dp12014
S'a'
I2363
sS's'
I17
sS'm'
(lp12015
I1
aI3
aI3
aI13
aI23
aI59
aI85
aI25
aI253
aI405
aI65
aI1625
aI4401
aI4679
aI14381
aI57833
aI30001
asS'd'
I6008
sa(dp12016
S'a'
I2378
sS's'
I17
sS'm'
(lp12017
I1
aI3
aI3
aI3
aI13
aI35
aI11
aI157
aI123
aI397
aI119
aI2513
aI1919
aI14583
aI5469
aI11463
aI94711
asS'd'
I6009
sa(dp12018
S'a'
I2383
sS's'
I17
sS'm'
(lp12019
I1
aI1
aI1
aI7
aI17
aI37
aI83
aI211
aI451
aI939
aI449
aI13
aI6671
aI1457
aI19855
aI15053
aI52327
asS'd'
I6010
sa(dp12020
S'a'
I2391
sS's'
I17
sS'm'
(lp12021
I1
aI1
aI5
aI3
aI9
aI57
aI39
aI183
aI331
aI451
aI1391
aI1865
aI7801
aI14293
aI29069
aI705
aI109497
asS'd'
I6011
sa(dp12022
S'a'
I2401
sS's'
I17
sS'm'
(lp12023
I1
aI3
aI7
aI7
aI23
aI21
aI85
aI81
aI255
aI9
aI1685
aI2879
aI6327
aI12675
aI31657
aI38877
aI74131
asS'd'
I6012
sa(dp12024
S'a'
I2408
sS's'
I17
sS'm'
(lp12025
I1
aI1
aI5
aI9
aI25
aI19
aI41
aI195
aI31
aI555
aI927
aI1445
aI593
aI11067
aI10819
aI17205
aI82037
asS'd'
I6013
sa(dp12026
S'a'
I2414
sS's'
I17
sS'm'
(lp12027
I1
aI3
aI1
aI13
aI1
aI35
aI29
aI71
aI323
aI705
aI53
aI3885
aI6223
aI1319
aI30853
aI59935
aI35949
asS'd'
I6014
sa(dp12028
S'a'
I2419
sS's'
I17
sS'm'
(lp12029
I1
aI1
aI7
aI3
aI27
aI63
aI67
aI31
aI149
aI61
aI1611
aI77
aI4271
aI3161
aI12493
aI38341
aI53837
asS'd'
I6015
sa(dp12030
S'a'
I2428
sS's'
I17
sS'm'
(lp12031
I1
aI1
aI1
aI15
aI27
aI53
aI31
aI249
aI429
aI925
aI1485
aI1855
aI4421
aI5703
aI10097
aI14827
aI36685
asS'd'
I6016
sa(dp12032
S'a'
I2441
sS's'
I17
sS'm'
(lp12033
I1
aI3
aI7
aI13
aI7
aI63
aI53
aI9
aI317
aI485
aI1679
aI3631
aI3745
aI5643
aI21615
aI45129
aI48027
asS'd'
I6017
sa(dp12034
S'a'
I2444
sS's'
I17
sS'm'
(lp12035
I1
aI1
aI1
aI1
aI17
aI43
aI19
aI163
aI441
aI847
aI937
aI959
aI6649
aI13071
aI1065
aI55193
aI129509
asS'd'
I6018
sa(dp12036
S'a'
I2461
sS's'
I17
sS'm'
(lp12037
I1
aI1
aI1
aI11
aI29
aI47
aI9
aI215
aI397
aI637
aI961
aI3139
aI2007
aI12603
aI27657
aI22825
aI72873
asS'd'
I6019
sa(dp12038
S'a'
I2480
sS's'
I17
sS'm'
(lp12039
I1
aI3
aI3
aI15
aI7
aI45
aI55
aI163
aI259
aI899
aI951
aI3245
aI4191
aI15813
aI20195
aI8361
aI54025
asS'd'
I6020
sa(dp12040
S'a'
I2483
sS's'
I17
sS'm'
(lp12041
I1
aI1
aI5
aI11
aI3
aI17
aI13
aI223
aI289
aI255
aI875
aI2937
aI1593
aI9729
aI21569
aI63199
aI83875
asS'd'
I6021
sa(dp12042
S'a'
I2486
sS's'
I17
sS'm'
(lp12043
I1
aI1
aI1
aI15
aI19
aI31
aI17
aI129
aI267
aI9
aI2015
aI3233
aI6799
aI12891
aI18473
aI37865
aI19547
asS'd'
I6022
sa(dp12044
S'a'
I2489
sS's'
I17
sS'm'
(lp12045
I1
aI1
aI5
aI5
aI5
aI29
aI81
aI37
aI357
aI539
aI1525
aI2839
aI8041
aI5569
aI4423
aI8907
aI35461
asS'd'
I6023
sa(dp12046
S'a'
I2490
sS's'
I17
sS'm'
(lp12047
I1
aI1
aI5
aI5
aI29
aI11
aI85
aI61
aI333
aI521
aI1111
aI3627
aI325
aI9805
aI17889
aI25655
aI39537
asS'd'
I6024
sa(dp12048
S'a'
I2518
sS's'
I17
sS'm'
(lp12049
I1
aI3
aI5
aI11
aI11
aI53
aI81
aI25
aI79
aI253
aI1963
aI287
aI7487
aI15045
aI21431
aI35417
aI102391
asS'd'
I6025
sa(dp12050
S'a'
I2527
sS's'
I17
sS'm'
(lp12051
I1
aI1
aI1
aI5
aI11
aI33
aI45
aI45
aI425
aI773
aI1817
aI4077
aI1471
aI11655
aI683
aI7115
aI92651
asS'd'
I6026
sa(dp12052
S'a'
I2540
sS's'
I17
sS'm'
(lp12053
I1
aI1
aI3
aI3
aI21
aI13
aI101
aI215
aI311
aI853
aI41
aI1007
aI5511
aI2581
aI25565
aI13155
aI117225
asS'd'
I6027
sa(dp12054
S'a'
I2558
sS's'
I17
sS'm'
(lp12055
I1
aI1
aI3
aI11
aI19
aI9
aI125
aI59
aI273
aI691
aI499
aI1547
aI567
aI10259
aI21963
aI48725
aI3601
asS'd'
I6028
sa(dp12056
S'a'
I2567
sS's'
I17
sS'm'
(lp12057
I1
aI1
aI3
aI7
aI27
aI31
aI39
aI125
aI317
aI625
aI1329
aI3947
aI3943
aI6889
aI2811
aI34055
aI1449
asS'd'
I6029
sa(dp12058
S'a'
I2571
sS's'
I17
sS'm'
(lp12059
I1
aI1
aI1
aI3
aI29
aI45
aI73
aI239
aI319
aI611
aI647
aI1839
aI5277
aI7807
aI3107
aI14683
aI20203
asS'd'
I6030
sa(dp12060
S'a'
I2574
sS's'
I17
sS'm'
(lp12061
I1
aI3
aI3
aI3
aI5
aI5
aI107
aI139
aI103
aI809
aI1343
aI4041
aI3273
aI1789
aI16205
aI47873
aI27803
asS'd'
I6031
sa(dp12062
S'a'
I2579
sS's'
I17
sS'm'
(lp12063
I1
aI3
aI1
aI9
aI21
aI23
aI13
aI131
aI105
aI741
aI1773
aI981
aI5633
aI14609
aI12281
aI50343
aI14317
asS'd'
I6032
sa(dp12064
S'a'
I2585
sS's'
I17
sS'm'
(lp12065
I1
aI1
aI1
aI5
aI11
aI5
aI125
aI171
aI109
aI555
aI159
aI905
aI691
aI12401
aI22817
aI41411
aI70113
asS'd'
I6033
sa(dp12066
S'a'
I2615
sS's'
I17
sS'm'
(lp12067
I1
aI3
aI3
aI9
aI31
aI37
aI109
aI231
aI59
aI615
aI799
aI319
aI2459
aI4521
aI8525
aI4827
aI22969
asS'd'
I6034
sa(dp12068
S'a'
I2639
sS's'
I17
sS'm'
(lp12069
I1
aI3
aI1
aI5
aI11
aI7
aI49
aI237
aI345
aI473
aI981
aI2073
aI6525
aI8805
aI13403
aI3659
aI69897
asS'd'
I6035
sa(dp12070
S'a'
I2641
sS's'
I17
sS'm'
(lp12071
I1
aI3
aI1
aI5
aI9
aI37
aI13
aI203
aI141
aI573
aI745
aI2613
aI5589
aI607
aI24483
aI38427
aI95775
asS'd'
I6036
sa(dp12072
S'a'
I2644
sS's'
I17
sS'm'
(lp12073
I1
aI1
aI3
aI1
aI23
aI61
aI75
aI57
aI299
aI191
aI805
aI2993
aI5175
aI12037
aI13649
aI58831
aI48791
asS'd'
I6037
sa(dp12074
S'a'
I2663
sS's'
I17
sS'm'
(lp12075
I1
aI3
aI7
aI13
aI31
aI57
aI13
aI219
aI185
aI717
aI1607
aI3785
aI4719
aI11583
aI29285
aI48207
aI92021
asS'd'
I6038
sa(dp12076
S'a'
I2667
sS's'
I17
sS'm'
(lp12077
I1
aI3
aI7
aI15
aI23
aI35
aI23
aI69
aI411
aI773
aI1549
aI1087
aI1685
aI15703
aI27193
aI62675
aI43505
asS'd'
I6039
sa(dp12078
S'a'
I2669
sS's'
I17
sS'm'
(lp12079
I1
aI1
aI5
aI3
aI25
aI19
aI97
aI75
aI493
aI549
aI1655
aI2881
aI4989
aI2765
aI4797
aI43143
aI113955
asS'd'
I6040
sa(dp12080
S'a'
I2672
sS's'
I17
sS'm'
(lp12081
I1
aI1
aI5
aI7
aI21
aI5
aI65
aI37
aI383
aI133
aI1907
aI3747
aI1525
aI5803
aI19977
aI50551
aI23157
asS'd'
I6041
sa(dp12082
S'a'
I2687
sS's'
I17
sS'm'
(lp12083
I1
aI1
aI1
aI11
aI15
aI61
aI59
aI109
aI489
aI901
aI1787
aI1611
aI6101
aI10653
aI3071
aI35643
aI56227
asS'd'
I6042
sa(dp12084
S'a'
I2700
sS's'
I17
sS'm'
(lp12085
I1
aI3
aI1
aI5
aI15
aI25
aI121
aI111
aI25
aI251
aI1467
aI1795
aI1631
aI13753
aI32391
aI14831
aI90739
asS'd'
I6043
sa(dp12086
S'a'
I2705
sS's'
I17
sS'm'
(lp12087
I1
aI1
aI1
aI13
aI23
aI55
aI119
aI147
aI45
aI871
aI1389
aI1929
aI1023
aI16131
aI10041
aI40055
aI23337
asS'd'
I6044
sa(dp12088
S'a'
I2724
sS's'
I17
sS'm'
(lp12089
I1
aI3
aI1
aI15
aI27
aI33
aI23
aI41
aI463
aI603
aI1633
aI3445
aI2007
aI5999
aI11175
aI18343
aI13159
asS'd'
I6045
sa(dp12090
S'a'
I2728
sS's'
I17
sS'm'
(lp12091
I1
aI3
aI1
aI9
aI17
aI15
aI107
aI63
aI493
aI411
aI293
aI3669
aI6143
aI3057
aI8253
aI25491
aI58907
asS'd'
I6046
sa(dp12092
S'a'
I2733
sS's'
I17
sS'm'
(lp12093
I1
aI3
aI5
aI11
aI1
aI43
aI5
aI117
aI127
aI813
aI1881
aI3711
aI2567
aI7819
aI5809
aI64471
aI104221
asS'd'
I6047
sa(dp12094
S'a'
I2741
sS's'
I17
sS'm'
(lp12095
I1
aI3
aI5
aI9
aI25
aI27
aI49
aI93
aI77
aI705
aI1773
aI1745
aI4605
aI16137
aI14621
aI62893
aI81637
asS'd'
I6048
sa(dp12096
S'a'
I2748
sS's'
I17
sS'm'
(lp12097
I1
aI3
aI1
aI15
aI9
aI29
aI41
aI101
aI291
aI763
aI1475
aI3185
aI3661
aI10351
aI26645
aI50375
aI59373
asS'd'
I6049
sa(dp12098
S'a'
I2751
sS's'
I17
sS'm'
(lp12099
I1
aI1
aI5
aI15
aI9
aI31
aI107
aI159
aI125
aI471
aI1023
aI2361
aI4805
aI8073
aI21563
aI14903
aI77801
asS'd'
I6050
sa(dp12100
S'a'
I2756
sS's'
I17
sS'm'
(lp12101
I1
aI3
aI7
aI1
aI27
aI17
aI75
aI129
aI71
aI697
aI551
aI1969
aI6597
aI13821
aI2605
aI61783
aI74791
asS'd'
I6051
sa(dp12102
S'a'
I2771
sS's'
I17
sS'm'
(lp12103
I1
aI1
aI7
aI15
aI17
aI27
aI49
aI47
aI59
aI47
aI1671
aI2535
aI1299
aI2387
aI24349
aI23661
aI91123
asS'd'
I6052
sa(dp12104
S'a'
I2774
sS's'
I17
sS'm'
(lp12105
I1
aI1
aI5
aI15
aI21
aI61
aI45
aI37
aI415
aI189
aI143
aI351
aI1815
aI3479
aI2399
aI56753
aI123893
asS'd'
I6053
sa(dp12106
S'a'
I2793
sS's'
I17
sS'm'
(lp12107
I1
aI1
aI3
aI7
aI7
aI19
aI93
aI249
aI335
aI305
aI1437
aI1329
aI2693
aI13201
aI9589
aI61513
aI115995
asS'd'
I6054
sa(dp12108
S'a'
I2796
sS's'
I17
sS'm'
(lp12109
I1
aI1
aI1
aI11
aI21
aI57
aI33
aI205
aI235
aI253
aI751
aI259
aI6029
aI9811
aI10231
aI36899
aI78035
asS'd'
I6055
sa(dp12110
S'a'
I2804
sS's'
I17
sS'm'
(lp12111
I1
aI1
aI1
aI11
aI13
aI25
aI115
aI195
aI111
aI913
aI1851
aI3283
aI6083
aI11717
aI2773
aI40727
aI493
asS'd'
I6056
sa(dp12112
S'a'
I2814
sS's'
I17
sS'm'
(lp12113
I1
aI3
aI3
aI9
aI9
aI17
aI83
aI137
aI465
aI671
aI1277
aI325
aI2767
aI12413
aI21977
aI47525
aI23041
asS'd'
I6057
sa(dp12114
S'a'
I2822
sS's'
I17
sS'm'
(lp12115
I1
aI1
aI1
aI11
aI15
aI47
aI65
aI219
aI271
aI197
aI297
aI3195
aI1325
aI9991
aI26385
aI46055
aI43151
asS'd'
I6058
sa(dp12116
S'a'
I2845
sS's'
I17
sS'm'
(lp12117
I1
aI1
aI1
aI13
aI31
aI21
aI39
aI89
aI127
aI629
aI367
aI2935
aI6259
aI6627
aI15691
aI55781
aI97251
asS'd'
I6059
sa(dp12118
S'a'
I2846
sS's'
I17
sS'm'
(lp12119
I1
aI1
aI7
aI13
aI11
aI45
aI65
aI75
aI211
aI785
aI1221
aI2087
aI7751
aI15619
aI25489
aI28195
aI69007
asS'd'
I6060
sa(dp12120
S'a'
I2850
sS's'
I17
sS'm'
(lp12121
I1
aI3
aI5
aI15
aI27
aI37
aI75
aI111
aI487
aI219
aI233
aI583
aI6433
aI15105
aI355
aI28331
aI21105
asS'd'
I6061
sa(dp12122
S'a'
I2855
sS's'
I17
sS'm'
(lp12123
I1
aI3
aI3
aI15
aI31
aI53
aI33
aI95
aI27
aI197
aI1727
aI1467
aI7115
aI15479
aI26873
aI31075
aI12793
asS'd'
I6062
sa(dp12124
S'a'
I2856
sS's'
I17
sS'm'
(lp12125
I1
aI3
aI7
aI1
aI19
aI3
aI19
aI105
aI225
aI599
aI737
aI107
aI7951
aI10193
aI31699
aI59207
aI85619
asS'd'
I6063
sa(dp12126
S'a'
I2867
sS's'
I17
sS'm'
(lp12127
I1
aI3
aI1
aI3
aI7
aI17
aI73
aI191
aI247
aI421
aI537
aI1473
aI189
aI4219
aI29993
aI25491
aI21189
asS'd'
I6064
sa(dp12128
S'a'
I2891
sS's'
I17
sS'm'
(lp12129
I1
aI3
aI7
aI7
aI13
aI21
aI33
aI95
aI147
aI699
aI943
aI2275
aI4093
aI6067
aI9063
aI25503
aI111085
asS'd'
I6065
sa(dp12130
S'a'
I2894
sS's'
I17
sS'm'
(lp12131
I1
aI1
aI7
aI9
aI13
aI47
aI123
aI121
aI347
aI467
aI225
aI957
aI2329
aI14075
aI29843
aI61753
aI97179
asS'd'
I6066
sa(dp12132
S'a'
I2902
sS's'
I17
sS'm'
(lp12133
I1
aI3
aI3
aI7
aI17
aI55
aI37
aI167
aI215
aI819
aI163
aI1747
aI4485
aI15991
aI28011
aI36351
aI106495
asS'd'
I6067
sa(dp12134
S'a'
I2908
sS's'
I17
sS'm'
(lp12135
I1
aI1
aI3
aI9
aI25
aI5
aI83
aI199
aI209
aI395
aI1757
aI1967
aI5739
aI2573
aI13989
aI32145
aI4847
asS'd'
I6068
sa(dp12136
S'a'
I2951
sS's'
I17
sS'm'
(lp12137
I1
aI3
aI3
aI13
aI11
aI21
aI25
aI223
aI239
aI569
aI1877
aI299
aI8089
aI3697
aI801
aI64775
aI26827
asS'd'
I6069
sa(dp12138
S'a'
I2970
sS's'
I17
sS'm'
(lp12139
I1
aI3
aI5
aI7
aI17
aI9
aI127
aI9
aI65
aI919
aI1073
aI2661
aI1337
aI10065
aI30099
aI30929
aI90067
asS'd'
I6070
sa(dp12140
S'a'
I2972
sS's'
I17
sS'm'
(lp12141
I1
aI3
aI1
aI13
aI25
aI41
aI35
aI251
aI279
aI351
aI111
aI3917
aI2815
aI7989
aI9895
aI54859
aI126355
asS'd'
I6071
sa(dp12142
S'a'
I2975
sS's'
I17
sS'm'
(lp12143
I1
aI1
aI3
aI7
aI17
aI61
aI13
aI73
aI335
aI831
aI703
aI37
aI2765
aI13169
aI12513
aI56301
aI13907
asS'd'
I6072
sa(dp12144
S'a'
I2976
sS's'
I17
sS'm'
(lp12145
I1
aI1
aI5
aI13
aI11
aI15
aI33
aI45
aI505
aI127
aI1723
aI17
aI4927
aI11453
aI28859
aI9671
aI80041
asS'd'
I6073
sa(dp12146
S'a'
I2981
sS's'
I17
sS'm'
(lp12147
I1
aI3
aI1
aI5
aI9
aI1
aI25
aI147
aI281
aI601
aI243
aI2687
aI5533
aI6725
aI11075
aI34807
aI24619
asS'd'
I6074
sa(dp12148
S'a'
I2986
sS's'
I17
sS'm'
(lp12149
I1
aI1
aI3
aI1
aI7
aI21
aI71
aI31
aI485
aI561
aI1361
aI1237
aI8171
aI15885
aI7941
aI4583
aI32851
asS'd'
I6075
sa(dp12150
S'a'
I2999
sS's'
I17
sS'm'
(lp12151
I1
aI3
aI7
aI1
aI5
aI35
aI95
aI155
aI283
aI959
aI577
aI1343
aI4269
aI13481
aI30819
aI40273
aI8711
asS'd'
I6076
sa(dp12152
S'a'
I3000
sS's'
I17
sS'm'
(lp12153
I1
aI3
aI7
aI3
aI1
aI53
aI77
aI45
aI215
aI537
aI1045
aI77
aI2791
aI3553
aI13273
aI23819
aI62263
asS'd'
I6077
sa(dp12154
S'a'
I3006
sS's'
I17
sS'm'
(lp12155
I1
aI3
aI1
aI15
aI29
aI59
aI7
aI145
aI85
aI3
aI251
aI2691
aI7547
aI11241
aI32295
aI24645
aI75739
asS'd'
I6078
sa(dp12156
S'a'
I3014
sS's'
I17
sS'm'
(lp12157
I1
aI1
aI5
aI9
aI19
aI9
aI39
aI163
aI303
aI233
aI2039
aI2027
aI7169
aI2773
aI28649
aI38317
aI66761
asS'd'
I6079
sa(dp12158
S'a'
I3028
sS's'
I17
sS'm'
(lp12159
I1
aI3
aI7
aI5
aI21
aI27
aI93
aI227
aI131
aI1019
aI1619
aI1497
aI4043
aI1131
aI25761
aI20173
aI99957
asS'd'
I6080
sa(dp12160
S'a'
I3031
sS's'
I17
sS'm'
(lp12161
I1
aI3
aI7
aI5
aI19
aI33
aI15
aI173
aI435
aI399
aI531
aI2001
aI3221
aI12627
aI10153
aI24421
aI61805
asS'd'
I6081
sa(dp12162
S'a'
I3035
sS's'
I17
sS'm'
(lp12163
I1
aI3
aI1
aI9
aI11
aI3
aI69
aI105
aI289
aI183
aI1103
aI831
aI2297
aI1613
aI18801
aI54395
aI54243
asS'd'
I6082
sa(dp12164
S'a'
I3037
sS's'
I17
sS'm'
(lp12165
I1
aI3
aI3
aI9
aI3
aI53
aI113
aI183
aI79
aI355
aI1629
aI1061
aI3713
aI4563
aI14365
aI43529
aI56073
asS'd'
I6083
sa(dp12166
S'a'
I3053
sS's'
I17
sS'm'
(lp12167
I1
aI3
aI7
aI11
aI31
aI39
aI107
aI139
aI187
aI873
aI225
aI33
aI4943
aI15837
aI225
aI6407
aI85967
asS'd'
I6084
sa(dp12168
S'a'
I3059
sS's'
I17
sS'm'
(lp12169
I1
aI3
aI1
aI11
aI17
aI47
aI93
aI233
aI119
aI699
aI1429
aI2845
aI2061
aI8887
aI20665
aI45497
aI33107
asS'd'
I6085
sa(dp12170
S'a'
I3065
sS's'
I17
sS'm'
(lp12171
I1
aI3
aI5
aI1
aI25
aI11
aI55
aI75
aI91
aI1009
aI1887
aI3167
aI515
aI15929
aI11659
aI57953
aI63401
asS'd'
I6086
sa(dp12172
S'a'
I3080
sS's'
I17
sS'm'
(lp12173
I1
aI1
aI3
aI15
aI27
aI59
aI103
aI53
aI353
aI553
aI2021
aI1543
aI2785
aI9373
aI14609
aI21213
aI19911
asS'd'
I6087
sa(dp12174
S'a'
I3091
sS's'
I17
sS'm'
(lp12175
I1
aI3
aI7
aI9
aI3
aI1
aI101
aI133
aI437
aI773
aI1399
aI1067
aI7419
aI1793
aI16589
aI3483
aI42065
asS'd'
I6088
sa(dp12176
S'a'
I3094
sS's'
I17
sS'm'
(lp12177
I1
aI3
aI7
aI1
aI25
aI57
aI127
aI113
aI65
aI577
aI1865
aI1527
aI6485
aI11273
aI15803
aI39625
aI75219
asS'd'
I6089
sa(dp12178
S'a'
I3109
sS's'
I17
sS'm'
(lp12179
I1
aI3
aI5
aI9
aI7
aI63
aI29
aI89
aI155
aI45
aI1029
aI2407
aI6783
aI4749
aI4849
aI26639
aI54059
asS'd'
I6090
sa(dp12180
S'a'
I3110
sS's'
I17
sS'm'
(lp12181
I1
aI3
aI7
aI9
aI25
aI13
aI113
aI41
aI267
aI767
aI1071
aI1689
aI269
aI14437
aI21255
aI39473
aI65771
asS'd'
I6091
sa(dp12182
S'a'
I3113
sS's'
I17
sS'm'
(lp12183
I1
aI3
aI1
aI15
aI5
aI3
aI77
aI43
aI391
aI763
aI59
aI1027
aI6263
aI3715
aI31061
aI43311
aI130725
asS'd'
I6092
sa(dp12184
S'a'
I3116
sS's'
I17
sS'm'
(lp12185
I1
aI3
aI7
aI7
aI21
aI51
aI127
aI71
aI229
aI171
aI397
aI1099
aI871
aI2717
aI1643
aI17363
aI125979
asS'd'
I6093
sa(dp12186
S'a'
I3136
sS's'
I17
sS'm'
(lp12187
I1
aI1
aI5
aI15
aI25
aI11
aI11
aI113
aI203
aI795
aI1703
aI3901
aI1113
aI12819
aI25345
aI46691
aI112313
asS'd'
I6094
sa(dp12188
S'a'
I3139
sS's'
I17
sS'm'
(lp12189
I1
aI3
aI7
aI5
aI1
aI59
aI91
aI81
aI325
aI483
aI595
aI1491
aI7455
aI6699
aI199
aI35597
aI59851
asS'd'
I6095
sa(dp12190
S'a'
I3141
sS's'
I17
sS'm'
(lp12191
I1
aI3
aI5
aI1
aI3
aI33
aI43
aI195
aI201
aI575
aI1395
aI1305
aI7001
aI2023
aI22419
aI15233
aI120355
asS'd'
I6096
sa(dp12192
S'a'
I3154
sS's'
I17
sS'm'
(lp12193
I1
aI1
aI3
aI3
aI15
aI37
aI81
aI59
aI87
aI675
aI199
aI3231
aI4473
aI5023
aI16753
aI51475
aI102113
asS'd'
I6097
sa(dp12194
S'a'
I3160
sS's'
I17
sS'm'
(lp12195
I1
aI1
aI7
aI9
aI13
aI39
aI65
aI9
aI51
aI565
aI1171
aI119
aI7875
aI12149
aI6565
aI56849
aI123235
asS'd'
I6098
sa(dp12196
S'a'
I3169
sS's'
I17
sS'm'
(lp12197
I1
aI3
aI3
aI7
aI15
aI45
aI53
aI93
aI111
aI533
aI1849
aI643
aI2265
aI10241
aI24741
aI11559
aI74333
asS'd'
I6099
sa(dp12198
S'a'
I3182
sS's'
I17
sS'm'
(lp12199
I1
aI3
aI1
aI1
aI11
aI61
aI75
aI51
aI5
aI199
aI535
aI279
aI5821
aI6005
aI2907
aI32521
aI74121
asS'd'
I6100
sa(dp12200
S'a'
I3187
sS's'
I17
sS'm'
(lp12201
I1
aI1
aI3
aI15
aI3
aI21
aI29
aI193
aI71
aI993
aI1719
aI1865
aI6135
aI7683
aI12171
aI29275
aI14539
asS'd'
I6101
sa(dp12202
S'a'
I3189
sS's'
I17
sS'm'
(lp12203
I1
aI1
aI1
aI7
aI7
aI13
aI1
aI61
aI315
aI431
aI1145
aI2067
aI5745
aI1641
aI1047
aI55111
aI129477
asS'd'
I6102
sa(dp12204
S'a'
I3190
sS's'
I17
sS'm'
(lp12205
I1
aI1
aI5
aI1
aI21
aI43
aI115
aI193
aI153
aI573
aI1181
aI3947
aI7809
aI11317
aI30649
aI56891
aI47741
asS'd'
I6103
sa(dp12206
S'a'
I3203
sS's'
I17
sS'm'
(lp12207
I1
aI1
aI5
aI7
aI19
aI15
aI61
aI239
aI109
aI683
aI395
aI2869
aI3103
aI1531
aI12019
aI45159
aI37525
asS'd'
I6104
sa(dp12208
S'a'
I3217
sS's'
I17
sS'm'
(lp12209
I1
aI1
aI5
aI7
aI29
aI55
aI45
aI7
aI353
aI659
aI591
aI3371
aI5777
aI8475
aI2743
aI47483
aI11983
asS'd'
I6105
sa(dp12210
S'a'
I3229
sS's'
I17
sS'm'
(lp12211
I1
aI3
aI1
aI3
aI13
aI17
aI39
aI195
aI43
aI5
aI1749
aI2559
aI5843
aI8719
aI21421
aI58511
aI105637
asS'd'
I6106
sa(dp12212
S'a'
I3236
sS's'
I17
sS'm'
(lp12213
I1
aI3
aI5
aI5
aI5
aI21
aI29
aI63
aI387
aI301
aI567
aI3325
aI2109
aI403
aI23053
aI24851
aI14493
asS'd'
I6107
sa(dp12214
S'a'
I3248
sS's'
I17
sS'm'
(lp12215
I1
aI1
aI3
aI3
aI17
aI57
aI107
aI131
aI85
aI855
aI1101
aI3199
aI7159
aI14739
aI4197
aI27943
aI113009
asS'd'
I6108
sa(dp12216
S'a'
I3257
sS's'
I17
sS'm'
(lp12217
I1
aI1
aI3
aI11
aI1
aI61
aI31
aI79
aI33
aI123
aI1509
aI507
aI6679
aI2279
aI8465
aI37279
aI17553
asS'd'
I6109
sa(dp12218
S'a'
I3278
sS's'
I17
sS'm'
(lp12219
I1
aI3
aI1
aI15
aI7
aI33
aI11
aI71
aI217
aI609
aI1661
aI3437
aI5497
aI13365
aI6247
aI649
aI26407
asS'd'
I6110
sa(dp12220
S'a'
I3283
sS's'
I17
sS'm'
(lp12221
I1
aI1
aI3
aI1
aI19
aI45
aI49
aI125
aI5
aI455
aI1669
aI4083
aI253
aI10101
aI27327
aI16401
aI120399
asS'd'
I6111
sa(dp12222
S'a'
I3289
sS's'
I17
sS'm'
(lp12223
I1
aI3
aI1
aI1
aI27
aI19
aI117
aI137
aI261
aI341
aI1697
aI457
aI7553
aI12169
aI30049
aI49281
aI36937
asS'd'
I6112
sa(dp12224
S'a'
I3292
sS's'
I17
sS'm'
(lp12225
I1
aI1
aI1
aI3
aI9
aI49
aI33
aI13
aI461
aI545
aI1537
aI2623
aI883
aI10921
aI5583
aI58997
aI114183
asS'd'
I6113
sa(dp12226
S'a'
I3302
sS's'
I17
sS'm'
(lp12227
I1
aI1
aI7
aI9
aI29
aI53
aI29
aI165
aI205
aI989
aI1347
aI2343
aI7505
aI7609
aI18503
aI51677
aI105993
asS'd'
I6114
sa(dp12228
S'a'
I3316
sS's'
I17
sS'm'
(lp12229
I1
aI1
aI1
aI13
aI1
aI29
aI59
aI121
aI297
aI659
aI1965
aI1765
aI5255
aI10971
aI32613
aI18763
aI41983
asS'd'
I6115
sa(dp12230
S'a'
I3328
sS's'
I17
sS'm'
(lp12231
I1
aI3
aI7
aI11
aI21
aI41
aI19
aI47
aI125
aI485
aI475
aI2745
aI4075
aI8101
aI31227
aI4679
aI115473
asS'd'
I6116
sa(dp12232
S'a'
I3333
sS's'
I17
sS'm'
(lp12233
I1
aI3
aI3
aI7
aI21
aI23
aI55
aI65
aI223
aI1001
aI317
aI1459
aI183
aI5139
aI26553
aI41471
aI116373
asS'd'
I6117
sa(dp12234
S'a'
I3337
sS's'
I17
sS'm'
(lp12235
I1
aI1
aI7
aI3
aI1
aI9
aI29
aI139
aI343
aI913
aI1993
aI3139
aI3791
aI5869
aI6057
aI23863
aI35737
asS'd'
I6118
sa(dp12236
S'a'
I3340
sS's'
I17
sS'm'
(lp12237
I1
aI3
aI3
aI3
aI7
aI21
aI77
aI197
aI239
aI467
aI35
aI591
aI1061
aI3417
aI31811
aI38825
aI124981
asS'd'
I6119
sa(dp12238
S'a'
I3368
sS's'
I17
sS'm'
(lp12239
I1
aI3
aI3
aI1
aI21
aI29
aI5
aI213
aI417
aI111
aI1681
aI1409
aI2899
aI16233
aI1053
aI51235
aI87767
asS'd'
I6120
sa(dp12240
S'a'
I3371
sS's'
I17
sS'm'
(lp12241
I1
aI1
aI5
aI3
aI13
aI47
aI61
aI203
aI223
aI73
aI1947
aI3613
aI5885
aI13567
aI7593
aI34329
aI68597
asS'd'
I6121
sa(dp12242
S'a'
I3376
sS's'
I17
sS'm'
(lp12243
I1
aI3
aI1
aI1
aI17
aI9
aI11
aI187
aI361
aI973
aI781
aI1835
aI1539
aI12917
aI21725
aI48279
aI115037
asS'd'
I6122
sa(dp12244
S'a'
I3385
sS's'
I17
sS'm'
(lp12245
I1
aI3
aI1
aI1
aI9
aI25
aI117
aI157
aI433
aI395
aI403
aI2183
aI3327
aI5427
aI7505
aI2673
aI77137
asS'd'
I6123
sa(dp12246
S'a'
I3386
sS's'
I17
sS'm'
(lp12247
I1
aI1
aI7
aI15
aI31
aI15
aI27
aI155
aI441
aI837
aI1877
aI3829
aI5139
aI16331
aI31183
aI15803
aI95699
asS'd'
I6124
sa(dp12248
S'a'
I3393
sS's'
I17
sS'm'
(lp12249
I1
aI1
aI7
aI15
aI5
aI51
aI77
aI179
aI289
aI727
aI1763
aI2529
aI6715
aI3967
aI29267
aI27293
aI67953
asS'd'
I6125
sa(dp12250
S'a'
I3399
sS's'
I17
sS'm'
(lp12251
I1
aI3
aI7
aI13
aI7
aI3
aI3
aI17
aI311
aI547
aI1465
aI1413
aI3937
aI2725
aI24523
aI12321
aI109763
asS'd'
I6126
sa(dp12252
S'a'
I3405
sS's'
I17
sS'm'
(lp12253
I1
aI3
aI5
aI15
aI9
aI5
aI87
aI135
aI281
aI97
aI2021
aI1903
aI8007
aI10321
aI27989
aI18993
aI110407
asS'd'
I6127
sa(dp12254
S'a'
I3414
sS's'
I17
sS'm'
(lp12255
I1
aI1
aI1
aI13
aI25
aI61
aI89
aI107
aI233
aI823
aI1375
aI3531
aI1757
aI1577
aI29457
aI1461
aI17217
asS'd'
I6128
sa(dp12256
S'a'
I3433
sS's'
I17
sS'm'
(lp12257
I1
aI1
aI1
aI13
aI17
aI17
aI27
aI193
aI485
aI759
aI145
aI3943
aI4183
aI14119
aI11217
aI3793
aI1935
asS'd'
I6129
sa(dp12258
S'a'
I3436
sS's'
I17
sS'm'
(lp12259
I1
aI1
aI1
aI3
aI13
aI31
aI101
aI227
aI311
aI363
aI1925
aI1525
aI5275
aI2385
aI15093
aI48769
aI121189
asS'd'
I6130
sa(dp12260
S'a'
I3448
sS's'
I17
sS'm'
(lp12261
I1
aI1
aI5
aI13
aI11
aI61
aI89
aI141
aI117
aI229
aI417
aI3935
aI7249
aI13869
aI30591
aI62763
aI67521
asS'd'
I6131
sa(dp12262
S'a'
I3467
sS's'
I17
sS'm'
(lp12263
I1
aI1
aI3
aI15
aI7
aI59
aI105
aI239
aI453
aI221
aI1101
aI395
aI2031
aI8941
aI23155
aI7077
aI125593
asS'd'
I6132
sa(dp12264
S'a'
I3469
sS's'
I17
sS'm'
(lp12265
I1
aI1
aI1
aI11
aI7
aI55
aI99
aI31
aI305
aI371
aI1035
aI577
aI4473
aI577
aI371
aI46093
aI69157
asS'd'
I6133
sa(dp12266
S'a'
I3472
sS's'
I17
sS'm'
(lp12267
I1
aI3
aI1
aI9
aI9
aI33
aI35
aI245
aI95
aI47
aI1623
aI2965
aI6849
aI7269
aI5321
aI31641
aI73321
asS'd'
I6134
sa(dp12268
S'a'
I3477
sS's'
I17
sS'm'
(lp12269
I1
aI1
aI1
aI15
aI21
aI61
aI65
aI65
aI159
aI151
aI625
aI2281
aI2993
aI1311
aI29757
aI24703
aI71029
asS'd'
I6135
sa(dp12270
S'a'
I3484
sS's'
I17
sS'm'
(lp12271
I1
aI3
aI5
aI15
aI29
aI59
aI29
aI69
aI351
aI901
aI631
aI3501
aI7031
aI703
aI20805
aI36437
aI94931
asS'd'
I6136
sa(dp12272
S'a'
I3494
sS's'
I17
sS'm'
(lp12273
I1
aI3
aI7
aI1
aI21
aI11
aI19
aI125
aI237
aI807
aI1651
aI2389
aI7347
aI11759
aI27151
aI38669
aI965
asS'd'
I6137
sa(dp12274
S'a'
I3505
sS's'
I17
sS'm'
(lp12275
I1
aI1
aI5
aI1
aI15
aI41
aI1
aI105
aI89
aI127
aI895
aI29
aI2339
aI15951
aI18633
aI2781
aI67269
asS'd'
I6138
sa(dp12276
S'a'
I3515
sS's'
I17
sS'm'
(lp12277
I1
aI1
aI5
aI15
aI25
aI7
aI3
aI33
aI375
aI447
aI203
aI2579
aI6145
aI14015
aI9939
aI52777
aI123181
asS'd'
I6139
sa(dp12278
S'a'
I3523
sS's'
I17
sS'm'
(lp12279
I1
aI3
aI1
aI15
aI29
aI7
aI7
aI27
aI451
aI869
aI107
aI2457
aI5557
aI11601
aI28957
aI36181
aI41419
asS'd'
I6140
sa(dp12280
S'a'
I3530
sS's'
I17
sS'm'
(lp12281
I1
aI1
aI1
aI7
aI1
aI57
aI33
aI213
aI329
aI763
aI815
aI169
aI623
aI155
aI20529
aI20603
aI73311
asS'd'
I6141
sa(dp12282
S'a'
I3543
sS's'
I17
sS'm'
(lp12283
I1
aI3
aI5
aI7
aI25
aI21
aI7
aI217
aI159
aI89
aI1373
aI1735
aI705
aI4093
aI13083
aI3855
aI55875
asS'd'
I6142
sa(dp12284
S'a'
I3559
sS's'
I17
sS'm'
(lp12285
I1
aI3
aI1
aI1
aI29
aI33
aI105
aI127
aI95
aI543
aI235
aI67
aI691
aI5015
aI22139
aI18251
aI89945
asS'd'
I6143
sa(dp12286
S'a'
I3568
sS's'
I17
sS'm'
(lp12287
I1
aI1
aI3
aI11
aI27
aI53
aI105
aI83
aI337
aI331
aI1571
aI1145
aI745
aI1845
aI17881
aI17697
aI88139
asS'd'
I6144
sa(dp12288
S'a'
I3577
sS's'
I17
sS'm'
(lp12289
I1
aI3
aI7
aI15
aI19
aI37
aI119
aI35
aI35
aI463
aI1925
aI1665
aI673
aI12193
aI12137
aI62371
aI10957
asS'd'
I6145
sa(dp12290
S'a'
I3578
sS's'
I17
sS'm'
(lp12291
I1
aI3
aI3
aI3
aI19
aI21
aI113
aI29
aI459
aI467
aI623
aI2661
aI857
aI16265
aI27509
aI46555
aI18867
asS'd'
I6146
sa(dp12292
S'a'
I3594
sS's'
I17
sS'm'
(lp12293
I1
aI3
aI7
aI5
aI17
aI49
aI123
aI41
aI85
aI673
aI41
aI1871
aI7649
aI8687
aI28269
aI64423
aI93675
asS'd'
I6147
sa(dp12294
S'a'
I3601
sS's'
I17
sS'm'
(lp12295
I1
aI3
aI3
aI3
aI7
aI23
aI101
aI171
aI181
aI527
aI65
aI2387
aI6629
aI6089
aI17387
aI46551
aI36143
asS'd'
I6148
sa(dp12296
S'a'
I3607
sS's'
I17
sS'm'
(lp12297
I1
aI1
aI5
aI1
aI13
aI51
aI21
aI251
aI139
aI429
aI1993
aI3767
aI1089
aI5459
aI19407
aI41747
aI41033
asS'd'
I6149
sa(dp12298
S'a'
I3608
sS's'
I17
sS'm'
(lp12299
I1
aI1
aI1
aI11
aI15
aI9
aI81
aI91
aI73
aI969
aI1513
aI2067
aI7959
aI2605
aI26641
aI37631
aI124571
asS'd'
I6150
sa(dp12300
S'a'
I3620
sS's'
I17
sS'm'
(lp12301
I1
aI1
aI3
aI15
aI29
aI15
aI5
aI57
aI247
aI901
aI527
aI3325
aI5859
aI11299
aI9871
aI63947
aI125247
asS'd'
I6151
sa(dp12302
S'a'
I3629
sS's'
I17
sS'm'
(lp12303
I1
aI3
aI1
aI5
aI1
aI35
aI75
aI21
aI307
aI43
aI1111
aI3299
aI1647
aI3585
aI31045
aI18217
aI95169
asS'd'
I6152
sa(dp12304
S'a'
I3644
sS's'
I17
sS'm'
(lp12305
I1
aI3
aI1
aI7
aI23
aI35
aI11
aI103
aI3
aI461
aI1915
aI4019
aI453
aI13111
aI26941
aI43091
aI22917
asS'd'
I6153
sa(dp12306
S'a'
I3656
sS's'
I17
sS'm'
(lp12307
I1
aI1
aI5
aI5
aI1
aI61
aI121
aI167
aI475
aI5
aI1749
aI887
aI2237
aI5055
aI7077
aI29453
aI17691
asS'd'
I6154
sa(dp12308
S'a'
I3664
sS's'
I17
sS'm'
(lp12309
I1
aI3
aI3
aI15
aI15
aI15
aI9
aI15
aI171
aI787
aI1965
aI577
aI4507
aI7325
aI20901
aI8557
aI111909
asS'd'
I6155
sa(dp12310
S'a'
I3670
sS's'
I17
sS'm'
(lp12311
I1
aI3
aI5
aI1
aI27
aI15
aI123
aI141
aI63
aI55
aI599
aI4095
aI1245
aI13919
aI27485
aI49977
aI74551
asS'd'
I6156
sa(dp12312
S'a'
I3680
sS's'
I17
sS'm'
(lp12313
I1
aI3
aI5
aI9
aI21
aI61
aI79
aI119
aI7
aI573
aI1923
aI2775
aI3127
aI12689
aI12135
aI53429
aI130163
asS'd'
I6157
sa(dp12314
S'a'
I3685
sS's'
I17
sS'm'
(lp12315
I1
aI3
aI3
aI13
aI27
aI41
aI67
aI249
aI447
aI277
aI311
aI775
aI8187
aI10161
aI12953
aI22885
aI121247
asS'd'
I6158
sa(dp12316
S'a'
I3686
sS's'
I17
sS'm'
(lp12317
I1
aI3
aI5
aI9
aI21
aI55
aI115
aI65
aI45
aI395
aI481
aI2063
aI6493
aI4199
aI19219
aI27119
aI62255
asS'd'
I6159
sa(dp12318
S'a'
I3695
sS's'
I17
sS'm'
(lp12319
I1
aI1
aI3
aI13
aI7
aI41
aI3
aI127
aI383
aI923
aI1725
aI1033
aI7731
aI11971
aI3089
aI46459
aI98369
asS'd'
I6160
sa(dp12320
S'a'
I3698
sS's'
I17
sS'm'
(lp12321
I1
aI1
aI3
aI11
aI13
aI39
aI39
aI149
aI309
aI311
aI1491
aI807
aI2109
aI363
aI14637
aI65429
aI124731
asS'd'
I6161
sa(dp12322
S'a'
I3703
sS's'
I17
sS'm'
(lp12323
I1
aI1
aI7
aI13
aI13
aI35
aI67
aI81
aI493
aI859
aI1177
aI237
aI4605
aI15319
aI16669
aI16661
aI21385
asS'd'
I6162
sa(dp12324
S'a'
I3710
sS's'
I17
sS'm'
(lp12325
I1
aI1
aI3
aI7
aI7
aI39
aI57
aI103
aI239
aI753
aI221
aI1611
aI1557
aI13317
aI27453
aI10245
aI33839
asS'd'
I6163
sa(dp12326
S'a'
I3714
sS's'
I17
sS'm'
(lp12327
I1
aI1
aI5
aI13
aI27
aI53
aI97
aI41
aI123
aI253
aI535
aI1839
aI5827
aI7587
aI1261
aI20313
aI65961
asS'd'
I6164
sa(dp12328
S'a'
I3726
sS's'
I17
sS'm'
(lp12329
I1
aI1
aI7
aI1
aI11
aI47
aI93
aI135
aI223
aI591
aI1087
aI3329
aI3293
aI14207
aI6187
aI54789
aI23781
asS'd'
I6165
sa(dp12330
S'a'
I3731
sS's'
I17
sS'm'
(lp12331
I1
aI3
aI7
aI7
aI25
aI21
aI97
aI105
aI269
aI515
aI1805
aI3711
aI3295
aI7307
aI21065
aI65205
aI116969
asS'd'
I6166
sa(dp12332
S'a'
I3733
sS's'
I17
sS'm'
(lp12333
I1
aI3
aI1
aI11
aI25
aI37
aI21
aI89
aI109
aI581
aI1055
aI2393
aI1291
aI1115
aI25545
aI36383
aI93605
asS'd'
I6167
sa(dp12334
S'a'
I3737
sS's'
I17
sS'm'
(lp12335
I1
aI3
aI7
aI1
aI27
aI13
aI113
aI11
aI395
aI473
aI943
aI4045
aI5507
aI15051
aI25203
aI2971
aI31961
asS'd'
I6168
sa(dp12336
S'a'
I3756
sS's'
I17
sS'm'
(lp12337
I1
aI1
aI5
aI5
aI27
aI35
aI57
aI219
aI67
aI949
aI659
aI203
aI5235
aI6509
aI13731
aI61533
aI54963
asS'd'
I6169
sa(dp12338
S'a'
I3759
sS's'
I17
sS'm'
(lp12339
I1
aI3
aI1
aI1
aI15
aI39
aI85
aI13
aI347
aI99
aI25
aI3595
aI3081
aI13617
aI14373
aI58909
aI102181
asS'd'
I6170
sa(dp12340
S'a'
I3767
sS's'
I17
sS'm'
(lp12341
I1
aI1
aI7
aI13
aI3
aI25
aI97
aI91
aI287
aI389
aI665
aI2981
aI2301
aI12625
aI4495
aI57489
aI68677
asS'd'
I6171
sa(dp12342
S'a'
I3776
sS's'
I17
sS'm'
(lp12343
I1
aI1
aI5
aI1
aI15
aI57
aI77
aI55
aI299
aI713
aI1457
aI3699
aI2807
aI5549
aI467
aI47367
aI8163
asS'd'
I6172
sa(dp12344
S'a'
I3785
sS's'
I17
sS'm'
(lp12345
I1
aI1
aI7
aI3
aI23
aI45
aI91
aI251
aI501
aI193
aI1121
aI2359
aI4781
aI12797
aI13713
aI55171
aI927
asS'd'
I6173
sa(dp12346
S'a'
I3793
sS's'
I17
sS'm'
(lp12347
I1
aI3
aI3
aI7
aI7
aI31
aI87
aI163
aI249
aI163
aI937
aI1293
aI4827
aI10299
aI31935
aI58787
aI80589
asS'd'
I6174
sa(dp12348
S'a'
I3812
sS's'
I17
sS'm'
(lp12349
I1
aI3
aI1
aI9
aI7
aI1
aI73
aI65
aI475
aI791
aI1429
aI3319
aI7149
aI433
aI10373
aI44061
aI121195
asS'd'
I6175
sa(dp12350
S'a'
I3815
sS's'
I17
sS'm'
(lp12351
I1
aI1
aI5
aI9
aI9
aI61
aI27
aI249
aI435
aI437
aI1329
aI2163
aI5859
aI13663
aI623
aI55569
aI94283
asS'd'
I6176
sa(dp12352
S'a'
I3824
sS's'
I17
sS'm'
(lp12353
I1
aI3
aI7
aI11
aI1
aI29
aI117
aI195
aI399
aI999
aI1705
aI1325
aI6043
aI9823
aI27335
aI30377
aI16627
asS'd'
I6177
sa(dp12354
S'a'
I3844
sS's'
I17
sS'm'
(lp12355
I1
aI1
aI1
aI15
aI5
aI11
aI63
aI185
aI15
aI741
aI1061
aI2961
aI3455
aI5
aI26587
aI54081
aI18107
asS'd'
I6178
sa(dp12356
S'a'
I3859
sS's'
I17
sS'm'
(lp12357
I1
aI1
aI5
aI7
aI29
aI57
aI17
aI203
aI501
aI177
aI49
aI2773
aI8069
aI12513
aI14437
aI64489
aI58661
asS'd'
I6179
sa(dp12358
S'a'
I3866
sS's'
I17
sS'm'
(lp12359
I1
aI3
aI3
aI9
aI11
aI23
aI121
aI3
aI415
aI447
aI1773
aI135
aI5901
aI4951
aI2683
aI437
aI126251
asS'd'
I6180
sa(dp12360
S'a'
I3872
sS's'
I17
sS'm'
(lp12361
I1
aI3
aI3
aI1
aI7
aI23
aI17
aI23
aI115
aI591
aI1075
aI3133
aI49
aI15183
aI10615
aI37857
aI122609
asS'd'
I6181
sa(dp12362
S'a'
I3884
sS's'
I17
sS'm'
(lp12363
I1
aI1
aI3
aI3
aI13
aI49
aI63
aI37
aI275
aI763
aI1135
aI2913
aI1563
aI11037
aI6693
aI18799
aI32089
asS'd'
I6182
sa(dp12364
S'a'
I3889
sS's'
I17
sS'm'
(lp12365
I1
aI3
aI5
aI11
aI7
aI29
aI59
aI45
aI227
aI941
aI1947
aI2733
aI797
aI10485
aI7071
aI14741
aI11451
asS'd'
I6183
sa(dp12366
S'a'
I3899
sS's'
I17
sS'm'
(lp12367
I1
aI1
aI1
aI9
aI21
aI19
aI77
aI97
aI75
aI991
aI187
aI1003
aI5619
aI11013
aI3931
aI19907
aI79723
asS'd'
I6184
sa(dp12368
S'a'
I3902
sS's'
I17
sS'm'
(lp12369
I1
aI1
aI7
aI13
aI1
aI57
aI61
aI177
aI443
aI227
aI1347
aI2665
aI2011
aI12329
aI14137
aI37795
aI63331
asS'd'
I6185
sa(dp12370
S'a'
I3909
sS's'
I17
sS'm'
(lp12371
I1
aI3
aI3
aI9
aI31
aI59
aI87
aI93
aI485
aI635
aI901
aI1845
aI6153
aI10797
aI1289
aI8989
aI41717
asS'd'
I6186
sa(dp12372
S'a'
I3913
sS's'
I17
sS'm'
(lp12373
I1
aI1
aI1
aI1
aI3
aI7
aI85
aI17
aI67
aI309
aI1891
aI435
aI303
aI8011
aI32127
aI54309
aI21457
asS'd'
I6187
sa(dp12374
S'a'
I3933
sS's'
I17
sS'm'
(lp12375
I1
aI3
aI7
aI1
aI29
aI27
aI41
aI239
aI293
aI717
aI1331
aI917
aI6145
aI7131
aI28199
aI35093
aI103683
asS'd'
I6188
sa(dp12376
S'a'
I3938
sS's'
I17
sS'm'
(lp12377
I1
aI3
aI7
aI3
aI21
aI63
aI65
aI233
aI257
aI789
aI1095
aI505
aI4557
aI16259
aI7397
aI24815
aI89529
asS'd'
I6189
sa(dp12378
S'a'
I3949
sS's'
I17
sS'm'
(lp12379
I1
aI3
aI3
aI11
aI29
aI41
aI55
aI17
aI335
aI715
aI779
aI2121
aI6393
aI8887
aI32753
aI45647
aI82665
asS'd'
I6190
sa(dp12380
S'a'
I3952
sS's'
I17
sS'm'
(lp12381
I1
aI1
aI1
aI11
aI27
aI47
aI71
aI13
aI141
aI283
aI967
aI3359
aI4309
aI6661
aI20481
aI23175
aI50835
asS'd'
I6191
sa(dp12382
S'a'
I3980
sS's'
I17
sS'm'
(lp12383
I1
aI3
aI3
aI7
aI3
aI25
aI19
aI241
aI409
aI573
aI1565
aI3355
aI1307
aI12205
aI18017
aI8271
aI117007
asS'd'
I6192
sa(dp12384
S'a'
I3991
sS's'
I17
sS'm'
(lp12385
I1
aI3
aI3
aI9
aI21
aI39
aI21
aI253
aI439
aI963
aI341
aI3637
aI2275
aI1845
aI11015
aI481
aI83369
asS'd'
I6193
sa(dp12386
S'a'
I3992
sS's'
I17
sS'm'
(lp12387
I1
aI3
aI7
aI9
aI31
aI29
aI29
aI163
aI111
aI983
aI571
aI713
aI2621
aI11569
aI13341
aI28341
aI130381
asS'd'
I6194
sa(dp12388
S'a'
I4002
sS's'
I17
sS'm'
(lp12389
I1
aI3
aI7
aI7
aI11
aI35
aI89
aI49
aI81
aI115
aI113
aI1857
aI3527
aI14819
aI6909
aI14659
aI23557
asS'd'
I6195
sa(dp12390
S'a'
I4008
sS's'
I17
sS'm'
(lp12391
I1
aI3
aI3
aI15
aI29
aI41
aI85
aI241
aI317
aI737
aI213
aI1667
aI5789
aI16321
aI13991
aI36165
aI124151
asS'd'
I6196
sa(dp12392
S'a'
I4011
sS's'
I17
sS'm'
(lp12393
I1
aI3
aI1
aI3
aI31
aI1
aI75
aI99
aI495
aI241
aI1499
aI1535
aI2033
aI2135
aI6699
aI58893
aI37031
asS'd'
I6197
sa(dp12394
S'a'
I4016
sS's'
I17
sS'm'
(lp12395
I1
aI1
aI7
aI9
aI25
aI15
aI101
aI23
aI477
aI563
aI1691
aI2655
aI2321
aI2323
aI4255
aI22055
aI99661
asS'd'
I6198
sa(dp12396
S'a'
I4034
sS's'
I17
sS'm'
(lp12397
I1
aI3
aI7
aI5
aI7
aI7
aI49
aI221
aI51
aI83
aI279
aI2205
aI2939
aI2119
aI14073
aI32839
aI108075
asS'd'
I6199
sa(dp12398
S'a'
I4036
sS's'
I17
sS'm'
(lp12399
I1
aI3
aI5
aI11
aI17
aI39
aI3
aI127
aI87
aI501
aI799
aI401
aI4439
aI9895
aI13017
aI64975
aI67177
asS'd'
I6200
sa(dp12400
S'a'
I4063
sS's'
I17
sS'm'
(lp12401
I1
aI3
aI3
aI9
aI17
aI41
aI59
aI95
aI283
aI309
aI83
aI1293
aI6385
aI5783
aI30115
aI33997
aI12531
asS'd'
I6201
sa(dp12402
S'a'
I4067
sS's'
I17
sS'm'
(lp12403
I1
aI3
aI5
aI3
aI7
aI31
aI69
aI171
aI225
aI409
aI1237
aI3343
aI835
aI8039
aI16723
aI37203
aI129047
asS'd'
I6202
sa(dp12404
S'a'
I4073
sS's'
I17
sS'm'
(lp12405
I1
aI3
aI3
aI15
aI17
aI23
aI107
aI1
aI105
aI135
aI1245
aI993
aI4101
aI7325
aI7425
aI17379
aI98121
asS'd'
I6203
sa(dp12406
S'a'
I4082
sS's'
I17
sS'm'
(lp12407
I1
aI1
aI7
aI9
aI27
aI5
aI67
aI111
aI75
aI531
aI243
aI2239
aI2527
aI4513
aI27059
aI40533
aI88169
asS'd'
I6204
sa(dp12408
S'a'
I4091
sS's'
I17
sS'm'
(lp12409
I1
aI3
aI5
aI7
aI21
aI63
aI57
aI15
aI75
aI679
aI1729
aI1845
aI6259
aI8531
aI18691
aI49321
aI101599
asS'd'
I6205
sa(dp12410
S'a'
I4093
sS's'
I17
sS'm'
(lp12411
I1
aI1
aI5
aI9
aI3
aI35
aI7
aI201
aI351
aI885
aI669
aI2339
aI5009
aI279
aI26469
aI54597
aI67933
asS'd'
I6206
sa(dp12412
S'a'
I4101
sS's'
I17
sS'm'
(lp12413
I1
aI3
aI5
aI13
aI27
aI5
aI85
aI161
aI141
aI733
aI1017
aI2021
aI6951
aI15595
aI21817
aI17243
aI88607
asS'd'
I6207
sa(dp12414
S'a'
I4113
sS's'
I17
sS'm'
(lp12415
I1
aI3
aI5
aI1
aI11
aI31
aI117
aI97
aI175
aI629
aI995
aI1207
aI2941
aI5825
aI5319
aI48191
aI9505
asS'd'
I6208
sa(dp12416
S'a'
I4120
sS's'
I17
sS'm'
(lp12417
I1
aI3
aI3
aI7
aI25
aI39
aI45
aI79
aI21
aI607
aI1593
aI1749
aI7951
aI10425
aI17491
aI16617
aI56903
asS'd'
I6209
sa(dp12418
S'a'
I4125
sS's'
I17
sS'm'
(lp12419
I1
aI1
aI1
aI5
aI15
aI41
aI107
aI115
aI79
aI693
aI919
aI3513
aI6793
aI6541
aI5545
aI58583
aI27963
asS'd'
I6210
sa(dp12420
S'a'
I4126
sS's'
I17
sS'm'
(lp12421
I1
aI3
aI7
aI11
aI21
aI19
aI123
aI1
aI441
aI531
aI359
aI2117
aI2465
aI11389
aI13489
aI32755
aI4577
asS'd'
I6211
sa(dp12422
S'a'
I4139
sS's'
I17
sS'm'
(lp12423
I1
aI1
aI5
aI13
aI7
aI7
aI7
aI127
aI201
aI377
aI1423
aI269
aI2611
aI3339
aI19153
aI25659
aI33069
asS'd'
I6212
sa(dp12424
S'a'
I4142
sS's'
I17
sS'm'
(lp12425
I1
aI3
aI7
aI1
aI13
aI35
aI45
aI5
aI313
aI739
aI1779
aI2983
aI1815
aI8817
aI14239
aI3921
aI57975
asS'd'
I6213
sa(dp12426
S'a'
I4144
sS's'
I17
sS'm'
(lp12427
I1
aI3
aI1
aI11
aI9
aI39
aI33
aI111
aI39
aI255
aI159
aI2345
aI2193
aI11475
aI12841
aI47579
aI90309
asS'd'
I6214
sa(dp12428
S'a'
I4147
sS's'
I17
sS'm'
(lp12429
I1
aI1
aI1
aI3
aI27
aI49
aI85
aI157
aI243
aI247
aI1473
aI323
aI4631
aI1787
aI15193
aI5533
aI104999
asS'd'
I6215
sa(dp12430
S'a'
I4153
sS's'
I17
sS'm'
(lp12431
I1
aI1
aI7
aI9
aI11
aI29
aI23
aI219
aI57
aI339
aI1797
aI409
aI6025
aI10569
aI27409
aI15147
aI130281
asS'd'
I6216
sa(dp12432
S'a'
I4154
sS's'
I17
sS'm'
(lp12433
I1
aI1
aI7
aI1
aI31
aI31
aI113
aI229
aI63
aI877
aI319
aI2655
aI3335
aI7743
aI19593
aI10089
aI28215
asS'd'
I6217
sa(dp12434
S'a'
I4164
sS's'
I17
sS'm'
(lp12435
I1
aI1
aI3
aI11
aI23
aI3
aI71
aI235
aI329
aI751
aI159
aI2579
aI5363
aI12681
aI20233
aI53855
aI16407
asS'd'
I6218
sa(dp12436
S'a'
I4174
sS's'
I17
sS'm'
(lp12437
I1
aI1
aI5
aI1
aI7
aI61
aI21
aI235
aI379
aI849
aI61
aI2969
aI6399
aI2655
aI21635
aI16955
aI58675
asS'd'
I6219
sa(dp12438
S'a'
I4182
sS's'
I17
sS'm'
(lp12439
I1
aI3
aI7
aI7
aI29
aI15
aI5
aI11
aI143
aI699
aI1875
aI2115
aI6633
aI6195
aI5829
aI53633
aI111221
asS'd'
I6220
sa(dp12440
S'a'
I4185
sS's'
I17
sS'm'
(lp12441
I1
aI3
aI7
aI11
aI19
aI41
aI17
aI219
aI483
aI829
aI1233
aI3183
aI6283
aI2363
aI25245
aI63075
aI82733
asS'd'
I6221
sa(dp12442
S'a'
I4188
sS's'
I17
sS'm'
(lp12443
I1
aI3
aI7
aI13
aI21
aI17
aI1
aI207
aI443
aI575
aI521
aI2585
aI6875
aI14871
aI14739
aI10211
aI127435
asS'd'
I6222
sa(dp12444
S'a'
I4191
sS's'
I17
sS'm'
(lp12445
I1
aI3
aI7
aI7
aI15
aI39
aI99
aI197
aI219
aI259
aI1723
aI3737
aI6625
aI849
aI887
aI41293
aI53825
asS'd'
I6223
sa(dp12446
S'a'
I4195
sS's'
I17
sS'm'
(lp12447
I1
aI3
aI3
aI3
aI5
aI3
aI75
aI155
aI189
aI935
aI85
aI2273
aI1375
aI4217
aI10709
aI58047
aI81689
asS'd'
I6224
sa(dp12448
S'a'
I4219
sS's'
I17
sS'm'
(lp12449
I1
aI3
aI5
aI5
aI27
aI27
aI107
aI229
aI179
aI887
aI91
aI421
aI7313
aI6495
aI451
aI43859
aI40033
asS'd'
I6225
sa(dp12450
S'a'
I4225
sS's'
I17
sS'm'
(lp12451
I1
aI3
aI5
aI11
aI25
aI49
aI121
aI73
aI169
aI311
aI1387
aI1037
aI6519
aI9317
aI26975
aI50627
aI46805
asS'd'
I6226
sa(dp12452
S'a'
I4228
sS's'
I17
sS'm'
(lp12453
I1
aI1
aI5
aI11
aI17
aI21
aI19
aI125
aI387
aI697
aI1017
aI1759
aI7295
aI9869
aI28241
aI9367
aI119255
asS'd'
I6227
sa(dp12454
S'a'
I4232
sS's'
I17
sS'm'
(lp12455
I1
aI1
aI7
aI5
aI29
aI27
aI87
aI187
aI95
aI625
aI933
aI1751
aI5253
aI313
aI30841
aI16349
aI67347
asS'd'
I6228
sa(dp12456
S'a'
I4246
sS's'
I17
sS'm'
(lp12457
I1
aI1
aI3
aI3
aI15
aI51
aI23
aI101
aI183
aI267
aI243
aI711
aI983
aI12461
aI17801
aI1429
aI47273
asS'd'
I6229
sa(dp12458
S'a'
I4255
sS's'
I17
sS'm'
(lp12459
I1
aI1
aI1
aI3
aI17
aI3
aI73
aI67
aI49
aI449
aI879
aI2559
aI401
aI11983
aI13697
aI12023
aI78855
asS'd'
I6230
sa(dp12460
S'a'
I4274
sS's'
I17
sS'm'
(lp12461
I1
aI3
aI7
aI15
aI25
aI25
aI43
aI81
aI141
aI161
aI595
aI621
aI1165
aI10869
aI22875
aI6741
aI90017
asS'd'
I6231
sa(dp12462
S'a'
I4283
sS's'
I17
sS'm'
(lp12463
I1
aI3
aI5
aI11
aI13
aI57
aI53
aI219
aI145
aI937
aI769
aI1961
aI4725
aI3335
aI12623
aI8335
aI46305
asS'd'
I6232
sa(dp12464
S'a'
I4286
sS's'
I17
sS'm'
(lp12465
I1
aI1
aI3
aI5
aI7
aI39
aI19
aI101
aI313
aI583
aI483
aI2515
aI125
aI5211
aI2559
aI11937
aI126717
asS'd'
I6233
sa(dp12466
S'a'
I4306
sS's'
I17
sS'm'
(lp12467
I1
aI3
aI1
aI7
aI7
aI1
aI117
aI49
aI231
aI133
aI381
aI697
aI927
aI8263
aI26529
aI64881
aI25059
asS'd'
I6234
sa(dp12468
S'a'
I4311
sS's'
I17
sS'm'
(lp12469
I1
aI1
aI1
aI15
aI11
aI25
aI77
aI149
aI233
aI215
aI1239
aI3045
aI99
aI11183
aI30279
aI32271
aI100943
asS'd'
I6235
sa(dp12470
S'a'
I4317
sS's'
I17
sS'm'
(lp12471
I1
aI1
aI5
aI7
aI31
aI25
aI1
aI51
aI221
aI607
aI1733
aI2145
aI6765
aI7011
aI16927
aI29257
aI2445
asS'd'
I6236
sa(dp12472
S'a'
I4321
sS's'
I17
sS'm'
(lp12473
I1
aI3
aI5
aI1
aI19
aI23
aI123
aI93
aI381
aI295
aI765
aI2335
aI8025
aI14003
aI4801
aI54243
aI57297
asS'd'
I6237
sa(dp12474
S'a'
I4324
sS's'
I17
sS'm'
(lp12475
I1
aI1
aI7
aI9
aI9
aI31
aI63
aI191
aI495
aI527
aI251
aI2119
aI1663
aI209
aI7445
aI1441
aI4075
asS'd'
I6238
sa(dp12476
S'a'
I4331
sS's'
I17
sS'm'
(lp12477
I1
aI3
aI5
aI5
aI13
aI17
aI97
aI79
aI369
aI55
aI677
aI2031
aI7315
aI4769
aI31659
aI21975
aI22061
asS'd'
I6239
sa(dp12478
S'a'
I4333
sS's'
I17
sS'm'
(lp12479
I1
aI3
aI3
aI7
aI3
aI63
aI121
aI243
aI39
aI917
aI1917
aI297
aI7241
aI1565
aI31675
aI14443
aI67239
asS'd'
I6240
sa(dp12480
S'a'
I4359
sS's'
I17
sS'm'
(lp12481
I1
aI3
aI7
aI1
aI13
aI25
aI51
aI65
aI145
aI475
aI1853
aI4023
aI5121
aI14411
aI15993
aI42165
aI13615
asS'd'
I6241
sa(dp12482
S'a'
I4360
sS's'
I17
sS'm'
(lp12483
I1
aI3
aI3
aI1
aI3
aI51
aI75
aI29
aI169
aI311
aI1309
aI2929
aI7669
aI1507
aI14605
aI32667
aI103861
asS'd'
I6242
sa(dp12484
S'a'
I4368
sS's'
I17
sS'm'
(lp12485
I1
aI3
aI7
aI1
aI23
aI37
aI89
aI211
aI137
aI495
aI1469
aI3425
aI1167
aI12429
aI27301
aI46857
aI83007
asS'd'
I6243
sa(dp12486
S'a'
I4373
sS's'
I17
sS'm'
(lp12487
I1
aI3
aI7
aI7
aI27
aI37
aI33
aI129
aI73
aI23
aI761
aI119
aI6217
aI4749
aI20835
aI47477
aI33665
asS'd'
I6244
sa(dp12488
S'a'
I4389
sS's'
I17
sS'm'
(lp12489
I1
aI1
aI3
aI5
aI29
aI35
aI79
aI21
aI183
aI933
aI43
aI3149
aI5273
aI12159
aI20695
aI5387
aI23569
asS'd'
I6245
sa(dp12490
S'a'
I4394
sS's'
I17
sS'm'
(lp12491
I1
aI1
aI5
aI5
aI3
aI11
aI57
aI205
aI349
aI657
aI1509
aI3693
aI5495
aI11865
aI13861
aI62215
aI94141
asS'd'
I6246
sa(dp12492
S'a'
I4413
sS's'
I17
sS'm'
(lp12493
I1
aI3
aI1
aI7
aI17
aI43
aI117
aI119
aI75
aI849
aI1247
aI643
aI2691
aI2289
aI9759
aI18683
aI68649
asS'd'
I6247
sa(dp12494
S'a'
I4422
sS's'
I17
sS'm'
(lp12495
I1
aI1
aI1
aI15
aI5
aI55
aI89
aI177
aI427
aI701
aI735
aI2993
aI5293
aI15395
aI567
aI5501
aI102393
asS'd'
I6248
sa(dp12496
S'a'
I4431
sS's'
I17
sS'm'
(lp12497
I1
aI3
aI3
aI15
aI5
aI37
aI73
aI111
aI9
aI141
aI407
aI1579
aI6691
aI11843
aI6377
aI64181
aI97347
asS'd'
I6249
sa(dp12498
S'a'
I4436
sS's'
I17
sS'm'
(lp12499
I1
aI1
aI5
aI1
aI9
aI17
aI71
aI127
aI285
aI929
aI1243
aI2605
aI359
aI14589
aI32603
aI39879
aI115901
asS'd'
I6250
sa(dp12500
S'a'
I4440
sS's'
I17
sS'm'
(lp12501
I1
aI3
aI7
aI15
aI3
aI27
aI91
aI121
aI47
aI631
aI1589
aI385
aI5997
aI14077
aI21285
aI33895
aI36985
asS'd'
I6251
sa(dp12502
S'a'
I4445
sS's'
I17
sS'm'
(lp12503
I1
aI3
aI3
aI9
aI1
aI47
aI89
aI79
aI213
aI27
aI547
aI1703
aI4035
aI13205
aI4341
aI21895
aI34247
asS'd'
I6252
sa(dp12504
S'a'
I4452
sS's'
I17
sS'm'
(lp12505
I1
aI3
aI5
aI7
aI9
aI9
aI47
aI89
aI231
aI857
aI297
aI2949
aI2715
aI1275
aI14427
aI20227
aI21569
asS'd'
I6253
sa(dp12506
S'a'
I4462
sS's'
I17
sS'm'
(lp12507
I1
aI3
aI1
aI3
aI15
aI57
aI61
aI183
aI377
aI477
aI1135
aI1729
aI2863
aI8607
aI29241
aI34983
aI84443
asS'd'
I6254
sa(dp12508
S'a'
I4469
sS's'
I17
sS'm'
(lp12509
I1
aI1
aI7
aI7
aI5
aI53
aI91
aI149
aI71
aI41
aI1025
aI3945
aI3989
aI15853
aI20903
aI26943
aI99841
asS'd'
I6255
sa(dp12510
S'a'
I4470
sS's'
I17
sS'm'
(lp12511
I1
aI3
aI3
aI3
aI29
aI21
aI59
aI217
aI483
aI257
aI331
aI657
aI2935
aI945
aI9821
aI42501
aI98087
asS'd'
I6256
sa(dp12512
S'a'
I4473
sS's'
I17
sS'm'
(lp12513
I1
aI3
aI5
aI3
aI17
aI39
aI123
aI103
aI109
aI957
aI853
aI3821
aI555
aI10869
aI27673
aI38315
aI83105
asS'd'
I6257
sa(dp12514
S'a'
I4479
sS's'
I17
sS'm'
(lp12515
I1
aI3
aI1
aI3
aI27
aI7
aI97
aI57
aI429
aI53
aI1791
aI1405
aI4113
aI8435
aI12845
aI21567
aI91559
asS'd'
I6258
sa(dp12516
S'a'
I4480
sS's'
I17
sS'm'
(lp12517
I1
aI3
aI3
aI1
aI17
aI61
aI125
aI77
aI225
aI395
aI945
aI3213
aI1363
aI15947
aI27049
aI4389
aI64037
asS'd'
I6259
sa(dp12518
S'a'
I4483
sS's'
I17
sS'm'
(lp12519
I1
aI1
aI1
aI3
aI15
aI51
aI15
aI189
aI449
aI989
aI939
aI985
aI6929
aI13779
aI25011
aI22277
aI72543
asS'd'
I6260
sa(dp12520
S'a'
I4489
sS's'
I17
sS'm'
(lp12521
I1
aI3
aI3
aI1
aI25
aI53
aI5
aI219
aI195
aI703
aI163
aI1405
aI821
aI6797
aI14329
aI1675
aI96653
asS'd'
I6261
sa(dp12522
S'a'
I4503
sS's'
I17
sS'm'
(lp12523
I1
aI1
aI7
aI13
aI7
aI1
aI45
aI135
aI369
aI125
aI711
aI2509
aI131
aI13663
aI29769
aI19497
aI116779
asS'd'
I6262
sa(dp12524
S'a'
I4519
sS's'
I17
sS'm'
(lp12525
I1
aI1
aI7
aI15
aI23
aI25
aI7
aI225
aI435
aI835
aI1981
aI2537
aI5727
aI15961
aI30089
aI58905
aI100339
asS'd'
I6263
sa(dp12526
S'a'
I4520
sS's'
I17
sS'm'
(lp12527
I1
aI3
aI7
aI3
aI19
aI9
aI79
aI63
aI371
aI419
aI1357
aI3649
aI7987
aI14541
aI6631
aI50555
aI84217
asS'd'
I6264
sa(dp12528
S'a'
I4525
sS's'
I17
sS'm'
(lp12529
I1
aI3
aI3
aI9
aI7
aI61
aI11
aI157
aI99
aI95
aI945
aI2803
aI1703
aI117
aI12891
aI21817
aI84259
asS'd'
I6265
sa(dp12530
S'a'
I4526
sS's'
I17
sS'm'
(lp12531
I1
aI3
aI7
aI7
aI25
aI37
aI111
aI99
aI65
aI599
aI1313
aI2557
aI5489
aI3625
aI7429
aI19309
aI78111
asS'd'
I6266
sa(dp12532
S'a'
I4533
sS's'
I17
sS'm'
(lp12533
I1
aI3
aI1
aI1
aI19
aI15
aI85
aI253
aI347
aI315
aI1349
aI983
aI2507
aI4155
aI15311
aI43535
aI101409
asS'd'
I6267
sa(dp12534
S'a'
I4552
sS's'
I17
sS'm'
(lp12535
I1
aI3
aI3
aI3
aI1
aI55
aI3
aI57
aI375
aI107
aI177
aI1673
aI6871
aI7137
aI10297
aI65363
aI42293
asS'd'
I6268
sa(dp12536
S'a'
I4581
sS's'
I17
sS'm'
(lp12537
I1
aI1
aI1
aI3
aI9
aI5
aI83
aI45
aI139
aI893
aI63
aI2859
aI6333
aI15591
aI18491
aI26387
aI25573
asS'd'
I6269
sa(dp12538
S'a'
I4585
sS's'
I17
sS'm'
(lp12539
I1
aI1
aI7
aI15
aI1
aI39
aI113
aI127
aI503
aI617
aI1367
aI1855
aI185
aI4233
aI5787
aI8265
aI42097
asS'd'
I6270
sa(dp12540
S'a'
I4591
sS's'
I17
sS'm'
(lp12541
I1
aI1
aI3
aI11
aI11
aI41
aI119
aI165
aI331
aI625
aI81
aI2495
aI7247
aI9139
aI15269
aI31447
aI128425
asS'd'
I6271
sa(dp12542
S'a'
I4594
sS's'
I17
sS'm'
(lp12543
I1
aI1
aI5
aI5
aI17
aI35
aI39
aI1
aI91
aI563
aI1841
aI2975
aI1233
aI3837
aI22145
aI36719
aI104503
asS'd'
I6272
sa(dp12544
S'a'
I4596
sS's'
I17
sS'm'
(lp12545
I1
aI1
aI7
aI3
aI23
aI35
aI77
aI69
aI271
aI487
aI921
aI2597
aI8011
aI13037
aI6001
aI20519
aI32673
asS'd'
I6273
sa(dp12546
S'a'
I4599
sS's'
I17
sS'm'
(lp12547
I1
aI1
aI1
aI1
aI29
aI17
aI11
aI145
aI473
aI877
aI813
aI727
aI6805
aI3563
aI13371
aI22169
aI17239
asS'd'
I6274
sa(dp12548
S'a'
I4612
sS's'
I17
sS'm'
(lp12549
I1
aI1
aI1
aI13
aI17
aI13
aI1
aI125
aI313
aI423
aI1079
aI2401
aI2325
aI2219
aI24071
aI25613
aI34163
asS'd'
I6275
sa(dp12550
S'a'
I4621
sS's'
I17
sS'm'
(lp12551
I1
aI1
aI5
aI7
aI29
aI33
aI53
aI215
aI11
aI555
aI555
aI1965
aI3643
aI5433
aI12923
aI59655
aI25339
asS'd'
I6276
sa(dp12552
S'a'
I4630
sS's'
I17
sS'm'
(lp12553
I1
aI3
aI3
aI3
aI23
aI37
aI119
aI117
aI459
aI359
aI1849
aI1019
aI433
aI15391
aI5625
aI52649
aI81313
asS'd'
I6277
sa(dp12554
S'a'
I4636
sS's'
I17
sS'm'
(lp12555
I1
aI3
aI3
aI1
aI21
aI31
aI121
aI161
aI113
aI667
aI863
aI105
aI3805
aI14459
aI28235
aI24543
aI89755
asS'd'
I6278
sa(dp12556
S'a'
I4640
sS's'
I17
sS'm'
(lp12557
I1
aI1
aI5
aI15
aI17
aI37
aI15
aI111
aI511
aI477
aI611
aI955
aI2591
aI16137
aI14179
aI30995
aI129575
asS'd'
I6279
sa(dp12558
S'a'
I4649
sS's'
I17
sS'm'
(lp12559
I1
aI3
aI3
aI3
aI21
aI49
aI25
aI37
aI287
aI263
aI851
aI1015
aI8133
aI9429
aI10959
aI64483
aI82533
asS'd'
I6280
sa(dp12560
S'a'
I4650
sS's'
I17
sS'm'
(lp12561
I1
aI1
aI5
aI1
aI25
aI19
aI49
aI159
aI155
aI443
aI975
aI1413
aI321
aI7871
aI22935
aI57303
aI124027
asS'd'
I6281
sa(dp12562
S'a'
I4660
sS's'
I17
sS'm'
(lp12563
I1
aI3
aI1
aI1
aI19
aI45
aI47
aI89
aI409
aI509
aI1249
aI2445
aI2053
aI3781
aI7517
aI61869
aI125137
asS'd'
I6282
sa(dp12564
S'a'
I4677
sS's'
I17
sS'm'
(lp12565
I1
aI1
aI5
aI13
aI27
aI57
aI45
aI43
aI361
aI329
aI1321
aI771
aI4665
aI12245
aI18993
aI15121
aI127485
asS'd'
I6283
sa(dp12566
S'a'
I4687
sS's'
I17
sS'm'
(lp12567
I1
aI3
aI3
aI7
aI3
aI41
aI127
aI75
aI485
aI821
aI497
aI2649
aI6423
aI12419
aI31421
aI9441
aI63645
asS'd'
I6284
sa(dp12568
S'a'
I4696
sS's'
I17
sS'm'
(lp12569
I1
aI1
aI3
aI5
aI19
aI61
aI91
aI35
aI311
aI287
aI449
aI3955
aI5805
aI5631
aI25613
aI55409
aI104545
asS'd'
I6285
sa(dp12570
S'a'
I4701
sS's'
I17
sS'm'
(lp12571
I1
aI3
aI7
aI11
aI27
aI19
aI27
aI53
aI19
aI35
aI1687
aI3923
aI3379
aI10435
aI15053
aI12343
aI89077
asS'd'
I6286
sa(dp12572
S'a'
I4705
sS's'
I17
sS'm'
(lp12573
I1
aI3
aI5
aI13
aI31
aI41
aI15
aI239
aI349
aI533
aI1771
aI737
aI6503
aI14355
aI18781
aI27805
aI79049
asS'd'
I6287
sa(dp12574
S'a'
I4706
sS's'
I17
sS'm'
(lp12575
I1
aI3
aI1
aI3
aI13
aI11
aI69
aI227
aI169
aI873
aI533
aI2217
aI1047
aI12415
aI12271
aI22447
aI14163
asS'd'
I6288
sa(dp12576
S'a'
I4711
sS's'
I17
sS'm'
(lp12577
I1
aI1
aI3
aI9
aI7
aI31
aI23
aI155
aI133
aI305
aI1569
aI521
aI201
aI10339
aI16999
aI29163
aI32817
asS'd'
I6289
sa(dp12578
S'a'
I4720
sS's'
I17
sS'm'
(lp12579
I1
aI1
aI1
aI5
aI31
aI57
aI43
aI223
aI121
aI803
aI357
aI1855
aI4321
aI10245
aI25725
aI2543
aI47395
asS'd'
I6290
sa(dp12580
S'a'
I4723
sS's'
I17
sS'm'
(lp12581
I1
aI3
aI5
aI9
aI3
aI5
aI47
aI189
aI217
aI899
aI1455
aI691
aI1277
aI7861
aI3627
aI14895
aI41109
asS'd'
I6291
sa(dp12582
S'a'
I4732
sS's'
I17
sS'm'
(lp12583
I1
aI3
aI7
aI3
aI29
aI9
aI37
aI63
aI453
aI709
aI921
aI771
aI8069
aI239
aI22639
aI59937
aI10635
asS'd'
I6292
sa(dp12584
S'a'
I4736
sS's'
I17
sS'm'
(lp12585
I1
aI3
aI7
aI1
aI11
aI51
aI79
aI131
aI225
aI757
aI549
aI1605
aI3921
aI1849
aI16307
aI29809
aI120597
asS'd'
I6293
sa(dp12586
S'a'
I4742
sS's'
I17
sS'm'
(lp12587
I1
aI3
aI7
aI7
aI1
aI45
aI33
aI185
aI23
aI881
aI1941
aI4093
aI4741
aI11633
aI2059
aI32007
aI11103
asS'd'
I6294
sa(dp12588
S'a'
I4748
sS's'
I17
sS'm'
(lp12589
I1
aI3
aI5
aI11
aI17
aI21
aI43
aI205
aI363
aI559
aI697
aI4057
aI631
aI6697
aI883
aI61705
aI102791
asS'd'
I6295
sa(dp12590
S'a'
I4754
sS's'
I17
sS'm'
(lp12591
I1
aI1
aI7
aI9
aI29
aI35
aI109
aI85
aI373
aI321
aI415
aI2969
aI6163
aI6999
aI9999
aI36435
aI125267
asS'd'
I6296
sa(dp12592
S'a'
I4759
sS's'
I17
sS'm'
(lp12593
I1
aI1
aI7
aI11
aI25
aI9
aI113
aI91
aI337
aI889
aI947
aI2093
aI5289
aI1367
aI13297
aI36155
aI21825
asS'd'
I6297
sa(dp12594
S'a'
I4769
sS's'
I17
sS'm'
(lp12595
I1
aI1
aI3
aI9
aI17
aI25
aI35
aI79
aI275
aI687
aI335
aI1181
aI7327
aI3729
aI1561
aI27441
aI114355
asS'd'
I6298
sa(dp12596
S'a'
I4787
sS's'
I17
sS'm'
(lp12597
I1
aI3
aI3
aI11
aI25
aI41
aI27
aI89
aI115
aI361
aI871
aI1497
aI5735
aI6365
aI1737
aI14277
aI63847
asS'd'
I6299
sa(dp12598
S'a'
I4807
sS's'
I17
sS'm'
(lp12599
I1
aI3
aI7
aI7
aI1
aI63
aI31
aI73
aI289
aI67
aI277
aI1821
aI4883
aI10795
aI11755
aI15471
aI105871
asS'd'
I6300
sa(dp12600
S'a'
I4814
sS's'
I17
sS'm'
(lp12601
I1
aI3
aI7
aI9
aI23
aI17
aI37
aI179
aI409
aI957
aI373
aI2393
aI2363
aI6735
aI28737
aI41927
aI115735
asS'd'
I6301
sa(dp12602
S'a'
I4837
sS's'
I17
sS'm'
(lp12603
I1
aI1
aI3
aI9
aI15
aI43
aI111
aI61
aI455
aI181
aI1643
aI3063
aI4311
aI13705
aI29993
aI21731
aI25243
asS'd'
I6302
sa(dp12604
S'a'
I4867
sS's'
I17
sS'm'
(lp12605
I1
aI1
aI1
aI15
aI13
aI13
aI69
aI187
aI91
aI395
aI209
aI3477
aI4649
aI7727
aI30557
aI14719
aI1953
asS'd'
I6303
sa(dp12606
S'a'
I4873
sS's'
I17
sS'm'
(lp12607
I1
aI1
aI1
aI15
aI9
aI39
aI119
aI193
aI459
aI135
aI567
aI25
aI4583
aI8401
aI22161
aI14771
aI74165
asS'd'
I6304
sa(dp12608
S'a'
I4879
sS's'
I17
sS'm'
(lp12609
I1
aI1
aI3
aI7
aI5
aI39
aI77
aI149
aI293
aI585
aI1245
aI3615
aI357
aI11613
aI13865
aI40227
aI41023
asS'd'
I6305
sa(dp12610
S'a'
I4884
sS's'
I17
sS'm'
(lp12611
I1
aI1
aI7
aI9
aI9
aI37
aI5
aI177
aI121
aI181
aI771
aI733
aI7683
aI4855
aI13629
aI8349
aI46137
asS'd'
I6306
sa(dp12612
S'a'
I4898
sS's'
I17
sS'm'
(lp12613
I1
aI1
aI3
aI13
aI3
aI37
aI73
aI69
aI281
aI109
aI563
aI1427
aI5127
aI8957
aI16749
aI41489
aI49531
asS'd'
I6307
sa(dp12614
S'a'
I4907
sS's'
I17
sS'm'
(lp12615
I1
aI1
aI7
aI11
aI29
aI63
aI79
aI127
aI95
aI809
aI1175
aI1567
aI6353
aI7505
aI26551
aI5073
aI53733
asS'd'
I6308
sa(dp12616
S'a'
I4910
sS's'
I17
sS'm'
(lp12617
I1
aI1
aI1
aI5
aI25
aI41
aI59
aI103
aI59
aI365
aI1111
aI3909
aI3749
aI14889
aI3639
aI10435
aI45407
asS'd'
I6309
sa(dp12618
S'a'
I4918
sS's'
I17
sS'm'
(lp12619
I1
aI1
aI1
aI5
aI3
aI61
aI93
aI199
aI97
aI779
aI67
aI241
aI6197
aI6785
aI16869
aI7573
aI46745
asS'd'
I6310
sa(dp12620
S'a'
I4924
sS's'
I17
sS'm'
(lp12621
I1
aI1
aI5
aI9
aI27
aI29
aI21
aI69
aI165
aI661
aI1245
aI1265
aI2979
aI9685
aI17781
aI23329
aI48029
asS'd'
I6311
sa(dp12622
S'a'
I4953
sS's'
I17
sS'm'
(lp12623
I1
aI1
aI1
aI7
aI7
aI23
aI39
aI197
aI169
aI561
aI499
aI2197
aI4371
aI157
aI6837
aI44635
aI94861
asS'd'
I6312
sa(dp12624
S'a'
I4956
sS's'
I17
sS'm'
(lp12625
I1
aI1
aI5
aI13
aI7
aI5
aI9
aI207
aI321
aI243
aI899
aI2967
aI3553
aI15413
aI8961
aI55039
aI6459
asS'd'
I6313
sa(dp12626
S'a'
I4965
sS's'
I17
sS'm'
(lp12627
I1
aI3
aI5
aI3
aI13
aI25
aI33
aI145
aI45
aI979
aI33
aI2211
aI7003
aI11147
aI11327
aI55151
aI30697
asS'd'
I6314
sa(dp12628
S'a'
I4966
sS's'
I17
sS'm'
(lp12629
I1
aI1
aI3
aI13
aI7
aI51
aI25
aI229
aI231
aI115
aI1815
aI3867
aI1533
aI15259
aI8067
aI64803
aI87535
asS'd'
I6315
sa(dp12630
S'a'
I4970
sS's'
I17
sS'm'
(lp12631
I1
aI1
aI3
aI3
aI21
aI51
aI101
aI49
aI227
aI393
aI1659
aI955
aI545
aI7395
aI31563
aI5499
aI130541
asS'd'
I6316
sa(dp12632
S'a'
I4972
sS's'
I17
sS'm'
(lp12633
I1
aI3
aI1
aI1
aI21
aI41
aI57
aI161
aI269
aI35
aI893
aI1817
aI857
aI7027
aI973
aI12529
aI46659
asS'd'
I6317
sa(dp12634
S'a'
I4983
sS's'
I17
sS'm'
(lp12635
I1
aI1
aI3
aI7
aI17
aI35
aI23
aI29
aI335
aI725
aI453
aI1051
aI6019
aI7595
aI29451
aI1853
aI116615
asS'd'
I6318
sa(dp12636
S'a'
I4989
sS's'
I17
sS'm'
(lp12637
I1
aI3
aI3
aI1
aI3
aI55
aI73
aI187
aI213
aI329
aI997
aI703
aI5829
aI7903
aI1081
aI33359
aI119123
asS'd'
I6319
sa(dp12638
S'a'
I4994
sS's'
I17
sS'm'
(lp12639
I1
aI3
aI3
aI15
aI29
aI55
aI15
aI17
aI245
aI117
aI1735
aI767
aI4457
aI8803
aI17621
aI26925
aI72487
asS'd'
I6320
sa(dp12640
S'a'
I5000
sS's'
I17
sS'm'
(lp12641
I1
aI3
aI5
aI3
aI25
aI7
aI119
aI139
aI159
aI199
aI317
aI3875
aI8115
aI7581
aI29239
aI50225
aI48459
asS'd'
I6321
sa(dp12642
S'a'
I5005
sS's'
I17
sS'm'
(lp12643
I1
aI3
aI7
aI11
aI11
aI41
aI107
aI225
aI395
aI545
aI259
aI2379
aI6709
aI11669
aI14545
aI43663
aI69979
asS'd'
I6322
sa(dp12644
S'a'
I5014
sS's'
I17
sS'm'
(lp12645
I1
aI3
aI5
aI13
aI23
aI45
aI73
aI137
aI447
aI305
aI117
aI2659
aI7989
aI233
aI31991
aI60495
aI571
asS'd'
I6323
sa(dp12646
S'a'
I5018
sS's'
I17
sS'm'
(lp12647
I1
aI3
aI7
aI9
aI1
aI37
aI31
aI1
aI433
aI701
aI159
aI3811
aI4529
aI6697
aI7121
aI31107
aI61555
asS'd'
I6324
sa(dp12648
S'a'
I5023
sS's'
I17
sS'm'
(lp12649
I1
aI3
aI5
aI5
aI13
aI21
aI81
aI63
aI95
aI741
aI1189
aI1567
aI1223
aI12371
aI28435
aI10537
aI53785
asS'd'
I6325
sa(dp12650
S'a'
I5039
sS's'
I17
sS'm'
(lp12651
I1
aI1
aI1
aI11
aI17
aI31
aI67
aI121
aI281
aI593
aI561
aI1759
aI387
aI9639
aI28595
aI22473
aI4935
asS'd'
I6326
sa(dp12652
S'a'
I5053
sS's'
I17
sS'm'
(lp12653
I1
aI3
aI7
aI3
aI5
aI43
aI59
aI151
aI351
aI263
aI297
aI423
aI1681
aI3785
aI15171
aI7145
aI57531
asS'd'
I6327
sa(dp12654
S'a'
I5054
sS's'
I17
sS'm'
(lp12655
I1
aI3
aI7
aI15
aI9
aI35
aI105
aI189
aI261
aI175
aI1669
aI1289
aI5401
aI12801
aI19585
aI48169
aI93195
asS'd'
I6328
sa(dp12656
S'a'
I5061
sS's'
I17
sS'm'
(lp12657
I1
aI1
aI7
aI1
aI31
aI41
aI23
aI237
aI151
aI549
aI1079
aI2933
aI5509
aI15593
aI1791
aI15757
aI44607
asS'd'
I6329
sa(dp12658
S'a'
I5065
sS's'
I17
sS'm'
(lp12659
I1
aI1
aI1
aI3
aI29
aI1
aI59
aI115
aI13
aI999
aI1179
aI3561
aI2749
aI10059
aI12861
aI6797
aI11793
asS'd'
I6330
sa(dp12660
S'a'
I5080
sS's'
I17
sS'm'
(lp12661
I1
aI3
aI3
aI7
aI11
aI5
aI23
aI217
aI101
aI775
aI1497
aI4047
aI2427
aI5117
aI9683
aI28895
aI27557
asS'd'
I6331
sa(dp12662
S'a'
I5083
sS's'
I17
sS'm'
(lp12663
I1
aI3
aI7
aI5
aI31
aI55
aI99
aI65
aI55
aI587
aI1271
aI2277
aI7947
aI12995
aI13149
aI4463
aI37625
asS'd'
I6332
sa(dp12664
S'a'
I5107
sS's'
I17
sS'm'
(lp12665
I1
aI1
aI7
aI11
aI3
aI63
aI23
aI191
aI125
aI365
aI1153
aI2657
aI6763
aI4557
aI21643
aI26885
aI36753
asS'd'
I6333
sa(dp12666
S'a'
I5119
sS's'
I17
sS'm'
(lp12667
I1
aI1
aI1
aI15
aI25
aI15
aI111
aI135
aI507
aI745
aI1947
aI2545
aI4329
aI14325
aI8187
aI52021
aI63401
asS'd'
I6334
sa(dp12668
S'a'
I5146
sS's'
I17
sS'm'
(lp12669
I1
aI1
aI3
aI3
aI27
aI25
aI19
aI211
aI393
aI467
aI1015
aI2495
aI7135
aI495
aI10385
aI26961
aI49325
asS'd'
I6335
sa(dp12670
S'a'
I5151
sS's'
I17
sS'm'
(lp12671
I1
aI1
aI3
aI5
aI15
aI35
aI3
aI203
aI337
aI337
aI703
aI1989
aI6869
aI6055
aI21095
aI4749
aI125669
asS'd'
I6336
sa(dp12672
S'a'
I5152
sS's'
I17
sS'm'
(lp12673
I1
aI1
aI5
aI1
aI31
aI39
aI57
aI101
aI419
aI717
aI1489
aI199
aI5729
aI3003
aI2607
aI64593
aI11515
asS'd'
I6337
sa(dp12674
S'a'
I5155
sS's'
I17
sS'm'
(lp12675
I1
aI3
aI7
aI13
aI15
aI3
aI33
aI61
aI17
aI433
aI1097
aI957
aI5351
aI3043
aI3679
aI44881
aI126909
asS'd'
I6338
sa(dp12676
S'a'
I5169
sS's'
I17
sS'm'
(lp12677
I1
aI1
aI3
aI11
aI5
aI1
aI121
aI175
aI119
aI367
aI399
aI2527
aI2157
aI2667
aI31069
aI24797
aI119621
asS'd'
I6339
sa(dp12678
S'a'
I5170
sS's'
I17
sS'm'
(lp12679
I1
aI3
aI1
aI7
aI27
aI47
aI115
aI229
aI455
aI775
aI73
aI837
aI1181
aI3457
aI4057
aI33907
aI67151
asS'd'
I6340
sa(dp12680
S'a'
I5176
sS's'
I17
sS'm'
(lp12681
I1
aI3
aI3
aI1
aI7
aI51
aI71
aI177
aI463
aI921
aI393
aI3137
aI1225
aI5709
aI303
aI20597
aI77581
asS'd'
I6341
sa(dp12682
S'a'
I5179
sS's'
I17
sS'm'
(lp12683
I1
aI3
aI5
aI3
aI31
aI1
aI93
aI53
aI177
aI433
aI1471
aI2191
aI4471
aI9211
aI19397
aI57727
aI60367
asS'd'
I6342
sa(dp12684
S'a'
I5182
sS's'
I17
sS'm'
(lp12685
I1
aI1
aI3
aI11
aI29
aI55
aI121
aI89
aI67
aI869
aI1631
aI2657
aI7357
aI7159
aI22449
aI16357
aI20077
asS'd'
I6343
sa(dp12686
S'a'
I5189
sS's'
I17
sS'm'
(lp12687
I1
aI3
aI7
aI15
aI11
aI39
aI127
aI63
aI211
aI359
aI971
aI1221
aI1909
aI9963
aI7827
aI60923
aI98495
asS'd'
I6344
sa(dp12688
S'a'
I5193
sS's'
I17
sS'm'
(lp12689
I1
aI1
aI7
aI9
aI23
aI47
aI47
aI85
aI307
aI471
aI1287
aI3825
aI5451
aI15151
aI15647
aI63043
aI92443
asS'd'
I6345
sa(dp12690
S'a'
I5196
sS's'
I17
sS'm'
(lp12691
I1
aI3
aI7
aI5
aI19
aI11
aI11
aI27
aI307
aI695
aI99
aI1037
aI1997
aI13673
aI591
aI8183
aI82197
asS'd'
I6346
sa(dp12692
S'a'
I5204
sS's'
I17
sS'm'
(lp12693
I1
aI3
aI5
aI5
aI3
aI53
aI109
aI227
aI503
aI855
aI1269
aI3903
aI5049
aI10647
aI21751
aI58707
aI78311
asS'd'
I6347
sa(dp12694
S'a'
I5207
sS's'
I17
sS'm'
(lp12695
I1
aI1
aI3
aI11
aI31
aI3
aI51
aI211
aI285
aI919
aI487
aI3393
aI3463
aI2271
aI8053
aI56791
aI33763
asS'd'
I6348
sa(dp12696
S'a'
I5211
sS's'
I17
sS'm'
(lp12697
I1
aI3
aI3
aI5
aI21
aI15
aI5
aI5
aI327
aI809
aI915
aI1365
aI7323
aI4247
aI31603
aI26261
aI80389
asS'd'
I6349
sa(dp12698
S'a'
I5220
sS's'
I17
sS'm'
(lp12699
I1
aI3
aI7
aI7
aI15
aI33
aI31
aI221
aI291
aI815
aI1307
aI929
aI3249
aI14573
aI13613
aI59509
aI59741
asS'd'
I6350
sa(dp12700
S'a'
I5258
sS's'
I17
sS'm'
(lp12701
I1
aI3
aI7
aI15
aI19
aI41
aI61
aI27
aI353
aI965
aI1901
aI87
aI2669
aI12757
aI29723
aI47165
aI16521
asS'd'
I6351
sa(dp12702
S'a'
I5265
sS's'
I17
sS'm'
(lp12703
I1
aI3
aI5
aI3
aI11
aI43
aI97
aI215
aI361
aI901
aI1425
aI4063
aI5327
aI14119
aI457
aI43145
aI107401
asS'd'
I6352
sa(dp12704
S'a'
I5271
sS's'
I17
sS'm'
(lp12705
I1
aI1
aI3
aI15
aI19
aI37
aI101
aI69
aI131
aI927
aI897
aI477
aI7641
aI4299
aI21213
aI26017
aI123801
asS'd'
I6353
sa(dp12706
S'a'
I5277
sS's'
I17
sS'm'
(lp12707
I1
aI3
aI7
aI7
aI19
aI5
aI11
aI51
aI277
aI985
aI1071
aI3437
aI6595
aI9547
aI11855
aI64249
aI30957
asS'd'
I6354
sa(dp12708
S'a'
I5278
sS's'
I17
sS'm'
(lp12709
I1
aI1
aI7
aI9
aI21
aI41
aI89
aI113
aI61
aI235
aI685
aI1419
aI7619
aI9863
aI21221
aI28685
aI53409
asS'd'
I6355
sa(dp12710
S'a'
I5282
sS's'
I17
sS'm'
(lp12711
I1
aI1
aI1
aI1
aI27
aI1
aI19
aI3
aI473
aI827
aI269
aI1659
aI2621
aI12347
aI13359
aI64687
aI99293
asS'd'
I6356
sa(dp12712
S'a'
I5296
sS's'
I17
sS'm'
(lp12713
I1
aI3
aI7
aI7
aI29
aI37
aI61
aI49
aI215
aI883
aI625
aI2671
aI3743
aI4517
aI2075
aI64865
aI58611
asS'd'
I6357
sa(dp12714
S'a'
I5299
sS's'
I17
sS'm'
(lp12715
I1
aI3
aI3
aI7
aI15
aI11
aI35
aI37
aI255
aI781
aI613
aI3587
aI7643
aI13081
aI32467
aI14427
aI15235
asS'd'
I6358
sa(dp12716
S'a'
I5319
sS's'
I17
sS'm'
(lp12717
I1
aI1
aI1
aI11
aI31
aI47
aI107
aI65
aI489
aI377
aI425
aI3453
aI2901
aI9999
aI7687
aI13311
aI103947
asS'd'
I6359
sa(dp12718
S'a'
I5328
sS's'
I17
sS'm'
(lp12719
I1
aI3
aI3
aI7
aI9
aI17
aI7
aI107
aI33
aI545
aI407
aI3335
aI7563
aI14315
aI32725
aI8483
aI69093
asS'd'
I6360
sa(dp12720
S'a'
I5343
sS's'
I17
sS'm'
(lp12721
I1
aI1
aI1
aI5
aI17
aI9
aI87
aI229
aI417
aI769
aI423
aI569
aI7073
aI8705
aI24487
aI63743
aI69807
asS'd'
I6361
sa(dp12722
S'a'
I5353
sS's'
I17
sS'm'
(lp12723
I1
aI3
aI1
aI9
aI1
aI29
aI75
aI25
aI483
aI259
aI1941
aI1533
aI8147
aI14127
aI24087
aI37475
aI130961
asS'd'
I6362
sa(dp12724
S'a'
I5364
sS's'
I17
sS'm'
(lp12725
I1
aI3
aI3
aI11
aI15
aI15
aI51
aI45
aI215
aI283
aI1687
aI185
aI4521
aI12205
aI13041
aI33283
aI77007
asS'd'
I6363
sa(dp12726
S'a'
I5368
sS's'
I17
sS'm'
(lp12727
I1
aI1
aI3
aI3
aI5
aI47
aI107
aI67
aI325
aI87
aI1831
aI2845
aI1645
aI1741
aI10811
aI8983
aI58515
asS'd'
I6364
sa(dp12728
S'a'
I5379
sS's'
I17
sS'm'
(lp12729
I1
aI3
aI1
aI13
aI19
aI17
aI1
aI151
aI411
aI915
aI1739
aI3781
aI4939
aI15767
aI25897
aI7205
aI17285
asS'd'
I6365
sa(dp12730
S'a'
I5381
sS's'
I17
sS'm'
(lp12731
I1
aI3
aI5
aI15
aI19
aI1
aI125
aI33
aI321
aI325
aI639
aI4013
aI967
aI4347
aI19743
aI13445
aI61229
asS'd'
I6366
sa(dp12732
S'a'
I5399
sS's'
I17
sS'm'
(lp12733
I1
aI3
aI3
aI13
aI13
aI37
aI71
aI85
aI51
aI775
aI973
aI739
aI4341
aI15707
aI12221
aI24321
aI48073
asS'd'
I6367
sa(dp12734
S'a'
I5415
sS's'
I17
sS'm'
(lp12735
I1
aI1
aI7
aI13
aI15
aI13
aI9
aI211
aI331
aI429
aI1323
aI3027
aI1091
aI13311
aI289
aI57789
aI93261
asS'd'
I6368
sa(dp12736
S'a'
I5422
sS's'
I17
sS'm'
(lp12737
I1
aI1
aI1
aI1
aI27
aI7
aI13
aI27
aI67
aI573
aI455
aI2353
aI113
aI11831
aI9069
aI4503
aI89291
asS'd'
I6369
sa(dp12738
S'a'
I5441
sS's'
I17
sS'm'
(lp12739
I1
aI1
aI1
aI7
aI21
aI63
aI47
aI39
aI419
aI991
aI1623
aI11
aI3153
aI12633
aI9425
aI65087
aI44935
asS'd'
I6370
sa(dp12740
S'a'
I5451
sS's'
I17
sS'm'
(lp12741
I1
aI3
aI1
aI7
aI23
aI11
aI15
aI11
aI99
aI543
aI1739
aI3955
aI5883
aI12469
aI7529
aI14177
aI1945
asS'd'
I6371
sa(dp12742
S'a'
I5456
sS's'
I17
sS'm'
(lp12743
I1
aI3
aI1
aI3
aI5
aI17
aI31
aI251
aI387
aI311
aI725
aI3827
aI6835
aI5065
aI3141
aI43441
aI87955
asS'd'
I6372
sa(dp12744
S'a'
I5462
sS's'
I17
sS'm'
(lp12745
I1
aI1
aI1
aI11
aI25
aI7
aI75
aI135
aI67
aI589
aI889
aI3429
aI155
aI9081
aI28653
aI8059
aI57251
asS'd'
I6373
sa(dp12746
S'a'
I5490
sS's'
I17
sS'm'
(lp12747
I1
aI3
aI5
aI15
aI21
aI15
aI103
aI149
aI311
aI407
aI1391
aI717
aI1765
aI14887
aI14381
aI37483
aI29587
asS'd'
I6374
sa(dp12748
S'a'
I5495
sS's'
I17
sS'm'
(lp12749
I1
aI3
aI5
aI5
aI19
aI31
aI93
aI5
aI507
aI193
aI1735
aI3841
aI7895
aI9853
aI10317
aI14867
aI49529
asS'd'
I6375
sa(dp12750
S'a'
I5501
sS's'
I17
sS'm'
(lp12751
I1
aI3
aI7
aI7
aI19
aI3
aI99
aI201
aI479
aI313
aI693
aI3435
aI5453
aI1157
aI23127
aI49005
aI20167
asS'd'
I6376
sa(dp12752
S'a'
I5502
sS's'
I17
sS'm'
(lp12753
I1
aI3
aI7
aI9
aI15
aI21
aI123
aI41
aI19
aI281
aI1837
aI2589
aI1003
aI1993
aI18345
aI10039
aI89325
asS'd'
I6377
sa(dp12754
S'a'
I5505
sS's'
I17
sS'm'
(lp12755
I1
aI3
aI5
aI1
aI19
aI21
aI77
aI151
aI145
aI951
aI2017
aI609
aI5847
aI4475
aI12439
aI6357
aI108277
asS'd'
I6378
sa(dp12756
S'a'
I5512
sS's'
I17
sS'm'
(lp12757
I1
aI1
aI1
aI9
aI17
aI21
aI91
aI91
aI111
aI951
aI497
aI1759
aI503
aI12787
aI25117
aI24323
aI96447
asS'd'
I6379
sa(dp12758
S'a'
I5523
sS's'
I17
sS'm'
(lp12759
I1
aI1
aI3
aI11
aI13
aI9
aI73
aI205
aI329
aI243
aI1187
aI829
aI2821
aI5563
aI14391
aI771
aI116441
asS'd'
I6380
sa(dp12760
S'a'
I5529
sS's'
I17
sS'm'
(lp12761
I1
aI1
aI1
aI1
aI11
aI57
aI39
aI221
aI41
aI521
aI1541
aI3515
aI2367
aI4179
aI21039
aI52943
aI11627
asS'd'
I6381
sa(dp12762
S'a'
I5548
sS's'
I17
sS'm'
(lp12763
I1
aI3
aI3
aI3
aI23
aI13
aI103
aI125
aI67
aI217
aI863
aI3755
aI213
aI12657
aI31399
aI3771
aI54107
asS'd'
I6382
sa(dp12764
S'a'
I5551
sS's'
I17
sS'm'
(lp12765
I1
aI3
aI3
aI7
aI3
aI9
aI107
aI217
aI497
aI935
aI519
aI3041
aI323
aI14895
aI5695
aI28789
aI36085
asS'd'
I6383
sa(dp12766
S'a'
I5553
sS's'
I17
sS'm'
(lp12767
I1
aI1
aI5
aI11
aI23
aI33
aI81
aI23
aI167
aI3
aI1683
aI2279
aI5365
aI847
aI14717
aI9689
aI64481
asS'd'
I6384
sa(dp12768
S'a'
I5565
sS's'
I17
sS'm'
(lp12769
I1
aI3
aI1
aI7
aI1
aI15
aI107
aI93
aI429
aI363
aI1745
aI1459
aI5879
aI8351
aI17527
aI44001
aI70293
asS'd'
I6385
sa(dp12770
S'a'
I5568
sS's'
I17
sS'm'
(lp12771
I1
aI3
aI3
aI9
aI27
aI55
aI125
aI211
aI141
aI827
aI1239
aI663
aI4803
aI11067
aI32039
aI28091
aI56421
asS'd'
I6386
sa(dp12772
S'a'
I5577
sS's'
I17
sS'm'
(lp12773
I1
aI3
aI5
aI5
aI7
aI13
aI125
aI231
aI427
aI483
aI967
aI549
aI3105
aI13919
aI3017
aI39207
aI23253
asS'd'
I6387
sa(dp12774
S'a'
I5578
sS's'
I17
sS'm'
(lp12775
I1
aI3
aI7
aI3
aI21
aI29
aI79
aI67
aI39
aI451
aI157
aI337
aI3585
aI3621
aI9545
aI31205
aI63201
asS'd'
I6388
sa(dp12776
S'a'
I5583
sS's'
I17
sS'm'
(lp12777
I1
aI3
aI1
aI1
aI29
aI25
aI77
aI57
aI167
aI899
aI95
aI2487
aI3743
aI5381
aI3637
aI56289
aI39453
asS'd'
I6389
sa(dp12778
S'a'
I5585
sS's'
I17
sS'm'
(lp12779
I1
aI1
aI1
aI9
aI29
aI19
aI41
aI97
aI75
aI199
aI1709
aI483
aI4099
aI3113
aI10953
aI20659
aI109273
asS'd'
I6390
sa(dp12780
S'a'
I5588
sS's'
I17
sS'm'
(lp12781
I1
aI3
aI5
aI15
aI13
aI9
aI83
aI43
aI111
aI789
aI965
aI4061
aI1239
aI14577
aI10113
aI26359
aI52609
asS'd'
I6391
sa(dp12782
S'a'
I5613
sS's'
I17
sS'm'
(lp12783
I1
aI3
aI5
aI5
aI11
aI39
aI113
aI31
aI457
aI119
aI725
aI831
aI4143
aI5675
aI27431
aI12431
aI94977
asS'd'
I6392
sa(dp12784
S'a'
I5614
sS's'
I17
sS'm'
(lp12785
I1
aI1
aI3
aI3
aI25
aI17
aI93
aI253
aI307
aI625
aI143
aI1061
aI4415
aI3563
aI3313
aI53527
aI29537
asS'd'
I6393
sa(dp12786
S'a'
I5616
sS's'
I17
sS'm'
(lp12787
I1
aI3
aI5
aI3
aI29
aI41
aI43
aI109
aI147
aI919
aI1675
aI465
aI6101
aI12251
aI28915
aI15397
aI85233
asS'd'
I6394
sa(dp12788
S'a'
I5622
sS's'
I17
sS'm'
(lp12789
I1
aI1
aI1
aI1
aI31
aI25
aI59
aI187
aI439
aI561
aI559
aI413
aI1917
aI9319
aI27475
aI49715
aI32953
asS'd'
I6395
sa(dp12790
S'a'
I5631
sS's'
I17
sS'm'
(lp12791
I1
aI1
aI7
aI13
aI23
aI31
aI95
aI231
aI141
aI207
aI1373
aI2173
aI2905
aI169
aI23825
aI55071
aI6147
asS'd'
I6396
sa(dp12792
S'a'
I5637
sS's'
I17
sS'm'
(lp12793
I1
aI1
aI7
aI13
aI15
aI39
aI43
aI117
aI321
aI297
aI661
aI2941
aI7359
aI11675
aI15483
aI24093
aI7269
asS'd'
I6397
sa(dp12794
S'a'
I5638
sS's'
I17
sS'm'
(lp12795
I1
aI3
aI3
aI13
aI9
aI59
aI51
aI49
aI81
aI563
aI745
aI1843
aI295
aI4689
aI19847
aI42137
aI63197
asS'd'
I6398
sa(dp12796
S'a'
I5668
sS's'
I17
sS'm'
(lp12797
I1
aI3
aI1
aI9
aI5
aI33
aI21
aI199
aI509
aI927
aI1777
aI1349
aI3593
aI1065
aI24943
aI55667
aI73539
asS'd'
I6399
sa(dp12798
S'a'
I5675
sS's'
I17
sS'm'
(lp12799
I1
aI3
aI1
aI11
aI17
aI15
aI91
aI21
aI59
aI587
aI1207
aI543
aI6669
aI10861
aI24755
aI1789
aI91249
asS'd'
I6400
sa(dp12800
S'a'
I5683
sS's'
I17
sS'm'
(lp12801
I1
aI3
aI7
aI15
aI13
aI47
aI57
aI147
aI381
aI1021
aI921
aI1347
aI3847
aI5969
aI9075
aI39081
aI127241
asS'd'
I6401
sa(dp12802
S'a'
I5695
sS's'
I17
sS'm'
(lp12803
I1
aI3
aI3
aI15
aI19
aI15
aI1
aI97
aI203
aI409
aI1745
aI1217
aI2199
aI7945
aI24361
aI41771
aI123127
asS'd'
I6402
sa(dp12804
S'a'
I5703
sS's'
I17
sS'm'
(lp12805
I1
aI3
aI3
aI5
aI17
aI17
aI43
aI255
aI179
aI717
aI1993
aI645
aI6527
aI1533
aI32719
aI27481
aI122425
asS'd'
I6403
sa(dp12806
S'a'
I5710
sS's'
I17
sS'm'
(lp12807
I1
aI3
aI5
aI9
aI13
aI59
aI15
aI157
aI373
aI937
aI27
aI3325
aI2297
aI89
aI10861
aI48615
aI16083
asS'd'
I6404
sa(dp12808
S'a'
I5715
sS's'
I17
sS'm'
(lp12809
I1
aI3
aI1
aI3
aI19
aI27
aI109
aI243
aI189
aI17
aI99
aI1879
aI695
aI11329
aI12467
aI6053
aI41749
asS'd'
I6405
sa(dp12810
S'a'
I5727
sS's'
I17
sS'm'
(lp12811
I1
aI1
aI5
aI5
aI23
aI41
aI103
aI69
aI171
aI917
aI1303
aI2101
aI617
aI10017
aI26525
aI11009
aI66137
asS'd'
I6406
sa(dp12812
S'a'
I5738
sS's'
I17
sS'm'
(lp12813
I1
aI1
aI1
aI9
aI21
aI45
aI47
aI171
aI455
aI257
aI411
aI4021
aI6995
aI12881
aI4793
aI51193
aI60775
asS'd'
I6407
sa(dp12814
S'a'
I5752
sS's'
I17
sS'm'
(lp12815
I1
aI3
aI7
aI5
aI25
aI31
aI89
aI53
aI321
aI593
aI1795
aI2435
aI3833
aI2767
aI17241
aI63373
aI25457
asS'd'
I6408
sa(dp12816
S'a'
I5767
sS's'
I17
sS'm'
(lp12817
I1
aI3
aI1
aI1
aI3
aI45
aI19
aI255
aI179
aI991
aI1407
aI3683
aI1435
aI6803
aI12215
aI12835
aI2005
asS'd'
I6409
sa(dp12818
S'a'
I5773
sS's'
I17
sS'm'
(lp12819
I1
aI3
aI7
aI3
aI17
aI5
aI117
aI251
aI71
aI983
aI1391
aI3499
aI5119
aI7257
aI7325
aI16565
aI6321
asS'd'
I6410
sa(dp12820
S'a'
I5776
sS's'
I17
sS'm'
(lp12821
I1
aI3
aI5
aI7
aI5
aI49
aI47
aI201
aI297
aI485
aI1879
aI2205
aI4903
aI13619
aI22537
aI5479
aI121625
asS'd'
I6411
sa(dp12822
S'a'
I5781
sS's'
I17
sS'm'
(lp12823
I1
aI1
aI3
aI5
aI27
aI27
aI87
aI61
aI145
aI943
aI343
aI1639
aI6307
aI4549
aI20765
aI33479
aI113697
asS'd'
I6412
sa(dp12824
S'a'
I5791
sS's'
I17
sS'm'
(lp12825
I1
aI1
aI3
aI9
aI17
aI5
aI101
aI129
aI305
aI653
aI1901
aI3901
aI6361
aI2369
aI7449
aI55259
aI75215
asS'd'
I6413
sa(dp12826
S'a'
I5792
sS's'
I17
sS'm'
(lp12827
I1
aI1
aI7
aI5
aI31
aI45
aI117
aI55
aI335
aI827
aI1309
aI2603
aI2111
aI11005
aI14747
aI56999
aI97373
asS'd'
I6414
sa(dp12828
S'a'
I5795
sS's'
I17
sS'm'
(lp12829
I1
aI1
aI7
aI11
aI29
aI29
aI81
aI175
aI169
aI453
aI293
aI2589
aI1057
aI15795
aI32397
aI65433
aI79455
asS'd'
I6415
sa(dp12830
S'a'
I5798
sS's'
I17
sS'm'
(lp12831
I1
aI1
aI1
aI5
aI11
aI7
aI13
aI249
aI29
aI407
aI1289
aI2385
aI8113
aI15327
aI4029
aI32005
aI105901
asS'd'
I6416
sa(dp12832
S'a'
I5801
sS's'
I17
sS'm'
(lp12833
I1
aI1
aI5
aI5
aI7
aI61
aI103
aI141
aI109
aI391
aI631
aI821
aI1479
aI14771
aI25057
aI1415
aI8081
asS'd'
I6417
sa(dp12834
S'a'
I5810
sS's'
I17
sS'm'
(lp12835
I1
aI3
aI1
aI1
aI9
aI37
aI17
aI231
aI501
aI745
aI1695
aI45
aI7797
aI2945
aI5529
aI34747
aI39069
asS'd'
I6418
sa(dp12836
S'a'
I5812
sS's'
I17
sS'm'
(lp12837
I1
aI1
aI7
aI9
aI21
aI59
aI103
aI103
aI33
aI875
aI723
aI3477
aI4729
aI7311
aI29979
aI60901
aI72187
asS'd'
I6419
sa(dp12838
S'a'
I5836
sS's'
I17
sS'm'
(lp12839
I1
aI3
aI3
aI3
aI15
aI63
aI93
aI237
aI203
aI635
aI1189
aI2035
aI6499
aI9943
aI9133
aI62977
aI29657
asS'd'
I6420
sa(dp12840
S'a'
I5839
sS's'
I17
sS'm'
(lp12841
I1
aI1
aI1
aI9
aI3
aI11
aI63
aI207
aI95
aI563
aI775
aI3009
aI7125
aI13141
aI4489
aI16343
aI120951
asS'd'
I6421
sa(dp12842
S'a'
I5841
sS's'
I17
sS'm'
(lp12843
I1
aI1
aI3
aI1
aI21
aI57
aI15
aI217
aI185
aI305
aI463
aI1597
aI6529
aI4989
aI14011
aI11265
aI131031
asS'd'
I6422
sa(dp12844
S'a'
I5867
sS's'
I17
sS'm'
(lp12845
I1
aI3
aI5
aI15
aI17
aI61
aI35
aI127
aI411
aI579
aI1349
aI615
aI3293
aI8475
aI9773
aI30635
aI117639
asS'd'
I6423
sa(dp12846
S'a'
I5870
sS's'
I17
sS'm'
(lp12847
I1
aI1
aI7
aI9
aI11
aI3
aI55
aI105
aI305
aI223
aI1899
aI2217
aI1261
aI9831
aI23693
aI3013
aI30489
asS'd'
I6424
sa(dp12848
S'a'
I5877
sS's'
I17
sS'm'
(lp12849
I1
aI3
aI7
aI15
aI15
aI29
aI1
aI99
aI67
aI293
aI499
aI1941
aI5303
aI1329
aI24547
aI14065
aI7927
asS'd'
I6425
sa(dp12850
S'a'
I5881
sS's'
I17
sS'm'
(lp12851
I1
aI1
aI5
aI11
aI17
aI55
aI71
aI49
aI499
aI435
aI985
aI2803
aI6139
aI1503
aI24167
aI47181
aI102529
asS'd'
I6426
sa(dp12852
S'a'
I5899
sS's'
I17
sS'm'
(lp12853
I1
aI3
aI5
aI1
aI19
aI53
aI71
aI17
aI63
aI469
aI1871
aI2051
aI357
aI11661
aI5689
aI36373
aI13379
asS'd'
I6427
sa(dp12854
S'a'
I5914
sS's'
I17
sS'm'
(lp12855
I1
aI1
aI5
aI1
aI27
aI47
aI23
aI247
aI59
aI381
aI1895
aI2453
aI3665
aI5487
aI24081
aI50501
aI91659
asS'd'
I6428
sa(dp12856
S'a'
I5925
sS's'
I17
sS'm'
(lp12857
I1
aI1
aI5
aI7
aI29
aI19
aI3
aI33
aI83
aI301
aI133
aI3603
aI5133
aI16171
aI22905
aI36271
aI10405
asS'd'
I6429
sa(dp12858
S'a'
I5929
sS's'
I17
sS'm'
(lp12859
I1
aI3
aI7
aI9
aI11
aI23
aI57
aI87
aI9
aI731
aI631
aI3703
aI2593
aI12851
aI7115
aI8801
aI108919
asS'd'
I6430
sa(dp12860
S'a'
I5943
sS's'
I17
sS'm'
(lp12861
I1
aI3
aI3
aI3
aI23
aI35
aI33
aI99
aI343
aI837
aI231
aI3921
aI6975
aI15093
aI15049
aI64623
aI123523
asS'd'
I6431
sa(dp12862
S'a'
I5949
sS's'
I17
sS'm'
(lp12863
I1
aI1
aI7
aI11
aI15
aI61
aI113
aI103
aI501
aI57
aI1345
aI3155
aI2965
aI4433
aI10605
aI43765
aI42169
asS'd'
I6432
sa(dp12864
S'a'
I5962
sS's'
I17
sS'm'
(lp12865
I1
aI1
aI7
aI13
aI7
aI53
aI91
aI121
aI229
aI127
aI103
aI833
aI7829
aI1571
aI10847
aI20861
aI101155
asS'd'
I6433
sa(dp12866
S'a'
I5969
sS's'
I17
sS'm'
(lp12867
I1
aI3
aI7
aI1
aI9
aI25
aI71
aI103
aI37
aI473
aI1133
aI1129
aI1651
aI6965
aI6937
aI16597
aI20439
asS'd'
I6434
sa(dp12868
S'a'
I5976
sS's'
I17
sS'm'
(lp12869
I1
aI1
aI5
aI9
aI1
aI9
aI47
aI131
aI285
aI967
aI1869
aI1075
aI8127
aI135
aI15575
aI38569
aI123729
asS'd'
I6435
sa(dp12870
S'a'
I5988
sS's'
I17
sS'm'
(lp12871
I1
aI1
aI7
aI9
aI5
aI31
aI33
aI227
aI347
aI41
aI2025
aI3755
aI857
aI7805
aI13121
aI38307
aI125825
asS'd'
I6436
sa(dp12872
S'a'
I5997
sS's'
I17
sS'm'
(lp12873
I1
aI3
aI5
aI7
aI11
aI11
aI19
aI55
aI23
aI627
aI1477
aI3093
aI2779
aI7653
aI7165
aI23053
aI76123
asS'd'
I6437
sa(dp12874
S'a'
I6006
sS's'
I17
sS'm'
(lp12875
I1
aI1
aI3
aI1
aI3
aI47
aI83
aI89
aI177
aI381
aI1247
aI141
aI7051
aI6443
aI27369
aI34323
aI43063
asS'd'
I6438
sa(dp12876
S'a'
I6010
sS's'
I17
sS'm'
(lp12877
I1
aI1
aI7
aI7
aI13
aI15
aI55
aI223
aI351
aI525
aI1051
aI3009
aI5443
aI11499
aI8335
aI37949
aI69149
asS'd'
I6439
sa(dp12878
S'a'
I6016
sS's'
I17
sS'm'
(lp12879
I1
aI1
aI1
aI3
aI13
aI61
aI89
aI33
aI129
aI921
aI1905
aI201
aI3141
aI5531
aI135
aI34103
aI56883
asS'd'
I6440
sa(dp12880
S'a'
I6022
sS's'
I17
sS'm'
(lp12881
I1
aI1
aI5
aI13
aI17
aI27
aI13
aI163
aI169
aI471
aI1263
aI1421
aI7015
aI7927
aI21027
aI58001
aI26739
asS'd'
I6441
sa(dp12882
S'a'
I6026
sS's'
I17
sS'm'
(lp12883
I1
aI1
aI1
aI15
aI19
aI49
aI109
aI207
aI245
aI49
aI1271
aI3635
aI2561
aI5091
aI24415
aI59195
aI67701
asS'd'
I6442
sa(dp12884
S'a'
I6031
sS's'
I17
sS'm'
(lp12885
I1
aI3
aI5
aI7
aI27
aI57
aI99
aI155
aI461
aI595
aI1859
aI1727
aI857
aI4993
aI31733
aI42141
aI10035
asS'd'
I6443
sa(dp12886
S'a'
I6040
sS's'
I17
sS'm'
(lp12887
I1
aI1
aI1
aI15
aI11
aI11
aI85
aI9
aI251
aI375
aI155
aI379
aI7501
aI12559
aI32583
aI36317
aI4675
asS'd'
I6444
sa(dp12888
S'a'
I6043
sS's'
I17
sS'm'
(lp12889
I1
aI1
aI5
aI13
aI19
aI57
aI81
aI69
aI201
aI293
aI593
aI3169
aI4519
aI9057
aI16685
aI12847
aI123797
asS'd'
I6445
sa(dp12890
S'a'
I6050
sS's'
I17
sS'm'
(lp12891
I1
aI3
aI1
aI5
aI5
aI1
aI19
aI243
aI345
aI661
aI561
aI3549
aI2541
aI5887
aI25879
aI41467
aI72799
asS'd'
I6446
sa(dp12892
S'a'
I6059
sS's'
I17
sS'm'
(lp12893
I1
aI1
aI5
aI13
aI15
aI51
aI67
aI61
aI79
aI89
aI447
aI1471
aI4915
aI10637
aI10901
aI48157
aI103545
asS'd'
I6447
sa(dp12894
S'a'
I6079
sS's'
I17
sS'm'
(lp12895
I1
aI3
aI5
aI13
aI31
aI25
aI73
aI129
aI435
aI659
aI1851
aI3595
aI753
aI7717
aI10927
aI30115
aI109221
asS'd'
I6448
sa(dp12896
S'a'
I6099
sS's'
I17
sS'm'
(lp12897
I1
aI1
aI1
aI3
aI25
aI3
aI121
aI43
aI349
aI205
aI1209
aI2671
aI6445
aI8755
aI7171
aI58631
aI74319
asS'd'
I6449
sa(dp12898
S'a'
I6101
sS's'
I17
sS'm'
(lp12899
I1
aI1
aI3
aI1
aI11
aI15
aI83
aI37
aI483
aI65
aI759
aI1835
aI3883
aI1693
aI30051
aI61077
aI1187
asS'd'
I6450
sa(dp12900
S'a'
I6105
sS's'
I17
sS'm'
(lp12901
I1
aI3
aI7
aI15
aI29
aI23
aI85
aI77
aI139
aI903
aI1821
aI943
aI6453
aI1523
aI18539
aI49039
aI110787
asS'd'
I6451
sa(dp12902
S'a'
I6108
sS's'
I17
sS'm'
(lp12903
I1
aI1
aI7
aI15
aI15
aI17
aI69
aI253
aI507
aI921
aI523
aI79
aI747
aI4011
aI25795
aI42029
aI88309
asS'd'
I6452
sa(dp12904
S'a'
I6124
sS's'
I17
sS'm'
(lp12905
I1
aI1
aI7
aI3
aI25
aI47
aI119
aI83
aI313
aI45
aI985
aI145
aI205
aI3407
aI9013
aI64517
aI115811
asS'd'
I6453
sa(dp12906
S'a'
I6132
sS's'
I17
sS'm'
(lp12907
I1
aI1
aI7
aI1
aI29
aI21
aI9
aI123
aI97
aI545
aI1987
aI2979
aI6901
aI12667
aI23325
aI63635
aI70593
asS'd'
I6454
sa(dp12908
S'a'
I6145
sS's'
I17
sS'm'
(lp12909
I1
aI3
aI7
aI3
aI23
aI45
aI81
aI255
aI41
aI29
aI1493
aI4065
aI3201
aI10479
aI17193
aI39999
aI55493
asS'd'
I6455
sa(dp12910
S'a'
I6146
sS's'
I17
sS'm'
(lp12911
I1
aI3
aI1
aI3
aI9
aI43
aI43
aI135
aI235
aI603
aI481
aI3139
aI2729
aI14759
aI7269
aI7995
aI110351
asS'd'
I6456
sa(dp12912
S'a'
I6151
sS's'
I17
sS'm'
(lp12913
I1
aI3
aI1
aI11
aI17
aI35
aI113
aI93
aI417
aI967
aI755
aI659
aI3115
aI16163
aI22997
aI38205
aI126961
asS'd'
I6457
sa(dp12914
S'a'
I6152
sS's'
I17
sS'm'
(lp12915
I1
aI1
aI7
aI11
aI29
aI57
aI81
aI235
aI93
aI869
aI475
aI825
aI6269
aI15819
aI14977
aI53057
aI116021
asS'd'
I6458
sa(dp12916
S'a'
I6158
sS's'
I17
sS'm'
(lp12917
I1
aI1
aI7
aI13
aI5
aI61
aI5
aI241
aI245
aI673
aI1651
aI3367
aI2355
aI713
aI20107
aI30133
aI735
asS'd'
I6459
sa(dp12918
S'a'
I6160
sS's'
I17
sS'm'
(lp12919
I1
aI1
aI5
aI9
aI21
aI3
aI121
aI241
aI129
aI703
aI1435
aI1943
aI5087
aI13123
aI30023
aI58287
aI50377
asS'd'
I6460
sa(dp12920
S'a'
I6163
sS's'
I17
sS'm'
(lp12921
I1
aI1
aI1
aI15
aI23
aI27
aI67
aI197
aI123
aI629
aI169
aI3303
aI1679
aI11051
aI16875
aI28055
aI12379
asS'd'
I6461
sa(dp12922
S'a'
I6165
sS's'
I17
sS'm'
(lp12923
I1
aI1
aI3
aI3
aI7
aI63
aI97
aI43
aI89
aI739
aI779
aI2893
aI7763
aI6351
aI26135
aI44647
aI127987
asS'd'
I6462
sa(dp12924
S'a'
I6170
sS's'
I17
sS'm'
(lp12925
I1
aI3
aI3
aI9
aI31
aI59
aI95
aI131
aI131
aI321
aI1125
aI127
aI4865
aI145
aI26237
aI47871
aI114549
asS'd'
I6463
sa(dp12926
S'a'
I6182
sS's'
I17
sS'm'
(lp12927
I1
aI3
aI3
aI13
aI21
aI3
aI33
aI17
aI445
aI693
aI1599
aI2517
aI1679
aI2237
aI15053
aI30983
aI106755
asS'd'
I6464
sa(dp12928
S'a'
I6196
sS's'
I17
sS'm'
(lp12929
I1
aI1
aI5
aI13
aI31
aI37
aI49
aI67
aI403
aI27
aI575
aI1795
aI3385
aI1067
aI585
aI60277
aI123189
asS'd'
I6465
sa(dp12930
S'a'
I6199
sS's'
I17
sS'm'
(lp12931
I1
aI3
aI1
aI15
aI13
aI35
aI23
aI247
aI493
aI305
aI363
aI451
aI4011
aI3679
aI18281
aI31751
aI127933
asS'd'
I6466
sa(dp12932
S'a'
I6200
sS's'
I17
sS'm'
(lp12933
I1
aI1
aI7
aI5
aI21
aI45
aI123
aI253
aI469
aI267
aI985
aI2349
aI3427
aI7653
aI25685
aI13747
aI531
asS'd'
I6467
sa(dp12934
S'a'
I6205
sS's'
I17
sS'm'
(lp12935
I1
aI1
aI5
aI11
aI7
aI59
aI105
aI209
aI27
aI847
aI593
aI3775
aI6165
aI1655
aI29867
aI28465
aI92193
asS'd'
I6468
sa(dp12936
S'a'
I6226
sS's'
I17
sS'm'
(lp12937
I1
aI3
aI1
aI11
aI7
aI25
aI101
aI81
aI233
aI311
aI9
aI2735
aI3951
aI485
aI10105
aI24489
aI649
asS'd'
I6469
sa(dp12938
S'a'
I6228
sS's'
I17
sS'm'
(lp12939
I1
aI3
aI1
aI7
aI27
aI5
aI115
aI243
aI295
aI659
aI215
aI1787
aI5131
aI2513
aI29201
aI21195
aI103383
asS'd'
I6470
sa(dp12940
S'a'
I6237
sS's'
I17
sS'm'
(lp12941
I1
aI3
aI5
aI13
aI29
aI21
aI7
aI57
aI345
aI467
aI1297
aI207
aI5115
aI335
aI6153
aI32959
aI125697
asS'd'
I6471
sa(dp12942
S'a'
I6247
sS's'
I17
sS'm'
(lp12943
I1
aI1
aI1
aI9
aI3
aI63
aI63
aI5
aI373
aI123
aI1265
aI2365
aI1623
aI1561
aI14805
aI17487
aI104787
asS'd'
I6472
sa(dp12944
S'a'
I6251
sS's'
I17
sS'm'
(lp12945
I1
aI3
aI1
aI5
aI15
aI13
aI55
aI69
aI251
aI341
aI463
aI2611
aI4793
aI12157
aI4669
aI11613
aI128705
asS'd'
I6473
sa(dp12946
S'a'
I6253
sS's'
I17
sS'm'
(lp12947
I1
aI3
aI7
aI13
aI19
aI7
aI93
aI149
aI453
aI693
aI1731
aI861
aI6971
aI943
aI18891
aI56547
aI34411
asS'd'
I6474
sa(dp12948
S'a'
I6256
sS's'
I17
sS'm'
(lp12949
I1
aI1
aI7
aI1
aI27
aI49
aI27
aI9
aI281
aI121
aI581
aI393
aI2583
aI1159
aI26989
aI39955
aI100765
asS'd'
I6475
sa(dp12950
S'a'
I6268
sS's'
I17
sS'm'
(lp12951
I1
aI1
aI3
aI9
aI3
aI43
aI97
aI207
aI311
aI617
aI1987
aI2559
aI2101
aI15791
aI30085
aI40713
aI41909
asS'd'
I6476
sa(dp12952
S'a'
I6272
sS's'
I17
sS'm'
(lp12953
I1
aI3
aI1
aI3
aI15
aI19
aI53
aI183
aI375
aI867
aI397
aI3203
aI4207
aI5381
aI25065
aI60357
aI88739
asS'd'
I6477
sa(dp12954
S'a'
I6275
sS's'
I17
sS'm'
(lp12955
I1
aI3
aI3
aI3
aI27
aI51
aI85
aI231
aI19
aI559
aI567
aI4049
aI4875
aI14201
aI11623
aI39763
aI57339
asS'd'
I6478
sa(dp12956
S'a'
I6281
sS's'
I17
sS'm'
(lp12957
I1
aI1
aI5
aI1
aI19
aI7
aI81
aI249
aI41
aI789
aI985
aI3725
aI4053
aI4255
aI9861
aI1609
aI29511
asS'd'
I6479
sa(dp12958
S'a'
I6289
sS's'
I17
sS'm'
(lp12959
I1
aI3
aI5
aI5
aI21
aI13
aI49
aI41
aI367
aI283
aI1161
aI2753
aI4733
aI3691
aI27931
aI53055
aI83625
asS'd'
I6480
sa(dp12960
S'a'
I6335
sS's'
I17
sS'm'
(lp12961
I1
aI3
aI5
aI11
aI29
aI47
aI95
aI51
aI265
aI85
aI385
aI833
aI7957
aI14985
aI7017
aI41937
aI41377
asS'd'
I6481
sa(dp12962
S'a'
I6338
sS's'
I17
sS'm'
(lp12963
I1
aI1
aI7
aI5
aI1
aI23
aI17
aI191
aI185
aI323
aI515
aI3183
aI7685
aI7361
aI21143
aI5227
aI110297
asS'd'
I6482
sa(dp12964
S'a'
I6355
sS's'
I17
sS'm'
(lp12965
I1
aI3
aI3
aI7
aI11
aI39
aI31
aI97
aI237
aI497
aI1649
aI3529
aI6153
aI5055
aI29021
aI35125
aI121581
asS'd'
I6483
sa(dp12966
S'a'
I6362
sS's'
I17
sS'm'
(lp12967
I1
aI3
aI5
aI3
aI17
aI47
aI105
aI75
aI55
aI343
aI595
aI2447
aI5575
aI10673
aI32015
aI37541
aI127867
asS'd'
I6484
sa(dp12968
S'a'
I6373
sS's'
I17
sS'm'
(lp12969
I1
aI3
aI1
aI7
aI19
aI39
aI31
aI135
aI167
aI979
aI219
aI1353
aI489
aI9667
aI27107
aI55565
aI72291
asS'd'
I6485
sa(dp12970
S'a'
I6386
sS's'
I17
sS'm'
(lp12971
I1
aI1
aI3
aI13
aI31
aI49
aI87
aI93
aI235
aI577
aI1551
aI2663
aI387
aI1129
aI26683
aI31285
aI15913
asS'd'
I6486
sa(dp12972
S'a'
I6388
sS's'
I17
sS'm'
(lp12973
I1
aI3
aI3
aI7
aI15
aI29
aI61
aI33
aI115
aI511
aI1781
aI2029
aI4265
aI6745
aI1467
aI34415
aI40907
asS'd'
I6487
sa(dp12974
S'a'
I6391
sS's'
I17
sS'm'
(lp12975
I1
aI1
aI7
aI5
aI1
aI55
aI13
aI129
aI167
aI937
aI79
aI2047
aI3589
aI1979
aI4153
aI15229
aI85745
asS'd'
I6488
sa(dp12976
S'a'
I6397
sS's'
I17
sS'm'
(lp12977
I1
aI1
aI7
aI15
aI15
aI25
aI89
aI129
aI31
aI435
aI1359
aI49
aI2659
aI2829
aI8741
aI25215
aI4239
asS'd'
I6489
sa(dp12978
S'a'
I6405
sS's'
I17
sS'm'
(lp12979
I1
aI3
aI5
aI3
aI11
aI39
aI95
aI239
aI187
aI615
aI1481
aI3509
aI1133
aI13497
aI24833
aI59635
aI45695
asS'd'
I6490
sa(dp12980
S'a'
I6406
sS's'
I17
sS'm'
(lp12981
I1
aI1
aI5
aI3
aI19
aI17
aI17
aI235
aI315
aI943
aI883
aI1381
aI7129
aI15709
aI9847
aI41183
aI116071
asS'd'
I6491
sa(dp12982
S'a'
I6410
sS's'
I17
sS'm'
(lp12983
I1
aI1
aI1
aI3
aI9
aI63
aI109
aI209
aI309
aI1015
aI1391
aI2617
aI1481
aI6483
aI4151
aI28063
aI49887
asS'd'
I6492
sa(dp12984
S'a'
I6417
sS's'
I17
sS'm'
(lp12985
I1
aI1
aI5
aI13
aI23
aI37
aI31
aI89
aI501
aI461
aI41
aI931
aI7863
aI15499
aI25635
aI16995
aI41651
asS'd'
I6493
sa(dp12986
S'a'
I6443
sS's'
I17
sS'm'
(lp12987
I1
aI1
aI1
aI9
aI29
aI29
aI125
aI161
aI219
aI439
aI1465
aI1615
aI7483
aI7497
aI1121
aI49693
aI30269
asS'd'
I6494
sa(dp12988
S'a'
I6457
sS's'
I17
sS'm'
(lp12989
I1
aI3
aI1
aI5
aI7
aI43
aI27
aI161
aI431
aI375
aI419
aI2995
aI527
aI8207
aI747
aI18491
aI15351
asS'd'
I6495
sa(dp12990
S'a'
I6468
sS's'
I17
sS'm'
(lp12991
I1
aI1
aI3
aI13
aI25
aI21
aI67
aI177
aI9
aI453
aI1171
aI65
aI2845
aI16147
aI12699
aI30905
aI122255
asS'd'
I6496
sa(dp12992
S'a'
I6475
sS's'
I17
sS'm'
(lp12993
I1
aI3
aI1
aI5
aI29
aI47
aI77
aI251
aI473
aI385
aI947
aI3239
aI5375
aI13617
aI10639
aI36005
aI95821
asS'd'
I6497
sa(dp12994
S'a'
I6486
sS's'
I17
sS'm'
(lp12995
I1
aI3
aI1
aI15
aI13
aI1
aI75
aI223
aI509
aI19
aI175
aI1541
aI637
aI5711
aI1097
aI44901
aI35277
asS'd'
I6498
sa(dp12996
S'a'
I6489
sS's'
I17
sS'm'
(lp12997
I1
aI3
aI3
aI7
aI3
aI27
aI17
aI151
aI39
aI583
aI391
aI2739
aI7339
aI2051
aI17005
aI49573
aI85969
asS'd'
I6499
sa(dp12998
S'a'
I6495
sS's'
I17
sS'm'
(lp12999
I1
aI3
aI1
aI11
aI3
aI25
aI119
aI125
aI17
aI629
aI201
aI2347
aI2923
aI1273
aI14871
aI58299
aI97667
asS'd'
I6500
sa(dp13000
S'a'
I6499
sS's'
I17
sS'm'
(lp13001
I1
aI1
aI7
aI1
aI31
aI39
aI11
aI121
aI339
aI667
aI1863
aI3479
aI1895
aI11319
aI5683
aI64969
aI9261
asS'd'
I6501
sa(dp13002
S'a'
I6505
sS's'
I17
sS'm'
(lp13003
I1
aI1
aI5
aI9
aI27
aI61
aI101
aI221
aI221
aI583
aI287
aI707
aI5931
aI4225
aI29537
aI46097
aI114361
asS'd'
I6502
sa(dp13004
S'a'
I6511
sS's'
I17
sS'm'
(lp13005
I1
aI1
aI1
aI9
aI23
aI47
aI1
aI35
aI85
aI1021
aI151
aI3153
aI3867
aI971
aI31573
aI4745
aI107639
asS'd'
I6503
sa(dp13006
S'a'
I6520
sS's'
I17
sS'm'
(lp13007
I1
aI1
aI7
aI13
aI15
aI15
aI63
aI37
aI291
aI907
aI411
aI1571
aI6415
aI7443
aI26635
aI27945
aI130909
asS'd'
I6504
sa(dp13008
S'a'
I6529
sS's'
I17
sS'm'
(lp13009
I1
aI3
aI1
aI9
aI21
aI13
aI77
aI147
aI485
aI107
aI235
aI481
aI2389
aI957
aI11493
aI53033
aI46373
asS'd'
I6505
sa(dp13010
S'a'
I6542
sS's'
I17
sS'm'
(lp13011
I1
aI3
aI5
aI7
aI3
aI55
aI125
aI237
aI205
aI411
aI1911
aI4053
aI5983
aI15489
aI29333
aI44727
aI62167
asS'd'
I6506
sa(dp13012
S'a'
I6547
sS's'
I17
sS'm'
(lp13013
I1
aI1
aI3
aI3
aI17
aI3
aI59
aI239
aI209
aI495
aI447
aI3427
aI3425
aI2347
aI10057
aI26147
aI52243
asS'd'
I6507
sa(dp13014
S'a'
I6550
sS's'
I17
sS'm'
(lp13015
I1
aI1
aI3
aI1
aI11
aI31
aI3
aI139
aI441
aI997
aI295
aI1267
aI2181
aI6047
aI32419
aI62657
aI24921
asS'd'
I6508
sa(dp13016
S'a'
I6554
sS's'
I17
sS'm'
(lp13017
I1
aI3
aI7
aI15
aI5
aI3
aI11
aI9
aI211
aI701
aI1987
aI2611
aI6195
aI14379
aI22919
aI15785
aI52149
asS'd'
I6509
sa(dp13018
S'a'
I6556
sS's'
I17
sS'm'
(lp13019
I1
aI1
aI7
aI9
aI7
aI27
aI35
aI253
aI343
aI679
aI103
aI1217
aI3983
aI8677
aI17671
aI41347
aI89355
asS'd'
I6510
sa(dp13020
S'a'
I6560
sS's'
I17
sS'm'
(lp13021
I1
aI1
aI1
aI5
aI7
aI55
aI111
aI115
aI231
aI999
aI773
aI2111
aI3617
aI2469
aI16967
aI60735
aI24557
asS'd'
I6511
sa(dp13022
S'a'
I6569
sS's'
I17
sS'm'
(lp13023
I1
aI3
aI5
aI1
aI29
aI5
aI77
aI217
aI131
aI307
aI473
aI3595
aI2713
aI6503
aI18459
aI57245
aI91897
asS'd'
I6512
sa(dp13024
S'a'
I6572
sS's'
I17
sS'm'
(lp13025
I1
aI3
aI5
aI13
aI9
aI33
aI93
aI31
aI59
aI343
aI1337
aI1971
aI7593
aI15629
aI22693
aI19885
aI4139
asS'd'
I6513
sa(dp13026
S'a'
I6590
sS's'
I17
sS'm'
(lp13027
I1
aI3
aI3
aI3
aI21
aI33
aI115
aI205
aI373
aI587
aI739
aI669
aI8065
aI5339
aI16507
aI29455
aI15863
asS'd'
I6514
sa(dp13028
S'a'
I6592
sS's'
I17
sS'm'
(lp13029
I1
aI3
aI5
aI11
aI9
aI43
aI45
aI41
aI91
aI87
aI19
aI1523
aI5059
aI9403
aI6739
aI36893
aI6395
asS'd'
I6515
sa(dp13030
S'a'
I6601
sS's'
I17
sS'm'
(lp13031
I1
aI1
aI5
aI15
aI19
aI43
aI81
aI3
aI401
aI621
aI1839
aI1443
aI179
aI8085
aI27021
aI7757
aI95011
asS'd'
I6516
sa(dp13032
S'a'
I6610
sS's'
I17
sS'm'
(lp13033
I1
aI3
aI5
aI15
aI19
aI21
aI45
aI167
aI77
aI977
aI309
aI431
aI3437
aI8327
aI12895
aI50521
aI68473
asS'd'
I6517
sa(dp13034
S'a'
I6632
sS's'
I17
sS'm'
(lp13035
I1
aI3
aI3
aI15
aI7
aI21
aI49
aI169
aI327
aI271
aI7
aI785
aI1767
aI14747
aI7083
aI65223
aI24213
asS'd'
I6518
sa(dp13036
S'a'
I6635
sS's'
I17
sS'm'
(lp13037
I1
aI1
aI5
aI9
aI9
aI51
aI101
aI197
aI507
aI839
aI1413
aI3131
aI331
aI15725
aI32293
aI60433
aI86759
asS'd'
I6519
sa(dp13038
S'a'
I6640
sS's'
I17
sS'm'
(lp13039
I1
aI1
aI7
aI1
aI17
aI39
aI127
aI201
aI341
aI607
aI1565
aI1615
aI1367
aI16043
aI28265
aI29139
aI63813
asS'd'
I6520
sa(dp13040
S'a'
I6643
sS's'
I17
sS'm'
(lp13041
I1
aI3
aI5
aI7
aI9
aI1
aI107
aI73
aI121
aI649
aI1385
aI3203
aI2897
aI8479
aI28519
aI34041
aI1359
asS'd'
I6521
sa(dp13042
S'a'
I6649
sS's'
I17
sS'm'
(lp13043
I1
aI1
aI7
aI7
aI21
aI55
aI19
aI13
aI415
aI647
aI2015
aI107
aI4167
aI5033
aI16849
aI41407
aI94387
asS'd'
I6522
sa(dp13044
S'a'
I6659
sS's'
I17
sS'm'
(lp13045
I1
aI3
aI5
aI13
aI31
aI27
aI107
aI95
aI425
aI679
aI55
aI3521
aI6737
aI11459
aI19995
aI64189
aI44323
asS'd'
I6523
sa(dp13046
S'a'
I6662
sS's'
I17
sS'm'
(lp13047
I1
aI1
aI3
aI9
aI17
aI47
aI29
aI167
aI17
aI63
aI5
aI2505
aI6483
aI14089
aI7127
aI7907
aI68555
asS'd'
I6524
sa(dp13048
S'a'
I6666
sS's'
I17
sS'm'
(lp13049
I1
aI1
aI5
aI3
aI29
aI3
aI87
aI107
aI227
aI893
aI1821
aI341
aI5481
aI13317
aI10637
aI8611
aI28625
asS'd'
I6525
sa(dp13050
S'a'
I6690
sS's'
I17
sS'm'
(lp13051
I1
aI1
aI1
aI13
aI11
aI19
aI59
aI157
aI397
aI103
aI1821
aI3913
aI3083
aI6053
aI1015
aI25475
aI94813
asS'd'
I6526
sa(dp13052
S'a'
I6692
sS's'
I17
sS'm'
(lp13053
I1
aI3
aI1
aI3
aI15
aI45
aI1
aI209
aI335
aI1015
aI539
aI2959
aI1711
aI2567
aI30169
aI147
aI25383
asS'd'
I6527
sa(dp13054
S'a'
I6704
sS's'
I17
sS'm'
(lp13055
I1
aI3
aI7
aI1
aI17
aI5
aI99
aI121
aI91
aI531
aI865
aI1667
aI5615
aI4729
aI7473
aI21445
aI37925
asS'd'
I6528
sa(dp13056
S'a'
I6713
sS's'
I17
sS'm'
(lp13057
I1
aI1
aI7
aI13
aI3
aI51
aI27
aI115
aI439
aI761
aI1121
aI1503
aI3047
aI2127
aI29253
aI48147
aI10813
asS'd'
I6529
sa(dp13058
S'a'
I6728
sS's'
I17
sS'm'
(lp13059
I1
aI3
aI7
aI15
aI1
aI51
aI33
aI161
aI509
aI159
aI1705
aI3365
aI7953
aI14027
aI3873
aI29609
aI33101
asS'd'
I6530
sa(dp13060
S'a'
I6731
sS's'
I17
sS'm'
(lp13061
I1
aI1
aI5
aI15
aI15
aI53
aI119
aI115
aI433
aI75
aI497
aI1259
aI1681
aI7715
aI24767
aI34647
aI82007
asS'd'
I6531
sa(dp13062
S'a'
I6734
sS's'
I17
sS'm'
(lp13063
I1
aI1
aI5
aI3
aI27
aI63
aI41
aI181
aI393
aI439
aI95
aI2765
aI7617
aI817
aI1311
aI18595
aI16921
asS'd'
I6532
sa(dp13064
S'a'
I6746
sS's'
I17
sS'm'
(lp13065
I1
aI3
aI1
aI15
aI31
aI7
aI57
aI89
aI371
aI745
aI475
aI3211
aI6893
aI10681
aI18547
aI28373
aI127787
asS'd'
I6533
sa(dp13066
S'a'
I6755
sS's'
I17
sS'm'
(lp13067
I1
aI3
aI5
aI13
aI5
aI55
aI45
aI167
aI147
aI833
aI765
aI1153
aI4037
aI8503
aI10751
aI49541
aI77489
asS'd'
I6534
sa(dp13068
S'a'
I6757
sS's'
I17
sS'm'
(lp13069
I1
aI3
aI1
aI11
aI11
aI7
aI45
aI167
aI431
aI759
aI1035
aI1367
aI1649
aI11711
aI4915
aI58915
aI72479
asS'd'
I6535
sa(dp13070
S'a'
I6764
sS's'
I17
sS'm'
(lp13071
I1
aI1
aI5
aI1
aI11
aI3
aI15
aI135
aI427
aI637
aI879
aI1667
aI6139
aI14759
aI25665
aI13083
aI67961
asS'd'
I6536
sa(dp13072
S'a'
I6772
sS's'
I17
sS'm'
(lp13073
I1
aI3
aI3
aI9
aI1
aI3
aI73
aI167
aI269
aI51
aI1481
aI3659
aI7863
aI7187
aI3951
aI10711
aI5909
asS'd'
I6537
sa(dp13074
S'a'
I6792
sS's'
I17
sS'm'
(lp13075
I1
aI1
aI3
aI3
aI9
aI53
aI101
aI209
aI109
aI691
aI1641
aI919
aI1083
aI6247
aI23041
aI44681
aI130105
asS'd'
I6538
sa(dp13076
S'a'
I6797
sS's'
I17
sS'm'
(lp13077
I1
aI3
aI7
aI5
aI21
aI55
aI127
aI9
aI437
aI225
aI1599
aI2575
aI5407
aI8099
aI20009
aI40339
aI110581
asS'd'
I6539
sa(dp13078
S'a'
I6821
sS's'
I17
sS'm'
(lp13079
I1
aI3
aI3
aI13
aI7
aI41
aI15
aI137
aI363
aI337
aI995
aI1215
aI3651
aI11011
aI27209
aI53927
aI78065
asS'd'
I6540
sa(dp13080
S'a'
I6822
sS's'
I17
sS'm'
(lp13081
I1
aI1
aI1
aI7
aI11
aI17
aI27
aI9
aI481
aI79
aI905
aI1297
aI811
aI10221
aI463
aI12979
aI114731
asS'd'
I6541
sa(dp13082
S'a'
I6831
sS's'
I17
sS'm'
(lp13083
I1
aI1
aI3
aI13
aI7
aI59
aI105
aI79
aI253
aI699
aI139
aI3823
aI4939
aI12955
aI32069
aI7255
aI18159
asS'd'
I6542
sa(dp13084
S'a'
I6834
sS's'
I17
sS'm'
(lp13085
I1
aI3
aI5
aI7
aI29
aI7
aI79
aI79
aI147
aI921
aI425
aI1423
aI5967
aI6397
aI17393
aI30009
aI84075
asS'd'
I6543
sa(dp13086
S'a'
I6851
sS's'
I17
sS'm'
(lp13087
I1
aI3
aI7
aI13
aI23
aI45
aI51
aI141
aI237
aI443
aI1101
aI309
aI4533
aI7479
aI22415
aI31517
aI120407
asS'd'
I6544
sa(dp13088
S'a'
I6858
sS's'
I17
sS'm'
(lp13089
I1
aI1
aI5
aI13
aI3
aI19
aI97
aI185
aI59
aI179
aI1343
aI2537
aI3165
aI16295
aI25005
aI49769
aI78007
asS'd'
I6545
sa(dp13090
S'a'
I6860
sS's'
I17
sS'm'
(lp13091
I1
aI3
aI7
aI15
aI11
aI53
aI127
aI195
aI309
aI121
aI1741
aI1415
aI225
aI15645
aI16365
aI38729
aI70061
asS'd'
I6546
sa(dp13092
S'a'
I6871
sS's'
I17
sS'm'
(lp13093
I1
aI3
aI7
aI11
aI29
aI35
aI47
aI109
aI179
aI3
aI849
aI2305
aI3035
aI15289
aI31569
aI28851
aI90057
asS'd'
I6547
sa(dp13094
S'a'
I6875
sS's'
I17
sS'm'
(lp13095
I1
aI1
aI7
aI1
aI13
aI27
aI93
aI119
aI439
aI45
aI623
aI1263
aI6595
aI6669
aI12981
aI64721
aI130109
asS'd'
I6548
sa(dp13096
S'a'
I6884
sS's'
I17
sS'm'
(lp13097
I1
aI1
aI7
aI13
aI5
aI43
aI43
aI99
aI395
aI417
aI795
aI3991
aI5601
aI13115
aI12803
aI52247
aI39245
asS'd'
I6549
sa(dp13098
S'a'
I6888
sS's'
I17
sS'm'
(lp13099
I1
aI3
aI3
aI3
aI15
aI61
aI85
aI91
aI407
aI391
aI359
aI3885
aI1925
aI4873
aI169
aI41727
aI129471
asS'd'
I6550
sa(dp13100
S'a'
I6894
sS's'
I17
sS'm'
(lp13101
I1
aI3
aI3
aI9
aI11
aI47
aI3
aI33
aI355
aI853
aI1329
aI1347
aI1995
aI8197
aI10015
aI787
aI66773
asS'd'
I6551
sa(dp13102
S'a'
I6919
sS's'
I17
sS'm'
(lp13103
I1
aI3
aI3
aI13
aI31
aI31
aI49
aI195
aI55
aI185
aI1743
aI3523
aI1781
aI8469
aI7623
aI55933
aI74953
asS'd'
I6552
sa(dp13104
S'a'
I6940
sS's'
I17
sS'm'
(lp13105
I1
aI3
aI5
aI15
aI29
aI31
aI5
aI45
aI149
aI71
aI2033
aI3171
aI4601
aI9941
aI15005
aI55709
aI74403
asS'd'
I6553
sa(dp13106
S'a'
I6950
sS's'
I17
sS'm'
(lp13107
I1
aI3
aI5
aI3
aI1
aI27
aI105
aI7
aI139
aI805
aI1877
aI915
aI1843
aI11897
aI29485
aI19275
aI44711
asS'd'
I6554
sa(dp13108
S'a'
I6959
sS's'
I17
sS'm'
(lp13109
I1
aI1
aI5
aI7
aI25
aI57
aI111
aI57
aI401
aI935
aI1685
aI2985
aI2015
aI13501
aI14581
aI53579
aI117011
asS'd'
I6555
sa(dp13110
S'a'
I6968
sS's'
I17
sS'm'
(lp13111
I1
aI1
aI5
aI11
aI13
aI47
aI63
aI137
aI145
aI77
aI1727
aI2629
aI7377
aI6311
aI537
aI13703
aI129503
asS'd'
I6556
sa(dp13112
S'a'
I6981
sS's'
I17
sS'm'
(lp13113
I1
aI1
aI7
aI9
aI5
aI49
aI67
aI51
aI163
aI989
aI845
aI7
aI2141
aI14467
aI3197
aI57581
aI121087
asS'd'
I6557
sa(dp13114
S'a'
I6988
sS's'
I17
sS'm'
(lp13115
I1
aI1
aI5
aI3
aI31
aI49
aI57
aI103
aI171
aI491
aI1109
aI1255
aI4353
aI11927
aI29525
aI16685
aI48469
asS'd'
I6558
sa(dp13116
S'a'
I6996
sS's'
I17
sS'm'
(lp13117
I1
aI1
aI1
aI3
aI7
aI29
aI17
aI111
aI339
aI747
aI763
aI179
aI7747
aI2483
aI18415
aI45301
aI25155
asS'd'
I6559
sa(dp13118
S'a'
I6999
sS's'
I17
sS'm'
(lp13119
I1
aI1
aI7
aI7
aI1
aI41
aI71
aI109
aI401
aI815
aI1311
aI3933
aI1349
aI13327
aI20847
aI44391
aI49721
asS'd'
I6560
sa(dp13120
S'a'
I7015
sS's'
I17
sS'm'
(lp13121
I1
aI1
aI1
aI15
aI27
aI57
aI39
aI129
aI391
aI701
aI619
aI3925
aI701
aI403
aI11821
aI30517
aI22035
asS'd'
I6561
sa(dp13122
S'a'
I7019
sS's'
I17
sS'm'
(lp13123
I1
aI1
aI5
aI11
aI21
aI49
aI109
aI101
aI497
aI417
aI73
aI2727
aI2899
aI2777
aI22161
aI35561
aI70211
asS'd'
I6562
sa(dp13124
S'a'
I7022
sS's'
I17
sS'm'
(lp13125
I1
aI1
aI3
aI3
aI15
aI43
aI1
aI159
aI41
aI833
aI55
aI2415
aI5009
aI9663
aI31295
aI29397
aI3187
asS'd'
I6563
sa(dp13126
S'a'
I7040
sS's'
I17
sS'm'
(lp13127
I1
aI1
aI3
aI7
aI27
aI5
aI113
aI187
aI453
aI753
aI1649
aI1605
aI2405
aI11791
aI4239
aI20915
aI54033
asS'd'
I6564
sa(dp13128
S'a'
I7045
sS's'
I17
sS'm'
(lp13129
I1
aI3
aI1
aI11
aI1
aI57
aI49
aI229
aI283
aI113
aI345
aI785
aI8009
aI11977
aI30169
aI63787
aI32011
asS'd'
I6565
sa(dp13130
S'a'
I7049
sS's'
I17
sS'm'
(lp13131
I1
aI1
aI7
aI3
aI5
aI59
aI57
aI91
aI327
aI685
aI219
aI1949
aI3095
aI8389
aI2035
aI11903
aI73461
asS'd'
I6566
sa(dp13132
S'a'
I7055
sS's'
I17
sS'm'
(lp13133
I1
aI1
aI3
aI3
aI19
aI59
aI19
aI37
aI453
aI1
aI1811
aI3263
aI1807
aI16147
aI24861
aI14003
aI31747
asS'd'
I6567
sa(dp13134
S'a'
I7073
sS's'
I17
sS'm'
(lp13135
I1
aI1
aI3
aI11
aI1
aI53
aI93
aI203
aI429
aI629
aI1931
aI1487
aI3301
aI8805
aI4901
aI2459
aI98555
asS'd'
I6568
sa(dp13136
S'a'
I7076
sS's'
I17
sS'm'
(lp13137
I1
aI1
aI7
aI5
aI21
aI5
aI37
aI135
aI159
aI749
aI1589
aI2631
aI8145
aI7279
aI28397
aI47113
aI82309
asS'd'
I6569
sa(dp13138
S'a'
I7085
sS's'
I17
sS'm'
(lp13139
I1
aI1
aI5
aI15
aI25
aI61
aI19
aI51
aI217
aI495
aI109
aI1179
aI2743
aI12107
aI12509
aI13003
aI94375
asS'd'
I6570
sa(dp13140
S'a'
I7091
sS's'
I17
sS'm'
(lp13141
I1
aI3
aI3
aI15
aI11
aI7
aI67
aI165
aI57
aI925
aI427
aI2549
aI7189
aI5917
aI13113
aI30933
aI62703
asS'd'
I6571
sa(dp13142
S'a'
I7103
sS's'
I17
sS'm'
(lp13143
I1
aI1
aI5
aI5
aI9
aI5
aI43
aI5
aI485
aI159
aI757
aI3979
aI4963
aI3389
aI29731
aI48477
aI113429
asS'd'
I6572
sa(dp13144
S'a'
I7112
sS's'
I17
sS'm'
(lp13145
I1
aI3
aI5
aI1
aI5
aI5
aI81
aI163
aI493
aI357
aI2005
aI1093
aI5951
aI1045
aI10569
aI40321
aI56881
asS'd'
I6573
sa(dp13146
S'a'
I7117
sS's'
I17
sS'm'
(lp13147
I1
aI3
aI1
aI5
aI7
aI29
aI11
aI7
aI7
aI13
aI1641
aI1031
aI4025
aI16337
aI24333
aI9589
aI37779
asS'd'
I6574
sa(dp13148
S'a'
I7118
sS's'
I17
sS'm'
(lp13149
I1
aI3
aI5
aI11
aI15
aI3
aI69
aI19
aI141
aI79
aI749
aI391
aI4505
aI6939
aI3079
aI3647
aI22363
asS'd'
I6575
sa(dp13150
S'a'
I7123
sS's'
I17
sS'm'
(lp13151
I1
aI3
aI3
aI3
aI29
aI3
aI7
aI189
aI183
aI513
aI1225
aI239
aI4203
aI9197
aI23507
aI33089
aI124433
asS'd'
I6576
sa(dp13152
S'a'
I7126
sS's'
I17
sS'm'
(lp13153
I1
aI3
aI3
aI13
aI27
aI37
aI81
aI221
aI287
aI891
aI1197
aI3501
aI539
aI2053
aI20509
aI48635
aI50269
asS'd'
I6577
sa(dp13154
S'a'
I7154
sS's'
I17
sS'm'
(lp13155
I1
aI1
aI5
aI7
aI13
aI3
aI35
aI79
aI3
aI885
aI343
aI3527
aI1043
aI7197
aI6973
aI8515
aI39315
asS'd'
I6578
sa(dp13156
S'a'
I7180
sS's'
I17
sS'm'
(lp13157
I1
aI3
aI3
aI9
aI21
aI53
aI79
aI225
aI229
aI759
aI457
aI293
aI953
aI12857
aI20483
aI3677
aI93839
asS'd'
I6579
sa(dp13158
S'a'
I7192
sS's'
I17
sS'm'
(lp13159
I1
aI3
aI5
aI3
aI5
aI17
aI45
aI107
aI153
aI279
aI761
aI1923
aI7013
aI2989
aI10137
aI19107
aI126897
asS'd'
I6580
sa(dp13160
S'a'
I7195
sS's'
I17
sS'm'
(lp13161
I1
aI3
aI1
aI3
aI23
aI53
aI91
aI1
aI133
aI729
aI13
aI2017
aI6933
aI7405
aI1255
aI49509
aI105571
asS'd'
I6581
sa(dp13162
S'a'
I7207
sS's'
I17
sS'm'
(lp13163
I1
aI3
aI5
aI1
aI9
aI45
aI35
aI153
aI209
aI289
aI1779
aI2557
aI315
aI981
aI15347
aI30391
aI16027
asS'd'
I6582
sa(dp13164
S'a'
I7208
sS's'
I17
sS'm'
(lp13165
I1
aI3
aI3
aI5
aI17
aI3
aI51
aI105
aI263
aI959
aI1255
aI1177
aI8143
aI10541
aI7687
aI38731
aI93561
asS'd'
I6583
sa(dp13166
S'a'
I7214
sS's'
I17
sS'm'
(lp13167
I1
aI1
aI1
aI13
aI19
aI1
aI15
aI135
aI447
aI847
aI663
aI3893
aI3539
aI6833
aI13265
aI62923
aI8375
asS'd'
I6584
sa(dp13168
S'a'
I7222
sS's'
I17
sS'm'
(lp13169
I1
aI3
aI1
aI15
aI31
aI11
aI105
aI1
aI91
aI523
aI1583
aI3493
aI2665
aI117
aI10757
aI29845
aI127201
asS'd'
I6585
sa(dp13170
S'a'
I7234
sS's'
I17
sS'm'
(lp13171
I1
aI1
aI1
aI3
aI29
aI49
aI9
aI103
aI309
aI605
aI1751
aI1981
aI833
aI3653
aI14001
aI16545
aI58513
asS'd'
I6586
sa(dp13172
S'a'
I7254
sS's'
I17
sS'm'
(lp13173
I1
aI1
aI5
aI9
aI1
aI19
aI117
aI71
aI237
aI765
aI249
aI1983
aI2289
aI6019
aI26505
aI31427
aI64333
asS'd'
I6587
sa(dp13174
S'a'
I7258
sS's'
I17
sS'm'
(lp13175
I1
aI1
aI3
aI11
aI15
aI31
aI5
aI207
aI347
aI143
aI11
aI1987
aI3569
aI2051
aI31051
aI22193
aI93289
asS'd'
I6588
sa(dp13176
S'a'
I7264
sS's'
I17
sS'm'
(lp13177
I1
aI1
aI3
aI5
aI13
aI15
aI5
aI73
aI457
aI611
aI673
aI2675
aI8071
aI13245
aI19443
aI14399
aI99599
asS'd'
I6589
sa(dp13178
S'a'
I7279
sS's'
I17
sS'm'
(lp13179
I1
aI1
aI1
aI9
aI11
aI5
aI103
aI231
aI31
aI457
aI1031
aI2257
aI3159
aI8323
aI31585
aI26163
aI45159
asS'd'
I6590
sa(dp13180
S'a'
I7282
sS's'
I17
sS'm'
(lp13181
I1
aI3
aI1
aI11
aI29
aI51
aI29
aI7
aI89
aI331
aI783
aI951
aI6353
aI15421
aI12801
aI8337
aI119171
asS'd'
I6591
sa(dp13182
S'a'
I7293
sS's'
I17
sS'm'
(lp13183
I1
aI1
aI3
aI13
aI23
aI57
aI63
aI43
aI505
aI1
aI657
aI4005
aI6327
aI7545
aI15455
aI27097
aI53649
asS'd'
I6592
sa(dp13184
S'a'
I7297
sS's'
I17
sS'm'
(lp13185
I1
aI1
aI1
aI5
aI31
aI7
aI51
aI107
aI175
aI461
aI1893
aI305
aI157
aI4819
aI18549
aI33087
aI9499
asS'd'
I6593
sa(dp13186
S'a'
I7322
sS's'
I17
sS'm'
(lp13187
I1
aI3
aI1
aI3
aI19
aI45
aI37
aI9
aI459
aI143
aI1327
aI3611
aI1899
aI15109
aI30151
aI17911
aI13233
asS'd'
I6594
sa(dp13188
S'a'
I7324
sS's'
I17
sS'm'
(lp13189
I1
aI1
aI5
aI15
aI19
aI49
aI11
aI227
aI375
aI661
aI665
aI259
aI3659
aI13723
aI28239
aI48159
aI59209
asS'd'
I6595
sa(dp13190
S'a'
I7351
sS's'
I17
sS'm'
(lp13191
I1
aI3
aI7
aI7
aI17
aI49
aI77
aI161
aI505
aI713
aI1521
aI935
aI3629
aI5033
aI26717
aI47199
aI3693
asS'd'
I6596
sa(dp13192
S'a'
I7363
sS's'
I17
sS'm'
(lp13193
I1
aI3
aI5
aI9
aI17
aI61
aI1
aI201
aI259
aI179
aI1637
aI2485
aI4995
aI2813
aI19923
aI43739
aI32183
asS'd'
I6597
sa(dp13194
S'a'
I7380
sS's'
I17
sS'm'
(lp13195
I1
aI1
aI3
aI5
aI1
aI23
aI125
aI61
aI225
aI703
aI2011
aI1013
aI6651
aI14029
aI27375
aI23301
aI80269
asS'd'
I6598
sa(dp13196
S'a'
I7384
sS's'
I17
sS'm'
(lp13197
I1
aI1
aI3
aI9
aI11
aI57
aI37
aI49
aI321
aI443
aI1055
aI1989
aI4755
aI8467
aI22001
aI18647
aI112617
asS'd'
I6599
sa(dp13198
S'a'
I7389
sS's'
I17
sS'm'
(lp13199
I1
aI3
aI1
aI5
aI5
aI39
aI21
aI139
aI101
aI583
aI1881
aI2599
aI4185
aI15679
aI22215
aI19093
aI76737
asS'd'
I6600
sa(dp13200
S'a'
I7396
sS's'
I17
sS'm'
(lp13201
I1
aI3
aI1
aI11
aI31
aI51
aI85
aI91
aI159
aI421
aI2005
aI1075
aI7757
aI12653
aI25489
aI3545
aI62961
asS'd'
I6601
sa(dp13202
S'a'
I7413
sS's'
I17
sS'm'
(lp13203
I1
aI1
aI1
aI15
aI27
aI57
aI75
aI151
aI357
aI571
aI395
aI299
aI5607
aI12865
aI2149
aI21059
aI120753
asS'd'
I6602
sa(dp13204
S'a'
I7417
sS's'
I17
sS'm'
(lp13205
I1
aI1
aI1
aI3
aI15
aI57
aI63
aI171
aI265
aI709
aI1089
aI677
aI7243
aI10207
aI9789
aI38431
aI130415
asS'd'
I6603
sa(dp13206
S'a'
I7431
sS's'
I17
sS'm'
(lp13207
I1
aI3
aI7
aI5
aI21
aI9
aI73
aI149
aI197
aI773
aI773
aI3931
aI4135
aI5671
aI2733
aI57173
aI90693
asS'd'
I6604
sa(dp13208
S'a'
I7443
sS's'
I17
sS'm'
(lp13209
I1
aI1
aI5
aI1
aI23
aI1
aI47
aI201
aI33
aI167
aI1643
aI4009
aI2687
aI5725
aI28733
aI27859
aI55163
asS'd'
I6605
sa(dp13210
S'a'
I7445
sS's'
I17
sS'm'
(lp13211
I1
aI1
aI5
aI15
aI25
aI11
aI57
aI139
aI471
aI625
aI1067
aI3647
aI6213
aI15605
aI23537
aI5005
aI32593
asS'd'
I6606
sa(dp13212
S'a'
I7450
sS's'
I17
sS'm'
(lp13213
I1
aI3
aI1
aI11
aI17
aI11
aI25
aI163
aI199
aI21
aI1775
aI3721
aI2845
aI15769
aI2381
aI27643
aI19909
asS'd'
I6607
sa(dp13214
S'a'
I7456
sS's'
I17
sS'm'
(lp13215
I1
aI3
aI5
aI5
aI21
aI41
aI23
aI125
aI401
aI483
aI535
aI925
aI7065
aI1727
aI3761
aI8485
aI3519
asS'd'
I6608
sa(dp13216
S'a'
I7466
sS's'
I17
sS'm'
(lp13217
I1
aI1
aI3
aI15
aI27
aI31
aI11
aI7
aI93
aI237
aI611
aI3635
aI4747
aI9751
aI20607
aI20473
aI73935
asS'd'
I6609
sa(dp13218
S'a'
I7468
sS's'
I17
sS'm'
(lp13219
I1
aI1
aI7
aI3
aI15
aI19
aI69
aI169
aI387
aI291
aI1981
aI635
aI3387
aI15817
aI20357
aI47537
aI107311
asS'd'
I6610
sa(dp13220
S'a'
I7474
sS's'
I17
sS'm'
(lp13221
I1
aI3
aI7
aI15
aI13
aI59
aI31
aI235
aI399
aI343
aI1265
aI2975
aI6839
aI13335
aI5397
aI58915
aI31313
asS'd'
I6611
sa(dp13222
S'a'
I7479
sS's'
I17
sS'm'
(lp13223
I1
aI1
aI7
aI1
aI3
aI35
aI81
aI243
aI387
aI421
aI1533
aI799
aI2615
aI13219
aI9041
aI26967
aI22677
asS'd'
I6612
sa(dp13224
S'a'
I7486
sS's'
I17
sS'm'
(lp13225
I1
aI1
aI7
aI15
aI17
aI41
aI89
aI115
aI67
aI569
aI1647
aI1831
aI5533
aI4629
aI1413
aI20037
aI97343
asS'd'
I6613
sa(dp13226
S'a'
I7497
sS's'
I17
sS'm'
(lp13227
I1
aI1
aI5
aI1
aI23
aI41
aI11
aI149
aI319
aI377
aI439
aI1237
aI4819
aI14303
aI14657
aI61711
aI129235
asS'd'
I6614
sa(dp13228
S'a'
I7508
sS's'
I17
sS'm'
(lp13229
I1
aI3
aI3
aI7
aI9
aI11
aI79
aI219
aI249
aI607
aI1453
aI2931
aI3407
aI13725
aI28289
aI42869
aI96759
asS'd'
I6615
sa(dp13230
S'a'
I7515
sS's'
I17
sS'm'
(lp13231
I1
aI1
aI5
aI11
aI7
aI9
aI101
aI51
aI11
aI893
aI697
aI1221
aI4237
aI1873
aI11191
aI25517
aI119861
asS'd'
I6616
sa(dp13232
S'a'
I7533
sS's'
I17
sS'm'
(lp13233
I1
aI1
aI3
aI11
aI23
aI23
aI19
aI245
aI485
aI317
aI1945
aI2339
aI193
aI9389
aI30709
aI33927
aI95089
asS'd'
I6617
sa(dp13234
S'a'
I7542
sS's'
I17
sS'm'
(lp13235
I1
aI1
aI3
aI1
aI27
aI55
aI5
aI81
aI63
aI185
aI223
aI3639
aI6093
aI10053
aI1793
aI11885
aI29307
asS'd'
I6618
sa(dp13236
S'a'
I7546
sS's'
I17
sS'm'
(lp13237
I1
aI1
aI7
aI13
aI15
aI41
aI33
aI133
aI467
aI457
aI213
aI3687
aI1313
aI2555
aI19487
aI44257
aI108667
asS'd'
I6619
sa(dp13238
S'a'
I7551
sS's'
I17
sS'm'
(lp13239
I1
aI1
aI3
aI5
aI31
aI51
aI53
aI115
aI449
aI273
aI1043
aI2743
aI1759
aI2013
aI28171
aI57091
aI76837
asS'd'
I6620
sa(dp13240
S'a'
I7569
sS's'
I17
sS'm'
(lp13241
I1
aI1
aI5
aI15
aI21
aI43
aI11
aI215
aI151
aI253
aI913
aI1889
aI2799
aI13787
aI3869
aI54413
aI50991
asS'd'
I6621
sa(dp13242
S'a'
I7572
sS's'
I17
sS'm'
(lp13243
I1
aI1
aI3
aI13
aI29
aI19
aI81
aI123
aI461
aI203
aI81
aI555
aI6601
aI15689
aI12637
aI41467
aI105343
asS'd'
I6622
sa(dp13244
S'a'
I7595
sS's'
I17
sS'm'
(lp13245
I1
aI1
aI3
aI13
aI7
aI21
aI75
aI111
aI47
aI481
aI1519
aI3299
aI6199
aI3501
aI31323
aI29215
aI45607
asS'd'
I6623
sa(dp13246
S'a'
I7603
sS's'
I17
sS'm'
(lp13247
I1
aI3
aI1
aI3
aI17
aI51
aI45
aI223
aI321
aI233
aI267
aI3333
aI3803
aI3099
aI4601
aI29061
aI88441
asS'd'
I6624
sa(dp13248
S'a'
I7605
sS's'
I17
sS'm'
(lp13249
I1
aI1
aI5
aI13
aI23
aI27
aI7
aI57
aI273
aI893
aI773
aI239
aI6357
aI15875
aI5497
aI21775
aI108291
asS'd'
I6625
sa(dp13250
S'a'
I7629
sS's'
I17
sS'm'
(lp13251
I1
aI3
aI1
aI15
aI25
aI17
aI11
aI229
aI175
aI909
aI691
aI3507
aI5247
aI2933
aI1741
aI35059
aI62841
asS'd'
I6626
sa(dp13252
S'a'
I7632
sS's'
I17
sS'm'
(lp13253
I1
aI3
aI5
aI1
aI29
aI7
aI11
aI69
aI345
aI87
aI99
aI3243
aI5669
aI11053
aI1185
aI6979
aI117069
asS'd'
I6627
sa(dp13254
S'a'
I7638
sS's'
I17
sS'm'
(lp13255
I1
aI3
aI5
aI11
aI13
aI33
aI23
aI183
aI89
aI475
aI643
aI2773
aI7899
aI15219
aI133
aI5073
aI129355
asS'd'
I6628
sa(dp13256
S'a'
I7648
sS's'
I17
sS'm'
(lp13257
I1
aI3
aI7
aI9
aI23
aI17
aI31
aI53
aI455
aI193
aI1695
aI2557
aI1645
aI12675
aI27857
aI50447
aI121335
asS'd'
I6629
sa(dp13258
S'a'
I7654
sS's'
I17
sS'm'
(lp13259
I1
aI1
aI3
aI11
aI15
aI19
aI41
aI57
aI305
aI235
aI1131
aI1165
aI1857
aI13667
aI19285
aI29755
aI118885
asS'd'
I6630
sa(dp13260
S'a'
I7663
sS's'
I17
sS'm'
(lp13261
I1
aI3
aI7
aI3
aI9
aI43
aI107
aI3
aI275
aI673
aI677
aI3769
aI3097
aI5497
aI24911
aI4617
aI80505
asS'd'
I6631
sa(dp13262
S'a'
I7675
sS's'
I17
sS'm'
(lp13263
I1
aI1
aI7
aI9
aI21
aI39
aI107
aI155
aI471
aI753
aI579
aI2929
aI4951
aI4245
aI25035
aI41795
aI86955
asS'd'
I6632
sa(dp13264
S'a'
I7693
sS's'
I17
sS'm'
(lp13265
I1
aI3
aI1
aI7
aI31
aI51
aI27
aI165
aI147
aI751
aI709
aI399
aI45
aI947
aI9893
aI32721
aI122127
asS'd'
I6633
sa(dp13266
S'a'
I7705
sS's'
I17
sS'm'
(lp13267
I1
aI3
aI3
aI1
aI31
aI31
aI73
aI59
aI351
aI293
aI845
aI3139
aI4177
aI3537
aI9465
aI20689
aI65837
asS'd'
I6634
sa(dp13268
S'a'
I7717
sS's'
I17
sS'm'
(lp13269
I1
aI3
aI5
aI9
aI27
aI29
aI13
aI115
aI417
aI435
aI465
aI1291
aI5225
aI11687
aI29207
aI39895
aI55443
asS'd'
I6635
sa(dp13270
S'a'
I7724
sS's'
I17
sS'm'
(lp13271
I1
aI3
aI3
aI15
aI29
aI49
aI111
aI179
aI221
aI565
aI787
aI1811
aI4055
aI7863
aI27273
aI32975
aI26985
asS'd'
I6636
sa(dp13272
S'a'
I7727
sS's'
I17
sS'm'
(lp13273
I1
aI1
aI1
aI7
aI15
aI49
aI121
aI145
aI277
aI27
aI149
aI965
aI4903
aI3497
aI32333
aI37217
aI105073
asS'd'
I6637
sa(dp13274
S'a'
I7735
sS's'
I17
sS'm'
(lp13275
I1
aI1
aI7
aI1
aI23
aI29
aI31
aI77
aI353
aI349
aI755
aI2081
aI4291
aI567
aI641
aI41751
aI20397
asS'd'
I6638
sa(dp13276
S'a'
I7761
sS's'
I17
sS'm'
(lp13277
I1
aI1
aI5
aI3
aI25
aI31
aI97
aI3
aI405
aI607
aI965
aI2981
aI3217
aI14695
aI25977
aI22457
aI113539
asS'd'
I6639
sa(dp13278
S'a'
I7767
sS's'
I17
sS'm'
(lp13279
I1
aI3
aI3
aI15
aI25
aI3
aI91
aI125
aI269
aI825
aI1163
aI2181
aI4247
aI6813
aI4699
aI35091
aI87771
asS'd'
I6640
sa(dp13280
S'a'
I7783
sS's'
I17
sS'm'
(lp13281
I1
aI1
aI5
aI9
aI25
aI23
aI113
aI145
aI71
aI31
aI1115
aI3729
aI6793
aI11869
aI26509
aI18779
aI99499
asS'd'
I6641
sa(dp13282
S'a'
I7784
sS's'
I17
sS'm'
(lp13283
I1
aI1
aI1
aI9
aI31
aI51
aI77
aI217
aI247
aI599
aI1541
aI3217
aI1383
aI5203
aI27971
aI23647
aI71823
asS'd'
I6642
sa(dp13284
S'a'
I7798
sS's'
I17
sS'm'
(lp13285
I1
aI1
aI5
aI7
aI17
aI35
aI113
aI73
aI475
aI511
aI35
aI1961
aI5311
aI2257
aI1935
aI58963
aI94349
asS'd'
I6643
sa(dp13286
S'a'
I7802
sS's'
I17
sS'm'
(lp13287
I1
aI3
aI1
aI7
aI27
aI31
aI67
aI253
aI95
aI883
aI1213
aI855
aI3429
aI15049
aI26715
aI56099
aI101797
asS'd'
I6644
sa(dp13288
S'a'
I7811
sS's'
I17
sS'm'
(lp13289
I1
aI1
aI3
aI5
aI9
aI9
aI61
aI57
aI511
aI537
aI1803
aI949
aI1327
aI3921
aI11297
aI13807
aI64629
asS'd'
I6645
sa(dp13290
S'a'
I7817
sS's'
I17
sS'm'
(lp13291
I1
aI1
aI5
aI1
aI31
aI57
aI105
aI161
aI309
aI283
aI1291
aI2737
aI7141
aI7497
aI25893
aI14453
aI35375
asS'd'
I6646
sa(dp13292
S'a'
I7823
sS's'
I17
sS'm'
(lp13293
I1
aI1
aI3
aI1
aI21
aI3
aI77
aI37
aI13
aI211
aI1863
aI1895
aI8035
aI5801
aI25981
aI12317
aI48375
asS'd'
I6647
sa(dp13294
S'a'
I7832
sS's'
I17
sS'm'
(lp13295
I1
aI3
aI7
aI7
aI25
aI45
aI13
aI77
aI185
aI553
aI1501
aI1349
aI5951
aI15581
aI32657
aI18467
aI128363
asS'd'
I6648
sa(dp13296
S'a'
I7837
sS's'
I17
sS'm'
(lp13297
I1
aI3
aI5
aI9
aI23
aI63
aI105
aI239
aI213
aI935
aI1331
aI3653
aI2775
aI6591
aI6067
aI34597
aI19217
asS'd'
I6649
sa(dp13298
S'a'
I7842
sS's'
I17
sS'm'
(lp13299
I1
aI3
aI7
aI13
aI15
aI19
aI79
aI91
aI391
aI637
aI1685
aI2263
aI3507
aI2025
aI2111
aI15875
aI14831
asS'd'
I6650
sa(dp13300
S'a'
I7853
sS's'
I17
sS'm'
(lp13301
I1
aI3
aI3
aI5
aI7
aI29
aI81
aI69
aI511
aI399
aI343
aI737
aI2833
aI1021
aI10471
aI18689
aI36181
asS'd'
I6651
sa(dp13302
S'a'
I7854
sS's'
I17
sS'm'
(lp13303
I1
aI1
aI5
aI11
aI21
aI17
aI39
aI137
aI145
aI857
aI583
aI789
aI8057
aI15995
aI32113
aI64163
aI37153
asS'd'
I6652
sa(dp13304
S'a'
I7856
sS's'
I17
sS'm'
(lp13305
I1
aI3
aI3
aI11
aI9
aI61
aI87
aI131
aI487
aI667
aI1295
aI493
aI4629
aI7719
aI18157
aI49715
aI2051
asS'd'
I6653
sa(dp13306
S'a'
I7861
sS's'
I17
sS'm'
(lp13307
I1
aI3
aI5
aI9
aI19
aI5
aI85
aI3
aI491
aI353
aI571
aI2829
aI4411
aI343
aI24781
aI62325
aI123959
asS'd'
I6654
sa(dp13308
S'a'
I7862
sS's'
I17
sS'm'
(lp13309
I1
aI1
aI7
aI13
aI13
aI39
aI11
aI31
aI413
aI285
aI27
aI2433
aI3307
aI6165
aI26565
aI40065
aI102655
asS'd'
I6655
sa(dp13310
S'a'
I7873
sS's'
I17
sS'm'
(lp13311
I1
aI1
aI5
aI11
aI25
aI45
aI7
aI97
aI9
aI973
aI1833
aI2537
aI1457
aI7389
aI24087
aI38061
aI122805
asS'd'
I6656
sa(dp13312
S'a'
I7874
sS's'
I17
sS'm'
(lp13313
I1
aI3
aI5
aI3
aI21
aI63
aI77
aI21
aI249
aI525
aI1145
aI1421
aI8011
aI3357
aI15051
aI30293
aI127017
asS'd'
I6657
sa(dp13314
S'a'
I7886
sS's'
I17
sS'm'
(lp13315
I1
aI1
aI5
aI3
aI13
aI53
aI81
aI185
aI303
aI307
aI1579
aI841
aI2277
aI607
aI10899
aI34209
aI215
asS'd'
I6658
sa(dp13316
S'a'
I7914
sS's'
I17
sS'm'
(lp13317
I1
aI3
aI3
aI13
aI17
aI1
aI125
aI145
aI205
aI763
aI127
aI1865
aI4129
aI849
aI27247
aI29845
aI36515
asS'd'
I6659
sa(dp13318
S'a'
I7927
sS's'
I17
sS'm'
(lp13319
I1
aI3
aI7
aI13
aI5
aI59
aI19
aI71
aI227
aI111
aI365
aI1309
aI5857
aI6035
aI32379
aI11303
aI127329
asS'd'
I6660
sa(dp13320
S'a'
I7936
sS's'
I17
sS'm'
(lp13321
I1
aI1
aI1
aI1
aI19
aI61
aI79
aI253
aI459
aI23
aI725
aI3615
aI4583
aI429
aI13215
aI31879
aI4523
asS'd'
I6661
sa(dp13322
S'a'
I7951
sS's'
I17
sS'm'
(lp13323
I1
aI1
aI1
aI7
aI19
aI13
aI53
aI41
aI243
aI107
aI1767
aI983
aI3483
aI2249
aI2209
aI58895
aI14805
asS'd'
I6662
sa(dp13324
S'a'
I7963
sS's'
I17
sS'm'
(lp13325
I1
aI1
aI1
aI9
aI5
aI63
aI31
aI85
aI119
aI307
aI633
aI3295
aI841
aI3495
aI22965
aI57587
aI108271
asS'd'
I6663
sa(dp13326
S'a'
I7966
sS's'
I17
sS'm'
(lp13327
I1
aI3
aI5
aI9
aI17
aI13
aI57
aI49
aI97
aI613
aI405
aI2637
aI3229
aI14253
aI4663
aI61345
aI33415
asS'd'
I6664
sa(dp13328
S'a'
I7976
sS's'
I17
sS'm'
(lp13329
I1
aI3
aI1
aI1
aI17
aI37
aI63
aI3
aI5
aI375
aI1073
aI3971
aI665
aI4445
aI153
aI20437
aI38513
asS'd'
I6665
sa(dp13330
S'a'
I7993
sS's'
I17
sS'm'
(lp13331
I1
aI3
aI3
aI15
aI5
aI9
aI77
aI161
aI409
aI461
aI443
aI567
aI5581
aI8623
aI27735
aI9041
aI5517
asS'd'
I6666
sa(dp13332
S'a'
I8001
sS's'
I17
sS'm'
(lp13333
I1
aI3
aI5
aI13
aI13
aI5
aI19
aI53
aI263
aI155
aI557
aI3973
aI6841
aI4829
aI30751
aI30025
aI121973
asS'd'
I6667
sa(dp13334
S'a'
I8004
sS's'
I17
sS'm'
(lp13335
I1
aI3
aI7
aI9
aI27
aI37
aI49
aI243
aI107
aI1013
aI1743
aI1509
aI4465
aI15415
aI4741
aI41409
aI72695
asS'd'
I6668
sa(dp13336
S'a'
I8013
sS's'
I17
sS'm'
(lp13337
I1
aI1
aI3
aI5
aI11
aI49
aI39
aI45
aI21
aI463
aI875
aI3681
aI1901
aI15325
aI24553
aI51369
aI82227
asS'd'
I6669
sa(dp13338
S'a'
I8021
sS's'
I17
sS'm'
(lp13339
I1
aI1
aI3
aI15
aI11
aI35
aI21
aI91
aI383
aI149
aI1815
aI911
aI4633
aI12027
aI12413
aI22307
aI120049
asS'd'
I6670
sa(dp13340
S'a'
I8026
sS's'
I17
sS'm'
(lp13341
I1
aI3
aI5
aI7
aI7
aI3
aI15
aI83
aI477
aI687
aI145
aI1705
aI6893
aI5233
aI20171
aI43337
aI72603
asS'd'
I6671
sa(dp13342
S'a'
I8028
sS's'
I17
sS'm'
(lp13343
I1
aI1
aI3
aI9
aI25
aI35
aI19
aI173
aI67
aI5
aI561
aI2139
aI4557
aI4911
aI26273
aI38409
aI22801
asS'd'
I6672
sa(dp13344
S'a'
I8031
sS's'
I17
sS'm'
(lp13345
I1
aI1
aI3
aI13
aI15
aI39
aI85
aI91
aI91
aI187
aI1851
aI1181
aI4049
aI16353
aI26525
aI43703
aI19415
asS'd'
I6673
sa(dp13346
S'a'
I8035
sS's'
I17
sS'm'
(lp13347
I1
aI3
aI1
aI9
aI13
aI41
aI77
aI179
aI415
aI705
aI693
aI3017
aI5847
aI16191
aI11435
aI28979
aI51839
asS'd'
I6674
sa(dp13348
S'a'
I8042
sS's'
I17
sS'm'
(lp13349
I1
aI1
aI3
aI5
aI23
aI15
aI3
aI159
aI269
aI67
aI625
aI4043
aI4701
aI1599
aI6467
aI10949
aI80073
asS'd'
I6675
sa(dp13350
S'a'
I8071
sS's'
I17
sS'm'
(lp13351
I1
aI3
aI3
aI15
aI7
aI43
aI81
aI157
aI393
aI321
aI1875
aI2801
aI4359
aI11703
aI1063
aI64015
aI109997
asS'd'
I6676
sa(dp13352
S'a'
I8085
sS's'
I17
sS'm'
(lp13353
I1
aI1
aI7
aI3
aI25
aI21
aI37
aI123
aI133
aI691
aI973
aI3115
aI2291
aI10519
aI13339
aI22751
aI85445
asS'd'
I6677
sa(dp13354
S'a'
I8092
sS's'
I17
sS'm'
(lp13355
I1
aI3
aI1
aI1
aI21
aI21
aI9
aI23
aI431
aI679
aI1873
aI289
aI4503
aI3939
aI14417
aI36081
aI18709
asS'd'
I6678
sa(dp13356
S'a'
I8102
sS's'
I17
sS'm'
(lp13357
I1
aI3
aI5
aI5
aI1
aI53
aI109
aI133
aI33
aI279
aI727
aI2233
aI3065
aI8557
aI7487
aI25797
aI109177
asS'd'
I6679
sa(dp13358
S'a'
I8105
sS's'
I17
sS'm'
(lp13359
I1
aI1
aI7
aI7
aI1
aI9
aI47
aI127
aI179
aI757
aI1985
aI547
aI169
aI13393
aI22669
aI58795
aI92897
asS'd'
I6680
sa(dp13360
S'a'
I8114
sS's'
I17
sS'm'
(lp13361
I1
aI3
aI5
aI11
aI17
aI21
aI95
aI219
aI263
aI579
aI1493
aI3283
aI5461
aI1235
aI1749
aI33325
aI36033
asS'd'
I6681
sa(dp13362
S'a'
I8123
sS's'
I17
sS'm'
(lp13363
I1
aI1
aI3
aI11
aI21
aI49
aI45
aI143
aI511
aI983
aI1933
aI965
aI7905
aI1925
aI27857
aI40723
aI68251
asS'd'
I6682
sa(dp13364
S'a'
I8131
sS's'
I17
sS'm'
(lp13365
I1
aI3
aI7
aI3
aI27
aI9
aI73
aI7
aI441
aI877
aI107
aI1599
aI4795
aI7251
aI6819
aI7671
aI21137
asS'd'
I6683
sa(dp13366
S'a'
I8140
sS's'
I17
sS'm'
(lp13367
I1
aI1
aI3
aI3
aI21
aI25
aI49
aI43
aI247
aI949
aI627
aI2859
aI2507
aI4787
aI11269
aI53221
aI126387
asS'd'
I6684
sa(dp13368
S'a'
I8145
sS's'
I17
sS'm'
(lp13369
I1
aI1
aI5
aI3
aI5
aI53
aI127
aI65
aI353
aI521
aI1701
aI2981
aI3201
aI611
aI13475
aI58015
aI2605
asS'd'
I6685
sa(dp13370
S'a'
I8157
sS's'
I17
sS'm'
(lp13371
I1
aI1
aI5
aI13
aI9
aI39
aI55
aI103
aI53
aI281
aI705
aI2433
aI6179
aI3381
aI31973
aI30267
aI91307
asS'd'
I6686
sa(dp13372
S'a'
I8158
sS's'
I17
sS'm'
(lp13373
I1
aI1
aI7
aI13
aI31
aI23
aI29
aI161
aI347
aI449
aI123
aI3427
aI5731
aI12691
aI15175
aI20487
aI74695
asS'd'
I6687
sa(dp13374
S'a'
I8185
sS's'
I17
sS'm'
(lp13375
I1
aI3
aI3
aI15
aI13
aI19
aI83
aI137
aI437
aI317
aI921
aI913
aI7979
aI6665
aI5313
aI1435
aI60271
asS'd'
I6688
sa(dp13376
S'a'
I8186
sS's'
I17
sS'm'
(lp13377
I1
aI3
aI5
aI7
aI19
aI23
aI31
aI131
aI421
aI95
aI1999
aI897
aI4839
aI1815
aI12387
aI45009
aI2609
asS'd'
I6689
sa(dp13378
S'a'
I8188
sS's'
I17
sS'm'
(lp13379
I1
aI1
aI1
aI7
aI3
aI53
aI121
aI33
aI47
aI283
aI813
aI3841
aI4449
aI2543
aI15211
aI59285
aI42551
asS'd'
I6690
sa(dp13380
S'a'
I8192
sS's'
I17
sS'm'
(lp13381
I1
aI3
aI1
aI13
aI9
aI43
aI37
aI167
aI93
aI417
aI213
aI2721
aI3395
aI2089
aI13743
aI32925
aI91147
asS'd'
I6691
sa(dp13382
S'a'
I8212
sS's'
I17
sS'm'
(lp13383
I1
aI3
aI7
aI5
aI31
aI25
aI97
aI25
aI19
aI11
aI543
aI1889
aI455
aI5765
aI9517
aI56963
aI131069
asS'd'
I6692
sa(dp13384
S'a'
I8219
sS's'
I17
sS'm'
(lp13385
I1
aI3
aI1
aI7
aI3
aI7
aI87
aI61
aI209
aI39
aI1303
aI1637
aI6687
aI8001
aI5003
aI47911
aI110657
asS'd'
I6693
sa(dp13386
S'a'
I8221
sS's'
I17
sS'm'
(lp13387
I1
aI1
aI5
aI3
aI11
aI25
aI99
aI77
aI379
aI843
aI1423
aI2933
aI7535
aI4181
aI32223
aI49327
aI48041
asS'd'
I6694
sa(dp13388
S'a'
I8235
sS's'
I17
sS'm'
(lp13389
I1
aI3
aI3
aI13
aI9
aI7
aI85
aI59
aI47
aI777
aI401
aI2449
aI2795
aI11289
aI25023
aI7725
aI73887
asS'd'
I6695
sa(dp13390
S'a'
I8237
sS's'
I17
sS'm'
(lp13391
I1
aI1
aI3
aI5
aI11
aI51
aI93
aI57
aI369
aI871
aI1175
aI2705
aI1253
aI5169
aI24691
aI14243
aI119667
asS'd'
I6696
sa(dp13392
S'a'
I8249
sS's'
I17
sS'm'
(lp13393
I1
aI3
aI1
aI3
aI5
aI7
aI33
aI171
aI359
aI115
aI1909
aI2003
aI1413
aI13829
aI3471
aI36185
aI118399
asS'd'
I6697
sa(dp13394
S'a'
I8260
sS's'
I17
sS'm'
(lp13395
I1
aI1
aI1
aI11
aI5
aI49
aI97
aI145
aI415
aI731
aI671
aI2309
aI7211
aI11359
aI22757
aI15415
aI70951
asS'd'
I6698
sa(dp13396
S'a'
I8264
sS's'
I17
sS'm'
(lp13397
I1
aI1
aI3
aI5
aI7
aI51
aI61
aI101
aI375
aI575
aI1321
aI2835
aI7569
aI9599
aI4707
aI7655
aI53417
asS'd'
I6699
sa(dp13398
S'a'
I8270
sS's'
I17
sS'm'
(lp13399
I1
aI3
aI1
aI15
aI9
aI63
aI25
aI117
aI203
aI5
aI1345
aI2571
aI5273
aI2059
aI4689
aI27237
aI23199
asS'd'
I6700
sa(dp13400
S'a'
I8282
sS's'
I17
sS'm'
(lp13401
I1
aI1
aI3
aI15
aI15
aI23
aI69
aI49
aI349
aI995
aI5
aI1565
aI903
aI10165
aI9565
aI6343
aI16653
asS'd'
I6701
sa(dp13402
S'a'
I8291
sS's'
I17
sS'm'
(lp13403
I1
aI1
aI3
aI9
aI21
aI15
aI69
aI9
aI463
aI69
aI1447
aI2347
aI5125
aI7479
aI18257
aI14405
aI51321
asS'd'
I6702
sa(dp13404
S'a'
I8293
sS's'
I17
sS'm'
(lp13405
I1
aI1
aI7
aI11
aI23
aI57
aI57
aI179
aI17
aI129
aI999
aI777
aI6281
aI1693
aI31885
aI31085
aI29237
asS'd'
I6703
sa(dp13406
S'a'
I8297
sS's'
I17
sS'm'
(lp13407
I1
aI3
aI5
aI1
aI25
aI55
aI15
aI21
aI199
aI271
aI1645
aI1719
aI2023
aI10049
aI15215
aI11959
aI44875
asS'd'
I6704
sa(dp13408
S'a'
I8298
sS's'
I17
sS'm'
(lp13409
I1
aI3
aI1
aI3
aI29
aI43
aI83
aI11
aI281
aI27
aI429
aI685
aI7189
aI9151
aI8665
aI9553
aI115293
asS'd'
I6705
sa(dp13410
S'a'
I8305
sS's'
I17
sS'm'
(lp13411
I1
aI3
aI1
aI7
aI17
aI43
aI125
aI11
aI189
aI803
aI713
aI683
aI7285
aI4455
aI18195
aI45333
aI32281
asS'd'
I6706
sa(dp13412
S'a'
I8306
sS's'
I17
sS'm'
(lp13413
I1
aI3
aI3
aI3
aI11
aI55
aI21
aI59
aI173
aI283
aI709
aI1561
aI5391
aI5097
aI24725
aI19217
aI13769
asS'd'
I6707
sa(dp13414
S'a'
I8311
sS's'
I17
sS'm'
(lp13415
I1
aI3
aI5
aI13
aI7
aI29
aI117
aI207
aI415
aI525
aI567
aI1741
aI3553
aI6729
aI433
aI17619
aI45971
asS'd'
I6708
sa(dp13416
S'a'
I8318
sS's'
I17
sS'm'
(lp13417
I1
aI1
aI7
aI7
aI3
aI23
aI43
aI43
aI213
aI823
aI609
aI1037
aI3797
aI4733
aI30717
aI61067
aI89581
asS'd'
I6709
sa(dp13418
S'a'
I8327
sS's'
I17
sS'm'
(lp13419
I1
aI3
aI5
aI7
aI11
aI7
aI7
aI241
aI379
aI217
aI739
aI2815
aI2549
aI14297
aI10283
aI1509
aI80613
asS'd'
I6710
sa(dp13420
S'a'
I8345
sS's'
I17
sS'm'
(lp13421
I1
aI1
aI1
aI1
aI17
aI45
aI53
aI229
aI193
aI893
aI1881
aI227
aI6751
aI7135
aI20823
aI36939
aI27667
asS'd'
I6711
sa(dp13422
S'a'
I8379
sS's'
I17
sS'm'
(lp13423
I1
aI3
aI3
aI1
aI15
aI39
aI27
aI217
aI101
aI949
aI1963
aI2213
aI2357
aI4129
aI11925
aI841
aI59259
asS'd'
I6712
sa(dp13424
S'a'
I8390
sS's'
I17
sS'm'
(lp13425
I1
aI1
aI3
aI3
aI5
aI53
aI59
aI255
aI421
aI1009
aI683
aI2171
aI6691
aI12489
aI20865
aI29363
aI70611
asS'd'
I6713
sa(dp13426
S'a'
I8394
sS's'
I17
sS'm'
(lp13427
I1
aI1
aI7
aI15
aI7
aI31
aI105
aI141
aI153
aI401
aI549
aI3045
aI5443
aI11147
aI18159
aI24283
aI21859
asS'd'
I6714
sa(dp13428
S'a'
I8414
sS's'
I17
sS'm'
(lp13429
I1
aI3
aI7
aI1
aI11
aI17
aI17
aI231
aI175
aI603
aI1915
aI111
aI3203
aI10627
aI9687
aI47235
aI87057
asS'd'
I6715
sa(dp13430
S'a'
I8417
sS's'
I17
sS'm'
(lp13431
I1
aI1
aI1
aI11
aI19
aI21
aI115
aI41
aI45
aI727
aI1523
aI739
aI3025
aI10321
aI27353
aI63139
aI16051
asS'd'
I6716
sa(dp13432
S'a'
I8432
sS's'
I17
sS'm'
(lp13433
I1
aI3
aI7
aI11
aI13
aI9
aI33
aI121
aI237
aI565
aI2043
aI2131
aI3079
aI12575
aI2187
aI14427
aI85939
asS'd'
I6717
sa(dp13434
S'a'
I8437
sS's'
I17
sS'm'
(lp13435
I1
aI3
aI1
aI15
aI21
aI19
aI91
aI227
aI485
aI49
aI101
aI15
aI1903
aI4039
aI23819
aI40001
aI66405
asS'd'
I6718
sa(dp13436
S'a'
I8441
sS's'
I17
sS'm'
(lp13437
I1
aI3
aI1
aI5
aI15
aI25
aI65
aI25
aI393
aI287
aI1435
aI1851
aI6437
aI5983
aI13769
aI37847
aI120907
asS'd'
I6719
sa(dp13438
S'a'
I8449
sS's'
I17
sS'm'
(lp13439
I1
aI3
aI7
aI15
aI15
aI21
aI97
aI37
aI359
aI155
aI807
aI1421
aI517
aI13135
aI2955
aI56979
aI52299
asS'd'
I6720
sa(dp13440
S'a'
I8456
sS's'
I17
sS'm'
(lp13441
I1
aI1
aI5
aI1
aI27
aI53
aI79
aI27
aI467
aI605
aI267
aI1193
aI31
aI6177
aI12369
aI32621
aI38319
asS'd'
I6721
sa(dp13442
S'a'
I8473
sS's'
I17
sS'm'
(lp13443
I1
aI1
aI1
aI11
aI27
aI15
aI15
aI231
aI205
aI677
aI331
aI133
aI3313
aI7193
aI8059
aI36449
aI21671
asS'd'
I6722
sa(dp13444
S'a'
I8489
sS's'
I17
sS'm'
(lp13445
I1
aI3
aI3
aI11
aI19
aI57
aI113
aI83
aI399
aI801
aI1843
aI2119
aI2779
aI14061
aI30901
aI28745
aI120903
asS'd'
I6723
sa(dp13446
S'a'
I8495
sS's'
I17
sS'm'
(lp13447
I1
aI1
aI1
aI11
aI5
aI27
aI121
aI247
aI467
aI251
aI1487
aI251
aI897
aI3171
aI28383
aI22473
aI1709
asS'd'
I6724
sa(dp13448
S'a'
I8522
sS's'
I17
sS'm'
(lp13449
I1
aI1
aI1
aI15
aI7
aI59
aI123
aI165
aI123
aI373
aI167
aI1323
aI5239
aI9027
aI13791
aI55593
aI78785
asS'd'
I6725
sa(dp13450
S'a'
I8524
sS's'
I17
sS'm'
(lp13451
I1
aI3
aI1
aI11
aI31
aI11
aI81
aI229
aI123
aI183
aI461
aI1751
aI5713
aI2615
aI27795
aI1657
aI39253
asS'd'
I6726
sa(dp13452
S'a'
I8529
sS's'
I17
sS'm'
(lp13453
I1
aI1
aI7
aI1
aI21
aI45
aI107
aI3
aI283
aI149
aI549
aI3731
aI6435
aI3595
aI32753
aI16079
aI84257
asS'd'
I6727
sa(dp13454
S'a'
I8545
sS's'
I17
sS'm'
(lp13455
I1
aI3
aI3
aI15
aI19
aI9
aI81
aI37
aI51
aI341
aI909
aI985
aI1503
aI12787
aI16129
aI37789
aI113515
asS'd'
I6728
sa(dp13456
S'a'
I8557
sS's'
I17
sS'm'
(lp13457
I1
aI3
aI5
aI13
aI3
aI33
aI127
aI219
aI369
aI341
aI1191
aI1305
aI567
aI2339
aI31221
aI49435
aI114927
asS'd'
I6729
sa(dp13458
S'a'
I8565
sS's'
I17
sS'm'
(lp13459
I1
aI1
aI7
aI15
aI29
aI47
aI103
aI107
aI257
aI15
aI2029
aI2133
aI2129
aI11235
aI29553
aI49139
aI33809
asS'd'
I6730
sa(dp13460
S'a'
I8572
sS's'
I17
sS'm'
(lp13461
I1
aI3
aI3
aI13
aI23
aI33
aI105
aI43
aI155
aI815
aI1087
aI2261
aI2781
aI3461
aI7371
aI4479
aI123093
asS'd'
I6731
sa(dp13462
S'a'
I8576
sS's'
I17
sS'm'
(lp13463
I1
aI1
aI1
aI13
aI17
aI7
aI89
aI107
aI143
aI349
aI637
aI3651
aI4153
aI12131
aI28393
aI45781
aI84133
asS'd'
I6732
sa(dp13464
S'a'
I8582
sS's'
I17
sS'm'
(lp13465
I1
aI3
aI5
aI11
aI31
aI47
aI105
aI101
aI267
aI403
aI1853
aI3977
aI3277
aI1737
aI15503
aI47365
aI14361
asS'd'
I6733
sa(dp13466
S'a'
I8594
sS's'
I17
sS'm'
(lp13467
I1
aI1
aI1
aI13
aI1
aI63
aI125
aI107
aI123
aI183
aI1027
aI3491
aI3597
aI15949
aI5779
aI34665
aI81257
asS'd'
I6734
sa(dp13468
S'a'
I8629
sS's'
I17
sS'm'
(lp13469
I1
aI3
aI1
aI9
aI13
aI5
aI125
aI41
aI389
aI73
aI1487
aI1983
aI957
aI12645
aI13983
aI7675
aI72711
asS'd'
I6735
sa(dp13470
S'a'
I8636
sS's'
I17
sS'm'
(lp13471
I1
aI3
aI7
aI5
aI17
aI5
aI25
aI63
aI211
aI591
aI261
aI2345
aI3883
aI4403
aI773
aI43963
aI93509
asS'd'
I6736
sa(dp13472
S'a'
I8668
sS's'
I17
sS'm'
(lp13473
I1
aI3
aI3
aI1
aI11
aI35
aI15
aI251
aI225
aI643
aI537
aI3769
aI7593
aI6113
aI1377
aI52185
aI81459
asS'd'
I6737
sa(dp13474
S'a'
I8678
sS's'
I17
sS'm'
(lp13475
I1
aI3
aI5
aI15
aI27
aI27
aI51
aI35
aI389
aI853
aI1437
aI2803
aI5739
aI1887
aI15099
aI3299
aI111827
asS'd'
I6738
sa(dp13476
S'a'
I8701
sS's'
I17
sS'm'
(lp13477
I1
aI1
aI3
aI15
aI25
aI63
aI31
aI201
aI79
aI131
aI31
aI3929
aI4195
aI13045
aI8681
aI48121
aI110723
asS'd'
I6739
sa(dp13478
S'a'
I8702
sS's'
I17
sS'm'
(lp13479
I1
aI1
aI5
aI7
aI11
aI43
aI101
aI57
aI69
aI271
aI189
aI3087
aI4893
aI11365
aI6945
aI14285
aI41961
asS'd'
I6740
sa(dp13480
S'a'
I8708
sS's'
I17
sS'm'
(lp13481
I1
aI1
aI7
aI9
aI21
aI61
aI41
aI123
aI25
aI947
aI1619
aI2895
aI7879
aI12397
aI17405
aI48139
aI71519
asS'd'
I6741
sa(dp13482
S'a'
I8712
sS's'
I17
sS'm'
(lp13483
I1
aI3
aI1
aI15
aI1
aI27
aI113
aI225
aI441
aI855
aI541
aI357
aI3111
aI4867
aI20571
aI30627
aI70123
asS'd'
I6742
sa(dp13484
S'a'
I8745
sS's'
I17
sS'm'
(lp13485
I1
aI3
aI5
aI3
aI5
aI33
aI103
aI1
aI21
aI93
aI383
aI407
aI5145
aI7857
aI20289
aI51943
aI16223
asS'd'
I6743
sa(dp13486
S'a'
I8754
sS's'
I17
sS'm'
(lp13487
I1
aI1
aI7
aI15
aI1
aI13
aI41
aI215
aI463
aI417
aI513
aI3417
aI1755
aI16165
aI7271
aI3101
aI54353
asS'd'
I6744
sa(dp13488
S'a'
I8759
sS's'
I17
sS'm'
(lp13489
I1
aI3
aI3
aI13
aI19
aI29
aI5
aI205
aI245
aI927
aI1249
aI773
aI3653
aI9959
aI357
aI40863
aI37289
asS'd'
I6745
sa(dp13490
S'a'
I8763
sS's'
I17
sS'm'
(lp13491
I1
aI3
aI3
aI7
aI3
aI5
aI85
aI241
aI29
aI627
aI1963
aI3133
aI1369
aI503
aI11449
aI4699
aI2573
asS'd'
I6746
sa(dp13492
S'a'
I8766
sS's'
I17
sS'm'
(lp13493
I1
aI1
aI7
aI15
aI3
aI35
aI47
aI157
aI413
aI437
aI1627
aI3953
aI947
aI12721
aI22209
aI34303
aI81237
asS'd'
I6747
sa(dp13494
S'a'
I8780
sS's'
I17
sS'm'
(lp13495
I1
aI1
aI5
aI5
aI1
aI45
aI47
aI245
aI253
aI349
aI1853
aI3481
aI6105
aI7267
aI3159
aI38833
aI117889
asS'd'
I6748
sa(dp13496
S'a'
I8783
sS's'
I17
sS'm'
(lp13497
I1
aI3
aI7
aI15
aI23
aI43
aI25
aI181
aI121
aI681
aI479
aI1239
aI6155
aI3317
aI9419
aI28717
aI44643
asS'd'
I6749
sa(dp13498
S'a'
I8786
sS's'
I17
sS'm'
(lp13499
I1
aI3
aI3
aI15
aI31
aI43
aI111
aI99
aI405
aI991
aI301
aI1689
aI7107
aI16131
aI16703
aI24059
aI40345
asS'd'
I6750
sa(dp13500
S'a'
I8798
sS's'
I17
sS'm'
(lp13501
I1
aI1
aI3
aI9
aI25
aI5
aI107
aI91
aI117
aI351
aI1595
aI163
aI3007
aI13743
aI24535
aI38671
aI29745
asS'd'
I6751
sa(dp13502
S'a'
I8804
sS's'
I17
sS'm'
(lp13503
I1
aI3
aI3
aI5
aI27
aI47
aI15
aI195
aI119
aI919
aI665
aI1903
aI1981
aI7753
aI21709
aI33699
aI15963
asS'd'
I6752
sa(dp13504
S'a'
I8819
sS's'
I17
sS'm'
(lp13505
I1
aI3
aI1
aI11
aI23
aI23
aI75
aI115
aI477
aI105
aI541
aI1111
aI209
aI13939
aI17129
aI7565
aI75415
asS'd'
I6753
sa(dp13506
S'a'
I8826
sS's'
I17
sS'm'
(lp13507
I1
aI1
aI1
aI11
aI7
aI61
aI123
aI201
aI305
aI713
aI779
aI2059
aI4899
aI13733
aI20529
aI15617
aI39833
asS'd'
I6754
sa(dp13508
S'a'
I8835
sS's'
I17
sS'm'
(lp13509
I1
aI1
aI7
aI11
aI21
aI7
aI63
aI113
aI213
aI871
aI375
aI29
aI1925
aI15237
aI7091
aI12229
aI8457
asS'd'
I6755
sa(dp13510
S'a'
I8838
sS's'
I17
sS'm'
(lp13511
I1
aI1
aI1
aI7
aI19
aI57
aI83
aI91
aI297
aI255
aI1993
aI63
aI5337
aI4569
aI21243
aI40867
aI46969
asS'd'
I6756
sa(dp13512
S'a'
I8856
sS's'
I17
sS'm'
(lp13513
I1
aI1
aI3
aI7
aI13
aI63
aI91
aI191
aI281
aI259
aI1367
aI3505
aI5885
aI10557
aI12423
aI56303
aI14731
asS'd'
I6757
sa(dp13514
S'a'
I8862
sS's'
I17
sS'm'
(lp13515
I1
aI1
aI5
aI15
aI27
aI15
aI29
aI67
aI115
aI287
aI253
aI1497
aI3739
aI2183
aI14427
aI44931
aI11547
asS'd'
I6758
sa(dp13516
S'a'
I8871
sS's'
I17
sS'm'
(lp13517
I1
aI3
aI1
aI9
aI25
aI61
aI25
aI113
aI137
aI819
aI781
aI3741
aI2457
aI7817
aI31209
aI20707
aI93007
asS'd'
I6759
sa(dp13518
S'a'
I8875
sS's'
I17
sS'm'
(lp13519
I1
aI1
aI7
aI3
aI5
aI13
aI23
aI3
aI365
aI77
aI1117
aI3061
aI4707
aI3013
aI27899
aI10887
aI78677
asS'd'
I6760
sa(dp13520
S'a'
I8890
sS's'
I17
sS'm'
(lp13521
I1
aI3
aI1
aI15
aI1
aI39
aI85
aI107
aI483
aI83
aI603
aI3121
aI1995
aI5241
aI32319
aI9515
aI94551
asS'd'
I6761
sa(dp13522
S'a'
I8892
sS's'
I17
sS'm'
(lp13523
I1
aI1
aI7
aI3
aI27
aI13
aI105
aI41
aI285
aI237
aI1589
aI517
aI2009
aI10833
aI1459
aI26217
aI50759
asS'd'
I6762
sa(dp13524
S'a'
I8898
sS's'
I17
sS'm'
(lp13525
I1
aI1
aI3
aI11
aI27
aI1
aI127
aI83
aI355
aI107
aI1003
aI657
aI4997
aI4123
aI13151
aI56601
aI122307
asS'd'
I6763
sa(dp13526
S'a'
I8927
sS's'
I17
sS'm'
(lp13527
I1
aI1
aI1
aI7
aI13
aI17
aI93
aI75
aI481
aI473
aI131
aI1359
aI4859
aI1319
aI23919
aI50079
aI128849
asS'd'
I6764
sa(dp13528
S'a'
I8928
sS's'
I17
sS'm'
(lp13529
I1
aI1
aI3
aI7
aI9
aI33
aI111
aI229
aI11
aI283
aI1089
aI3049
aI1635
aI959
aI19109
aI62821
aI105391
asS'd'
I6765
sa(dp13530
S'a'
I8945
sS's'
I17
sS'm'
(lp13531
I1
aI3
aI1
aI3
aI9
aI47
aI49
aI169
aI343
aI929
aI1379
aI1985
aI5867
aI6053
aI12179
aI39727
aI116053
asS'd'
I6766
sa(dp13532
S'a'
I8952
sS's'
I17
sS'm'
(lp13533
I1
aI3
aI3
aI15
aI27
aI39
aI61
aI113
aI439
aI719
aI1313
aI3701
aI4793
aI10275
aI2943
aI32405
aI95457
asS'd'
I6767
sa(dp13534
S'a'
I8955
sS's'
I17
sS'm'
(lp13535
I1
aI1
aI1
aI1
aI27
aI49
aI121
aI171
aI319
aI365
aI1593
aI1655
aI63
aI6257
aI18097
aI35285
aI112245
asS'd'
I6768
sa(dp13536
S'a'
I8965
sS's'
I17
sS'm'
(lp13537
I1
aI3
aI1
aI1
aI19
aI33
aI89
aI235
aI281
aI519
aI1867
aI525
aI4475
aI12059
aI26611
aI14789
aI59541
asS'd'
I6769
sa(dp13538
S'a'
I8972
sS's'
I17
sS'm'
(lp13539
I1
aI3
aI1
aI15
aI1
aI51
aI65
aI71
aI131
aI599
aI117
aI2459
aI7421
aI7157
aI24393
aI48139
aI53701
asS'd'
I6770
sa(dp13540
S'a'
I8977
sS's'
I17
sS'm'
(lp13541
I1
aI1
aI7
aI7
aI1
aI41
aI57
aI191
aI207
aI329
aI43
aI1235
aI5671
aI12243
aI22549
aI40751
aI104513
asS'd'
I6771
sa(dp13542
S'a'
I8990
sS's'
I17
sS'm'
(lp13543
I1
aI3
aI5
aI13
aI15
aI21
aI55
aI187
aI283
aI209
aI1511
aI1329
aI6665
aI15953
aI4521
aI16879
aI57625
asS'd'
I6772
sa(dp13544
S'a'
I8996
sS's'
I17
sS'm'
(lp13545
I1
aI1
aI5
aI3
aI3
aI53
aI75
aI123
aI291
aI663
aI1893
aI3669
aI4903
aI8575
aI27971
aI46977
aI56357
asS'd'
I6773
sa(dp13546
S'a'
I9025
sS's'
I17
sS'm'
(lp13547
I1
aI3
aI1
aI5
aI27
aI41
aI19
aI199
aI489
aI197
aI439
aI3299
aI6315
aI6957
aI15809
aI35297
aI5559
asS'd'
I6774
sa(dp13548
S'a'
I9037
sS's'
I17
sS'm'
(lp13549
I1
aI3
aI5
aI1
aI3
aI25
aI109
aI191
aI33
aI543
aI125
aI2309
aI429
aI14059
aI3149
aI45747
aI47357
asS'd'
I6775
sa(dp13550
S'a'
I9040
sS's'
I17
sS'm'
(lp13551
I1
aI1
aI3
aI11
aI15
aI61
aI109
aI103
aI305
aI1
aI1479
aI2781
aI6521
aI8921
aI23681
aI9583
aI87257
asS'd'
I6776
sa(dp13552
S'a'
I9049
sS's'
I17
sS'm'
(lp13553
I1
aI1
aI7
aI15
aI5
aI19
aI121
aI139
aI177
aI967
aI1363
aI705
aI211
aI11877
aI22457
aI34563
aI7801
asS'd'
I6777
sa(dp13554
S'a'
I9062
sS's'
I17
sS'm'
(lp13555
I1
aI1
aI7
aI13
aI9
aI21
aI103
aI95
aI483
aI567
aI5
aI2095
aI4659
aI2447
aI23521
aI27273
aI85867
asS'd'
I6778
sa(dp13556
S'a'
I9068
sS's'
I17
sS'm'
(lp13557
I1
aI3
aI5
aI15
aI23
aI55
aI13
aI237
aI275
aI113
aI1431
aI2931
aI5165
aI5317
aI5625
aI51865
aI42177
asS'd'
I6779
sa(dp13558
S'a'
I9076
sS's'
I17
sS'm'
(lp13559
I1
aI3
aI3
aI7
aI1
aI23
aI15
aI171
aI303
aI43
aI1137
aI1255
aI3843
aI9049
aI1799
aI7075
aI2115
asS'd'
I6780
sa(dp13560
S'a'
I9079
sS's'
I17
sS'm'
(lp13561
I1
aI1
aI7
aI5
aI23
aI53
aI75
aI129
aI1
aI511
aI793
aI265
aI6535
aI9641
aI25173
aI9449
aI46949
asS'd'
I6781
sa(dp13562
S'a'
I9099
sS's'
I17
sS'm'
(lp13563
I1
aI3
aI3
aI1
aI19
aI39
aI51
aI173
aI5
aI281
aI2047
aI4065
aI3225
aI14587
aI16947
aI1459
aI87227
asS'd'
I6782
sa(dp13564
S'a'
I9107
sS's'
I17
sS'm'
(lp13565
I1
aI3
aI7
aI13
aI13
aI53
aI39
aI115
aI403
aI37
aI1533
aI2727
aI2229
aI8291
aI18687
aI59553
aI37629
asS'd'
I6783
sa(dp13566
S'a'
I9114
sS's'
I17
sS'm'
(lp13567
I1
aI3
aI1
aI9
aI3
aI55
aI63
aI191
aI147
aI321
aI1287
aI2419
aI6881
aI2249
aI11141
aI54839
aI50263
asS'd'
I6784
sa(dp13568
S'a'
I9123
sS's'
I17
sS'm'
(lp13569
I1
aI1
aI5
aI3
aI9
aI61
aI85
aI139
aI1
aI409
aI633
aI53
aI163
aI14677
aI13043
aI12253
aI106939
asS'd'
I6785
sa(dp13570
S'a'
I9126
sS's'
I17
sS'm'
(lp13571
I1
aI1
aI7
aI3
aI19
aI3
aI7
aI165
aI497
aI621
aI1563
aI1267
aI8113
aI2383
aI17205
aI13337
aI102547
asS'd'
I6786
sa(dp13572
S'a'
I9137
sS's'
I17
sS'm'
(lp13573
I1
aI3
aI3
aI13
aI15
aI29
aI23
aI31
aI481
aI535
aI471
aI2125
aI331
aI9421
aI29799
aI27097
aI5307
asS'd'
I6787
sa(dp13574
S'a'
I9149
sS's'
I17
sS'm'
(lp13575
I1
aI1
aI1
aI1
aI31
aI45
aI47
aI139
aI235
aI509
aI889
aI685
aI1855
aI13599
aI24431
aI62105
aI109509
asS'd'
I6788
sa(dp13576
S'a'
I9150
sS's'
I17
sS'm'
(lp13577
I1
aI3
aI1
aI7
aI3
aI13
aI25
aI197
aI111
aI45
aI1815
aI1031
aI4803
aI349
aI32369
aI40837
aI111529
asS'd'
I6789
sa(dp13578
S'a'
I9155
sS's'
I17
sS'm'
(lp13579
I1
aI1
aI7
aI1
aI27
aI9
aI3
aI73
aI403
aI321
aI967
aI2713
aI6953
aI16123
aI8611
aI48651
aI120635
asS'd'
I6790
sa(dp13580
S'a'
I9161
sS's'
I17
sS'm'
(lp13581
I1
aI3
aI5
aI3
aI3
aI25
aI69
aI231
aI249
aI393
aI1141
aI1721
aI7071
aI3711
aI15627
aI21815
aI104735
asS'd'
I6791
sa(dp13582
S'a'
I9162
sS's'
I17
sS'm'
(lp13583
I1
aI3
aI1
aI11
aI19
aI63
aI77
aI5
aI55
aI481
aI1021
aI119
aI3941
aI1227
aI10997
aI29513
aI18923
asS'd'
I6792
sa(dp13584
S'a'
I9167
sS's'
I17
sS'm'
(lp13585
I1
aI3
aI7
aI5
aI1
aI11
aI13
aI99
aI365
aI797
aI1993
aI699
aI3091
aI11401
aI3659
aI15339
aI90395
asS'd'
I6793
sa(dp13586
S'a'
I9172
sS's'
I17
sS'm'
(lp13587
I1
aI3
aI5
aI7
aI31
aI43
aI55
aI143
aI273
aI379
aI1189
aI1689
aI4811
aI5159
aI3281
aI63819
aI57065
asS'd'
I6794
sa(dp13588
S'a'
I9186
sS's'
I17
sS'm'
(lp13589
I1
aI1
aI1
aI13
aI9
aI25
aI9
aI3
aI461
aI281
aI959
aI2439
aI3187
aI4837
aI13857
aI20221
aI29733
asS'd'
I6795
sa(dp13590
S'a'
I9188
sS's'
I17
sS'm'
(lp13591
I1
aI1
aI7
aI11
aI31
aI17
aI13
aI101
aI81
aI921
aI1329
aI2421
aI2747
aI9435
aI23313
aI7093
aI7547
asS'd'
I6796
sa(dp13592
S'a'
I9191
sS's'
I17
sS'm'
(lp13593
I1
aI1
aI3
aI3
aI9
aI51
aI67
aI95
aI511
aI1011
aI1519
aI4089
aI5001
aI1351
aI15367
aI50665
aI92111
asS'd'
I6797
sa(dp13594
S'a'
I9198
sS's'
I17
sS'm'
(lp13595
I1
aI1
aI5
aI13
aI27
aI43
aI115
aI77
aI439
aI589
aI31
aI915
aI7027
aI697
aI25143
aI1443
aI59093
asS'd'
I6798
sa(dp13596
S'a'
I9200
sS's'
I17
sS'm'
(lp13597
I1
aI1
aI7
aI3
aI17
aI5
aI107
aI117
aI133
aI649
aI1309
aI2979
aI969
aI9789
aI12597
aI24507
aI106825
asS'd'
I6799
sa(dp13598
S'a'
I9205
sS's'
I17
sS'm'
(lp13599
I1
aI1
aI7
aI13
aI1
aI27
aI97
aI35
aI431
aI183
aI199
aI2619
aI515
aI89
aI20281
aI30291
aI97977
asS'd'
I6800
sa(dp13600
S'a'
I9206
sS's'
I17
sS'm'
(lp13601
I1
aI1
aI7
aI1
aI31
aI9
aI35
aI11
aI359
aI21
aI1875
aI3227
aI1307
aI15691
aI17343
aI21163
aI84671
asS'd'
I6801
sa(dp13602
S'a'
I9215
sS's'
I17
sS'm'
(lp13603
I1
aI3
aI1
aI11
aI29
aI21
aI47
aI137
aI441
aI841
aI1641
aI3283
aI1371
aI8835
aI16287
aI45009
aI13779
asS'd'
I6802
sa(dp13604
S'a'
I9227
sS's'
I17
sS'm'
(lp13605
I1
aI1
aI3
aI9
aI23
aI53
aI1
aI99
aI473
aI649
aI447
aI2589
aI5667
aI15579
aI6497
aI44321
aI46993
asS'd'
I6803
sa(dp13606
S'a'
I9232
sS's'
I17
sS'm'
(lp13607
I1
aI1
aI7
aI9
aI31
aI63
aI95
aI81
aI197
aI373
aI1027
aI3959
aI7189
aI13369
aI17287
aI53643
aI12673
asS'd'
I6804
sa(dp13608
S'a'
I9241
sS's'
I17
sS'm'
(lp13609
I1
aI3
aI1
aI5
aI25
aI61
aI79
aI183
aI489
aI725
aI1077
aI1147
aI113
aI7357
aI27505
aI529
aI61855
asS'd'
I6805
sa(dp13610
S'a'
I9244
sS's'
I17
sS'm'
(lp13611
I1
aI1
aI7
aI11
aI19
aI35
aI73
aI223
aI125
aI765
aI1303
aI2887
aI7861
aI14839
aI9537
aI27027
aI94327
asS'd'
I6806
sa(dp13612
S'a'
I9251
sS's'
I17
sS'm'
(lp13613
I1
aI3
aI1
aI3
aI17
aI35
aI63
aI233
aI317
aI133
aI1837
aI3339
aI4351
aI10071
aI5005
aI13245
aI34327
asS'd'
I6807
sa(dp13614
S'a'
I9254
sS's'
I17
sS'm'
(lp13615
I1
aI3
aI1
aI3
aI17
aI13
aI59
aI113
aI247
aI1015
aI1831
aI3391
aI6337
aI6853
aI7145
aI64309
aI40109
asS'd'
I6808
sa(dp13616
S'a'
I9275
sS's'
I17
sS'm'
(lp13617
I1
aI3
aI5
aI13
aI15
aI23
aI65
aI203
aI241
aI545
aI1521
aI1253
aI3171
aI7777
aI21145
aI565
aI87813
asS'd'
I6809
sa(dp13618
S'a'
I9283
sS's'
I17
sS'm'
(lp13619
I1
aI1
aI5
aI15
aI31
aI9
aI9
aI145
aI409
aI155
aI409
aI2935
aI5817
aI11427
aI32617
aI38167
aI69465
asS'd'
I6810
sa(dp13620
S'a'
I9285
sS's'
I17
sS'm'
(lp13621
I1
aI1
aI5
aI11
aI19
aI31
aI43
aI85
aI97
aI931
aI687
aI1501
aI3991
aI2215
aI11091
aI64735
aI56999
asS'd'
I6811
sa(dp13622
S'a'
I9303
sS's'
I17
sS'm'
(lp13623
I1
aI1
aI1
aI3
aI7
aI11
aI101
aI21
aI345
aI829
aI531
aI1475
aI6617
aI1187
aI26885
aI32135
aI9733
asS'd'
I6812
sa(dp13624
S'a'
I9304
sS's'
I17
sS'm'
(lp13625
I1
aI3
aI5
aI11
aI7
aI49
aI79
aI197
aI57
aI15
aI1845
aI1485
aI6167
aI10887
aI17083
aI59367
aI7411
asS'd'
I6813
sa(dp13626
S'a'
I9313
sS's'
I17
sS'm'
(lp13627
I1
aI3
aI7
aI5
aI9
aI33
aI7
aI91
aI311
aI847
aI1435
aI3573
aI3693
aI5369
aI26817
aI30105
aI115337
asS'd'
I6814
sa(dp13628
S'a'
I9314
sS's'
I17
sS'm'
(lp13629
I1
aI3
aI1
aI9
aI25
aI43
aI65
aI69
aI225
aI337
aI575
aI1979
aI5555
aI8499
aI8127
aI33035
aI52549
asS'd'
I6815
sa(dp13630
S'a'
I9320
sS's'
I17
sS'm'
(lp13631
I1
aI1
aI3
aI11
aI17
aI29
aI71
aI99
aI379
aI145
aI1067
aI2561
aI7635
aI5647
aI32491
aI56621
aI93603
asS'd'
I6816
sa(dp13632
S'a'
I9328
sS's'
I17
sS'm'
(lp13633
I1
aI1
aI5
aI13
aI25
aI43
aI75
aI237
aI407
aI393
aI1219
aI3651
aI7719
aI11685
aI26123
aI62767
aI1043
asS'd'
I6817
sa(dp13634
S'a'
I9333
sS's'
I17
sS'm'
(lp13635
I1
aI1
aI7
aI15
aI13
aI59
aI9
aI163
aI273
aI225
aI873
aI3201
aI633
aI6121
aI18777
aI58763
aI77731
asS'd'
I6818
sa(dp13636
S'a'
I9337
sS's'
I17
sS'm'
(lp13637
I1
aI3
aI7
aI7
aI3
aI7
aI99
aI155
aI279
aI991
aI799
aI753
aI7205
aI9567
aI23643
aI38263
aI19083
asS'd'
I6819
sa(dp13638
S'a'
I9338
sS's'
I17
sS'm'
(lp13639
I1
aI3
aI7
aI11
aI11
aI29
aI65
aI3
aI207
aI575
aI253
aI2407
aI7935
aI11323
aI23239
aI1923
aI47737
asS'd'
I6820
sa(dp13640
S'a'
I9340
sS's'
I17
sS'm'
(lp13641
I1
aI1
aI5
aI9
aI25
aI47
aI1
aI25
aI397
aI1009
aI193
aI4031
aI3023
aI2029
aI10561
aI32363
aI104405
asS'd'
I6821
sa(dp13642
S'a'
I9353
sS's'
I17
sS'm'
(lp13643
I1
aI3
aI7
aI9
aI19
aI55
aI63
aI179
aI385
aI97
aI461
aI3393
aI8137
aI8929
aI17621
aI9611
aI58925
asS'd'
I6822
sa(dp13644
S'a'
I9356
sS's'
I17
sS'm'
(lp13645
I1
aI1
aI1
aI7
aI1
aI17
aI127
aI45
aI157
aI529
aI809
aI3545
aI5173
aI5083
aI13325
aI52295
aI91261
asS'd'
I6823
sa(dp13646
S'a'
I9364
sS's'
I17
sS'm'
(lp13647
I1
aI1
aI7
aI9
aI25
aI49
aI99
aI79
aI157
aI535
aI1569
aI2195
aI1725
aI1187
aI18423
aI47957
aI10043
asS'd'
I6824
sa(dp13648
S'a'
I9373
sS's'
I17
sS'm'
(lp13649
I1
aI1
aI3
aI7
aI3
aI31
aI83
aI45
aI199
aI665
aI1261
aI3497
aI7885
aI5761
aI17187
aI12041
aI12867
asS'd'
I6825
sa(dp13650
S'a'
I9374
sS's'
I17
sS'm'
(lp13651
I1
aI3
aI1
aI7
aI3
aI55
aI73
aI215
aI41
aI1011
aI1883
aI1051
aI7293
aI1881
aI27435
aI29459
aI130933
asS'd'
I6826
sa(dp13652
S'a'
I9378
sS's'
I17
sS'm'
(lp13653
I1
aI1
aI3
aI9
aI21
aI31
aI113
aI209
aI35
aI771
aI365
aI3151
aI787
aI3845
aI26555
aI13823
aI36951
asS'd'
I6827
sa(dp13654
S'a'
I9380
sS's'
I17
sS'm'
(lp13655
I1
aI3
aI7
aI15
aI13
aI21
aI119
aI91
aI15
aI251
aI1337
aI2715
aI1665
aI3451
aI8309
aI11033
aI127159
asS'd'
I6828
sa(dp13656
S'a'
I9389
sS's'
I17
sS'm'
(lp13657
I1
aI3
aI1
aI9
aI9
aI63
aI5
aI145
aI357
aI9
aI859
aI1565
aI1141
aI14689
aI25121
aI41337
aI83357
asS'd'
I6829
sa(dp13658
S'a'
I9395
sS's'
I17
sS'm'
(lp13659
I1
aI1
aI7
aI11
aI13
aI63
aI57
aI151
aI33
aI595
aI2025
aI571
aI4713
aI11019
aI26771
aI16221
aI92439
asS'd'
I6830
sa(dp13660
S'a'
I9412
sS's'
I17
sS'm'
(lp13661
I1
aI3
aI3
aI15
aI29
aI49
aI93
aI131
aI167
aI835
aI33
aI263
aI93
aI8475
aI16139
aI61237
aI95081
asS'd'
I6831
sa(dp13662
S'a'
I9422
sS's'
I17
sS'm'
(lp13663
I1
aI1
aI7
aI13
aI1
aI57
aI43
aI91
aI485
aI841
aI1415
aI3083
aI2665
aI8521
aI9825
aI59955
aI21763
asS'd'
I6832
sa(dp13664
S'a'
I9439
sS's'
I17
sS'm'
(lp13665
I1
aI1
aI1
aI1
aI29
aI47
aI63
aI107
aI439
aI847
aI537
aI2011
aI7571
aI3699
aI23961
aI54887
aI92681
asS'd'
I6833
sa(dp13666
S'a'
I9450
sS's'
I17
sS'm'
(lp13667
I1
aI3
aI7
aI5
aI27
aI41
aI105
aI161
aI95
aI821
aI451
aI2627
aI4687
aI1899
aI18851
aI35167
aI6869
asS'd'
I6834
sa(dp13668
S'a'
I9452
sS's'
I17
sS'm'
(lp13669
I1
aI1
aI1
aI11
aI7
aI7
aI13
aI163
aI399
aI471
aI1587
aI2561
aI1241
aI5365
aI27189
aI49883
aI68101
asS'd'
I6835
sa(dp13670
S'a'
I9482
sS's'
I17
sS'm'
(lp13671
I1
aI3
aI7
aI9
aI19
aI5
aI119
aI251
aI151
aI359
aI235
aI2387
aI3919
aI7135
aI17591
aI1053
aI6265
asS'd'
I6836
sa(dp13672
S'a'
I9487
sS's'
I17
sS'm'
(lp13673
I1
aI1
aI5
aI9
aI13
aI25
aI43
aI23
aI453
aI693
aI517
aI1235
aI1045
aI4299
aI27877
aI3733
aI72269
asS'd'
I6837
sa(dp13674
S'a'
I9489
sS's'
I17
sS'm'
(lp13675
I1
aI1
aI7
aI1
aI27
aI43
aI103
aI249
aI487
aI67
aI855
aI3239
aI2157
aI8121
aI4701
aI37803
aI49971
asS'd'
I6838
sa(dp13676
S'a'
I9499
sS's'
I17
sS'm'
(lp13677
I1
aI1
aI3
aI13
aI1
aI37
aI125
aI115
aI365
aI57
aI1419
aI4085
aI7039
aI10079
aI14991
aI48861
aI61979
asS'd'
I6839
sa(dp13678
S'a'
I9501
sS's'
I17
sS'm'
(lp13679
I1
aI1
aI5
aI5
aI3
aI35
aI109
aI19
aI219
aI653
aI1219
aI1625
aI6847
aI11271
aI4525
aI56341
aI57801
asS'd'
I6840
sa(dp13680
S'a'
I9508
sS's'
I17
sS'm'
(lp13681
I1
aI3
aI7
aI5
aI31
aI19
aI37
aI73
aI185
aI13
aI1723
aI1139
aI5919
aI11717
aI27161
aI13635
aI51765
asS'd'
I6841
sa(dp13682
S'a'
I9515
sS's'
I17
sS'm'
(lp13683
I1
aI1
aI1
aI1
aI19
aI61
aI53
aI111
aI215
aI189
aI1199
aI591
aI943
aI2111
aI17171
aI15621
aI128459
asS'd'
I6842
sa(dp13684
S'a'
I9518
sS's'
I17
sS'm'
(lp13685
I1
aI1
aI7
aI9
aI17
aI61
aI101
aI159
aI85
aI537
aI15
aI1427
aI6139
aI4091
aI32639
aI28655
aI115385
asS'd'
I6843
sa(dp13686
S'a'
I9520
sS's'
I17
sS'm'
(lp13687
I1
aI1
aI7
aI5
aI23
aI31
aI125
aI7
aI151
aI967
aI1079
aI4059
aI3287
aI11673
aI19307
aI49469
aI65981
asS'd'
I6844
sa(dp13688
S'a'
I9526
sS's'
I17
sS'm'
(lp13689
I1
aI3
aI3
aI1
aI29
aI59
aI95
aI119
aI31
aI427
aI1653
aI721
aI5509
aI6385
aI17043
aI45133
aI74155
asS'd'
I6845
sa(dp13690
S'a'
I9537
sS's'
I17
sS'm'
(lp13691
I1
aI1
aI3
aI9
aI13
aI61
aI35
aI189
aI1
aI559
aI119
aI3719
aI4137
aI1369
aI19147
aI10923
aI43909
asS'd'
I6846
sa(dp13692
S'a'
I9552
sS's'
I17
sS'm'
(lp13693
I1
aI3
aI3
aI13
aI1
aI41
aI31
aI185
aI451
aI379
aI29
aI153
aI4121
aI13153
aI4171
aI36993
aI109241
asS'd'
I6847
sa(dp13694
S'a'
I9571
sS's'
I17
sS'm'
(lp13695
I1
aI1
aI1
aI9
aI15
aI41
aI99
aI17
aI21
aI93
aI649
aI2765
aI6955
aI10843
aI12547
aI64989
aI63713
asS'd'
I6848
sa(dp13696
S'a'
I9588
sS's'
I17
sS'm'
(lp13697
I1
aI1
aI7
aI5
aI5
aI5
aI73
aI187
aI473
aI235
aI1907
aI409
aI7335
aI4429
aI7493
aI20703
aI14505
asS'd'
I6849
sa(dp13698
S'a'
I9613
sS's'
I17
sS'm'
(lp13699
I1
aI1
aI3
aI11
aI27
aI59
aI17
aI103
aI337
aI117
aI1241
aI951
aI3701
aI10407
aI16741
aI46531
aI56485
asS'd'
I6850
sa(dp13700
S'a'
I9619
sS's'
I17
sS'm'
(lp13701
I1
aI1
aI3
aI15
aI11
aI51
aI111
aI189
aI137
aI939
aI97
aI1563
aI851
aI13949
aI1375
aI41463
aI61445
asS'd'
I6851
sa(dp13702
S'a'
I9622
sS's'
I17
sS'm'
(lp13703
I1
aI1
aI7
aI9
aI19
aI39
aI117
aI173
aI165
aI547
aI483
aI361
aI6819
aI15093
aI13631
aI29785
aI29593
asS'd'
I6852
sa(dp13704
S'a'
I9637
sS's'
I17
sS'm'
(lp13705
I1
aI3
aI3
aI5
aI15
aI39
aI41
aI249
aI455
aI79
aI233
aI3133
aI405
aI9487
aI23161
aI32751
aI117743
asS'd'
I6853
sa(dp13706
S'a'
I9652
sS's'
I17
sS'm'
(lp13707
I1
aI1
aI5
aI15
aI7
aI63
aI7
aI57
aI127
aI349
aI1913
aI1145
aI3371
aI3733
aI30971
aI35717
aI60935
asS'd'
I6854
sa(dp13708
S'a'
I9655
sS's'
I17
sS'm'
(lp13709
I1
aI1
aI7
aI11
aI7
aI57
aI49
aI63
aI51
aI233
aI855
aI2125
aI6961
aI15011
aI28503
aI40549
aI47175
asS'd'
I6855
sa(dp13710
S'a'
I9661
sS's'
I17
sS'm'
(lp13711
I1
aI3
aI7
aI1
aI25
aI49
aI35
aI39
aI237
aI545
aI1637
aI1401
aI3279
aI10499
aI14463
aI34973
aI29485
asS'd'
I6856
sa(dp13712
S'a'
I9664
sS's'
I17
sS'm'
(lp13713
I1
aI3
aI3
aI13
aI7
aI13
aI79
aI141
aI55
aI277
aI843
aI3087
aI2339
aI6855
aI10635
aI13021
aI11273
asS'd'
I6857
sa(dp13714
S'a'
I9669
sS's'
I17
sS'm'
(lp13715
I1
aI3
aI1
aI1
aI11
aI39
aI51
aI255
aI119
aI691
aI559
aI3287
aI5485
aI791
aI19283
aI51027
aI8061
asS'd'
I6858
sa(dp13716
S'a'
I9681
sS's'
I17
sS'm'
(lp13717
I1
aI3
aI7
aI7
aI3
aI59
aI119
aI241
aI185
aI81
aI1843
aI2313
aI7471
aI15689
aI2271
aI59781
aI107439
asS'd'
I6859
sa(dp13718
S'a'
I9682
sS's'
I17
sS'm'
(lp13719
I1
aI3
aI3
aI3
aI17
aI63
aI93
aI217
aI329
aI39
aI583
aI3031
aI4315
aI4623
aI12557
aI42063
aI11877
asS'd'
I6860
sa(dp13720
S'a'
I9688
sS's'
I17
sS'm'
(lp13721
I1
aI1
aI1
aI1
aI15
aI57
aI37
aI233
aI387
aI639
aI37
aI425
aI637
aI1577
aI16449
aI33665
aI80417
asS'd'
I6861
sa(dp13722
S'a'
I9697
sS's'
I17
sS'm'
(lp13723
I1
aI1
aI1
aI15
aI25
aI1
aI67
aI159
aI423
aI961
aI959
aI417
aI5657
aI8417
aI8127
aI29251
aI105893
asS'd'
I6862
sa(dp13724
S'a'
I9700
sS's'
I17
sS'm'
(lp13725
I1
aI3
aI5
aI15
aI31
aI9
aI87
aI217
aI259
aI771
aI1663
aI2899
aI1531
aI7849
aI1961
aI61487
aI55399
asS'd'
I6863
sa(dp13726
S'a'
I9715
sS's'
I17
sS'm'
(lp13727
I1
aI1
aI3
aI9
aI21
aI13
aI39
aI107
aI89
aI811
aI449
aI2569
aI4617
aI8977
aI1649
aI37721
aI48943
asS'd'
I6864
sa(dp13728
S'a'
I9722
sS's'
I17
sS'm'
(lp13729
I1
aI3
aI7
aI15
aI15
aI59
aI63
aI195
aI287
aI677
aI269
aI1715
aI3545
aI3269
aI5231
aI46433
aI25921
asS'd'
I6865
sa(dp13730
S'a'
I9727
sS's'
I17
sS'm'
(lp13731
I1
aI1
aI5
aI7
aI19
aI27
aI57
aI221
aI243
aI47
aI1791
aI2309
aI2751
aI4403
aI7083
aI34223
aI64905
asS'd'
I6866
sa(dp13732
S'a'
I9734
sS's'
I17
sS'm'
(lp13733
I1
aI1
aI1
aI15
aI1
aI63
aI119
aI155
aI383
aI649
aI429
aI3857
aI7309
aI9823
aI9539
aI8933
aI128573
asS'd'
I6867
sa(dp13734
S'a'
I9740
sS's'
I17
sS'm'
(lp13735
I1
aI3
aI7
aI11
aI17
aI19
aI99
aI19
aI321
aI415
aI1501
aI2123
aI6119
aI9705
aI11397
aI39521
aI34327
asS'd'
I6868
sa(dp13736
S'a'
I9743
sS's'
I17
sS'm'
(lp13737
I1
aI1
aI5
aI15
aI29
aI37
aI9
aI95
aI417
aI19
aI1637
aI2949
aI4961
aI10743
aI9619
aI16045
aI48083
asS'd'
I6869
sa(dp13738
S'a'
I9745
sS's'
I17
sS'm'
(lp13739
I1
aI1
aI1
aI11
aI21
aI17
aI57
aI23
aI247
aI201
aI1781
aI779
aI2207
aI2511
aI4829
aI13847
aI77593
asS'd'
I6870
sa(dp13740
S'a'
I9757
sS's'
I17
sS'm'
(lp13741
I1
aI3
aI1
aI13
aI7
aI1
aI95
aI87
aI223
aI73
aI1129
aI383
aI1355
aI4965
aI29645
aI63465
aI76281
asS'd'
I6871
sa(dp13742
S'a'
I9761
sS's'
I17
sS'm'
(lp13743
I1
aI3
aI3
aI13
aI3
aI47
aI33
aI123
aI155
aI621
aI1019
aI1817
aI4083
aI4723
aI24701
aI47503
aI18007
asS'd'
I6872
sa(dp13744
S'a'
I9762
sS's'
I17
sS'm'
(lp13745
I1
aI1
aI7
aI15
aI13
aI41
aI73
aI93
aI379
aI923
aI1183
aI2475
aI5901
aI10599
aI10053
aI9941
aI112107
asS'd'
I6873
sa(dp13746
S'a'
I9767
sS's'
I17
sS'm'
(lp13747
I1
aI1
aI3
aI3
aI13
aI35
aI59
aI231
aI45
aI1011
aI1101
aI2467
aI2703
aI10305
aI12575
aI7587
aI25737
asS'd'
I6874
sa(dp13748
S'a'
I9768
sS's'
I17
sS'm'
(lp13749
I1
aI3
aI7
aI1
aI21
aI31
aI9
aI55
aI373
aI779
aI397
aI1551
aI5139
aI16339
aI1769
aI10413
aI74059
asS'd'
I6875
sa(dp13750
S'a'
I9774
sS's'
I17
sS'm'
(lp13751
I1
aI1
aI7
aI15
aI7
aI3
aI67
aI179
aI411
aI217
aI1219
aI13
aI1577
aI13463
aI12263
aI41465
aI83001
asS'd'
I6876
sa(dp13752
S'a'
I9786
sS's'
I17
sS'm'
(lp13753
I1
aI3
aI7
aI1
aI21
aI53
aI7
aI187
aI395
aI777
aI391
aI737
aI47
aI12681
aI16749
aI26507
aI49415
asS'd'
I6877
sa(dp13754
S'a'
I9796
sS's'
I17
sS'm'
(lp13755
I1
aI1
aI5
aI7
aI5
aI57
aI93
aI53
aI419
aI731
aI825
aI487
aI45
aI9199
aI20947
aI56067
aI45343
asS'd'
I6878
sa(dp13756
S'a'
I9820
sS's'
I17
sS'm'
(lp13757
I1
aI3
aI3
aI9
aI31
aI41
aI35
aI133
aI63
aI293
aI1503
aI51
aI3111
aI15711
aI15051
aI1965
aI64951
asS'd'
I6879
sa(dp13758
S'a'
I9823
sS's'
I17
sS'm'
(lp13759
I1
aI1
aI5
aI9
aI9
aI47
aI53
aI229
aI405
aI621
aI1795
aI1923
aI6609
aI6983
aI1695
aI18021
aI71893
asS'd'
I6880
sa(dp13760
S'a'
I9839
sS's'
I17
sS'm'
(lp13761
I1
aI1
aI5
aI9
aI23
aI13
aI107
aI13
aI149
aI759
aI1113
aI1329
aI1747
aI14159
aI16705
aI61841
aI82955
asS'd'
I6881
sa(dp13762
S'a'
I9844
sS's'
I17
sS'm'
(lp13763
I1
aI3
aI3
aI9
aI25
aI49
aI31
aI145
aI481
aI609
aI1847
aI1485
aI6345
aI7859
aI21231
aI37303
aI69975
asS'd'
I6882
sa(dp13764
S'a'
I9851
sS's'
I17
sS'm'
(lp13765
I1
aI3
aI1
aI15
aI13
aI49
aI59
aI221
aI27
aI517
aI431
aI3961
aI6401
aI8483
aI10161
aI37453
aI128237
asS'd'
I6883
sa(dp13766
S'a'
I9853
sS's'
I17
sS'm'
(lp13767
I1
aI1
aI3
aI1
aI3
aI55
aI37
aI111
aI263
aI735
aI655
aI2831
aI2219
aI9449
aI8413
aI49585
aI91355
asS'd'
I6884
sa(dp13768
S'a'
I9863
sS's'
I17
sS'm'
(lp13769
I1
aI3
aI7
aI1
aI31
aI33
aI7
aI55
aI261
aI977
aI1215
aI1967
aI7297
aI14815
aI27009
aI35001
aI89671
asS'd'
I6885
sa(dp13770
S'a'
I9864
sS's'
I17
sS'm'
(lp13771
I1
aI1
aI7
aI11
aI13
aI21
aI33
aI151
aI195
aI373
aI181
aI1631
aI355
aI7857
aI12555
aI7531
aI50417
asS'd'
I6886
sa(dp13772
S'a'
I9877
sS's'
I17
sS'm'
(lp13773
I1
aI3
aI1
aI15
aI19
aI25
aI79
aI195
aI237
aI385
aI1531
aI2509
aI4371
aI16103
aI3575
aI62265
aI124251
asS'd'
I6887
sa(dp13774
S'a'
I9884
sS's'
I17
sS'm'
(lp13775
I1
aI3
aI1
aI11
aI5
aI61
aI21
aI159
aI51
aI37
aI845
aI3075
aI8039
aI14269
aI10505
aI36369
aI73793
asS'd'
I6888
sa(dp13776
S'a'
I9888
sS's'
I17
sS'm'
(lp13777
I1
aI3
aI5
aI9
aI11
aI43
aI67
aI57
aI271
aI451
aI989
aI3705
aI2481
aI10717
aI10861
aI63785
aI10183
asS'd'
I6889
sa(dp13778
S'a'
I9897
sS's'
I17
sS'm'
(lp13779
I1
aI3
aI3
aI5
aI13
aI29
aI119
aI171
aI439
aI459
aI479
aI3173
aI3781
aI11131
aI6827
aI53925
aI119939
asS'd'
I6890
sa(dp13780
S'a'
I9915
sS's'
I17
sS'm'
(lp13781
I1
aI3
aI7
aI3
aI27
aI21
aI1
aI167
aI79
aI305
aI1283
aI1903
aI5483
aI5727
aI17911
aI16075
aI97629
asS'd'
I6891
sa(dp13782
S'a'
I9925
sS's'
I17
sS'm'
(lp13783
I1
aI3
aI1
aI3
aI23
aI21
aI29
aI185
aI227
aI295
aI915
aI2033
aI6269
aI2089
aI20785
aI15207
aI115675
asS'd'
I6892
sa(dp13784
S'a'
I9949
sS's'
I17
sS'm'
(lp13785
I1
aI3
aI7
aI15
aI11
aI15
aI65
aI103
aI249
aI27
aI1805
aI2079
aI4797
aI2535
aI16865
aI61449
aI90923
asS'd'
I6893
sa(dp13786
S'a'
I9954
sS's'
I17
sS'm'
(lp13787
I1
aI3
aI7
aI9
aI27
aI41
aI77
aI181
aI457
aI677
aI633
aI1601
aI8085
aI2431
aI7957
aI55913
aI38677
asS'd'
I6894
sa(dp13788
S'a'
I9960
sS's'
I17
sS'm'
(lp13789
I1
aI1
aI5
aI7
aI11
aI37
aI3
aI221
aI79
aI895
aI1023
aI653
aI3925
aI12755
aI19729
aI18221
aI91123
asS'd'
I6895
sa(dp13790
S'a'
I9965
sS's'
I17
sS'm'
(lp13791
I1
aI3
aI1
aI5
aI23
aI61
aI119
aI191
aI425
aI41
aI853
aI3497
aI6915
aI1927
aI5513
aI55303
aI4895
asS'd'
I6896
sa(dp13792
S'a'
I9978
sS's'
I17
sS'm'
(lp13793
I1
aI3
aI5
aI3
aI7
aI35
aI47
aI243
aI167
aI821
aI267
aI2149
aI5797
aI6329
aI32495
aI51037
aI18313
asS'd'
I6897
sa(dp13794
S'a'
I9986
sS's'
I17
sS'm'
(lp13795
I1
aI1
aI7
aI9
aI23
aI29
aI79
aI205
aI115
aI839
aI1217
aI479
aI1601
aI9681
aI1
aI35293
aI28731
asS'd'
I6898
sa(dp13796
S'a'
I9992
sS's'
I17
sS'm'
(lp13797
I1
aI3
aI3
aI5
aI31
aI17
aI31
aI161
aI35
aI953
aI377
aI451
aI7985
aI11371
aI15115
aI60131
aI27033
asS'd'
I6899
sa(dp13798
S'a'
I9995
sS's'
I17
sS'm'
(lp13799
I1
aI1
aI3
aI9
aI15
aI19
aI43
aI215
aI327
aI429
aI145
aI1837
aI725
aI14775
aI10465
aI7367
aI21271
asS'd'
I6900
sa(dp13800
S'a'
I10005
sS's'
I17
sS'm'
(lp13801
I1
aI3
aI7
aI13
aI31
aI17
aI85
aI49
aI487
aI795
aI1679
aI599
aI3783
aI3195
aI2683
aI53475
aI38603
asS'd'
I6901
sa(dp13802
S'a'
I10026
sS's'
I17
sS'm'
(lp13803
I1
aI1
aI1
aI7
aI19
aI11
aI71
aI143
aI443
aI199
aI1117
aI3445
aI6429
aI12037
aI13751
aI43609
aI101563
asS'd'
I6902
sa(dp13804
S'a'
I10031
sS's'
I17
sS'm'
(lp13805
I1
aI3
aI5
aI7
aI29
aI63
aI65
aI87
aI305
aI721
aI851
aI2235
aI4987
aI3051
aI23015
aI1281
aI15755
asS'd'
I6903
sa(dp13806
S'a'
I10040
sS's'
I17
sS'm'
(lp13807
I1
aI1
aI3
aI9
aI17
aI3
aI57
aI47
aI223
aI305
aI1409
aI235
aI4379
aI5779
aI27695
aI22535
aI9387
asS'd'
I6904
sa(dp13808
S'a'
I10051
sS's'
I17
sS'm'
(lp13809
I1
aI1
aI3
aI11
aI25
aI33
aI75
aI141
aI155
aI699
aI85
aI1729
aI2551
aI7101
aI7739
aI18025
aI100819
asS'd'
I6905
sa(dp13810
S'a'
I10057
sS's'
I17
sS'm'
(lp13811
I1
aI3
aI3
aI13
aI5
aI45
aI63
aI83
aI141
aI383
aI1931
aI3343
aI7397
aI4823
aI28893
aI41279
aI67805
asS'd'
I6906
sa(dp13812
S'a'
I10072
sS's'
I17
sS'm'
(lp13813
I1
aI3
aI5
aI7
aI19
aI29
aI97
aI67
aI177
aI583
aI1783
aI4007
aI5087
aI805
aI30999
aI23197
aI117553
asS'd'
I6907
sa(dp13814
S'a'
I10096
sS's'
I17
sS'm'
(lp13815
I1
aI3
aI5
aI1
aI25
aI41
aI33
aI109
aI511
aI449
aI653
aI995
aI5881
aI2163
aI13689
aI54385
aI97419
asS'd'
I6908
sa(dp13816
S'a'
I10102
sS's'
I17
sS'm'
(lp13817
I1
aI3
aI3
aI13
aI25
aI17
aI49
aI77
aI497
aI659
aI783
aI3513
aI3735
aI3541
aI573
aI50237
aI99247
asS'd'
I6909
sa(dp13818
S'a'
I10105
sS's'
I17
sS'm'
(lp13819
I1
aI3
aI1
aI7
aI17
aI13
aI37
aI169
aI19
aI965
aI289
aI455
aI6855
aI11233
aI7553
aI7007
aI57389
asS'd'
I6910
sa(dp13820
S'a'
I10115
sS's'
I17
sS'm'
(lp13821
I1
aI1
aI7
aI11
aI5
aI15
aI11
aI177
aI75
aI243
aI453
aI3861
aI3091
aI4625
aI12489
aI11537
aI74199
asS'd'
I6911
sa(dp13822
S'a'
I10124
sS's'
I17
sS'm'
(lp13823
I1
aI1
aI5
aI13
aI17
aI21
aI23
aI57
aI343
aI985
aI1755
aI3947
aI3899
aI11847
aI19321
aI62295
aI51265
asS'd'
I6912
sa(dp13824
S'a'
I10139
sS's'
I17
sS'm'
(lp13825
I1
aI1
aI3
aI9
aI19
aI37
aI31
aI243
aI433
aI725
aI535
aI3733
aI33
aI7885
aI1425
aI41919
aI66507
asS'd'
I6913
sa(dp13826
S'a'
I10145
sS's'
I17
sS'm'
(lp13827
I1
aI3
aI5
aI11
aI15
aI11
aI25
aI255
aI93
aI33
aI71
aI2389
aI1855
aI317
aI12773
aI13311
aI81927
asS'd'
I6914
sa(dp13828
S'a'
I10148
sS's'
I17
sS'm'
(lp13829
I1
aI3
aI1
aI3
aI7
aI55
aI21
aI175
aI357
aI235
aI1679
aI931
aI2051
aI14213
aI20539
aI38049
aI122513
asS'd'
I6915
sa(dp13830
S'a'
I10157
sS's'
I17
sS'm'
(lp13831
I1
aI1
aI5
aI15
aI5
aI51
aI127
aI79
aI297
aI135
aI1423
aI2783
aI7229
aI14451
aI27619
aI7299
aI49189
asS'd'
I6916
sa(dp13832
S'a'
I10158
sS's'
I17
sS'm'
(lp13833
I1
aI1
aI1
aI3
aI5
aI13
aI9
aI209
aI455
aI483
aI1745
aI323
aI789
aI7645
aI26373
aI61659
aI23671
asS'd'
I6917
sa(dp13834
S'a'
I10163
sS's'
I17
sS'm'
(lp13835
I1
aI1
aI1
aI9
aI23
aI63
aI99
aI91
aI377
aI275
aI275
aI3005
aI1563
aI5945
aI23825
aI33211
aI52753
asS'd'
I6918
sa(dp13836
S'a'
I10180
sS's'
I17
sS'm'
(lp13837
I1
aI1
aI1
aI1
aI31
aI55
aI31
aI109
aI481
aI581
aI771
aI197
aI6155
aI3465
aI8451
aI25925
aI41159
asS'd'
I6919
sa(dp13838
S'a'
I10187
sS's'
I17
sS'm'
(lp13839
I1
aI3
aI7
aI13
aI5
aI33
aI113
aI161
aI265
aI493
aI1723
aI513
aI5111
aI10177
aI21755
aI5321
aI58831
asS'd'
I6920
sa(dp13840
S'a'
I10198
sS's'
I17
sS'm'
(lp13841
I1
aI1
aI7
aI1
aI21
aI33
aI117
aI183
aI89
aI689
aI1253
aI2215
aI6565
aI3079
aI16343
aI22427
aI96447
asS'd'
I6921
sa(dp13842
S'a'
I10208
sS's'
I17
sS'm'
(lp13843
I1
aI1
aI1
aI5
aI15
aI61
aI5
aI139
aI111
aI463
aI573
aI1907
aI4615
aI14975
aI5715
aI51017
aI69827
asS'd'
I6922
sa(dp13844
S'a'
I10214
sS's'
I17
sS'm'
(lp13845
I1
aI1
aI1
aI13
aI3
aI3
aI117
aI249
aI25
aI361
aI1177
aI2901
aI1601
aI11381
aI18981
aI44811
aI47117
asS'd'
I6923
sa(dp13846
S'a'
I10220
sS's'
I17
sS'm'
(lp13847
I1
aI1
aI5
aI3
aI29
aI5
aI49
aI221
aI247
aI57
aI553
aI1889
aI479
aI15581
aI7035
aI7293
aI53065
asS'd'
I6924
sa(dp13848
S'a'
I10237
sS's'
I17
sS'm'
(lp13849
I1
aI3
aI3
aI3
aI15
aI49
aI91
aI187
aI213
aI981
aI1417
aI211
aI3719
aI13693
aI17671
aI16691
aI57147
asS'd'
I6925
sa(dp13850
S'a'
I10238
sS's'
I17
sS'm'
(lp13851
I1
aI1
aI7
aI9
aI7
aI17
aI109
aI185
aI459
aI769
aI1783
aI899
aI885
aI2291
aI30023
aI26315
aI7337
asS'd'
I6926
sa(dp13852
S'a'
I10241
sS's'
I17
sS'm'
(lp13853
I1
aI1
aI5
aI11
aI11
aI31
aI73
aI191
aI95
aI25
aI1953
aI1387
aI1077
aI7547
aI9661
aI57739
aI76799
asS'd'
I6927
sa(dp13854
S'a'
I10244
sS's'
I17
sS'm'
(lp13855
I1
aI1
aI7
aI13
aI23
aI41
aI69
aI177
aI407
aI699
aI1055
aI3653
aI1239
aI8113
aI12823
aI1803
aI117815
asS'd'
I6928
sa(dp13856
S'a'
I10251
sS's'
I17
sS'm'
(lp13857
I1
aI1
aI1
aI15
aI1
aI55
aI71
aI133
aI401
aI593
aI605
aI2855
aI4569
aI3533
aI14141
aI65457
aI125655
asS'd'
I6929
sa(dp13858
S'a'
I10253
sS's'
I17
sS'm'
(lp13859
I1
aI1
aI7
aI9
aI31
aI55
aI53
aI11
aI65
aI17
aI561
aI925
aI1561
aI8929
aI19859
aI57111
aI12777
asS'd'
I6930
sa(dp13860
S'a'
I10256
sS's'
I17
sS'm'
(lp13861
I1
aI3
aI3
aI11
aI7
aI59
aI125
aI205
aI473
aI655
aI1429
aI337
aI6829
aI7551
aI27873
aI11667
aI39231
asS'd'
I6931
sa(dp13862
S'a'
I10259
sS's'
I17
sS'm'
(lp13863
I1
aI3
aI3
aI9
aI13
aI23
aI25
aI161
aI443
aI545
aI1967
aI1895
aI6929
aI5975
aI17801
aI41769
aI30429
asS'd'
I6932
sa(dp13864
S'a'
I10266
sS's'
I17
sS'm'
(lp13865
I1
aI3
aI7
aI13
aI15
aI1
aI99
aI43
aI45
aI451
aI21
aI639
aI7121
aI4781
aI2813
aI419
aI17761
asS'd'
I6933
sa(dp13866
S'a'
I10284
sS's'
I17
sS'm'
(lp13867
I1
aI1
aI5
aI13
aI11
aI9
aI53
aI83
aI443
aI441
aI1601
aI3177
aI1913
aI12211
aI25835
aI1733
aI4793
asS'd'
I6934
sa(dp13868
S'a'
I10290
sS's'
I17
sS'm'
(lp13869
I1
aI3
aI3
aI1
aI13
aI15
aI11
aI187
aI471
aI699
aI1751
aI3279
aI2305
aI15259
aI31541
aI21357
aI73763
asS'd'
I6935
sa(dp13870
S'a'
I10331
sS's'
I17
sS'm'
(lp13871
I1
aI3
aI5
aI9
aI23
aI11
aI125
aI57
aI261
aI479
aI879
aI719
aI3221
aI2943
aI10593
aI11521
aI83979
asS'd'
I6936
sa(dp13872
S'a'
I10334
sS's'
I17
sS'm'
(lp13873
I1
aI3
aI7
aI13
aI3
aI39
aI119
aI135
aI85
aI417
aI1675
aI971
aI7577
aI12709
aI20407
aI26105
aI97021
asS'd'
I6937
sa(dp13874
S'a'
I10350
sS's'
I17
sS'm'
(lp13875
I1
aI1
aI5
aI11
aI15
aI63
aI83
aI141
aI281
aI663
aI1745
aI2775
aI5605
aI9127
aI553
aI7177
aI115969
asS'd'
I6938
sa(dp13876
S'a'
I10355
sS's'
I17
sS'm'
(lp13877
I1
aI1
aI7
aI1
aI19
aI47
aI7
aI165
aI87
aI95
aI361
aI1879
aI6351
aI2861
aI9103
aI37489
aI24525
asS'd'
I6939
sa(dp13878
S'a'
I10357
sS's'
I17
sS'm'
(lp13879
I1
aI3
aI3
aI11
aI9
aI21
aI51
aI149
aI375
aI967
aI1583
aI1427
aI1223
aI11611
aI7481
aI36619
aI128429
asS'd'
I6940
sa(dp13880
S'a'
I10367
sS's'
I17
sS'm'
(lp13881
I1
aI1
aI5
aI1
aI3
aI31
aI7
aI217
aI453
aI565
aI1517
aI2847
aI6937
aI1197
aI24339
aI44311
aI66843
asS'd'
I6941
sa(dp13882
S'a'
I10368
sS's'
I17
sS'm'
(lp13883
I1
aI1
aI5
aI3
aI3
aI17
aI127
aI59
aI3
aI905
aI531
aI1179
aI3559
aI5175
aI24627
aI60941
aI129457
asS'd'
I6942
sa(dp13884
S'a'
I10377
sS's'
I17
sS'm'
(lp13885
I1
aI1
aI1
aI7
aI15
aI15
aI1
aI31
aI373
aI643
aI279
aI3831
aI4881
aI9763
aI17641
aI43219
aI83109
asS'd'
I6943
sa(dp13886
S'a'
I10388
sS's'
I17
sS'm'
(lp13887
I1
aI3
aI3
aI9
aI5
aI21
aI41
aI71
aI371
aI201
aI573
aI1481
aI3631
aI10783
aI6679
aI1089
aI117347
asS'd'
I6944
sa(dp13888
S'a'
I10407
sS's'
I17
sS'm'
(lp13889
I1
aI1
aI7
aI7
aI5
aI25
aI73
aI63
aI173
aI197
aI147
aI981
aI1491
aI1597
aI11733
aI14285
aI74021
asS'd'
I6945
sa(dp13890
S'a'
I10421
sS's'
I17
sS'm'
(lp13891
I1
aI1
aI5
aI11
aI17
aI15
aI3
aI175
aI391
aI503
aI1745
aI319
aI791
aI5607
aI18173
aI37319
aI92025
asS'd'
I6946
sa(dp13892
S'a'
I10434
sS's'
I17
sS'm'
(lp13893
I1
aI3
aI1
aI1
aI9
aI37
aI43
aI81
aI439
aI951
aI805
aI251
aI4625
aI15617
aI13715
aI62263
aI3827
asS'd'
I6947
sa(dp13894
S'a'
I10439
sS's'
I17
sS'm'
(lp13895
I1
aI3
aI1
aI1
aI25
aI21
aI67
aI191
aI499
aI205
aI1355
aI105
aI1637
aI563
aI22291
aI9045
aI6545
asS'd'
I6948
sa(dp13896
S'a'
I10440
sS's'
I17
sS'm'
(lp13897
I1
aI1
aI5
aI5
aI9
aI3
aI75
aI75
aI287
aI303
aI1767
aI1789
aI3437
aI4637
aI9605
aI2537
aI64935
asS'd'
I6949
sa(dp13898
S'a'
I10443
sS's'
I17
sS'm'
(lp13899
I1
aI1
aI3
aI3
aI1
aI51
aI27
aI155
aI375
aI149
aI885
aI187
aI1551
aI13109
aI27011
aI57301
aI41047
asS'd'
I6950
sa(dp13900
S'a'
I10446
sS's'
I17
sS'm'
(lp13901
I1
aI1
aI7
aI5
aI21
aI23
aI1
aI81
aI163
aI231
aI2039
aI1519
aI1279
aI15379
aI25549
aI6711
aI81499
asS'd'
I6951
sa(dp13902
S'a'
I10457
sS's'
I17
sS'm'
(lp13903
I1
aI1
aI3
aI5
aI3
aI37
aI71
aI243
aI165
aI365
aI379
aI351
aI4649
aI4287
aI13395
aI30329
aI78383
asS'd'
I6952
sa(dp13904
S'a'
I10469
sS's'
I17
sS'm'
(lp13905
I1
aI3
aI1
aI1
aI25
aI63
aI27
aI215
aI223
aI699
aI2029
aI3737
aI5947
aI7287
aI20813
aI4931
aI19345
asS'd'
I6953
sa(dp13906
S'a'
I10476
sS's'
I17
sS'm'
(lp13907
I1
aI1
aI3
aI15
aI21
aI7
aI25
aI187
aI219
aI53
aI1749
aI1797
aI3533
aI14307
aI53
aI11095
aI75469
asS'd'
I6954
sa(dp13908
S'a'
I10479
sS's'
I17
sS'm'
(lp13909
I1
aI1
aI3
aI13
aI27
aI31
aI91
aI121
aI481
aI291
aI915
aI535
aI4291
aI5271
aI12181
aI55921
aI125917
asS'd'
I6955
sa(dp13910
S'a'
I10481
sS's'
I17
sS'm'
(lp13911
I1
aI3
aI1
aI1
aI3
aI29
aI21
aI251
aI361
aI747
aI997
aI2989
aI1809
aI7235
aI17855
aI31027
aI100689
asS'd'
I6956
sa(dp13912
S'a'
I10494
sS's'
I17
sS'm'
(lp13913
I1
aI3
aI7
aI1
aI21
aI13
aI49
aI93
aI183
aI673
aI881
aI1931
aI7009
aI2565
aI26021
aI53815
aI19807
asS'd'
I6957
sa(dp13914
S'a'
I10501
sS's'
I17
sS'm'
(lp13915
I1
aI1
aI7
aI13
aI9
aI23
aI47
aI237
aI487
aI843
aI1357
aI919
aI1753
aI903
aI2911
aI31527
aI73027
asS'd'
I6958
sa(dp13916
S'a'
I10505
sS's'
I17
sS'm'
(lp13917
I1
aI1
aI1
aI1
aI25
aI33
aI97
aI241
aI421
aI375
aI73
aI2541
aI6231
aI14659
aI15335
aI5915
aI110791
asS'd'
I6959
sa(dp13918
S'a'
I10516
sS's'
I17
sS'm'
(lp13919
I1
aI3
aI3
aI7
aI21
aI17
aI97
aI125
aI7
aI271
aI167
aI475
aI4887
aI1847
aI30173
aI25913
aI36659
asS'd'
I6960
sa(dp13920
S'a'
I10532
sS's'
I17
sS'm'
(lp13921
I1
aI1
aI3
aI15
aI15
aI37
aI67
aI5
aI463
aI423
aI823
aI941
aI1551
aI14175
aI15377
aI6017
aI118297
asS'd'
I6961
sa(dp13922
S'a'
I10541
sS's'
I17
sS'm'
(lp13923
I1
aI1
aI1
aI7
aI31
aI51
aI71
aI127
aI73
aI517
aI881
aI3205
aI6219
aI11213
aI14783
aI64275
aI70033
asS'd'
I6962
sa(dp13924
S'a'
I10547
sS's'
I17
sS'm'
(lp13925
I1
aI3
aI1
aI5
aI17
aI17
aI57
aI107
aI359
aI999
aI1415
aI757
aI4743
aI7775
aI14111
aI20075
aI73269
asS'd'
I6963
sa(dp13926
S'a'
I10550
sS's'
I17
sS'm'
(lp13927
I1
aI3
aI5
aI3
aI21
aI57
aI87
aI43
aI307
aI777
aI717
aI3329
aI4159
aI12545
aI31355
aI31329
aI41377
asS'd'
I6964
sa(dp13928
S'a'
I10591
sS's'
I17
sS'm'
(lp13929
I1
aI3
aI7
aI15
aI25
aI43
aI19
aI147
aI487
aI517
aI977
aI3625
aI2311
aI14173
aI15167
aI56563
aI110417
asS'd'
I6965
sa(dp13930
S'a'
I10597
sS's'
I17
sS'm'
(lp13931
I1
aI3
aI3
aI11
aI23
aI1
aI67
aI157
aI461
aI169
aI231
aI1977
aI5657
aI865
aI711
aI24213
aI76895
asS'd'
I6966
sa(dp13932
S'a'
I10602
sS's'
I17
sS'm'
(lp13933
I1
aI1
aI7
aI13
aI5
aI37
aI51
aI165
aI331
aI97
aI431
aI3819
aI1379
aI12083
aI27521
aI19689
aI100119
asS'd'
I6967
sa(dp13934
S'a'
I10610
sS's'
I17
sS'm'
(lp13935
I1
aI1
aI7
aI15
aI29
aI21
aI59
aI193
aI397
aI467
aI951
aI3037
aI2955
aI13235
aI20981
aI63865
aI30069
asS'd'
I6968
sa(dp13936
S'a'
I10619
sS's'
I17
sS'm'
(lp13937
I1
aI3
aI3
aI5
aI7
aI49
aI41
aI143
aI319
aI71
aI353
aI2159
aI3043
aI15317
aI24095
aI12017
aI64393
asS'd'
I6969
sa(dp13938
S'a'
I10631
sS's'
I17
sS'm'
(lp13939
I1
aI1
aI5
aI13
aI25
aI45
aI57
aI153
aI311
aI805
aI953
aI1763
aI5655
aI3961
aI12085
aI58761
aI76533
asS'd'
I6970
sa(dp13940
S'a'
I10646
sS's'
I17
sS'm'
(lp13941
I1
aI1
aI3
aI15
aI29
aI19
aI71
aI107
aI203
aI221
aI1173
aI1597
aI1179
aI9649
aI21659
aI10463
aI8195
asS'd'
I6971
sa(dp13942
S'a'
I10655
sS's'
I17
sS'm'
(lp13943
I1
aI1
aI3
aI9
aI31
aI29
aI53
aI151
aI247
aI577
aI543
aI459
aI8141
aI5613
aI12029
aI24199
aI118603
asS'd'
I6972
sa(dp13944
S'a'
I10665
sS's'
I17
sS'm'
(lp13945
I1
aI3
aI1
aI5
aI1
aI55
aI103
aI23
aI405
aI5
aI181
aI3805
aI1103
aI13389
aI6725
aI48733
aI99639
asS'd'
I6973
sa(dp13946
S'a'
I10673
sS's'
I17
sS'm'
(lp13947
I1
aI1
aI5
aI9
aI1
aI47
aI115
aI231
aI151
aI885
aI427
aI2849
aI361
aI12969
aI705
aI41711
aI53587
asS'd'
I6974
sa(dp13948
S'a'
I10674
sS's'
I17
sS'm'
(lp13949
I1
aI1
aI3
aI11
aI9
aI3
aI11
aI231
aI77
aI775
aI657
aI2721
aI3431
aI11919
aI10425
aI29405
aI91561
asS'd'
I6975
sa(dp13950
S'a'
I10680
sS's'
I17
sS'm'
(lp13951
I1
aI1
aI1
aI5
aI5
aI7
aI79
aI41
aI181
aI333
aI963
aI3117
aI7703
aI2259
aI16671
aI51139
aI27997
asS'd'
I6976
sa(dp13952
S'a'
I10693
sS's'
I17
sS'm'
(lp13953
I1
aI3
aI7
aI7
aI13
aI55
aI59
aI157
aI377
aI711
aI1475
aI1509
aI1375
aI6825
aI13729
aI28613
aI109199
asS'd'
I6977
sa(dp13954
S'a'
I10700
sS's'
I17
sS'm'
(lp13955
I1
aI3
aI3
aI3
aI13
aI11
aI51
aI1
aI67
aI609
aI467
aI2161
aI7693
aI9019
aI1847
aI27969
aI74863
asS'd'
I6978
sa(dp13956
S'a'
I10721
sS's'
I17
sS'm'
(lp13957
I1
aI1
aI3
aI3
aI11
aI33
aI87
aI217
aI239
aI505
aI1451
aI2801
aI1417
aI695
aI29883
aI15877
aI99969
asS'd'
I6979
sa(dp13958
S'a'
I10727
sS's'
I17
sS'm'
(lp13959
I1
aI3
aI3
aI5
aI3
aI61
aI9
aI171
aI57
aI547
aI2003
aI2335
aI2259
aI3205
aI5639
aI21721
aI25893
asS'd'
I6980
sa(dp13960
S'a'
I10746
sS's'
I17
sS'm'
(lp13961
I1
aI3
aI1
aI3
aI19
aI15
aI83
aI69
aI47
aI897
aI627
aI2839
aI7123
aI8567
aI14707
aI13159
aI125139
asS'd'
I6981
sa(dp13962
S'a'
I10748
sS's'
I17
sS'm'
(lp13963
I1
aI3
aI7
aI11
aI1
aI59
aI53
aI33
aI135
aI1009
aI1829
aI3011
aI1245
aI421
aI28909
aI45517
aI55071
asS'd'
I6982
sa(dp13964
S'a'
I10757
sS's'
I17
sS'm'
(lp13965
I1
aI1
aI5
aI9
aI3
aI27
aI