import tensorflow as tf
import numpy as np

class VATTransformer(tf.keras.Model):
    def __init__(self, num_layers, d_model, num_heads, dff, vocab_size, pe_size, std_dev, rate):
        """
        Variational Auto-Transformer model
        :param num_layers: Number of encoder/decoder layers
        :param d_model: Model dimensionality
        :param num_heads: Number of attention heads
        :param dff: Number of nodes in the feed forward layers
        :param vocab_size: Number of tokens in the vocabulary
        :param pe_size: Positions to precompute for positional embedding
        :param std_dev: Scaling factor for standard deviation in (0, 1]
        :param rate: Dropout rate
        """
        super(VATTransformer, self).__init__()

        self.d_model = d_model
        self.std_dev = std_dev

        self.encoder = Encoder(num_layers, d_model, num_heads, dff, vocab_size, pe_size, rate)

        self.decoder = Decoder(num_layers, d_model, num_heads, dff,
                               vocab_size, pe_size, rate)

        self.final_layer = tf.keras.layers.Dense(vocab_size)

        self.hidden2mean = tf.keras.layers.Dense(d_model)
        self.hidden2logv = tf.keras.layers.Dense(d_model)

    def call(self, inp, tar, training, enc_padding_mask, look_ahead_mask, dec_padding_mask):
        enc_output = self.encoder(inp, training, enc_padding_mask)

        mean = self.hidden2mean(enc_output)
        logv = self.hidden2logv(enc_output)  # estimates log(sigma^2)

        epsilon = tf.random.normal(shape=tf.shape(mean), mean=0.0, stddev=self.std_dev)
        if training:
            std = tf.math.exp(0.5 * logv)  # compute back to sigma
            z = std * epsilon + mean
        else:
            z = mean

        dec_output = self.decoder(tar, z, training, look_ahead_mask, dec_padding_mask)

        final_output = self.final_layer(dec_output)

        return final_output, mean, logv, z


class Encoder(tf.keras.layers.Layer):
    """
    The full encoder with multiple individual encoder layers
    According to Vaswani et al.
    Code originally from https://www.tensorflow.org/tutorials/text/transformer
    """
    def __init__(self, num_layers, d_model, num_heads, dff, vocab_size,
                 maximum_position_encoding, rate=0.1):
        super(Encoder, self).__init__()
        self.d_model = d_model
        self.num_layers = num_layers

        self.embedding = tf.keras.layers.Embedding(vocab_size, d_model)
        self.pos_encoding = positional_encoding(maximum_position_encoding,
                                                self.d_model)

        self.enc_layers = [EncoderLayer(d_model, num_heads, dff, rate)
                           for _ in range(num_layers)]

        self.dropout = tf.keras.layers.Dropout(rate)

    def call(self, x, training, mask):
        seq_len = tf.shape(x)[1]

        # adding embedding and positional encoding
        x = self.embedding(x)  # (batch_size, seq_len, d_model)
        x *= tf.math.sqrt(tf.cast(self.d_model, tf.float32))
        x += self.pos_encoding[:, :seq_len, :]

        x = self.dropout(x, training=training)

        for i in range(self.num_layers):
            x = self.enc_layers[i](x, training, mask)

        return x  # (batch_size, seq_len, d_model)


class Decoder(tf.keras.layers.Layer):
    """
    The full decoder with multiple individual decoder layers
    According to Vaswani et al.
    Code originally from https://www.tensorflow.org/tutorials/text/transformer
    """
    def __init__(self, num_layers, d_model, num_heads, dff, vocab_size,
                 maximum_position_encoding, rate=0.1):
        super(Decoder, self).__init__()

        self.d_model = d_model
        self.num_layers = num_layers

        self.embedding = tf.keras.layers.Embedding(vocab_size, d_model)
        self.pos_encoding = positional_encoding(maximum_position_encoding, d_model)

        self.dec_layers = [DecoderLayer(d_model, num_heads, dff, rate)
                           for _ in range(num_layers)]
        self.dropout = tf.keras.layers.Dropout(rate)

    def call(self, x, enc_output, training,
             look_ahead_mask, padding_mask):
        seq_len = tf.shape(x)[1]

        x = self.embedding(x)  # (batch_size, target_seq_len, d_model)
        x *= tf.math.sqrt(tf.cast(self.d_model, tf.float32))
        x += self.pos_encoding[:, :seq_len, :]

        x = self.dropout(x, training=training)

        for i in range(self.num_layers):
            x = self.dec_layers[i](x, enc_output, training, look_ahead_mask, padding_mask)

        return x


class EncoderLayer(tf.keras.layers.Layer):
    """
    One encoder layer with self-attention
    According to Vaswani et al.
    Code originally from https://www.tensorflow.org/tutorials/text/transformer
    """
    def __init__(self, d_model, num_heads, dff, rate=0.1):
        super(EncoderLayer, self).__init__()

        self.mha = MultiHeadAttention(d_model, num_heads)
        self.ffn = point_wise_feed_forward_network(d_model, dff)

        self.layernorm1 = tf.keras.layers.LayerNormalization(epsilon=1e-6)
        self.layernorm2 = tf.keras.layers.LayerNormalization(epsilon=1e-6)

        self.dropout1 = tf.keras.layers.Dropout(rate)
        self.dropout2 = tf.keras.layers.Dropout(rate)

    def call(self, x, training, mask):
        attn_output = self.mha(x, x, x, mask)  # (batch_size, seq_len, d_model)
        attn_output = self.dropout1(attn_output, training=training)
        out1 = self.layernorm1(x + attn_output)  # (batch_size, seq_len, d_model)

        ffn_output = self.ffn(out1)  # (batch_size, seq_len, d_model)
        ffn_output = self.dropout2(ffn_output, training=training)
        out2 = self.layernorm2(out1 + ffn_output)  # (batch_size, seq_len, d_model)

        return out2


class DecoderLayer(tf.keras.layers.Layer):
    """
    One decoder layer with self-attention
    According to Vaswani et al.
    Code originally from https://www.tensorflow.org/tutorials/text/transformer
    """
    def __init__(self, d_model, num_heads, dff, rate=0.1):
        super(DecoderLayer, self).__init__()

        self.mha1 = MultiHeadAttention(d_model, num_heads)
        self.mha2 = MultiHeadAttention(d_model, num_heads)

        self.ffn = point_wise_feed_forward_network(d_model, dff)

        self.layernorm1 = tf.keras.layers.LayerNormalization(epsilon=1e-6)
        self.layernorm2 = tf.keras.layers.LayerNormalization(epsilon=1e-6)
        self.layernorm3 = tf.keras.layers.LayerNormalization(epsilon=1e-6)

        self.dropout1 = tf.keras.layers.Dropout(rate)
        self.dropout2 = tf.keras.layers.Dropout(rate)
        self.dropout3 = tf.keras.layers.Dropout(rate)

    def call(self, x, enc_output, training, look_ahead_mask, padding_mask):
        attn1 = self.mha1(x, x, x, look_ahead_mask)  # (batch_size, target_seq_len, d_model)
        attn1 = self.dropout1(attn1, training=training)
        out1 = self.layernorm1(attn1 + x)

        attn2 = self.mha2(
            enc_output, enc_output, out1, padding_mask)  # (batch_size, target_seq_len, d_model)
        attn2 = self.dropout2(attn2, training=training)
        out2 = self.layernorm2(attn2 + out1)  # (batch_size, target_seq_len, d_model)

        ffn_output = self.ffn(out2)  # (batch_size, target_seq_len, d_model)
        ffn_output = self.dropout3(ffn_output, training=training)
        out3 = self.layernorm3(ffn_output + out2)  # (batch_size, target_seq_len, d_model)

        return out3

# Code originally from https://www.tensorflow.org/tutorials/text/transformer
def point_wise_feed_forward_network(d_model, dff):
    return tf.keras.Sequential([
        tf.keras.layers.Dense(dff, activation='relu'),  # (batch_size, seq_len, dff)
        tf.keras.layers.Dense(d_model)  # (batch_size, seq_len, d_model)
    ])

# Attention
class MultiHeadAttention(tf.keras.layers.Layer):
    """
    According to Vaswani et al.
    Code originally from https://www.tensorflow.org/tutorials/text/transformer
    """
    def __init__(self, d_model, num_heads):
        super(MultiHeadAttention, self).__init__()
        self.num_heads = num_heads
        self.d_model = d_model

        assert d_model % self.num_heads == 0

        self.depth = d_model // self.num_heads

        self.wq = tf.keras.layers.Dense(d_model)
        self.wk = tf.keras.layers.Dense(d_model)
        self.wv = tf.keras.layers.Dense(d_model)

        self.dense = tf.keras.layers.Dense(d_model)

    def split_heads(self, x, batch_size):
        """Split the last dimension into (num_heads, depth).
        Transpose the result such that the shape is (batch_size, num_heads, seq_len, depth)
        """
        x = tf.reshape(x, (batch_size, -1, self.num_heads, self.depth))
        return tf.transpose(x, perm=[0, 2, 1, 3])

    def call(self, v, k, q, mask):
        batch_size = tf.shape(q)[0]

        q = self.wq(q)  # (batch_size, seq_len, d_model)
        k = self.wk(k)  # (batch_size, seq_len, d_model)
        v = self.wv(v)  # (batch_size, seq_len, d_model)

        q = self.split_heads(q, batch_size)  # (batch_size, num_heads, seq_len_q, depth)
        k = self.split_heads(k, batch_size)  # (batch_size, num_heads, seq_len_k, depth)
        v = self.split_heads(v, batch_size)  # (batch_size, num_heads, seq_len_v, depth)

        # scaled_attention.shape == (batch_size, num_heads, seq_len_q, depth)
        scaled_attention = scaled_dot_product_attention(q, k, v, mask)

        # (batch_size, seq_len_q, num_heads, depth)
        scaled_attention = tf.transpose(scaled_attention, perm=[0, 2, 1, 3])
        # (batch_size, seq_len_q, d_model)
        concat_attention = tf.reshape(scaled_attention, (batch_size, -1, self.d_model))
        # (batch_size, seq_len_q, d_model)
        output = self.dense(concat_attention)

        return output

def scaled_dot_product_attention(q, k, v, mask):
        """Calculate the attention weights.
        q: query shape == (..., seq_len_q, depth)
        k: key shape == (..., seq_len_k, depth)
        v: value shape == (..., seq_len_v, depth_v)
        mask: Float tensor with shape broadcastable to (..., seq_len_q, seq_len_k). Defaults to None.

        According to Vaswani et al.
        Code originally from https://www.tensorflow.org/tutorials/text/transformer
        """

        matmul_qk = tf.matmul(q, k, transpose_b=True)  # (..., seq_len_q, seq_len_k)

        # scale matmul_qk
        dk = tf.cast(tf.shape(k)[-1], tf.float32)
        scaled_attention_logits = matmul_qk / tf.math.sqrt(dk)

        # add the mask to the scaled tensor
        if mask is not None:
            scaled_attention_logits += (mask * -1e9)

        # softmax is normalized on the last axis (seq_len_k) so that the scores add up to 1
        attention_weights = tf.nn.softmax(scaled_attention_logits, axis=-1)  # (..., seq_len_q, seq_len_k)

        output = tf.matmul(attention_weights, v)  # (..., seq_len_q, depth_v)

        return output


# Positional encoding
# Code originally from https://www.tensorflow.org/tutorials/text/transformer
def get_angles(pos, i, d_model):
    angle_rates = 1 / np.power(10000, (2 * (i // 2)) / np.float32(d_model))
    return pos * angle_rates

# Code originally from https://www.tensorflow.org/tutorials/text/transformer
def positional_encoding(position, d_model):
    angle_rads = get_angles(np.arange(position)[:, np.newaxis],
                            np.arange(d_model)[np.newaxis, :],
                            d_model)

    # apply sin to even indices in the array; 2i
    angle_rads[:, 0::2] = np.sin(angle_rads[:, 0::2])

    # apply cos to odd indices in the array; 2i+1
    angle_rads[:, 1::2] = np.cos(angle_rads[:, 1::2])

    pos_encoding = angle_rads[np.newaxis, ...]

    return tf.cast(pos_encoding, dtype=tf.float32)



# Masking
# Code originally from https://www.tensorflow.org/tutorials/text/transformer
def create_padding_mask(seq):
    seq = tf.cast(tf.math.equal(seq, 0), tf.float32)
    # add extra dimensions to add the padding to the attention logits.
    return seq[:, tf.newaxis, tf.newaxis, :]  # (batch_size, 1, 1, seq_len)

def create_combined_mask(tar):
    tar_padding_mask = create_padding_mask(tar)
    size = tf.shape(tar)[1]
    look_ahead_mask = 1 - tf.linalg.band_part(tf.ones((size, size)), -1, 0)
    combined_mask = tf.maximum(tar_padding_mask, look_ahead_mask)
    return combined_mask

def create_masks(inp, tar):
    # Encoder padding mask
    enc_padding_mask = create_padding_mask(inp)
    # Used in the 2nd attention block in the decoder.
    # This padding mask is used to mask the encoder outputs.
    dec_padding_mask = create_padding_mask(inp)
    # Used in the 1st attention block in the decoder.
    # It is used to pad and mask future tokens in the input received by the decoder.
    combined_mask = create_combined_mask(tar)

    return enc_padding_mask, combined_mask, dec_padding_mask


# Training: Loss function, learning rate
loss_object = tf.keras.losses.SparseCategoricalCrossentropy(from_logits=True, reduction='none')

def kl_anneal_function(step, k=0.00025, x0=100000):
    """
    step: current training step
    k: keep rate in [0, 1]
    x0: offset (in number of steps)

    According to Bowman et al.
    """
    return 1.0 / (1 + tf.exp(-k * (step - x0)))


def loss_function_vae(real, pred, logv, mean, step, std_scale=0.4, k=0.00025, x0=100000):
    # reconstruction term
    mask = tf.math.logical_not(tf.math.equal(real, 0))
    reconstruction_loss = loss_object(real, pred)

    mask = tf.cast(mask, dtype=reconstruction_loss.dtype)
    reconstruction_loss *= mask
    reconstruction_loss = tf.reduce_mean(reconstruction_loss)  # mean of losses

    # regularisation term
    kl_loss = -1 * tf.reduce_sum(0.5 * logv - (tf.math.pow(mean, 2) + tf.math.exp(logv)) / (2 * tf.math.pow(std_scale, 2)) - tf.math.log(std_scale) + 0.5, axis=2)
    kl_loss = tf.reduce_mean(kl_loss)  # mean of losses
    kl_weight = kl_anneal_function(step, k, x0)

    return reconstruction_loss, kl_loss, kl_weight


class CustomSchedule(tf.keras.optimizers.schedules.LearningRateSchedule):
    """
    Noam optimizer
    According to Vaswani et al.
    Code originally from https://www.tensorflow.org/tutorials/text/transformer
    """
    def __init__(self, d_model, warmup_steps=100000, coef=1.0):
        super(CustomSchedule, self).__init__()

        self.d_model = d_model
        self.d_model = tf.cast(self.d_model, tf.float32)

        self.warmup_steps = warmup_steps
        self.coef = coef
        self.learning_rate = 0

    def __call__(self, step):
        arg1 = tf.math.rsqrt(step+1)
        arg2 = (step+1) * (self.warmup_steps ** -1.5)

        self.learning_rate = self.coef * tf.math.rsqrt(self.d_model) * tf.math.minimum(arg1, arg2)
        return self.learning_rate