import argparse
import os
import random
import shutil
import time
import warnings

import torch
import torch.nn as nn
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.optim
import torch.multiprocessing as mp
import torch.utils.data
import torch.utils.data.distributed
import torchvision.transforms as transforms
import torchvision.datasets as datasets
import torchvision.models as models
import logging
import pickle 
import pathlib
import numpy as np
from sklearn.metrics import classification_report, confusion_matrix, accuracy_score, roc_auc_score
from collections import Counter, defaultdict

model_names = sorted(name for name in models.__dict__
    if name.islower() and not name.startswith("__")
    and callable(models.__dict__[name]))

def get_args_parser():
    parser = argparse.ArgumentParser(description='PyTorch ImageNet Training')
    parser.add_argument('--data', metavar='DIR', default='/data/GQA/metaDataset', 
                        help='path to dataset')
    parser.add_argument('-a', '--arch', metavar='ARCH', default='resnet18',
                        choices=model_names,
                        help='model architecture: ' +
                            ' | '.join(model_names) +
                            ' (default: resnet18)')
    parser.add_argument('--num-classes', default=2, type=int, metavar='N',
                        help='number of meta tasks used (default: 2, binary classification)')
    parser.add_argument('-j', '--workers', default=4, type=int, metavar='N',
                        help='number of data loading workers (default: 4)')
    parser.add_argument('--epochs', default=90, type=int, metavar='N',
                        help='number of total epochs to run')
    parser.add_argument('--start-epoch', default=0, type=int, metavar='N',
                        help='manual epoch number (useful on restarts)')
    parser.add_argument('-b', '--batch-size', default=256, type=int,
                        metavar='N',
                        help='mini-batch size (default: 256), this is the total '
                            'batch size of all GPUs on the current node when '
                            'using Data Parallel or Distributed Data Parallel')
    parser.add_argument('--lr', '--learning-rate', default=0.1, type=float,
                        metavar='LR', help='initial learning rate', dest='lr')
    parser.add_argument('--momentum', default=0.9, type=float, metavar='M',
                        help='momentum')
    parser.add_argument('--wd', '--weight-decay', default=0.0, type=float,
                        metavar='W', help='weight decay (default: 1e-4)',
                        dest='weight_decay')
    parser.add_argument('-p', '--print-freq', default=10, type=int,
                        metavar='N', help='print frequency (default: 10)')
    parser.add_argument('--resume', default='', type=str, metavar='PATH',
                        help='path to latest checkpoint (default: none)')
    parser.add_argument('--custom-pretrain', default='model_best.pth.tar', type=str, metavar='PATH',
                        help='path to latest checkpoint (default: none)')
    parser.add_argument('-e', '--evaluate', dest='evaluate', action='store_true',
                        help='evaluate model on validation set')
    parser.add_argument('--pretrained', dest='pretrained', action='store_true',
                        help='use pre-trained model')
    parser.add_argument('--world-size', default=-1, type=int,
                        help='number of nodes for distributed training')
    parser.add_argument('--rank', default=-1, type=int,
                        help='node rank for distributed training')
    parser.add_argument('--dist-url', default='tcp://224.66.41.62:23456', type=str,
                        help='url used to set up distributed training')
    parser.add_argument('--dist-backend', default='nccl', type=str,
                        help='distributed backend')
    parser.add_argument('--seed', default=None, type=int,
                        help='seed for initializing training. ')
    parser.add_argument('--gpu', default=None, type=int,
                        help='GPU id to use.')
    parser.add_argument('--multiprocessing-distributed', action='store_true',
                        help='Use multi-processing distributed training to launch '
                            'N processes per node, which has N GPUs. This is the '
                            'fastest way to use PyTorch for either single node or '
                            'multi node data parallel training')
    parser.add_argument('--lr-drop', default=30, type=int)

    ##################################
    # Logging 
    ##################################
    parser.add_argument('--log-name', default='ouptut.log', type=str, metavar='PATH',
                        help='path to the log file (default: output.log)')
    parser.add_argument('--output_dir', default='outputTmp',
                        help='path where to save, empty for no saving')


    return parser


best_acc1 = 0


def main(fn_call_training):
    parser = get_args_parser()
    args = parser.parse_args()

    pathlib.Path(args.output_dir).mkdir(parents=True, exist_ok=True)


    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')

    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely '
                      'disable data parallelism.')

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    ngpus_per_node = torch.cuda.device_count()
    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args, fn_call_training))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args, fn_call_training)


"""
fn_call_training is the training logits' wrapper
"""
def main_worker(gpu, ngpus_per_node, args, fn_call_training):
    global best_acc1
    args.gpu = gpu

    ##################################
    # Logging setting
    ##################################
    
    logging.basicConfig(
        filename=os.path.join( args.output_dir, args.log_name),
        filemode='w',
        format='%(asctime)s: %(levelname)s: [%(filename)s:%(lineno)d]: %(message)s',
        level=logging.INFO)
    warnings.filterwarnings("ignore")

    print(str(args))
    logging.info( str(args) )

    if args.gpu is not None:
        print("Use GPU: {} for training".format(args.gpu))

    if args.distributed:
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(backend=args.dist_backend, init_method=args.dist_url,
                                world_size=args.world_size, rank=args.rank)
    # create model
    if args.pretrained:
        print("=> using pre-trained model '{}'".format(args.arch))
        model = models.__dict__[args.arch](pretrained=True)
    else:
        print("=> creating model '{}'".format(args.arch))
        model = models.__dict__[args.arch]()

    ##################################
    # Hack the resumed pre-train here
    # custom-pretrain, without cat and dogs. 
    ##################################
    # optionally resume from a checkpoint
    if args.custom_pretrain:
        if os.path.isfile(args.custom_pretrain):
            print("=> loading custom pretrainend checkpoint '{}'".format(args.custom_pretrain))
            checkpoint = torch.load(args.custom_pretrain)
            # original saved file with DataParallel
            state_dict = checkpoint['state_dict']
            # create new OrderedDict that does not contain `module.`
            from collections import OrderedDict
            new_state_dict = OrderedDict()
            for k, v in state_dict.items():
                name = k[7:] # remove `module.`
                new_state_dict[name] = v
            # load params
            model.load_state_dict(new_state_dict)
        else:
            print("=> no checkpoint found at '{}'".format(args.custom_pretrain))

    ##################################
    # Hack the final layer here 
    ##################################
    num_classes = args.num_classes
    model.fc = nn.Linear(512, num_classes)

    if not torch.cuda.is_available():
        print('using CPU, this will be slow')
    elif args.distributed:
        # For multiprocessing distributed, DistributedDataParallel constructor
        # should always set the single device scope, otherwise,
        # DistributedDataParallel will use all available devices.
        if args.gpu is not None:
            torch.cuda.set_device(args.gpu)
            model.cuda(args.gpu)
            # When using a single GPU per process and per
            # DistributedDataParallel, we need to divide the batch size
            # ourselves based on the total number of GPUs we have
            args.batch_size = int(args.batch_size / ngpus_per_node)
            args.workers = int((args.workers + ngpus_per_node - 1) / ngpus_per_node)
            model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu])
        else:
            model.cuda()
            # DistributedDataParallel will divide and allocate batch_size to all
            # available GPUs if device_ids are not set
            model = torch.nn.parallel.DistributedDataParallel(model)
    elif args.gpu is not None:
        torch.cuda.set_device(args.gpu)
        model = model.cuda(args.gpu)
    else:
        # DataParallel will divide and allocate batch_size to all available GPUs
        if args.arch.startswith('alexnet') or args.arch.startswith('vgg'):
            model.features = torch.nn.DataParallel(model.features)
            model.cuda()
        else:
            model = torch.nn.DataParallel(model).cuda()


    ##################################
    # Handle Imbalanced Classes: 
    ##################################

    # weights = torch.tensor([1./1159, 1./1642], dtype=torch.float32)
    # weights = weights / weights.sum()
    # weights = weights.cuda(args.gpu)
    # criterion = nn.CrossEntropyLoss(weight=weights).cuda(args.gpu)
            
    # define loss function (criterion) and optimizer
    criterion = nn.CrossEntropyLoss().cuda(args.gpu)

    optimizer = torch.optim.SGD(model.parameters(), args.lr,
                                momentum=args.momentum,
                                weight_decay=args.weight_decay)

    # optionally resume from a checkpoint
    if args.resume:
        if os.path.isfile(args.resume):
            print("=> loading checkpoint '{}'".format(args.resume))
            if args.gpu is None:
                checkpoint = torch.load(args.resume)
            else:
                # Map model to be loaded to specified single gpu.
                loc = 'cuda:{}'.format(args.gpu)
                checkpoint = torch.load(args.resume, map_location=loc)
            args.start_epoch = checkpoint['epoch']
            best_acc1 = checkpoint['best_acc1']
            if args.gpu is not None:
                # best_acc1 may be from a checkpoint from a different GPU
                best_acc1 = best_acc1.to(args.gpu)
            model.load_state_dict(checkpoint['state_dict'])
            optimizer.load_state_dict(checkpoint['optimizer'])
            print("=> loaded checkpoint '{}' (epoch {})"
                  .format(args.resume, checkpoint['epoch']))
        else:
            print("=> no checkpoint found at '{}'".format(args.resume))

    cudnn.benchmark = True

    ##################################
    # Above: prepare model, optimizer, loss function
    # Below: prepare data and data loader, training logits
    ##################################
    fn_call_training(model, criterion, optimizer, args)    
    return


##################################
# wrapper function for val data loader 
##################################
def get_val_loader(dataset_dir, args):
    normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                     std=[0.229, 0.224, 0.225])
    ret_dataset = datasets.ImageFolder(
            os.path.join(args.data, dataset_dir),
            transforms.Compose([
            transforms.Resize(256),
            transforms.CenterCrop(224),
            transforms.ToTensor(),
            normalize,
        ]))
    ret_loader = torch.utils.data.DataLoader(
        ret_dataset,
        batch_size=args.batch_size * 2, shuffle=False,
        num_workers=args.workers, pin_memory=True)
    return ret_loader, ret_dataset
    
def load_data_and_train(model, criterion, optimizer, args):

    global best_acc1

    # Data loading code
    traindir = os.path.join(args.data, 'train')
    normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                     std=[0.229, 0.224, 0.225])

    train_dataset = datasets.ImageFolder(
        traindir,
        transforms.Compose([
            transforms.RandomResizedCrop(224),
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor(),
            normalize,
        ]))

    if args.distributed:
        train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
    else:
        train_sampler = None

    train_loader = torch.utils.data.DataLoader(
        train_dataset, batch_size=args.batch_size, shuffle=(train_sampler is None),
        num_workers=args.workers, pin_memory=True, sampler=train_sampler)


    ##################################
    # train in-order set. 
    ##################################
    train_unshuffled_loader, train_unshuffled_dataset = get_val_loader(dataset_dir='train', args=args)
    
    ##################################
    # EMR, out of domain val acc, containing other. 
    ##################################
    FLAG_VAL_out_of_domain = True 
    # FLAG_VAL_out_of_domain = False
    if FLAG_VAL_out_of_domain:
        val_out_of_domain_loader, val_out_of_domain_dataset = get_val_loader(dataset_dir='val_out_of_domain', args=args)


    ##################################
    # EMR, val_sg folder, random split test set 
    ##################################
    val_valsg_loader, _ = get_val_loader(dataset_dir='val_valsg', args=args)

    ##################################
    # EMR, out of domain val acc, containing other. 
    ##################################
    val_in_domain_loader, _ = get_val_loader(dataset_dir='val_in_domain', args=args)


    if args.evaluate:
        print('train Dataset Cartography')
        logging.info('train Dataset Cartograph')
        acc1, _ = validate(train_unshuffled_loader, model, criterion, args, dumpResult=True)
        # Report every-group acc, worst-set acc 
        report_every_set_acc(train_unshuffled_dataset, args, split='train')
        

        if FLAG_VAL_out_of_domain:
            print('out-of-domain val')
            logging.info('out-of-domain val')
            acc1, _ = validate(val_out_of_domain_loader, model, criterion, args, dumpResult=True)
            # Report every-group acc, worst-set acc 
            report_every_set_acc(val_out_of_domain_dataset, args)

        print('random split val')
        logging.info('random split val')
        _ = validate(val_valsg_loader, model, criterion, args, dumpResult=False)
        print('in-domain val')
        logging.info('in-domain val')
        _ = validate(val_in_domain_loader, model, criterion, args, dumpResult=False)

        return

    for epoch in range(args.start_epoch, args.epochs):
        if args.distributed:
            train_sampler.set_epoch(epoch)
        adjust_learning_rate(optimizer, epoch, args)

        # train for one epoch
        train(train_loader, model, criterion, optimizer, epoch, args)

        print('train Dataset Cartography')
        logging.info('train Dataset Cartograph')
        acc1, _ = validate(train_unshuffled_loader, model, criterion, args, dumpResult=True)
        # Report every-group acc, worst-set acc 
        report_every_set_acc(train_unshuffled_dataset, args, split='train')

        if FLAG_VAL_out_of_domain:
            # evaluate on validation set
            print('out-of-domain val')
            logging.info('out-of-domain val')
            acc1, _ = validate(val_out_of_domain_loader, model, criterion, args, dumpResult=True)
            # Report every-group acc, worst-set acc 
            report_every_set_acc(val_out_of_domain_dataset, args)

        print('random split val')
        logging.info('random split val')
        _ = validate(val_valsg_loader, model, criterion, args, dumpResult=False)
        print('in-domain val')
        logging.info('in-domain  val')
        _ = validate(val_in_domain_loader, model, criterion, args, dumpResult=False)

        # remember best acc@1 and save checkpoint
        is_best = acc1 > best_acc1
        best_acc1 = max(acc1, best_acc1)

        if not args.multiprocessing_distributed or (args.multiprocessing_distributed
                and args.rank % ngpus_per_node == 0):
            save_checkpoint({
                'epoch': epoch + 1,
                'arch': args.arch,
                'state_dict': model.state_dict(),
                'best_acc1': best_acc1,
                'optimizer' : optimizer.state_dict(),
            }, is_best, args=args)


def train(train_loader, model, criterion, optimizer, epoch, args):
    batch_time = AverageMeter('Time', ':6.3f')
    data_time = AverageMeter('Data', ':6.3f')
    losses = AverageMeter('Loss', ':.4e')
    top1 = AverageMeter('Acc@1', ':6.2f')
    top5 = AverageMeter('Acc@5', ':6.2f')
    progress = ProgressMeter(
        len(train_loader),
        [batch_time, data_time, losses, top1, top5],
        prefix="Epoch: [{}]".format(epoch))

    # switch to train mode
    model.train()

    end = time.time()
    for i, (images, target) in enumerate(train_loader):
        # measure data loading time
        data_time.update(time.time() - end)

        if args.gpu is not None:
            images = images.cuda(args.gpu, non_blocking=True)
        if torch.cuda.is_available():
            target = target.cuda(args.gpu, non_blocking=True)

        # compute output
        output = model(images)
        loss = criterion(output, target)

        # measure accuracy and record loss
        # acc1, acc5 = accuracy(output, target, topk=(1, 5))
        acc1 = accuracy(output, target, topk=(1,))[0]
        losses.update(loss.item(), images.size(0))
        top1.update(acc1[0], images.size(0))
        # top5.update(acc5[0], images.size(0))

        # compute gradient and do SGD step
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if i % args.print_freq == 0:
            progress.display(i)

        # break # inspect every batch's improvement 


global_train_dataset_cartography_list = []
def cal_dataset_cargography(my_dataset, imageID_to_group):
    ##################################
    # First calculate for each datapoint
    ##################################
    cargography_confidence_array = [] # for each data point 
    cargography_variability_array = [] # for each data point 
    group_to_cargography_confidence = defaultdict(list)
    group_to_cargography_variability = defaultdict(list)

    for idx, sample in enumerate(my_dataset.samples):
        image_path, target = sample
        imageID = image_path.split('/')[-1].split('.')[0] # image_path = IMAGE_DATA_FOLDER + imageID + '.jpg'
        tmp_score_array = []
        for epoch_idx in range(len(global_train_dataset_cartography_list)):
            assert target==0 or target==1
            if target==1:
                tmp_score_array.append(global_train_dataset_cartography_list[epoch_idx][idx])
            else:
                tmp_score_array.append(1 - global_train_dataset_cartography_list[epoch_idx][idx])
        
        cargography_confidence_array.append( np.mean(tmp_score_array) )
        cargography_variability_array.append( np.std(tmp_score_array) )

        for groups_local in imageID_to_group[imageID]:
            group_to_cargography_confidence[groups_local].append( np.mean(tmp_score_array) )
            group_to_cargography_variability[groups_local].append( np.std(tmp_score_array) )

    ##################################
    # And then aggregate across groups 
    # TODO: add a figure here too 
    # https://github.com/allenai/cartography/blob/c7865383e421a91611c2f4e79d1ffbfb7850f4f4/cartography/selection/train_dy_filtering.py
    ##################################

    return group_to_cargography_confidence, group_to_cargography_variability


def report_every_set_acc(my_dataset, args, split='val'):

    ##################################
    # 1. Load pkl that stores the prediction scores 
    # And the pkl that stores the meta-dataset-structure
    ##################################
    with open(os.path.join( args.output_dir, 'model_validate_dump.pkl'), "rb") as pkl_file:
        load_data = pickle.load( pkl_file )
        target_all = load_data['target_all']
        pred_score_all = load_data['pred_score_all']

    ##################################
    # Load Meta-Data
    ##################################
    # meta_data_path = os.path.join( args.data, 'imageID_to_group.pkl')
    meta_data_path = os.path.join( args.data, 'imageID_to_node.pkl')
    with open(meta_data_path, "rb") as pkl_file:
        imageID_to_group = pickle.load( pkl_file ) # defaultdict(set)

    ##################################
    # Keep all for train - Dataset Cartography
    # TODO: add a post-hoc analysis here. print at every train epoch. - for all subsets 
    ##################################
    if split == 'train':
        global_train_dataset_cartography_list.append(pred_score_all) # save for all epoches 
        with open(os.path.join( args.output_dir, 'model_train_cartography_dump.pkl'), "wb") as pkl_file:
            pickle.dump( {
                "global_train_dataset_cartography_list": global_train_dataset_cartography_list, 
                "imageID_to_train_group": imageID_to_group, 
                "my_dataset_samples": my_dataset.samples
                }, 
                pkl_file 
            )
        group_to_cargography_confidence, group_to_cargography_variability = cal_dataset_cargography(my_dataset=my_dataset, imageID_to_group=imageID_to_group)


    ##################################
    # 2. Sanity check datapoint order with ground truth labels.     
    # my_dataset.targets: list 
    # my_dataset.samples: list to 2-tuples
    # target_all: np.array
    # pred_score_all: np.array
    ##################################
    assert len(pred_score_all) == len(my_dataset.samples), 'len(pred_score_all) {} len(my_dataset.samples) {}'.format(len(pred_score_all), len(my_dataset.samples))
    # print('my_dataset.samples', my_dataset.samples)

    ##################################
    # 3. iterate through all data points, and collect prediction scores, and labels. 
    ##################################
    group_to_preds = defaultdict(lambda: defaultdict(list))
    for idx, sample in enumerate(my_dataset.samples):
        image_path, target = sample
        imageID = image_path.split('/')[-1].split('.')[0] # image_path = IMAGE_DATA_FOLDER + imageID + '.jpg'
        assert target_all[idx] == target
        assert len(imageID_to_group[imageID])>0, str(image_path) + ' ' + str(target)
        for groups_local in imageID_to_group[imageID]:
            group_to_preds[groups_local]['target'].append(target) 
            group_to_preds[groups_local]['pred_score'].append(pred_score_all[idx]) 

    ##################################
    # 4. reduce collected data by acc, or auc. 
    # And report each group in the sorted order of acc. 
    ##################################
    # print('group_to_preds.keys()', group_to_preds.keys())
    group_accs = list()
    for groups_local in sorted(group_to_preds.keys()):
        groups_local_target = np.array(group_to_preds[groups_local]['target'])
        groups_local_pred_scores = np.array(group_to_preds[groups_local]['pred_score'])
        groups_local_pred_labels = (groups_local_pred_scores > 0.5)
        # group_to_acc[groups_local] = accuracy_score(groups_local_target, groups_local_pred_labels)
        group_accs.append(
            (groups_local, accuracy_score(groups_local_target, groups_local_pred_labels), len(groups_local_target))
            # name, acc, group_size 
        )
        # print(
        #     "accuracy {:.3f}".format(accuracy_score(groups_local_target, groups_local_pred_labels)),
        #     '\t',
        #     'size:', len(groups_local_target), 
        #     '\t', 
        #     groups_local,
        # )
    
    ##################################
    # 5. report each group in the sorted order of acc. 
    ##################################
    group_accs.sort(key=lambda x: x[1], reverse=True)
    for tup in group_accs:
        groups_local, acc, group_size = tup
        info_str = "accuracy {:.3f} \t size: {} \t {}".format(acc, group_size, groups_local)
        if split == 'train':
            # train - dataset cargography
            info_str += '\t confidence: {:.3f} \t variability: {:.3f}'.format(
                np.mean(group_to_cargography_confidence[groups_local]), 
                np.mean(group_to_cargography_variability[groups_local])
            )
        print(info_str)
        logging.info(info_str)

    return 
    

def validate(val_loader, model, criterion, args, dumpResult):
    batch_time = AverageMeter('Time', ':6.3f')
    losses = AverageMeter('Loss', ':.4e')
    top1 = AverageMeter('Acc@1', ':6.2f')
    top5 = AverageMeter('Acc@5', ':6.2f')
    progress = ProgressMeter(
        len(val_loader),
        [batch_time, losses, top1, top5],
        prefix='Test: ')

    # switch to evaluate mode
    model.eval()

    ##################################
    # Also, output the confusion matrix 
    ##################################
    nb_classes = args.num_classes
    my_confusion_matrix = torch.zeros(nb_classes, nb_classes)

    ##################################
    # Fields to be stored for postprocessing 
    ##################################
    target_all = []
    pred_score_all = [] 


    with torch.no_grad():
        end = time.time()
        for i, (images, target) in enumerate(val_loader):
            if args.gpu is not None:
                images = images.cuda(args.gpu, non_blocking=True)
            if torch.cuda.is_available():
                target = target.cuda(args.gpu, non_blocking=True)

            # compute output
            output = model(images)
            loss = criterion(output, target)

            ##################################
            # Confusion Matrix
            ##################################
            _, preds = torch.max(output, 1)
            for t, p in zip(target.view(-1), preds.view(-1)):
                    my_confusion_matrix[t.long(), p.long()] += 1

            # measure accuracy and record loss
            # acc1, acc5 = accuracy(output, target, topk=(1, 5))
            acc1 = accuracy(output, target, topk=(1, ))[0]
            losses.update(loss.item(), images.size(0))
            top1.update(acc1[0], images.size(0))
            # top5.update(acc5[0], images.size(0))

            # measure elapsed time
            batch_time.update(time.time() - end)
            end = time.time()

            # if i % args.print_freq == 0:
            #     progress.display(i)

            ##################################
            # For analysis
            ##################################
            output_scores = torch.nn.functional.softmax(output, dim=-1)
            positive_scores = output_scores[:,1]

            target_all.append( target.cpu().numpy() )
            pred_score_all.append( positive_scores.cpu().numpy() )

        target_all = np.concatenate( target_all, axis=0)
        pred_score_all = np.concatenate( pred_score_all, axis=0)

        dump_result_dict = {
                    "target_all": target_all, 
                    "pred_score_all": pred_score_all, 
                    }
        if dumpResult is True:
            with open(os.path.join( args.output_dir, 'model_validate_dump.pkl'), "wb") as pkl_file:
                pickle.dump(
                    dump_result_dict, 
                    pkl_file, 
                )
        
        # a large analysis here 
        pred_label = (pred_score_all>0.5)
        print("accuracy {:.3f}".format(accuracy_score(target_all, pred_label)),
        '\t',
        "roc_auc_score {:.3f}".format(roc_auc_score(target_all, pred_score_all)), 
        )
        print("confusion_matrix\n{}".format(confusion_matrix(target_all, pred_label)))
        print("classification_report\n{}".format(classification_report(target_all, pred_label)))

        # TODO: this should also be done with the ProgressMeter
        # print(' * Acc@1 {top1.avg:.3f} Acc@5 {top5.avg:.3f}'
        #       .format(top1=top1, top5=top5))
        print('VAL * Acc@1 {top1.avg:.3f}'
              .format(top1=top1))

        # if is_main_process():
        logging.info("accuracy {:.3f}".format(accuracy_score(target_all, pred_label)))
        logging.info(
            "roc_auc_score {:.3f}".format( roc_auc_score(target_all, pred_score_all) )
        )
        logging.info("confusion_matrix\n{}".format(confusion_matrix(target_all, pred_label)))
        logging.info("classification_report\n{}".format(classification_report(target_all, pred_label)))
        logging.info('VAL * Acc@1 {top1.avg:.3f}'
            .format(top1=top1))



        # TODO: this should also be done with the ProgressMeter
        print(' * Acc@1 {top1.avg:.3f} Acc@5 {top5.avg:.3f}'
              .format(top1=top1, top5=top5))

    return top1.avg, dump_result_dict


def save_checkpoint(state, is_best, filename='checkpoint.pth.tar', args=None): 
    torch.save(state, os.path.join( args.output_dir, filename) )
    if is_best:
        shutil.copyfile(
            os.path.join( args.output_dir, filename), 
            os.path.join( args.output_dir, 'model_best.pth.tar')
            )


class AverageMeter(object):
    """Computes and stores the average and current value"""
    def __init__(self, name, fmt=':f'):
        self.name = name
        self.fmt = fmt
        self.reset()

    def reset(self):
        self.val = 0
        self.avg = 0
        self.sum = 0
        self.count = 0

    def update(self, val, n=1):
        self.val = val
        self.sum += val * n
        self.count += n
        self.avg = self.sum / self.count

    def __str__(self):
        fmtstr = '{name} {val' + self.fmt + '} ({avg' + self.fmt + '})'
        return fmtstr.format(**self.__dict__)


class ProgressMeter(object):
    def __init__(self, num_batches, meters, prefix=""):
        self.batch_fmtstr = self._get_batch_fmtstr(num_batches)
        self.meters = meters
        self.prefix = prefix

    def display(self, batch):
        entries = [self.prefix + self.batch_fmtstr.format(batch)]
        entries += [str(meter) for meter in self.meters]
        print('\t'.join(entries))
        ##################################
        # Save to logging 
        ##################################
        logging.info('\t'.join(entries))

    def _get_batch_fmtstr(self, num_batches):
        num_digits = len(str(num_batches // 1))
        fmt = '{:' + str(num_digits) + 'd}'
        return '[' + fmt + '/' + fmt.format(num_batches) + ']'


def adjust_learning_rate(optimizer, epoch, args):
    """Sets the learning rate to the initial LR decayed by 10 every 30 epochs"""
    # lr = args.lr * (0.1 ** (epoch // 30))
    lr = args.lr * (0.1 ** (epoch // args.lr_drop))
    for param_group in optimizer.param_groups:
        param_group['lr'] = lr


def accuracy(output, target, topk=(1,)):
    """Computes the accuracy over the k top predictions for the specified values of k"""
    with torch.no_grad():
        maxk = max(topk)
        batch_size = target.size(0)

        _, pred = output.topk(maxk, 1, True, True)
        pred = pred.t()
        correct = pred.eq(target.view(1, -1).expand_as(pred))

        res = []
        for k in topk:
            correct_k = correct[:k].reshape(-1).float().sum(0, keepdim=True)
            res.append(correct_k.mul_(100.0 / batch_size))
        return res


if __name__ == '__main__':
    main(fn_call_training=load_data_and_train)