
"""
Aggregate the subset influence results. 
"""
import numpy as np
import matplotlib
matplotlib.use('Agg')
import pickle
import os

import matplotlib.pyplot as plt


from metaDatasetBaseline import * 


def get_subset_loss_array(
    pred_score_all, 
    target_all, 
    val_out_of_domain_dataset, 
    groups_local_vocab, 
    imageID_to_group,
    ):
    group_to_pred_loss = defaultdict(list)
    my_dataset = val_out_of_domain_dataset
    for idx, sample in enumerate(my_dataset.samples):
        image_path, target = sample
        imageID = image_path.split('/')[-1].split('.')[0] # image_path = IMAGE_DATA_FOLDER + imageID + '.jpg'

        if target_all[idx] == 1:
            loss_item = -np.log(pred_score_all[idx])
        elif target_all[idx] == 0:
            loss_item = -np.log(1 - pred_score_all[idx])
        
        for groups_local in imageID_to_group[imageID]:
            group_to_pred_loss[groups_local].append(loss_item)

    loss_array = []
    for groups_local in groups_local_vocab:
        loss_array.append(
            np.mean(group_to_pred_loss[groups_local])
        )
    loss_array = np.array(loss_array)
    return loss_array


def get_total_loss(
    pred_score_all, 
    target_all, 
    ):
    total_loss = 0
    for pred, target in zip(pred_score_all, target_all):
        if target == 1:
            loss_item = -np.log(pred)
        elif target == 0:
            loss_item = -np.log(1 - pred)
        total_loss += loss_item

    total_loss /= len(pred_score_all)

    return total_loss


def heatmap(data, row_labels, col_labels, ax=None,
            cbar_kw={}, cbarlabel="", **kwargs):
    """
    Create a heatmap from a numpy array and two lists of labels.

    Parameters
    ----------
    data
        A 2D numpy array of shape (N, M).
    row_labels
        A list or array of length N with the labels for the rows.
    col_labels
        A list or array of length M with the labels for the columns.
    ax
        A `matplotlib.axes.Axes` instance to which the heatmap is plotted.  If
        not provided, use current axes or create a new one.  Optional.
    cbar_kw
        A dictionary with arguments to `matplotlib.Figure.colorbar`.  Optional.
    cbarlabel
        The label for the colorbar.  Optional.
    **kwargs
        All other arguments are forwarded to `imshow`.
    """

    if not ax:
        ax = plt.gca()

    # Plot the heatmap
    im = ax.imshow(data, vmin=-1, vmax=1, **kwargs)
    print('data', data)


    # Create colorbar
    cbar = ax.figure.colorbar(im, ax=ax, shrink=0.7, pad=0.02, **cbar_kw)
    cbar.ax.set_ylabel(cbarlabel, rotation=-90, va="bottom", fontsize=18) 

    # We want to show all ticks...
    ax.set_xticks(np.arange(data.shape[1]))
    ax.set_yticks(np.arange(data.shape[0]))
    # ... and label them with the respective list entries.
    ax.set_xticklabels(col_labels, fontsize=18) # 16
    ax.set_yticklabels(row_labels, fontsize=18)

    # Let the horizontal axes labeling appear on top.
    ax.tick_params(top=True, bottom=False,
                   labeltop=True, labelbottom=False)

    # Rotate the tick labels and set their alignment.
    plt.setp(ax.get_xticklabels(), 
    # rotation=-45, 
    rotation=-90, 
    ha="right",
             rotation_mode="anchor")
    
    # Turn spines off and create white grid.
    # ax.spines[:].set_visible(False)

    ax.set_xticks(np.arange(data.shape[1]+1)-.5, minor=True)
    ax.set_yticks(np.arange(data.shape[0]+1)-.5, minor=True)
    ax.grid(which="minor", color="w", linestyle='-', linewidth=3)
    ax.tick_params(which="minor", bottom=False, left=False)

    return im, cbar


def annotate_heatmap(im, data=None, valfmt="{x:.2f}",
                     textcolors=("black", "white"),
                     threshold=None, **textkw):
    """
    A function to annotate a heatmap.

    Parameters
    ----------
    im
        The AxesImage to be labeled.
    data
        Data used to annotate.  If None, the image's data is used.  Optional.
    valfmt
        The format of the annotations inside the heatmap.  This should either
        use the string format method, e.g. "$ {x:.2f}", or be a
        `matplotlib.ticker.Formatter`.  Optional.
    textcolors
        A pair of colors.  The first is used for values below a threshold,
        the second for those above.  Optional.
    threshold
        Value in data units according to which the colors from textcolors are
        applied.  If None (the default) uses the middle of the colormap as
        separation.  Optional.
    **kwargs
        All other arguments are forwarded to each call to `text` used to create
        the text labels.
    """

    if not isinstance(data, (list, np.ndarray)):
        data = im.get_array()

    # Normalize the threshold to the images color range.
    if threshold is not None:
        threshold = im.norm(threshold)
    else:
        threshold = im.norm(data.max())/2.

    # Set default alignment to center, but allow it to be
    # overwritten by textkw.
    kw = dict(horizontalalignment="center",
              verticalalignment="center")
    kw.update(textkw)

    # Get the formatter in case a string is supplied
    if isinstance(valfmt, str):
        valfmt = matplotlib.ticker.StrMethodFormatter(valfmt)

    # Loop over the data and create a `Text` for each "pixel".
    # Change the text's color depending on the data.
    texts = []
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            kw.update(color=textcolors[int(im.norm(data[i, j]) > threshold)])
            text = im.axes.text(j, i, valfmt(data[i, j], None), **kw)
            texts.append(text)

    return texts



def get_influence_mat(trail_path='./influence/trial0', epoch_num=599):

    ##################################
    # Dummay Arg for Debugging
    ##################################
    class DummyArg():
        data = '/data/GQA/metaDataset/'
        output_dir = './outputTmp'
        batch_size = 16
    args = DummyArg()

    ##################################
    # Load evaluation logs. 
    ##################################
    with open(os.path.join(trail_path, 'subset_influence_results_epoch_{}.pkl'.format(epoch_num)), "rb") as pkl_file:
        subset_influence_batch_results = pickle.load(pkl_file)
    
        sample_schedule = subset_influence_batch_results['sample_schedule']
        target_all = subset_influence_batch_results['target']
        batch_results = subset_influence_batch_results['batch_results']


    ##################################
    # Load Meta-Data
    ##################################
    # meta_data_path = os.path.join( args.data, 'imageID_to_group.pkl')
    meta_data_path = os.path.join( args.data, 'imageID_to_node.pkl')
    with open(meta_data_path, "rb") as pkl_file:
        imageID_to_group = pickle.load( pkl_file ) # defaultdict(set)


    ##################################
    # load out of domain val dataset   
    ##################################
    val_out_of_domain_dataset = datasets.ImageFolder(
            os.path.join(args.data, 'val_out_of_domain'),
            None,
    )

    ##################################
    # re-build the training group vocab 
    ##################################
    training_groups = set()
    for batch_ImageIDs in sample_schedule:
        for ImageID in batch_ImageIDs:
            for groups_local in imageID_to_group[ImageID]:
                training_groups.add(groups_local)

    training_groups_local_vocab = sorted(set(training_groups))
    N_training_groups = len(training_groups_local_vocab)
    # print('training_groups_local_vocab', training_groups_local_vocab)

    ##################################
    # re-build the val group vocab 
    ##################################
    val_groups = set()
    for image_path, target in val_out_of_domain_dataset.samples:
        ImageID = image_path.split('/')[-1].split('.')[0] # image_path = IMAGE_DATA_FOLDER + imageID + '.jpg'
        for groups_local in imageID_to_group[ImageID]:
            val_groups.add(groups_local)
    val_groups_local_vocab = training_groups_local_vocab + sorted(set(val_groups) - set(training_groups))
    N_val_groups = len(val_groups_local_vocab)
    # print('val_groups', val_groups_local_vocab)


    # training_groups_local_vocab = sorted(set(training_groups))[-30:]
    
    train_cat_nodes = [
        'cat(computer)', 
        'cat(keyboard)', 
        'cat(drawer)', 'cat(cabinet)',
        'cat(faucet)', 
        'cat(sink)', 
    ]

    train_dog_nodes = [
        'dog(carpet)', 'dog(sheet)', 
        'dog(blanket)', 'dog(curtain)',  
        'dog(truck)', 'dog(frisbee)', 
    ]
    training_groups_local_vocab = train_cat_nodes + train_dog_nodes
    N_training_groups = len(training_groups_local_vocab)

    val_cat_nodes = [
        'cat(fence)',
        'cat(television)',
    ]

    val_dog_nodes = [
        'dog(television)',
        'dog(animal)', 
    ]


    val_groups_local_vocab = training_groups_local_vocab + val_cat_nodes + val_dog_nodes 
    N_val_groups = len(val_groups_local_vocab)


    # process each batch, look up imageID. 
    # do not know groups_local_vocab in advance. 

    ##################################
    # Iteration of epoches/batches
    ##################################
    start_batch_id = 37 * 1
    previous_loss_array = get_subset_loss_array(
        batch_results[start_batch_id-1], #pred_score_all, 
        target_all, 
        val_out_of_domain_dataset, 
        val_groups_local_vocab, 
        imageID_to_group,
        )

    X_list = []
    y_all_list = []
    for batch_id in range(start_batch_id, len(batch_results)): 
        batch_ImageIDs = sample_schedule[batch_id]
        # TODO: translate batch_ImageIDs to group array
        training_group_array = np.zeros(N_training_groups)
        for ImageID in batch_ImageIDs:
            for groups_local in imageID_to_group[ImageID]:
                if groups_local not in training_groups_local_vocab:
                    continue 
                training_group_idx = training_groups_local_vocab.index(groups_local)
                training_group_array[training_group_idx] += 1/len(imageID_to_group[ImageID])

        ##################################  
        # Get delta loss 
        # digest the validation resutls - an inner loop 
        ##################################
        loss_array = get_subset_loss_array(
            batch_results[batch_id], #pred_score_all, 
            target_all, 
            val_out_of_domain_dataset, 
            val_groups_local_vocab, 
            imageID_to_group,
            )
        delta_loss_array = previous_loss_array - loss_array
        previous_loss_array = loss_array

        ##################################
        # Book keeping
        ##################################
        X_list.append(training_group_array)
        y_all_list.append(delta_loss_array)

    ##################################
    # Regression Analysis for each val subset 
    ##################################
    from sklearn.linear_model import LinearRegression
    X = np.stack(X_list, axis=0) # (n_samples, n_features)
    y = np.stack(y_all_list, axis=0) # (n_samples, n_targets)
    print('X',X,'y',y)
    reg = LinearRegression().fit(X, y)
    group_influence_matrix = reg.coef_.T # coef_: (n_targets, n_features)

    return group_influence_matrix, training_groups_local_vocab, val_groups_local_vocab


if __name__ == '__main__':
    
    group_influence_matrix, training_groups_local_vocab, val_groups_local_vocab = get_influence_mat(trail_path='./outputTmp_1/', epoch_num=4)

    # ##################################
    # # Normalize the matrix. divide each row by the count 
    # ##################################
    group_influence_matrix = group_influence_matrix / (group_influence_matrix.max())

    ##################################
    # Plot
    ##################################
    # plt.rcParams.update({'figure.figsize': (15, 10)})
    plt.rcParams.update({'figure.figsize': (10, 8)})
    fig, ax = plt.subplots()
    
    im, cbar = heatmap(group_influence_matrix, training_groups_local_vocab, val_groups_local_vocab, ax=ax,
                    cmap="RdYlGn", cbarlabel="Normalized Performance Contribution")
    fig.tight_layout()
    plt.show()
    os.makedirs('./influence/', exist_ok=True)
    plt.savefig( './influence/' + "influence_matrix.png")

