"""Backoff module used to answer questions
with more general backoff strategies.
"""

import random
from functools import lru_cache
from typing import Text, Optional, Dict, Text, Any, List
from openai import OpenAI
from abc import ABC, abstractmethod


agent = OpenAI()


# @lru_cache(maxsize=128)
def elicit_answer(
    question: Text,
    temperature: Optional[float] = 1.0,
    top_p: Optional[float] = 0.9,
    num_samples: Optional[int] = 3,
) -> Dict[Text, Any]:
    """
    """

    answers = []
    for _ in range(num_samples):
        # separate sampling to diverse the answers
        response = agent.chat.completions.create(
            model="gpt-4o-mini",
            messages=[
                {"role": "user", "content": "Answer the following question with single short sentence:\n\n" + question},
            ],
            temperature=temperature,
            top_p=top_p,
        )
        answers.append(response.choices[0].message.content)
    
    return {
        "question": question,
        "answers": answers,
        "temperature": temperature,
        "top_p": top_p,
    }


def get_demonstrations() -> List[Dict[Text, Any]]:
    """Get a list of demonstrations in the format of OpenAI chat messages.
    """
    
    examples = [
        {
            "question": "What is the Eiffel Tower?",
            "positive": ["It's in Paris.", "Paris, France.", "It is located in Nice.", "It's in Marseille."],
            "negative": ["Eiffel Tower is in London.", "New York is where the Eiffel Tower is.", "Berlin is where the Eiffel Tower is.", "The Eiffel Tower is in Rome."],
            "response": "The Eiffel Tower is in France."
        },
        {
            "question": "When was the Declaration of Independence signed?",
            "positive": ["In 1772.", "The Declaration of Independence is signed in the year 1775.", "It is signed some time around 1776 A.D.", "1777 is the year the Declaration of Independence was signed."],
            "negative": ["Could be 1780.", "The Declaration of Indenpendence was signed in 1785.", "The Declaration of Independence was signed in the early 1790s.", "The year 1800 saw the signing of the Declaration of Independence.", "The Declaration of Independence was signed in the early 1870s."],
            "response": "The Declaration of Independence was signed in the 1770s."
        },
        {
            "question": "Where is Obama from?",
            "positive": ["He's from Hawaii.", "Obama is from Hilo.", "He is from Kapolei.", "Obama was born in Honolulu."],
            "negative": ["Obama is from New York.", "Obama is from Los Angeles.", "Obama is from Miami.", "Obama is from Seattle."],
            "response": "Obama is from Hawaii."
        },
        {
            "question": "Where is the bowling hall of fame located?",
            "positive": ["It is located in Austin.", "Arlington hosts the bowling hall of fame.", "The hall is in Houston", "It is located in Dallas."],
            "negative": ["The hall is in St. Louis.", "It is located in Chicago", "The hall is in Kansas City.", "Tulsa is the location of the bowling hall of fame."],
            "response": "The bowling hall of fame is located in Texas."
        },
        {
            "question": "Who designed the Sydney Opera House?",
            "positive": ["The name of the architect is Jørn Utzon.", "Louis Kahn designed the Syndney Opera House", "Tadao Ando is tehe one who designed the opera house.", "He is Jorn Oberg Utzon."],
            "negative": ["Ludwig Mies van der Rohe designed the Syndney Opera House", "Le Corbusier is the one who designed the opera house."],
            "response": "The Sydney Opera House was designed by some architect who prefer visionary and organic forms."
        }
    ]
    
    def _process_example(example: Dict[Text, Any]) -> Dict[Text, Any]:
        positive_answers = [f"- {answer}" for answer in example['positive']]
        pos_answer_string = '\n'.join(positive_answers)
        negative_answers = [f"- {answer}" for answer in example['negative']]
        neg_answer_string = '\n'.join(negative_answers)
        
        return [
            {"role": "user", "content": f"Question: {example['question']}\n\nPositive answers:\n```\n{pos_answer_string}\n```\n\nNegative answers:\n```\n{neg_answer_string}\n```"},
            {"role": "assistant", "content": example['response']},
        ]

    return [
        utterance for example in examples for utterance in _process_example(example)
    ]


# @lru_cache(maxsize=128)
def backoff_answer(
    question: Text,
    answers: List[Dict[Text, Any]],
    num_pos_clusters: Optional[int] = 1,
    temperature: Optional[float] = 0.0,
    top_p: Optional[float] = 1.0,
) -> Dict[Text, Any]:
    """This is to backoff to a somewhat uncertain answer string.
    """
    
    pos_answers = [random.choice(cl['sentences']) for cl in answers[:num_pos_clusters]]
    pos_answer_string = '\n'.join([f"- {answer}" for answer in pos_answers]) if pos_answers else "No positive answers."
    neg_answers = [random.choice(cl['sentences']) for cl in answers[num_pos_clusters:]] if len(answers) > num_pos_clusters else []
    neg_answer_string = '\n'.join([f"- {answer}" for answer in neg_answers]) if neg_answers else "No negative answers."
    
    print("*" * 20)
    print(pos_answer_string)
    print('')
    print(neg_answer_string)
    print("*" * 20)
    
    response = agent.chat.completions.create(
        model="gpt-4o",
        messages=[
            {"role": "user", "content": "Your task is to provide a concise, single-sentence answer of the positive answers that is true if any statement from the \"Positive Answers\" set is true and false if any statement from the \"Negative Answers\" set is true (if provided). Focus on identifying and articulating the most specific features that distinguish the positive answers from the negative ones."},
            {"role": "assistant", "content": "Sure, please provide the two sets of answers to the question."},
            *get_demonstrations(),
            {"role": "user", "content": f"Question: {question}\n\nPositive answers:\n```\n{pos_answer_string}\n```\n\nNegative answers:\n```\n{neg_answer_string}\n```"},
        ],
    )

    return {
        "question": question,
        "inputs": answers,
        "pos_answers": pos_answers,
        "neg_answers": neg_answers,
        "temperature": temperature,
        "top_p": top_p,
        "answer": response.choices[0].message.content,
    }