""" Deduplicate negative answers in the set """


from dataclasses import dataclass
from overrides import overrides
from typing import Union, Text, List, Dict, Optional, Callable, Any

from langchain_core.runnables.config import RunnableConfig
from langchain_core.runnables.base import Runnable
from langchain.prompts import (
    ChatPromptTemplate,
    FewShotChatMessagePromptTemplate,
)
from langchain_core.output_parsers import BaseOutputParser
import re

# TODO: use customer downloaded examples for example selector
from langchain_interface.example_selectors import ConstantExampleSelector, ExampleSelector
from langchain_interface.steps import Step, FewShotStep
from langchain_interface.instances.instance import LLMResponse


@dataclass(frozen=True, eq=True)
class CoverageCheckRepsonse(LLMResponse):
    is_covered: bool
    

class CoverageCheckOutputParser(BaseOutputParser[CoverageCheckRepsonse]):
    def parse(self, text: Text) -> CoverageCheckRepsonse:
        """ """
        return CoverageCheckRepsonse(
            messages=text,
            is_covered=text.strip().lower() == "true"
        )
    
    @property
    def _type(self) -> Text:
        return "coverage-check"
    
    
class CoverageCheckStep(FewShotStep):
    """ """
    
    def __init__(self, example_selector: ExampleSelector = None):
        """ """
        if example_selector is None:
            example_selector = ConstantExampleSelector()
            examples = [
                {
                    "src": "early 1920s",
                    "tgt": "1923",
                    "is_covered": True
                },
                {
                    "src": "television and theatre",
                    "tgt": "theatre and television",
                    "is_covered": True
                },
                {
                    "src": "July 9, 2004",
                    "tgt": "July 8, 2004",
                    "is_covered": False
                },
                {
                    "src": "black or white",
                    "tgt": "black",
                    "is_covered": True
                },
                # {
                #     "src": "Tom and Jerry",
                #     "tgt": "Tom",
                #     "is_covered": False
                # }
            ]
            
            for example in examples:
                example_selector.add_example(example)
            
        super().__init__(example_selector=example_selector)
        
    @overrides
    def get_prompt_template(self) -> Runnable:
        """ """
        
        instruction_prompt = (
            "Given two phrases A and B, determine if the set denoted by B is a subset of the set denoted by A. "
            "Return \"True\" if B is a subset of A, and \"False\" otherwise without any explanation."
        )

        example_prompt = ChatPromptTemplate.from_messages(
            [
                ("human", "If an entity can be denoted by the phrase {src}, can it be denoted by the phrase {tgt}?"),
                ("ai", "{is_covered}")
            ]
        )
        
        fewshot_prompt_template = FewShotChatMessagePromptTemplate(
            example_prompt=example_prompt,
            example_selector=self._example_selector
        )
        
        return ChatPromptTemplate.from_messages(
            [
                ("human", instruction_prompt),
                fewshot_prompt_template,
                ("human", "If an entity can be denoted by the phrase {src}, can it be denoted by the phrase {tgt}?")
            ]
        )
        
    @overrides
    def get_output_parser(self) -> Runnable:
        """ """
        return CoverageCheckOutputParser()