"""
"""

from typing import (
    Any,
    Dict,
    List,
    Callable,
    ParamSpec,
    Tuple,
    Union,
    Generator,
    Iterable,
    TypeVar,
    Text,
    Optional
)
import ujson as json
import transformers
import os


_T = TypeVar("_T")
_RETURN_TYPE = TypeVar("_RETURN_TYPE")
_P = ParamSpec("_P")


def checkpoint(iterable: Iterable[_T], filepath: Text) -> Generator[Tuple[_T], None, None]:
    """Providing context manage for checkpointing an iterable.

    if the file does exist, it will read the file and yield the items,
    otherwise it will write the items to the file.
    """

    if not os.path.exists(filepath):
        os.makedirs(os.path.dirname(filepath), exist_ok=True)
        with open(filepath, 'w') as file_:
            for item in iterable:
                file_.write(json.dumps(item) + '\n')
                yield item
    else:
        with open(filepath, 'r') as file_:
            for line in file_:
                yield json.loads(line)

                
def deconstruct(func: Callable[_P, _RETURN_TYPE]) -> Callable[[Dict[Text, Any]], _RETURN_TYPE]:
    """
    """

    def wrapper(kwargs: Dict[Text, Any]) -> _RETURN_TYPE:
        return func(**kwargs)

    return wrapper


def batchify(pairs, batch_size):
    current_batch = []

    for item in pairs:
        current_batch.append(item)
        if len(current_batch) == batch_size:
            yield current_batch
            current_batch = []
    
    if current_batch:
        yield current_batch
        
