""" """
        
from functools import lru_cache
from typing import (
    Text, List, Union, Optional,
    Iterable, Generator
)
import ujson as json
from dataclasses import dataclass, field


@dataclass(frozen=True, eq=True)
class AttachingAnswer:
    answer: Text
    answer_id: int


@dataclass(eq=True)
class AnswerAttachedCluster:
    cluster_name: Text
    answers: List[AttachingAnswer]
    multiplicity: int = field(init=False)

    def __post_init__(self):
        self.multiplicity = len(self.answers)
        

@dataclass(frozen=True, eq=True)
class SimpleQAClusterAttachedItem:
    question: Text
    answer_type: Text
    gold_answer: Text
    clusters: List[AnswerAttachedCluster]
    
    
class SimpleQAAttachedDataReader:
    def __init__(self, data_path: Union[List[Text], Text]):
        if not isinstance(data_path, List):
            data_path = [data_path]
            
        self._data_paths = data_path

    def _parse_item(self, line_text: Text) -> SimpleQAClusterAttachedItem:
        """ """
        line_dict = json.loads(line_text)
        # line_dict.pop("messages")
        return SimpleQAClusterAttachedItem(
            question=line_dict['question'],
            answer_type=line_dict['answer_type'],
            gold_answer=line_dict['gold_answer'],
            clusters=[
                AnswerAttachedCluster(
                    cluster_name=cluster['cluster'],
                    answers=[
                        AttachingAnswer(
                            answer=attaching_answer['answer'],
                            answer_id=attaching_answer['aidx']
                        )
                        for attaching_answer in cluster['answers']
                    ]
                )
                for cluster in line_dict['clusters'] if cluster['answers']
            ]
        )
        
    @lru_cache(maxsize=3)
    def _read_file(self, file_path: Text) -> List[SimpleQAClusterAttachedItem]:
        with open(file_path, "r", encoding='utf-8') as file_:
            return [self._parse_item(line) for line in file_]
        
    def __iter__(self) -> Iterable[SimpleQAClusterAttachedItem]:
        for file_path in self._data_paths:
            for item in self._read_file(file_path):
                yield item