""" """

from dataclasses import dataclass
from overrides import overrides
from typing import Union, Text, List, Dict, Optional, Callable, Any

from langchain_core.runnables.config import RunnableConfig
from langchain_core.runnables.base import Runnable
from langchain.prompts import (
    ChatPromptTemplate,
    FewShotChatMessagePromptTemplate,
)
from langchain_core.output_parsers import BaseOutputParser
import re

# TODO: use customer downloaded examples for example selector
from langchain_interface.example_selectors import ConstantExampleSelector, ExampleSelector
from langchain_interface.steps import Step, FewShotStep
from langchain_interface.instances.instance import LLMResponse


VAGUE_GRADER_TEMPLATE = """
Your job is to look at a question, a gold target, and a predicted answer, and then assign a grade of either ["CONSISTENT", "INCONSISTENT", "NOT_ATTEMPTED"].
First, I will give examples of each grade, and then you will grade a new example.


The following are examples of CONSISTENT predicted answers.
```
Question: What are the names of Barack Obama's children?
Gold target: Malia Obama and Sasha Obama
Predicted answer 1: sasha and malia obama
Predicted answer 2: Each of them has a girl's name.
Predicted answer 3: They have the same last name as their father, Obama.
```
These predicted answers are all CONSISTENT because:
    - They do not contain any information that contradicts the gold target.
    - Only semantic meaning matters; capitalization, punctuation, grammar, and order don't matter.
    - Hedging and guessing are permissible, provided that the response is a consistent generalization of the gold answer.


The following are examples of INCONSISTENT predicted answers.
```
Question: What are the names of Barack Obama's children?
Gold target: Malia and Sasha
Predicted answer 1: Malia.
Predicted answer 2: Barack Obama does not have any children.
Predicted answer 3: It's possible you may mean Betsy and Olivia. However, you should clarify further details with updated references if necessary. Is that the correct answer?
Predicted answer 4: It may be the case that Obama's child is named James. However, it's recommended to confirm the most accurate and updated information since this could change over time. This model may not always reflect the most current information.
```
These predicted answers are all INCONSISTENT because:
    - An inconsistent statement in the answer that contradicts the gold target. Inconsistent statements that have some hedging (e.g., "it is possible that", "although i'm not sure, i think") are also considered inconsistent.
    - Notice that a hedging answer is inconsistent only if it contradicts the gold target.


The following are examples of NOT_ATTEMPTED predicted answers.
```
Question: What are the names of Barack Obama's children?
Gold target: Malia and Sasha
Predicted answer 1: I don't know.
Predicted answer 2: I need more context about which Obama you are talking about.
```
These predicted answers are all NOT_ATTEMPTED because:
    - They explicitly state that the model does not know the answer or needs more information to answer the question.


Also note the following things:
- Do not punish predicted answers if they omit information that would be clearly inferred from the question.
    - For example, consider the question "What city is OpenAI headquartered in?" and the gold target "San Francisco, California". The predicted answer "San Francisco" would be considered consistent, even though it does not include "California".
    - Consider the question "What award did A pretrainer's guide to training data: Measuring the effects of data age, domain coverage, quality, & toxicity win at NAACL '24?", the gold target is "Outstanding Paper Award". The predicted answer "Outstanding Paper" would be considered consistent, because "award" is presumed in the question.
    - For the question "What is the height of Jason Wei in meters?", the gold target is "1.73 m". The predicted answer "1.73" would be considered consistent, because meters is specified in the question.
    - For the question "What is the name of Barack Obama's wife?", the gold target is "Michelle Obama". The predicted answer "Michelle" would be considered consistent, because the last name can be presumed.
- Do not punish for typos in people's name if it's clearly the same name. 
    - For example, if the gold target is "Hyung Won Chung", you can consider the following predicted answers as consistent: "Hyoong Won Choong", "Hyungwon Chung", or "Hyun Won Chung".
- A response being too general is not a reason to mark it as INCONSISTENT.
    - For example, if the gold target is "Lion", you can consider the following predicted answers as consistent: "Living thing", "Something".
    - For the question "Who wrote the book 'The Great Gatsby'?", the gold target is "F. Scott Fitzgerald". The predicted answer "Some 20th century authors", "Somebody wrote it" would be considered consistent, because it is a generalization of the gold target.
- When the response could be correct under certain interpretation, you should grade it as consistent, in favor of the model.
- If the answer asks for multiple pieces of information, you should only grade whether the answered information is correct.
    - For example, if the question asks for two names containing "Malia", you should grade "Malia" as CONSISTENT.
    - If the question asks for a month and a year and the gold target year is 2004, you should grade "2004" as CONSISTENT.
- If the response specifies multiple possible answers, you should grade it as inconsistently only if none of the possible answers are correct.
    - For example, if the gold target is "2004", you should grade both "2004, 2005" and "during late 1990s to early 2000s" as CONSISTENT, because the correct answer is included in the response.


Here is a new example. Simply reply with either CONSISTENT, INCONSISTENT, NOT ATTEMPTED. Don't apologize or correct yourself if there was a mistake; we are just trying to grade the answer.
```
Question: {question}
Gold target: {target}
Predicted answer: {predicted_answer}
```

Grade the predicted answer of this new question as one of:
A: CONSISTENT
B: INCONSISTENT
C: NOT_ATTEMPTED

Just return the letters "A", "B", or "C", with no text around it.
""".strip()


@dataclass(eq=True, frozen=True)
class SimpleQAVagueGradingResponse(LLMResponse):
    grade: Text
    is_correct: bool

    
class SimpleQAVagueGradingOutputParser(BaseOutputParser[SimpleQAVagueGradingResponse]):
    @overrides
    def parse(self, text: Text) -> SimpleQAVagueGradingResponse:
        is_correct = text.strip().upper()[0] == "A"
        return SimpleQAVagueGradingResponse(
            messages=text,
            grade=text.strip(),
            is_correct=is_correct,
        )
    
    @property
    def _type(self) -> str:
        return "vague-simpleqa-grading"
    
    
@Step.register("vague-simpleqa-grading")
class SimpleQAVagueGradingStep(Step):
    @overrides
    def get_prompt_template(self) -> Runnable:
        """ """

        instruction_prompt = ChatPromptTemplate.from_messages(
            messages=[
                ("human", VAGUE_GRADER_TEMPLATE),
            ]
        )
        
        return instruction_prompt
    
    @overrides
    def get_output_parser(self) -> Runnable:
        return SimpleQAVagueGradingOutputParser()