import torch
import torch.nn as nn
import torch.nn.functional as F
from models.networks import Conv1d, ConvDW, ConvTransposeDW, Flatten, Unflatten
from models.base_vae_model import BaseModel


class CVAE(BaseModel):
    def __init__(self, latent_size=1024, num_classes=50, pretrained_encoder=False):
        super().__init__(latent_size)

        if pretrained_encoder:
            self.encoder = nn.Sequential(
                self._create_pretrained_encoder(),
                Flatten()
            )
        else:
            self.encoder = nn.Sequential(
                Conv1d(512, 512),  # conv5_4/sep
                ConvDW(512, 512, 1),  # conv5_5
                ConvDW(512, 1024, 2),  # conv5_6
                ConvDW(1024, 1024, 1),  # conv6
                nn.MaxPool2d(kernel_size=4),
                Flatten()
            )

        self.ext_enc = nn.Linear(1024, 256)
        self.mu = nn.Linear(256, self.latent_size)
        self.logvar = nn.Linear(256, self.latent_size)

        self.decoder = nn.Sequential(
            nn.Linear(self.latent_size + num_classes, 1024),
            Unflatten(1024, 1, 1),
            nn.UpsamplingBilinear2d(scale_factor=4),
            nn.ReLU(),
            ConvTransposeDW(1024, 1024, 1),  # deconv6
            ConvTransposeDW(1024, 512, 2),  # deconv5_6
            ConvTransposeDW(512, 512, 1),  # deconv5_5
            Conv1d(512, 512)  # conv5_4/sep
        )

    def forward(self, x, y_one_hot=None):
        """
        :param x: the batch of features to be reconstructed
        :param y_one_hot: the one-hot encoding vectors of labels
        :return: the reconstructed x features.
        """
        assert y_one_hot is not None, "labels should be provided to the forward method"
        mu, logvar = self.encode(x)
        z = self.reparameterize(mu, logvar)
        z = torch.cat((z, y_one_hot), dim=1)
        return self.decode(z), mu, logvar

    def encode(self, x):
        h = self.encoder(x)
        h = F.tanh(self.ext_enc(h))
        mu, logvar = self.mu(h), self.logvar(h)
        return mu, logvar

    def decode(self, z):
        z = self.decoder(z)
        return z

    def reparameterize(self, mu, logvar):
        if self.training:
            std = torch.exp(0.5 * logvar)
            eps = torch.randn_like(std)
            res = eps.mul(std).add_(mu)
            return res
        else:
            return mu
