import hashlib
import json
from pathlib import Path
from typing import Union, Callable, Any, Optional, Sequence, List, Dict, Protocol
import joblib
from joblib import Memory


def generic_cache_wrapper(keys: Any, value: Any = None):
    return value


class GenericCache(Protocol):
    def get_from_cache(self, keys: Dict) -> Union[None, Any]:
        ...

    def add_to_cache(self, keys: Dict, value: Any) -> None:
        ...


class JoblibCache:
    def __init__(self, root_path: Union[str, Path]):
        self.root_path: Path = Path(root_path)

        self.root_path.mkdir(parents=True, exist_ok=True)

        self.memory = Memory(str(self.root_path / 'joblib_cache'))
        self.cached_wrapper = self.memory.cache(generic_cache_wrapper, ignore=['value'])

        self.__cache_keys_as_str: List[str] = []
        if (self.root_path / 'cache_keys.dat').exists():
            self.__cache_keys_as_str = joblib.load(str(self.root_path / 'cache_keys.dat'))

    def get_from_cache(self, keys: Dict):
        key_as_str = json.dumps(keys, sort_keys=True, indent=2)
        if key_as_str in self.__cache_keys_as_str:
            return self.cached_wrapper(keys, value=None)
        else:
            return None

    def add_to_cache(self, keys: Dict, value: Any):
        key_as_str = json.dumps(keys, sort_keys=True, indent=2)
        if key_as_str not in self.__cache_keys_as_str:
            self.__cache_keys_as_str.append(key_as_str)
            self.cached_wrapper(keys, value=value)
            joblib.dump(self.__cache_keys_as_str, str(self.root_path / 'cache_keys.dat'))


class NoOpCache:
    def __init__(self):
        pass

    def get_from_cache(self, keys: Dict):
        return None

    def add_to_cache(self, keys: Dict, value: Any):
        pass
