
import numpy as np
from sklearn.base import BaseEstimator, ClassifierMixin
from sklearn.utils.multiclass import unique_labels
from keras.models import Model
from keras.layers import Dense, Input
from nanDense import nanDense


class NNClasifier(BaseEstimator, ClassifierMixin):
    
     def __init__(self, X, use_nanDense=True, use_c = False):
         
         """
         
         :param X: n by p input matrix
         :param use_c: Flag for using compensation weight, defaults to False


         """
         
         self.X = X
         self.use_c = use_c
         self.use_nanDense = use_nanDense
         input_layer = Input(shape=(X.shape[1]), name='Input_Layer')
         if use_nanDense:
             dense_layer1 = nanDense(int(np.round(X.shape[1]/2)), activation='relu',
                    name='Dense_Layer1', use_c=self.use_c)(input_layer) 
         else:
             dense_layer1 = Dense(int(np.round(X.shape[1]/2)), activation='relu',
                    name='Dense_Layer1')(input_layer) 
         dense_layer2 = Dense(2, activation='relu', name='Dense_Layer2')(dense_layer1)
         output_layer = Dense(1, activation='sigmoid', name='Output_Layer')(dense_layer2)
         model = Model(inputs=input_layer, outputs=output_layer)
         #model.summary()
         
         self.model = model
         
     def fit(self, X, y, **kwargs):
         """
         
         :param X: n by p input matrix
         :param y: n-vector of binary targets

         """
         
         epochs = kwargs.get('epochs', 100) 
         batch_size = kwargs.get('batch_size', int(np.round(X.shape[0]/100)))
         self.classes_ = unique_labels(y)
         self.X_ = X
         self.y_ = y
         metrics = ['AUC']
         loss = 'binary_crossentropy'
         self.model.compile(loss=loss, metrics=metrics, optimizer='sgd')  
         self.model.fit(X, y, epochs=epochs, batch_size=batch_size, 
                               verbose=0)  
         return self

     def predict(self, X):
         """
         
         :param X: n by p input matrix
         :return: y: predicticted binary target

         """

         y = self.model.predict(X)  
         
         return y
     
     def predict_proba(self, X):
         """
         
         :param X: n by p input matrix
         :return:y: predicticted probaility

         """

         y = self.model.predict(X)  
         probs = np.zeros([y.shape[0],2])
         probs[:,0:1] = 1 - y
         probs[:,1:] = y
         
         return probs

    

class NNRegressor(BaseEstimator, ClassifierMixin):

     def __init__(self, X, use_nanDense=True, use_c=False):
         
         """
         
         :param X: n by p input matrix
         :param use_c: Flag for using compensation weight, defaults to False


         """
         
         self.X = X
         self.use_c = use_c
         self.use_nanDense = use_nanDense
         input_layer = Input(shape=(X.shape[1]), name='Input_Layer')
         if use_nanDense:
             dense_layer1 = nanDense(int(np.round(X.shape[1]/2)), activation='relu',
                    name='Dense_Layer1', use_c=self.use_c)(input_layer) 
         else:
             dense_layer1 = Dense(int(np.round(X.shape[1]/2)), activation='relu',
                    name='Dense_Layer1')(input_layer) 
         dense_layer2 = Dense(2, activation='relu', name='Dense_Layer2')(dense_layer1)
         output_layer = Dense(1, name='Output_Layer')(dense_layer2)
         model = Model(inputs=input_layer, outputs=output_layer)
         
         self.model = model
         
     def fit(self, X, y, **kwargs):
         """
         
         :param X: n by p input matrix
         :param y: n-vector of outputs

         """
         
         epochs = kwargs.get('epochs', 100) 
         batch_size = kwargs.get('batch_size', int(np.round(X.shape[0]/100)))
         self.X_ = X
         self.y_ = y
         metrics = ['mse']
         loss = 'mse'
         self.model.compile(loss=loss, metrics=metrics, optimizer='sgd')  
         self.model.fit(X, y, epochs=epochs, batch_size=batch_size, 
                               verbose=0)  
         return self

     def predict(self, X):
         """
         
         :param X: n by p input matrix
         :return: y: n-vector of predicticted outputs

         """

         y = self.model.predict(X)  
         
         return y
     

         
class sterlizedNeuron(BaseEstimator, ClassifierMixin):

     def __init__(self, use_nanDense=True, use_c=False):
         """
         
         :param use_c: Flag to use compensation weight, defaults to False
         
         """
         
         self.use_c = use_c
         input_layer = Input(shape=(2), name='Input_Layer')
         if use_nanDense:
             sterlized_neuron = nanDense(4, activation='tanh',
                    name='sterlized_neuron', use_c=use_c)(input_layer) 
         else:
             sterlized_neuron = Dense(4, activation='tanh',
                    name='sterlized_neuron')(input_layer) 
         
         output = Dense(1, activation='sigmoid', name='output')(sterlized_neuron) 
        
         model = Model(inputs=input_layer, outputs=output)
         model.summary()
         
         self.model = model
         
     def fit(self, X, y, **kwargs):
         """
         
         :param X: n by p matrix of inputs
         :param y: vector of binary targets
         
         """
         
         epochs = kwargs.get('epochs', 100) 
         batch_size = kwargs.get('batch_size', 10)
         self.classes_ = unique_labels(y)
         self.X_ = X
         self.y_ = y
         metrics = ['AUC']
         loss = 'binary_crossentropy'
         self.model.compile(loss=loss, metrics=metrics, optimizer='sgd')  
         self.model.fit(X, y, epochs=epochs, batch_size=batch_size, verbose=0)  
         return self

     def predict(self, X):
         """
         
         :param X: n by p matrix of inputs
         :return: y predicticted binary target

         """

         y = self.model.predict(X)  
         
         return y
     
     def predict_proba(self, X):
         """
         
         :param X: n by p matrix of inputs
         :return: y predicticted class probability

         """

         y = self.model.predict(X)  
         probs = np.zeros([y.shape[0],2])
         probs[:,0:1] = 1 - y
         probs[:,1:] = y
         
         return probs
     