import os
import sammo
from sammo.runners import OpenAIChat, AnthropicChat, GeminiChat
from sammo.throttler import AtMost

SHARED_PARAMS = {"equivalence_class": "exact", "timeout": 240, "use_cached_timeouts": True, "max_timeout_retries": 3}

MODELS = {
    OpenAIChat: {
        "models": {
            "4.1-nano": {"full": "gpt-4.1-nano-2025-04-14", "pretty": "GPT-4.1 nano"},
            "4.1-mini": {"full": "gpt-4.1-mini", "pretty": "GPT-4.1 mini"},
            "4o-mini": {"full": "gpt-4o-mini-2024-07-18", "pretty": "GPT-4o mini"},
            "4o": {"full": "gpt-4o-2024-11-20", "pretty": "GPT-4o"},
            "o1-mini": {"full": "o1-mini-2024-09-12", "pretty": "GPT o1 mini"},
            "o3-mini": {"full": "o3-mini-2025-01-31", "pretty": "GPT o3 mini"},
            "o3": {"full": "o3-2025-04-16", "pretty": "o3"}
        },
        "api_config": {"api_key": os.getenv("OPENAI_API_KEY")},
        "params": {"rate_limit": [AtMost(10, "running", 1), AtMost(10, "calls", 1)]},
    },
    AnthropicChat: {
        "models": {
            "sonnet-3.7": {"full": "claude-3-7-sonnet-20250219", "pretty": "Claude 3.7 Sonnet"},
            "sonnet-3.5": {"full": "claude-3-5-sonnet-20241022", "pretty": "Claude 3.5 Sonnet"},
            "haiku-3.5": {"full": "claude-3-5-haiku-20241022", "pretty": "Claude 3.5 Haiku"},
            "haiku-3": {"full": "claude-3-haiku-20240307", "pretty": "Claude 3 Haiku"},
        },
        "api_config": {"api_key": os.getenv("ANTHROPIC_API_KEY"), "anthropic-version": "2023-06-01"},
        "params": {"max_context_window": 4096, "rate_limit": [AtMost(10, "running", 1), AtMost(2, "calls", 1)]},
    },
    GeminiChat: {
        "models": {
            "g-1.5-pro": {"full": "gemini-1.5-pro-002", "pretty": "Gemini 1.5 Pro"},
            "g-1.5-flash": {"full": "gemini-1.5-flash-002", "pretty": "Gemini 1.5 Flash"},
            "g-1.5-flash-8b": {"full": "gemini-1.5-flash-8b-001", "pretty": "Gemini 1.5 Flash 8B"},
            "g-2.5-flash": {"full": "gemini-2.5-flash-preview-04-17", "pretty": "Gemini 2.5 Flash"},
            "g-2.5-pro": {"full": "gemini-2.5-pro-preview-05-06", "pretty": "Gemini 2.5 Pro"},
        },
        "api_config": {"api_key": os.getenv("GOOGLE_API_KEY")},
        "params": {"rate_limit": [AtMost(10, "running", 1), AtMost(3, "rejected", 10)]},
    },
}


def lookup_model(model_shorthand):
    model_shorthand = model_shorthand.strip().lower()
    model_cls = None
    for llm_provider, config in MODELS.items():
        if model_shorthand in config["models"]:
            model = config["models"][model_shorthand]
            model_cls = llm_provider
    return {
        "model_class": model_cls,
        "model_id": model["full"],
        "pretty_name": model["pretty"],
        "api_config": MODELS[model_cls]["api_config"],
        "params": {**SHARED_PARAMS, **MODELS[model_cls]["params"]},
    }


def get_runner(model_shorthand, **kwargs):
    record = lookup_model(model_shorthand)
    model_alias = model_shorthand.strip().lower().replace(".", "")
    cache_file = (sammo.utils.MAIN_PATH / "cache" / f"{model_alias}.tsv")

    return record["model_class"](
        model_id=record["model_id"],
        api_config=record["api_config"],
        cache=cache_file,
        **record["params"],
        **kwargs,
    )
