#
# Copyright (C) 2023, Inria
# GRAPHDECO research group, https://team.inria.fr/graphdeco
# All rights reserved.
#
# This software is free for non-commercial, research and evaluation use 
# under the terms of the LICENSE.md file.
#
# For inquiries contact  george.drettakis@inria.fr
#

import torch
import math
from diff_gaussian_rasterization import GaussianRasterizationSettings, GaussianRasterizer
from scene.gaussian_model import GaussianModel
from utils.sh_utils import eval_sh
import time

def render(viewpoint_camera, pc : GaussianModel, pipe, bg_color : torch.Tensor, scaling_modifier = 1.0, override_color = None, numofgaussians = -1):
    """
    Render the scene. 
    
    Background tensor (bg_color) must be on GPU!
    """
    # print(viewpoint_camera.image_height, viewpoint_camera.image_width)
 
    # Create zero tensor. We will use it to make pytorch return gradients of the 2D (screen-space) means
    screenspace_points = torch.zeros_like(pc.get_xyz, dtype=pc.get_xyz.dtype, requires_grad=True, device="cuda") + 0
    if numofgaussians == -1:
        numofgaussians = screenspace_points.shape[0]
    screenspace_points = screenspace_points[:numofgaussians]

    try:
        screenspace_points.retain_grad()
    except:
        pass

    # Set up rasterization configuration
    tanfovx = math.tan(viewpoint_camera.FoVx * 0.5)
    tanfovy = math.tan(viewpoint_camera.FoVy * 0.5)

    raster_settings = GaussianRasterizationSettings(
        image_height=int(viewpoint_camera.image_height),
        image_width=int(viewpoint_camera.image_width),
        tanfovx=tanfovx,
        tanfovy=tanfovy,
        bg=bg_color,
        scale_modifier=scaling_modifier,
        viewmatrix=viewpoint_camera.world_view_transform,
        projmatrix=viewpoint_camera.full_proj_transform,
        sh_degree=pc.active_sh_degree,
        campos=viewpoint_camera.camera_center,
        prefiltered=False,
        debug=pipe.debug
    )

    rasterizer = GaussianRasterizer(raster_settings=raster_settings)

    means3D = pc.get_xyz[:numofgaussians]
    means2D = screenspace_points[:numofgaussians]
    opacity = pc.get_opacity[:numofgaussians]

    # If precomputed 3d covariance is provided, use it. If not, then it will be computed from
    # scaling / rotation by the rasterizer.
    scales = None
    rotations = None
    cov3D_precomp = None
    if pipe.compute_cov3D_python:
        cov3D_precomp = pc.get_covariance(scaling_modifier)[:numofgaussians]
    else:
        scales = pc.get_scaling[:numofgaussians]
        rotations = pc.get_rotation[:numofgaussians]

    # If precomputed colors are provided, use them. Otherwise, if it is desired to precompute colors
    # from SHs in Python, do it. If not, then SH -> RGB conversion will be done by rasterizer.
    shs = None
    colors_precomp = None
    if colors_precomp is None:
        if pipe.convert_SHs_python:
            shs_view = pc.get_features.transpose(1, 2).view(-1, 3, (pc.max_sh_degree+1)**2)
            dir_pp = (pc.get_xyz - viewpoint_camera.camera_center.repeat(pc.get_features.shape[0], 1))
            dir_pp_normalized = dir_pp/dir_pp.norm(dim=1, keepdim=True)
            sh2rgb = eval_sh(pc.active_sh_degree, shs_view, dir_pp_normalized)
            colors_precomp = torch.clamp_min(sh2rgb + 0.5, 0.0)
        else:
            shs = pc.get_features[:numofgaussians]
    else:
        colors_precomp = override_color

    # Rasterize visible Gaussians to image, obtain their radii (on screen). 
    # print(f"Rendering...{means3D.shapemea}")
    # print(f"Before rendering: {torch.cuda.memory_allocated() / (1024 ** 2):.2f} MB")

    rendered_image, radii = rasterizer(
        means3D = means3D,
        means2D = means2D,
        shs = shs,
        colors_precomp = colors_precomp,
        opacities = opacity,
        scales = scales,
        rotations = rotations,
        cov3D_precomp = cov3D_precomp)
    # print(means3D.shape)

    torch.cuda.empty_cache()

    # Those Gaussians that were frustum culled or had a radius of 0 were not visible.
    # They will be excluded from value updates used in the splitting criteria.
    return {"render": rendered_image,
            "viewspace_points": screenspace_points,
            "visibility_filter" : radii > 0,
            "radii": radii}


def render_multigs(viewpoint_camera, pc1 : GaussianModel, pc2 : GaussianModel, pipe, bg_color : torch.Tensor, scaling_modifier = 1.0, override_color = None):
    """
    Render the scene. 
    
    Background tensor (bg_color) must be on GPU!
    """
 
    # Create zero tensor. We will use it to make pytorch return gradients of the 2D (screen-space) means
    # pc1 is the base model, does not need to be trained, pc2 is the model to be trained
    screenspace_points_1 = torch.zeros_like(pc1.get_xyz, dtype=pc1.get_xyz.dtype, requires_grad=True, device="cuda") + 0
    screenspace_points_2 = torch.zeros_like(pc2.get_xyz, dtype=pc2.get_xyz.dtype, requires_grad=True, device="cuda") + 0
    try:
        screenspace_points_2.retain_grad()
    except:
        pass
    screenspace_points = torch.cat((screenspace_points_1, screenspace_points_2), dim=0)
    

    # Set up rasterization configuration
    tanfovx = math.tan(viewpoint_camera.FoVx * 0.5)
    tanfovy = math.tan(viewpoint_camera.FoVy * 0.5)

    raster_settings = GaussianRasterizationSettings(
        image_height=int(viewpoint_camera.image_height),
        image_width=int(viewpoint_camera.image_width),
        tanfovx=tanfovx,
        tanfovy=tanfovy,
        bg=bg_color,
        scale_modifier=scaling_modifier,
        viewmatrix=viewpoint_camera.world_view_transform,
        projmatrix=viewpoint_camera.full_proj_transform,
        sh_degree=pc1.active_sh_degree,
        campos=viewpoint_camera.camera_center,
        prefiltered=False,
        debug=pipe.debug
    )

    rasterizer = GaussianRasterizer(raster_settings=raster_settings)

    means3D_1 = pc1.get_xyz
    means2D = screenspace_points
    opacity_1 = pc1.get_opacity
    means3D_2 = pc2.get_xyz
    opacity_2 = pc2.get_opacity
    opacity = torch.cat((opacity_1, opacity_2), dim=0)
    means3D = torch.cat((means3D_1, means3D_2), dim=0)


    # If precomputed 3d covariance is provided, use it. If not, then it will be computed from
    # scaling / rotation by the rasterizer.
    scales = None
    rotations = None
    cov3D_precomp = None
    if pipe.compute_cov3D_python:
        cov3D_precomp_1 = pc1.get_covariance(scaling_modifier)
        cov3D_precomp_2 = pc2.get_covariance(scaling_modifier)
        conv3D_precomp = torch.cat((cov3D_precomp_1, cov3D_precomp_2), dim=0)
    else:
        scales_1 = pc1.get_scaling
        rotations_1 = pc1.get_rotation
        scales_2 = pc2.get_scaling
        rotations_2 = pc2.get_rotation
        scales = torch.cat((scales_1, scales_2), dim=0)
        rotations = torch.cat((rotations_1, rotations_2), dim=0)

    # If precomputed colors are provided, use them. Otherwise, if it is desired to precompute colors
    # from SHs in Python, do it. If not, then SH -> RGB conversion will be done by rasterizer.
    shs = None
    colors_precomp = None
    if colors_precomp is None:
        # if pipe.convert_SHs_python:
        #     shs_view = pc.get_features.transpose(1, 2).view(-1, 3, (pc.max_sh_degree+1)**2)
        #     dir_pp = (pc.get_xyz - viewpoint_camera.camera_center.repeat(pc.get_features.shape[0], 1))
        #     dir_pp_normalized = dir_pp/dir_pp.norm(dim=1, keepdim=True)
        #     sh2rgb = eval_sh(pc.active_sh_degree, shs_view, dir_pp_normalized)
        #     colors_precomp = torch.clamp_min(sh2rgb + 0.5, 0.0)
        # else:
        shs_1 = pc1.get_features
        shs_2 = pc2.get_features
        shs = torch.cat((shs_1, shs_2), dim=0)
    else:
        colors_precomp = override_color

    # Rasterize visible Gaussians to image, obtain their radii (on screen). 
    # print(f"Rendering...{means3D.shapemea}")
    # print(f"Before rendering: {torch.cuda.memory_allocated() / (1024 ** 2):.2f} MB")
    rendered_image, radii = rasterizer(
        means3D = means3D,
        means2D = means2D,
        shs = shs,
        colors_precomp = colors_precomp,
        opacities = opacity,
        scales = scales,
        rotations = rotations,
        cov3D_precomp = cov3D_precomp)
    
    radii = radii[screenspace_points_1.shape[0]:]

    # Those Gaussians that were frustum culled or had a radius of 0 were not visible.
    # They will be excluded from value updates used in the splitting criteria.
    return {"render": rendered_image,
            "viewspace_points": screenspace_points,
            "visibility_filter" : radii > 0,
            "radii": radii}
