#! /bin/bash

FRAMAC=frama-c
TIMEOUT=1200
PROCESSES=8
MACHINE=$(hostname)
SCRIPT=all_lemmas_proofs_Cl+.script

function kill_provers {
    for solver in why3server cvc4 alt-ergo ; do
	for pid in $(pgrep $solver) ; do
	    echo "Kill: " $pid ;
	    kill -9 $pid ;
	done ;
    done
}

PROVERS=why3
FUNCTIONS=array_pop,array_push,list_remove,list_add,list_final_add,list_init,list_head,list_tail,list_pop,list_length,list_item_next,list_push,list_chop,list_copy
LEMMA_FUNCTIONS=index_of_not_in_subrange,index_of_unexisting_item,index_of_up_unexisting_item,index_of_inter_existing_item,index_of,index_of_bounds_weak,index_of_existing_item_weak,linked_n_starting_from_null_empty,linked_n_all_elements_valid,linked_n_first_valid,linked_n_bounds,linked_n_valid_range,linked_n_next_of_all_indexes,linked_n_before_last,linked_n_split_segment,linked_n_split_segment_direct,linked_n_split_segment_right,linked_n_split_segment_right_direct,linked_n_merge_segment,linked_n_merge_segment_right,linked_n_all_elements
ALL_FUNCTIONS=$(echo $FUNCTIONS | tr ',' '\n')
STRUCTURES=list,list_ptr,list_point

WITHOUT_LEMMAS=no
WITHOUT_FUNCTIONS=no
WITHOUT_VALID_TESTS=no
WITH_INVALID_TESTS=no

function print_help {
    echo "By default, this script, on each data structure \"list\" \"list_ptr\" \"list_point\""
    echo "    - runs the coq proofs of the lemmas"
    echo "    - runs the proofs of all functions in the module"
    echo "    - runs the proofs of the valid user code functions"
    echo "    - skips the proofs of the invalid user code functions (that takes a lot of time since it has to fail)"

    echo "OPTIONS:"
    echo "    -gui:"
    echo "        in first position, makes Frama-C start in GUI mode"
    echo "    -only-one-structure :"
    echo "        only run the specified verification on \"list\""
    echo "    -without-lemmas :"
    echo "        skips the proofs of the lemmas"
    echo "    -only-functions=[functions,comma_separated]"
    echo "        only execute the proof of the specified functions"
    echo "        deactivate all other proofs (lemmas and tests)"
    echo "        only prove for the \"list\" structure"
    echo "    -without-functions=[functions,comma_separated]"
    echo "        skips the (specified) functions"
    echo "        If no function is specified, skips all of them"
    echo "    -without-valid-tests"
    echo "        skips the valid user code functions"
    echo "    -with-invalid-tests"
    echo "        force the proofs of the invalid tests"
}

for opt in "$@" ; do
    list=$(echo "$opt" | cut -s -d\= -f2)
    opt=$(echo "$opt" | cut -d\= -f1)
    
    case $opt in
	-help)
	    print_help
	    exit
	    ;;
	-gui)
	    FRAMAC=frama-c-gui
	    ;;
	-only-one-structure)
	    STRUCTURES=list
	    ;;
	-without-lemmas)
	    WITHOUT_LEMMAS=yes
	    ;;
	-without-functions)
	    if [ "$list" == "" ] ; then
		WITHOUT_FUNCTIONS=yes
	    else
		WITHOUT_FUNCTIONS=some
		REMOVE_FUNCTIONS=$list
	    fi
	    ;;
	-only-functions)
	    if [ "$list" == "" ] ; then
		echo "-only-functions cannot be used without arguments"
		exit
	    else
		ALL_FUNCTIONS=$list
		WITHOUT_LEMMAS=yes
		WITHOUT_VALID_TESTS=yes
		WITH_INVALID_TESTS=no
		STRUCTURES=list
		break
	    fi
	    ;;
	-without-valid-tests)
	    WITHOUT_VALID_TESTS=yes
	    ;;
	-with-invalid-tests)
	    WITH_INVALID_TESTS=yes
	    ;;
	*)
	    echo "Invalid argument: $opt"
	    exit
    esac
done

if [ $WITHOUT_FUNCTIONS == some ] ; then
    REMOVE_FUNCTIONS=$(echo $REMOVE_FUNCTIONS | tr ',' '\n')
    FUNCTIONS=$(printf "$ALL_FUNCTIONS\n$REMOVE_FUNCTIONS" | sort -d | uniq -u)
else
    FUNCTIONS=$ALL_FUNCTIONS
fi
FUNCTIONS=$(echo $FUNCTIONS | tr ' ' ',')
STRUCTURES=$(echo $STRUCTURES | tr ',' '\n')

function proofs_no_lemmas_for {
    (time\
	 $FRAMAC $1.c\
	 -wp\
	 -wp-rte\
	 -wp-gen\
	 -wp-out wp_out\
	 -wp-par=$PROCESSES\
	 -wp-prover=$PROVERS\
	 -wp-timeout=$TIMEOUT\
	 -wp-prop=-@lemma\
	 -wp-fct=$FUNCTIONS,$LEMMA_FUNCTIONS
	 #\
	 #-then\
	 #-wp-rte -wp\
	 #-wp-par=$PROCESSES\
	 #-wp-prover=$PROVERS\
	 #-wp-timeout=$TIMEOUT\
	 #-wp-prop=-@lemma\
	 #-wp-fct=$FUNCTIONS,$LEMMA_FUNCTIONS
    ) | tee verification_$1_${MACHINE}.txt ;
    kill_provers
}

function proofs_only_lemmas_for {
    (time\
	 $FRAMAC $1.c\
	 -wp\
	 -wp-par=$PROCESSES\
	 -wp-prover=alt-ergo,coq\
	 -wp-timeout=30\
	 -wp-prop=@lemma\
	 -wp-script=$SCRIPT
    ) | tee verification_$1_lemmas_${MACHINE}.txt
    kill_provers
}

function valid_tests_for {
    (time\
	 user_code_tests/script.sh "$1.c" user_code_tests/valid.c
    ) | tee verification_$1_success_tests_${MACHINE}.txt
    kill_provers
}

function invalid_tests_for {
    (time\
	 user_code_tests/script.sh "$1.c" user_code_tests/invalid.c
    ) | tee verification_$1_success_tests_${MACHINE}.txt
    kill_provers
}

for example in $STRUCTURES ; do
    #if [ $WITHOUT_LEMMAS == no ] ; then
    #    proofs_only_lemmas_for $example
    #fi
    #if [ ! $WITHOUT_FUNCTIONS == yes ] ; then
	proofs_no_lemmas_for $example
    #fi
    #if [ $WITHOUT_VALID_TESTS == no ] ; then
    #    valid_tests_for $example
    #fi
    #if [ $WITH_INVALID_TESTS == yes ] ; then
    #    invalid_tests_for $example
    #fi
done
