/*
 * Copyright (c) 2004, Swedish Institute of Computer Science.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * This file is part of the Contiki operating system.
 *
 * Author: Adam Dunkels <adam@sics.se>
 *
 */

/**
 * \addtogroup mem
 * @{
 */


/**
 * \defgroup memb Memory block management functions
 *
 * The memory block allocation routines provide a simple yet powerful
 * set of functions for managing a set of memory blocks of fixed
 * size. A set of memory blocks is statically declared with the
 * MEMB() macro. Memory blocks are allocated from the declared
 * memory by the memb_alloc() function, and are deallocated with the
 * memb_free() function.
 *
 * @{
 */


/**
 * \file
 *         Memory block allocation routines.
 * \author
 *         Adam Dunkels <adam@sics.se>
 *
 */

#ifndef MEMB_H_
#define MEMB_H_

// #include "sys/cc.h"

/**
 * Declare a memory block.
 *
 * This macro is used to statically declare a block of memory that can
 * be used by the block allocation functions. The macro statically
 * declares a C array with a size that matches the specified number of
 * blocks and their individual sizes.
 *
 * Example:
 \code
MEMB(connections, struct connection, 16);
 \endcode
 *
 * \param name The name of the memory block (later used with
 * memb_init(), memb_alloc() and memb_free()).
 *
 * \param structure The name of the struct that the memory block holds
 *
 * \param num The total number of memory chunks in the block.
 *
 */
#define MEMB(name, structure, num) \
        static char CC_CONCAT(name,_memb_count)[num]; \
        static structure CC_CONCAT(name,_memb_mem)[num]; \
        static struct memb name = {sizeof(structure), num, \
                                          CC_CONCAT(name,_memb_count), \
                                          (void *)CC_CONCAT(name,_memb_mem)}

struct memb {
  unsigned short size;
  unsigned short num;
  char *count;
  void *mem;
};

#include "logic_defs_memb.h"

/**
 * Initialize a memory block that was declared with MEMB().
 *
 * \param m A memory block previously declared with MEMB().
 */
/*@
  requires valid_memb(m);
  ensures valid_memb(m);
  ensures _memb_empty(m);
  assigns m->count[0 .. (m->num - 1)];
  assigns *((char*) m->mem + (0 .. (m->size * m->num - 1)));
*/
void  memb_init(struct memb *m);

/**
 * Allocate a memory block from a block of memory declared with MEMB().
 *
 * \param m A memory block previously declared with MEMB().
 */
/*@
  requires valid_memb(m);
  ensures valid_memb(m);
  assigns m->count[0 .. (m->num - 1)];
  behavior free_found:
    assumes \exists integer i; 0 <= i < m->num && m->count[i] == 0;
    ensures \exists integer i;
      0 <= i < m->num &&
      \old(m->count[i]) == 0 &&
      m->count[i] == 1 &&
      \result == (char*) m->mem + (i * m->size) &&
      \forall integer j; (0 <= j < i || i < j < m->num) ==> m->count[j] == \old(m->count[j]);
    ensures \valid((char*) \result + (0 .. (m->size - 1)));
    ensures _memb_numfree(m) == \old(_memb_numfree(m)) - 1;
    ensures _memb_allocated(m, \result);
  behavior full:
    assumes _memb_full(m);
    ensures \forall integer i; 0 <= i < m->num ==> m->count[i] == \old(m->count[i]);
    ensures _memb_numfree(m) == \old(_memb_numfree(m));
    ensures \result == NULL;
  complete behaviors;
  disjoint behaviors;
*/
void *memb_alloc(struct memb *m);

/**
 * Deallocate a memory block from a memory block previously declared
 * with MEMB().
 *
 * \param m m A memory block previously declared with MEMB().
 *
 * \param ptr A pointer to the memory block that is to be deallocated.
 *
 * \return The new reference count for the memory block (should be 0
 * if successfully deallocated) or -1 if the pointer "ptr" did not
 * point to a legal memory block.
 */
/*@
  requires valid_memb(m);
  ensures valid_memb(m);
  assigns m->count[_memb_index(m, ptr)];
  behavior alloc_found:
    assumes _memb_has(m, ptr) && _memb_allocated(m, ptr);
    ensures !_memb_allocated(m, ptr);
    ensures _memb_numfree(m) == \old(_memb_numfree(m)) + 1;
    ensures \result == 0;
  behavior already_free:
    assumes _memb_has(m, ptr) && !_memb_allocated(m, ptr);
    ensures !_memb_allocated(m, ptr);
    ensures _memb_numfree(m) == \old(_memb_numfree(m));
    ensures \result == 0;
  behavior elem_notfound:
    assumes !_memb_has(m, ptr);
    ensures m->count[_memb_index(m, ptr)] == \old(m->count[_memb_index(m, ptr)]);
    ensures _memb_numfree(m) == \old(_memb_numfree(m));
    ensures \result == -1;
  complete behaviors;
  disjoint behaviors;
*/
char  memb_free(struct memb *m, void *ptr);

/*@
  requires valid_memb(m);
  ensures \result <==> (m->mem <= ptr && (char*) ptr < (char*) m->mem + (m->num * m->size));
  assigns \nothing;
*/
int memb_inmemb(struct memb *m, void *ptr);

/*@
  requires valid_memb(m);
  ensures \result == _memb_numfree(m);
  assigns \nothing;
*/
int  memb_numfree(struct memb *m);

/** @} */
/** @} */

#endif /* MEMB_H_ */
