#include <stdint.h>
#include <stdlib.h>

#include "calc_sum.c"

// BEGIN{bridge}
/*@ // An axiomatic for floating-point exponential and logarithm
  @ // the specifications are given in exp_log.mlw, module ExpLogApprox
  @
  @ axiomatic ExpLogApprox __attribute__ ((j3_theory ("exp_log.ExpLogApprox"))) {
  @
  @ // Validity range on exponential, assuming 0.0 < exp_max_value
  @ logic real exp_max_value __attribute__((j3_symbol("ExpLogApprox.exp_max_value")));
  @
  @ // Error parameter on exponential, assuming 0. <=. exp_error <=. 0.5
  @ logic real exp_error __attribute__((j3_symbol("ExpLogApprox.exp_error")));
  @
  @ // Error parameter on logarithm, assuming 0. <=. log_error <=. 1.
  @ logic real log_error __attribute__((j3_symbol("ExpLogApprox.log_error")));
  @
  @ // the approximate exponential, on unbounded doubles
  @ // spec: for all x, |u_exp(x) - exp(x)| <= exp(x) * exp_error
  @ logic udouble u_exp(udouble x) __attribute__((j3_symbol("ExpLogApprox.u_exp")));
  @
  @ // Validity range on logarithm, assuming 0.0 < log_max_value
  @ logic real log_max_value __attribute__((j3_symbol("ExpLogApprox.log_max_value")));
  @
  @ // the approximate logarithm, on unbounded doubles
  @ // spec: for all x, 0 < x ==> |u_log(x) - log(x)| <= exp(x) * exp_error
  @ logic udouble u_log(udouble x) __attribute__((j3_symbol("ExpLogApprox.u_log")));
  @ }
  @*/

/*@ // An axiomatic for LSE
  @
  @ axiomatic LSE __attribute__ ((j3_theory ("j3bridge.LSE"))) {
  @
  @ logic udouble u_sum_of_u_exp(double a[MAX_SIZE], integer m, integer n)
  @         __attribute__((j3_symbol("LSE.u_sum_of_u_exp")));
  @
  @ logic udouble u_lse(double a[MAX_SIZE], integer size)
  @         __attribute__((j3_symbol("LSE.u_lse")));
  @
  @ logic real lse_exact(double a[MAX_SIZE], integer size)
  @        __attribute__((j3_symbol("LSE.lse_exact")));
  @
  @ // predicate no_overflow_pre(double s, double a[MAX_SIZE], integer size_ub, integer m, integer n)
  @ //             __attribute__((j3_symbol("J3Axiomatic.no_overflow_pre")));
  @
  @ // predicate no_overflow_post(double s, integer size_ub)
  @ //             __attribute__((j3_symbol("J3Axiomatic.no_overflow_post")));
  @ }
  @*/
// END{bridge}

// BEGIN{explog}
/*@ requires \abs(x) <= exp_max_value;
  @ ensures to_udouble(\result) == u_exp(to_udouble(x));
  @ assigns \nothing;
  @*/
extern double exp_approx(double x);

/*@ requires 0 < x <= log_max_value;
  @ ensures to_udouble(\result) == u_log(to_udouble(x));
  @ assigns \nothing;
  @*/
extern double log_approx(double x);
// END{explog}


// BEGIN{code}
/*@ requires 0 < size <= MAX_SIZE;
  @ requires \initialized (&a[0..size-1]);
  @ // to fit exp_approx pre-condition
  @ requires \forall integer i; 0 <= i < size ==> \abs(a[i]) <= exp_max_value;
  @ // additional requirement to prevent overflow on addition
  @ requires exp_max_value <= 701.0;
  @ // the hypothesis (6) of Theorem 3.4
  @ requires \exp(exp_max_value) * (1.0 + exp_error) * size * (1.0 + (eps * (size - 1)))
             <= log_max_value;
  @ // additional requirement to prevent overflow on addition
  @ requires log_max_value <= 0x1p1023;   // close to max double
  @ // the result is equal to the WhyML def of LSE on udouble
  @ ensures to_udouble(\result) == u_lse(a, size);
  @ // the accuracy property
  @ ensures \abs(\result - lse_exact(a, size)) <=
  @   log_error * \abs(lse_exact(a,size))
  @   - \log(1 - (exp_error + eps * (size - 1) * (1 + exp_error))) * (1 +
  @     log_error);
  @*/
double log_sum_exp(size_t size) {
  int i;
  double s = 0.0;

  /*@ loop invariant 0 <= i <= size;
    @ // to prove the first post-condition
    @ loop invariant to_udouble(s) == u_sum_of_u_exp(a, 0, i);
    @ // for absence of overflow on the sum
    @ loop invariant \forall integer j; 0 <= j < i ==>
    @   \abs(to_real(u_exp(to_udouble(a[j])))) <= \exp(exp_max_value) * (1.0 + exp_error) ;
    @ // for calling log:
    @ loop invariant (i == 0 ? s == 0.0 : 0.0 < s);
    @ loop assigns i, s;
    @ loop variant (size - i);
    @*/
  for (i = 0; i < size; i++) {
    //@ assert 0.0 <= to_real(u_exp(to_udouble(a[i]))) ;
    //@ assert to_real(to_udouble(a[i])) <= exp_max_value ;
    //@ assert \exp(to_real(to_udouble(a[i]))) <= \exp(exp_max_value) ;
    /*@ assert \abs(to_real(u_exp(to_udouble(a[i]))) - \exp(to_real(to_udouble(a[i]))))
      @        <= \exp(exp_max_value) * exp_error ;
      @*/
    //@ assert to_real(u_exp(to_udouble(a[i]))) <= \exp(exp_max_value) * (1.0+exp_error) ;
    //@ assert usum_double_bound(u_sum_of_u_exp(a, 0, i), \exp(exp_max_value) * (1.0 + exp_error), size);
    s += exp_approx(a[i]);
  }

  //@ assert usum_double_bound(to_udouble(s), \exp(exp_max_value) * (1.0 + exp_error), size);
  return log_approx(s);
}
// END{code}
