#include <stdint.h>
#include <stdlib.h>

// BEGIN{udouble}
/*@ axiomatic UDouble __attribute__ ((j3_theory ("udouble.UDouble"))) {
  @
  @   // logic type for unbounded floats in double format
  @   type udouble __attribute__((j3_symbol("UDouble.udouble")));
  @
  @   // injection of unbounded floats into real numbers
  @   logic real to_real(udouble x) __attribute__((j3_symbol("UDouble.to_real")));
  @
  @   // the epsilon constant, 2^{-53} / (1 + 2^{-53})
  @   logic real eps __attribute__((j3_symbol("UDouble.eps")));
  @
  @ }
  @*/
// END{udouble}

// BEGIN{bridge_udouble}
/*@ axiomatic J3UDouble __attribute__ ((j3_theory ("j3bridge.UDouble"))) {
  @
  @   // conversion of a J3 safe double to an unbounded double
  @   logic udouble to_udouble(double z)
  @           __attribute__((j3_symbol("J3UDouble.to_udouble")));
  @ }
  @*/
// END{bridge_udouble}

// BEGIN{bridge_sum}
#define MAX_SIZE 1024

/*@ axiomatic J3Sum __attribute__ ((j3_theory ("j3bridge.Sum"))) {
  @
  @   // sum of doubles, result as an unbounded double
  @   // usum_double(a,n,m) = a[n] + ... + a[m-1]
  @   //   where + is the rounding sum of udoubles
  @   logic udouble usum_double(double a[MAX_SIZE], integer n, integer m)
  @           __attribute__((j3_symbol("J3Sum.usum_double")));
  @
  @   // sum of doubles as a real number, without rounding
  @   // real_sum(a,n,m) = a[n] + ... + a[m-1]
  @   //   where + is the mathematical addition of real numbers
  @   logic real real_sum(double a[MAX_SIZE], integer n, integer m)
  @           __attribute__((j3_symbol("J3Sum.real_sum")));
  @
  @   // sum of absolute values, as a real number, without rounding
  @   // real_sum_of_abs(a,n,m) = |a[n]| + ... + |a[m-1]|
  @   //   where + is the mathematical addition of real numbers
  @   logic real real_sum_of_abs(double a[MAX_SIZE], integer n, integer m)
  @           __attribute__((j3_symbol("J3Sum.real_sum_of_abs")));
// END{bridge_sum}
  @
// BEGIN{ax_no_overflow}
  @   // predicate expressing a constant bound, defined in WhyML as:
  @   //   \abs(s) <= max * size * (1. + eps * (size - 1));
  @   predicate usum_double_bound(udouble s, real max, integer size)
  @           __attribute__((j3_symbol("J3Sum.usum_double_bound")));
// END{ax_no_overflow}
  @
  @   // predicate expressing a constant bound, defined in WhyML as:
  @   //   \abs(s) <= max * size * (1. + eps * (size - 1));
  @   // can't do that because no function in logic supported by J3
  @   // predicate usum_double_bound3((integer -> udouble) a, integer m, integer m, real max)
  @   //        __attribute__((j3_symbol("J3Sum.usum_double_bound3")));
// BEGIN{bridge_sum}
  @
  @ }
  @*/
// END{bridge_sum}



// BEGIN{code}
double a[MAX_SIZE];

#define MAX_VALUE 0x1p1012

/*@ requires 0 < size <= MAX_SIZE;
  @ requires \forall integer i; 0 <= i < size ==> \abs(a[i]) <= MAX_VALUE;
  @ requires \initialized (&a[0..]);
  @ ensures to_udouble(\result) == usum_double(a, 0, size);
  @ ensures \abs(\result - real_sum(a,0,size))
  @           <=  (size-1) * eps * real_sum_of_abs(a,0,size);
  @*/
double sum(size_t size) {
  int i;
  double s = 0.0;

  /*@ loop invariant 0 <= i <= size;
    @ loop invariant to_udouble(s) == usum_double(a, 0, i);
    @ loop assigns i, s;
    @ loop variant size - i;
    @*/
  for (i = 0; i < size; i++) {
    //@ assert usum_double_bound(to_udouble(s), MAX_VALUE, MAX_SIZE);
    s += a[i];
  }
  return s;
}
// END{code}
