#!/bin/env python3
import hashlib
import subprocess
import json
import queue
import threading
import time
import os

def get_hash(file_path):
    """Calculate the MD5 hash of a file."""
    hash_md5 = hashlib.md5()
    with open(file_path, "rb") as f:
        for chunk in iter(lambda: f.read(4096), b""):
            hash_md5.update(chunk)
    return hash_md5.hexdigest()

FILES = {}
import os

for root, dirs, files in os.walk('./data/why3/examples'):
    for file in files:
        if file.endswith('.mlw'):
            file_path = os.path.join(root, file)
            hash = get_hash(file_path)
            if hash in FILES:
                print(f"Duplicate file: {file_path} and {FILES[hash]}")
            else:
                FILES[hash] = file_path
                print(f"Added file: {file_path}")

class Why3Error(Exception):
    pass

def why_can_load(file):
    file = os.path.relpath(file)
    file_name = os.path.splitext(os.path.basename(file))[0]
    dir = os.path.dirname(file)
    session_dir = os.path.join(dir, file_name)
    if os.path.exists(session_dir):
        for root, dirs, files in os.walk(session_dir, topdown=False):
            for name in files:
                os.remove(os.path.join(root, name))
            for name in dirs:
                os.rmdir(os.path.join(root, name))
        os.rmdir(session_dir)
    cmd = ['why3', 'session', 'create', '-o', session_dir, file]
    result = subprocess.run(cmd, capture_output=True, text=True)

    if result.returncode != 0:
        raise Why3Error(f"why3 session create failed with return code {result.returncode}")
    
    if dir in ['data/why3/examples', 'data/why3/gallery']:
        cmd = ['why3', 'shell', file]
    else:
        cmd = ['why3', 'shell', '-L', dir, file]

    #file = os.path.relpath(file)
    process = subprocess.Popen(cmd, 
                                stdin=subprocess.PIPE,
                                stdout=subprocess.PIPE,
                                stderr=subprocess.PIPE)
    
    def dump_error():
        stderr = process.stderr.read().decode()
        if stderr:
            print("Error output from why3:")
            print(stderr)
    
    # 使用线程和队列实现超时读取
    line_queue = queue.Queue()
    
    def read_thread():
        try:
            while True:
                line = process.stdout.readline().decode('utf-8', errors='replace').strip()
                if line:
                    line_queue.put(line)
                elif process.poll() is not None:
                    break
        except Exception as e:
            line_queue.put(e)
    
    # 启动读取线程
    reader_thread = threading.Thread(target=read_thread, daemon=True)
    reader_thread.start()
    
    def read_line():
        try:
            line = line_queue.get(timeout=1.0)  # 1秒超时
            if not line and process.poll() is not None:
                dump_error()
                raise Why3Error(f"why3 shell process terminated unexpectedly with return code {process.returncode}")
            if isinstance(line, Exception):
                raise line
            return line
        except queue.Empty:
            return ''
    def read_until_start():
        start_time = time.time()
        while True:
            line = read_line()
            if line.endswith("OH!MY#GOD$I'M^GOING&TO*START"):
                break
            if time.time() - start_time > 1:
                return False
        return True
    ret = read_until_start()
    process.stdin.close()
    process.terminate()
    process.wait()
    return ret


NOT_SEEN = []

for root, dirs, files in os.walk('./data/why3/gallery'):
    for file in files:
        if file.endswith('.mlw'):
            file_path = os.path.join(root, file)
            hash = get_hash(file_path)
            if hash not in FILES and why_can_load(file_path):
                NOT_SEEN.append(file_path)
                print(f"New file: {file_path}")

