import os
import argparse
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt

import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.utils.data import Dataset, DataLoader, TensorDataset

import pytorch_lightning as pl
from sklearn.metrics import mean_squared_error, r2_score
from scipy.stats import pearsonr
from tqdm import tqdm

# Constants
LEARNING_RATE = 0.0001
BATCH_SIZE = 64 * 2
EPOCHS = 15
WEIGHT_DECAY = 0
GAMMA = None
STEP_SIZE = None
SCHEDULER_FLAG = False
SCHEDULER_STEP_FLAG = False
LOSS_TYPE = 'MSE'
LOSS_CALCULATION_MP_FLAG = True

class MLP(nn.Module):

    def __init__(self, num_features, num_classes, hidden_layer_dim=128, p_dropout=.1):
        super(MLP, self).__init__()

        self.linear1 = nn.Linear(num_features, hidden_layer_dim)
        self.leakyReLU = nn.LeakyReLU()
        self.dropout = nn.Dropout(p=p_dropout)
        self.linear2 = nn.Linear(hidden_layer_dim, num_classes)

    def forward(self, x, _0, _1, _2):
        x = x.view(x.size(0), -1).float()
        out = self.linear1(x)
        out = self.leakyReLU(out)
        out = self.dropout(out)
        out = self.linear2(out)
        return out


def evaluate_predictions(y_true, y_pred):
    y_true_np = y_true.detach().cpu().numpy()
    y_pred_np = y_pred.detach().cpu().numpy()
    mse = mean_squared_error(y_true_np, y_pred_np)
    corr = np.corrcoef(y_true_np.flatten(), y_pred_np.flatten())[0, 1]
    r2 = r2_score(y_true_np, y_pred_np)
    return mse, corr, r2


def test(model, test_loader):
    all_predictions = []
    all_targets = []
    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
    model.to(device)
    model.eval()
    with torch.no_grad():
        for inputs, targets in test_loader:
            inputs = inputs.to(device)
            targets = targets.to(device)
            outputs = model(inputs)
            all_predictions.append(outputs)
            all_targets.append(targets)
    all_predictions = torch.cat(all_predictions, dim=0)
    all_targets = torch.cat(all_targets, dim=0)
    mse, corr, r2 = evaluate_predictions(all_targets, all_predictions)
    print(f"Test - MSE: {mse}, Correlation: {corr}, R2: {r2}")


def train(train_loader, val_loader, lr, decay, num_epochs, scheduler_flag=None, scheduler_step_flag=None, step_size=None, gamma=None, loss_type='MSE', loss_calculation_mp_flag=True, file_name_prefix=None):
    model = MLP(num_features=np.prod(train_loader.dataset[0][0].shape), num_classes=1)

    train_losses, valid_losses = [], []
    train_mse, train_r2, train_corr = [], [], []
    valid_mse, valid_r2, valid_corr = [], [], []

    optimizer = torch.optim.Adam(model.parameters(), lr=lr, weight_decay=decay)
    if loss_type == 'MSE':
        criterion = torch.nn.MSELoss()
    elif loss_type == 'L1':
        criterion = torch.nn.L1Loss()

    if scheduler_flag:
        scheduler = torch.optim.lr_scheduler.StepLR(optimizer, step_size=step_size, gamma=gamma)

    epochs = num_epochs
    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
    model.to(device)

    for epoch in range(epochs):
        print(f"Epoch {epoch + 1}/{epochs}")
        model.train()
        total_loss = 0
        for idx, (inputs, targets) in enumerate(tqdm(train_loader, desc=f"Epoch {epoch + 1}/{epochs}")):
            targets = targets.unsqueeze(1)
            inputs, targets = inputs.to(device), targets.to(device)
            optimizer.zero_grad()
            outputs = model(inputs)
            loss = criterion(outputs, targets)
            loss.backward()
            optimizer.step()
            total_loss += loss.item()
        if scheduler_step_flag:
            scheduler.step()
        avg_train_loss = total_loss / len(train_loader)
        train_losses.append(avg_train_loss)
        model.eval()
        valid_loss = 0
        all_targets, all_outputs = [], []
        for inputs, targets in val_loader:
            targets = targets.unsqueeze(1)
            inputs, targets = inputs.to(device), targets.to(device)
            outputs = model(inputs)
            all_targets.extend(targets.cpu().numpy())
            all_outputs.extend(outputs.detach().cpu().numpy())
        avg_valid_loss = valid_loss / len(val_loader)
        valid_losses.append(avg_valid_loss)
        train_mse_val = mean_squared_error(all_targets, all_outputs)
        train_r2_val = r2_score(all_targets, all_outputs)
        train_corr_val, _ = pearsonr(np.array(all_targets).flatten(), np.array(all_outputs).flatten())
        valid_mse.append(train_mse_val)
        valid_r2.append(train_r2_val)
        valid_corr.append(train_corr_val)
        all_targets, all_outputs = [], []
        for inputs, targets in train_loader:
            targets = targets.unsqueeze(1)
            inputs, targets = inputs.to(device), targets.to(device)
            outputs = model(inputs)
            all_targets.extend(targets.cpu().numpy())
            all_outputs.extend(outputs.detach().cpu().numpy())
        train_mse_t = mean_squared_error(all_targets, all_outputs)
        train_r2_t = r2_score(all_targets, all_outputs)
        train_corr_t, _ = pearsonr(np.array(all_targets).flatten(), np.array(all_outputs).flatten())
        train_mse.append(train_mse_t)
        train_r2.append(train_r2_t)
        train_corr.append(train_corr_t)
        print(f"Epoch {epoch + 1}, Training Loss: {avg_train_loss}, Validation Loss: {avg_valid_loss}")
        print(f"Training - MSE: {train_mse_t}, R2: {train_r2_t}, Corr: {train_corr_t}")
        print(f"Validation - MSE: {train_mse_val}, R2: {train_r2_val}, Corr: {train_corr_val}")
    plt.figure(figsize=(10, 5))
    plt.plot(train_losses, label='Training Loss')
    plt.plot(valid_losses, label='Validation Loss')
    plt.title('Training and Validation Loss per Epoch')
    plt.xlabel('Epochs')
    plt.ylabel('Loss')
    plt.legend()
    if file_name_prefix:
        plt.savefig(f"{file_name_prefix}_loss.png")
    plt.show()
    train_df = pd.DataFrame({'Epoch': range(1, epochs + 1), 'MSE': train_mse, 'R2': train_r2, 'Corr': train_corr, 'Type': 'Training'})
    valid_df = pd.DataFrame({'Epoch': range(1, epochs + 1), 'MSE': valid_mse, 'R2': valid_r2, 'Corr': valid_corr, 'Type': 'Validation'})
    metrics_df = pd.concat([train_df, valid_df])
    return model


def run_experiment(train_loader, val_loader, test_loader, PRED_LEN, LEARNING_RATE, WEIGHT_DECAY, BATCH_SIZE, EPOCHS, LOSS_TYPE, LOSS_CALCULATION_MP_FLAG, SCHEDULER_FLAG=False, SCHEDULER_STEP_FLAG=False, STEP_SIZE=None, GAMMA=None):
    seed = 1
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    np.random.seed(seed)
    file_name_prefix = f'pred{PRED_LEN}_lr{LEARNING_RATE}_wd{WEIGHT_DECAY}_bs{BATCH_SIZE}_ep{EPOCHS}_loss{LOSS_TYPE}_mp{LOSS_CALCULATION_MP_FLAG}'
    if SCHEDULER_FLAG:
        file_name_prefix += f'_sch{SCHEDULER_FLAG}_ss{STEP_SIZE}_g{GAMMA}'
    model = train(train_loader, val_loader, lr=LEARNING_RATE, decay=WEIGHT_DECAY, num_epochs=EPOCHS, scheduler_flag=SCHEDULER_FLAG, scheduler_step_flag=SCHEDULER_STEP_FLAG, step_size=STEP_SIZE, gamma=GAMMA, loss_type=LOSS_TYPE, loss_calculation_mp_flag=LOSS_CALCULATION_MP_FLAG, file_name_prefix=file_name_prefix)
    test(model, test_loader)
    file_name = f'model_weights_{file_name_prefix}.pth'
    torch.save(model.state_dict(), file_name)


