# DeepLOB: Deep Convolutional Neural Networks for Limit Order Books
# Source: https://ieeexplore.ieee.org/stamp/stamp.jsp?arnumber=8673598


import pytorch_lightning as pl
from torch import nn
import torch


class DeepLob(pl.LightningModule):
    def __init__(self, num_classes, num_features):
        super().__init__()
        self.num_classes = num_classes

        # convolution blocks
        self.conv1 = nn.Sequential(
            nn.Conv2d(in_channels=1, out_channels=32, kernel_size=(1, 2), stride=(1, 2)),
            nn.LeakyReLU(negative_slope=0.01),
            # nn.Tanh(),
            nn.BatchNorm2d(32),
            nn.Conv2d(in_channels=32, out_channels=32, kernel_size=(4, 1)),
            nn.LeakyReLU(negative_slope=0.01),
            nn.BatchNorm2d(32),
            nn.Conv2d(in_channels=32, out_channels=32, kernel_size=(4, 1)),
            nn.LeakyReLU(negative_slope=0.01),
            nn.BatchNorm2d(32),
        )

        self.conv2 = nn.Sequential(
            nn.Conv2d(in_channels=32, out_channels=32, kernel_size=(1, 2), stride=(1, 2)),
            nn.Tanh(),
            nn.BatchNorm2d(32),
            nn.Conv2d(in_channels=32, out_channels=32, kernel_size=(4, 1)),
            nn.Tanh(),
            nn.BatchNorm2d(32),
            nn.Conv2d(in_channels=32, out_channels=32, kernel_size=(4, 1)),
            nn.Tanh(),
            nn.BatchNorm2d(32),
        )

        self.conv3 = nn.Sequential(
            nn.Conv2d(in_channels=32, out_channels=32, kernel_size=(1, (num_features //2)//2)),
            nn.LeakyReLU(negative_slope=0.01),
            nn.BatchNorm2d(32),
            nn.Conv2d(in_channels=32, out_channels=32, kernel_size=(4, 1)),
            nn.LeakyReLU(negative_slope=0.01),
            nn.BatchNorm2d(32),
            nn.Conv2d(in_channels=32, out_channels=32, kernel_size=(4, 1)),
            nn.LeakyReLU(negative_slope=0.01),
            nn.BatchNorm2d(32),
        )

        # inception modules
        self.inp1 = nn.Sequential(
            nn.Conv2d(in_channels=32, out_channels=64, kernel_size=(1, 1), padding='same'),
            nn.LeakyReLU(negative_slope=0.01),
            nn.BatchNorm2d(64),
            nn.Conv2d(in_channels=64, out_channels=64, kernel_size=(3, 1), padding='same'),
            nn.LeakyReLU(negative_slope=0.01),
            nn.BatchNorm2d(64),
        )

        self.inp2 = nn.Sequential(
            nn.Conv2d(in_channels=32, out_channels=64, kernel_size=(1, 1), padding='same'),
            nn.LeakyReLU(negative_slope=0.01),
            nn.BatchNorm2d(64),
            nn.Conv2d(in_channels=64, out_channels=64, kernel_size=(5, 1), padding='same'),
            nn.LeakyReLU(negative_slope=0.01),
            nn.BatchNorm2d(64),
        )

        self.inp3 = nn.Sequential(
            nn.MaxPool2d((3, 1), stride=(1, 1), padding=(1, 0)),
            nn.Conv2d(in_channels=32, out_channels=64, kernel_size=(1, 1), padding='same'),
            nn.LeakyReLU(negative_slope=0.01),
            nn.BatchNorm2d(64),
        )

        # lstm layers
        self.lstm = nn.LSTM(input_size=192, hidden_size=64, num_layers=1, batch_first=True)
        self.fc1 = nn.Linear(64, self.num_classes)

        self.softmax = nn.Softmax(dim=1)

    def forward(self, x):
        x = x[:, None, :, :]  # none stands for the channel
        # print("x initial: {}".format(x.shape))

        x = self.conv1(x)
        # print("x after conv1: {}".format(x.shape))
        x = self.conv2(x)
        # print("x after conv2: {}".format(x.shape))
        x = self.conv3(x)
        # print("x after conv3: {}".format(x.shape))

        x_inp1 = self.inp1(x)
        x_inp2 = self.inp2(x)
        x_inp3 = self.inp3(x)
        # print("x_inp1 shape: {}".format(x_inp1.shape))
        x = torch.cat((x_inp1, x_inp2, x_inp3), dim=1)
        # print("after cat x shape: {}".format(x.shape))
        # x = torch.transpose(x, 1, 2)
        x = x.permute(0, 2, 1, 3)
        # print("x after permute shape: {}".format(x.shape))
        x = torch.reshape(x, (-1, x.shape[1], x.shape[2]))
        # print("x shape before lstm: {}".format(x.shape))
        out, _ = self.lstm(x)
        # print("out shape after lstm: {}".format(out.shape))

        out = out[:, -1, :]
        # print(out.shape)
        out = self.fc1(out)
        # print("final shape: {}".format(out.shape))

        return out
