import csv

import statistics
from math import sqrt

import matplotlib
import matplotlib.pyplot as plt
import matplotlib.colors as mcolors

import numpy as np

matplotlib.rcParams['font.family'] = 'sans serif'
matplotlib.rcParams['text.color'] = 'black'

# print(plt.rcParams['figure.figsize'])

plt.rcParams['figure.figsize'] = [16, 4.8]

"""
['Timestamp',
'SUNET ID:',
'Layout',
'Algo',
'The AI followed my lead when making decisions.', -> followed lead
'The AI agent frequently blocked my progress.', -> blocked
"I adapted to the AI's strategy over time.", -> user adapted
'The AI was consistent in its actions.', -> consistent
"I felt like I needed to follow the AI's lead when making decisions.", -> user followed
'The AI always made reasonable actions throughout the game.', -> reasonable
'I would like to collaborate with this AI in future Overcooked tasks.', -> recommended
"The AI's actions were human-like.", -> human-like
'I trusted the AI agent in making good decisions.', -> trustworthy
'The AI agent was better than me at this game.', -> strong
'Other comments or observations?']
"""

LAYOUT_IND = 2
ALGO_IND = 3

POS_LABELS = {
    "Followed User": 4,
    "Consistent": 7,
    "Reasonable": 9,
    "Recommended": 10,
    "Human-like": 11,
    "Trustworthy": 12,
    "Strong": 13
}

NEG_LABELS = {
    "Blocked User": 5,
    "User Adapted": 6,
    "User Followed": 8
}

LAYOUT_MAPPING = {
    "Cramped Room (simple)": "Cramped Room",
    "Coordination Ring (random1)": "Coordination Ring"
}

ALGO_LIST = ['SP', 'ADAP', 'XP', 'MP']
ALGO_COLORS = [
    mcolors.CSS4_COLORS['lightgray'],
    mcolors.CSS4_COLORS['gray'],
    mcolors.CSS4_COLORS['khaki'],
    '#ff9b00'
]

ALGO_MAPPING = {
    'AI S': 'SP',
    'AI D': 'ADAP',
    'AI X': 'XP',
    'AI M': 'MP'
}

# [layout][algo][question]
POS_SCORES = {L: {
    a: {
        q: [] for q in POS_LABELS.keys()
    } for a in ALGO_LIST
} for L in LAYOUT_MAPPING.values()}

NEG_SCORES = {L: {
    a: {
        q: [] for q in NEG_LABELS.keys()
    } for a in ALGO_LIST
} for L in LAYOUT_MAPPING.values()}


def parsecsv():
    first_row = None
    with open('user_feedback.csv', 'r') as csv_file:
        reader = csv.reader(csv_file)
        first_row = reader.__next__()
        print(first_row)
        for row in reader:
            layout = LAYOUT_MAPPING[row[LAYOUT_IND]]
            algo = ALGO_MAPPING[row[ALGO_IND]]

            for q in POS_LABELS:
                POS_SCORES[layout][algo][q].append(
                    int(row[POS_LABELS[q]])
                )

            for q in NEG_LABELS:
                NEG_SCORES[layout][algo][q].append(
                    int(row[NEG_LABELS[q]])
                )
        # print(POS_SCORES)
        # print(NEG_SCORES)


def get_mean(values):
    return sum(values) / len(values)


def get_stdev(values):
    return statistics.pstdev(values) / sqrt(len(values))


def make_plots():
    spacing = 0.4
    indp = np.arange(len(POS_LABELS.keys()))
    indn = np.arange(len(NEG_LABELS.keys()))
    fullind = np.concatenate((indp, indn + len(indp) + spacing))
    print(fullind)
    width = 0.2

    for L in LAYOUT_MAPPING.values():
        plt.clf()
        # plt.title(f"{L}: User Survey", fontsize=20)
        plt.ylabel("Score", color="black", fontsize=25)
        for i, A in enumerate(ALGO_LIST):
            algo_scores = []
            algo_std = []
            for Q in POS_LABELS.keys():
                algo_scores.append(get_mean(POS_SCORES[L][A][Q]))
                algo_std.append(get_stdev(POS_SCORES[L][A][Q]))

            for Q in NEG_LABELS.keys():
                algo_scores.append(get_mean(NEG_SCORES[L][A][Q]))
                algo_std.append(get_stdev(NEG_SCORES[L][A][Q]))

            plt.bar(fullind + width*i, algo_scores, width, yerr=algo_std, color=ALGO_COLORS[i])

        for pos in ['right', 'top']:
            plt.gca().spines[pos].set_visible(False)
        plt.gca().spines['bottom'].set_color('black')
        plt.gca().spines['left'].set_color('black')

        plt.gca().tick_params(axis='x', colors='black')
        plt.gca().tick_params(axis='y', colors='black')
        plt.xticks(fullind + width * 1.5, list(POS_LABELS.keys()) + list(NEG_LABELS.keys()), fontsize=25, rotation=20, ha='right')

        plt.yticks(fontsize=20)

        plt.tight_layout()
        
        layout_file = L.replace(" ", "_").lower()
        plt.savefig(f"{layout_file}_feedback_scores.pdf", bbox_inches='tight')
        plt.show()


def main():
    parsecsv()

    make_plots()


if __name__ == '__main__':
    main()
