#ours

import abc
from collections import OrderedDict
import time

import gtimer as gt
import numpy as np

from rlkit.core import logger, eval_util
from rlkit.data_management.env_replay_buffer import MultiTaskReplayBuffer
from rlkit.data_management.path_builder import PathBuilder
from rlkit.samplers.in_place import InPlacePathSampler
from rlkit.torch import pytorch_util as ptu

import copy
import random

def mean_with_max(l,mx,size=10):
    return np.mean(l+[mx]*(size-len(l)))

def cal_posterior(ret,ret_step,n_epi,ret_window,phi0,mu0,phi):
    weight=np.ones(len(ret_step))#1-(n_epi-np.array(ret_step))/ret_window
    return (phi0*mu0+phi*(weight*np.array(ret)).sum())/(phi0+phi*weight.sum()),1/(phi0+phi*weight.sum())

def big_prob(p1,p2,size=100):
    rn1 = np.random.normal(loc=p1[0], scale=p1[1], size=size)
    rn2 = np.random.normal(loc=p2[0], scale=p2[1], size=size)
    return (rn1>rn2).mean()

class MetaRLAlgorithm(metaclass=abc.ABCMeta):
    def __init__(
            self,
            env,
            agent,
            train_tasks,
            eval_tasks,
            meta_batch=64,
            num_iterations=100,
            num_train_steps_per_itr=1000,
            num_initial_steps=100,
            num_tasks_sample=100,
            num_steps_prior=100,
            num_steps_posterior=100,
            num_extra_rl_steps_posterior=100,
            num_evals=10,
            num_steps_per_eval=1000,
            batch_size=1024,
            embedding_batch_size=1024,
            embedding_mini_batch_size=1024,
            max_path_length=1000,
            discount=0.99,
            replay_buffer_size=1000000,
            reward_scale=1,
            num_exp_traj_eval=1,
            update_post_train=1,
            eval_deterministic=True,
            render=False,
            save_replay_buffer=False,
            save_algorithm=False,
            save_environment=False,
            render_eval_paths=False,
            dump_eval_paths=False,
            plotter=None,
    ):
        """
        :param env: training env
        :param agent: agent that is conditioned on a latent variable z that rl_algorithm is responsible for feeding in
        :param train_tasks: list of tasks used for training
        :param eval_tasks: list of tasks used for eval

        see default experiment config file for descriptions of the rest of the arguments
        """
        self.fake_tasks=2000#200
        
        self.env = env
        self.agent = agent
        self.agent_bf = copy.deepcopy(agent).cuda() # Can potentially use a different policy purely for exploration rather than also solving tasks, currently not being used
        self.train_tasks = train_tasks
        self.eval_tasks = eval_tasks
        self.meta_batch = meta_batch
        self.num_iterations = 500#2000#num_iterations
        self.num_train_steps_per_itr = num_train_steps_per_itr
        self.num_initial_steps = num_initial_steps
        self.num_tasks_sample = num_tasks_sample
        self.num_steps_prior = num_steps_prior
        self.num_steps_posterior = num_steps_posterior
        self.num_extra_rl_steps_posterior = num_extra_rl_steps_posterior
        self.num_evals = num_evals
        self.num_steps_per_eval = num_steps_per_eval
        self.batch_size = batch_size
        self.embedding_batch_size = embedding_batch_size
        self.embedding_mini_batch_size = embedding_mini_batch_size
        self.max_path_length = max_path_length
        self.discount = discount
        self.replay_buffer_size = replay_buffer_size
        self.reward_scale = reward_scale
        self.update_post_train = update_post_train
        self.num_exp_traj_eval = num_exp_traj_eval
        self.eval_deterministic = eval_deterministic
        self.render = render
        self.save_replay_buffer = save_replay_buffer
        self.save_algorithm = save_algorithm
        self.save_environment = save_environment

        self.eval_statistics = None
        self.render_eval_paths = render_eval_paths
        self.dump_eval_paths = dump_eval_paths
        self.plotter = plotter

        self.sampler = InPlacePathSampler(
            env=env,
            policy=agent,
            max_path_length=self.max_path_length,
        )

        # separate replay buffers for
        # - training RL update
        # - training encoder update
        self.replay_buffer = MultiTaskReplayBuffer(
                self.replay_buffer_size,
                env,
                np.arange(self.fake_tasks),#self.train_tasks,
            )

        self.enc_replay_buffer = MultiTaskReplayBuffer(
                self.replay_buffer_size,
                env,
                np.arange(self.fake_tasks),#self.train_tasks,
        )

        self._n_env_steps_total = 0
        self._n_train_steps_total = 0
        self._n_rollouts_total = 0
        self._do_train_time = 0
        self._epoch_start_time = None
        self._algo_start_time = None
        self._old_table_keys = None
        self._current_path_builder = PathBuilder()
        self._exploration_paths = []
        
        # self.switch_freq=2000#500#100
        self.switch_cnt=0
        # self.true_idx = 0#np.random.randint(len(self.train_tasks))#
        self.explore_ret=[]
        self.explore_ret_step=[]
        self.context_ret=[]
        self.context_ret_step=[]
        # self.phase_change=False
        # self.phase='context'
        self.ret_window=100#50
        self.reset_freq=50
        np.random.seed(1)
        if len(self.train_tasks)==2:
            self.true_idx_list=[i%2 for i in range(100000)]
        else:
            self.true_idx_list=np.random.randint(0,len(self.train_tasks),100000)
        self.true_idx_pt=0
        self.true_idx=self.true_idx_list[self.true_idx_pt]
        self.max_explore=-10000
        self.max_context=-10000
        self.n_explore,self.n_context=0,0
        self.keep_len=10
        
        self.bayes=True
        if self.bayes:
            self.phi0=1#10#100
            self.phi=1
            self.phi0_estimax=10#100
            self.phi_estimax=1
            self.mu0_estimax=10000
            # self.max_len=10000
            
        # self.relabel_percentage=0#1#0.5

    def make_exploration_policy(self, policy):
         return policy

    def make_eval_policy(self, policy):
        return policy

    def sample_task(self, is_eval=False):
        '''
        sample task randomly
        '''
        if is_eval:
            idx = np.random.randint(len(self.eval_tasks))
        else:
            idx = np.random.randint(len(self.train_tasks))
        return idx

    def reset_task(self,idx):
        if type(self.train_tasks[idx]) is not int:#ml10
            # print('debug',idx,self.train_tasks[idx])
            self.env=self.train_tasks[idx]
            self.sampler.env=self.env
            if self.phase_change or self.phase=='expl':
                self.sampler_expl.env=self.env
        elif hasattr(self.env,'tasks_pool'):#ml1
            self.env.set_task(self.env.tasks_pool[idx])
        else:
            self.env.reset_task(idx)

    def train(self):
        '''
        meta-training loop
        '''
        self.pretrain()
        params = self.get_epoch_snapshot(-1)
        logger.save_itr_params(-1, params)
        gt.reset()
        gt.set_def_unique(False)
        self._current_path_builder = PathBuilder()

        idx=-1
        avail_idx=[]
        # at each iteration, we first collect data from tasks, perform meta-updates, then try to evaluate
        recent_list=[]
        for it_ in gt.timed_for(
                range(self.num_iterations),
                save_itrs=True,
        ):
            # self.ret_window=100+400*(it_/self.num_iterations)#decay
            self._start_epoch(it_)
            self.training_mode(True)
            # if it_ == 0:
            #     print('collecting initial pool of data for train and eval')
            #     # temp for evaluating
            #     for idx in self.train_tasks:
            #         self.task_idx = idx
            #         self.env.reset_task(idx)
            #         self.collect_data(self.num_initial_steps, 1, np.inf)
            # Sample data from train tasks.
            current_list=[]
            for i in range(self.num_tasks_sample):
                n_epi=it_*self.num_tasks_sample+i
                logger.record_tabular("Epoch", it_)
                logger.record_tabular("n_epi", n_epi)
                if True:#not self.bayes:
                    while len(self.explore_ret)>0 and self.explore_ret_step[0]<n_epi-self.ret_window:
                        self.explore_ret=self.explore_ret[1:]
                        self.explore_ret_step=self.explore_ret_step[1:]
                    while len(self.context_ret)>0 and self.context_ret_step[0]<n_epi-self.ret_window:
                        self.context_ret=self.context_ret[1:]
                        self.context_ret_step=self.context_ret_step[1:]
                if self.phase_change:
                    if self.bayes:
                        # if len(self.explore_ret)<self.keep_len and len(self.context_ret)<self.keep_len:
                        #     if random.random()<0.5:
                        #         self.phase='explore'
                        #     else:
                        #         self.phase='context'
                        # elif len(self.explore_ret)<self.keep_len:
                        #     self.phase='explore'
                        # elif len(self.context_ret)<self.keep_len:
                        #     self.phase='context'
                        #else:
                        if True:
                            # self.explore_ret,self.explore_ret_step=self.explore_ret[-self.max_len:],self.explore_ret_step[-self.max_len:]
                            # self.explore_ret,self.explore_ret_step=self.explore_ret[-self.max_len:],self.explore_ret_step[-self.max_len:]
                            
                            # self.mu0_estimax=max(self.max_context,self.max_explore)
                            
                            explore_estimax=(self.phi0_estimax*self.mu0_estimax+self.phi_estimax*self.n_explore*self.max_explore)/(self.phi0_estimax+self.phi_estimax*self.n_explore)
                            explore_post=cal_posterior(self.explore_ret,self.explore_ret_step,n_epi,self.ret_window,phi0=self.phi0,mu0=explore_estimax,phi=self.phi)
                            # explore_post=cal_posterior(self.explore_ret,self.explore_ret_step,n_epi,n_epi,phi0=self.phi0,mu0=explore_estimax,phi=self.phi)
                            print('explore',self.mu0_estimax,explore_estimax,self.explore_ret,self.explore_ret_step,n_epi,explore_post)
                            
                            context_estimax=(self.phi0_estimax*self.mu0_estimax+self.phi_estimax*self.n_context*self.max_context)/(self.phi0_estimax+self.phi_estimax*self.n_context)
                            context_post=cal_posterior(self.context_ret,self.context_ret_step,n_epi,self.ret_window,phi0=self.phi0,mu0=context_estimax,phi=self.phi)
                            # context_post=cal_posterior(self.context_ret,self.context_ret_step,n_epi,n_epi,phi0=self.phi0,mu0=context_estimax,phi=self.phi)
                            print('context',self.mu0_estimax,context_estimax,self.context_ret,self.context_ret_step,n_epi,context_post)
                            
                            if random.random()<big_prob(explore_post,context_post):
                                self.phase='explore'
                            else:
                                self.phase='context'
                    else:
                        if len(self.explore_ret)<self.keep_len and len(self.context_ret)<self.keep_len:
                            if random.random()<0.5:
                                self.phase='explore'
                            else:
                                self.phase='context'
                        elif len(self.explore_ret)<self.keep_len:
                            self.phase='explore'
                        elif len(self.context_ret)<self.keep_len:
                            self.phase='context'
                        elif np.mean(self.explore_ret)<np.mean(self.context_ret):
                        
                        # if mean_with_max(self.explore_ret,self.max_explore)<mean_with_max(self.context_ret,self.max_context):
                            self.phase='context'
                        else:
                            self.phase='explore'
                else:
                    pass#self.phase='context'
                logger.record_tabular("phase", self.phase)
                
                # idx = np.random.randint(len(self.train_tasks))
                idx=(idx+1)%self.fake_tasks
                current_list.append(idx)
                recent_list.append(idx)
                if idx not in avail_idx:
                    avail_idx.append(idx)
                print('switch check',self.switch_cnt,self.switch_freq)
                if self.switch_cnt>=self.switch_freq:
                    # self.true_idx = (self.true_idx+1)%len(self.train_tasks)#np.random.randint(len(self.train_tasks))#
                    self.true_idx_pt+=1
                    self.true_idx=self.true_idx_list[self.true_idx_pt]
                    self.switch_cnt=0
                # elif self.switch_cnt>self.switch_freq:
                #     raise
                self.switch_cnt+=self.num_steps_prior//200+self.num_extra_rl_steps_posterior//200
                
                self.task_idx = idx
                self.reset_task(self.true_idx)#self.env.reset_task(self.true_idx)
                self.enc_replay_buffer.task_buffers[idx].clear()
                self.replay_buffer.task_buffers[idx].clear()

                # collect some trajectories with z ~ prior
                if self.num_steps_prior > 0:
                    ret=self.collect_data(self.num_steps_prior, 1, np.inf)
                    logger.record_tabular('prior_return', ret)
                # # collect some trajectories with z ~ posterior
                # if self.num_steps_posterior > 0:
                #     self.collect_data(self.num_steps_posterior, 1, self.update_post_train)
                # even if encoder is trained only on samples from the prior, the policy needs to learn to handle z ~ posterior
                if self.num_extra_rl_steps_posterior > 0:
                    if self.phase=='explore':
                        ret=self.collect_data_expl(self.num_extra_rl_steps_posterior, 1, self.update_post_train, add_to_enc_buffer=False)
                    elif self.phase=='context':
                        ret=self.collect_data(self.num_extra_rl_steps_posterior, 1, self.update_post_train, add_to_enc_buffer=False)
                    
                    logger.record_tabular('non_sta_return', ret)
                    logger.record_tabular('task_idx', self.true_idx)
                    
                    if self.phase=='explore':
                        self.explore_ret=self.explore_ret[-(self.keep_len-1):]+[ret]
                        self.explore_ret_step=self.explore_ret_step[-(self.keep_len-1):]+[n_epi]
                        self.max_explore=max(self.max_explore,ret)
                    elif self.phase=='context':
                        self.context_ret=self.context_ret[-(self.keep_len-1):]+[ret]
                        self.context_ret_step=self.context_ret_step[-(self.keep_len-1):]+[n_epi]
                        self.max_context=max(self.max_context,ret)
                        
                    logger.record_tabular('explore_ret', self.explore_ret)
                    logger.record_tabular('explore_ret_step', self.explore_ret_step)
                    logger.record_tabular('context_ret', self.context_ret)
                    logger.record_tabular('context_ret_step', self.context_ret_step)
                
                logger.dump_tabular(with_prefix=False, with_timestamp=False)

            # Sample train tasks and compute gradient updates on parameters.
            if True:#because context-based dynamics is trained here#self.phase_change or self.phase=='context':
                for train_step in range(self.num_train_steps_per_itr):
                    indices = np.random.choice(avail_idx, self.meta_batch)
                    if self.phase_change or self.phase=='explore' or random.random()>self.relabel_percentage:
                        self._do_training(indices)
                    else:
                    # if self.relabel and not self.phase_change and self.phase=='context':
                        self._do_training(indices,relabel_avail_idx=np.random.choice(avail_idx, len(indices),replace=False) if len(avail_idx)>len(indices) else avail_idx)
                    
                    self._n_train_steps_total += 1
                gt.stamp('train')
            if self.phase_change or self.phase=='explore':
                if self.reset_expl and n_epi%self.reset_freq==0:
                    if (n_epi//self.reset_freq)%2==0:
                        self.reset_expl1()
                    else:
                        self.reset_expl2()
                # Sample train tasks and compute gradient updates on parameters.
                for train_step in range(self.num_train_steps_per_itr):
                    # based on current list
                    indices = np.random.choice(current_list, self.meta_batch,replace=False) if len(current_list)>self.meta_batch else current_list
                    
                    #based on recent list
                    # recent_list=recent_list[-100:]
                    # indices = np.random.choice(recent_list, self.meta_batch,replace=False) if len(current_list)>self.meta_batch else current_list
                    
                    if random.random()<self.relabel_percentage:
                        self._do_training_expl(indices,relabel_avail_idx=np.random.choice(avail_idx, len(indices),replace=False) if len(avail_idx)>len(indices) else avail_idx)
                    else:
                        self._do_training_expl(indices)
                    # self._n_train_steps_total += 1
                gt.stamp('train')

            self.training_mode(False)

            # eval
            # self._try_to_eval(it_)
            # gt.stamp('eval')

            self._end_epoch()

    def pretrain(self):
        """
        Do anything before the main training phase.
        """
        pass

    def collect_data(self, num_samples, resample_z_rate, update_posterior_rate, add_to_enc_buffer=True):
        '''
        get trajectories from current env in batch mode with given policy
        collect complete trajectories until the number of collected transitions >= num_samples

        :param agent: policy to rollout
        :param num_samples: total number of transitions to sample
        :param resample_z_rate: how often to resample latent context z (in units of trajectories)
        :param update_posterior_rate: how often to update q(z | c) from which z is sampled (in units of trajectories)
        :param add_to_enc_buffer: whether to add collected data to encoder replay buffer
        '''
        # start from the prior
        self.agent.clear_z()

        num_transitions = 0
        all_paths=[]
        while num_transitions < num_samples:
            paths, n_samples = self.sampler.obtain_samples(max_samples=num_samples - num_transitions,
                                                                max_trajs=update_posterior_rate,
                                                                accum_context=False,
                                                                resample=resample_z_rate)
            num_transitions += n_samples
            self.replay_buffer.add_paths(self.task_idx, paths)
            if add_to_enc_buffer:
                self.enc_replay_buffer.add_paths(self.task_idx, paths)
            if update_posterior_rate != np.inf:
                context = self.sample_context(self.task_idx)
                self.agent.infer_posterior(context)
            
            # print('db',paths,len(paths))
            if update_posterior_rate==1:
                assert len(paths)==1
            all_paths+=paths
        self._n_env_steps_total += num_transitions
        gt.stamp('sample')
        
        return eval_util.get_average_returns(all_paths)
    
    def collect_data_expl(self, num_samples, resample_z_rate, update_posterior_rate, add_to_enc_buffer=True):
        '''
        get trajectories from current env in batch mode with given policy
        collect complete trajectories until the number of collected transitions >= num_samples

        :param agent: policy to rollout
        :param num_samples: total number of transitions to sample
        :param resample_z_rate: how often to resample latent context z (in units of trajectories)
        :param update_posterior_rate: how often to update q(z | c) from which z is sampled (in units of trajectories)
        :param add_to_enc_buffer: whether to add collected data to encoder replay buffer
        '''
        # start from the prior
        self.expl_agent1.clear_z()
        self.expl_agent2.clear_z()

        num_transitions = 0
        all_paths=[]
        while num_transitions < num_samples:
            paths, n_samples = self.sampler_expl.obtain_samples(max_samples=num_samples - num_transitions,
                                                                max_trajs=update_posterior_rate,
                                                                accum_context=False,
                                                                resample=resample_z_rate)
            num_transitions += n_samples
            self.replay_buffer.add_paths(self.task_idx, paths)
            if add_to_enc_buffer:
                self.enc_replay_buffer.add_paths(self.task_idx, paths)
            # if update_posterior_rate != np.inf:
            #     context = self.sample_context(self.task_idx)
            #     self.expl_agent.infer_posterior(context)
            
            # print('db',paths,len(paths))
            if update_posterior_rate==1:
                assert len(paths)==1
            all_paths+=paths
        self._n_env_steps_total += num_transitions
        gt.stamp('sample')
        
        return eval_util.get_average_returns(all_paths)

    def _try_to_eval(self, epoch):
        logger.save_extra_data(self.get_extra_data_to_save(epoch))
        if self._can_evaluate():
            self.evaluate(epoch)

            params = self.get_epoch_snapshot(epoch)
            logger.save_itr_params(epoch, params)
            table_keys = logger.get_table_key_set()
            if self._old_table_keys is not None:
                assert table_keys == self._old_table_keys, (
                    "Table keys cannot change from iteration to iteration."
                )
            self._old_table_keys = table_keys

            logger.record_tabular(
                "Number of train steps total",
                self._n_train_steps_total,
            )
            logger.record_tabular(
                "Number of env steps total",
                self._n_env_steps_total,
            )
            logger.record_tabular(
                "Number of rollouts total",
                self._n_rollouts_total,
            )

            times_itrs = gt.get_times().stamps.itrs
            train_time = times_itrs['train'][-1]
            sample_time = times_itrs['sample'][-1]
            eval_time = times_itrs['eval'][-1] if epoch > 0 else 0
            epoch_time = train_time + sample_time + eval_time
            total_time = gt.get_times().total

            logger.record_tabular('Train Time (s)', train_time)
            logger.record_tabular('(Previous) Eval Time (s)', eval_time)
            logger.record_tabular('Sample Time (s)', sample_time)
            logger.record_tabular('Epoch Time (s)', epoch_time)
            logger.record_tabular('Total Train Time (s)', total_time)

            logger.record_tabular("Epoch", epoch)
            logger.dump_tabular(with_prefix=False, with_timestamp=False)
        else:
            logger.log("Skipping eval for now.")

    def _can_evaluate(self):
        """
        One annoying thing about the logger table is that the keys at each
        iteration need to be the exact same. So unless you can compute
        everything, skip evaluation.

        A common example for why you might want to skip evaluation is that at
        the beginning of training, you may not have enough data for a
        validation and training set.

        :return:
        """
        # eval collects its own context, so can eval any time
        return True

    def _can_train(self):
        return all([self.replay_buffer.num_steps_can_sample(idx) >= self.batch_size for idx in self.train_tasks])

    def _get_action_and_info(self, agent, observation):
        """
        Get an action to take in the environment.
        :param observation:
        :return:
        """
        agent.set_num_steps_total(self._n_env_steps_total)
        return agent.get_action(observation,)

    def _start_epoch(self, epoch):
        self._epoch_start_time = time.time()
        self._exploration_paths = []
        self._do_train_time = 0
        logger.push_prefix('Iteration #%d | ' % epoch)

    def _end_epoch(self):
        logger.log("Epoch Duration: {0}".format(
            time.time() - self._epoch_start_time
        ))
        # logger.log("Started Training: {0}".format(self._can_train()))
        logger.pop_prefix()

    ##### Snapshotting utils #####
    def get_epoch_snapshot(self, epoch):
        data_to_save = dict(
            epoch=epoch,
            exploration_policy=self.exploration_policy,
        )
        if self.save_environment:
            data_to_save['env'] = self.training_env
        return data_to_save

    def get_extra_data_to_save(self, epoch):
        """
        Save things that shouldn't be saved every snapshot but rather
        overwritten every time.
        :param epoch:
        :return:
        """
        if self.render:
            self.training_env.render(close=True)
        data_to_save = dict(
            epoch=epoch,
        )
        if self.save_environment:
            data_to_save['env'] = self.training_env
        if self.save_replay_buffer:
            data_to_save['replay_buffer'] = self.replay_buffer
        if self.save_algorithm:
            data_to_save['algorithm'] = self
        return data_to_save

    def collect_paths(self, idx, epoch, run):
        self.task_idx = idx
        self.reset_task(self.true_idx)#self.env.reset_task(self.true_idx)

        self.agent.clear_z()
        paths = []
        num_transitions = 0
        num_trajs = 0
        while num_transitions < self.num_steps_per_eval:
            path, num = self.sampler.obtain_samples(deterministic=self.eval_deterministic, max_samples=self.num_steps_per_eval - num_transitions, max_trajs=1, accum_context=True)
            paths += path
            num_transitions += num
            num_trajs += 1
            if num_trajs >= self.num_exp_traj_eval:
                self.agent.infer_posterior(self.agent.context)

        if self.sparse_rewards:
            for p in paths:
                sparse_rewards = np.stack(e['sparse_reward'] for e in p['env_infos']).reshape(-1, 1)
                p['rewards'] = sparse_rewards

        goal = self.env._goal
        for path in paths:
            path['goal'] = goal # goal

        # save the paths for visualization, only useful for point mass
        if self.dump_eval_paths:
            logger.save_extra_data(paths, path='eval_trajectories/task{}-epoch{}-run{}'.format(idx, epoch, run))

        return paths

    def _do_eval(self, indices, epoch):
        final_returns = []
        online_returns = []
        for idx in indices:
            all_rets = []
            for r in range(self.num_evals):
                paths = self.collect_paths(idx, epoch, r)
                all_rets.append([eval_util.get_average_returns([p]) for p in paths])
            final_returns.append(np.mean([a[-1] for a in all_rets]))
            # record online returns for the first n trajectories
            n = min([len(a) for a in all_rets])
            all_rets = [a[:n] for a in all_rets]
            all_rets = np.mean(np.stack(all_rets), axis=0) # avg return per nth rollout
            online_returns.append(all_rets)
        n = min([len(t) for t in online_returns])
        online_returns = [t[:n] for t in online_returns]
        return final_returns, online_returns

    def evaluate(self, epoch):
        if self.eval_statistics is None:
            self.eval_statistics = OrderedDict()

        ### sample trajectories from prior for debugging / visualization
        if self.dump_eval_paths:
            # 100 arbitrarily chosen for visualizations of point_robot trajectories
            # just want stochasticity of z, not the policy
            self.agent.clear_z()
            prior_paths, _ = self.sampler.obtain_samples(deterministic=self.eval_deterministic, max_samples=self.max_path_length * 20,
                                                        accum_context=False,
                                                        resample=1)
            logger.save_extra_data(prior_paths, path='eval_trajectories/prior-epoch{}'.format(epoch))

        ### train tasks
        # eval on a subset of train tasks for speed
        indices = np.random.choice(self.train_tasks, len(self.eval_tasks))
        eval_util.dprint('evaluating on {} train tasks'.format(len(indices)))
        ### eval train tasks with posterior sampled from the training replay buffer
        train_returns = []
        for idx in indices:
            self.task_idx = idx
            self.reset_task(self.true_idx)#self.env.reset_task(self.true_idx)
            paths = []
            for _ in range(self.num_steps_per_eval // self.max_path_length):
                context = self.sample_context(idx)
                self.agent.infer_posterior(context)
                p, _ = self.sampler.obtain_samples(deterministic=self.eval_deterministic, max_samples=self.max_path_length,
                                                        accum_context=False,
                                                        max_trajs=1,
                                                        resample=np.inf)
                paths += p

            if self.sparse_rewards:
                for p in paths:
                    sparse_rewards = np.stack(e['sparse_reward'] for e in p['env_infos']).reshape(-1, 1)
                    p['rewards'] = sparse_rewards

            train_returns.append(eval_util.get_average_returns(paths))
        train_returns = np.mean(train_returns)
        ### eval train tasks with on-policy data to match eval of test tasks
        train_final_returns, train_online_returns = self._do_eval(indices, epoch)
        eval_util.dprint('train online returns')
        eval_util.dprint(train_online_returns)

        ### test tasks
        eval_util.dprint('evaluating on {} test tasks'.format(len(self.eval_tasks)))
        test_final_returns, test_online_returns = self._do_eval(self.eval_tasks, epoch)
        eval_util.dprint('test online returns')
        eval_util.dprint(test_online_returns)

        # save the final posterior
        self.agent.log_diagnostics(self.eval_statistics)

        if hasattr(self.env, "log_diagnostics"):
            self.env.log_diagnostics(paths, prefix=None)

        avg_train_return = np.mean(train_final_returns)
        avg_test_return = np.mean(test_final_returns)
        avg_train_online_return = np.mean(np.stack(train_online_returns), axis=0)
        avg_test_online_return = np.mean(np.stack(test_online_returns), axis=0)
        self.eval_statistics['AverageTrainReturn_all_train_tasks'] = train_returns
        self.eval_statistics['AverageReturn_all_train_tasks'] = avg_train_return
        self.eval_statistics['AverageReturn_all_test_tasks'] = avg_test_return
        # logger.save_extra_data(avg_train_online_return, path='online-train-epoch{}'.format(epoch))
        # logger.save_extra_data(avg_test_online_return, path='online-test-epoch{}'.format(epoch))

        for key, value in self.eval_statistics.items():
            logger.record_tabular(key, value)
        self.eval_statistics = None

        if self.render_eval_paths:
            self.env.render_paths(paths)

        if self.plotter:
            self.plotter.draw()

    @abc.abstractmethod
    def training_mode(self, mode):
        """
        Set training mode to `mode`.
        :param mode: If True, training will happen (e.g. set the dropout
        probabilities to not all ones).
        """
        pass

    @abc.abstractmethod
    def _do_training(self):
        """
        Perform some update, e.g. perform one gradient step.
        :return:
        """
        pass

