
# -- python imports --
import numpy as np
from pathlib import Path

# -- this package --
import vnlb

# -- helper imports --
from vnlb.testing.data_loader import load_dataset
from vnlb.testing.file_io import save_images
from vnlb.exps.basic import format_exp_config,create_experiment_grid

# -- weights and biases --
import wandb

def run_exp():

    exps,order = create_experiment_grid()
    for e,exp in enumerate(exps):

        exp.use_clean = True
        exp.use_imread = [False,False]
        exp.aggreBoost = [False,False]

        # -- format exp --
        config,del_fields = format_exp_config(exp)

        # -- init wandb --
        # wandb_run = wandb.init(
        #     project="vnlb-assess-clean",
        #     notes="how do similar patches impact denoising result?",
        #     tags=["explore", "vnlb", "similar search"],
        #     config=exp,
        #     reinit = True
        # )
        wandb_run = None

        # -- exec --
        exec_experiment(wandb_run,config,del_fields)

        # -- finish --
        wandb_run.finish()

def exec_experiment(wandb_run,config,del_fields):

    # -- load & denoise a video --
    print("Running example script.")
    vnlb.check_omp_num_threads()

    # -- load clean --
    nframes = config.nframes
    clean = load_dataset("davis_64x64",vnlb=False)[0]['clean'].copy()[:nframes]

    # -- add noise --
    np.random.seed(config.random_seed)
    sigma = config['sigma'][0]
    noisy = np.random.normal(clean.copy(),scale=sigma)

    # -- clean some fields for vnlb params --
    use_clean = config['use_clean']
    for field in del_fields:
        del config[field]

    # -- init params --
    params = vnlb.swig.setVnlbParams(noisy.shape,sigma,params=config)

    # -- TV-L1 Optical Flow --
    fflow,bflow = vnlb.swig.runPyFlow(noisy,sigma)

    # -- Video Non-Local Bayes --
    if use_clean: img_clean = clean
    else: img_clean = None
    result = vnlb.cpu.runPythonVnlb(noisy,sigma,{},params,img_clean)

    # -- unpack --
    denoised = result['denoised']
    basic = result['basic']

    # -- compute denoising quality --
    psnrs = vnlb.utils.compute_psnrs(clean,denoised)
    psnrs_basic = vnlb.utils.compute_psnrs(clean,basic)

    # -- compare with original  --
    noisy_psnrs = vnlb.utils.compute_psnrs(clean,noisy)

    # -- report outputs --
    print("Denoised PSNRs:")
    print(psnrs)

    print("Basic PSNRs:")
    print(psnrs_basic)

    print("Starting PSNRs:")
    print(noisy_psnrs)

    # -- log to wandb --
    wandb.log({f"ave_psnrs": np.mean(psnrs).item(),
               f"ave_psnrs_basic": np.mean(psnrs_basic).item()})
    for t in range(len(psnrs)):
        wandb.log({f"psnrs_{t}": psnrs[t],
                   f"psnrs_basic_{t}": psnrs_basic[t]})

if __name__ == "__main__":
    run_exp()
