import java.util.*;
import java.io.*;

class InfiniteExemplarModelExperiment {
    
    /*****************************************************************/
    /*                     Instance Variables                        */
    /*****************************************************************/

    // Need some subclasses that implement an inference algorithm.
    public ArrayList<InfiniteExemplarModel> models_;
    public ArrayList<Parameters> parameters_;   // Algorithm parameters

    // Data parameters
    public int numVars_;
    public int dimension_;
    public String dataset_;
    public double sigma_;
    public double g0Sigma_;
    public int instance_;
    public int[] trueLabels_;
    public double diagonalDelta_;
    public double sFactor_;
    public double alpha_;

    /*****************************************************************/
    /*                            Methods                            */
    /*****************************************************************/

    /**
     * By default, read parameters from "experiment_parameters.txt".
     */
    public InfiniteExemplarModelExperiment(String parameterFile) {
	models_ = new ArrayList<InfiniteExemplarModel>();
	parameters_ = new ArrayList<Parameters>();
	InitializeFromFile(parameterFile);
    }

    /**
     * Actually run experiments
     */
    public void RunExperiments() {
	// Initialize then run
	for (int i = 0; i < models_.size(); i++) {
	    models_.get(i).LogToSystemErr();
	    models_.get(i).MAPInference();
	}
    }

    /**
     * Load settings from an experiment settings file.
     */
    protected void InitializeFromFile(String filename) {
	numVars_ = -1;
	diagonalDelta_ = 0;
	sFactor_ = 1;
	alpha_ = 1;

	try {
	    BufferedReader input = 
		new BufferedReader(new FileReader(filename));

	    // There's just one line to read
	    String line ;
	    while ( (line = input.readLine()) != null) {
		String[] entries = line.trim().split("\\s+");		
		
		// Allow blank lines
		if (entries.length == 0) continue;
		
		// Otherwise, there must be at least 2 entries per line
		assert(entries.length >= 2);

		// Data set parameters
		if (entries[0].equals("dataset")) {
		    dataset_ = entries[1];
		} else if (entries[0].equals("sigma")) {
		    sigma_ = Double.parseDouble(entries[1]);
		} else if (entries[0].equals("numVars")) {
		    numVars_ = Integer.parseInt(entries[1]);
		} else if (entries[0].equals("dimension")) {
		    dimension_ = Integer.parseInt(entries[1]);
		} else if (entries[0].equals("instance")) {
		    instance_ = Integer.parseInt(entries[1]);
		} else if (entries[0].equals("diagonalDelta")) {
		    diagonalDelta_ = Double.parseDouble(entries[1]);
		    System.err.println("diagonalDelta = " + diagonalDelta_);
		} else if (entries[0].equals("sFactor")) {
		    sFactor_ = Double.parseDouble(entries[1]);
		    System.err.println("sfactor = " + sFactor_);
		} else if (entries[0].equals("alpha")) {
		    alpha_ = Double.parseDouble(entries[1]);
		    System.err.println("alpha = " + alpha_);

		} else if (entries[0].equals("algorithmFile")) {
		    // Add an algorithm to the set of experiments.  There needs to be
		    // at least one in order for this to do anything.
		    assert numVars_ != -1 : 
		    "numVars must come before algorithms in " + filename;
		    
		    // Add a new model to the experiment
		    InfiniteExemplarModel m = null;
		    Parameters p = null;
		    if (entries[1].equals("DPAP")) {
			m = new DPAffinityPropagation(numVars_);
			p = new DPAPParameters(entries[2]);
		    } else if (entries[1].equals("AP")) {
			m = new AffinityPropagation(numVars_);
			System.err.println("Don't have APParameters written yet");
			System.exit(1);
		    } else if (entries[1].equals("MG")) {
			m = new MaximizingGibbsSampler(numVars_);
			p = new MGParameters(entries[2]);
		    } else {
			assert false : "Unknown algorithm: " + entries[1];
		    }
		    m.LoadParametersFromFile(entries[2]);
		    m.LoadSimilaritiesFromFile(SimilaritiesFilename());
		    m.alpha_ = alpha_;
		    //m.AddToDiagonal(Math.log(alpha_));
		    m.AddToDiagonal(diagonalDelta_);
		    m.MultiplySimilarities(sFactor_);
		    m.LoadTrueLabelsFromFile(TrueLabelsFilename());
		    models_.add(m);
		    parameters_.add(p);
		} else {
		    System.err.println("Warning: unknown parameter " +
				       entries[0] + ", value = " + entries[1]);
		}
	    }
	}
	catch (Exception ex){
	    ex.printStackTrace();
	    System.exit(1);
	}
    }

    /**
     * Filename portion for dataset-specific parameters
     */
    public String BaseFilename() {
	String filename = "";
	int instance = 0;

	filename += dataset_ + "/";
	filename += numVars_ + "_vars/";
	filename += "dim_" + dimension_ + "/";
	filename += "sigma" + sigma_ +"/";
	filename += instance_;

	return filename;
	// Never overwrite results, even if we have the same
	// settings for all parameters.
	// while (true) {
	//    File f = new File(fullName + instance + ".txt");
	//    if (!f.exists()) break;
	//}

    }

    public String ResultsFilename() {
	return "results/" + BaseFilename() + ".txt";
    }

    public String PointsFilename() {
	return "data/" + BaseFilename() + "_points.txt";
    }

    public String SimilaritiesFilename() {
	return "data/" + BaseFilename() + "_similarities.txt";
    }

    public String TrueLabelsFilename() {
	return "data/" + BaseFilename() + "_labels.txt";
    }

    public void WriteResultsToCSV(String filename) {
	try {
	    FileWriter fstream = new FileWriter(filename);
	    BufferedWriter fout = new BufferedWriter(fstream);
	    
	    for (int i =-1; i < models_.size(); i++) {
		// Write out basic parameters... all need to be numbers so MATLAB
		// will read this eventually.
		fout.write(numVars_ + "," + sigma_ + "," + instance_ + "," + i);

		// Write the full assignment out
		int[] assignment;
		boolean converged;
		if (i == -1) {
		    converged = models_.get(0).MessagesConverged();
		    assignment = models_.get(0).TrueLabels();
		} else {
		    converged = models_.get(i).MessagesConverged();
		    assignment = models_.get(i).CurrentAssignments();
		}
		if (converged) {
		    fout.write(",1");
		} else {
		    fout.write(",-1");
		}
		for (int j = 0; j < numVars_; j++) {
		    fout.write("," + assignment[j]);
		}
		fout.write("\n");
	    }

	    fout.close();
	}
	catch (Exception e){
	    System.err.println("Error: " + e.getMessage());
	}
    }

    /**
     * Print a row of a spreadsheet to standard out.
     */
    public void DisplayResults() {
	System.out.println("NumVars,Sigma,Instance,ShortName,NumClusters," +
			   "LogProb,GenerativeLogProb,RandIndex,Comments");
	for (int i = 0; i < models_.size(); i++) {
	    InfiniteExemplarModel m = models_.get(i);
	    int[] assignment = m.CurrentAssignments();	    
	    double randIndex = m.RandIndex(assignment, m.TrueLabels());
	    double logProb = m.LogProb(assignment);
	    double generativeLogProb = m.LogProb(m.TrueLabels());
	    System.out.println(numVars_ + "," + sigma_ + "," + instance_ + "," + 
			       m.ShortName() + "," + m.CurrentExemplars().size() + ","
			       + logProb + "," + randIndex + "," + generativeLogProb + 
			       "," + parameters_.get(i).FileName());
	}
    }


    /*****************************************************************/
    /*                              Main                             */
    /*****************************************************************/

    public static void main(String args[]) {
	InfiniteExemplarModelExperiment e = 
	    new InfiniteExemplarModelExperiment(args[0]);

	e.RunExperiments();

	e.WriteResultsToCSV(args[1]);

	e.DisplayResults();

    }
    

}
