package common;

public final class Date {

	private final long serialNumber;

	public Date() {
		serialNumber = 0;
	}

	// constructor taking a serial number
	public Date(long serialNumber) {
		this.serialNumber = serialNumber;
	}

	// constructor taking Day, Month, Year
	public Date(int day, Month month, int year) {
		this(day, month.value(), year);

	}

	// constructor taking month as short value
	public Date(int day, int month, int year) {
		boolean isLeap = isLeap(year);
		int offset = monthOffset(month, isLeap);

		serialNumber = day + offset + yearOffset(year);

	}
	
	public Date(Date date) {
		this(date.getSerialNumber());
	}

	public static boolean isLeap(int year) {
		boolean[] yearIsLeap = {
				// 1900-1909
				true,
				false,
				false,
				false,
				true,
				false,
				false,
				false,
				true,
				false,
				// 1910-1919
				false,
				false,
				true,
				false,
				false,
				false,
				true,
				false,
				false,
				false,
				// 1920-1929
				true,
				false,
				false,
				false,
				true,
				false,
				false,
				false,
				true,
				false,
				// 1930-1939
				false,
				false,
				true,
				false,
				false,
				false,
				true,
				false,
				false,
				false,
				// 1940-1949
				true,
				false,
				false,
				false,
				true,
				false,
				false,
				false,
				true,
				false,
				// 1950-1959
				false,
				false,
				true,
				false,
				false,
				false,
				true,
				false,
				false,
				false,
				// 1960-1969
				true,
				false,
				false,
				false,
				true,
				false,
				false,
				false,
				true,
				false,
				// 1970-1979
				false,
				false,
				true,
				false,
				false,
				false,
				true,
				false,
				false,
				false,
				// 1980-1989
				true,
				false,
				false,
				false,
				true,
				false,
				false,
				false,
				true,
				false,
				// 1990-1999
				false,
				false,
				true,
				false,
				false,
				false,
				true,
				false,
				false,
				false,
				// 2000-2009
				true,
				false,
				false,
				false,
				true,
				false,
				false,
				false,
				true,
				false,
				// 2010-2019
				false, false,
				true,
				false,
				false,
				false,
				true,
				false,
				false,
				false,
				// 2020-2029
				true, false, false,
				false,
				true,
				false,
				false,
				false,
				true,
				false,
				// 2030-2039
				false, false, true, false,
				false,
				false,
				true,
				false,
				false,
				false,
				// 2040-2049
				true, false, false, false, true,
				false,
				false,
				false,
				true,
				false,
				// 2050-2059
				false, false, true, false, false, false,
				true,
				false,
				false,
				false,
				// 2060-2069
				true, false, false, false, true, false, false,
				false,
				true,
				false,
				// 2070-2079
				false, false, true, false, false, false, true, false,
				false,
				false,
				// 2080-2089
				true, false, false, false, true, false, false, false, true,
				false,
				// 2090-2099
				false, false, true, false, false, false, true, false, false,
				false,
				// 2100
				false };
		return yearIsLeap[year - 1900];
	}

	public int monthLength(int month, boolean isLeap) {
		int[] monthLength = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
		int[] monthLengthLeap = { 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30,
				31 };

		return (isLeap ? monthLengthLeap[month - 1] : monthLength[month - 1]);
	}

	public static int monthOffset(int month, boolean isLeap) {
		int[] MonthOffset = { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304,
				334, 365 };
		int[] MonthLeapOffset = { 0, 31, 60, 91, 121, 152, 182, 213, 244, 274,
				305, 335, 366 };

		return (isLeap ? MonthLeapOffset[month - 1] : MonthOffset[month - 1]);
	}

	public static int yearOffset(int year) {
		int[] yearOffset = {
				// 1900-1909
				0,
				366,
				731,
				1096,
				1461,
				1827,
				2192,
				2557,
				2922,
				3288,
				// 1910-1919
				3653,
				4018,
				4383,
				4749,
				5114,
				5479,
				5844,
				6210,
				6575,
				6940,
				// 1920-1929
				7305,
				7671,
				8036,
				8401,
				8766,
				9132,
				9497,
				9862,
				10227,
				10593,
				// 1930-1939
				10958,
				11323,
				11688,
				12054,
				12419,
				12784,
				13149,
				13515,
				13880,
				14245,
				// 1940-1949
				14610,
				14976,
				15341,
				15706,
				16071,
				16437,
				16802,
				17167,
				17532,
				17898,
				// 1950-1959
				18263,
				18628,
				18993,
				19359,
				19724,
				20089,
				20454,
				20820,
				21185,
				21550,
				// 1960-1969
				21915,
				22281,
				22646,
				23011,
				23376,
				23742,
				24107,
				24472,
				24837,
				25203,
				// 1970-1979
				25568,
				25933,
				26298,
				26664,
				27029,
				27394,
				27759,
				28125,
				28490,
				28855,
				// 1980-1989
				29220,
				29586,
				29951,
				30316,
				30681,
				31047,
				31412,
				31777,
				32142,
				32508,
				// 1990-1999
				32873,
				33238,
				33603,
				33969,
				34334,
				34699,
				35064,
				35430,
				35795,
				36160,
				// 2000-2009
				36525,
				36891,
				37256,
				37621,
				37986,
				38352,
				38717,
				39082,
				39447,
				39813,
				// 2010-2019
				40178, 40543,
				40908,
				41274,
				41639,
				42004,
				42369,
				42735,
				43100,
				43465,
				// 2020-2029
				43830, 44196, 44561,
				44926,
				45291,
				45657,
				46022,
				46387,
				46752,
				47118,
				// 2030-2039
				47483, 47848, 48213, 48579,
				48944,
				49309,
				49674,
				50040,
				50405,
				50770,
				// 2040-2049
				51135, 51501, 51866, 52231, 52596,
				52962,
				53327,
				53692,
				54057,
				54423,
				// 2050-2059
				54788, 55153, 55518, 55884, 56249, 56614,
				56979,
				57345,
				57710,
				58075,
				// 2060-2069
				58440, 58806, 59171, 59536, 59901, 60267, 60632,
				60997,
				61362,
				61728,
				// 2070-2079
				62093, 62458, 62823, 63189, 63554, 63919, 64284, 64650,
				65015,
				65380,
				// 2080-2089
				65745, 66111, 66476, 66841, 67206, 67572, 67937, 68302, 68667,
				69033,
				// 2090-2099
				69398, 69763, 70128, 70494, 70859, 71224, 71589, 71955, 72320,
				72685,
				// 2100
				73050 };
		return yearOffset[year - 1900];
	}

	public Weekday weekday() {
		int weekday = (int) (serialNumber % 7);
		switch (weekday) {
		case 0:
			return Weekday.Saturday;
		case 1:
			return Weekday.Sunday;
		case 2:
			return Weekday.Monday;
		case 3:
			return Weekday.Tuesday;
		case 4:
			return Weekday.Wednesday;
		case 5:
			return Weekday.Thursday;
		case 6:
			return Weekday.Friday;
		default:
			return null;
		}
	}

	public int year() {
		int year = (int) (1900 + (serialNumber / 365));
		if (serialNumber <= yearOffset(year))
			year--;
		return year;
	}

	public int dayOfYear() {
		return (int) (serialNumber - yearOffset(year()));
	}

	public int dayOfMonth() {
		return dayOfYear() - monthOffset(month().value(), isLeap(year()));
	}

	public Month month() {
		int day = dayOfYear();
		int month = day / 30 + 1;
		boolean leap = isLeap(year());

		while (day <= monthOffset(month, leap))
			month--;
		while (day > monthOffset(month + 1, leap))
			month++;

		switch (month) {
		case 1:
			return Month.January;
		case 2:
			return Month.February;
		case 3:
			return Month.March;
		case 4:
			return Month.April;
		case 5:
			return Month.May;
		case 6:
			return Month.June;
		case 7:
			return Month.July;
		case 8:
			return Month.August;
		case 9:
			return Month.September;
		case 10:
			return Month.October;
		case 11:
			return Month.November;
		case 12:
			return Month.December;
		default:
			return null;
		}
	}

	public long getSerialNumber() {
		return serialNumber;
	}

	public boolean isEndOfMonth() {
		return (this.dayOfMonth() == monthLength(this.month().value(),
				isLeap(this.year())));
	}

	public Date lastDayOfMonth() {
		Month month = this.month();
		int year = this.year();

		int lastDayOfMonth = monthLength(month.value(), isLeap(year));

		return new Date(lastDayOfMonth, month, year);

	}

	public Date increment() {
		return new Date (serialNumber + 1);
	}

	public Date decrement() {
		return new Date (serialNumber - 1);
	}

	/** advance by a certain number of time units */
	public Date advance(int n, TimeUnit units) {
		switch (units) {
		case Days:
			return new Date (serialNumber + n);
		case Weeks:
			return new Date (serialNumber + 7L * n);
		case Months:
			int d1 = dayOfMonth();
			int m = month().value() + n;
			int y = year();
			while (m > 12) {
				m -= 12;
				y += 1;
			}
			while (m < 1) {
				m += 12;
				y -= 1;
			}
			int length = monthLength(m, isLeap(y));
			if (d1 > length)
				d1 = length;
			return new Date(d1, m, y);
		case Years:
			int d2 = dayOfMonth();
			Month m2 = month();
			int y2 = year() + n;
			if (d2 == 29 && m2 == Month.February && !isLeap(y2))
				d2 = 28;
			return new Date(d2, m2, y2);
		default:
			return new Date();
		}
	}

	public String toString() {
		return dayOfMonth() + "/" + month().value() + "/" + year();
	}

	public boolean isEqual(Date date) {
		if (serialNumber == date.getSerialNumber())
			return true;
		else
			return false;
	}

	/*
	 * Date class is a generic superclass that has all days as business days
	 * except for Saturdays and Sundays
	 */
	public boolean isBusinessDay() {
		if (this.weekday() == Weekday.Saturday
				|| this.weekday() == Weekday.Sunday)
			return false;
		else
			return true;
	}

	public static Date applyConvention(Date d, BusinessDayConvention convention) {
		long serialNumber = d.getSerialNumber();
		Date tmp = new Date(serialNumber);
		
		switch (convention) {

		case Preceding:
			while (!tmp.isBusinessDay()) {
				serialNumber--;
				tmp = new Date(serialNumber);
			}
			break;
		
		case ModifiedPreceding:
			while (!tmp.isBusinessDay()) {
				serialNumber--;
				tmp = new Date(serialNumber);
			}
			if(tmp.month() != d.month()) {
				serialNumber++;
				tmp = new Date(serialNumber);
				while (!tmp.isBusinessDay()) {
					serialNumber++;
					tmp = new Date(serialNumber);
				}
			}
			break;
			
		case Following:
			while (!tmp.isBusinessDay()) {
				serialNumber++;
				tmp = new Date(serialNumber);
			}
			break;
			
		case ModifiedFollowing:
			while (!tmp.isBusinessDay()) {
				serialNumber++;
				tmp = new Date(serialNumber);
			}
			if(tmp.month() != d.month()) {
				serialNumber--;
				tmp = new Date(serialNumber);
				while (!tmp.isBusinessDay()) {
					serialNumber--;
					tmp = new Date(serialNumber);
				}
			}
			break;
		}
		
		return tmp;
	}

	public double dayCount(Date d1, DayCountConvention convention) {
		switch (convention) {
		case ACT_365:
			return ((d1.getSerialNumber() - serialNumber) / 365.0);
		case ACT_360:
			return ((d1.getSerialNumber() - serialNumber) / 360.0);
		case Day30_365:
			return ((d1.dayOfMonth() + 30 - dayOfMonth() + 30 * (12
					* (d1.year() - year()) + d1.month().value() - month()
					.value())) / 365.0);
		case Day30_360:
			return ((d1.dayOfMonth() + 30 - dayOfMonth() + 30 * (12
					* (d1.year() - year()) + d1.month().value() - month()
					.value())) / 360.0);
		default:
			return ((d1.getSerialNumber() - serialNumber) / 365.0);

		}
	}

	public static Date setDateToToday() {
		java.util.GregorianCalendar calendar = new java.util.GregorianCalendar();
		int day = calendar.get(java.util.Calendar.DATE);
		int month = calendar.get(java.util.Calendar.MONTH) + 1;
		int year = calendar.get(java.util.Calendar.YEAR);

		boolean leap = isLeap(year);
		int monOffset = monthOffset(month, leap);
		long serialNumber = day + monOffset + yearOffset(year);
		return new Date(serialNumber);
	}
}
