package credit;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import rates.InterestComposition;
import rates.TypeOfRate;
import rates.YieldCurve;
import rates.YieldPoint;

import common.CachedValue;
import common.Currency;
import common.Date;
import common.Frequency;

public class CreditCurve {
	public final static int CC_MAX_NUM_SPREADS = 30;

	public final static double CC_DEFAULT_RECOVERY_RATE = 0.40;

	public final static Frequency CC_DEFAULT_FREQUENCY = Frequency.Annual;

	public final static Currency CC_DEFAULT_CURRENCY = Currency.USD;

	public final static String CC_DEFAULT_NAME = "CreditCurve";

	private CreditSpreadPoint[] spreads;

	private CachedValue[] survivalProbability;

	private CachedValue[] defaultProbability;

	private CachedValue[] swapFees;

	private double recoveryRate;

	private Currency currency;

	private Frequency frequency;

	private YieldCurve underlying;

	private YieldCurve combined;

	private String name;

	public CreditCurve() {
		spreads = new CreditSpreadPoint[CC_MAX_NUM_SPREADS];
		survivalProbability = new CachedValue[CC_MAX_NUM_SPREADS];
		defaultProbability = new CachedValue[CC_MAX_NUM_SPREADS];
		swapFees = new CachedValue[CC_MAX_NUM_SPREADS];
	}

	public CreditCurve(YieldCurve yc, CreditSpreadPoint[] csp, String name,
			double recoveryRate, Currency currency, Frequency frequency) {
		spreads = new CreditSpreadPoint[CC_MAX_NUM_SPREADS];
		survivalProbability = new CachedValue[CC_MAX_NUM_SPREADS];
		defaultProbability = new CachedValue[CC_MAX_NUM_SPREADS];
		swapFees = new CachedValue[CC_MAX_NUM_SPREADS];

		this.recoveryRate = recoveryRate;
		this.currency = currency;
		this.frequency = frequency;
		this.name = name;
		
		CreditSpreadPoint[] tmpSpreadPoints = new CreditSpreadPoint[csp.length];
		for (int i = 0; i < csp.length; i++) {
			tmpSpreadPoints[i] = new CreditSpreadPoint(csp[i]);
		}

		underlying = new YieldCurve(yc.getPointsIntheMarketCurve(), yc.getName());
		YieldCurve spreadCurve = createSpreadCurve(underlying, tmpSpreadPoints);
		combined = combineUnderlyingAndSpreads(underlying, spreadCurve);
		resampleSpread();
	}

	public CreditCurve(double flatRate, double flatSpread, String name,
			double recoveryRate, Currency currency, Frequency frequency) {
		spreads = new CreditSpreadPoint[CC_MAX_NUM_SPREADS];
		survivalProbability = new CachedValue[CC_MAX_NUM_SPREADS];
		defaultProbability = new CachedValue[CC_MAX_NUM_SPREADS];
		swapFees = new CachedValue[CC_MAX_NUM_SPREADS];

		this.recoveryRate = recoveryRate;
		this.currency = currency;
		this.frequency = frequency;
		this.name = name;

		underlying = new YieldCurve(flatRate);
		YieldCurve spreadCurve = new YieldCurve(flatSpread);
		combined = combineUnderlyingAndSpreads(underlying, spreadCurve);
		assignFlatSpread(flatSpread);
	}

	public CreditCurve(YieldCurve yc, double flatSpread, String name,
			double recoveryRate, Currency currency, Frequency frequency) {
		spreads = new CreditSpreadPoint[CC_MAX_NUM_SPREADS];
		survivalProbability = new CachedValue[CC_MAX_NUM_SPREADS];
		defaultProbability = new CachedValue[CC_MAX_NUM_SPREADS];
		swapFees = new CachedValue[CC_MAX_NUM_SPREADS];

		this.recoveryRate = recoveryRate;
		this.currency = currency;
		this.frequency = frequency;
		this.name = name;

		underlying = new YieldCurve(yc.getPointsIntheMarketCurve(), yc.getName());
		YieldCurve spreadCurve = new YieldCurve(flatSpread);
		combined = combineUnderlyingAndSpreads(underlying, spreadCurve);
		assignFlatSpread(flatSpread);
	}

	public CreditCurve(YieldPoint[] yp, CreditSpreadPoint[] csp, String name,
			double recoveryRate, Currency currency, Frequency frequency) {
		spreads = new CreditSpreadPoint[CC_MAX_NUM_SPREADS];
		survivalProbability = new CachedValue[CC_MAX_NUM_SPREADS];
		defaultProbability = new CachedValue[CC_MAX_NUM_SPREADS];
		swapFees = new CachedValue[CC_MAX_NUM_SPREADS];

		this.recoveryRate = recoveryRate;
		this.currency = currency;
		this.frequency = frequency;
		this.name = name;

		CreditSpreadPoint[] tmpSpreadPoints = new CreditSpreadPoint[csp.length];
		for (int i = 0; i < csp.length; i++) {
			tmpSpreadPoints[i] = new CreditSpreadPoint(csp[i]);
		}
		YieldPoint[] tmpYieldPoints = new YieldPoint[yp.length];
		for (int i = 0; i < yp.length; i++) {
			tmpYieldPoints[i] = new YieldPoint(yp[i]);
		}
		
		underlying = new YieldCurve(tmpYieldPoints, name);
		YieldCurve spreadCurve = createSpreadCurve(underlying, tmpSpreadPoints);
		combined = combineUnderlyingAndSpreads(underlying, spreadCurve);
		resampleSpread();
	}
	
	public CreditCurve(CreditCurve creditCurve) {
		this(creditCurve.getUnderlyingYieldCurve(), creditCurve.getCreditSpreadPoints(),
				creditCurve.getName(), creditCurve.getRecoveryRate(), creditCurve.getCurrency(), creditCurve.getFrequency());
	}

	public YieldCurve createSpreadCurve(YieldCurve underlying,
			CreditSpreadPoint[] csp) {
		
		CreditSpreadPoint[] tmpSpreadPoints = new CreditSpreadPoint[csp.length];
		for (int i = 0; i < csp.length; i++) {
			tmpSpreadPoints[i] = new CreditSpreadPoint(csp[i]);
		}
		YieldPoint[] tmpSpreads = new YieldPoint[tmpSpreadPoints.length];

		for (int i = 0; i < tmpSpreadPoints.length; i++) {
			double rate;
			if (tmpSpreadPoints[i].getSpreadType() == CreditSpreadType.Absolute) {
				rate = tmpSpreadPoints[i].getRate()
						- underlying.spotRate(tmpSpreadPoints[i].getMaturity());
			} else {
				rate = tmpSpreadPoints[i].getRate();
			}
			tmpSpreads[i] = new YieldPoint(rate, tmpSpreadPoints[i].getMaturity(), TypeOfRate.Cash);
		}
		

		YieldCurve result = new YieldCurve(tmpSpreads, "tmpspreads");

		return result;
	}

	public YieldCurve combineUnderlyingAndSpreads(YieldCurve underlying,
			YieldCurve spreadCurve) {
		YieldCurve yc = new YieldCurve(underlying);
		YieldCurve sc = new YieldCurve(spreadCurve);
		int numTermValues = 0;

		double[] underlyingMaturities = yc.getMaturitiesInTheZCBCurve();
		double[] spreadMaturities = sc.getMaturitiesInTheZCBCurve();

		double[] termValues = mergeArrays(underlyingMaturities,
				spreadMaturities);
		numTermValues = termValues.length;

		YieldPoint[] combinedyp = new YieldPoint[numTermValues];
		for (int i = 0; i < numTermValues; i++) {
			double rate;
			double maturity;
			rate = yc.spotRate(termValues[i]) + sc.spotRate(termValues[i]);
			maturity = termValues[i];
			combinedyp[i] = new YieldPoint(rate, maturity, TypeOfRate.Cash);
		}

		return new YieldCurve(combinedyp, "combinedcurve");
	}

	public void assignFlatSpread(double rate) {
		int size = spreads.length;

		for (int i = 0; i < size; i++) {
			spreads[i] = new CreditSpreadPoint(rate, i, CreditSpreadType.Relative);
		}
	}

	public void resampleSpread() {
		int size = spreads.length;

		for (int i = 0; i < size; i++) {
			spreads[i] = new CreditSpreadPoint(combined.spotRate(i) - underlying.spotRate(i), i, CreditSpreadType.Relative);
		}
	}

	public double creditSpread(double maturity) {
		return combined.spotRate(maturity) - underlying.spotRate(maturity);
	}

	public double creaditSpread(Date maturityDate) {
		Date date = new Date(maturityDate);
		return combined.spotRate(date)
				- underlying.spotRate(date);
	}

	public double timeOfCurrentSpread(double maturity) {
		return spreads[indexOfCurrentSpread(maturity)].getMaturity();
	}

	public int indexOfCurrentSpread(double maturity) {
		int i = 0;
		int result = 0;

		while (i < spreads.length && spreads[i].getMaturity() < maturity)
			i++;

		result = (i == spreads.length ? i - 1 : i);

		return result;
	}

	public double timeOfPreviousSpread(double maturity) {
		return spreads[indexOfPreviousSpread(maturity)].getMaturity();
	}

	public int indexOfPreviousSpread(double maturity) {
		int i = 0;

		while (i < spreads.length && spreads[i].getMaturity() < maturity)
			i++;
		i = (i - 1 < 0 ? 0 : i - 1);

		return i;
	}

	public double survivalProbability(double maturity) {
		double result = 0;

		if (maturity == 0)
			// at time 0 survival prob = 100%
			result = 1;
		else {
			int i = indexOfCurrentSpread(maturity);
			if ((survivalProbability[i]==null) || (!survivalProbability[i].isCached())) {
				double T = timeOfCurrentSpread(maturity);
				double Tminus = timeOfPreviousSpread(maturity);
				double Sminus = survivalProbability(Tminus);
				double q = defaultProbability(T);
				survivalProbability[i] = new CachedValue(Sminus * (1 - q));
			}
			result = survivalProbability[i].getValue();
		}
		return result;
	}

	public double cumulativeDefaultProbability(double maturity) {
		return 1 - survivalProbability(maturity);
	}

	public double swapFees(double maturity) {
		double result = 0;

		if (maturity == 0)
			result = 0;
		else {
			int i = indexOfCurrentSpread(maturity);
			if ((swapFees[i]==null) || (!swapFees[i].isCached())) {
				double T = timeOfCurrentSpread(maturity);
				double Tminus = timeOfPreviousSpread(maturity);
				double Fminus = swapFees(Tminus);
				double Sminus = survivalProbability(Tminus);
				double q = defaultProbability(T);
				double DF = underlying.discountFactor(T);
				double CS = creditSpread(T);
				double CSminus = creditSpread(Tminus);
				swapFees[i] = new CachedValue(Fminus * CS / CSminus + DF * CS * Sminus * (1 - q));
			}
			result = swapFees[i].getValue();
		}
		return result;
	}

	public double defaultProbability(double maturity) {
		double result = 0;

		if (maturity == 0)
			result = 0;
		else {
			int i = indexOfCurrentSpread(maturity);
			if ((defaultProbability[i]==null) || (!defaultProbability[i].isCached())) {
				double T = timeOfCurrentSpread(maturity);
				double Tminus = timeOfPreviousSpread(maturity);
				double Fminus = swapFees(Tminus);
				double Sminus = survivalProbability(Tminus);
				double DF = underlying.discountFactor(T);
				double Tdefault = T - (T - Tminus) / 2;
				double DFd = underlying.discountFactor(Tdefault);
				double CS = creditSpread(T);
				double CSminus = creditSpread(Tminus);
				double num = Fminus * (CS / CSminus - 1) + DF * CS * Sminus;
				double denom = Sminus * (DFd * (1 - recoveryRate) + DF * CS);
				defaultProbability[i] = new CachedValue(num / denom);
			}
			result = defaultProbability[i].getValue();
		}
		return result;
	}

	public double hazardRate(double maturity) {
		return defaultProbability(maturity);
	}

	public double spotRate(double maturity) {
		return combined.spotRate(maturity);
	}

	public double spotRate(Date maturityDate) {
		Date matDate = new Date(maturityDate);
		return combined.spotRate(matDate);
	}

	public double discountFactor(double maturity,
			InterestComposition composition) {
		return combined.discountFactor(maturity, composition);
	}

	public double discountFactor(Date maturityDate,
			InterestComposition composition) {
		Date matDate = new Date(maturityDate);
		return combined.discountFactor(matDate, composition);
	}

	public double riskyDiscountFactor(double maturity,
			InterestComposition composition) {
		double DF = underlying.discountFactor(maturity, composition);
		double RDF = DF * survivalProbability(maturity);
		return RDF;
	}

	public double forwardRate(double forwardStart, double lengthAfterStart,
			InterestComposition composition) {
		return combined
				.forwardRate(forwardStart, lengthAfterStart, composition);
	}

	public double forwardRate(Date forwardStart, Date forwardEnd,
			InterestComposition composition) {
		Date fwdStart = new Date(forwardStart);
		Date fwdEnd = new Date(forwardEnd);
		return combined.forwardRate(fwdStart, fwdEnd, composition);
	}

	public double[] getMaturitiesInTheZCBCurve() {
		return combined.getMaturitiesInTheZCBCurve();
	}

	public double[] getRatesInTheZCBCurve() {
		return combined.getRatesInTheZCBCurve();
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public double getRecoveryRate() {
		return recoveryRate;
	}

	public void setRecoveryRate(double recoveryRate) {
		this.recoveryRate = recoveryRate;
	}

	public Currency getCurrency() {
		return currency;
	}

	public void setCurrency(Currency currency) {
		this.currency = currency;
	}

	public Frequency getFrequency() {
		return frequency;
	}

	public void setFrequency(Frequency frequency) {
		this.frequency = frequency;
	}

	public YieldCurve getUnderlyingYieldCurve() {
		YieldCurve underlyingYieldCurve;
		underlyingYieldCurve = new YieldCurve(underlying.getPointsIntheMarketCurve(), underlying.getName());
		return underlyingYieldCurve;
	}
	
	public CreditSpreadPoint[] getCreditSpreadPoints() {
		CreditSpreadPoint[] tmpSpreadPoints = new CreditSpreadPoint[spreads.length];
		for (int i = 0; i < spreads.length; i++) {
			tmpSpreadPoints[i] = new CreditSpreadPoint(spreads[i]);
		}
		return tmpSpreadPoints;
	}

	// utility method to merge two arrays of doubles while removing duplicates
	private double[] mergeArrays(double[] firstarray, double[] secondarray) {
		Hashtable<Double, Double> hash = new Hashtable<Double, Double>();

		for (int i = 0; i < firstarray.length; i++)
			hash.put(new Double(firstarray[i]), new Double(firstarray[i]));
		for (int i = 0; i < secondarray.length; i++)
			hash.put(new Double(secondarray[i]), new Double(secondarray[i]));

		Enumeration enumeration = hash.keys();

		Vector<Double> doubleVector = new Vector<Double>(firstarray.length
				+ secondarray.length);
		int index = 0;
		while (enumeration.hasMoreElements()) {
			doubleVector.add((Double) enumeration.nextElement());
			index++;
		}

		double[] doubleArrayResult = new double[doubleVector.size()];

		for (int i = 0; i < doubleVector.size(); i++)
			doubleArrayResult[i] = doubleVector.get(i).doubleValue();

		return doubleArrayResult;
	}


}
