package rates;

import common.BusinessDayConvention;
import common.Date;
import common.TimeUnit;

public class SwapLeg {

	private Date[] dateSchedule;

	private double[] flowSchedule;

	/**
	 * Constructor that takes a notional and the constant value of which the
	 * notional should be decreasing every time. If endDate!=startDate +
	 * multiple of frequency then endDate is reduced until it satisfies the
	 * condition
	 */
	public SwapLeg(Date startDate, double frequency, Date endDate,
			double notional, double amortizingConstant,
			BusinessDayConvention convention) {
		int i = 1;
		long serialEnd = endDate.getSerialNumber();
		long serialCurrent = startDate.getSerialNumber();
		int size = (int) ((serialEnd - serialCurrent) * frequency / 365) + 1;
		dateSchedule = new Date[size];
		flowSchedule = new double[size];

		dateSchedule[0] = new Date(startDate);
		// adjusting to exclude weekend days: saturday and sunday
		// ideally the isBusinessDay method of Date should accept
		// as argument a calendar the allows knowing which days
		// are business days and which are not.
		while (!dateSchedule[0].isBusinessDay())
			dateSchedule[0] = dateSchedule[0].increment();
		flowSchedule[0] = 0.0;

		while (serialCurrent <= serialEnd && i < size) {
			Date current;
			int adv = (int) (365 / frequency);
			current = Date.applyConvention(startDate.advance(adv, TimeUnit.Days), convention);
			serialCurrent = current.getSerialNumber();
			flowSchedule[i] = notional - (i - 1) * amortizingConstant;
			dateSchedule[i] = current;
			while (!dateSchedule[i].isBusinessDay())
				dateSchedule[i].increment();
			i++;
		}
	}

	/**
	 * For more general swaps : takes an array of notionals for every period and
	 * an array of dates
	 */
	public SwapLeg(Date[] dates, double[] notionals) {
		dateSchedule = new Date[dates.length];
		for (int i = 0; i < dates.length; i++) {
			dateSchedule[i] = new Date(dates[i]);
		}
		
		flowSchedule = new double[notionals.length];
		for (int i = 0; i < notionals.length; i++) {
			flowSchedule[i] = notionals[i];
		}
	}

	/** Return the size of the arrays */
	public int getSize() {
		return dateSchedule.length;
	}

	/** Return dates array */
	public Date[] getDates() {
		Date[] tmpDateSchedule = new Date[dateSchedule.length];
		for (int i = 0; i < dateSchedule.length; i++) {
			tmpDateSchedule[i] = new Date(dateSchedule[i]);
		}
		return tmpDateSchedule;
	}

	/** Return amounts array */
	public double[] getAmounts() {
		double[] tmpAmounts = new double[flowSchedule.length];
		for (int i = 0; i < flowSchedule.length; i++) {
			tmpAmounts[i] = flowSchedule[i];
		}
		return tmpAmounts;
	}
}
