package rates;

import java.util.Arrays;
import java.util.Vector;

import Jama.Matrix;

import common.Date;
import common.DayCountConvention;

public class YieldCurve {
	public static final double defaultshiftfactorForShortRate = 0.0001; // 1 bp

	public static final int YC_DEFAULT_NUBMER_POINTS = 15;

	public static final int YC_MAX_NUMBER_POINTS = 50;

	private YieldPoint[] marketRates;

	private YieldPoint[] ZCBRates;

	private String name;

	public YieldCurve() {
		name = "NONAME";
		marketRates = new YieldPoint[YC_MAX_NUMBER_POINTS];
		ZCBRates = new YieldPoint[YC_MAX_NUMBER_POINTS];

		assignFlatRate(0.0);
		computeZCBRatesBootstrap();
	}

	public YieldCurve(double rate) {
		name = "NONAME";
		marketRates = new YieldPoint[YC_MAX_NUMBER_POINTS];
		ZCBRates = new YieldPoint[YC_MAX_NUMBER_POINTS];

		assignFlatRate(rate);
		computeZCBRatesBootstrap();
	}

	public YieldCurve(YieldPoint[] yieldPoints, String name) {
		this.name = name;
		
		marketRates = new YieldPoint[yieldPoints.length];
		for (int i = 0; i < yieldPoints.length; i++) {
			marketRates[i] = new YieldPoint(yieldPoints[i].getRate(),
					yieldPoints[i].getMaturity(), yieldPoints[i].getRateType(),
					yieldPoints[i].getDayCountConvention());
		}
		
		ZCBRates = new YieldPoint[yieldPoints.length];
		for (int i = 0; i < yieldPoints.length; i++) {
			ZCBRates[i] = new YieldPoint(yieldPoints[i].getRate(),
					yieldPoints[i].getMaturity(), yieldPoints[i].getRateType(),
					yieldPoints[i].getDayCountConvention());
		}

		sortMarketRatesByMaturity();
		sortCashSwap();
		computeZCBRatesBootstrap();
	}
	
	public YieldCurve(YieldCurve yc) {
		this(yc.getPointsIntheMarketCurve(), yc.getName());
	}

	/** For flat rate curves, set the flat rate value */
	public void assignFlatRate(double rate) {
		int size = marketRates.length;

		for (int i = 0; i < size; i++) {
			marketRates[i] = new YieldPoint(rate, i, TypeOfRate.Cash, DayCountConvention.ACT_360);
		}
	}

	/** sets an existing rate at a certain level */
	public void assignZCBRateAtIndex(double rate, int index) {
		if (index < ZCBRates.length) {
			YieldPoint yp = new YieldPoint(rate, ZCBRates[index].getMaturity(),
					ZCBRates[index].getRateType(), ZCBRates[index].getDayCountConvention());
			ZCBRates[index] = yp;;
		}
			
	}

	/** for risk management purposes, shifts the yield curve */
	public YieldCurve shiftZCBRateCurve(double shiftFactor) {
		String shiftName = name + " " + shiftFactor + " shifted curve";

		YieldCurve resultShifted = new YieldCurve(ZCBRates, shiftName);
		for (int i = 0; i < ZCBRates.length; i++)
			resultShifted.assignZCBRateAtIndex(ZCBRates[i].getRate()
					+ shiftFactor, i);

		return resultShifted;
	}

	/**
	 * for risk mngmt purposes, rotates the yc with a ref to how you move the
	 * shortest rate, around which rate
	 */
	public YieldCurve rotateZCBRateCurve(double moveInShortestRate,
			double maturityOfRotation) {
		String rotatedName = name + " " + moveInShortestRate + " rotated curve";

		YieldCurve resultRotated = new YieldCurve(ZCBRates, rotatedName);
		double move = 0.;
		double difmat = ZCBRates[0].getMaturity()
				- Math.min(2 * maturityOfRotation,
						ZCBRates[ZCBRates.length - 1].getMaturity());
		double summat = ZCBRates[0].getMaturity()
				+ Math.min(2 * maturityOfRotation,
						ZCBRates[ZCBRates.length - 1].getMaturity());
		for (int i = 0; i < ZCBRates.length; i++) {
			move = (moveInShortestRate / difmat)
					* (2 * ZCBRates[i].getMaturity() - summat);
			resultRotated.assignZCBRateAtIndex(ZCBRates[i].getRate() + move, i);
		}
		return resultRotated;
	}

	/**
	 * Calculates the spot ZCB rate
	 * 
	 * @param maturity :
	 *            if it is exact it just gives the result from a Point, else an
	 *            interpolated one based on interpolator
	 */
	public double spotRate(double maturity) {
		int i = 0;

		while (i < ZCBRates.length && ZCBRates[i].getMaturity() < maturity)
			i++;

		if (i == ZCBRates.length)
			// longer term than we know. Best approximation is the last
			// YieldPoint we have
			return ZCBRates[i - 1].getRate();
		else if (i == 0)
			// shorter term than we know. Best approximation is the first
			// YieldPoint we have
			return ZCBRates[i].getRate();
		else {
			// we do linear interpolation
			// www.riskglossary.com/link/interpolation.htm
			double rmin = ZCBRates[i - 1].getRate();
			double tmin = ZCBRates[i - 1].getMaturity();
			double rmax = ZCBRates[i].getRate();
			double tmax = ZCBRates[i].getMaturity();

			return ((tmax - maturity) * rmin + (maturity - tmin) * rmax)
					/ (tmax - tmin);
		}
	}

	/**
	 * Calculates the spot ZCB rate
	 * 
	 * @param maturityDate :
	 *            (Date) maturity of the ZCB
	 */
	public double spotRate(Date maturityDate) {
		Date matDate = new Date(maturityDate);
		Date today = Date.setDateToToday();

		return spotRate(today.dayCount(matDate, DayCountConvention.Day30_360));
	}

	/**
	 * Return the maturities present in the market curve, both from the Cash and
	 * Swap Points
	 */
	public double[] getMaturitiesInTheMarketCurve() {
		int size = marketRates.length;
		double[] maturities = new double[size];

		for (int i = 0; i < size; i++)
			maturities[i] = marketRates[i].getMaturity();

		return maturities;
	}

	/**
	 * Return the maturities present in the zero curve.
	 */
	public double[] getMaturitiesInTheZCBCurve() {
		int size = ZCBRates.length;
		double[] maturities = new double[size];

		for (int i = 0; i < size; i++)
			maturities[i] = ZCBRates[i].getMaturity();

		return maturities;
	}

	/**
	 * Return the rates present in the market curve, both from the Cash and Swap
	 * Points
	 */
	public double[] getRatesInTheMarketCurve() {
		int size = marketRates.length;
		double[] rates = new double[size];

		for (int i = 0; i < size; i++)
			rates[i] = marketRates[i].getRate();

		return rates;

	}

	/**
	 * Return the rates present in the ZCB curve.
	 */
	public double[] getRatesInTheZCBCurve() {
		int size = ZCBRates.length;
		double[] rates = new double[size];

		for (int i = 0; i < size; i++)
			rates[i] = ZCBRates[i].getRate();

		return rates;

	}

	public double discountFactor(double maturity) {
		return discountFactor(maturity, InterestComposition.Continuous);
	}

	/**
	 * Calculates the discountFactor
	 * 
	 * @param maturity :
	 *            (double) maturity of the ZCB
	 */
	public double discountFactor(double maturity,
			InterestComposition interestComposition) {
		switch (interestComposition) {
		case Discrete:
			return Math.pow(1 / (1 + spotRate(maturity)), maturity);
		case Continuous:
			return Math.exp(-1 * maturity * spotRate(maturity));
			// default case we assume continuous compounding
		default:
			return Math.exp(-1 * maturity * spotRate(maturity));
		}
	}

	/**
	 * Calculates the discountFactor
	 * 
	 * @param maturity :
	 *            (Date) maturity of the ZCB
	 */
	public double discountFactor(Date maturityDate,
			InterestComposition composition) {
		Date matDate = new Date(maturityDate);
		Date today = Date.setDateToToday();
		return discountFactor(today.dayCount(matDate,
				DayCountConvention.Day30_360), composition);
	}

	/**
	 * Calculates the discountFactor
	 */
	public double forwardDiscountFactor(double forwardStart,
			double lengthOfContract, InterestComposition composition) {
		switch (composition) {
		case Discrete:
			return Math.pow(1 / (1 + forwardRate(forwardStart, lengthOfContract,
					composition)), lengthOfContract);
		case Continuous:
			return Math.exp(-1 * lengthOfContract
					* forwardRate(forwardStart, lengthOfContract, composition));
		default:
			return Math.exp(-1 * lengthOfContract
					* forwardRate(forwardStart, lengthOfContract, composition));
		}
	}

	/**
	 * 
	 * @param forwardStart
	 * @param lengthOfContract
	 * @param composition :
	 *            Interest Composition
	 * @return
	 */
	public double forwardRate(double forwardStart, double lengthOfContract,
			InterestComposition composition) {
		switch (composition) {
		case Discrete:
			return Math.pow(discountFactor(forwardStart, composition)
					/ discountFactor(forwardStart + lengthOfContract,
							composition), 1 / lengthOfContract) - 1;
		case Continuous:
			return (spotRate(forwardStart + lengthOfContract)
					* (forwardStart + lengthOfContract) - spotRate(forwardStart)
					* forwardStart)
					/ lengthOfContract;
		default:
			return (spotRate(forwardStart + lengthOfContract)
					* (forwardStart + lengthOfContract) - spotRate(forwardStart)
					* forwardStart)
					/ lengthOfContract;

		}
	}

	/**
	 * 
	 * @param forwardStart :
	 *            (Date) start of forward
	 * @param forwardEnd :
	 *            (Date) end of forward
	 * @param composition :
	 *            Interest Composition
	 */
	public double forwardRate(Date forwardStart, Date forwardEnd,
			InterestComposition composition) {
		Date fwdStart = new Date(forwardStart);
		Date fwdEnd = new Date(forwardEnd);
		Date today = Date.setDateToToday();

		return forwardRate(today.dayCount(fwdStart,
				DayCountConvention.Day30_360), fwdStart.dayCount(
						fwdEnd, DayCountConvention.Day30_360), composition);
	}

	public YieldCurve forwardZCBCurve(double forwardStart) {
		int size = ZCBRates.length;
		Vector<YieldPoint> fwdPts = new Vector<YieldPoint>(size);
		int index = 0;
		int start = 0;

		while (index < size && forwardStart >= ZCBRates[index].getRate())
			index++;
		if (index == size)
			// forward term too high
			return new YieldCurve();
		else {
			start = index;
			for (int i = start; i < size - start; i++) {
				double rate = forwardRate(forwardStart, ZCBRates[i]
						.getMaturity()
						- forwardStart, InterestComposition.Continuous);
				double mat = ZCBRates[i].getMaturity() - forwardStart;
				YieldPoint tmpYP = new YieldPoint(rate, mat, TypeOfRate.Cash);
				fwdPts.add(tmpYP);
			}
		}

		YieldPoint[] res = new YieldPoint[fwdPts.size()];
		for (int i = 0; i < fwdPts.size(); i++)
			res[i] = fwdPts.get(i);

		return new YieldCurve(res, name + "_FwdCurve");
	}

	public String getName() {
		return name;
	}
	
	public YieldPoint[] getPointsIntheMarketCurve() {
		YieldPoint[] tmpPoints = new YieldPoint[marketRates.length];
		for (int i = 0; i < marketRates.length; i++) {
			tmpPoints[i] = new YieldPoint(marketRates[i].getRate(),
					marketRates[i].getMaturity(), marketRates[i].getRateType(),
					marketRates[i].getDayCountConvention());
		}
		return tmpPoints;
	}

	public void setName(String name) {
		this.name = name;
	}

	/**
	 * for know maturities, we can return the market Point as it is
	 * 
	 * @param maturity :
	 *            for unknown maturities returns null
	 */
	public YieldPoint getPointAtMaturity(double maturity) {
		int size = marketRates.length;
		int index = 0;
		int indexFound = 0;
		YieldPoint yieldPoint;
		boolean isFound = false;

		while (index < size && !isFound) {
			if (marketRates[index].getMaturity() == maturity) {
				isFound = true;
				indexFound = index;
			}
			index++;
		}

		if (isFound)
			yieldPoint = new YieldPoint(marketRates[indexFound]);
		else
			yieldPoint = new YieldPoint();

		return yieldPoint;
	}

	/**
	 * sorts market rates by maturity
	 * 
	 */
	private void sortMarketRatesByMaturity() {
		int size = marketRates.length;
		Vector<YieldPoint> tempStorage = new Vector<YieldPoint>(size);
		double[] toSort = new double[size];

		for (int index = 0; index < size; index++)
			toSort[index] = marketRates[index].getMaturity();

		Arrays.sort(toSort);

		for (int index = 0; index < size; index++) {
			tempStorage.add(getPointAtMaturity(toSort[index]));
		}

		for (int index = 0; index < size; index++) {
			marketRates[index] = tempStorage.get(index);
		}
	}

	/**
	 * Routine to make sure the short term rates (cash) are before the mid/long
	 * term (swap)
	 */
	private void sortCashSwap() {
		Vector<YieldPoint> tempStorage = new Vector<YieldPoint>(
				marketRates.length);

		for (int i = 0; i < marketRates.length; i++) {
			if (marketRates[i].getRateType() == TypeOfRate.Cash)
				tempStorage.add(marketRates[i]);
		}
		for (int i = 0; i < marketRates.length; i++) {
			if (marketRates[i].getRateType() == TypeOfRate.Swap)
				tempStorage.add(marketRates[i]);
		}
		for (int index = 0; index < marketRates.length; index++) {
			marketRates[index] = tempStorage.get(index);
		}
	}

	/**
	 * needed in the bootstrap method to be able to back the ZC
	 */
	private YieldPoint[] getSwapRates() {
		Vector<YieldPoint> tempStorage = new Vector<YieldPoint>(
				YC_MAX_NUMBER_POINTS);

		tempStorage.add(new YieldPoint(spotRate(1), 1, TypeOfRate.Swap));
		int index = 0;
		int count = 0;

		while (index < marketRates.length
				&& marketRates[index].getRateType() == TypeOfRate.Cash)
			index++;
		while (index < marketRates.length) {
			tempStorage.add(marketRates[index]);
			count++;
			index++;
		}

		YieldPoint[] swapRates = new YieldPoint[tempStorage.size()];
		for (int i = 0; i < tempStorage.size(); i++) {
			swapRates[i] = tempStorage.get(i);
		}

		return swapRates;
	}

	/**
	 * 1Y difference in swaps -> ZC's easily backed up The method assumes that
	 * rates are sorted by ascending maturity The market curve is 1Y by 1Y - we
	 * should fill in the gaps by linear interpolation
	 */
	private YieldPoint[] getSequentSwapRates() {
		YieldPoint[] tempStorage = getSwapRates();
		int size = tempStorage.length;
		YieldPoint[] res = new YieldPoint[(int) tempStorage[size - 1]
				.getMaturity()];
		int i = 1, diff = 0, j = 0, nextMat = 0, prevMat = 0;

		while (i < size
				&& 
				Math.abs(tempStorage[i - 1].getMaturity()-(tempStorage[i].getMaturity() - 1)) < 0.0000001) {
			res[j] = tempStorage[i - 1];
			i++;
			j++;
		}

		while (j < res.length && i < size) {
			if (tempStorage[i].getMaturity() == (double) (j + 1)) {
				res[j] = tempStorage[i];
				i++;
			} else {
				// linear interpolation
				double mat = (double) (j + 1);
				nextMat = (int) tempStorage[i].getMaturity();
				prevMat = (int) tempStorage[i - 1].getMaturity();
				diff = nextMat - prevMat;
				double rate = ((tempStorage[i - 1].getRate()
						* (nextMat - j - 1) + tempStorage[i].getRate()
						* (j + 1 - prevMat))
						/ diff);
				res[j] = new YieldPoint(rate, mat, TypeOfRate.Swap);
			}
			j++;
		}
		return res;
	}

	private void computeZCBRatesBootstrap() {

		YieldPoint[] tempStorage = new YieldPoint[marketRates.length];

		for (int i = 0; i < marketRates.length; i++)
			tempStorage[i] = marketRates[i];

		if (marketRates[marketRates.length - 1].getRateType() != TypeOfRate.Cash) {
			double[] discountFactors = SequentDiscountFactorsByInvertSwapMatrix();
			int start = 0;
			int size = discountFactors.length;

			while (tempStorage[start].getRateType() == TypeOfRate.Cash)
				start++;

			ZCBRates = new YieldPoint[size + start - 1];

			for (int i = 0; i < start; i++)
				ZCBRates[i] = tempStorage[i];

			for (int i = 1; i < size; i++) {
				double rate = ((double) Math.pow(1 / discountFactors[i],
						1 / ((double) (i + 1))) - 1);
				double mat = ((double) i + 1);
				YieldPoint tmp = new YieldPoint(rate, mat, TypeOfRate.Cash);

				ZCBRates[i + start - 1] = tmp;
			}
		}
		// as all market rates are cash, we copy marketRates into zcbRates as is
		else {
			ZCBRates = new YieldPoint[marketRates.length];
			for (int i = 0; i < marketRates.length; i++)
				ZCBRates[i] = new YieldPoint(marketRates[i].getRate(),
						marketRates[i].getMaturity(), marketRates[i].getRateType(),
						marketRates[i].getDayCountConvention());
		}
	}

	private double[] SequentDiscountFactorsByInvertSwapMatrix() {
		YieldPoint[] tempStorage = getSequentSwapRates();
		int size = tempStorage.length;

		Matrix M = new Matrix(size, size, 0.0);
		for (int i = 0; i < size; i++) {
			for (int j = 0; j < size; j++) {
				if (i == j)
					M.set(i, j, 1 + tempStorage[i].getRate());
				else if (i > j)
					M.set(i, j, tempStorage[i].getRate());
				else
					M.set(i, j, 0.0);
			}
		}

		M = M.inverse();

		double[] res = new double[size];

		for (int i = 0; i < size; i++)
			for (int j = 0; j < size; j++)
				res[i] += M.get(i, j);

		return res;
	}
}