package tests;

import junit.framework.TestCase;
import rates.InterestComposition;
import rates.TypeOfRate;
import rates.YieldCurve;
import rates.YieldPoint;

import common.Currency;
import common.Frequency;

import credit.CreditCurve;
import credit.CreditSpreadPoint;
import credit.CreditSpreadType;

public class CreditCurveTest extends TestCase {

	private final double[] marketRates = { 0.041, 0.0429, 0.045106, 0.045805,
			0.04614, 0.04641, 0.04671, 0.04698, 0.04721, 0.04746, 0.047735,
			0.04801, 0.04921, 0.04985, 0.0502 };

	private final double[] maturities = { 0.25, 0.5, 1, 2, 3, 4, 5, 6, 7, 8, 9,
			10, 15, 20, 30 };

	private final TypeOfRate[] typeOfRates = { TypeOfRate.Cash,
			TypeOfRate.Cash, TypeOfRate.Cash, TypeOfRate.Swap, TypeOfRate.Swap,
			TypeOfRate.Swap, TypeOfRate.Swap, TypeOfRate.Swap, TypeOfRate.Swap,
			TypeOfRate.Swap, TypeOfRate.Swap, TypeOfRate.Swap, TypeOfRate.Swap,
			TypeOfRate.Swap, TypeOfRate.Swap };

	private final double[] creditSpreadRates = { 0.045821, 0.0468, 0.0506,
			0.0569 };

	private final double[] creditSpreadMaturities = { 2, 5, 10, 20 };

	private final CreditSpreadType[] creditSpreadTypes = {
			CreditSpreadType.Absolute, CreditSpreadType.Absolute,
			CreditSpreadType.Absolute, CreditSpreadType.Absolute };

	private YieldCurve yieldCurve;

	private YieldPoint[] yieldPoints;

	private CreditSpreadPoint[] creditPoints;

	private CreditCurve creditCurve;

	public void setUp() throws Exception {
		super.setUp();

		yieldPoints = new YieldPoint[marketRates.length];
		for (int i = 0; i < marketRates.length; i++)
			yieldPoints[i] = new YieldPoint(marketRates[i], maturities[i],
					typeOfRates[i]);

		creditPoints = new CreditSpreadPoint[creditSpreadRates.length];
		for (int i = 0; i < creditSpreadRates.length; i++)
			creditPoints[i] = new CreditSpreadPoint(creditSpreadRates[i],
					creditSpreadMaturities[i], creditSpreadTypes[i]);

		creditCurve = new CreditCurve(yieldPoints, creditPoints,
				"testcreditcurve", 0.40, Currency.USD, Frequency.Annual);
		yieldCurve = creditCurve.getUnderlyingYieldCurve();
	}

	public void testUnderlyingSpotRate() {
		double calculation;
		boolean result = false;

		calculation = yieldCurve.spotRate(3.72);
		result = calculation >= 0.0463701 && calculation <= 0.0463702;
		assertTrue(result);

		calculation = yieldCurve.spotRate(14);
		result = calculation >= 0.0494384 && calculation <= 0.0494385;
		assertTrue(result);
	}

	public void testUnderlyingDiscountFactor() {
		double calculation;
		boolean result = false;

		calculation = yieldCurve.discountFactor(5.6,
				InterestComposition.Continuous);
		result = calculation >= 0.7688138 && calculation <= 0.7688142;
		assertTrue(result);

		calculation = yieldCurve.discountFactor(5.6,
				InterestComposition.Discrete);
		result = calculation >= 0.7734288 && calculation <= 0.7734292;
		assertTrue(result);

		calculation = yieldCurve.discountFactor(1.5);
		result = calculation >= 0.934078 && calculation <= 0.934079;
		assertTrue(result);
	}

	public void testUnderlyingForwardRate() {
		double calculation;
		boolean result = false;

		calculation = yieldCurve.forwardRate(1.5, 2.5,
				InterestComposition.Continuous);
		result = calculation >= 0.0470409 && calculation <= 0.0470410;
		assertTrue(result);

		calculation = yieldCurve.forwardRate(1.5, 2.5,
				InterestComposition.Discrete);
		result = calculation >= 0.0470413 && calculation <= 0.0470414;
		assertTrue(result);
	}

	public void testHullOFOD() {
		double[] defaultProb = { 0, 0.0200704, 0.0200704, 0.0200704, 0.0200704,
				0.0200704 };
		double[] survivalProb = { 1, 0.97993, 0.960262, 0.940989, 0.922103,
				0.903596 };
		double flatRate = 0.05;
		double flatSpread = 0.0126;

		CreditCurve flatCreditCurve = new CreditCurve(flatRate, flatSpread,
				"testcreditcurve", CreditCurve.CC_DEFAULT_RECOVERY_RATE,
				CreditCurve.CC_DEFAULT_CURRENCY,
				CreditCurve.CC_DEFAULT_FREQUENCY);

		// special case for time t0
		double result = flatCreditCurve.defaultProbability(0);
		assertEquals(result, 0.0);
		result = flatCreditCurve.survivalProbability(0);
		assertEquals(result, 1.0);

		for (int i = 1; i <= 5; i++) {
			double result1 = flatCreditCurve.defaultProbability(i);
			double result2 = flatCreditCurve.survivalProbability(i);
			boolean test1 = Math.abs(result1 - defaultProb[i]) <= 0.000001;
			boolean test2 = Math.abs(result2 - survivalProb[i]) <= 0.000001;
			assertTrue(test1);
			assertTrue(test2);
		}
	}
}
