package game; 
import java.util.Set;
import java.util.HashMap;
import java.util.Iterator;

/*
 * Class Room - a room in an adventure game.
 *
 * This class is part of the "Escape from Alcatraz" application. 
 *
 * A "Room" represents one location in the scenery of the game.  It is 
 * connected to other rooms via exits.  For each existing exit, the room 
 * stores a reference to the neighboring room.
 * 
 * @author  Michael Kolling and David J. Barnes
 * @version 1.0 (February 2002)
 */

public class Room 
{   private String roomImage;                   //stores the name of the room's image, with no extension. all extensions of images used in this game are .jpg.
    private String description;                 //short description of the room
    private HashMap <String, Room> exits;        // stores exits of this room.
    private Item item=null;                     //the item present in the room
    private Character character=null;           //the character present in the room. 
    
        //a room can have only a single character and a single item

    /**
     * Create a room described "description" with a given image. 
     * Initially, it has no exits. "description" is something like 
     * "in a kitchen" or "in an open court yard".
     */
    public Room(String roomImage,String description) 
    {   this.roomImage = roomImage;
        this.description = description;
        exits = new HashMap <String, Room> ();
 
		
    }
    
    
       /**
        * add new characters to the room
        * @param newCharacter
        */
     public void setCharacter(Character newCharacter){
         this.character=newCharacter;
         
     }
     
     /**
      * adds new item to the room
      * @param newItem
      */
     public void setItem(Item newItem){
         this.item=newItem;
     }
     
     /**
      * checks if there are any characters in the room. used to avoid the chance of a null pointer exception
      * @return boolean
      */
     public boolean isCharacterPresent()
     {
         return !(character == null); //returns true if characters are present and false if characters are absent
     }
  
     /**
      * used to check if there are any items in the room. 
      * @return boolean
      */
     public boolean isItemPresent()
     { return !(item==null);         //returns true if items are present and false if characters are absent
     }
     
     
     /**
      * removes a character from a room .. it is called if the player shoots a character in the room
      */
     public void deleteCharacter(){
         this.character=null;
     }
     
     /**
      * removes item from the room. This is called if the player picks up an item from the room. 
      */
     public void deleteItem(){
         this.item=null;
     }
     
     /**
      * accessor method to get a character object in the room
      * @return Character
      */
     public Character getCharacter(){
         return this.character;
     }
     
     
     /**
      * accessor method to get the image file name of the character in the room
      * @return path string of character image
      */
     public String getRoomCharacterImage()
     {
         return character.getCharacterImage();
     }
     
     
     /**
      * accessor method used to get the item image file path
      * @return String with the path of the item image
      */
     public String getRoomItemImage()
     {
         return item.getItemName();
     }
     
     /**
      * returns the items details, description
      * @return String with the description
      */
     public String getItemDetails(){
         return item.getItemDescription();
     }
     
     /**
      * returns the iten Object
      * @return Item
      */
     public Item getItem(){
         return this.item;
     }
     
     /** this is used to get the image title path of the room.
      * 
      * @return String path to the image title.
      */
     public String getRoomImageTitle()
    {
        return roomImage;
    }
     
     
    /**
     * sets the exits in a room
     * @param direction String
     * @param neighbor Room
     */ 
    public void setExit(String direction, Room neighbor) 
    {
        exits.put(direction, neighbor);
    }

    /**
     * Return the description of the room (the one that was defined in the
     * constructor).
     * @return String only the room description
     */
    public String getShortDescription()
    {
        return description;
    }
    

    /**
     * Return a long description of this room, in the form:
     *   @return returns long description You are in the kitchen.
     *     Exits: north west
     */
    public String getLongDescription()
    {
        return "You are " + description + ".\n" + getExitString();
    }

    
    /**
     * Return a string describing the room's exits, for example
     * "Exits: north west".
     * @return String  (the exit String.)
     */
    private String getExitString()
    {
        String returnString = "Exits:";
        for (Object key : exits.keySet()) {
            returnString += " " + key;
        }
        return returnString;
    }

    /**
     * Return the room that is reached if we go from this room in direction
     * "direction". If there is no room in that direction, return null.
     * @return Room gets the room in the specified direction
     */
    public Room getExit(String direction) 
    {
        return (Room)exits.get(direction);
    }

	
}
