package repository;

import java.util.Date;
import java.util.List;
import org.hibernate.*;
import org.hibernate.criterion.Expression;
import data.*;
import exception.*;

/**
 * DAO providing basic methods for managing ForumThread-Objects.
 * @author Ferdinand Kaiser, 11.06.08
 */
public class ForumThreadDAO {

	private Session session;

	/**
	 * This constructor is used by the DAOManager, its a factory that initializes all DAOs with the current session.
	 * Note that every session.commit() will close the session, the DAOManager must initialize the DAOs again.
	 * Use a Open Session in View pattern to prevent this.
	 */
	public ForumThreadDAO(){
		this.session=HibernateUtil.getSessionFactory().getCurrentSession();
	}	

	/**
	 * @param owner the threads owner
	 * @param title title of the thread
	 * @param creationDate date when thread was created
	 * @param text the threads content
	 * @param lastEdit date when thread was last edited
	 * @param lastEdit date when the whole thread was last updated (edited or comment added)
	 * @param visible
	 * @return ForumThread object 
	 * @throws MissingValueException if there is a not-null value that is null	 * 
	 */
	public ForumThread createAndStore(User owner, String title, String text) 
	throws MissingValueException{
		ForumThread forumthread=new ForumThread(owner, title, new Date(), text, new Date(),new Date(), true);
		try{
			session.save(forumthread);
		}
		catch(PropertyValueException ex){
			throw new MissingValueException("A not-null value is possibly null!");
		}
		return forumthread;
	}

	/**
	 * 
	 * @return a list of all ForumThreads
	 * @throws NoSuchArticleException if there aren't forumthreads
	 */
	public List<ForumThread> getList() throws NoSuchForumThreadException{
		List<ForumThread> result=session.createQuery("from ForumThread order by creationDate DESC").list();
		if (result.size()==0) throw new NoSuchForumThreadException();
		return result;
	}

	/**
	 *  Retrieves given amount of ForumThreads
	 * @param start sets the first row to retrieve, starting from 0
	 * @param count sets the maximum number of results to retrieve
	 * @return
	 */
	public List<ForumThread> getList(int start, int count)throws NoSuchForumThreadException{
		Query query=session.createQuery("from Forumthread order by creationDate DESC");
		query.setFirstResult(start);
		query.setMaxResults(count);
		List<ForumThread> result=query.list();
		if (result.size()==0) throw new NoSuchForumThreadException();
		return result;
	}	

	/**
	 * gets ForumThread by its unique id
	 * @param id unique id
	 * @return forumthread
	 * @throws NoSuchForumThreadException if there isn't a ForumThread  with this id
	 */
	public ForumThread  getById(int id) throws NoSuchForumThreadException{
		ForumThread  result=(ForumThread)session.get(ForumThread.class,id);
		if (result==null) throw new NoSuchForumThreadException();
		return result;
	}

	/**
	 * filters all ForumThreads by a given owner
	 * @param owner owner to filter by
	 * @return list of all ForumThreads from this owner
	 * @throws NoSuchEventException if there are no ForumThreads from this owner
	 */
	public List<ForumThread> getByOwner(User owner) throws NoSuchForumThreadException{
		Criteria criteria = session.createCriteria(ForumThread.class);
		criteria.add(Expression.eq("owner",owner));
		List<ForumThread> result=criteria.list();
		if (result.size()==0) throw new NoSuchForumThreadException();
		return result;
	}	

}
