package repository;

import manager.DAOManager;
import org.hibernate.Query;
import org.hibernate.Session;
import data.*;
import exception.*;
import org.hibernate.PropertyValueException;
import java.util.List;

/**
 * DAO providing basic methods for managing Tagclass-Objects.
 * @author Ferdinand Kaiser, 25.05.2008
 */
public class TagDAO {

	private Session session;

	/**
	 * This constructor is used by the DAOManager, its a factory that initializes all DAOs with the current session.
	 * Note that every session.commit() will close the session, the DAOManager must initialize the DAOs again.
	 * Use a Open Session in View pattern to prevent this.
	 */
	public TagDAO(){
		this.session=HibernateUtil.getSessionFactory().getCurrentSession();
	}	

	/**
	 * Creates an saves a tag, if it doesn't already exist and assigns it to the specified article
	 * @param name name of the tag
	 * @param parentId id of the article
	 * @return tag instance
	 * @throws MissingValueException if there is a not-null value that is null	 * 
	 */
	public Tag createArticleTag(String name, int parentId) throws NoSuchArticleException,MissingValueException{
		Tag tag;
		try{
			tag=DAOManager.getTagDAO().getByName(name);	
		}
		catch(NoSuchTagException ex){
			tag=new Tag(name);	
		}
		DAOManager.getArticleDAO().getById(parentId).getTags().add(tag);
		try{
			session.save(tag);
		}
		catch(PropertyValueException ex){
			throw new MissingValueException("A not-null value is possibly null!");
		}
		return tag;
	}

	/**
	 * Creates an saves a tag, if it doesn't already exist and assigns it to the specified event
	 * @param name name of the tag
	 * @param parentId id of the event
	 * @return tag instance
	 */
	public Tag createEventTag(String name, int parentId) throws NoSuchEventException,MissingValueException{
		Tag tag;
		try{
			tag=DAOManager.getTagDAO().getByName(name);	
		}
		catch(NoSuchTagException ex){
			tag=new Tag(name);	
		}
		DAOManager.getEventDAO().getById(parentId).getTags().add(tag);
		try{
			session.save(tag);
		}
		catch(PropertyValueException ex){
			throw new MissingValueException("A not-null value is possibly null!");
		}
		return tag;
	}

	/**
	 * returns a tag with the specified name
	 * @param name name of the tag
	 * @return tag instance
	 * @throws NoSuchTagException if there isn't a tag with the specified name
	 */
	public Tag getByName(String name) throws NoSuchTagException{
		Query query=session.createQuery("from Tag  t where t.name = :name");
		query.setString("name",name);
		List<Tag> tags=query.list();
		if (tags.size()==0) throw new NoSuchTagException();
		return tags.get(0);

	}

	/**
	 * gets an tag by its unique id
	 * @param id unique id
	 * @return tag instance
	 * @throws NoSuchTagException if there isn't a tag with the specified id
	 */
	public Tag getById(int id) throws NoSuchTagException{
		Tag result=(Tag)session.load(Article.class,id);
		if (result==null) throw new NoSuchTagException();
		return result;
	}	

	/**
	 * @return a list of all tags
	 * @throws NoSuchTagException if there aren't any tags
	 */
	public List<Tag> getList() throws NoSuchTagException{
		List<Tag> result=session.createQuery("from Tag").list();
		if (result.size()==0) throw new NoSuchTagException();
		return result;
	}

	/**
	 * Retrieves given amount of Tags
	 * @param start sets the first row to retrieve, starting from 0
	 * @param count sets the maximum number of results to retrieve
	 * @return
	 */
	public List<Tag> getList(int start, int count)throws NoSuchTagException{
		Query query=session.createQuery("from Tag order by name DESC");
		query.setFirstResult(start);
		query.setMaxResults(count);
		List<Tag> result=query.list();
		if (result.size()==0) throw new NoSuchTagException();
		return result;
	}	

}
