package repository;


import java.util.List;
import java.util.Date;
import org.hibernate.Session;
import org.hibernate.criterion.*; 
import data.*;
import exception.*;
import java.security.*;
import org.hibernate.PropertyValueException;

/**
 * DAO providing basic methods for managing Userclass-Objects.
 *
 * @author Ferdinand Kaiser, 06.04.08
 *
 */
public class UserDAO {

	private Session session;

	/**
	 * This constructor is used by the DAOManager, its a factory that initializes all DAOs with the current session.
	 * Note that every session.commit() will close the session, the DAOManager must initialize the DAOs again.
	 * Use a Open Session in View pattern to prevent this.
	 */
	public UserDAO(){
		this.session=HibernateUtil.getSessionFactory().getCurrentSession();
	}

	/**
	 * Creates user object and saves it, sets creationDate with actual date and sets
	 * visible property to true
	 * @param firstName firstname of the user
	 * @param lastName lastname of the user
	 * @param emailAddress emailaddress of the user
	 * @param nickName nickname of the user
	 * @param password the users password
	 * @param role the role the user holds
	 * @return user instance
	 * @throws MissingValueException if there is a not-null value that is null	 * 
	 */
	public User createAndStore(String firstName, String lastName, String emailAddress,
			String nickName, String password, RoleEnum role) throws MissingValueException{
		String pwhash=null;
		User user=null;
		try{
			pwhash=MD5Util.getHash(password);
		}
		catch(NoSuchAlgorithmException ex){
			ex.printStackTrace();
		}
		try{
			user=new User(firstName, lastName, emailAddress, nickName, pwhash, null,new Date(), role, true);
			session.save(user);
		}
		catch(PropertyValueException ex){
			throw new MissingValueException("A not-null value is possibly null!");
		}
		return user;
	}

	/**
	 * Deactivates the user with the given id
	 * @param id unique id
	 * @throws NoSuchUserException if there isn't a user with this id
	 */
	public void deactivate(int id) throws NoSuchUserException{
		User user=getById(id);
		user.setVisible(false);
		session.update(user);
	}

	/**
	 * Reactivates the user with the given id
	 * @param id unique id
	 * @throws NoSuchUserException if there isn't a user with this id
	 */
	public void reactivate(int id) throws NoSuchUserException{
		User user=getById(id);
		user.setVisible(true);
		session.update(user);
	}

	/**
	 * @return a list of all users
	 * @throws NoSuchUserException if there aren't any users
	 */
	public List<User> getList() throws NoSuchUserException{
		List<User> result=session.createCriteria(User.class).list();
		if (result==null) throw new NoSuchUserException();
		return result;
	}

	/**
	 * filters the users by a given nickname
	 * @param nickname the users nickname
	 * @return 
	 * @throws NoSuchUserException if there isn't a user with this nickname
	 */
	public User getByNickName(String nickname) throws NoSuchUserException{
		User result=(User) session.createCriteria(User.class)
		.add(Restrictions.like("nickName", nickname)
				.ignoreCase())
				.uniqueResult();
		if (result==null) throw new NoSuchUserException();
		return result;
	}

	/**
	 * @param nickname
	 * @return true if there is already a user with this nickname
	 */
	public boolean isRegistered(String nickname){
		boolean result=false;
		try{
			result=(getByNickName(nickname)!=null);	
		}
		catch(NoSuchUserException ex){
		}
		return result;
	}

	/**
	 * gets user by its unique id
	 * @param id unique id
	 * @return user
	 * @throws NoSuchUserException if there isn't a user with this id 
	 */
	public User getById(int id) throws NoSuchUserException{
		User result=(User)session.get(User.class,id);
		if (result==null) throw new NoSuchUserException();
		return result;
	}

}
