package mks;

import hudson.EnvVars;
import hudson.Extension;
import hudson.FilePath;
import hudson.Launcher;
import hudson.Util;
import hudson.model.AbstractBuild;
import hudson.model.AbstractProject;
import hudson.model.BuildListener;
import hudson.model.Describable;
import hudson.model.Node;
import hudson.model.Run;
import hudson.model.TaskListener;
import hudson.scm.ChangeLogParser;
import hudson.scm.PollingResult;
import hudson.scm.SCM;
import hudson.scm.SCMDescriptor;
import hudson.scm.SCMRevisionState;
import hudson.util.FormValidation;
import hudson.util.ListBoxModel;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import javax.servlet.ServletException;
import mks.changelog.MksChangeLogParser;
import mks.changelog.MksRlogOutputParser;
import mks.cmd.CmdMakeWritable;
import mks.cmd.CommandRunner;
import mks.cmd.MksCmdCreateSandbox;
import mks.cmd.MksCmdDropSandbox;
import mks.cmd.MksCmdFreeze;
import mks.cmd.MksCmdProjectInfo;
import mks.cmd.MksCmdRlog;
import mks.cmd.MksCmdRsync;
import mks.cmd.MksCmdTestSandbox;
import mks.cmd.MksCmdThaw;
import mks.config.MksAuthTypeListBoxModel;
import mks.config.MksBuildTypeListBoxModel;
import mks.config.JobSettings;
import mks.config.PreviousRunSettings;
import mks.config.PreviousRunSettings.CONFIG_CHANGE;
import mks.config.Project;
import mks.config.TimeoutSettings;
import mks.exceptions.InvalidSandboxException;
import mks.publisher.MksRecorder;
import net.sf.json.JSONObject;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.QueryParameter;
import org.kohsuke.stapler.StaplerRequest;


/**
 *
 * @author James Sheets
 */
public class MksScm
extends SCM
{

    // Stapler bound variables
    private final JobSettings jobSettings;

    
    @DataBoundConstructor
    public MksScm(JobSettings jobSettings)
    {
        this.jobSettings = jobSettings;
    }


    public JobSettings getJobSettings()
    {
        return jobSettings;
    }


    @Override
    public FilePath getModuleRoot(FilePath workspace, AbstractBuild build)
    {
        Project proj1 = getJobSettings().getProjects().get(0);
        return MksUtils.getSandboxLocation(workspace, proj1);
    }


    @Override
    public FilePath[] getModuleRoots(FilePath workspace, AbstractBuild build)
    {
        int projects = getJobSettings().getProjects().size();
        FilePath[] paths = new FilePath[ projects ];
        for (int i=0; i < projects; i++)
        {
            Project proj = getJobSettings().getProjects().get(i);
            paths[i] = MksUtils.getSandboxLocation(workspace, proj);
        }

        return paths;
    }


    @Override
    public boolean requiresWorkspaceForPolling()
    {
        // todo: modify program so this works false? affects compareRemoteRevisionWith methods
        return true;
    }


    @Override
    public boolean supportsPolling()
    {
        return true;
    }


    @Override
    public SCMRevisionState calcRevisionsFromBuild(AbstractBuild<?, ?> ab, Launcher lnchr, TaskListener tl)
    throws IOException, InterruptedException
    {
        return new MksRevisionState( ab.getTimestamp() );
    }


    @Override
    protected PollingResult compareRemoteRevisionWith( AbstractProject<?, ?> project, Launcher launcher,
            FilePath workspace, TaskListener listener, SCMRevisionState baseline )
    throws IOException, InterruptedException
    {
        listener.getLogger().println( "Checking for changes" );

        // We had a non-MKS version control system previously?  Force a build
        if ( !(baseline instanceof MksRevisionState) )
        {
            listener.getLogger().println( "Unknown revision state - forcing build" );
            return PollingResult.BUILD_NOW;
        }

        else if (project.getLastBuild() == null)
        {
            listener.getLogger().println( "No previous build found - forcing build" );
            return PollingResult.BUILD_NOW;
        }

        // Note: We don't use the SCMRevisionState, because MKS doesn't track project revisions
        // We'd have to mark the revision of every file in that class, then compare to see if there
        // were any differences.  Simpler to just see if there were changes to project vs workspace
        else
        {
            // We specifically don't want to check only non-fail builds.
            // If the triggered build failed, we could get stuck in a build-fail loop
            final Date lastBuildDate = project.getLastBuild().getTimestamp().getTime();
            //final Date lastBuildDate = ((MksRevisionState)baseline).getBuildDate();

            final ByteArrayOutputStream rlogOutput = new ByteArrayOutputStream();

            // run the rlog command to see if there were changes
            try
            {
                // Append each changelog line (in all projects) to our output steam
                for ( int jobNumber=0; jobNumber < jobSettings.getProjects().size(); jobNumber++ )
                {
                    final MksCmdRlog rlog = new MksCmdRlog(jobSettings, jobNumber, lastBuildDate );
                    CommandRunner runner = new CommandRunner( listener, workspace, launcher );
                    runner.run( rlog, rlogOutput );
                }
            }
            catch ( Throwable th )
            {
                // TODO: Can I just rethrow the exception, and reinstate the finally?
                th.printStackTrace( listener.getLogger() );
                return PollingResult.NO_CHANGES;
            }

            final String output = rlogOutput.toString();
            IOUtils.closeQuietly( rlogOutput );
            
            return MksRlogOutputParser.hasChanges( output )
                ? PollingResult.SIGNIFICANT
                : PollingResult.NO_CHANGES;
        }
    }
    

    public boolean getChanges( AbstractBuild build, CommandRunner runner, File changelogFile, int jobNumber )
    throws Throwable
    {
        runner.listener.getLogger().println( "" );
        runner.listener.getLogger().println( "Changes since last non-fail build" );

        final ByteArrayOutputStream rlogOutput = new ByteArrayOutputStream();
        Date lastNonFailBuild = new Date();
        boolean hasChanges = false;

        // Query MKS for changes since last non-fail build.
        // Specifically want the last non-fail build in this instance
        Run<?, ?> lsb = build.getPreviousNotFailedBuild();
        if ( null != lsb ) lastNonFailBuild = lsb.getTimestamp().getTime();
        
        final MksCmdRlog rlog = new MksCmdRlog(jobSettings, jobNumber, lastNonFailBuild);
        runner.run( rlog, rlogOutput );

        final String output = rlogOutput.toString();

        // Parse our output log to determine if there were any changes
        // Save changes to a log file
        if ( MksRlogOutputParser.hasChanges( output ) )
        {
            MksRlogOutputParser
                    .parse(build, output, runner.listener)
                    .save(changelogFile, runner.workspace, jobSettings, jobNumber);
            hasChanges = true;
        }
        else
        {
            runner.listener.getLogger().println( "No Changes." );
        }

        IOUtils.closeQuietly( rlogOutput );

        return hasChanges;
    }
    
    
    @Override
    public void buildEnvVars(AbstractBuild<?, ?> ab, Map<String,String> env)
    {
        // Add an env var for each sandbox
        final FilePath workspace = ab.getWorkspace();
        for ( Project project : jobSettings.getProjects())
        {
            final FilePath sandboxProjectFile = MksUtils.getSandboxLocation(workspace, project);
            env.put("Mks.Sandbox." + project.getSandboxName(), sandboxProjectFile.getRemote() );
        }
    }


    @Override
    public boolean checkout(AbstractBuild<?, ?> ab, Launcher lnchr, FilePath fp, BuildListener bl, File changeLogFile)
    throws IOException, InterruptedException
    {
        bl.getLogger().println( "" );
        
        boolean success = true;
        final EnvVars envVars = ab.getEnvironment( bl );
        final FilePath workspace = ab.getWorkspace();
        final CommandRunner runner = new CommandRunner( bl, workspace, lnchr );
        final PreviousRunSettings previousRunSettings = 
                new PreviousRunSettings( jobSettings, workspace, bl );

        // TODO: put in a switch to allow user to skip fetching the changelog? Or document instead...
        boolean skipChangelog = Boolean.parseBoolean( envVars.get( "MKS_SKIP_FETCH_CHANGELOG" ) );
        
        // Check to see if the MKS Post Build Actions is enabled
        final Describable mksPublisher =
                ab.getProject().getPublishersList().get( MksRecorder.class );
        boolean mksPostBuildEnabled = mksPublisher != null;

        try
        {
            final CONFIG_CHANGE hasConfigChanges = previousRunSettings.getConfigChanges();

            // Check if there was a change to the job configuration which
            // requires us to drop our sandboxes
            if ( ab.getPreviousBuild() != null
            && workspace.exists()
            && hasConfigChanges != CONFIG_CHANGE.NONE )
            {
                bl.getLogger().println("Detected config changes that affect registered sandboxes" );
                dropAllPossibleSandboxes( runner );
                workspace.deleteContents();
            }

            // Save config info so we can check for changes in next build
            previousRunSettings.saveConfig();

            // Test/Freeze/Drop/Checkout/Resync projects
            int loopCount = 0;
            for ( Project project : jobSettings.getProjects())
            {
                final FilePath sandboxDir = MksUtils.getSandboxLocation(workspace, project);

                // Drop the sandbox and delete the files if the sandbox somehow
                // isn't registered correctly
                if ( MksUtils.appearsIsRegisteredSandbox(sandboxDir) )
                {
                    try
                    {
                        runner.runSilent( new MksCmdTestSandbox(jobSettings, sandboxDir.getRemote()) );
                    }
                    catch ( InvalidSandboxException _ )
                    {
                        bl.getLogger().println( "Unregistered project found.  Deleting workspace contents." );
                        // Don't do deleteRecursive here, because we need an empty folder to exist
                        sandboxDir.deleteRecursive();
                    }
                }
                
                // Echo out the project info.
                runner.run( new MksCmdProjectInfo(jobSettings, loopCount) );

                // Check if we should freeze the project
                if( jobSettings.getFreeze() )
                {
                    // Can't checkpoint if post-build job isn't enabled
                    if ( !mksPostBuildEnabled )
                    {
                        bl.getLogger().println( "" );
                        bl.error( "MKS Source Integrity Post-Build Actions " +
                                "is not enabled.  Skipping Freeze/Thaw." );
                    }
                    // Can't freeze projects checked out by a revision number
                    else if ( !project.getBuildType().equals(MksBuildTypeListBoxModel.PROJECT_REVISION) )
                    {
                        runner.run( new MksCmdFreeze(jobSettings, loopCount) );
                        project.setFrozen(true);
                    }
                }

                // Note: we may want to remove this if stmt, and leave what's in the else
                if ( !MksUtils.appearsIsRegisteredSandbox(sandboxDir) && loopCount <= 0)
                {
                    createEmptyChangeLog( changeLogFile, bl, "changelog" );
                }
                else
                {
                    if ( skipChangelog )
                    {
                        // We don't create an empty changelog if we skip
                        bl.getLogger().println( "Skipping changelog fetch" );
                        bl.getLogger().println( "" );
                    }
                    else
                    {
                        // If there were changes, this will write them out to our log
                        // make sure we don't overwrite a log file with a non-empty one
                        // when there's more than 1 project path set
                        boolean hasChanged = getChanges( ab, runner, changeLogFile, loopCount );
                        if ( !hasChanged && loopCount <= 0 )
                        {
                            createEmptyChangeLog( changeLogFile, bl, "changelog" );
                        }
                    }
                }

                // Resync or checkout
                if ( sandboxDir.child("project.pj").exists() )
                {
                    // Clean the workspace
                    if ( jobSettings.getCleanBeforeResync() )
                    {
                        runner.run( new MksCmdDropSandbox(jobSettings, sandboxDir.getRemote()) );
                        sandboxDir.deleteRecursive();
                    }

                    runner.run( new MksCmdRsync(jobSettings, sandboxDir.getRemote()) );
                }
                else
                {
                    runner.run( new MksCmdCreateSandbox(jobSettings, loopCount, sandboxDir.getRemote()) );
                }

                loopCount++;
                bl.getLogger().println();
            } // end config path loop

            // Check if we should make all files in sandboxes writable
            if ( jobSettings.getMakeWritable() )
            {
                runner.run( new CmdMakeWritable(lnchr) );
            }
        }
        catch(Throwable th)
        {
            bl.getLogger().println();
            th.printStackTrace( bl.getLogger() );
            success = false;
        }
        finally
        {
            // Thaw all frozen projects after a failure
            if ( !success )
            {
                for (int i=0; i < jobSettings.getProjects().size(); i++)
                {
                    Project project = jobSettings.getProjects().get(i);
                    try
                    {
                        if ( project.isFrozen() )
                        {
                            runner.run( new MksCmdThaw(jobSettings, i) );
                        }
                    }
                    catch ( Throwable _ )
                    {
                        bl.error( String.format("Unable to thaw %s project during build failure cleanup", project.getSandboxName()) );
                    }
                }
            }
            return success;
        }
    }
    

    @Override
    public boolean processWorkspaceBeforeDeletion( AbstractProject<?,?> project, FilePath workspace, Node node )
    throws IOException, InterruptedException
    {
        // Don't allow any errors to keep the workspace from being deleted.
        // If a sandbox doesn't get dropped, we can clean it up through MKS's GUI
        try
        {
            TaskListener listener = TaskListener.NULL;
            Launcher launcher = node.createLauncher( listener );
            CommandRunner runner = new CommandRunner( listener, workspace, launcher );

            dropAllPossibleSandboxes( runner );
        }
        catch ( Throwable _ )
        {
            // Throwables are already logged
        }

        return true;
    }


    @Override
    public ChangeLogParser createChangeLogParser()
    {
        return new MksChangeLogParser();
    }


    @Override
    public DescriptorImpl getDescriptor()
    {
        return (DescriptorImpl)super.getDescriptor();
    }


    public void dropAllPossibleSandboxes( final CommandRunner runner )
    throws Throwable
    {
        final FilePath workspace = runner.workspace;
        final TaskListener listener = runner.listener;
        final List<FilePath> subDirs = workspace.listDirectories();

        listener.getLogger().println( "Attempting to drop any found sandboxes" );

        // The config path for the sandbox may no longer be in our project
        // Find any project.pj files in the direct sub-dirs of our workspace
        // and try to drop them
        for ( FilePath subDir : subDirs )
        {
            if ( MksUtils.appearsIsRegisteredSandbox(subDir) )
            {
                runner.run( new MksCmdDropSandbox(jobSettings, subDir.getRemote() ) );
            }
        }

        listener.getLogger().println( "" );
    }




    @Extension
    public static final class DescriptorImpl
    extends SCMDescriptor<MksScm>
    {
        // To persist global configuration information,
        // simply store it in a field and call save().
        // If you don't want fields to be persisted, use <tt>transient</tt>.
        private String executable;
        private TimeoutSettings timeoutSettings = new TimeoutSettings();


        @Override
        public String getDisplayName()
        {
            return "MKS Source Integrity";
        }

        
        public DescriptorImpl()
        {
            super( MksScm.class, null );
            load();
        }


        @Override
        public boolean configure(StaplerRequest req, JSONObject o)
        throws FormException
        {
            // TODO: I could probably rewrite this so stapler could bind to this class...
            // return req.bindJSON(TimeoutSettings.class, o);
            
            // Persist global settings
            executable = Util.fixEmptyAndTrim( o.getString("executable") );
            timeoutSettings.INFO = o.optInt("info");
            timeoutSettings.DROP = o.optInt("drop");
            timeoutSettings.CREATE_SANDBOX = o.optInt("createSandbox");
            timeoutSettings.FREEZE = o.optInt("freeze");
            timeoutSettings.THAW = o.optInt("thaw");
            timeoutSettings.RESYNC = o.optInt("resync");
            timeoutSettings.RLOG = o.optInt("rlog");
            timeoutSettings.CHECKPOINT = o.optInt("checkpoint");
            timeoutSettings.TEST = o.optInt("test");
            save();
            return super.configure(req,o);
        }


        public FormValidation doExecutableCheck(@QueryParameter String value)
        throws IOException, ServletException
        {
            return FormValidation.validateExecutable(value);
        }


        public String getExecutable()
        {
            return StringUtils.defaultString(executable, "si");
        }
        

        public TimeoutSettings getTimeoutSettings()
        {
            return timeoutSettings;
        }
        
        
        @Override
        public MksScm newInstance(StaplerRequest req, JSONObject formData)
        throws FormException
        {
            System.out.println(formData.toString(2));

            /*List<Project> validate = (List<Project>)req.bindJSONToList(
                    Project.class, formData.getJSONArray("projects"));

            if (validate.size() > 1)
            {
                for (int i=0; i < validate.size()-1; i++)
                {
                    String currSandboxName = validate.get(i).getSandboxName();
                    String nextSandboxName = validate.get(i+1).getSandboxName();
                    if (currSandboxName.equalsIgnoreCase(nextSandboxName))
                    {
                        // Ugly, but better than nothing
                        throw new FormException("MKS sandbox names must be unique", "projects");
                    }
                }
            }*/

            return req.bindJSON(MksScm.class, formData);
        }


        public ListBoxModel doFillBuildTypeItems()
        {
            return new MksBuildTypeListBoxModel();
        }


        public ListBoxModel doFillAuthTypeItems()
        {
            return new MksAuthTypeListBoxModel();
        }

        
        public FormValidation doCheckUsername(@QueryParameter String value, @QueryParameter String authType)
        throws IOException, ServletException
        {
            return authType.equalsIgnoreCase(MksAuthTypeListBoxModel.CURRENT_USER) || !StringUtils.isBlank(value)
                ? FormValidation.ok()
                : FormValidation.error("You must specify a username");
        }


        public FormValidation doCheckPassword(@QueryParameter String value, @QueryParameter String authType)
        throws IOException, ServletException
        {
            return authType.equalsIgnoreCase(MksAuthTypeListBoxModel.CURRENT_USER) || !StringUtils.isBlank(value)
                ? FormValidation.ok()
                : FormValidation.error("You must specify a password");
        }


        /**
         * Performs on-the-fly validation of the form field 'name'.
         *
         * @param value
         *      This parameter receives the value that the user has typed.
         * @return
         *      Indicates the outcome of the validation. This is sent to the browser.
         */
        public FormValidation doCheckServer(@QueryParameter String value, @QueryParameter String authType)
        throws IOException, ServletException
        {
            return authType.equalsIgnoreCase(MksAuthTypeListBoxModel.CURRENT_USER) || !StringUtils.isBlank(value)
                ? FormValidation.ok()
                : FormValidation.error("You must set the MKS server host");
        }


        public FormValidation doCheckPort(@QueryParameter String value, @QueryParameter String authType)
        throws IOException, ServletException
        {
            try
            {
                int port = Integer.valueOf(value).intValue();
                return authType.equalsIgnoreCase(MksAuthTypeListBoxModel.CURRENT_USER) || port > 0
                    ? FormValidation.ok()
                    : FormValidation.error("Invalid port number");
            }
            catch(NumberFormatException _)
            {
                return FormValidation.error("Invalid port number");
            }
        }


        public FormValidation doCheckSandboxName(@QueryParameter String value)
        throws IOException, ServletException
        {
            
            return !StringUtils.isBlank(value)
                ? FormValidation.ok()
                : FormValidation.error("You must specify a sandbox name");
        }


        public FormValidation doCheckConfigPath(@QueryParameter String value)
        throws IOException, ServletException
        {
            return value.length() > 0
                ? FormValidation.ok()
                : FormValidation.error("You must specify the project's MKS ConfigPath");
        }

        public FormValidation doCheckBuildNumber(@QueryParameter String value, @QueryParameter String buildType)
        throws IOException, ServletException
        {
            return value.length() > 0 || buildType.equalsIgnoreCase(MksBuildTypeListBoxModel.TIP)
                ? FormValidation.ok()
                : FormValidation.error("You must specify a valid project checkpoint");
        }

    }
}