package mks.builder;

import hudson.EnvVars;
import hudson.Extension;
import hudson.Launcher;
import hudson.model.AbstractBuild;
import hudson.model.AbstractProject;
import hudson.model.BuildListener;
import hudson.tasks.BuildStepDescriptor;
import hudson.tasks.Builder;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.logging.Level;
import java.util.logging.Logger;
import mks.MksScm;
import mks.MksUtils;
import mks.cmd.CommandRunner;
import mks.cmd.MksCmdCi;
import mks.cmd.MksCmdCo;
import mks.cmd.MksCmdRlog_ModifiedMembers;
import net.sf.json.JSONObject;
import org.apache.commons.lang.StringUtils;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.StaplerRequest;

/**
 *
 * @author James Sheets
 */
public class MksCommitChanges 
extends Builder
{
    private final String sandbox;
    private final String issueNumber;
    //private final String checkoutParameters;
    //private final String checkinParameters;
    
    @DataBoundConstructor
    public MksCommitChanges(String sandbox, String issueNumber)//, String checkoutParameters, String checkinParameters)
    {
        this.sandbox = sandbox;
        this.issueNumber = issueNumber;
        //this.checkoutParameters = checkoutParameters;
        //this.checkinParameters = checkinParameters;
    }
    
    public String getSandbox() { return sandbox; }
    public String getIssueNumber() { return issueNumber; }
    //public String getCheckoutParameters() { return checkoutParameters; }
    //public String getCheckinParameters() { return checkinParameters; }
    
    @Override
    public boolean perform(AbstractBuild<?, ?> build, Launcher launcher, BuildListener listener) 
    throws InterruptedException, IOException
    {
        final PrintStream logger = listener.getLogger();
        boolean success = true;
        
        // Only supported for MKS Source Integrity SCM
        if ( !MksUtils.isUsingMksSandbox(build.getProject().getScm(), logger) )
        {
            return true;
        }
        
        final EnvVars envVars = build.getEnvironment( listener );
        
        // Check to see if it's an environment variable; else use the raw imput
        String envVarKey = MksUtils.stripEnvVarTokens( getSandbox() );
        String sandboxDir = ( envVars.containsKey( envVarKey ) ) 
                            ? envVars.get( envVarKey )
                            : getSandbox();
        
        if (sandboxDir == null)
        {
            listener.getLogger().println("Unable to find sandbox");
            return false;
        }
        
        final MksScm scm = MksScm.class.cast( build.getProject().getScm() );
        final CommandRunner runner = new CommandRunner( listener, build.getWorkspace(), launcher );
        
        
        try {
            final ByteArrayOutputStream rlogOutput = new ByteArrayOutputStream();
            runner.run( new MksCmdRlog_ModifiedMembers(scm.getJobSettings(), sandboxDir), rlogOutput );
            String[] addedFiles = rlogOutput.toString().split("\\r?\\n");
            
            for (String member :addedFiles)
            {
                if (StringUtils.isNotEmpty(member))
                {
                    runner.run( new MksCmdCo(scm.getJobSettings(), sandboxDir, member, getIssueNumber(), "Hudson checkout") );
                    runner.run( new MksCmdCi(scm.getJobSettings(), sandboxDir, member, getIssueNumber(), "Hudson checkin") );
                }
            }
        } catch (Throwable ex) {
            Logger.getLogger(MksDropFiles.class.getName()).log(Level.SEVERE, null, ex);
            success = false;
        }
        
        return success;
    }

    @Extension
    public static class DescriptorImpl
    extends BuildStepDescriptor<Builder> 
    {

        @Override
        public boolean isApplicable(Class<? extends AbstractProject> type) 
        {
            // TODO: only if MKS source control type is selected!
            return true;
        }

        @Override
        public String getDisplayName() 
        {
            return "MKS: Commit changes to source";
        }
        
        @Override
        public Builder newInstance(StaplerRequest req, JSONObject formData) 
        throws FormException
        {
            return req.bindJSON(MksCommitChanges.class, formData);
        }
        
    }
}
