package mks.changelog;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;

import org.kohsuke.stapler.export.Exported;
import org.kohsuke.stapler.export.ExportedBean;

import hudson.model.User;
import hudson.scm.ChangeLogSet;
import hudson.scm.ChangeLogSet.AffectedFile;
import hudson.scm.EditType;
import hudson.scm.ChangeLogSet.Entry;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import mks.MksUtils;
import org.apache.commons.lang.StringUtils;


/*
 * @author James Sheets
 */
@ExportedBean(defaultVisibility = 999)
public class MksChangeLogEntry
extends Entry
{
    private String member;
    private String mksuser;
    private String revision;
    private Date date;
    private String changePackageId;
    private String changePackageDesc;
    private String comment;
    private String action;


    public MksChangeLogEntry() { }


    public MksChangeLogEntry( MksChangeLogSet parent )
    {
        super();
        setParent( parent );
    }


    @Override
    public Collection<String> getAffectedPaths()
    {
        return Collections.singletonList( member );
    }


    /**
     * Hudson user - best guess based on the Login name parsed out from the MKS user.
     */
    @Override
    @Exported
    public User getAuthor()
    {
        User u = User.getUnknown();
        if ( mksuser != null )
        {
            String mksLogin = mksuser.substring( mksuser.indexOf( "(" )+1, mksuser.lastIndexOf( ")") ).trim();
            if ( !mksLogin.isEmpty() )
            {
                u = User.get( mksLogin );
            }
        }
        return u;
    }


    @Exported
    public EditType getEditType()
    {
        if ( "delete".equalsIgnoreCase( action ) ) {
            return EditType.DELETE;
        }
        else if ( "add".equalsIgnoreCase( action ) ) {
            return EditType.ADD;
        }
        return EditType.EDIT;
    }


    /**
     * @return The modified file
     */
    @Exported
    public String getMember()
    {
        return member;
    }


    /**
     * @param member The modified file
     */
    public void setMember( String member )
    {
        this.member = member;
    }


    /**
     * @return rev The revision number of the file
     */
    @Exported
    public String getRevision()
    {
        return revision;
    }


    /**
     * @param rev The revision number of the file
     */
    public void setRevision( String rev )
    {
        revision = rev;
    }


    /**
     * @return The date the file was modified
     */
    @Exported
    public Date getDate()
    {
        return this.date;
    }


    /**
     * @param date The date the file was modified
     */
    public void setDate( Date date )
    {
        this.date = date;
    }


    /**
     * Called by jelly to display the date in a simpler format than the default
     * @return Formatted date displayed to user in Hudson
     */
    public String getFormattedDate()
    {
        return MksUtils.formatRlogOutputDate(date);
    }


    /**
     * Used by MksChangeLogParser to read the date in from our XML file
     * @param date
     */
    public void setDateStr( String date )
    {
        try {
            SimpleDateFormat sdf = (SimpleDateFormat) SimpleDateFormat.getInstance();
            sdf.applyPattern( MksUtils.RLOG_OUTPUT_DATE_FORMAT );
            this.date = sdf.parse( date );
        }
        catch ( ParseException ex ) {
            Logger.getLogger(MksChangeLogEntry.class.getName()).log(Level.SEVERE, null, ex);
        }
    }


    /**
     * used by MksChangeLogParser to read the date from an MKS rlog output
     * @param date  
     */
    public void setMksDateStr( String date )
    {
        try {
            SimpleDateFormat sdf = (SimpleDateFormat) SimpleDateFormat.getInstance();
            sdf.applyPattern( MksUtils.RLOG_SEARCH_DATE_FORMAT );
            this.date = sdf.parse( date );
        }
        catch ( ParseException ex ) {
            Logger.getLogger(MksChangeLogEntry.class.getName()).log(Level.SEVERE, null, ex);
        }
    }
    

    /**
     * @return The Change Package ID
     */
    @Exported
    public String getCpid()
    {
        return changePackageId;
    }


    /**
     * @param cpid The Change Package ID
     */
    public void setCpid( String cpid )
    {
        changePackageId = cpid;
    }


    /**
     * @return The Change Package description
     */
    public String getCpDesc()
    {
        return changePackageDesc;
    }


    /**
     * @param changePackageDesc The Change Package description
     */
    public void setCpDesc( String changePackageDesc )
    {
        this.changePackageDesc = changePackageDesc;
    }


    /**
     * @return The commit message for this revision
     */
    @Override
    public String getMsg()
    {
        return comment;
    }


    /**
     * @param comment The commit message for this revision
     */
    public void setMsg( String comment )
    {
        this.comment = comment;
    }


    /**
     * @return The committer's MKS user name
     */
    @Exported
    public String getMksuser()
    {
        return mksuser;
    }


    /**
     * @param mksuser The committer's MKS user name
     */
    public void setMksuser( String mksuser )
    {
        this.mksuser = mksuser;
    }


    /**
     * @return The file edit type
     */
    @Exported
    public String getAction()
    {
        return action;
    }


    /**
     * @param action The file edit type
     */
    public void setAction( String action )
    {
        this.action = action;
    }


    @Override
    public String getMsgAnnotated()
    {
        return getMember();
    }


    @Override
    public String toString()
    {
        String nl = System.getProperty("line.separator");
        
        StringBuilder sb = new StringBuilder( "Modification: " );
        sb.append( action ).append( " " );
        sb.append( mksuser ).append( " " );
        sb.append( changePackageId ).append( " " );
        sb.append( getFormattedDate() ).append( " " );
        sb.append( revision ).append( " " );
        sb.append( comment ).append( nl );
        return sb.toString();
    }


    @Override
    public Collection<ChangeLogSet.AffectedFile> getAffectedFiles()
    {
        final ArrayList<ChangeLogSet.AffectedFile> returnList =
                new ArrayList<ChangeLogSet.AffectedFile>(1);

        returnList.add(
            new AffectedFile()
            {
                public String getPath()
                {
                    return StringUtils.substringBeforeLast( member, "/" );
                }

                public EditType getEditType()
                {
                    return MksChangeLogEntry.this.getEditType();
                }
            }
        );

        return returnList;
    }

}
