package mks.cmd;

import hudson.FilePath;
import hudson.Launcher;
import hudson.Launcher.ProcStarter;
import hudson.model.StreamBuildListener;
import hudson.model.TaskListener;
import hudson.util.ForkOutputStream;
import hudson.util.IOUtils;
import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.concurrent.TimeUnit;
import javax.naming.TimeLimitExceededException;
import mks.exceptions.ConnectionException;
import mks.exceptions.InvalidSandboxException;
import org.apache.commons.lang.StringUtils;

/**
 * Run a command, and verify that it was successful.
 * @author James Sheets
 */
public class CommandRunner
{
    public final TaskListener listener;
    public final FilePath workspace;
    public final Launcher launcher;

    /**
     * Invalid project
     */
    private final static String INVALID_PROJECT_SEARCH = "is not registered with the system";
    /**
     * Process timed out
     */
    private final static String TIMEOUT_SEARCH = "Timeout after";
    /**
     * Occasionally, communication with MKS via command line seems to stop until we log out and back in.  Cause unknown.
     */
    private final static String CONNECTION_REFUSED_SEARCH = "Could not establish server connection";
    /**
     * Authorization failed during login (MKS 2007, or was this a mistake?)
     */
    private final static String AUTHORIZATION_FAILED_SEARCH = "Authorization failed";
    /**
     * Authentication failed during login
     */
    private final static String AUTHENTICATION_FAILED_SEARCH = "Authentication failed";
    /**
     * This can happen if the account running Hudson doesn't have permission to access MKS
     */
    private final static String HUDSON_USER_PERMISSION_SEARCH = "Attempt to launch MKS Integrity Client Timed out";
    /**
     * Registry lock errors.  Random MKS problem
     */
    private final static String REGISTRY_LOCK_SEARCH = "CannotGetRegistryLock";


    public CommandRunner( final TaskListener listener, final FilePath workspace, final Launcher launcher )
    {
        this.listener = listener;
        this.workspace = workspace;
        this.launcher = launcher;
    }


    public void checkForErrors(int result, String capturedOutput)
    throws Throwable
    {
        // We're using Current GUI Session login settings, but there's no user logged into GUI
//        boolean nouser = StringUtils.containsIgnoreCase( capturedOutput, NO_CURRENT_USER_SEARCH );
//        if ( nouser )
//        {
//            throw new NoCurrentUserException( "No current user logged into MKS GUI tool" );
//        }

        // Check for a timeout
        boolean timeout = StringUtils.containsIgnoreCase( capturedOutput, TIMEOUT_SEARCH );
        if ( timeout )
        {
            throw new TimeLimitExceededException( "Operation timed out" );
        }

        boolean hudsonUserPermission = StringUtils.containsIgnoreCase( capturedOutput, HUDSON_USER_PERMISSION_SEARCH );
        if ( hudsonUserPermission )
        {
            throw new TimeLimitExceededException( "Interaction with MKS timed out. Hudson may be running under a different account than MKS." );
        }

        // Parse output to determine if this is an invalid project
        boolean invalidProject = StringUtils.containsIgnoreCase( capturedOutput, INVALID_PROJECT_SEARCH );
        if ( invalidProject )
        {
            throw new InvalidSandboxException( "Unregistered sandbox" );
        }

        // Bad connection  (invalid username/password/port/etc, server down, and so on)
        boolean connectionRefused = StringUtils.containsIgnoreCase( capturedOutput, CONNECTION_REFUSED_SEARCH );
        boolean authFailed = StringUtils.containsIgnoreCase( capturedOutput, AUTHORIZATION_FAILED_SEARCH );
        boolean authFailed2 = StringUtils.containsIgnoreCase( capturedOutput, AUTHENTICATION_FAILED_SEARCH );
        if ( connectionRefused || authFailed || authFailed2 )
        {
            throw new ConnectionException( "Connection failed" );
        }
        
        boolean registryLock = StringUtils.containsIgnoreCase( capturedOutput, REGISTRY_LOCK_SEARCH );
        if (registryLock)
        {
            throw new Exception("MKS CannotGetRegistryLock error");
        }

        if (result > 0)
        {
            throw new Exception("Error executing command.  Exit code: " + result);
            //throw new Exception(capturedOutput);
        }
    }


    /**
     * Execute the command, and print result to console
     * 
     * @param cmd
     * @throws Throwable
     */
    public void run( final AbstractCmd cmd )
    throws Throwable
    {
        final ByteArrayOutputStream ignoreOutput = new ByteArrayOutputStream();

        try
        {
            run(cmd, ignoreOutput);
        }
        catch(Throwable th)
        {
            throw th;
        }
        finally
        {
            IOUtils.closeQuietly( ignoreOutput );
        }
    }


    /**
     * Execute the command, and print result to console and capturedOutput
     *
     * @param cmd
     * @param captureOutput
     * @throws Throwable
     */
    public void run( final AbstractCmd cmd, final OutputStream captureOutput )
    throws Throwable
    {
        final PrintStream listenerLogger = listener.getLogger();
        final ForkOutputStream fos = new ForkOutputStream( captureOutput, listenerLogger );
        final StreamBuildListener lis = new StreamBuildListener( fos );
        
        try
        {
            listener.getLogger().println( "" );
            listener.getLogger().println( cmd.getDescription() );

            ProcStarter runit = launcher.launch()
                                .cmds( cmd )
                                .masks( cmd.toMaskArray() )
                                .pwd( workspace)
                                .stdout( fos );

            // Timeout error will be pumped to lis->out
            int result = ( cmd.getTimeout() <= 0 )
                    ? runit.join()
                    : runit.start().joinWithTimeout( cmd.getTimeout(), TimeUnit.SECONDS, lis );

            checkForErrors( result, captureOutput.toString() );
        }
        catch(Throwable th)
        {
            throw th;
        }
    }


    /**
     * Execute the command, and suppress output to the console
     * 
     * @param cmd
     * @throws Throwable
     */
    public void runSilent( final AbstractCmd cmd )
    throws Throwable
    {
        // Do not print output to logger
        final ByteArrayOutputStream out = new ByteArrayOutputStream();
        final StreamBuildListener lis = new StreamBuildListener( out );

        try
        {
            ProcStarter runit = launcher.launch()
                                .cmds( cmd )
                                .masks( cmd.toMaskArray() )
                                .pwd( workspace )
                                .stdout( out );

            // Timeout error will be pumped to lis->out
            int result = ( cmd.getTimeout() <= 0 )
                    ? runit.join()
                    : runit.start().joinWithTimeout( cmd.getTimeout(), TimeUnit.SECONDS, lis );

            checkForErrors( result, out.toString() );
        }
        catch(Throwable th)
        {
            throw th;
        }
        finally
        {
            IOUtils.closeQuietly( out );
        }
    }

}
