package hudson.plugins.si;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;

import org.apache.commons.lang.time.FastDateFormat;
import org.kohsuke.stapler.export.Exported;
import org.kohsuke.stapler.export.ExportedBean;

import hudson.model.User;
import hudson.scm.ChangeLogSet;
import hudson.scm.ChangeLogSet.AffectedFile;
import hudson.scm.EditType;
import hudson.scm.ChangeLogSet.Entry;
import java.util.ArrayList;
import org.apache.commons.lang.StringUtils;


/*
 * @author Michael Rack
 */
@ExportedBean(defaultVisibility = 999)
public class SourceIntegrityChangeLogEntry
extends Entry
{
    private String member;
    private String mksuser;
    private String revision;
    private Date date;
    private String changePackageId;
    private String changePackageDesc;
    private String comment;
    private String action;


    public SourceIntegrityChangeLogEntry() { }


    public SourceIntegrityChangeLogEntry( SourceIntegrityChangeLogSet parent )
    {
        super();
        setParent( parent );
    }


    /**
     * @see hudson.scm.ChangeLogSet.Entry#setParent(hudson.scm.ChangeLogSet)
     */
    @SuppressWarnings("unchecked")
    @Override
    protected void setParent( ChangeLogSet parent )
    {
        super.setParent( parent );
    }


    @Override
    public Collection<String> getAffectedPaths()
    {
        return Collections.singletonList( member );
    }


    /**
     * Hudson user - best guess based on the Login name parsed out from the MKS user.
     */
    @Override
    @Exported
    public User getAuthor()
    {
        User u = User.getUnknown();
        if ( mksuser != null ) 
        {
            String mksLogin = mksuser.substring( mksuser.indexOf( "(" )+1, mksuser.lastIndexOf( ")") ).trim();
            if ( !mksLogin.isEmpty() )
            {
                u = User.get( mksLogin );
            }
        }
        return u;
    }


    @Exported
    public EditType getEditType()
    {
        if ( "delete".equalsIgnoreCase( action ) ) {
            return EditType.DELETE;
        }
        else if ( "add".equalsIgnoreCase( action ) ) {
            return EditType.ADD;
        }
        return EditType.EDIT;
    }


    @Exported
    public String getMember()
    {
        return member;
    }


    public void setMember( String member )
    {
        this.member = member;
    }


    @Exported
    public String getRevision()
    {
        return revision;
    }


    public void setRevision( String rev )
    {
        revision = rev;
    }


    @Exported
    public Date getDate()
    {
        return this.date;
    }


    public void setDate( Date date )
    {
        this.date = date;
    }


    /**
     * Called by jelly to display the date in a simpler format than the default
     * @return
     */
    public String getFormattedDate()
    {
        return FastDateFormat.getInstance( SourceIntegrityRlogCommand.MKS_DATE_FORMAT ).format( date );
    }


    /**
     * Used by SourceIntegrityChangeLogparser to read the date in from our XML file
     * @param date
     */
    public void setDateStr( String date )
    {
        try {
            SimpleDateFormat sdf = (SimpleDateFormat) SimpleDateFormat.getInstance();
            sdf.applyPattern( SourceIntegrityRlogCommand.MKS_DATE_FORMAT  );
            this.date = sdf.parse( date );
        }
        catch ( ParseException px ) {
            px.printStackTrace();
        }
    }


    @Exported
    public String getCpid()
    {
        return changePackageId;
    }


    public void setCpid( String cpid )
    {
        changePackageId = cpid;
    }


    public String getCpDesc()
    {
        return changePackageDesc;
    }


    public void setCpDesc( String changePackageDesc )
    {
        this.changePackageDesc = changePackageDesc;
    }


    @Override
    public String getMsg()
    {
        return comment;
    }


    public void setMsg( String comment )
    {
        this.comment = comment;
    }
    

    @Exported
    public String getMksuser()
    {
        return mksuser;
    }


    public void setMksuser( String auth )
    {
        mksuser = auth;
    }


    @Exported
    public String getAction()
    {
        return action;
    }


    public void setAction( String action )
    {
        this.action = action;
    }


    @Override
    public String getMsgAnnotated()
    {
        return getMember();
    }


    @Override
    public String toString()
    {
        StringBuilder sb = new StringBuilder( "Modification: " );
        sb.append( action ).append( " " );
        sb.append( mksuser ).append( " " );
        sb.append( changePackageId ).append( " " );
        sb.append( FastDateFormat.getInstance( SourceIntegrityRlogCommand.MKS_DATE_FORMAT ).format( date ) ).append( " " );
        sb.append( revision ).append( " " );
        sb.append( comment ).append( "\n" );
        return sb.toString();
    }

    
    @Override
    public Collection<ChangeLogSet.AffectedFile> getAffectedFiles()
    {
        final ArrayList<ChangeLogSet.AffectedFile> returnList =
                new ArrayList<ChangeLogSet.AffectedFile>(1);

        returnList.add(
            new AffectedFile()
            {
                public String getPath()
                {
                    return StringUtils.substringBeforeLast( member, "/" );
                }

                public EditType getEditType()
                {
                    return SourceIntegrityChangeLogEntry.this.getEditType();
                }
            }
        );
        
        return returnList;
    }

}
