package hudson.plugins.si;

import hudson.EnvVars;
import hudson.FilePath;
import hudson.Launcher;
import hudson.Util;
import hudson.model.AbstractBuild;
import hudson.model.AbstractProject;
import hudson.model.BuildListener;
import hudson.model.Describable;
import hudson.model.Hudson;
import hudson.model.Node;
import hudson.model.Run;
import hudson.model.TaskListener;
import hudson.scm.ChangeLogParser;
import hudson.scm.PollingResult;
import hudson.scm.SCM;
import hudson.scm.SCMDescriptor;
import hudson.scm.SCMRevisionState;
import hudson.util.FormFieldValidator;
import hudson.util.Scrambler;
import java.io.BufferedReader;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.servlet.ServletException;

import net.sf.json.JSONObject;

import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.commons.lang.StringUtils;
import org.apache.tools.ant.util.FileUtils;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.StaplerRequest;
import org.kohsuke.stapler.StaplerResponse;

/*
 * @author Michael Rack
 */
public class SourceIntegrityScm
extends SCM
implements Serializable
{
    private static final String MKS_NAME = "MKS Source Integrity";

    private String _server;
    private int _port;
    private String[] _paths;
    private String _username;
    private String _password;
    private boolean _freeze = false;
    private boolean _cleanBeforeResync = false;
    private boolean _makeWritable = true;
    private String _authType;

    /**
     * Timeout settings - in seconds - for various shell commands
     */
    private static class TIMEOUT_SETTINGS {
        private static int TEST         = 5*60;     // 5 min
        private static int INFO         = 5*60;     // 5 min
        private static int DROP         = 20*60;    // 20 min
        private static int FREEZE       = 40*60;    // 40 min
        private static int THAW         = 40*60;    // 40 min
        private static int CHECKOUT     = 40*60;    // 40 min
        private static int RESYNC       = 40*60;    // 40 min
        private static int RLOG         = 40*60;    // 40 min
        private static int WRITEABLE    = 0; // no timeout
    }

    /**
     * Denote a configuration change to the job which would require
     * the sandboxes to be dropped
     */
    public enum CONFIG_CHANGE { NONE, WORKSPACE, CONFIGPATHS };
    

    /**
     * Constructor
     */
    @DataBoundConstructor
    public SourceIntegrityScm( String server, int port, String[] paths, String username,
            String password, boolean freeze, boolean cleanBeforeResync, boolean makeWritable, String authType )
    {
        _freeze = false;
        _cleanBeforeResync = false;
        _makeWritable = true;
        _server = server;
        _port = port;
        _username = username;
        _password = Scrambler.scramble( password );
        _paths = paths;
        _freeze = freeze;
        _cleanBeforeResync = cleanBeforeResync;
        _makeWritable = makeWritable;
        _authType = authType;
    }


    /**
     * Return the saved paths to config screen
     * @return
     */
    public String[] getPaths()
    {
        return _paths;
    }


    /**
     * @return the saved server to the config screen
     */
    public String getServer()
    {
        return _server;
    }


    /**
     * @return the saved port to the config screen
     */
    public int getPort()
    {
        return _port;
    }


    /**
     * @return the saved username to the config screen
     */
    public String getUsername()
    {
        if ( getAuthType().equals("currentUser") )
            return null;
        else
            return _username;
    }


    /**
     * @return the saved password to the config screen
     */
    public String getPassword()
    {
        if ( getAuthType().equals("currentUser") )
            return null;
        else
            return Scrambler.descramble( _password );
    }


    /**
     * @return the saved freeze state to the config screen
     */
    public boolean getFreeze()
    {
        return _freeze;
    }


    /**
     * @return the saved cleanBeforeResync state to the config screen
     */
    public boolean getCleanBeforeResync()
    {
        return _cleanBeforeResync;
    }


    /**
     * @return the saved makeWritable state to the config screen
     */
    public boolean getMakeWritable()
    {
        return _makeWritable;
    }


    /**
     * @return the authType to the config screen
     */
    public String getAuthType()
    {
        return _authType;
    }


    /**
     * Read in our configuation paths from the previous build, and compare
     * them to our current job settings.  Don't know how to serialize a variable
     * I don't want users to configure on the screen, hence writing to a file
     * @param workspace
     * @return
     */
    public CONFIG_CHANGE getConfigChanges( FilePath workspace, TaskListener listener )
    {
        CONFIG_CHANGE hasChanges = CONFIG_CHANGE.NONE;
        
        final File configPathFile =
                new File( workspace.getParent() + File.separator + "ConfigPaths.txt" );
        BufferedReader reader = null;
        if ( configPathFile.exists() )
        {
            List<String> foundPaths = new ArrayList<String>();
            String[] currPaths = getPaths();

            try
            {
                reader = new BufferedReader( new FileReader( configPathFile ) );

                // Get the first line - the workspace
                String line = StringUtils.trimToNull( reader.readLine() );

                // Workspace has changed, so our sandbox doesn't match the
                // registered path for the sandbox anymore
                if ( !line.equalsIgnoreCase( workspace.getRemote() ) )
                {
                    hasChanges = CONFIG_CHANGE.WORKSPACE;
                }
                else
                {
                    while ((line = StringUtils.trimToNull( reader.readLine() )) != null)
                    {
                        foundPaths.add( line );
                    }

                    // There's been a change to the sandbox config paths
                    if ( foundPaths.size() != currPaths.length )
                    {
                        hasChanges = CONFIG_CHANGE.CONFIGPATHS;
                    }
                    else if ( foundPaths.size() > 0 )
                    {
                        for ( int i=0; i < foundPaths.size(); i++ )
                        {
                            if ( !foundPaths.get(i).equalsIgnoreCase( currPaths[i] ))
                            {
                                hasChanges = CONFIG_CHANGE.CONFIGPATHS;
                                break;
                            }
                        }
                    }
                }
            }
            catch ( IOException ex )
            {
                listener.error( ex.getMessage() );
            }
            finally
            {
                FileUtils.close( reader );
            }
        }
        return hasChanges;
    }


    /**
     * Write out our configuation paths to a file
     * @param workspace
     * @return
     */
    public boolean saveConfig( FilePath workspace, TaskListener listener )
    {
        boolean success = false;
        final File configPathFile =
                new File( workspace.getParent() + File.separator + "ConfigPaths.txt" );
        PrintWriter writer = null;
        try
        {
            writer = new PrintWriter( new FileWriter( configPathFile ) );
            writer.println( workspace.getRemote() );
            for ( String configPath : getPaths() ) {
                writer.println( configPath );
            }
            return true;
        }
        catch ( IOException ex )
        {
            listener.error( ex.getMessage() );
        }
        finally
        {
            FileUtils.close( writer );
        }
        return success;
    }


    /**
     * Checkout the latest version of the project
     * @param build
     * @param launcher
     * @param workspace
     * @param listener
     * @param changelogFile
     * @return
     * @throws java.io.IOException
     * @throws java.lang.InterruptedException
     */
    @SuppressWarnings("unchecked")
    @Override
    public boolean checkout( AbstractBuild build, Launcher launcher,
            FilePath workspace, BuildListener listener, File changelogFile )
    throws IOException, InterruptedException
    {
        listener.getLogger().println( "" );

        final EnvVars envVars = build.getEnvironment( listener );
        
        boolean isFrozen = false;
        boolean success = true;
        boolean skipChangelog = Boolean.parseBoolean( envVars.get( "MKS_SKIP_FETCH_CHANGELOG" ) );
        
        // Check to see if the MKS Post Build Actions is enabled.
        final Describable mksPublisher =
                build.getProject().getPublishersList().get( SourceIntegrityPublisher.class );
        boolean mksPostBuildEnabled = mksPublisher == null;
        
        final SourceIntegrityLauncher siLauncher =
                new SourceIntegrityLauncher( listener, workspace, launcher );
        
        try
        {
            final CONFIG_CHANGE hasConfigChanges = getConfigChanges( workspace, listener );
            // Check if there was a change to the job configuration which
            // requires us to drop our sandboxes
            if ( build.getPreviousBuild() != null
            && workspace.exists()
            && hasConfigChanges != CONFIG_CHANGE.NONE )
            {
                listener.getLogger().println( 
                        "Detected config changes that affect registered sandboxes" );
                dropAllPossibleSandboxes( siLauncher );
                workspace.deleteContents();
            }

            // Save config info so we can check for changes in next build
            saveConfig( workspace, listener );

            // Test/Freeze/Drop/Checkout/Resync projects
            int loopCount = 0;
            for ( String configPath : getPaths() )
            {
                final FilePath sandbox = getSandboxWorkspace( workspace, configPath );

                // Drop the sandbox and delete the files if the sandbox somehow
                // isn't registered correctly
                if ( sandbox.child("project.pj").exists() )
                {
                    testSandbox( siLauncher, sandbox );
                }

                // Echo out the project info.
                projectInfo( siLauncher, configPath );

                // Check if we should freeze the project
                // Requires MKS Post-Build Actions to be enabled
                if( getFreeze() )
                {
                    if ( !mksPostBuildEnabled )
                    {
                        listener.error( "MKS Source Integrity Post-Build Actions " +
                                "is not enabled.  Skipping Freeze/Thaw." );
                        listener.getLogger().println( "" );
                    }
                    else
                    {
                        freezeProject( siLauncher, configPath );
                        isFrozen = true;
                    }
                }

                // Note: we may want to remove this if stmt, and leave what's in the else
                if ( !sandbox.child("project.pj").exists() && loopCount <= 0)
                {
                    createEmptyChangeLog( changelogFile, listener, "changelog" );
                }
                else
                {
                    if ( !skipChangelog )
                    {
                        // If there were changes, this will write them out to our log
                        // make sure we don't overwrite a log file with a non-empty one
                        // when there's more than 1 project path set
                        boolean hasChanged = getChanges( build, siLauncher, changelogFile, configPath, sandbox.getRemote() );
                        if ( !hasChanged && loopCount <= 0 )
                        {
                            createEmptyChangeLog( changelogFile, listener, "changelog" );
                        }
                    }
                    else
                    {
                        // We don't create an empty changelog if we skip
                        listener.getLogger().println( "Skipping changelog fetch" );
                        listener.getLogger().println( "" );
                    }
                }

                // Resync or checkout
                if ( sandbox.child("project.pj").exists() )
                {
                    // Clean the workspace
                    if ( getCleanBeforeResync() )
                    {
                        dropSandbox( siLauncher, true, sandbox.getRemote() );
                        sandbox.deleteRecursive();
                    }
                    
                    resyncSandbox( siLauncher, sandbox.getRemote() );
                }
                else
                {
                    checkoutSandbox( siLauncher, configPath, sandbox.getRemote() );
                }
                
                loopCount++;
                listener.getLogger().println( "" );
            } // end config path loop
            
            // Check if we should make all files in sandboxes writable
            if ( getMakeWritable() )
            {
                makeWritable( build, siLauncher );
            }
        }
//        catch ( IOException ioe )
//        {
//            success = false;
//            throw ioe;
//        }
//        catch ( InterruptedException ie )
//        {
//            success = false;
//            throw ie;
//        }
        catch ( Throwable th )
        {
            th.printStackTrace( listener.getLogger() );
            success = false;
        }
        finally
        {
            // Thaw a frozen project after a failure
            if ( isFrozen && !success )
            {
                try 
                {
                    for ( String configPath : getPaths() )
                    {
                        thawProject( siLauncher, configPath );
                        isFrozen = false;
                    }
                }
                catch ( Throwable _ )
                {
                    listener.error( "Unable to thaw project during build failure cleanup" );
                }
            }
            return success;
        }
    }


    public void testSandbox( SourceIntegrityLauncher siLauncher, FilePath sandbox )
    throws Throwable
    {
        siLauncher.getListener().getLogger().println( "Check sandbox integrity" );
        try
        {
            SourceIntegrityTestSandbox testSandbox = new SourceIntegrityTestSandbox(
                    getServer(), getPort(), getUsername(), getPassword(), sandbox.getRemote() );
            siLauncher.runSilent( testSandbox, TIMEOUT_SETTINGS.TEST );
        }
        catch ( InvalidSandboxException _ )
        {
            siLauncher.getListener().getLogger().println( "Unregistered project found.  Deleting workspace contents." );
            // Don't do deleteRecursive here, because we need an empty folder to exist
            sandbox.deleteContents();
        }
        siLauncher.getListener().getLogger().println( "" );
    }
    

    public void projectInfo( SourceIntegrityLauncher siLauncher, String configPath )
    throws Throwable
    {
        siLauncher.getListener().getLogger().println( "Project information" );

        final OutputStream captureOutput = new ByteArrayOutputStream();
        SourceIntegrityProjectInfo projectInfo = new SourceIntegrityProjectInfo(
                getServer(), getPort(), getUsername(), getPassword(), configPath );
        siLauncher.run( projectInfo, null, captureOutput, TIMEOUT_SETTINGS.INFO );
        FileUtils.close( captureOutput );

        siLauncher.getListener().getLogger().println( "" );
    }


    public boolean getChanges( AbstractBuild build, SourceIntegrityLauncher siLauncher, File changelogFile, String configPath, String sandbox )
    throws Throwable
    {
        siLauncher.getListener().getLogger().println( "Changes since last non-fail build" );
        
        final ByteArrayOutputStream captureOutput = new ByteArrayOutputStream();
        Date lastNonFailBuild = new Date();
        boolean hasChanges = false;

        // Query MKS for changes since last non-fail build
        Run<?, ?> lsb = build.getPreviousNotFailedBuild();
        if ( null != lsb ) lastNonFailBuild = lsb.getTimestamp().getTime();
        final SourceIntegrityRlogCommand rlog = new SourceIntegrityRlogCommand(
                getServer(), getPort(), getUsername(), getPassword(), configPath, sandbox, lastNonFailBuild );
        captureOutput.reset();
        siLauncher.run( rlog, null, captureOutput, TIMEOUT_SETTINGS.RLOG );
        
        // Parse our output log to determine if there were any changes
        // Save changes to a log file
        if ( SourceIntegrityRlogCommand.changesDetected( captureOutput.toString() ) )
        {
            String sandboxDir = StringUtils.substringAfterLast( sandbox, File.separator );
            SourceIntegrityChangeLogSet changeSet = SourceIntegrityRlogCommand.parse(
                    build, captureOutput.toString(), siLauncher.getListener() );
            SourceIntegrityRlogCommand.writeChangeLog( changelogFile, changeSet, configPath, sandboxDir );
            hasChanges = true;
        }
        else
        {
            siLauncher.getListener().getLogger().println( "No Changes." );
        }
        
        FileUtils.close( captureOutput );
        siLauncher.getListener().getLogger().println( "" );
        return hasChanges;
    }


    public void dropAllPossibleSandboxes( SourceIntegrityLauncher siLauncher )
    throws Throwable
    {
        final FilePath workspace = siLauncher.getWorkspace();
        final TaskListener listener = siLauncher.getListener();
        final List<FilePath> subDirs = workspace.listDirectories();

        listener.getLogger().println( "Dropping all possible sandboxes" );

        // The config path for the sandbox may no longer be in our project
        // Find any project.pj files in the direct sub-dirs of our workspace
        // and try to drop them
        for ( FilePath subDir : subDirs )
        {
            if ( subDir.exists() && subDir.child("project.pj").exists() )
            {
                dropSandbox( siLauncher, false, subDir.getRemote() );
            }
        }
        
        listener.getLogger().println( "" );
    }

    
    public void dropSandbox( SourceIntegrityLauncher siLauncher, boolean failOnError, String sandbox )
    throws Throwable
    {
        siLauncher.getListener().getLogger().println( "Dropping sandbox" );

        final ByteArrayOutputStream captureOutput = new ByteArrayOutputStream();
        SourceIntegrityCommand dropSandbox = new SourceIntegrityDropSandboxCommand(
                getServer(), getPort(), getUsername(), getPassword(), sandbox );
        siLauncher.run( dropSandbox, null, captureOutput, TIMEOUT_SETTINGS.DROP );
        FileUtils.close( captureOutput );

        siLauncher.getListener().getLogger().println( "" );
    }


    public void checkoutSandbox( SourceIntegrityLauncher siLauncher, String configPath, String sandbox )
    throws Throwable
    {
        siLauncher.getListener().getLogger().println( "Creating sandbox" );

        final ByteArrayOutputStream captureOutput = new ByteArrayOutputStream();
        SourceIntegrityCreateSandboxCommand createSandbox = new SourceIntegrityCreateSandboxCommand(
                getServer(), getPort(), getUsername(), getPassword(), configPath, sandbox );
        siLauncher.run( createSandbox, null, captureOutput, TIMEOUT_SETTINGS.CHECKOUT );
        FileUtils.close( captureOutput );

        siLauncher.getListener().getLogger().println( "" );
    }


    public void resyncSandbox( SourceIntegrityLauncher siLauncher, String sandbox )
    throws Throwable
    {
        siLauncher.getListener().getLogger().println( "Resync sandbox" );

        final ByteArrayOutputStream captureOutput = new ByteArrayOutputStream();
        SourceIntegrityResyncCommand resyncSandbox = new SourceIntegrityResyncCommand(
                getServer(), getPort(), getUsername(), getPassword(), sandbox );
        siLauncher.run( resyncSandbox, null, captureOutput, TIMEOUT_SETTINGS.RESYNC );
        FileUtils.close( captureOutput );

        siLauncher.getListener().getLogger().println( "" );

        // TODO: Do we need to delete the workspace if there's an error during resync?
    }


    public void makeWritable( AbstractBuild build, SourceIntegrityLauncher siLauncher )
    throws Throwable
    {
        //attrib -R WORKSPACE\*.* /s /d
        siLauncher.getListener().getLogger().println( "Setting workspace writable" );

        final ByteArrayOutputStream captureOutput = new ByteArrayOutputStream();
        final SourceIntegrityMakeWritableCommand makeWritableCommand =
                new SourceIntegrityMakeWritableCommand( siLauncher.getLauncher() );
        siLauncher.run( makeWritableCommand, null, captureOutput, TIMEOUT_SETTINGS.WRITEABLE);
        FileUtils.close( captureOutput );

        siLauncher.getListener().getLogger().println( "" );
    }


    public void freezeProject( SourceIntegrityLauncher siLauncher, String configPath )
    throws Throwable
    {
        siLauncher.getListener().getLogger().println( "Freezing project" );

        final ByteArrayOutputStream captureOutput = new ByteArrayOutputStream();
        final SourceIntegrityFreezeCommand freezeCommand = new SourceIntegrityFreezeCommand(
            getServer(), getPort(), getUsername(), getPassword(), configPath);
        siLauncher.run( freezeCommand, null, captureOutput, TIMEOUT_SETTINGS.FREEZE );
        FileUtils.close( captureOutput );

        siLauncher.getListener().getLogger().println( "" );
    }
    

    public void thawProject( SourceIntegrityLauncher siLauncher, String configPath )
    throws Throwable
    {
        siLauncher.getListener().getLogger().println( "Thawing project" );

        final ByteArrayOutputStream captureOutput = new ByteArrayOutputStream();
        final SourceIntegrityThawCommand thawCmd = new SourceIntegrityThawCommand(
                getServer(), getPort(), getUsername(), getPassword(), configPath );
        siLauncher.run( thawCmd, null, captureOutput, TIMEOUT_SETTINGS.THAW );
        FileUtils.close( captureOutput );

        siLauncher.getListener().getLogger().println( "" );
    }


    @Override
    public ChangeLogParser createChangeLogParser()
    {
        return new SourceIntegrityChangeLogParser();
    }
    

    @Override
    public SCMRevisionState calcRevisionsFromBuild( AbstractBuild<?, ?> build, Launcher launcher, TaskListener listener )
    throws IOException, InterruptedException
    {
        return new SourceIntegrityRevisionState( build.getTimestamp() );
    }


    @Override
    protected PollingResult compareRemoteRevisionWith( AbstractProject<?, ?> project, Launcher launcher,
            FilePath workspace, TaskListener listener, SCMRevisionState baseline )
    throws IOException, InterruptedException
    {
        //if (baseline instanceof SourceIntegrityRevisionState)
        //  SourceIntegrityRevisionState currRev = (SourceIntegrityRevisionState)baseline;
        //if (project.getLastBuild()!=null)
        //    SourceIntegrityRevisionState prevRev =
        //      (SourceIntegrityRevisionState)calcRevisionsFromBuild( project.getLastBuild(), launcher, listener );
        //if ( currRev.equals(prevRev) ) return PollingResult.SIGNIFICANT;

        listener.getLogger().println( "Checking for changes" );

        // We had a non-MKS version control system previously?  Force a build
        if ( !(baseline instanceof SourceIntegrityRevisionState) )
        {
            listener.getLogger().println( "Unknown revision state - forcing build" );
            return PollingResult.BUILD_NOW;
        }

        else if (project.getLastBuild() == null)
        {
            listener.getLogger().println( "No previous build found - forcing build" );
            return PollingResult.BUILD_NOW;
        }

        // Note: We don't use the SCMRevisionState, because MKS doesn't track project revisions
        // We'd have to mark the revision of every file in that class, then compare to see if there
        // were any differences.  Simpler to just see if there were changes to project vs workspace
        else
        {
            final ByteArrayOutputStream consoleStream = new ByteArrayOutputStream();
            
            //run the rlog command to see if there were changes
            try
            {
                for ( String configPath : getPaths() )
                {
                    final FilePath projSandbox = getSandboxWorkspace( workspace, configPath );
                    final FilePath sandbox = projSandbox.child( "project.pj" );

                    SourceIntegrityRlogCommand rlog = new SourceIntegrityRlogCommand(
                            getServer(), getPort(), getUsername(), getPassword(), configPath
                            ,sandbox.getRemote() , project.getLastBuild().getTimestamp().getTime() );
                    SourceIntegrityLauncher siLauncher =
                            new SourceIntegrityLauncher( listener, workspace, launcher );
                    siLauncher.run( rlog, null, consoleStream, TIMEOUT_SETTINGS.RLOG );
                }
            }
            catch ( Throwable th )
            {
                // TODO: Can I just rethrow the exception, and reinstate the finally?
                //listener.error( "An error occurred while retrieving change log: " + th.getMessage() );
                th.printStackTrace( listener.getLogger() );
                return PollingResult.NO_CHANGES;
            }
            //finally
            //{
                final String output = consoleStream.toString();
                FileUtils.close( consoleStream );
                return SourceIntegrityRlogCommand.changesDetected( output )
                    ? PollingResult.SIGNIFICANT : PollingResult.NO_CHANGES;
            //}
        }
    }


    /**
     * @param workspace
     *      This should be the root workspace for this project
     * @param configPath
     *      The Configuration Path for the sandbox you want to retrieve.
     *      Must be one of the paths entered in the config screen
     * @return
     *      The FilePath of the sandbox for the given Configuration Path.
     */
    public FilePath getSandboxWorkspace(FilePath workspace, String configPath)
    {
        // Each project will be named $WORKSPACE/sb1, $WORKSPACE/sb2, etc,
        int iter = 0;
        for (String path : getPaths())
        {
            iter++;
            if ( path.equals(configPath) )
                return new FilePath( workspace, "project"+iter );
        }
        return null;
    }


    @Override
    public FilePath getModuleRoot(FilePath workspace)
    {
        return getSandboxWorkspace(workspace, getPaths()[0]);
    }


    @Override
    public FilePath[] getModuleRoots(FilePath workspace)
    {
        List<FilePath> projectRoots = new ArrayList<FilePath>();
        for (String configPath : getPaths())
        {
            projectRoots.add( getSandboxWorkspace(workspace,configPath) );
        }

        return (FilePath[]) projectRoots.toArray( new FilePath[0] );
    }


    @Override
    public boolean processWorkspaceBeforeDeletion( AbstractProject<?,?> project, FilePath workspace, Node node )
    throws IOException, InterruptedException
    {
        // Don't allow any errors to keep the workspace from being deleted.
        // If a sandbox doesn't get dropped, we can clean it up through MKS's GUI
        try
        {
            TaskListener listener = TaskListener.NULL;
            Launcher launcher = node.createLauncher( listener );
            SourceIntegrityLauncher siLauncher =
                    new SourceIntegrityLauncher( listener, workspace, launcher );

            dropAllPossibleSandboxes( siLauncher );
        }
        catch ( Throwable _ )
        {
            // Should be logged elswhere
        }

        return true;
    }


    @Override
    public boolean requiresWorkspaceForPolling()
    {
        return true;
    }


    @Override
    public boolean supportsPolling()
    {
        return true;
    }


    @Override
    public DescriptorImpl getDescriptor()
    {
        return PluginImpl.SI_DESCRIPTOR;
    }


    /**
     * Hook into Jelly interface
     */
    public static class DescriptorImpl
    extends SCMDescriptor<SourceIntegrityScm>
    {

        private String siExecutable;


        /**
         * Constructor
         */
        public DescriptorImpl()
        {
            super( SourceIntegrityScm.class, null );
            load();
        }


        /**
         * Creates a new instance of the SourceIntegrityScm class, feeding
         * in the values from our configuration screen
         * @param req
         * @param formData
         * @return
         * @throws hudson.model.Descriptor.FormException
         */
        @Override
        public SCM newInstance( StaplerRequest req, JSONObject formData )
        throws FormException
        {
            String server = req.getParameter( "si.server" );
            int port = Integer.valueOf(req.getParameter("si.port")).intValue();
            String[] configPaths = req.getParameterValues("si.paths");
            String username = req.getParameter("si.username").trim();
            String password = req.getParameter("si.password").trim();
            boolean freeze = "on".equals( req.getParameter( "si.freeze" ) );
            boolean cleanBeforeResync = "on".equals( req.getParameter( "si.cleanBeforeResync" ) );
            boolean makeWritable = "on".equals( req.getParameter( "si.makeWritable" ) );
            String authType = req.getParameter("si.authType").trim();

            // Create and save our MKS instance
            return new SourceIntegrityScm(server, port, configPaths, username, password
                    , freeze, cleanBeforeResync, makeWritable, authType);
        }


        /**
         * Verify that the MKS Source Integrity executable exists
         * @param req
         * @param rsp
         * @throws IOException
         * @throws ServletException
         */
        public void doExecutableCheck( StaplerRequest req, StaplerResponse rsp )
        throws IOException, ServletException
        {
            new FormFieldValidator.Executable( req, rsp ).process();
        }


        /**
         * This method is called by our jelly scripts to ensure the parameters required
         * by those scripts are satisfield
         * @param req
         * @param rsp
         * @throws IOException
         * @throws ServletException
         */
        public void doFieldCheck( StaplerRequest req, StaplerResponse rsp )
        throws IOException, ServletException
        {
            Hudson.getInstance().doFieldCheck( req, rsp );
        }


        /**
         * The name of the Plugin, to display in Hudson
         * @return
         */
        @Override
        public String getDisplayName()
        {
            return MKS_NAME;
        }


        /**
         * The path to the plugin's help file, to display in Hudson
         * @return
         */
        @Override
        public String getHelpFile()
        {
            return "/plugin/si/help-scm.html";
        }


        public String getSiExecutable()
        {
            if(siExecutable == null)
                return "si";
            else
                return siExecutable;
        }


        @Override
        public boolean configure( StaplerRequest req, net.sf.json.JSONObject json )
        throws FormException
        {
            siExecutable = Util.fixEmptyAndTrim( req.getParameter( "si.siExecutable" ) );
            save();
            return super.configure(req, json);
        }
    }
}
