package Joo;

/**
*	This file is part of joodbms.	
*	Joodbms is free software: you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation, either version 3 of the License, or
*	(at your option) any later version.
*	
*	Joodbms is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*	
*	You should have received a copy of the GNU General Public License
*	along with joodbms.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Used to establish a connection to a joo database and perform connection specific tasks
 * 
 * @author Rob Carr
 * @version 1.0.0
 */

import java.io.*;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Scanner;

public class Joo implements IJooConnection, IJooInteraction
{	
	private boolean connected;
	private String dbPath;
	private String dbName;
	private String dbDescription;
	private String dbCreated;
	
	@Override
	public boolean connect(String location) {
		/*
		 * If a connection to the joo database is successfully made then the instance variables
		 * will be populated with the appropriate data from the databases db.joo file.
		 * 
		 * Should any exceptions be caught (i.e. an IO exception when trying to access the file),
		 * the connected attribute will be set to false to indicate no connection exists and the
		 * function will return false.
		 */
		try
		{
			FileInputStream fis = new FileInputStream(location + "/db.joo");
			DataInputStream dis = new DataInputStream(fis);
			BufferedReader buffer = new BufferedReader(new InputStreamReader(dis));
			this.dbName = buffer.readLine();
			this.dbDescription = buffer.readLine();
			this.dbCreated = buffer.readLine();
			this.dbPath = location;
			this.connected = true;
			return true;
		}
		catch (Exception e)
		{
			this.connected = false;
			return false;
		}
	}

	@Override
	public String getPath() {
		return this.dbPath;
	}

	@Override
	public boolean isConnected() {
		return this.connected;
	}
	
	public String toString(){
		return this.dbName + ", " + this.dbDescription + ", " + this.dbCreated;
	}
	
	public Joo()
	{
	}
	
	public Joo(String location){
		connect(location);
	}
	
	@Override
	public boolean clusterExists(String cluster) {
		/*
		 * If a connection has been established the method will check to see if
		 * the cluster exists by accessing the cluster.joo file inside the cluster's
		 * private folder. Should the file exist the method will return true indicating
		 * that the cluster does indeed exists; if the file does not exist the method
		 * will return false, indicating that the cluster does not exist.
		 */
		if(isConnected())
		{
			File f = new File(this.dbPath + "/clusters/" + cluster + "/cluster.joo");
			if(f.exists())
			{
				return true;
			}
			else
			{
				return false;
			}
		}
		else
		{
			return false;
		}
	}

	@Override
	public boolean createCluster(String cluster) {
		/*
		 * Prior to the creation of a cluster, the clusterExists method is called to
		 * ensure that a cluster with the same name does not already exist.
		 */
		if(clusterExists(cluster))
		{
			return false;
		}
		else
		{
			/*
			 * Should the clusterExists function return false (indicating no name collisions
			 * were found) the cluster directory and the cluster.joo directory will be created.
			 * 
			 * If an IOException is caught the method will return false to indicate the cluster
			 * was not created, and no files or directories will be created.
			 */
			File f = new File(this.dbPath + "/clusters/" + cluster);
			if(f.mkdir())
			{
				f = new File(this.dbPath + "/clusters/" + cluster + "/cluster.joo");
				try {
					if(f.createNewFile())
					{
						/*
						 * cluster.joo is given a default value of "0" as the first object
						 * in the cluster should be given the ID 0
						 */
						FileWriter fs = new FileWriter(f);
						BufferedWriter fo = new BufferedWriter(fs);
						fo.write("0");
						fo.flush();
						return true;	
					}
					else
					{
						return false;
					}
				} catch (IOException e) {
					return false;
				}
			}
			else
			{
				return false;
			}
		}
	}
	
	@Override
	public Collection<String> getClusters() {
		if(isConnected())
		{
			/*
			 * If a connection to a database is established then the method
			 * will create a new File object that points to the clusters directory
			 * of the database and subsequently fetching an array of File objects
			 * which represent the directories found within the clusters directory. 
			 */
			File dbpath = new File(this.dbPath + "/clusters");
			File[] clusters = dbpath.listFiles();
			
			/*
			 * Once the array of Files has been made, each one will be added to the
			 * LinkedList that will be returned after being built.
			 */
			Collection<String> clusterCollection = new LinkedList<String>();
			for(File cluster : clusters)
			{
				if(cluster.isDirectory())
				{
					clusterCollection.add(cluster.getName());
				}
			}
			return clusterCollection;
		}
		else
		{
			// If no connection exists then return null
			return null;
		}
	}
	
	@Override
	public Object getObject(String cluster, int oid) {
		if(clusterExists(cluster))
		{
			try {
				FileInputStream fis = new FileInputStream(this.dbPath + "/clusters/" + cluster + "/" + oid + ".dat");
				ObjectInputStream ois = new ObjectInputStream(fis);
				Object obj = ois.readObject();
				return obj;
			} catch (FileNotFoundException e) {
				e.printStackTrace();
				return null;
			} catch (IOException e) {
				e.printStackTrace();
				return null;
			} catch (ClassNotFoundException e) {
				e.printStackTrace();
				return null;
			}	
		}
		else
		{
			return null;
		}
	}

	@Override
	public boolean purgeCluster(String cluster) {
		if(clusterExists(cluster))
		{
			File c = new File(this.dbPath + "/clusters/" + cluster);
			if(c.exists())
			{
				File[] f = c.listFiles();
				for(File cf : f)
				{
					cf.delete();
				}
	
				try
				{
					File idFile = new File(this.dbPath + "/clusters/" + cluster + "/cluster.joo");
					if(!idFile.createNewFile())
					{
						return false;
					}
					Writer output = new BufferedWriter(new FileWriter(idFile));
					output.write("0");
					output.close();
				}
				catch(IOException e)
				{
					return false;
				}
				
				return true;
			}
			else
			{
				return false;
			}
		}
		else
		{
			return false;
		}
	}

	private void deleteFile(File f)
	{
		if(f.exists())
		{
			if(f.isDirectory())
			{
				File[] c = f.listFiles();
				for(File cf : c )
				{
					deleteFile(cf);
				}
				f.delete();
			}
			else
			{
				f.delete();
			}
		}
	}
	
	@Override
	public boolean purgeDatabase() {
		File c = new File(this.dbPath + "/clusters");
		if(c.exists())
		{
			File[] f = c.listFiles();
			for(File cf : f)
			{
				deleteFile(cf);
			}
			
			return true;
		}
		else
		{
			return false;
		}
	}

	@Override
	public boolean removeCluster(String cluster) {
		if(clusterExists(cluster))
		{
			File c = new File(this.dbPath + "/clusters/" + cluster);
			if(c.exists())
			{
				File[] f = c.listFiles();
				for(File cf : f)
				{
					cf.delete();
				}
				
				return c.delete();
			}
			else
			{
				return false;
			}
		}
		else
		{
			return false;
		}
	}
	

	@Override
	public boolean removeObject(String cluster, int oid) {
		if(clusterExists(cluster))
		{
			File o = new File(this.dbPath + "/clusters/" + cluster + "/" + oid + ".dat");
			if(o.exists())
			{
				return o.delete();
			}
			else
			{
				return false;
			}
		}
		else
		{
			return false;
		}
	}
	

	@Override
	public boolean storeObject(String cluster, Object object) {
		if(clusterExists(cluster))
		{
			File idFile = new File(this.dbPath + "/clusters/" + cluster + "/cluster.joo");
			Scanner s;
			try {
				s = new Scanner(idFile);
			} catch (FileNotFoundException e1) {
				return false;
			}
			int id = s.nextInt();
			s.close();
			FileOutputStream fos;
			try {
				fos = new FileOutputStream(this.dbPath + "/clusters/" + cluster + "/" + id + ".dat");
				ObjectOutputStream oos = new ObjectOutputStream(fos);
				oos.writeObject(object);
				oos.flush();
				oos.close();
				id++;
				Writer output = new BufferedWriter(new FileWriter(idFile));
				output.write(Integer.toString(id));
				output.close();
				return true;
			} catch (FileNotFoundException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				return false;
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				return false;
			}			
		}
		else
		{
			return false;
		}
	}
}
