package sensorBoard.sensor;

import java.io.IOException;
import java.util.Vector;

import sensorBoard.ISensorBoardComponent;
import server.interfaces.ISensorBoardManager;

import com.CharNotValidException;
import com.Event;
import com.LedsManager;
import com.ParamNotFoundException;
import com.ParamsLabel;
import com.sun.spot.sensorboard.EDemoBoard;
import com.sun.spot.sensorboard.peripheral.ITemperatureInput;
import com.sun.spot.resources.transducers.LEDColor;

/**
 * Provides the access to the node Temperature sensor. Provides information about the temperature values 
 * and builds instances of TemperatureListener if it's necessary.
 * @author Francesco Aiello, Antonio Guerrieri, Raffaele Gravina
 *
 */
public class TemperatureSensor implements ISensorBoardComponent {
	
	private byte [] handlableEvents = {Event.TMP_CURRENT, 
			Event.TMP_THRESHOLDCHANGED, 
			Event.TMP_THRESHOLDEXCEDEED,
			Event.TMP_TERMINATION};
	private ITemperatureInput temperatureSensor;
	private ISensorBoardManager sensorBoardManager;
	private Vector temperatureListeners;
	
	/**
	 * The constructor needs to know own sensorBoardManager and the Accelerometer class provided by SunSPOT SDK
	 * @param sensorBoardManager own sensorBoardManager
	 */
	public TemperatureSensor (ISensorBoardManager sensorBoardManager) {
		this.sensorBoardManager = sensorBoardManager;
		this.temperatureSensor = EDemoBoard.getInstance().getADCTemperature();
		this.temperatureListeners = new Vector();
	}
	
	public boolean IsEventHandlable(byte eventName) {
		for (int i = 0; i < handlableEvents.length; i++) 
			if (this.handlableEvents[i] == eventName)
				return true;
		return false;
	}
	
	public void handleEvent(Event event) {
		
		
		try {
			byte eventName = event.getName();
			switch (eventName) {
				case Event.TMP_CURRENT:
					if (event.getType() == Event.NOW) {
						if (event.getParam(ParamsLabel.TMP_CELSIUS).equals("true"))
							event.setParam(ParamsLabel.TMP_TEMPERATURE_VALUE, "" + this.temperatureSensor.getCelsius());
						else
							event.setParam(ParamsLabel.TMP_TEMPERATURE_VALUE, "" + this.temperatureSensor.getFahrenheit());
						this.sensorBoardManager.send(event);
					}
				break;
				case Event.TMP_THRESHOLDEXCEDEED:
					if (event.getType() != Event.NOW && this.temperatureListeners.size() == 0) {
						String sensorID = this.sensorBoardManager.generateSensorBoardComponentID();
						this.temperatureListeners.addElement(new TemperatureListener(this, sensorID, event));
						Event sensorIDEvent = new Event("", event.getTargetID(), Event.TMP_ID, Event.NOW);
						sensorIDEvent.setParam(ParamsLabel.SNS_ID, sensorID);
						this.sensorBoardManager.send(sensorIDEvent);
					}
					else {
						Event sensorBoardComponentNotAvailableEvent = new Event("", event.getTargetID(), Event.TMP_NOT_AVAILABLE, Event.NOW);
						this.sensorBoardManager.send(sensorBoardComponentNotAvailableEvent);
					}
				break;
				case Event.TMP_TERMINATION:
					String sensorID;
					try {
						sensorID = event.getParam(ParamsLabel.SNS_ID);
						if (!sensorID.equals("")) {
							if (this.terminateListener(sensorID))
								this.sensorBoardManager.send(event);
						}
					}
					catch (ParamNotFoundException e) {
						LedsManager.error(LEDColor.YELLOW);
						e.printStackTrace();
					}
				break;
			}
		}
		catch (IOException e) {
			LedsManager.error(LEDColor.YELLOW);
			e.printStackTrace();
		}
		catch (ParamNotFoundException e) {
			LedsManager.error(LEDColor.YELLOW);
			e.printStackTrace();
		}
		catch (CharNotValidException e) {
			LedsManager.error(LEDColor.YELLOW);
			e.printStackTrace();
		}
		
	}
	
	public void send(Event backEvent) {
		this.sensorBoardManager.send(backEvent);
	}
	
	public boolean terminateListener(String sensorID) {
		for (int i = 0; i < this.temperatureListeners.size(); i++) {
			TemperatureListener temperatureListener = (TemperatureListener)this.temperatureListeners.elementAt(i);
			if ( temperatureListener.getSensorBoardComponentID().equals(sensorID) ) {
				this.temperatureListeners.removeElementAt(i);
				return temperatureListener.terminateListener(); 
			}
		}
		return false;
	}
}
