/*
nbtostring - NetBeans toString() Generator
Copyright (C) 2009 by Simon Martinelli, Gampelen, Switzerland

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package ch.simas.nbtostring;

import ch.simas.nbtostring.builder.BuilderOptions;
import ch.simas.nbtostring.builder.ToStringBuilder;
import ch.simas.nbtostring.builder.ToStringBuilderFactory;
import ch.simas.nbtostring.ui.ElementNode;
import ch.simas.nbtostring.ui.ElementNode.Description;
import ch.simas.nbtostring.ui.ErrorPanel;
import ch.simas.nbtostring.ui.ToStringPanel;
import ch.simas.nbtostring.util.Problem;
import com.sun.source.tree.AnnotationTree;
import org.netbeans.spi.editor.codegen.CodeGenerator;
import com.sun.source.tree.ClassTree;
import com.sun.source.tree.ExpressionTree;
import com.sun.source.tree.MethodTree;
import com.sun.source.tree.Tree;
import com.sun.source.tree.TypeParameterTree;
import com.sun.source.tree.VariableTree;
import com.sun.source.util.SourcePositions;
import com.sun.source.util.TreePath;
import java.awt.Dialog;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.lang.model.element.Element;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.Modifier;
import javax.lang.model.element.TypeElement;
import javax.lang.model.element.VariableElement;
import javax.lang.model.type.TypeKind;
import javax.lang.model.util.ElementFilter;
import javax.lang.model.util.Elements;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import org.netbeans.api.java.source.CompilationController;
import org.netbeans.api.java.source.Task;
import org.netbeans.api.java.source.ElementHandle;
import org.netbeans.api.java.source.JavaSource;
import org.netbeans.api.java.source.ModificationResult;
import org.netbeans.api.java.source.TreeMaker;
import org.netbeans.api.java.source.WorkingCopy;
import org.netbeans.editor.GuardedDocument;
import org.netbeans.editor.GuardedException;
import org.openide.DialogDescriptor;
import org.openide.DialogDisplayer;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.util.Exceptions;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;

// TODO: we should really think about to move all the UI stuff out of here
/**
 * Implementation of a toString() method generator
 * 
 * @author Simon
 */
public class ToStringGenerator implements CodeGenerator {

    public static final String TOSTRING = "toString";
    private final EvaluationContainer evaluationContainer;

    /**
     * Constructor
     *
     * @param evalContainer the container having all evaluation results
     */
    private ToStringGenerator(EvaluationContainer evalContainer) {
        this.evaluationContainer = evalContainer;
    }

    /**
     * Returns the name to be displayed in the context menu
     * F
     * @return
     */
    public String getDisplayName() {
        return NbBundle.getMessage(ToStringGenerator.class, "LBL_displayname");
    }

    /**
     * Method to be invoked when the users select the toString() generator
     */
    public void invoke() {

        JTextComponent component = evaluationContainer.getComponent();

        final int caretOffset = component.getCaretPosition();
        final ToStringPanel panel = new ToStringPanel(evaluationContainer.getDescription(), evaluationContainer.getfDescription());
        String title = "toString()";

        // TODO: all the following checks have to be moved to a separate method or maybe class

        Problem problem = null;
        if (!FileUtil.toFile(evaluationContainer.getFileObject()).canWrite()) {
            problem = new Problem(true, NbBundle.getMessage(ToStringGenerator.class, "MSG_targetFileReadOnly",
                    FileUtil.toFile(evaluationContainer.getFileObject()).getName()));
            problem = chainProblem(problem, null);
        }

        // let's have a look if there's already the toString() method and notify user if so...
        if (evaluationContainer.getExistingToString() != null) {
            Problem p = new Problem(false, NbBundle.getMessage(ToStringGenerator.class, "MSG_toString_exist", evaluationContainer.getClassElement().getSimpleName()));
            problem = chainProblem(problem, p);
        }

        if (evaluationContainer.getFinalSuper() != null) {
            Problem p = new Problem(false, NbBundle.getMessage(ToStringGenerator.class, "MSG_finalInSuper", evaluationContainer.getSuperClass().getSimpleName()));
            problem = chainProblem(problem, p);
        }

        if (problem != null) {
            ErrorPanel ep = new ErrorPanel(problem);
            boolean onlyNonFatal = !problem.isFatal();
            while (onlyNonFatal && problem.getNext() != null) {
                problem = problem.getNext();
                onlyNonFatal = !problem.isFatal();
            }

            DialogDescriptor dialogDescriptor = this.createDialogDescriptor(ep, title, onlyNonFatal, false);
            Dialog problemDialog = DialogDisplayer.getDefault().createDialog(dialogDescriptor);
            problemDialog.setVisible(true);

            // in case we have problems the cancel button is default
            if (dialogDescriptor.getValue() == dialogDescriptor.getDefaultValue()) {
                return;
            }
        }

        DialogDescriptor dialogDescriptor = this.createDialogDescriptor(panel, title, true, true);
        Dialog dialog = DialogDisplayer.getDefault().createDialog(dialogDescriptor);
        dialog.setVisible(true);
        if (dialogDescriptor.getValue() == dialogDescriptor.getDefaultValue()) {
            JavaSource js = JavaSource.forDocument(component.getDocument());
            if (js != null) {
                try {
                    ModificationResult mr = js.runModificationTask(new Task<WorkingCopy>() {

                        public void run(WorkingCopy copy) throws IOException {
                            copy.toPhase(JavaSource.Phase.ELEMENTS_RESOLVED);
                            TreePath path = copy.getTreeUtilities().pathFor(caretOffset);
                            path = ToStringGenerator.getPathElementOfKind(Tree.Kind.CLASS, path);
                            int idx = findClassMemberIndex(copy, (ClassTree) path.getLeaf(), caretOffset);
                            ArrayList<Element> elements = new ArrayList<Element>();
                            for (ElementHandle<? extends Element> elementHandle : panel.getVariables()) {
                                elements.add(elementHandle.resolve(copy));
                            }

                            final BuilderOptions options = new BuilderOptions(elements, panel.isArrayToString(), panel.isChainedAppend(),
                                panel.insertOverride(), panel.getToStringBuilderType(), idx);

                            generateToString(copy, path, options);
                        }
                    });
                    this.guardedCommit(component, mr);
                } catch (IOException ex) {
                    Exceptions.printStackTrace(ex);
                }
            }
        }
    }

    private Problem chainProblem(Problem orig, Problem next) {
        Problem problem;

        if (orig == null) {
            return next;
        }
        if (next == null) {
            return orig;
        }
        problem = orig;
        while (problem.getNext() != null) {
            problem = problem.getNext();
        }
        problem.setNext(next);
        return orig;
    }

    /**
     * Generates the toString method and may replaces an existing one if needed.
     *
     * @param wc the working copy of the sources
     * @param path the path of the file
     * @param options the options to use
     */
    private void generateToString(WorkingCopy wc, TreePath path, final BuilderOptions options) {

        assert path.getLeaf().getKind() == Tree.Kind.CLASS;
        TypeElement te = (TypeElement) wc.getTrees().getElement(path);
        if (te != null) {
        int index = options.getPositionOfMethod();

            TreeMaker make = wc.getTreeMaker();
            ClassTree clazz = (ClassTree) path.getLeaf();

            List<Tree> members = new ArrayList<Tree>(clazz.getMembers());
            // use an iterator to prevent concurrent modification
            for (Iterator<Tree> treeIt = members.iterator(); treeIt.hasNext();) {
                Tree member = treeIt.next();
                if (member.getKind().equals(Tree.Kind.METHOD)) {
                    MethodTree mt = (MethodTree) member;
                    // this may looks strange, but I've seen code with methods like:
                    // public String toString(Object o) {}
                    // and I think we shouldn't remove them :)
                    // so we should ensure to get right toString() method, means
                    // a return value and no parameters
                    if (mt.getName().contentEquals(TOSTRING) && mt.getParameters().isEmpty() &&
                            mt.getReturnType() != null && mt.getReturnType().getKind() == Tree.Kind.IDENTIFIER) {
                        treeIt.remove();
                        // decrease the index to use, as we else will get an ArrayIndexOutOfBounds (if added at the end of a class)
                        index--;
                        break;
                    }
                }
            }

            ToStringBuilder tsb = ToStringBuilderFactory.createToStringBuilder(options.getBuilderType());


            Set<Modifier> mods = EnumSet.of(Modifier.PUBLIC);
            List<AnnotationTree> annotations = new ArrayList<AnnotationTree>();
            if (options.isAddOverride()) {
                AnnotationTree newAnnotation = make.Annotation(
                        make.Identifier("Override"),
                        Collections.<ExpressionTree>emptyList());
                annotations.add(newAnnotation);
            }
            TypeElement element = wc.getElements().getTypeElement("java.lang.String");
            ExpressionTree returnType = make.QualIdent(element);

            MethodTree method = make.Method(make.Modifiers(mods, annotations), TOSTRING, returnType, Collections.<TypeParameterTree>emptyList(),
                    Collections.<VariableTree>emptyList(), Collections.<ExpressionTree>emptyList(),
                    tsb.buildToString(wc, clazz.getSimpleName().toString(), options), null);

            members.add(index, method);

            ClassTree nue = make.Class(clazz.getModifiers(), clazz.getSimpleName(), clazz.getTypeParameters(), clazz.getExtendsClause(),
                    (List<ExpressionTree>) clazz.getImplementsClause(), members);
            wc.rewrite(clazz, nue);
        }
    }

    /**
     * TODO
     *
     * @param wc
     * @param clazz
     * @param offset
     * @return
     */
    private int findClassMemberIndex(WorkingCopy wc, ClassTree clazz, int offset) {

        int index = 0;
        SourcePositions sp = wc.getTrees().getSourcePositions();
        GuardedDocument gdoc = null;
        try {
            Document doc = wc.getDocument();
            if (doc != null && doc instanceof GuardedDocument) {
                gdoc = (GuardedDocument) doc;
            }
        } catch (IOException ioe) {
        }

        Tree lastMember = null;
        for (Tree tree : clazz.getMembers()) {
            if (offset <= sp.getStartPosition(wc.getCompilationUnit(), tree)) {
                if (gdoc == null) {
                    break;
                }
                int pos = (int) (lastMember != null ? sp.getEndPosition(wc.getCompilationUnit(), lastMember) : sp.getStartPosition(wc.getCompilationUnit(), clazz));
                pos = gdoc.getGuardedBlockChain().adjustToBlockEnd(pos);
                if (pos <= sp.getStartPosition(wc.getCompilationUnit(), tree)) {
                    break;
                }
            }
            index++;
            lastMember = tree;
        }
        return index;
    }

    /**
     * TODO
     *
     * @param component
     * @param mr
     * @throws IOException
     */
    private void guardedCommit(JTextComponent component, ModificationResult mr) throws IOException {

        try {
            mr.commit();

        } catch (IOException e) {
            if (e.getCause() instanceof GuardedException) {
                String message = NbBundle.getMessage(ToStringGenerator.class, "ERR_CannotApplyGuarded");
                org.netbeans.editor.Utilities.setStatusBoldText(component, message);
            }
        }
    }

    /**
     * TODO
     *
     * @param content
     * @param label
     * @param enableGenerate {@code true} if the generate button should be enabled
     * @param generateIsDefault {@code true} if the generate button should be the default one, otherwise the cancel
     * button is used as default button
     * @return
     */
    private DialogDescriptor createDialogDescriptor(JComponent content, String label, boolean enableGenerate, boolean generateIsDefault) {

        JButton[] buttons = new JButton[2];
        buttons[0] = new JButton(NbBundle.getMessage(ToStringGenerator.class, "LBL_generate_button"));
        buttons[0].getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(ToStringGenerator.class, "A11Y_Generate"));
        buttons[0].setEnabled(enableGenerate);
        buttons[1] = new JButton(NbBundle.getMessage(ToStringGenerator.class, "LBL_cancel_button"));
        return new DialogDescriptor(content, label, true, buttons, buttons[generateIsDefault ? 0 : 1], DialogDescriptor.DEFAULT_ALIGN, null, null);
    }

    /**
     * TODO
     *
     * @param kind
     * @param path
     * @return
     */
    private static TreePath getPathElementOfKind(Tree.Kind kind, TreePath path) {

        return getPathElementOfKind(EnumSet.of(kind), path);
    }

    /**
     * TODO
     *
     * @param kinds
     * @param path
     * @return
     */
    private static TreePath getPathElementOfKind(EnumSet<Tree.Kind> kinds, TreePath path) {

        while (path != null) {
            if (kinds.contains(path.getLeaf().getKind())) {
                return path;
            }
            path = path.getParentPath();
        }
        return null;
    }

    public static class Factory implements CodeGenerator.Factory {

        private static final String ERROR = "<error>";

        public List<? extends CodeGenerator> create(Lookup context) {
            ArrayList<CodeGenerator> ret = new ArrayList<CodeGenerator>();
            JTextComponent component = context.lookup(JTextComponent.class);

            EvaluationContainer evaluationContainer = new EvaluationContainer(component);

            CompilationController controller = context.lookup(CompilationController.class);
            TreePath path = context.lookup(TreePath.class);
            path = path != null ? ToStringGenerator.getPathElementOfKind(Tree.Kind.CLASS, path) : null;
            if (component == null || controller == null || path == null) {
                return ret;
            }
            try {
                controller.toPhase(JavaSource.Phase.ELEMENTS_RESOLVED);
            } catch (IOException ioe) {
                return ret;
            }
            evaluationContainer.setFileObject(controller.getFileObject());
            Elements elements = controller.getElements();
            TypeElement typeElement = (TypeElement) controller.getTrees().getElement(path);
            if (typeElement == null || !typeElement.getKind().isClass()) {
                return ret;
            }
            evaluationContainer.setClassElement(typeElement);
            
            // we create separate mappings here for a complete list and list of fields as else
            // we have to clone the element in the selector panel later
            final Map<Element, List<ElementNode.Description>> complete = new LinkedHashMap<Element, List<ElementNode.Description>>();
            final Map<Element, List<ElementNode.Description>> fieldsOnly = new LinkedHashMap<Element, List<ElementNode.Description>>();

            final List<? extends Element> enclosedElements = typeElement.getEnclosedElements();
            Map<String, List<ExecutableElement>> methods = new HashMap<String, List<ExecutableElement>>();
            for (ExecutableElement method : ElementFilter.methodsIn(elements.getAllMembers(typeElement))) {

                boolean isEnclosedToString = false;
                if (enclosedElements.contains(method) && "toString".equals(method.getSimpleName().toString()) && method.getParameters().isEmpty()) {
                    evaluationContainer.setExistingToString(method);
                    isEnclosedToString = true;
                }

                if (!isEnclosedToString && method.getParameters().isEmpty() && method.getReturnType().getKind() != TypeKind.VOID &&
                        !"clone".equals(method.getSimpleName().toString())) {

                    addDescription(method, complete);
                }

                // check the method is not overriden as final in a super class
                // if so - return immediately
                // we can do it this way, as ElementFilter.methodsIn() delivers all methods, even the inherited ones
                // an override of toString in the current class was already checked in the above block
                //
                // notes on the conditions:
                // 1. it has to have the name toString()
                // 2. no parameters must be present
                // 3. it has to be final
                // 4. and of course it must not be defined in this file
                if (TOSTRING.equals(method.getSimpleName().toString()) && method.getParameters().isEmpty() &&
                        method.getModifiers().contains(Modifier.FINAL) && !enclosedElements.contains(method)) {
                    evaluationContainer.setFinalSuper(controller.getTrees().getTree(method));

                    evaluationContainer.setSuperClass(controller.getElementUtilities().enclosingTypeElement(method));
                }

                List<ExecutableElement> l = methods.get(method.getSimpleName().toString());
                if (l == null) {
                    l = new ArrayList<ExecutableElement>();
                    methods.put(method.getSimpleName().toString(), l);
                }
                l.add(method);
            }
            
            for (final VariableElement variableElement : ElementFilter.fieldsIn(elements.getAllMembers(typeElement))) {
                if (ERROR.contentEquals(variableElement.getSimpleName())) {
                    continue;
                }
                addDescription(variableElement, complete);
                addDescription(variableElement, fieldsOnly);

            }

            Description plainList = getPlainList(complete, typeElement);
            if(plainList != null) {
                evaluationContainer.setCompleteDescription(plainList);
            }

            plainList = getPlainList(fieldsOnly, typeElement);
            if(plainList != null) {
                evaluationContainer.setFieldsOnlyDescription(plainList);
            }

            ret.add(new ToStringGenerator(evaluationContainer));
            return ret;
        }

        private Description getPlainList(final Map<Element, List<ElementNode.Description>> mapToCreateFrom, final Element typeElement) {
            if (!mapToCreateFrom.isEmpty()) {
                final List<ElementNode.Description> descriptions = new ArrayList<ElementNode.Description>();
                // the map is not used any further, so we remove the typeElement's mapping
                // and add it after the loop to ensure, the typeElement's elements will
                // be on top of throws ereturn list later
                List<Description> head = mapToCreateFrom.remove(typeElement);
                for (final Map.Entry<Element, List<ElementNode.Description>> entry : mapToCreateFrom.entrySet()) {
                    descriptions.add(ElementNode.Description.create(entry.getKey(), entry.getValue(), false, false));
                }
                descriptions.add(ElementNode.Description.create(typeElement, head, false, false));

                Collections.reverse(descriptions);
                return ElementNode.Description.create(typeElement, descriptions, false, false);
            }
            return null;
        }

        private void addDescription(final Element element, final Map<Element, List<ElementNode.Description>> mapToAddTo) {
            final ElementNode.Description description = ElementNode.Description.create(element, null, true, false);
            List<ElementNode.Description> descriptions = mapToAddTo.get(element.getEnclosingElement());

            if (descriptions == null) {
                descriptions = new ArrayList<ElementNode.Description>();
                mapToAddTo.put(element.getEnclosingElement(), descriptions);
            }
            descriptions.add(description);
        }
    }

    /**
     * A class used to map all the evaluation results, so we do not need to pass all that results to the constructor.
     */
    // TODO: may we should refactor this class out
    // TODO: add documentation
    private static class EvaluationContainer {

        private JTextComponent component;
        private ElementNode.Description cDescription;
        private ElementNode.Description fDescription;

        public Description getfDescription() {
            return fDescription;
        }

        public void setFieldsOnlyDescription(Description fDescription) {
            this.fDescription = fDescription;
        }
        private Element existingToString;
        private MethodTree finalSuperImpl;
        private Element classElement;
        private TypeElement superClass;
        private FileObject fileObject;

        public FileObject getFileObject() {
            return fileObject;
        }

        public void setFileObject(FileObject fileObject) {
            this.fileObject = fileObject;
        }

        public TypeElement getSuperClass() {
            return superClass;
        }

        public void setSuperClass(TypeElement superClass) {
            this.superClass = superClass;
        }

        public Element getClassElement() {
            return classElement;
        }

        public void setClassElement(Element classElement) {
            this.classElement = classElement;
        }

        public MethodTree getFinalSuper() {
            return finalSuperImpl;
        }

        public void setFinalSuper(MethodTree finalSuper) {
            this.finalSuperImpl = finalSuper;
        }

        protected EvaluationContainer(JTextComponent component) {
            this.component = component;
        }

        public Element getExistingToString() {
            return existingToString;
        }

        public void setExistingToString(Element existingToString) {
            this.existingToString = existingToString;
        }

        public JTextComponent getComponent() {
            return component;
        }

        public Description getDescription() {
            return cDescription;
        }

        public void setCompleteDescription(Description description) {
            this.cDescription = description;
        }
    }
}
