/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package com.ohell.threads;

import com.ohell.communication.servertoclient.GameStartingObject;
import com.ohell.communication.servertoclient.InitNextRoundObject;
import com.ohell.data.Announce;
import com.ohell.data.Dealer;
import com.ohell.data.GameTable;
import com.ohell.data.OhellGame;
import com.ohell.data.Player;
import com.ohell.data.PlayerInRound;
import com.ohell.data.Round;
import com.ohell.relations.GamePlayerRel;
import com.ohell.server.service.AnnounceService;
import com.ohell.server.service.CardsService;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 *
 * @author vasko
 */
public class OhellTableCommunicationMediator {

    /*can be used for validation of number of players sitting on the table*/
    private boolean isGameStarted = false;
    private static final int PLAYERS_COUNT = 4;
    private GameTable table;
    private List<OhellPlayerThread> playerThreads;
    private int readyPlayersCount;
    private OhellGame game;

    private Map<Integer, Boolean> roundOverMap;

    public OhellTableCommunicationMediator(GameTable table) {
        this.table = table;
        playerThreads = new ArrayList<OhellPlayerThread>(PLAYERS_COUNT);
        readyPlayersCount = 0;

        roundOverMap = new HashMap<Integer, Boolean>();
        refreshRoundOverMap();
    }

    private void refreshRoundOverMap(){
        roundOverMap.clear();
        roundOverMap.put(0, false);
        roundOverMap.put(1, false);
        roundOverMap.put(2, false);
        roundOverMap.put(3, false);
    }

    public OhellGame getGame() {
        return game;
    }

    public void setGame(OhellGame game) {
        this.game = game;
    }

    public GameTable getTable() {
        return table;
    }

    public void setTable(GameTable table) {
        this.table = table;
    }

    public synchronized void updateCommunicationState(long senderID, Object o) {
        for (OhellPlayerThread ohellPlayerThread : playerThreads) {
            if (ohellPlayerThread.getPlayer() != null && senderID != ohellPlayerThread.getPlayer().getId()) {

                //here all appropriate observers of the ChatMediator are updated with the new ChatState
                ohellPlayerThread.sendObject(o);
            }
        }
    }

    //IMPORTANT - synchronization is needed - game may start inappropriately or multiple times
    public synchronized void updateReadyPlayerState(boolean playerIsReady) {
        if (playerIsReady) {
            if (!isGameStarted) {
                readyPlayersCount++;
                if (readyPlayersCount == PLAYERS_COUNT) {
                    //this call should be executed in separate thread, at the moment it is executed in
                    //the last ready player thread - can this lag for that player
                    startGame();
                    isGameStarted = true;
                }
            } else {
                //TODO : game has already started - handle error case
            }
        } else {
            readyPlayersCount--;
        }
    }

    public synchronized void updateRoundOverState(int playerSeatIndex){
        System.out.println("RoundOver submited by: "+playerSeatIndex);
        roundOverMap.put(playerSeatIndex, true);
        if(!roundOverMap.containsValue(false)){
            //Round is confirmed to be over by all players - init next round            
            if(!game.isGameOver()){
                System.out.println("game not over - will send next round because current is over.");
                refreshRoundOverMap();
                game.initNextRound();
                sendInitNextRoundObject();
            }else{
                sendGameOverMessage();
            }
        }else{
            //nothing here
            System.out.println("RoundOver submitted by seatIndex: "+playerSeatIndex);
        }
    }

    private void sendGameOverMessage(){
        //TODO : implement GameOver server behaviour.
    }

    private void startGame() {
        List<Player> players = new ArrayList<Player>();
        for (int i = 0; i < playerThreads.size(); i++) {
            players.add(playerThreads.get(i).getPlayer());
        }

        Dealer dealer = new Dealer(CardsService.getInstance().getCardsFromDB());
        //players are shuffled in the Game constructor
        game = new OhellGame(players, dealer);

        sendGameStartObject();

        
        sendInitNextRoundObject();
    }

    public synchronized void sendInitNextRoundObject() {        
        Round currentRound = game.currentRound();
        System.out.println("sendInitNextRoundObject(): "+currentRound.getRoundNumber());
        for (PlayerInRound playerInRound : currentRound.getPlayers()) {
            InitNextRoundObject inro = new InitNextRoundObject(
                    currentRound.getRoundNumber(), playerInRound.getCards(), currentRound.getTrump());

            for (OhellPlayerThread ohellPlayerThread : playerThreads) {
                if (ohellPlayerThread.getPlayer() != null && playerInRound.getPlayer().getId() == ohellPlayerThread.getPlayer().getId()) {

                    //here all appropriate observers of the ChatMediator are updated with the new ChatState
                    ohellPlayerThread.sendObject(inro);
                }
            }
        }
    }

    /**
     * Prepare client for the game - send Announces and the SeatIndex of the player in the currently starting game.
     * With this seatIndex, the client application will determine when is the time for the player to interact.
     */
    private void sendGameStartObject() {
        List<Announce> announces = AnnounceService.getInstance().getAnnouncesFromDB();
        GameStartingObject gso = new GameStartingObject();
        gso.setAnnounces(announces);

        for (OhellPlayerThread ohellPlayerThread : playerThreads) {
            Player p = ohellPlayerThread.getPlayer();
            long id = p.getId();
            int seatIndex = getSeatIndex(p);
            gso.getSeatIndexMap().put(seatIndex, id);
        }
        for (OhellPlayerThread ohellPlayerThread : playerThreads) {
            ohellPlayerThread.sendObject(gso);
        }
    }

    private int getSeatIndex(Player player) {
        for (GamePlayerRel gamePlayerRel : game.getGamePlayerRels()) {
            if (gamePlayerRel.getPlayer().equals(player)) {
                return gamePlayerRel.getSeatIndex();
            }
        }
        //this should never be reached
        return -1;
    }

    public long getPlayerIdFromSeatIndex(int seatIndex) {
        for (GamePlayerRel gamePlayerRel : game.getGamePlayerRels()) {
            if (gamePlayerRel.getSeatIndex() == seatIndex) {
                return gamePlayerRel.getPlayer().getId();
            }
        }
        //this should never be reached
        return -1;
    }

    public boolean subscribeCommunicationObserver(OhellPlayerThread player) {
        if (playerThreads.size() >= PLAYERS_COUNT) {
            return false;
        } else {
            table.getCurrentPlayers().add(player.getPlayer());
            return playerThreads.add(player);
        }
    }

    public boolean unsubscribeCommunicationObserver(OhellPlayerThread player) {
        return playerThreads.remove(player);
    }
}
