package control;

import java.awt.event.KeyEvent;
import model.interfaces.commandTypes.ArmyArgs;
import model.interfaces.commandTypes.DirectionArgs;
import model.interfaces.commandTypes.MoveArgs;
import model.interfaces.commandTypes.NoArgs;
import model.interfaces.selectable.ControllableModel;
import model.interfaces.selectable.ControllablePlayer;
import model.interfaces.selectable.SelectableCommand;
import model.interfaces.selectable.SelectableStructure;
import view.View;

/**
 * Manages the keys for the StructureOverview. The arrowkeys are used to select
 * different structures.
 * 
 * @author Ross Nichols
 */

class StructureOverviewKeyAdapter extends GameKeyAdapter
{
    private View view;
    private ControllablePlayer player;
    private SelectableStructure[] structures;
    private SelectableStructure structure;
    
    /**
     * Upon construction, files away the Player and View.
     */
    public StructureOverviewKeyAdapter( ControllableModel m, View v ) {
        player = m.getPlayer();
        view = v;
    }
    
    /**
     * Upon activation, the overview is opened. Nothing is highlighted by
     * default.
     */
    protected void doActivation()
    {
        System.out.println( "SO activated" );
        structures = player.getStructures();
        structure = null;
        view.openOverview( "Structure Overview", transform( structures ), new String[]
        {
            "Overview"
        } );
        view.closeCommandQueue();
        view.setSelectionIndex( -1 );
    }
    
    /**
     * Upon deactivation, the overview is closed.
     */
    protected void doDeactivation()
    {
        System.out.println( "SO deactivated" );
        view.closeOverview();
    }
    
    /**
     * Transformes a SS[] into a String[] based on its ID and its name.
     */
    private String[] transform( SelectableStructure[] structures )
    {
        String[] ret = new String[ structures.length ];
        for ( int i = 0; i != ret.length; ++i )
            ret[ i ] = String.format( "%s %d", structures[ i ].getName(), structures[ i ].getID() );
        
        return ret;
    }
    
    /**
     * Transforms a SC[] into a String[] based on its name.
     */
    private String[] transform( SelectableCommand[] commands )
    {
        String[] ret = new String[ commands.length ];
        for ( int i = 0; i != ret.length; ++i )
            ret[ i ] = commands[ i ].getName();
        
        return ret;
    }
    
    /**
     * Returns the index of an object in an array of the same type.
     */
    private < T > int indexOf( T[] a, T t )
    {
        for ( int i = 0; i != a.length; ++i )
            if ( a[ i ] == t ) return i;
        return -1;
    }
    
    /**
     * Processes the input.
     * 
     * UP/DOWN traverse the list. D disbands the selected army. N flags the
     * system to expect a number corresponding to the ID of a new RP. 1-0 adds a
     * new RP of that ID, if the system was correctly flagged.
     */
    public void keyPressed( KeyEvent e )
    {
        int code = e.getKeyCode();
        if ( (code == KeyEvent.VK_UP || code == KeyEvent.VK_DOWN) && structures.length > 0 )
        {
            int index;
            if ( structure == null )
            {
                index = code == KeyEvent.VK_DOWN ? 0 : structures.length - 1;
            }
            else
            {
                int offset = code == KeyEvent.VK_DOWN ? 1 : -1;
                index = (indexOf( structures, structure ) + offset) % structures.length;
                if ( index < 0 ) index += structures.length;
            }
            
            structure = structures[ index ];
            view.setSelectionIndex( index );
            view.openCommandQueue( transform( structure.getQueue() ) );
        }
    }
    
    /**
     * On tick, make sure our army still exists and update its command queue.
     */
    public void onTick()
    {
        structures = player.getStructures();
        view.setOverview( transform( structures ) );
        int index = indexOf( structures, structure );
        
        if ( index == -1 ) structure = null;
        
        if ( structure == null )
        {
            view.setSelectionIndex( -1 );
            view.closeCommandQueue();
        }
        else
        {
            view.setSelectionIndex( index );
            view.setCommandQueue( transform( structure.getQueue() ) );
        }
    }
    
    public void onGameBegin()
    {
        // not needed
    }
    
    public void onGameLose()
    {
        // not needed
    }
    
    public void onGameWin()
    {
        // not needed
    }
    
    public void visitMoveArgs( MoveArgs args )
    {
        // not needed
    }
    
    public void visitArmyArgs( ArmyArgs args )
    {
        // not needed
    }
    
    public void visitDirectionArgs( DirectionArgs args )
    {
        // not needed
    }
    
    public void visitNoArgs( NoArgs args )
    {
        // not needed
    }
}
