package chat;

import java.io.InputStream;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Locale;

import net.jxta.discovery.DiscoveryEvent;
import net.jxta.discovery.DiscoveryListener;
import net.jxta.discovery.DiscoveryService;
import net.jxta.document.AdvertisementFactory;
import net.jxta.endpoint.Message;
import net.jxta.endpoint.StringMessageElement;
import net.jxta.id.IDFactory;
import net.jxta.peergroup.PeerGroup;
import net.jxta.peergroup.PeerGroupFactory;
import net.jxta.pipe.InputPipe;
import net.jxta.pipe.OutputPipe;
import net.jxta.pipe.PipeMsgEvent;
import net.jxta.pipe.PipeMsgListener;
import net.jxta.pipe.PipeService;
import net.jxta.protocol.DiscoveryResponseMsg;
import net.jxta.protocol.PeerAdvertisement;
import net.jxta.protocol.PipeAdvertisement;
import net.jxta.rendezvous.RendezVousService;

import org.apache.log4j.Logger;



/**
 * Classe principal para aplicao do chat P2P, implementa as interfaces
 * PipeMsgListener para recebimento de eventos de pipes e DiscoveryListener para
 * eventos de Discovery de novos peers.
 */
public class ChatApp implements PipeMsgListener, DiscoveryListener {
	/**
	 * Nome do elemento que definir qual advertisement de pipe pertence a este
	 * chat.
	 */
	public static final String CHAT_NAME_TAG = "ChatMack";
	/**
	 * Tempo de vida do advertisement de pipe na rede. Aps isso o chat parar
	 * de funcionar.
	 */
	private static final long ADV_LIFETIME = 210 * 1000;
	/**
	 * 80 Frame responsvel por exibir e enviar as mensagens para outros
	 * usurios.
	 */
	private static ChatFrame chatFrame;
	/**
	 * Frame responsvel por recuperar o nick escolhido.
	 */
	private static ChatLoginFrame chatLoginFrame;
	/**
	 * Log 'chat' definido em classes/log4j.xml.
	 */
	private static Logger log = Logger.getLogger("chat");
	/**
	 * Nome do elemento que conter mensagem enviada por um usurio.
	 */
	private static final String MESSAGE_TAG = "ChatSenderMessage";
	/**
	 * Nome do elemento que conter o nome do enviador da mensagem
	 */
	private static final String SENDER_NAME = "ChatSenderName";
	/**
	 * Tempo de espera antes de tentar recuperar aps um advertisement aps um
	 * publish.
	 */
	private static final int WAITING_TIME = 5 * 1000;
	private DiscoveryService discovery;
	private PeerGroup group;
	/**
	 * Indica se o flush de advertisements j foi realizado.
	 */
	private boolean isFlushed;
	/**
	 * Indica se o usurio atual j esta logado.
	 */
	private boolean loggedIn;
	/**
	 * Nick escolhido pelo usurio.
	 */
	private String nick;
	private PipeService pipeSvc;
	private RendezVousService rdv;
	/**
	 * Ir procurar remotamente novos advertisements.
	 */
	private Thread thread;
	/**
	 * 81 Utilizado para recebimento das mensagens enviadas para este usurio.
	 */
	private PipeAdvertisement userAdv;

	/**
	 * Mtodo para exibio formatada de data
	 * 
	 * @return String contendo data no formato dd/mm/aa hh:mm
	 * @see <code>java.text.DateFormat#getDateTimeInstance(int, int,
java.util.Locale)</code>
	 */
	public static String getShortDate() {
		DateFormat df = DateFormat.getDateTimeInstance(DateFormat.SHORT,
				DateFormat.SHORT, new Locale("pt", "BR"));
		return df.format(new Date(System.currentTimeMillis()));
	}

	public static void main(String args[]) {
		ChatApp chat = new ChatApp();
		log.debug("Aplicao iniciada...");
		chat.startJxta();
	}

	/**
	 * Cria um input pipe e adiciona a prpria classe como ouvinte de eventos
	 * recebido por este pipe
	 */
	private void createInputPipe() {
		InputPipe pipeIn = null;
		try {
			pipeIn = pipeSvc.createInputPipe(userAdv, this);
		} catch (Exception e) {
			log.fatal("Impossvel criar pipe de entrada:" + e.getMessage());
			System.exit(-1);
		}
		log.info("Pipe de entrada criado com sucesso...");
	}

	/**
	 * Recebe os eventos de discovery para esta classe. Apenas registra no log
	 * os advertisements recebidos pertencente a este chat, e mostra quais foram
	 * as respostas recebidas e por quem foram enviadas
	 * 
	 * @param ev
	 *            Evento de Discovery recebido
	 */
	public void discoveryEvent(DiscoveryEvent ev) {
		DiscoveryResponseMsg res = ev.getResponse();
		String name = "Unknown";
		PeerAdvertisement peerAdv = res.getPeerAdvertisement();
		if (peerAdv != null) {
			name = peerAdv.getName();
		}
		log.debug(res.getResponseCount()+ " resposta[s] de discovery recebida[s] de "+ name+ "...");
		PipeAdvertisement adv = null;
		Enumeration enum1 = res.getAdvertisements();
		if (enum1 != null) {
			while (enum1.hasMoreElements()) {
				try {
					adv = (PipeAdvertisement) enum1.nextElement();
					log.debug("Pipe para usurio '"+ adv.getName()+ "' recuperado no evento dediscovery:\n"+ adv.toString());
				} catch (Exception e) {
					
				}
			}
		}
}

	/**
	 * Procura por advertisements de pipe de um determinado usurio localmente e
	 * remotamente
	 * 
	 * @param name
	 *            Nick de qual usurio que desejamos procurar advs
	 * @return <code>PipeAdvertisement</code> contendo o advertisement
	 *         recuperado para este usurio ou null, caso nenhum tenha sido
	 *         encontrado.
	 * @see #findUserAdvLocal(String).
	 * @see #findUserAdvRemote(String).
	 */
	private PipeAdvertisement findUserAdv(String name) {
		PipeAdvertisement adv = findUserAdvLocal(name);
		if (adv == null) {
			adv = findUserAdvRemote(name);
			adv = findUserAdvLocal(name);
		}
		return adv;
	}

	/**
	 * Realiza o flush dos advertisements locais, e recupera os advertisements
	 * ainda validos que correspondam a esta aplicao de chat, procurando por
	 * um usurio especfico.
	 * 
	 * @param name
	 *            Nick do usurio desejado.
	 * @return null ou PipeAdvertiment j atribudo a este usurio
	 *         anteriormente.
	 */
	private PipeAdvertisement findUserAdvLocal(String name) {
		Enumeration enum1 = null;
		try {
			if(isFlushed){
				discovery.flushAdvertisements(null,DiscoveryService.ADV);
				log.info(" flush de advertisements executados com sucesso...");
			}
		enum1 = discovery.getLocalAdvertisements(DiscoveryService.ADV,"Name",CHAT_NAME_TAG + ":" + nick);
		if (enum1 != null) {
			PipeAdvertisement adv = null;
			while (enum1.hasMoreElements()) {
				try {
					adv = (PipeAdvertisement)
							enum1.nextElement();
					if ((CHAT_NAME_TAG + ":" + name).equals(adv.getName())) {
						return adv;
					}
				} catch (Exception e) {
					
				}
			}
		}
	} catch (Exception e) {
		log.error("Erro ao recuperar advertisements de cache:" +e.getMessage());
	}
		return null;
	}

	/**
	 * Procura remotamente os advertisements de pipe desta aplicao e que
	 * correspondam a um determinado usurio informado.
	 * 
	 * @param name
	 *            Nick do usurio desejado
	 * @return null ou <code>PipeAdvertiment</code> deste usurio vlido na rede
	 *         JXTA
	 */
	private PipeAdvertisement findUserAdvRemote(String name) {
		PipeAdvertisement adv = null;
		try {
			// ache todos os advertisiments locais de pipes publicados por esta
			// aplicao....
			discovery.getRemoteAdvertisements(null, DiscoveryService.ADV,
					"Name", "*" + CHAT_NAME_TAG + ":*", 2);
			try {
				Thread.sleep(WAITING_TIME);
			} catch (Exception e) {

			}
		} catch (Exception e) {
			log.error("Erro ao recuperar advertisements remotos:"
					+ e.getMessage());
		}
		return null;
	}

	/**
	 * Recupera todos os advertisements locais desta aplicao, verificando qual
	 * corresponde a PipeAdvertisement.
	 * 
	 * @return <code>Collection</code> contendo todos os PipeAdvertisement
	 *         encontrados.
	 */
	private Collection getAllPipeLocalAdvertisements() {
		Collection pipes = new ArrayList();
		try {
			Enumeration enum1 = discovery.getLocalAdvertisements(DiscoveryService.ADV,"Name","*" + CHAT_NAME_TAG + ":*");
			if (enum1 != null) {
				while (enum1.hasMoreElements()) {
					try {
						PipeAdvertisement adv = (PipeAdvertisement)
								enum1.nextElement();
						pipes.add(adv);
					} catch (Exception e) {
						continue;
					}
				}
			}
		} catch (Exception e) {
			log.error("Erro ao recuperar cache de advertisements:" +e.getMessage());
		}
		return pipes;
	}

	public String getNick() {
		return nick;
	}

	/**
	 * Efetua o login na aplicao de chat, criando um PipeAdvertisement para
	 * este usurio, e publicando este pipe localmente e remotamente na rede
	 * JXTA. Aps realizada a publicao do advertisement,o frame que exibe e
	 * permite o envio de mensagens  exibido, uma mensagem para todos  enviada
	 * alertando a presena de um novo usurio, e uma thread  iniciada para
	 * descoberta de novos advertisements na rede JXTA. Caso haja algum problema
	 * antes de terminar a rotina, a aplicao  encerrada com cdigo de erro
	 * -1.
	 * 
	 * @param name
	 *            Nick escolhido pelo usurio na tela de login
	 * @return <code>int</code> , -1 caso o usurio j tenha realizado o login
	 *         anteriormente, ou 1 caso tudo tenha sido realizado com sucesso.
	 *         85
	 */
	public int loginChat(String name) {
		if (!loggedIn) {
			this.nick = name;
			discovery.addDiscoveryListener(this);
			PipeAdvertisement adv = findUserAdv(name);
			if (adv != null) {
				log.fatal("Erro, Advertisiment j existente para este usurio, saindo da aplicao...");
				System.exit(-1);
			}
			try {//criando um adv de pipe para este user...
				adv = (PipeAdvertisement) 
						AdvertisementFactory.newAdvertisement(PipeAdvertisement.getAdvertisementType());
				adv.setPipeID(IDFactory.newPipeID(group.getPeerGroupID()));
				adv.setName(CHAT_NAME_TAG + ":" + name);
				adv.setType(PipeService.PropagateType);
			} catch (Exception e) {
				log.fatal("Impossvel criar advertisement de pipe:" +e.getMessage());
				System.exit(-1);
			}
			try {
				discovery.publish(adv, ADV_LIFETIME,ADV_LIFETIME);
				discovery.remotePublish(adv,DiscoveryService.ADV);
			} catch (Exception e) {
				log.fatal("Impossvel publicar advertisement de pipe:"+ e.getMessage());
				System.exit(-1);
			}adv = findUserAdvLocal(name);
			if (adv == null) {
				log.fatal("Impossvel recuperar advertisement de pipe para este usurio...");
				System.exit(-1);
			}
			userAdv = adv;
			loggedIn = true;
			createInputPipe();
			chatFrame.setTitle(getNick() + " - JXTA Chat P2P ");
			chatFrame.show();
			sendMessageToAll(" Entrei no chat....");
			//thread que vai recuperar novos pipes de tempos em tempos...
			log.info("Criando thread para Discovery de pipes...");
			DiscoveryPipes discoveryPipes = new DiscoveryPipes(discovery);
			} else {
				log.fatal("Ateno , voc j esta logado no chat...");
				return -1;
			}
		return 1;
	}

	public void pipeMsgEvent(PipeMsgEvent pipeMsgEvent) {
		try {
			Message message = pipeMsgEvent.getMessage();
			String from = message.getMessageElement(SENDER_NAME).toString();
			String msg = message.getMessageElement(MESSAGE_TAG).toString();
			chatFrame.printMessage(msg, from);
		} catch (Exception e) {
			log.error("Erro ao receber evento de pipe:" + e.getMessage());
		}
	}

	public void sendMessageToAll(String texto) {
		Collection pipes = getAllPipeLocalAdvertisements();
		if (pipes != null) {
			Iterator it = pipes.iterator();
			while (it.hasNext()) {
				PipeAdvertisement adv = (PipeAdvertisement) it.next();
				OutputPipe pipeOut = null;
				try {
					pipeOut = pipeSvc.createOutputPipe(adv, 100);
				} catch (Exception e) {
					log.error("Erro ao criar OutputPipe para '" + adv.getName()
							+ "':" + e.getMessage());
				}
				if (pipeOut != null) {
					Message msg = null;
					String userInput = null;
					InputStream inputStream = null;
					try {
					    msg = (Message) pipeSvc.createInputPipe(adv, this);
						// inputStream=new ByteArrayInputStream()
						StringMessageElement sme = new StringMessageElement(SENDER_NAME, nick, null);
						StringMessageElement sme2 = new StringMessageElement(MESSAGE_TAG, texto, null);
						msg.replaceMessageElement(sme);
						msg.replaceMessageElement(sme2);
						pipeOut.send(msg);
					} catch (Exception e) {
						log.error("Erro ao enviar mensagem para '"
								+ adv.getName() + "':" + e.getMessage());
					}
				}
			}
		}
	}

	public void startJxta() {
		try {
			group = PeerGroupFactory.newNetPeerGroup();
			log.info("NetPeerGroup iniciado com sucesso...");
		} catch (Exception e) {
			log.fatal("Erro ao iniciar netPeerGroup:" + e.getMessage());
			System.exit(-1);
		}
		discovery = group.getDiscoveryService();
		rdv = group.getRendezVousService();
		log.info("Rendezvous service recuperado...");
		while (rdv.isConnectedToRendezVous()) {
			try {
				log.info("Aguardando conexo com Rendezvous...");
				Thread.sleep(2000);
			} catch (InterruptedException e) {

			}
		}
		log.info("Conexo com Rendezvous estabelecida, recuperando servio de pipe...");
		pipeSvc = group.getPipeService();
		chatLoginFrame = new ChatLoginFrame(this);
		chatFrame = new ChatFrame(this);
	}
}
