/*
 * Copyright 2009 Zero Separation
 *
 *     This file is part of PDSSQLService.
 *
 *  PDSSQLService is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  PDSSQLService is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with PDSSQLService.  If not, see <http://www.gnu.org/licenses/>.
 *
 */


package com.zero_separation.pds.sql;

/**
 * The SQLManager provides access to a connected SQL server from Project
 * Darkstar. Queries can be issued from a Darkstar Task, processed
 * asynchronously and then the result received as a new Darkstar Task.
 *
 * @author Tim Boura - Zero Separation
 */
public interface SQLManager {


    /**
     * Create a new SQL connection for use by this PDS server. New connections
     * can be used from any node in the cluster so long as the required driver
     * and connectivity configuration is in place.
     *
     * @param driver The class name for the driver. (e.g. com.mysql.jdbc.Driver)
     * @param connectionURL The JDBC connection string to use to connect to the database
     * @return A new SQLConnection object that can be used to make queries to the
     *  specified database.
     */
    SQLConnection createConnection(String driver, String connectionURL);


    /**
     * Perform the specified query with no notification on success or failure
     * and no result set.
     *
     * Once this method call has completed the SQLStatement object can be modified
     * or deleted without changing the query that will be performed.
     *
     * @param connection The SQL connection to use
     * @param statement The statement to execute
     */
    void performQuery(SQLConnection connection, SQLStatement statement);


    /**
     * Perform the specified statement and then call the specified SQLResultHandler
     * with the results.
     *
     * Once this method call has completed the SQLStatement object can be modified
     * or deleted without changing the query that will be performed.
     *
     * @param connection The SQL connection to use
     * @param statement The statement to execute
     * @param resultHandler The object to call back
     */
    void performQuery(SQLConnection connection, SQLStatement statement, SQLResultHandler resultHandler);


}
