/*
 * Copyright 2009 Zero Separation
 *
 *     This file is part of PDSSQLService.
 *
 *  PDSSQLService is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  PDSSQLService is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with PDSSQLService.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package com.zero_separation.pds.sql.test;

import com.sun.sgs.app.AppContext;
import com.sun.sgs.app.AppListener;
import com.sun.sgs.app.ClientSession;
import com.sun.sgs.app.ClientSessionListener;
import com.sun.sgs.app.ExceptionRetryStatus;
import com.sun.sgs.app.ManagedObject;
import com.sun.sgs.app.ManagerNotFoundException;
import com.sun.sgs.app.Task;
import com.zero_separation.pds.sql.SQLConnection;
import com.zero_separation.pds.sql.SQLManager;
import com.zero_separation.pds.sql.SQLResult;
import com.zero_separation.pds.sql.SQLResultHandler;
import com.zero_separation.pds.sql.SQLStatement;
import java.io.Serializable;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * A very basic test of the PDS SQL Service.
 * 
 * In order to set up the test environment:
 *      Extract the PDS server to the same folder containing the root PDSSQLService folder
 *      Install the MySQL ConnectorJ jar file in the PDS libs folder.
 *      Install the jar file containing AsyncTaskService to the PDS extensions folder
 *      Build PDSSQLService and install the jar file in the PDS extensions folder.
 *      Create a MySQL server on localhost
 *      Create an empty database called pdssqltest in that server
 *      Create a user called "PDSSQLTest" with password "testing" on the server
 *      Give the user PDSSQLTest full access to the empty database
 *
 * In order to run the test (on windows):
 *      Run either:
 *          TestSQLService.bat
 *      or:
 *          TestSQLService To File.bat
 *
 * The second will cause all logging output to be written to a file "output.txt"
 * in the current folder.
 *
 * After a failed run of the test it may be necessary to delete all the contents
 * of the SQL database and/or the PDS datastore. Both should always be empty for
 * the start of a test run. If the test runs succesfully it will clean up after
 * itself.
 *
 * The test will keep running in a loop until stopped. Info messages are progress
 * updates SEVERE messages indicate a test has failed.
 *
 * @author Tim Boura - Zero Separation
 */
public class TestSQLService implements Serializable, AppListener {

    /** The version of the serialized form of this class. */
    private static final long serialVersionUID = 1L;

    /** Used for all logging from this class. */
    private static final Logger logger =
        Logger.getLogger(TestSQLService.class.getName());

    public static class TimerTask implements Serializable, Task, ManagedObject {

        int testPhase;
        SQLConnection validConnection;

        TimerTask() {
            logger.log(Level.INFO, "Starting test");
            testPhase = 0;

            SQLManager sql;

            try {
                sql = AppContext.getManager(SQLManager.class);
            }
            catch (ManagerNotFoundException ex) {
                logger.log(Level.SEVERE, "SQL Manager not installed!", ex);
                return;
            }

            // Create the connection here to check serialisation/deserialisation
            // of connections works correctly.
            validConnection = sql.createConnection(
                    "com.mysql.jdbc.Driver",
                    "jdbc:mysql://localhost/pdssqltest?user=PDSSQLTest&password=testing");

            AppContext.getDataManager().setBinding("TestTimerTask", this);
            AppContext.getTaskManager().schedulePeriodicTask(this, 10000, 10000);
        }


        public void run() {

            SQLManager sql;

            try {
                sql = AppContext.getManager(SQLManager.class);
            }
            catch (ManagerNotFoundException ex) {
                logger.log(Level.SEVERE, "SQL Manager not installed!", ex);
                return;
            }

            switch (testPhase++) {
                case 0:
                    logger.log(Level.INFO, "Creating database table to use.\n");
                    validConnection.performQuery(new SQLStatement("DROP TABLE IF EXISTS `Foobar`"),
                            new CreateTableResultHandler(validConnection));
                    break;

                case 1:
                   /* Pre-check
                    *
                    * Update/Query with some parameters not set
                    * Set parameter with index out of range
                    */
                    try {
                        sql.performQuery(validConnection, new SQLStatement("select * from foobar where `key` = ?"));
                        logger.log(Level.SEVERE, "INCORRECT: Sent a query with one ? and no paramaters - with no failure");
                    } catch (IllegalArgumentException ex) {
                        logger.log(Level.INFO, "Correctly got an illegal argument exception for a query with one ? and no paramaters");
                    }

                    try {
                        validConnection.performQuery(new SQLStatement("select * from foobar where `key` = ?"));
                        logger.log(Level.SEVERE, "INCORRECT: Sent a query with one ? and no paramaters - with no failure");
                    } catch (IllegalArgumentException ex) {
                        logger.log(Level.INFO, "Correctly got an illegal argument exception for a query with one ? and no paramaters");
                    }

                    SQLStatement statement = new SQLStatement("select * from foobar where `key`=? and value like ?");
                    statement.setInt(1, 17);
                    try {
                        validConnection.performQuery(statement);
                        logger.log(Level.SEVERE, "INCORRECT: Sent a query with two ? and one paramater - with no failure");
                    } catch (IllegalArgumentException ex) {
                        logger.log(Level.INFO, "Correctly got an illegal argument exception for a query with two ? and one paramater");
                    }

                    try {
                        statement.setInt(-3, 17);
                        logger.log(Level.SEVERE, "INCORRECT: Set negative index with no exception thrown!");
                    } catch (IndexOutOfBoundsException ex) {
                        logger.log(Level.INFO, "IndexOutOfBoundsException correctly thrown");
                    }
                    try {
                        statement.setInt(20, 71);
                        logger.log(Level.SEVERE, "INCORRECT: Set too high index with no exception thrown!");
                    } catch (IndexOutOfBoundsException ex) {
                        logger.log(Level.INFO, "IndexOutOfBoundsException correctly thrown");
                    }

                    statement.setString(2, "bob");
                    try {
                        validConnection.performQuery(statement);
                        logger.log(Level.INFO, "Successfully sent a query with two paramaters");
                    } catch (IllegalArgumentException ex) {
                        logger.log(Level.INFO, "INCORRECT: Got an illegal argument exception for a query with both params set");
                    }

                    break;

                case 2:
                   /* Connection
                    *
                    * Connecting to invalid driver
                    * Connecting to non-existent server
                    * Connecting to server with invalid login details
                    * Connecting to non-existing database inside server
                    * Connecting succesfully
                    */
                    SQLStatement query = new SQLStatement("select * from Foobar");
                    logger.log(Level.INFO, "Set up basic statement to test connections.");


                    SQLConnection conn = sql.createConnection(
                            "invaliddriver",
                            "jdbc:mysql://localhost/pdssqltest?user=PDSSQLTest&password=testing");

                    conn.performQuery(query,
                            new ShouldSucceedResultHandler(false, testPhase, 0, SQLResult.Result.CONNECTION_FAILURE));
                    logger.log(Level.INFO, "Performing query with invalid driver.");

                    conn = sql.createConnection(
                            "com.mysql.jdbc.Driver",
                            "jdbc:mysql://nosuchserver/pdssqltest?user=PDSSQLTest&password=testing");
                    logger.log(Level.INFO, "Performing query with invalid connection.");

                    conn.performQuery(query,
                            new ShouldSucceedResultHandler(false, testPhase, 1, SQLResult.Result.CONNECTION_FAILURE));

                    conn = sql.createConnection(
                            "com.mysql.jdbc.Driver",
                            "jdbc:mysql://localhost/pdssqltest?user=PDSSQLTest&password=wrongpassword");
                    logger.log(Level.INFO, "Performing query with invalid password.");

                    conn.performQuery(query,
                            new ShouldSucceedResultHandler(false, testPhase, 2, SQLResult.Result.CONNECTION_FAILURE));

                    conn = sql.createConnection(
                            "com.mysql.jdbc.Driver",
                            "jdbc:mysql://localhost/wrongdatabase?user=PDSSQLTest&password=testing");
                    logger.log(Level.INFO, "Performing query with invalid database.");

                    conn.performQuery(query,
                            new ShouldSucceedResultHandler(false, testPhase, 3, SQLResult.Result.CONNECTION_FAILURE));

                    logger.log(Level.INFO, "Performing valid query.");
                    conn = validConnection;
                    conn.performQuery(query,
                            new ShouldSucceedResultHandler(true, testPhase, 4, SQLResult.Result.SUCCESS));
                    break;

                case 3:
                   /* Update
                    *
                    * Invalid update
                    * Update with no callback/return function
                    * Update that changes 0 rows
                    * Update that returns generated keys
                    */

                    logger.log(Level.INFO, "Performing invalid update.");
                    query = new SQLStatement("update Foobar set bob=? where `key`=?");
                    query.setInt(1, 11);
                    query.setInt(2, 12);
                    validConnection.performQuery(query,
                            new ShouldSucceedResultHandler(false, testPhase, 0, SQLResult.Result.SQL_FAILURE));

                    logger.log(Level.INFO, "Performing update that changes 0 rows.");
                    query = new SQLStatement("update Foobar set value=? where `key`=?");
                    query.setString(1, "bob");
                    query.setInt(2, 12);
                    validConnection.performQuery(query,
                            new ShouldSucceedResultHandler(true, testPhase, 1, SQLResult.Result.SUCCESS));

                    logger.log(Level.INFO, "Performing update that changes multiple rows.");
                    query = new SQLStatement("insert into Foobar (Value) values (?), (?), (?), (?), (?);", Statement.RETURN_GENERATED_KEYS);
                    query.setString(1, "time");
                    query.setString(2, "is");
                    query.setString(3, "the");
                    query.setString(4, "simplest");
                    query.setString(5, "thing");
                    validConnection.performQuery(query,
                            new GeneratedKeysResultHandler(testPhase, 2, 1, 5));
                    break;

                case 4:
                   /* Update part two
                    *
                    * Update that changes >0 rows
                    * Change query after sending
                    */
                    query = new SQLStatement("update Foobar set Value=? where `key`=?");
                    query.setString(1, "complex");
                    query.setInt(2, 4);
                    validConnection.performQuery(query,
                            new RowsModifiedResultHandler(testPhase, 0, 1));
                    query.setString(1, "a");
                    query.setInt(2, 3);
                    validConnection.performQuery(query,
                            new RowsModifiedResultHandler(testPhase, 1, 1));

                case 5:
                   /* Query
                    *
                    * Query that returns no results
                    * Query that returns results
                    */
                    query = new SQLStatement("select * from Foobar where Value like ?");
                    query.setString(1, "artichoke");
                    validConnection.performQuery(query,
                            new QueryResultHandler(testPhase, 0, new String[0]));

                    String[] expectedResult = {"time", "is", "a", "complex", "thing"};

                    query.setString(1, "%");
                    validConnection.performQuery(query,
                            new QueryResultHandler(testPhase, 0, expectedResult));

                    break;

                case 6:
                    /*
                     * Query and retrieve results using named columns
                     */
                    logger.log(Level.INFO, "Checking named columns.");
                    query = new SQLStatement("select `Key`, `Value` from Foobar");
                    validConnection.performQuery(query,
                            new NamedColumnsResultHandler(testPhase, 5));

                    break;

                case 7:
                    /*
                     * Test results are as expected when the transactional task
                     * recieving the results gets retried.
                     */
                    logger.log(Level.INFO, "Checking transactional task retry.");
                    query = new SQLStatement("select `Key`, `Value` from Foobar");
                    validConnection.performQuery(query,
                            new TransactionalRetryResultHandler(testPhase, "time"));
                    break;

                case 8:
                   /* Clean Up
                    *
                    * Delete tables etc
                    */
                    logger.log(Level.INFO, "Cleaning up test environment (dropping sql tables).");
                    validConnection.performQuery(new SQLStatement(
                            "DROP TABLE `pdssqltest`.`Foobar`"),
                            new ShouldSucceedResultHandler(true, testPhase, 0, SQLResult.Result.SUCCESS));
                    testPhase = 0;
                    break;

            }

        }

    }

    public static class ShouldSucceedResultHandler implements SQLResultHandler, Serializable {
        boolean shouldSucceed;
        SQLResult.Result expectedResult;
        int stage;
        int testCount;

        ShouldSucceedResultHandler(boolean shouldSucceed, int stage, int testCount, SQLResult.Result expectedResult) {
            this.shouldSucceed = shouldSucceed;
            this.stage = stage;
            this.testCount = testCount;
            this.expectedResult = expectedResult;
        }

        public void SQLQueryResult(SQLResult result) {
            // Check we are properly running in a transaction by accessing the data manager
            AppContext.getDataManager().getBinding("TestTimerTask");

            if (shouldSucceed && result.getResult()==SQLResult.Result.SUCCESS) {
                logger.log(Level.INFO, stage+", "+testCount+": Correctly succeeded");
            } else {
                if (result.getResult() == expectedResult)
                    logger.log(Level.INFO, stage+", "+testCount+": Failed (as expected) with Result "+result.getResult().toString());
                else {
                    logger.log(Level.SEVERE, stage+", "+testCount+": INCORRECTLY Failed with Result "+result.getResult()+
                            ", Failure:"+result.getFailure()+"\n"+result.getQuery());
                }
            }
        }

    }

    public static class GeneratedKeysResultHandler implements SQLResultHandler, Serializable {
        int stage;
        int testCount;
        int firstKey;
        int lastKey;

        GeneratedKeysResultHandler(int stage, int testCount, int firstKey, int lastKey) {
            this.stage = stage;
            this.testCount = testCount;
            this.firstKey = firstKey;
            this.lastKey = lastKey;
        }

        public void SQLQueryResult(SQLResult result) {
            // Check we are properly running in a transaction by accessing the data manager
            AppContext.getDataManager().getBinding("TestTimerTask");

            if (result.getResult()==SQLResult.Result.SUCCESS) {
                ResultSet keys = result.getGeneratedKeys();
                if (keys == null) {
                    logger.log(Level.INFO, stage+", "+testCount+": Succeeded but INCORRECTLY had no generated keys"+"\n"+result.getQuery());
                } else {
                    try {
                        while (keys.next()) {
                            if (keys.getRow()>lastKey-firstKey+1)
                                logger.log(Level.SEVERE, stage+", "+testCount+": INCORRECTLY got row "+keys.getRow()+" when at maximum expected "+(lastKey-firstKey+1)+" rows");
                            if (keys.getInt(1)<firstKey)
                                logger.log(Level.SEVERE, stage+", "+testCount+": INCORRECTLY got key "+keys.getInt(1)+" when at minimum expected "+firstKey);
                            if (keys.getInt(1)>lastKey)
                                logger.log(Level.SEVERE, stage+", "+testCount+": INCORRECTLY got key "+keys.getInt(1)+" when at minimum expected "+lastKey);
                        }

                        keys.last();

                        if (keys.getRow()<lastKey-firstKey) {
                            logger.log(Level.SEVERE, stage+", "+testCount+": INCORRECTLY only got "+keys.getRow()+" keys when expected "+(lastKey-firstKey)+" rows"+"\n"+result.getQuery());
                        }
                    } catch (SQLException ex) {
                        logger.log(Level.SEVERE, stage+", "+testCount+": INCORRECTLY got SQL Exception"+ ex+"\n"+result.getQuery());
                    }
                }
                logger.log(Level.INFO, stage+", "+testCount+": Correctly succeeded");
            } else {
                logger.log(Level.SEVERE, stage+", "+testCount+": INCORRECTLY Failed with Result "+result.getResult()+
                        ", Failure:"+result.getFailure()+"\n"+result.getQuery());
            }
        }

    }


    public static class CreateTableResultHandler implements SQLResultHandler, Serializable {

        SQLConnection validConnection;

        public CreateTableResultHandler(SQLConnection validConnection) {
            this.validConnection = validConnection;
        }

        public void SQLQueryResult(SQLResult result) {
            if (result.getResult() == SQLResult.Result.SUCCESS) {
                logger.log(Level.INFO, "Successfully dropped old table, creating new one.");
                validConnection.performQuery(new SQLStatement(
                        "CREATE TABLE `pdssqltest`.`Foobar` ("+
                        " `Key` INTEGER UNSIGNED NOT NULL AUTO_INCREMENT,"+
                        "  `Value` VARCHAR(45) NOT NULL,"+
                        "  PRIMARY KEY (`Key`)"+
                        ")"),
                        new ShouldSucceedResultHandler(true, 0, 1, SQLResult.Result.SUCCESS));
            } else
                logger.log(Level.SEVERE, "INCORRECTLY could not drop if exists old table: "+result.getResult()+"\n"+result.getQuery()+"\n",result.getFailure());
        }
    }

    public static class RowsModifiedResultHandler implements SQLResultHandler, Serializable {

        int stage;
        int testCount;
        int expectedModified;

        public RowsModifiedResultHandler(int stage, int testCount, int expectedModified) {
            this.stage = stage;
            this.testCount = testCount;
            this.expectedModified = expectedModified;
        }

        public void SQLQueryResult(SQLResult result) {
            if (result.getResult() == SQLResult.Result.SUCCESS) {
                if (result.getUpdateCount() == expectedModified) {
                    logger.log(Level.INFO, stage+", "+testCount+": Correctly succeeded with "+expectedModified+" rows changed");
                } else {
                    logger.log(Level.SEVERE, stage+", "+testCount+": Succeeded but INCORRECTLY had "+result.getUpdateCount()+" modified rows, expected "+expectedModified+"\n"+result.getQuery());
                }
            } else
                logger.log(Level.SEVERE, stage+", "+testCount+": INCORRECTLY Failed with Result "+result.getResult()+
                        ", Failure:"+result.getFailure()+"\n"+result.getQuery());
        }
    }

    public static class QueryResultHandler implements SQLResultHandler, Serializable {
        int stage;
        int testCount;
        String[] expectedResult;

        QueryResultHandler(int stage, int testCount, String[] expectedResult) {
            this.stage = stage;
            this.testCount = testCount;
            this.expectedResult = expectedResult;
        }

        public void SQLQueryResult(SQLResult result) {
            // Check we are properly running in a transaction by accessing the data manager
            AppContext.getDataManager().getBinding("TestTimerTask");

            if (result.getResult()==SQLResult.Result.SUCCESS) {
                ResultSet results = result.getResultSet();
                try {
                    while (results.next()) {
                        if (!results.getString(2).equals(expectedResult[results.getInt(1)-1])) {
                            logger.log(Level.SEVERE, stage+", "+testCount+": INCORRECTLY got result "+results.getInt(1)+", "+results.getString(2));
                        }
                    }

                    results.last();

                    if (results.getRow()<expectedResult.length) {
                        logger.log(Level.SEVERE, stage+", "+testCount+": INCORRECTLY only got "+results.getRow()+" rows when expected "+expectedResult.length+" rows"+"\n"+result.getQuery());
                    }
                } catch (SQLException ex) {
                    logger.log(Level.SEVERE, stage+", "+testCount+": INCORRECTLY got SQL Exception"+ ex+"\n"+result.getQuery());
                }
                logger.log(Level.INFO, stage+", "+testCount+": Correctly succeeded");
            } else {
                logger.log(Level.SEVERE, stage+", "+testCount+": INCORRECTLY Failed with Result "+result.getResult()+
                        ", Failure:"+result.getFailure()+"\n"+result.getQuery());
            }
        }

    }

    public static class NamedColumnsResultHandler implements SQLResultHandler, Serializable {
        int stage;
        int expectedResultCount;

        NamedColumnsResultHandler(int stage, int expectedResultCount) {
            this.stage = stage;
            this.expectedResultCount = expectedResultCount;
        }

        public void SQLQueryResult(SQLResult result) {
            // Check we are properly running in a transaction by accessing the data manager
            AppContext.getDataManager().getBinding("TestTimerTask");

            if (result.getResult()==SQLResult.Result.SUCCESS) {
                ResultSet results = result.getResultSet();
                try {
                    while (results.next()) {
                        if (results.getInt(1) != results.getInt("Key")) {
                            logger.log(Level.SEVERE, stage+": INCORRECTLY Integer result of column 1 ("+results.getInt(1)+") != column \"Key\" ("+results.getInt("Key")+")");
                        }
                        if (!results.getString(2).equals(results.getString("Value"))) {
                            logger.log(Level.SEVERE, stage+": INCORRECTLY String result of column 2 ("+results.getString(2)+") != column \"Key\" ("+results.getString("Value")+")");
                        }
                    }

                    results.last();

                    if (results.getRow()<expectedResultCount) {
                        logger.log(Level.SEVERE, stage+": INCORRECTLY only got "+results.getRow()+" rows when expected "+expectedResultCount+" rows\n"+result.getQuery());
                    }
                } catch (SQLException ex) {
                    logger.log(Level.SEVERE, stage+": INCORRECTLY got SQL Exception"+ ex+"\n"+result.getQuery());
                }
                logger.log(Level.INFO, stage+": Correctly succeeded");
            } else {
                logger.log(Level.SEVERE, stage+": INCORRECTLY Failed with Result "+result.getResult()+
                        ", Failure:"+result.getFailure()+"\n"+result.getQuery());
            }
        }

    }

    /**
     * WARNING: Uses static counter, only one instance of this class can be active
     * at a time!
     */
    public static class TransactionalRetryResultHandler implements SQLResultHandler, Serializable {
        int stage;
        String expectedResult;
        static int staticFailureCount;

        TransactionalRetryResultHandler(int stage, String expectedResult) {
            this.stage = stage;
            this.expectedResult = expectedResult;
            staticFailureCount = 0;
        }

        public void SQLQueryResult(SQLResult result) {
            // Check we are properly running in a transaction by accessing the data manager
            AppContext.getDataManager().getBinding("TestTimerTask");

            if (result.getResult()==SQLResult.Result.SUCCESS) {
                ResultSet results = result.getResultSet();
                try {
                    if (results.getRow()!=0) {
                        logger.log(Level.SEVERE, stage+", "+staticFailureCount+": INCORRECTLY at row "+results.getRow()+" not 0\n"+result.getQuery());
                    }
                    if (results.next() == false) {
                        logger.log(Level.SEVERE, stage+", "+staticFailureCount+": INCORRECTLY not able to get first row of results\n"+result.getQuery());
                    } else if (!results.getString(2).equals(expectedResult)) {
                        logger.log(Level.SEVERE, stage+", "+staticFailureCount+": INCORRECTLY got result "+results.getInt(1)+", "+results.getString(2));
                    }

                } catch (SQLException ex) {
                    logger.log(Level.SEVERE, stage+", "+staticFailureCount+": INCORRECTLY got SQL Exception"+ ex+"\n"+result.getQuery());
                }
                if (staticFailureCount<5) {
                    logger.log(Level.INFO, stage+", "+staticFailureCount+": Completed processing, throwing retriable exception");
                    staticFailureCount++;
                    throw new RetriableException();
                } else {
                    logger.log(Level.INFO, stage+", "+staticFailureCount+": Completed processing, throwing non-retriable exception");
                    staticFailureCount=0;

                    throw new NonRetriableException();
                }

            } else {
                logger.log(Level.SEVERE, stage+", "+staticFailureCount+": INCORRECTLY Failed with Result "+result.getResult()+
                        ", Failure:"+result.getFailure()+"\n"+result.getQuery());
            }
        }

        static class RetriableException extends RuntimeException implements ExceptionRetryStatus {

            public boolean shouldRetry() {
                return true;
            }

        }
        static class NonRetriableException extends RuntimeException {

        }

    }


    /**
     * This function is called once and only once when the server first starts up
     * in order to initialise it. It is not called after server reboots unless the
     * database is deleted restoring everything to basic settings.
     *
     * This function runs a number of tests against indexed maps.
     *
     * @param properties The properties with which the server was started up.
     */
    public void initialize(Properties properties) {
        TimerTask task = new TimerTask();

        AppContext.getDataManager().setBinding("TestTimer", task);
    }


    public ClientSessionListener loggedIn(ClientSession session) {
        return null;
    }


}

